import React, { useState, useRef, useEffect, useCallback } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  Dimensions,
  TextInput,
  ScrollView,
  KeyboardAvoidingView,
  Keyboard,
  Platform,
  Image,
  ActivityIndicator,
  StatusBar,
  Modal,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation, useRoute } from '@react-navigation/native';
import { Dropdown } from 'react-native-element-dropdown';
import LinearGradient from 'react-native-linear-gradient';
import DateTimePicker from '@react-native-community/datetimepicker';
import AsyncStorage from '@react-native-async-storage/async-storage';
import database from '@react-native-firebase/database';
import auth from '@react-native-firebase/auth';
import Toast from 'react-native-toast-message';
import NetInfo from '@react-native-community/netinfo';

const { width, height } = Dimensions.get('window');

const CustomInputField = ({
  label,
  value,
  onChangeText,
  isRequired = false,
  showError = false,
  errorMessage = '',
  icon,
  onIconPress,
  editable = true,
  showSelectButton = false,
  onSelectPress,
  noBackground = false,
  noInput = false,
  onFocus,
  onLayout,
  onPress,
  style,
  inputStyle,
  accessibilityLabel,
  accessibilityRole = 'text',
  ...props
}) => (
  <View style={[inputStyles.container, noInput && { marginBottom: width * 0.02 }, style]} onLayout={onLayout}>
    {noInput ? (
      <View style={inputStyles.noInputContainer}>
        <Text style={inputStyles.label}>
          {label}
          {isRequired && <Text style={inputStyles.requiredError}> *</Text>}
        </Text>
        <TouchableOpacity
          style={inputStyles.selectButton}
          onPress={onSelectPress}
          accessibilityLabel={`Select ${label}`}
          accessibilityRole="button"
        >
          <Text style={inputStyles.selectText}>Select</Text>
          <MaterialIcons name="arrow-drop-down" size={width * 0.06} color="#333" />
        </TouchableOpacity>
      </View>
    ) : (
      <>
        <Text style={inputStyles.label}>
          {label}
          {isRequired && <Text style={inputStyles.requiredError}> *</Text>}
        </Text>
        <TouchableOpacity
          style={inputStyles.inputContainer}
          onPress={!editable ? onPress : undefined}
          activeOpacity={!editable ? 0.7 : 1}
        >
          <TextInput
            style={[
              inputStyles.input,
              showError ? inputStyles.inputError : null,
              !editable && inputStyles.disabledInput,
              noBackground && inputStyles.noBackground,
              inputStyle,
            ]}
            value={value}
            onChangeText={onChangeText}
            editable={editable}
            placeholderTextColor="#888"
            onFocus={onFocus}
            accessibilityLabel={accessibilityLabel || label}
            accessibilityRole={accessibilityRole}
            {...props}
          />
          {icon && (
            <TouchableOpacity
              style={inputStyles.inputIcon}
              onPress={onIconPress || onPress}
              accessibilityLabel={`Select ${label}`}
              accessibilityRole="button"
            >
              {icon}
            </TouchableOpacity>
          )}
          {showSelectButton && (
            <TouchableOpacity
              style={inputStyles.selectButton}
              onPress={onSelectPress}
              accessibilityLabel={`Select ${label}`}
              accessibilityRole="button"
            >
              <Text style={inputStyles.selectText}>Select</Text>
              <MaterialIcons name="arrow-drop-down" size={width * 0.06} color="#333" />
            </TouchableOpacity>
          )}
        </TouchableOpacity>
        {showError && <Text style={inputStyles.errorText}>{errorMessage}</Text>}
      </>
    )}
  </View>
);

// Helper function to show toast messages
const showToast = (type, title, message) => {
  Toast.show({
    type: type,
    text1: title,
    text2: message,
    position: 'bottom',
  });
};

const AdminHrAppliApply = () => {
  const navigation = useNavigation();
  const route = useRoute();
  const scrollViewRef = useRef(null);
  const [isOnline, setIsOnline] = useState(true);
  const [fullName, setFullName] = useState('');
  const [employeeNo, setEmployeeNo] = useState('');
  const [country, setCountry] = useState([]);
  const [project, setProject] = useState([]);
  const [department, setDepartment] = useState('');
  const [nrcNo, setNrcNo] = useState('');
  const [applicationType, setApplicationType] = useState(null);
  const [applicationTypeOpen, setApplicationTypeOpen] = useState(false);
  const [isKeyboardVisible, setIsKeyboardVisible] = useState(false);
  const [keyboardHeight, setKeyboardHeight] = useState(0);

  // State variables for Leave Application fields
  const [leaveStartDate, setLeaveStartDate] = useState('');
  const [leaveEndDate, setLeaveEndDate] = useState('');
  const [reportDate, setReportDate] = useState('');
  const [reasonOfLeave, setReasonOfLeave] = useState('');
  const [addressWhileOnLeave, setAddressWhileOnLeave] = useState('');
  const [otherContactName, setOtherContactName] = useState('');
  const [otherContactCell, setOtherContactCell] = useState('');
  const [signatureDate, setSignatureDate] = useState('');
  const [leaveSignature, setLeaveSignature] = useState(null);
  const [leaveSignatureError, setLeaveSignatureError] = useState(false);
  const [leaveSignatureLoading, setLeaveSignatureLoading] = useState(false);

  // State variables for Funeral Grant Form fields
  const [funeralSpouseChildData, setFuneralSpouseChildData] = useState([
    { id: '1', date: '', description: 'STANDARD COFFIN', amount: '' },
    { id: '2', date: '', description: 'CASH', amount: '' },
    { id: '3', date: '', description: 'TRANSPORT', amount: '' },
  ]);
  const [funeralMotherFatherData, setFuneralMotherFatherData] = useState([
    { id: '1', date: '', description: 'CASH', amount: '' },
    { id: '2', date: '', description: 'TRANSPORT', amount: '' },
  ]);
  const [deceasedName, setDeceasedName] = useState('');
  const [deceasedRelationship, setDeceasedRelationship] = useState('');
  const [deceasedDate, setDeceasedDate] = useState('');
  const [funeralEmployeeSignatureDate, setFuneralEmployeeSignatureDate] = useState('');
  const [funeralFromDate, setFuneralFromDate] = useState('');
  const [funeralToDate, setFuneralToDate] = useState('');
  const [selectedFuneralRelationshipType, setSelectedFuneralRelationshipType] = useState(null);
  const [funeralRelationshipTypeOpen, setFuneralRelationshipTypeOpen] = useState(false);
  const [funeralSignature, setFuneralSignature] = useState(null);
  const [funeralSignatureError, setFuneralSignatureError] = useState(false);
  const [funeralSignatureLoading, setFuneralSignatureLoading] = useState(false);

  // State variables for Salary Advance Form fields
  const [salaryAdvanceDepartment, setSalaryAdvanceDepartment] = useState('');
  const [salaryAdvanceAmount, setSalaryAdvanceAmount] = useState('');
  const [selectedSites, setSelectedSites] = useState([]);
  const [salaryAdvanceOtherSite, setSalaryAdvanceOtherSite] = useState('');
  const [selectedRecoveryPeriod, setSelectedRecoveryPeriod] = useState(null);
  const [selectedReasons, setSelectedReasons] = useState([]);
  const [salaryAdvanceOtherReason, setSalaryAdvanceOtherReason] = useState('');
  const [salaryAdvanceSignatureDate, setSalaryAdvanceSignatureDate] = useState('');
  const [salaryAdvanceSignature, setSalaryAdvanceSignature] = useState(null);
  const [salaryAdvanceSignatureError, setSalaryAdvanceSignatureError] = useState(false);
  const [salaryAdvanceSignatureLoading, setSalaryAdvanceSignatureLoading] = useState(false);

  // State variables for Application for Appeal fields
  const [appealFormDate, setAppealFormDate] = useState('');
  const [appealManNo, setAppealManNo] = useState('');
  const [appealName, setAppealName] = useState('');
  const [appealSanction, setAppealSanction] = useState('');
  const [appealOffence, setAppealOffence] = useState('');
  const [appealOffenceDate, setAppealOffenceDate] = useState('');
  const [appealGrounds, setAppealGrounds] = useState('');
  const [appellantSignature, setAppellantSignature] = useState(null);
  const [appealDateOfAppeal, setAppealDateOfAppeal] = useState('');
  const [appellantSignatureError, setAppellantSignatureError] = useState(false);
  const [appellantSignatureLoading, setAppellantSignatureLoading] = useState(false);

  // State for Grievance Record fields
  const [grievanceDate, setGrievanceDate] = useState('');
  const [grievanceComplaints, setGrievanceComplaints] = useState([
    { name: '', mineNo: '', occupationGrade: '' },
  ]);
  const [grievancePlaceOfWork, setGrievancePlaceOfWork] = useState('');
  const [grievanceSupervisorName, setGrievanceSupervisorName] = useState('');
  const [grievanceDetails, setGrievanceDetails] = useState('');

  // State for Hospital Attendance Form
  const [selectedClinics, setSelectedClinics] = useState([]);
  const [otherClinic, setOtherClinic] = useState('');
  const [selectedShift, setSelectedShift] = useState('');

  // State for DatePicker
  const [showDatePicker, setShowDatePicker] = useState(false);
  const [datePickerValue, setDatePickerValue] = useState(new Date());
  const [currentEditingDateField, setCurrentEditingDateField] = useState(null);
  const [currentTableId, setCurrentTableId] = useState(null);
  const [datePickerMaxDate, setDatePickerMaxDate] = useState(null);

  // Country and Project dropdown options (copied from CreateIncidentReport)
  const countryItems = [
    { label: 'Zambia', value: 'Zambia' },
    { label: 'DRC', value: 'DRC' },
  ];
  const projectOptionsByCountry = {
    Zambia: [
      { label: 'Kansanshi', value: 'Kansanshi' },
      { label: 'Kalumbila', value: 'Kalumbila' },
      { label: 'Kimteto', value: 'Kimteto' },
      { label: 'Kobold', value: 'Kobold' },
      { label: 'FQM Exploration', value: 'FQM Exploration' },
      { label: 'Mimosa', value: 'Mimosa' },
    ],
    DRC: [
      { label: 'Lubumbashi', value: 'Lubumbashi' },
      { label: 'Musompo', value: 'Musompo' },
      { label: 'IME', value: 'IME' },
      { label: 'Kamoa', value: 'Kamoa' },
    ],
  };

  // Add state for country and project dropdowns
  const [selectedCountry, setSelectedCountry] = useState('Zambia');
  const [selectedProject, setSelectedProject] = useState('');
  const [projectItems, setProjectItems] = useState(projectOptionsByCountry['Zambia']);

  // Set default country to Zambia on mount if not set
  useEffect(() => {
    if (!selectedCountry) {
      setSelectedCountry('Zambia');
    }
  }, []);

  // Update project options when country changes
  useEffect(() => {
    setProjectItems(projectOptionsByCountry[selectedCountry] || []);
    setSelectedProject(''); // Reset project when country changes
  }, [selectedCountry]);

  const applicationTypes = [
    { label: 'Leave Application', value: 'leave_application' },
    { label: 'Funeral Grant Form', value: 'funeral_grant_form' },
    { label: 'Salary Advance Application', value: 'salary_advance_application' },
    { label: 'Application for Appeal', value: 'application_for_appeal' },
    { label: 'GRIEVANCE RECORD', value: 'grievance_record' },
    { label: 'Hospital Attendance Form', value: 'hospital_attendance_form' },
  ];

  const funeralRelationshipOptions = [
    { label: 'SPOUSE AND BIOLOGICAL CHILD', value: 'spouse_child' },
    { label: 'MOTHER AND FATHER', value: 'mother_father' },
    { label: 'BOTH', value: 'both' },
  ];

  const sitesOptions = [
    { label: 'KALUMBILA', value: 'kalumbila' },
    { label: 'KANSANSHI', value: 'kansanshi' },
    { label: 'KIMITETO', value: 'kimteto' },
    { label: 'OTHERS', value: 'others' },
  ];

  const reasonsOptions = [
    { label: 'Medical Assistance', value: 'medical_assistance' },
    { label: 'Funeral', value: 'funeral' },
    { label: 'School Fees', value: 'school_fees' },
    { label: 'Tools Recovery', value: 'tools_recovery' },
    { label: 'Others', value: 'others' },
  ];

  const recoveryPeriodOptions = [
    { label: '1 Instalment', value: '1_instalment' },
    { label: '2 Instalments', value: '2_instalments' },
    { label: '3 Instalments', value: '3_instalments' },
  ];

  const clinicOptions = [
    { label: 'Marybegg Clinic Kalumbila', value: 'marybegg_kalumbila' },
    { label: 'Solwezi General Hospital', value: 'solwezi_general' },
    { label: 'Kalumbila District Hospital', value: 'kalumbila_district' },
    { label: 'Hilltop Hospital', value: 'hilltop' },
    { label: 'Marybegg Clinic Kansanshi', value: 'marybegg_kansanshi' },
    { label: 'Others', value: 'others' },
  ];

  const [layoutPositions, setLayoutPositions] = useState({});

  // Network status handling
  useEffect(() => {
    const unsubscribe = NetInfo.addEventListener((state) => {
      const newOnlineStatus = state.isConnected || false;
      setIsOnline(newOnlineStatus);
      if (newOnlineStatus) {
        syncOfflineApplications();
      }
    });

    return () => unsubscribe();
  }, []);

  // Show instruction toast when component mounts
  useEffect(() => {
    const timer = setTimeout(() => {
      showToast('info', 'Instructions', 'Please select an application type to proceed with your application.');
    }, 1000); // Show after 1 second to ensure component is fully loaded

    return () => clearTimeout(timer);
  }, []);

  // Fetch user data from Firebase and AsyncStorage
  useEffect(() => {
    const fetchUserData = async () => {
      try {
        // Retrieve and validate userUid
        const userUid = await AsyncStorage.getItem('userUid');
        if (!userUid) {
          showToast('error', 'Authentication Error', 'User ID not found. Please log in again.');
          navigation.reset({ index: 0, routes: [{ name: 'Login' }] });
          return;
        }

        const currentUser = auth().currentUser;
        if (!currentUser || currentUser.uid !== userUid) {
          showToast('error', 'Authentication Error', 'Invalid session. Please log in again.');
          await AsyncStorage.multiRemove(['userUid', 'userData']);
          navigation.reset({ index: 0, routes: [{ name: 'Login' }] });
          return;
        }

        if (!isOnline) {
          // Load cached data when offline
          const cachedData = await AsyncStorage.getItem('userData');
          if (cachedData) {
            const parsedData = JSON.parse(cachedData);
            setFullName(parsedData.name || '');
            setEmployeeNo(parsedData.employeeNumber || '');
            setCountry(Array.isArray(parsedData.countries) ? parsedData.countries : []);
            setProject(Array.isArray(parsedData.projects) ? parsedData.projects : []);
            setDepartment(parsedData.department || '');
            setNrcNo(parsedData.nrcIdNumber || '');
            showToast('info', 'Offline Mode', 'Loaded cached user data.');
          } else {
            showToast('warning', 'Offline Mode', 'No cached user data available.');
          }
          return;
        }

        // Fetch from Firebase
        const userRef = database().ref(`/Globalusers/admins/${userUid}`);
        userRef.on('value', (snapshot) => {
          const userData = snapshot.val();
          if (userData) {
            setFullName(userData.name || '');
            setEmployeeNo(userData.employeeNumber || '');
            setCountry(Array.isArray(userData.countries) ? userData.countries : []);
            setProject(Array.isArray(userData.projects) ? userData.projects : []);
            setDepartment(userData.department || '');
            setNrcNo(userData.nrcIdNumber || '');
            // Cache data in AsyncStorage
            AsyncStorage.setItem('userData', JSON.stringify({
              name: userData.name,
              employeeNumber: userData.employeeNumber,
              countries: userData.countries,
              projects: userData.projects,
              department: userData.department,
              nrcIdNumber: userData.nrcIdNumber,
            }));
          } else {
            // Fallback to AsyncStorage
            const cachedData =  AsyncStorage.getItem('userData');
            if (cachedData) {
              const parsedData = JSON.parse(cachedData);
              setFullName(parsedData.name || '');
              setEmployeeNo(parsedData.employeeNumber || '');
              setCountry(Array.isArray(parsedData.countries) ? parsedData.countries : []);
              setProject(Array.isArray(parsedData.projects) ? parsedData.projects : []);
              setDepartment(parsedData.department || '');
              setNrcNo(parsedData.nrcIdNumber || '');
              showToast('warning', 'Offline Mode', 'Loaded cached user data due to missing database entry.');
            } else {
              showToast('error', 'Error', 'User data not found in database or cache.');
            }
          }
        });

        return () => userRef.off();
      } catch (error) {
        console.error('Error fetching user data:', error);
        // Fallback to AsyncStorage
        const cachedData = await AsyncStorage.getItem('userData');
        if (cachedData) {
          const parsedData = JSON.parse(cachedData);
          setFullName(parsedData.name || '');
          setEmployeeNo(parsedData.employeeNumber || '');
          setCountry(Array.isArray(parsedData.countries) ? parsedData.countries : []);
          setProject(Array.isArray(parsedData.projects) ? parsedData.projects : []);
          setDepartment(parsedData.department || '');
          setNrcNo(parsedData.nrcIdNumber || '');
          showToast('warning', 'Offline Mode', 'Loaded cached user data due to network error.');
        } else {
          showToast('error', 'Error', 'Failed to load user data. Please check your connection.');
        }
      }
    };

    fetchUserData();
  }, [navigation, isOnline]);

  // Keyboard handling
  useEffect(() => {
    const keyboardDidShowListener = Keyboard.addListener('keyboardDidShow', (e) => {
      setIsKeyboardVisible(true);
      setKeyboardHeight(e.endCoordinates.height);
    });
    const keyboardDidHideListener = Keyboard.addListener('keyboardDidHide', () => {
      setIsKeyboardVisible(false);
      setKeyboardHeight(0);
    });

    return () => {
      keyboardDidShowListener.remove();
      keyboardDidHideListener.remove();
    };
  }, []);

  const handleInputFocus = useCallback(
    (field) => () => {
      if (scrollViewRef.current && layoutPositions[field]) {
        const offset = layoutPositions[field] - (height * 0.1 + keyboardHeight);
        scrollViewRef.current.scrollTo({ y: offset, animated: true });
      }
    },
    [layoutPositions, keyboardHeight]
  );

  const handleLayout = (field) => (event) => {
    const layout = event.nativeEvent.layout;
    setLayoutPositions((prev) => ({ ...prev, [field]: layout.y }));
  };

  const formatDate = (date) => {
    if (!date) return '';
    const d = new Date(date);
    const year = d.getFullYear();
    const month = String(d.getMonth() + 1).padStart(2, '0');
    const day = String(d.getDate()).padStart(2, '0');
    return `${year}-${month}-${day}`;
  };

  const onDateChange = (event, selectedDate) => {
    if (event.type === 'dismissed') {
      setShowDatePicker(false);
      return;
    }

    const currentDate = selectedDate || datePickerValue;
    setShowDatePicker(false);
    setDatePickerValue(currentDate);

    if (currentTableId) {
      if (currentTableId.startsWith('funeralSpouseChild')) {
        const updatedData = funeralSpouseChildData.map((item) =>
          item.id === currentTableId.split('-')[1] ? { ...item, date: formatDate(currentDate) } : item
        );
        setFuneralSpouseChildData(updatedData);
      } else if (currentTableId.startsWith('funeralMotherFather')) {
        const updatedData = funeralMotherFatherData.map((item) =>
          item.id === currentTableId.split('-')[1] ? { ...item, date: formatDate(currentDate) } : item
        );
        setFuneralMotherFatherData(updatedData);
      }
      setCurrentTableId(null);
    } else {
      switch (currentEditingDateField) {
        case 'leaveStartDate':
          setLeaveStartDate(formatDate(currentDate));
          break;
        case 'leaveEndDate':
          if (leaveStartDate && new Date(currentDate) < new Date(leaveStartDate)) {
            showToast('error', 'Error', 'End date must be after start date.');
            return;
          }
          setLeaveEndDate(formatDate(currentDate));
          break;
        case 'reportDate':
          if (leaveEndDate && new Date(currentDate) <= new Date(leaveEndDate)) {
            showToast('error', 'Error', 'Report date must be after end date.');
            return;
          }
          setReportDate(formatDate(currentDate));
          break;
        case 'signatureDate':
          setSignatureDate(formatDate(currentDate));
          break;
        case 'deceasedDate':
          setDeceasedDate(formatDate(currentDate));
          break;
        case 'funeralEmployeeSignatureDate':
          setFuneralEmployeeSignatureDate(formatDate(currentDate));
          break;
        case 'funeralFromDate':
          setFuneralFromDate(formatDate(currentDate));
          break;
        case 'funeralToDate':
          if (funeralFromDate && new Date(currentDate) < new Date(funeralFromDate)) {
            showToast('error', 'Error', 'To date must be after from date.');
            return;
          }
          setFuneralToDate(formatDate(currentDate));
          break;
        case 'salaryAdvanceSignatureDate':
          setSalaryAdvanceSignatureDate(formatDate(currentDate));
          break;
        case 'appealFormDate':
          setAppealFormDate(formatDate(currentDate));
          break;
        case 'appealOffenceDate':
          setAppealOffenceDate(formatDate(currentDate));
          break;
        case 'appealDateOfAppeal':
          setAppealDateOfAppeal(formatDate(currentDate));
          break;
        case 'grievanceDate':
          setGrievanceDate(formatDate(currentDate));
          break;
        default:
          break;
      }
    }
  };

  const showMode = (currentMode, field, tableId = null) => {
    setShowDatePicker(true);
    setCurrentEditingDateField(field);
    setCurrentTableId(tableId);

    let maxDate = null;
    if (
      [
        'leaveEndDate',
        'reportDate',
        'funeralToDate',
        'appealDateOfAppeal',
        'signatureDate',
        'funeralEmployeeSignatureDate',
        'salaryAdvanceSignatureDate',
        'appealFormDate',
      ].includes(field)
    ) {
      maxDate = null; // Allow future dates
    } else {
      maxDate = new Date(); // Restrict to today or past
    }
    setDatePickerMaxDate(maxDate);

    let initialDate = new Date();
    if (tableId) {
      if (tableId.startsWith('funeralSpouseChild')) {
        const item = funeralSpouseChildData.find((d) => d.id === tableId.split('-')[1]);
        initialDate = item && item.date ? new Date(item.date) : new Date();
      } else if (tableId.startsWith('funeralMotherFather')) {
        const item = funeralMotherFatherData.find((d) => d.id === tableId.split('-')[1]);
        initialDate = item && item.date ? new Date(item.date) : new Date();
      }
    } else {
      switch (field) {
        case 'leaveStartDate':
          initialDate = leaveStartDate ? new Date(leaveStartDate) : new Date();
          break;
        case 'leaveEndDate':
          initialDate = leaveEndDate ? new Date(leaveEndDate) : new Date();
          break;
        case 'reportDate':
          initialDate = reportDate ? new Date(reportDate) : new Date();
          break;
        case 'signatureDate':
          initialDate = signatureDate ? new Date(signatureDate) : new Date();
          break;
        case 'deceasedDate':
          initialDate = deceasedDate ? new Date(deceasedDate) : new Date();
          break;
        case 'funeralEmployeeSignatureDate':
          initialDate = funeralEmployeeSignatureDate ? new Date(funeralEmployeeSignatureDate) : new Date();
          break;
        case 'funeralFromDate':
          initialDate = funeralFromDate ? new Date(funeralFromDate) : new Date();
          break;
        case 'funeralToDate':
          initialDate = funeralToDate ? new Date(funeralToDate) : new Date();
          break;
        case 'salaryAdvanceSignatureDate':
          initialDate = salaryAdvanceSignatureDate ? new Date(salaryAdvanceSignatureDate) : new Date();
          break;
        case 'appealFormDate':
          initialDate = appealFormDate ? new Date(appealFormDate) : new Date();
          break;
        case 'appealOffenceDate':
          initialDate = appealOffenceDate ? new Date(appealOffenceDate) : new Date();
          break;
        case 'appealDateOfAppeal':
          initialDate = appealDateOfAppeal ? new Date(appealDateOfAppeal) : new Date();
          break;
        case 'grievanceDate':
          initialDate = grievanceDate ? new Date(grievanceDate) : new Date();
          break;
      }
    }
    setDatePickerValue(initialDate);
  };

  const showDatePickerDialog = (field, tableId = null) => {
    showMode('date', field, tableId);
  };

  const handleFuneralAmountChange = (tableType, id, value) => {
    const cleanedValue = value.replace(/[^0-9.]/g, '');
    const parts = cleanedValue.split('.');
    if (parts.length > 2) return;
    if (parts[1] && parts[1].length > 2) return;

    if (tableType === 'spouseChild') {
      const updatedData = funeralSpouseChildData.map((item) =>
        item.id === id ? { ...item, amount: cleanedValue } : item
      );
      setFuneralSpouseChildData(updatedData);
    } else if (tableType === 'motherFather') {
      const updatedData = funeralMotherFatherData.map((item) =>
        item.id === id ? { ...item, amount: cleanedValue } : item
      );
      setFuneralMotherFatherData(updatedData);
    }
  };

  const calculateTotal = (data) => {
    return data.reduce((sum, item) => sum + (parseFloat(item.amount) || 0), 0).toFixed(2);
  };

  const toggleSiteSelection = (value) => {
    setSelectedSites((prev) => {
      if (prev.includes(value)) {
        return prev.filter((item) => item !== value);
      } else {
        return [...prev, value];
      }
    });
  };

  const toggleReasonSelection = (value) => {
    setSelectedReasons((prev) => {
      if (prev.includes(value)) {
        return prev.filter((item) => item !== value);
      } else {
        return [...prev, value];
      }
    });
  };

  const handleRecoveryPeriodSelection = (value) => {
    setSelectedRecoveryPeriod(value);
  };

  const handleOpenSignaturePad = (signatureType) => {
    let initialSignature;
    switch (signatureType) {
      case 'leave':
        initialSignature = leaveSignature;
        setLeaveSignatureLoading(true);
        break;
      case 'funeral':
        initialSignature = funeralSignature;
        setFuneralSignatureLoading(true);
        break;
      case 'salaryAdvance':
        initialSignature = salaryAdvanceSignature;
        setSalaryAdvanceSignatureLoading(true);
        break;
      case 'appeal':
        initialSignature = appellantSignature;
        setAppellantSignatureLoading(true);
        break;
      default:
        initialSignature = null;
    }

    const handleSignatureResult = (signature) => {
      if (signature) {
        switch (signatureType) {
          case 'leave':
            setLeaveSignature(signature);
            setLeaveSignatureError(false);
            break;
          case 'funeral':
            setFuneralSignature(signature);
            setFuneralSignatureError(false);
            break;
          case 'salaryAdvance':
            setSalaryAdvanceSignature(signature);
            setSalaryAdvanceSignatureError(false);
            break;
          case 'appeal':
            setAppellantSignature(signature);
            setAppellantSignatureError(false);
            break;
        }
      }
      switch (signatureType) {
        case 'leave':
          setLeaveSignatureLoading(false);
          break;
        case 'funeral':
          setFuneralSignatureLoading(false);
          break;
        case 'salaryAdvance':
          setSalaryAdvanceSignatureLoading(false);
          break;
        case 'appeal':
          setAppellantSignatureLoading(false);
          break;
      }
    };

    navigation.navigate('SignatureHRappl', {
      signatureTitle: `${signatureType.charAt(0).toUpperCase() + signatureType.slice(1)} Signature`,
      signatureType,
      initialSignature,
      onSignatureSave: handleSignatureResult,
    });
  };

  // Handle form submission
  const handleSubmit = async () => {
    if (submitting) return; // Prevent double submit
    setSubmitting(true);
    
    // Reset modal states
    setEmailStatus({
      creatorEmail: { sent: false, status: 'pending', email: '' },
      supervisorEmails: { sent: false, count: 0, status: 'pending', emails: [] }
    });
    setShowLoadingModal(true);
    initializeProcessingSteps();
    
    try {
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) {
        showToast('error', 'Authentication Error', 'User ID not found. Please log in again.');
        navigation.reset({ index: 0, routes: [{ name: 'Login' }] });
        return;
      }

      const currentUser = auth().currentUser;
      if (!currentUser || currentUser.uid !== userUid) {
        showToast('error', 'Authentication Error', 'Invalid session. Please log in again.');
        await AsyncStorage.multiRemove(['userUid', 'userData']);
        navigation.reset({ index: 0, routes: [{ name: 'Login' }] });
        return;
      }

      if (!applicationType) {
        showToast('error', 'Error', 'Please select an application type.');
        return;
      }

      let hasError = false;

      // Validate common fields
      if (!fullName) {
        showToast('error', 'Error', 'Full name is required.');
        hasError = true;
      }
      if (!employeeNo) {
        showToast('error', 'Error', 'Employee number is required.');
        hasError = true;
      }
      if (!selectedCountry) {
        showToast('error', 'Error', 'Country is required.');
        hasError = true;
      }
      if (!selectedProject) {
        showToast('error', 'Error', 'Project is required.');
        hasError = true;
      }
      if (!department) {
        showToast('error', 'Error', 'Department is required.');
        hasError = true;
      }
      if (!nrcNo) {
        showToast('error', 'Error', 'NRC number is required.');
        hasError = true;
      }

      if (applicationType === 'leave_application') {
        if (!leaveStartDate) {
          showToast('error', 'Error', 'Please select a start date for leave.');
          hasError = true;
        }
        if (!leaveEndDate) {
          showToast('error', 'Error', 'Please select an end date for leave.');
          hasError = true;
        }
        if (!reportDate) {
          showToast('error', 'Error', 'Please select a report date.');
          hasError = true;
        }
        if (!reasonOfLeave) {
          showToast('error', 'Error', 'Please provide a reason for leave.');
          hasError = true;
        }
        if (!addressWhileOnLeave) {
          showToast('error', 'Error', 'Please provide an address while on leave.');
          hasError = true;
        }
        if (!signatureDate) {
          showToast('error', 'Error', 'Please select a signature date.');
          hasError = true;
        }
        if (!leaveSignature) {
          setLeaveSignatureError(true);
          showToast('error', 'Error', 'Please provide a signature for the leave application.');
          hasError = true;
        }
      } else if (applicationType === 'funeral_grant_form') {
        if (!funeralFromDate) {
          showToast('error', 'Error', 'Please select a from date for the funeral grant.');
          hasError = true;
        }
        if (!funeralToDate) {
          showToast('error', 'Error', 'Please select a to date for the funeral grant.');
          hasError = true;
        }
        if (!selectedFuneralRelationshipType) {
          showToast('error', 'Error', 'Please select a relationship type for the funeral grant.');
          hasError = true;
        }
        if (!deceasedName) {
          showToast('error', 'Error', "Please provide the deceased's name.");
          hasError = true;
        }
        if (!deceasedRelationship) {
          showToast('error', 'Error', 'Please provide the relationship with the deceased.');
          hasError = true;
        }
        if (!deceasedDate) {
          showToast('error', 'Error', 'Please select the deceased date.');
          hasError = true;
        }
        if (!funeralEmployeeSignatureDate) {
          showToast('error', 'Error', 'Please select a signature date for the funeral grant.');
          hasError = true;
        }
        if (!funeralSignature) {
          setFuneralSignatureError(true);
          showToast('error', 'Error', 'Please provide a signature for the funeral grant.');
          hasError = true;
        }
        if (selectedFuneralRelationshipType === 'spouse_child') {
          const hasAmount = funeralSpouseChildData.some((item) => item.amount);
          if (!hasAmount) {
            showToast('error', 'Error', 'Please provide at least one amount for spouse/child funeral grant.');
            hasError = true;
          }
        } else if (selectedFuneralRelationshipType === 'mother_father') {
          const hasAmount = funeralMotherFatherData.some((item) => item.amount);
          if (!hasAmount) {
            showToast('error', 'Error', 'Please provide at least one amount for mother/father funeral grant.');
            hasError = true;
          }
        } else if (selectedFuneralRelationshipType === 'both') {
          const hasSpouseChildAmount = funeralSpouseChildData.some((item) => item.amount);
          const hasMotherFatherAmount = funeralMotherFatherData.some((item) => item.amount);
          if (!hasSpouseChildAmount) {
            showToast('error', 'Error', 'Please provide at least one amount for spouse/child funeral grant.');
            hasError = true;
          }
          if (!hasMotherFatherAmount) {
            showToast('error', 'Error', 'Please provide at least one amount for mother/father funeral grant.');
            hasError = true;
          }
        }
      } else if (applicationType === 'salary_advance_application') {
        if (!salaryAdvanceDepartment) {
          showToast('error', 'Error', 'Please provide the department for salary advance.');
          hasError = true;
        }
        if (!salaryAdvanceAmount) {
          showToast('error', 'Error', 'Please provide the amount for salary advance.');
          hasError = true;
        }
        if (selectedSites.length === 0) {
          showToast('error', 'Error', 'Please select at least one site for salary advance.');
          hasError = true;
        }
        if (selectedSites.includes('others') && !salaryAdvanceOtherSite) {
          showToast('error', 'Error', 'Please specify the other site for salary advance.');
          hasError = true;
        }
        if (!selectedRecoveryPeriod) {
          showToast('error', 'Error', 'Please select a recovery period for salary advance.');
          hasError = true;
        }
        if (selectedReasons.length === 0) {
          showToast('error', 'Error', 'Please select at least one reason for salary advance.');
          hasError = true;
        }
        if (selectedReasons.includes('others') && !salaryAdvanceOtherReason) {
          showToast('error', 'Error', 'Please specify the other reason for salary advance.');
          hasError = true;
        }
        if (!salaryAdvanceSignatureDate) {
          showToast('error', 'Error', 'Please select a signature date for salary advance.');
          hasError = true;
        }
        if (!salaryAdvanceSignature) {
          setSalaryAdvanceSignatureError(true);
          showToast('error', 'Error', 'Please provide a signature for the salary advance.');
          hasError = true;
        }
      } else if (applicationType === 'application_for_appeal') {
        if (!appealFormDate) {
          showToast('error', 'Error', 'Please select the form date for appeal.');
          hasError = true;
        }
        if (!appealManNo) {
          showToast('error', 'Error', 'Please provide the man number for appeal.');
          hasError = true;
        }
        if (!appealName) {
          showToast('error', 'Error', 'Please provide the name for appeal.');
          hasError = true;
        }
        if (!appealSanction) {
          showToast('error', 'Error', 'Please provide the sanction for appeal.');
          hasError = true;
        }
        if (!appealOffence) {
          showToast('error', 'Error', 'Please provide the offence for appeal.');
          hasError = true;
        }
        if (!appealOffenceDate) {
          showToast('error', 'Error', 'Please select the offence date for appeal.');
          hasError = true;
        }
        if (!appealGrounds) {
          showToast('error', 'Error', 'Please provide the grounds for appeal.');
          hasError = true;
        }
        if (!appealDateOfAppeal) {
          showToast('error', 'Error', 'Please select the date of appeal.');
          hasError = true;
        }
        if (!appellantSignature) {
          setAppellantSignatureError(true);
          showToast('error', 'Error', 'Please provide a signature for the appeal.');
          hasError = true;
        }
      } else if (applicationType === 'grievance_record') {
        if (!grievanceDate) {
          showToast('error', 'Error', 'Please select the date for the grievance record.');
          hasError = true;
        }
        if (grievanceComplaints.length === 0 || grievanceComplaints.every((c) => !c.name && !c.mineNo && !c.occupationGrade)) {
          showToast('error', 'Error', 'Please provide at least one valid complainant for the grievance record.');
          hasError = true;
        }
        if (!grievancePlaceOfWork) {
          showToast('error', 'Error', 'Please provide the place of work for the grievance record.');
          hasError = true;
        }
        if (!grievanceSupervisorName) {
          showToast('error', 'Error', 'Please provide the supervisor name for the grievance record.');
          hasError = true;
        }
        if (!grievanceDetails) {
          showToast('error', 'Error', 'Please provide the details of the grievance for the grievance record.');
          hasError = true;
        }
      } else if (applicationType === 'hospital_attendance_form') {
        if (selectedClinics.length === 0) {
          showToast('error', 'Error', 'Please select at least one clinic for hospital attendance.');
          hasError = true;
        }
        if (selectedClinics.includes('others') && !otherClinic) {
          showToast('error', 'Error', 'Please specify the other clinic for hospital attendance.');
          hasError = true;
        }
        if (!selectedShift) {
          showToast('error', 'Error', 'Please select a shift for hospital attendance.');
          hasError = true;
        }
      }

      if (hasError) {
        setSubmitting(false);
        setShowLoadingModal(false);
        return;
      }

      let applicationData = {
        applicationType,
        fullName,
        employeeNo,
        country: selectedCountry,
        project: selectedProject,
        department,
        nrcNo,
        submittedAt: new Date().toISOString(),
        status: 'Pending',
        Supervisor: { name: 'Pending:Supervisor', status: 'Pending' },
        HrManager: { name: 'Pending:HrManager', status: 'Pending' },
      };

      if (applicationType === 'salary_advance_application') {
        applicationData = {
          ...applicationData,
          FinanceManager: { name: 'Pending:FinanceManager', status: 'Pending' },
        };
      }

      if (applicationType === 'leave_application') {
        applicationData = {
          ...applicationData,
          leaveStartDate,
          leaveEndDate,
          reportDate,
          reasonOfLeave,
          addressWhileOnLeave,
          otherContactName,
          otherContactCell,
          signatureDate,
          leaveSignature,
        };
      } else if (applicationType === 'funeral_grant_form') {
        applicationData = {
          ...applicationData,
          funeralFromDate,
          funeralToDate,
          selectedFuneralRelationshipType,
          funeralSpouseChildData:
            selectedFuneralRelationshipType === 'spouse_child' || selectedFuneralRelationshipType === 'both'
              ? funeralSpouseChildData
              : null,
          funeralMotherFatherData:
            selectedFuneralRelationshipType === 'mother_father' || selectedFuneralRelationshipType === 'both'
              ? funeralMotherFatherData
              : null,
          deceasedName,
          deceasedRelationship,
          deceasedDate,
          funeralEmployeeSignatureDate,
          funeralSignature,
        };
      } else if (applicationType === 'salary_advance_application') {
        applicationData = {
          ...applicationData,
          salaryAdvanceDepartment,
          salaryAdvanceAmount,
          selectedSites,
          salaryAdvanceOtherSite: selectedSites.includes('others') ? salaryAdvanceOtherSite : '',
          selectedRecoveryPeriod,
          selectedReasons,
          salaryAdvanceOtherReason: selectedReasons.includes('others') ? salaryAdvanceOtherReason : '',
          salaryAdvanceSignatureDate,
          salaryAdvanceSignature,
        };
      } else if (applicationType === 'application_for_appeal') {
        applicationData = {
          ...applicationData,
          appealFormDate,
          appealManNo,
          appealName,
          appealSanction,
          appealOffence,
          appealOffenceDate,
          appealGrounds,
          appellantSignature,
          appealDateOfAppeal,
        };
      } else if (applicationType === 'grievance_record') {
        applicationData = {
          ...applicationData,
          grievanceDate,
          grievanceComplaints,
          grievancePlaceOfWork,
          grievanceSupervisorName,
          grievanceDetails,
        };
      } else if (applicationType === 'hospital_attendance_form') {
        applicationData = {
          ...applicationData,
          selectedClinics,
          otherClinic,
          selectedShift,
        };
      }

      if (isOnline) {
        // Step 1: Validate data
        setCurrentStepIndex(0);
        await new Promise(resolve => setTimeout(resolve, 500));
        
        // Get current user data for email
        const userData = await AsyncStorage.getItem('userData');
        const parsedUserData = userData ? JSON.parse(userData) : {};
        
        // Try multiple ways to get user email
        let userEmail = parsedUserData.email || '';
        
        // If no email in cached data, try to get from Firebase
        if (!userEmail) {
          try {
            const userSnapshot = await database().ref(`/Globalusers/admins/${userUid}`).once('value');
            const firebaseUserData = userSnapshot.val();
            userEmail = firebaseUserData?.email || '';
          } catch (error) {
            console.error('Error getting user email from Firebase:', error);
          }
        }
        
        console.log('📧 User email for confirmation:', userEmail);
        
        // Step 2: Save to database (this triggers the Firebase function)
        setCurrentStepIndex(1);
        setLoadingStep('Saving application to database...');
        const applicationRef = database().ref(`/GlobalHrApplications/admins/${applicationType}/${userUid}`);
        const newApplicationRef = await applicationRef.push(applicationData);
        await new Promise(resolve => setTimeout(resolve, 1000));
        
        // Step 3: Get matching supervisors
        setCurrentStepIndex(2);
        setLoadingStep('Finding matching supervisors...');
        const matchingSupervisors = await getSupervisorsByCountryProject(selectedCountry, selectedProject);
        const supervisorEmails = matchingSupervisors.map(s => s.email).filter(email => email && email !== 'No email');
        
        // Update email status with found supervisors
        setEmailStatus(prev => ({
          ...prev,
          supervisorEmails: { 
            sent: false, 
            count: supervisorEmails.length, 
            status: supervisorEmails.length > 0 ? 'processing' : 'no_match',
            emails: supervisorEmails
          }
        }));
        await new Promise(resolve => setTimeout(resolve, 1000));
        
        // Step 4: Send confirmation email to applicant
        setCurrentStepIndex(3);
        setLoadingStep('Sending confirmation email to you...');
        let confirmationEmailSent = false;
        if (userEmail) {
          try {
            confirmationEmailSent = await sendConfirmationEmail(userEmail, {
              applicationType,
              fullName,
              employeeNo
            });
            
            console.log('📧 Confirmation email result:', confirmationEmailSent ? 'SUCCESS' : 'FAILED');
          } catch (error) {
            console.error('📧 Confirmation email error:', error);
            confirmationEmailSent = false;
          }
          
          // Update creator email status
          setEmailStatus(prev => ({
            ...prev,
            creatorEmail: { 
              sent: confirmationEmailSent, 
              status: confirmationEmailSent ? 'success' : 'failed',
              email: userEmail 
            }
          }));
        } else {
          console.log('⚠️ No user email found for confirmation');
          setEmailStatus(prev => ({
            ...prev,
            creatorEmail: { 
              sent: false, 
              status: 'no_email',
              email: '' 
            }
          }));
        }
        await new Promise(resolve => setTimeout(resolve, 1000));
        
        // Step 5: Firebase function processes supervisor emails automatically
        setCurrentStepIndex(4);
        setLoadingStep('Firebase system sending emails to supervisors...');
        
        // Show processing status for supervisor emails
        if (supervisorEmails.length > 0) {
          setEmailStatus(prev => ({
            ...prev,
            supervisorEmails: { 
              ...prev.supervisorEmails,
              sent: false, 
              status: 'processing'
            }
          }));
        }
        
        // Wait for Firebase function to process (it runs automatically)
        await new Promise(resolve => setTimeout(resolve, 3000));
        
        // Update supervisor email status (Firebase function handles the actual sending)
        if (supervisorEmails.length > 0) {
          setEmailStatus(prev => ({
            ...prev,
            supervisorEmails: { 
              ...prev.supervisorEmails,
              sent: true, 
              status: 'success'
            }
          }));
        }
        
        // Step 6: Finalize
        setCurrentStepIndex(5);
        setLoadingStep('Finalizing submission...');
        await new Promise(resolve => setTimeout(resolve, 1000));
        
        // Hide loading modal and show success modal
        setShowLoadingModal(false);
        setShowSuccessModal(true);
        setSubmitting(false);
      } else {
        // Cache for offline submission
        setLoadingStep('Saving for offline sync...');
        await new Promise(resolve => setTimeout(resolve, 1000));
        
        const cachedApplications = await AsyncStorage.getItem('offlineApplications');
        const applications = cachedApplications ? JSON.parse(cachedApplications) : [];
        applications.push({ ...applicationData, applicationId: `offline_${Date.now()}`, userUid });
        await AsyncStorage.setItem('offlineApplications', JSON.stringify(applications));
        
        setEmailStatus({
          creatorEmail: { sent: false, status: 'offline', email: '' },
          supervisorEmails: { sent: false, count: 0, status: 'offline', emails: [] }
        });
        
        setShowLoadingModal(false);
        setShowSuccessModal(true);
        setSubmitting(false);
      }
    } catch (error) {
      console.error('Error submitting application:', error);
      setSubmitting(false);
      setShowLoadingModal(false);
      showToast('error', 'Error', 'Failed to submit application. Please try again.');
    }
  };

  // Sync offline applications
  const syncOfflineApplications = async () => {
    try {
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) return;

      const cachedApplications = await AsyncStorage.getItem('offlineApplications');
      if (cachedApplications) {
        const applications = JSON.parse(cachedApplications);
        for (const app of applications) {
          if (app.userUid === userUid) {
            const applicationRef = database().ref(`/GlobalHrApplications/admins/${app.applicationType}/${userUid}`);
            await applicationRef.push({ ...app, submittedAt: new Date().toISOString() });
          }
        }
        await AsyncStorage.removeItem('offlineApplications');
        showToast('success', 'Success', 'Offline applications synced successfully.');
      }
    } catch (error) {
      console.error('Error syncing offline applications:', error);
      showToast('error', 'Error', 'Failed to sync offline applications.');
    }
  };

  // Render selected items for country and project
  const renderSelectedItems = (selectedArray) => {
    if (!selectedArray || selectedArray.length === 0) {
      return <Text style={styles.noSelectionText}>No items selected</Text>;
    }
    return (
      <View style={styles.selectedItemsContainer}>
        {selectedArray.map((item, index) => (
          <Text key={index} style={styles.selectedItemText}>
            {item}
            {index < selectedArray.length - 1 ? ', ' : ''}
          </Text>
        ))}
      </View>
    );
  };

  const [submitting, setSubmitting] = useState(false);
  
  // Modal states
  const [showLoadingModal, setShowLoadingModal] = useState(false);
  const [showSuccessModal, setShowSuccessModal] = useState(false);
  const [emailStatus, setEmailStatus] = useState({
    creatorEmail: { sent: false, status: 'pending', email: '' },
    supervisorEmails: { sent: false, count: 0, status: 'pending', emails: [] }
  });
  const [loadingStep, setLoadingStep] = useState('');
  const [processingSteps, setProcessingSteps] = useState([]);
  const [currentStepIndex, setCurrentStepIndex] = useState(0);

  // Helper function to format application type for display
  const formatApplicationType = (type) => {
    const typeMap = {
      'leave_application': 'Leave Application',
      'funeral_grant_form': 'Funeral Grant Form',
      'salary_advance_application': 'Salary Advance Application',
      'application_for_appeal': 'Application for Appeal',
      'grievance_record': 'Grievance Record',
      'hospital_attendance_form': 'Hospital Attendance Form'
    };
    
    return typeMap[type] || type?.replace(/_/g, ' ').replace(/\b\w/g, l => l.toUpperCase()) || 'Application';
  };

  // Get supervisors matching country and project
  const getSupervisorsByCountryProject = async (country, project) => {
    try {
      console.log('🔍 Getting supervisors for:', { country, project });
      
      const response = await fetch('https://us-central1-titan-drilling-1f8e9.cloudfunctions.net/debugHrSupervisors', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({ country, project })
      });
      
      const result = await response.json();
      
      if (result.success && result.debugInfo) {
        const supervisors = result.debugInfo.matchingSupervisors || [];
        console.log(`Found ${supervisors.length} matching supervisors`);
        return supervisors;
      }
      
      return [];
    } catch (error) {
      console.error('Error getting supervisors:', error);
      return [];
    }
  };

  // Send confirmation email to applicant with retry mechanism
  const sendConfirmationEmail = async (email, applicationData) => {
    const maxRetries = 3;
    let lastError = null;
    
    for (let attempt = 1; attempt <= maxRetries; attempt++) {
      try {
        console.log(`📧 Sending confirmation email to: ${email} (attempt ${attempt}/${maxRetries})`);
        
        const response = await fetch('https://us-central1-titan-drilling-1f8e9.cloudfunctions.net/testHrApplicationEmail', {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
          },
          body: JSON.stringify({
            email: email,
            applicationType: applicationData.applicationType,
            fullName: applicationData.fullName,
            employeeNo: applicationData.employeeNo
          })
        });
        
        if (!response.ok) {
          throw new Error(`HTTP ${response.status}: ${response.statusText}`);
        }
        
        const result = await response.json();
        console.log('📧 Confirmation email result:', result);
        
        if (result.success) {
          return true;
        } else {
          throw new Error(result.error || 'Email sending failed');
        }
        
      } catch (error) {
        lastError = error;
        console.error(`❌ Confirmation email attempt ${attempt} failed:`, error.message);
        
        if (attempt < maxRetries) {
          // Wait before retry (exponential backoff)
          const delay = Math.pow(2, attempt) * 1000; // 2s, 4s, 8s
          console.log(`⏳ Retrying confirmation email in ${delay}ms...`);
          await new Promise(resolve => setTimeout(resolve, delay));
        }
      }
    }
    
    console.error('❌ All confirmation email attempts failed:', lastError);
    return false;
  };

  // Initialize processing steps
  const initializeProcessingSteps = () => {
    const steps = [
      'Validating application data...',
      'Saving application to database...',
      'Finding matching supervisors...',
      'Sending confirmation email...',
      'Processing supervisor notifications...',
      'Finalizing submission...'
    ];
    
    setProcessingSteps(steps);
    setCurrentStepIndex(0);
    setLoadingStep(steps[0]);
  };

  return (
    <View style={styles.container}>
      <LinearGradient
        colors={['#025383', '#015185']}
        style={styles.header}
        start={{ x: 0, y: 0 }}
        end={{ x: 1, y: 0 }}
      >
        <StatusBar
          translucent
          backgroundColor="transparent"
          barStyle="light-content"
        />
        <View style={styles.headerContent}>
          <TouchableOpacity
            style={styles.backButton}
            onPress={() => navigation.goBack()}
            accessibilityLabel="Go back"
            accessibilityRole="button"
          >
            <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
          </TouchableOpacity>
          <Text style={styles.headerTitle}>Application</Text>
          <TouchableOpacity
            style={styles.menuButton}
            accessibilityLabel="Open menu"
            accessibilityRole="button"
          >
            {/* <MaterialIcons name="menu" size={width * 0.06} color="#fff" /> */}
          </TouchableOpacity>
        </View>
      </LinearGradient>

      <KeyboardAvoidingView
        behavior={Platform.OS === 'ios' ? 'padding' : 'height'}
        style={styles.keyboardAvoidingView}
        keyboardVerticalOffset={Platform.OS === 'ios' ? height * 0.1 : 0}
      >
        <ScrollView
          ref={scrollViewRef}
          contentContainerStyle={[styles.content, { paddingBottom: keyboardHeight + width * 0.1 }]}
          keyboardShouldPersistTaps="handled"
          showsVerticalScrollIndicator={false}
          scrollEnabled={true}
        >
          <CustomInputField
            label="Full name"
            value={fullName}
            placeholder="Full name"
            onChangeText={setFullName}
            editable={false}
            onFocus={handleInputFocus('fullName')}
            onLayout={handleLayout('fullName')}
            isRequired
            showError={!fullName}
            errorMessage="Full name is required"
          />

          <CustomInputField
            label="Employee no"
            value={employeeNo}
            placeholder="Employee no"
            onChangeText={setEmployeeNo}
            keyboardType="numeric"
            editable={false}
            onFocus={handleInputFocus('employeeNo')}
            onLayout={handleLayout('employeeNo')}
            isRequired
            showError={!employeeNo}
            errorMessage="Employee number is required"
          />

          <View style={inputStyles.container} onLayout={handleLayout('country')}>
            <Text style={inputStyles.label}>
              Country
              <Text style={inputStyles.requiredError}> *</Text>
            </Text>
            <Dropdown
              style={[styles.dropdown]}
              containerStyle={styles.dropdownContainer}
              placeholderStyle={styles.dropdownText}
              selectedTextStyle={styles.dropdownText}
              data={countryItems}
              labelField="label"
              valueField="value"
              placeholder="Select Country"
              value={selectedCountry}
              onChange={item => setSelectedCountry(item.value)}
              maxHeight={height * 0.5}
              showsVerticalScrollIndicator={true}
              accessibilityLabel="Select country"
              accessibilityRole="combobox"
            />
            {!selectedCountry && <Text style={inputStyles.errorText}>Country is required</Text>}
          </View>

          <View style={inputStyles.container} onLayout={handleLayout('project')}>
            <Text style={inputStyles.label}>
              Project
              <Text style={inputStyles.requiredError}> *</Text>
            </Text>
            <Dropdown
              style={[styles.dropdown, { marginTop: height * 0.01 }]}
              containerStyle={styles.dropdownContainer}
              placeholderStyle={styles.dropdownText}
              selectedTextStyle={styles.dropdownText}
              data={projectItems}
              labelField="label"
              valueField="value"
              placeholder="Select Project"
              value={selectedProject}
              onChange={item => setSelectedProject(item.value)}
              maxHeight={height * 0.5}
              showsVerticalScrollIndicator={true}
              accessibilityLabel="Select project"
              accessibilityRole="combobox"
              disable={!selectedCountry}
            />
            {!selectedProject && <Text style={inputStyles.errorText}>Project is required</Text>}
          </View>

          <CustomInputField
            label="Department"
            value={department}
            placeholder="Department"
            onChangeText={setDepartment}
            editable={false}
            onFocus={handleInputFocus('department')}
            onLayout={handleLayout('department')}
            isRequired
            showError={!department}
            errorMessage="Department is required"
          />

          <CustomInputField
            label="NRC no"
            value={nrcNo}
            placeholder="NRC no"
            onChangeText={setNrcNo}
            keyboardType="numeric"
            editable={false}
            onFocus={handleInputFocus('nrcNo')}
            onLayout={handleLayout('nrcNo')}
            isRequired
            showError={!nrcNo}
            errorMessage="NRC number is required"
          />

          <View style={[styles.inputRow, { zIndex: 9999 }]} onLayout={handleLayout('applicationType')}>
            <Dropdown
              style={[styles.dropdown, { zIndex: 8000 }]}
              containerStyle={[styles.dropdownContainer, { zIndex: 9000 }]}
              placeholderStyle={styles.dropdownText}
              selectedTextStyle={styles.dropdownText}
              itemTextStyle={styles.dropdownText}
              data={applicationTypes}
              labelField="label"
              valueField="value"
              placeholder="Application type"
              value={applicationType}
              onChange={(item) => {
                setApplicationType(item.value);
                setApplicationTypeOpen(false);
              }}
              onFocus={() => setApplicationTypeOpen(true)}
              onBlur={() => setApplicationTypeOpen(false)}
              maxHeight={height * 0.5}
              showsVerticalScrollIndicator={true}
              nestedScrollEnabled={true}
              renderRightIcon={() => (
                <MaterialIcons name="arrow-drop-down" size={width * 0.06} color="#333" />
              )}
              accessibilityLabel="Select application type"
              accessibilityRole="combobox"
            />
          </View>

          {applicationType === 'leave_application' && (
            <>
              <View style={styles.twoColumnRow}>
                <CustomInputField
                  label="Date"
                  value={leaveStartDate}
                  placeholder="Select Start Date"
                  onPress={() => showDatePickerDialog('leaveStartDate')}
                  editable={false}
                  icon={<MaterialIcons name="calendar-today" size={width * 0.05} color="#333" />}
                  onIconPress={() => showDatePickerDialog('leaveStartDate')}
                  onFocus={handleInputFocus('leaveStartDate')}
                  onLayout={handleLayout('leaveStartDate')}
                  style={styles.halfWidthInput}
                  isRequired
                  showError={!leaveStartDate && leaveSignatureError}
                  errorMessage="Start date is required"
                />
                <CustomInputField
                  label="To"
                  value={leaveEndDate}
                  placeholder="Select End Date"
                  onPress={() => showDatePickerDialog('leaveEndDate')}
                  editable={false}
                  icon={<MaterialIcons name="calendar-today" size={width * 0.05} color="#333" />}
                  onIconPress={() => showDatePickerDialog('leaveEndDate')}
                  onFocus={handleInputFocus('leaveEndDate')}
                  onLayout={handleLayout('leaveEndDate')}
                  style={styles.halfWidthInput}
                  isRequired
                  showError={!leaveEndDate && leaveSignatureError}
                  errorMessage="End date is required"
                />
              </View>

              <CustomInputField
                label="Report date"
                value={reportDate}
                placeholder="Select Report Date"
                onPress={() => showDatePickerDialog('reportDate')}
                editable={false}
                icon={<MaterialIcons name="calendar-today" size={width * 0.05} color="#333" />}
                onIconPress={() => showDatePickerDialog('reportDate')}
                onFocus={handleInputFocus('reportDate')}
                onLayout={handleLayout('reportDate')}
                isRequired
                showError={!reportDate && leaveSignatureError}
                errorMessage="Report date is required"
              />

              <CustomInputField
                label="Reason of leave"
                value={reasonOfLeave}
                placeholder="Reason of leave"
                onChangeText={setReasonOfLeave}
                onFocus={handleInputFocus('reasonOfLeave')}
                onLayout={handleLayout('reasonOfLeave')}
                isRequired
                showError={!reasonOfLeave && leaveSignatureError}
                errorMessage="Reason is required"
              />

              <CustomInputField
                label="Address while on leave"
                value={addressWhileOnLeave}
                placeholder="Address while on leave"
                onChangeText={setAddressWhileOnLeave}
                onFocus={handleInputFocus('addressWhileOnLeave')}
                onLayout={handleLayout('addressWhileOnLeave')}
                isRequired
                showError={!addressWhileOnLeave && leaveSignatureError}
                errorMessage="Address is required"
              />

              <Text style={inputStyles.label}>Other contacts</Text>
              <View style={styles.twoColumnRow}>
                <CustomInputField
                  label="Name"
                  value={otherContactName}
                  placeholder="Name"
                  onChangeText={setOtherContactName}
                  onFocus={handleInputFocus('otherContactName')}
                  onLayout={handleLayout('otherContactName')}
                  style={styles.halfWidthInput}
                />
                <CustomInputField
                  label="Cell"
                  value={otherContactCell}
                  placeholder="Cell"
                  onChangeText={setOtherContactCell}
                  keyboardType="phone-pad"
                  onFocus={handleInputFocus('otherContactCell')}
                  onLayout={handleLayout('otherContactCell')}
                  style={styles.halfWidthInput}
                />
              </View>

              <View style={styles.signatureContainer} onLayout={handleLayout('employeeSignature')}>
                <Text style={inputStyles.label}>Employee's Signature</Text>
                <TouchableOpacity
                  style={[styles.signaturePad, leaveSignatureError && styles.signaturePadError]}
                  onPress={() => handleOpenSignaturePad('leave')}
                  accessibilityLabel="Capture employee signature"
                  accessibilityRole="button"
                >
                  {leaveSignatureLoading ? (
                    <ActivityIndicator size="small" color="#015185" />
                  ) : leaveSignature ? (
                    <Image
                      source={{ uri: leaveSignature }}
                      style={styles.capturedSignatureImage}
                      resizeMode="contain"
                      onError={(e) => {
                        console.error('Error loading leave signature:', e.nativeEvent.error);
                        showToast('error', 'Error', 'Failed to load signature.');
                      }}
                    />
                  ) : (
                    <Text style={styles.signatureText}>Tap to sign</Text>
                  )}
                </TouchableOpacity>
                {leaveSignatureError && <Text style={inputStyles.errorText}>Signature is required</Text>}
              </View>

              <CustomInputField
                label="Date"
                value={signatureDate}
                placeholder="Select Date"
                onPress={() => showDatePickerDialog('signatureDate')}
                editable={false}
                icon={<MaterialIcons name="calendar-today" size={width * 0.05} color="#333" />}
                onIconPress={() => showDatePickerDialog('signatureDate')}
                onFocus={handleInputFocus('signatureDate')}
                onLayout={handleLayout('signatureDate')}
                isRequired
                showError={!signatureDate && leaveSignatureError}
                errorMessage="Signature date is required"
              />
            </>
          )}

          {applicationType === 'funeral_grant_form' && (
            <>
              <View style={styles.twoColumnRow}>
                <CustomInputField
                  label="From Date"
                  value={funeralFromDate}
                  placeholder="Select From Date"
                  onPress={() => showDatePickerDialog('funeralFromDate')}
                  editable={false}
                  icon={<MaterialIcons name="calendar-today" size={width * 0.05} color="#333" />}
                  onIconPress={() => showDatePickerDialog('funeralFromDate')}
                  onFocus={handleInputFocus('funeralFromDate')}
                  onLayout={handleLayout('funeralFromDate')}
                  style={styles.halfWidthInput}
                  isRequired
                  showError={!funeralFromDate && funeralSignatureError}
                  errorMessage="From date is required"
                />
                <CustomInputField
                  label="To Date"
                  value={funeralToDate}
                  placeholder="Select To Date"
                  onPress={() => showDatePickerDialog('funeralToDate')}
                  editable={false}
                  icon={<MaterialIcons name="calendar-today" size={width * 0.05} color="#333" />}
                  onIconPress={() => showDatePickerDialog('funeralToDate')}
                  onFocus={handleInputFocus('funeralToDate')}
                  onLayout={handleLayout('funeralToDate')}
                  style={styles.halfWidthInput}
                  isRequired
                  showError={!funeralToDate && funeralSignatureError}
                  errorMessage="To date is required"
                />
              </View>

              <View style={styles.inputRow} onLayout={handleLayout('funeralRelationshipType')}>
                <Dropdown
                  style={[styles.dropdown, { zIndex: 7000 }]}
                  containerStyle={[styles.dropdownContainer, { zIndex: 9000 }]}
                  placeholderStyle={styles.dropdownText}
                  selectedTextStyle={styles.dropdownText}
                  itemTextStyle={styles.dropdownText}
                  data={funeralRelationshipOptions}
                  labelField="label"
                  valueField="value"
                  placeholder="Select Relationship Type"
                  value={selectedFuneralRelationshipType}
                  onChange={(item) => {
                    setSelectedFuneralRelationshipType(item.value);
                    setFuneralRelationshipTypeOpen(false);
                  }}
                  onFocus={() => setFuneralRelationshipTypeOpen(true)}
                  onBlur={() => setFuneralRelationshipTypeOpen(false)}
                  maxHeight={height * 0.5}
                  showsVerticalScrollIndicator={true}
                  nestedScrollEnabled={true}
                  renderRightIcon={() => (
                    <MaterialIcons name="arrow-drop-down" size={width * 0.06} color="#333" />
                  )}
                  accessibilityLabel="Select relationship type"
                  accessibilityRole="combobox"
                />
              </View>

              {selectedFuneralRelationshipType === 'spouse_child' && (
                <>
                  <Text style={styles.sectionTitle}>
                    RELATIONSHIP: SPOUSE AND BIOLOGICAL CHILD (EMPLOYEE HIM/HERSELF)
                  </Text>
                  <View style={styles.table}>
                    <View style={styles.tableRow}>
                      <Text style={[styles.tableHeaderCell, styles.dateCell]}>DATE</Text>
                      <Text style={[styles.tableHeaderCell, styles.descriptionCell]}>DESCRIPTION</Text>
                      <Text style={[styles.tableHeaderCell, styles.amountCell]}>AMOUNT</Text>
                    </View>
                    {funeralSpouseChildData.map((item) => (
                      <View key={item.id} style={styles.tableRow}>
                        <View style={[styles.tableDataCell, styles.dateCell]}>
                          <TouchableOpacity
                            style={styles.tableDateButton}
                            onPress={() =>
                              showDatePickerDialog('funeralSpouseChildDate', `funeralSpouseChild-${item.id}`)
                            }
                          >
                            <Text style={styles.tableDateText}>{item.date || 'Select Date'}</Text>
                            <MaterialIcons name="calendar-today" size={width * 0.04} color="#333" />
                          </TouchableOpacity>
                        </View>
                        <View style={[styles.tableDataCell, styles.descriptionCell]}>
                          <Text style={styles.tableCellText}>{item.description}</Text>
                        </View>
                        <View style={[styles.tableDataCell, styles.amountCell]}>
                          <TextInput
                            style={styles.tableInput}
                            value={item.amount}
                            onChangeText={(text) => handleFuneralAmountChange('spouseChild', item.id, text)}
                            keyboardType="numeric"
                            placeholder="K"
                          />
                        </View>
                      </View>
                    ))}
                    <View style={styles.tableRow}>
                      <Text style={[styles.tableTotalCell, styles.dateCell]}></Text>
                      <Text style={[styles.tableTotalCell, styles.descriptionCell]}>TOTAL</Text>
                      <Text style={[styles.tableTotalCell, styles.amountCell]}>
                        K {calculateTotal(funeralSpouseChildData)}
                      </Text>
                    </View>
                  </View>
                </>
              )}

              {selectedFuneralRelationshipType === 'mother_father' && (
                <>
                  <Text style={[styles.sectionTitle, { marginTop: width * 0.06 }]}>
                    RELATIONSHIP: MOTHER AND FATHER
                  </Text>
                  <View style={styles.table}>
                    <View style={styles.tableRow}>
                      <Text style={[styles.tableHeaderCell, styles.dateCell]}>DATE</Text>
                      <Text style={[styles.tableHeaderCell, styles.descriptionCell]}>DESCRIPTION</Text>
                      <Text style={[styles.tableHeaderCell, styles.amountCell]}>AMOUNT</Text>
                    </View>
                    {funeralMotherFatherData.map((item) => (
                      <View key={item.id} style={styles.tableRow}>
                        <View style={[styles.tableDataCell, styles.dateCell]}>
                          <TouchableOpacity
                            style={styles.tableDateButton}
                            onPress={() =>
                              showDatePickerDialog('funeralMotherFatherDate', `funeralMotherFather-${item.id}`)
                            }
                          >
                            <Text style={styles.tableDateText}>{item.date || 'Select Date'}</Text>
                            <MaterialIcons name="calendar-today" size={width * 0.04} color="#333" />
                          </TouchableOpacity>
                        </View>
                        <View style={[styles.tableDataCell, styles.descriptionCell]}>
                          <Text style={styles.tableCellText}>{item.description}</Text>
                        </View>
                        <View style={[styles.tableDataCell, styles.amountCell]}>
                          <TextInput
                            style={styles.tableInput}
                            value={item.amount}
                            onChangeText={(text) => handleFuneralAmountChange('motherFather', item.id, text)}
                            keyboardType="numeric"
                            placeholder="K"
                          />
                        </View>
                      </View>
                    ))}
                    <View style={styles.tableRow}>
                      <Text style={[styles.tableTotalCell, styles.dateCell]}></Text>
                      <Text style={[styles.tableTotalCell, styles.descriptionCell]}>TOTAL</Text>
                      <Text style={[styles.tableTotalCell, styles.amountCell]}>
                        K {calculateTotal(funeralMotherFatherData)}
                      </Text>
                    </View>
                  </View>
                </>
              )}

              {selectedFuneralRelationshipType === 'both' && (
                <>
                  <Text style={styles.sectionTitle}>
                    RELATIONSHIP: SPOUSE AND BIOLOGICAL CHILD (EMPLOYEE HIM/HERSELF)
                  </Text>
                  <View style={styles.table}>
                    <View style={styles.tableRow}>
                      <Text style={[styles.tableHeaderCell, styles.dateCell]}>DATE</Text>
                      <Text style={[styles.tableHeaderCell, styles.descriptionCell]}>DESCRIPTION</Text>
                      <Text style={[styles.tableHeaderCell, styles.amountCell]}>AMOUNT</Text>
                    </View>
                    {funeralSpouseChildData.map((item) => (
                      <View key={item.id} style={styles.tableRow}>
                        <View style={[styles.tableDataCell, styles.dateCell]}>
                          <TouchableOpacity
                            style={styles.tableDateButton}
                            onPress={() =>
                              showDatePickerDialog('funeralSpouseChildDate', `funeralSpouseChild-${item.id}`)
                            }
                          >
                            <Text style={styles.tableDateText}>{item.date || 'Select Date'}</Text>
                            <MaterialIcons name="calendar-today" size={width * 0.04} color="#333" />
                          </TouchableOpacity>
                        </View>
                        <View style={[styles.tableDataCell, styles.descriptionCell]}>
                          <Text style={styles.tableCellText}>{item.description}</Text>
                        </View>
                        <View style={[styles.tableDataCell, styles.amountCell]}>
                          <TextInput
                            style={styles.tableInput}
                            value={item.amount}
                            onChangeText={(text) => handleFuneralAmountChange('spouseChild', item.id, text)}
                            keyboardType="numeric"
                            placeholder="K"
                          />
                        </View>
                      </View>
                    ))}
                    <View style={styles.tableRow}>
                      <Text style={[styles.tableTotalCell, styles.dateCell]}></Text>
                      <Text style={[styles.tableTotalCell, styles.descriptionCell]}>TOTAL</Text>
                      <Text style={[styles.tableTotalCell, styles.amountCell]}>
                        K {calculateTotal(funeralSpouseChildData)}
                      </Text>
                    </View>
                  </View>

                  <Text style={[styles.sectionTitle, { marginTop: width * 0.06 }]}>
                    RELATIONSHIP: MOTHER AND FATHER
                  </Text>
                  <View style={styles.table}>
                    <View style={styles.tableRow}>
                      <Text style={[styles.tableHeaderCell, styles.dateCell]}>DATE</Text>
                      <Text style={[styles.tableHeaderCell, styles.descriptionCell]}>DESCRIPTION</Text>
                      <Text style={[styles.tableHeaderCell, styles.amountCell]}>AMOUNT</Text>
                    </View>
                    {funeralMotherFatherData.map((item) => (
                      <View key={item.id} style={styles.tableRow}>
                        <View style={[styles.tableDataCell, styles.dateCell]}>
                          <TouchableOpacity
                            style={styles.tableDateButton}
                            onPress={() =>
                              showDatePickerDialog('funeralMotherFatherDate', `funeralMotherFather-${item.id}`)
                            }
                          >
                            <Text style={styles.tableDateText}>{item.date || 'Select Date'}</Text>
                            <MaterialIcons name="calendar-today" size={width * 0.04} color="#333" />
                          </TouchableOpacity>
                        </View>
                        <View style={[styles.tableDataCell, styles.descriptionCell]}>
                          <Text style={styles.tableCellText}>{item.description}</Text>
                        </View>
                        <View style={[styles.tableDataCell, styles.amountCell]}>
                          <TextInput
                            style={styles.tableInput}
                            value={item.amount}
                            onChangeText={(text) => handleFuneralAmountChange('motherFather', item.id, text)}
                            keyboardType="numeric"
                            placeholder="K"
                          />
                        </View>
                      </View>
                    ))}
                    <View style={styles.tableRow}>
                      <Text style={[styles.tableTotalCell, styles.dateCell]}></Text>
                      <Text style={[styles.tableTotalCell, styles.descriptionCell]}>TOTAL</Text>
                      <Text style={[styles.tableTotalCell, styles.amountCell]}>
                        K {calculateTotal(funeralMotherFatherData)}
                      </Text>
                    </View>
                  </View>
                </>
              )}

              <CustomInputField
                label="DECEASED'S NAME"
                value={deceasedName}
                placeholder="Deceased's Name"
                onChangeText={setDeceasedName}
                onFocus={handleInputFocus('deceasedName')}
                onLayout={handleLayout('deceasedName')}
                isRequired
                showError={!deceasedName && funeralSignatureError}
                errorMessage="Deceased's name is required"
              />
              <CustomInputField
                label="RELATIONSHIP"
                value={deceasedRelationship}
                placeholder="Relationship"
                onChangeText={setDeceasedRelationship}
                onFocus={handleInputFocus('deceasedRelationship')}
                onLayout={handleLayout('deceasedRelationship')}
                isRequired
                showError={!deceasedRelationship && funeralSignatureError}
                errorMessage="Relationship is required"
              />
              <CustomInputField
                label="DATE"
                value={deceasedDate}
                placeholder="Date"
                onPress={() => showDatePickerDialog('deceasedDate')}
                editable={false}
                icon={<MaterialIcons name="calendar-today" size={width * 0.05} color="#333" />}
                onIconPress={() => showDatePickerDialog('deceasedDate')}
                onFocus={handleInputFocus('deceasedDate')}
                onLayout={handleLayout('deceasedDate')}
                isRequired
                showError={!deceasedDate && funeralSignatureError}
                errorMessage="Deceased date is required"
              />

              <View style={styles.signatureContainer} onLayout={handleLayout('funeralEmployeeSignature')}>
                <Text style={inputStyles.label}>EMPLOYEE SIGNATURE</Text>
                <TouchableOpacity
                  style={[styles.signaturePad, funeralSignatureError && styles.signaturePadError]}
                  onPress={() => handleOpenSignaturePad('funeral')}
                  accessibilityLabel="Capture employee signature for funeral grant"
                  accessibilityRole="button"
                >
                  {funeralSignatureLoading ? (
                    <ActivityIndicator size="small" color="#015185" />
                  ) : funeralSignature ? (
                    <Image
                      source={{ uri: funeralSignature }}
                      style={styles.capturedSignatureImage}
                      resizeMode="contain"
                      onError={(e) => {
                        console.error('Error loading funeral signature:', e.nativeEvent.error);
                        showToast('error', 'Error', 'Failed to load signature.');
                      }}
                    />
                  ) : (
                    <Text style={styles.signatureText}>Tap to sign</Text>
                  )}
                </TouchableOpacity>
                {funeralSignatureError && <Text style={inputStyles.errorText}>Signature is required</Text>}
              </View>
              <CustomInputField
                label="DATE"
                value={funeralEmployeeSignatureDate}
                placeholder="Date"
                onPress={() => showDatePickerDialog('funeralEmployeeSignatureDate')}
                editable={false}
                icon={<MaterialIcons name="calendar-today" size={width * 0.05} color="#333" />}
                onIconPress={() => showDatePickerDialog('funeralEmployeeSignatureDate')}
                onFocus={handleInputFocus('funeralEmployeeSignatureDate')}
                onLayout={handleLayout('funeralEmployeeSignatureDate')}
                isRequired
                showError={!funeralEmployeeSignatureDate && funeralSignatureError}
                                errorMessage="Signature date is required"
              />
            </>
          )}

          {applicationType === 'salary_advance_application' && (
            <>
              <CustomInputField
                label="Department"
                value={salaryAdvanceDepartment}
                placeholder="Department"
                onChangeText={setSalaryAdvanceDepartment}
                onFocus={handleInputFocus('salaryAdvanceDepartment')}
                onLayout={handleLayout('salaryAdvanceDepartment')}
                isRequired
                showError={!salaryAdvanceDepartment && salaryAdvanceSignatureError}
                errorMessage="Department is required"
              />
              <CustomInputField
                label="Amount"
                value={salaryAdvanceAmount}
                placeholder="Amount"
                onChangeText={(text) => {
                  const cleanedValue = text.replace(/[^0-9.]/g, '');
                  const parts = cleanedValue.split('.');
                  if (parts.length > 2) return;
                  if (parts[1] && parts[1].length > 2) return;
                  setSalaryAdvanceAmount(cleanedValue);
                }}
                keyboardType="numeric"
                onFocus={handleInputFocus('salaryAdvanceAmount')}
                onLayout={handleLayout('salaryAdvanceAmount')}
                isRequired
                showError={!salaryAdvanceAmount && salaryAdvanceSignatureError}
                errorMessage="Amount is required"
              />
              <View style={inputStyles.container} onLayout={handleLayout('sites')}>
                <Text style={inputStyles.label}>
                  Sites<Text style={inputStyles.requiredError}> *</Text>
                </Text>
                <View style={styles.checkboxContainer}>
                  {sitesOptions.map((site) => (
                    <TouchableOpacity
                      key={site.value}
                      style={styles.checkboxRow}
                      onPress={() => toggleSiteSelection(site.value)}
                      accessibilityLabel={`Select ${site.label}`}
                      accessibilityRole="checkbox"
                      accessibilityState={{ checked: selectedSites.includes(site.value) }}
                    >
                      <View
                        style={[
                          styles.checkbox,
                          selectedSites.includes(site.value) && styles.checkboxSelected,
                        ]}
                      >
                        {selectedSites.includes(site.value) && (
                          <MaterialIcons name="check" size={width * 0.04} color="#fff" />
                        )}
                      </View>
                      <Text style={styles.checkboxLabel}>{site.label}</Text>
                    </TouchableOpacity>
                  ))}
                </View>
                {selectedSites.length === 0 && salaryAdvanceSignatureError && (
                  <Text style={inputStyles.errorText}>At least one site is required</Text>
                )}
              </View>
              {selectedSites.includes('others') && (
                <CustomInputField
                  label="Other Site"
                  value={salaryAdvanceOtherSite}
                  placeholder="Specify other site"
                  onChangeText={setSalaryAdvanceOtherSite}
                  onFocus={handleInputFocus('salaryAdvanceOtherSite')}
                  onLayout={handleLayout('salaryAdvanceOtherSite')}
                  isRequired
                  showError={selectedSites.includes('others') && !salaryAdvanceOtherSite && salaryAdvanceSignatureError}
                  errorMessage="Other site is required"
                />
              )}
              <View style={inputStyles.container} onLayout={handleLayout('recoveryPeriod')}>
                <Text style={inputStyles.label}>
                  Recovery Period<Text style={inputStyles.requiredError}> *</Text>
                </Text>
                <View style={styles.radioContainer}>
                  {recoveryPeriodOptions.map((option) => (
                    <TouchableOpacity
                      key={option.value}
                      style={styles.radioRow}
                      onPress={() => handleRecoveryPeriodSelection(option.value)}
                      accessibilityLabel={`Select ${option.label}`}
                      accessibilityRole="radio"
                      accessibilityState={{ selected: selectedRecoveryPeriod === option.value }}
                    >
                      <View
                        style={[
                          styles.radio,
                          selectedRecoveryPeriod === option.value && styles.radioSelected,
                        ]}
                      >
                        {selectedRecoveryPeriod === option.value && (
                          <View style={styles.radioInner} />
                        )}
                      </View>
                      <Text style={styles.radioLabel}>{option.label}</Text>
                    </TouchableOpacity>
                  ))}
                </View>
                {!selectedRecoveryPeriod && salaryAdvanceSignatureError && (
                  <Text style={inputStyles.errorText}>Recovery period is required</Text>
                )}
              </View>
              <View style={inputStyles.container} onLayout={handleLayout('reasons')}>
                <Text style={inputStyles.label}>
                  Reasons<Text style={inputStyles.requiredError}> *</Text>
                </Text>
                <View style={styles.checkboxContainer}>
                  {reasonsOptions.map((reason) => (
                    <TouchableOpacity
                      key={reason.value}
                      style={styles.checkboxRow}
                      onPress={() => toggleReasonSelection(reason.value)}
                      accessibilityLabel={`Select ${reason.label}`}
                      accessibilityRole="checkbox"
                      accessibilityState={{ checked: selectedReasons.includes(reason.value) }}
                    >
                      <View
                        style={[
                          styles.checkbox,
                          selectedReasons.includes(reason.value) && styles.checkboxSelected,
                        ]}
                      >
                        {selectedReasons.includes(reason.value) && (
                          <MaterialIcons name="check" size={width * 0.04} color="#fff" />
                        )}
                      </View>
                      <Text style={styles.checkboxLabel}>{reason.label}</Text>
                    </TouchableOpacity>
                  ))}
                </View>
                {selectedReasons.length === 0 && salaryAdvanceSignatureError && (
                  <Text style={inputStyles.errorText}>At least one reason is required</Text>
                )}
              </View>
              {selectedReasons.includes('others') && (
                <CustomInputField
                  label="Other Reason"
                  value={salaryAdvanceOtherReason}
                  placeholder="Specify other reason"
                  onChangeText={setSalaryAdvanceOtherReason}
                  onFocus={handleInputFocus('salaryAdvanceOtherReason')}
                  onLayout={handleLayout('salaryAdvanceOtherReason')}
                  isRequired
                  showError={selectedReasons.includes('others') && !salaryAdvanceOtherReason && salaryAdvanceSignatureError}
                  errorMessage="Other reason is required"
                />
              )}
              <View style={styles.signatureContainer} onLayout={handleLayout('salaryAdvanceSignature')}>
                <Text style={inputStyles.label}>Employee Signature</Text>
                <TouchableOpacity
                  style={[styles.signaturePad, salaryAdvanceSignatureError && styles.signaturePadError]}
                  onPress={() => handleOpenSignaturePad('salaryAdvance')}
                  accessibilityLabel="Capture employee signature for salary advance"
                  accessibilityRole="button"
                >
                  {salaryAdvanceSignatureLoading ? (
                    <ActivityIndicator size="small" color="#015185" />
                  ) : salaryAdvanceSignature ? (
                    <Image
                      source={{ uri: salaryAdvanceSignature }}
                      style={styles.capturedSignatureImage}
                      resizeMode="contain"
                      onError={(e) => {
                        console.error('Error loading salary advance signature:', e.nativeEvent.error);
                        showToast('error', 'Error', 'Failed to load signature.');
                      }}
                    />
                  ) : (
                    <Text style={styles.signatureText}>Tap to sign</Text>
                  )}
                </TouchableOpacity>
                {salaryAdvanceSignatureError && <Text style={inputStyles.errorText}>Signature is required</Text>}
              </View>
              <CustomInputField
                label="Date"
                value={salaryAdvanceSignatureDate}
                placeholder="Select Date"
                onPress={() => showDatePickerDialog('salaryAdvanceSignatureDate')}
                editable={false}
                icon={<MaterialIcons name="calendar-today" size={width * 0.05} color="#333" />}
                onIconPress={() => showDatePickerDialog('salaryAdvanceSignatureDate')}
                onFocus={handleInputFocus('salaryAdvanceSignatureDate')}
                onLayout={handleLayout('salaryAdvanceSignatureDate')}
                isRequired
                showError={!salaryAdvanceSignatureDate && salaryAdvanceSignatureError}
                errorMessage="Signature date is required"
              />
            </>
          )}

          {applicationType === 'application_for_appeal' && (
            <>
              <CustomInputField
                label="Date"
                value={appealFormDate}
                placeholder="Select Date"
                onPress={() => showDatePickerDialog('appealFormDate')}
                editable={false}
                icon={<MaterialIcons name="calendar-today" size={width * 0.05} color="#333" />}
                onIconPress={() => showDatePickerDialog('appealFormDate')}
                onFocus={handleInputFocus('appealFormDate')}
                onLayout={handleLayout('appealFormDate')}
                isRequired
                showError={!appealFormDate && appellantSignatureError}
                errorMessage="Form date is required"
              />
              <CustomInputField
                label="Man No"
                value={appealManNo}
                placeholder="Man No"
                onChangeText={setAppealManNo}
                keyboardType="numeric"
                onFocus={handleInputFocus('appealManNo')}
                onLayout={handleLayout('appealManNo')}
                isRequired
                showError={!appealManNo && appellantSignatureError}
                errorMessage="Man number is required"
              />
              <CustomInputField
                label="Name"
                value={appealName}
                placeholder="Name"
                onChangeText={setAppealName}
                onFocus={handleInputFocus('appealName')}
                onLayout={handleLayout('appealName')}
                isRequired
                showError={!appealName && appellantSignatureError}
                errorMessage="Name is required"
              />
              <CustomInputField
                label="Sanction"
                value={appealSanction}
                placeholder="Sanction"
                onChangeText={setAppealSanction}
                onFocus={handleInputFocus('appealSanction')}
                onLayout={handleLayout('appealSanction')}
                isRequired
                showError={!appealSanction && appellantSignatureError}
                errorMessage="Sanction is required"
              />
              <CustomInputField
                label="Offence"
                value={appealOffence}
                placeholder="Offence"
                onChangeText={setAppealOffence}
                onFocus={handleInputFocus('appealOffence')}
                onLayout={handleLayout('appealOffence')}
                isRequired
                showError={!appealOffence && appellantSignatureError}
                errorMessage="Offence is required"
              />
              <CustomInputField
                label="Date of Offence"
                value={appealOffenceDate}
                placeholder="Select Date"
                onPress={() => showDatePickerDialog('appealOffenceDate')}
                editable={false}
                icon={<MaterialIcons name="calendar-today" size={width * 0.05} color="#333" />}
                onIconPress={() => showDatePickerDialog('appealOffenceDate')}
                onFocus={handleInputFocus('appealOffenceDate')}
                onLayout={handleLayout('appealOffenceDate')}
                isRequired
                showError={!appealOffenceDate && appellantSignatureError}
                errorMessage="Offence date is required"
              />
              <CustomInputField
                label="Grounds for Appeal"
                value={appealGrounds}
                placeholder="Grounds for Appeal"
                onChangeText={setAppealGrounds}
                multiline
                numberOfLines={4}
                onFocus={handleInputFocus('appealGrounds')}
                onLayout={handleLayout('appealGrounds')}
                isRequired
                showError={!appealGrounds && appellantSignatureError}
                errorMessage="Grounds for appeal is required"
                inputStyle={{ height: height * 0.15, textAlignVertical: 'top' }}
              />
              <View style={styles.signatureContainer} onLayout={handleLayout('appellantSignature')}>
                <Text style={inputStyles.label}>Appellant Signature</Text>
                <TouchableOpacity
                  style={[styles.signaturePad, appellantSignatureError && styles.signaturePadError]}
                  onPress={() => handleOpenSignaturePad('appeal')}
                  accessibilityLabel="Capture appellant signature"
                  accessibilityRole="button"
                >
                  {appellantSignatureLoading ? (
                    <ActivityIndicator size="small" color="#015185" />
                  ) : appellantSignature ? (
                    <Image
                      source={{ uri: appellantSignature }}
                      style={styles.capturedSignatureImage}
                      resizeMode="contain"
                      onError={(e) => {
                        console.error('Error loading appeal signature:', e.nativeEvent.error);
                        showToast('error', 'Error', 'Failed to load signature.');
                      }}
                    />
                  ) : (
                    <Text style={styles.signatureText}>Tap to sign</Text>
                  )}
                </TouchableOpacity>
                {appellantSignatureError && <Text style={inputStyles.errorText}>Signature is required</Text>}
              </View>
              <CustomInputField
                label="Date of Appeal"
                value={appealDateOfAppeal}
                placeholder="Select Date"
                onPress={() => showDatePickerDialog('appealDateOfAppeal')}
                editable={false}
                icon={<MaterialIcons name="calendar-today" size={width * 0.05} color="#333" />}
                onIconPress={() => showDatePickerDialog('appealDateOfAppeal')}
                onFocus={handleInputFocus('appealDateOfAppeal')}
                onLayout={handleLayout('appealDateOfAppeal')}
                isRequired
                showError={!appealDateOfAppeal && appellantSignatureError}
                errorMessage="Date of appeal is required"
              />
            </>
          )}

          {applicationType === 'grievance_record' && (
            <>
              <CustomInputField
                label="Date"
                value={grievanceDate}
                placeholder="Select Date"
                onPress={() => showDatePickerDialog('grievanceDate')}
                editable={false}
                icon={<MaterialIcons name="calendar-today" size={width * 0.05} color="#333" />}
                onIconPress={() => showDatePickerDialog('grievanceDate')}
                onFocus={handleInputFocus('grievanceDate')}
                onLayout={handleLayout('grievanceDate')}
                isRequired
                showError={!grievanceDate}
                errorMessage="Date is required"
              />
              <Text style={styles.sectionTitle}>Complainant(s)</Text>
              {grievanceComplaints.map((complainant, index) => (
                <View key={index} style={styles.complainantContainer}>
                  <CustomInputField
                    label="Name"
                    value={complainant.name}
                    placeholder="Name"
                    onChangeText={(text) => {
                      const updatedComplaints = [...grievanceComplaints];
                      updatedComplaints[index].name = text;
                      setGrievanceComplaints(updatedComplaints);
                    }}
                    onFocus={handleInputFocus(`grievanceComplainantName${index}`)}
                    onLayout={handleLayout(`grievanceComplainantName${index}`)}
                    style={styles.complainantInput}
                  />
                  <CustomInputField
                    label="Mine No"
                    value={complainant.mineNo}
                    placeholder="Mine No"
                    onChangeText={(text) => {
                      const updatedComplaints = [...grievanceComplaints];
                      updatedComplaints[index].mineNo = text;
                      setGrievanceComplaints(updatedComplaints);
                    }}
                    keyboardType="numeric"
                    onFocus={handleInputFocus(`grievanceComplainantMineNo${index}`)}
                    onLayout={handleLayout(`grievanceComplainantMineNo${index}`)}
                    style={styles.complainantInput}
                  />
                  <CustomInputField
                    label="Occupation/Grade"
                    value={complainant.occupationGrade}
                    placeholder="Occupation/Grade"
                    onChangeText={(text) => {
                      const updatedComplaints = [...grievanceComplaints];
                      updatedComplaints[index].occupationGrade = text;
                      setGrievanceComplaints(updatedComplaints);
                    }}
                    onFocus={handleInputFocus(`grievanceComplainantOccupation${index}`)}
                    onLayout={handleLayout(`grievanceComplainantOccupation${index}`)}
                    style={styles.complainantInput}
                  />
                  {grievanceComplaints.length > 1 && (
                    <TouchableOpacity
                      style={styles.removeComplainantButton}
                      onPress={() => {
                        const updatedComplaints = grievanceComplaints.filter((_, i) => i !== index);
                        setGrievanceComplaints(updatedComplaints);
                      }}
                      accessibilityLabel="Remove complainant"
                      accessibilityRole="button"
                    >
                      <MaterialIcons name="remove-circle" size={width * 0.05} color="#ff4444" />
                    </TouchableOpacity>
                  )}
                </View>
              ))}
              <TouchableOpacity
                style={styles.addComplainantButton}
                onPress={() => {
                  setGrievanceComplaints([...grievanceComplaints, { name: '', mineNo: '', occupationGrade: '' }]);
                }}
                accessibilityLabel="Add complainant"
                accessibilityRole="button"
              >
                <Text style={styles.addComplainantText}>Add Complainant</Text>
              </TouchableOpacity>
              <CustomInputField
                label="Place of Work"
                value={grievancePlaceOfWork}
                placeholder="Place of Work"
                onChangeText={setGrievancePlaceOfWork}
                onFocus={handleInputFocus('grievancePlaceOfWork')}
                onLayout={handleLayout('grievancePlaceOfWork')}
                isRequired
                showError={!grievancePlaceOfWork}
                errorMessage="Place of work is required"
              />
              <CustomInputField
                label="Supervisor Name"
                value={grievanceSupervisorName}
                placeholder="Supervisor Name"
                onChangeText={setGrievanceSupervisorName}
                onFocus={handleInputFocus('grievanceSupervisorName')}
                onLayout={handleLayout('grievanceSupervisorName')}
                isRequired
                showError={!grievanceSupervisorName}
                errorMessage="Supervisor name is required"
              />
              <CustomInputField
                label="Details of Grievance"
                value={grievanceDetails}
                placeholder="Details of Grievance"
                onChangeText={setGrievanceDetails}
                multiline
                numberOfLines={4}
                onFocus={handleInputFocus('grievanceDetails')}
                onLayout={handleLayout('grievanceDetails')}
                isRequired
                showError={!grievanceDetails}
                errorMessage="Details of grievance is required"
                inputStyle={{ height: height * 0.15, textAlignVertical: 'top' }}
              />
            </>
          )}

          {applicationType === 'hospital_attendance_form' && (
            <>
              <View style={inputStyles.container} onLayout={handleLayout('clinics')}>
                <Text style={inputStyles.label}>
                  Clinic(s)<Text style={inputStyles.requiredError}> *</Text>
                </Text>
                <View style={styles.checkboxContainer}>
                  {clinicOptions.map((clinic) => (
                    <TouchableOpacity
                      key={clinic.value}
                      style={styles.checkboxRow}
                      onPress={() => {
                        setSelectedClinics((prev) =>
                          prev.includes(clinic.value)
                            ? prev.filter((item) => item !== clinic.value)
                            : [...prev, clinic.value]
                        );
                      }}
                      accessibilityLabel={`Select ${clinic.label}`}
                      accessibilityRole="checkbox"
                      accessibilityState={{ checked: selectedClinics.includes(clinic.value) }}
                    >
                      <View
                        style={[
                          styles.checkbox,
                          selectedClinics.includes(clinic.value) && styles.checkboxSelected,
                        ]}
                      >
                        {selectedClinics.includes(clinic.value) && (
                          <MaterialIcons name="check" size={width * 0.04} color="#fff" />
                        )}
                      </View>
                      <Text style={styles.checkboxLabel}>{clinic.label}</Text>
                    </TouchableOpacity>
                  ))}
                </View>
                {selectedClinics.length === 0 && (
                  <Text style={inputStyles.errorText}>At least one clinic is required</Text>
                )}
              </View>
              {selectedClinics.includes('others') && (
                <CustomInputField
                  label="Other Clinic"
                  value={otherClinic}
                  placeholder="Specify other clinic"
                  onChangeText={setOtherClinic}
                  onFocus={handleInputFocus('otherClinic')}
                  onLayout={handleLayout('otherClinic')}
                  isRequired
                  showError={selectedClinics.includes('others') && !otherClinic}
                  errorMessage="Other clinic is required"
                />
              )}
              <View style={inputStyles.container} onLayout={handleLayout('shift')}>
                <Text style={inputStyles.label}>
                  Shift<Text style={inputStyles.requiredError}> *</Text>
                </Text>
                <View style={styles.radioContainer}>
                  {['Day', 'Night'].map((shift) => (
                    <TouchableOpacity
                      key={shift}
                      style={styles.radioRow}
                      onPress={() => setSelectedShift(shift)}
                      accessibilityLabel={`Select ${shift} shift`}
                      accessibilityRole="radio"
                      accessibilityState={{ selected: selectedShift === shift }}
                    >
                      <View
                        style={[styles.radio, selectedShift === shift && styles.radioSelected]}
                      >
                        {selectedShift === shift && <View style={styles.radioInner} />}
                      </View>
                      <Text style={styles.radioLabel}>{shift}</Text>
                    </TouchableOpacity>
                  ))}
                </View>
                {!selectedShift && <Text style={inputStyles.errorText}>Shift is required</Text>}
              </View>
            </>
          )}

          {showDatePicker && (
            <DateTimePicker
              value={datePickerValue}
              mode="date"
              display={Platform.OS === 'ios' ? 'inline' : 'default'}
              onChange={onDateChange}
              maximumDate={datePickerMaxDate}
              accessibilityLabel={`Select date for ${currentEditingDateField}`}
            />
          )}

          <TouchableOpacity
            style={[styles.submitButton, submitting && { opacity: 0.6 }]}
            onPress={handleSubmit}
            accessibilityLabel="Submit application"
            accessibilityRole="button"
            disabled={submitting}
          >
            <LinearGradient
              colors={['#025383', '#015185']}
              style={styles.submitButtonGradient}
              start={{ x: 0, y: 0 }}
              end={{ x: 1, y: 0 }}
            >
              <Text style={styles.submitButtonText}>Submit Application</Text>
            </LinearGradient>
          </TouchableOpacity>
        </ScrollView>
      </KeyboardAvoidingView>

      {/* Loading Modal */}
      <Modal
        visible={showLoadingModal}
        transparent={true}
        animationType="fade"
        onRequestClose={() => {}}
      >
        <View style={modalStyles.modalOverlay}>
          <View style={modalStyles.loadingModalContainer}>
            <View style={modalStyles.loadingHeader}>
              <MaterialIcons name="upload" size={width * 0.08} color="#015185" />
              <Text style={modalStyles.loadingTitle}>Processing Application</Text>
            </View>
            
            <View style={modalStyles.loadingContent}>
              <ActivityIndicator size="large" color="#015185" style={modalStyles.loadingSpinner} />
              <Text style={modalStyles.loadingStep}>{loadingStep}</Text>
              
              <View style={modalStyles.progressContainer}>
                <View style={modalStyles.progressBar}>
                  <View 
                    style={[
                      modalStyles.progressFill, 
                      { width: `${((currentStepIndex + 1) / processingSteps.length) * 100}%` }
                    ]} 
                  />
                </View>
                <Text style={modalStyles.progressText}>
                  Step {currentStepIndex + 1} of {processingSteps.length}
                </Text>
              </View>
            </View>
          </View>
        </View>
      </Modal>

      {/* Success Modal */}
      <Modal
        visible={showSuccessModal}
        transparent={true}
        animationType="slide"
        onRequestClose={() => setShowSuccessModal(false)}
      >
        <View style={modalStyles.modalOverlay}>
          <View style={modalStyles.successModalContainer}>
            <View style={modalStyles.successHeader}>
              <MaterialIcons name="check-circle" size={width * 0.12} color="#00B894" />
              <Text style={modalStyles.successTitle}>Application Submitted!</Text>
              <Text style={modalStyles.successSubtitle}>
                Your {formatApplicationType(applicationType)} has been submitted successfully
              </Text>
            </View>
            
            <View style={modalStyles.emailStatusContainer}>
              <Text style={modalStyles.emailStatusTitle}>Email Notifications</Text>
              
              {/* Creator Email Status */}
              <View style={modalStyles.emailStatusRow}>
                <View style={modalStyles.emailStatusLeft}>
                  <MaterialIcons 
                    name={emailStatus.creatorEmail.status === 'success' ? 'check-circle' : 
                          emailStatus.creatorEmail.status === 'failed' ? 'error' :
                          emailStatus.creatorEmail.status === 'no_email' ? 'email' :
                          emailStatus.creatorEmail.status === 'offline' ? 'cloud-off' : 'schedule'} 
                    size={width * 0.05} 
                    color={emailStatus.creatorEmail.status === 'success' ? '#00B894' : 
                           emailStatus.creatorEmail.status === 'failed' ? '#ff4444' :
                           emailStatus.creatorEmail.status === 'no_email' ? '#ffd205' :
                           emailStatus.creatorEmail.status === 'offline' ? '#ffd205' : '#015185'} 
                  />
                  <View style={modalStyles.emailStatusTextContainer}>
                    <Text style={modalStyles.emailStatusText}>Confirmation email to you</Text>
                    {emailStatus.creatorEmail.email ? (
                      <Text style={modalStyles.emailAddressText}>{emailStatus.creatorEmail.email}</Text>
                    ) : null}
                  </View>
                </View>
                <Text style={[
                  modalStyles.emailStatusBadge,
                  emailStatus.creatorEmail.status === 'success' && modalStyles.emailStatusSuccess,
                  emailStatus.creatorEmail.status === 'failed' && modalStyles.emailStatusFailed,
                  emailStatus.creatorEmail.status === 'no_email' && modalStyles.emailStatusWarning,
                  emailStatus.creatorEmail.status === 'offline' && modalStyles.emailStatusOffline
                ]}>
                  {emailStatus.creatorEmail.status === 'success' ? 'Sent' : 
                   emailStatus.creatorEmail.status === 'failed' ? 'Failed' :
                   emailStatus.creatorEmail.status === 'no_email' ? 'No Email' :
                   emailStatus.creatorEmail.status === 'offline' ? 'Offline' : 'Pending'}
                </Text>
              </View>
              
              {/* Supervisor Email Status */}
              <View style={modalStyles.emailStatusRow}>
                <View style={modalStyles.emailStatusLeft}>
                  <MaterialIcons 
                    name={emailStatus.supervisorEmails.status === 'success' ? 'check-circle' : 
                          emailStatus.supervisorEmails.status === 'processing' ? 'schedule' :
                          emailStatus.supervisorEmails.status === 'no_match' ? 'info' :
                          emailStatus.supervisorEmails.status === 'offline' ? 'cloud-off' : 'schedule'} 
                    size={width * 0.05} 
                    color={emailStatus.supervisorEmails.status === 'success' ? '#00B894' : 
                           emailStatus.supervisorEmails.status === 'processing' ? '#015185' :
                           emailStatus.supervisorEmails.status === 'no_match' ? '#ffd205' :
                           emailStatus.supervisorEmails.status === 'offline' ? '#ffd205' : '#015185'} 
                  />
                  <View style={modalStyles.emailStatusTextContainer}>
                    <Text style={modalStyles.emailStatusText}>
                      {emailStatus.supervisorEmails.status === 'no_match' ? 
                        'No matching supervisors found' : 
                        `Emails to supervisors (${selectedCountry} - ${selectedProject})`}
                    </Text>
                    {emailStatus.supervisorEmails.emails && emailStatus.supervisorEmails.emails.length > 0 ? (
                      <View style={modalStyles.emailListContainer}>
                        {emailStatus.supervisorEmails.emails.slice(0, 3).map((email, index) => (
                          <Text key={index} style={modalStyles.emailAddressText}>{email}</Text>
                        ))}
                        {emailStatus.supervisorEmails.emails.length > 3 ? (
                          <Text style={modalStyles.emailAddressText}>
                            +{emailStatus.supervisorEmails.emails.length - 3} more...
                          </Text>
                        ) : null}
                      </View>
                    ) : null}
                  </View>
                </View>
                <Text style={[
                  modalStyles.emailStatusBadge,
                  emailStatus.supervisorEmails.status === 'success' && modalStyles.emailStatusSuccess,
                  emailStatus.supervisorEmails.status === 'processing' && modalStyles.emailStatusProcessing,
                  emailStatus.supervisorEmails.status === 'no_match' && modalStyles.emailStatusWarning,
                  emailStatus.supervisorEmails.status === 'offline' && modalStyles.emailStatusOffline
                ]}>
                  {emailStatus.supervisorEmails.status === 'success' ? 
                    `Sent (${emailStatus.supervisorEmails.count})` : 
                   emailStatus.supervisorEmails.status === 'processing' ? 'Processing...' :
                   emailStatus.supervisorEmails.status === 'no_match' ? 'No Match' :
                   emailStatus.supervisorEmails.status === 'offline' ? 'Offline' : 'Pending'}
                </Text>
              </View>
            </View>
            
            {/* Email Recipients List */}
            {(emailStatus.creatorEmail.sent || emailStatus.supervisorEmails.sent) && (
              <View style={modalStyles.emailRecipientsContainer}>
                <Text style={modalStyles.emailRecipientsTitle}>📧 Emails Sent To:</Text>
                
                {emailStatus.creatorEmail.sent && emailStatus.creatorEmail.email && (
                  <View style={modalStyles.emailRecipientRow}>
                    <MaterialIcons name="person" size={width * 0.04} color="#00B894" />
                    <View style={modalStyles.emailRecipientInfo}>
                      <Text style={modalStyles.emailRecipientLabel}>You (Confirmation)</Text>
                      <Text style={modalStyles.emailRecipientAddress}>{emailStatus.creatorEmail.email}</Text>
                    </View>
                    <MaterialIcons name="check-circle" size={width * 0.04} color="#00B894" />
                  </View>
                )}
                
                {emailStatus.supervisorEmails.sent && emailStatus.supervisorEmails.emails.length > 0 && (
                  <>
                    <View style={modalStyles.emailRecipientRow}>
                      <MaterialIcons name="supervisor-account" size={width * 0.04} color="#015185" />
                      <View style={modalStyles.emailRecipientInfo}>
                        <Text style={modalStyles.emailRecipientLabel}>
                          Supervisors ({emailStatus.supervisorEmails.count} recipients)
                        </Text>
                        <Text style={modalStyles.emailRecipientSubtext}>
                          Matching {selectedCountry} - {selectedProject}
                        </Text>
                      </View>
                      <MaterialIcons name="check-circle" size={width * 0.04} color="#00B894" />
                    </View>
                    
                    <View style={modalStyles.supervisorEmailsList}>
                      {emailStatus.supervisorEmails.emails.slice(0, 5).map((email, index) => (
                        <Text key={index} style={modalStyles.supervisorEmailItem}>
                          • {email}
                        </Text>
                      ))}
                      {emailStatus.supervisorEmails.emails.length > 5 && (
                        <Text style={modalStyles.supervisorEmailItem}>
                          • +{emailStatus.supervisorEmails.emails.length - 5} more supervisors...
                        </Text>
                      )}
                    </View>
                  </>
                )}
              </View>
            )}
            
            <View style={modalStyles.applicationDetailsContainer}>
              <Text style={modalStyles.applicationDetailsTitle}>Application Details</Text>
              <View style={modalStyles.applicationDetailRow}>
                <Text style={modalStyles.applicationDetailLabel}>Type:</Text>
                <Text style={modalStyles.applicationDetailValue}>{formatApplicationType(applicationType)}</Text>
              </View>
              <View style={modalStyles.applicationDetailRow}>
                <Text style={modalStyles.applicationDetailLabel}>Employee:</Text>
                <Text style={modalStyles.applicationDetailValue}>{fullName} ({employeeNo})</Text>
              </View>
              <View style={modalStyles.applicationDetailRow}>
                <Text style={modalStyles.applicationDetailLabel}>Location:</Text>
                <Text style={modalStyles.applicationDetailValue}>{selectedCountry} - {selectedProject}</Text>
              </View>
              <View style={modalStyles.applicationDetailRow}>
                <Text style={modalStyles.applicationDetailLabel}>Status:</Text>
                <Text style={[modalStyles.applicationDetailValue, modalStyles.statusPending]}>Pending Review</Text>
              </View>
              <View style={modalStyles.applicationDetailRow}>
                <Text style={modalStyles.applicationDetailLabel}>Submitted:</Text>
                <Text style={modalStyles.applicationDetailValue}>
                  {new Date().toLocaleDateString('en-GB')} {new Date().toLocaleTimeString()}
                </Text>
              </View>
            </View>
            
            <View style={modalStyles.successActions}>
              <TouchableOpacity
                style={modalStyles.successButton}
                onPress={() => {
                  setShowSuccessModal(false);
                  navigation.goBack();
                }}
                accessibilityLabel="Close and go back"
                accessibilityRole="button"
              >
                <LinearGradient
                  colors={['#00B894', '#00A085']}
                  style={modalStyles.successButtonGradient}
                  start={{ x: 0, y: 0 }}
                  end={{ x: 1, y: 0 }}
                >
                  <MaterialIcons name="check" size={width * 0.05} color="#fff" />
                  <Text style={modalStyles.successButtonText}>Done</Text>
                </LinearGradient>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  backButton: {
    padding: width * 0.02,
  },
  headerTitle: {
    flex: 1,
    fontSize: width * 0.05,
    fontWeight: 'bold',
    color: '#fff',
    textAlign: 'center',
  },
  menuButton: {
    padding: width * 0.02,
  },
  keyboardAvoidingView: {
    flex: 1,
  },
  content: {
    paddingHorizontal: width * 0.04,
    paddingTop: height * 0.02,
  },
  inputRow: {
    marginBottom: height * 0.02,
  },
  twoColumnRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
  },
  halfWidthInput: {
    width: '48%',
  },
  dropdown: {
    borderWidth: 1,
    borderColor: '#ccc',
    borderRadius: 8,
    paddingHorizontal: width * 0.03,
    paddingVertical: height * 0.015,
    backgroundColor: '#fff',
  },
  dropdownContainer: {
    borderWidth: 1,
    borderColor: '#ccc',
    borderRadius: 8,
    backgroundColor: '#fff',
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 2,
  },
  dropdownText: {
    fontSize: width * 0.04,
    color: '#333',
  },
  selectedDisplayBox: {
    borderWidth: 1,
    borderColor: '#ccc',
    borderRadius: 8,
    padding: width * 0.03,
    minHeight: height * 0.06,
    justifyContent: 'center',
  },
  selectedItemsContainer: {
    flexDirection: 'row',
    flexWrap: 'wrap',
  },
  selectedItemText: {
    fontSize: width * 0.04,
    color: '#333',
  },
  noSelectionText: {
    fontSize: width * 0.04,
    color: '#888',
  },
  sectionTitle: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#333',
    marginBottom: height * 0.015,
  },
  table: {
    borderWidth: 1,
    borderColor: '#ccc',
    borderRadius: 8,
    marginBottom: height * 0.02,
  },
  tableRow: {
    flexDirection: 'row',
    borderBottomWidth: 1,
    borderColor: '#ccc',
  },
  tableHeaderCell: {
    fontSize: width * 0.04,
    fontWeight: 'bold',
    color: '#333',
    padding: width * 0.03,
    textAlign: 'center',
  },
  tableDataCell: {
    padding: width * 0.03,
    justifyContent: 'center',
    alignItems: 'center',
  },
  dateCell: {
    width: '30%',
  },
  descriptionCell: {
    width: '40%',
  },
  amountCell: {
    width: '30%',
  },
  tableDateButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    padding: width * 0.02,
    backgroundColor: '#f5f5f5',
    borderRadius: 4,
  },
  tableDateText: {
    fontSize: width * 0.035,
    color: '#333',
    marginRight: width * 0.01,
  },
  tableCellText: {
    fontSize: width * 0.035,
    color: '#333',
    textAlign: 'center',
  },
  tableInput: {
    fontSize: width * 0.035,
    color: '#333',
    textAlign: 'center',
    borderWidth: 1,
    borderColor: '#ccc',
    borderRadius: 4,
    padding: width * 0.02,
  },
  tableTotalCell: {
    fontSize: width * 0.04,
    fontWeight: 'bold',
    color: '#333',
    padding: width * 0.03,
    textAlign: 'center',
  },
  checkboxContainer: {
    marginTop: height * 0.01,
  },
  checkboxRow: {
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: height * 0.01,
  },
  checkbox: {
    width: width * 0.05,
    height: width * 0.05,
    borderWidth: 1,
    borderColor: '#333',
    borderRadius: 4,
    justifyContent: 'center',
    alignItems: 'center',
    marginRight: width * 0.02,
  },
  checkboxSelected: {
    backgroundColor: '#015185',
    borderColor: '#015185',
  },
  checkboxLabel: {
    fontSize: width * 0.04,
    color: '#333',
  },
  radioContainer: {
    marginTop: height * 0.01,
  },
  radioRow: {
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: height * 0.01,
  },
  radio: {
    width: width * 0.05,
    height: width * 0.05,
    borderWidth: 1,
    borderColor: '#333',
    borderRadius: width * 0.025,
    justifyContent: 'center',
    alignItems: 'center',
    marginRight: width * 0.02,
  },
  radioSelected: {
    borderColor: '#015185',
  },
  radioInner: {
    width: width * 0.03,
    height: width * 0.03,
    backgroundColor: '#015185',
    borderRadius: width * 0.015,
  },
  radioLabel: {
    fontSize: width * 0.04,
    color: '#333',
  },
  signatureContainer: {
    marginBottom: height * 0.02,
  },
  signaturePad: {
    height: height * 0.15,
    borderWidth: 1,
    borderColor: '#ccc',
    borderRadius: 8,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: '#f5f5f5',
  },
  signaturePadError: {
    borderColor: '#ff4444',
  },
  signatureText: {
    fontSize: width * 0.04,
    color: '#888',
  },
  capturedSignatureImage: {
    width: '100%',
    height: '100%',
    borderRadius: 8,
  },
  complainantContainer: {
    borderWidth: 1,
    borderColor: '#ccc',
    borderRadius: 8,
    padding: width * 0.03,
    marginBottom: height * 0.02,
    position: 'relative',
  },
  complainantInput: {
    marginBottom: height * 0.01,
  },
  removeComplainantButton: {
    position: 'absolute',
    top: width * 0.02,
    right: width * 0.02,
  },
  addComplainantButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    padding: width * 0.03,
    backgroundColor: '#f5f5f5',
    borderRadius: 8,
    marginBottom: height * 0.02,
  },
  addComplainantText: {
    fontSize: width * 0.04,
    color: '#015185',
    fontWeight: '600',
  },
  submitButton: {
    marginTop: height * 0.02,
    marginBottom: height * 0.04,
    borderRadius: 8,
    overflow: 'hidden',
  },
  submitButtonGradient: {
    paddingVertical: height * 0.02,
    alignItems: 'center',
  },
  submitButtonText: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#fff',
  },
});

const inputStyles = StyleSheet.create({
  container: {
    marginBottom: height * 0.02,
  },
  label: {
    fontSize: width * 0.04,
    fontWeight: '600',
    color: '#333',
    marginBottom: height * 0.005,
  },
  inputContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    borderWidth: 1,
    borderColor: '#ccc',
    borderRadius: 8,
    backgroundColor: '#fff',
  },
  input: {
    flex: 1,
    fontSize: width * 0.04,
    color: '#333',
    paddingHorizontal: width * 0.03,
    paddingVertical: height * 0.015,
  },
  inputError: {
    borderColor: '#ff4444',
  },
  disabledInput: {
    backgroundColor: '#f5f5f5',
    color: '#888',
  },
  noBackground: {
    backgroundColor: 'transparent',
  },
  inputIcon: {
    padding: width * 0.03,
  },
  selectButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingHorizontal: width * 0.03,
    paddingVertical: height * 0.01,
  },
  selectText: {
    fontSize: width * 0.04,
    color: '#333',
    marginRight: width * 0.01,
  },
  errorText: {
    fontSize: width * 0.035,
    color: '#ff4444',
    marginTop: height * 0.005,
  },
  requiredError: {
    color: '#ff4444',
  },
  noInputContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
  },
});

const modalStyles = StyleSheet.create({
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  loadingModalContainer: {
    backgroundColor: '#fff',
    borderRadius: 12,
    padding: width * 0.06,
    width: width * 0.85,
    maxWidth: 400,
    alignItems: 'center',
  },
  loadingHeader: {
    alignItems: 'center',
    marginBottom: height * 0.03,
  },
  loadingTitle: {
    fontSize: width * 0.05,
    fontWeight: 'bold',
    color: '#015185',
    marginTop: height * 0.01,
  },
  loadingContent: {
    alignItems: 'center',
    width: '100%',
  },
  loadingSpinner: {
    marginBottom: height * 0.02,
  },
  loadingStep: {
    fontSize: width * 0.04,
    color: '#333',
    textAlign: 'center',
    marginBottom: height * 0.02,
    minHeight: height * 0.03,
  },
  progressContainer: {
    width: '100%',
    alignItems: 'center',
  },
  progressBar: {
    width: '100%',
    height: 6,
    backgroundColor: '#e0e0e0',
    borderRadius: 3,
    overflow: 'hidden',
    marginBottom: height * 0.01,
  },
  progressFill: {
    height: '100%',
    backgroundColor: '#015185',
    borderRadius: 3,
  },
  progressText: {
    fontSize: width * 0.035,
    color: '#666',
  },
  successModalContainer: {
    backgroundColor: '#fff',
    borderRadius: 12,
    padding: width * 0.06,
    width: width * 0.9,
    maxWidth: 450,
    maxHeight: height * 0.8,
  },
  successHeader: {
    alignItems: 'center',
    marginBottom: height * 0.03,
    paddingBottom: height * 0.02,
    borderBottomWidth: 1,
    borderBottomColor: '#e0e0e0',
  },
  successTitle: {
    fontSize: width * 0.055,
    fontWeight: 'bold',
    color: '#00B894',
    marginTop: height * 0.01,
    marginBottom: height * 0.005,
  },
  successSubtitle: {
    fontSize: width * 0.04,
    color: '#666',
    textAlign: 'center',
    lineHeight: width * 0.055,
  },
  emailStatusContainer: {
    marginBottom: height * 0.025,
  },
  emailStatusTitle: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: height * 0.015,
  },
  emailStatusRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingVertical: height * 0.012,
    paddingHorizontal: width * 0.03,
    backgroundColor: '#f8f9fa',
    borderRadius: 8,
    marginBottom: height * 0.01,
  },
  emailStatusLeft: {
    flexDirection: 'row',
    alignItems: 'center',
    flex: 1,
  },
  emailStatusTextContainer: {
    flex: 1,
    marginLeft: width * 0.02,
  },
  emailStatusText: {
    fontSize: width * 0.038,
    color: '#333',
    fontWeight: '500',
  },
  emailAddressText: {
    fontSize: width * 0.032,
    color: '#666',
    fontStyle: 'italic',
    marginTop: height * 0.002,
  },
  emailListContainer: {
    marginTop: height * 0.005,
  },
  emailStatusBadge: {
    fontSize: width * 0.035,
    fontWeight: '600',
    paddingHorizontal: width * 0.025,
    paddingVertical: height * 0.005,
    borderRadius: 12,
    backgroundColor: '#e0e0e0',
    color: '#666',
    textAlign: 'center',
    minWidth: width * 0.15,
  },
  emailStatusSuccess: {
    backgroundColor: '#d4edda',
    color: '#155724',
  },
  emailStatusFailed: {
    backgroundColor: '#f8d7da',
    color: '#721c24',
  },
  emailStatusProcessing: {
    backgroundColor: '#d1ecf1',
    color: '#0c5460',
  },
  emailStatusWarning: {
    backgroundColor: '#fff3cd',
    color: '#856404',
  },
  emailStatusOffline: {
    backgroundColor: '#ffeaa7',
    color: '#b8860b',
  },
  emailRecipientsContainer: {
    backgroundColor: '#f8f9fa',
    borderRadius: 8,
    padding: width * 0.04,
    marginBottom: height * 0.025,
    borderLeftWidth: 4,
    borderLeftColor: '#00B894',
  },
  emailRecipientsTitle: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: height * 0.015,
  },
  emailRecipientRow: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: height * 0.008,
    marginBottom: height * 0.01,
  },
  emailRecipientInfo: {
    flex: 1,
    marginLeft: width * 0.02,
    marginRight: width * 0.02,
  },
  emailRecipientLabel: {
    fontSize: width * 0.038,
    fontWeight: '600',
    color: '#333',
  },
  emailRecipientAddress: {
    fontSize: width * 0.035,
    color: '#666',
    fontStyle: 'italic',
  },
  emailRecipientSubtext: {
    fontSize: width * 0.033,
    color: '#888',
    fontStyle: 'italic',
  },
  supervisorEmailsList: {
    marginLeft: width * 0.06,
    marginTop: height * 0.01,
    paddingLeft: width * 0.03,
    borderLeftWidth: 2,
    borderLeftColor: '#e0e0e0',
  },
  supervisorEmailItem: {
    fontSize: width * 0.033,
    color: '#666',
    marginBottom: height * 0.003,
    fontFamily: 'monospace',
  },
  applicationDetailsContainer: {
    backgroundColor: '#f8f9fa',
    borderRadius: 8,
    padding: width * 0.04,
    marginBottom: height * 0.025,
  },
  applicationDetailsTitle: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: height * 0.015,
  },
  applicationDetailRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingVertical: height * 0.008,
    borderBottomWidth: 1,
    borderBottomColor: '#e0e0e0',
  },
  applicationDetailLabel: {
    fontSize: width * 0.038,
    color: '#666',
    fontWeight: '500',
    flex: 0.4,
  },
  applicationDetailValue: {
    fontSize: width * 0.038,
    color: '#333',
    fontWeight: '500',
    flex: 0.6,
    textAlign: 'right',
  },
  statusPending: {
    color: '#ffd205',
    fontWeight: 'bold',
  },
  successActions: {
    alignItems: 'center',
  },
  successButton: {
    borderRadius: 8,
    overflow: 'hidden',
    width: '100%',
  },
  successButtonGradient: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingVertical: height * 0.018,
    paddingHorizontal: width * 0.06,
  },
  successButtonText: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#fff',
    marginLeft: width * 0.02,
  },
});

export default AdminHrAppliApply;