import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  TextInput,
  Image,
  Dimensions,
  Animated,
  Modal,
  FlatList,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import Ionicons from 'react-native-vector-icons/Ionicons';
import { useNavigation, useFocusEffect } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import database from '@react-native-firebase/database';
import Toast from 'react-native-toast-message';
import NetInfo from '@react-native-community/netinfo';

const { width } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.45;

const IncidentReport = () => {
  const navigation = useNavigation();
  const [searchQuery, setSearchQuery] = useState('');
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [incidentReports, setIncidentReports] = useState([]);
  const [filteredReports, setFilteredReports] = useState([]);
  const [archivedReports, setArchivedReports] = useState([]);
  const [filteredArchivedReports, setFilteredArchivedReports] = useState([]);
  const [isRefreshing, setIsRefreshing] = useState(false);
  const [dropdownVisible, setDropdownVisible] = useState(false);
  const [selectedReport, setSelectedReport] = useState(null);
  const [dropdownPosition, setDropdownPosition] = useState({ x: 0, y: 0, openUp: false });
  const [confirmDeleteVisible, setConfirmDeleteVisible] = useState(false);
  const [reportToDelete, setReportToDelete] = useState(null);
  const [showArchived, setShowArchived] = useState(false);
  const [syncMessage, setSyncMessage] = useState('Cloud Data Sync in Progress');
  const [reportedByName, setReportedByName] = useState('N/A');
  const [permissionsModalVisible, setPermissionsModalVisible] = useState(false);
  const [userPermissions, setUserPermissions] = useState({
    countries: [],
    projects: [],
    incidentReportPermissions: [],
  });
  const [currentUserUid, setCurrentUserUid] = useState(null);
  const [hasOnlyMineView, setHasOnlyMineView] = useState(false);
  const [hasCreatePermission, setHasCreatePermission] = useState(false);
  const [hasDeletePermission, setHasDeletePermission] = useState(false);
  const [hasEditPermission, setHasEditPermission] = useState(false);
  const [hasViewPermission, setHasViewPermission] = useState(false);

  const rotateAnim = useRef(new Animated.Value(0)).current;
  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  const fetchReportedByName = async () => {
    try {
      const userName = await AsyncStorage.getItem('userName');
      if (userName) {
        setReportedByName(userName);
      } else {
        setReportedByName('N/A');
      }
    } catch (error) {
      console.error('Error fetching reported by name:', error);
      setReportedByName('N/A');
    }
  };

  const fetchUserPermissions = async () => {
    try {
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) {
        throw new Error('User not authenticated. Please log in again.');
      }
      
      setCurrentUserUid(userUid);

      const userSnapshot = await database()
        .ref(`/Globalusers/admins/${userUid}`)
        .once('value');
      const userData = userSnapshot.val();

      if (!userData) {
        throw new Error('User data not found.');
      }

      const incidentReportModule = userData.modules?.find(
        (mod) => mod.module === 'incidentReport'
      );

      const permissions = incidentReportModule?.permissions || [];
      
      setUserPermissions({
        countries: userData.countries || [],
        projects: userData.projects || [],
        incidentReportPermissions: permissions,
      });

      // Set permission flags
      setHasOnlyMineView(permissions.includes('onlyMineView'));
      setHasCreatePermission(permissions.includes('create'));
      setHasDeletePermission(permissions.includes('delete'));
      setHasEditPermission(permissions.includes('edit'));
      setHasViewPermission(permissions.includes('view'));
      
    } catch (error) {
      console.error('Error fetching user permissions:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to fetch user permissions.',
        position: 'top',
      });
      setUserPermissions({
        countries: [],
        projects: [],
        incidentReportPermissions: [],
      });
    }
  };

  const isReportEmpty = (report) => {
    const fields = [
      report.incidentNumber,
      report.incidentDate,
      report.incidentArea,
      report.severity,
      report.incidentCategory,
      report.selectedIncidents && Object.values(report.selectedIncidents).some(Boolean),
    ];
    return fields.every(
      (field) =>
        field === undefined ||
        field === null ||
        field === '' ||
        field === 'N/A' ||
        field === false
    );
  };

  // Helper: filter reports according to current permissions
  const filterReportsByPermissions = (allReports, permissions, onlyMine, userUid) => {
    let filtered = allReports;
    // Filter by country and project permissions
    if (permissions.countries.length > 0 || permissions.projects.length > 0) {
      filtered = filtered.filter((report) => {
        const countryMatch = permissions.countries.length === 0 || permissions.countries.includes(report.country);
        const projectMatch = permissions.projects.length === 0 || permissions.projects.includes(report.project);
        return countryMatch && projectMatch;
      });
    }
    // Only mine view
    if (onlyMine && userUid) {
      filtered = filtered.filter((report) => report.adminUid === userUid);
    }
    return filtered;
  };

  const fetchIncidentReports = async (showModal = false) => {
    if (showModal) {
      setSyncMessage('Cloud Data Sync in Progress');
      setSyncModalVisible(true);
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    }

    try {
      await fetchReportedByName();
      await fetchUserPermissions();

      // Load from GlobalIncidentReports/admins structure
      const adminsSnapshot = await database()
        .ref('/GlobalIncidentReports/admins')
        .once('value');
      const adminsData = adminsSnapshot.val();
      const reports = [];
      
      console.log('Raw database data:', adminsData);
      
      if (adminsData) {
        Object.keys(adminsData).forEach((adminKey) => {
          if (adminKey === 'lastIncidentNumber') return;
          const userReports = adminsData[adminKey];
          console.log(`Processing admin ${adminKey}:`, userReports);
          if (userReports) {
            Object.keys(userReports).forEach((reportId) => {
              const reportData = userReports[reportId];
              if (reportData) {
                // Flatten the data structure for home page compatibility
                const flattenedReport = {
                  id: reportId,
                  adminUid: adminKey,
                  // Flatten step1 data to top level
                  incidentNumber: reportData.incidentNumber || reportData.step1?.incidentNumber,
                  incidentDate: reportData.incidentDate || reportData.step1?.incidentDate,
                  incidentArea: reportData.incidentArea || reportData.step1?.incidentArea,
                  severity: reportData.severity || reportData.step1?.severity,
                  incidentCategory: reportData.incidentCategory || reportData.step1?.incidentCategory,
                  country: reportData.country || reportData.step1?.country,
                  project: reportData.project || reportData.step1?.project,
                  selectedIncidents: reportData.selectedIncidents || reportData.step1?.selectedIncidents,
                  involvesEquipment: reportData.involvesEquipment || reportData.step1?.involvesEquipment,
                  vehicleId: reportData.vehicleId || reportData.step1?.vehicleId,
                  vehicleName: reportData.vehicleName || reportData.step1?.vehicleName,
                  personInvolved: reportData.personInvolved || reportData.step1?.personInvolved,
                  coordinates: reportData.coordinates || reportData.step1?.coordinates,
                  otherIncidentNote: reportData.otherIncidentNote || reportData.step1?.otherIncidentNote,
                  createdAt: reportData.createdAt || reportData.step1?.createdAt,
                  isArchived: reportData.isArchived || false,
                  pdfDownloadUrl: reportData.pdfDownloadUrl || reportData.step2?.pdfDownloadUrl,
                  // Keep original structure for compatibility
                  step1: reportData.step1,
                  step2: reportData.step2,
                };
                reports.push(flattenedReport);
                console.log('Added report:', flattenedReport.incidentNumber);
              }
            });
          }
        });
      }

      console.log('Total reports loaded:', reports.length);
      const nonEmptyReports = reports.filter((report) => !isReportEmpty(report));
      console.log('Non-empty reports:', nonEmptyReports.length);
      
      // Always filter by up-to-date permissions
      const filtered = filterReportsByPermissions(
        nonEmptyReports,
        {
          countries: userPermissions.countries,
          projects: userPermissions.projects,
        },
        hasOnlyMineView,
        currentUserUid
      );
      console.log('Filtered reports:', filtered.length);
      
      // Sort reports by creation date (newest first)
      const sortedReports = filtered.sort((a, b) => {
        const dateA = new Date(a.createdAt || a.incidentDate || 0);
        const dateB = new Date(b.createdAt || b.incidentDate || 0);
        return dateB.getTime() - dateA.getTime(); // Newest first
      });
      
      const activeReports = sortedReports.filter((report) => !report.isArchived);
      const archivedReports = sortedReports.filter((report) => report.isArchived);
      
      console.log('Active reports:', activeReports.length);
      console.log('Archived reports:', archivedReports.length);
      
      setIncidentReports(activeReports);
      setFilteredReports(activeReports);
      setArchivedReports(archivedReports);
      setFilteredArchivedReports(archivedReports);
      await AsyncStorage.setItem('incidentReports', JSON.stringify(filtered));
    } catch (error) {
      console.error('Error fetching incident reports:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to fetch incident reports from the cloud.',
        position: 'top',
      });
      const localReports = await AsyncStorage.getItem('incidentReports');
      const parsedLocalReports = localReports ? JSON.parse(localReports) : [];
      const activeLocalReports = parsedLocalReports.filter((report) => !report.isArchived);
      const archivedLocalReports = parsedLocalReports.filter((report) => report.isArchived);
      setIncidentReports(activeLocalReports);
      setFilteredReports(activeLocalReports);
      setArchivedReports(archivedLocalReports);
      setFilteredArchivedReports(archivedLocalReports);
    } finally {
      if (showModal) {
        setSyncModalVisible(false);
        rotateAnim.setValue(0);
      }
    }
  };

  // On mount, fetch permissions and reports, then filter
  useFocusEffect(
    React.useCallback(() => {
      const loadAll = async () => {
        // Use the same logic as fetchIncidentReports but without modal
        await fetchIncidentReports(false);
      };
      loadAll();
    }, [])
  );

  const handleSearch = (text) => {
    setSearchQuery(text);
    if (text) {
      const filteredActive = incidentReports.filter((report) =>
        [
          report.incidentNumber,
          report.country,
          report.project,
          report.vehicleId,
          report.vehicleName,
          report.personInvolved,
          report.incidentArea,
          report.severity,
          report.incidentCategory,
          ...Object.keys(report.selectedIncidents || {}).filter(
            (key) => report.selectedIncidents[key]
          ),
        ].some(
          (field) =>
            field && field.toLowerCase().includes(text.toLowerCase())
        )
      );
      const filteredArchived = archivedReports.filter((report) =>
        [
          report.incidentNumber,
          report.country,
          report.project,
          report.vehicleId,
          report.vehicleName,
          report.personInvolved,
          report.incidentArea,
          report.severity,
          report.incidentCategory,
          ...Object.keys(report.selectedIncidents || {}).filter(
            (key) => report.selectedIncidents[key]
          ),
        ].some(
          (field) =>
            field && field.toLowerCase().includes(text.toLowerCase())
        )
      );
      
      // Sort search results by creation date (newest first)
      const sortedActive = filteredActive.sort((a, b) => {
        const dateA = new Date(a.createdAt || a.incidentDate || 0);
        const dateB = new Date(b.createdAt || b.incidentDate || 0);
        return dateB.getTime() - dateA.getTime();
      });
      
      const sortedArchived = filteredArchived.sort((a, b) => {
        const dateA = new Date(a.createdAt || a.incidentDate || 0);
        const dateB = new Date(b.createdAt || b.incidentDate || 0);
        return dateB.getTime() - dateA.getTime();
      });
      
      setFilteredReports(sortedActive);
      setFilteredArchivedReports(sortedArchived);
    } else {
      setFilteredReports(incidentReports);
      setFilteredArchivedReports(archivedReports);
    }
  };

  const handleRefresh = () => {
    setIsRefreshing(true);
    fetchIncidentReports(true);
    setIsRefreshing(false);
  };

  const handleViewPDF = (pdfDownloadUrl) => {
    if (!hasViewPermission && !hasEditPermission) {
      Toast.show({
        type: 'error',
        text1: 'Permission Denied',
        text2: 'You do not have permission to view PDF reports.',
        position: 'top',
      });
      return;
    }

    if (!pdfDownloadUrl || typeof pdfDownloadUrl !== 'string') {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'No PDF available for this report.',
        position: 'top',
      });
      return;
    }

    if (!pdfDownloadUrl.startsWith('https://firebasestorage.googleapis.com/')) {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Invalid PDF URL for this report.',
        position: 'top',
      });
      return;
    }

    try {
      navigation.navigate('PdfviewPage', {
        pdfUrl: pdfDownloadUrl,
        onGoBack: () => navigation.navigate('IncidentReport'),
      });
    } catch (error) {
      console.error('Error navigating to PDF viewer:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to open the PDF. Please try again.',
        position: 'top',
      });
    }
  };

  const handleArchiveReport = async () => {
    if (!selectedReport) return;

    setDropdownVisible(false);
    setSyncMessage(`Archiving Incident Report ${selectedReport.incidentNumber || 'N/A'}`);
    setSyncModalVisible(true);

    Animated.loop(
      Animated.timing(rotateAnim, {
        toValue: 1,
        duration: 1000,
        useNativeDriver: true,
      })
    ).start();

    try {
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) {
        throw new Error('User not authenticated. Please log in again.');
      }

      await database()
        .ref(`/GlobalIncidentReports/admins/${userUid}/${selectedReport.id}`)
        .update({ isArchived: true });

      const updatedActiveReports = incidentReports.filter(
        (report) => report.id !== selectedReport.id
      );
      const updatedArchivedReports = [
        ...archivedReports,
        { ...selectedReport, isArchived: true },
      ];

      // Sort both lists by creation date (newest first)
      const sortedActive = updatedActiveReports.sort((a, b) => {
        const dateA = new Date(a.createdAt || a.incidentDate || 0);
        const dateB = new Date(b.createdAt || b.incidentDate || 0);
        return dateB.getTime() - dateA.getTime();
      });
      
      const sortedArchived = updatedArchivedReports.sort((a, b) => {
        const dateA = new Date(a.createdAt || a.incidentDate || 0);
        const dateB = new Date(b.createdAt || b.incidentDate || 0);
        return dateB.getTime() - dateA.getTime();
      });

      setIncidentReports(sortedActive);
      setFilteredReports(sortedActive);
      setArchivedReports(sortedArchived);
      setFilteredArchivedReports(sortedArchived);

      const allReports = [...updatedActiveReports, ...updatedArchivedReports];
      await AsyncStorage.setItem('incidentReports', JSON.stringify(allReports));

      Toast.show({
        type: 'success',
        text1: 'Success',
        text2: `Incident Report ${selectedReport.incidentNumber || 'N/A'} archived successfully.`,
        position: 'top',
      });
    } catch (error) {
      console.error('Error archiving incident report:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to archive the incident report. Please try again.',
        position: 'top',
      });
    } finally {
      setSyncModalVisible(false);
      rotateAnim.setValue(0);
      setSelectedReport(null);
    }
  };

  const handleUnarchiveReport = async (report) => {
    setSyncMessage(`Unarchiving Incident Report ${report.incidentNumber || 'N/A'}`);
    setSyncModalVisible(true);

    Animated.loop(
      Animated.timing(rotateAnim, {
        toValue: 1,
        duration: 1000,
        useNativeDriver: true,
      })
    ).start();

    try {
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) {
        throw new Error('User not authenticated. Please log in again.');
      }

      await database()
        .ref(`/GlobalIncidentReports/admins/${userUid}/${report.id}`)
        .update({ isArchived: false });

      const updatedArchivedReports = archivedReports.filter(
        (r) => r.id !== report.id
      );
      const updatedActiveReports = [
        ...incidentReports,
        { ...report, isArchived: false },
      ];

      // Sort both lists by creation date (newest first)
      const sortedActive = updatedActiveReports.sort((a, b) => {
        const dateA = new Date(a.createdAt || a.incidentDate || 0);
        const dateB = new Date(b.createdAt || b.incidentDate || 0);
        return dateB.getTime() - dateA.getTime();
      });
      
      const sortedArchived = updatedArchivedReports.sort((a, b) => {
        const dateA = new Date(a.createdAt || a.incidentDate || 0);
        const dateB = new Date(b.createdAt || b.incidentDate || 0);
        return dateB.getTime() - dateA.getTime();
      });

      setArchivedReports(sortedArchived);
      setFilteredArchivedReports(sortedArchived);
      setIncidentReports(sortedActive);
      setFilteredReports(sortedActive);

      const allReports = [...updatedActiveReports, ...updatedArchivedReports];
      await AsyncStorage.setItem('incidentReports', JSON.stringify(allReports));

      Toast.show({
        type: 'success',
        text1: 'Success',
        text2: `Incident Report ${report.incidentNumber || 'N/A'} unarchived successfully.`,
        position: 'top',
      });
    } catch (error) {
      console.error('Error unarchiving incident report:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to unarchive the incident report. Please try again.',
        position: 'top',
      });
    } finally {
      setSyncModalVisible(false);
      rotateAnim.setValue(0);
    }
  };

  const handleDeleteReport = async () => {
    if (!reportToDelete) return;

    if (!hasDeletePermission) {
      Toast.show({
        type: 'error',
        text1: 'Permission Denied',
        text2: 'You do not have permission to delete incident reports.',
        position: 'top',
      });
      setConfirmDeleteVisible(false);
      setReportToDelete(null);
      return;
    }

    setConfirmDeleteVisible(false);
    setDropdownVisible(false);
    setSyncMessage(`Deleting Incident Report ${reportToDelete.incidentNumber || 'N/A'}`);
    setSyncModalVisible(true);

    Animated.loop(
      Animated.timing(rotateAnim, {
        toValue: 1,
        duration: 1000,
        useNativeDriver: true,
      })
    ).start();

    try {
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) {
        throw new Error('User not authenticated. Please log in again.');
      }

      // Delete from global incident reports
      await database()
        .ref(`/GlobalIncidentReports/admins/${userUid}/${reportToDelete.id}`)
        .remove();

      let updatedActiveReports = incidentReports;
      let updatedArchivedReports = archivedReports;
      if (reportToDelete.isArchived) {
        updatedArchivedReports = archivedReports.filter(
          (report) => report.id !== reportToDelete.id
        );
      } else {
        updatedActiveReports = incidentReports.filter(
          (report) => report.id !== reportToDelete.id
        );
      }
      setIncidentReports(updatedActiveReports);
      setFilteredReports(updatedActiveReports);
      setArchivedReports(updatedArchivedReports);
      setFilteredArchivedReports(updatedArchivedReports);

      const allReports = [...updatedActiveReports, ...updatedArchivedReports];
      await AsyncStorage.setItem('incidentReports', JSON.stringify(allReports));

      Toast.show({
        type: 'success',
        text1: 'Success',
        text2: `Incident Report ${reportToDelete.incidentNumber || 'N/A'} deleted successfully.`,
        position: 'top',
      });
    } catch (error) {
      console.error('Error deleting incident report:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to delete the incident report. Please try again.',
        position: 'top',
      });
    } finally {
      setSyncModalVisible(false);
      rotateAnim.setValue(0);
      setReportToDelete(null);
    }
  };

  const handleUploadToCloud = async (report) => {
    if (!report) return;

    setDropdownVisible(false);
    setSyncMessage(`Uploading Incident Report ${report.incidentNumber || 'N/A'} to Cloud`);
    setSyncModalVisible(true);

    Animated.loop(
      Animated.timing(rotateAnim, {
        toValue: 1,
        duration: 1000,
        useNativeDriver: true,
      })
    ).start();

    try {
      const isConnected = await NetInfo.fetch();
      if (!isConnected.isConnected) {
        Toast.show({
          type: 'error',
          text1: 'No Internet Connection',
          text2: 'Please check your internet connection and try again.',
          position: 'top',
        });
        return;
      }

      // Upload the report to Firebase
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) {
        throw new Error('User not authenticated. Please log in again.');
      }
      await database()
        .ref(`/GlobalIncidentReports/admins/${userUid}/${report.id}`)
        .set(report);

      Toast.show({
        type: 'success',
        text1: 'Success',
        text2: 'Incident report uploaded to cloud successfully.',
        position: 'top',
      });
    } catch (error) {
      console.error('Error uploading incident report to cloud:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to upload incident report to cloud. Please try again.',
        position: 'top',
      });
    } finally {
      setSyncModalVisible(false);
      rotateAnim.setValue(0);
    }
  };

  const formatDate = (dateValue) => {
    if (!dateValue) {
      console.warn('Date is missing for report:', dateValue);
      return 'N/A';
    }

    let dateObj;
    if (typeof dateValue === 'number') {
      dateObj = new Date(dateValue);
    } else if (typeof dateValue === 'string') {
      dateObj = new Date(dateValue);
    } else {
      console.warn('Invalid date format:', dateValue);
      return 'N/A';
    }

    if (isNaN(dateObj.getTime())) {
      console.warn('Invalid date:', dateValue);
      return 'N/A';
    }

    try {
      // Try with timeZone, but fallback if error
      return dateObj.toLocaleString('en-IN', {
        day: '2-digit',
        month: 'short',
        year: 'numeric',
        hour: '2-digit',
        minute: '2-digit',
        hour12: true,
        timeZone: 'Asia/Kolkata',
      });
    } catch (e) {
      // Fallback: no timeZone
      return dateObj.toLocaleString('en-IN', {
        day: '2-digit',
        month: 'short',
        year: 'numeric',
        hour: '2-digit',
        minute: '2-digit',
        hour12: true,
      });
    }
  };

  const getSelectedIncidentsList = (selectedIncidents) => {
    if (!selectedIncidents || typeof selectedIncidents !== 'object') {
      return 'N/A';
    }
    const selected = Object.keys(selectedIncidents).filter(
      (key) => selectedIncidents[key]
    );
    return selected.length > 0 ? selected.join(', ') : 'None';
  };

  const canPerformActionOnReport = (report) => {
    if (!report) return false; // Prevents crash if report is null/undefined
    // If user has onlyMineView permission, they can only perform actions on their own reports
    if (hasOnlyMineView && currentUserUid) {
      return report.adminUid === currentUserUid;
    }
    // If no onlyMineView restriction, user can perform actions on any report they have permission for
    return true;
  };

  const renderIncidentReport = ({ item }) => {
    const hasAnyPermission = hasViewPermission || hasEditPermission || hasDeletePermission;
    const canPerformAction = canPerformActionOnReport(item);
    
    return (
      <View style={styles.reportCard}>
        <View style={styles.cardHeader}>
          <Text style={styles.cardTitle}>{String(item.incidentNumber || 'N/A')}</Text>
        </View>
        <View style={styles.cardRow}>
          <Text style={styles.cardLabel}>Incident Date</Text>
          <Text style={styles.cardValue}>{String(formatDate(item.incidentDate))}</Text>
        </View>
        <View style={styles.cardRow}>
          <Text style={styles.cardLabel}>Incident Area</Text>
          <Text style={styles.cardValue}>{String(item.incidentArea || 'N/A')}</Text>
        </View>
        <View style={styles.cardRow}>
          <Text style={styles.cardLabel}>Severity</Text>
          <Text style={styles.cardValue}>{String(item.severity || 'N/A')}</Text>
        </View>
        <View style={styles.cardRow}>
          <Text style={styles.cardLabel}>Involves Equipment</Text>
          <Text style={styles.cardValue}>{item.involvesEquipment ? 'Yes' : 'No'}</Text>
        </View>
        {item.involvesEquipment && (
          <>
            <View style={styles.cardRow}>
              <Text style={styles.cardLabel}>Equipment #</Text>
              <Text style={styles.cardValue}>{String(item.vehicleId || 'N/A')}</Text>
            </View>
            <View style={styles.cardRow}>
              <Text style={styles.cardLabel}>Equipment Name</Text>
              <Text style={styles.cardValue}>{String(item.vehicleName || 'N/A')}</Text>
            </View>
          </>
        )}
        {/* Created At row with 3-dots menu at the right */}
        <View style={styles.cardRow}>
          <Text style={styles.cardLabel}>Created At</Text>
          <View style={{ flexDirection: 'row', alignItems: 'center', flex: 1, justifyContent: 'space-between' }}>
            <Text style={styles.cardValue}>{String(formatDate(item.createdAt))}</Text>
            {hasAnyPermission && canPerformAction && (
              <TouchableOpacity
                onPress={(event) => {
                  setSelectedReport(item);
                  const { pageX, pageY } = event.nativeEvent;
                  const dropdownHeight = 280;
                  const screenHeight = Dimensions.get('window').height;
                  let top = pageY + 2;
                  let openUp = false;
                  if (pageY + 2 + dropdownHeight + 20 > screenHeight) {
                    top = pageY - dropdownHeight - 2;
                    openUp = true;
                  }
                  setDropdownPosition({ x: pageX - 200 + 24, y: top, openUp });
                  setDropdownVisible(true);
                }}
                style={styles.menuButton}
              >
                <MaterialIcons name="more-vert" size={24} color="#333" />
              </TouchableOpacity>
            )}
          </View>
        </View>
      </View>
    );
  };

  const renderArchivedReport = ({ item }) => {
    return (
      <View style={[styles.reportCard, { backgroundColor: '#F5F5F5' }]}>
        <View style={styles.cardHeader}>
          <Text style={styles.cardTitle}>{String(item.incidentNumber || 'N/A')}</Text>
          <TouchableOpacity
            onPress={() => handleUnarchiveReport(item)}
            style={styles.menuButton}
          >
            <MaterialIcons name="unarchive" size={24} color="#3481BC" />
          </TouchableOpacity>
        </View>
        <View style={styles.cardRow}>
          <Text style={styles.cardLabel}>Incident Date</Text>
          <Text style={styles.cardValue}>{String(formatDate(item.incidentDate))}</Text>
        </View>
        <View style={styles.cardRow}>
          <Text style={styles.cardLabel}>Country</Text>
          <Text style={styles.cardValue}>{String(item.country || 'N/A')}</Text>
        </View>
        <View style={styles.cardRow}>
          <Text style={styles.cardLabel}>Project</Text>
          <Text style={styles.cardValue}>{String(item.project || 'N/A')}</Text>
        </View>
        <View style={styles.cardRow}>
          <Text style={styles.cardLabel}>Incident Area</Text>
          <Text style={styles.cardValue}>{String(item.incidentArea || 'N/A')}</Text>
        </View>
        <View style={styles.cardRow}>
          <Text style={styles.cardLabel}>Severity</Text>
          <Text style={styles.cardValue}>{String(item.severity || 'N/A')}</Text>
        </View>
        <View style={styles.cardRow}>
          <Text style={styles.cardLabel}>Incident Category</Text>
          <Text style={styles.cardValue}>{String(item.incidentCategory || 'N/A')}</Text>
        </View>
        <View style={styles.cardRow}>
          <Text style={styles.cardLabel}>Types of Event</Text>
          <Text style={styles.cardValue}>{String(getSelectedIncidentsList(item.selectedIncidents))}</Text>
        </View>
        {item.selectedIncidents?.['Other'] && (
          <View style={styles.cardRow}>
            <Text style={styles.cardLabel}>Other Incident Note</Text>
            <Text style={styles.cardValue}>{String(item.otherIncidentNote || 'N/A')}</Text>
          </View>
        )}
        <View style={styles.cardRow}>
          <Text style={styles.cardLabel}>Involves Equipment</Text>
          <Text style={styles.cardValue}>{item.involvesEquipment ? 'Yes' : 'No'}</Text>
        </View>
        {item.involvesEquipment && (
          <>
            <View style={styles.cardRow}>
              <Text style={styles.cardLabel}>Equipment #</Text>
              <Text style={styles.cardValue}>{String(item.vehicleId || 'N/A')}</Text>
            </View>
            <View style={styles.cardRow}>
              <Text style={styles.cardLabel}>Equipment Name</Text>
              <Text style={styles.cardValue}>{String(item.vehicleName || 'N/A')}</Text>
            </View>
          </>
        )}
        <View style={styles.cardRow}>
          <Text style={styles.cardLabel}>Person Involved</Text>
          <Text style={styles.cardValue}>{String(item.personInvolved || 'N/A')}</Text>
        </View>
        <View style={styles.cardRow}>
          <Text style={styles.cardLabel}>Coordinates</Text>
          <Text style={styles.cardValue}>{String(item.coordinates || 'N/A')}</Text>
        </View>
        <View style={styles.cardRow}>
          <Text style={styles.cardLabel}>Created At</Text>
          <Text style={styles.cardValue}>{String(formatDate(item.createdAt))}</Text>
        </View>
      </View>
    );
  };

  const getDropdownData = (report) => {
    if (!report) return [];
    const canPerformAction = canPerformActionOnReport(report);
    
    return [
      ...(hasViewPermission || hasEditPermission ? [{
        label: 'View Report',
        value: 'view',
        icon: 'picture-as-pdf',
        action: () => {
          if (selectedReport) {
            handleViewPDF(selectedReport.pdfDownloadUrl);
          }
        },
      }] : []),
      {
        label: 'Upload to Cloud',
        value: 'upload',
        icon: 'cloud-upload',
        action: () => {
          if (selectedReport) {
            handleUploadToCloud(selectedReport);
          }
        },
      },
      ...(hasDeletePermission && canPerformAction ? [{
        label: 'Delete',
        value: 'delete',
        icon: 'delete',
        action: () => {
          if (selectedReport) {
            setReportToDelete(selectedReport);
            setConfirmDeleteVisible(true);
          }
        },
      }] : []),
    ];
  };

  return (
    <View style={styles.container}>
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity onPress={() => navigation.reset({
              index: 0,
              routes: [
                { name: 'MainApp' },
              ],
            })}>
              <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>Incident Reports</Text>
          </View>
          <View style={styles.headerRight}>
            <TouchableOpacity style={styles.iconButton} onPress={handleRefresh}>
              <Image
                source={require('../../../Images/adminhome/equp/cloud.png')}
                style={styles.refreshIcon}
              />
            </TouchableOpacity>
            <TouchableOpacity
              style={styles.iconButton}
              onPress={() => setPermissionsModalVisible(true)}
            >
              <Image
                source={require('../../../Images/adminhome/inspreport/mailico.png')}
                style={styles.icons12}
              />
            </TouchableOpacity>
          </View>
        </View>
        <View style={styles.searchContainer}>
          <View style={styles.searchWrapper}>
            <TextInput
              style={styles.searchInput}
              placeholder="Search by title or type..."
              placeholderTextColor="#888"
              value={searchQuery}
              onChangeText={handleSearch}
            />
            <MaterialIcons name="search" size={width * 0.05} color="#888" style={styles.bladeIcon} />
          </View>
          <TouchableOpacity
            style={styles.filterButton}
            onPress={() => {
              Toast.show({
                type: 'info',
                text1: 'Filter',
                text2: 'Filter functionality not implemented yet.',
                position: 'top',
              });
            }}
          >
            <MaterialIcons name="filter-list" size={width * 0.06} color="#fff" />
          </TouchableOpacity>
          <TouchableOpacity
            style={styles.archiveButton}
            onPress={() => setShowArchived(!showArchived)}
          >
            <MaterialIcons 
              name={showArchived ? "archive" : "unarchive"} 
              size={width * 0.06} 
              color="#fff" 
            />
          </TouchableOpacity>
        </View>
      </LinearGradient>
      <View style={styles.mainContent}>
        {filteredReports.length === 0 && (!showArchived || filteredArchivedReports.length === 0) ? (
          <>
            <Text style={styles.defaultText}>You don't have any incident reports</Text>
            <View style={styles.questionContainer}>
              <MaterialIcons name="help-outline" size={width * 0.05} color="#333" />
              <Text style={styles.questionText}>How To Create Incident Report?</Text>
            </View>
          </>
        ) : (
          <>
            <FlatList
              data={filteredReports}
              renderItem={renderIncidentReport}
              keyExtractor={(item) => `active-${item.id}`}
              contentContainerStyle={styles.reportList}
              refreshing={isRefreshing}
              onRefresh={handleRefresh}
            />
            {showArchived && filteredArchivedReports.length > 0 && (
              <>
                <Text style={styles.sectionHeader}>Archived Reports</Text>
                <FlatList
                  data={filteredArchivedReports}
                  renderItem={renderArchivedReport}
                  keyExtractor={(item) => `archived-${item.id}`}
                  contentContainerStyle={styles.reportList}
                />
              </>
            )}
          </>
        )}
        <Modal
          visible={dropdownVisible}
          transparent={true}
          animationType="none"
          onRequestClose={() => setDropdownVisible(false)}
        >
          <TouchableOpacity
            style={{ flex: 1, backgroundColor: 'transparent' }}
            onPress={() => setDropdownVisible(false)}
            activeOpacity={1}
          >
            <View
              style={[
                styles.dropdownContainer,
                {
                  position: 'absolute',
                  top: dropdownPosition.y,
                  left: dropdownPosition.x,
                  width: 220,
                  zIndex: 9999,
                },
              ]}
            >
              {selectedReport && getDropdownData(selectedReport).map((option, idx) => (
                <TouchableOpacity
                  key={idx}
                  style={styles.dropdownItem}
                  onPress={() => {
                    option.action();
                    if (option.label !== 'Delete') {
                      setDropdownVisible(false);
                    }
                  }}
                  activeOpacity={0.8}
                >
                  <MaterialIcons
                    name={option.icon}
                    size={22}
                    color={option.value === 'delete' ? '#e74c3c' : '#015185'}
                  />
                  <Text
                    style={[
                      styles.dropdownText,
                      { color: option.value === 'delete' ? '#e74c3c' : '#015185' },
                    ]}
                  >
                    {option.label}
                  </Text>
                </TouchableOpacity>
              ))}
            </View>
          </TouchableOpacity>
        </Modal>
        <Modal
          visible={confirmDeleteVisible}
          transparent={true}
          animationType="fade"
          onRequestClose={() => setConfirmDeleteVisible(false)}
        >
          <View style={styles.modalOverlay}>
            <View style={styles.confirmModalContent}>
              <Text style={styles.confirmModalTitle}>Confirm Deletion</Text>
              <Text style={styles.confirmModalText}>
                Are you sure you want to delete the incident report{' '}
                <Text style={styles.boldText}>
                  {reportToDelete?.incidentNumber || 'N/A'}
                </Text>
                ? This action cannot be undone.
              </Text>
              <View style={styles.modalButtons}>
                <TouchableOpacity
                  style={[styles.modalButton, styles.cancelButton]}
                  onPress={() => {
                    setConfirmDeleteVisible(false);
                    setReportToDelete(null);
                    setDropdownVisible(false);
                  }}
                >
                  <Text style={styles.modalButtonText}>Cancel</Text>
                </TouchableOpacity>
                <TouchableOpacity
                  style={[styles.modalButton, styles.deleteButton]}
                  onPress={handleDeleteReport}
                >
                  <Text style={[styles.modalButtonText, { color: '#fff' }]}>Delete</Text>
                </TouchableOpacity>
              </View>
            </View>
          </View>
        </Modal>
        <Modal
          visible={permissionsModalVisible}
          transparent={true}
          animationType="fade"
          onRequestClose={() => setPermissionsModalVisible(false)}
        >
          <View style={styles.modalOverlay}>
            <View style={styles.permissionsModalContent}>
              <Text style={styles.permissionsModalTitle}>User Permissions</Text>
              <View style={styles.permissionsSection}>
                <Text style={styles.permissionsLabel}>Countries</Text>
                <Text style={styles.permissionsValue}>
                  {userPermissions.countries.length > 0
                    ? userPermissions.countries.join(', ')
                    : 'None'}
                </Text>
              </View>
              <View style={styles.permissionsSection}>
                <Text style={styles.permissionsLabel}>Projects</Text>
                <Text style={styles.permissionsValue}>
                  {userPermissions.projects.length > 0
                    ? userPermissions.projects.join(', ')
                    : 'None'}
                </Text>
              </View>
              <View style={styles.permissionsSection}>
                <Text style={styles.permissionsLabel}>Incident Report Permissions</Text>
                <Text style={styles.permissionsValue}>
                  {userPermissions.incidentReportPermissions.length > 0
                    ? userPermissions.incidentReportPermissions.join(', ')
                    : 'None'}
                </Text>
              </View>
              <View style={styles.permissionsSection}>
                <Text style={styles.permissionsLabel}>Permission Details</Text>
                <Text style={styles.permissionsValue}>
                  Create: {hasCreatePermission ? 'Yes' : 'No'}{'\n'}
                  View: {hasViewPermission ? 'Yes' : 'No'}{'\n'}
                  Edit: {hasEditPermission ? 'Yes' : 'No'}{'\n'}
                  Delete: {hasDeletePermission ? 'Yes' : 'No'}{'\n'}
                  Only Mine View: {hasOnlyMineView ? 'Yes' : 'No'}
                </Text>
              </View>
              <TouchableOpacity
                style={[styles.modalButton, styles.cancelButton]}
                onPress={() => setPermissionsModalVisible(false)}
              >
                <Text style={styles.modalButtonText}>Close</Text>
              </TouchableOpacity>
            </View>
          </View>
        </Modal>
        <Modal visible={syncModalVisible} transparent animationType="fade">
          <View style={styles.syncModalContainer}>
            <View style={styles.syncModalContent}>
              <Text style={styles.syncModalText}>Please Wait</Text>
              <View style={styles.syncIcondiv}>
                <Animated.Image
                  source={require('../../../Images/adminlogin/sync.png')}
                  style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
                />
                <Text style={styles.syncModalTextadd}>{syncMessage}</Text>
              </View>
            </View>
          </View>
        </Modal>
      </View>
      {hasCreatePermission && (
        <View style={styles.buttonContainer}>
          <LinearGradient colors={['#3481BC', '#025383']} style={styles.buttonGradient}>
            <TouchableOpacity
              style={styles.startButton}
              onPress={() => navigation.navigate('CreateIncidentReport')}
            >
              <Ionicons name="add" size={width * 0.06} color="#fff" />
              <Text style={styles.buttonText}>Create Incident Report</Text>
            </TouchableOpacity>
          </LinearGradient>
        </View>
      )}
      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.05,
    fontWeight: 'bold',
    marginLeft: width * 0.025,
  },
  iconButton: {
    padding: width * 0.012,
    marginLeft: width * 0.025,
  },
  icons12: {
    width: width * 0.07,
    height: width * 0.06,
  },
  searchContainer: {
    flexDirection: 'row',
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
    alignItems: 'center',
    width: '100%',
  },
  searchWrapper: {
    flex: 1,
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#fff',
    borderRadius: width * 0.09,
    paddingHorizontal: width * 0.025,
    marginRight: width * 0.025,
  },
  searchInput: {
    flex: 1,
    height: width * 0.1,
    color: '#333',
    fontSize: width * 0.04,
  },
  bladeIcon: {
    marginLeft: width * 0.012,
  },
  iconContainer: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  filterButton: {
    padding: width * 0.015,
    marginRight: width * 0.015,
  },
  archiveButton: {
    padding: width * 0.015,
  },
  mainContent: {
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
    paddingBottom: width * 0.2,
  },
  sectionHeader: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#3481BC',
    marginVertical: width * 0.03,
  },
  sectionHeader11: {
    // Empty style for consistency
  },
  defaultText: {
    fontSize: width * 0.045,
    color: '#888',
    textAlign: 'center',
    marginVertical: width * 0.05,
  },
  reportCard: {
    backgroundColor: '#fff',
    borderRadius: 8,
    marginBottom: 16,
    padding: 0,
    borderWidth: 1,
    borderColor: '#e0e0e0',
    overflow: 'visible',
    elevation: 2,
  },
  cardHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    padding: 12,
    borderBottomWidth: 1,
    borderColor: '#f0f0f0',
  },
  cardTitle: {
    fontSize: 20,
    color: '#3481BC',
    fontWeight: 'bold',
  },
  cloudIconContainer: {
    position: 'relative',
    alignItems: 'center',
    justifyContent: 'center',
  },
  tickContainer: {
    position: 'absolute',
    top: '50%',
    left: '50%',
    transform: [{ translateX: -8 }, { translateY: -8 }],
  },
  menuButton: {
    padding: 4,
  },
  cardRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: 12,
    paddingVertical: 8,
    borderBottomWidth: 1,
    borderColor: '#f0f0f0',
  },
  cardLabel: {
    fontSize: 16,
    color: '#333',
    fontWeight: '500',
    flex: 1,
  },
  cardValue: {
    fontSize: 16,
    color: '#333',
    flex: 1,
    textAlign: 'left',
  },
  dropdownContainer: {
    backgroundColor: '#fff',
    borderRadius: 3,
    paddingVertical: 4,
    paddingHorizontal: 0,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 4,
    elevation: 5,
    padding: 10,
  },
  dropdownItem: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 11,
    paddingHorizontal: 11,
  },
  dropdownText: {
    marginLeft: 12,
    fontSize: 16,
  },
  buttonContainer: {
    position: 'absolute',
    bottom: '10%',
    width: '100%',
    alignItems: 'center',
  },
  buttonGradient: {
    borderRadius: width * 0.012,
    overflow: 'hidden',
  },
  startButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.05,
  },
  buttonText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: 'bold',
    marginLeft: width * 0.012,
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: width * 0.025,
    padding: width * 0.05,
    alignItems: 'center',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.25,
    shadowRadius: 4,
  },
  syncModalText: {
    fontSize: width * 0.045,
    marginBottom: width * 0.025,
    color: '#333',
    fontWeight: '600',
  },
  syncIcon: {
    width: width * 0.06,
    height: width * 0.06,
    marginRight: width * 0.02,
  },
  syncModalTextadd: {
    fontSize: width * 0.035,
    color: '#333',
    fontWeight: '500',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    width: '100%',
    marginBottom: width * 0.025,
  },
  modalOverlay: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: 'rgba(0,0,0,0.5)',
  },
  confirmModalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    width: '80%',
    alignItems: 'center',
  },
  confirmModalTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#333',
    marginBottom: 10,
  },
  confirmModalText: {
    fontSize: 16,
    color: '#333',
    textAlign: 'center',
    marginBottom: 20,
  },
  boldText: {
    fontWeight: 'bold',
  },
  modalButtons: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    width: '100%',
  },
  modalButton: {
    flex: 1,
    paddingVertical: 10,
    alignItems: 'center',
    borderRadius: 5,
    marginHorizontal: 5,
  },
  cancelButton: {
    backgroundColor: '#f0f0f0',
  },
  deleteButton: {
    backgroundColor: '#FF3A3A',
  },
  modalButtonText: {
    fontSize: 16,
    color: '#333',
    fontWeight: '600',
  },
  permissionsModalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    width: '80%',
    alignItems: 'center',
  },
  permissionsModalTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#3481BC',
    marginBottom: 15,
  },
  permissionsSection: {
    width: '100%',
    marginBottom: 15,
  },
  permissionsLabel: {
    fontSize: 16,
    fontWeight: '600',
    color: '#333',
    marginBottom: 5,
  },
  permissionsValue: {
    fontSize: 16,
    color: '#333',
    textAlign: 'left',
  },
  noReportsContainer: {
    alignItems: 'center',
    justifyContent: 'center',
    paddingVertical: 40,
  },
  permissionNote: {
    fontSize: 14,
    color: '#666',
    textAlign: 'center',
    marginTop: 8,
    fontStyle: 'italic',
  },
  refreshButton: {
    padding: width * 0.015,
    justifyContent: 'center',
    alignItems: 'center',
  },
  refreshIcon: {
    width: width * 0.07,
    height: width * 0.07,
    resizeMode: 'contain',
  },
  questionContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    marginTop: width * 0.02,
  },
  questionText: {
    fontSize: width * 0.04,
    color: '#666',
    marginLeft: width * 0.02,
  },
  reportList: {
    paddingBottom: width * 0.02,
  },
});

export default IncidentReport;