import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  Dimensions,
  ScrollView,
  Modal,
  FlatList,
  Animated,
  TextInput,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import database from '@react-native-firebase/database';
import AsyncStorage from '@react-native-async-storage/async-storage';
import Toast from 'react-native-toast-message';
import { Dropdown } from 'react-native-element-dropdown';

const { width, height } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.45;

const SyncModal = ({ visible, message = 'Loading' }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateAnim.interpolate({ inputRange: [0, 1], outputRange: ['0deg', '360deg'] }) }] }]}
            />
            <Text style={styles.syncModalTextadd}>{message}</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

const DROPDOWN_OPTIONS = [
  { label: 'View PDF Report', value: 'view-pdf', icon: <MaterialIcons name="picture-as-pdf" size={22} color="#015185" /> },
  { label: 'Delete Report', value: 'delete', icon: <MaterialIcons name="delete" size={22} color="#e74c3c" /> },
];

const UniInspectionFilterKPI = () => {
  const navigation = useNavigation();
  const [allHSEReports, setAllHSEReports] = useState([]);
  const [filteredReports, setFilteredReports] = useState([]);
  const [loading, setLoading] = useState(true);
  const [searchQuery, setSearchQuery] = useState('');
  
  // Filter states
  const [filterModalVisible, setFilterModalVisible] = useState(false);
  const [activeFilters, setActiveFilters] = useState({
    countries: [],
    projects: [],
    creators: [],
    dateRange: 'all', // all, today, week, month
  });
  const [availableCountries, setAvailableCountries] = useState([]);
  const [availableProjects, setAvailableProjects] = useState([]);
  const [availableCreators, setAvailableCreators] = useState([]);

  useEffect(() => {
    loadAllHSEReports();
  }, []);

  // Extract unique countries, projects, and creators from reports
  useEffect(() => {
    if (allHSEReports.length > 0) {
      const countries = new Set();
      const projects = new Set();
      const creators = new Set();
      
      allHSEReports.forEach(report => {
        const country = report?.step1?.country;
        const project = report?.step1?.project;
        const creatorName = report?.creatorName || 'Unknown';
        const creatorEmail = report?.creatorEmail || '';
        
        if (country && (country === 'DRC' || country === 'Zambia')) {
          countries.add(country);
        }
        if (project) projects.add(project);
        if (creatorName) {
          creators.add(JSON.stringify({ name: creatorName, email: creatorEmail }));
        }
      });
      
      setAvailableCountries(Array.from(countries).sort());
      setAvailableProjects(Array.from(projects).sort());
      setAvailableCreators(
        Array.from(creators)
          .map(c => JSON.parse(c))
          .sort((a, b) => a.name.localeCompare(b.name))
      );
    }
  }, [allHSEReports]);

  // Apply filters
  useEffect(() => {
    let filtered = [...allHSEReports];
    
    // Apply search filter
    if (searchQuery.trim() !== '') {
      filtered = filtered.filter(report => {
        const query = searchQuery.toLowerCase();
        const requestNumber = report.id || report.step1?.requestNumber || '';
        const documentName = report.step1?.selectedChecklist?.document_name || '';
        const documentId = report.step1?.selectedChecklist?.document_id || '';
        const creatorName = report.creatorName || '';
        
        return requestNumber.toLowerCase().includes(query) ||
               documentName.toLowerCase().includes(query) ||
               documentId.toLowerCase().includes(query) ||
               creatorName.toLowerCase().includes(query);
      });
    }
    
    // Apply country filter
    if (activeFilters.countries.length > 0) {
      filtered = filtered.filter(report => {
        const country = report?.step1?.country;
        return activeFilters.countries.includes(country);
      });
    }
    
    // Apply project filter
    if (activeFilters.projects.length > 0) {
      filtered = filtered.filter(report => {
        const project = report?.step1?.project;
        return activeFilters.projects.includes(project);
      });
    }
    
    // Apply creator filter
    if (activeFilters.creators.length > 0) {
      filtered = filtered.filter(report => {
        const creatorName = report?.creatorName || 'Unknown';
        return activeFilters.creators.includes(creatorName);
      });
    }
    
    // Apply date range filter
    if (activeFilters.dateRange !== 'all') {
      const now = new Date();
      const today = new Date(now.getFullYear(), now.getMonth(), now.getDate());
      
      filtered = filtered.filter(report => {
        if (!report.createdAt) return false;
        const reportDate = new Date(report.createdAt);
        
        switch (activeFilters.dateRange) {
          case 'today':
            return reportDate >= today;
          case 'week':
            const weekAgo = new Date(today);
            weekAgo.setDate(weekAgo.getDate() - 7);
            return reportDate >= weekAgo;
          case 'month':
            const monthAgo = new Date(today);
            monthAgo.setMonth(monthAgo.getMonth() - 1);
            return reportDate >= monthAgo;
          default:
            return true;
        }
      });
    }
    
    setFilteredReports(filtered);
  }, [searchQuery, allHSEReports, activeFilters]);

  const loadAllHSEReports = async () => {
    try {
      setLoading(true);
      
      const hseRef = database().ref('/GlobalHSEInspections/admins');
      const hseSnapshot = await hseRef.once('value');
      const hseData = hseSnapshot.val() || {};
      
      const reportsArray = [];
      
      // Get all admin users to map creator info
      const usersSnapshot = await database().ref('/Globalusers/admins').once('value');
      const usersData = usersSnapshot.val() || {};
      
      Object.entries(hseData).forEach(([adminId, reports]) => {
        if (reports && typeof reports === 'object') {
          const adminInfo = usersData[adminId] || {};
          const creatorName = adminInfo.name || 'Unknown';
          const creatorEmail = adminInfo.email || '';
          
          Object.entries(reports).forEach(([reportId, report]) => {
            // Skip metadata fields
            if (reportId === 'Reportidnum') return;
            
            reportsArray.push({
              ...report,
              id: reportId,
              adminId,
              creatorName,
              creatorEmail,
              createdAt: report.createdAt || new Date().toISOString(),
            });
          });
        }
      });
      
      // Sort by creation date (newest first)
      reportsArray.sort((a, b) => {
        const dateA = a.createdAt ? new Date(a.createdAt).getTime() : 0;
        const dateB = b.createdAt ? new Date(b.createdAt).getTime() : 0;
        return dateB - dateA;
      });
      
      setAllHSEReports(reportsArray);
      
    } catch (error) {
      console.error('Error loading HSE reports:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to load HSE reports.',
        position: 'top',
      });
    } finally {
      setLoading(false);
    }
  };

  const toggleCountryFilter = (country) => {
    setActiveFilters(prev => ({
      ...prev,
      countries: prev.countries.includes(country)
        ? prev.countries.filter(c => c !== country)
        : [...prev.countries, country]
    }));
  };

  const toggleProjectFilter = (project) => {
    setActiveFilters(prev => ({
      ...prev,
      projects: prev.projects.includes(project)
        ? prev.projects.filter(p => p !== project)
        : [...prev.projects, project]
    }));
  };

  const toggleCreatorFilter = (creatorName) => {
    setActiveFilters(prev => ({
      ...prev,
      creators: prev.creators.includes(creatorName)
        ? prev.creators.filter(c => c !== creatorName)
        : [...prev.creators, creatorName]
    }));
  };

  const setDateRangeFilter = (range) => {
    setActiveFilters(prev => ({
      ...prev,
      dateRange: range
    }));
  };

  const clearAllFilters = () => {
    setActiveFilters({
      countries: [],
      projects: [],
      creators: [],
      dateRange: 'all',
    });
    setFilterModalVisible(false);
  };

  const applyFilters = () => {
    setFilterModalVisible(false);
  };

  const hasActiveFilters = () => {
    return activeFilters.countries.length > 0 || 
           activeFilters.projects.length > 0 || 
           activeFilters.creators.length > 0 ||
           activeFilters.dateRange !== 'all';
  };

  const handleViewPDF = (report) => {
    if (report.pdfDownloadUrl) {
      navigation.navigate('HSEViewPDF', {
        pdfUrl: report.pdfDownloadUrl,
        onGoBack: () => {
          loadAllHSEReports();
        }
      });
    } else {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'PDF not available for this report.',
        position: 'top',
      });
    }
  };

  const handleDeleteReport = (report) => {
    // Implement delete functionality if needed
    Toast.show({
      type: 'info',
      text1: 'Delete',
      text2: 'Delete functionality coming soon.',
      position: 'top',
    });
  };

  const handleDropdownAction = (action, report) => {
    switch (action) {
      case 'view-pdf':
        handleViewPDF(report);
        break;
      case 'delete':
        handleDeleteReport(report);
        break;
      default:
        break;
    }
  };

  const renderReportCard = ({ item, index }) => {
    const dropdownPosition = index > filteredReports.length - 4 ? 'top' : 'auto';
    
    const requestNumber = item.id || item.step1?.requestNumber || 'N/A';
    const documentName = item.step1?.selectedChecklist?.document_name || 'HSE Report';
    const documentId = item.step1?.selectedChecklist?.document_id || 'HSE';
    const country = item.step1?.country || 'N/A';
    const project = item.step1?.project || 'N/A';
    const creatorName = item.creatorName || 'Unknown';
    const createdAt = item.createdAt 
      ? new Date(item.createdAt).toLocaleString('en-US', { 
          month: 'short', 
          day: '2-digit', 
          year: 'numeric', 
          hour: '2-digit', 
          minute: '2-digit' 
        })
      : 'Unknown Date';
    
    return (
      <View style={styles.reportCard}>
        <View style={styles.cardHeader}>
          <View style={styles.cardLeft}>
            <View style={styles.cardInfo}>
              <Text style={styles.titleText}>{documentName}</Text>
              <Text style={styles.documentIdText}>{documentId}</Text>
            </View>
          </View>
          <View style={styles.dropdownContainer}>
            <Dropdown
              style={styles.dropdown}
              data={DROPDOWN_OPTIONS}
              labelField="label"
              valueField="value"
              value={null}
              onChange={(option) => handleDropdownAction(option.value, item)}
              renderRightIcon={() => <MaterialIcons name="more-vert" size={22} color="#333" />}
              renderItem={(option) => (
                <View style={styles.dropdownItem}>
                  {option.icon}
                  <Text style={[
                    styles.dropdownText,
                    { color: option.value === 'delete' ? '#e74c3c' : '#015185' },
                  ]}>
                    {option.label}
                  </Text>
                </View>
              )}
              placeholder=""
              showsVerticalScrollIndicator={false}
              dropdownPosition={dropdownPosition}
              containerStyle={styles.dropdownContainerStyle}
              selectedTextStyle={{ display: 'none' }}
            />
          </View>
        </View>
        
        {/* Details Rows */}
        <View style={styles.detailsList}>
          <View style={styles.detailRow}>
            <Text style={styles.detailLabel}>Request #</Text>
            <Text style={styles.detailValuePrimary}>{requestNumber}</Text>
          </View>
          <View style={styles.detailRow}>
            <Text style={styles.detailLabel}>Country</Text>
            <Text style={styles.detailValuePrimary}>{country}</Text>
          </View>
          <View style={styles.detailRow}>
            <Text style={styles.detailLabel}>Project</Text>
            <Text style={styles.detailValuePrimary}>{project}</Text>
          </View>
          <View style={styles.detailRow}>
            <Text style={styles.detailLabel}>Created By</Text>
            <Text style={styles.detailValuePrimary}>{creatorName}</Text>
          </View>
          <View style={styles.detailRow}>
            <Text style={styles.detailLabel}>Date</Text>
            <Text style={styles.detailValuePrimary}>{createdAt}</Text>
          </View>
        </View>

        {item.pdfDownloadUrl && (
          <View style={styles.pdfIndicator}>
            <MaterialIcons name="picture-as-pdf" size={16} color="#3481BC" />
            <Text style={styles.pdfText}>PDF Available</Text>
          </View>
        )}
      </View>
    );
  };

  return (
    <View style={styles.container}>
      {/* Header */}
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity onPress={() => navigation.goBack()}>
              <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>HSE Reports</Text>
          </View>
          <View style={styles.headerRight}>
            <TouchableOpacity onPress={loadAllHSEReports} style={styles.refreshButton}>
              <MaterialIcons name="refresh" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
          </View>
        </View>

        <View style={styles.searchContainer}>
          <View style={styles.searchWrapper}>
            <TextInput
              style={styles.searchInput}
              placeholder="Search by request #, document, or creator..."
              placeholderTextColor="#888"
              value={searchQuery}
              onChangeText={setSearchQuery}
            />
            <MaterialIcons name="search" size={width * 0.05} color="#888" style={styles.bladeIcon} />
          </View>
          <TouchableOpacity
            style={[styles.filterButton, hasActiveFilters() && styles.filterButtonActive]}
            onPress={() => setFilterModalVisible(true)}
          >
            <MaterialIcons 
              name="filter-list" 
              size={width * 0.06} 
              color={hasActiveFilters() ? "#fff" : "#333"} 
            />
            {hasActiveFilters() && (
              <View style={styles.filterBadge}>
                <Text style={styles.filterBadgeText}>
                  {(activeFilters.countries.length + activeFilters.projects.length + activeFilters.creators.length + (activeFilters.dateRange !== 'all' ? 1 : 0))}
                </Text>
              </View>
            )}
          </TouchableOpacity>
        </View>
      </LinearGradient>

      {/* Main Content */}
      <ScrollView style={styles.scrollView} showsVerticalScrollIndicator={false}>
        <View style={styles.content}>
          <View style={styles.sectionContainer}>
            <View style={styles.sectionHeader}>
              <View>
                <Text style={styles.sectionTitle}>
                  {searchQuery.length > 0 ? 'Search Results' : hasActiveFilters() ? 'Filtered Reports' : 'All HSE Reports'}
                </Text>
                {(searchQuery.length > 0 || hasActiveFilters()) && (
                  <Text style={styles.resultCount}>
                    {filteredReports.length} of {allHSEReports.length} reports
                  </Text>
                )}
              </View>
              {(searchQuery.length > 0 || hasActiveFilters()) && (
                <TouchableOpacity 
                  onPress={() => {
                    setSearchQuery('');
                    clearAllFilters();
                  }} 
                  style={styles.clearSearchButton}
                >
                  <MaterialIcons name="close" size={16} color="#666" />
                  <Text style={styles.clearSearchText}>Clear All</Text>
                </TouchableOpacity>
              )}
            </View>
            
            {/* Active Filters Display */}
            {hasActiveFilters() && (
              <View style={styles.activeFiltersContainer}>
                <Text style={styles.activeFiltersLabel}>Active Filters:</Text>
                <View style={styles.activeFiltersChips}>
                  {activeFilters.dateRange !== 'all' && (
                    <View style={styles.activeFilterChip}>
                      <Text style={styles.activeFilterChipText}>
                        {activeFilters.dateRange === 'today' ? 'Today' : 
                         activeFilters.dateRange === 'week' ? 'Last 7 Days' : 'Last 30 Days'}
                      </Text>
                    </View>
                  )}
                  {activeFilters.countries.map(country => (
                    <View key={country} style={styles.activeFilterChip}>
                      <Text style={styles.activeFilterChipText}>{country}</Text>
                    </View>
                  ))}
                  {activeFilters.projects.map(project => (
                    <View key={project} style={styles.activeFilterChip}>
                      <Text style={styles.activeFilterChipText}>{project}</Text>
                    </View>
                  ))}
                  {activeFilters.creators.map(creator => (
                    <View key={creator} style={styles.activeFilterChip}>
                      <Text style={styles.activeFilterChipText}>{creator}</Text>
                    </View>
                  ))}
                </View>
              </View>
            )}
            
            {loading ? (
              <View style={styles.loadingContainer}>
                <MaterialIcons name="hourglass-empty" size={48} color="#ccc" />
                <Text style={styles.loadingText}>Loading reports...</Text>
              </View>
            ) : filteredReports.length > 0 ? (
              <FlatList
                data={filteredReports}
                keyExtractor={(item) => item.id}
                renderItem={renderReportCard}
                showsVerticalScrollIndicator={false}
                scrollEnabled={false}
              />
            ) : (
              <View style={styles.emptyContainer}>
                <MaterialIcons name="search" size={64} color="#ccc" />
                <Text style={styles.emptyTitle}>
                  {searchQuery.length > 0 || hasActiveFilters() ? 'No Reports Found' : 'No HSE Reports'}
                </Text>
                <Text style={styles.emptySubtext}>
                  {searchQuery.length > 0 || hasActiveFilters()
                    ? 'Try adjusting your filters'
                    : 'No HSE reports available'
                  }
                </Text>
              </View>
            )}
          </View>
        </View>
      </ScrollView>

      <SyncModal visible={loading} message="Loading HSE Reports..." />
      
      {/* Filter Modal */}
      <Modal
        visible={filterModalVisible}
        animationType="slide"
        transparent={true}
        onRequestClose={() => setFilterModalVisible(false)}
      >
        <View style={styles.filterModalOverlay}>
          <View style={styles.filterModalContainer}>
            {/* Header */}
            <View style={styles.filterModalHeader}>
              <Text style={styles.filterModalTitle}>Filter HSE Reports</Text>
              <TouchableOpacity onPress={() => setFilterModalVisible(false)}>
                <MaterialIcons name="close" size={24} color="#333" />
              </TouchableOpacity>
            </View>

            <ScrollView style={styles.filterModalContent} showsVerticalScrollIndicator={false}>
              {/* Date Range Filter */}
              <View style={styles.filterSection}>
                <Text style={styles.filterSectionTitle}>Date Range</Text>
                <View style={styles.filterChipsContainer}>
                  {[
                    { label: 'All Time', value: 'all' },
                    { label: 'Today', value: 'today' },
                    { label: 'Last 7 Days', value: 'week' },
                    { label: 'Last 30 Days', value: 'month' },
                  ].map(option => (
                    <TouchableOpacity
                      key={option.value}
                      style={[
                        styles.filterChip,
                        activeFilters.dateRange === option.value && styles.filterChipActive
                      ]}
                      onPress={() => setDateRangeFilter(option.value)}
                    >
                      <Text style={[
                        styles.filterChipText,
                        activeFilters.dateRange === option.value && styles.filterChipTextActive
                      ]}>
                        {option.label}
                      </Text>
                    </TouchableOpacity>
                  ))}
                </View>
              </View>

              {/* Country Filter */}
              <View style={styles.filterSection}>
                <Text style={styles.filterSectionTitle}>
                  Countries {activeFilters.countries.length > 0 && `(${activeFilters.countries.length})`}
                </Text>
                {availableCountries.length > 0 ? (
                  <View style={styles.filterChipsContainer}>
                    {availableCountries.map(country => (
                      <TouchableOpacity
                        key={country}
                        style={[
                          styles.filterChip,
                          activeFilters.countries.includes(country) && styles.filterChipActive
                        ]}
                        onPress={() => toggleCountryFilter(country)}
                      >
                        <Text style={[
                          styles.filterChipText,
                          activeFilters.countries.includes(country) && styles.filterChipTextActive
                        ]}>
                          {country}
                        </Text>
                        {activeFilters.countries.includes(country) && (
                          <MaterialIcons name="check" size={16} color="#fff" style={{ marginLeft: 4 }} />
                        )}
                      </TouchableOpacity>
                    ))}
                  </View>
                ) : (
                  <Text style={styles.noFilterOptions}>No countries available</Text>
                )}
              </View>

              {/* Project Filter */}
              <View style={styles.filterSection}>
                <Text style={styles.filterSectionTitle}>
                  Projects {activeFilters.projects.length > 0 && `(${activeFilters.projects.length})`}
                </Text>
                {availableProjects.length > 0 ? (
                  <View style={styles.filterChipsContainer}>
                    {availableProjects.map(project => (
                      <TouchableOpacity
                        key={project}
                        style={[
                          styles.filterChip,
                          activeFilters.projects.includes(project) && styles.filterChipActive
                        ]}
                        onPress={() => toggleProjectFilter(project)}
                      >
                        <Text style={[
                          styles.filterChipText,
                          activeFilters.projects.includes(project) && styles.filterChipTextActive
                        ]}>
                          {project}
                        </Text>
                        {activeFilters.projects.includes(project) && (
                          <MaterialIcons name="check" size={16} color="#fff" style={{ marginLeft: 4 }} />
                        )}
                      </TouchableOpacity>
                    ))}
                  </View>
                ) : (
                  <Text style={styles.noFilterOptions}>No projects available</Text>
                )}
              </View>

              {/* Creator/Person Filter */}
              <View style={styles.filterSection}>
                <Text style={styles.filterSectionTitle}>
                  Created By {activeFilters.creators.length > 0 && `(${activeFilters.creators.length})`}
                </Text>
                {availableCreators.length > 0 ? (
                  <View style={styles.filterChipsContainer}>
                    {availableCreators.map(creator => (
                      <TouchableOpacity
                        key={creator.name}
                        style={[
                          styles.filterChip,
                          activeFilters.creators.includes(creator.name) && styles.filterChipActive
                        ]}
                        onPress={() => toggleCreatorFilter(creator.name)}
                      >
                        <View style={styles.creatorChipContent}>
                          <MaterialIcons name="person" size={16} color={activeFilters.creators.includes(creator.name) ? "#fff" : "#015185"} />
                          <Text style={[
                            styles.filterChipText,
                            activeFilters.creators.includes(creator.name) && styles.filterChipTextActive,
                            { marginLeft: 6 }
                          ]}>
                            {creator.name}
                          </Text>
                        </View>
                        {activeFilters.creators.includes(creator.name) && (
                          <MaterialIcons name="check" size={16} color="#fff" style={{ marginLeft: 4 }} />
                        )}
                      </TouchableOpacity>
                    ))}
                  </View>
                ) : (
                  <Text style={styles.noFilterOptions}>No creators available</Text>
                )}
              </View>
            </ScrollView>

            {/* Footer Buttons */}
            <View style={styles.filterModalFooter}>
              <TouchableOpacity 
                style={styles.filterClearButton}
                onPress={clearAllFilters}
              >
                <Text style={styles.filterClearButtonText}>Clear All</Text>
              </TouchableOpacity>
              <TouchableOpacity 
                style={styles.filterApplyButton}
                onPress={applyFilters}
              >
                <LinearGradient
                  colors={['#3481BC', '#025383']}
                  style={styles.filterApplyGradient}
                >
                  <Text style={styles.filterApplyButtonText}>Apply Filters</Text>
                </LinearGradient>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>
      
      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#f8f9fa',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.045,
    fontWeight: '500',
    marginLeft: width * 0.025,
  },
  refreshButton: {
    padding: width * 0.015,
    justifyContent: 'center',
    alignItems: 'center',
  },
  searchContainer: {
    flexDirection: 'row',
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
    alignItems: 'center',
  },
  searchWrapper: {
    flex: 1,
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#fff',
    borderRadius: width * 0.09,
    paddingHorizontal: width * 0.025,
    marginRight: width * 0.025,
  },
  searchInput: {
    flex: 1,
    height: width * 0.1,
    color: '#333',
    fontSize: width * 0.04,
  },
  bladeIcon: {
    marginLeft: width * 0.012,
  },
  filterButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    backgroundColor: '#fff',
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.03,
    borderRadius: width * 0.02,
    marginHorizontal: width * 0.01,
    borderWidth: 1,
    borderColor: '#ddd',
    position: 'relative',
  },
  filterButtonActive: {
    backgroundColor: '#015185',
    borderColor: '#015185',
  },
  filterBadge: {
    position: 'absolute',
    top: -6,
    right: -6,
    backgroundColor: '#e74c3c',
    borderRadius: 10,
    minWidth: 20,
    height: 20,
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 4,
  },
  filterBadgeText: {
    color: '#fff',
    fontSize: 10,
    fontWeight: 'bold',
  },
  scrollView: {
    flex: 1,
  },
  content: {
    padding: 16,
  },
  sectionContainer: {
    backgroundColor: '#fff',
    borderRadius: 16,
    padding: 16,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
    elevation: 3,
  },
  sectionHeader: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    marginBottom: 20,
  },
  sectionTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#333',
  },
  resultCount: {
    fontSize: 12,
    color: '#666',
    marginTop: 4,
    fontWeight: '500',
  },
  clearSearchButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingHorizontal: 12,
    paddingVertical: 6,
    backgroundColor: '#f0f0f0',
    borderRadius: 16,
  },
  clearSearchText: {
    fontSize: 12,
    color: '#666',
    marginLeft: 4,
    fontWeight: '500',
  },
  activeFiltersContainer: {
    marginBottom: 16,
    paddingTop: 8,
  },
  activeFiltersLabel: {
    fontSize: 12,
    color: '#666',
    fontWeight: '600',
    marginBottom: 8,
  },
  activeFiltersChips: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    gap: 6,
  },
  activeFilterChip: {
    paddingVertical: 4,
    paddingHorizontal: 10,
    borderRadius: 12,
    backgroundColor: '#e3f2fd',
    borderWidth: 1,
    borderColor: '#015185',
  },
  activeFilterChipText: {
    fontSize: 11,
    color: '#015185',
    fontWeight: '600',
  },
  loadingContainer: {
    alignItems: 'center',
    padding: 40,
  },
  loadingText: {
    fontSize: 16,
    color: '#666',
    marginTop: 12,
  },
  emptyContainer: {
    alignItems: 'center',
    padding: 40,
  },
  emptyTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#333',
    marginTop: 16,
    marginBottom: 8,
  },
  emptySubtext: {
    fontSize: 14,
    color: '#666',
    textAlign: 'center',
  },
  reportCard: {
    backgroundColor: '#f8f9fa',
    borderRadius: 12,
    padding: 20,
    marginBottom: 12,
    borderWidth: 1,
    borderColor: '#e9ecef',
  },
  cardHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'flex-start',
  },
  cardLeft: {
    flex: 1,
  },
  cardInfo: {
    flex: 1,
  },
  titleText: {
    fontSize: 16,
    fontWeight: '600',
    color: '#333',
  },
  documentIdText: {
    fontSize: 14,
    color: '#015185',
    fontWeight: '500',
    marginTop: 4,
  },
  dropdownContainer: {
    justifyContent: 'center',
    alignItems: 'center',
  },
  dropdown: {
    width: 40,
    backgroundColor: 'transparent',
    borderWidth: 0,
  },
  dropdownContainerStyle: {
    width: 220,
    borderRadius: 3,
    paddingVertical: 4,
    paddingHorizontal: 0,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 4,
    elevation: 5,
    padding: 10,
    left: '40%',
  },
  dropdownItem: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 11,
    paddingHorizontal: 11,
  },
  dropdownText: {
    marginLeft: 12,
    fontSize: 16,
  },
  detailsList: {
    marginTop: 12,
    backgroundColor: '#fff',
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#e9ecef',
    overflow: 'hidden',
  },
  detailRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingHorizontal: 12,
    paddingVertical: 10,
    borderBottomWidth: 1,
    borderBottomColor: '#f1f1f1',
  },
  detailLabel: {
    color: '#777',
    fontSize: 13,
  },
  detailValuePrimary: {
    color: '#015185',
    fontSize: 13,
    fontWeight: '600',
  },
  pdfIndicator: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#f0f8ff',
    paddingHorizontal: 12,
    paddingVertical: 6,
    borderRadius: 16,
    alignSelf: 'flex-start',
    marginTop: 12,
  },
  pdfText: {
    fontSize: 12,
    color: '#3481BC',
    fontWeight: '500',
    marginLeft: 4,
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 22,
    height: 22,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  
  // Filter Modal Styles
  filterModalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'flex-end',
  },
  filterModalContainer: {
    backgroundColor: '#fff',
    borderTopLeftRadius: 20,
    borderTopRightRadius: 20,
    maxHeight: height * 0.8,
  },
  filterModalHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    padding: 20,
    borderBottomWidth: 1,
    borderBottomColor: '#e9ecef',
  },
  filterModalTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#333',
  },
  filterModalContent: {
    maxHeight: height * 0.55,
    paddingHorizontal: 20,
  },
  filterSection: {
    marginTop: 20,
    marginBottom: 16,
  },
  filterSectionTitle: {
    fontSize: 16,
    fontWeight: '600',
    color: '#333',
    marginBottom: 12,
  },
  filterChipsContainer: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    gap: 8,
  },
  filterChip: {
    paddingVertical: 8,
    paddingHorizontal: 16,
    borderRadius: 20,
    backgroundColor: '#f8f9fa',
    borderWidth: 1,
    borderColor: '#dee2e6',
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: 8,
  },
  filterChipActive: {
    backgroundColor: '#015185',
    borderColor: '#015185',
  },
  filterChipText: {
    fontSize: 14,
    color: '#333',
    fontWeight: '500',
  },
  filterChipTextActive: {
    color: '#fff',
    fontWeight: '600',
  },
  creatorChipContent: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  noFilterOptions: {
    fontSize: 14,
    color: '#999',
    fontStyle: 'italic',
    textAlign: 'center',
    paddingVertical: 12,
  },
  filterModalFooter: {
    flexDirection: 'row',
    padding: 20,
    borderTopWidth: 1,
    borderTopColor: '#e9ecef',
    gap: 12,
  },
  filterClearButton: {
    flex: 1,
    paddingVertical: 14,
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#dee2e6',
    alignItems: 'center',
    justifyContent: 'center',
  },
  filterClearButtonText: {
    fontSize: 16,
    fontWeight: '600',
    color: '#666',
  },
  filterApplyButton: {
    flex: 1,
    borderRadius: 8,
    overflow: 'hidden',
  },
  filterApplyGradient: {
    paddingVertical: 14,
    alignItems: 'center',
    justifyContent: 'center',
  },
  filterApplyButtonText: {
    fontSize: 16,
    fontWeight: '600',
    color: '#fff',
  },
});

export default UniInspectionFilterKPI;

