import React, { useState, useEffect } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  TextInput,
  Platform,
  Alert,
  Linking,
  StatusBar,
  Dimensions,
} from 'react-native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import { useNavigation, useRoute } from '@react-navigation/native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import Modal from 'react-native-modal';
import * as Animatable from 'react-native-animatable';
import AudioRecorderPlayer from 'react-native-audio-recorder-player';
import RNFS from 'react-native-fs';
import axios from 'axios';
import { check, request, PERMISSIONS, RESULTS } from 'react-native-permissions';
import LinearGradient from 'react-native-linear-gradient';

const audioRecorderPlayer = new AudioRecorderPlayer();

const { width } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.24;

const DoinspecNoteadd = () => {
  const [note, setNote] = useState('');
  const [isRecording, setIsRecording] = useState(false);
  const [transcribedText, setTranscribedText] = useState('');
  const [recordingPath, setRecordingPath] = useState('');
  const navigation = useNavigation();
  const route = useRoute();
  const { itemKey, inspectionId } = route.params || {};

  const MAX_RECORDING_DURATION = 30000; // 30 seconds (in milliseconds)
  const MAX_FILE_SIZE = 10 * 1024 * 1024; // 10 MB in bytes

  useEffect(() => {
    loadNote();

    // Cleanup on component unmount
    return () => {
      if (isRecording) {
        stopRecording();
      }
      if (recordingPath) {
        RNFS.unlink(recordingPath).catch((err) =>
          console.error('Error deleting recording file:', err)
        );
      }
    };
  }, [recordingPath]);

  const loadNote = async () => {
    try {
      const savedNotes = JSON.parse(
        await AsyncStorage.getItem(`inspectionNotes_${inspectionId}`) || '{}'
      );
      if (savedNotes[itemKey]) {
        setNote(savedNotes[itemKey] || '');
      }
    } catch (error) {
      console.error('Error loading note:', error);
    }
  };

  const saveNote = async () => {
    try {
      const savedNotes = JSON.parse(
        await AsyncStorage.getItem(`inspectionNotes_${inspectionId}`) || '{}'
      );
      savedNotes[itemKey] = note;
      await AsyncStorage.setItem(
        `inspectionNotes_${inspectionId}`,
        JSON.stringify(savedNotes)
      );
      navigation.goBack();
    } catch (error) {
      console.error('Error saving note:', error);
      Alert.alert('Error', 'Failed to save note.');
    }
  };

  const checkPermissions = async () => {
    const permissionsToCheck = [];

    const microphonePermission =
      Platform.OS === 'ios'
        ? PERMISSIONS.IOS.MICROPHONE
        : PERMISSIONS.ANDROID.RECORD_AUDIO;
    permissionsToCheck.push(microphonePermission);

    if (Platform.OS === 'android') {
      const androidVersion = parseInt(Platform.Version, 10);

      if (androidVersion <= 29) {
        permissionsToCheck.push(PERMISSIONS.ANDROID.WRITE_EXTERNAL_STORAGE);
        permissionsToCheck.push(PERMISSIONS.ANDROID.READ_EXTERNAL_STORAGE);
      } else if (androidVersion >= 33) {
        permissionsToCheck.push(PERMISSIONS.ANDROID.READ_MEDIA_AUDIO);
      } else {
        permissionsToCheck.push(PERMISSIONS.ANDROID.READ_EXTERNAL_STORAGE);
      }
    }

    for (const permission of permissionsToCheck) {
      const result = await check(permission);
      if (result === RESULTS.DENIED) {
        const requestResult = await request(permission);
        if (requestResult !== RESULTS.GRANTED) {
          Alert.alert(
            'Permission Denied',
            `The app requires ${permission} to record and process audio. Please grant the permission in settings.`,
            [
              { text: 'Cancel', style: 'cancel' },
              { text: 'Open Settings', onPress: () => Linking.openSettings() },
            ]
          );
          return false;
        }
      } else if (result !== RESULTS.GRANTED) {
        Alert.alert(
          'Permission Denied',
          `The app requires ${permission} to record and process audio. Please grant the permission in settings.`,
          [
            { text: 'Cancel', style: 'cancel' },
            { text: 'Open Settings', onPress: () => Linking.openSettings() },
          ]
        );
        return false;
      }
    }
    return true;
  };

  const startRecording = async () => {
    const hasPermissions = await checkPermissions();
    if (!hasPermissions) {
      return;
    }

    try {
      setIsRecording(true);
      setTranscribedText('');

      const path = `${RNFS.DocumentDirectoryPath}/recording.${Platform.OS === 'ios' ? 'm4a' : 'wav'}`;
      setRecordingPath(path);

      const result = await audioRecorderPlayer.startRecorder(path);
      console.log('Recording started:', result);

      setTimeout(() => {
        if (isRecording) {
          stopRecording();
          Alert.alert('Recording Stopped', 'Maximum recording duration (30 seconds) reached.');
        }
      }, MAX_RECORDING_DURATION);
    } catch (error) {
      console.error('Error starting recording:', error);
      setIsRecording(false);
      Alert.alert('Error', 'Failed to start recording. Please try again.');
    }
  };

  const stopRecording = async () => {
    try {
      const filePath = await audioRecorderPlayer.stopRecorder();
      console.log('Recording stopped, file path:', filePath);
      setIsRecording(false);
      if (filePath) {
        transcribeAudio(filePath);
      } else {
        setTranscribedText('No audio file recorded.');
        Alert.alert('Error', 'No audio file was recorded. Please try again.');
      }
    } catch (error) {
      console.error('Error stopping recording:', error);
      setIsRecording(false);
      Alert.alert('Error', 'Failed to stop recording. Please try again.');
    }
  };

  const transcribeAudio = async (filePath) => {
    try {
      const fileExists = await RNFS.exists(filePath);
      if (!fileExists) {
        throw new Error('Audio file not found.');
      }

      const fileStats = await RNFS.stat(filePath);
      if (fileStats.size > MAX_FILE_SIZE) {
        throw new Error('Audio file exceeds 10 MB limit.');
      }

      const audioBase64 = await RNFS.readFile(filePath, 'base64');

      const payload = {
        config: {
          encoding: Platform.OS === 'ios' ? 'LINEAR16' : 'LINEAR16',
          sampleRateHertz: 16000,
          languageCode: 'en-US',
        },
        audio: {
          content: audioBase64,
        },
      };

      const response = await axios.post(
        `https://speech.googleapis.com/v1/speech:recognize?key=AIzaSyBUk_ddoTBctL3gcniqiEer0rRVEzTjVx0`,
        payload,
        { timeout: 30000 }
      );

      const transcription =
        response.data.results?.[0]?.alternatives?.[0]?.transcript || '';
      console.log('Transcription:', transcription);

      if (transcription) {
        setTranscribedText(transcription);
        setNote((prevNote) => prevNote + (prevNote ? ' ' : '') + transcription);
      } else {
        setTranscribedText('No transcription available.');
        Alert.alert('Info', 'No speech was detected in the audio.');
      }
    } catch (error) {
      console.error('Error transcribing audio:', error.response?.data || error.message);
      setTranscribedText('Failed to transcribe audio.');
      Alert.alert('Error', 'Failed to transcribe audio. Please check your network and try again.');
    }
  };

  return (
    <View style={styles.container}>
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity
              style={styles.backButton}
              onPress={() => navigation.goBack()}
              activeOpacity={0.8}
            >
              <MaterialIcons name="arrow-back" size={24} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>Add Note</Text>
          </View>
          <View style={styles.headerRight} />
        </View>
      </LinearGradient>
      <LinearGradient colors={['#3481BC', '#025383']} style={styles.noteContainer}>
        <Text style={styles.noteLabel}>Note</Text>
        <View style={styles.inputContainer}>
          <TextInput
            style={styles.noteInput}
            value={note}
            onChangeText={setNote}
            placeholder="Enter your note here"
            placeholderTextColor="#e0e0e0"
            multiline
            numberOfLines={5}
          />
          <TouchableOpacity
            style={styles.micButton}
            onPress={isRecording ? stopRecording : startRecording}
          >
            <MaterialIcons
              name={isRecording ? 'mic-off' : 'mic'}
              size={24}
              color={isRecording ? '#FF4B2B' : '#035484'}
            />
          </TouchableOpacity>
        </View>
      </LinearGradient>
      <View style={styles.buttonContainer}>
        <TouchableOpacity style={styles.cancelButton} onPress={() => navigation.goBack()}>
          <Text style={styles.buttonText}>CANCEL</Text>
        </TouchableOpacity>
        <TouchableOpacity style={styles.doneButton} onPress={saveNote}>
          <Text style={styles.doneButtonText}>DONE</Text>
        </TouchableOpacity>
      </View>

      <Modal isVisible={isRecording} backdropOpacity={0.5}>
        <View style={styles.modalContainer}>
          <Animatable.View
            animation="pulse"
            iterationCount="infinite"
            style={styles.micIconContainer}
          >
            <MaterialIcons name="mic" size={50} color="#FF4B2B" />
          </Animatable.View>
          <Text style={styles.recordingText}>Recording...</Text>
          <Text style={styles.transcribingText}>Speak now (max 30 seconds)...</Text>
          <TouchableOpacity style={styles.stopButton} onPress={stopRecording}>
            <Text style={styles.stopButtonText}>Stop Recording</Text>
          </TouchableOpacity>
        </View>
      </Modal>

      <Modal isVisible={!!transcribedText && !isRecording} backdropOpacity={0.5}>
        <View style={styles.modalContainer}>
          <Text style={styles.recordingText}>Transcription Result</Text>
          <Text style={styles.transcribedText}>{transcribedText}</Text>
          <TouchableOpacity
            style={styles.stopButton}
            onPress={() => setTranscribedText('')}
          >
            <Text style={styles.stopButtonText}>Close</Text>
          </TouchableOpacity>
        </View>
      </Modal>
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#f0f0f0',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: Platform.OS === 'ios' ? width * 0.08 : width * 0.04,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
    width: width * 0.07,
    height: width * 0.07,
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.05,
    fontWeight: 'bold',
    marginLeft: width * 0.025,
  },
  noteContainer: {
    width: '94%',
    alignSelf: 'center',
    paddingHorizontal: 15,
    paddingVertical: 10,
    borderRadius: 12,
    marginTop: 20,
    marginBottom: 12,
    backgroundColor: '#3481BC',
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
  },
  noteLabel: {
    fontSize: 16,
    color: '#fff',
    marginBottom: 5,
  },
  inputContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    position: 'relative',
  },
  noteInput: {
    flex: 1,
    backgroundColor: '#fff',
    borderRadius: 8,
    padding: 12,
    paddingRight: 40,
    fontSize: 16,
    minHeight: 100,
    textAlignVertical: 'top',
    color: '#222',
    borderWidth: 1,
    borderColor: '#e0e0e0',
  },
  micButton: {
    position: 'absolute',
    right: 10,
    bottom: 10,
    padding: 5,
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'center',
    paddingVertical: 10,
  },
  cancelButton: {
    paddingVertical: 10,
    paddingHorizontal: 28,
    marginHorizontal: 10,
    backgroundColor: '#fff',
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#035484',
    alignItems: 'center',
    justifyContent: 'center',
    elevation: 2,
  },
  doneButton: {
    paddingVertical: 10,
    paddingHorizontal: 28,
    marginHorizontal: 10,
    backgroundColor: '#035484',
    borderRadius: 8,
    alignItems: 'center',
    justifyContent: 'center',
    elevation: 2,
  },
  buttonText: {
    fontSize: 16,
    color: '#035484',
    fontWeight: 'bold',
  },
  doneButtonText: {
    fontSize: 16,
    color: '#fff',
    fontWeight: 'bold',
  },
  modalContainer: {
    backgroundColor: '#fff',
    padding: 20,
    borderRadius: 12,
    alignItems: 'center',
    justifyContent: 'center',
  },
  micIconContainer: {
    marginBottom: 20,
  },
  recordingText: {
    fontSize: 18,
    fontWeight: '600',
    color: '#333',
    marginBottom: 10,
  },
  transcribingText: {
    fontSize: 16,
    color: '#666',
    marginBottom: 20,
    fontStyle: 'italic',
  },
  transcribedText: {
    fontSize: 16,
    color: '#333',
    marginBottom: 20,
    textAlign: 'center',
  },
  stopButton: {
    backgroundColor: '#FF4B2B',
    paddingVertical: 10,
    paddingHorizontal: 20,
    borderRadius: 8,
  },
  stopButtonText: {
    fontSize: 16,
    color: '#fff',
    fontWeight: '600',
  },
  backButton: {
    padding: 5,
  },
});

export default DoinspecNoteadd;