import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  TextInput,
  Image,
  Dimensions,
  FlatList,
  Share,
  Modal,
  Alert,
  Animated,
  ActivityIndicator,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import Ionicons from 'react-native-vector-icons/Ionicons';
import { useNavigation, useFocusEffect, useIsFocused, useRoute } from '@react-navigation/native';
import database from '@react-native-firebase/database';
import NetInfo from '@react-native-community/netinfo';
import Toast from 'react-native-toast-message';
import AsyncStorage from '@react-native-async-storage/async-storage';
import { Dropdown } from 'react-native-element-dropdown';
import { getEmailStatistics } from './ModulesNotificationEmail';

// Error Boundary Component
class ErrorBoundary extends React.Component {
  state = { hasError: false };

  static getDerivedStateFromError(error) {
    return { hasError: true };
  }

  render() {
    if (this.state.hasError) {
      return <Text style={styles.defaultText}>Something went wrong. Please restart the app.</Text>;
    }
    return this.props.children;
  }
}

const { width, height } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.45;
const DROPDOWN_ITEM_HEIGHT = width * 0.13;

const DROPDOWN_OPTIONS = [
  { label: 'View Report', value: 'view', icon: <MaterialIcons name="picture-as-pdf" size={22} color="#2196f3" /> },
  { label: 'Share', value: 'share', icon: <MaterialIcons name="share" size={22} color="#2196f3" /> },
  { label: 'Create Job Card', value: 'create-work-order', icon: <MaterialIcons name="build" size={22} color="#607D8B" /> },
  { label: 'Corrective Action', value: 'corrective', icon: <MaterialIcons name="radio-button-checked" size={22} color="#4CAF50" /> },
  { label: 'Archive Inspection', value: 'archive', icon: <MaterialIcons name="archive" size={22} color="#2196f3" /> },
  { label: 'Risk Assessment', value: 'risk-assessment', icon: <MaterialIcons name="warning" size={22} color="#FF9800" /> },
  { label: 'Delete', value: 'delete', icon: <MaterialIcons name="delete" size={22} color="#e74c3c" /> },
];

const SyncModal = ({ visible, onClose, message = 'Loading Reports' }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
            />
            <Text style={styles.syncModalTextadd}>{message}</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

const UserInfoModal = ({ visible, onClose, userInfo }) => {
  return (
    <Modal visible={visible} transparent animationType="slide" onRequestClose={onClose}>
      <View style={styles.modalOverlay}>
        <View style={styles.modalContent}>
          <Text style={styles.modalTitle}>User Information</Text>
          <View style={styles.userInfoItem}>
            <Text style={styles.userInfoLabel}>Countries:</Text>
            <Text style={styles.userInfoValue}>{userInfo.countries.length > 0 ? userInfo.countries.join(', ') : 'N/A'}</Text>
          </View>
          <View style={styles.userInfoItem}>
            <Text style={styles.userInfoLabel}>Projects:</Text>
            <Text style={styles.userInfoValue}>{userInfo.projects.length > 0 ? userInfo.projects.join(', ') : 'N/A'}</Text>
          </View>
          <View style={styles.userInfoItem}>
            <Text style={styles.userInfoLabel}>Inspection Permissions:</Text>
            <Text style={styles.userInfoValue}>{userInfo.inspectionPermissions.length > 0 ? userInfo.inspectionPermissions.join(', ') : 'None'}</Text>
          </View>
          <View style={styles.userInfoItem}>
            <Text style={styles.userInfoLabel}>Employee Number:</Text>
            <Text style={styles.userInfoValue}>{userInfo.employeeNumber || 'N/A'}</Text>
          </View>
          <View style={styles.userInfoItem}>
            <Text style={styles.userInfoLabel}>Name:</Text>
            <Text style={styles.userInfoValue}>{userInfo.name || 'N/A'}</Text>
          </View>
          <TouchableOpacity style={styles.modalCloseButton} onPress={onClose}>
            <Text style={styles.modalCloseText}>Close</Text>
          </TouchableOpacity>
        </View>
      </View>
    </Modal>
  );
};

const Inspection_Report_Home = () => {
  const navigation = useNavigation();
  const route = useRoute();
  const isFocused = useIsFocused();
  const [searchQuery, setSearchQuery] = useState('');
  const [selectedFilter, setSelectedFilter] = useState('');
  const [filterModalVisible, setFilterModalVisible] = useState(false);
  const [allReports, setAllReports] = useState([]);
  const [reports, setReports] = useState([]);
  const [selectedReport, setSelectedReport] = useState(null);
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [isRefreshing, setIsRefreshing] = useState(false);
  const [isOnline, setIsOnline] = useState(true);
  const [userInfoModalVisible, setUserInfoModalVisible] = useState(false);
  const [isProcessingAction, setIsProcessingAction] = useState(false);
  const [userInfo, setUserInfo] = useState({
    countries: [],
    projects: [],
    inspectionPermissions: [],
    employeeNumber: '',
    name: '',
  });
  const flatListRef = useRef(null);
  const isMounted = useRef(true);
  const [loadingUserInfo, setLoadingUserInfo] = useState(true);
  const PAGE_SIZE = 6;
  const [page, setPage] = useState(1);

  // Handle notification data from deep linking
  useEffect(() => {
    if (route.params?.notificationData) {
      const notificationData = route.params.notificationData;
      console.log('🔍 Inspection report notification data received:', notificationData);
      
      // Show a toast message to indicate the notification was received
      Toast.show({
        type: 'info',
        text1: '🔍 Inspection Report Notification',
        text2: `Opening inspection report: ${notificationData.requestNumber || 'Unknown'}`,
        position: 'top',
        visibilityTime: 3000,
      });
      
      // You can add additional logic here to highlight or filter to the specific report
      // For now, we'll just show the notification and let the user browse normally
    }
  }, [route.params?.notificationData]);
  const [hasMore, setHasMore] = useState(true);
  const [loadingMore, setLoadingMore] = useState(false);
  const [emailStats, setEmailStats] = useState(null);

  const filterOptions = ['Location', 'Inspector', 'Equipment', 'Project'];

  const checkNetworkStatus = async () => {
    try {
      const netInfo = await NetInfo.fetch();
      return netInfo.isConnected;
    } catch (error) {
      console.error('Error checking network status:', error);
      return false;
    }
  };

  const loadEmailStatistics = async () => {
    try {
      const stats = await getEmailStatistics();
      if (isMounted.current) {
        setEmailStats(stats);
      }
    } catch (error) {
      console.error('Error loading email statistics:', error);
    }
  };

  const showToast = (type, text1, text2) => {
    if (!isMounted.current) return;
    Toast.show({
      type,
      text1,
      text2,
      position: 'bottom',
      visibilityTime: 3000,
      autoHide: true,
    });
  };

  const queueOperation = async (operation) => {
    try {
      const pendingOperationsRaw = await AsyncStorage.getItem('pendingOperations');
      const pendingOperations = pendingOperationsRaw ? JSON.parse(pendingOperationsRaw) : [];
      pendingOperations.push(operation);
      await AsyncStorage.setItem('pendingOperations', JSON.stringify(pendingOperations));
      showToast('info', 'Offline Mode', 'Operation queued for sync when online.');
    } catch (error) {
      console.error('Error queuing operation:', error);
      showToast('error', 'Error', 'Failed to queue operation.');
    }
  };

  const fetchUserInfo = async () => {
    try {
      const userId = await AsyncStorage.getItem('userUid');
      if (!userId) {
        showToast('error', 'Authentication Error', 'No user is logged in.');
        navigation.navigate('AdminLoginScreen');
        return;
      }
      const userRef = database().ref(`/Globalusers/admins/${userId}`);
      const userSnapshot = await userRef.once('value');
      const userData = userSnapshot.val();
      if (!userData) {
        showToast('error', 'Authentication Error', 'User not found in admin database.');
        navigation.navigate('AdminLoginScreen');
        return;
      }
      const inspectionModule = userData.modules?.find((module) => module.module === 'inspectionReport');
      if (isMounted.current) {
        setUserInfo({
          countries: userData.countries || [],
          projects: userData.projects || [],
          inspectionPermissions: inspectionModule ? inspectionModule.permissions || [] : [],
          employeeNumber: userData.employeeNumber || '',
          name: userData.name || '',
        });
      }
    } catch (error) {
      console.error('Error fetching user info:', error);
      showToast('error', 'Error', 'Failed to load user information.');
    } finally {
      if (isMounted.current) {
        setLoadingUserInfo(false);
      }
    }
  };

  const syncPendingOperations = async () => {
    if (!(await checkNetworkStatus())) return;

    try {
      const userId = await AsyncStorage.getItem('userUid');
      if (!userId) {
        showToast('error', 'Authentication Error', 'User not authenticated.');
        return;
      }

      const userRef = database().ref(`/Globalusers/admins/${userId}`);
      const userSnapshot = await userRef.once('value');
      const userData = userSnapshot.val();
      if (!userData) {
        showToast('error', 'Authentication Error', 'User not found in admin database.');
        return;
      }

      const pendingOperationsRaw = await AsyncStorage.getItem('pendingOperations');
      if (!pendingOperationsRaw) return;

      const pendingOperations = JSON.parse(pendingOperationsRaw);
      if (pendingOperations.length === 0) return;

      let updatedReports = [];
      for (const operation of pendingOperations) {
        try {
          if (operation.type === 'delete') {
            const reportRef = database().ref(`/GlobalInspectionReport/admins/${operation.userId}/${operation.reportId}`);
            const correctiveRef = database().ref(`/GlobalCorrectiveActions/admins/${operation.userId}/${operation.reportId}`);
            await Promise.all([reportRef.remove(), correctiveRef.remove()]);
            await AsyncStorage.removeItem(`report_${operation.reportId}`);
          } else if (operation.type === 'create') {
            const reportRef = database().ref(`/GlobalInspectionReport/admins/${operation.userId}/${operation.report.id}`);
            await reportRef.set(operation.report);
            await AsyncStorage.setItem(`report_${operation.report.id}`, JSON.stringify(operation.report));
            updatedReports.push(operation.report);
          }
        } catch (error) {
          console.error(`Error syncing operation ${operation.type}:`, error);
          continue;
        }
      }

      if (isMounted.current) {
        setAllReports((prev) => {
          const allReports = [...prev, ...updatedReports].filter((report, index, self) =>
            index === self.findIndex((r) => r.id === report.id)
          );
          return allReports;
        });
      }
      await AsyncStorage.setItem('pendingOperations', JSON.stringify([]));
      showToast('success', 'Sync Complete', 'Offline operations synced successfully.');
    } catch (error) {
      console.error('Error syncing pending operations:', error);
      showToast('error', 'Sync Error', 'Failed to sync operations. Will retry when online.');
    }
  };

  const loadReports = async () => {
    setSyncModalVisible(true);
    setIsRefreshing(true);
    try {
      const reportsRef = database().ref(`/GlobalInspectionReport/admins`);
      const reportsSnapshot = await reportsRef.once('value');
      const adminsData = reportsSnapshot.val();

      let reportsArray = [];
      if (adminsData) {
        Object.entries(adminsData).forEach(([adminId, reportsObj]) => {
          if (reportsObj) {
            Object.entries(reportsObj).forEach(([reportId, report]) => {
              reportsArray.push({
                ...report,
                id: reportId,
                adminId,
                createdAt: report.step3?.createdAt || new Date().toISOString(),
              });
            });
          }
        });
      }

      const workOrdersRef = database().ref(`/GlobalWorkOrders/admins`);
      const workOrdersSnapshot = await workOrdersRef.once('value');
      const workOrdersAdminsData = workOrdersSnapshot.val() || {};

      let allWorkOrders = [];
      Object.entries(workOrdersAdminsData).forEach(([adminId, adminWorkOrders]) => {
        if (adminWorkOrders) {
          Object.entries(adminWorkOrders).forEach(([woId, wo]) => {
            if (woId !== 'lastRequestNumber') {
              allWorkOrders.push({ ...wo, id: woId, adminId: adminId });
            }
          });
        }
      });

      const correctiveRef = database().ref(`/GlobalCorrectiveActions/admins`);
      const correctiveSnapshot = await correctiveRef.once('value');
      const correctiveAdminsData = correctiveSnapshot.val() || {};
      const correctiveByReportId = {};
      Object.values(correctiveAdminsData).forEach(adminCorrectives => {
        if (adminCorrectives) {
          Object.entries(adminCorrectives).forEach(([reportId, corrective]) => {
            if (corrective?.pdfDownloadUrl) {
              correctiveByReportId[reportId] = corrective;
            }
          });
        }
      });

      // Map risk assessments to reports by id for quick linking
      let riskAssessmentsByReportId = {};
      try {
        const riskRef = database().ref(`/GlobalRiskAssessments/admins`);
        const riskSnap = await riskRef.once('value');
        const riskAdmins = riskSnap.val() || {};
        Object.entries(riskAdmins).forEach(([adminId, raObj]) => {
          if (raObj) {
            Object.entries(raObj).forEach(([reportId, ra]) => {
              if (reportId !== 'lastRequestNumber') {
                riskAssessmentsByReportId[`${adminId}:${reportId}`] = { ...ra, id: reportId, adminId };
              }
            });
          }
        });
      } catch (e) {
        // ignore if risk assessments not accessible
      }

      reportsArray = reportsArray.map(report => {
        const workOrder = allWorkOrders.find(wo => wo.linkInspection && wo.linkInspection.id === report.id);
        const correctiveAction = correctiveByReportId[report.id];
        const riskAssessment = riskAssessmentsByReportId[`${report.adminId}:${report.id}`];
        return {
          ...report,
          ...(workOrder ? { workOrder } : {}),
          ...(correctiveAction ? { correctiveAction } : {}),
          ...(riskAssessment ? { riskAssessment } : {}),
        };
      });

      setAllReports(reportsArray);

      if (!isOnline && reportsArray.length > 0) {
        showToast('info', 'Offline Mode', 'Reports loaded from local storage.');
      } else if (reportsArray.length === 0 && !isOnline) {
        showToast('info', 'Offline Mode', 'No reports available offline.');
      }
    } catch (error) {
      console.error('Error loading reports:', error);
      let errorMessage = 'Failed to load reports.';
      if (!isOnline) {
        errorMessage = 'Offline mode. Reports loaded from local storage if available.';
      }
      showToast('error', 'Error', errorMessage);
    } finally {
      if (isMounted.current) {
        setSyncModalVisible(false);
        setIsRefreshing(false);
      }
    }
  };

  useEffect(() => {
    const unsubscribe = NetInfo.addEventListener((state) => {
      setIsOnline(state.isConnected);
      if (state.isConnected && isMounted.current) {
        syncPendingOperations();
        fetchUserInfo();
        loadReports();
      }
    });

    const loadInitialData = async () => {
      await new Promise((resolve) => setTimeout(resolve, 1000));
      checkNetworkStatus().then((connected) => {
        setIsOnline(connected);
        if (connected) {
          syncPendingOperations();
          fetchUserInfo();
          loadReports();
          loadEmailStatistics();
        } else {
          fetchUserInfo();
          loadReports();
          loadEmailStatistics();
        }
      });
    };
    loadInitialData();

    return () => {
      isMounted.current = false;
      unsubscribe();
    };
  }, []);

  useEffect(() => {
    if (isFocused) {
      setSyncModalVisible(true);
      setTimeout(async () => {
        await fetchUserInfo();
        await loadReports();
        setSyncModalVisible(false);
      }, 1000);
    }
  }, [isFocused]);

  useEffect(() => {
    if (!userInfo || !userInfo.countries || !userInfo.projects) return;

    const filterReports = async () => {
      let userUid = await AsyncStorage.getItem('userUid');
      let filtered = allReports.filter(
        (report) =>
          userInfo.countries.includes(report.step1?.country) &&
          userInfo.projects.includes(report.step1?.project)
      );

      if (userInfo.inspectionPermissions.includes('onlyMineView')) {
        filtered = filtered.filter((report) => report.adminId === userUid);
      }

      if (searchQuery.trim() !== '') {
        filtered = filtered.filter((report) => {
          if (selectedFilter === 'Location') {
            return report.step1?.country?.toLowerCase().includes(searchQuery.toLowerCase());
          } else if (selectedFilter === 'Inspector') {
            return report.step1?.inspector?.toLowerCase().includes(searchQuery.toLowerCase());
          } else if (selectedFilter === 'Equipment') {
            return report.step1?.selectedEquipment?.equipmentName?.toLowerCase().includes(searchQuery.toLowerCase());
          } else if (selectedFilter === 'Project') {
            return report.step1?.project?.toLowerCase().includes(searchQuery.toLowerCase());
          }
          return (
            report.step1?.country?.toLowerCase().includes(searchQuery.toLowerCase()) ||
            report.step1?.inspector?.toLowerCase().includes(searchQuery.toLowerCase()) ||
            report.step1?.selectedEquipment?.equipmentName?.toLowerCase().includes(searchQuery.toLowerCase()) ||
            report.step1?.project?.toLowerCase().includes(searchQuery.toLowerCase())
          );
        });
      }

      const sorted = filtered.sort((a, b) => new Date(b.createdAt) - new Date(a.createdAt));
      const paginated = sorted.slice(0, page * PAGE_SIZE);
      setReports(paginated);
      setHasMore(sorted.length > paginated.length);
    };
    filterReports();
  }, [allReports, userInfo, searchQuery, selectedFilter, page]);

  const handleLoadMore = () => {
    if (!loadingMore && hasMore) {
      setLoadingMore(true);
      setTimeout(() => {
        setPage((prev) => prev + 1);
        setLoadingMore(false);
      }, 300);
    }
  };

  const handleRefresh = () => {
    setIsRefreshing(true);
    setSyncModalVisible(true);
    setPage(1);
    loadReports();
    setSearchQuery('');
    setSelectedFilter('');
  };

  const handleSearch = (text) => {
    setSearchQuery(text);
    setPage(1);
  };

  const handleViewReport = async (report) => {
    if (!report) {
      showToast('error', 'Error', 'No report selected.');
      return;
    }

    if (!userInfo.inspectionPermissions.includes('view') && !userInfo.inspectionPermissions.includes('onlyMineView')) {
      showToast('error', 'Permission Denied', 'You do not have permission to view reports.');
      return;
    }

    try {
      const adminId = report.adminId;
      if (!adminId) {
        showToast('error', 'Error', 'Report admin ID not found.');
        return;
      }
      let pdfUrl = null;
      if (isOnline) {
        const reportRef = database().ref(`/GlobalInspectionReport/admins/${adminId}/${report.id}`);
        const snapshot = await reportRef.once('value');
        const reportData = snapshot.val();
        pdfUrl = reportData?.step3?.pdfDownloadUrl;
      } else {
        const reportRaw = await AsyncStorage.getItem(`report_${report.id}`);
        if (reportRaw) {
          const report = JSON.parse(reportRaw);
          pdfUrl = report.step3?.pdfPath || report.step3?.pdfDownloadUrl;
        }
      }

      if (!pdfUrl) {
        showToast('error', 'Error', isOnline ? 'PDF URL not available.' : 'PDF not available offline.');
        return;
      }

      if (!isOnline) {
        showToast('info', 'Offline Mode', 'Viewing locally stored report.');
      }

      navigation.navigate('PdfviewPage', {
        pdfUrl,
        onGoBack: () => navigation.navigate('Inspection_Report_Home'),
      });
    } catch (error) {
      console.error('Error fetching PDF URL:', error);
      let errorMessage = 'Could not fetch the PDF URL. Please try again.';
      if (!isOnline) {
        errorMessage = 'Offline mode. Please connect to the internet to view the PDF.';
      }
      showToast('error', 'Error', errorMessage);
    }
  };

  const handleViewCorrectivePDF = async (report) => {
    if (!report) {
      showToast('error', 'Error', 'No report selected.');
      return;
    }

    const pdfUrl = report.correctiveAction?.pdfDownloadUrl;
    if (!pdfUrl) {
      showToast('error', 'Error', 'No Corrective Action PDF URL found for this report.');
      return;
    }

    try {
      if (!isOnline) {
        showToast('info', 'Offline Mode', 'Viewing locally stored corrective action.');
      }
      navigation.navigate('PdfviewPage', {
        pdfUrl,
        onGoBack: () => navigation.navigate('Inspection_Report_Home'),
      });
    } catch (error) {
      console.error('Error fetching corrective PDF URL:', error);
      let errorMessage = 'Could not fetch the corrective PDF URL. Please try again.';
      if (!isOnline) {
        errorMessage = 'Offline mode. Please connect to the internet to view the PDF.';
      }
      showToast('error', 'Error', errorMessage);
    }
  };

  const handleViewRiskAssessmentPDF = async (report) => {
    if (!report) {
      showToast('error', 'Error', 'No report selected.');
      return;
    }
    try {
      const adminId = report.adminId;
      if (!adminId) {
        showToast('error', 'Error', 'Report admin ID not found.');
        return;
      }
      let pdfUrl = null;
      if (isOnline) {
        const raRef = database().ref(`/GlobalRiskAssessments/admins/${adminId}/${report.id}`);
        const snap = await raRef.once('value');
        const ra = snap.val();
        pdfUrl = ra?.step3?.pdfDownloadUrl || report.riskAssessment?.step3?.pdfDownloadUrl;
      } else {
        // No offline cache yet for risk assessment PDFs
        pdfUrl = report.riskAssessment?.step3?.pdfDownloadUrl || null;
      }
      if (!pdfUrl) {
        showToast('error', 'Error', isOnline ? 'Risk Assessment PDF not available.' : 'Risk Assessment PDF not available offline.');
        return;
      }
      navigation.navigate('PdfviewPage', {
        pdfUrl,
        onGoBack: () => navigation.navigate('Inspection_Report_Home'),
      });
    } catch (error) {
      console.error('Error fetching Risk Assessment PDF URL:', error);
      let errorMessage = 'Could not fetch the Risk Assessment PDF URL. Please try again.';
      if (!isOnline) {
        errorMessage = 'Offline mode. Please connect to the internet to view the Risk Assessment PDF.';
      }
      showToast('error', 'Error', errorMessage);
    }
  };

  const handleViewWorkOrderPDF = async (report) => {
    if (!report?.workOrder?.pdfDownloadUrl) {
      showToast('error', 'Error', 'No Work Order PDF URL found for this report.');
      return;
    }

    try {
      const userId = await AsyncStorage.getItem('userUid');
      if (!userId) {
        showToast('error', 'Authentication Error', 'No user is logged in.');
        navigation.navigate('AdminLoginScreen');
        return;
      }
      let pdfUrl = null;
      if (isOnline) {
        const workOrderRef = database().ref(`/GlobalWorkOrders/admins/${report.workOrder.adminId}/${report.workOrder.id}`);
        const snapshot = await workOrderRef.once('value');
        const workOrderData = snapshot.val();
        pdfUrl = workOrderData?.pdfDownloadUrl;
      } else {
        const reportRaw = await AsyncStorage.getItem(`report_${report.id}`);
        if (reportRaw) {
          const report = JSON.parse(reportRaw);
          pdfUrl = report.workOrder?.pdfDownloadUrl;
        }
      }

      if (!pdfUrl) {
        showToast('error', 'Error', isOnline ? 'PDF URL not available for this work order.' : 'Work Order PDF not available offline.');
        return;
      }

      if (!isOnline) {
        showToast('info', 'Offline Mode', 'Viewing locally stored work order.');
      }

      navigation.navigate('PdfviewPage', {
        pdfUrl,
        onGoBack: () => navigation.navigate('Inspection_Report_Home'),
      });
    } catch (error) {
      console.error('Error fetching work order PDF URL:', error);
      let errorMessage = 'Could not fetch the work order PDF URL. Please try again.';
      if (!isOnline) {
        errorMessage = 'Offline mode. Please connect to the internet to view the PDF.';
      }
      showToast('error', 'Error', errorMessage);
    }
  };

  const handleShare = async (report) => {
    if (!report) {
      showToast('error', 'Error', 'No report selected.');
      return;
    }

    try {
      let shareContent = null;
      if (isOnline && report.step3?.pdfDownloadUrl) {
        shareContent = {
          url: report.step3.pdfDownloadUrl,
          title: `Inspection Report ${report.step1.requestNumber}`,
          message: `Inspection Report ${report.step1.requestNumber} generated on ${formatDate(report.step3?.createdAt)}`,
        };
      } else {
        const reportRaw = await AsyncStorage.getItem(`report_${report.id}`);
        if (reportRaw) {
          const report = JSON.parse(reportRaw);
          shareContent = {
            message: `Inspection Report ${report.step1.requestNumber}\nLocation: ${report.step1.country}\nProject: ${report.step1.project}\nInspector: ${report.step1.inspector}\nEquipment: ${report.step1.selectedEquipment?.equipmentName}\nCreated: ${formatDate(report.step3?.createdAt)}`,
          };
        }
      }

      if (!shareContent) {
        showToast('error', 'Error', 'Unable to share report. Data not available.');
        return;
      }

      await Share.share(shareContent);
      showToast('success', 'Success', 'Report shared successfully.');
    } catch (error) {
      console.error('Error sharing report:', error);
      showToast('error', 'Error', 'Failed to share report.');
    }
  };

  const handleCreateWorkOrder = async (report) => {
    if (!report) {
      showToast('error', 'Error', 'No report selected.');
      return;
    }

    if (report.workOrder) {
      Alert.alert('Job Card Exists', 'A job card has already been created for this inspection.');
      return;
    }

    try {
      if (!isOnline) {
        showToast('error', 'Error', 'Offline mode. Cannot create work order.');
        return;
      }

      const userId = await AsyncStorage.getItem('userUid');
      if (!userId) {
        showToast('error', 'Authentication Error', 'No user is logged in.');
        navigation.navigate('AdminLoginScreen');
        return;
      }

      const userRef = database().ref(`/Globalusers/admins/${userId}`);
      const userSnapshot = await userRef.once('value');
      if (!userSnapshot.exists()) {
        throw new Error('User not found in admin database');
      }

      const workOrdersRef = database().ref(`/GlobalJobInspections/${userId}`);
      const snapshot = await workOrdersRef.once('value');
      const workOrdersData = snapshot.val();
      let workOrderExists = false;

      if (workOrdersData) {
        Object.values(workOrdersData).forEach((workOrder) => {
          if (workOrder.linkInspection?.id === report.id) {
            workOrderExists = true;
          }
        });
      }

      if (workOrderExists) {
        showToast('error', 'Error', 'A work order already exists for this inspection.');
        return;
      }

      navigation.navigate('CreateInsWorkOrder', {
        selectedReport: report,
        selectedEquipment: report.step1?.selectedEquipment,
        onRefresh: handleRefresh,
        userId,
        onGoBack: () => navigation.navigate('Inspection_Report_Home'),
      });
    } catch (error) {
      console.error('Error checking work order existence:', error);
      showToast('error', 'Error', 'Failed to verify work order status.');
    }
  };

  const handleCorrectiveAction = async (report) => {
    if (!report) {
      showToast('error', 'Error', 'No report selected.');
      return;
    }

    if (!report.step2?.selections || !report.step2?.sections) {
      Alert.alert('Error', 'Report is missing required data for corrective action.');
      return;
    }

    try {
      const userId = await AsyncStorage.getItem('userUid');
      if (!userId) {
        showToast('error', 'Authentication Error', 'No user is logged in.');
        navigation.navigate('AdminLoginScreen');
        return;
      }

      const correctiveRef = database().ref(`/GlobalCorrectiveActions/admins/${userId}/${report.id}`);
      const snapshot = await correctiveRef.once('value');
      const correctiveData = snapshot.val();

      navigation.navigate('CorrectiveactionHeader', {
        report,
        isUpdate: !!correctiveData,
        userId,
        onRefresh: handleRefresh,
        onGoBack: () => navigation.navigate('Inspection_Report_Home'),
      });
    } catch (error) {
      console.error('Error checking corrective action:', error);
      showToast('error', 'Error', 'Failed to load corrective action data.');
    }
  };

  const handleArchive = async (report) => {
    if (!report) {
      showToast('error', 'Error', 'No report selected.');
      return;
    }
    Alert.alert('Archive', 'This feature is not yet implemented.');
  };

  const handleDelete = async (report) => {
    if (!report) {
      showToast('error', 'Error', 'No report selected.');
      return;
    }

    Alert.alert(
      'Delete Report',
      `Are you sure you want to delete Inspection #${report.step1.requestNumber}?`,
      [
        { text: 'Cancel', style: 'cancel' },
        {
          text: 'Delete',
          style: 'destructive',
          onPress: async () => {
            try {
              const userId = await AsyncStorage.getItem('userUid');
              if (!userId) {
                showToast('error', 'Authentication Error', 'No user is logged in.');
                return;
              }
              if (isOnline) {
                await Promise.all([
                  database().ref(`/GlobalInspectionReport/admins/${userId}/${report.id}`).remove(),
                  database().ref(`/GlobalCorrectiveActions/admins/${userId}/${report.id}`).remove(),
                ]);
                await AsyncStorage.removeItem(`report_${report.id}`);
                showToast('success', 'Success', 'Report deleted successfully.');
              } else {
                await queueOperation({
                  type: 'delete',
                  reportId: report.id,
                  userId,
                });
                await AsyncStorage.removeItem(`report_${report.id}`);
                showToast('info', 'Offline Mode', 'Deletion queued and will sync when online.');
              }

              if (isMounted.current) {
                setReports((prev) => prev.filter((r) => r.id !== report.id));
                setAllReports((prev) => prev.filter((r) => r.id !== report.id));
              }
            } catch (error) {
              console.error('Error deleting report:', error);
              let errorMessage = 'Failed to delete report.';
              if (!isOnline) {
                errorMessage = 'Offline mode. Deletion will sync when online.';
              }
              showToast('error', 'Error', errorMessage);
            }
          },
        },
      ]
    );
  };

  const handleRiskAssessment = async (report) => {
    if (!report) {
      showToast('error', 'Error', 'No report selected.');
      return;
    }
    navigation.navigate('Hazards_step1', {
      report,
      onGoBack: () => navigation.navigate('Inspection_Report_Home'),
    });
  };

  const handleWorkOrderNavigation = async (workOrderId, adminId) => {
    if (!workOrderId) {
      showToast('error', 'Error', 'Work order ID is missing.');
      return;
    }

    if (!adminId) {
      showToast('error', 'Error', 'Admin ID is missing.');
      return;
    }

    try {
      const userId = await AsyncStorage.getItem('userUid');
      if (!userId) {
        showToast('error', 'Authentication Error', 'No user is logged in.');
        navigation.navigate('AdminLoginScreen');
        return;
      }
      if (!isOnline) {
        showToast('error', 'Error', 'Offline mode. Please connect to the internet to view the work order.');
        return;
      }

      const workOrderRef = database().ref(`/GlobalWorkOrders/admins/${adminId}/${workOrderId}`);
      const snapshot = await workOrderRef.once('value');
      const workOrderData = snapshot.val();

      if (!workOrderData) {
        showToast('error', 'Error', 'Work order not found in the database.');
        return;
      }

      navigation.navigate('WorkerDashboardhome', {
        workOrderId,
        adminUid: adminId,
        onGoBack: () => navigation.navigate('Inspection_Report_Home'),
      });
    } catch (error) {
      console.error('Error navigating to work order:', error);
      let errorMessage = 'Failed to load work order details.';
      if (!isOnline) {
        errorMessage = 'Offline mode. Please connect to the internet to view the work order.';
      }
      showToast('error', 'Error', errorMessage);
    }
  };

  const handleDropdownAction = async (action, report) => {
    try {
      setIsProcessingAction(true);
      await new Promise(resolve => setTimeout(resolve, 100));
      
      switch (action) {
        case 'view':
          await handleViewReport(report);
          break;
        case 'share':
          await handleShare(report);
          break;
        case 'create-work-order':
          await handleCreateWorkOrder(report);
          break;
        case 'corrective':
          await handleCorrectiveAction(report);
          break;
        case 'archive':
          await handleArchive(report);
          break;
        case 'view-risk-assessment':
          await handleViewRiskAssessmentPDF(report);
          break;
        case 'risk-assessment':
          await handleRiskAssessment(report);
          break;
        case 'delete':
          await handleDelete(report);
          break;
        default:
          break;
      }
    } catch (error) {
      console.error('Error processing action:', error);
      showToast('error', 'Error', 'Failed to process action.');
    } finally {
      setIsProcessingAction(false);
    }
  };

  const formatDate = (isoString) => {
    if (!isoString) return 'N/A';
    const date = new Date(isoString);
    return date.toLocaleString('en-US', {
      day: '2-digit',
      month: 'short',
      year: 'numeric',
      hour: '2-digit',
      minute: '2-digit',
      hour12: true,
    }).replace(',', '');
  };

  const renderReportItem = ({ item, index }) => {
    const dropdownPosition = index > reports.length - 4 ? 'top' : 'auto';
    let filteredOptions = DROPDOWN_OPTIONS.filter(option => {
      if (option.value === 'delete' && !userInfo.inspectionPermissions.includes('delete')) {
        return false;
      }
      return true;
    });

    // If a Risk Assessment PDF exists, show "View Risk Assessment" and hide the create entry
    const raPdf = item?.riskAssessment?.step3?.pdfDownloadUrl;
    if (raPdf) {
      // Keep existing "Risk Assessment" (for updates) and add a dedicated view option
      filteredOptions.push({
        label: 'View Risk Assessment',
        value: 'view-risk-assessment',
        icon: <MaterialIcons name="picture-as-pdf" size={22} color="#FF9800" />,
      });
    }

    return (
      <View style={styles.reportItem}>
        <View style={styles.reportHeader}>
          <View style={{ flexDirection: 'row', alignItems: 'center' }}>
            <Text style={styles.reportTitle}>{item.step1?.requestNumber || 'N/A'}</Text>
          </View>
          <TouchableOpacity style={styles.statusButton}>
            <Text style={styles.statusButtonText}>{item.step3?.equipmentStatus || 'N/A'}</Text>
          </TouchableOpacity>
        </View>
        <View style={styles.reportDetails}>
          <View style={{ flexDirection: 'row', alignItems: 'stretch' }}>
            <View style={{ flex: 1 }}>
              <View style={styles.detailItem}>
                <Text style={styles.detailLabel}>Inspector</Text>
                <Text style={[styles.detailValue, styles.detailValueLeft]}>{item.step1?.inspector || 'N/A'}</Text>
              </View>
              <View style={styles.detailItem}>
                <Text style={styles.detailLabel}>Location</Text>
                <Text style={[styles.detailValue, styles.detailValueLeft]}>{item.step1?.country || 'N/A'}</Text>
              </View>
            </View>
            <View style={{ justifyContent: 'center', alignItems: 'center', width: 70 }}>
              <View style={styles.faultBadgeContainer}>
                <Text style={styles.faultBadgeText}>{item.step3?.faultyItems || 0}</Text>
                <Text style={styles.faultBadgeLabel}>Fault</Text>
              </View>
            </View>
          </View>
          <View style={styles.detailItem}>
            <Text style={styles.detailLabel}>Project</Text>
            <Text style={styles.detailValue}>{item.step1?.project || 'N/A'}</Text>
          </View>
          <View style={styles.detailItem}>
            <Text style={styles.detailLabel}>Equipment</Text>
            <Text style={styles.detailValue}>{item.step1?.selectedEquipment?.equipmentName || 'N/A'}</Text>
          </View>
          <View style={styles.detailItem}>
            <Text style={styles.detailLabel}>Corrective Action</Text>
            <Text style={[styles.detailValue, { color: item.correctiveAction ? '#2ecc40' : '#e74c3c' }]}>{item.correctiveAction ? 'Taken' : 'Not taken'}</Text>
          </View>
          {item.correctiveActionNumber && (
            <View style={styles.detailItem}>
              <Text style={styles.detailLabel}>Corrective Action No</Text>
              <TouchableOpacity onPress={() => handleDropdownAction('corrective', item)}>
                <Text style={[styles.detailValue, { color: '#0078D4', textDecorationLine: 'underline' }]}>{item.correctiveActionNumber}</Text>
              </TouchableOpacity>
            </View>
          )}
          {item.correctiveAction?.pdfDownloadUrl && (
            <View style={styles.detailItem}>
              <Text style={styles.detailLabel}>Corrective PDF</Text>
              <TouchableOpacity onPress={() => handleViewCorrectivePDF(item)}>
                <Text style={[styles.detailValue, { color: '#0078D4', textDecorationLine: 'underline' }]}>View</Text>
              </TouchableOpacity>
            </View>
          )}
          {item.workOrder?.pdfDownloadUrl && (
            <View style={styles.detailItem}>
              <Text style={styles.detailLabel}>Job Card PDF</Text>
              <TouchableOpacity onPress={() => handleViewWorkOrderPDF(item)}>
                <Text style={[styles.detailValue, { color: '#0078D4', textDecorationLine: 'underline' }]}>View</Text>
              </TouchableOpacity>
            </View>
          )}
          {item.workOrder?.id && (
            <View style={styles.detailItem}>
              <Text style={styles.detailLabel}>Job Card Track</Text>
              <TouchableOpacity onPress={() => handleWorkOrderNavigation(item.workOrder.id, item.workOrder.adminId)}>
                <Text style={[styles.detailValue, { color: '#0078D4', textDecorationLine: 'underline' }]}>Track</Text>
              </TouchableOpacity>
            </View>
          )}
          <View style={styles.detailItem}>
            <Text style={[styles.detailLabel, { color: '#666', marginRight: 10 }]}>Date</Text>
            <View style={styles.dateContainer}>
              <Text style={styles.detailValue}>{formatDate(item.step3?.createdAt)}</Text>
              <Dropdown
                style={styles.dropdown}
                data={filteredOptions}
                labelField="label"
                valueField="value"
                value={null}
                onChange={async (option) => {
                  await new Promise(resolve => setTimeout(resolve, 50));
                  handleDropdownAction(option.value, item);
                }}
                renderRightIcon={() => <MaterialIcons name="more-vert" size={22} color="#333" />}
                renderItem={(option) => (
                  <View style={styles.dropdownItem}>
                    {option.icon}
                    <Text
                      style={[
                        styles.dropdownText,
                        { color: option.value === 'delete' ? '#e74c3c' : '#015185' },
                      ]}
                    >
                      {option.label}
                    </Text>
                  </View>
                )}
                placeholder=""
                showsVerticalScrollIndicator={false}
                dropdownPosition={dropdownPosition}
                containerStyle={styles.dropdownContainer}
                itemTextStyle={styles.dropdownItemText}
                selectedTextStyle={{ display: 'none' }}
              />
            </View>
          </View>
        </View>
      </View>
    );
  };

  return (
    <ErrorBoundary>
      <View style={styles.container}>
        <LinearGradient
          colors={['#3481BC', '#025383']}
          style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
        >
          <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
          <View style={styles.headerContent}>
            <View style={styles.headerLeft}>
              <TouchableOpacity
                onPress={() => {
                  Alert.alert(
                    'Go Back',
                    'Are you sure you want to go back? Your changes may not be saved.',
                    [
                      { text: 'No', style: 'cancel' },
                      {
                        text: 'Yes',
                        onPress: () => navigation.reset({
                          index: 0,
                          routes: [{ name: 'MainApp' }],
                        }),
                        style: 'destructive',
                      },
                    ]
                  );
                }}
              >
                <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
              </TouchableOpacity>
              <Text style={styles.headerTitle}>Inspection Report</Text>
            </View>
            <View style={styles.headerRight}>
              <TouchableOpacity style={styles.iconButton} onPress={handleRefresh}>
                <Image
                  source={require('../../../Images/adminhome/equp/cloud_1.png')}
                  style={styles.icons12}
                />
              </TouchableOpacity>
              <TouchableOpacity
                style={styles.iconButton}
                onPress={() => setUserInfoModalVisible(true)}
              >
                <MaterialIcons name="lock" size={width * 0.06} color="#fff" />
              </TouchableOpacity>
            </View>
          </View>
          <View style={styles.searchContainer}>
            <View style={styles.searchWrapper}>
              <TextInput
                style={styles.searchInput}
                placeholder={`Search${selectedFilter ? ' by ' + selectedFilter : '...'}`}
                placeholderTextColor="#888"
                value={searchQuery}
                onChangeText={handleSearch}
              />
              <MaterialIcons name="search" size={width * 0.05} color="#888" style={styles.bladeIcon} />
            </View>
            <TouchableOpacity
              style={styles.filterButton}
              onPress={() => setFilterModalVisible(true)}
            >
              <MaterialIcons name="filter-list" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
          </View>
        </LinearGradient>
        
        {/* Email Statistics Section removed as per user request */}
        
        <View style={styles.mainContent}>
          {loadingUserInfo ? (
            <Text style={styles.defaultText}>Loading...</Text>
          ) : !userInfo.inspectionPermissions.includes('view') && !userInfo.inspectionPermissions.includes('onlyMineView') ? (
            <Text style={styles.defaultText}>You do not have permission to view inspection records.</Text>
          ) : reports.length > 0 ? (
            <>
              <FlatList
                ref={flatListRef}
                data={reports}
                keyExtractor={(item) => item.id.toString()}
                renderItem={renderReportItem}
                ListEmptyComponent={<Text style={styles.defaultText}>No inspection records found</Text>}
                contentContainerStyle={styles.reportList}
                refreshing={isRefreshing}
                onRefresh={handleRefresh}
                onEndReached={handleLoadMore}
                onEndReachedThreshold={0.5}
                ListFooterComponent={loadingMore && hasMore ? <View style={{ padding: 16 }}><ActivityIndicator size="small" color="#3481BC" /></View> : null}
              />
              {/* List of reports with both Job Card and Corrective Action (below the list) */}
              {/* {(() => {
                const reportsWithBoth = reports.filter(
                  report => report.workOrder && report.correctiveAction
                );
                const countWithBoth = reportsWithBoth.length;
                return (
                  <View style={{ marginVertical: 16, alignItems: 'center' }}>
                    <Text style={{ fontWeight: 'bold', color: '#015185', fontSize: 15 }}>
                      Linked Job Card & Corrective Action: {countWithBoth}
                    </Text>
                    {countWithBoth > 0 && (
                      <Text style={{ color: '#444', fontSize: 13, marginTop: 4, textAlign: 'center' }}>
                        {reportsWithBoth.map(r => r.step1?.requestNumber || r.id).join(', ')}
                      </Text>
                    )}
                  </View>
                );
              })()} */}
            </>
          ) : (
            <>
              <Text style={styles.defaultText}>You don't have any inspection records</Text>
              <View style={styles.questionContainer}>
                <MaterialIcons name="help-outline" size={width * 0.05} color="#333" />
                <Text style={styles.questionText}>How To Create Inspection?</Text>
              </View>
            </>
          )}
        </View>
        {userInfo.inspectionPermissions.includes('create') && (
          <View style={styles.buttonContainer}>
            <LinearGradient colors={['#3481BC', '#025383']} style={styles.buttonGradient}>
              <TouchableOpacity
                style={styles.startButton}
                onPress={async () => {
                  try {
                    const userId = await AsyncStorage.getItem('userUid');
                    if (!userId) {
                      showToast('error', 'Authentication Error', 'No user is logged in.');
                      navigation.navigate('AdminLoginScreen');
                      return;
                    }
                    await AsyncStorage.removeItem('inspectionStep1');
                    await AsyncStorage.setItem('equipmentStoragePath', `GlobalEquipments/admins/${userId}`);
                    navigation.navigate('InsreportHeader', {
                      onGoBack: () => navigation.navigate('Inspection_Report_Home'),
                    });
                  } catch (error) {
                    console.error('Error initiating new inspection:', error);
                    showToast('error', 'Error', 'Failed to start new inspection.');
                  }
                }}
              >
                <Ionicons name="add" size={width * 0.06} color="#fff" />
                <Text style={styles.buttonText}>Start New Inspection</Text>
              </TouchableOpacity>
            </LinearGradient>
          </View>
        )}
        <Modal
          visible={filterModalVisible}
          transparent={true}
          animationType="slide"
          onRequestClose={() => setFilterModalVisible(false)}
        >
          <View style={styles.modalOverlay}>
            <View style={styles.modalContent}>
              <Text style={styles.modalTitle}>Filter By</Text>
              {filterOptions.map((filter) => (
                <TouchableOpacity
                  key={filter}
                  style={styles.modalOption}
                  onPress={() => {
                    setSelectedFilter(filter);
                    setFilterModalVisible(false);
                  }}
                >
                  <Text style={styles.modalOptionText}>{filter}</Text>
                </TouchableOpacity>
              ))}
              <TouchableOpacity
                style={styles.modalCloseButton}
                onPress={() => setFilterModalVisible(false)}
              >
                <Text style={styles.modalCloseText}>Close</Text>
              </TouchableOpacity>
            </View>
          </View>
        </Modal>
        <UserInfoModal
          visible={userInfoModalVisible}
          onClose={() => setUserInfoModalVisible(false)}
          userInfo={userInfo}
        />
        <SyncModal visible={syncModalVisible} onClose={() => setSyncModalVisible(false)} />
        <SyncModal visible={isProcessingAction} message="Processing Action..." />
        <Toast />
      </View>
    </ErrorBoundary>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.05,
    fontWeight: 'bold',
    marginLeft: width * 0.025,
  },
  iconButton: {
    padding: width * 0.013,
    marginLeft: width * 0.035,
  },
  icons12: {
    width: 30,
    height: width * 0.06,
  },
  searchContainer: {
    flexDirection: 'row',
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
    alignItems: 'center',
    width: '100%',
  },
  searchWrapper: {
    flex: 1,
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#fff',
    borderRadius: width * 0.09,
    paddingHorizontal: width * 0.025,
    marginRight: width * 0.025,
  },
  searchInput: {
    flex: 1,
    height: width * 0.1,
    color: '#333',
    fontSize: width * 0.04,
  },
  bladeIcon: {
    marginLeft: width * 0.012,
  },
  filterButton: {
    padding: width * 0.012,
  },
  mainContent: {
    flex: 1,
    paddingHorizontal: width * 0.04,
  },
  defaultText: {
    fontSize: width * 0.045,
    color: '#888',
    textAlign: 'center',
    marginVertical: width * 0.05,
  },
  questionContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#e0e0e0',
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.04,
    borderRadius: width * 0.012,
    marginTop: width * 0.025,
  },
  questionText: {
    fontSize: width * 0.04,
    color: '#333',
    marginLeft: width * 0.025,
  },
  reportList: {
    paddingVertical: width * 0.025,
    paddingBottom: width * 0.3,
  },
  reportItem: {
    marginVertical: width * 0.012,
    backgroundColor: '#F2F2F2',
    borderRadius: width * 0.012,
    borderWidth: 1,
    borderColor: '#eee',
  },
  reportHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingVertical: width * 0.025,
    backgroundColor: '#E9E9E9',
    paddingHorizontal: width * 0.05,
    borderTopLeftRadius: width * 0.012,
    borderTopRightRadius: width * 0.012,
  },
  reportTitle: {
    fontSize: width * 0.045,
    color: '#015185',
    fontWeight: '500',
  },
  statusButton: {
    backgroundColor: '#015185',
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.03,
    borderRadius: width * 0.012,
    border: '2px solid #015185',
  },
  statusButtonText: {
    color: '#fff',
    fontSize: width * 0.035,
    fontWeight: '500',
  },
  reportDetails: {
    paddingHorizontal: width * 0.05,
  },
  detailItem: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingVertical: width * 0.012,
    borderBottomWidth: 1,
    borderBottomColor: '#ddd',
  },
  detailLabel: {
    fontSize: width * 0.04,
    color: '#666',
    fontWeight: '400',
    width: '40%',
    marginVertical: width * 0.008,
  },
  detailValue: {
    fontSize: width * 0.04,
    color: '#015185',
    fontWeight: '500',
  },
  detailValueLeft: {
    textAlign: 'left',
    width: '60%',
  },
  dateContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'flex-end',
    width: '60%',
  },
  dropdown: {
    width: 40,
    backgroundColor: 'transparent',
    borderWidth: 0,
  },
  buttonContainer: {
    position: 'absolute',
    bottom: '10%',
    width: '100%',
    alignItems: 'center',
  },
  buttonGradient: {
    borderRadius: width * 0.012,
    overflow: 'hidden',
  },
  startButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.05,
  },
  buttonText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: 'bold',
    marginLeft: width * 0.012,
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    backgroundColor: '#fff',
    borderRadius: width * 0.025,
    padding: width * 0.05,
    width: '80%',
  },
  modalTitle: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#333',
    marginBottom: width * 0.025,
    textAlign: 'center',
  },
  modalOption: {
    paddingVertical: width * 0.025,
  },
  modalOptionText: {
    fontSize: width * 0.04,
    color: '#333',
  },
  modalCloseButton: {
    marginTop: width * 0.025,
    alignItems: 'center',
  },
  modalCloseText: {
    fontSize: width * 0.04,
    color: '#0078D4',
    fontWeight: '600',
  },
  dropdownContainer: {
    width: 220,
    borderRadius: 3,
    paddingVertical: 4,
    paddingHorizontal: 0,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 4,
    elevation: 5,
    padding: 10,
    left: '40%',
  },
  dropdownItem: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 11,
    paddingHorizontal: 11,
  },
  dropdownText: {
    marginLeft: 12,
    fontSize: 16,
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 24,
    height: 24,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  faultBadgeContainer: {
    flexDirection: 'column',
    alignItems: 'center',
    justifyContent: 'center',
    backgroundColor: '#fff',
    borderRadius: width * 0.025,
    marginLeft: width * 0.03,
    paddingHorizontal: width * 0.025,
    paddingVertical: width * 0.01,
    minWidth: width * 0.09,
    minHeight: width * 0.07,
  },
  faultBadgeText: {
    color: '#e74c3c',
    fontWeight: 'bold',
    fontSize: width * 0.045,
    textAlign: 'center',
    lineHeight: width * 0.05,
  },
  faultBadgeLabel: {
    color: '#e74c3c',
    fontSize: width * 0.03,
    fontWeight: '600',
    textAlign: 'center',
    marginTop: -2,
  },
  userInfoItem: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingVertical: width * 0.015,
    borderBottomWidth: 1,
    borderBottomColor: '#ddd',
  },
  userInfoLabel: {
    fontSize: width * 0.04,
    color: '#666',
    fontWeight: '400',
    width: '40%',
  },
  userInfoValue: {
    fontSize: width * 0.04,
    color: '#015185',
    fontWeight: '500',
    width: '60%',
  },
  emailStatsContainer: {
    backgroundColor: '#fff',
    marginHorizontal: width * 0.04,
    marginTop: width * 0.025,
    borderRadius: width * 0.012,
    borderWidth: 1,
    borderColor: '#eee',
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 1 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
  },
  emailStatsHeader: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
    backgroundColor: '#f8f9fa',
    borderTopLeftRadius: width * 0.012,
    borderTopRightRadius: width * 0.012,
  },
  emailStatsTitle: {
    fontSize: width * 0.04,
    fontWeight: '600',
    color: '#015185',
    marginLeft: width * 0.025,
  },
  emailStatsContent: {
    flexDirection: 'row',
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
  },
  emailStatItem: {
    flex: 1,
    alignItems: 'center',
  },
  emailStatLabel: {
    fontSize: width * 0.032,
    color: '#666',
    marginBottom: width * 0.01,
    textAlign: 'center',
  },
  emailStatValue: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    textAlign: 'center',
  },
});

export default Inspection_Report_Home;