import React, { useState, useEffect } from 'react';
import { View, Text, StyleSheet, FlatList, TouchableOpacity, Dimensions } from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import RiskManagementHeader from './RiskManagementHeader';
import { useNavigation, useRoute } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import database from '@react-native-firebase/database';


const TASKS = [
  { id: '1', title: 'Abrasive Surface' },
  { id: '2', title: 'Confined Space' },
  { id: '3', title: 'Dust' },
  { id: '4', title: 'Electricity  LV' },
  { id: '5', title: 'Electricity HV' },
  { id: '6', title: 'Elevated Work Area' },
  { id: '7', title: 'Falling Objects' },
  { id: '8', title: 'Flying Objects' },
  { id: '9', title: 'Fumes' },
  { id: '10', title: 'Hazardous Chemicals' },
  { id: '11', title: 'Heavy Objects' },
  { id: '12', title: 'High Pressure -  Hydraulic' },
  { id: '13', title: 'High Pressure – Air' },
  { id: '14', title: 'High Pressure – Water' },
  { id: '15', title: 'Hot Objects' },
  { id: '16', title: 'Hydraulic Failure' },
  { id: '17', title: 'Loud Noise' },
  { id: '18', title: 'Low Oxygen' },
  { id: '19', title: 'Poor Housekeeping' },
  { id: '20', title: 'Radiation' },
  { id: '21', title: 'Rotating Parts' },
  { id: '22', title: 'Sharp Objects' },
  { id: '23', title: 'Steam' },
  { id: '24', title: 'Uneven Surface' },
  { id: '25', title: 'Unexpected Movement' },
];

// Global risks list to show in a dropdown for the selected hazard
const RISKS = [
  { id: 'r1', title: 'Amputation' },
  { id: 'r2', title: 'Burns' },
  { id: 'r3', title: 'Caught Between' },
  { id: 'r4', title: 'Choking' },
  { id: 'r5', title: 'Contact With / Between (Pinch)' },
  { id: 'r6', title: 'Contamination – Soil' },
  { id: 'r7', title: 'Contamination – Oil' },
  { id: 'r8', title: 'Crush' },
  { id: 'r9', title: 'Drowning' },
  { id: 'r10', title: 'Electrocution' },
  { id: 'r11', title: 'Entangled' },
  { id: 'r12', title: 'Fall from Heights' },
  { id: 'r13', title: 'Impairment' },
  { id: 'r14', title: 'Laceration' },
  { id: 'r15', title: 'Over Exertion' },
  { id: 'r16', title: 'Penetrated by' },
  { id: 'r17', title: 'Poisoning' },
  { id: 'r18', title: 'Slip & Fall' },
  { id: 'r19', title: 'Sprain' },
  { id: 'r20', title: 'Strain' },
  { id: 'r21', title: 'Struck Against' },
  { id: 'r22', title: 'Struck By' },
  { id: 'r23', title: 'Thermal Burns' },
  { id: 'r24', title: 'Trip' },
];

const Hazards_step1 = (props) => {
  const navigation = useNavigation();
  const route = useRoute();
  const linkedReport = route.params?.report || null;
  const [checked, setChecked] = useState({});
  const [currentStep, setCurrentStep] = useState(1);
  const [selectedHazardId, setSelectedHazardId] = useState(null);
  // Which hazard's risk list is expanded
  const [expandedHazardId, setExpandedHazardId] = useState(null);
  // Map: hazardId -> { riskId: true }
  const [hazardRiskSelections, setHazardRiskSelections] = useState({});

  // Persist Step 1 data locally
  const persistStep1 = async (nextChecked = null, nextSelections = null) => {
    try {
      const selectedHazardIds = Object.keys(nextChecked ?? checked).filter((id) => (nextChecked ?? checked)[id]);
      const toStore = {
        selectedHazardIds,
        hazardRiskSelections: nextSelections ?? hazardRiskSelections,
      };
      await AsyncStorage.setItem('riskMgmt_step1', JSON.stringify(toStore));
      // Ensure meta is stored to point Firebase to /admins/{uid}/{inspectionId}
      if (linkedReport) {
        await AsyncStorage.setItem('riskMgmt_meta', JSON.stringify({ reportId: linkedReport.id, adminId: linkedReport.adminId }));
      } else {
        try {
          const userUid = await AsyncStorage.getItem('userUid');
          const step1Raw = await AsyncStorage.getItem('inspectionStep1');
          const step1 = step1Raw ? JSON.parse(step1Raw) : null;
          if (userUid && step1?.id) {
            await AsyncStorage.setItem('riskMgmt_meta', JSON.stringify({ reportId: step1.id, adminId: userUid }));
          }
        } catch (e) {
          // ignore
        }
      }
    } catch (e) {
      // noop
    }
  };

  // Load existing (Firebase first when linked, else Async fallback)
  useEffect(() => {
    (async () => {
      try {
        if (linkedReport && linkedReport.adminId && linkedReport.id) {
          try {
            const ref = database().ref(`/GlobalRiskAssessments/admins/${linkedReport.adminId}/${linkedReport.id}`);
            const snap = await ref.once('value');
            const data = snap.val();
            if (data?.step1) {
              if (Array.isArray(data.step1.selectedHazardIds)) {
                const nextChecked = {};
                data.step1.selectedHazardIds.forEach((id) => (nextChecked[id] = true));
                setChecked(nextChecked);
              }
              if (data.step1.hazardRiskSelections) setHazardRiskSelections(data.step1.hazardRiskSelections);
              return; // prefer Firebase when available
            }
          } catch (e) {}
        }
        // Fallback: attempt to load from Firebase using current user and local inspection id
        try {
          const userUid = await AsyncStorage.getItem('userUid');
          const step1Raw = await AsyncStorage.getItem('inspectionStep1');
          const step1Local = step1Raw ? JSON.parse(step1Raw) : null;
          if (userUid && step1Local?.id) {
            const refLocal = database().ref(`/GlobalRiskAssessments/admins/${userUid}/${step1Local.id}`);
            const snapLocal = await refLocal.once('value');
            const dataLocal = snapLocal.val();
            if (dataLocal?.step1) {
              if (Array.isArray(dataLocal.step1.selectedHazardIds)) {
                const nextChecked = {};
                dataLocal.step1.selectedHazardIds.forEach((id) => (nextChecked[id] = true));
                setChecked(nextChecked);
              }
              if (dataLocal.step1.hazardRiskSelections) setHazardRiskSelections(dataLocal.step1.hazardRiskSelections);
              return;
            }
          }
        } catch (e) {}
        const saved = await AsyncStorage.getItem('riskMgmt_step1');
        if (saved) {
          const parsed = JSON.parse(saved);
          if (parsed?.selectedHazardIds) {
            const nextChecked = {};
            parsed.selectedHazardIds.forEach((id) => (nextChecked[id] = true));
            setChecked(nextChecked);
          }
          if (parsed?.hazardRiskSelections) setHazardRiskSelections(parsed.hazardRiskSelections);
        }
      } catch (e) {
        // ignore
      }
    })();
  }, []);

  const toggleCheck = (id) => {
    setChecked((prev) => {
      const wasSelected = !!prev[id];
      const willSelect = !prev[id];
      const next = { ...prev, [id]: willSelect };
      setSelectedHazardId(id);
      // Expand this hazard when selected; collapse if unselected
      if (willSelect) {
        setExpandedHazardId(id);
      } else if (expandedHazardId === id) {
        setExpandedHazardId(null);
      }
      persistStep1(next, null);
      return next;
    });
  };

  const toggleRiskForSelectedHazard = (riskId) => {
    if (!selectedHazardId) return;
    setHazardRiskSelections((prev) => {
      const cur = prev[selectedHazardId] || {};
      const nextForHazard = { ...cur, [riskId]: !cur[riskId] };
      const next = { ...prev, [selectedHazardId]: nextForHazard };
      persistStep1(null, next);
      return next;
    });
  };

  const renderRisksDropdown = () => {
    if (!selectedHazardId) return null;
    const selectedHazard = TASKS.find(t => t.id === selectedHazardId);
    const selectedMap = hazardRiskSelections[selectedHazardId] || {};
    return (
      <View style={styles.dropdownContainer}>
        <TouchableOpacity style={styles.dropdownHeader} onPress={() => setIsRiskOpen((v) => !v)}>
          <Text style={styles.dropdownTitle}>Risks</Text>
          <View style={{ flexDirection: 'row', alignItems: 'center' }}>
            <Text style={styles.dropdownSubTitle}>For: {selectedHazard?.title || ''}  ·  Tap to {isRiskOpen ? 'hide' : 'show'}</Text>
            <MaterialIcons name={isRiskOpen ? 'expand-less' : 'expand-more'} size={22} color="#015185" />
          </View>
        </TouchableOpacity>
        {isRiskOpen && (
          <>
            <View style={styles.divider} />
            {RISKS.map((risk) => {
              const isSelected = !!selectedMap[risk.id];
              return (
                <TouchableOpacity key={risk.id} onPress={() => toggleRiskForSelectedHazard(risk.id)} style={styles.dropdownRow}>
                  <Text style={styles.dropdownText}>{risk.title}</Text>
                  <View style={styles.checkboxOuterSmall}>
                    {isSelected ? (
                      <View style={styles.checkboxInnerSmall}>
                        <MaterialIcons name="check" size={14} color="#3481BC" />
                      </View>
                    ) : null}
                  </View>
                </TouchableOpacity>
              );
            })}
          </>
        )}
      </View>
    );
  };

  const renderItem = ({ item }) => {
    const isSelected = !!checked[item.id];
    const isActive = selectedHazardId === item.id;
    const isExpanded = expandedHazardId === item.id;
    const selectedMap = hazardRiskSelections[item.id] || {};
    const selectedRiskNames = Object.keys(selectedMap)
      .filter((rid) => selectedMap[rid])
      .map((rid) => (RISKS.find((r) => r.id === rid)?.title || rid));
    return (
      <View>
        <TouchableOpacity onPress={() => toggleCheck(item.id)} style={styles.card}>
          <View style={{ flex: 1 }}>
            <Text style={styles.cardTitle} numberOfLines={1}>{item.title}</Text>
            {item.subtitle && <Text style={styles.cardSubtitle}>{item.subtitle}</Text>}
          </View>
          <View style={styles.checkboxOuter}>
            {isSelected ? (
              <View style={styles.checkboxInner}>
                <MaterialIcons name="check" size={16} color="#3481BC" />
              </View>
            ) : null}
          </View>
        </TouchableOpacity>

        {isSelected && (
          <View style={styles.dropdownContainer}>
            <TouchableOpacity
              style={styles.dropdownHeader}
              onPress={() => {
                setSelectedHazardId(item.id);
                setExpandedHazardId((cur) => (cur === item.id ? null : item.id));
              }}
            >
              <Text style={styles.dropdownTitle}>Risks</Text>
              <View style={{ flexDirection: 'row', alignItems: 'center' }}>
                <Text style={styles.dropdownSubTitle}>For: {item.title}  ·  Tap to {isExpanded ? 'hide' : 'show'}</Text>
                <MaterialIcons name={isExpanded ? 'expand-less' : 'expand-more'} size={22} color="#015185" />
              </View>
            </TouchableOpacity>
            {!!selectedRiskNames.length && (
              <View style={styles.chipsRow}>
                {selectedRiskNames.slice(0, 3).map((n) => (
                  <View key={n} style={styles.chip}><Text style={styles.chipText}>{n}</Text></View>
                ))}
                {selectedRiskNames.length > 3 && (
                  <View style={styles.chip}><Text style={styles.chipText}>+{selectedRiskNames.length - 3} more</Text></View>
                )}
              </View>
            )}
            {isExpanded && (
              <>
                <View style={styles.divider} />
                {RISKS.map((risk) => {
                  const isRiskChecked = !!selectedMap[risk.id];
                  return (
                    <TouchableOpacity key={risk.id} onPress={() => toggleRiskForSelectedHazard(risk.id)} style={styles.dropdownRow}>
                      <Text style={styles.dropdownText}>{risk.title}</Text>
                      <View style={styles.checkboxOuterSmall}>
                        {isRiskChecked ? (
                          <View style={styles.checkboxInnerSmall}>
                            <MaterialIcons name="check" size={14} color="#3481BC" />
                          </View>
                        ) : null}
                      </View>
                    </TouchableOpacity>
                  );
                })}
              </>
            )}
          </View>
        )}
      </View>
    );
  };

  return (
    <View style={styles.container}>
      {/* Risk Management Header */}
      <RiskManagementHeader 
        currentStep={currentStep} 
        setCurrentStep={setCurrentStep} 
      />
      
      {/* Content */}
      <View style={styles.content}>
        {/* Header */}
        <View style={styles.headerRow}>
          <Text style={[styles.headerTitle, { color: '#3481BC' }]}>Hazards</Text>
          {/* Removed the icons next to the Hazards heading */}
        </View>
        {/* Task List */}
        <FlatList
          data={TASKS}
          renderItem={renderItem}
          keyExtractor={(item) => item.id}
          contentContainerStyle={{ paddingBottom: 100 }}
          showsVerticalScrollIndicator={false}
        />
      </View>
      {/* Bottom Buttons */}
      <View style={styles.buttonContainer}>
        <TouchableOpacity style={styles.cancelButton} onPress={() => navigation.goBack()}>
          <MaterialIcons style={styles.bottomicon} name="chevron-left" size={16} color="#015185" />
          <Text style={styles.cancelButtonText}>Cancel</Text>
        </TouchableOpacity>
        <TouchableOpacity
          style={styles.nextButton}
          onPress={() => navigation.navigate('Risk_step2', {
            selectedHazards: TASKS.filter(t => checked[t.id]),
            hazardRiskSelections,
            report: linkedReport || null,
          })}
        >
          <Text style={styles.nextButtonText}>Next</Text>
          <MaterialIcons style={styles.bottomicon} name="chevron-right" size={16} color="#015185" />
        </TouchableOpacity>
      </View>
    </View>
  );
};

const { width } = Dimensions.get('window');

const styles = StyleSheet.create({
  container: {
    flex: 1,
  },
  content: {
    flex: 1,
  },
  headerRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: 20,
    paddingTop: 16,
    paddingBottom: 16,
  },
  headerTitle: {
    fontSize: 22,
    fontWeight: 'bold',
  },
  headerIcons: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  card: {
    backgroundColor: '#e0e0e0', // grey background
    borderRadius: 16,
    flexDirection: 'row',
    alignItems: 'center',
    padding: 16,
    marginHorizontal: 16,
    marginBottom: 12,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.15,
    shadowRadius: 4,
    elevation: 2,
    justifyContent: 'space-between',
  },
  dropdownContainer: {
    marginHorizontal: 16,
    marginTop: -4,
    marginBottom: 12,
    backgroundColor: '#fff',
    borderRadius: 12,
    borderWidth: 1,
    borderColor: '#e0e0e0',
  },
  dropdownHeader: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: 14,
    paddingVertical: 10,
  },
  dropdownTitle: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '700',
  },
  dropdownSubTitle: {
    color: '#707173',
    fontSize: 13,
    fontWeight: '500',
  },
  divider: {
    height: 1,
    backgroundColor: '#eee',
    marginHorizontal: 10,
  },
  dropdownRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: 14,
    paddingVertical: 10,
    borderBottomWidth: 1,
    borderBottomColor: '#f2f2f2',
  },
  dropdownText: {
    color: '#333',
    fontSize: 14,
    flex: 1,
    paddingRight: 10,
  },
  chipsRow: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    paddingHorizontal: 14,
    paddingBottom: 8,
    gap: 6,
  },
  chip: {
    backgroundColor: '#e6f0fa',
    borderRadius: 12,
    paddingHorizontal: 10,
    paddingVertical: 4,
    borderWidth: 1,
    borderColor: '#c7d8ee',
  },
  chipText: {
    color: '#015185',
    fontSize: 12,
    fontWeight: '600',
  },
  checkboxOuter: {
    width: 24,
    height: 24,
    borderRadius: 12,
    borderWidth: 2,
    borderColor: '#3481BC',
    marginRight: 0,
    alignItems: 'center',
    justifyContent: 'center',
    backgroundColor: 'transparent', // remove background
  },
  checkboxInner: {
    width: 20,
    height: 20,
    borderRadius: 10,
    backgroundColor: 'transparent', // remove background
    alignItems: 'center',
    justifyContent: 'center',
  },
  checkboxOuterSmall: {
    width: 20,
    height: 20,
    borderRadius: 10,
    borderWidth: 2,
    borderColor: '#3481BC',
    alignItems: 'center',
    justifyContent: 'center',
    backgroundColor: 'transparent',
  },
  checkboxInnerSmall: {
    width: 16,
    height: 16,
    borderRadius: 8,
    backgroundColor: 'transparent',
    alignItems: 'center',
    justifyContent: 'center',
  },
  cardTitle: {
    color: '#3481BC', // blue like heading
    fontSize: 16,
    fontWeight: '600',
  },
  cardSubtitle: {
    color: '#3481BC', // blue like heading
    fontSize: 12,
    marginTop: 2,
  },
  fab: {
    position: 'absolute',
    bottom: 24,
    right: 24,
    backgroundColor: '#fff',
    width: 56,
    height: 56,
    borderRadius: 28,
    alignItems: 'center',
    justifyContent: 'center',
    elevation: 6,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 4,
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: 15,
    paddingVertical: 15,
    backgroundColor: '#F2F2F2',
    borderTopWidth: 1,
    borderTopColor: '#ddd',
    position: 'absolute',
    bottom: 0,
    left: 0,
    right: 0,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: -2 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
    paddingBottom: 30,
  },
  cancelButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingBottom: 30,
    paddingHorizontal: 15,
  },
  cancelButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  nextButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingBottom: 30,
    borderRadius: 5,
  },
  nextButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  bottomicon: {
    marginTop: 2,
  },
});

export default Hazards_step1;