import React, { useState, useEffect, useCallback } from 'react';
import {
  View,
  Text,
  StyleSheet,
  Dimensions,
  StatusBar,
  TouchableOpacity,
  ScrollView,
  Modal,
  Animated,
  Platform,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import LinearGradient from 'react-native-linear-gradient';
import AsyncStorage from '@react-native-async-storage/async-storage';
import { useNavigation } from '@react-navigation/native';
import database from '@react-native-firebase/database';
import auth from '@react-native-firebase/auth';
import NetInfo from '@react-native-community/netinfo';
import Toast from 'react-native-toast-message';
import { ToastAndroid } from 'react-native';

const { width } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.3;

const LoaderModal = ({ visible, message = 'Loading KPI Data...' }) => {
  const rotateAnim = React.useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={{ flex: 1, backgroundColor: 'rgba(0,0,0,0.5)', justifyContent: 'center', alignItems: 'center' }}>
        <View style={{ backgroundColor: '#fff', borderRadius: 10, padding: 30, alignItems: 'center' }}>
          <Text style={{ fontSize: 18, color: '#015185', marginBottom: 10 }}>{message}</Text>
          <Animated.Image
            source={require('../../../Images/adminlogin/sync.png')}
            style={{ width: 40, height: 40, transform: [{ rotate: rotateInterpolate }] }}
          />
        </View>
      </View>
    </Modal>
  );
};

// Helper to color status
const getStatusColor = (status) => {
  switch (status) {
    case 'Available': return '#2ecc71';
    case 'In Maintenance': return '#e67e22';
    case 'Maintenance Required': return '#f1c40f';
    case 'Breakdown': return '#e74c3c';
    case 'Deployed': return '#3498db';
    default: return '#666';
  }
};

const Masterdatakpi = () => {
  const navigation = useNavigation();
  const [equipmentList, setEquipmentList] = useState([]);
  const [selectedCountry, setSelectedCountry] = useState('All Countries');
  const [userInfo, setUserInfo] = useState({
    countries: [],
    projects: [],
    equipmentPermissions: [],
    employeeNumber: '',
    name: '',
    userUid: '',
  });
  const [isLoading, setIsLoading] = useState(true);
  const [modalVisible, setModalVisible] = useState(false);

  const showToast = (type, text1, text2) => {
    if (Platform.OS === 'android') {
      ToastAndroid.show(`${text1}: ${text2}`, ToastAndroid.LONG);
    } else {
      Toast.show({ type, text1, text2, position: 'bottom' });
    }
  };

  const checkNetworkStatus = async () => {
    try {
      const netInfo = await NetInfo.fetch();
      return netInfo.isConnected;
    } catch (error) {
      console.error('Error checking network status:', error);
      return false;
    }
  };

  const fetchUserInfo = async () => {
    try {
      const user = auth().currentUser;
      if (!user) {
        showToast('error', 'Error', 'User not authenticated. Please log in.');
        navigation.replace('AdminLogin');
        return null;
      }

      let userUid = await AsyncStorage.getItem('userUid');
      let userData = null;

      if (userUid) {
        const userRef = database().ref(`/Globalusers/admins/${userUid}`);
        const snapshot = await userRef.once('value');
        userData = snapshot.val();
      }

      if (!userData) {
        const email = user.email.replace(/\./g, '_');
        const userRef = database().ref(`/Globalusers/admins/${email}`);
        const snapshot = await userRef.once('value');
        userData = snapshot.val();
        if (userData && userData.userUid) {
          userUid = userData.userUid;
          await AsyncStorage.setItem('userUid', userUid);
        } else {
          userUid = email;
          await AsyncStorage.setItem('userUid', userUid);
        }
      }

      if (!userData) {
        showToast('error', 'Error', 'User data not found.');
        return null;
      }

      const equipmentModule = userData.modules?.find(
        (module) => module.module?.toLowerCase() === 'equipments'
      ) || { permissions: [] };

      const userInfo = {
        countries: userData.countries || [],
        projects: userData.projects || [],
        equipmentPermissions: equipmentModule.permissions || [],
        employeeNumber: userData.employeeNumber || '',
        name: userData.name || '',
        userUid: userUid || user.uid,
      };

      await AsyncStorage.setItem('userInfo', JSON.stringify(userInfo));
      setUserInfo(userInfo);
      return userInfo;
    } catch (error) {
      console.error('Error fetching user info:', error);
      showToast('error', 'Error', 'Failed to fetch user permissions.');
      return null;
    }
  };

  const filterEquipmentByPermission = useCallback((equipmentArr, userInfoObj) => {
    if (!equipmentArr || !userInfoObj) return [];

    if (userInfoObj.equipmentPermissions.includes('onlyMineView')) {
      const creatorIdentifier = userInfoObj.userUid;
      return equipmentArr
        .filter(
          (item) =>
            (item.createdBy === creatorIdentifier || !item.createdBy) &&
            userInfoObj.countries.includes(item.country)
        )
        .map((item) => ({
          ...item,
          createdBy: item.createdBy || userInfoObj.userUid,
          equipmentName: item.equipmentName
            ? item.equipmentName.charAt(0).toUpperCase() + item.equipmentName.slice(1).toLowerCase()
            : item.equipmentName,
        }));
    } else if (userInfoObj.equipmentPermissions.includes('view')) {
      return equipmentArr
        .filter((item) => userInfoObj.countries.includes(item.country))
        .map((item) => ({
          ...item,
          equipmentName: item.equipmentName
            ? item.equipmentName.charAt(0).toUpperCase() + item.equipmentName.slice(1).toLowerCase()
            : item.equipmentName,
        }));
    } else {
      return [];
    }
  }, []);

  const loadEquipmentData = async () => {
    setIsLoading(true);
    try {
      const user = auth().currentUser;
      if (!user) {
        showToast('error', 'Error', 'User not authenticated. Please log in.');
        navigation.replace('AdminLogin');
        return;
      }

      const userInfoData = await fetchUserInfo();
      if (!userInfoData) {
        setIsLoading(false);
        return;
      }

      const isOnline = await checkNetworkStatus();
      let parsedData = [];

      if (isOnline) {
        if (userInfoData.equipmentPermissions.includes('onlyMineView')) {
          const equipmentRef = database().ref(`/equipment/${userInfoData.userUid}`);
          const snapshot = await equipmentRef.once('value');
          const equipmentData = snapshot.val() || {};

          parsedData = Object.keys(equipmentData).map((key) => ({
            ...equipmentData[key],
            firebaseKey: key,
          }));

          const userRef = database().ref(`/Globalusers/admins/${userInfoData.userUid}`);
          const userSnapshot = await userRef.once('value');
          const userData = userSnapshot.val();
          if (!userData || !userData.countries || userData.countries.length === 0) {
            parsedData = [];
          } else {
            parsedData = parsedData.filter((item) =>
              userData.countries.includes(item.country)
            );
          }
        } else {
          const globalRef = database().ref('EquipmentsDataJSON/GlobalEquipmentJsonData');
          const snapshot = await globalRef.once('value');
          const data = snapshot.val() || [];

          parsedData = Array.isArray(data)
            ? data.map((item, index) => ({
                ...item,
                firebaseKey: `equip_${index}`,
              }))
            : [];
        }

        parsedData = parsedData
          .filter((item, index, self) => self.findIndex((t) => t.id === item.id) === index)
          .sort((a, b) => {
            const dateA = a.updatedAt ? new Date(a.updatedAt) : new Date(a.createdAt);
            const dateB = b.updatedAt ? new Date(b.updatedAt) : new Date(b.createdAt);
            return dateB - dateA;
          });

        const filteredData = filterEquipmentByPermission(parsedData, userInfoData);
        parsedData = filteredData;
        await AsyncStorage.setItem('equipment', JSON.stringify(parsedData));
      } else {
        const savedData = await AsyncStorage.getItem('equipment');
        parsedData = savedData ? JSON.parse(savedData) : [];
        const filteredData = filterEquipmentByPermission(parsedData, userInfoData);
        parsedData = filteredData;
        parsedData.sort((a, b) => {
          const dateA = a.updatedAt ? new Date(a.updatedAt) : new Date(a.createdAt);
          const dateB = b.updatedAt ? new Date(b.updatedAt) : new Date(b.createdAt);
          return dateB - dateA;
        });
        showToast('info', 'Offline', 'Equipment data loaded from local storage.');
      }

      const storedCountry = await AsyncStorage.getItem('selectedCountry');
      setSelectedCountry(storedCountry || 'All Countries');

      let finalEquipment = parsedData;
      if (storedCountry && storedCountry !== 'All Countries') {
        finalEquipment = parsedData.filter((item) => item.country === storedCountry);
      }

      setEquipmentList(finalEquipment);
    } catch (error) {
      console.error('Error loading equipment:', error);
      showToast('error', 'Error', 'Failed to load equipment data.');
    } finally {
      setIsLoading(false);
    }
  };

  useEffect(() => {
    loadEquipmentData();
  }, []);

  const calculateKPIs = () => {
    const totalEquipments = equipmentList.length;
    const activeEquipments = equipmentList.filter(
      (item) => item.status === 'Available' || item.status === 'Deployed'
    ).length;
    const inMaintenance = equipmentList.filter(
      (item) => item.status === 'In Maintenance'
    ).length;
    const maintenanceRequired = equipmentList.filter(
      (item) => item.status === 'Maintenance Required'
    ).length;
    const allBreakdowns = equipmentList.filter(
      (item) => item.status === 'Breakdown'
    ).length;
    const deployed = equipmentList.filter(
      (item) => item.status === 'Deployed'
    ).length;
    const teamAllBreakdowns = allBreakdowns; // Team KPI based on breakdowns

    return {
      totalEquipments,
      activeEquipments,
      inMaintenance,
      maintenanceRequired,
      allBreakdowns,
      deployed,
      teamAllBreakdowns,
    };
  };

  const kpis = calculateKPIs();

  if (isLoading) {
    return (
      <>
        <LoaderModal visible={true} message="Loading KPI Data..." />
      </>
    );
  }

  // Modal for user info and equipment permission summary
  const renderPermissionModal = () => (
    <Modal visible={modalVisible} transparent animationType="fade">
      <View style={{ flex: 1, backgroundColor: 'rgba(0,0,0,0.5)', justifyContent: 'center', alignItems: 'center' }}>
        <View style={{ backgroundColor: '#fff', borderRadius: 10, padding: 24, alignItems: 'center', minWidth: '75%' }}>
          <Text style={{ fontSize: 20, color: '#015185', fontWeight: 'bold', marginBottom: 12 }}>User Permissions</Text>
          <View style={{ alignSelf: 'stretch', marginBottom: 10 }}>
            <Text style={{ fontWeight: 'bold', color: '#333', fontSize: 16 }}>Countries:</Text>
            <Text style={{ color: '#015185', fontSize: 15, marginBottom: 6 }}>{userInfo.countries.join(', ') || 'None'}</Text>
            <Text style={{ fontWeight: 'bold', color: '#333', fontSize: 16 }}>Projects:</Text>
            <Text style={{ color: '#015185', fontSize: 15, marginBottom: 6 }}>{userInfo.projects.join(', ') || 'None'}</Text>
            <Text style={{ fontWeight: 'bold', color: '#333', fontSize: 16 }}>Equipment Permissions:</Text>
            <Text style={{ color: '#015185', fontSize: 15, marginBottom: 10 }}>{userInfo.equipmentPermissions.join(', ') || 'None'}</Text>
          </View>
          <View style={{ alignSelf: 'stretch', marginBottom: 10 }}>
            <Text style={{ fontWeight: 'bold', color: '#015185', fontSize: 16, marginBottom: 6 }}>Equipment Status Summary</Text>
            <View style={{ flexDirection: 'row', justifyContent: 'space-between', backgroundColor: '#F5F5F5', borderRadius: 8, padding: 10, marginBottom: 8 }}>
              <View style={{ alignItems: 'center' }}>
                <MaterialIcons name="view-list" size={22} color="#015185" />
                <Text style={{ fontSize: 12, color: '#333' }}>Total</Text>
                <Text style={{ fontWeight: 'bold', color: '#015185' }}>{kpis.totalEquipments}</Text>
              </View>
              <View style={{ alignItems: 'center' }}>
                <MaterialIcons name="check-circle" size={22} color="#015185" />
                <Text style={{ fontSize: 12, color: '#333' }}>Available</Text>
                <Text style={{ fontWeight: 'bold', color: '#015185' }}>{equipmentList.filter(item => item.status === 'Available').length}</Text>
              </View>
              <View style={{ alignItems: 'center' }}>
                <MaterialIcons name="build" size={22} color="#015185" />
                <Text style={{ fontSize: 12, color: '#333' }}>In Maint.</Text>
                <Text style={{ fontWeight: 'bold', color: '#015185' }}>{kpis.inMaintenance}</Text>
              </View>
              <View style={{ alignItems: 'center' }}>
                <MaterialIcons name="warning" size={22} color="#015185" />
                <Text style={{ fontSize: 12, color: '#333' }}>Maint. Req.</Text>
                <Text style={{ fontWeight: 'bold', color: '#015185' }}>{kpis.maintenanceRequired}</Text>
              </View>
              <View style={{ alignItems: 'center' }}>
                <MaterialIcons name="cancel" size={22} color="#015185" />
                <Text style={{ fontSize: 12, color: '#333' }}>Breakdown</Text>
                <Text style={{ fontWeight: 'bold', color: '#015185' }}>{kpis.allBreakdowns}</Text>
              </View>
              <View style={{ alignItems: 'center' }}>
                <MaterialIcons name="flight-takeoff" size={22} color="#015185" />
                <Text style={{ fontSize: 12, color: '#333' }}>Deployed</Text>
                <Text style={{ fontWeight: 'bold', color: '#015185' }}>{kpis.deployed}</Text>
              </View>
            </View>
          </View>
          <TouchableOpacity style={{ backgroundColor: '#3481BC', paddingVertical: 10, paddingHorizontal: 30, borderRadius: 8, marginTop: 10 }} onPress={() => setModalVisible(false)}>
            <Text style={{ color: '#fff', fontWeight: 'bold', fontSize: 16 }}>Close</Text>
          </TouchableOpacity>
        </View>
      </View>
    </Modal>
  );

  return (
    <View style={styles.container}>
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
        start={{ x: 0, y: 0 }}
        end={{ x: 1, y: 0 }}
      >
        <StatusBar
          translucent
          backgroundColor="transparent"
          barStyle="light-content"
        />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity onPress={() => navigation.goBack()}>
              <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>KPI's</Text>
          </View>
         <TouchableOpacity onPress={() => setModalVisible(true)}>
           <MaterialIcons name="lock" size={width * 0.06} color="#fff" />
         </TouchableOpacity>
        </View>
      </LinearGradient>

      <ScrollView style={styles.content}>
        <View style={styles.section}>
          <View style={styles.sectionHeader}>
            <Text style={styles.sectionTitle}>Equipments Summary</Text>
          </View>
          <View style={styles.item}>
            <MaterialIcons name="view-list" size={width * 0.06} color="#015185" />
            <Text style={styles.label}>Total Equipments</Text>
            <Text style={styles.value}>{kpis.totalEquipments}</Text>
          </View>
          <View style={styles.item}>
            <MaterialIcons name="check-circle" size={width * 0.06} color="#015185" />
            <Text style={styles.label}>Available</Text>
            <Text style={styles.value}>{equipmentList.filter(item => item.status === 'Available').length}</Text>
          </View>
          <View style={styles.item}>
            <MaterialIcons name="build" size={width * 0.06} color="#015185" />
            <Text style={styles.label}>In Maintenance</Text>
            <Text style={styles.value}>{kpis.inMaintenance}</Text>
          </View>
          <View style={styles.item}>
            <MaterialIcons name="warning" size={width * 0.06} color="#015185" />
            <Text style={styles.label}>Maintenance Required</Text>
            <Text style={styles.value}>{kpis.maintenanceRequired}</Text>
          </View>
          <View style={styles.item}>
            <MaterialIcons name="cancel" size={width * 0.06} color="#015185" />
            <Text style={styles.label}>Breakdown</Text>
            <Text style={styles.value}>{kpis.allBreakdowns}</Text>
          </View>
          <View style={styles.item}>
            <MaterialIcons name="flight-takeoff" size={width * 0.06} color="#015185" />
            <Text style={styles.label}>Deployed</Text>
            <Text style={styles.value}>{kpis.deployed}</Text>
          </View>
        </View>

        {/* Equipment List Section (exclude Available) */}
        <View style={styles.section}>
          <View style={styles.sectionHeader}>
            <Text style={styles.sectionTitle}>Equipment List (Non-Available)</Text>
          </View>
          {equipmentList.filter(item => item.status !== 'Available').length === 0 ? (
            <Text style={{ color: '#888', textAlign: 'center', marginVertical: 10 }}>No non-available equipment found.</Text>
          ) : (
            equipmentList.filter(item => item.status !== 'Available').map((item, idx) => (
              <View key={item.id || idx} style={{ flexDirection: 'row', alignItems: 'center', paddingVertical: 8, borderBottomWidth: 1, borderBottomColor: '#eee' }}>
                <MaterialIcons name="build" size={22} color="#015185" style={{ marginRight: 10 }} />
                <View style={{ flex: 1 }}>
                  <Text style={{ fontSize: 15, color: '#015185', fontWeight: '600' }}>{item.equipmentName || item.equipmentNumber || 'Unnamed'}</Text>
                  <Text style={{ fontSize: 13, color: '#666' }}>Status: <Text style={{ color: getStatusColor(item.status), fontWeight: 'bold' }}>{item.status || 'N/A'}</Text></Text>
                </View>
              </View>
            ))
          )}
        </View>

        <View style={styles.section}>
          <View style={styles.sectionHeader}>
            <Text style={styles.sectionTitle}>Team KPI</Text>
          </View>
          <View style={styles.item}>
            <MaterialIcons name="warning" size={width * 0.06} color="#015185" />
            <Text style={styles.label}>All Breakdowns</Text>
            <Text style={styles.value}>{kpis.teamAllBreakdowns}</Text>
          </View>
        </View>
      </ScrollView>
      <Toast />
     {renderPermissionModal()}
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#ECF0F3',
  },
  header: {
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.045,
    fontWeight: '500',
    marginLeft: width * 0.025,
  },
  content: {
    flex: 1,
    padding: width * 0.04,
  },
  section: {
    backgroundColor: '#F5F5F5',
    borderRadius: width * 0.02,
    marginBottom: width * 0.04,
    padding: width * 0.04,
    elevation: 1,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 1 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
  },
  sectionHeader: {
    paddingBottom: width * 0.02,
    borderBottomWidth: 1,
    borderBottomColor: '#E0E0E0',
    marginBottom: width * 0.02,
  },
  sectionTitle: {
    fontSize: width * 0.045,
    color: '#015185',
    fontWeight: '600',
  },
  item: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: width * 0.035,
    borderBottomWidth: 1,
    borderBottomColor: '#E0E0E0',
  },
  label: {
    flex: 1,
    fontSize: width * 0.038,
    color: '#333333',
    marginLeft: width * 0.03,
    fontWeight: '400',
  },
  value: {
    fontSize: width * 0.038,
    color: '#015185',
    fontWeight: '500',
    marginLeft: width * 0.02,
  },
  loadingContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
  },
  loadingText: {
    fontSize: width * 0.04,
    color: '#333',
  },
});

export default Masterdatakpi;