import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  TextInput,
  Image,
  Dimensions,
  Animated,
  Modal,
  FlatList,
  ScrollView,
  KeyboardAvoidingView,
  Platform,
  RefreshControl,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation, useRoute } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import database from '@react-native-firebase/database';
import NetInfo from '@react-native-community/netinfo';
import Toast from 'react-native-toast-message';

const { width } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.3;
const MARGIN = 8;

// Reusing CustomInputField from UpdateRequestMaintenance
const CustomInputField = ({
  label,
  value,
  onChangeText,
  isRequired = false,
  showError = false,
  errorMessage = '',
  icon,
  onIconPress,
  editable = true,
  showSelectButton = false,
  onSelectPress,
  noBackground = false,
  multiline = false,
  tags = [],
  onRemoveTag,
  noBorder = false,
  ...props
}) => (
  <View style={inputStyles.container}>
    <View style={inputStyles.labelContainer}>
      <Text style={inputStyles.label}>
        {label}
        {isRequired && <Text style={inputStyles.requiredError}> *</Text>}
      </Text>
      {showSelectButton && (
        <TouchableOpacity style={inputStyles.labelSelectButton} onPress={onSelectPress}>
          <Text style={inputStyles.selectText}>Select</Text>
          <MaterialIcons name="chevron-right" size={width * 0.05} color="#015185" />
        </TouchableOpacity>
      )}
    </View>
    <View style={inputStyles.inputContainer}>
      {tags.length > 0 && (
        <View style={inputStyles.tagsContainer}>
          {tags.map((item, index) => (
            <View key={index} style={inputStyles.tag}>
              <Text style={inputStyles.tagText}>{item.name}</Text>
              {onRemoveTag && (
                <TouchableOpacity
                  style={inputStyles.tagCloseButton}
                  onPress={() => onRemoveTag(item)}
                >
                  <View style={inputStyles.tagCloseCircle}>
                    <MaterialIcons name="close" size={width * 0.035} color="#fff" />
                  </View>
                </TouchableOpacity>
              )}
            </View>
          ))}
        </View>
      )}
      <TextInput
        style={[
          inputStyles.input,
          showError ? inputStyles.inputError : null,
          !editable && inputStyles.disabledInput,
          noBackground && inputStyles.noBackground,
          multiline && inputStyles.multilineInput,
          tags.length > 0 && inputStyles.inputWithTags,
          noBorder && inputStyles.noBorder,
        ]}
        value={value}
        onChangeText={onChangeText}
        editable={editable}
        placeholderTextColor="#888"
        multiline={multiline}
        {...props}
      />
      {icon && (
        <TouchableOpacity style={inputStyles.inputIcon} onPress={onIconPress}>
          {icon}
        </TouchableOpacity>
      )}
    </View>
    {showError && <Text style={inputStyles.errorText}>{errorMessage}</Text>}
  </View>
);

const ViewMRequest = () => {
  const navigation = useNavigation();
  const route = useRoute();
  const { requestId } = route.params || {}; // Get requestId from navigation params
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [workOrderTitle, setWorkOrderTitle] = useState('');
  const [memo, setMemo] = useState('');
  const [priority, setPriority] = useState(null);
  const [assignTo, setAssignTo] = useState([]);
  const [selectedRequestType, setSelectedRequestType] = useState(null);
  const [selectedEquipment, setSelectedEquipment] = useState(null);
  const [tasks, setTasks] = useState([]);
  const [requestNumber, setRequestNumber] = useState('');
  const [userId, setUserId] = useState(null);
  const [country, setCountry] = useState('');
  const [project, setProject] = useState('');
  const [status, setStatus] = useState('');
  const [note, setNote] = useState('');
  const [createdAt, setCreatedAt] = useState('');
  const [isRefreshing, setIsRefreshing] = useState(false);

  const priorityOptions = [
    { label: 'Low', image: require('../../../Images/adminhome/inspreport/Priority1.png') },
    { label: 'Medium', image: require('../../../Images/adminhome/inspreport/Priority2.png') },
    { label: 'High', image: require('../../../Images/adminhome/inspreport/Priority3.png') },
    { label: 'Emergency', image: require('../../../Images/adminhome/inspreport/Priority4.png') },
  ];

  // Loader animation state
  const rotateAnim = useRef(new Animated.Value(0)).current;
  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });
  const animationRef = useRef(null);

  const checkNetworkStatus = async () => {
    try {
      const netInfo = await NetInfo.fetch();
      return netInfo.isConnected;
    } catch (error) {
      console.error('Error checking network status:', error);
      return false;
    }
  };

  const showToast = (type, text1, text2) => {
    Toast.show({
      type,
      text1,
      text2,
      position: 'bottom',
      visibilityTime: 3000,
      autoHide: true,
    });
  };

  // Fetch user ID and request data
  const loadUserIdAndRequest = async () => {
    try {
      const uid = await AsyncStorage.getItem('userUid');
      if (uid) {
        setUserId(uid);
        if (requestId) {
          setSyncModalVisible(true);
          let animation;
          if (!animationRef.current) {
            animation = Animated.loop(
              Animated.timing(rotateAnim, {
                toValue: 1,
                duration: 1000,
                useNativeDriver: true,
              })
            );
            animationRef.current = animation;
            animation.start();
          } else {
            animationRef.current.start();
          }
          const isOnline = await checkNetworkStatus();
          let request = null;

          if (isOnline) {
            const snapshot = await database()
              .ref(`GlobalRequestMaintenancereport/admins/${uid}/${requestId}`)
              .once('value');
            request = snapshot.val();
            if (request) {
              const storedRequests = await AsyncStorage.getItem('globalMaintenanceRequests');
              let requests = storedRequests ? JSON.parse(storedRequests) : [];
              requests = requests.filter(req => req.id !== requestId);
              requests.push(request);
              await AsyncStorage.setItem('globalMaintenanceRequests', JSON.stringify(requests));
            }
          } else {
            const storedRequests = await AsyncStorage.getItem('globalMaintenanceRequests');
            const requests = storedRequests ? JSON.parse(storedRequests) : [];
            request = requests.find(req => req.id === requestId);
            showToast('info', 'Offline', 'Using offline data. Will sync when online.');
          }

          if (request) {
            setWorkOrderTitle(request.workOrderTitle || '');
            setMemo(request.memo || '');
            setPriority(request.priority || null);
            setAssignTo(request.assignTo || []);
            setSelectedRequestType(request.requestType || null);
            setSelectedEquipment(request.equipment || null);
            setTasks(request.tasks || []);
            setRequestNumber(request.requestNumber || '');
            setCountry(request.country || '');
            setProject(request.project || '');
            setStatus(request.status || 'Requested');
            setNote(request.note || '');
            setCreatedAt(request.createdAt || '');
          } else {
            showToast('error', 'Error', 'Maintenance request not found.');
            navigation.goBack();
          }
          setSyncModalVisible(false);
          if (animationRef.current) {
            animationRef.current.stop();
            rotateAnim.setValue(0);
          }
        } else {
          showToast('error', 'Error', 'No request ID provided.');
          navigation.goBack();
        }
      } else {
        showToast('error', 'Error', 'User not logged in. Please log in again.');
        navigation.replace('AdminLogin');
      }
    } catch (error) {
      console.error('Error fetching request data:', error);
      showToast('error', 'Error', 'Failed to load request data.');
      setSyncModalVisible(false);
      if (animationRef.current) {
        animationRef.current.stop();
        rotateAnim.setValue(0);
      }
      navigation.goBack();
    }
  };

  useEffect(() => {
    loadUserIdAndRequest();
    // Clean up animation on unmount
    return () => {
      if (animationRef.current) {
        animationRef.current.stop();
        rotateAnim.setValue(0);
      }
    };
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [navigation, requestId]);

  // Pull-to-refresh handler
  const handleRefresh = async () => {
    setIsRefreshing(true);
    await loadUserIdAndRequest();
    setIsRefreshing(false);
  };

  const handleCancel = () => {
    navigation.reset({
          index: 1,
          routes: [
            { name: 'MainApp' },
            { name: 'RequestmaintenanceHome' },
          ],
        });
  };

  return (
    <KeyboardAvoidingView
      style={styles.container}
      behavior={Platform.OS === 'ios' ? 'padding' : 'height'}
      keyboardVerticalOffset={Platform.OS === 'ios' ? 0 : 20}
    >
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity onPress={() => navigation.goBack()}>
              <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>View Request</Text>
          </View>
          <View style={styles.headerRight}>
            <TouchableOpacity
              style={styles.iconButton}
              onPress={handleCancel}
            >
              <Text style={styles.createButtonText}>Cancel</Text>
            </TouchableOpacity>
          </View>
        </View>
      </LinearGradient>

      <ScrollView
        style={styles.mainContent}
        keyboardShouldPersistTaps="handled"
        contentContainerStyle={{ paddingBottom: width * 0.1 }}
        refreshControl={
          <RefreshControl
            refreshing={isRefreshing}
            onRefresh={handleRefresh}
            tintColor="#015185"
            colors={['#015185']}
          />
        }
      >
        {/* Optional: Custom refresh button at the top of the content */}
        <TouchableOpacity
          style={{ alignSelf: 'center', marginVertical: 10 }}
          onPress={handleRefresh}
          disabled={isRefreshing}
        >
       
        </TouchableOpacity>
        <View style={styles.formGroup}>
          <View style={styles.requestNumberRow}>
            <Text style={styles.requestNumberLabel}>Request #</Text>
            <Text style={styles.requestNumberValue}>{requestNumber}</Text>
          </View>
        </View>

        <View style={styles.formGroup}>
          <View style={styles.selectRow}>
            <Text style={styles.selectLabel}>Status</Text>
            <Text style={styles.selectText}>{status}</Text>
          </View>
        </View>

        <View style={styles.formGroup}>
          <View style={styles.selectRow}>
            <Text style={styles.selectLabel}>Country</Text>
            <Text style={styles.selectText}>{country || 'N/A'}</Text>
          </View>
        </View>

        <View style={styles.formGroup}>
          <View style={styles.selectRow}>
            <Text style={styles.selectLabel}>Project</Text>
            <Text style={styles.selectText}>{project || 'N/A'}</Text>
          </View>
        </View>

        <View style={styles.formGroup}>
          <CustomInputField
            label="Work Order Title"
            value={workOrderTitle}
            editable={false}
            placeholder="Work Order Title"
          />
        </View>

        <View style={styles.formGroup}>
          <CustomInputField
            label="Memo"
            value={memo}
            editable={false}
            multiline
            placeholder="Memo"
          />
        </View>

        <View style={styles.formGroup}>
          <Text style={styles.label}>Priority</Text>
          <View style={styles.priorityContainer}>
            {priorityOptions.map((option) => {
              const isSelected = priority === option.label;
              return (
                <View key={option.label} style={styles.priorityItem}>
                  <View
                    style={[
                      styles.priorityButton,
                      isSelected ? styles.priorityButtonSelected : styles.priorityButtonUnselected,
                    ]}
                  >
                    <Image source={option.image} style={styles.priorityImage} />
                    <Text style={isSelected ? styles.priorityTextSelected : styles.priorityText}>
                      {option.label}
                    </Text>
                  </View>
                </View>
              );
            })}
          </View>
        </View>

        <View style={styles.formGroup}>
          <View style={styles.selectRow}>
            <Text style={styles.selectLabel}>Vehicle</Text>
            <Text style={styles.selectText}>
              {selectedEquipment?.equipmentName || 'No Vehicle Selected'}
            </Text>
          </View>
          {selectedEquipment && (
            <View style={{backgroundColor:'#f0f0f0', borderRadius:8, padding:10, marginTop:6}}>
              <Text style={{fontWeight:'bold', color:'#015185', fontSize:16}}>{selectedEquipment.equipmentName || 'N/A'}</Text>
              <Text style={{color:'#333'}}>Number: {selectedEquipment.equipmentNumber || 'N/A'}</Text>
              <Text style={{color:'#333'}}>Model: {selectedEquipment.model || 'N/A'}</Text>
              <Text style={{color:'#333'}}>Meter: {selectedEquipment.meterReading} {selectedEquipment.meterUnit}</Text>
              <Text style={{color:'#333'}}>Status: {selectedEquipment.status || 'N/A'}</Text>
            </View>
          )}
        </View>

        <View style={styles.formGroup}>
          <View style={styles.selectRow}>
            <Text style={styles.selectLabel}>Assign to</Text>
            <Text style={styles.selectText}>
              {assignTo.length > 0 ? assignTo.map(item => item.fullName || item.name).join(', ') : 'No Assignees Selected'}
            </Text>
          </View>
        </View>

        <View style={styles.formGroup}>
          <View style={styles.selectRow}>
            <Text style={styles.selectLabel}>Request Type</Text>
            <Text style={styles.selectText}>
              {selectedRequestType || 'No Request Type Selected'}
            </Text>
          </View>
        </View>

        <View style={styles.formGroup}>
          <View style={styles.selectRow}>
            <Text style={styles.selectLabel}>Created At</Text>
            <Text style={styles.selectText}>
              {createdAt ? new Date(createdAt).toLocaleString('en-GB', {
                day: '2-digit',
                month: 'short',
                hour: '2-digit',
                minute: '2-digit',
                hour12: true,
              }).replace(',', '') : 'N/A'}
            </Text>
          </View>
        </View>

        <View style={styles.formGroup}>
          <View style={styles.row}>
            <Text style={styles.label}>Tasks</Text>
          </View>
          {tasks.length > 0 ? (
            <View style={styles.taskListContainer}>
              {tasks.map((task) => (
                <View key={task.id} style={styles.taskCard}>
                  <View style={styles.taskCardHeader}>
                    <Text style={styles.taskCardTitle}>{task.title}</Text>
                    <View style={styles.taskCardStatusPill}>
                      <Text style={styles.taskCardStatusText}>Initiated</Text>
                    </View>
                  </View>
                  <Text style={styles.taskCardType}>{task.description || 'Repair'}</Text>
                  <MaterialIcons name="person" size={22} color="#888" style={{ marginTop: 2 }} />
                </View>
              ))}
            </View>
          ) : (
            <Text style={styles.selectedText}>No Tasks Available</Text>
          )}
        </View>

        <View style={styles.formGroup}>
          <View style={styles.selectRow}>
            <Text style={styles.selectLabel}>Note</Text>
            <Text style={styles.selectText}>{note || 'N/A'}</Text>
          </View>
        </View>
      </ScrollView>

      {/* Sync Modal */}
      <Modal visible={syncModalVisible} transparent animationType="fade">
        <View style={styles.syncModalContainer}>
          <View style={styles.syncModalContent}>
            <Text style={styles.syncModalText}>Please Wait</Text>
            <View style={styles.syncIcondiv}>
              <Animated.Image
                source={require('../../../Images/adminlogin/sync.png')}
                style={[
                  styles.syncIcon,
                  { transform: [{ rotate: rotateInterpolate }] },
                ]}
              />
              <Text style={styles.syncModalTextadd}>Loading Request Data...</Text>
            </View>
          </View>
        </View>
      </Modal>

      <Toast />
    </KeyboardAvoidingView>
  );
};

// Reusing inputStyles from UpdateRequestMaintenance
const inputStyles = StyleSheet.create({
  container: {
    marginBottom: width * 0.02,
  },
  labelContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 4,
  },
  label: {
    fontSize: width * 0.038,
    color: '#015185',
    fontWeight: '500',
  },
  requiredError: {
    color: 'red',
    fontWeight: 'bold',
  },
  inputContainer: {
    position: 'relative',
    flexDirection: 'row',
    alignItems: 'center',
    flexWrap: 'wrap',
  },
  tagsContainer: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    marginBottom: width * 0.02,
    paddingHorizontal: 12,
  },
  tag: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#e0e0e0',
    borderRadius: width * 0.05,
    paddingVertical: width * 0.01,
    paddingHorizontal: width * 0.02,
    marginRight: width * 0.02,
    marginBottom: width * 0.01,
  },
  tagText: {
    fontSize: width * 0.035,
    color: '#015185',
    marginRight: width * 0.01,
  },
  tagCloseButton: {
    padding: width * 0.005,
  },
  tagCloseCircle: {
    width: width * 0.035,
    height: width * 0.035,
    borderRadius: (width * 0.035) / 2,
    backgroundColor: '#d32f2f',
    justifyContent: 'center',
    alignItems: 'center',
  },
  input: {
    flex: 1,
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    paddingHorizontal: 12,
    paddingVertical: 14,
    fontSize: width * 0.04,
    backgroundColor: '#fff',
  },
  multilineInput: {
    height: width * 0.2,
    textAlignVertical: 'top',
  },
  inputError: {
    borderColor: 'red',
  },
  disabledInput: {
    backgroundColor: '#f5f5f5',
  },
  noBackground: {
    backgroundColor: 'transparent',
  },
  noBorder: {
    borderWidth: 0,
  },
  inputWithTags: {
    paddingTop: 10,
    paddingBottom: 10,
  },
  errorText: {
    color: 'red',
    fontSize: width * 0.032,
    marginTop: 2,
    marginLeft: 2,
  },
  inputIcon: {
    position: 'absolute',
    right: 12,
    top: '50%',
    transform: [{ translateY: -width * 0.03 }],
  },
  labelSelectButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: width * 0.01,
  },
  selectText: {
    fontSize: width * 0.038,
    color: '#015185',
    marginRight: width * 0.01,
  },
});

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.05,
    fontWeight: 'bold',
    marginLeft: width * 0.025,
  },
  iconButton: {
    padding: width * 0.012,
    marginLeft: width * 0.025,
    backgroundColor: '#fff',
    borderRadius: 5,
  },
  createButtonText: {
    color: '#015185',
    fontSize: width * 0.04,
    fontWeight: '500',
  },
  mainContent: {
    flex: 1,
  },
  formGroup: {
    marginBottom: width * 0.03,
    paddingHorizontal: width * 0.03,
    paddingVertical: width * 0.015,
  },
  label: {
    fontSize: width * 0.04,
    fontWeight: '600',
    color: '#333',
    marginBottom: width * 0.02,
  },
  requestNumberRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    backgroundColor: '#F2F2F2',
    borderRadius: 6,
    paddingHorizontal: 16,
    paddingVertical: 14,
    marginBottom: width * 0.02,
  },
  requestNumberLabel: {
    fontSize: width * 0.038,
    color: '#015185',
    fontWeight: '500',
  },
  requestNumberValue: {
    fontSize: width * 0.038,
    color: '#025383',
    fontWeight: 'bold',
  },
  priorityContainer: {
    flexDirection: 'row',
    justifyContent: 'space-around',
    marginBottom: width * 0.02,
    backgroundColor: '#fff',
    borderRadius: 5,
    width: '100%',
  },
  priorityItem: {
    alignItems: 'center',
  },
  priorityButton: {
    padding: 8,
    borderRadius: 5,
    borderWidth: 1,
    borderColor: '#015185',
    width: 85,
    height: 60,
    justifyContent: 'center',
    alignItems: 'center',
  },
  priorityButtonSelected: {
    backgroundColor: '#015185',
  },
  priorityButtonUnselected: {
    backgroundColor: '#FFFFFF',
  },
  priorityImage: {
    width: 24,
    height: 24,
  },
  priorityText: {
    fontSize: 10,
    color: '#333',
    marginTop: 3,
    textAlign: 'center',
  },
  priorityTextSelected: {
    fontSize: 10,
    color: '#fff',
    fontWeight: '600',
    marginTop: 3,
    textAlign: 'center',
  },
  row: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 30,
    height: 30,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  taskListContainer: {
    marginTop: width * 0.025,
    maxHeight: width * 0.3,
  },
  taskCard: {
    backgroundColor: '#fff',
    borderRadius: 8,
    padding: width * 0.035,
    marginBottom: width * 0.025,
    borderWidth: 1,
    borderColor: '#eee',
  },
  taskCardHeader: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
  },
  taskCardTitle: {
    fontSize: width * 0.04,
    color: '#222',
    fontWeight: '500',
    flex: 1,
  },
  taskCardStatusPill: {
    backgroundColor: '#bfc2c5',
    borderRadius: 16,
    paddingHorizontal: 14,
    paddingVertical: 4,
    marginLeft: 8,
  },
  taskCardStatusText: {
    color: '#fff',
    fontWeight: '500',
    fontSize: width * 0.035,
  },
  taskCardType: {
    color: '#e6b800',
    fontWeight: '500',
    fontSize: width * 0.037,
    marginTop: 6,
    marginBottom: 2,
  },
  selectRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    backgroundColor: '#F5F5F5',
    borderRadius: 4,
    paddingVertical: width * 0.035,
    paddingHorizontal: width * 0.03,
    marginBottom: width * 0.01,
  },
  selectLabel: {
    fontSize: width * 0.038,
    color: '#707173',
    fontWeight: '500',
  },
  selectText: {
    color: '#015185',
    fontSize: width * 0.038,
    fontWeight: '500',
  },
  selectedText: {
    fontSize: width * 0.035,
    color: '#333',
    marginTop: width * 0.01,
    textAlign: 'left',
  },
});

export default ViewMRequest;