import React, { useState, useEffect, useRef, useCallback } from 'react';
import {
  View,
  Text,
  Image,
  StyleSheet,
  SafeAreaView,
  TouchableOpacity,
  Animated,
  ActivityIndicator,
  Modal,
  ScrollView,
  Dimensions,
  KeyboardAvoidingView,
  Platform,
  ToastAndroid,
  Alert,
} from 'react-native';
import { useNavigation } from '@react-navigation/native';
import auth from '@react-native-firebase/auth';
import database from '@react-native-firebase/database';
import storage from '@react-native-firebase/storage';
import LinearGradient from 'react-native-linear-gradient';
import NetInfo from '@react-native-community/netinfo';
import { TextInput } from 'react-native-paper';
import { GoogleSignin } from '@react-native-google-signin/google-signin';
import Toast from 'react-native-toast-message';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import DropDownPicker from 'react-native-dropdown-picker';
import DateTimePicker from '@react-native-community/datetimepicker';
import { launchCamera, launchImageLibrary } from 'react-native-image-picker';
import { PermissionsAndroid } from 'react-native';
import AsyncStorage from '@react-native-async-storage/async-storage';

// Get screen dimensions for responsive design
const { width } = Dimensions.get('window');
const scale = (size) => (width / 375) * size;

// Define allowed projects by country (exact per requirement)
const ZAMBIA_PROJECTS = [
  { label: 'Kansanshi', value: 'Kansanshi' },
  { label: 'Kalumbila', value: 'Kalumbila' },
  { label: 'Kimteto', value: 'Kimteto' },
  { label: 'Kobold', value: 'Kobold' },
  { label: 'FQM Exploration', value: 'FQM Exploration' },
  { label: 'Mimosa', value: 'Mimosa' },
];
const DRC_PROJECTS = [
  { label: 'Lubumbashi', value: 'Lubumbashi' },
  { label: 'Musompo', value: 'Musompo' },
  { label: 'IME', value: 'IME' },
  { label: 'Kamoa', value: 'Kamoa' },
];
const ALL_POSSIBLE_PROJECTS = [...ZAMBIA_PROJECTS, ...DRC_PROJECTS];

// Reusable Gradient Button Component
const GradientButton = ({ onPress, title, style, isLoading }) => {
  return (
    <LinearGradient colors={['#015185', '#3481BC']} style={[styles.gradientButton, style]}>
      <TouchableOpacity style={styles.buttonTouchable} onPress={onPress} activeOpacity={0.8} disabled={isLoading}>
        {isLoading ? (
          <ActivityIndicator size="small" color="#fff" />
        ) : (
          <Text style={styles.googleButtonText}>{title}</Text>
        )}
      </TouchableOpacity>
    </LinearGradient>
  );
};

// Custom Animated Input Component
const AnimatedInput = ({
  label,
  value,
  onChangeText,
  secureTextEntry,
  keyboardType,
  autoCapitalize,
  leftIconName,
  rightIconName,
  onRightIconPress,
  onBlur,
  maxLength,
  inputRef,
  hasError,
  onFocus,
  editable = true,
}) => {
  const [isFocused, setIsFocused] = useState(false);
  const animatedLabel = useRef(new Animated.Value(value ? 1 : 0)).current;
  const animatedBorderWidth = useRef(new Animated.Value(1)).current;
  const animatedBorderColor = useRef(new Animated.Value(0)).current;

  useEffect(() => {
    Animated.parallel([
      Animated.timing(animatedLabel, {
        toValue: isFocused || value ? 1 : 0,
        duration: 200,
        useNativeDriver: false,
      }),
      Animated.timing(animatedBorderWidth, {
        toValue: isFocused ? 2 : 1,
        duration: 200,
        useNativeDriver: false,
      }),
      Animated.timing(animatedBorderColor, {
        toValue: hasError ? 2 : isFocused || value ? 1 : 0,
        duration: 200,
        useNativeDriver: false,
      }),
    ]).start();
  }, [isFocused, value, hasError]);

  const labelStyle = {
    position: 'absolute',
    left: scale(45),
    top: animatedLabel.interpolate({
      inputRange: [0, 1],
      outputRange: [scale(13), scale(-8)],
    }),
    fontSize: animatedLabel.interpolate({
      inputRange: [0, 1],
      outputRange: [scale(16), scale(12)],
    }),
    color: animatedLabel.interpolate({
      inputRange: [0, 1],
      outputRange: ['#888', '#015185'],
    }),
    backgroundColor: '#fff',
    paddingHorizontal: scale(4),
    zIndex: 2,
  };

  const borderColor = animatedBorderColor.interpolate({
    inputRange: [0, 1, 2],
    outputRange: ['#ddd', '#015185', '#ff0000'],
  });

  return (
    <View style={styles.inputContainer} ref={inputRef}>
      <Animated.View
        style={[
          styles.inputGradient,
          {
            borderWidth: animatedBorderWidth,
            borderColor: borderColor,
          },
        ]}
      >
        <Animated.Text style={labelStyle}>{label}</Animated.Text>
        <TextInput
          mode="outlined"
          value={value}
          onChangeText={onChangeText}
          secureTextEntry={secureTextEntry}
          keyboardType={keyboardType}
          autoCapitalize={autoCapitalize}
          style={styles.input}
          onFocus={() => {
            setIsFocused(true);
            onFocus && onFocus();
          }}
          onBlur={() => {
            setIsFocused(false);
            onBlur && onBlur();
          }}
          editable={editable}
          left={
            leftIconName && (
              <TextInput.Icon
                icon={() => <MaterialIcons name={leftIconName} size={scale(24)} color="#015185" />}
              />
            )
          }
          right={
            rightIconName && (
              <TextInput.Icon
                icon={() => (
                  <TouchableOpacity onPress={onRightIconPress}>
                    <MaterialIcons name={rightIconName} size={scale(24)} color="#015185" />
                  </TouchableOpacity>
                )}
              />
            )
          }
          outlineStyle={{
            borderWidth: 0,
            borderRadius: scale(6),
          }}
          theme={{
            colors: {
              primary: 'transparent',
              text: '#333',
              placeholder: '#888',
            },
          }}
          maxLength={maxLength}
        />
      </Animated.View>
    </View>
  );
};

// SyncModal Component
const SyncModal = ({ visible, onClose, message }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
            />
            <Text style={styles.syncModalTextadd}>{message}</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

// Helper function for platform-specific toasts
const showToast = (type, title, message) => {
  if (Platform.OS === 'android') {
    ToastAndroid.showWithGravity(message, ToastAndroid.LONG, ToastAndroid.BOTTOM);
  } else {
    // Map 'warn' type to 'info' since 'warn' doesn't exist in react-native-toast-message
    const toastType = type === 'warn' ? 'info' : type;
    Toast.show({
      type: toastType,
      text1: title,
      text2: message,
      position: 'bottom',
    });
  }
};

const Basic_Details_TeamTab1 = ({ setActiveStep, teamData = {}, updateTeamData }) => {
  const navigation = useNavigation();
  // Authentication state
  const [name, setName] = useState(teamData.name || '');
  const [email, setEmail] = useState(teamData.email || '');
  const [password, setPassword] = useState(teamData.password || '');
  const [mobile, setMobile] = useState(teamData.mobile || '');
  const [showPassword, setShowPassword] = useState(false);
  const [nameError, setNameError] = useState(false);
  const [nameErrorMessage, setNameErrorMessage] = useState('');
  const [emailError, setEmailError] = useState(false);
  const [emailErrorMessage, setEmailErrorMessage] = useState('');
  const [passwordError, setPasswordError] = useState(false);
  const [passwordErrorMessage, setPasswordErrorMessage] = useState('');
  const [mobileError, setMobileError] = useState(false);
  const [mobileErrorMessage, setMobileErrorMessage] = useState('');
  const [isLoadingSignUp, setIsLoadingSignUp] = useState(false);
  const [isNetworkError, setIsNetworkError] = useState(false);
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [syncModalMessage, setSyncModalMessage] = useState('');
  const [successModalVisible, setSuccessModalVisible] = useState(false);
  const [successMessage, setSuccessMessage] = useState('');
  const [profilePic, setProfilePic] = useState(teamData.profilePic || null);
  const [department, setDepartment] = useState(teamData.department || '');
  const [nrcIdNumber, setNrcIdNumber] = useState(teamData.nrcIdNumber || '');
  const [engagementDate, setEngagementDate] = useState(
    teamData.engagementDate ? new Date(teamData.engagementDate) : new Date()
  );
  const [titanInductionDate, setTitanInductionDate] = useState(
    teamData.titanInductionDate ? new Date(teamData.titanInductionDate) : new Date()
  );
  const [clientInduction, setClientInduction] = useState(
    teamData.clientInduction ? new Date(teamData.clientInduction) : new Date()
  );
  const [dateExamined, setDateExamined] = useState(
    teamData.dateExamined ? new Date(teamData.dateExamined) : new Date()
  );
  const [expiryMedicals, setExpiryMedicals] = useState(
    teamData.expiryMedicals ? new Date(teamData.expiryMedicals) : new Date()
  );
  const [driverLicenseNumber, setDriverLicenseNumber] = useState(teamData.driverLicenseNumber || '');
  const [driverLicenseExpiry, setDriverLicenseExpiry] = useState(
    teamData.driverLicenseExpiry ? new Date(teamData.driverLicenseExpiry) : new Date()
  );
  const [clientDrivingDate, setClientDrivingDate] = useState(
    teamData.clientDrivingDate ? new Date(teamData.clientDrivingDate) : new Date()
  );
  const [dateCompletedFirstAid, setDateCompletedFirstAid] = useState(
    teamData.dateCompletedFirstAid ? new Date(teamData.dateCompletedFirstAid) : new Date()
  );
  const [levelFirstAid, setLevelFirstAid] = useState(teamData.levelFirstAid || '');
  const [expiryDateFirstAid, setExpiryDateFirstAid] = useState(
    teamData.expiryDateFirstAid ? new Date(teamData.expiryDateFirstAid) : new Date()
  );
  const [driverLicenseCode, setDriverLicenseCode] = useState(teamData.driverLicenseCode || '');
  const [uploadModalVisible, setUploadModalVisible] = useState(false);
  const [currentUploadField, setCurrentUploadField] = useState('');
  const [isUploading, setIsUploading] = useState(false);
  const [showDatePicker, setShowDatePicker] = useState(false);
  const [datePickerValue, setDatePickerValue] = useState(new Date());
  const [currentEditingDateField, setCurrentEditingDateField] = useState('');
  const [datePickerMaxDate, setDatePickerMaxDate] = useState(null);
  const [datePickerMinDate, setDatePickerMinDate] = useState(new Date(1900, 0, 1));
  const [departmentError, setDepartmentError] = useState(false);
  const [departmentErrorMessage, setDepartmentErrorMessage] = useState('');
  const [nrcIdNumberError, setNrcIdNumberError] = useState(false);
  const [nrcIdNumberErrorMessage, setNrcIdNumberErrorMessage] = useState('');
  const [employeeNumber, setEmployeeNumber] = useState(teamData.employeeNumber || '');
  const [employeeNumberError, setEmployeeNumberError] = useState(false);
  const [employeeNumberErrorMessage, setEmployeeNumberErrorMessage] = useState('');

  const networkErrorAnim = useRef(new Animated.Value(-scale(50))).current;
  const scrollViewRef = useRef(null);
  const nameInputRef = useRef(null);
  const emailInputRef = useRef(null);
  const passwordInputRef = useRef(null);
  const mobileInputRef = useRef(null);
  const departmentInputRef = useRef(null);
  const nrcIdNumberInputRef = useRef(null);
  const clientInductionInputRef = useRef(null);
  const driverLicenseNumberInputRef = useRef(null);
  const levelFirstAidInputRef = useRef(null);
  const employeeNumberInputRef = useRef(null);

  // Dropdown state
  const [countries, setCountries] = useState(teamData.countries || []);
  const [countryOpen, setCountryOpen] = useState(false);
  const [countryItems, setCountryItems] = useState([
    { label: 'Zambia', value: 'Zambia' },
    { label: 'DRC', value: 'DRC' },
  ]);

  const [projects, setProjects] = useState(teamData.projects || []);
  const [projectOpen, setProjectOpen] = useState(false);
  const [projectItems, setProjectItems] = useState(ALL_POSSIBLE_PROJECTS);

  const [modules, setModules] = useState(teamData.modules || []);
  const [moduleOpen, setModuleOpen] = useState(false);
  const [moduleItems, setModuleItems] = useState([
    { label: 'Equipments', value: 'Equipments' },
    { label: 'Documents', value: 'Documents' },
    { label: 'Team Employ', value: 'teamEmploy' },
    { label: 'Inspection Report', value: 'inspectionReport' },
    { label: 'DVIR Report', value: 'dvirReport' },
    { label: 'Checklist', value: 'checklist' },
    { label: 'Incident Report', value: 'incidentReport' },
    { label: 'Daily Report', value: 'dailyReport' },
    { label: 'Request Maintenance', value: 'requestMaintenance' },
    { label: 'Jobcard', value: 'jobcard' },
    { label: 'Service Schedule', value: 'serviceSchedule' },
    { label: 'Fuel Log', value: 'fuelLog' },
    { label: 'Logbook', value: 'logbook' },
    { label: 'HR Form', value: 'hrForm' },
    { label: 'HR Applications', value: 'hrApplications' },
    { label: 'Team KPI', value: 'teamKpi' },
  ]);

  const [role, setRole] = useState(teamData.role || 'Operator');
  const [roleOpen, setRoleOpen] = useState(false);
  const roles = [
    { label: 'Admin', value: 'ProAdmin' },
    { label: 'CEO', value: 'CEO' },
    { label: 'Country Manager', value: 'Country Manager' },
    { label: 'Operations Manager', value: 'Operations Manager' },
    { label: 'Health & Safety Manager', value: 'Health & Safety Manager' },
    { label: 'Project Manager', value: 'Project Manager' },
    { label: 'Site Manager', value: 'Site Manager' },
    { label: 'Supervisor', value: 'Supervisor' },
    { label: 'HrManager', value: 'HrManager' },
    { label: 'FinanceManager', value: 'FinanceManager' },
    { label: 'Driller', value: 'Driller' },
    { label: 'Trainee Driller', value: 'Trainee Driller' },
    { label: 'Junior Driller', value: 'Junior Driller' },
    { label: 'Coremarker', value: 'Coremarker' },
    { label: 'Assistant', value: 'Assistant' },
    { label: 'Workshop Manager', value: 'Workshop Manager' },
    { label: 'Workshop Foreman', value: 'Workshop Foreman' },
    { label: 'Maintenance Manager', value: 'Maintenance Manager' },
    { label: 'Mechanical Engineer', value: 'Mechanical Engineer' },
    { label: 'Boilermaker', value: 'Boilermaker' },
    { label: 'Welder', value: 'Welder' },
    { label: 'Mechanic', value: 'Mechanic' },
    { label: 'Machinist', value: 'Machinist' },
    { label: 'Electrician', value: 'Electrician' },
    { label: 'Housekeeper', value: 'Housekeeper' },
    { label: 'Cook', value: 'Cook' },
    { label: 'Stores Officer', value: 'Stores Officer' },
    { label: 'General Worker', value: 'General Worker' },
    
  ];

  // Update available projects when countries change
  useEffect(() => {
    const projectMap = {
      Zambia: ZAMBIA_PROJECTS,
      DRC: DRC_PROJECTS,
    };
    const merged = (countries || [])
      .map((c) => projectMap[c] || [])
      .flat()
      .reduce((acc, item) => {
        if (!acc.find((i) => i.value === item.value)) acc.push(item);
        return acc;
      }, []);
    const nextItems = merged.length > 0 ? merged : [];
    setProjectItems(nextItems);
    // Keep only valid selected projects
    setProjects((prev) => prev.filter((p) => nextItems.find((i) => i.value === p)));
  }, [countries]);

  // Initialize permissions for new modules (only if completely empty)
  useEffect(() => {
    const newPermissions = { ...modulePermissions };
    let hasChanges = false;

    modules.forEach((mod) => {
      if (!newPermissions[mod]) {
        newPermissions[mod] = []; // Start with empty permissions, let user choose
        hasChanges = true;
      }
      // Always update the ref
      permissionRefs.current[mod] = newPermissions[mod] || [];
    });

    if (hasChanges) {
      setModulePermissions(newPermissions);
    }
  }, [modules]);

  const permissionOptions = [
    { label: 'All Permissions', value: 'all' },
    { label: 'Create', value: 'create' },
    { label: 'Edit', value: 'edit' },
    { label: 'Delete', value: 'delete' },
    { label: 'View', value: 'view' },
    { label: 'Update', value: 'update' },
    { label: 'Only Mine View', value: 'onlyMineView' },
  ];

  const individualPermissions = ['create', 'edit', 'delete', 'view', 'update', 'onlyMineView'];

  const [modulePermissions, setModulePermissions] = useState(teamData.modulePermissions || {});
  const [permissionOpen, setPermissionOpen] = useState({});
  const permissionRefs = useRef({});
  const [errors, setErrors] = useState({
    country: false,
    project: false,
    module: false,
    permissions: {},
    role: false,
    department: false,
    nrcIdNumber: false,
  });

  // Network error animation
  const showNetworkError = () => {
    setIsNetworkError(true);
    Animated.timing(networkErrorAnim, {
      toValue: 0,
      duration: 300,
      useNativeDriver: true,
    }).start();

    setTimeout(() => {
      Animated.timing(networkErrorAnim, {
        toValue: -scale(50),
        duration: 300,
        useNativeDriver: true,
      }).start(() => setIsNetworkError(false));
    }, 3000);
  };

  // Configure Google Sign-In
  useEffect(() => {
    GoogleSignin.configure({
      webClientId: '749788854975-rhkmepo20c1n038ftnreustj8br62qbj.apps.googleusercontent.com',
      androidClientId: '749788854975-1vqr2rje9a19hl56ohheqvr32hl6045p.apps.googleusercontent.com',
      iosClientId: '749788854975-fsan3f653h4fffue813uqc38h6mu7ipe.apps.googleusercontent.com',
      offlineAccess: true,
    });
  }, []);

  // Update module permissions
  useEffect(() => {
    const updatedPermissions = { ...modulePermissions };
    const updatedErrors = { ...errors.permissions };

    modules.forEach((mod) => {
      updatedPermissions[mod] = updatedPermissions[mod] || [];
      updatedErrors[mod] = updatedPermissions[mod].length === 0;
    });

    Object.keys(updatedPermissions).forEach((mod) => {
      if (!modules.includes(mod)) {
        delete updatedPermissions[mod];
        delete updatedErrors[mod];
      }
    });

    setModulePermissions(updatedPermissions);
    setErrors((prev) => ({ ...prev, permissions: updatedErrors }));
  }, [modules]);

  // Debug permission changes and sync refs
  useEffect(() => {
    console.log('Module permissions updated:', modulePermissions);
    
    // Keep refs in sync with state
    Object.keys(modulePermissions).forEach((module) => {
      permissionRefs.current[module] = modulePermissions[module] || [];
    });
  }, [modulePermissions]);

  // Debug modal state changes and force show on iOS if needed
  useEffect(() => {
    console.log('Success modal state changed:', successModalVisible);
    
    // Force show modal on iOS if it should be visible but isn't showing
    if (Platform.OS === 'ios' && successModalVisible) {
      setTimeout(() => {
        if (!successModalVisible) {
          console.log('Forcing modal to show on iOS');
          setSuccessModalVisible(true);
        }
      }, 100);
    }
  }, [successModalVisible]);

  // Sync offline data and ensure admin password is stored
  useEffect(() => {
    const checkNetwork = async () => {
      const state = await NetInfo.fetch();
      if (state.isConnected) {
        syncOfflineData();
      }
    };

    const ensureAdminPasswordStored = async () => {
      try {
        // Check if admin password is already stored
        const storedPassword = await AsyncStorage.getItem('adminPassword');
        if (!storedPassword) {
          // Try to get admin password from current auth user
          const currentUser = auth().currentUser;
          if (currentUser && currentUser.email) {
            // Get admin data from database to retrieve password
            const adminRef = database().ref(`/Globalusers/admins/${currentUser.uid}`);
            const snapshot = await adminRef.once('value');
            const adminData = snapshot.val();
            
            if (adminData && adminData.password) {
              // Store admin password in AsyncStorage
              await AsyncStorage.setItem('adminPassword', adminData.password);
              await AsyncStorage.setItem('userEmail', adminData.email);
              console.log('Admin password stored from database');
            }
          }
        }
      } catch (error) {
        console.error('Error ensuring admin password storage:', error);
      }
    };

    checkNetwork();
    ensureAdminPasswordStored();

    const unsubscribe = NetInfo.addEventListener((state) => {
      if (state.isConnected) {
        syncOfflineData();
      }
    });

    return () => unsubscribe();
  }, []);

  const syncOfflineData = async () => {
    try {
      const pendingUploads = JSON.parse(await AsyncStorage.getItem('pendingUploads') || '[]');
      if (pendingUploads.length > 0) {
        for (const upload of pendingUploads) {
          const { adminUid, tempId, field, file } = upload;
          const reference = storage().ref(file.name);
          await reference.putFile(file.uri);
          const downloadURL = await reference.getDownloadURL();
          await database()
            .ref(`/TeamMember/temp/${adminUid}/${tempId}/${field}`)
            .set({ url: downloadURL, name: file.name, type: file.type });
        }
        await AsyncStorage.setItem('pendingUploads', '[]');
        showToast('success', 'Sync Success', 'Offline uploads have been synced.');
      }

      const pendingTeamData = JSON.parse(await AsyncStorage.getItem('pendingTeamData') || '[]');
      if (pendingTeamData.length > 0) {
        for (const data of pendingTeamData) {
          const { adminUid, tempId, data: teamData, authUid } = data;
          if (authUid) {
            // Store in the correct path if we have an authUid
            await database()
              .ref(`/Globalusers/admins/${authUid}`)
              .set(teamData);
          } else {
            // Fallback to temp storage if no authUid
            await database()
              .ref(`/TeamMember/temp/${adminUid}/${tempId}`)
              .set(teamData);
          }
        }
        await AsyncStorage.setItem('pendingTeamData', '[]');
        showToast('success', 'Sync Success', 'Offline team data has been synced.');
      }
    } catch (error) {
      console.error('Error syncing offline data:', error);
      showToast('error', 'Sync Error', 'Failed to sync offline data.');
    }
  };

  // Function to restore admin session after user creation
  const restoreAdminSession = async () => {
    try {
      // First try to get admin password from AsyncStorage (stored during login)
      const adminPassword = await AsyncStorage.getItem('adminPassword');
      const adminEmail = await AsyncStorage.getItem('userEmail');
      
      if (adminPassword && adminEmail) {
        console.log('Attempting to restore admin session for:', adminEmail);
        
        try {
          await auth().signInWithEmailAndPassword(adminEmail, adminPassword);
          console.log('Admin session restored successfully');
          showToast('success', 'Session Restored', 'Admin session restored successfully!');
          return true;
        } catch (signInError) {
          console.error('Failed to restore admin session:', signInError);
          showToast('info', 'Session Warning', 'Could not restore admin session. Please sign in again.');
        }
      } else {
        console.log('No admin password or email found in AsyncStorage');
        showToast('info', 'Session Info', 'Admin session info not found. You may need to sign in again.');
      }
      
      return false;
    } catch (error) {
      console.error('Error restoring admin session:', error);
      return false;
    }
  };

  // Function to check auth user creation status
  const checkAuthUserCreationStatus = async (reqId) => {
    try {
      const snapshot = await database().ref(`/AdminTasks/createAuthUser/${reqId}`).once('value');
      const data = snapshot.val();
      if (data) {
        console.log('Auth user creation status:', data.status);
        if (data.status === 'completed' && data.authUid) {
          console.log('Auth user created successfully with UID:', data.authUid);
          showToast('success', 'Auth User Created', `User account created successfully!\nAuth UID: ${data.authUid}`);
          return data.authUid;
        } else if (data.status === 'failed') {
          console.error('Auth user creation failed:', data.error);
          showToast('error', 'Auth User Failed', `User account creation failed: ${data.error || 'Unknown error'}`);
          return null;
        } else if (data.status === 'processing') {
          showToast('info', 'Auth User Processing', 'User account creation in progress...');
          return null;
        } else {
          showToast('info', 'Auth User Pending', 'User account creation is pending...');
          return null;
        }
      } else {
        showToast('info', 'Auth User Status', 'No auth user creation task found');
        return null;
      }
    } catch (error) {
      console.error('Error checking auth user creation status:', error);
      showToast('error', 'Status Check Error', 'Failed to check auth user creation status');
      return null;
    }
  };

  const checkNetworkStatus = async () => {
    try {
      const netInfo = await NetInfo.fetch();
      if (!netInfo.isConnected) {
        showNetworkError();
        showToast('error', 'Network Error', 'Internet connection not available');
        return false;
      }
      return true;
    } catch (error) {
      console.error('Network check error:', error);
      showNetworkError();
      showToast('error', 'Network Error', 'Failed to check network status');
      return false;
    }
  };

  // Validation functions
  const validateName = (name) => {
    const nameEmpty = !name.trim();
    const nameTooLong = name.length > 13;

    setNameError(nameEmpty || nameTooLong);
    if (nameEmpty) {
      setNameErrorMessage('Name is required');
    } else if (nameTooLong) {
      setNameErrorMessage('Name cannot exceed 13 characters');
    } else {
      setNameErrorMessage('');
    }

    return !(nameEmpty || nameTooLong);
  };

  const validateEmail = (email) => {
    const emailEmpty = !email.trim();
    const emailInvalid = emailEmpty ? false : !/\S+@\S+\.\S+/.test(email);
    const emailTooLong = email.length > 40;

    setEmailError(emailEmpty || emailInvalid || emailTooLong);
    if (emailEmpty) {
      setEmailErrorMessage('Email is required');
    } else if (emailInvalid) {
      setEmailErrorMessage('Enter a valid email address');
    } else if (emailTooLong) {
      setEmailErrorMessage('Email cannot exceed 40 characters');
    } else {
      setEmailErrorMessage('');
    }

    return !(emailEmpty || emailInvalid || emailTooLong);
  };

  const validatePassword = (password) => {
    const passwordEmpty = !password.trim();
    const passwordShort = passwordEmpty ? false : password.length < 8;
    const passwordNoSpecialChar = passwordEmpty ? false : !/[!@#$%^&*(),.?":{}|<>]/.test(password);
    const passwordTooLong = password.length > 15;

    setPasswordError(passwordEmpty || passwordShort || passwordNoSpecialChar || passwordTooLong);
    if (passwordEmpty) {
      setPasswordErrorMessage('Password is required');
    } else if (passwordShort) {
      setPasswordErrorMessage('Password must be at least 8 characters');
    } else if (passwordNoSpecialChar) {
      setPasswordErrorMessage('Password needs a special character');
    } else if (passwordTooLong) {
      setPasswordErrorMessage('Password cannot exceed 15 characters');
    } else {
      setPasswordErrorMessage('');
    }

    return !(passwordEmpty || passwordShort || passwordNoSpecialChar || passwordTooLong);
  };

  const validateMobile = (mobile) => {
    const mobileEmpty = !mobile.trim();
    const mobileInvalid = mobileEmpty ? false : !/^\+?[1-9]\d{9,14}$/.test(mobile);
    const mobileTooLong = mobile.length > 15;

    setMobileError(mobileEmpty || mobileInvalid || mobileTooLong);
    if (mobileEmpty) {
      setMobileErrorMessage('Mobile number is required');
    } else if (mobileInvalid) {
      setMobileErrorMessage('Enter a valid mobile number');
    } else if (mobileTooLong) {
      setMobileErrorMessage('Mobile number cannot exceed 15 characters');
    } else {
      setMobileErrorMessage('');
    }

    return !(mobileEmpty || mobileInvalid || mobileTooLong);
  };

  const validateDepartment = (department) => {
    const departmentEmpty = !department.trim();
    setDepartmentError(departmentEmpty);
    setDepartmentErrorMessage(departmentEmpty ? 'Department is required' : '');
    return !departmentEmpty;
  };

  const validateNrcIdNumber = (nrcIdNumber) => {
    const nrcEmpty = !nrcIdNumber.trim();
    setNrcIdNumberError(nrcEmpty);
    setNrcIdNumberErrorMessage(nrcEmpty ? 'NRC / ID Number is required' : '');
    return !nrcEmpty;
  };

  const validateEmployeeNumber = (number) => {
    const numberEmpty = !number.trim();
    const numberTooLong = number.length > 20;
    setEmployeeNumberError(numberEmpty || numberTooLong);
    if (numberEmpty) {
      setEmployeeNumberErrorMessage('Employee Number is required');
    } else if (numberTooLong) {
      setEmployeeNumberErrorMessage('Employee Number cannot exceed 20 characters');
    } else {
      setEmployeeNumberErrorMessage('');
    }
    return !(numberEmpty || numberTooLong);
  };

  const validateDropdowns = () => {
    const newErrors = {
      country: countries.length === 0,
      project: projects.length === 0,
      module: modules.length === 0,
      permissions: {},
      role: !role,
      department: !department.trim(),
      nrcIdNumber: !nrcIdNumber.trim(),
    };

    modules.forEach((mod) => {
      const perms = modulePermissions[mod] || [];
      newErrors.permissions[mod] = perms.length === 0;
    });

    setErrors(newErrors);

    return (
      !newErrors.country &&
      !newErrors.project &&
      !newErrors.module &&
      Object.values(newErrors.permissions).every((v) => v === false) &&
      !newErrors.role &&
      !newErrors.department &&
      !newErrors.nrcIdNumber
    );
  };

  // Date picker functions
  const formatDate = (date) => {
    if (!date || isNaN(date.getTime())) return '';
    const d = new Date(date);
    const year = d.getFullYear();
    const month = String(d.getMonth() + 1).padStart(2, '0');
    const day = String(d.getDate()).padStart(2, '0');
    return `${year}-${month}-${day}`;
  };

  const onDateChange = (event, selectedDate) => {
    try {
      if (Platform.OS === 'android') {
        // Android: Close picker immediately and handle date selection
        setShowDatePicker(false);
        
        if (event.type === 'dismissed') {
          setCurrentEditingDateField('');
          setDatePickerMaxDate(null);
          return;
        }

        const currentDate = selectedDate || datePickerValue;
        if (!currentDate || isNaN(currentDate.getTime())) {
          console.warn('Invalid date selected:', currentDate);
          return;
        }

        // Apply the selected date immediately for Android
        applySelectedDate(currentDate);
        
      } else {
        // iOS: Keep modal open, just update the picker value
        if (event.type === 'dismissed') {
          setShowDatePicker(false);
          setCurrentEditingDateField('');
          setDatePickerMaxDate(null);
          return;
        }

        const currentDate = selectedDate || datePickerValue;
        if (!currentDate || isNaN(currentDate.getTime())) {
          console.warn('Invalid date selected:', currentDate);
          return;
        }

        // Don't close the modal, just update the picker value
        setDatePickerValue(currentDate);
      }

    } catch (error) {
      console.error('Error in onDateChange:', error);
      showToast('error', 'Date Error', 'Failed to update date. Please try again.');
    }
  };

  // Function to apply selected date to the appropriate field
  const applySelectedDate = (selectedDate) => {
    try {
      if (!selectedDate || isNaN(selectedDate.getTime())) {
        showToast('error', 'Date Error', 'Please select a valid date');
        return;
      }

      // Apply the selected date to the appropriate field
      switch (currentEditingDateField) {
        case 'engagementDate':
          setEngagementDate(selectedDate);
          if (selectedDate > titanInductionDate) {
            showToast('error', 'Validation Error', 'Engagement Date must be before Titan Induction Date.');
          }
          if (selectedDate > driverLicenseExpiry) {
            showToast('error', 'Validation Error', "Engagement Date must be before Driver's License Expiry.");
          }
          break;
        case 'titanInductionDate':
          setTitanInductionDate(selectedDate);
          if (selectedDate < engagementDate) {
            showToast('error', 'Validation Error', 'Titan Induction Date must be after Engagement Date.');
          }
          break;
        case 'clientInduction':
          setClientInduction(selectedDate);
          if (selectedDate < engagementDate) {
            showToast('error', 'Validation Error', 'Client Induction Date must be after Engagement Date.');
          }
          break;
        case 'dateExamined':
          setDateExamined(selectedDate);
          if (selectedDate > expiryMedicals) {
            showToast('error', 'Validation Error', 'Date Examined must be before Medical Expiry Date.');
          }
          break;
        case 'expiryMedicals':
          setExpiryMedicals(selectedDate);
          if (selectedDate <= dateExamined) {
            showToast('error', 'Validation Error', 'Medical Expiry Date must be after Date Examined.');
          }
          break;
        case 'driverLicenseExpiry':
          setDriverLicenseExpiry(selectedDate);
          if (selectedDate <= engagementDate) {
            showToast('error', 'Validation Error', "Driver's License Expiry Date must be after Engagement Date.");
          }
          break;
        case 'clientDrivingDate':
          setClientDrivingDate(selectedDate);
          break;
        case 'dateCompletedFirstAid':
          setDateCompletedFirstAid(selectedDate);
          if (selectedDate > expiryDateFirstAid) {
            showToast('error', 'Validation Error', 'Date Completed must be before First Aid Expiry Date.');
          }
          break;
        case 'expiryDateFirstAid':
          setExpiryDateFirstAid(selectedDate);
          if (selectedDate <= dateCompletedFirstAid) {
            showToast('error', 'Validation Error', 'First Aid Expiry Date must be after Date Completed.');
          }
          break;
        default:
          break;
      }

      // Reset state
      setCurrentEditingDateField('');
      setDatePickerMaxDate(null);
      setDatePickerMinDate(new Date(1900, 0, 1));
      setDatePickerValue(new Date());
      
      showToast('success', 'Date Updated', 'Date has been updated successfully');
      
    } catch (error) {
      console.error('Error in applySelectedDate:', error);
      showToast('error', 'Date Error', 'Failed to update date. Please try again.');
    }
  };

  // Function to handle when Done button is pressed (iOS only)
  const handleDatePickerDone = () => {
    try {
      const selectedDate = datePickerValue;
      if (!selectedDate || isNaN(selectedDate.getTime())) {
        showToast('error', 'Date Error', 'Please select a valid date');
        return;
      }

      // Apply the selected date
      applySelectedDate(selectedDate);

      // Close the modal and reset state
      setShowDatePicker(false);
      
    } catch (error) {
      console.error('Error in handleDatePickerDone:', error);
      showToast('error', 'Date Error', 'Failed to update date. Please try again.');
    }
  };

  const showMode = (currentMode, field) => {
    try {
      if (!field) {
        console.warn('No field specified for date picker');
        return;
      }

      // Set minimum date based on field type
      let minDate = new Date(1900, 0, 1); // Default minimum
      let maxDate = null;
      
      if (['expiryMedicals', 'driverLicenseExpiry', 'expiryDateFirstAid'].includes(field)) {
        // Expiry dates must be from today onwards (future only)
        minDate = new Date();
        maxDate = null; // Allow future dates
      } else if (['titanInductionDate'].includes(field)) {
        // Induction dates can be from today onwards
        minDate = new Date();
        maxDate = null;
      } else {
        // Other dates can be past or today
        minDate = new Date(1900, 0, 1);
        maxDate = new Date(); // Restrict to today or past
      }
      
      setDatePickerMaxDate(maxDate);
      setDatePickerMinDate(minDate);

      let initialDate = new Date();
      switch (field) {
        case 'engagementDate':
          initialDate = engagementDate && !isNaN(engagementDate.getTime()) ? new Date(engagementDate) : new Date();
          break;
        case 'titanInductionDate':
          initialDate = titanInductionDate && !isNaN(titanInductionDate.getTime()) ? new Date(titanInductionDate) : new Date();
          break;
        case 'clientInduction':
          initialDate = clientInduction && !isNaN(clientInduction.getTime()) ? new Date(clientInduction) : new Date();
          break;
        case 'dateExamined':
          initialDate = dateExamined && !isNaN(dateExamined.getTime()) ? new Date(dateExamined) : new Date();
          break;
        case 'expiryMedicals':
          initialDate = expiryMedicals && !isNaN(expiryMedicals.getTime()) ? new Date(expiryMedicals) : new Date();
          break;
        case 'driverLicenseExpiry':
          initialDate = driverLicenseExpiry && !isNaN(driverLicenseExpiry.getTime()) ? new Date(driverLicenseExpiry) : new Date();
          break;
        case 'clientDrivingDate':
          initialDate = clientDrivingDate && !isNaN(clientDrivingDate.getTime()) ? new Date(clientDrivingDate) : new Date();
          break;
        case 'dateCompletedFirstAid':
          initialDate = dateCompletedFirstAid && !isNaN(dateCompletedFirstAid.getTime()) ? new Date(dateCompletedFirstAid) : new Date();
          break;
        case 'expiryDateFirstAid':
          initialDate = expiryDateFirstAid && !isNaN(expiryDateFirstAid.getTime()) ? new Date(expiryDateFirstAid) : new Date();
          break;
        default:
          initialDate = new Date();
          break;
      }

      // Ensure the date is valid and not 1970
      if (isNaN(initialDate.getTime()) || initialDate.getFullYear() === 1970) {
        initialDate = new Date();
      }

      // For expiry dates, ensure they start from today if no valid date is set
      if (['expiryMedicals', 'driverLicenseExpiry', 'expiryDateFirstAid'].includes(field)) {
        if (initialDate < new Date()) {
          initialDate = new Date();
        }
      }

      setDatePickerValue(initialDate);
      
      // On Android, show native date picker directly
      // On iOS, show custom modal
      if (Platform.OS === 'android') {
        setShowDatePicker(true);
        setCurrentEditingDateField(field);
      } else {
        setShowDatePicker(true);
        setCurrentEditingDateField(field);
      }
    } catch (error) {
      console.error('Error in showMode:', error);
      showToast('error', 'Date Picker Error', 'Failed to open date picker. Please try again.');
    }
  };

  const showDatePickerDialog = (field) => {
    try {
      showMode('date', field);
    } catch (error) {
      console.error('Error in showDatePickerDialog:', error);
      showToast('error', 'Date Picker Error', 'Failed to open date picker. Please try again.');
    }
  };

  // Document upload functions
  const requestCameraPermission = async () => {
    if (Platform.OS === 'android') {
      try {
        const granted = await PermissionsAndroid.request(
          PermissionsAndroid.PERMISSIONS.CAMERA,
          {
            title: 'Camera Permission',
            message: 'This app needs access to your camera to take photos.',
            buttonPositive: 'OK',
            buttonNegative: 'Cancel',
          }
        );
        return granted === PermissionsAndroid.RESULTS.GRANTED;
      } catch (error) {
        console.error('Error requesting camera permission:', error);
        showToast('error', 'Error', 'Failed to request camera permission.');
        return false;
      }
    }
    return true;
  };

  const requestStoragePermission = async () => {
    if (Platform.OS === 'android') {
      try {
        const permission = Platform.Version >= 33
          ? PermissionsAndroid.PERMISSIONS.READ_MEDIA_IMAGES
          : PermissionsAndroid.PERMISSIONS.READ_EXTERNAL_STORAGE;
        const granted = await PermissionsAndroid.request(permission, {
          title: 'Storage Permission',
          message: 'This app needs access to your storage to select files.',
          buttonPositive: 'OK',
          buttonNegative: 'Cancel',
        });
        return granted === PermissionsAndroid.RESULTS.GRANTED;
      } catch (error) {
        console.error('Error requesting storage permissions:', error);
        showToast('error', 'Error', 'Failed to request storage permissions.');
        return false;
      }
    }
    return true;
  };

  const handlePickImage = async () => {
    setIsUploading(true);
    const hasPermission = await requestStoragePermission();
    if (!hasPermission) {
      showToast('error', 'Permission Denied', 'Storage permission is required to pick a file.');
      setIsUploading(false);
      return;
    }

    try {
      const result = await launchImageLibrary({ mediaType: 'photo', quality: 0.8 });
      if (!result.didCancel && !result.errorCode) {
        const file = result.assets[0];
        await uploadFile(file, 'image');
      }
    } catch (error) {
      console.error('Error picking image:', error);
      showToast('error', 'Error', 'Failed to pick image.');
    } finally {
      setIsUploading(false);
      setUploadModalVisible(false);
    }
  };

  const handlePickPDF = async () => {
    setIsUploading(true);
    const hasPermission = await requestStoragePermission();
    if (!hasPermission) {
      showToast('error', 'Permission Denied', 'Storage permission is required to pick a file.');
      setIsUploading(false);
      return;
    }

    try {
      const result = await launchImageLibrary({ mediaType: 'mixed' });
      if (!result.didCancel && !result.errorCode) {
        const file = result.assets[0];
        if (file.type === 'application/pdf') {
          await uploadFile(file, 'pdf');
        } else {
          showToast('error', 'Invalid File', 'Please select a PDF file.');
        }
      }
    } catch (error) {
      console.error('Error picking PDF:', error);
      showToast('error', 'Error', 'Failed to pick PDF.');
    } finally {
      setIsUploading(false);
      setUploadModalVisible(false);
    }
  };

  const handleTakePhoto = async () => {
    setIsUploading(true);
    const hasPermission = await requestCameraPermission();
    if (!hasPermission) {
      showToast('error', 'Permission Denied', 'Camera permission is required to take a photo.');
      setIsUploading(false);
      return;
    }

    try {
      const result = await launchCamera({ mediaType: 'photo', quality: 0.8 });
      if (!result.didCancel && !result.errorCode) {
        const file = result.assets[0];
        await uploadFile(file, 'image');
      }
    } catch (error) {
      console.error('Error taking photo:', error);
      showToast('error', 'Error', 'Failed to take photo.');
    } finally {
      setIsUploading(false);
      setUploadModalVisible(false);
    }
  };

  const uploadFile = async (file, fileType) => {
    try {
      const adminUid = await AsyncStorage.getItem('userUid');
      if (!adminUid) {
        showToast('error', 'Error', 'Administrator not authenticated.');
        return;
      }

      const tempId = teamData.tempId || `temp_${Date.now()}`;
      const extension = file.uri.split('.').pop().toLowerCase();
      const fileName = `teamMembers/${adminUid}/${tempId}/${currentUploadField}/${Date.now()}.${extension}`;
      const reference = storage().ref(fileName);

      if (currentUploadField === 'profilePic' && profilePic) {
        await storage().ref(profilePic.name).delete().catch((err) => console.warn('Error deleting old profile pic:', err));
      }

      await reference.putFile(file.uri);
      const downloadURL = await reference.getDownloadURL();

      const doc = { uri: file.uri, url: downloadURL, name: fileName, type: fileType };
      const updatedTeamData = {
        ...teamData,
        [currentUploadField]: currentUploadField === 'profilePic' ? doc : [...(teamData[currentUploadField] || []), doc],
        tempId,
      };

      updateTeamData(updatedTeamData);
      if (currentUploadField === 'profilePic') {
        setProfilePic(doc);
      }

      if (await checkNetworkStatus()) {
        await database()
          .ref(`/TeamMember/temp/${adminUid}/${tempId}/${currentUploadField}`)
          .set(currentUploadField === 'profilePic' ? doc : updatedTeamData[currentUploadField]);
      } else {
        const pendingUploads = JSON.parse(await AsyncStorage.getItem('pendingUploads') || '[]');
        pendingUploads.push({
          adminUid,
          tempId,
          field: currentUploadField,
          file: { uri: file.uri, name: fileName, type: fileType },
        });
        await AsyncStorage.setItem('pendingUploads', JSON.stringify(pendingUploads));
        showToast('info', 'Offline Mode', 'File upload queued. Will sync when online.');
      }

      showToast('success', 'Success', 'File uploaded successfully.');
    } catch (error) {
      console.error('Error uploading file:', error);
      showToast('error', 'Error', 'Failed to upload file.');
      throw error;
    }
  };

  const handleRemoveDocument = async (field, index) => {
    try {
      const updatedDocs = Array.isArray(teamData[field]) ? teamData[field].filter((_, i) => i !== index) : null;
      const updatedTeamData = {
        ...teamData,
        [field]: updatedDocs,
      };
      updateTeamData(updatedTeamData);

      if (field === 'profilePic') {
        setProfilePic(null);
        if (await checkNetworkStatus() && teamData[field]) {
          await storage().ref(teamData[field].name).delete().catch((err) => console.warn('Error deleting profile pic:', err));
        }
      }

      if (await checkNetworkStatus()) {
        const adminUid = await AsyncStorage.getItem('userUid');
        const tempId = teamData.tempId;
        await database()
          .ref(`/TeamMember/temp/${adminUid}/${tempId}/${field}`)
          .set(updatedDocs ? updatedDocs : null);
      }
      showToast('success', 'Success', 'Document removed successfully.');
    } catch (error) {
      console.error('Error removing document:', error);
      showToast('error', 'Error', 'Failed to remove document.');
    }
  };

  const handleUploadPress = (field) => {
    setCurrentUploadField(field);
    setUploadModalVisible(true);
  };

  // Handle permission changes
  const handlePermissionChange = (module, callback) => {
    let newPermissions = [];
    
    // Handle the callback properly for multiple selection
    if (typeof callback === 'function') {
      // Get current permissions for this module from ref to avoid stale state
      const currentPermissions = permissionRefs.current[module] || [];
      // Call the callback function to get the new selection
      newPermissions = callback(currentPermissions);
    } else {
      // If it's a direct value, use it
      newPermissions = Array.isArray(callback) ? [...callback] : [];
    }

    // Ensure newPermissions is always an array
    if (!Array.isArray(newPermissions)) {
      newPermissions = [];
    }

    // Handle "All Permissions" logic
    if (newPermissions.includes('all')) {
      // If "All Permissions" is selected, include all individual permissions
      newPermissions = ['all', ...individualPermissions];
    }

    console.log('Permission change for module:', module, 'Current:', permissionRefs.current[module], 'New:', newPermissions);

    // Update the ref immediately
    permissionRefs.current[module] = newPermissions;

    setModulePermissions((prev) => ({
      ...prev,
      [module]: newPermissions,
    }));

    setErrors((prev) => ({
      ...prev,
      permissions: {
        ...prev.permissions,
        [module]: newPermissions.length === 0,
      },
    }));
  };

  // Handle next step with database storage
  const handleNextStep = async () => {
    const isNameValid = validateName(name);
    const isEmailValid = validateEmail(email);
    const isPasswordValid = validatePassword(password);
    const isMobileValid = validateMobile(mobile);
    const isDepartmentValid = validateDepartment(department);
    const isNrcIdNumberValid = validateNrcIdNumber(nrcIdNumber);
    const isEmployeeNumberValid = validateEmployeeNumber(employeeNumber);
    const isDropdownValid = validateDropdowns();

    if (
      !isNameValid ||
      !isEmailValid ||
      !isPasswordValid ||
      !isMobileValid ||
      !isDepartmentValid ||
      !isNrcIdNumberValid ||
      !isEmployeeNumberValid ||
      !isDropdownValid
    ) {
      showToast('error', 'Validation Error', 'Please correct all input errors');
      Alert.alert('Error', 'Please fill all required fields correctly.');
      return;
    }

    if (
      isNaN(engagementDate.getTime()) ||
      isNaN(titanInductionDate.getTime()) ||
      isNaN(dateExamined.getTime()) ||
      isNaN(expiryMedicals.getTime()) ||
      isNaN(driverLicenseExpiry.getTime()) ||
      isNaN(clientDrivingDate.getTime()) ||
      isNaN(dateCompletedFirstAid.getTime()) ||
      isNaN(expiryDateFirstAid.getTime())
    ) {
      showToast('error', 'Validation Error', 'All dates must be valid.');
      return;
    }

    if (expiryMedicals <= dateExamined) {
      showToast('error', 'Validation Error', 'Medical Expiry Date must be after Date Examined.');
      return;
    }

    if (driverLicenseExpiry <= engagementDate) {
      showToast('error', 'Validation Error', "Driver's License Expiry Date must be after Engagement Date.");
      return;
    }

    if (expiryDateFirstAid <= dateCompletedFirstAid) {
      showToast('error', 'Validation Error', 'First Aid Expiry Date must be after Date Completed.');
      return;
    }

    setIsLoadingSignUp(true);
    setSyncModalVisible(true);
    setSyncModalMessage('Creating user');

    try {
      if (!(await checkNetworkStatus())) {
        const tempId = teamData.tempId || `temp_${Date.now()}`;
        const updatedTeamData = {
          ...teamData,
          name,
          email,
          password,
          mobile,
          role,
          countries,
          projects,
          modules: modules.map((mod) => ({
            module: mod,
            permissions: modulePermissions[mod].filter((perm) => perm !== 'all'),
          })),
          department,
          nrcIdNumber,
          employeeNumber,
          engagementDate: engagementDate.toISOString(),
          titanInductionDate: titanInductionDate.toISOString(),
          clientInduction: clientInduction.toISOString(),
          dateExamined: dateExamined.toISOString(),
          expiryMedicals: expiryMedicals.toISOString(),
          driverLicenseNumber,
          driverLicenseCode,
          driverLicenseExpiry: driverLicenseExpiry.toISOString(),
          clientDrivingDate: clientDrivingDate.toISOString(),
          dateCompletedFirstAid: dateCompletedFirstAid.toISOString(),
          levelFirstAid,
          expiryDateFirstAid: expiryDateFirstAid.toISOString(),
          profilePic,
          medicalDocs: teamData.medicalDocs || [],
          driverLicenseDocs: teamData.driverLicenseDocs || [],
          defensiveDrivingDocs: teamData.defensiveDrivingDocs || [],
          tempId,
        };

        const pendingTeamData = JSON.parse(await AsyncStorage.getItem('pendingTeamData') || '[]');
        pendingTeamData.push({ 
          adminUid: await AsyncStorage.getItem('userUid'), 
          tempId, 
          data: updatedTeamData,
          authUid: null // Will be generated when online
        });
        await AsyncStorage.setItem('pendingTeamData', JSON.stringify(pendingTeamData));
        updateTeamData(updatedTeamData);
        showToast('info', 'Offline Mode', 'Data saved locally. Will sync when online.');
        // Remove all related AsyncStorage data for this user/tempId and all form data
        try {
          const adminUid = await AsyncStorage.getItem('userUid');
          const tempId = teamData.tempId || '';
          // Remove all pendingTeamData and pendingUploads
          await AsyncStorage.removeItem('pendingTeamData');
          await AsyncStorage.removeItem('pendingUploads');
          // Remove any draft/form data for this user/tempId
          const draftKeys = [
            `teamDataDraft_${adminUid}_${tempId}`,
            `teamDataForm_${adminUid}_${tempId}`,
            `teamDataDraft`,
            `teamDataForm`,
            // Remove selected modules keys as well
            `selectedModules`,
            `selectedModules_${adminUid}_${tempId}`,
          ];
          for (const key of draftKeys) {
            await AsyncStorage.removeItem(key);
          }
        } catch (cleanupErr) {
          console.warn('Error cleaning up AsyncStorage after team member creation:', cleanupErr);
        }
        // Show success modal for offline mode
        setSuccessMessage('Data saved locally. Will sync when online.');
        
        // Close sync modal first
        setSyncModalVisible(false);
        
        // Force modal to show on iOS with multiple attempts
        setSuccessModalVisible(true);
        
        // Additional attempts to ensure modal shows on iOS
        setTimeout(() => {
          console.log('First attempt to show success modal (offline)');
          setSuccessModalVisible(true);
        }, 200);
        
        setTimeout(() => {
          console.log('Second attempt to show success modal (offline)');
          setSuccessModalVisible(true);
        }, 500);
        
        // Auto-close modal after 5 seconds on iOS to ensure navigation works
        if (Platform.OS === 'ios') {
          setTimeout(() => {
            console.log('Auto-closing modal on iOS after 5 seconds (offline)');
            handleSuccessModalClose();
          }, 5000);
          
          // Fallback: show alert if modal doesn't work after 2 seconds
          setTimeout(() => {
            if (successModalVisible) {
              console.log('Modal is visible, proceeding normally (offline)');
            } else {
              console.log('Modal not visible, showing fallback alert (offline)');
              Alert.alert(
                'Success!',
                'Data saved locally. Will sync when online.',
                [
                  {
                    text: 'OK',
                    onPress: () => navigation.goBack(),
                  },
                ]
              );
            }
          }, 2000);
        }
        return;
      }

      // Prepare user data for database
      const userData = {
        name,
        email,
        password, // Add password to database
        mobile,
        role,
        loginStatus: false,
        lastLogin: null,
        countries,
        projects,
        modules: modules.map((mod) => ({
          module: mod,
          permissions: modulePermissions[mod].filter((perm) => perm !== 'all'),
        })),
        department,
        nrcIdNumber,
        employeeNumber,
        engagementDate: engagementDate.toISOString(),
        titanInductionDate: titanInductionDate.toISOString(),
        clientInduction,
        dateExamined: dateExamined.toISOString(),
        expiryMedicals: expiryMedicals.toISOString(),
        driverLicenseNumber,
        driverLicenseCode,
        driverLicenseExpiry: driverLicenseExpiry.toISOString(),
        clientDrivingDate: clientDrivingDate.toISOString(),
        dateCompletedFirstAid: dateCompletedFirstAid.toISOString(),
        levelFirstAid,
        expiryDateFirstAid: expiryDateFirstAid.toISOString(),
        profilePic,
        medicalDocs: teamData.medicalDocs || [],
        driverLicenseDocs: teamData.driverLicenseDocs || [],
        defensiveDrivingDocs: teamData.defensiveDrivingDocs || [],
      };

      // Create Auth user directly to get the UID, then restore admin session
      let authUid = null;
      try {
        // Store current admin session info BEFORE creating new user
        const currentAdminUser = auth().currentUser;
        const adminEmail = currentAdminUser?.email;
        const adminUid = currentAdminUser?.uid;
        
        // Get admin password from AsyncStorage (stored during login)
        const adminPassword = await AsyncStorage.getItem('adminPassword');
        
        console.log('Current admin user:', adminEmail, adminUid);
        console.log('Admin password available:', !!adminPassword);
        
        if (!adminPassword) {
          // Try to get admin email from AsyncStorage as fallback
          const storedAdminEmail = await AsyncStorage.getItem('userEmail');
          if (storedAdminEmail) {
            console.log('Using stored admin email:', storedAdminEmail);
            // For now, we'll proceed without password restoration
            // The user will need to log in again after user creation
            console.warn('Admin password not found in AsyncStorage. User will need to log in again after user creation.');
          } else {
            throw new Error('Admin password not stored. Cannot restore session after user creation.');
          }
        }
        
        // Create the authentication user
        const userCredential = await auth().createUserWithEmailAndPassword(email, password);
        authUid = userCredential.user.uid;
        console.log('Auth user created successfully with UID:', authUid);
        
        // Sign out the newly created user
        await auth().signOut();
        console.log('New user signed out');
        
        // IMMEDIATELY sign back in as admin using stored password
        if (adminPassword) {
          try {
            await auth().signInWithEmailAndPassword(adminEmail, adminPassword);
            console.log('Admin session restored successfully after user creation');
            showToast('success', 'Session Restored', 'Admin session restored successfully!');
          } catch (signInError) {
            console.error('Failed to restore admin session:', signInError);
            throw new Error('Failed to restore admin session after user creation');
          }
        } else {
          console.warn('Admin password not available - user will need to log in again');
          showToast('info', 'Session Info', 'User created successfully. Please log in again to continue.');
        }
        
        showToast('success', 'Auth User Created', `User account created successfully!\nAuth UID: ${authUid}`);
        
      } catch (authError) {
        console.error('Auth user creation error:', authError);
        
        if (authError.code === 'auth/email-already-in-use') {
          showToast('error', 'Auth Error', 'This email is already in use by another user');
        } else if (authError.code === 'auth/weak-password') {
          showToast('error', 'Auth Error', 'Password is too weak');
        } else if (authError.code === 'auth/invalid-email') {
          showToast('error', 'Auth Error', 'Invalid email address');
        } else if (authError.message.includes('Admin password not stored')) {
          showToast('error', 'Session Error', 'Admin password not stored. Please log in again.');
        } else if (authError.message.includes('Failed to restore admin session')) {
          showToast('error', 'Session Error', 'Failed to restore admin session. Please log in again.');
        } else {
          showToast('error', 'Auth Error', `Failed to create authentication user: ${authError.message}`);
        }
        
        // If auth creation fails, we can't proceed
        throw authError;
      }

      // Now store user data in the correct path using the Auth UID
      console.log('Storing user data in Firebase Realtime Database at path:', `/Globalusers/admins/${authUid}`);
      await database().ref(`/Globalusers/admins/${authUid}`).set(userData);
      console.log('User data stored successfully in Firebase Realtime Database');

      // Update parent component's teamData
      const newTeamData = {
        ...teamData,
        name,
        email,
        password,
        mobile,
        role,
        countries,
        projects,
        modules,
        modulePermissions,
        department,
        nrcIdNumber,
        employeeNumber,
        engagementDate: engagementDate.toISOString(),
        titanInductionDate: titanInductionDate.toISOString(),
        clientInduction,
        dateExamined: dateExamined.toISOString(),
        expiryMedicals: expiryMedicals.toISOString(),
        driverLicenseNumber,
        driverLicenseCode,
        driverLicenseExpiry: driverLicenseExpiry.toISOString(),
        clientDrivingDate: clientDrivingDate.toISOString(),
        dateCompletedFirstAid: dateCompletedFirstAid.toISOString(),
        levelFirstAid,
        expiryDateFirstAid: expiryDateFirstAid.toISOString(),
        profilePic,
        medicalDocs: teamData.medicalDocs || [],
        driverLicenseDocs: teamData.driverLicenseDocs || [],
        defensiveDrivingDocs: teamData.defensiveDrivingDocs || [],
        userId: authUid, // Use the actual Auth UID
        authUid: authUid, // Also store as authUid for clarity
      };
      if (updateTeamData) updateTeamData(newTeamData);

      // Show success modal instead of toast
      setSuccessMessage(`User is created successfully with login!\n\nAuth UID: ${authUid}\nData stored in /Globalusers/admins/${authUid}/\n\nAdmin session restored automatically.`);
      
      // Close sync modal first
      setSyncModalVisible(false);
      
      // Force modal to show on iOS with multiple attempts
      setSuccessModalVisible(true);
      
      // Additional attempts to ensure modal shows on iOS
      setTimeout(() => {
        console.log('First attempt to show success modal');
        setSuccessModalVisible(true);
      }, 200);
      
      setTimeout(() => {
        console.log('Second attempt to show success modal');
        setSuccessModalVisible(true);
      }, 500);
      
      // Auto-close modal after 5 seconds on iOS to ensure navigation works
      if (Platform.OS === 'ios') {
        setTimeout(() => {
          console.log('Auto-closing modal on iOS after 5 seconds');
          handleSuccessModalClose();
        }, 5000);
        
        // Fallback: show alert if modal doesn't work after 2 seconds
        setTimeout(() => {
          if (successModalVisible) {
            console.log('Modal is visible, proceeding normally');
          } else {
            console.log('Modal not visible, showing fallback alert');
            Alert.alert(
              'Success!',
              `User is created successfully with login!\n\nAuth UID: ${authUid}\nData stored in /Globalusers/admins/${authUid}/\n\nAdmin session restored automatically.`,
              [
                {
                  text: 'OK',
                  onPress: () => navigation.goBack(),
                },
              ]
            );
          }
        }, 2000);
      }

      // Remove all related AsyncStorage data for this user/tempId and all form data
      try {
        const adminUid = await AsyncStorage.getItem('userUid');
        const tempId = teamData.tempId || '';
        // Remove all pendingTeamData and pendingUploads
        await AsyncStorage.removeItem('pendingTeamData');
        await AsyncStorage.removeItem('pendingUploads');
        // Remove any draft/form data for this user/tempId
        const draftKeys = [
          `teamDataDraft_${adminUid}_${tempId}`,
          `teamDataForm_${adminUid}_${tempId}`,
          `teamDataDraft`,
          `teamDataForm`,
          // Remove selected modules keys as well
          `selectedModules`,
          `selectedModules_${adminUid}_${tempId}`,
        ];
        for (const key of draftKeys) {
          await AsyncStorage.removeItem(key);
        }
      } catch (cleanupErr) {
        console.warn('Error cleaning up AsyncStorage after team member creation:', cleanupErr);
      }
      // Don't navigate back here, let the success modal handle it
    } catch (error) { 
      console.error('Team member creation error:', {
        message: error.message,
        code: error.code,
        details: JSON.stringify(error, null, 2),
      });
      const errorMessages = {
        'auth/email-already-in-use': 'This email is already in use',
        'auth/invalid-email': 'Invalid email address',
        'auth/weak-password': 'Password is too weak',
        'auth/too-many-requests': 'Too many attempts. Please try again later',
        default: error.message || 'An unexpected error occurred',
      };
      showToast('error', 'Creation Error', errorMessages[error.code] || errorMessages.default);
    } finally {
      setIsLoadingSignUp(false);
      setSyncModalVisible(false);
    }
  };

  // Handle back step
  const handleBackStep = () => {
    Alert.alert(
      'Confirm Exit',
      'Are you sure you want to exit? Any unsaved changes will be lost.',
      [
        { text: 'Cancel', style: 'cancel' },
        { text: 'OK', onPress: () => navigation.goBack() },
      ],
      { cancelable: true }
    );
  };

  const handleSuccessModalClose = () => {
    console.log('Closing success modal');
    setSuccessModalVisible(false);
    setSuccessMessage('');
    
    // Force navigation back on iOS
    setTimeout(() => {
      navigation.goBack();
    }, 100);
  };

  const toggleShowPassword = () => {
    setShowPassword(!showPassword);
  };

  // Handle input focus to scroll
  const handleInputFocus = useCallback(
    (inputRef) => () => {
      setTimeout(() => {
        inputRef.current?.measureLayout(
          scrollViewRef.current.getScrollResponder(),
          (x, y) => {
            scrollViewRef.current.scrollTo({ y: y - scale(100), animated: true });
          },
          (error) => {
            console.error('Error measuring layout:', error);
          }
        );
      }, 300);
    },
    []
  );

  const renderDocumentPreviews = (field) => {
    const docs = teamData[field] || [];
    if (docs.length === 0) return null;

    return (
      <View style={styles.documentPreviewContainer}>
        {docs.map((doc, index) => (
          <View key={index} style={styles.documentPreviewItem}>
            {doc.type === 'pdf' ? (
              <View style={styles.pdfIconContainer}>
                <MaterialIcons name="picture-as-pdf" size={scale(40)} color="#ff4444" />
              </View>
            ) : (
              <Image source={{ uri: doc.uri }} style={styles.documentPreviewImage} />
            )}
            <TouchableOpacity
              style={styles.removeDocumentButton}
              onPress={() => handleRemoveDocument(field, index)}
            >
              <MaterialIcons name="close" size={scale(18)} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.documentName} numberOfLines={1} ellipsizeMode="middle">
              {doc.name.split('/').pop()}
            </Text>
          </View>
        ))}
      </View>
    );
  };

  const renderProfilePicturePreview = () => {
    if (!profilePic) return null;

    return (
      <View style={styles.profilePicturePreview}>
        <Image source={{ uri: profilePic.uri }} style={styles.profilePictureImage} />
        <TouchableOpacity
          style={styles.removePictureButton}
          onPress={() => handleRemoveDocument('profilePic', 0)}
        >
          <MaterialIcons name="close" size={scale(18)} color="#fff" />
        </TouchableOpacity>
      </View>
    );
  };

  return (
    <SafeAreaView style={styles.container}>
      {isNetworkError && (
        <Animated.View
          style={[styles.networkErrorBanner, { transform: [{ translateY: networkErrorAnim }] }]}
        >
          <Text style={styles.networkErrorText}>Internet connection not available</Text>
        </Animated.View>
      )}

      <KeyboardAvoidingView
        style={{ flex: 1 }}
        behavior={Platform.OS === 'ios' ? 'padding' : 'height'}
        keyboardVerticalOffset={Platform.OS === 'ios' ? scale(100) : scale(20)}
      >
        <ScrollView
          ref={scrollViewRef}
          contentContainerStyle={styles.scrollContent}
          keyboardShouldPersistTaps="handled"
          nestedScrollEnabled={true}
          disableScrollViewPanResponder={true}
          showsVerticalScrollIndicator={false}
        >
          <View style={styles.header}>
            <TouchableOpacity
              style={styles.syncButton}
              onPress={() => {
                setSyncModalVisible(true);
                setSyncModalMessage('Syncing Cloud Data');
                setTimeout(() => {
                  setSyncModalVisible(false);
                  showToast('success', 'Sync Complete', 'Data synchronized successfully');
                }, 2000);
              }}
            >
              <Image
                source={require('../../../Images/adminhome/equp/cloud.png')}
                style={styles.refreshIcon}
              />
            </TouchableOpacity>
          </View>

          <View style={styles.contentContainer}>
            <View style={styles.formContainer}>
              {/* Employee Profile Picture */}
              <Text style={styles.sectionTitle}>Employee Profile Picture</Text>
              <View style={styles.profilePictureContainer}>
                {renderProfilePicturePreview()}
                <GradientButton
                  title={profilePic ? 'Replace Picture' : 'Upload Picture'}
                  onPress={() => handleUploadPress('profilePic')}
                  style={[styles.uploadButton, profilePic && styles.replaceButton]}
                />
              </View>

              {/* Authentication Inputs */}
              <Text style={styles.sectionTitle}>Authentication Details</Text>
              <View style={styles.mainemailpass}>
                <AnimatedInput
                  inputRef={nameInputRef}
                  label="Full Name"
                  value={name}
                  onChangeText={(text) => {
                    setName(text);
                    validateName(text);
                  }}
                  autoCapitalize="words"
                  leftIconName="person"
                  hasError={nameError}
                  onBlur={() => validateName(name)}
                  maxLength={13}
                  onFocus={handleInputFocus(nameInputRef)}
                />
                {nameError && <Text style={styles.errorText}>{nameErrorMessage}</Text>}
              </View>

              <View style={styles.mainemailpass}>
                <AnimatedInput
                  inputRef={emailInputRef}
                  label="Email"
                  value={email}
                  onChangeText={(text) => {
                    setEmail(text);
                    validateEmail(text);
                  }}
                  keyboardType="email-address"
                  autoCapitalize="none"
                  leftIconName="email"
                  hasError={emailError}
                  onBlur={() => validateEmail(email)}
                  maxLength={40}
                  onFocus={handleInputFocus(emailInputRef)}
                />
                {emailError && <Text style={styles.errorText}>{emailErrorMessage}</Text>}
              </View>

              <View style={styles.mainemailpass}>
                <AnimatedInput
                  inputRef={mobileInputRef}
                  label="Mobile"
                  value={mobile}
                  onChangeText={(text) => {
                    setMobile(text);
                    validateMobile(text);
                  }}
                  keyboardType="phone-pad"
                  leftIconName="phone"
                  hasError={mobileError}
                  onBlur={() => validateMobile(mobile)}
                  maxLength={15}
                  onFocus={handleInputFocus(mobileInputRef)}
                />
                {mobileError && <Text style={styles.errorText}>{mobileErrorMessage}</Text>}
              </View>

              <View style={styles.mainemailpass}>
                <AnimatedInput
                  inputRef={passwordInputRef}
                  label="Password"
                  value={password}
                  onChangeText={(text) => {
                    setPassword(text);
                    validatePassword(text);
                  }}
                  secureTextEntry={!showPassword}
                  autoCapitalize="none"
                  leftIconName="lock"
                  rightIconName={showPassword ? 'visibility' : 'visibility-off'}
                  onRightIconPress={toggleShowPassword}
                  hasError={passwordError}
                  onBlur={() => validatePassword(password)}
                  maxLength={15}
                  onFocus={handleInputFocus(passwordInputRef)}
                />
                {passwordError && <Text style={styles.errorText}>{passwordErrorMessage}</Text>}
              </View>

              {/* Role Dropdown */}
              <View style={inputStyles.container}>
                <Text style={inputStyles.label}>
                  Role <Text style={inputStyles.requiredError}>*</Text>
                </Text>
                <DropDownPicker
                  open={roleOpen}
                  value={role}
                  items={roles}
                  setOpen={(open) => {
                    setRoleOpen(open);
                    if (open) {
                      setCountryOpen(false);
                      setProjectOpen(false);
                      setModuleOpen(false);
                      setPermissionOpen({});
                    }
                  }}
                  setValue={setRole}
                  setItems={() => {}}
                  placeholder="Select Role"
                  zIndex={7000}
                  listMode="SCROLLVIEW"
                  scrollViewProps={{
                    nestedScrollEnabled: true,
                  }}
                  style={[inputStyles.input, errors.role ? inputStyles.inputError : null]}
                  dropDownContainerStyle={[styles.dropdownContainer, { maxHeight: scale(200) }]}
                  textStyle={styles.dropdownText}
                  placeholderStyle={styles.placeholderStyle}
                  badgeDotColors={['#015185']}
                  badgeTextStyle={styles.badgeText}
                  badgeStyle={styles.badge}
                />
                {errors.role && <Text style={inputStyles.errorText}>Role is required</Text>}
              </View>

              {/* General Details */}
              <Text style={styles.sectionTitle}>General Details</Text>
              <View style={styles.mainemailpass}>
                <AnimatedInput
                  inputRef={employeeNumberInputRef}
                  label="Employee Number"
                  value={employeeNumber}
                  onChangeText={(text) => {
                    setEmployeeNumber(text);
                    validateEmployeeNumber(text);
                  }}
                  hasError={employeeNumberError}
                  onBlur={() => validateEmployeeNumber(employeeNumber)}
                  maxLength={20}
                  onFocus={handleInputFocus(employeeNumberInputRef)}
                  leftIconName="badge"
                />
                {employeeNumberError && <Text style={styles.errorText}>{employeeNumberErrorMessage}</Text>}
              </View>
              <View style={styles.mainemailpass}>
                <AnimatedInput
                  inputRef={departmentInputRef}
                  label="Department"
                  value={department}
                  onChangeText={(text) => {
                    setDepartment(text);
                    validateDepartment(text);
                  }}
                  hasError={departmentError}
                  onBlur={() => validateDepartment(department)}
                  onFocus={handleInputFocus(departmentInputRef)}
                  leftIconName="business"
                />
                {departmentError && <Text style={styles.errorText}>{departmentErrorMessage}</Text>}
              </View>

              <View style={styles.mainemailpass}>
                <AnimatedInput
                  inputRef={nrcIdNumberInputRef}
                  label="NRC / ID Number"
                  value={nrcIdNumber}
                  onChangeText={(text) => {
                    setNrcIdNumber(text);
                    validateNrcIdNumber(text);
                  }}
                  hasError={nrcIdNumberError}
                  onBlur={() => validateNrcIdNumber(nrcIdNumber)}
                  onFocus={handleInputFocus(nrcIdNumberInputRef)}
                  leftIconName="credit-card"
                />
                {nrcIdNumberError && <Text style={styles.errorText}>{nrcIdNumberErrorMessage}</Text>}
              </View>

              <View style={styles.mainemailpass}>
                <AnimatedInput
                  label="Engagement Date"
                  value={formatDate(engagementDate)}
                  editable={false}
                  leftIconName="calendar-today"
                  rightIconName="calendar-today"
                  onRightIconPress={() => showDatePickerDialog('engagementDate')}
                />
              </View>

              {/* Country Dropdown */}
              <View style={inputStyles.container}>
                <Text style={inputStyles.label}>
                  Countries <Text style={inputStyles.requiredError}>*</Text>
                </Text>
                <DropDownPicker
                  multiple={true}
                  open={countryOpen}
                  value={countries}
                  items={countryItems}
                  setOpen={(open) => {
                    setCountryOpen(open);
                    if (open) {
                      setProjectOpen(false);
                      setModuleOpen(false);
                      setPermissionOpen({});
                      setRoleOpen(false);
                    }
                  }}
                  setValue={setCountries}
                  setItems={setCountryItems}
                  placeholder="Select Countries"
                  zIndex={10000}
                  listMode="SCROLLVIEW"
                  scrollViewProps={{
                    nestedScrollEnabled: true,
                  }}
                  style={[inputStyles.input, errors.country ? inputStyles.inputError : null]}
                  dropDownContainerStyle={[styles.dropdownContainer, { maxHeight: scale(200) }]}
                  textStyle={styles.dropdownText}
                  placeholderStyle={styles.placeholderStyle}
                  badgeDotColors={['#015185']}
                  badgeTextStyle={styles.badgeText}
                  badgeStyle={styles.badge}
                />
                {errors.country && (
                  <Text style={inputStyles.errorText}>At least one country is required</Text>
                )}
                <View style={styles.chipContainer}>
                  {countries.length > 0 ? (
                    countries.map((country, index) => (
                      <View key={index} style={styles.chip}>
                        <Text style={styles.chipText}>{country}</Text>
                      </View>
                    ))
                  ) : (
                    <Text style={styles.noItemsText}>No countries selected</Text>
                  )}
                </View>
              </View>

              {/* Project Dropdown */}
              <View style={inputStyles.container}>
                <Text style={inputStyles.label}>
                  Projects <Text style={inputStyles.requiredError}>*</Text>
                </Text>
                <DropDownPicker
                  multiple={true}
                  open={projectOpen}
                  value={projects}
                  items={projectItems}
                  setOpen={(open) => {
                    setProjectOpen(open);
                    if (open) {
                      setCountryOpen(false);
                      setModuleOpen(false);
                      setPermissionOpen({});
                      setRoleOpen(false);
                    }
                  }}
                  setValue={setProjects}
                  setItems={setProjectItems}
                  placeholder="Select Projects"
                  zIndex={9000}
                  listMode="SCROLLVIEW"
                  scrollViewProps={{
                    nestedScrollEnabled: true,
                  }}
                  style={[inputStyles.input, errors.project ? inputStyles.inputError : null]}
                  dropDownContainerStyle={[styles.dropdownContainer, { maxHeight: scale(200) }]}
                  textStyle={styles.dropdownText}
                  placeholderStyle={styles.placeholderStyle}
                  badgeDotColors={['#015185']}
                  badgeTextStyle={styles.badgeText}
                  badgeStyle={styles.badge}
                />
                {errors.project && (
                  <Text style={inputStyles.errorText}>At least one project is required</Text>
                )}
                <View style={styles.chipContainer}>
                  {projects.length > 0 ? (
                    projects.map((project, index) => (
                      <View key={index} style={styles.chip}>
                        <Text style={styles.chipText}>{project}</Text>
                      </View>
                    ))
                  ) : (
                    <Text style={styles.noItemsText}>No projects selected</Text>
                  )}
                </View>
              </View>

              {/* Modules Dropdown */}
              <View style={inputStyles.container}>
                <Text style={inputStyles.label}>
                  Modules <Text style={inputStyles.requiredError}>*</Text>
                </Text>
                <DropDownPicker
                  multiple={true}
                  open={moduleOpen}
                  value={modules}
                  items={moduleItems}
                  setOpen={(open) => {
                    setModuleOpen(open);
                    if (open) {
                      setCountryOpen(false);
                      setProjectOpen(false);
                      setPermissionOpen({});
                      setRoleOpen(false);
                    }
                  }}
                  setValue={setModules}
                  setItems={setModuleItems}
                  placeholder="Select Modules"
                  zIndex={8000}
                  listMode="SCROLLVIEW"
                  scrollViewProps={{
                    nestedScrollEnabled: true,
                  }}
                  style={[inputStyles.input, errors.module ? inputStyles.inputError : null]}
                  dropDownContainerStyle={[styles.dropdownContainer, { maxHeight: scale(200) }]}
                  textStyle={styles.dropdownText}
                  placeholderStyle={styles.placeholderStyle}
                  badgeDotColors={['#015185']}
                  badgeTextStyle={styles.badgeText}
                  badgeStyle={styles.badge}
                />
                {errors.module && (
                  <Text style={inputStyles.errorText}>At least one module is required</Text>
                )}
                <View style={styles.chipContainer}>
                  {modules.length > 0 ? (
                    modules.map((module, index) => (
                      <View key={module} style={styles.moduleChipContainer}>
                        <View style={styles.chip}>
                          <Text style={styles.chipText}>{module}</Text>
                        </View>
                        <View style={styles.permissionContainer}>
                          <Text style={styles.label}>
                            Permissions for {module} <Text style={styles.requiredError}>*</Text>
                          </Text>
                          <DropDownPicker
                            multiple={true}
                            open={permissionOpen[module] || false}
                            value={modulePermissions[module] || []}
                            items={permissionOptions}
                            setOpen={(open) => {
                              // Close all other permission dropdowns first
                              setPermissionOpen((prev) => {
                                const newState = {};
                                Object.keys(prev).forEach((key) => {
                                  newState[key] = false;
                                });
                                newState[module] = open;
                                return newState;
                              });
                            }}
                            setValue={(callback) => {
                              console.log('Permission change for module:', module, 'callback:', callback);
                              handlePermissionChange(module, callback);
                            }}
                            setItems={() => {}}
                            placeholder={`Select Permissions for ${module}`}
                            zIndex={8000 - (index + 1)}
                            listMode="SCROLLVIEW"
                            scrollViewProps={{
                              nestedScrollEnabled: true,
                            }}
                            style={[
                              inputStyles.input,
                              errors.permissions[module] ? inputStyles.inputError : null,
                            ]}
                            dropDownContainerStyle={[styles.dropdownContainer, { maxHeight: scale(200) }]}
                            textStyle={styles.dropdownText}
                            placeholderStyle={styles.placeholderStyle}
                            badgeDotColors={['#015185']}
                            badgeTextStyle={styles.badgeText}
                            badgeStyle={styles.badge}
                            closeAfterSelecting={false}
                            min={0}
                            max={permissionOptions.length}
                            searchable={false}
                            autoScroll={true}
                            multipleText={modulePermissions[module]?.length > 0 ? `${modulePermissions[module].length} items selected` : ''}
                            showTickIcon={true}
                            tickIconStyle={{
                              width: scale(20),
                              height: scale(20),
                              tintColor: '#015185',
                            }}
                          />
                          {errors.permissions[module] && (
                            <Text style={inputStyles.errorText}>
                              At least one permission is required for {module}
                            </Text>
                          )}
                          <View style={styles.chipContainer}>
                            {modulePermissions[module]?.length > 0 ? (
                              modulePermissions[module].map((permission, permIndex) => (
                                <View key={permIndex} style={styles.chip}>
                                  <Text style={styles.chipText}>{permission}</Text>
                                </View>
                              ))
                            ) : (
                              <Text style={styles.noItemsText}>No permissions selected</Text>
                            )}
                          </View>
                        </View>
                      </View>
                    ))
                  ) : (
                    <Text style={styles.noItemsText}>No modules selected</Text>
                  )}
                </View>
              </View>

              {/* Titan Induction */}
              <Text style={styles.sectionTitle}>Titan Induction</Text>
              <View style={styles.mainemailpass}>
                <AnimatedInput
                  label="Titan Induction Date"
                  value={formatDate(titanInductionDate)}
                  editable={false}
                  leftIconName="calendar-today"
                  rightIconName="calendar-today"
                  onRightIconPress={() => showDatePickerDialog('titanInductionDate')}
                />
              </View>
              <View style={styles.mainemailpass}>
                <AnimatedInput
                  label="Client Induction"
                  value={formatDate(clientInduction)}
                  editable={false}
                  leftIconName="calendar-today"
                  rightIconName="calendar-today"
                  onRightIconPress={() => showDatePickerDialog('clientInduction')}
                />
              </View>

              {/* Medicals */}
              <Text style={styles.sectionTitle}>Medicals</Text>
              <View style={styles.mainemailpass}>
                <AnimatedInput
                  label="Date Examined"
                  value={formatDate(dateExamined)}
                  editable={false}
                  leftIconName="calendar-today"
                  rightIconName="calendar-today"
                  onRightIconPress={() => showDatePickerDialog('dateExamined')}
                />
              </View>
              <View style={styles.mainemailpass}>
                <AnimatedInput
                  label="Expiry"
                  value={formatDate(expiryMedicals)}
                  editable={false}
                  leftIconName="calendar-today"
                  rightIconName="calendar-today"
                  onRightIconPress={() => showDatePickerDialog('expiryMedicals')}
                />
              </View>
              <GradientButton
                title="Upload Medical Docs"
                onPress={() => handleUploadPress('medicalDocs')}
                style={styles.uploadButton}
              />
              {renderDocumentPreviews('medicalDocs')}

              {/* Driver's License */}
              <Text style={styles.sectionTitle}>Driver's License</Text>
              <View style={styles.mainemailpass}>
                <AnimatedInput
                  inputRef={driverLicenseNumberInputRef}
                  label="Number"
                  value={driverLicenseNumber}
                  onChangeText={setDriverLicenseNumber}
                  onFocus={handleInputFocus(driverLicenseNumberInputRef)}
                  leftIconName="credit-card"
                />
              </View>
              <View style={styles.mainemailpass}>
                <AnimatedInput
                  label="Code"
                  value={driverLicenseCode}
                  onChangeText={setDriverLicenseCode}
                  leftIconName="vpn-key"
                />
              </View>
              <View style={styles.mainemailpass}>
                <AnimatedInput
                  label="Expiry"
                  value={formatDate(driverLicenseExpiry)}
                  editable={false}
                  leftIconName="calendar-today"
                  rightIconName="calendar-today"
                  onRightIconPress={() => showDatePickerDialog('driverLicenseExpiry')}
                />
              </View>
              <GradientButton
                title="Upload License Docs"
                onPress={() => handleUploadPress('driverLicenseDocs')}
                style={styles.uploadButton}
              />
              {renderDocumentPreviews('driverLicenseDocs')}

              {/* Defensive Driving Data */}
              <Text style={styles.sectionTitle}>Defensive Driving Data</Text>
              <View style={styles.mainemailpass}>
                <AnimatedInput
                  label="Client Driving Date"
                  value={formatDate(clientDrivingDate)}
                  editable={false}
                  leftIconName="calendar-today"
                  rightIconName="calendar-today"
                  onRightIconPress={() => showDatePickerDialog('clientDrivingDate')}
                />
              </View>
              <GradientButton
                title="Upload Driving Docs"
                onPress={() => handleUploadPress('defensiveDrivingDocs')}
                style={styles.uploadButton}
              />
              {renderDocumentPreviews('defensiveDrivingDocs')}

              {/* First Aid Training */}
              <Text style={styles.sectionTitle}>First Aid Training</Text>
              <View style={styles.mainemailpass}>
                <AnimatedInput
                  label="Date Completed"
                  value={formatDate(dateCompletedFirstAid)}
                  editable={false}
                  leftIconName="calendar-today"
                  rightIconName="calendar-today"
                  onRightIconPress={() => showDatePickerDialog('dateCompletedFirstAid')}
                />
              </View>
              <View style={styles.mainemailpass}>
                <AnimatedInput
                  inputRef={levelFirstAidInputRef}
                  label="Level"
                  value={levelFirstAid}
                  onChangeText={setLevelFirstAid}
                  onFocus={handleInputFocus(levelFirstAidInputRef)}
                  leftIconName="medical-services"
                />
              </View>
              <View style={styles.mainemailpass}>
                <AnimatedInput
                  label="Expiry Date"
                  value={formatDate(expiryDateFirstAid)}
                  editable={false}
                  leftIconName="calendar-today"
                  rightIconName="calendar-today"
                  onRightIconPress={() => showDatePickerDialog('expiryDateFirstAid')}
                />
              </View>
            </View>
          </View>
        </ScrollView>
        {/* Bottom Back/Next Buttons */}
        <View style={styles.buttonContainer}>
          <TouchableOpacity
            style={styles.flatButton}
            onPress={handleBackStep}
            disabled={isLoadingSignUp}
          >
            <Text style={styles.flatButtonText}>{'< Cancel'}</Text>
          </TouchableOpacity>
          <TouchableOpacity
            style={styles.flatButton}
            onPress={handleNextStep}
            disabled={isLoadingSignUp}
          >
            <Text style={styles.flatButtonText}>{'Submit'}</Text>
          </TouchableOpacity>
        </View>
      </KeyboardAvoidingView>

      {/* Custom Date Picker Modal - Only for iOS */}
      {Platform.OS === 'ios' && (
        <Modal
          animationType="slide"
          transparent={true}
          visible={showDatePicker}
          onRequestClose={() => {
            setShowDatePicker(false);
            setCurrentEditingDateField('');
            setDatePickerMaxDate(null);
            setDatePickerMinDate(new Date(1900, 0, 1));
          }}
          statusBarTranslucent={true}
          presentationStyle="pageSheet"
        >
          <View style={styles.datePickerContainer}>
            <View style={styles.datePickerHeader}>
              <Text style={styles.datePickerTitle}>
                Select {currentEditingDateField ? currentEditingDateField.replace(/([A-Z])/g, ' $1').replace(/^./, str => str.toUpperCase()) : 'Date'}
              </Text>
              <TouchableOpacity
                style={styles.datePickerCloseButton}
                onPress={handleDatePickerDone}
              >
                <Text style={styles.datePickerCloseText}>Done</Text>
              </TouchableOpacity>
            </View>
            <View style={styles.datePickerContent}>
              <DateTimePicker
                testID="dateTimePicker"
                value={datePickerValue}
                mode="date"
                display="spinner"
                onChange={onDateChange}
                maximumDate={datePickerMaxDate}
                minimumDate={datePickerMinDate}
                accentColor="#025383"
                textColor="#333"
                style={styles.datePicker}
                themeVariant="light"
              />
            </View>
          </View>
        </Modal>
      )}

      {/* Native Android Date Picker - Only for Android */}
      {Platform.OS === 'android' && showDatePicker && (
        <DateTimePicker
          testID="dateTimePicker"
          value={datePickerValue}
          mode="date"
          display="default"
          onChange={onDateChange}
          maximumDate={datePickerMaxDate}
          minimumDate={datePickerMinDate}
          accentColor="#025383"
          textColor="#333"
        />
      )}

      <Modal
        visible={uploadModalVisible}
        transparent
        animationType="slide"
        onRequestClose={() => setUploadModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>Upload Document</Text>
            {isUploading ? (
              <View style={styles.loaderContainer}>
                <ActivityIndicator size="large" color="#015185" />
                <Text style={styles.loaderText}>Uploading...</Text>
              </View>
            ) : (
              <>
                <ScrollView style={styles.uploadList}>
                  {currentUploadField === 'profilePic' ? (
                    <>
                      <TouchableOpacity
                        style={styles.uploadOption}
                        onPress={handlePickImage}
                        disabled={isUploading}
                      >
                        <MaterialIcons name="photo-library" size={scale(24)} color="#333" />
                        <Text style={styles.uploadOptionText}>Pick Image</Text>
                      </TouchableOpacity>
                      <TouchableOpacity
                        style={styles.uploadOption}
                        onPress={handleTakePhoto}
                        disabled={isUploading}
                      >
                        <MaterialIcons name="camera-alt" size={scale(24)} color="#333" />
                        <Text style={styles.uploadOptionText}>Take Photo</Text>
                      </TouchableOpacity>
                    </>
                  ) : (
                    <>
                      <TouchableOpacity
                        style={styles.uploadOption}
                        onPress={handlePickImage}
                        disabled={isUploading}
                      >
                        <MaterialIcons name="photo-library" size={scale(24)} color="#333" />
                        <Text style={styles.uploadOptionText}>Pick Image</Text>
                      </TouchableOpacity>
                      <TouchableOpacity
                        style={styles.uploadOption}
                        onPress={handleTakePhoto}
                        disabled={isUploading}
                      >
                        <MaterialIcons name="camera-alt" size={scale(24)} color="#333" />
                        <Text style={styles.uploadOptionText}>Take Photo</Text>
                      </TouchableOpacity>
                      <TouchableOpacity
                        style={styles.uploadOption}
                        onPress={handlePickPDF}
                        disabled={isUploading}
                      >
                        <MaterialIcons name="picture-as-pdf" size={scale(24)} color="#333" />
                        <Text style={styles.uploadOptionText}>Pick PDF</Text>
                      </TouchableOpacity>
                    </>
                  )}
                </ScrollView>
                <TouchableOpacity
                  style={styles.modalCloseButton}
                  onPress={() => setUploadModalVisible(false)}
                  disabled={isUploading}
                >
                  <Text style={styles.modalCloseText}>Cancel</Text>
                </TouchableOpacity>
              </>
            )}
          </View>
        </View>
      </Modal>

           <SyncModal
        visible={syncModalVisible}
        onClose={() => setSyncModalVisible(false)}
        message={syncModalMessage}
      />

      {/* Success Modal */}
      <Modal
        visible={successModalVisible}
        transparent
        animationType="fade"
        onRequestClose={handleSuccessModalClose}
        statusBarTranslucent={true}
        presentationStyle={Platform.OS === 'ios' ? 'overFullScreen' : 'overFullScreen'}
        hardwareAccelerated={true}
        onShow={() => console.log('Success modal shown')}
        onDismiss={() => console.log('Success modal dismissed')}
      >
        <View style={styles.successModalOverlay}>
          <View style={styles.successModalContent}>
            <View style={styles.successIconContainer}>
              <MaterialIcons name="check-circle" size={scale(60)} color="#4CAF50" />
            </View>
            <Text style={styles.successModalTitle}>Success!</Text>
            <Text style={styles.successModalMessage}>{successMessage}</Text>
            <TouchableOpacity
              style={styles.successModalButton}
              onPress={handleSuccessModalClose}
            >
              <Text style={styles.successModalButtonText}>Go Back</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      <Toast />
    </SafeAreaView>
  );
};

// Styles
const inputStyles = StyleSheet.create({
  container: {
    marginBottom: scale(20),
    marginHorizontal: scale(10),
  },
  label: {
    fontSize: scale(14),
    color: '#015185',
    fontWeight: '500',
    marginBottom:scale(4)
  
  },
  requiredError: {
    color: 'red',
    fontWeight: 'bold',
  },
  input: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: scale(4),
    paddingHorizontal: scale(12),
    paddingVertical: scale(14),
    fontSize: scale(16),
    backgroundColor: '#fff',
    alignItems:'center',
    

  },
  inputError: {
    borderColor: 'red',
  },

  errorText: {
    color: 'red',
    fontSize: scale(12),
    marginTop: scale(2),
    marginLeft: scale(2),
  },
});

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  scrollContent: {
    paddingHorizontal: scale(5),
    paddingBottom: scale(50),
    flexGrow: 1,
  },
  header: {
    flexDirection: 'row',
    justifyContent: 'flex-end',
    alignItems: 'center',
    padding: scale(10),
  },
  syncButton: {
    padding: scale(5),
  },
  refreshIcon: {
    width: scale(32),
    height: scale(32),
    resizeMode: 'contain',
  },
  cloudIcon: {
    width: scale(32),
    height: scale(32),
  },
  contentContainer: {
    flex: 1,
    justifyContent: 'center',
  },
  formContainer: {
    width: '100%',
  },
  sectionTitle: {
    fontSize: scale(18),
    fontWeight: 'bold',
    color: '#015185',
    marginVertical: scale(10),
    marginHorizontal: scale(10),
  },
  inputContainer: {
    marginBottom: scale(20),
    marginHorizontal: scale(8),
    
   
  },
  inputGradient: {
    borderRadius: scale(8),
    backgroundColor: '#fff',
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
  },
  input: {
    backgroundColor: '#fff',
    height: scale(50),
    borderRadius: scale(8),
    fontSize: scale(16),
    justifyContent: 'center',
    
 
  },
  gradientButton: {
    borderRadius: scale(6),
    marginVertical: scale(8),
    justifyContent: 'center',
    alignSelf: 'center',
  },
  buttonTouchable: {
    alignItems: 'center',
    justifyContent: 'center',
    height: scale(48),
    borderRadius: scale(6),
    width: '100%',
  },
  googleButtonText: {
    color: '#fff',
    fontSize: scale(14),
    fontWeight: 'bold',
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingHorizontal: scale(20),
    paddingBottom: scale(20),
    borderTopWidth: 1,
    borderColor: '#eee',
    backgroundColor: '#fff',
   
  },
  networkErrorBanner: {
    backgroundColor: '#ff4444',
    paddingVertical: scale(10),
    alignItems: 'center',
    zIndex: 1000,
  },
  networkErrorText: {
    color: '#fff',
    fontSize: scale(14),
    fontWeight: 'bold',
  },
  errorText: {
    color: 'red',
    fontSize: scale(12),
    marginLeft: scale(10),
    marginTop: scale(2),
    marginBottom: scale(8),
    position: 'absolute',
    bottom: scale(-10),
  },
  mainemailpass: {
    position: 'relative',
    marginBottom: scale(10),
    

  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '80%',
    backgroundColor: '#fff',
    borderRadius: scale(10),
    padding: scale(20),
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: scale(18),
    color: '#333',
    marginBottom: scale(10),
    fontWeight: '600',
  },
  syncIcon: {
    width: scale(24),
    height: scale(24),
    marginRight: scale(8),
  },
  syncModalTextadd: {
    fontSize: scale(14),
    color: '#555',
    fontWeight: '500',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  dropdownContainer: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: scale(5),
    backgroundColor: '#fff',
  },
  dropdownText: {
    fontSize: scale(14),
    color: '#333',
  },
  placeholderStyle: {
    fontSize: scale(16),
    color: '#888',

  },
  disabledDropdown: {
    backgroundColor: '#e0e0e0',
  },
  badge: {
    backgroundColor: '#e6f0fa',
    borderRadius: scale(12),
    paddingHorizontal: scale(8),
    paddingVertical: scale(4),
    margin: scale(2),
    borderWidth: 1,
    borderColor: '#015185',
  },
  badgeText: {
    fontSize: scale(12),
    color: '#015185',
    fontWeight: '500',
  },
  chipContainer: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    marginTop: scale(8),
  },
  chip: {
    backgroundColor: '#e6f0fa',
    borderRadius: scale(12),
    paddingHorizontal: scale(10),
    paddingVertical: scale(6),
    marginRight: scale(8),
    marginBottom: scale(8),
    borderWidth: 1,
    borderColor: '#015185',
  },
  chipText: {
    fontSize: scale(12),
    color: '#015185',
    fontWeight: '500',
   
  },
  noItemsText: {
    fontSize: scale(14),
    color: '#888',
    marginTop: scale(8),
  },
  moduleChipContainer: {
    marginBottom: scale(20),
    width: '100%',
  },
  permissionContainer: {
    marginTop: scale(8),
    marginLeft: scale(12),
  },
  label: {
    fontSize: scale(14),
    color: '#015185',
    fontWeight: '500',
  },
  requiredError: {
    color: 'red',
    fontWeight: 'bold',
  },
  flatButton: {
    flex: 1,
    alignItems: 'center',
    justifyContent: 'center',
    paddingVertical: scale(14),
    backgroundColor: 'transparent',
  },
  flatButtonText: {
    color: '#0078D4',
    fontSize: scale(16),
    fontWeight: 'bold',
  },
  profilePictureContainer: {
    alignItems: 'center',
    marginVertical: scale(10),
  },
  profilePicturePreview: {
    position: 'relative',
    marginBottom: scale(10),
  },
  profilePictureImage: {
    width: scale(100),
    height: scale(100),
    borderRadius: scale(50),
    borderWidth: 2,
    borderColor: '#015185',
  },
  removePictureButton: {
    position: 'absolute',
    top: scale(-5),
    right: scale(-5),
    backgroundColor: '#ff4444',
    borderRadius: scale(12),
    padding: scale(2),
  },
uploadButton: {
  width: scale(150),
  textAlign:'center',
  marginHorizontal: scale(10),
  alignSelf: 'flex-start',
},
  replaceButton: {
    backgroundColor: '#ff4444',
  },
  documentPreviewContainer: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    marginTop: scale(10),
    marginHorizontal: scale(10),
  },
  documentPreviewItem: {
    width: scale(100),
    marginRight: scale(10),
    marginBottom: scale(10),
    alignItems: 'center',
  },
  documentPreviewImage: {
    width: scale(80),
    height: scale(80),
    borderRadius: scale(5),
    borderWidth: 1,
    borderColor: '#015185',
  },
  pdfIconContainer: {
    width: scale(80),
    height: scale(80),
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: '#f0f0f0',
    borderRadius: scale(5),
    borderWidth: 1,
    borderColor: '#015185',
  },
  removeDocumentButton: {
    position: 'absolute',
    top: scale(-5),
    right: scale(-5),
    backgroundColor: '#ff4444',
    borderRadius: scale(12),
    padding: scale(2),
  },
  documentName: {
    fontSize: scale(10),
    color: '#333',
    marginTop: scale(5),
    textAlign: 'center',
    width: scale(80),
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    width: '80%',
    backgroundColor: '#fff',
    borderRadius: scale(10),
    padding: scale(20),
    alignItems: 'center',
  },
  modalTitle: {
    fontSize: scale(18),
    fontWeight: 'bold',
    color: '#333',
    marginBottom: scale(20),
  },
  uploadList: {
    maxHeight: scale(200),
    width: '100%',
  },
  uploadOption: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: scale(10),
    paddingHorizontal: scale(15),
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
  },
  uploadOptionText: {
    fontSize: scale(16),
    color: '#333',
    marginLeft: scale(10),
  },
  modalCloseButton: {
    marginTop: scale(20),
    paddingVertical: scale(10),
    paddingHorizontal: scale(20),
    backgroundColor: '#ff4444',
    borderRadius: scale(5),
  },
  modalCloseText: {
    color: '#fff',
    fontSize: scale(16),
    fontWeight: 'bold',
  },
  loaderContainer: {
    alignItems: 'center',
    justifyContent: 'center',
    paddingVertical: scale(20),
  },
  loaderText: {
    fontSize: scale(16),
    color: '#333',
    marginTop: scale(10),
  },
  datePickerContainer: {
    backgroundColor: '#fff',
    borderRadius: scale(10),
    marginHorizontal: scale(20),
    marginVertical: Platform.OS === 'ios' ? scale(50) : scale(100),
    maxHeight: Platform.OS === 'ios' ? '80%' : undefined,
    width: Platform.OS === 'ios' ? '90%' : scale(300),
    alignSelf: 'center',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.25,
    shadowRadius: 3.84,
    overflow: 'hidden',
  },
  datePicker: {
    width: Platform.OS === 'ios' ? '100%' : scale(300),
    height: Platform.OS === 'ios' ? scale(200) : undefined,
  },
  datePickerCloseButton: {
    paddingVertical: scale(8),
    paddingHorizontal: scale(16),
    backgroundColor: '#015185',
    borderRadius: scale(6),
    minWidth: scale(60),
    alignItems: 'center',
  },
  datePickerCloseText: {
    color: '#fff',
    fontSize: scale(14),
    fontWeight: 'bold',
  },
  datePickerHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    padding: scale(15),
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
    width: '100%',
    backgroundColor: '#f8f9fa',
  },
  datePickerTitle: {
    fontSize: scale(16),
    fontWeight: 'bold',
    color: '#015185',
    flex: 1,
  },
  datePickerContent: {
    padding: scale(20),
    alignItems: 'center',
    justifyContent: 'center',
  },
  successModalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
    position: 'absolute',
    top: 0,
    left: 0,
    right: 0,
    bottom: 0,
    zIndex: 9999,
    ...(Platform.OS === 'ios' && {
      position: 'fixed',
      width: '100%',
      height: '100%',
    }),
  },
  successModalContent: {
    width: '85%',
    backgroundColor: '#fff',
    borderRadius: scale(15),
    padding: scale(25),
    alignItems: 'center',
    elevation: 10,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 4 },
    shadowOpacity: 0.3,
    shadowRadius: 8,
  },
  successIconContainer: {
    marginBottom: scale(15),
  },
  successModalTitle: {
    fontSize: scale(24),
    fontWeight: 'bold',
    color: '#4CAF50',
    marginBottom: scale(15),
    textAlign: 'center',
  },
  successModalMessage: {
    fontSize: scale(16),
    color: '#333',
    textAlign: 'center',
    lineHeight: scale(22),
    marginBottom: scale(25),
  },
  successModalButton: {
    backgroundColor: '#4CAF50',
    paddingVertical: scale(12),
    paddingHorizontal: scale(30),
    borderRadius: scale(8),
    minWidth: scale(120),
    alignItems: 'center',
  },
  successModalButtonText: {
    color: '#fff',
    fontSize: scale(16),
    fontWeight: 'bold',
  },
});

export default Basic_Details_TeamTab1;
