import React, { useEffect, useState } from 'react';
import {
  View,
  Text,
  Image,
  TouchableOpacity,
  StyleSheet,
  Dimensions,
  Platform,
  Linking,
  Modal,
  ToastAndroid,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import { useNavigation } from '@react-navigation/native';
import { check, request, PERMISSIONS, RESULTS } from 'react-native-permissions';
import AsyncStorage from '@react-native-async-storage/async-storage';
import Toast from 'react-native-toast-message';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';

const { width, height } = Dimensions.get('window');

// Helper function for platform-specific toasts
const showToast = (type, title, message) => {
  if (Platform.OS === 'android') {
    ToastAndroid.showWithGravity(
      message,
      ToastAndroid.LONG,
      ToastAndroid.BOTTOM
    );
  } else {
    Toast.show({
      type,
      text1: title,
      text2: message,
      position: 'bottom',
    });
  }
};

const Login = () => {
  const navigation = useNavigation();
  const [permissionModalVisible, setPermissionModalVisible] = useState(false);
  const [modalMessage, setModalMessage] = useState('');

  useEffect(() => {
    checkPermissions();
  }, []);

  const checkPermissions = async () => {
    try {
      // Check if permissions were already requested
      const permissionsRequested = await AsyncStorage.getItem('permissionsRequested');
      if (permissionsRequested === 'true') {
        // Verify if all permissions are granted
        const allGranted = await verifyPermissions();
        if (allGranted) {
          console.log('All permissions already granted');
          return;
        }
      }

      // Request permissions if not fully granted or not requested
      requestPermissions();
    } catch (error) {
      console.error('Error checking permissions:', error);
      showToast('error', 'Error', 'Failed to check permissions. Please try again.');
    }
  };

  const verifyPermissions = async () => {
    const permissions = Platform.OS === 'android'
      ? Platform.Version >= 33
        ? [PERMISSIONS.ANDROID.READ_MEDIA_IMAGES]
        : Platform.Version >= 30
        ? [
            PERMISSIONS.ANDROID.READ_EXTERNAL_STORAGE,
            PERMISSIONS.ANDROID.WRITE_EXTERNAL_STORAGE,
            PERMISSIONS.ANDROID.MANAGE_EXTERNAL_STORAGE,
          ]
        : [
            PERMISSIONS.ANDROID.READ_EXTERNAL_STORAGE,
            PERMISSIONS.ANDROID.WRITE_EXTERNAL_STORAGE,
          ]
      : [];

    if (permissions.length === 0) {
      return true; // No permissions required for iOS or other platforms
    }

    let allGranted = true;
    for (const permission of permissions) {
      const result = await check(permission);
      if (result !== RESULTS.GRANTED) {
        allGranted = false;
        break;
      }
    }
    return allGranted;
  };

  const requestPermissions = async () => {
    try {
      const permissions = Platform.OS === 'android'
        ? Platform.Version >= 33
          ? [PERMISSIONS.ANDROID.READ_MEDIA_IMAGES]
          : Platform.Version >= 30
          ? [
              PERMISSIONS.ANDROID.READ_EXTERNAL_STORAGE,
              PERMISSIONS.ANDROID.WRITE_EXTERNAL_STORAGE,
              PERMISSIONS.ANDROID.MANAGE_EXTERNAL_STORAGE,
            ]
          : [
              PERMISSIONS.ANDROID.READ_EXTERNAL_STORAGE,
              PERMISSIONS.ANDROID.WRITE_EXTERNAL_STORAGE,
            ]
        : [];

      if (permissions.length === 0) {
        console.log('No permissions required for this platform');
        await AsyncStorage.setItem('permissionsRequested', 'true');
        return;
      }

      let allGranted = true;

      for (const permission of permissions) {
        if (permission === PERMISSIONS.ANDROID.MANAGE_EXTERNAL_STORAGE && Platform.Version >= 30) {
          const result = await check(permission);
          console.log(`Checking permission ${permission}: ${result}`);

          if (result === RESULTS.DENIED || result === RESULTS.LIMITED) {
            setModalMessage(
              'Please enable "Allow all files access" in Settings to save documents and manage files.'
            );
            setPermissionModalVisible(true);
            allGranted = false;
          } else if (result === RESULTS.BLOCKED) {
            setModalMessage(
              'Storage permission is blocked. Please enable it in Settings > Apps > Titan Drilling > Permissions.'
            );
            setPermissionModalVisible(true);
            allGranted = false;
          }
          continue;
        }

        const result = await check(permission);
        console.log(`Checking permission ${permission}: ${result}`);

        if (result === RESULTS.DENIED) {
          const requestResult = await request(permission);
          console.log(`Requested permission ${permission}: ${requestResult}`);

          if (requestResult !== RESULTS.GRANTED) {
            allGranted = false;
            setModalMessage(
              'Storage permission is required to save reports and manage files. Please enable it in Settings.'
            );
            setPermissionModalVisible(true);
          }
        } else if (result === RESULTS.BLOCKED || result === RESULTS.LIMITED) {
          allGranted = false;
          setModalMessage(
            'Storage permission is blocked. Please enable it in Settings > Apps > Titan Drilling > Permissions.'
          );
          setPermissionModalVisible(true);
        }
      }

      if (allGranted) {
        console.log('All storage permissions granted');
        showToast('success', 'Success', 'Storage permissions enabled successfully.');
        await AsyncStorage.setItem('permissionsRequested', 'true');
      } else {
        console.warn('Some storage permissions were not granted');
        await AsyncStorage.setItem('permissionsRequested', 'true');
      }
    } catch (error) {
      console.error('Error requesting permissions:', error);
      showToast('error', 'Permission Error', `Failed to process storage permission: ${error.message}`);
      await AsyncStorage.setItem('permissionsRequested', 'true');
    }
  };

  const handleLoginPress = async () => {
    try {
      // Check if ProAdminLogined exists in AsyncStorage
      const proAdminLogined = await AsyncStorage.getItem('ProAdminLogined');
      if (proAdminLogined === 'true') {
        navigation.replace('MainApp');
      } else {
        navigation.navigate('AdminLoginScreen');
      }
    } catch (error) {
      console.error('Error checking admin login status:', error);
      showToast('error', 'Error', 'Failed to verify admin login status. Please try again.');
      navigation.navigate('AdminLoginScreen');
    }
  };

  const handleTeamMemberLoginPress = async () => {
    try {
      // Check if TeamMemberLogined exists in AsyncStorage
      const teamMemberLogined = await AsyncStorage.getItem('TeamMemberLogined');
      if (teamMemberLogined === 'true') {
        navigation.replace('MainApp');
      } else {
        navigation.navigate('TeamMemberLogin');
      }
    } catch (error) {
      console.error('Error checking team member login status:', error);
      showToast('error', 'Error', 'Failed to verify team member login status. Please try again.');
      navigation.navigate('TeamMemberLogin');
    }
  };

  return (
    <View style={styles.container}>
      {/* Permission Denied Modal */}
      <Modal
        visible={permissionModalVisible}
        transparent
        animationType="fade"
        onRequestClose={() => setPermissionModalVisible(false)}
      >
        <View style={styles.modalContainer}>
          <View style={styles.modalContent}>
            <MaterialIcons name="warning" size={40} color="#ff0000" style={styles.modalIcon} />
            <Text style={styles.modalTitle}>Permission Required</Text>
            <Text style={styles.modalMessage}>{modalMessage}</Text>
            <View style={styles.modalButtonContainer}>
              <TouchableOpacity
                style={styles.modalButton}
                onPress={() => {
                  setPermissionModalVisible(false);
                  Linking.openSettings();
                }}
              >
                <LinearGradient
                  colors={['#015185', '#3481BC']}
                  style={styles.modalButtonGradient}
                >
                  <Text style={styles.modalButtonText}>Open Settings</Text>
                </LinearGradient>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.modalButton, styles.cancelButton]}
                onPress={() => setPermissionModalVisible(false)}
              >
                <Text style={styles.modalCancelText}>Cancel</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      {/* Top Half: Full-Width Image and Buttons */}
      <View style={styles.topSection}>
        <Image
          source={require('../../../../Images/adminlogin/titanlogin1.png')}
          style={styles.topImage}
          resizeMode="cover"
        />
      </View>

      <View style={styles.loginbuttons}>
        <TouchableOpacity style={styles.loginButton} onPress={handleLoginPress}>
          <LinearGradient
            colors={['#015185', '#3481BC']}
            style={styles.gradient}
            start={{ x: 0, y: 0 }}
            end={{ x: 1, y: 1 }}
          >
            <Text style={styles.buttonText}>Admin Login</Text>
          </LinearGradient>
        </TouchableOpacity>

        {/* Team Member Login Button */}
        <TouchableOpacity style={styles.teamMemberButton} onPress={handleTeamMemberLoginPress}>
          <LinearGradient
            colors={['#228B22', '#32CD32']}
            style={styles.gradient}
            start={{ x: 0, y: 0 }}
            end={{ x: 1, y: 0 }}
          >
            <Text style={styles.buttonText}>Team Member Login</Text>
          </LinearGradient>
        </TouchableOpacity>
      </View>

      {/* Bottom Half: Two Images */}
      <View style={styles.bottomSection}>
        <View style={styles.imageRow}>
          <Image
            source={require('../../../../Images/adminlogin/titanlogin3.png')}
            style={styles.bottomImage}
            resizeMode="contain"
          />
          <Image
            source={require('../../../../Images/adminlogin/titanlogin2.png')}
            style={styles.bottomImage}
            resizeMode="contain"
          />
        </View>
      </View>
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  topSection: {
    height: height * 0.55,
    justifyContent: 'center',
    alignItems: 'center',
  },
  topImage: {
    width: '100%',
    height: '70%',
  },
  loginButton: {
    width: '90%',
    height: height * 0.07,
    borderRadius: 10,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.3,
    shadowRadius: 3,
    marginBottom: 10,
  },
  teamMemberButton: {
    width: '90%',
    height: height * 0.07,
    borderRadius: 10,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.3,
    shadowRadius: 3,
  },
  gradient: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    borderRadius: 10,
  },
  buttonText: {
    color: '#fff',
    fontSize: 18,
    fontWeight: 'bold',
  },
  bottomSection: {
    height: height * 0.35,
    justifyContent: 'center',
    alignItems: 'center',
  },
  imageRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    width: '90%',
  },
  bottomImage: {
    width: '40%',
    height: height * 0.2,
  },
  loginbuttons: {
    alignItems: 'center',
  },
  modalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    width: '80%',
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.3,
    shadowRadius: 3,
  },
  modalIcon: {
    marginBottom: 10,
  },
  modalTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: 10,
  },
  modalMessage: {
    fontSize: 16,
    color: '#333',
    textAlign: 'center',
    marginBottom: 20,
  },
  modalButtonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    width: '100%',
  },
  modalButton: {
    flex: 1,
    marginHorizontal: 5,
  },
  modalButtonGradient: {
    paddingVertical: 12,
    borderRadius: 8,
    alignItems: 'center',
  },
  modalButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: 'bold',
  },
  cancelButton: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 8,
    alignItems: 'center',
    justifyContent: 'center',
    paddingVertical: 12,
  },
  modalCancelText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: 'bold',
  },
});

export default Login;