import React, { useEffect, useState, useRef, useCallback } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  Modal,
  Animated,
  Platform,
  Image,
  SafeAreaView,
} from 'react-native';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import database from '@react-native-firebase/database';
import Toast from 'react-native-toast-message';
import NetInfo from '@react-native-community/netinfo';

// SyncModal Component (Processing Modal)
const SyncModal = ({ visible, onClose }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
            />
            <Text style={styles.syncModalTextadd}>Processing Logout</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

// Helper function to show toast messages
const showToast = (type, title, message) => {
  Toast.show({
    type: type,
    text1: title,
    text2: message,
    position: 'bottom',
  });
};

// Helper function to clear login-related AsyncStorage data
const clearLoginStorage = async () => {
  try {
    await AsyncStorage.removeItem('userEmail');
    await AsyncStorage.removeItem('userPassword');
    await AsyncStorage.removeItem('TeamMemberAdminuserUid');
    await AsyncStorage.removeItem('teamMemberId');
    await AsyncStorage.removeItem('teamMemberUserId');
    await AsyncStorage.removeItem('Loginsuccess-01');
    await AsyncStorage.removeItem('TeammemberLogin');
    await AsyncStorage.removeItem('teamMemberLoginTimestamp');
    await AsyncStorage.removeItem('sessionId');
    await AsyncStorage.removeItem('sessionStartTime');
    await AsyncStorage.removeItem('pendingSessionSync');
    await AsyncStorage.removeItem('selectedCountry');
    await AsyncStorage.removeItem('teamMemberData');
  } catch (error) {
    console.error('Error clearing AsyncStorage:', error);
  }
};

// Helper function to check network status
const checkNetworkStatus = async () => {
  try {
    const netInfo = await NetInfo.fetch();
    return netInfo.isConnected;
  } catch (error) {
    console.error('Error checking network status:', error);
    return false;
  }
};

const TeamAccount = () => {
  const navigation = useNavigation();
  const [isLoading, setIsLoading] = useState(true);
  const [userData, setUserData] = useState({
    email: 'N/A',
    fullName: 'N/A',
    role: 'N/A',
    adminEmail: 'N/A',
    teamMemberId: 'N/A',
    userId: 'N/A',
    employeeNumber: 'N/A',
    project: 'N/A',
    profilePicUrl: null,
  });
  const [isLogoutModalVisible, setIsLogoutModalVisible] = useState(false);
  const [isProcessingModalVisible, setIsProcessingModalVisible] = useState(false);
  const fadeAnim = useRef(new Animated.Value(0)).current;
  const isMounted = useRef(true);

  // Fetch team member and admin details
  const fetchAccountDetails = useCallback(async () => {
    if (!isMounted.current) return;

    try {
      // Retrieve data from AsyncStorage
      const email = await AsyncStorage.getItem('userEmail');
      const teamMemberId = await AsyncStorage.getItem('teamMemberId');
      const adminUid = await AsyncStorage.getItem('TeamMemberAdminuserUid');
      const userId = await AsyncStorage.getItem('teamMemberUserId');

      if (!email || !teamMemberId || !adminUid || !userId) {
        showToast('error', 'Error', 'Session data missing. Please log in again.');
        navigation.replace('TeamMemberLogin');
        return;
      }

      const isOnline = await checkNetworkStatus();
      let newUserData = {
        email,
        fullName: 'N/A',
        role: 'N/A',
        adminEmail: 'N/A',
        teamMemberId,
        userId,
        employeeNumber: 'N/A',
        project: 'N/A',
        profilePicUrl: null,
      };

      // Try to load cached data
      const cachedData = await AsyncStorage.getItem('teamMemberData');
      if (cachedData) {
        const parsedData = JSON.parse(cachedData);
        newUserData = { ...newUserData, ...parsedData };
      }

      if (!isOnline) {
        showToast('info', 'Offline Mode', 'Displaying cached data.');
        setUserData(newUserData);
        setIsLoading(false);
        Animated.timing(fadeAnim, {
          toValue: 1,
          duration: 500,
          useNativeDriver: true,
        }).start();
        return;
      }

      // Fetch team member details from Firebase
      const teamMemberRef = database().ref(`/TeamMember/employ/${adminUid}/${teamMemberId}`);
      const teamMemberSnapshot = await teamMemberRef.once('value');
      const teamMemberData = teamMemberSnapshot.val();

      if (!teamMemberData) {
        showToast('error', 'Error', 'Team member data not found.');
        await clearLoginStorage();
        navigation.replace('TeamMemberLogin');
        return;
      }

      newUserData.email = teamMemberData.email || email;
      newUserData.fullName = teamMemberData.fullName || 'N/A';
      newUserData.role = teamMemberData.role || 'N/A';

      // Fetch global team member data
      const globalTeamMemberRef = database().ref(`/GlobalTeamMemberInformation/${teamMemberId}`);
      const globalSnapshot = await globalTeamMemberRef.once('value');
      const globalTeamMemberData = globalSnapshot.val();
      if (globalTeamMemberData) {
        newUserData.employeeNumber = globalTeamMemberData.employeeNumber || 'N/A';
        newUserData.project = Array.isArray(globalTeamMemberData.project)
          ? globalTeamMemberData.project.join(', ')
          : globalTeamMemberData.project || 'N/A';
        newUserData.profilePicUrl = globalTeamMemberData.profilePic?.url || null;
      }

      // Fetch admin details from Firebase
      const adminRef = database().ref(`/users/${adminUid}`);
      const adminSnapshot = await adminRef.once('value');
      const adminData = adminSnapshot.val();
      newUserData.adminEmail = adminData?.email || 'N/A';

      setUserData(newUserData);
      await AsyncStorage.setItem('teamMemberData', JSON.stringify(newUserData));

      // Animate content fade-in
      Animated.timing(fadeAnim, {
        toValue: 1,
        duration: 500,
        useNativeDriver: true,
      }).start();
    } catch (error) {
      console.error('Error fetching account details:', error);
      showToast('error', 'Error', 'Failed to load account details.');
      const cachedData = await AsyncStorage.getItem('teamMemberData');
      if (cachedData) {
        setUserData(JSON.parse(cachedData));
      }
    } finally {
      setIsLoading(false);
    }
  }, [navigation, fadeAnim]);

  useEffect(() => {
    fetchAccountDetails();
    const unsubscribeNetInfo = NetInfo.addEventListener((state) => {
      if (state.isConnected && isMounted.current) {
        fetchAccountDetails();
      }
    });

    return () => {
      isMounted.current = false;
      unsubscribeNetInfo();
    };
  }, [fetchAccountDetails]);

  // Handle logout
  const handleLogout = async () => {
    if (!isMounted.current) return;

    setIsLogoutModalVisible(false);
    setIsProcessingModalVisible(true);
    try {
      const isOnline = await checkNetworkStatus();
      if (isOnline) {
        const userId = await AsyncStorage.getItem('teamMemberUserId');
        if (userId) {
          await database()
            .ref(`/TeamMemberLoginMobile/users/${userId}`)
            .update({ lastLogout: new Date().toISOString() });
        }
      }

      await clearLoginStorage();
      showToast('success', 'Success', 'Logged out successfully');
      if (Platform.OS === 'ios') {
        await new Promise(resolve => setTimeout(resolve, 300));
      }
      navigation.replace('TeamMemberLogin');
    } catch (error) {
      console.error('Logout error:', error);
      showToast('error', 'Logout Error', 'Failed to log out. Please try again.');
    } finally {
      if (isMounted.current) {
        setIsProcessingModalVisible(false);
      }
      if (Platform.OS === 'ios') {
        await new Promise(resolve => setTimeout(resolve, 200));
      }
    }
  };

  if (isLoading) {
    return (
      <SafeAreaView style={styles.container}>
        <View style={styles.loadingContainer}>
          <MaterialIcons name="hourglass-empty" size={40} color="#015185" />
          <Text style={styles.loadingText}>Loading Account Details...</Text>
        </View>
      </SafeAreaView>
    );
  }

  return (
    <SafeAreaView style={styles.container}>
      <Animated.View style={[styles.contentContainer, { opacity: fadeAnim }]}>
        {/* User Section */}
        <View style={styles.userSection}>
          <Text style={styles.userLabel}>Team Member</Text>
          <View style={styles.profileImageContainer}>
            {userData.profilePicUrl ? (
              <Image
                source={{ uri: userData.profilePicUrl }}
                style={styles.profileImage}
                accessibilityLabel="Profile Picture"
                onError={() => setUserData(prev => ({ ...prev, profilePicUrl: null }))}
              />
            ) : (
              <MaterialIcons
                name="account-circle"
                size={60}
                color="#fff"
                style={styles.profileIcon}
                accessibilityLabel="Default Profile Picture"
              />
            )}
          </View>
          <Text style={styles.email}>{userData.email}</Text>
          <View style={styles.infoRow}>
            <Text style={styles.label}>Name:</Text>
            <Text style={styles.value}>{userData.fullName}</Text>
          </View>
          <View style={styles.infoRow}>
            <Text style={styles.label}>Role:</Text>
            <Text style={styles.value}>{userData.role}</Text>
          </View>
          <View style={styles.infoRow}>
            <Text style={styles.label}>Email:</Text>
            <Text style={styles.value}>{userData.email}</Text>
          </View>
          <View style={styles.infoRow}>
            <Text style={styles.label}>Admin Email:</Text>
            <Text style={styles.value}>{userData.adminEmail}</Text>
          </View>
          <View style={styles.infoRow}>
            <Text style={styles.label}>Team Member ID:</Text>
            <Text style={styles.value}>{userData.teamMemberId}</Text>
          </View>
          <View style={styles.infoRow}>
            <Text style={styles.label}>User ID:</Text>
            <Text style={styles.value}>{userData.userId}</Text>
          </View>
          <View style={styles.infoRow}>
            <Text style={styles.label}>Employee ID:</Text>
            <Text style={styles.value}>{userData.employeeNumber}</Text>
          </View>
          <View style={styles.infoRow}>
            <Text style={styles.label}>Project:</Text>
            <Text style={styles.value}>{userData.project}</Text>
          </View>
        </View>

        {/* Logout Button */}
        <TouchableOpacity
          style={styles.logoutButton}
          onPress={() => setIsLogoutModalVisible(true)}
          accessibilityRole="button"
          accessibilityLabel="Logout"
          accessibilityHint="Log out of your account"
        >
          <Text style={styles.logoutText}>LOGOUT</Text>
        </TouchableOpacity>

        {/* Logout Confirmation Modal */}
        <Modal
          visible={isLogoutModalVisible}
          transparent
          animationType="fade"
          onRequestClose={() => setIsLogoutModalVisible(false)}
        >
          <View style={styles.modalOverlay}>
            <View style={styles.modalContent}>
              <Text style={styles.modalTitle}>Logout Confirmation</Text>
              <Text style={styles.modalMessage}>Are you sure you want to logout?</Text>
              <View style={styles.modalButtons}>
                <TouchableOpacity
                  style={[styles.modalButton, styles.cancelButton]}
                  onPress={() => setIsLogoutModalVisible(false)}
                  accessibilityRole="button"
                  accessibilityLabel="Cancel Logout"
                  accessibilityHint="Cancel the logout action"
                >
                  <Text style={styles.modalButtonText}>Cancel</Text>
                </TouchableOpacity>
                <TouchableOpacity
                  style={[styles.modalButton, styles.submitButton]}
                  onPress={handleLogout}
                  accessibilityRole="button"
                  accessibilityLabel="Confirm Logout"
                  accessibilityHint="Confirm and proceed with logout"
                >
                  <Text style={styles.modalButtonText}>Logout</Text>
                </TouchableOpacity>
              </View>
            </View>
          </View>
        </Modal>

        {/* Processing Modal */}
        <SyncModal
          visible={isProcessingModalVisible}
          onClose={() => setIsProcessingModalVisible(false)}
        />
        <Toast />
      </Animated.View>
    </SafeAreaView>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  contentContainer: {
    flex: 1,
  },
  loadingContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
  },
  loadingText: {
    marginTop: 10,
    fontSize: 16,
    color: '#333',
  },
  userSection: {
    backgroundColor: '#015185',
    padding: 20,
    alignItems: 'center',
  },
  userLabel: {
    color: '#fff',
    fontSize: 16,
    fontWeight: 'bold',
  },
  profileImageContainer: {
    marginVertical: 10,
    alignItems: 'center',
  },
  profileImage: {
    width: 60,
    height: 60,
    borderRadius: 30,
    borderWidth: 2,
    borderColor: '#fff',
  },
  profileIcon: {
    width: 60,
    height: 60,
  },
  email: {
    color: '#fff',
    fontSize: 14,
    marginVertical: 5,
  },
  infoRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    width: '100%',
    marginTop: 10,
  },
  label: {
    color: '#fff',
    fontSize: 14,
    fontWeight: 'bold',
    flex: 1,
  },
  value: {
    color: '#fff',
    fontSize: 14,
    textAlign: 'right',
    flex: 1,
  },
  logoutButton: {
    alignItems: 'center',
    paddingVertical: 15,
    marginTop: 20,
  },
  logoutText: {
    fontSize: 16,
    fontWeight: 'bold',
    color: '#FF0000',
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    width: '80%',
    alignItems: 'center',
  },
  modalTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#333',
    marginBottom: 10,
  },
  modalMessage: {
    fontSize: 16,
    color: '#333',
    marginBottom: 20,
    textAlign: 'center',
  },
  modalButtons: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    width: '100%',
  },
  modalButton: {
    flex: 1,
    paddingVertical: 10,
    borderRadius: 5,
    marginHorizontal: 5,
    alignItems: 'center',
  },
  cancelButton: {
    backgroundColor: '#888',
  },
  submitButton: {
    backgroundColor: '#015185',
  },
  modalButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: 'bold',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    color: '#333',
    marginBottom: 10,
  },
  syncIcon: {
    width: 30,
    height: 30,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
});

export default TeamAccount;