const functions = require('firebase-functions/v1');
const admin = require('firebase-admin');
const nodemailer = require('nodemailer');

// Initialize Firebase Admin if not already initialized
if (!admin.apps.length) {
  admin.initializeApp();
}

const db = admin.database();

// Email configuration - Using Gmail SMTP server
const emailConfig = {
  host: 'smtp.gmail.com',
  port: 587,
  secure: false,
  auth: {
    user: 'mandeep.owninfotech@gmail.com',
    pass: 'aeqa purb mxtl hkhh',
  },
};

// Create transporter
const transporter = nodemailer.createTransport(emailConfig);

/**
 * Global Request Maintenance Notification Function
 * Handles both email and push notifications for maintenance requests
 * Triggers when maintenance request is created at /GlobalRequestMaintenancereport/admins/{userId}/{requestId}
 */
exports.GlobalRequestmaintenanceNotify = functions.database
  .ref('/GlobalRequestMaintenancereport/admins/{userId}/{requestId}')
  .onCreate(async (snapshot, context) => {
    try {
      console.log('🔧 FIREBASE FUNCTION TRIGGERED! 🔧');
      console.log('Path:', context.params);
      console.log('Snapshot exists:', !!snapshot);
      
      const requestData = snapshot.val();
      const { userId, requestId } = context.params;

      console.log('Maintenance request created:', { userId, requestId, requestData });

      if (!requestData) {
        console.log('No request data found');
        return null;
      }

      // Get the maintenance request details
      const {
        requestNumber,
        workOrderTitle,
        memo,
        priority,
        country,
        project,
        assignTo,
        requestType,
        equipment,
        tasks,
        createdAt,
        userId: creatorUserId
      } = requestData;
      
      if (!requestNumber || !workOrderTitle) {
        console.log('Missing required request data');
        return null;
      }
      
      // Debug logging
      console.log('=== MAINTENANCE REQUEST EMAIL NOTIFICATION DEBUG ===');
      console.log('Extracted data:', {
        requestNumber,
        workOrderTitle,
        priority,
        country,
        project,
        requestType,
        equipment: equipment?.equipmentName || 'N/A'
      });

      // Get ALL users to notify (including admins, supervisors, and regular users)
      const allUsersToNotify = await getAllUsersToNotify();
      
      // Get request creator's email
      const requestCreator = await getRequestCreator(userId);

      // Prepare email content
      const emailContent = prepareMaintenanceEmailContent({
        requestData,
        requestNumber,
        workOrderTitle,
        priority,
        country,
        project,
        requestType,
        equipment,
        assignTo,
        memo,
        tasks,
        createdAt,
        requestCreator
      });

      // Log all recipients for debugging
      console.log('=== EMAIL RECIPIENTS SUMMARY ===');
      console.log('ProAdmins (primary):', allUsersToNotify.proAdmins.length);
      console.log('Admins (BCC):', allUsersToNotify.admins.length);
      console.log('Request creator:', requestCreator ? 'Yes' : 'No');
      console.log('Total recipients:', allUsersToNotify.totalRecipients + (requestCreator ? 1 : 0));
      console.log('================================');

      // Send emails to ProAdmins (primary recipients) with Admins in BCC
      if (allUsersToNotify.proAdmins.length > 0) {
        console.log(`📧 Sending emails to ${allUsersToNotify.proAdmins.length} ProAdmins with ${allUsersToNotify.admins.length} Admins in BCC...`);
        
        // Get all admin emails for BCC
        const bccEmails = allUsersToNotify.admins.map(admin => admin.email);
        
        // Send to first ProAdmin as primary recipient, others in CC, all Admins in BCC
        const primaryRecipient = allUsersToNotify.proAdmins[0];
        const ccRecipients = allUsersToNotify.proAdmins.slice(1);
        
        const emailResult = await sendEmailWithBCC(
          primaryRecipient, 
          ccRecipients, 
          bccEmails, 
          emailContent, 
          'Maintenance Request Notification'
        );
        console.log('📧 Email sending result:', emailResult);
      } else if (allUsersToNotify.admins.length > 0) {
        // If no ProAdmins, send to first Admin as primary with others in CC
        console.log(`📧 No ProAdmins found. Sending to ${allUsersToNotify.admins.length} Admins...`);
        const primaryRecipient = allUsersToNotify.admins[0];
        const ccRecipients = allUsersToNotify.admins.slice(1);
        
        const emailResult = await sendEmailWithBCC(
          primaryRecipient, 
          ccRecipients, 
          [], 
          emailContent, 
          'Maintenance Request Notification'
        );
        console.log('📧 Email sending result:', emailResult);
      }

      // Send email to request creator
      if (requestCreator) {
        console.log(`📧 Sending email to request creator: ${requestCreator.email}`);
        const creatorResult = await sendBulkEmails([requestCreator], emailContent, 'Maintenance Request - Your Request Created');
        console.log('📧 Request creator email result:', creatorResult);
      }

      // 🔔 SEND FCM PUSH NOTIFICATIONS TO ALL USERS
      console.log('🔔 Starting FCM push notifications...');
      await sendFCMNotificationsToAllUsers(requestData, allUsersToNotify, requestCreator, {
        requestNumber,
        workOrderTitle,
        priority,
        country,
        project,
        requestType,
        equipment: equipment?.equipmentName || 'N/A'
      });

      console.log('✅ Email notifications and FCM push notifications sent successfully');
      return null;

    } catch (error) {
      console.error('Error in GlobalRequestmaintenanceNotify:', error);
      return null;
    }
  });

/**
 * Get ALL users with admin/supervisor roles - NO country/project checking
 * This function gets EVERYONE based ONLY on role and email
 * Optimized to complete within 4 seconds
 */
async function getAllUsersToNotify() {
  try {
    console.log('🔍 Getting ALL users to notify (based on role only) - 4 second timeout');
    
    // Add timeout to ensure function completes within 4 seconds
    const timeoutPromise = new Promise((_, reject) => {
      setTimeout(() => reject(new Error('Function timeout: Exceeded 4 seconds')), 4000);
    });
    
    const userFetchPromise = (async () => {
    const usersSnapshot = await db.ref('/Globalusers/admins').once('value');
    const users = usersSnapshot.val() || {};
    
    console.log(`Total users found in database: ${Object.keys(users).length}`);
    
      const proAdmins = []; // Primary recipients
      const admins = []; // BCC recipients
    const skippedUsers = [];

    for (const [userId, userData] of Object.entries(users)) {
      if (!userData) {
        console.log(`Skipping user ${userId}: No user data`);
        continue;
      }

      const { role, email, name } = userData;
      
      // Skip users without email
      if (!email) {
        console.log(`Skipping user ${userId}: No email address`);
        skippedUsers.push({ userId, reason: 'No email address' });
        continue;
      }

      // Check user role (case-insensitive)
      const userRole = role || '';
        const { isProAdmin, isAdmin } = checkRoleFast(userRole);
      
      // Debug: Log exact role value for troubleshooting
      console.log(`\n👤 Checking user ${userId}:`);
      console.log(`   Name: ${name || 'Unknown'}`);
      console.log(`   Email: ${email}`);
      console.log(`   Role: ${userRole}`);
        console.log(`   Is ProAdmin: ${isProAdmin}`);
      console.log(`   Is Admin: ${isAdmin}`);
      
      if (isProAdmin) {
          // ProAdmins get emails as primary recipients
          proAdmins.push({
          email,
          name: name || 'Unknown User',
          role: userRole,
          userId,
            reason: 'ProAdmin - primary recipient'
          });
          console.log(`   ✅ ADDED to ProAdmin list (primary)`);
        } else if (isAdmin) {
          // Admins get emails as BCC recipients
          admins.push({
          email,
          name: name || 'Unknown User',
          role: userRole,
          userId,
            reason: 'Admin - BCC recipient'
        });
          console.log(`   ✅ ADDED to Admin list (BCC)`);
      } else {
          // Regular users don't get emails
          skippedUsers.push({ userId, name, role: userRole, reason: 'Regular user - no email based on role' });
          console.log(`   ❌ SKIPPED (regular user)`);
        }
      }
    
    console.log(`\n📊 FINAL RESULTS:`);
    console.log(`Total users in database: ${Object.keys(users).length}`);
      console.log(`ProAdmins (primary): ${proAdmins.length}`);
      console.log(`Admins (BCC): ${admins.length}`);
    console.log(`Users skipped: ${skippedUsers.length}`);
    
      console.log(`\n📧 PROADMINS (PRIMARY RECIPIENTS):`);
      proAdmins.forEach((user, index) => {
      console.log(`${index + 1}. ${user.name} (${user.role}) - ${user.email}`);
        console.log(`   Reason: ${user.reason}`);
    });
    
      console.log(`\n📧 ADMINS (BCC RECIPIENTS):`);
      admins.forEach((user, index) => {
      console.log(`${index + 1}. ${user.name} (${user.role}) - ${user.email}`);
        console.log(`   Reason: ${user.reason}`);
    });
    
    console.log(`\n❌ SKIPPED USERS:`);
    skippedUsers.forEach((user, index) => {
      console.log(`${index + 1}. ${user.name || 'Unknown'} (${user.role || 'No role'}) - ${user.reason}`);
    });
    
      return {
        proAdmins,
        admins,
        totalRecipients: proAdmins.length + admins.length
      };
    })();
    
    // Race between timeout and user fetch
    const result = await Promise.race([userFetchPromise, timeoutPromise]);
    console.log('✅ User fetch completed successfully within 4 seconds');
    return result;

  } catch (error) {
    if (error.message.includes('timeout')) {
      console.error('❌ Function timeout: Exceeded 4 seconds limit');
      console.log('⚠️ Returning empty results due to timeout');
      return { proAdmins: [], admins: [], totalRecipients: 0 };
    } else {
    console.error('❌ Error getting all users to notify:', error);
      return { proAdmins: [], admins: [], totalRecipients: 0 };
    }
  }
}

/**
 * Fast role checking function - optimized for speed
 */
function checkRoleFast(role) {
  if (!role || typeof role !== 'string') return { isProAdmin: false, isAdmin: false };
  
  const roleLower = role.toLowerCase();
  
  // Check for ProAdmin first (most important)
  const isProAdmin = roleLower.includes('proadmin') || 
                     roleLower.includes('pro admin') || 
                     roleLower === 'proadmin';
  
  // Check for Admin (if not ProAdmin)
  const isAdmin = !isProAdmin && (
    roleLower.includes('admin') || 
    roleLower.includes('administrator') || 
    roleLower.includes('ceo') || 
    roleLower.includes('manager')
  );
  
  return { isProAdmin, isAdmin };
}

/**
 * Get request creator's information
 */
async function getRequestCreator(userId) {
  try {
    console.log('Getting request creator for userId:', userId);
    
    const userSnapshot = await db.ref(`/Globalusers/admins/${userId}`).once('value');
    const userData = userSnapshot.val();
    
    if (!userData || !userData.email) {
      console.log('No user data or email found for userId:', userId);
      return null;
    }
    
    const requestCreator = {
      email: userData.email,
      name: userData.name || userData.fullName || 'Request Creator',
      role: userData.role || 'User',
      userId: userId
    };
    
    console.log('Found request creator:', requestCreator);
    return requestCreator;
    
  } catch (error) {
    console.error('Error getting request creator:', error);
    return null;
  }
}

/**
 * Prepare maintenance request email content
 */
function prepareMaintenanceEmailContent(data) {
  const {
    requestData,
    requestNumber,
    workOrderTitle,
    priority,
    country,
    project,
    requestType,
    equipment,
    assignTo,
    memo,
    tasks,
    createdAt,
    requestCreator
  } = data;

  const emailSubject = `Maintenance Request: ${requestNumber} - ${priority || 'Maintenance Request'}`;
  
  const emailBody = `
    <html>
      <body style="font-family: Arial, sans-serif; line-height: 1.6; color: #333;">
        <div style="max-width: 600px; margin: 0 auto; padding: 20px;">
          <div style="background-color: #3481BC; color: white; padding: 20px; text-align: center; border-radius: 8px 8px 0 0;">
            <h1 style="margin: 0; font-size: 24px;">Maintenance Request</h1>
            <p style="margin: 10px 0 0 0; font-size: 16px;">${requestNumber}</p>
          </div>
          
          <div style="background-color: #f8f9fa; padding: 20px; border: 1px solid #dee2e6;">
            <h2 style="color: #3481BC; margin-top: 0;">Request Summary</h2>
            
            <table style="width: 100%; border-collapse: collapse; margin-bottom: 20px;">
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold; width: 40%;">Request Number:</td>
                <td style="padding: 8px;">${requestNumber}</td>
              </tr>
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold;">Job card Title:</td>
                <td style="padding: 8px;">${workOrderTitle}</td>
              </tr>
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold;">Country:</td>
                <td style="padding: 8px;">${country || 'N/A'}</td>
              </tr>
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold;">Project:</td>
                <td style="padding: 8px;">${project || 'N/A'}</td>
              </tr>
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold;">Equipment:</td>
                <td style="padding: 8px;">${equipment?.equipmentName || 'N/A'}</td>
              </tr>
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold;">Priority:</td>
                <td style="padding: 8px;">${priority || 'N/A'}</td>
              </tr>
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold;">Request Type:</td>
                <td style="padding: 8px;">${requestType || 'N/A'}</td>
              </tr>
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold;">Created By:</td>
                <td style="padding: 8px;">${requestCreator?.name || 'Unknown'}</td>
              </tr>
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold;">Date:</td>
                <td style="padding: 8px;">${new Date(createdAt).toLocaleDateString()}</td>
              </tr>
              ${assignTo && assignTo.length > 0 ? `
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold;">Assigned To:</td>
                <td style="padding: 8px;">${assignTo.map(a => a.fullName || a.name).join(', ')}</td>
              </tr>
              ` : ''}
              ${memo ? `
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold;">Memo:</td>
                <td style="padding: 8px;">${memo}</td>
              </tr>
              ` : ''}
            </table>
            
            ${tasks && tasks.length > 0 ? `
              <div style="background-color: #e9ecef; padding: 15px; border-radius: 5px; margin: 20px 0;">
                <h3 style="color: #3481BC; margin-top: 0;">Tasks (${tasks.length})</h3>
                ${tasks.map((task, index) => `
                  <div style="margin-bottom: 10px; padding: 10px; background-color: white; border-radius: 3px;">
                    <strong>Task ${index + 1}:</strong> ${task.title || 'N/A'}<br>
                    <small style="color: #666;">Type: ${task.description || 'N/A'}</small>
                    ${task.assignees && task.assignees.length > 0 ? `
                      <br><small style="color: #666;">Assigned to: ${task.assignees.map(a => a.fullName).join(', ')}</small>
                    ` : ''}
                  </div>
                `).join('')}
              </div>
            ` : ''}
            
            <div style="background-color: #e9ecef; padding: 15px; border-radius: 5px; margin-top: 20px;">
              <p style="margin: 0; font-size: 14px; color: #6c757d; text-align: center;">
                This is an automated notification from the Titan Drilling Maintenance Management System.<br>
                Please do not reply to this email.
              </p>
            </div>
          </div>
        </div>
      </body>
    </html>
  `;

  return {
    subject: emailSubject,
    html: emailBody,
    text: convertHtmlToText(emailBody)
  };
}

/**
 * Convert HTML to plain text
 */
function convertHtmlToText(html) {
  return html
    .replace(/<[^>]*>/g, '')
    .replace(/&nbsp;/g, ' ')
    .replace(/&amp;/g, '&')
    .replace(/&lt;/g, '<')
    .replace(/&gt;/g, '>')
    .replace(/&quot;/g, '"')
    .replace(/\s+/g, ' ')
    .trim();
}

/**
 * Send bulk emails
 */
async function sendBulkEmails(recipients, emailContent, emailType) {
  try {
    console.log(`🚀 Starting email queue for ${emailType}`);
    console.log(`📧 Total recipients: ${recipients.length}`);
    
    if (recipients.length === 0) {
      console.log('⚠️ No recipients to send emails to');
      return { success: true, sent: 0, failed: 0, results: [] };
    }
    
    const results = [];
    let successCount = 0;
    let failureCount = 0;
    
    // Process emails in batches to avoid overwhelming the SMTP server
    const batchSize = 3; // Send 3 emails at a time
    const batches = [];
    
    for (let i = 0; i < recipients.length; i += batchSize) {
      batches.push(recipients.slice(i, i + batchSize));
    }
    
    console.log(`📦 Processing ${batches.length} batches of ${batchSize} emails each`);
    
    for (let batchIndex = 0; batchIndex < batches.length; batchIndex++) {
      const batch = batches[batchIndex];
      console.log(`\n📦 Processing batch ${batchIndex + 1}/${batches.length} (${batch.length} emails)`);
      
      // Process batch with delay to avoid rate limiting
      const batchPromises = batch.map(async (recipient, index) => {
        try {
          console.log(`\n📧 Sending email ${batchIndex * batchSize + index + 1}/${recipients.length} to: ${recipient.email}`);
          
          const mailOptions = {
            from: "Titan Drilling Maintenance Management System <support@titandrillingzm.com>",
            to: recipient.email,
            subject: `Maintenance Request - ${emailType}`,
            html: emailContent.html,
            text: emailContent.text,
            attachments: emailContent.attachments || []
          };

          const info = await transporter.sendMail(mailOptions);
          
          const result = {
            email: recipient.email,
            name: recipient.name,
            success: true,
            messageId: info.messageId,
            batch: batchIndex + 1,
            emailNumber: batchIndex * batchSize + index + 1
          };
          
          successCount++;
          console.log(`✅ Email sent successfully to ${recipient.email} (Message ID: ${info.messageId})`);
          
          // Log successful email
          await logEmailSent(recipient, emailType, info.messageId);
          
          return result;
          
        } catch (error) {
          console.error(`❌ Failed to send email to ${recipient.email}:`, error);
          
          const result = {
            email: recipient.email,
            name: recipient.name,
            success: false,
            error: error.message,
            batch: batchIndex + 1,
            emailNumber: batchIndex * batchSize + index + 1
          };
          
          failureCount++;
          return result;
        }
      });
      
      // Wait for all emails in this batch to complete
      const batchResults = await Promise.all(batchPromises);
      results.push(...batchResults);
      
      // Add delay between batches to avoid overwhelming SMTP server
      if (batchIndex < batches.length - 1) {
        console.log(`⏳ Waiting 3 seconds before next batch...`);
        await new Promise(resolve => setTimeout(resolve, 3000));
      }
    }
    
    console.log(`\n🎉 EMAIL QUEUE COMPLETED!`);
    console.log(`✅ Successful: ${successCount}`);
    console.log(`❌ Failed: ${failureCount}`);
    console.log(`📊 Total: ${recipients.length}`);
    
    return {
      success: true,
      sent: successCount,
      failed: failureCount,
      total: recipients.length,
      results: results
    };
    
  } catch (error) {
    console.error('❌ Critical error in email queue:', error);
    return {
      success: false,
      sent: 0,
      failed: recipients.length,
      total: recipients.length,
      error: error.message,
      results: []
    };
  }
}

/**
 * Send email with BCC
 */
async function sendEmailWithBCC(primaryRecipient, ccRecipients, bccRecipients, emailContent, emailType) {
  try {
    console.log(`🚀 Sending email with BCC for ${emailType}`);
    console.log(`📧 Primary Recipient: ${primaryRecipient.email}`);
    console.log(`📧 CC Recipients: ${ccRecipients.length}`);
    console.log(`📧 BCC Recipients: ${bccRecipients.length}`);

    const mailOptions = {
      from: "Titan Drilling Maintenance Management System <support@titandrillingzm.com>",
      to: primaryRecipient.email,
      cc: ccRecipients.length > 0 ? ccRecipients.map(r => r.email) : undefined,
      bcc: bccRecipients.length > 0 ? bccRecipients : undefined,
      subject: `Maintenance Request - ${emailType}`,
      html: emailContent.html,
      text: emailContent.text,
      attachments: emailContent.attachments || []
    };

    const info = await transporter.sendMail(mailOptions);
    console.log(`✅ Email sent successfully with BCC to ${primaryRecipient.email} (Message ID: ${info.messageId})`);
    
    // Log email sent for primary recipient
    await logEmailSent(primaryRecipient, emailType, info.messageId);
    
    // Log emails sent for CC recipients
    for (const ccRecipient of ccRecipients) {
      await logEmailSent(ccRecipient, emailType, info.messageId);
    }
    
    // Log emails sent for BCC recipients
    for (const bccRecipient of bccRecipients) {
      await logEmailSent({ email: bccRecipient, name: 'BCC Recipient', role: 'Admin' }, emailType, info.messageId);
    }
    
    return { 
      success: true, 
      messageId: info.messageId,
      primaryRecipient: primaryRecipient.email,
      ccCount: ccRecipients.length,
      bccCount: bccRecipients.length
    };
  } catch (error) {
    console.error(`❌ Failed to send email with BCC to ${primaryRecipient.email}:`, error);
    return { success: false, error: error.message };
  }
}

/**
 * Log email sent in Firebase
 */
async function logEmailSent(recipient, emailType, messageId) {
  try {
    const emailLog = {
      recipient: recipient.email,
      recipientName: recipient.name,
      recipientRole: recipient.role,
      emailType,
      messageId,
      sentAt: admin.database.ServerValue.TIMESTAMP,
      status: 'sent'
    };

    await db.ref('/emailLogs/maintenanceRequests').push(emailLog);
  } catch (error) {
    console.error('Error logging email:', error);
  }
}

/**
 * Send FCM push notifications to all users
 */
async function sendFCMNotificationsToAllUsers(requestData, allUsersToNotify, requestCreator, extractedData) {
  try {
    console.log('=== SENDING FCM PUSH NOTIFICATIONS ===');
    
    const { requestNumber, workOrderTitle, priority, country, project, requestType, equipment } = extractedData;
    
    // Get all users with FCM tokens
    const allUsersSnapshot = await admin.database()
      .ref('/Globalusers/admins')
      .once('value');
    
    if (!allUsersSnapshot.exists()) {
      console.log('⚠️ No admin users found in database');
      return { success: false, error: 'No admin users found' };
    }
    
    const allUsers = allUsersSnapshot.val() || {};
    const results = [];
    let totalUsers = 0;
    let successCount = 0;
    let failureCount = 0;
    let noDevicesCount = 0;
    let invalidTokenCount = 0;
    
    // Prepare push notification data
    const pushData = {
      type: 'maintenance_request_created',
      requestId: requestData.id || 'unknown',
      country: country,
      project: project,
      requestNumber: requestNumber,
      workOrderTitle: workOrderTitle,
      priority: priority,
      requestType: requestType,
      equipment: equipment,
      creatorEmail: requestCreator?.email || 'Unknown',
      creatorName: requestCreator?.name || 'Unknown',
      source: 'maintenance_request',
      priority: 'high'
    };
    
    const pushTitle = `🔧 New Maintenance Request: ${requestNumber}`;
    const pushBody = `${requestCreator?.name || 'Unknown'} created maintenance request for ${project} - ${equipment || 'Equipment'}`;
    
    // Process each user
    for (const [userId, userData] of Object.entries(allUsers)) {
      totalUsers++;
      
      try {
        console.log(`🔍 Checking user ${userId} (${userData.name || 'Unknown'}) for FCM tokens...`);
        
        let devices = {};
        
        // First check if there are devices in the new multi-device structure
        if (userData.devices && Object.keys(userData.devices).length > 0) {
          devices = userData.devices;
          console.log(`📱 User ${userId} has ${Object.keys(devices).length} devices in multi-device structure:`, Object.keys(devices));
        }
        // Fallback to the old single FCM token structure
        else if (userData.NotificationFCMtoken) {
          devices = {
            FCM: {
              token: userData.NotificationFCMtoken,
              fcmStatus: 'active',
              lastTokenUpdate: userData.lastTokenUpdate || new Date().toISOString(),
              deviceInfo: {
                platform: 'legacy',
                deviceId: 'legacy',
                timestamp: new Date().toISOString()
              }
            }
          };
          console.log(`📱 User ${userId} has legacy FCM token structure`);
        }
        // Check for fcmTokens array (another legacy structure)
        else if (userData.fcmTokens && Array.isArray(userData.fcmTokens) && userData.fcmTokens.length > 0) {
          devices = {};
          userData.fcmTokens.forEach((token, index) => {
            if (token && token.trim() !== '') {
              devices[`FCM_${index}`] = {
                token: token,
                fcmStatus: 'active',
                lastTokenUpdate: new Date().toISOString(),
                deviceInfo: {
                  platform: 'legacy_array',
                  deviceId: `legacy_${index}`,
                  timestamp: new Date().toISOString()
                }
              };
            }
          });
          console.log(`📱 User ${userId} has ${Object.keys(devices).length} tokens in fcmTokens array`);
        }
        
        if (Object.keys(devices).length === 0) {
          noDevicesCount++;
          console.log(`⚠️ User ${userId} has no FCM tokens`);
          results.push({ userId, status: 'no_devices', message: 'No FCM tokens found' });
          continue;
        }
        
        // Send notification to each device
        const deviceResults = [];
        let deviceSuccessCount = 0;
        let deviceFailureCount = 0;
        
        for (const [deviceKey, deviceData] of Object.entries(devices)) {
          if (deviceData.token && deviceData.fcmStatus === 'active') {
            try {
              const fcmResult = await sendFcmToToken(
                deviceData.token, 
                pushTitle,
                pushBody,
                pushData,
                deviceKey
              );
              
              if (fcmResult.success) {
                deviceSuccessCount++;
                deviceResults.push({ 
                  deviceKey, 
                  status: 'success', 
                  messageId: fcmResult.messageId 
                });
              } else {
                deviceFailureCount++;
                deviceResults.push({ 
                  deviceKey, 
                  status: 'failed', 
                  error: fcmResult.error 
                });
                
                // If token is invalid, mark device as inactive
                if (fcmResult.code === 'messaging/invalid-registration-token' || 
                    fcmResult.code === 'messaging/registration-token-not-registered') {
                  await markDeviceAsInactive(userId, deviceKey);
                  invalidTokenCount++;
                }
              }
            } catch (error) {
              deviceFailureCount++;
              deviceResults.push({ 
                deviceKey, 
                status: 'error', 
                error: error.message 
              });
            }
          }
        }
        
        if (deviceSuccessCount > 0) {
          successCount++;
          results.push({ 
            userId, 
            status: 'success', 
            devices: deviceResults,
            successCount: deviceSuccessCount,
            failureCount: deviceFailureCount
          });
        } else {
          failureCount++;
          results.push({ 
            userId, 
            status: 'failed', 
            devices: deviceResults,
            error: 'All device notifications failed'
          });
        }
        
      } catch (error) {
        failureCount++;
        console.error(`❌ Error processing user ${userId}:`, error);
        results.push({ userId, status: 'error', error: error.message });
      }
    }
    
    console.log(`📊 Push notification summary: ${successCount}/${totalUsers} users successful, ${failureCount} failed, ${noDevicesCount} no devices, ${invalidTokenCount} invalid tokens`);
    
    // Store push notification record
    const pushRecord = {
      requestId: requestData.id || 'unknown',
      type: 'maintenance_request_created',
      title: pushTitle,
      body: pushBody,
      data: pushData,
      sentAt: new Date().toISOString(),
      totalUsers,
      successCount,
      failureCount,
      noDevicesCount,
      invalidTokenCount,
      results,
      status: 'completed'
    };
    
    await admin.database().ref(`/GlobalPushNotifications/maintenance_requests/${requestData.id || 'unknown'}`).set(pushRecord);
    
    return {
      success: true,
      totalUsers,
      successCount,
      failureCount,
      noDevicesCount,
      invalidTokenCount,
      message: 'Push notifications sent successfully'
    };
    
  } catch (error) {
    console.error('Error sending FCM push notifications:', error);
    return {
      success: false,
      error: error.message,
      message: 'Failed to send push notifications'
    };
  }
}

/**
 * Send FCM to token helper function
 */
async function sendFcmToToken(token, title, body, data = {}, deviceKey = 'unknown') {
  try {
    const message = {
      token: token,
      notification: {
        title: title,
        body: body,
      },
      data: {
        type: data.type || 'maintenance_request_created',
        requestId: data.requestId || 'unknown',
        country: data.country || '',
        project: data.project || '',
        requestNumber: data.requestNumber || '',
        workOrderTitle: data.workOrderTitle || '',
        priority: data.priority || '',
        requestType: data.requestType || '',
        equipment: data.equipment || '',
        creatorEmail: data.creatorEmail || '',
        creatorName: data.creatorName || '',
        source: data.source || 'maintenance_request',
        priority: data.priority || 'high',
        deviceKey: deviceKey,
        timestamp: new Date().toISOString()
      },
      android: {
        notification: {
          channelId: 'maintenance_requests',
          importance: 'high',
          priority: 'high',
          icon: 'ic_notification',
          color: '#3481BC',
          sound: 'default',
          vibrateTimingsMillis: [0, 250, 250, 250],
          lightSettings: {
            color: '#3481BC',
            lightOnDurationMillis: 1000,
            lightOffDurationMillis: 1000,
          },
        },
        priority: 'high',
      },
      apns: {
        payload: {
          aps: {
            'content-available': 1,
            'mutable-content': 1,
            sound: 'default',
            badge: 1,
            alert: {
              title: title,
              body: body,
            },
            category: 'MAINTENANCE_REQUEST',
          },
        },
        headers: {
          'apns-priority': '10',
          'apns-push-type': 'alert',
        },
      },
    };

    console.log(`📱 Sending maintenance FCM notification to device ${deviceKey} with token:`, token.substring(0, 20) + '...');
    const response = await admin.messaging().send(message);
    
    console.log(`✅ FCM maintenance notification sent successfully to device ${deviceKey}`);
    return { success: true, messageId: response };
    
  } catch (error) {
    console.error(`❌ Error sending FCM maintenance notification to device ${deviceKey}:`, error);
    
    // Handle specific FCM errors
    if (error.code === 'messaging/invalid-registration-token' || 
        error.code === 'messaging/registration-token-not-registered') {
      console.log(`🔄 FCM token is invalid or expired for device ${deviceKey}, marking as inactive`);
      return { success: false, error: 'Invalid or expired FCM token', code: error.code };
    }
    
    return { success: false, error: error.message, code: error.code };
  }
}

/**
 * Mark device as inactive when FCM token is invalid
 */
async function markDeviceAsInactive(userId, deviceKey) {
  try {
    const deviceRef = admin.database().ref(`/Globalusers/admins/${userId}/devices/${deviceKey}`);
    await deviceRef.update({
      fcmStatus: 'inactive',
      lastError: 'Invalid FCM token',
      lastErrorAt: new Date().toISOString()
    });
    console.log(`🔄 Marked device ${deviceKey} as inactive for user ${userId}`);
  } catch (error) {
    console.error(`❌ Error marking device ${deviceKey} as inactive for user ${userId}:`, error);
  }
}

module.exports = { GlobalRequestmaintenanceNotify: exports.GlobalRequestmaintenanceNotify };