const functions = require('firebase-functions/v1');
const admin = require('firebase-admin');
const nodemailer = require('nodemailer');

// Initialize Firebase Admin if not already initialized
if (!admin.apps.length) {
  admin.initializeApp();
}

// Mail transport (same credentials as other functions)
const mailTransport = nodemailer.createTransport({
  host: 'smtp.gmail.com',
  port: 587,
  secure: false,
  auth: {
    user: 'mandeep.owninfotech@gmail.com',
    pass: 'aeqa purb mxtl hkhh',
  },
});

async function sendEmail(toEmail, subject, html) {
  const mailOptions = {
    from: 'Titan Drilling <support@titandrillingzm.com>',
    to: toEmail,
    subject,
    html,
  };
  return mailTransport.sendMail(mailOptions);
}

// Helper to send FCM message to a single token
async function sendFcmToToken(token, title, body, data = {}) {
  const message = {
    token,
    notification: { title, body },
    data: Object.entries(data).reduce((acc, [k, v]) => ({ ...acc, [k]: String(v) }), {}),
    android: { priority: 'high' },
    apns: { payload: { aps: { sound: 'default' } } },
  };
  return admin.messaging().send(message);
}

// Role helpers (small copy of logic used elsewhere to keep behaviour consistent)
function isManagerRole(role) {
  if (!role) return false;
  if (role.toLowerCase().includes('manager')) return true;
  const managerRoles = ['CEO','Country Manager','Manager','Project Manager','Operations Manager','Site Manager','Field Manager','Regional Manager','Area Manager','Department Manager','Team Manager','Senior Manager','General Manager','Executive Manager'];
  return managerRoles.includes(role);
}

function isAlwaysEligibleRole(role) {
  return ['admin','ProAdmin','CEO'].includes(role);
}

function isEligibleRoleForHSE(role) {
  if (!role) return false;
  const eligible = ['admin','ProAdmin','Supervisor','CEO','Country Manager','Manager','Project Manager','Operations Manager','Site Manager','Field Manager','Regional Manager','Area Manager','Department Manager','Team Manager','Senior Manager','General Manager','Executive Manager'];
  if (eligible.includes(role)) return true;
  if (role.toLowerCase().includes('manager')) return true;
  return false;
}

// Main function: triggers when an HSE report node is created
exports.sendHseReportNotify = functions
  .runWith({ memory: '512MB', timeoutSeconds: 120 })
  .database
  .ref('/GlobalHSEInspections/admins/{userId}/{reportId}')
  .onCreate(async (snap, ctx) => {
    const reportData = snap.val();
    const { userId, reportId } = ctx.params;
    const now = new Date().toISOString();

    if (!reportData) {
      console.log('No HSE report data found');
      return null;
    }

    try {
      // Get report creator details
      const creatorSnapshot = await admin.database().ref(`/Globalusers/admins/${userId}`).once('value');
      const creatorData = creatorSnapshot.val() || {};
      const creatorEmail = creatorData.email || '';
      const creatorName = creatorData.name || 'Team Member';

      // Extract useful fields (step1/step3 expected)
      const step1 = reportData.step1 || reportData.selectedStep1 || {};
      const step3 = reportData.step3 || {};
      const country = step1.country || '';
      const project = step1.project || '';
      const checklistTitle = step1.selectedChecklist?.document_name || step1.checklistTitle || 'HSE Inspection Report';
      const pdfUrl = reportData.pdfDownloadUrl || step3.pdfDownloadUrl || null;

      // Prepare email content
      const emailSubject = `HSE Report: ${reportId || 'N/A'} - ${checklistTitle}`;
      const emailHtml = `
        <div style="font-family: Arial, sans-serif; max-width:600px; margin:0 auto;">
          <div style="background:#015185;color:white;padding:15px;text-align:center"><h2 style="margin:0">HSE Inspection Report</h2></div>
          <div style="padding:15px;background:#f9f9f9">
            <p>Report ID: <strong>${reportId}</strong></p>
            <p>Checklist: <strong>${checklistTitle}</strong></p>
            <p>Country / Project: <strong>${country} / ${project}</strong></p>
            <p>Created by: <strong>${creatorName}</strong></p>
            ${pdfUrl ? `<p style="text-align:center;margin:20px 0"><a href="${pdfUrl}" style="background:#015185;color:#fff;padding:10px 16px;border-radius:6px;text-decoration:none">View PDF Report</a></p>` : '<p>No PDF available yet.</p>'}
            <p>This is an automated message from Titan Drilling.</p>
          </div>
        </div>
      `;

      // Collect recipients: ProAdmins/CEO always, Supervisors/Managers matching country+project
      const allAdminsSnap = await admin.database().ref('/Globalusers/admins').once('value');
      const allAdmins = allAdminsSnap.val() || {};
      const proAdminEmails = [];
      const supervisorEmails = [];

      Object.entries(allAdmins).forEach(([adminId, adminData]) => {
        if (!adminData || !adminData.email) return;
        const role = adminData.role || '';
        const userCountries = adminData.countries || [];
        const userProjects = adminData.projects || [];

        if (role === 'ProAdmin' || role === 'CEO') {
          proAdminEmails.push(adminData.email);
        } else if (role === 'Supervisor' || isManagerRole(role) || role.toLowerCase().includes('manager')) {
          const hasCountry = Array.isArray(userCountries) && userCountries.includes(country);
          const hasProject = Array.isArray(userProjects) && userProjects.includes(project);
          if (hasCountry && hasProject) {
            supervisorEmails.push(adminData.email);
          }
        }
      });

      // Send email to creator
      if (creatorEmail) {
        try {
          await sendEmail(creatorEmail, emailSubject, emailHtml);
          console.log('Email sent to creator:', creatorEmail);
        } catch (e) {
          console.error('Failed to send email to creator:', e);
        }
      }

      // Send to proAdmins and supervisors (dedupe)
      const toEmails = Array.from(new Set([...(proAdminEmails || []), ...(supervisorEmails || [])]));
      for (const to of toEmails) {
        if (to && to !== creatorEmail) {
          try {
            await sendEmail(to, emailSubject, emailHtml);
            console.log('Email sent to:', to);
          } catch (e) {
            console.error('Failed to send email to:', to, e);
          }
        }
      }

      // Prepare push notification payload
      const notificationTitle = `📝 HSE Report: ${reportId || ''}`;
      const notificationBody = `${creatorName} submitted ${checklistTitle} for ${project || 'Unknown Project'}`;

      // Filter users for push: similar to email filtering but require active devices
      const eligibleUsers = [];
      for (const [adminId, adminData] of Object.entries(allAdmins)) {
        if (!adminData) continue;
        const role = adminData.role || '';
        if (!isEligibleRoleForHSE(role)) continue;

        // If always eligible, collect
        if (isAlwaysEligibleRole(role)) {
          // collect devices
        }

        const userCountries = adminData.countries || [];
        const userProjects = adminData.projects || [];

        if (isAlwaysEligibleRole(role) || ((role === 'Supervisor' || isManagerRole(role)) && Array.isArray(userCountries) && Array.isArray(userProjects) && userCountries.includes(country) && userProjects.includes(project))) {
          const devices = adminData.devices || {};
          const activeDevices = Object.entries(devices).filter(([_, d]) => d && (d.token || d.fcmToken) && (d.fcmStatus === 'active' || d.fcmStatus === undefined));
          if (activeDevices.length > 0) {
            eligibleUsers.push({ adminId, adminData, devices: activeDevices });
          }
        }
      }

      // Send notifications and save to user's notifications node
      for (const user of eligibleUsers) {
        try {
          // Save notification in DB
          const notificationRef = admin.database().ref(`/Globalusers/admins/${user.adminId}/notifications`).push();
          await notificationRef.set({
            title: notificationTitle,
            body: notificationBody.substring(0, 200),
            type: 'hse_report',
            data: { reportId, action: 'view_hse_report', pdfUrl: pdfUrl || '' },
            timestamp: Date.now(),
            isRead: false,
            createdAt: now,
          });

          // Send FCM to each device
          for (const [deviceKey, deviceObj] of user.devices) {
            const token = deviceObj.token || deviceObj.fcmToken;
            if (!token) continue;
            try {
              await sendFcmToToken(token, notificationTitle, notificationBody, { reportId, action: 'view_hse_report' });
              console.log(`FCM sent to ${user.adminId} device ${deviceKey}`);
            } catch (fcmErr) {
              console.error(`Failed FCM to ${user.adminId} device ${deviceKey}:`, fcmErr);
              if (fcmErr && (fcmErr.code === 'messaging/invalid-registration-token' || fcmErr.code === 'messaging/registration-token-not-registered')) {
                await admin.database().ref(`/Globalusers/admins/${user.adminId}/devices/${deviceKey}/fcmStatus`).set('inactive');
              }
            }
          }
        } catch (err) {
          console.error('Error notifying user:', user.adminId, err);
        }
      }

      // Store audit record for HSE report notifications
      const auditRecord = {
        reportId,
        userId,
        creatorEmail,
        creatorName,
        country,
        project,
        pdfUrl: pdfUrl || null,
        emailsSentTo: toEmails,
        pushNotifiedCount: eligibleUsers.length,
        sentAt: now,
        status: 'sent'
      };
      await admin.database().ref(`/GlobalHSENotifications/${reportId}`).set(auditRecord);

      console.log('HSE report notifications processing completed for', reportId);

      return { success: true, reportId, emails: toEmails.length, pushes: eligibleUsers.length };

    } catch (error) {
      console.error('Error in sendHseReportNotify:', error);
      await snap.ref.update({ notificationError: error.message || String(error), notificationErrorAt: now });
      return { success: false, error: error.message };
    }
  });


