const functions = require('firebase-functions/v1');
const admin = require('firebase-admin');
const nodemailer = require('nodemailer');

// Initialize Firebase Admin if not already initialized
if (!admin.apps.length) {
  admin.initializeApp();
}

const db = admin.database();

// Email configuration - Using Gmail SMTP server
const emailConfig = {
  host: 'smtp.gmail.com',
  port: 587,
  secure: false,
  auth: {
    user: 'mandeep.owninfotech@gmail.com',
    pass: 'aeqa purb mxtl hkhh',
  },
};

// Create transporter
const transporter = nodemailer.createTransport(emailConfig);

/**
 * Firebase Function: Inspection Report Email Notification
 * DISABLED - Now using sendInspectionReportEmails in index.js instead
 * This prevents duplicate triggers and memory issues
 */
exports.inspectionReportEmailNotify = functions.database
  .ref('/GlobalInspectionReport/admins/{userId}/{reportId}')
  .onCreate(async (snapshot, context) => {
    console.log('⚠️ inspectionReportEmailNotify is DISABLED - using sendInspectionReportEmails instead');
    return null;
    
    /* DISABLED CODE BELOW - DO NOT EXECUTE
    
    // Old code commented out to prevent execution
    try {
      console.log('🔥 FIREBASE FUNCTION TRIGGERED! 🔥');
      console.log('Path:', context.params);
      console.log('Snapshot exists:', !!snapshot);
      
      const reportData = snapshot.val();
      const { userId, reportId } = context.params;

      console.log('Inspection report created:', { userId, reportId, reportData });

      if (!reportData) {
        console.log('No report data found');
        return null;
      }

      // Get the inspection report details
      const { step1, step2, step3 } = reportData;
      
      if (!step1) {
        console.log('No step1 data found in report');
        return null;
      }
      
      const {
        country,
        project,
        inspector,
        selectedEquipment,
        requestNumber,
        coordinates,
        gpsAddress
      } = step1;
      
      const checklistTitle = step2?.checklistTitle || 'Inspection Report';
      const createdAt = step3?.createdAt || new Date().toISOString();
      
      // Debug logging
      console.log('=== EMAIL NOTIFICATION DEBUG ===');
      console.log('Extracted data:', {
        country,
        project,
        inspector,
        requestNumber,
        checklistTitle,
        equipmentName: selectedEquipment?.equipmentName
      });

      // Get inspection results from step2 data
      const inspectionResults = step2 || {};
      
      // Get PDF URL from step3 data
      const pdfUrl = step3?.pdfDownloadUrl || null;

      // Get ALL users to notify (including admins, supervisors, and regular users)
      const allUsersToNotify = await getAllUsersToNotify();
      
      // Get report creator's email
      const reportCreator = await getReportCreator(userId);

      // Prepare email content
      const emailContent = prepareEmailContent({
        reportData,
        inspectionResults,
        pdfUrl,
        requestNumber,
        country,
        project,
        inspector,
        selectedEquipment,
        checklistTitle,
        createdAt,
        coordinates,
        gpsAddress
      });

      // Log all recipients for debugging
      console.log('=== EMAIL RECIPIENTS SUMMARY ===');
      console.log('ProAdmins (primary):', allUsersToNotify.proAdmins.length);
      console.log('Admins (BCC):', allUsersToNotify.admins.length);
      console.log('Report creator:', reportCreator ? 'Yes' : 'No');
      console.log('Total recipients:', allUsersToNotify.totalRecipients + (reportCreator ? 1 : 0));
      console.log('================================');

      // Send emails to ProAdmins (primary recipients) with Admins in BCC
      if (allUsersToNotify.proAdmins.length > 0) {
        console.log(`📧 Sending emails to ${allUsersToNotify.proAdmins.length} ProAdmins with ${allUsersToNotify.admins.length} Admins in BCC...`);
        
        // Get all admin emails for BCC
        const bccEmails = allUsersToNotify.admins.map(admin => admin.email);
        
        // Send to first ProAdmin as primary recipient, others in CC, all Admins in BCC
        const primaryRecipient = allUsersToNotify.proAdmins[0];
        const ccRecipients = allUsersToNotify.proAdmins.slice(1);
        
        const emailResult = await sendEmailWithBCC(
          primaryRecipient, 
          ccRecipients, 
          bccEmails, 
          emailContent, 
          'Inspection Report Notification'
        );
        console.log('📧 Email sending result:', emailResult);
      } else if (allUsersToNotify.admins.length > 0) {
        // If no ProAdmins, send to first Admin as primary with others in CC
        console.log(`📧 No ProAdmins found. Sending to ${allUsersToNotify.admins.length} Admins...`);
        const primaryRecipient = allUsersToNotify.admins[0];
        const ccRecipients = allUsersToNotify.admins.slice(1);
        
        const emailResult = await sendEmailWithBCC(
          primaryRecipient, 
          ccRecipients, 
          [], 
          emailContent, 
          'Inspection Report Notification'
        );
        console.log('📧 Email sending result:', emailResult);
      }

      // Send email to report creator
      if (reportCreator) {
        console.log(`📧 Sending email to report creator: ${reportCreator.email}`);
        const creatorResult = await sendBulkEmails([reportCreator], emailContent, 'Inspection Report - Your Report Created');
        console.log('📧 Report creator email result:', creatorResult);
      }

      // 🔔 SEND FCM PUSH NOTIFICATIONS TO ALL USERS
      console.log('🔔 Starting FCM push notifications...');
      await sendFCMNotificationsToAllUsers(reportData, allUsersToNotify, reportCreator, {
        country,
        project,
        inspector,
        requestNumber,
        selectedEquipment,
        checklistTitle
      });

      console.log('✅ Email notifications and FCM push notifications sent successfully');
      return null;

    } catch (error) {
      console.error('Error in inspectionReportEmailNotify:', error);
      return null;
    }
    
    END OF DISABLED CODE */
  });

/**
 * Get inspection results from Firebase
 */
async function getInspectionResults(userId, reportId) {
  try {
    const snapshot = await db.ref(`/inspectionResults/${userId}/${reportId}`).once('value');
    return snapshot.val() || {};
  } catch (error) {
    console.error('Error getting inspection results:', error);
    return {};
  }
}

/**
 * Get PDF URL from Firebase Storage
 */
async function getPDFUrl(userId, reportId) {
  try {
    const bucket = admin.storage().bucket();
    const file = bucket.file(`inspection-reports/${userId}/${reportId}/report.pdf`);
    const [url] = await file.getSignedUrl({
      action: 'read',
      expires: Date.now() + 7 * 24 * 60 * 60 * 1000 // 7 days
    });
    return url;
  } catch (error) {
    console.error('Error getting PDF URL:', error);
    return null;
  }
}

/**
 * Get users to notify based on country and project permissions
 */
async function getUsersToNotify(country, project) {
  try {
    const usersSnapshot = await db.ref('/Globalusers/admins').once('value');
    const users = usersSnapshot.val() || {};
    
    const usersToNotify = [];

    for (const [userId, userData] of Object.entries(users)) {
      if (!userData || !userData.modules) continue;

      // Check if user has permissions for this country/project
      const hasPermission = checkUserPermissions(userData, country, project);
      
      if (hasPermission) {
        usersToNotify.push({
          email: userData.email,
          name: userData.name,
          role: userData.role,
          userId
        });
      }
    }

    console.log(`Found ${usersToNotify.length} users to notify for ${country}/${project}`);
    return usersToNotify;

  } catch (error) {
    console.error('Error getting users to notify:', error);
    return [];
  }
}

/**
 * Check if user has permissions for the given country and project
 */
function checkUserPermissions(userData, country, project) {
  try {
    const { countries, projects, modules } = userData;
    
    // Check if user has access to the country
    if (!countries || !Array.isArray(countries) || !countries.includes(country)) {
      return false;
    }
    
    // Check if user has access to the project
    if (!projects || !Array.isArray(projects) || !projects.includes(project)) {
      return false;
    }
    
    // Check if user has inspection report module permissions
    if (!modules || !Array.isArray(modules)) {
      return false;
    }
    
    const hasInspectionModule = modules.some(module => {
      if (typeof module === 'string') {
        return module === 'inspectionReport' || module === 'Inspection Report';
      } else if (module.module) {
        return module.module === 'inspectionReport' || module.module === 'Inspection Report';
      }
      return false;
    });
    
    if (!hasInspectionModule) {
      return false;
    }
    
    return true;

  } catch (error) {
    console.error('Error checking user permissions:', error);
    return false;
  }
}

/**
 * Fast role checking function - optimized for speed
 */
function checkRoleFast(role) {
  if (!role || typeof role !== 'string') return { isProAdmin: false, isAdmin: false };
  
  const roleLower = role.toLowerCase();
  
  // Check for ProAdmin first (most important)
  const isProAdmin = roleLower.includes('proadmin') || 
                     roleLower.includes('pro admin') || 
                     roleLower === 'proadmin';
  
  // Check for Admin (if not ProAdmin)
  const isAdmin = !isProAdmin && (
    roleLower.includes('admin') || 
    roleLower.includes('administrator') || 
    roleLower.includes('ceo') || 
    roleLower.includes('manager')
  );
  
  return { isProAdmin, isAdmin };
}

/**
 * Get ALL users with admin/supervisor roles - NO country/project checking
 * This function gets EVERYONE based ONLY on role and email
 * Optimized to complete within 4 seconds
 */
async function getAllUsersToNotify() {
  try {
    console.log('🔍 Getting ALL users to notify (based on role only) - 4 second timeout');
    
    // Add timeout to ensure function completes within 4 seconds
    const timeoutPromise = new Promise((_, reject) => {
      setTimeout(() => reject(new Error('Function timeout: Exceeded 4 seconds')), 4000);
    });
    
    const userFetchPromise = (async () => {
    const usersSnapshot = await db.ref('/Globalusers/admins').once('value');
    const users = usersSnapshot.val() || {};
    
    console.log(`Total users found in database: ${Object.keys(users).length}`);
    
      const proAdmins = []; // Primary recipients
      const admins = []; // BCC recipients
    const skippedUsers = [];

    for (const [userId, userData] of Object.entries(users)) {
      if (!userData) {
        console.log(`Skipping user ${userId}: No user data`);
        continue;
      }

      const { role, email, name } = userData;
      
      // Skip users without email
      if (!email) {
        console.log(`Skipping user ${userId}: No email address`);
        skippedUsers.push({ userId, reason: 'No email address' });
        continue;
      }

      // Check user role (case-insensitive)
      const userRole = role || '';
        const { isProAdmin, isAdmin } = checkRoleFast(userRole);
      
      // Debug: Log exact role value for troubleshooting
      console.log(`\n👤 Checking user ${userId}:`);
      console.log(`   Name: ${name || 'Unknown'}`);
      console.log(`   Email: ${email}`);
      console.log(`   Role: ${userRole}`);
        console.log(`   Is ProAdmin: ${isProAdmin}`);
      console.log(`   Is Admin: ${isAdmin}`);
      
      if (isProAdmin) {
          // ProAdmins get emails as primary recipients
          proAdmins.push({
          email,
          name: name || 'Unknown User',
          role: userRole,
          userId,
            reason: 'ProAdmin - primary recipient'
          });
          console.log(`   ✅ ADDED to ProAdmin list (primary)`);
        } else if (isAdmin) {
          // Admins get emails as BCC recipients
          admins.push({
          email,
          name: name || 'Unknown User',
          role: userRole,
          userId,
            reason: 'Admin - BCC recipient'
        });
          console.log(`   ✅ ADDED to Admin list (BCC)`);
      } else {
          // Regular users don't get emails
          skippedUsers.push({ userId, name, role: userRole, reason: 'Regular user - no email based on role' });
          console.log(`   ❌ SKIPPED (regular user)`);
        }
      }
    
    console.log(`\n📊 FINAL RESULTS:`);
    console.log(`Total users in database: ${Object.keys(users).length}`);
      console.log(`ProAdmins (primary): ${proAdmins.length}`);
      console.log(`Admins (BCC): ${admins.length}`);
    console.log(`Users skipped: ${skippedUsers.length}`);
    
      console.log(`\n📧 PROADMINS (PRIMARY RECIPIENTS):`);
      proAdmins.forEach((user, index) => {
      console.log(`${index + 1}. ${user.name} (${user.role}) - ${user.email}`);
        console.log(`   Reason: ${user.reason}`);
    });
    
      console.log(`\n📧 ADMINS (BCC RECIPIENTS):`);
      admins.forEach((user, index) => {
      console.log(`${index + 1}. ${user.name} (${user.role}) - ${user.email}`);
        console.log(`   Reason: ${user.reason}`);
    });
    
    console.log(`\n❌ SKIPPED USERS:`);
    skippedUsers.forEach((user, index) => {
      console.log(`${index + 1}. ${user.name || 'Unknown'} (${user.role || 'No role'}) - ${user.reason}`);
    });
    
      return {
        proAdmins,
        admins,
        totalRecipients: proAdmins.length + admins.length
      };
    })();
    
    // Race between timeout and user fetch
    const result = await Promise.race([userFetchPromise, timeoutPromise]);
    console.log('✅ User fetch completed successfully within 4 seconds');
    return result;

  } catch (error) {
    if (error.message.includes('timeout')) {
      console.error('❌ Function timeout: Exceeded 4 seconds limit');
      console.log('⚠️ Returning empty results due to timeout');
      return { proAdmins: [], admins: [], totalRecipients: 0 };
    } else {
    console.error('❌ Error getting all users to notify:', error);
      return { proAdmins: [], admins: [], totalRecipients: 0 };
    }
  }
}

/**
 * Get admin and supervisor users with matching country+project permissions
 */
async function getAdminAndSupervisorUsers(country, project) {
  try {
    console.log('Getting admin and supervisor users for:', { country, project });
    
    const usersSnapshot = await db.ref('/Globalusers/admins').once('value');
    const users = usersSnapshot.val() || {};
    
    const adminAndSupervisorUsers = [];

    for (const [userId, userData] of Object.entries(users)) {
      if (!userData) continue;

      const { role, countries, projects, email, name, modules } = userData;
      
      console.log(`Checking user ${userId} for admin/supervisor:`, { role, countries, projects, email, name });
      
      // Check if user is admin or supervisor (case-insensitive)
      const userRole = role || '';
      const isAdmin = userRole.toLowerCase().includes('admin') || userRole.toLowerCase().includes('administrator');
      const isSupervisor = userRole.toLowerCase().includes('Supervisor');
      
      if (!isAdmin && !isSupervisor) {
        console.log(`User ${userId} is not admin/supervisor (role: ${userRole})`);
        continue;
      }
      
      console.log(`User ${userId} is admin/supervisor (role: ${userRole})`);
      
      // Check if user has access to the EXACT country from inspection report
      const hasCountryAccess = countries && Array.isArray(countries) && countries.includes(country);
      console.log(`User ${userId} country access for ${country}:`, hasCountryAccess, 'User countries:', countries);
      
      // Check if user has access to the EXACT project from inspection report
      const hasProjectAccess = projects && Array.isArray(projects) && projects.includes(project);
      console.log(`User ${userId} project access for ${project}:`, hasProjectAccess, 'User projects:', projects);
      
      // Both country AND project must match exactly
      if (hasCountryAccess && hasProjectAccess) {
        console.log(`User ${userId} has both country and project access`);
        
        // Check if user has inspection report module access
        const hasInspectionModule = modules && Array.isArray(modules) && modules.some(module => {
          if (typeof module === 'string') {
            return module === 'inspectionReport' || module === 'Inspection Report';
          } else if (module.module) {
            return module.module === 'inspectionReport' || module.module === 'Inspection Report';
          }
          return false;
        });
        
        console.log(`User ${userId} inspection module access:`, hasInspectionModule, 'User modules:', modules);
        
        if (hasInspectionModule) {
          adminAndSupervisorUsers.push({
            email,
            name: name || 'Unknown User',
            role: userRole,
            userId,
            userCountries: countries,
            userProjects: projects
          });
          console.log(`User ${userId} added to admin/supervisor list`);
        } else {
          console.log(`User ${userId} does not have inspection report module access`);
        }
      } else {
        console.log(`User ${userId} does not have both country and project access`);
      }
    }

    console.log(`=== ADMIN/SUPERVISOR RESULT ===`);
    console.log(`Found ${adminAndSupervisorUsers.length} admin/supervisor users for ${country}/${project}:`);
    adminAndSupervisorUsers.forEach(user => {
      console.log(`- ${user.name} (${user.role}): ${user.email}`);
      console.log(`  Countries: ${user.userCountries?.join(', ')}`);
      console.log(`  Projects: ${user.userProjects?.join(', ')}`);
    });
    console.log(`=================================`);
    
    return adminAndSupervisorUsers;

  } catch (error) {
    console.error('Error getting admin and supervisor users:', error);
    return [];
  }
}

/**
 * Get report creator's information
 */
async function getReportCreator(userId) {
  try {
    console.log('Getting report creator for userId:', userId);
    
    const userSnapshot = await db.ref(`/Globalusers/${userId}`).once('value');
    const userData = userSnapshot.val();
    
    if (!userData || !userData.email) {
      console.log('No user data or email found for userId:', userId);
      return null;
    }
    
    const reportCreator = {
      email: userData.email,
      name: userData.name || userData.fullName || 'Report Creator',
      role: userData.role || 'User',
      userId: userId
    };
    
    console.log('Found report creator:', reportCreator);
    return reportCreator;
    
  } catch (error) {
    console.error('Error getting report creator:', error);
    return null;
  }
}

/**
 * Prepare email content
 */
function prepareEmailContent(data) {
  const {
    reportData,
    inspectionResults,
    pdfUrl,
    requestNumber,
    country,
    project,
    inspector,
    selectedEquipment,
    checklistTitle,
    createdAt,
    coordinates,
    gpsAddress
  } = data;

  // Calculate summary statistics
  const totalItems = inspectionResults.sections ? 
    inspectionResults.sections.reduce((total, section) => total + (section.subItems?.length || 0), 0) : 0;
  
  const faultyItems = inspectionResults.faultyItems || 0;
  const passedItems = totalItems - faultyItems;

  const emailSubject = `Inspection Report: ${requestNumber} - ${checklistTitle}`;
  
  const emailBody = `
    <html>
      <body style="font-family: Arial, sans-serif; line-height: 1.6; color: #333;">
        <div style="max-width: 600px; margin: 0 auto; padding: 20px;">
          <div style="background-color: #015185; color: white; padding: 20px; text-align: center; border-radius: 8px 8px 0 0;">
            <h1 style="margin: 0; font-size: 24px;">Inspection Report</h1>
            <p style="margin: 10px 0 0 0; font-size: 16px;">${requestNumber}</p>
          </div>
          
          <div style="background-color: #f8f9fa; padding: 20px; border: 1px solid #dee2e6;">
            <h2 style="color: #015185; margin-top: 0;">Report Summary</h2>
            
            <table style="width: 100%; border-collapse: collapse; margin-bottom: 20px;">
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold; width: 40%;">Request Number:</td>
                <td style="padding: 8px;">${requestNumber}</td>
              </tr>
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold;">Country:</td>
                <td style="padding: 8px;">${country}</td>
              </tr>
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold;">Project:</td>
                <td style="padding: 8px;">${project}</td>
              </tr>
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold;">Inspector:</td>
                <td style="padding: 8px;">${inspector}</td>
              </tr>
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold;">Equipment:</td>
                <td style="padding: 8px;">${selectedEquipment?.equipmentName || 'N/A'}</td>
              </tr>
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold;">Checklist:</td>
                <td style="padding: 8px;">${checklistTitle}</td>
              </tr>
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold;">Date:</td>
                <td style="padding: 8px;">${new Date(createdAt).toLocaleDateString()}</td>
              </tr>
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold;">Location:</td>
                <td style="padding: 8px;">${coordinates || 'N/A'}</td>
              </tr>
            </table>
            
            <h3 style="color: #015185;">Inspection Results</h3>
            <div style="background-color: white; padding: 15px; border-radius: 5px; margin-bottom: 20px;">
              <div style="display: flex; justify-content: space-around; text-align: center;">
                <div>
                  <div style="font-size: 24px; font-weight: bold; color: #28a745;">${passedItems}</div>
                  <div style="font-size: 14px; color: #666;">Passed Items</div>
                </div>
                <div>
                  <div style="font-size: 24px; font-weight: bold; color: #dc3545;">${faultyItems}</div>
                  <div style="font-size: 14px; color: #666;">Faulty Items</div>
                </div>
                <div>
                  <div style="font-size: 24px; font-weight: bold; color: #015185;">${totalItems}</div>
                  <div style="font-size: 14px; color: #666;">Total Items</div>
                </div>
              </div>
            </div>
            
            ${pdfUrl ? `
              <div style="text-align: center; margin: 20px 0;">
                <a href="${pdfUrl}" style="background-color: #007bff; color: white; padding: 12px 24px; text-decoration: none; border-radius: 5px; display: inline-block;">
                  📄 View Full Report PDF
                </a>
              </div>
            ` : ''}
            
            <div style="background-color: #e9ecef; padding: 15px; border-radius: 5px; margin-top: 20px;">
              <p style="margin: 0; font-size: 14px; color: #6c757d; text-align: center;">
                This is an automated notification from the Titan Drilling Inspection System.<br>
                Please do not reply to this email.
              </p>
            </div>
          </div>
        </div>
      </body>
    </html>
  `;

  return {
    subject: emailSubject,
    html: emailBody,
    text: convertHtmlToText(emailBody)
  };
}

/**
 * Convert HTML to plain text
 */
function convertHtmlToText(html) {
  return html
    .replace(/<[^>]*>/g, '')
    .replace(/&nbsp;/g, ' ')
    .replace(/&amp;/g, '&')
    .replace(/&lt;/g, '<')
    .replace(/&gt;/g, '>')
    .replace(/&quot;/g, '"')
    .replace(/\s+/g, ' ')
    .trim();
}

/**
 * Send bulk emails
 */
async function sendBulkEmails(recipients, emailContent, emailType) {
  try {
    console.log(`🚀 Starting email queue for ${emailType}`);
    console.log(`📧 Total recipients: ${recipients.length}`);
    
    if (recipients.length === 0) {
      console.log('⚠️ No recipients to send emails to');
      return { success: true, sent: 0, failed: 0, results: [] };
    }
    
    const results = [];
    let successCount = 0;
    let failureCount = 0;
    
    // Process emails in batches to avoid overwhelming the SMTP server
    const batchSize = 3; // Send 3 emails at a time
    const batches = [];
    
    for (let i = 0; i < recipients.length; i += batchSize) {
      batches.push(recipients.slice(i, i + batchSize));
    }
    
    console.log(`📦 Processing ${batches.length} batches of ${batchSize} emails each`);
    
    for (let batchIndex = 0; batchIndex < batches.length; batchIndex++) {
      const batch = batches[batchIndex];
      console.log(`\n📦 Processing batch ${batchIndex + 1}/${batches.length} (${batch.length} emails)`);
      
      // Process batch with delay to avoid rate limiting
      const batchPromises = batch.map(async (recipient, index) => {
        try {
          console.log(`\n📧 Sending email ${batchIndex * batchSize + index + 1}/${recipients.length} to: ${recipient.email}`);
          
          const mailOptions = {
            from: "Titan Drilling Inspection System <support@titandrillingzm.com>",
            to: recipient.email,
            subject: `Inspection Report - ${emailType}`,
            html: emailContent.html,
            text: emailContent.text,
            attachments: emailContent.attachments || []
          };

          const info = await transporter.sendMail(mailOptions);
          
          const result = {
            email: recipient.email,
            name: recipient.name,
            success: true,
            messageId: info.messageId,
            batch: batchIndex + 1,
            emailNumber: batchIndex * batchSize + index + 1
          };
          
          successCount++;
          console.log(`✅ Email sent successfully to ${recipient.email} (Message ID: ${info.messageId})`);
          
          // Log successful email
          await logEmailSent(recipient, emailType, info.messageId);
          
          return result;
          
        } catch (error) {
          console.error(`❌ Failed to send email to ${recipient.email}:`, error);
          
          const result = {
            email: recipient.email,
            name: recipient.name,
            success: false,
            error: error.message,
            batch: batchIndex + 1,
            emailNumber: batchIndex * batchSize + index + 1
          };
          
          failureCount++;
          return result;
        }
      });
      
      // Wait for all emails in this batch to complete
      const batchResults = await Promise.all(batchPromises);
      results.push(...batchResults);
      
      // Add delay between batches to avoid overwhelming SMTP server
      if (batchIndex < batches.length - 1) {
        console.log(`⏳ Waiting 3 seconds before next batch...`);
        await new Promise(resolve => setTimeout(resolve, 3000));
      }
    }
    
    console.log(`\n🎉 EMAIL QUEUE COMPLETED!`);
    console.log(`✅ Successful: ${successCount}`);
    console.log(`❌ Failed: ${failureCount}`);
    console.log(`📊 Total: ${recipients.length}`);
    
    return {
      success: true,
      sent: successCount,
      failed: failureCount,
      total: recipients.length,
      results: results
    };
    
  } catch (error) {
    console.error('❌ Critical error in email queue:', error);
    return {
      success: false,
      sent: 0,
      failed: recipients.length,
      total: recipients.length,
      error: error.message,
      results: []
    };
  }
}

/**
 * Send email with BCC
 */
async function sendEmailWithBCC(primaryRecipient, ccRecipients, bccRecipients, emailContent, emailType) {
  try {
    console.log(`🚀 Sending email with BCC for ${emailType}`);
    console.log(`📧 Primary Recipient: ${primaryRecipient.email}`);
    console.log(`📧 CC Recipients: ${ccRecipients.length}`);
    console.log(`📧 BCC Recipients: ${bccRecipients.length}`);

    const mailOptions = {
      from: "Titan Drilling Inspection System <support@titandrillingzm.com>",
      to: primaryRecipient.email,
      cc: ccRecipients.length > 0 ? ccRecipients.map(r => r.email) : undefined,
      bcc: bccRecipients.length > 0 ? bccRecipients : undefined,
      subject: `Inspection Report - ${emailType}`,
      html: emailContent.html,
      text: emailContent.text,
      attachments: emailContent.attachments || []
    };

    const info = await transporter.sendMail(mailOptions);
    console.log(`✅ Email sent successfully with BCC to ${primaryRecipient.email} (Message ID: ${info.messageId})`);
    
    // Log email sent for primary recipient
    await logEmailSent(primaryRecipient, emailType, info.messageId);
    
    // Log emails sent for CC recipients
    for (const ccRecipient of ccRecipients) {
      await logEmailSent(ccRecipient, emailType, info.messageId);
    }
    
    // Log emails sent for BCC recipients
    for (const bccRecipient of bccRecipients) {
      await logEmailSent({ email: bccRecipient, name: 'BCC Recipient', role: 'Admin' }, emailType, info.messageId);
    }
    
    return { 
      success: true, 
      messageId: info.messageId,
      primaryRecipient: primaryRecipient.email,
      ccCount: ccRecipients.length,
      bccCount: bccRecipients.length
    };
  } catch (error) {
    console.error(`❌ Failed to send email with BCC to ${primaryRecipient.email}:`, error);
    return { success: false, error: error.message };
  }
}

/**
 * Log email sent in Firebase
 */
async function logEmailSent(recipient, emailType, messageId) {
  try {
    const emailLog = {
      recipient: recipient.email,
      recipientName: recipient.name,
      recipientRole: recipient.role,
      emailType,
      messageId,
      sentAt: admin.database.ServerValue.TIMESTAMP,
      status: 'sent'
    };

    await db.ref('/emailLogs/inspectionReports').push(emailLog);
  } catch (error) {
    console.error('Error logging email:', error);
  }
}

/**
 * Test function to debug user filtering based on country and project
 */
exports.testUserFiltering = functions.https.onCall(async (data, context) => {
  try {
    // Check if user is authenticated
    if (!context.auth) {
      throw new functions.https.HttpsError('unauthenticated', 'User must be authenticated');
    }

    const { country, project } = data;
    
    if (!country || !project) {
      throw new functions.https.HttpsError('invalid-argument', 'Country and project are required');
    }

    console.log('Testing user filtering for:', { country, project });

    // Test the filtering functions
    const usersToNotify = await getUsersToNotify(country, project);
    const supervisorsAndAdmins = await getSupervisorsAndAdmins(country, project);
    const adminAndSupervisorUsers = await getAdminAndSupervisorUsers(country, project);
    
    // Get all users for comparison
    const allUsersSnapshot = await db.ref('/Globalusers/admins').once('value');
    const allUsers = allUsersSnapshot.val() || {};
    
    const allUsersList = Object.entries(allUsers).map(([userId, userData]) => {
      const userCountries = userData.countries || [];
      const userProjects = userData.projects || [];
      const userRole = userData.role || 'No role';
      
      const hasCountry = userCountries.includes(country);
      const hasProject = userProjects.includes(project);
      const isSupervisor = userRole.toLowerCase().includes('Supervisor');
      const isAdmin = userRole.toLowerCase().includes('admin') || userRole.toLowerCase().includes('administrator');
      const isProAdmin = userRole.toLowerCase().includes('proadmin') || userRole.toLowerCase().includes('pro admin');
      
      return {
        userId,
        name: userData.name || 'Unknown',
        email: userData.email || 'No email',
        role: userRole,
        countries: userCountries,
        projects: userProjects,
        hasCountry,
        hasProject,
        isSupervisor,
        isAdmin,
        isProAdmin,
        shouldGetEmail: (isSupervisor || isAdmin || isProAdmin) && (hasCountry || hasProject),
        accessType: hasCountry && hasProject ? 'Full Access' : (hasCountry || hasProject ? 'Partial Access' : 'No Access')
      };
    });

    // Show detailed breakdown
    const supervisorsList = allUsersList.filter(u => u.isSupervisor);
    const adminsList = allUsersList.filter(u => u.isAdmin);
    const proAdminsList = allUsersList.filter(u => u.isProAdmin);
    const usersWithAccess = allUsersList.filter(u => u.hasCountry || u.hasProject);

    return {
      success: true,
      testData: {
        country,
        project,
        totalUsers: allUsersList.length,
        usersToNotify: usersToNotify.length,
        supervisorsAndAdmins: supervisorsAndAdmins.length,
        adminAndSupervisorUsers: adminAndSupervisorUsers.length,
        totalRecipients: usersToNotify.length + supervisorsAndAdmins.length + adminAndSupervisorUsers.length
      },
      breakdown: {
        supervisors: supervisorsList.length,
        admins: adminsList.length,
        proAdmins: proAdminsList.length,
        usersWithAccess: usersWithAccess.length
      },
      allUsers: allUsersList,
      filteredUsers: {
        usersToNotify,
        supervisorsAndAdmins,
        adminAndSupervisorUsers
      },
      debug: {
        supervisorsList,
        adminsList,
        proAdminsList,
        usersWithAccess
      }
    };

  } catch (error) {
    console.error('Error in test function:', error);
    throw new functions.https.HttpsError('internal', error.message);
  }
});

/**
 * Show all roles in database (no auth required)
 */
exports.showAllRoles = functions.https.onCall(async (data, context) => {
  try {
    console.log('🔍 Showing all roles in database');
    
    // Get all users
    const usersSnapshot = await db.ref('/Globalusers/admins').once('value');
    const users = usersSnapshot.val() || {};
    
    console.log(`📊 Found ${Object.keys(users).length} users in database`);
    
    // Get unique roles with detailed analysis
    const allRoles = new Set();
    const roleCounts = {};
    const roleAnalysis = {};
    
    Object.entries(users).forEach(([userId, userData]) => {
      const role = userData.role || 'No role';
      allRoles.add(role);
      roleCounts[role] = (roleCounts[role] || 0) + 1;
      
      // Analyze each role for email eligibility
      const roleLower = role.toLowerCase();
      const isSupervisor = roleLower.includes('Supervisor');
      const isProAdmin = roleLower.includes('proadmin') || roleLower.includes('pro admin');
      const isAdmin = roleLower.includes('admin') || roleLower.includes('administrator') || roleLower.includes('ceo') || roleLower.includes('manager');
      
      roleAnalysis[role] = {
        isSupervisor,
        isProAdmin,
        isAdmin,
        shouldGetEmail: isSupervisor || isProAdmin || isAdmin,
        emailReason: isSupervisor || isProAdmin || isAdmin ? 
          `${role} - gets email based on role` : 
          `${role} - no email (regular user)`
      };
    });
    
    const uniqueRoles = Array.from(allRoles).sort();
    
    console.log(`\n📋 UNIQUE ROLES FOUND:`);
    uniqueRoles.forEach(role => {
      const analysis = roleAnalysis[role];
      console.log(`   ${role}: ${roleCounts[role]} users - ${analysis.emailReason}`);
    });
    
    // Count email-eligible users
    const totalEmailUsers = Object.values(roleAnalysis).filter(a => a.shouldGetEmail).length;
    const totalNonEmailUsers = Object.values(roleAnalysis).filter(a => !a.shouldGetEmail).length;
    
    console.log(`\n📧 EMAIL ELIGIBILITY SUMMARY:`);
    console.log(`   Total users: ${Object.keys(users).length}`);
    console.log(`   Users who will get emails: ${totalEmailUsers}`);
    console.log(`   Users who won't get emails: ${totalNonEmailUsers}`);
    
    return {
      success: true,
      totalUsers: Object.keys(users).length,
      uniqueRoles: uniqueRoles,
      roleCounts: roleCounts,
      roleAnalysis: roleAnalysis,
      emailSummary: {
        totalUsers: Object.keys(users).length,
        emailEligible: totalEmailUsers,
        notEmailEligible: totalNonEmailUsers
      }
    };

  } catch (error) {
    console.error('❌ Error showing roles:', error);
    throw new functions.https.HttpsError('internal', error.message);
  }
});

/**
 * Simple test function to check user data structure (no auth required for debugging)
 */
exports.debugUserData = functions.https.onCall(async (data, context) => {
  try {
    console.log('🔍 Debug user data (based on role only)');
    
    // Get all users
    const usersSnapshot = await db.ref('/Globalusers/admins').once('value');
    const users = usersSnapshot.val() || {};
    
    console.log(`📊 Found ${Object.keys(users).length} users in database`);
    
          const debugData = Object.entries(users).map(([userId, userData]) => {
        const userRole = userData.role || 'No role';
        const isSupervisor = userRole.toLowerCase().includes('Supervisor');
        const isAdmin = userRole.toLowerCase().includes('admin') || userRole.toLowerCase().includes('administrator') || userRole.toLowerCase().includes('ceo') || userRole.toLowerCase().includes('manager');
        const isProAdmin = userRole.toLowerCase().includes('proadmin') || userRole.toLowerCase().includes('pro admin');
      
      return {
        userId,
        name: userData.name || 'Unknown',
        email: userData.email || 'No email',
        role: userRole,
        isSupervisor,
        isAdmin,
        isProAdmin,
        shouldGetEmail: isSupervisor || isAdmin || isProAdmin,
        reason: isSupervisor || isAdmin || isProAdmin ? 
          `${userRole} - gets email based on role` : 
          'Regular user - no email based on role'
      };
    });

    // Filter users who should get emails
    const usersToNotify = debugData.filter(user => user.shouldGetEmail);
    const usersNotified = debugData.filter(user => !user.shouldGetEmail);

    console.log(`📧 Users who should get emails: ${usersToNotify.length}`);
    console.log(`❌ Users who won't get emails: ${usersNotified.length}`);

    return {
      success: true,
      totalUsers: debugData.length,
      usersToNotify: usersToNotify.length,
      usersNotified: usersNotified.length,
      allUsers: debugData,
      summary: {
        totalUsers: debugData.length,
        usersWithEmails: usersToNotify.length,
        usersWithoutEmails: usersNotified.length,
        supervisors: debugData.filter(u => u.isSupervisor).length,
        admins: debugData.filter(u => u.isAdmin).length,
        proAdmins: debugData.filter(u => u.isProAdmin).length
      }
    };

  } catch (error) {
    console.error('❌ Error in debug function:', error);
    throw new functions.https.HttpsError('internal', error.message);
  }
});

/**
 * Test function to verify role checking and email distribution
 */
exports.testRoleCheckingAndEmailDistribution = functions.https.onCall(async (data, context) => {
  try {
    console.log('🧪 Testing role checking and email distribution');
    
    // Test the fast role checking function
    const testRoles = [
      'proadmin',
      'ProAdmin',
      'pro admin',
      'Pro Admin',
      'admin',
      'Admin',
      'administrator',
      'Administrator',
      'ceo',
      'CEO',
      'manager',
      'Manager',
      'user',
      'User',
      'regular',
      'Regular'
    ];
    
    console.log('📋 Testing role checking:');
    testRoles.forEach(role => {
      const result = checkRoleFast(role);
      console.log(`   ${role}: ProAdmin=${result.isProAdmin}, Admin=${result.isAdmin}`);
    });
    
    // Test the user fetching function
    const usersResult = await getAllUsersToNotify();
    
    console.log('\n📊 User fetching results:');
    console.log(`   ProAdmins: ${usersResult.proAdmins.length}`);
    console.log(`   Admins: ${usersResult.admins.length}`);
    console.log(`   Total: ${usersResult.totalRecipients}`);
    
    // Show sample users
    if (usersResult.proAdmins.length > 0) {
      console.log('\n📧 Sample ProAdmins:');
      usersResult.proAdmins.slice(0, 3).forEach((user, index) => {
        console.log(`   ${index + 1}. ${user.name} (${user.role}) - ${user.email}`);
      });
    }
    
    if (usersResult.admins.length > 0) {
      console.log('\n📧 Sample Admins:');
      usersResult.admins.slice(0, 3).forEach((user, index) => {
        console.log(`   ${index + 1}. ${user.name} (${user.role}) - ${user.email}`);
      });
    }
    
    return {
      success: true,
      roleTestResults: testRoles.map(role => ({
        role,
        ...checkRoleFast(role)
      })),
      userResults: usersResult,
      message: 'Role checking and user fetching test completed successfully'
    };

  } catch (error) {
    console.error('❌ Error in test function:', error);
    throw new functions.https.HttpsError('internal', error.message);
  }
});

/**
 * Manual trigger function for testing
 */
exports.manualInspectionReportEmail = functions.https.onCall(async (data, context) => {
  try {
    // Check if user is authenticated
    if (!context.auth) {
      throw new functions.https.HttpsError('unauthenticated', 'User must be authenticated');
    }

    const { reportId, adminId } = data;
    
    if (!reportId || !adminId) {
      throw new functions.https.HttpsError('invalid-argument', 'reportId and adminId are required');
    }

    // Get the report data
    const reportSnapshot = await db.ref(`/GlobalInspectionReport/admins/${adminId}/${reportId}`).once('value');
    const reportData = reportSnapshot.val();
    
    if (!reportData) {
      throw new functions.https.HttpsError('not-found', 'Inspection report not found');
    }

    // Trigger the email notification manually
    const result = await exports.inspectionReportEmailNotify({
      ref: db.ref(`/GlobalInspectionReport/admins/${adminId}/${reportId}`),
      val: () => reportData
    }, { params: { adminId, reportId } });

    return { success: true, message: 'Email notification triggered successfully' };

      } catch (error) {
    console.error('Error in manual trigger:', error);
    throw new functions.https.HttpsError('internal', error.message);
  }
});

/**
 * Test function to verify FCM push notifications for inspection reports
 */
exports.testInspectionReportPushNotification = functions.https.onCall(async (data, context) => {
  try {
    // Check if user is authenticated
    if (!context.auth) {
      throw new functions.https.HttpsError('unauthenticated', 'User must be authenticated');
    }

    console.log('🧪 Testing inspection report push notification system...');
    
    // Create test report data
    const testReportData = {
      id: `test_${Date.now()}`,
      requestNumber: 'TEST-ZAM-KOB-IR-001',
      country: 'Zambia',
      project: 'Kobold',
      inspector: 'Test Inspector',
      selectedEquipment: {
        equipmentName: 'Test Equipment'
      },
      checklistTitle: 'Test Checklist',
      createdAt: new Date().toISOString()
    };

    // Get all users to notify
    const allUsersToNotify = await getAllUsersToNotify();
    
    // Send test FCM notifications
    const result = await sendFCMNotificationsToAllUsers(testReportData, allUsersToNotify, null);
    
    console.log('🧪 Test FCM notification result:', result);
    
    return {
      success: true,
      message: 'Test push notifications sent successfully',
      result: result,
      testData: testReportData
    };

  } catch (error) {
    console.error('❌ Error in test function:', error);
    throw new functions.https.HttpsError('internal', error.message);
  }
});

/**
 * Send FCM push notifications to all users
 */
async function sendFCMNotificationsToAllUsers(reportData, allUsersToNotify, reportCreator, extractedData) {
  try {
    console.log('=== SENDING FCM PUSH NOTIFICATIONS ===');
    
    const { country, project, inspector, requestNumber, selectedEquipment, checklistTitle } = extractedData;
    
    // Get all users with FCM tokens
    const allUsersSnapshot = await admin.database()
      .ref('/Globalusers/admins')
      .once('value');
    
    if (!allUsersSnapshot.exists()) {
      console.log('⚠️ No admin users found in database');
      return { success: false, error: 'No admin users found' };
    }
    
    const allUsers = allUsersSnapshot.val() || {};
    const results = [];
    let totalUsers = 0;
    let successCount = 0;
    let failureCount = 0;
    let noDevicesCount = 0;
    let invalidTokenCount = 0;
    
    // Prepare push notification data
    const pushData = {
      type: 'inspection_report_created',
      reportId: reportData.id || 'unknown',
      country: country,
      project: project,
      inspector: inspector,
      requestNumber: requestNumber,
      equipmentName: selectedEquipment?.equipmentName || 'N/A',
      checklistTitle: checklistTitle,
      creatorEmail: reportCreator?.email || 'Unknown',
      creatorName: reportCreator?.name || 'Unknown',
      source: 'inspection_report',
      priority: 'high'
    };
    
    const pushTitle = `🔍 New Inspection Report: ${requestNumber}`;
    const pushBody = `${inspector} created inspection report for ${project} - ${selectedEquipment?.equipmentName || 'Equipment'}`;
    
    // Process each user
    for (const [userId, userData] of Object.entries(allUsers)) {
      totalUsers++;
      
      try {
        console.log(`🔍 Checking user ${userId} (${userData.name || 'Unknown'}) for FCM tokens...`);
        
        let devices = {};
        
        // First check if there are devices in the new multi-device structure
        if (userData.devices && Object.keys(userData.devices).length > 0) {
          devices = userData.devices;
          console.log(`📱 User ${userId} has ${Object.keys(devices).length} devices in multi-device structure:`, Object.keys(devices));
        }
        // Fallback to the old single FCM token structure
        else if (userData.NotificationFCMtoken) {
          devices = {
            FCM: {
              token: userData.NotificationFCMtoken,
              fcmStatus: 'active',
              lastTokenUpdate: userData.lastTokenUpdate || new Date().toISOString(),
              deviceInfo: {
                platform: 'legacy',
                deviceId: 'legacy',
                timestamp: new Date().toISOString()
              }
            }
          };
          console.log(`📱 User ${userId} has legacy FCM token structure`);
        }
        // Check for fcmTokens array (another legacy structure)
        else if (userData.fcmTokens && Array.isArray(userData.fcmTokens) && userData.fcmTokens.length > 0) {
          devices = {};
          userData.fcmTokens.forEach((token, index) => {
            if (token && token.trim() !== '') {
              devices[`FCM_${index}`] = {
                token: token,
                fcmStatus: 'active',
                lastTokenUpdate: new Date().toISOString(),
                deviceInfo: {
                  platform: 'legacy_array',
                  deviceId: `legacy_${index}`,
                  timestamp: new Date().toISOString()
                }
              };
            }
          });
          console.log(`📱 User ${userId} has ${Object.keys(devices).length} tokens in fcmTokens array`);
        }
        
        if (Object.keys(devices).length === 0) {
          noDevicesCount++;
          console.log(`⚠️ User ${userId} has no FCM tokens`);
          results.push({ userId, status: 'no_devices', message: 'No FCM tokens found' });
          continue;
        }
        
        // Send notification to each device
        const deviceResults = [];
        let deviceSuccessCount = 0;
        let deviceFailureCount = 0;
        
        for (const [deviceKey, deviceData] of Object.entries(devices)) {
          if (deviceData.token && deviceData.fcmStatus === 'active') {
            try {
              const fcmResult = await sendFcmToToken(
                deviceData.token, 
                pushTitle,
                pushBody,
                pushData,
                deviceKey
              );
              
              if (fcmResult.success) {
                deviceSuccessCount++;
                deviceResults.push({ 
                  deviceKey, 
                  status: 'success', 
                  messageId: fcmResult.messageId 
                });
              } else {
                deviceFailureCount++;
                deviceResults.push({ 
                  deviceKey, 
                  status: 'failed', 
                  error: fcmResult.error 
                });
                
                // If token is invalid, mark device as inactive
                if (fcmResult.code === 'messaging/invalid-registration-token' || 
                    fcmResult.code === 'messaging/registration-token-not-registered') {
                  await markDeviceAsInactive(userId, deviceKey);
                  invalidTokenCount++;
                }
              }
            } catch (error) {
              deviceFailureCount++;
              deviceResults.push({ 
                deviceKey, 
                status: 'error', 
                error: error.message 
              });
            }
          }
        }
        
        if (deviceSuccessCount > 0) {
          successCount++;
          results.push({ 
            userId, 
            status: 'success', 
            devices: deviceResults,
            successCount: deviceSuccessCount,
            failureCount: deviceFailureCount
          });
        } else {
          failureCount++;
          results.push({ 
            userId, 
            status: 'failed', 
            devices: deviceResults,
            error: 'All device notifications failed'
          });
        }
        
      } catch (error) {
        failureCount++;
        console.error(`❌ Error processing user ${userId}:`, error);
        results.push({ userId, status: 'error', error: error.message });
      }
    }
    
    console.log(`📊 Push notification summary: ${successCount}/${totalUsers} users successful, ${failureCount} failed, ${noDevicesCount} no devices, ${invalidTokenCount} invalid tokens`);
    
    // Store push notification record
    const pushRecord = {
      reportId: reportData.id || 'unknown',
      type: 'inspection_report_created',
      title: pushTitle,
      body: pushBody,
      data: pushData,
      sentAt: new Date().toISOString(),
      totalUsers,
      successCount,
      failureCount,
      noDevicesCount,
      invalidTokenCount,
      results,
      status: 'completed'
    };
    
    await admin.database().ref(`/GlobalPushNotifications/inspection_reports/${reportData.id || 'unknown'}`).set(pushRecord);
    
    return {
      success: true,
      totalUsers,
      successCount,
      failureCount,
      noDevicesCount,
      invalidTokenCount,
      message: 'Push notifications sent successfully'
    };
    
  } catch (error) {
    console.error('Error sending FCM push notifications:', error);
    return {
      success: false,
      error: error.message,
      message: 'Failed to send push notifications'
    };
  }
}

/**
 * Send FCM to token helper function
 */
async function sendFcmToToken(token, title, body, data = {}, deviceKey = 'unknown') {
  try {
    const message = {
      token: token,
      notification: {
        title: title,
        body: body,
      },
      data: {
        type: data.type || 'inspection_report_created',
        reportId: data.reportId || 'unknown',
        country: data.country || '',
        project: data.project || '',
        inspector: data.inspector || '',
        requestNumber: data.requestNumber || '',
        equipmentName: data.equipmentName || '',
        equipmentStatus: data.equipmentStatus || '',
        faultyItems: data.faultyItems || '',
        creatorEmail: data.creatorEmail || '',
        creatorName: data.creatorName || '',
        source: data.source || 'inspection_report',
        priority: data.priority || 'high',
        deviceKey: deviceKey,
        timestamp: new Date().toISOString()
      },
      android: {
        notification: {
          channelId: 'inspection_reports',
          importance: 'high',
          priority: 'high',

          icon: 'ic_notification',
          color: '#FF6B35',
          sound: 'default',
          vibrateTimingsMillis: [0, 250, 250, 250],
          lightSettings: {
            color: '#FF6B35',
            lightOnDurationMillis: 1000,
            lightOffDurationMillis: 1000,
          },
        },
        priority: 'high',
      },
      apns: {
        payload: {
          aps: {
            'content-available': 1,
            'mutable-content': 1,
            sound: 'default',
            badge: 1,
            alert: {
              title: title,
              body: body,
            },
            category: 'INSPECTION_REPORT',
          },
        },
        headers: {
          'apns-priority': '10',
          'apns-push-type': 'alert',
        },
      },
    };

    console.log(`📱 Sending inspection FCM notification to device ${deviceKey} with token:`, token.substring(0, 20) + '...');
    const response = await admin.messaging().send(message);
    
    console.log(`✅ FCM inspection notification sent successfully to device ${deviceKey}`);
    return { success: true, messageId: response };
    
  } catch (error) {
    console.error(`❌ Error sending FCM inspection notification to device ${deviceKey}:`, error);
    
    // Handle specific FCM errors
    if (error.code === 'messaging/invalid-registration-token' || 
        error.code === 'messaging/registration-token-not-registered') {
      console.log(`🔄 FCM token is invalid or expired for device ${deviceKey}, marking as inactive`);
      return { success: false, error: 'Invalid or expired FCM token', code: error.code };
    }
    
    return { success: false, error: error.message, code: error.code };
  }
}

/**
 * Mark device as inactive when FCM token is invalid
 */
async function markDeviceAsInactive(userId, deviceKey) {
  try {
    const deviceRef = admin.database().ref(`/Globalusers/admins/${userId}/devices/${deviceKey}`);
    await deviceRef.update({
      fcmStatus: 'inactive',
      lastError: 'Invalid FCM token',
      lastErrorAt: new Date().toISOString()
    });
    console.log(`🔄 Marked device ${deviceKey} as inactive for user ${userId}`);
  } catch (error) {
    console.error(`❌ Error marking device ${deviceKey} as inactive for user ${userId}:`, error);
  }
}

module.exports = exports;
