const functions = require('firebase-functions/v1');
const admin = require('firebase-admin');
const nodemailer = require('nodemailer');

// Initialize Firebase Admin if not already initialized
if (!admin.apps.length) {
  admin.initializeApp();
}

const db = admin.database();

// Email configuration - Using Gmail SMTP server
const emailConfig = {
  host: 'smtp.gmail.com',
  port: 587,
  secure: false,
  auth: {
    user: 'mandeep.owninfotech@gmail.com',
    pass: 'aeqa purb mxtl hkhh',
  },
};

// Create transporter
const transporter = nodemailer.createTransport(emailConfig);

/**
 * New User Creation Notification Function
 * Sends welcome email with login credentials when a new user is created
 * Triggers when a new user is added to /Globalusers/admins/{userId}
 */
exports.NewUserCreateNotify = functions.database
  .ref('/Globalusers/admins/{userId}')
  .onCreate(async (snapshot, context) => {
    try {
      console.log('👤 NEW USER CREATION NOTIFICATION TRIGGERED! 👤');
      console.log('Path:', context.params);
      console.log('Function version: v1.0 - New user creation notifications');
      
      const userData = snapshot.val();
      const { userId } = context.params;

      console.log('New user created:', { userId, email: userData.email });

      if (!userData) {
        console.log('No user data found');
        return null;
      }

      // Extract user information
      const {
        name,
        email,
        password,
        mobile,
        role,
        department,
        nrcIdNumber,
        employeeNumber,
        countries,
        projects,
        modules,
        engagementDate,
        titanInductionDate,
        clientInduction,
        dateExamined,
        expiryMedicals,
        driverLicenseNumber,
        driverLicenseCode,
        driverLicenseExpiry,
        clientDrivingDate,
        dateCompletedFirstAid,
        levelFirstAid,
        expiryDateFirstAid
      } = userData;
      
      if (!email || !name) {
        console.log('Missing required user data (email or name)');
        return null;
      }

      // Prepare email content
      const emailContent = prepareWelcomeEmailContent({
        name,
        email,
        password,
        mobile,
        role,
        department,
        nrcIdNumber,
        employeeNumber,
        countries,
        projects,
        modules,
        engagementDate,
        titanInductionDate,
        clientInduction,
        dateExamined,
        expiryMedicals,
        driverLicenseNumber,
        driverLicenseCode,
        driverLicenseExpiry,
        clientDrivingDate,
        dateCompletedFirstAid,
        levelFirstAid,
        expiryDateFirstAid
      });

      // Send email to the new user
      console.log(`📧 Sending welcome email to new user: ${email}`);
      const emailResult = await sendWelcomeEmail(email, emailContent);
      console.log('📧 Welcome email result:', emailResult);

      // Log email sent in Firebase
      await logEmailSent(email, 'new_user_creation', emailResult.messageId);

      console.log('✅ New user creation notification sent successfully');
      return null;

    } catch (error) {
      console.error('Error in NewUserCreateNotify:', error);
      return null;
    }
  });

/**
 * Prepare welcome email content for new user
 */
function prepareWelcomeEmailContent(userData) {
  const {
    name,
    email,
    password,
    mobile,
    role,
    department,
    nrcIdNumber,
    employeeNumber,
    countries,
    projects,
    modules,
    engagementDate,
    titanInductionDate,
    clientInduction,
    dateExamined,
    expiryMedicals,
    driverLicenseNumber,
    driverLicenseCode,
    driverLicenseExpiry,
    clientDrivingDate,
    dateCompletedFirstAid,
    levelFirstAid,
    expiryDateFirstAid
  } = userData;

  const emailSubject = `Welcome to Titan Drilling - Your Account Details`;
  
  const emailBody = `
    <html>
      <body style="font-family: Arial, sans-serif; line-height: 1.6; color: #333;">
        <div style="max-width: 600px; margin: 0 auto; padding: 20px;">
          <div style="background-color: #015185; color: white; padding: 20px; text-align: center; border-radius: 8px 8px 0 0;">
            <h1 style="margin: 0; font-size: 24px;">Welcome to Titan Drilling!</h1>
            <p style="margin: 10px 0 0 0; font-size: 16px;">Your account has been created successfully</p>
          </div>
          
          <div style="background-color: #f8f9fa; padding: 20px; border: 1px solid #dee2e6;">
            <div style="background-color: #d4edda; padding: 15px; border-left: 4px solid #28a745; margin-bottom: 20px;">
              <h2 style="color: #28a745; margin-top: 0;">
                🎉 Account Created Successfully
              </h2>
              <p style="margin: 0;">
                <strong>Welcome ${name}!</strong> Your account has been created and you can now access the Titan Drilling system.
              </p>
            </div>
            
            <div style="background-color: #fff3cd; padding: 15px; border-left: 4px solid #ffc107; margin-bottom: 20px;">
              <h3 style="color: #856404; margin-top: 0;">🔐 Login Credentials</h3>
              <table style="width: 100%; border-collapse: collapse;">
                <tr style="border-bottom: 1px solid #dee2e6;">
                  <td style="padding: 8px; font-weight: bold; width: 40%;">Email:</td>
                  <td style="padding: 8px; font-family: monospace; background-color: #f8f9fa;">${email}</td>
                </tr>
                <tr style="border-bottom: 1px solid #dee2e6;">
                  <td style="padding: 8px; font-weight: bold;">Password:</td>
                  <td style="padding: 8px; font-family: monospace; background-color: #f8f9fa;">${password}</td>
                </tr>
                <tr>
                  <td style="padding: 8px; font-weight: bold;">Role:</td>
                  <td style="padding: 8px; background-color: #f8f9fa;">${formatRole(role)}</td>
                </tr>
              </table>
              <p style="margin: 10px 0 0 0; font-size: 14px; color: #856404;">
                <strong>Important:</strong> Please change your password after your first login for security purposes.
              </p>
            </div>
            
            <table style="width: 100%; border-collapse: collapse; margin-bottom: 20px;">
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold; width: 40%;">Full Name:</td>
                <td style="padding: 8px;">${name}</td>
              </tr>
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold;">Mobile:</td>
                <td style="padding: 8px;">${mobile || 'N/A'}</td>
              </tr>
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold;">Employee Number:</td>
                <td style="padding: 8px;">${employeeNumber || 'N/A'}</td>
              </tr>
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold;">Department:</td>
                <td style="padding: 8px;">${department || 'N/A'}</td>
              </tr>
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold;">NRC/ID Number:</td>
                <td style="padding: 8px;">${nrcIdNumber || 'N/A'}</td>
              </tr>
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold;">Countries:</td>
                <td style="padding: 8px;">${formatArray(countries) || 'N/A'}</td>
              </tr>
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold;">Projects:</td>
                <td style="padding: 8px;">${formatArray(projects) || 'N/A'}</td>
              </tr>
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold;">Modules:</td>
                <td style="padding: 8px;">${formatModules(modules) || 'N/A'}</td>
              </tr>
            </table>
            
            ${getEmploymentDetails(engagementDate, titanInductionDate, clientInduction)}
            
            ${getMedicalDetails(dateExamined, expiryMedicals)}
            
            ${getDrivingDetails(driverLicenseNumber, driverLicenseCode, driverLicenseExpiry, clientDrivingDate)}
            
            ${getFirstAidDetails(dateCompletedFirstAid, levelFirstAid, expiryDateFirstAid)}
            
            <div style="text-align: center; margin: 20px 0;">
              <div style="background-color: #e9ecef; padding: 15px; border-radius: 5px;">
                <p style="margin: 0; font-size: 14px; color: #6c757d;">
                  <strong>Next Steps:</strong><br>
                  1. Download the Titan Drilling mobile app<br>
                  2. Use the provided credentials to log in<br>
                  3. Change your password for security<br>
                  4. Complete your profile setup<br>
                  5. Contact your supervisor if you have any questions
                </p>
              </div>
            </div>
            
            <div style="background-color: #e9ecef; padding: 15px; border-radius: 5px; margin-top: 20px;">
              <p style="margin: 0; font-size: 14px; color: #6c757d; text-align: center;">
                This is an automated notification from the Titan Drilling HR Management System.<br>
                Please do not reply to this email. For support, contact your supervisor or HR department.
              </p>
            </div>
          </div>
        </div>
      </body>
    </html>
  `;

  return {
    subject: emailSubject,
    html: emailBody,
    text: convertHtmlToText(emailBody)
  };
}

/**
 * Get employment details section
 */
function getEmploymentDetails(engagementDate, titanInductionDate, clientInduction) {
  if (!engagementDate && !titanInductionDate && !clientInduction) {
    return '';
  }

  return `
    <div style="background-color: #e9ecef; padding: 15px; border-radius: 5px; margin: 20px 0;">
      <h3 style="color: #015185; margin-top: 0;">Employment Details</h3>
      ${engagementDate ? `<p><strong>Engagement Date:</strong> ${formatDate(engagementDate)}</p>` : ''}
      ${titanInductionDate ? `<p><strong>Titan Induction Date:</strong> ${formatDate(titanInductionDate)}</p>` : ''}
      ${clientInduction ? `<p><strong>Client Induction Date:</strong> ${formatDate(clientInduction)}</p>` : ''}
    </div>
  `;
}

/**
 * Get medical details section
 */
function getMedicalDetails(dateExamined, expiryMedicals) {
  if (!dateExamined && !expiryMedicals) {
    return '';
  }

  return `
    <div style="background-color: #e9ecef; padding: 15px; border-radius: 5px; margin: 20px 0;">
      <h3 style="color: #015185; margin-top: 0;">Medical Details</h3>
      ${dateExamined ? `<p><strong>Date Examined:</strong> ${formatDate(dateExamined)}</p>` : ''}
      ${expiryMedicals ? `<p><strong>Medical Expiry:</strong> ${formatDate(expiryMedicals)}</p>` : ''}
    </div>
  `;
}

/**
 * Get driving details section
 */
function getDrivingDetails(driverLicenseNumber, driverLicenseCode, driverLicenseExpiry, clientDrivingDate) {
  if (!driverLicenseNumber && !driverLicenseCode && !driverLicenseExpiry && !clientDrivingDate) {
    return '';
  }

  return `
    <div style="background-color: #e9ecef; padding: 15px; border-radius: 5px; margin: 20px 0;">
      <h3 style="color: #015185; margin-top: 0;">Driving Details</h3>
      ${driverLicenseNumber ? `<p><strong>License Number:</strong> ${driverLicenseNumber}</p>` : ''}
      ${driverLicenseCode ? `<p><strong>License Code:</strong> ${driverLicenseCode}</p>` : ''}
      ${driverLicenseExpiry ? `<p><strong>License Expiry:</strong> ${formatDate(driverLicenseExpiry)}</p>` : ''}
      ${clientDrivingDate ? `<p><strong>Client Driving Date:</strong> ${formatDate(clientDrivingDate)}</p>` : ''}
    </div>
  `;
}

/**
 * Get first aid details section
 */
function getFirstAidDetails(dateCompletedFirstAid, levelFirstAid, expiryDateFirstAid) {
  if (!dateCompletedFirstAid && !levelFirstAid && !expiryDateFirstAid) {
    return '';
  }

  return `
    <div style="background-color: #e9ecef; padding: 15px; border-radius: 5px; margin: 20px 0;">
      <h3 style="color: #015185; margin-top: 0;">First Aid Training</h3>
      ${dateCompletedFirstAid ? `<p><strong>Date Completed:</strong> ${formatDate(dateCompletedFirstAid)}</p>` : ''}
      ${levelFirstAid ? `<p><strong>Level:</strong> ${levelFirstAid}</p>` : ''}
      ${expiryDateFirstAid ? `<p><strong>Expiry Date:</strong> ${formatDate(expiryDateFirstAid)}</p>` : ''}
    </div>
  `;
}

/**
 * Format role for display
 */
function formatRole(role) {
  if (!role) return 'N/A';
  
  const roleMap = {
    'ProAdmin': 'Administrator',
    'CEO': 'Chief Executive Officer',
    'Country Manager': 'Country Manager',
    'Operations Manager': 'Operations Manager',
    'Health & Safety Manager': 'Health & Safety Manager',
    'Project Manager': 'Project Manager',
    'Site Manager': 'Site Manager',
    'Supervisor': 'Supervisor',
    'HrManager': 'HR Manager',
    'FinanceManager': 'Finance Manager',
    'Driller': 'Driller',
    'Trainee Driller': 'Trainee Driller',
    'Junior Driller': 'Junior Driller',
    'Coremarker': 'Coremarker',
    'Assistant': 'Assistant',
    'Workshop Manager': 'Workshop Manager',
    'Workshop Foreman': 'Workshop Foreman',
    'Maintenance Manager': 'Maintenance Manager',
    'Mechanical Engineer': 'Mechanical Engineer',
    'Boilermaker': 'Boilermaker',
    'Welder': 'Welder',
    'Mechanic': 'Mechanic',
    'Machinist': 'Machinist',
    'Electrician': 'Electrician',
    'Housekeeper': 'Housekeeper',
    'Cook': 'Cook',
    'Stores Officer': 'Stores Officer',
    'General Worker': 'General Worker'
  };
  
  return roleMap[role] || role;
}

/**
 * Format array for display
 */
function formatArray(arr) {
  if (!Array.isArray(arr) || arr.length === 0) return '';
  return arr.join(', ');
}

/**
 * Format modules for display
 */
function formatModules(modules) {
  if (!Array.isArray(modules) || modules.length === 0) return '';
  
  const moduleMap = {
    'Equipments': 'Equipment Management',
    'Documents': 'Document Management',
    'teamEmploy': 'Team Employment',
    'inspectionReport': 'Inspection Reports',
    'dvirReport': 'DVIR Reports',
    'checklist': 'Checklists',
    'incidentReport': 'Incident Reports',
    'dailyReport': 'Daily Reports',
    'requestMaintenance': 'Maintenance Requests',
    'jobcard': 'Job Cards',
    'serviceSchedule': 'Service Schedules',
    'fuelLog': 'Fuel Logs',
    'logbook': 'Logbooks',
    'hrForm': 'HR Forms',
    'hrApplications': 'HR Applications',
    'teamKpi': 'Team KPIs'
  };
  
  return modules.map(module => moduleMap[module] || module).join(', ');
}

/**
 * Format date for display
 */
function formatDate(dateString) {
  if (!dateString) return 'N/A';
  try {
    const date = new Date(dateString);
    return date.toLocaleDateString('en-GB', {
      year: 'numeric',
      month: 'long',
      day: 'numeric',
    });
  } catch (error) {
    return 'N/A';
  }
}

/**
 * Convert HTML to plain text
 */
function convertHtmlToText(html) {
  return html
    .replace(/<[^>]*>/g, '')
    .replace(/&nbsp;/g, ' ')
    .replace(/&amp;/g, '&')
    .replace(/&lt;/g, '<')
    .replace(/&gt;/g, '>')
    .replace(/&quot;/g, '"')
    .replace(/\s+/g, ' ')
    .trim();
}

/**
 * Send welcome email to new user
 */
async function sendWelcomeEmail(email, emailContent) {
  try {
    console.log(`📧 Sending welcome email to: ${email}`);
    
    const mailOptions = {
      from: "Titan Drilling HR Management System <support@titandrillingzm.com>",
      to: email,
      subject: emailContent.subject,
      html: emailContent.html,
      text: emailContent.text
    };

    // Retry mechanism for email sending
    let info;
    let retryCount = 0;
    const maxRetries = 3;
    
    while (retryCount < maxRetries) {
      try {
        info = await transporter.sendMail(mailOptions);
        break; // Success, exit retry loop
      } catch (retryError) {
        retryCount++;
        console.log(`⚠️ Email attempt ${retryCount}/${maxRetries} failed for ${email}: ${retryError.message}`);
        
        if (retryCount < maxRetries) {
          // Wait before retry (exponential backoff)
          const delay = Math.pow(2, retryCount) * 1000; // 2s, 4s, 8s
          console.log(`⏳ Retrying in ${delay}ms...`);
          await new Promise(resolve => setTimeout(resolve, delay));
        } else {
          throw retryError; // Final attempt failed
        }
      }
    }
    
    console.log(`✅ Welcome email sent successfully to ${email} (Message ID: ${info.messageId})${retryCount > 0 ? ` after ${retryCount} retries` : ''}`);
    
    return {
      success: true,
      messageId: info.messageId,
      retryCount: retryCount
    };
    
  } catch (error) {
    console.error(`❌ Failed to send welcome email to ${email}:`, error);
    return {
      success: false,
      error: error.message
    };
  }
}

/**
 * Log email sent in Firebase
 */
async function logEmailSent(recipientEmail, emailType, messageId) {
  try {
    const emailLog = {
      recipient: recipientEmail,
      emailType,
      messageId,
      sentAt: admin.database.ServerValue.TIMESTAMP,
      status: 'sent'
    };

    await db.ref('/emailLogs/newUserCreations').push(emailLog);
  } catch (error) {
    console.error('Error logging email:', error);
  }
}

// Function is already exported above as exports.NewUserCreateNotify
