# DVIR Report Email Notification System

This Firebase function automatically sends email notifications when DVIR (Driver Vehicle Inspection Report) reports are created, with PDF attachments and proper user permission filtering.

## 📋 Overview

The `dvirReportEmailNotify` function:
- **Triggers automatically** when a new DVIR report is created in Firebase
- **Sends emails with PDF attachments** to relevant users based on their permissions
- **Filters recipients** by role (ProAdmin > Admin > Regular users)
- **Includes supervisors and proadmins** in the email notifications
- **Logs all email activity** for tracking and debugging
- **Sends FCM push notifications** to all users with valid tokens

## 🔧 Setup Requirements

### 1. Email Configuration

Configure Firebase environment variables:

```bash
# Set up email credentials
firebase functions:config:set email.user="your-email@gmail.com"
firebase functions:config:set email.pass="your-app-password"
```

**Note:** Use an App Password for Gmail, not your regular password.

### 2. Firebase Database Structure

The function expects this data structure:

```
/Globalusers/admins/{userId}/
├── email: "user@example.com"
├── name: "User Name"
├── role: "Admin|Supervisor|ProAdmin|User"
├── countries: ["Zambia", "DRC"]
├── projects: ["Kobold", "Kansanshi"]
└── modules: [
    {
      module: "dvirReport",
      permissions: ["read", "write"]
    }
  ]

/GlobalDvirReports/admins/{adminId}/inspectionReports/{reportId}/
├── step1:
│   ├── country: "Zambia"
│   ├── project: "Kobold"
│   ├── inspector: "John Doe"
│   ├── selectedEquipment: {...}
│   ├── requestNumber: "ZAM-KOB-DVIR-001"
│   └── inspectionTypes: {DVIR: true}
├── step2:
│   ├── items: [...]
│   └── itemStates: {...}
└── step3:
    ├── pdfDownloadUrl: "https://..."
    ├── signatureDownloadUrl: "https://..."
    └── createdAt: "2024-01-01T00:00:00.000Z"
```

### 3. Dependencies

Ensure these packages are in `functions/package.json`:

```json
{
  "dependencies": {
    "firebase-admin": "^12.6.0",
    "firebase-functions": "^6.0.1",
    "nodemailer": "^6.9.14"
  }
}
```

## 🚀 How It Works

### Automatic Trigger

1. **Report Creation**: When a new DVIR report is saved to Firebase
2. **Function Trigger**: Firebase function automatically executes
3. **User Filtering**: Finds users with admin/supervisor roles
4. **Email Generation**: Creates HTML email with report details and PDF link
5. **Email Sending**: Sends to ProAdmins (primary) and Admins (BCC)
6. **Push Notifications**: Sends FCM notifications to all users
7. **Logging**: Records all email activity in Firebase

### Permission-Based Filtering

Users receive emails based on their roles:
- ✅ **ProAdmins**: Primary recipients (TO field)
- ✅ **Admins**: BCC recipients (hidden from others)
- ✅ **Report Creator**: Always receives a copy
- ✅ **All Users**: Receive FCM push notifications

### **Recipient Categories**

The system sends emails to **3 categories** of users:

1. **ProAdmins**: Users with roles containing "proadmin" or "pro admin" (Primary recipients)
2. **Admins**: Users with roles containing "admin", "administrator", "ceo", or "manager" (BCC recipients)
3. **Report Creator**: The user who created the DVIR report (always receives a copy)

**Total Recipients**: Sum of all 3 categories

## 📧 Email Template

The email includes:

### Header
- Company branding
- Report request number

### Report Summary Table
- Request Number
- Country & Project
- Inspector Name
- Equipment Details
- Inspection Type (DVIR/DOT)
- Date

### Inspection Results
- Visual summary with pass/fail counts
- Total items inspected

### PDF Download
- Direct link to full PDF report (7-day expiry)

### Footer
- Company copyright
- Automated notification disclaimer

## 🔔 Push Notifications

### FCM Push Notifications
- **Title**: "🚛 New DVIR Report: {requestNumber}"
- **Body**: "{inspector} created DVIR report for {project} - {equipment}"
- **Data payload** includes report details for app processing
- **High priority** notifications with sound and vibration
- **Multi-device support** - sends to all user devices

## 🔍 Testing

### Manual Testing

You can test the email function manually:

```javascript
// Call this from your app or Firebase console
const testEmailNotification = functions.httpsCallable('dvirReportEmailNotify');

testEmailNotification({
  reportId: 'your-report-id',
  adminId: 'your-admin-id'
}).then((result) => {
  console.log('Test result:', result.data);
}).catch((error) => {
  console.error('Test error:', error);
});
```

## 📊 Monitoring

### Email Logs

All email activity is logged in Firebase:

```
/emailLogs/dvirReports/{logId}/
├── recipient: "user@example.com"
├── recipientName: "User Name"
├── recipientRole: "Supervisor"
├── emailType: "DVIR Report Notification"
├── messageId: "email-message-id"
├── sentAt: timestamp
└── status: "sent"
```

### Push Notification Logs

All push notifications are logged in Firebase:

```
/GlobalPushNotifications/dvir_reports/{reportId}/
├── type: "dvir_report_created"
├── title: "🚛 New DVIR Report: ZAM-KOB-DVIR-001"
├── body: "John Doe created DVIR report for Kobold - Vehicle ABC"
├── data: {...}
├── sentAt: timestamp
├── totalUsers: 25
├── successCount: 23
├── failureCount: 2
└── results: [...]
```

### Function Logs

Monitor function execution in Firebase Console:
1. Go to Firebase Console
2. Functions → Logs
3. Filter by `dvirReportEmailNotify`

## 🛠️ Troubleshooting

### Common Issues

1. **Emails not sending**
   - Check email configuration: `firebase functions:config:get`
   - Verify Gmail App Password is correct
   - Check function logs for errors

2. **Users not receiving emails**
   - Verify user has correct role (ProAdmin/Admin)
   - Check that user's email is valid in Firebase
   - Ensure user is in `/Globalusers/admins/` path

3. **PDF links not working**
   - Check Firebase Storage permissions
   - Verify PDF was uploaded successfully
   - Check that storage path is correct

4. **Push notifications not working**
   - Check FCM token validity
   - Verify user has devices with active tokens
   - Check Firebase Cloud Messaging configuration

### Debug Commands

```bash
# View current configuration
firebase functions:config:get

# View function logs
firebase functions:log --only dvirReportEmailNotify

# Deploy functions
firebase deploy --only functions
```

## 🔒 Security

- **Email credentials** are stored securely in Firebase Functions config
- **PDF links** have 7-day expiry for security
- **User permissions** are checked before sending emails
- **Email logs** track all notifications for audit trail
- **FCM tokens** are validated and invalid tokens are marked inactive

## 📝 Customization

### Email Template

To customize the email template, edit the `prepareEmailContent()` function in `GlobalDVIRnotify.js`:

```javascript
// Modify email subject
const emailSubject = `Custom DVIR Subject: ${requestNumber}`;

// Modify email body HTML
const emailBody = `
  <!-- Your custom HTML template -->
`;
```

### Permission Logic

To modify who receives emails, edit the `checkRoleFast()` and `getAllUsersToNotify()` functions.

### Email Provider

To use a different email provider, modify the `emailConfig` object:

```javascript
const emailConfig = {
  service: 'outlook', // or other provider
  auth: {
    user: functions.config().email?.user,
    pass: functions.config().email?.pass
  }
};
```

## 📞 Support

For issues or questions:
1. Check Firebase Function logs
2. Verify email configuration
3. Test with manual trigger function
4. Review permission structure in Firebase Database

---

**Note**: This system automatically sends emails when DVIR reports are created. Ensure your email quotas and Firebase usage limits are appropriate for your expected volume.
