# Filter Delete Unused User Data Function

This function helps clean up unused admin user data from the Realtime Database by comparing it with Firebase Authentication users.

## What it does

- Compares admin users in `/Globalusers/admins/` with users in Firebase Authentication
- Removes admin user data that doesn't exist in Authentication
- Keeps only admin users that have valid Authentication accounts

## Functions Available

### 1. `filterDeleteUnusedUserData` (Callable Function)
- **Type**: HTTPS Callable
- **Authentication**: Required
- **Usage**: Call from your app using Firebase Functions SDK

```javascript
// From your React Native app
import { getFunctions, httpsCallable } from 'firebase/functions';

const functions = getFunctions();
const filterDeleteUnusedUserData = httpsCallable(functions, 'filterDeleteUnusedUserData');

// Call the function
const result = await filterDeleteUnusedUserData();
console.log(result.data);
```

### 2. `cleanupUnusedAdminUsers` (HTTP Function)
- **Type**: HTTPS Request
- **Authentication**: Not required
- **Usage**: Call via HTTP request

```bash
# Direct HTTP call
curl -X POST https://your-project.cloudfunctions.net/cleanupUnusedAdminUsers
```

## How to Deploy

1. **Deploy the functions:**
   ```bash
   cd functions
   firebase deploy --only functions
   ```

2. **Test locally first:**
   ```bash
   # Run the test script to see what will be deleted
   node test_cleanup_function.js
   ```

## Response Format

Both functions return the same response format:

```json
{
  "success": true,
  "message": "Successfully deleted X unused admin users",
  "deletedCount": 5,
  "remainingCount": 10,
  "deletedUids": ["uid1", "uid2", "uid3"]
}
```

## Safety Features

- **Dry Run Capability**: The test script shows what will be deleted without actually deleting
- **Detailed Logging**: All operations are logged for debugging
- **Error Handling**: Comprehensive error handling with meaningful messages
- **Authentication Check**: Ensures only valid users remain in the database

## Example Usage

1. **Check what will be deleted:**
   ```bash
   node test_cleanup_function.js
   ```

2. **Deploy and run:**
   ```bash
   firebase deploy --only functions
   ```

3. **Call the function:**
   ```bash
   curl -X POST https://titan-drilling-1f8e9.cloudfunctions.net/cleanupUnusedAdminUsers
   ```

## Database Structure

The function works with this structure:
```
/Globalusers/admins/
  ├── uid1/ (valid - exists in Auth)
  ├── uid2/ (valid - exists in Auth)
  └── uid3/ (invalid - doesn't exist in Auth) ← This gets deleted
```

## Security Notes

- The callable function requires authentication
- The HTTP function is open but includes detailed logging
- All operations are logged for audit purposes
- Consider adding additional security checks if needed

## Troubleshooting

- Check Firebase console logs for detailed execution logs
- Ensure service account has proper permissions
- Verify database rules allow the operations
- Test with the provided test script first
