@extends('welcome')

@section('title', 'Account')

@section('content')
<main class="documents-container">
   
    
    <main class="documents-content">
        <div class="content-box">
            <div class="action-bar">
                <div class="action-content">
                    <div class="page-title">
                        <h1>Document Management</h1>
                <p>Manage your documents, attachments, and file storage</p>

                    </div>
                    <div class="action-buttons">
                        <div class="search-section">
                            <div class="search-container">
                                <svg class="search-icon" xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" aria-hidden="true">
                                    <circle cx="11" cy="11" r="8"/>
                                    <path d="m21 21-4.35-4.35"/>
                                </svg>
                                <input type="text" class="search-input" placeholder="Search documents..." aria-label="Search documents">
                            </div>
                        </div>
                        <button class="btn-create" data-bs-toggle="modal" data-bs-target="#add-documents" aria-label="Add new document">
                            <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" aria-hidden="true">
                                <path d="M14.5 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V7.5L14.5 2z"/>
                                <polyline points="14,2 14,8 20,8"/>
                                <line x1="16" y1="13" x2="8" y2="13"/>
                                <line x1="16" y1="17" x2="8" y2="17"/>
                                <polyline points="10,9 9,9 8,9"/>
                            </svg>
                            Add Documents
                        </button>
                        <button class="btn-secondary" aria-label="Sort documents">
                            <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" aria-hidden="true">
                                <path d="M3 6h18"/>
                                <path d="M7 12h10"/>
                                <path d="M10 18h4"/>
                            </svg>
                            Sort
                        </button>
                        <button class="btn-secondary" aria-label="Export documents">
                            <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" aria-hidden="true">
                                <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"/>
                                <polyline points="7,10 12,15 17,10"/>
                                <line x1="12" y1="15" x2="12" y2="3"/>
                            </svg>
                            Export
                        </button>
                    </div>
                </div>
            </div>
            
            <div class="table-section">
                <div class="table-container">
                    <table class="documents-table" role="table" aria-label="Documents table">
                        <thead class="table-header">
                            <tr role="row">
                                <th scope="col" role="columnheader">Action</th>
                                <th scope="col" role="columnheader">Source</th>
                                <th scope="col" role="columnheader">Type</th>
                                <th scope="col" role="columnheader">Description</th>
                                <th scope="col" role="columnheader">Team Member</th>
                                <th scope="col" role="columnheader">Vehicle Details</th>
                                <th scope="col" role="columnheader">Created Date</th>
                                <th scope="col" role="columnheader">Added By</th>
                                <th scope="col" role="columnheader">Attachment</th>
                            </tr>
                        </thead>
                        <tbody class="table-body">
                            <tr class="empty-row" role="row">
                                <td colspan="9" role="cell">
                                    <div class="empty-state">
                                        <svg xmlns="http://www.w3.org/2000/svg" width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round" aria-hidden="true">
                                            <path d="M14.5 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V7.5L14.5 2z"/>
                                            <polyline points="14,2 14,8 20,8"/>
                                            <line x1="16" y1="13" x2="8" y2="13"/>
                                            <line x1="16" y1="17" x2="8" y2="17"/>
                                            <polyline points="10,9 9,9 8,9"/>
                                        </svg>
                                        <p>No documents found</p>
                                        <span>Add your first document to get started</span>
                                    </div>
                                </td>
                            </tr>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </main>
</main>

<script>
    (function() {
        'use strict';

        // Configuration object
        const CONFIG = {
            selectors: {
                sidebar: '.sidebar',
                searchInput: '.search-input',
                modals: '.modal',
                forms: 'form',
                fileInput: '.file-input',
                uploadArea: '.upload-area'
            },
            classes: {
                collapsed: 'collapsed',
                show: 'show',
                active: 'active'
            },
            messages: {
                fileUploaded: 'File uploaded successfully',
                formSaved: 'Document saved successfully',
                formError: 'Error saving document',
                searchResults: 'Search completed'
            },
            timing: {
                notificationDelay: 100,
                notificationDuration: 3000,
                animationDuration: 300
            },
            zIndex: {
                notification: 10001
            }
        };

        // DOM elements cache
        let elements = {};

        // Initialize DOM elements
        function initElements() {
            elements = {
                sidebar: document.querySelector(CONFIG.selectors.sidebar),
                searchInput: document.querySelector(CONFIG.selectors.searchInput),
                modals: document.querySelectorAll(CONFIG.selectors.modals),
                forms: document.querySelectorAll(CONFIG.selectors.forms),
                fileInput: document.querySelector(CONFIG.selectors.fileInput),
                uploadArea: document.querySelector(CONFIG.selectors.uploadArea)
            };
        }

        // Toggle sidebar
    function toggleSidebar() {
            if (elements.sidebar) {
                elements.sidebar.classList.toggle(CONFIG.classes.collapsed);
            }
        }

        // Show notification
        function showNotification(message, type = 'info') {
            const notification = document.createElement('div');
            notification.className = `documents-notification documents-notification-${type}`;
            notification.textContent = message;

            notification.style.cssText = `
                position: fixed;
                top: 80px;
                right: 20px;
                padding: 12px 20px;
                border-radius: 0;
                color: white;
                font-weight: 500;
                font-size: 14px;
                z-index: ${CONFIG.zIndex.notification};
                opacity: 0;
                transform: translateX(100%);
                transition: all ${CONFIG.timing.animationDuration}ms ease;
                box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
                font-family: var(--font-family, 'Inter', sans-serif);
            `;

            // Set background based on type
            const backgrounds = {
                success: 'linear-gradient(135deg, #10b981 0%, #059669 100%)',
                error: 'linear-gradient(135deg, #ef4444 0%, #dc2626 100%)',
                warning: 'linear-gradient(135deg, #f59e0b 0%, #d97706 100%)',
                info: 'linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%)'
            };

            notification.style.background = backgrounds[type] || backgrounds.info;

            document.body.appendChild(notification);

            // Animate in
            setTimeout(() => {
                notification.style.opacity = '1';
                notification.style.transform = 'translateX(0)';
            }, CONFIG.timing.notificationDelay);

            // Animate out and remove
            setTimeout(() => {
                notification.style.opacity = '0';
                notification.style.transform = 'translateX(100%)';
                setTimeout(() => {
                    if (notification.parentNode) {
                        document.body.removeChild(notification);
                    }
                }, CONFIG.timing.animationDuration);
            }, CONFIG.timing.notificationDuration);
        }

        // Initialize search functionality
        function initSearch() {
            if (elements.searchInput) {
                elements.searchInput.addEventListener('input', function() {
                    const searchTerm = this.value.toLowerCase();
                    // Simulate search functionality
                    if (searchTerm.length > 2) {
                        showNotification(CONFIG.messages.searchResults, 'info');
                    }
                });
            }
        }

        // Initialize file upload functionality
        function initFileUpload() {
            if (elements.uploadArea && elements.fileInput) {
                // Drag and drop functionality
                elements.uploadArea.addEventListener('dragover', function(e) {
                    e.preventDefault();
                    this.classList.add('drag-over');
                });

                elements.uploadArea.addEventListener('dragleave', function(e) {
                    e.preventDefault();
                    this.classList.remove('drag-over');
                });

                elements.uploadArea.addEventListener('drop', function(e) {
                    e.preventDefault();
                    this.classList.remove('drag-over');
                    
                    const files = e.dataTransfer.files;
                    if (files.length > 0) {
                        elements.fileInput.files = files;
                        showNotification(CONFIG.messages.fileUploaded, 'success');
                    }
                });

                // Click to upload
                elements.uploadArea.addEventListener('click', function() {
                    elements.fileInput.click();
                });

                // File input change
                elements.fileInput.addEventListener('change', function() {
                    if (this.files.length > 0) {
                        showNotification(CONFIG.messages.fileUploaded, 'success');
                    }
                });
            }
        }

        // Initialize form handlers
        function initFormHandlers() {
            elements.forms.forEach(form => {
                form.addEventListener('submit', function(e) {
                    e.preventDefault();
                    
                    // Simulate form submission
                    showNotification(CONFIG.messages.formSaved, 'success');
                    
                    // Close modal if it's inside one
                    const modal = this.closest('.modal');
                    if (modal) {
                        const bsModal = bootstrap.Modal.getInstance(modal);
                        if (bsModal) {
                            bsModal.hide();
                        }
                    }
                });
            });
        }

        // Initialize modal handlers
        function initModalHandlers() {
            elements.modals.forEach(modal => {
                modal.addEventListener('hidden.bs.modal', function() {
                    // Reset form when modal is closed
                    const form = this.querySelector('form');
                    if (form) {
                        form.reset();
                    }
                });
            });
        }

        // Initialize the application
        function init() {
            initElements();
            initSearch();
            initFileUpload();
            initFormHandlers();
            initModalHandlers();
        }

        // Make functions globally accessible
        window.toggleSidebar = toggleSidebar;
        window.showNotification = showNotification;

        // Initialize when DOM is ready
        document.addEventListener('DOMContentLoaded', init);
    })();
</script>
<!-- Modern Add Documents Modal -->
<div class="modal fade" id="add-documents" tabindex="-1" aria-labelledby="addDocumentsModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h4 class="modal-title" id="addDocumentsModalLabel">Add Documents</h4>
                <button type="button" class="modal-close" data-bs-dismiss="modal" aria-label="Close">
                    <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <line x1="18" y1="6" x2="6" y2="18"/>
                        <line x1="6" y1="6" x2="18" y2="18"/>
                    </svg>
                </button>
            </div>
            <div class="modal-body">
                <form class="document-form" action="">
                    <div class="form-grid">
                        <div class="form-group">
                            <label for="document_of" class="form-label">Document of *</label>
                            <select class="form-select" id="document_of" name="document_of" required>
                                <option value="">Select Document of</option>
                                <option value="vehicle">Vehicle</option>
                                <option value="team">Team</option>
                            </select>
                        </div>
                        <div class="form-group">
                            <label for="attachment_type" class="form-label">Attachment Type *</label>
                            <select class="form-select" id="attachment_type" name="attachment_type" required>
                                <option value="">Select Document Type</option>
                                <option value="driving_licence">Driving Licence</option>
                                <option value="health_insurance">Health Insurance</option>
                                <option value="drug_test">Drug Test Certificate</option>
                                <option value="medical_cert">Medical Certificate</option>
                                <option value="id_card">ID Card</option>
                                <option value="miscellaneous">Miscellaneous</option>
                            </select>
                        </div>
                        <div class="form-group">
                            <label for="description" class="form-label">Description *</label>
                            <input type="text" class="form-control" id="description" name="description" placeholder="Enter document description" required>
                        </div>
                        <div class="form-group">
                            <label for="vehicle" class="form-label">Vehicle *</label>
                            <div class="input-with-select">
                                <input type="text" class="form-control" id="vehicle" name="vehicle" placeholder="Select vehicle from inventory list" readonly>
                                <button type="button" class="btn-select" data-bs-toggle="modal" data-bs-target="#select-checklist">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <path d="M3 3h18v18H3zM9 9h6v6H9z"/>
                                    </svg>
                                </button>
                            </div>
                        </div>
                    </div>
                    
                    <div class="upload-section">
                        <div class="section-header">
                            <h3>File Upload</h3>
                        </div>
                        <div class="upload-area">
                            <svg xmlns="http://www.w3.org/2000/svg" width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round">
                                <path d="M14.5 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V7.5L14.5 2z"/>
                                <polyline points="14,2 14,8 20,8"/>
                                <line x1="16" y1="13" x2="8" y2="13"/>
                                <line x1="16" y1="17" x2="8" y2="17"/>
                                <polyline points="10,9 9,9 8,9"/>
                            </svg>
                            <p>Drag and drop your document here</p>
                            <span>or</span>
                            <input type="file" class="file-input" id="document_file" name="document_file" accept=".pdf,.doc,.docx,.jpg,.jpeg,.png" required>
                            <label for="document_file" class="file-label">
                                <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M12 5v9m-5 0H5a1 1 0 0 0-1 1v4a1 1 0 0 0 1 1h14a1 1 0 0 0 1-1v-4a1 1 0 0 0-1-1h-2M8 9l4-5l4 5m1 8h.01"/>
                                </svg>
                                Choose File
                            </label>
                            <div class="file-info">
                                <small>Supported formats: PDF, DOC, DOCX, JPG, PNG (Max 10MB)</small>
                            </div>
                        </div>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn-cancel" data-bs-dismiss="modal">
                    <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <line x1="18" y1="6" x2="6" y2="18"/>
                        <line x1="6" y1="6" x2="18" y2="18"/>
                    </svg>
                    Cancel
                </button>
                <button type="submit" class="btn-save" form="document-form">
                    <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <path d="M12 5v9m-5 0H5a1 1 0 0 0-1 1v4a1 1 0 0 0 1 1h14a1 1 0 0 0 1-1v-4a1 1 0 0 0-1-1h-2M8 9l4-5l4 5m1 8h.01"/>
                    </svg>
                    Upload Document
                </button>
            </div>
        </div>
    </div>
</div>
<!-- Modern Vehicle Selection Modal -->
<div class="modal fade" id="select-checklist" tabindex="-1" aria-labelledby="vehicleSelectionModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h4 class="modal-title" id="vehicleSelectionModalLabel">Select Vehicle</h4>
                <button type="button" class="modal-close" data-bs-dismiss="modal" aria-label="Close">
                    <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <line x1="18" y1="6" x2="6" y2="18"/>
                        <line x1="6" y1="6" x2="18" y2="18"/>
                    </svg>
                </button>
            </div>
            <div class="modal-body">
                <form class="vehicle-selection-form" action="">
                    <div class="search-section">
                        <div class="search-container">
                            <svg class="search-icon" xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                <circle cx="11" cy="11" r="8"/>
                                <path d="m21 21-4.35-4.35"/>
                            </svg>
                            <input type="text" class="search-input" placeholder="Search vehicles...">
                        </div>
                    </div>
                    <div class="empty-state">
                        <svg xmlns="http://www.w3.org/2000/svg" width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round">
                            <path d="M16 21v-2a4 4 0 0 0-4-4H6a4 4 0 0 0-4 4v2"/>
                            <circle cx="9" cy="7" r="4"/>
                            <line x1="19" y1="8" x2="19" y2="14"/>
                            <line x1="22" y1="11" x2="16" y2="11"/>
                        </svg>
                        <p>No vehicles found</p>
                        <span>No vehicles available for selection</span>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn-cancel" data-bs-dismiss="modal">
                    <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <line x1="18" y1="6" x2="6" y2="18"/>
                        <line x1="6" y1="6" x2="18" y2="18"/>
                    </svg>
                    Cancel
                </button>
                <button type="submit" class="btn-save" form="vehicle-selection-form">
                    <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <path d="M20 6L9 17l-5-5"/>
                    </svg>
                    Select
                </button>
            </div>
        </div>
    </div>
</div>
<script>
    var dropdown = document.getElementsByClassName("dropdown-btn");
    var i;

    for (i = 0; i < dropdown.length; i++) {
        dropdown[i].addEventListener("click", function() {
            this.classList.toggle("active");
            var dropdownContent = this.nextElementSibling;
            if (dropdownContent.style.display === "block") {
                dropdownContent.style.display = "none";
            } else {
                dropdownContent.style.display = "block";
            }
        });
    }
</script>
@endsection

<style>
/* Import Google Fonts for better typography */
@import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');

        /* ===== CSS VARIABLES ===== */
        :root {
            /* Primary Colors */
            --primary-color: #0055a5;
            --primary-hover: rgba(255, 255, 255, 0.2);
            --secondary-color: #64748b;
            --secondary-hover: #475569;
            --success-color: #10b981;
            --success-hover: #059669;
            --warning-color: #f59e0b;
            --warning-hover: #d97706;
            --danger-color: #ef4444;
            --danger-hover: #dc2626;
            --info-color: #3b82f6;
            --info-hover: #1d4ed8;

            /* Text Colors */
            --text-primary: #0f172a;
            --text-secondary: #64748b;
            --text-muted: #94a3b8;
            --text-light: #cbd5e1;
            --text-white: #ffffff;

            /* Background Colors */
            --bg-primary: #f8fafc;
            --bg-secondary: #f1f5f9;
            --bg-white: #ffffff;
            --bg-hover: #f8fafc;
            --bg-gradient: linear-gradient(135deg, #1e293b 0%, #334155 100%);

            /* Border Colors */
            --border-color: #e2e8f0;
            --border-light: #f1f5f9;

            /* Shadows */
            --shadow-sm: 0 1px 3px 0 rgba(0, 0, 0, 0.1), 0 1px 2px 0 rgba(0, 0, 0, 0.06);
            --shadow-md: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
            --shadow-lg: 0 10px 25px rgba(0, 0, 0, 0.1);

            /* Border Radius */
            --radius-sm: 0;
            --radius-md: 0;
            --radius-lg: 0;

            /* Transitions */
            --transition-base: all 0.2s ease;
            --transition-slow: all 0.3s ease;

            /* Typography */
            --font-family: 'Inter', sans-serif;

            /* Z-Index */
            --z-index-modal: 1050;
            --z-index-notification: 10001;
        }

        /* ===== RESET & BASE STYLES ===== */
        * {
    margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: var(--font-family);
            background-color: var(--bg-primary);
            color: var(--text-primary);
            line-height: 1.6;
        }

        /* ===== LAYOUT CONTAINERS ===== */
        .documents-container {
            min-height: 100vh;
            display: flex;
            flex-direction: column;
        }

        .documents-header {
            background: var(--bg-white);
            border-bottom: 1px solid var(--border-color);
            padding: 1.5rem 2rem;
            box-shadow: var(--shadow-sm);
}

.page-title h1 {
            font-size: 1.875rem;
    font-weight: 700;
            color: var(--text-primary);
            margin: 0;
            line-height: 1.2;
}

.page-title p {
            color: var(--text-secondary);
            margin: 0.5rem 0 0 0;
            font-size: 1rem;
        }

.documents-content {
height: 100vh;
}

.content-box {
            background: var(--bg-white);
            border: 1px solid var(--border-color);
            box-shadow: var(--shadow-sm);
            overflow: hidden;
        }

.action-bar {
    display: flex;
    justify-content: space-between;
    align-items: center;
            padding: 1.5rem 2rem;
            background: var(--bg-white);
            border-bottom: 1px solid var(--border-color);
        }

        .page-title h1 {
            font-size: 1.5rem;
            font-weight: 600;
            color: var(--text-primary);
    margin: 0;
}

.page-title h2 {
            font-size: 1.25rem;
            font-weight: 600;
            color: var(--text-primary);
            margin: 0 0 0.5rem 0;
}

        /* ===== ACTION BUTTONS ===== */
.action-buttons {
    display: flex;
    align-items: center;
    gap: 1rem;
}

.search-section {
    position: relative;
}

.search-container {
    position: relative;
    display: flex;
    align-items: center;
}

.search-icon {
    position: absolute;
    left: 1rem;
            color: var(--text-muted);
            z-index: 1;
}

.search-input {
    padding: 0.75rem 1rem 0.75rem 3rem;
            border: 1px solid var(--border-color);
            border-radius: var(--radius-sm);
    font-size: 0.875rem;
            font-family: var(--font-family);
            background: var(--bg-white);
            color: var(--text-primary);
            width: 300px;
            transition: var(--transition-base);
}

.search-input:focus {
    outline: none;
            border-color: var(--primary-color);
            box-shadow: 0 0 0 3px rgba(0, 85, 165, 0.1);
}

.search-input::placeholder {
            color: var(--text-muted);
}

        /* ===== BUTTONS ===== */
        .btn {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.75rem 1.5rem;
    border: none;
            border-radius: var(--radius-sm);
    font-size: 0.875rem;
    font-weight: 600;
            font-family: var(--font-family);
    cursor: pointer;
            transition: var(--transition-base);
            text-decoration: none;
    white-space: nowrap;
}

        .btn:focus {
            outline: none;
            box-shadow: 0 0 0 3px rgba(0, 85, 165, 0.1);
        }

        .btn-create {
            background: var(--primary-color);
            color: var(--text-white);
        }

.btn-create:hover {
            background: var(--primary-hover);
    transform: translateY(-1px);
            box-shadow: var(--shadow-md);
}

.btn-secondary {
            background: var(--secondary-color);
            color: var(--text-white);
}

.btn-secondary:hover {
            background: var(--secondary-hover);
    transform: translateY(-1px);
            box-shadow: var(--shadow-md);
        }

        .btn-success {
            background: var(--success-color);
            color: var(--text-white);
        }

        .btn-success:hover {
            background: var(--success-hover);
            transform: translateY(-1px);
            box-shadow: var(--shadow-md);
        }

        .btn-danger {
            background: var(--danger-color);
            color: var(--text-white);
        }

        .btn-danger:hover {
            background: var(--danger-hover);
            transform: translateY(-1px);
            box-shadow: var(--shadow-md);
        }

        .btn-cancel {
            background: var(--secondary-color);
            color: var(--text-white);
        }

        .btn-cancel:hover {
            background: var(--secondary-hover);
        }

        .btn-save {
            background: var(--success-color);
            color: var(--text-white);
        }

        .btn-save:hover {
            background: var(--success-hover);
        }

        /* ===== TABLES ===== */
.table-section {
            padding: 0;
}

.table-container {
            overflow-x: auto;
}

.documents-table {
    width: 100%;
    border-collapse: collapse;
            margin-bottom: 0;
        }

        .documents-table th,
        .documents-table td {
            padding: 1rem;
            text-align: left;
            border-bottom: 1px solid var(--border-color);
            vertical-align: middle;
        }

        .documents-table th {
            background: var(--bg-secondary);
    font-weight: 600;
            color: var(--text-primary);
    font-size: 0.875rem;
    text-transform: uppercase;
    letter-spacing: 0.05em;
        }

        .documents-table td {
            color: var(--text-secondary);
            font-size: 0.875rem;
        }

        .documents-table tbody tr:hover {
            background: var(--bg-hover);
}

/* ===== EMPTY STATE ===== */
.empty-state {
    display: flex;
    flex-direction: column;
    align-items: center;
    justify-content: center;
            padding: 3rem 2rem;
            text-align: center;
}

.empty-state svg {
            color: var(--text-muted);
            margin-bottom: 1rem;
}

.empty-state p {
            font-size: 1.125rem;
    font-weight: 600;
            color: var(--text-primary);
            margin: 0 0 0.5rem 0;
}

.empty-state span {
            color: var(--text-muted);
    font-size: 0.875rem;
        }

        /* ===== MODALS ===== */
        .modal-content {
            border: none;
            border-radius: var(--radius-md);
            box-shadow: var(--shadow-lg);
        }

        .modal-header {
            background: var(--bg-white);
            border-bottom: 1px solid var(--border-color);
            padding: 1.5rem 2rem;
        }

        .modal-title {
            font-size: 1.25rem;
            font-weight: 600;
            color: var(--text-primary);
    margin: 0;
        }

        .modal-close {
            background: none;
            border: none;
            color: var(--text-muted);
            cursor: pointer;
            padding: 0.5rem;
            border-radius: var(--radius-sm);
            transition: var(--transition-base);
        }

        .modal-close:hover {
            background: var(--bg-hover);
            color: var(--text-primary);
        }

        .modal-body {
            padding: 2rem;
            background: var(--bg-white);
        }

        .modal-footer {
            background: var(--bg-secondary);
            border-top: 1px solid var(--border-color);
            padding: 1.5rem 2rem;
            display: flex;
            gap: 0.75rem;
            justify-content: flex-end;
        }

        /* ===== FORMS ===== */
        .form-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 1.5rem;
            margin-bottom: 2rem;
        }

        .form-group {
            margin-bottom: 1.5rem;
        }

        .form-label {
            display: block;
            margin-bottom: 0.5rem;
            font-weight: 600;
            color: var(--text-primary);
            font-size: 0.875rem;
        }

        .form-control {
            width: 100%;
            padding: 0.75rem 1rem;
            border: 1px solid var(--border-color);
            border-radius: var(--radius-sm);
            font-size: 0.875rem;
            font-family: var(--font-family);
            transition: var(--transition-base);
            background: var(--bg-white);
            color: var(--text-primary);
        }

        .form-control:focus {
            outline: none;
            border-color: var(--primary-color);
            box-shadow: 0 0 0 3px rgba(0, 85, 165, 0.1);
        }

        .form-control::placeholder {
            color: var(--text-muted);
        }

        .form-select {
            background-image: url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' fill='none' viewBox='0 0 20 20'%3e%3cpath stroke='%236b7280' stroke-linecap='round' stroke-linejoin='round' stroke-width='1.5' d='m6 8 4 4 4-4'/%3e%3c/svg%3e");
            background-position: right 0.5rem center;
            background-repeat: no-repeat;
            background-size: 1.5em 1.5em;
            padding-right: 2.5rem;
        }

        .input-with-select {
            position: relative;
    display: flex;
        }

        .input-with-select .form-control {
            padding-right: 3rem;
        }

        .btn-select {
            position: absolute;
            right: 0.5rem;
            top: 50%;
            transform: translateY(-50%);
    background: none;
    border: none;
            color: var(--text-muted);
    cursor: pointer;
            padding: 0.5rem;
            border-radius: var(--radius-sm);
            transition: var(--transition-base);
        }

        .btn-select:hover {
            background: var(--bg-hover);
            color: var(--text-primary);
        }

        /* ===== FILE UPLOAD ===== */
        .upload-section {
            margin-top: 2rem;
        }

        .section-header h3 {
            font-size: 1.125rem;
            font-weight: 600;
            color: var(--text-primary);
            margin: 0 0 1rem 0;
        }

        .upload-area {
            border: 2px dashed var(--border-color);
            border-radius: var(--radius-md);
            padding: 3rem 2rem;
            text-align: center;
            background: var(--bg-secondary);
            cursor: pointer;
            transition: var(--transition-base);
        }

        .upload-area:hover,
        .upload-area.drag-over {
            border-color: var(--primary-color);
            background: rgba(0, 85, 165, 0.05);
        }

        .upload-area svg {
            color: var(--text-muted);
            margin-bottom: 1rem;
        }

        .upload-area p {
            font-size: 1rem;
            font-weight: 600;
            color: var(--text-primary);
            margin: 0 0 0.5rem 0;
        }

        .upload-area span {
            color: var(--text-muted);
            font-size: 0.875rem;
            margin: 0 0.5rem;
        }

        .file-input {
            display: none;
        }

        .file-label {
            display: inline-flex;
    align-items: center;
            gap: 0.5rem;
            padding: 0.75rem 1.5rem;
            background: var(--primary-color);
            color: var(--text-white);
            border-radius: var(--radius-sm);
            cursor: pointer;
            font-size: 0.875rem;
            font-weight: 600;
            transition: var(--transition-base);
            margin-top: 1rem;
        }

        .file-label:hover {
            background: var(--primary-hover);
            transform: translateY(-1px);
        }

        .file-info {
            margin-top: 1rem;
        }

        .file-info small {
            color: var(--text-muted);
            font-size: 0.75rem;
        }

        /* ===== NOTIFICATIONS ===== */
        .documents-notification {
            position: fixed;
            top: 80px;
            right: 20px;
            padding: 12px 20px;
            border-radius: var(--radius-sm);
            color: var(--text-white);
            font-weight: 500;
            font-size: 14px;
            z-index: var(--z-index-notification);
            opacity: 0;
            transform: translateX(100%);
            transition: var(--transition-slow);
            box-shadow: var(--shadow-lg);
            font-family: var(--font-family);
        }

        .documents-notification-success {
            background: linear-gradient(135deg, var(--success-color) 0%, var(--success-hover) 100%);
        }

        .documents-notification-error {
            background: linear-gradient(135deg, var(--danger-color) 0%, var(--danger-hover) 100%);
        }

        .documents-notification-warning {
            background: linear-gradient(135deg, var(--warning-color) 0%, var(--warning-hover) 100%);
        }

        .documents-notification-info {
            background: linear-gradient(135deg, var(--info-color) 0%, var(--info-hover) 100%);
        }

        /* ===== RESPONSIVE DESIGN ===== */
        @media (max-width: 768px) {
            .documents-header {
                padding: 1rem;
            }

            .documents-content {
                padding: 1rem;
            }

            .action-bar {
                flex-direction: column;
                gap: 1rem;
                align-items: stretch;
                padding: 1rem;
            }

            .action-buttons {
                flex-direction: column;
                gap: 0.75rem;
            }

            .search-input {
                width: 100%;
            }

            .form-grid {
                grid-template-columns: 1fr;
            }

            .modal-body {
                padding: 1rem;
            }

            .modal-footer {
                flex-direction: column;
                gap: 0.5rem;
            }

            .documents-table {
                font-size: 0.75rem;
            }

            .documents-table th,
            .documents-table td {
                padding: 0.5rem;
            }
        }

        @media (max-width: 576px) {
            .page-title h1 {
                font-size: 1.5rem;
            }

            .btn {
                width: 100%;
                justify-content: center;
            }

            .modal-header,
            .modal-body,
            .modal-footer {
                padding: 1rem;
            }

            .upload-area {
                padding: 2rem 1rem;
            }
        }

        /* ===== ACCESSIBILITY ===== */
        @media (prefers-reduced-motion: reduce) {
            * {
                animation-duration: 0.01ms !important;
                animation-iteration-count: 1 !important;
                transition-duration: 0.01ms !important;
            }
        }

        .btn:focus-visible,
        .form-control:focus-visible,
        .search-input:focus-visible {
            outline: 2px solid var(--primary-color);
            outline-offset: 2px;
        }

        /* ===== PRINT STYLES ===== */
        @media print {
            .documents-header,
            .action-bar,
            .modal,
            .btn {
                display: none !important;
            }

            .documents-content {
                padding: 0;
            }

            .content-box {
                box-shadow: none;
                border: 1px solid #000;
            }
        }
        .input-with-select {
    position: relative;
    display: flex;
}

.input-with-select .form-control {
    flex: 1;
    border-right: none;
}

.btn-select {
    background: #f8fafc;
    border: 2px solid #e2e8f0;
    border-left: none;
    padding: 0.75rem;
    cursor: pointer;
    transition: all 0.2s ease;
    display: flex;
    align-items: center;
    justify-content: center;
}

.btn-select:hover {
    background: #f1f5f9;
    border-color: #cbd5e1;
}

/* ===== UPLOAD SECTION ===== */
.upload-section {
    margin-top: 2rem;
    padding-top: 2rem;
    border-top: 2px solid #e2e8f0;
}

.section-header {
    margin-bottom: 1.5rem;
}

.section-header h3 {
    font-size: 1.25rem;
    font-weight: 700;
    color: #1e293b;
    margin: 0;
}

.upload-area {
    border: 2px dashed #cbd5e1;
    border-radius: 0;
    padding: 3rem 2rem;
    text-align: center;
    background: #f8fafc;
    transition: all 0.2s ease;
    cursor: pointer;
}

.upload-area:hover {
    border-color: #3b82f6;
    background: #f0f9ff;
}

.upload-area svg {
    color: #cbd5e1;
    margin-bottom: 1rem;
}

.upload-area p {
    color: #64748b;
    font-size: 1rem;
    font-weight: 600;
    margin: 0 0 0.5rem 0;
}

.upload-area span {
    color: #94a3b8;
    font-size: 0.875rem;
    margin: 0 0 1rem 0;
    display: block;
}

.file-input {
    display: none;
}

.file-label {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.75rem 1.5rem;
    background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%);
    color: #ffffff;
    border: none;
    border-radius: 0;
    font-size: 0.875rem;
    font-weight: 600;
    font-family: 'Inter', sans-serif;
    cursor: pointer;
    transition: all 0.2s ease;
    box-shadow: 0 2px 4px rgba(59, 130, 246, 0.2);
}

.file-label:hover {
    background: linear-gradient(135deg, #1d4ed8 0%, #1e40af 100%);
    transform: translateY(-1px);
    box-shadow: 0 4px 8px rgba(59, 130, 246, 0.3);
}

.file-info {
    margin-top: 1rem;
}

.file-info small {
    color: #94a3b8;
    font-size: 0.75rem;
    font-weight: 400;
}

/* ===== MODAL FOOTER BUTTONS ===== */
.btn-save {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.75rem 1.5rem;
    background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%);
    color: #ffffff;
    border: none;
    border-radius: 0;
    font-size: 0.875rem;
    font-weight: 600;
    font-family: 'Inter', sans-serif;
    cursor: pointer;
    transition: all 0.2s ease;
    box-shadow: 0 2px 4px rgba(59, 130, 246, 0.2);
}

.btn-save:hover {
    background: linear-gradient(135deg, #1d4ed8 0%, #1e40af 100%);
    transform: translateY(-1px);
    box-shadow: 0 4px 8px rgba(59, 130, 246, 0.3);
}

.btn-cancel {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.75rem 1.5rem;
    background: #ffffff;
    color: #64748b;
    border: 2px solid #e2e8f0;
    border-radius: 0;
    font-size: 0.875rem;
    font-weight: 600;
    font-family: 'Inter', sans-serif;
    cursor: pointer;
    transition: all 0.2s ease;
}

.btn-cancel:hover {
    background: #f8fafc;
    border-color: #cbd5e1;
    color: #475569;
}

/* ===== RESPONSIVE DESIGN ===== */
@media (max-width: 1200px) {
    .documents-container {
        padding: 16px;
    }
    
    .content-box {
        padding: 24px 24px 24px 32px;
        margin-left: 0.25rem;
    }
    
    .form-grid {
        grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
        gap: 1.25rem;
    }
    
    .modal-dialog.modal-lg {
        max-width: 90%;
    }
}

@media (max-width: 992px) {
    .documents-container {
        padding: 12px;
    }
    
    .content-box {
        padding: 20px 20px 20px 28px;
        margin-left: 0.125rem;
    }
    
    .page-title h1 {
        font-size: 28px;
    }
    
    .page-title h2 {
        font-size: 22px;
    }
    
    .action-content {
        flex-direction: column;
        align-items: flex-start;
        gap: 1.25rem;
    }
    
    .action-buttons {
        width: 100%;
        justify-content: flex-start;
        flex-wrap: wrap;
        gap: 0.75rem;
    }
    
    .search-input {
        min-width: 250px;
        width: 100%;
    }
    
    .form-grid {
        grid-template-columns: 1fr;
        gap: 1rem;
    }
    
    .table-container {
        overflow-x: auto;
    }
    
    .documents-table {
        min-width: 800px;
    }
    
    .modal-dialog.modal-lg {
        max-width: 95%;
    }
}

@media (max-width: 768px) {
    .documents-container {
        padding: 8px;
    }
    
    .content-box {
        padding: 16px 16px 16px 20px;
        margin-left: 0;
    }
    
    .page-title h1 {
        font-size: 24px;
    }
    
    .page-title h2 {
        font-size: 20px;
    }
    
    .action-content {
        flex-direction: column;
        align-items: stretch;
        gap: 1rem;
    }
    
    .action-buttons {
        flex-direction: column;
        gap: 0.75rem;
    }
    
    .search-section {
        width: 100%;
    }
    
    .search-input {
        min-width: 100%;
        width: 100%;
    }
    
    .btn-create, .btn-secondary {
        width: 100%;
        justify-content: center;
        padding: 0.875rem 1.25rem;
        font-size: 0.875rem;
        font-weight: 600;
        min-height: 48px;
    }
    
    .form-grid {
        grid-template-columns: 1fr;
        gap: 0.875rem;
    }
    
    .form-group {
        margin-bottom: 0.75rem;
    }
    
    .form-label {
        font-size: 0.8rem;
        margin-bottom: 0.375rem;
    }
    
    .form-control, .form-select {
        padding: 0.625rem 0.875rem;
        font-size: 0.8rem;
    }
    
    .input-with-select {
        flex-direction: column;
    }
    
    .input-with-select .form-control {
        border-right: 2px solid #e2e8f0;
        border-bottom: none;
        margin-bottom: 0.5rem;
    }
    
    .btn-select {
        border-left: 2px solid #e2e8f0;
        border-top: none;
        padding: 0.625rem;
        justify-content: center;
    }
    
    .table-container {
        overflow-x: auto;
        -webkit-overflow-scrolling: touch;
        border: 1px solid #e2e8f0;
    }
    
    .documents-table {
        min-width: 700px;
    }
    
    .table-header th,
    .table-body td {
        padding: 0.75rem 0.875rem;
        font-size: 0.8rem;
    }
    
    .table-header th {
        font-size: 0.75rem;
        font-weight: 600;
    }
    
    .empty-state {
        padding: 2rem 1rem;
        min-height: 150px;
    }
    
    .empty-state p {
        font-size: 0.875rem;
    }
    
    .empty-state span {
        font-size: 0.8rem;
    }
    
    .modal-dialog {
        margin: 0.5rem;
        width: 95%;
    }
    
    .modal-header, .modal-body, .modal-footer {
        padding: 1rem 1.25rem;
    }
    
    .modal-title {
        font-size: 1.25rem;
    }
    
    .upload-area {
        padding: 1.5rem 1rem;
    }
    
    .upload-area p {
        font-size: 0.875rem;
    }
    
    .upload-area span {
        font-size: 0.8rem;
    }
    
    .file-label {
        padding: 0.625rem 1.25rem;
        font-size: 0.8rem;
    }
    
    .btn-save, .btn-cancel {
        padding: 0.625rem 1.25rem;
        font-size: 0.8rem;
    }
}

@media (max-width: 576px) {
    .documents-container {
        padding: 4px;
    }
    
    .content-box {
        padding: 12px 12px 12px 16px;
    }
    
    .page-title h1 {
        font-size: 20px;
    }
    
    .page-title h2 {
        font-size: 18px;
    }
    
    .action-buttons {
        gap: 0.5rem;
    }
    
    .btn-create, .btn-secondary {
        padding: 0.75rem 1rem;
        font-size: 0.8rem;
        min-height: 44px;
    }
    
    .search-input {
        padding: 0.625rem 0.875rem 0.625rem 2.5rem;
        font-size: 0.8rem;
    }
    
    .search-icon {
        left: 0.875rem;
        width: 16px;
        height: 16px;
    }
    
    .form-grid {
        gap: 0.75rem;
    }
    
    .form-group {
        margin-bottom: 0.625rem;
    }
    
    .form-label {
        font-size: 0.75rem;
    }
    
    .form-control, .form-select {
        padding: 0.5rem 0.75rem;
        font-size: 0.75rem;
    }
    
    .documents-table {
        min-width: 600px;
    }
    
    .table-header th,
    .table-body td {
        padding: 0.625rem 0.75rem;
        font-size: 0.75rem;
    }
    
    .table-header th {
        font-size: 0.7rem;
    }
    
    .empty-state {
        padding: 1.5rem 0.75rem;
        min-height: 120px;
    }
    
    .empty-state svg {
        width: 40px;
        height: 40px;
    }
    
    .empty-state p {
        font-size: 0.8rem;
    }
    
    .empty-state span {
        font-size: 0.75rem;
    }
    
    .modal-dialog {
        margin: 0.25rem;
        width: 98%;
    }
    
    .modal-header, .modal-body, .modal-footer {
        padding: 0.75rem 1rem;
    }
    
    .modal-title {
        font-size: 1.125rem;
    }
    
    .upload-area {
        padding: 1.25rem 0.75rem;
    }
    
    .upload-area svg {
        width: 40px;
        height: 40px;
    }
    
    .upload-area p {
        font-size: 0.8rem;
    }
    
    .upload-area span {
        font-size: 0.75rem;
    }
    
    .file-label {
        padding: 0.5rem 1rem;
        font-size: 0.75rem;
    }
    
    .file-info small {
        font-size: 0.7rem;
    }
    
    .btn-save, .btn-cancel {
        padding: 0.5rem 1rem;
        font-size: 0.75rem;
    }
}

@media (max-width: 480px) {
    .documents-container {
        padding: 2px;
    }
    
    .content-box {
        padding: 8px 8px 8px 12px;
    }
    
    .page-title h1 {
        font-size: 18px;
    }
    
    .page-title h2 {
        font-size: 16px;
    }
    
    .btn-create, .btn-secondary {
        padding: 0.625rem 0.875rem;
        font-size: 0.75rem;
        min-height: 40px;
    }
    
    .search-input {
        padding: 0.5rem 0.75rem 0.5rem 2.25rem;
        font-size: 0.75rem;
    }
    
    .search-icon {
        left: 0.75rem;
        width: 14px;
        height: 14px;
    }
    
    .form-grid {
        gap: 0.625rem;
    }
    
    .form-group {
        margin-bottom: 0.5rem;
    }
    
    .form-label {
        font-size: 0.7rem;
    }
    
    .form-control, .form-select {
        padding: 0.375rem 0.625rem;
        font-size: 0.7rem;
    }
    
    .documents-table {
        min-width: 500px;
    }
    
    .table-header th,
    .table-body td {
        padding: 0.5rem 0.625rem;
        font-size: 0.7rem;
    }
    
    .table-header th {
        font-size: 0.65rem;
    }
    
    .empty-state {
        padding: 1rem 0.5rem;
        min-height: 100px;
    }
    
    .empty-state svg {
        width: 32px;
        height: 32px;
    }
    
    .empty-state p {
        font-size: 0.75rem;
    }
    
    .empty-state span {
        font-size: 0.7rem;
    }
    
    .modal-dialog {
        margin: 0.125rem;
        width: 100%;
    }
    
    .modal-header, .modal-body, .modal-footer {
        padding: 0.625rem 0.75rem;
    }
    
    .modal-title {
        font-size: 1rem;
    }
    
    .upload-area {
        padding: 1rem 0.5rem;
    }
    
    .upload-area svg {
        width: 32px;
        height: 32px;
    }
    
    .upload-area p {
        font-size: 0.75rem;
    }
    
    .upload-area span {
        font-size: 0.7rem;
    }
    
    .file-label {
        padding: 0.375rem 0.875rem;
        font-size: 0.7rem;
    }
    
    .file-info small {
        font-size: 0.65rem;
    }
    
    .btn-save, .btn-cancel {
        padding: 0.375rem 0.875rem;
        font-size: 0.7rem;
    }
}

/* ===== TABLET LANDSCAPE OPTIMIZATION ===== */
@media (max-width: 1024px) and (orientation: landscape) {
    .documents-container {
        padding: 12px;
    }
    
    .content-box {
        padding: 16px 16px 16px 24px;
    }
    
    .action-content {
        flex-direction: row;
        align-items: center;
    }
    
    .action-buttons {
        flex-direction: row;
        flex-wrap: wrap;
    }
    
    .search-input {
        min-width: 200px;
    }
}

/* ===== MOBILE LANDSCAPE OPTIMIZATION ===== */
@media (max-width: 896px) and (orientation: landscape) {
    .documents-container {
        padding: 8px;
    }
    
    .content-box {
        padding: 12px 12px 12px 16px;
    }
    
    .page-title h1 {
        font-size: 20px;
    }
    
    .action-content {
        flex-direction: row;
        align-items: center;
    }
    
    .action-buttons {
        flex-direction: row;
        overflow-x: auto;
    }
    
    .btn-create, .btn-secondary {
        white-space: nowrap;
        min-width: auto;
    }
}

/* ===== TABLE RESPONSIVE IMPROVEMENTS ===== */
@media (max-width: 768px) {
    .documents-table {
        font-size: 0.8rem;
    }
    
    .table-header th:first-child,
    .table-body td:first-child {
        position: sticky;
        left: 0;
        background: #f8fafc;
        z-index: 10;
        border-right: 1px solid #e2e8f0;
    }
    
    .table-header th:first-child {
        background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%);
    }
}

/* ===== MODAL RESPONSIVE IMPROVEMENTS ===== */
@media (max-width: 768px) {
    .modal {
        align-items: flex-start;
        padding-top: 20px;
    }
    
    .modal-dialog {
        margin: 0;
    }
    
    .modal-content {
        border-radius: 0;
    }
    
    .modal-footer {
        flex-direction: column;
        gap: 0.5rem;
    }
    
    .btn-save, .btn-cancel {
        width: 100%;
        justify-content: center;
    }
}

/* ===== ACCESSIBILITY IMPROVEMENTS ===== */
@media (prefers-reduced-motion: reduce) {
    * {
        transition: none !important;
        animation: none !important;
    }
}

/* ===== HIGH DPI DISPLAYS ===== */
@media (-webkit-min-device-pixel-ratio: 2), (min-resolution: 192dpi) {
    .search-icon,
    .btn-create svg,
    .btn-secondary svg,
    .modal-close svg,
    .btn-save svg,
    .btn-cancel svg {
        image-rendering: -webkit-optimize-contrast;
        image-rendering: crisp-edges;
    }
}

/* ===== ACCESSIBILITY ===== */
@media (prefers-reduced-motion: reduce) {
    * {
        animation-duration: 0.01ms !important;
        animation-iteration-count: 1 !important;
        transition-duration: 0.01ms !important;
    }
}

/* ===== FOCUS STYLES ===== */
button:focus-visible,
input:focus-visible,
select:focus-visible {
    outline: 2px solid #3b82f6;
    outline-offset: 2px;
}

/* ===== PRINT STYLES ===== */
@media print {
    .action-bar,
    .modal {
        display: none !important;
    }
    
    .documents-container {
        background: white;
    }
    
    .documents-header {
        background: white;
        color: black;
    }
}

/* ===== ENHANCED RESPONSIVE DESIGN ===== */
@media (max-width: 1400px) {
    .documents-container {
        padding: 18px;
    }
    
    .documents-header {
        margin-bottom: 20px;
    }
    
    .header-content {
        padding: 28px;
    }
}

@media (max-width: 1200px) {
    .documents-container {
        padding: 16px;
    }
    
    .header-content {
        padding: 24px;
    }
    
    .documents-content {
        padding: 20px;
    }
    
    .documents-grid {
        grid-template-columns: 1fr;
        gap: 20px;
    }
}

@media (max-width: 992px) {
    .documents-container {
        padding: 14px;
    }
    
    .documents-header {
        margin-bottom: 16px;
    }
    
    .header-content {
        padding: 20px;
    }
    
    .page-title h1 {
        font-size: 28px;
    }
    
    .page-title p {
        font-size: 15px;
    }
    
    .documents-content {
        padding: 16px;
    }
    
    .documents-grid {
        gap: 16px;
    }
    
    .document-section {
        padding: 16px;
    }
    
    .document-section h3 {
        font-size: 20px;
    }
}

@media (max-width: 768px) {
    .documents-container {
        padding: 12px;
    }
    
    .header-content {
        padding: 16px;
    }
    
    .page-title h1 {
        font-size: 24px;
    }
    
    .page-title p {
        font-size: 14px;
    }
    
    .documents-content {
        padding: 12px;
    }
    
    .documents-grid {
        gap: 12px;
    }
    
    .document-section {
        padding: 12px;
    }
    
    .document-section h3 {
        font-size: 18px;
    }
    
    .btn-primary,
    .btn-secondary {
        padding: 10px 16px;
        font-size: 14px;
    }
}

@media (max-width: 576px) {
    .documents-container {
        padding: 8px;
    }
    
    .header-content {
        padding: 12px;
    }
    
    .page-title h1 {
        font-size: 20px;
    }
    
    .page-title p {
        font-size: 13px;
    }
    
    .documents-content {
        padding: 10px;
    }
    
    .documents-grid {
        gap: 10px;
    }
    
    .document-section {
        padding: 10px;
    }
    
    .document-section h3 {
        font-size: 16px;
    }
    
    .btn-primary,
    .btn-secondary {
        padding: 8px 12px;
        font-size: 13px;
    }
}

@media (max-width: 480px) {
    .documents-container {
        padding: 6px;
    }
    
    .header-content {
        padding: 10px;
    }
    
    .page-title h1 {
        font-size: 18px;
    }
    
    .page-title p {
        font-size: 12px;
    }
    
    .documents-content {
        padding: 8px;
    }
    
    .documents-grid {
        gap: 8px;
    }
    
    .document-section {
        padding: 8px;
    }
    
    .document-section h3 {
        font-size: 14px;
    }
    
    .btn-primary,
    .btn-secondary {
        padding: 6px 10px;
        font-size: 12px;
    }
}
</style>



<script>
// Modern Documents Management Functionality
document.addEventListener('DOMContentLoaded', function() {
    // Search functionality
    const searchInput = document.querySelector('.search-input');
    if (searchInput) {
        searchInput.addEventListener('input', function(e) {
            const searchTerm = e.target.value.toLowerCase();
            // Add search logic here
            console.log('Searching for:', searchTerm);
        });
    }
    
    // Form validation
    const documentForm = document.querySelector('.document-form');
    if (documentForm) {
        documentForm.addEventListener('submit', function(e) {
            e.preventDefault();
            // Add form submission logic here
            console.log('Document form submitted');
        });
    }
    
    // File upload handling
    const fileInput = document.querySelector('.file-input');
    if (fileInput) {
        fileInput.addEventListener('change', function(e) {
            const file = e.target.files[0];
            if (file) {
                console.log('File selected:', file.name);
                // Update upload area with file info
                const uploadArea = document.querySelector('.upload-area');
                if (uploadArea) {
                    uploadArea.style.borderColor = '#10b981';
                    uploadArea.style.backgroundColor = '#f0fdf4';
                }
            }
        });
    }
    
    // Drag and drop functionality
    const uploadArea = document.querySelector('.upload-area');
    if (uploadArea) {
        uploadArea.addEventListener('dragover', function(e) {
            e.preventDefault();
            uploadArea.style.borderColor = '#3b82f6';
            uploadArea.style.backgroundColor = '#f0f9ff';
        });
        
        uploadArea.addEventListener('dragleave', function(e) {
            e.preventDefault();
            uploadArea.style.borderColor = '#cbd5e1';
            uploadArea.style.backgroundColor = '#f8fafc';
        });
        
        uploadArea.addEventListener('drop', function(e) {
            e.preventDefault();
            const files = e.dataTransfer.files;
            if (files.length > 0) {
                const fileInput = document.querySelector('.file-input');
                if (fileInput) {
                    fileInput.files = files;
                    fileInput.dispatchEvent(new Event('change'));
                }
            }
            uploadArea.style.borderColor = '#cbd5e1';
            uploadArea.style.backgroundColor = '#f8fafc';
        });
    }
    
    // Modal handling
    const modals = document.querySelectorAll('.modal');
    modals.forEach(modal => {
        modal.addEventListener('hidden.bs.modal', function() {
            // Reset form when modal is closed
            const form = this.querySelector('form');
            if (form) {
                form.reset();
            }
            // Reset upload area
            const uploadArea = this.querySelector('.upload-area');
            if (uploadArea) {
                uploadArea.style.borderColor = '#cbd5e1';
                uploadArea.style.backgroundColor = '#f8fafc';
            }
        });
    });
});
</script>