<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class VehicleController extends Controller
{
    public function inventory()
    {
        return view('vehicles.Vehicles_Inventory');
    }
    
    /**
     * Get all equipment from MongoDB API with pagination and filters
     */
    public function getAllEquipment(Request $request)
    {
        try {
            $page = $request->get('page', 1);
            $limit = $request->get('limit', 20);
            $status = $request->get('status', '');
            
            $mongoUrl = env('MONGODB_EQUIPMENT_GET_ALL_URL', 'http://168.231.113.219:5001/EquipmentsDataJSON/GlobalEquipmentJsonData');
            
            Log::info('Fetching equipment from MongoDB', [
                'url' => $mongoUrl,
                'page' => $page,
                'limit' => $limit,
                'status' => $status
            ]);
            
            $response = Http::timeout(10)->get($mongoUrl);
            
            if ($response->successful()) {
                $data = $response->json();
                $allEquipment = $data['data'] ?? [];
                
                // Filter by status if provided
                if ($status && $status !== 'all') {
                    $allEquipment = array_filter($allEquipment, function($item) use ($status) {
                        return isset($item['status']) && strtolower($item['status']) === strtolower($status);
                    });
                    $allEquipment = array_values($allEquipment); // Re-index array
                }
                
                // Calculate pagination
                $totalRecords = count($allEquipment);
                $totalPages = ceil($totalRecords / $limit);
                $offset = ($page - 1) * $limit;
                
                // Get paginated data
                $paginatedData = array_slice($allEquipment, $offset, $limit);
                
                Log::info('Equipment fetched successfully', [
                    'total' => $totalRecords,
                    'page' => $page,
                    'totalPages' => $totalPages
                ]);
                
                return response()->json([
                    'success' => true,
                    'data' => $paginatedData,
                    'pagination' => [
                        'currentPage' => (int)$page,
                        'totalPages' => $totalPages,
                        'totalRecords' => $totalRecords,
                        'limit' => (int)$limit,
                        'hasNext' => $page < $totalPages,
                        'hasPrev' => $page > 1
                    ]
                ]);
            }
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch equipment'
            ], 500);
            
        } catch (\Exception $e) {
            Log::error('Exception in getAllEquipment', [
                'message' => $e->getMessage()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Delete equipment from MongoDB
     */
    public function deleteEquipment($equipmentId)
    {
        try {
            $mongoUrl = env('MONGODB_EQUIPMENT_DELETE_URL', 'http://168.231.113.219:5001/equipment') . '/' . $equipmentId;
            
            Log::info('Deleting equipment from MongoDB', ['equipmentId' => $equipmentId]);
            
            $response = Http::timeout(10)->delete($mongoUrl);
            
            if ($response->successful()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Equipment deleted successfully'
                ]);
            }
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete equipment'
            ], 500);
            
        } catch (\Exception $e) {
            Log::error('Exception in deleteEquipment', [
                'message' => $e->getMessage(),
                'equipmentId' => $equipmentId
            ]);
            
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }

    public function overview()
    {
        return view('vehicles.Vehicles_Overview');
    }

    public function addVehicle()
    {
        return view('vehicles.add-vehicle');
    }

    public function dvirDotInspection(Request $request)
    {
        try {
            // Fetch DVIR data from the API endpoint
            $response = Http::get('https://titan-drilling-1f8e9-default-rtdb.firebaseio.com/GlobalDvirReports/admins/cahj3jJqe5Z16pQdLGSWrNgm84o1/inspectionReports.json');
            
            if ($response->successful()) {
                $dvirData = $response->json() ?? [];
                
                // Process the data to match the required table structure
                $processedData = [];
                
                if (is_array($dvirData)) {
                    foreach ($dvirData as $reportId => $report) {
                        if (is_array($report)) {
                            // Fetch requestNumber from the nested path if step1 exists
                            $requestNumber = 'N/A';
                            if (isset($report['step1']) && isset($report['step1']['requestNumber'])) {
                                $requestNumber = $report['step1']['requestNumber'];
                            } elseif (isset($report['requestNumber'])) {
                                $requestNumber = $report['requestNumber'];
                            }
                            
                            // Fetch vehicleNumber from step1/selectedEquipment/vehicleNumber
                            $vehicleNumber = 'N/A';
                            if (isset($report['step1']) && isset($report['step1']['selectedEquipment']) && isset($report['step1']['selectedEquipment']['vehicleNumber'])) {
                                $vehicleNumber = $report['step1']['selectedEquipment']['vehicleNumber'];
                            } elseif (isset($report['vehicle']) || isset($report['vehicleNumber']) || isset($report['vehicleName'])) {
                                $vehicleNumber = $report['vehicle'] ?? $report['vehicleNumber'] ?? $report['vehicleName'] ?? 'N/A';
                            }
                            
                            // Fetch location from step1/country
                            $location = 'N/A';
                            if (isset($report['step1']) && isset($report['step1']['country'])) {
                                $location = $report['step1']['country'];
                            } elseif (isset($report['location']) || isset($report['site']) || isset($report['siteLocation'])) {
                                $location = $report['location'] ?? $report['site'] ?? $report['siteLocation'] ?? 'N/A';
                            }
                            
                            // Fetch inspector from step1/inspector
                            $inspector = 'N/A';
                            if (isset($report['step1']) && isset($report['step1']['inspector'])) {
                                $inspector = $report['step1']['inspector'];
                            } elseif (isset($report['inspector']) || isset($report['inspectorName']) || isset($report['operator'])) {
                                $inspector = $report['inspector'] ?? $report['inspectorName'] ?? $report['operator'] ?? 'N/A';
                            }
                            
                            // Log the extracted data for debugging (only for first few records)
                            if (count($processedData) < 3) {
                                \Log::info("DVIR Data Extraction for Report ID: $reportId", [
                                    'requestNumber' => $requestNumber,
                                    'vehicleNumber' => $vehicleNumber,
                                    'location' => $location,
                                    'inspector' => $inspector,
                                    'hasStep1' => isset($report['step1']),
                                    'step1Keys' => isset($report['step1']) ? array_keys($report['step1']) : [],
                                    'hasSelectedEquipment' => isset($report['step1']['selectedEquipment']),
                                    'selectedEquipmentKeys' => isset($report['step1']['selectedEquipment']) ? array_keys($report['step1']['selectedEquipment']) : []
                                ]);
                            }
                            
                            // Extract faulty items count from step3
                            $faultyItems = 0;
                            if (isset($report['step3']['faultyItems'])) {
                                $faultyItems = $report['step3']['faultyItems'];
                            } elseif (isset($report['step2']['faultyItems'])) {
                                $faultyItems = $report['step2']['faultyItems'];
                            } elseif (isset($report['faultyItems'])) {
                                $faultyItems = $report['faultyItems'];
                            }
                            
                            // Determine status based on available data
                            $status = 'Completed';
                            if (isset($report['status'])) {
                                $status = $report['status'];
                            } elseif (isset($report['isCompleted']) && $report['isCompleted'] === false) {
                                $status = 'Pending';
                            } elseif ($faultyItems > 0 || (isset($report['fault']) && $report['fault'] !== 'No faults found')) {
                                $status = 'Issues Found';
                            }
                            
                            $processedReport = [
                                'id' => $reportId,
                                'reportNumber' => $requestNumber,
                                'date' => $report['date'] ?? $report['createdAt'] ?? $report['timestamp'] ?? 'N/A',
                                'vehicle' => $vehicleNumber,
                                'location' => $location,
                                'inspector' => $inspector,
                                'status' => $status,
                                'faultyItems' => $faultyItems,
                                'fault' => $report['fault'] ?? $report['faults'] ?? $report['issues'] ?? 'No faults found',
                                'correctiveAction' => $report['correctiveAction'] ?? $report['correctiveActions'] ?? $report['actions'] ?? 'No actions specified'
                            ];
                            
                            $processedData[] = $processedReport;
                        }
                    }
                }
                
                // Sort by date (newest first)
                usort($processedData, function($a, $b) {
                    try {
                        $dateA = strtotime($a['date']);
                        $dateB = strtotime($b['date']);
                        return $dateB - $dateA;
                    } catch (\Exception $e) {
                        return 0;
                    }
                });
                
                // Add serial numbers
                foreach ($processedData as $index => &$report) {
                    $report['serial_number'] = $index + 1;
                }
                
                // Pagination logic
                $perPage = $request->get('per_page', 10); // Default 10 records per page
                $currentPage = $request->get('page', 1);
                $totalRecords = count($processedData);
                $totalPages = ceil($totalRecords / $perPage);
                
                // Calculate offset
                $offset = ($currentPage - 1) * $perPage;
                
                // Get paginated data
                $paginatedData = array_slice($processedData, $offset, $perPage);
                
                // Pagination info
                $pagination = [
                    'current_page' => $currentPage,
                    'per_page' => $perPage,
                    'total_records' => $totalRecords,
                    'total_pages' => $totalPages,
                    'has_previous' => $currentPage > 1,
                    'has_next' => $currentPage < $totalPages,
                    'previous_page' => $currentPage > 1 ? $currentPage - 1 : null,
                    'next_page' => $currentPage < $totalPages ? $currentPage + 1 : null,
                    'start_record' => $offset + 1,
                    'end_record' => min($offset + $perPage, $totalRecords)
                ];
                
            } else {
                $paginatedData = [];
                $pagination = [
                    'current_page' => 1,
                    'per_page' => 10,
                    'total_records' => 0,
                    'total_pages' => 0,
                    'has_previous' => false,
                    'has_next' => false,
                    'previous_page' => null,
                    'next_page' => null,
                    'start_record' => 0,
                    'end_record' => 0
                ];
                \Log::error('Failed to fetch DVIR reports: ' . $response->status());
            }
        } catch (\Exception $e) {
            $paginatedData = [];
            $pagination = [
                'current_page' => 1,
                'per_page' => 10,
                'total_records' => 0,
                'total_pages' => 0,
                'has_previous' => false,
                'has_next' => false,
                'previous_page' => null,
                'next_page' => null,
                'start_record' => 0,
                'end_record' => 0
            ];
            \Log::error('Error fetching DVIR reports: ' . $e->getMessage());
        }

        return view('dvir_dot_inspection.DVIR_DOT_Inspection', compact('paginatedData', 'pagination'));
    }

    public function testDvirDataStructure()
    {
        try {
            // Fetch DVIR data from the API endpoint
            $response = Http::get('https://titan-drilling-1881a-default-rtdb.firebaseio.com/GlobalDvirReports/admins/cahj3jJqe5Z16pQdLGSWrNgm84o1/inspectionReports.json');
            
            if ($response->successful()) {
                $dvirData = $response->json() ?? [];
                
                $analysis = [
                    'total_reports' => is_array($dvirData) ? count($dvirData) : 0,
                    'sample_reports' => []
                ];
                
                if (is_array($dvirData) && !empty($dvirData)) {
                    // Analyze first few reports
                    $sampleReports = array_slice($dvirData, 0, 3, true);
                    
                    foreach ($sampleReports as $reportId => $report) {
                        if (is_array($report)) {
                            $reportAnalysis = [
                                'report_id' => $reportId,
                                'has_step1' => isset($report['step1']),
                                'step1_keys' => isset($report['step1']) ? array_keys($report['step1']) : [],
                                'request_number_paths' => [
                                    'step1.requestNumber' => isset($report['step1']['requestNumber']) ? $report['step1']['requestNumber'] : 'Not found',
                                    'requestNumber' => isset($report['requestNumber']) ? $report['requestNumber'] : 'Not found'
                                ],
                                'vehicle_paths' => [
                                    'step1.selectedEquipment.vehicleNumber' => isset($report['step1']['selectedEquipment']['vehicleNumber']) ? $report['step1']['selectedEquipment']['vehicleNumber'] : 'Not found',
                                    'vehicle' => isset($report['vehicle']) ? $report['vehicle'] : 'Not found',
                                    'vehicleNumber' => isset($report['vehicleNumber']) ? $report['vehicleNumber'] : 'Not found'
                                ],
                                'location_paths' => [
                                    'step1.country' => isset($report['step1']['country']) ? $report['step1']['country'] : 'Not found',
                                    'location' => isset($report['location']) ? $report['location'] : 'Not found'
                                ],
                                'inspector_paths' => [
                                    'step1.inspector' => isset($report['step1']['inspector']) ? $report['step1']['inspector'] : 'Not found',
                                    'inspector' => isset($report['inspector']) ? $report['inspector'] : 'Not found'
                                ]
                            ];
                            
                            $analysis['sample_reports'][$reportId] = $reportAnalysis;
                        }
                    }
                }
                
                return response()->json([
                    'success' => true,
                    'analysis' => $analysis,
                    'message' => 'DVIR data structure analysis completed'
                ]);
                
            } else {
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to fetch DVIR data: ' . $response->status()
                ], 500);
            }
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error analyzing DVIR data: ' . $e->getMessage()
            ], 500);
        }
    }

    public function testEquipmentData()
    {
        $data = $this->fetchEquipmentData();
        
        // Get raw data for first equipment item to show structure
        $rawData = [];
        if (!empty($data)) {
            $firstEquipmentId = $data[0]['id'];
            $rawData = [
                'main_equipment_data' => Http::get("https://titan-drilling-1881a-default-rtdb.firebaseio.com/EquipmentsDataJSON/GlobalEquipmentJsonData/{$firstEquipmentId}.json")->json()
            ];
        }
        
        return response()->json([
            'success' => true,
            'count' => count($data),
            'data' => $data,
            'raw_firebase_data' => Http::get('https://titan-drilling-1881a-default-rtdb.firebaseio.com/EquipmentsDataJSON/GlobalEquipmentJsonData.json')->json(),
            'sample_equipment_structure' => $rawData
        ]);
    }

    private function fetchEquipmentData()
    {
        try {
            $equipmentUrl = 'https://titan-drilling-1f8e9-default-rtdb.firebaseio.com/EquipmentsDataJSON/GlobalEquipmentJsonData.json';
            
            Log::info('Fetching equipment data from Firebase', ['url' => $equipmentUrl]);
            
            $response = Http::timeout(30)->get($equipmentUrl);
            
            if (!$response->successful()) {
                Log::error('Failed to fetch equipment data from Firebase', [
                    'status' => $response->status(),
                    'body' => $response->body()
                ]);
                return [];
            }

            $equipmentData = $response->json() ?? [];
            
            if (empty($equipmentData)) {
                Log::info('No equipment data found at the specified path');
                return [];
            }

            Log::info('Successfully fetched equipment data', ['count' => count($equipmentData)]);

            // Process the data to extract required fields for vehicles inventory
            $processedData = [];
            
            foreach ($equipmentData as $equipmentId => $equipment) {
                if (is_array($equipment)) {
                    $processedEquipment = [
                        'id' => $equipmentId,
                        'vehicleNumber' => $equipment['vehicleNumber'] ?? $equipment['equipmentNumber'] ?? $equipment['id'] ?? 'N/A',
                        'vehicleName' => $equipment['equipmentName'] ?? $equipment['vehicleName'] ?? $equipment['name'] ?? 'N/A',
                        'year' => $equipment['year'] ?? $equipment['manufactureYear'] ?? 'N/A',
                        'model' => $equipment['model'] ?? $equipment['equipmentModel'] ?? 'N/A',
                        'category' => $equipment['category'] ?? $equipment['vehicleType'] ?? $equipment['equipmentCategory'] ?? $equipment['type'] ?? 'N/A',
                        'manufacturer' => $equipment['make'] ?? $equipment['manufacturer'] ?? 'N/A',
                        'chassisNumber' => $equipment['chassisNumber'] ?? $equipment['serialNumber'] ?? 'N/A',
                        'vinNumber' => $equipment['vinNumber'] ?? $equipment['vin'] ?? 'N/A',
                        'licensePlate' => $equipment['registrationNumber'] ?? $equipment['licensePlate'] ?? $equipment['plateNumber'] ?? 'N/A',
                        'status' => $equipment['status'] ?? $equipment['equipmentStatus'] ?? 'Available',
                        'meterReading' => $equipment['meterReading'] ?? $equipment['meter'] ?? 'N/A',
                        'meterUnit' => $equipment['meterType'] ?? $equipment['meterUnit'] ?? $equipment['unit'] ?? 'Miles',
                        'lastServiceDate' => $equipment['lastServiceDate'] ?? $equipment['lastMaintenance'] ?? 'N/A',
                        'lastServiceMeter' => $equipment['lastServiceReading'] ?? $equipment['lastServiceMeter'] ?? 'N/A',
                        'site' => $equipment['country'] ?? $equipment['site'] ?? $equipment['location'] ?? 'N/A',
                        'operator' => $equipment['operatorName'] ?? $equipment['operator'] ?? $equipment['driver'] ?? 'N/A',
                        'businessUnit' => $equipment['businessUnit'] ?? $equipment['unit'] ?? 'N/A',
                        'assetValue' => $equipment['fleetValue'] ?? $equipment['assetValue'] ?? $equipment['value'] ?? 'N/A',
                        'purchaseDate' => $equipment['purchaseDate'] ?? $equipment['acquisitionDate'] ?? 'N/A',
                        'warrantyDate' => $equipment['warrantyDate'] ?? $equipment['warrantyExpiry'] ?? 'N/A',
                        'registrationExpDate' => $equipment['registrationExpDate'] ?? $equipment['registrationExpiry'] ?? 'N/A',
                        'maintenancePriority' => $equipment['maintenancePriority'] ?? $equipment['priority'] ?? 'Normal',
                        'ownershipMode' => $equipment['ownershipMode'] ?? $equipment['ownership'] ?? 'N/A',
                        'notes' => $equipment['comment'] ?? $equipment['notes'] ?? $equipment['description'] ?? $equipment['comments'] ?? 'N/A'
                    ];
                    
                    $processedData[] = $processedEquipment;
                }
            }

            return $processedData;

        } catch (\Exception $e) {
            Log::error('Error fetching equipment data: ' . $e->getMessage());
            return [];
        }
    }

    public function storeEquipment(Request $request)
    {
        try {
            $data = $request->input('data');
            
            // Generate a unique key for the new equipment
            $uniqueKey = 'EQ-' . time() . '-' . uniqid();
            
            // Store the data with the unique key at the new path
            $response = Http::put('https://titan-drilling-1f8e9-default-rtdb.firebaseio.com/EquipmentsDataJSON/GlobalEquipmentJsonData/' . $uniqueKey . '.json', $data);
            
            if ($response->successful()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Equipment stored successfully',
                    'key' => $uniqueKey,
                    'data' => $data
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to store equipment data',
                    'status' => $response->status()
                ], 500);
            }
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error storing equipment data: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Fetch equipment data from Firebase for the work order creation
     */
    public function getFirebaseEquipment()
    {
        try {
            $response = Http::get('https://titan-drilling-1f8e9-default-rtdb.firebaseio.com/EquipmentsDataJSON/GlobalEquipmentJsonData.json');
            
            if ($response->successful()) {
                $equipmentData = $response->json() ?? [];
                
                // Transform the data to match the expected format
                $transformedEquipment = [];
                
                if (is_array($equipmentData)) {
                    foreach ($equipmentData as $equipmentId => $equipment) {
                        if (is_array($equipment)) {
                            $transformedEquipment[] = [
                                'id' => $equipmentId,
                                'equipmentName' => $equipment['equipmentName'] ?? $equipment['name'] ?? 'Unknown Equipment',
                                'equipmentNumber' => $equipment['equipmentNumber'] ?? $equipment['equipmentId'] ?? $equipmentId,
                                'model' => $equipment['model'] ?? $equipment['equipmentModel'] ?? 'N/A',
                                'serialNumber' => $equipment['serialNumber'] ?? $equipment['serial'] ?? 'N/A',
                                'meterReading' => $equipment['meterReading'] ?? $equipment['currentMeter'] ?? '0',
                                'meterUnit' => $equipment['meterUnit'] ?? $equipment['unit'] ?? 'Hours',
                                'status' => $equipment['status'] ?? $equipment['equipmentStatus'] ?? 'Active',
                                'location' => $equipment['location'] ?? $equipment['site'] ?? $equipment['country'] ?? 'N/A',
                                'equipmentType' => $equipment['equipmentType'] ?? $equipment['type'] ?? 'N/A',
                                'manufacturer' => $equipment['manufacturer'] ?? 'N/A',
                                'year' => $equipment['year'] ?? 'N/A'
                            ];
                        }
                    }
                }
                
                Log::info('Firebase Equipment Data Fetched', [
                    'total_equipment' => count($transformedEquipment),
                    'sample_equipment' => count($transformedEquipment) > 0 ? $transformedEquipment[0] : 'No data'
                ]);
                
                return response()->json([
                    'success' => true,
                    'message' => 'Equipment data fetched successfully',
                    'equipment' => $transformedEquipment,
                    'count' => count($transformedEquipment)
                ]);
            } else {
                Log::error('Failed to fetch equipment data from Firebase', [
                    'status' => $response->status(),
                    'body' => $response->body()
                ]);
                
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to fetch equipment data from Firebase',
                    'status' => $response->status()
                ], $response->status());
            }
        } catch (\Exception $e) {
            Log::error('Exception while fetching equipment data from Firebase', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Exception occurred while fetching equipment data: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Store Work Order to Firebase
     */
    public function storeWorkOrderToFirebase(Request $request)
    {
        try {
            $request->validate([
                'path' => 'required|string',
                'data' => 'required|array'
            ]);

            $path = $request->input('path');
            $data = $request->input('data');

            // Ensure the path starts with the correct Firebase structure
            if (!str_starts_with($path, '/GlobalWorkOrders/admins/')) {
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid path format. Must start with /GlobalWorkOrders/admins/'
                ]);
            }

            // Convert the path to Firebase format (remove leading slash and add .json)
            $firebasePath = ltrim($path, '/') . '.json';
            
            // Firebase Realtime Database URL
            $firebaseUrl = 'https://titan-drilling-1881a-default-rtdb.firebaseio.com/' . $firebasePath;

            // Send data to Firebase
            $response = Http::put($firebaseUrl, $data);

            if ($response->successful()) {
                Log::info('Work Order saved to Firebase successfully', [
                    'path' => $path,
                    'workOrderId' => basename($path),
                    'response' => $response->body(),
                    'timestamp' => now()
                ]);

                return response()->json([
                    'success' => true,
                    'message' => 'Work Order saved successfully',
                    'workOrderId' => basename($path),
                    'path' => $path
                ]);
            } else {
                Log::error('Failed to save Work Order to Firebase', [
                    'path' => $path,
                    'response' => $response->body(),
                    'status' => $response->status()
                ]);

                return response()->json([
                    'success' => false,
                    'message' => 'Failed to save to Firebase: ' . $response->status()
                ]);
            }
        } catch (\Exception $e) {
            Log::error('Error storing Work Order to Firebase: ' . $e->getMessage(), [
                'path' => $request->input('path'),
                'data' => $request->input('data')
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Error storing Work Order: ' . $e->getMessage()
            ]);
        }
    }
}