@extends('welcome')

@section('title', 'Equipment KPI')

@section('styles')
<link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
<link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
@endsection

@section('content')
<div class="equipment-kpi-container">
    <!-- Loading Overlay - Matching Inspection Design -->
    <div id="loading-overlay" class="sync-modal" style="display: none;">
        <div class="sync-content">
            <div class="sync-icon">⏳</div>
            <div class="sync-text">Please Wait</div>
            <div class="sync-subtext">Loading Equipment Data...</div>
        </div>
    </div>
    
    <!-- Header - Matching Inspection Design -->
    <div class="header-gradient">
        <div class="header-content">
            <div class="header-left">
                <a href="{{ url()->previous() }}" class="back-button">
                    <i class="fas fa-arrow-left"></i>
                </a>
                <h1 class="header-title">Equipment KPI Dashboard</h1>
            </div>
            <div class="header-right">
                <button type="button" class="create-btn" id="exportExcelBtn">
                    <i class="fas fa-file-excel"></i>
                    Export to Excel
                </button>
                <button type="button" class="create-btn" id="refreshDataBtn">
                    <i class="fas fa-sync-alt"></i>
                    Refresh
                </button>
            </div>
        </div>
    </div>
    
    <div class="equipment-kpi-content">
        <div class="content-box">
            <!-- Time Period Selector -->
            <div class="time-period-selector">
                <div class="period-buttons">
                    <button class="period-btn" data-period="today">Today</button>
                    <button class="period-btn active" data-period="7days">Last 7 Days</button>
                    <button class="period-btn" data-period="30days">Last 30 Days</button>
                    <button class="period-btn" data-period="90days">Last 90 Days</button>
                    <button class="period-btn" data-period="custom">Custom Date</button>
                </div>
                
                <!-- Custom Date Range Picker -->
                <div class="custom-date-range" id="custom-date-range" style="display: none;">
                    <div class="date-range-inputs">
                        <div class="date-input-group">
                            <label for="start-date">Start Date:</label>
                            <input type="date" id="start-date" class="date-input">
                        </div>
                        <div class="date-input-group">
                            <label for="end-date">End Date:</label>
                            <input type="date" id="end-date" class="date-input">
                        </div>
                        <button class="btn-apply-dates" id="apply-custom-dates">Apply</button>
                        <button class="btn-clear-dates" id="clear-custom-dates">Clear</button>
                    </div>
                </div>
            </div>
            
            <!-- Modern Tabs -->
            <div class="modern-tabs">
                <div class="tab-navigation">
                    <button class="tab-btn active" data-tab="overview">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <rect x="3" y="3" width="18" height="18" rx="2" ry="2"/>
                            <path d="M9 9h6v6H9z"/>
                        </svg>
                        Overview
                    </button>
                    <button class="tab-btn" data-tab="by-category">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <path d="M3 3h18v18H3zM9 9h6v6H9z"/>
                        </svg>
                        By Category
                    </button>
                    <button class="tab-btn" data-tab="by-status">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <circle cx="12" cy="12" r="10"/>
                            <path d="M12 6v6l4 2"/>
                        </svg>
                        By Status
                    </button>
                    <button class="tab-btn" data-tab="details">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <line x1="8" y1="6" x2="21" y2="6"/>
                            <line x1="8" y1="12" x2="21" y2="12"/>
                            <line x1="8" y1="18" x2="21" y2="18"/>
                            <line x1="3" y1="6" x2="3.01" y2="6"/>
                            <line x1="3" y1="12" x2="3.01" y2="12"/>
                            <line x1="3" y1="18" x2="3.01" y2="18"/>
                        </svg>
                        Equipment Details
                    </button>
                </div>

                <!-- Tab Content -->
                <div class="tab-content">
                    <!-- Overview Tab -->
                    <div id="overview" class="tab-pane active">
                        <div class="kpi-cards-grid">
                            <div class="kpi-card" data-kpi="total-equipment">
                                <div class="kpi-icon">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <rect x="1" y="3" width="15" height="13"/>
                                        <polygon points="16 8 20 8 23 11 23 16 16 16 16 8"/>
                                        <circle cx="5.5" cy="18.5" r="2.5"/>
                                        <circle cx="18.5" cy="18.5" r="2.5"/>
                                    </svg>
                                </div>
                                <div class="kpi-details">
                                    <h3 class="kpi-title">Total Equipment</h3>
                                    <p class="kpi-value" id="total-equipment-count">0</p>
                                    <p class="kpi-subtitle">All registered equipment</p>
                                </div>
                            </div>

                            <div class="kpi-card" data-kpi="available-equipment">
                                <div class="kpi-icon" style="background: linear-gradient(135deg, #10b981 0%, #059669 100%);">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <path d="M22 11.08V12a10 10 0 1 1-5.93-9.14"/>
                                        <polyline points="22 4 12 14.01 9 11.01"/>
                                    </svg>
                                </div>
                                <div class="kpi-details">
                                    <h3 class="kpi-title">Available</h3>
                                    <p class="kpi-value" id="available-equipment-count">0</p>
                                    <p class="kpi-subtitle">Ready for use</p>
                                </div>
                            </div>

                            <div class="kpi-card" data-kpi="in-use-equipment">
                                <div class="kpi-icon" style="background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <circle cx="12" cy="12" r="10"/>
                                        <polyline points="12 6 12 12 16 14"/>
                                    </svg>
                                </div>
                                <div class="kpi-details">
                                    <h3 class="kpi-title">In Use</h3>
                                    <p class="kpi-value" id="in-use-equipment-count">0</p>
                                    <p class="kpi-subtitle">Currently operating</p>
                                </div>
                            </div>

                            <div class="kpi-card" data-kpi="maintenance-equipment">
                                <div class="kpi-icon" style="background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%);">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <path d="M14.7 6.3a1 1 0 0 0 0 1.4l1.6 1.6a1 1 0 0 0 1.4 0l3.77-3.77a6 6 0 0 1-7.94 7.94l-6.91 6.91a2.12 2.12 0 0 1-3-3l6.91-6.91a6 6 0 0 1 7.94-7.94l-3.76 3.76z"/>
                                    </svg>
                                </div>
                                <div class="kpi-details">
                                    <h3 class="kpi-title">In Maintenance</h3>
                                    <p class="kpi-value" id="maintenance-equipment-count">0</p>
                                    <p class="kpi-subtitle">Under repair/service</p>
                                </div>
                            </div>
                        </div>

                        <!-- Charts Section -->
                        <div class="charts-grid">
                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Equipment by Category</h3>
                                    <button class="chart-expand-btn">
                                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                            <polyline points="15 3 21 3 21 9"/>
                                            <polyline points="9 21 3 21 3 15"/>
                                            <line x1="21" y1="3" x2="14" y2="10"/>
                                            <line x1="3" y1="21" x2="10" y2="14"/>
                                        </svg>
                                    </button>
                                </div>
                                <div class="chart-body">
                                    <canvas id="categoryChart"></canvas>
                                </div>
                            </div>

                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Equipment by Status</h3>
                                    <button class="chart-expand-btn">
                                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                            <polyline points="15 3 21 3 21 9"/>
                                            <polyline points="9 21 3 21 3 15"/>
                                            <line x1="21" y1="3" x2="14" y2="10"/>
                                            <line x1="3" y1="21" x2="10" y2="14"/>
                                        </svg>
                                    </button>
                                </div>
                                <div class="chart-body">
                                    <canvas id="statusChart"></canvas>
                                </div>
                            </div>
                        </div>

                        <!-- Utilization Rate -->
                        <div class="utilization-card">
                            <div class="utilization-header">
                                <h3>Equipment Utilization Rate</h3>
                                <span class="utilization-percentage" id="utilization-rate">0%</span>
                            </div>
                            <div class="utilization-bar">
                                <div class="utilization-progress" id="utilization-progress" style="width: 0%"></div>
                            </div>
                            <div class="utilization-info">
                                <span><span id="in-use-count">0</span> in use</span>
                                <span>out of <span id="total-count">0</span> total equipment</span>
                            </div>
                        </div>
                    </div>

                    <!-- By Category Tab -->
                    <div id="by-category" class="tab-pane">
                        <div class="category-stats-grid" id="category-stats-grid">
                            <!-- Will be populated dynamically -->
                        </div>
                        <div class="chart-card">
                            <div class="chart-header">
                                <h3>Category Distribution</h3>
                            </div>
                            <div class="chart-body">
                                <canvas id="categoryDistributionChart"></canvas>
                            </div>
                        </div>
                        <div class="data-table-container">
                            <h3>Equipment by Category</h3>
                            <table class="data-table" id="category-table">
                                <thead>
                                    <tr>
                                        <th>Category</th>
                                        <th>Count</th>
                                        <th>Available</th>
                                        <th>In Use</th>
                                        <th>In Maintenance</th>
                                        <th>Percentage</th>
                                    </tr>
                                </thead>
                                <tbody id="category-table-body">
                                    <!-- Will be populated dynamically -->
                                </tbody>
                            </table>
                        </div>
                    </div>

                    <!-- By Status Tab -->
                    <div id="by-status" class="tab-pane">
                        <div class="status-stats-grid" id="status-stats-grid">
                            <!-- Will be populated dynamically -->
                        </div>
                        <div class="chart-card">
                            <div class="chart-header">
                                <h3>Status Distribution Over Time</h3>
                            </div>
                            <div class="chart-body">
                                <canvas id="statusTimelineChart"></canvas>
                            </div>
                        </div>
                    </div>

                    <!-- Equipment Details Tab -->
                    <div id="details" class="tab-pane">
                        <div class="table-controls">
                            <div class="search-box">
                                <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <circle cx="11" cy="11" r="8"/>
                                    <path d="m21 21-4.35-4.35"/>
                                </svg>
                                <input type="text" id="equipment-search" placeholder="Search equipment...">
                            </div>
                            <div class="filter-controls">
                                <select id="category-filter" class="filter-select">
                                    <option value="">All Categories</option>
                                </select>
                                <select id="status-filter" class="filter-select">
                                    <option value="">All Statuses</option>
                                    <option value="Available">Available</option>
                                    <option value="In Use">In Use</option>
                                    <option value="Maintenance">Maintenance</option>
                                    <option value="Maintenance Required">Maintenance Required</option>
                                    <option value="Out of Service">Out of Service</option>
                                </select>
                            </div>
                        </div>
                        <div class="data-table-container">
                            <table class="data-table" id="equipment-details-table">
                                <thead>
                                    <tr>
                                        <th>Vehicle Number</th>
                                        <th>Name</th>
                                        <th>Registration</th>
                                        <th>Make & Model</th>
                                        <th>Category</th>
                                        <th>Status</th>
                                        <th>Meter Reading</th>
                                        <th>Last Updated</th>
                                    </tr>
                                </thead>
                                <tbody id="equipment-details-body">
                                    <!-- Will be populated dynamically -->
                                </tbody>
                            </table>
                        </div>
                        <div class="table-pagination">
                            <button class="pagination-btn" id="prev-page" disabled>Previous</button>
                            <span class="pagination-info">
                                Page <span id="current-page">1</span> of <span id="total-pages">1</span>
                            </span>
                            <button class="pagination-btn" id="next-page">Next</button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<style>
/* CSS Variables for consistent theming - Matching Inspection KPI */
:root {
    --primary-color: #0055a5;
    --primary-hover: #003f7d;
    --secondary-color: #6c757d;
    --success-color: #28a745;
    --warning-color: #ffc107;
    --danger-color: #dc3545;
    --info-color: #17a2b8;
    --text-primary: #212529;
    --text-secondary: #6c757d;
    --text-muted: #64748b;
    --text-light: #cbd5e1;
    --bg-primary: #ffffff;
    --bg-secondary: #f8f9fa;
    --bg-light: #f8fafc;
    --bg-white: #ffffff;
    --border-color: #dee2e6;
    --border-light: #e2e8f0;
    --shadow-sm: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
    --shadow-md: 0 0.5rem 1rem rgba(0, 0, 0, 0.15);
    --shadow-lg: 0 1rem 3rem rgba(0, 0, 0, 0.175);
    --radius-sm: 0.25rem;
    --radius-md: 0.375rem;
    --radius-lg: 0.5rem;
    --transition-base: all 0.15s ease-in-out;
    --font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
    --z-index-notification: 10000;
    --z-index-modal: 1055;
}

/* ===== MAIN CONTAINER ===== */
.equipment-kpi-container {
    min-height: 100vh;
    background: var(--bg-light);
    font-family: var(--font-family);
    padding: 0;
    margin: 0;
    overflow-x: hidden;
    overflow-y: visible;
    color: var(--text-primary);
}

/* ===== HEADER STYLES - Matching Inspection Design ===== */
.header-gradient {
    background: #0055a5;
    padding: 20px 0;
    box-shadow: 0 4px 12px rgba(0, 85, 165, 0.3);
}

.header-content {
    width: 100%;
    padding: 0 20px;
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.header-left {
    display: flex;
    align-items: center;
    gap: 15px;
}

.back-button {
    color: white;
    font-size: 20px;
    text-decoration: none;
    padding: 8px;
    border-radius: 50%;
    transition: var(--transition-base);
}

.back-button:hover {
    background-color: rgba(255, 255, 255, 0.1);
    color: white;
}

.header-title {
    color: white;
    font-size: 28px;
    font-weight: 700;
    margin: 0;
}

.header-right {
    display: flex;
    align-items: center;
    gap: 15px;
}

.create-btn {
    background: white;
    color: var(--primary-color);
    border: none;
    padding: 12px 24px;
    border-radius: var(--radius-md);
    font-size: 16px;
    font-weight: 600;
    cursor: pointer;
    transition: var(--transition-base);
    box-shadow: var(--shadow-sm);
    display: inline-flex;
    align-items: center;
    gap: 8px;
}

.create-btn:hover {
    background: rgba(255, 255, 255, 0.9);
    transform: translateY(-2px);
    box-shadow: var(--shadow-md);
}

.create-btn i {
    font-size: 18px;
}

/* ===== MAIN CONTENT ===== */
.equipment-kpi-content {
    padding: 30px 20px;
    background: var(--bg-light);
    min-height: auto;
    overflow: visible;
}

.content-box {
    background: white;
    padding: 30px;
    border-radius: var(--radius-lg);
    box-shadow: var(--shadow-md);
    border: 1px solid var(--border-light);
    font-family: var(--font-family);
    margin-bottom: 1rem;
    margin: 0;
    overflow: hidden;
}

/* Loading/Sync Modal - Matching Inspection Design */
.sync-modal {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0, 0, 0, 0.5);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 10001;
}

.sync-content {
    background: white;
    border-radius: var(--radius-lg);
    padding: 40px;
    text-align: center;
    max-width: 300px;
    box-shadow: var(--shadow-lg);
}

.sync-icon {
    width: 40px;
    height: 40px;
    margin: 0 auto 20px;
    font-size: 40px;
    animation: spin 1s linear infinite;
}

@keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}

.sync-text {
    font-size: 18px;
    font-weight: 600;
    color: var(--text-primary);
    margin-bottom: 10px;
}

.sync-subtext {
    font-size: 14px;
    color: var(--text-secondary);
}

/* ===== TIME PERIOD SELECTOR ===== */
.time-period-selector {
    margin-bottom: 2rem;
}

.period-buttons {
    display: flex;
    gap: 0.5rem;
    flex-wrap: wrap;
}

.period-btn {
    padding: 12px 24px;
    background: white;
    color: var(--text-primary);
    border: 1px solid var(--border-color);
    border-radius: var(--radius-md);
    font-size: 14px;
    font-weight: 500;
    font-family: var(--font-family);
    cursor: pointer;
    transition: var(--transition-base);
    white-space: nowrap;
    box-shadow: var(--shadow-sm);
}

.period-btn:hover {
    background: var(--bg-light);
    border-color: var(--primary-color);
    color: var(--primary-color);
    transform: translateY(-1px);
}

.period-btn.active {
    background: var(--primary-color);
    color: white;
    border-color: var(--primary-color);
    box-shadow: 0 4px 12px rgba(0, 85, 165, 0.3);
}

/* Custom Date Range Picker */
.custom-date-range {
    margin-top: 1rem;
    padding: 1.5rem;
    background: white;
    border: 1px solid var(--border-light);
    border-radius: var(--radius-lg);
    box-shadow: var(--shadow-md);
    animation: slideDown 0.3s ease-out;
}

@keyframes slideDown {
    from {
        opacity: 0;
        transform: translateY(-10px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.date-range-inputs {
    display: flex;
    align-items: end;
    gap: 1rem;
    flex-wrap: wrap;
}

.date-input-group {
    display: flex;
    flex-direction: column;
    gap: 8px;
}

.date-input-group label {
    font-size: 14px;
    font-weight: 500;
    color: var(--text-primary);
    font-family: var(--font-family);
}

.date-input {
    padding: 12px 16px;
    border: 1px solid var(--border-color);
    border-radius: var(--radius-md);
    font-size: 14px;
    font-weight: 400;
    color: var(--text-primary);
    background: white;
    font-family: var(--font-family);
    transition: var(--transition-base);
    min-width: 180px;
    cursor: pointer;
}

.date-input:focus {
    outline: none;
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgba(0, 85, 165, 0.1);
}

.btn-apply-dates,
.btn-clear-dates {
    padding: 12px 24px;
    border: none;
    border-radius: var(--radius-md);
    font-size: 14px;
    font-weight: 600;
    font-family: var(--font-family);
    cursor: pointer;
    transition: var(--transition-base);
    white-space: nowrap;
    box-shadow: var(--shadow-sm);
}

.btn-apply-dates {
    background: var(--primary-color);
    color: white;
}

.btn-apply-dates:hover {
    background: var(--primary-hover);
    transform: translateY(-2px);
    box-shadow: var(--shadow-md);
}

.btn-clear-dates {
    background: white;
    color: var(--text-primary);
    border: 1px solid var(--border-color);
}

.btn-clear-dates:hover {
    background: var(--bg-light);
    border-color: var(--primary-color);
    color: var(--primary-color);
    transform: translateY(-1px);
}

/* ===== MODERN TABS ===== */
.modern-tabs {
    margin-bottom: 2rem;
}

.tab-navigation {
    display: flex;
    gap: 5px;
    flex-wrap: wrap;
    border-bottom: 1px solid var(--border-light);
    padding-bottom: 0;
    margin-bottom: 2rem;
    overflow-x: auto;
}

.tab-btn {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 12px 16px;
    background: transparent;
    color: var(--text-secondary);
    border: none;
    border-radius: 0;
    font-size: 14px;
    font-weight: 500;
    font-family: var(--font-family);
    cursor: pointer;
    transition: var(--transition-base);
    white-space: nowrap;
    position: relative;
    border-bottom: 2px solid transparent;
}

.tab-btn:hover {
    color: var(--primary-color);
    background: var(--bg-light);
}

.tab-btn.active {
    color: var(--primary-color);
    background: var(--bg-light);
    border-bottom-color: var(--primary-color);
}

.tab-btn.active::after {
    content: none;
}

/* ===== TAB CONTENT ===== */
.tab-content {
    min-height: 400px;
}

.tab-pane {
    display: none;
}

.tab-pane.active {
    display: block;
}

/* ===== KPI CARDS ===== */
.kpi-cards-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
    gap: 1.5rem;
    margin-bottom: 2rem;
}

.kpi-card {
    background: white;
    border: 1px solid var(--border-light);
    border-radius: var(--radius-lg);
    padding: 1.5rem;
    box-shadow: var(--shadow-sm);
    transition: var(--transition-base);
    display: flex;
    align-items: center;
    gap: 1rem;
}

.kpi-card:hover {
    box-shadow: var(--shadow-md);
    transform: translateY(-2px);
    border-color: var(--primary-color);
}

.kpi-icon {
    width: 56px;
    height: 56px;
    background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%);
    border-radius: var(--radius-md);
    display: flex;
    align-items: center;
    justify-content: center;
    color: var(--bg-white);
    flex-shrink: 0;
}

.kpi-icon svg {
    stroke: white;
}

.kpi-details {
    flex: 1;
}

.kpi-title {
    font-size: 14px;
    font-weight: 500;
    color: var(--text-secondary);
    margin: 0 0 8px 0;
}

.kpi-value {
    font-size: 32px;
    font-weight: 700;
    color: var(--primary-color);
    margin: 0 0 4px 0;
    line-height: 1;
}

.kpi-subtitle {
    font-size: 12px;
    font-weight: 500;
    color: var(--text-secondary);
    margin: 0;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

/* ===== CHARTS SECTION ===== */
.charts-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
    gap: 1.5rem;
    margin-bottom: 2rem;
}

.chart-card {
    background: white;
    border: 1px solid var(--border-light);
    border-radius: var(--radius-lg);
    padding: 1.5rem;
    box-shadow: var(--shadow-sm);
    transition: var(--transition-base);
}

.chart-card:hover {
    box-shadow: var(--shadow-md);
    transform: translateY(-1px);
    border-color: var(--primary-color);
}

.chart-header {
    margin-bottom: 1rem;
    padding-bottom: 0.75rem;
    border-bottom: 1px solid var(--border-light);
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.chart-header h3 {
    font-size: 16px;
    font-weight: 600;
    color: var(--text-primary);
    margin: 0;
}

.chart-expand-btn {
    background: #f3f4f6;
    border: none;
    width: 36px;
    height: 36px;
    border-radius: 8px;
    display: flex;
    align-items: center;
    justify-content: center;
    cursor: pointer;
    transition: var(--transition-base);
}

.chart-expand-btn:hover {
    background: #e5e7eb;
}

.chart-expand-btn svg {
    stroke: #6b7280;
}

.chart-body {
    position: relative;
    height: 250px;
    display: flex;
    align-items: center;
    justify-content: center;
}

.chart-body canvas {
    max-width: 100%;
    max-height: 100%;
}

/* ===== UTILIZATION CARD ===== */
.utilization-card {
    background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%);
    border-radius: var(--radius-lg);
    padding: 2rem;
    color: white;
    margin-bottom: 2rem;
    box-shadow: var(--shadow-md);
}

.utilization-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 1.5rem;
}

.utilization-header h3 {
    font-size: 20px;
    font-weight: 600;
    margin: 0;
}

.utilization-percentage {
    font-size: 32px;
    font-weight: 700;
}

.utilization-bar {
    background: rgba(255, 255, 255, 0.2);
    height: 12px;
    border-radius: 6px;
    overflow: hidden;
    margin-bottom: 1rem;
}

.utilization-progress {
    height: 100%;
    background: white;
    border-radius: 6px;
    transition: width 0.5s ease;
}

.utilization-info {
    display: flex;
    justify-content: space-between;
    font-size: 14px;
    opacity: 0.9;
}

/* ===== CATEGORY/STATUS STATS GRID ===== */
.category-stats-grid,
.status-stats-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 1.5rem;
    margin-bottom: 2rem;
}

.stat-card {
    background: white;
    border: 1px solid var(--border-light);
    border-radius: var(--radius-lg);
    padding: 1.5rem;
    text-align: center;
    transition: var(--transition-base);
    box-shadow: var(--shadow-sm);
}

.stat-card:hover {
    border-color: var(--primary-color);
    box-shadow: var(--shadow-md);
    transform: translateY(-2px);
}

.stat-card h4 {
    font-size: 14px;
    color: var(--text-secondary);
    margin: 0 0 10px 0;
    font-weight: 500;
}

.stat-card .stat-value {
    font-size: 36px;
    font-weight: 700;
    color: var(--primary-color);
    margin: 0;
}

/* ===== DATA TABLES ===== */
.data-table-container {
    margin-bottom: 2rem;
    overflow-x: auto;
    border-radius: var(--radius-lg);
    border: 1px solid var(--border-light);
}

.data-table-container h3 {
    font-size: 18px;
    font-weight: 600;
    color: var(--text-primary);
    margin: 0 0 1rem 0;
}

.data-table {
    width: 100%;
    border-collapse: collapse;
    background: white;
    border-radius: var(--radius-lg);
    overflow: hidden;
}

.data-table thead {
    background: var(--primary-color);
    color: white;
}

.data-table th {
    padding: 1rem;
    text-align: left;
    font-size: 14px;
    font-weight: 600;
    font-family: var(--font-family);
}

.data-table tbody tr {
    border-bottom: 1px solid var(--border-light);
    transition: var(--transition-base);
}

.data-table tbody tr:hover {
    background: var(--bg-light);
}

.data-table tbody tr:last-child {
    border-bottom: none;
}

.data-table td {
    padding: 1rem;
    font-size: 14px;
    color: var(--text-secondary);
}

/* ===== STATUS BADGES ===== */
.status-badge {
    display: inline-block;
    padding: 6px 14px;
    border-radius: var(--radius-md);
    font-size: 12px;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.status-badge.available {
    background: #d1fae5;
    color: #065f46;
}

.status-badge.in-use {
    background: #fef3c7;
    color: #92400e;
}

.status-badge.maintenance,
.status-badge.in-maintenance,
.status-badge.maintenance-required {
    background: #fee2e2;
    color: #991b1b;
}

.status-badge.out-of-service {
    background: #e5e7eb;
    color: #374151;
}

/* ===== TABLE CONTROLS ===== */
.table-controls {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 1.5rem;
    gap: 1rem;
    flex-wrap: wrap;
}

.search-box {
    display: flex;
    align-items: center;
    gap: 10px;
    padding: 12px 16px;
    background: white;
    border: 1px solid var(--border-color);
    border-radius: var(--radius-md);
    flex: 1;
    min-width: 250px;
    transition: var(--transition-base);
}

.search-box:focus-within {
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgba(0, 85, 165, 0.1);
}

.search-box svg {
    stroke: var(--text-muted);
}

.search-box input {
    flex: 1;
    border: none;
    outline: none;
    font-size: 14px;
    color: var(--text-primary);
    font-family: var(--font-family);
}

.filter-controls {
    display: flex;
    gap: 12px;
}

.filter-select {
    padding: 12px 16px;
    border: 1px solid var(--border-color);
    border-radius: var(--radius-md);
    font-size: 14px;
    color: var(--text-secondary);
    background: white;
    cursor: pointer;
    outline: none;
    transition: var(--transition-base);
    font-family: var(--font-family);
}

.filter-select:hover,
.filter-select:focus {
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgba(0, 85, 165, 0.1);
}

/* ===== PAGINATION ===== */
.table-pagination {
    display: flex;
    justify-content: center;
    align-items: center;
    gap: 1rem;
    padding: 1.5rem 0;
}

.pagination-btn {
    padding: 10px 20px;
    border: 1px solid var(--border-color);
    background: white;
    color: var(--text-secondary);
    border-radius: var(--radius-md);
    cursor: pointer;
    font-size: 14px;
    font-weight: 500;
    font-family: var(--font-family);
    transition: var(--transition-base);
    box-shadow: var(--shadow-sm);
}

.pagination-btn:hover:not(:disabled) {
    border-color: var(--primary-color);
    color: var(--primary-color);
    transform: translateY(-1px);
}

.pagination-btn:disabled {
    opacity: 0.5;
    cursor: not-allowed;
}

.pagination-info {
    font-size: 14px;
    color: var(--text-secondary);
    font-weight: 500;
}

/* ===== RESPONSIVE DESIGN ===== */
@media (max-width: 992px) {
    .kpi-cards-grid {
        grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
        gap: 16px;
    }
    
    .charts-grid {
        grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
        gap: 16px;
    }
}

@media (max-width: 768px) {
    .header-content {
        padding: 0 15px;
    }
    
    .header-title {
        font-size: 24px;
    }
    
    .header-right {
        gap: 10px;
    }
    
    .create-btn {
        padding: 10px 16px;
        font-size: 14px;
    }
    
    .equipment-kpi-content {
        padding: 20px 15px;
    }
    
    .content-box {
        padding: 20px;
    }
    
    .time-period-selector {
        margin-bottom: 1.5rem;
    }
    
    .period-buttons {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(120px, 1fr));
        gap: 8px;
    }
    
    .period-btn {
        padding: 10px 12px;
        font-size: 13px;
        min-height: 40px;
        text-align: center;
    }
    
    .tab-navigation {
        flex-direction: column;
        gap: 0;
        border-bottom: 1px solid var(--border-color);
    }
    
    .tab-btn {
        justify-content: flex-start;
        padding: 12px 16px;
        font-size: 14px;
        min-height: 44px;
        border-bottom: 1px solid var(--border-color);
    }
    
    .kpi-cards-grid {
        grid-template-columns: 1fr;
        gap: 12px;
    }
    
    .kpi-card {
        padding: 16px;
    }
    
    .charts-grid {
        grid-template-columns: 1fr;
        gap: 12px;
    }
    
    .chart-card {
        padding: 16px;
    }
    
    .chart-body {
        height: 200px;
    }
    
    .table-controls {
        flex-direction: column;
        align-items: stretch;
    }

    .search-box {
        width: 100%;
    }

    .filter-controls {
        width: 100%;
        flex-direction: column;
    }

    .filter-select {
        width: 100%;
    }
}

@media (max-width: 576px) {
    .header-title {
        font-size: 20px;
    }
    
    .header-right {
        flex-direction: column;
        gap: 8px;
        align-items: stretch;
    }
    
    .create-btn {
        width: 100%;
        justify-content: center;
        padding: 10px 12px;
        font-size: 13px;
    }
    
    .content-box {
        padding: 14px;
    }
    
    .period-buttons {
        grid-template-columns: repeat(auto-fit, minmax(100px, 1fr));
        gap: 6px;
    }
    
    .period-btn {
        padding: 8px 10px;
        font-size: 12px;
        min-height: 36px;
    }
    
    .custom-date-range {
        padding: 0.875rem;
    }
    
    .date-range-inputs {
        flex-direction: column;
        gap: 0.75rem;
    }
    
    .date-input-group {
        width: 100%;
    }
    
    .date-input {
        width: 100%;
        min-width: auto;
    }
    
    .btn-apply-dates,
    .btn-clear-dates {
        width: 100%;
        justify-content: center;
    }
}

@media (max-width: 480px) {
    .header-content {
        flex-direction: column;
        align-items: flex-start;
        gap: 15px;
    }
    
    .header-title {
        font-size: 18px;
    }
    
    .back-button {
        font-size: 18px;
    }
    
    .header-right {
        width: 100%;
    }
    
    .create-btn {
        font-size: 12px;
        padding: 9px 12px;
    }
    
    .content-box {
        padding: 12px;
    }
    
    .period-buttons {
        grid-template-columns: 1fr;
        gap: 6px;
    }
    
    .custom-date-range {
        padding: 0.75rem;
    }
    
    .date-input {
        padding: 9px;
        font-size: 13px;
    }
    
    .btn-apply-dates,
    .btn-clear-dates {
        padding: 9px 12px;
        font-size: 12px;
    }
}

/* ===== ACCESSIBILITY ===== */
@media (prefers-reduced-motion: reduce) {
    * {
        animation-duration: 0.01ms !important;
        animation-iteration-count: 1 !important;
        transition-duration: 0.01ms !important;
    }
}

/* ===== FOCUS STYLES ===== */
button:focus-visible,
input:focus-visible,
select:focus-visible {
    outline: 2px solid #3b82f6;
    outline-offset: 2px;
}

/* ===== PRINT STYLES ===== */
@media print {
    .header-gradient,
    .period-buttons,
    .tab-navigation {
        display: none !important;
    }
    
    .equipment-kpi-container {
        background: white;
    }
    
    .kpi-card,
    .chart-card {
        border: 1px solid #000;
        break-inside: avoid;
    }
}
</style>

<script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.0/dist/chart.umd.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/xlsx@0.18.5/dist/xlsx.full.min.js"></script>
<script>
let equipmentData = [];
let allEquipmentData = []; // Store all data without time filter
let filteredData = [];
let currentPage = 1;
const itemsPerPage = 10;
let currentTimePeriod = '7days'; // Default time period
let customStartDate = null;
let customEndDate = null;

// MongoDB API Configuration
const MONGODB_EQUIPMENT_API = 'http://168.231.113.219:5001';

// Initialize on page load
document.addEventListener('DOMContentLoaded', function() {
    initializeDashboard();
});

async function initializeDashboard() {
    showLoading();
    await fetchEquipmentData();
    hideLoading();
    updateDashboard();
    setupEventListeners();
}

function showLoading() {
    document.getElementById('loading-overlay').style.display = 'flex';
}

function hideLoading() {
    document.getElementById('loading-overlay').style.display = 'none';
}

// Fetch equipment data from MongoDB
async function fetchEquipmentData() {
    try {
        console.log('🔄 Fetching equipment data from MongoDB...');
        console.log('🔗 API URL:', `${MONGODB_EQUIPMENT_API}/EquipmentsDataJSON/GlobalEquipmentJsonData`);
        
        const response = await fetch(`${MONGODB_EQUIPMENT_API}/EquipmentsDataJSON/GlobalEquipmentJsonData`, {
            method: 'GET',
            headers: {
                'Content-Type': 'application/json',
                'Accept': 'application/json'
            }
        });
        
        console.log('📡 Response status:', response.status);
        
        if (!response.ok) {
            const errorText = await response.text();
            console.error('❌ Response error:', errorText);
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        
        const data = await response.json();
        console.log('📥 MongoDB Equipment response:', data);
        console.log('📥 Response structure:', {
            hasSuccess: !!data.success,
            hasData: !!data.data,
            dataType: Array.isArray(data.data) ? 'array' : typeof data.data,
            count: data.count
        });
        
        if (!data) {
            console.warn('⚠️ No equipment data found');
            allEquipmentData = [];
            equipmentData = [];
            return;
        }
        
        // MongoDB returns {success: true, data: [...], count: 670}
        // Extract the data array
        let rawData = data;
        if (data.success && data.data) {
            rawData = data.data;
            console.log('✅ Extracted data array from response');
        }
        
        console.log('📦 Processing equipment data...', {
            isArray: Array.isArray(rawData),
            length: Array.isArray(rawData) ? rawData.length : 'N/A',
            type: typeof rawData
        });
        
        // MongoDB returns data as ARRAY, not object!
        // Check if it's an array or object and handle accordingly
        let tempData = [];
        
        if (Array.isArray(rawData)) {
            console.log('✅ Processing as ARRAY (MongoDB format)');
            // MongoDB format: data is an array
            tempData = rawData.map((value, index) => {
                if (typeof value !== 'object' || value === null) {
                    return null;
                }
                
                return {
                    firebaseKey: value.firebaseKey || value._id || value.id,
                    id: value.id || value._id,
                    _id: value._id || value.id,
                    index: value.index || index,
                    equipmentName: value.equipmentName || value.vehicleNumber || value.name || 'Unknown',
                    vehicleNumber: value.vehicleNumber || value.registrationNumber || value.equipmentNumber || value.id,
                    registrationNumber: value.registrationNumber || value.vehicleNumber || 'N/A',
                    model: value.model || 'N/A',
                    make: value.make || 'N/A',
                    category: value.category || value.vehicleType || value.mainCategory || value.equipmentCategory || 'Uncategorized',
                    vehicleType: value.vehicleType || value.category || value.mainCategory || 'Uncategorized',
                    mainCategory: value.mainCategory || value.category || 'Uncategorized',
                    status: value.status || 'Available',
                    meterReading: value.meterReading || value.odometerReading || 0,
                    odometerReading: value.odometerReading || value.meterReading || 0,
                    meterType: value.meterType || 'KM',
                    engineNumber: value.engineNumber || 'N/A',
                    vinNumber: value.vinNumber || value.chassisNumber || 'N/A',
                    country: value.country || 'N/A',
                    project: value.project || 'N/A',
                    comment: value.comment || value.notes || '',
                    isActive: value.isActive !== undefined ? value.isActive : true,
                    createdBy: value.createdBy || value.adminUid || '',
                    timestamp: value.timestamp || value.createdAt || value.uploadTimestamp || value.uploadedAt || null,
                    createdAt: value.createdAt || value.timestamp || value.uploadTimestamp || value.uploadedAt || null,
                    updatedAt: value.updatedAt || value.timestamp || value.createdAt || null
                };
            }).filter(item => item !== null);
        } else if (typeof rawData === 'object') {
            console.log('⚠️ Processing as OBJECT (Legacy Firebase format)');
            // Legacy Firebase format: data is an object with keys
            tempData = Object.entries(rawData).map(([key, value], index) => {
                if (typeof value !== 'object' || value === null) {
                    return null;
                }
                
                return {
                    firebaseKey: value.firebaseKey || value._id || key,
                    id: value.id || value._id || key,
                    _id: value._id || value.id || key,
                    index: value.index || index,
                    equipmentName: value.equipmentName || value.vehicleNumber || value.name || 'Unknown',
                    vehicleNumber: value.vehicleNumber || value.registrationNumber || value.equipmentNumber || value.id || key,
                    registrationNumber: value.registrationNumber || value.vehicleNumber || 'N/A',
                    model: value.model || 'N/A',
                    make: value.make || 'N/A',
                    category: value.category || value.vehicleType || value.mainCategory || value.equipmentCategory || 'Uncategorized',
                    vehicleType: value.vehicleType || value.category || value.mainCategory || 'Uncategorized',
                    mainCategory: value.mainCategory || value.category || 'Uncategorized',
                    status: value.status || 'Available',
                    meterReading: value.meterReading || value.odometerReading || 0,
                    odometerReading: value.odometerReading || value.meterReading || 0,
                    meterType: value.meterType || 'KM',
                    engineNumber: value.engineNumber || 'N/A',
                    vinNumber: value.vinNumber || value.chassisNumber || 'N/A',
                    country: value.country || 'N/A',
                    project: value.project || 'N/A',
                    comment: value.comment || value.notes || '',
                    isActive: value.isActive !== undefined ? value.isActive : true,
                    createdBy: value.createdBy || value.adminUid || '',
                    timestamp: value.timestamp || value.createdAt || value.uploadTimestamp || null,
                    createdAt: value.createdAt || value.timestamp || value.uploadTimestamp || null,
                    updatedAt: value.updatedAt || value.timestamp || value.createdAt || null
                };
            }).filter(item => item !== null);
        } else {
            console.error('❌ Unknown data format:', typeof rawData);
            tempData = [];
        }
        
        // Show ALL equipment - don't filter by isActive
        allEquipmentData = tempData;
        
        console.log('📊 Equipment data after processing:', {
            total: allEquipmentData.length,
            active: allEquipmentData.filter(e => e.isActive).length,
            inactive: allEquipmentData.filter(e => !e.isActive).length
        });
        
        console.log('📊 Total equipment records:', allEquipmentData.length);
        
        // Log first 3 samples for debugging
        if (allEquipmentData.length > 0) {
            allEquipmentData.slice(0, 3).forEach((eq, idx) => {
                console.log(`📄 Sample equipment ${idx + 1}:`, {
                    id: eq.id,
                    equipmentName: eq.equipmentName,
                    vehicleNumber: eq.vehicleNumber,
                    category: eq.category,
                    status: eq.status,
                    country: eq.country,
                    project: eq.project
                });
            });
        }
        
        // Apply initial time filter
        applyTimeFilter(currentTimePeriod);
        
        console.log('✅ Equipment data loaded:', allEquipmentData.length, 'total items');
        console.log('📊 Filtered data:', equipmentData.length, 'items for', currentTimePeriod);
        
        if (allEquipmentData.length === 0) {
            showNotification('No equipment data found', 'info');
        } else {
            showNotification(`✅ Loaded ${allEquipmentData.length} equipment record(s) from MongoDB`, 'success');
        }
        
    } catch (error) {
        console.error('❌ Error fetching equipment data from MongoDB:', error);
        showNotification('Failed to load equipment data. Please try again later.', 'error');
        allEquipmentData = [];
        equipmentData = [];
        filteredData = [];
    }
}

// Apply time period filter
function applyTimeFilter(period) {
    console.log(`🔍 Applying time filter: ${period}`);
    console.log(`📊 Total equipment before filter: ${allEquipmentData.length}`);
    
    // FOR EQUIPMENT: Show ALL equipment regardless of date
    // Equipment doesn't have meaningful creation dates for filtering
    // We want to show all registered equipment in the system
    
    equipmentData = [...allEquipmentData];
    filteredData = [...equipmentData];
    
    console.log(`✅ Equipment filter applied: ${period}`);
    console.log(`📊 Showing ALL equipment: ${equipmentData.length} items`);
    console.log(`💡 Note: Equipment KPI shows all registered equipment regardless of date filter`);
}

// Update entire dashboard
function updateDashboard() {
    console.log('📊 Updating Equipment dashboard...');
    console.log('📊 Equipment data count:', equipmentData.length);
    
    updateOverviewTab();
    updateCategoryTab();
    updateStatusTab();
    updateDetailsTab();
    
    console.log('✅ Equipment dashboard updated');
}

// Update Overview Tab
function updateOverviewTab() {
    console.log('📊 Updating overview tab with', equipmentData.length, 'equipment');
    
    const total = equipmentData.length;
    
    // Count by status (case-insensitive for MongoDB compatibility)
    const available = equipmentData.filter(e => {
        const status = (e.status || '').toLowerCase();
        return status === 'available' || status === 'active' || status === 'operational';
    }).length;
    
    const inUse = equipmentData.filter(e => {
        const status = (e.status || '').toLowerCase();
        return status === 'in use' || status === 'in-use' || status === 'active';
    }).length;
    
    const maintenance = equipmentData.filter(e => {
        const status = (e.status || '').toLowerCase();
        return status === 'maintenance' || 
               status === 'in maintenance' || 
               status === 'maintenance required' ||
               status === 'under maintenance';
    }).length;
    
    console.log('📊 Equipment counts:', {
        total,
        available,
        inUse,
        maintenance
    });
    
    // Update KPI cards with null checks
    const totalEl = document.getElementById('total-equipment-count');
    const availableEl = document.getElementById('available-equipment-count');
    const inUseEl = document.getElementById('in-use-equipment-count');
    const maintenanceEl = document.getElementById('maintenance-equipment-count');
    
    if (totalEl) totalEl.textContent = total;
    if (availableEl) availableEl.textContent = available;
    if (inUseEl) inUseEl.textContent = inUse;
    if (maintenanceEl) maintenanceEl.textContent = maintenance;
    
    // Update utilization
    const utilizationRate = total > 0 ? ((inUse / total) * 100).toFixed(1) : 0;
    const utilizationRateEl = document.getElementById('utilization-rate');
    const utilizationProgressEl = document.getElementById('utilization-progress');
    const inUseCountEl = document.getElementById('in-use-count');
    const totalCountEl = document.getElementById('total-count');
    
    if (utilizationRateEl) utilizationRateEl.textContent = utilizationRate + '%';
    if (utilizationProgressEl) utilizationProgressEl.style.width = utilizationRate + '%';
    if (inUseCountEl) inUseCountEl.textContent = inUse;
    if (totalCountEl) totalCountEl.textContent = total;
    
    console.log('📊 Utilization rate:', utilizationRate + '%');
    
    // Update charts
    updateCategoryChart();
    updateStatusChart();
}

// Update Category Chart
function updateCategoryChart() {
    const categoryCount = {};
    equipmentData.forEach(e => {
        const category = e.category || e.mainCategory || e.vehicleType || 'Uncategorized';
        categoryCount[category] = (categoryCount[category] || 0) + 1;
    });
    
    console.log('📊 Category counts:', categoryCount);
    
    const ctx = document.getElementById('categoryChart');
    
    if (window.categoryChartInstance) {
        window.categoryChartInstance.destroy();
    }
    
    window.categoryChartInstance = new Chart(ctx, {
        type: 'bar',
        data: {
            labels: Object.keys(categoryCount),
            datasets: [{
                label: 'Equipment Count',
                data: Object.values(categoryCount),
                backgroundColor: 'rgba(102, 126, 234, 0.8)',
                borderColor: 'rgba(102, 126, 234, 1)',
                borderWidth: 2,
                borderRadius: 8
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: {
                    display: false
                }
            },
            scales: {
                y: {
                    beginAtZero: true,
                    ticks: {
                        stepSize: 1
                    }
                }
            }
        }
    });
}

// Update Status Chart
function updateStatusChart() {
    const statusCount = {};
    equipmentData.forEach(e => {
        const status = e.status || 'Unknown';
        statusCount[status] = (statusCount[status] || 0) + 1;
    });
    
    console.log('📊 Status counts:', statusCount);
    
    const ctx = document.getElementById('statusChart');
    if (!ctx) {
        console.warn('⚠️ Status chart canvas not found');
        return;
    }
    
    if (window.statusChartInstance) {
        window.statusChartInstance.destroy();
    }
    
    const colors = {
        'Available': '#10b981',
        'Active': '#06b6d4',
        'In Use': '#f59e0b',
        'Maintenance': '#ef4444',
        'In Maintenance': '#ef4444',
        'Maintenance Required': '#ef4444',
        'Under Maintenance': '#ef4444',
        'Out of Service': '#6b7280',
        'Operational': '#10b981'
    };
    
    window.statusChartInstance = new Chart(ctx, {
        type: 'doughnut',
        data: {
            labels: Object.keys(statusCount),
            datasets: [{
                data: Object.values(statusCount),
                backgroundColor: Object.keys(statusCount).map(status => colors[status] || '#667eea'),
                borderWidth: 2,
                borderColor: '#ffffff'
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: {
                    position: 'bottom'
                }
            }
        }
    });
}

// Update Category Tab
function updateCategoryTab() {
    const categoryStats = {};
    
    equipmentData.forEach(e => {
        const category = e.category || e.mainCategory || e.vehicleType || 'Uncategorized';
        if (!categoryStats[category]) {
            categoryStats[category] = {
                total: 0,
                available: 0,
                inUse: 0,
                maintenance: 0
            };
        }
        
        categoryStats[category].total++;
        
        if (e.status === 'Available') categoryStats[category].available++;
        else if (e.status === 'In Use') categoryStats[category].inUse++;
        else if (e.status === 'Maintenance' || e.status === 'In Maintenance' || e.status === 'Maintenance Required') categoryStats[category].maintenance++;
    });
    
    // Update stats grid
    const statsGrid = document.getElementById('category-stats-grid');
    statsGrid.innerHTML = '';
    
    Object.entries(categoryStats).forEach(([category, stats]) => {
        const card = document.createElement('div');
        card.className = 'stat-card';
        card.innerHTML = `
            <h4>${category}</h4>
            <p class="stat-value">${stats.total}</p>
        `;
        statsGrid.appendChild(card);
    });
    
    // Update table
    const tableBody = document.getElementById('category-table-body');
    tableBody.innerHTML = '';
    
    const total = equipmentData.length;
    
    Object.entries(categoryStats).forEach(([category, stats]) => {
        const percentage = total > 0 ? ((stats.total / total) * 100).toFixed(1) : 0;
        const row = document.createElement('tr');
        row.innerHTML = `
            <td><strong>${category}</strong></td>
            <td>${stats.total}</td>
            <td>${stats.available}</td>
            <td>${stats.inUse}</td>
            <td>${stats.maintenance}</td>
            <td>${percentage}%</td>
        `;
        tableBody.appendChild(row);
    });
    
    // Update category distribution chart
    updateCategoryDistributionChart(categoryStats);
    
    // Update category filter dropdown
    const categoryFilter = document.getElementById('category-filter');
    categoryFilter.innerHTML = '<option value="">All Categories</option>';
    Object.keys(categoryStats).forEach(category => {
        const option = document.createElement('option');
        option.value = category;
        option.textContent = category;
        categoryFilter.appendChild(option);
    });
}

// Update Category Distribution Chart
function updateCategoryDistributionChart(categoryStats) {
    const ctx = document.getElementById('categoryDistributionChart');
    
    if (window.categoryDistributionChartInstance) {
        window.categoryDistributionChartInstance.destroy();
    }
    
    window.categoryDistributionChartInstance = new Chart(ctx, {
        type: 'pie',
        data: {
            labels: Object.keys(categoryStats),
            datasets: [{
                data: Object.values(categoryStats).map(s => s.total),
                backgroundColor: [
                    '#667eea', '#f59e0b', '#10b981', '#ef4444', 
                    '#8b5cf6', '#ec4899', '#06b6d4', '#84cc16'
                ],
                borderWidth: 2,
                borderColor: '#ffffff'
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: {
                    position: 'bottom'
                }
            }
        }
    });
}

// Update Status Tab
function updateStatusTab() {
    const statusStats = {};
    
    equipmentData.forEach(e => {
        const status = e.status || 'Unknown';
        statusStats[status] = (statusStats[status] || 0) + 1;
    });
    
    const statsGrid = document.getElementById('status-stats-grid');
    statsGrid.innerHTML = '';
    
    const statusColors = {
        'Available': '#10b981',
        'In Use': '#f59e0b',
        'Maintenance': '#ef4444',
        'In Maintenance': '#ef4444',
        'Maintenance Required': '#ef4444',
        'Out of Service': '#6b7280'
    };
    
    Object.entries(statusStats).forEach(([status, count]) => {
        const card = document.createElement('div');
        card.className = 'stat-card';
        card.style.borderColor = statusColors[status] || '#667eea';
        card.innerHTML = `
            <h4>${status}</h4>
            <p class="stat-value" style="color: ${statusColors[status] || '#667eea'}">${count}</p>
        `;
        statsGrid.appendChild(card);
    });
    
    // Update status timeline chart
    updateStatusTimelineChart();
}

// Update Status Timeline Chart (placeholder for now)
function updateStatusTimelineChart() {
    const ctx = document.getElementById('statusTimelineChart');
    
    if (window.statusTimelineChartInstance) {
        window.statusTimelineChartInstance.destroy();
    }
    
    // For now, showing current status distribution
    const statusCount = {};
    equipmentData.forEach(e => {
        const status = e.status || 'Unknown';
        statusCount[status] = (statusCount[status] || 0) + 1;
    });
    
    window.statusTimelineChartInstance = new Chart(ctx, {
        type: 'bar',
        data: {
            labels: Object.keys(statusCount),
            datasets: [{
                label: 'Current Status Distribution',
                data: Object.values(statusCount),
                backgroundColor: ['#10b981', '#f59e0b', '#ef4444', '#6b7280'],
                borderWidth: 2,
                borderRadius: 8
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            scales: {
                y: {
                    beginAtZero: true,
                    ticks: {
                        stepSize: 1
                    }
                }
            }
        }
    });
}

// Update Details Tab
function updateDetailsTab() {
    const tbody = document.getElementById('equipment-details-body');
    tbody.innerHTML = '';
    
    const start = (currentPage - 1) * itemsPerPage;
    const end = start + itemsPerPage;
    const paginatedData = filteredData.slice(start, end);
    
    paginatedData.forEach(equipment => {
        const row = document.createElement('tr');
        
        const statusClass = equipment.status.toLowerCase().replace(/\s+/g, '-');
        const updatedDate = equipment.updatedAt ? new Date(equipment.updatedAt).toLocaleDateString() : 'N/A';
        
        const displayName = equipment.equipmentName || equipment.make + ' ' + equipment.model;
        const meterValue = equipment.meterReading && equipment.meterReading < 1e100 ? 
            equipment.meterReading.toFixed(0) : '0';
        
        row.innerHTML = `
            <td><strong>${equipment.vehicleNumber}</strong></td>
            <td>${displayName}</td>
            <td>${equipment.registrationNumber}</td>
            <td>${equipment.make} ${equipment.model}</td>
            <td>${equipment.category}</td>
            <td><span class="status-badge ${statusClass}">${equipment.status}</span></td>
            <td>${meterValue} ${equipment.meterType}</td>
            <td>${updatedDate}</td>
        `;
        
        tbody.appendChild(row);
    });
    
    updatePagination();
}

// Update Pagination
function updatePagination() {
    const totalPages = Math.ceil(filteredData.length / itemsPerPage);
    
    document.getElementById('current-page').textContent = currentPage;
    document.getElementById('total-pages').textContent = totalPages;
    
    document.getElementById('prev-page').disabled = currentPage === 1;
    document.getElementById('next-page').disabled = currentPage === totalPages || totalPages === 0;
}

// Setup Event Listeners
function setupEventListeners() {
    const customDateRange = document.getElementById('custom-date-range');
    const startDateInput = document.getElementById('start-date');
    const endDateInput = document.getElementById('end-date');
    const applyDatesBtn = document.getElementById('apply-custom-dates');
    const clearDatesBtn = document.getElementById('clear-custom-dates');
    
    // Set max date to today for date inputs
    const today = new Date().toISOString().split('T')[0];
    if (startDateInput) startDateInput.max = today;
    if (endDateInput) endDateInput.max = today;
    
    // When start date changes, update end date minimum
    if (startDateInput && endDateInput) {
        startDateInput.addEventListener('change', () => {
            if (startDateInput.value) {
                endDateInput.min = startDateInput.value;
                // If end date is before start date, clear it
                if (endDateInput.value && new Date(endDateInput.value) < new Date(startDateInput.value)) {
                    endDateInput.value = '';
                }
            }
        });
    }
    
    // Apply custom dates
    if (applyDatesBtn) {
        applyDatesBtn.addEventListener('click', () => {
            const startDate = startDateInput.value;
            const endDate = endDateInput.value;
            
            if (!startDate || !endDate) {
                showNotification('Please select both start and end dates', 'error');
                return;
            }
            
            if (new Date(startDate) > new Date(endDate)) {
                showNotification('Start date cannot be after end date', 'error');
                return;
            }
            
            customStartDate = startDate;
            customEndDate = endDate;
            currentTimePeriod = 'custom';
            
            // Apply filter
            applyTimeFilter('custom');
            currentPage = 1;
            updateDashboard();
            
            showNotification(`Showing data from ${new Date(startDate).toLocaleDateString()} to ${new Date(endDate).toLocaleDateString()}`, 'success');
        });
    }
    
    // Clear custom dates
    if (clearDatesBtn) {
        clearDatesBtn.addEventListener('click', () => {
            startDateInput.value = '';
            endDateInput.value = '';
            customStartDate = null;
            customEndDate = null;
            
            // Switch back to default (7 days)
            currentTimePeriod = '7days';
            document.querySelectorAll('.period-btn').forEach(btn => {
                btn.classList.remove('active');
                if (btn.getAttribute('data-period') === '7days') {
                    btn.classList.add('active');
                }
            });
            
            // Hide custom date range picker
            if (customDateRange) {
                customDateRange.style.display = 'none';
            }
            
            applyTimeFilter('7days');
            currentPage = 1;
            updateDashboard();
            showNotification('Custom date range cleared', 'info');
        });
    }
    
    // Time Period Buttons - NOW CLICKABLE AND FUNCTIONAL
    document.querySelectorAll('.period-btn').forEach(btn => {
        btn.addEventListener('click', function() {
            const period = this.getAttribute('data-period');
            
            console.log('📅 Period button clicked:', period);
            
            // Update active state - VISUAL FEEDBACK
            document.querySelectorAll('.period-btn').forEach(b => b.classList.remove('active'));
            this.classList.add('active');
            
            // Show/hide custom date range picker
            if (period === 'custom') {
                if (customDateRange) {
                    customDateRange.style.display = 'block';
                }
                
                // Set default dates if not already set
                if (!startDateInput.value || !endDateInput.value) {
                    // Default to last 7 days
                    const end = new Date();
                    const start = new Date();
                    start.setDate(end.getDate() - 7);
                    
                    startDateInput.value = start.toISOString().split('T')[0];
                    endDateInput.value = end.toISOString().split('T')[0];
                }
                
                currentTimePeriod = 'custom';
                showNotification('Please select custom date range and click Apply', 'info');
                // Don't update dashboard yet, wait for user to apply dates
                return;
            } else {
                // Hide custom date range picker for other periods
                if (customDateRange) {
                    customDateRange.style.display = 'none';
                }
                
                // Apply time filter
                currentTimePeriod = period;
                applyTimeFilter(period);
                currentPage = 1; // Reset to first page
                updateDashboard();
                
                // Show notification with proper count
                const periodText = period === 'today' ? 'Today' : 
                                  period === '7days' ? 'Last 7 Days' :
                                  period === '30days' ? 'Last 30 Days' :
                                  period === '90days' ? 'Last 90 Days' : period;
                showNotification(`✅ Showing ${equipmentData.length} equipment records (${periodText})`, 'success');
            }
        });
    });
    
    // Tab Navigation
    document.querySelectorAll('.tab-btn').forEach(btn => {
        btn.addEventListener('click', function() {
            const targetTab = this.getAttribute('data-tab');
            
            document.querySelectorAll('.tab-btn').forEach(b => b.classList.remove('active'));
            document.querySelectorAll('.tab-pane').forEach(p => p.classList.remove('active'));
            
            this.classList.add('active');
            document.getElementById(targetTab).classList.add('active');
        });
    });
    
    // Search
    document.getElementById('equipment-search').addEventListener('input', function(e) {
        const searchTerm = e.target.value.toLowerCase();
        filterData();
    });
    
    // Filters
    document.getElementById('category-filter').addEventListener('change', filterData);
    document.getElementById('status-filter').addEventListener('change', filterData);
    
    // Pagination
    document.getElementById('prev-page').addEventListener('click', () => {
        if (currentPage > 1) {
            currentPage--;
            updateDetailsTab();
        }
    });
    
    document.getElementById('next-page').addEventListener('click', () => {
        const totalPages = Math.ceil(filteredData.length / itemsPerPage);
        if (currentPage < totalPages) {
            currentPage++;
            updateDetailsTab();
        }
    });
    
    // Refresh Button
    document.getElementById('refreshDataBtn').addEventListener('click', async () => {
        showNotification('Refreshing equipment data...', 'info');
        showLoading();
        await fetchEquipmentData();
        hideLoading();
        updateDashboard();
    });
    
    // Export Button
    document.getElementById('exportExcelBtn').addEventListener('click', exportToExcel);
}

// Filter Data
function filterData() {
    const searchTerm = document.getElementById('equipment-search').value.toLowerCase();
    const categoryFilter = document.getElementById('category-filter').value;
    const statusFilter = document.getElementById('status-filter').value;
    
    filteredData = equipmentData.filter(equipment => {
        const matchesSearch = 
            equipment.equipmentName.toLowerCase().includes(searchTerm) ||
            equipment.vehicleNumber.toLowerCase().includes(searchTerm) ||
            equipment.model.toLowerCase().includes(searchTerm) ||
            equipment.id.toLowerCase().includes(searchTerm);
        
        const matchesCategory = !categoryFilter || equipment.category === categoryFilter;
        const matchesStatus = !statusFilter || equipment.status === statusFilter;
        
        return matchesSearch && matchesCategory && matchesStatus;
    });
    
    currentPage = 1;
    updateDetailsTab();
}

// Export to Excel
async function exportToExcel() {
    try {
        console.log('📊 Exporting equipment data to Excel...');
        
        if (equipmentData.length === 0) {
            showNotification('No data to export', 'error');
            return;
        }
        
        showNotification('Preparing Excel export...', 'info');
        
        // Prepare equipment data
        const excelData = equipmentData.map((equipment, index) => {
            const updatedDate = equipment.updatedAt ? new Date(equipment.updatedAt).toLocaleString('en-GB', {
                day: '2-digit',
                month: '2-digit',
                year: 'numeric',
                hour: '2-digit',
                minute: '2-digit'
            }) : 'N/A';
            
            const createdDate = equipment.timestamp || equipment.createdAt ? 
                new Date(equipment.timestamp || equipment.createdAt).toLocaleString('en-GB', {
                    day: '2-digit',
                    month: '2-digit',
                    year: 'numeric',
                    hour: '2-digit',
                    minute: '2-digit'
                }) : 'N/A';
            
            const displayName = equipment.equipmentName || `${equipment.make} ${equipment.model}`;
            const meterValue = equipment.meterReading && equipment.meterReading < 1e100 ? 
                equipment.meterReading.toFixed(0) : '0';
            
            return {
                'S.No': index + 1,
                'ID': equipment.id || equipment._id || 'N/A',
                'Vehicle Number': equipment.vehicleNumber || 'N/A',
                'Equipment Name': displayName,
                'Registration Number': equipment.registrationNumber || 'N/A',
                'Make': equipment.make || 'N/A',
                'Model': equipment.model || 'N/A',
                'Category': equipment.category || equipment.mainCategory || 'N/A',
                'Vehicle Type': equipment.vehicleType || 'N/A',
                'Status': equipment.status || 'N/A',
                'Meter Reading': `${meterValue} ${equipment.meterType || 'KM'}`,
                'Meter Type': equipment.meterType || 'KM',
                'Engine Number': equipment.engineNumber || 'N/A',
                'VIN Number': equipment.vinNumber || 'N/A',
                'Country': equipment.country || 'N/A',
                'Project': equipment.project || 'N/A',
                'Is Active': equipment.isActive ? 'Yes' : 'No',
                'Comment': equipment.comment || 'N/A',
                'Created By': equipment.createdBy || 'N/A',
                'Created At': createdDate,
                'Last Updated': updatedDate
            };
        });
    
    // Create workbook
    const wb = XLSX.utils.book_new();
    const ws = XLSX.utils.json_to_sheet(excelData);
    
        // Set column widths for better readability
        const colWidths = [
            { wch: 6 },   // S.No
            { wch: 15 },  // ID
            { wch: 18 },  // Vehicle Number
            { wch: 25 },  // Equipment Name
            { wch: 18 },  // Registration Number
            { wch: 15 },  // Make
            { wch: 15 },  // Model
            { wch: 18 },  // Category
            { wch: 18 },  // Vehicle Type
            { wch: 20 },  // Status
            { wch: 15 },  // Meter Reading
            { wch: 12 },  // Meter Type
            { wch: 20 },  // Engine Number
            { wch: 20 },  // VIN Number
            { wch: 12 },  // Country
            { wch: 15 },  // Project
            { wch: 10 },  // Is Active
            { wch: 30 },  // Comment
            { wch: 20 },  // Created By
            { wch: 20 },  // Created At
            { wch: 20 }   // Last Updated
        ];
        ws['!cols'] = colWidths;
        
        // Add equipment data sheet
        XLSX.utils.book_append_sheet(wb, ws, 'Equipment Data');
        
        // Create summary sheet (case-insensitive status matching)
        const total = equipmentData.length;
        const available = equipmentData.filter(e => {
            const status = (e.status || '').toLowerCase();
            return status === 'available' || status === 'active' || status === 'operational';
        }).length;
        const inUse = equipmentData.filter(e => {
            const status = (e.status || '').toLowerCase();
            return status === 'in use' || status === 'in-use' || status === 'active';
        }).length;
        const maintenance = equipmentData.filter(e => {
            const status = (e.status || '').toLowerCase();
            return status === 'maintenance' || 
                   status === 'in maintenance' || 
                   status === 'maintenance required' ||
                   status === 'under maintenance';
        }).length;
        
        const summaryData = [
            { 'Metric': 'Total Equipment', 'Value': total },
            { 'Metric': 'Available', 'Value': available },
            { 'Metric': 'In Use', 'Value': inUse },
            { 'Metric': 'In Maintenance', 'Value': maintenance },
            { 'Metric': 'Utilization Rate', 'Value': total > 0 ? `${((inUse / total) * 100).toFixed(1)}%` : '0%' },
            { 'Metric': 'Date Range', 'Value': getCurrentDateRangeText() },
            { 'Metric': 'Export Date', 'Value': new Date().toLocaleString('en-GB', {
                day: '2-digit',
                month: '2-digit',
                year: 'numeric',
                hour: '2-digit',
                minute: '2-digit'
            }) }
        ];
        
        const wsSummary = XLSX.utils.json_to_sheet(summaryData);
        wsSummary['!cols'] = [{ wch: 25 }, { wch: 30 }];
        XLSX.utils.book_append_sheet(wb, wsSummary, 'Summary');
        
        // Generate filename with current date and time range
        const dateRangeText = getCurrentDateRangeText().replace(/\s+/g, '_');
        const timestamp = new Date().toISOString().slice(0, 10);
        const filename = `Equipment_KPI_${dateRangeText}_${timestamp}.xlsx`;
        
        // Export file
        XLSX.writeFile(wb, filename);
        
        showNotification(`✅ Exported ${total} equipment record(s) to Excel successfully!`, 'success');
        console.log('✅ Excel export completed:', filename);
    } catch (error) {
        console.error('❌ Error exporting to Excel:', error);
        showNotification('Error exporting to Excel: ' + error.message, 'error');
    }
}

// Get current date range as text for filename
function getCurrentDateRangeText() {
    if (currentTimePeriod === 'today') {
        return 'Today';
    } else if (currentTimePeriod === 'custom') {
        if (customStartDate && customEndDate) {
            const start = new Date(customStartDate).toLocaleDateString('en-GB').replace(/\//g, '-');
            const end = new Date(customEndDate).toLocaleDateString('en-GB').replace(/\//g, '-');
            return `Custom_${start}_to_${end}`;
        }
        return 'All_Data';
    } else {
        return `Last_${currentTimePeriod.replace('days', '')}_Days`;
    }
}

// Notification system for user feedback
function showNotification(message, type = 'info') {
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.innerHTML = `
        <div class="notification-content">
            <span>${message}</span>
            <button class="notification-close" aria-label="Close notification">&times;</button>
        </div>
    `;
    
    document.body.appendChild(notification);
    
    // Auto remove after 5 seconds
    setTimeout(() => removeNotification(notification), 5000);
    
    // Close button functionality
    notification.querySelector('.notification-close').addEventListener('click', () => {
        removeNotification(notification);
    });
}

function removeNotification(notification) {
    if (notification && notification.parentNode) {
        notification.parentNode.removeChild(notification);
    }
}

// Add notification styles
const notificationStyles = document.createElement('style');
notificationStyles.textContent = `
    .notification {
        position: fixed;
        top: 20px;
        right: 20px;
        z-index: var(--z-index-notification);
        padding: 16px;
        border-radius: 8px;
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
        font-family: var(--font-family);
        font-weight: 500;
        max-width: 400px;
        min-width: 300px;
        animation: slideIn 0.3s ease;
        background: white;
    }
    
    .notification-success {
        border-left: 4px solid #28a745;
    }
    
    .notification-error {
        border-left: 4px solid #dc3545;
    }
    
    .notification-info {
        border-left: 4px solid #17a2b8;
    }
    
    .notification-content {
        display: flex;
        justify-content: space-between;
        align-items: center;
        gap: 1rem;
    }
    
    .notification-content span {
        color: #333;
        font-size: 14px;
    }
    
    .notification-close {
        background: none;
        border: none;
        font-size: 18px;
        color: #999;
        cursor: pointer;
        padding: 0;
        line-height: 1;
        margin-left: 12px;
    }
    
    .notification-close:hover {
        color: #666;
    }
    
    @keyframes slideIn {
        from {
            transform: translateX(100%);
            opacity: 0;
        }
        to {
            transform: translateX(0);
            opacity: 1;
        }
    }
`;
document.head.appendChild(notificationStyles);
</script>
@endsection

