@extends('welcome')

@section('title', 'Funeral Applications')

@section('styles')
<meta name="csrf-token" content="{{ csrf_token() }}">
<link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
<link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
<style>
/* CSS Variables for consistent theming */
:root {
    --primary-color: #0055a5;
    --primary-hover: #003f7d;
    --secondary-color: #6c757d;
    --success-color: #28a745;
    --warning-color: #ffc107;
    --danger-color: #dc3545;
    --info-color: #17a2b8;
    --text-primary: #212529;
    --text-secondary: #6c757d;
    --bg-primary: #ffffff;
    --bg-secondary: #f8f9fa;
    --bg-light: #f8fafc;
    --border-color: #dee2e6;
    --border-light: #e2e8f0;
    --shadow-sm: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
    --shadow-md: 0 0.5rem 1rem rgba(0, 0, 0, 0.15);
    --radius-sm: 0.25rem;
    --radius-md: 0.375rem;
    --radius-lg: 0.5rem;
    --transition-base: all 0.15s ease-in-out;
    --font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
}

/* Reset and base styles */
* {
    margin: 0;
    padding: 0;
    box-sizing: border-box;
}

body {
    font-family: var(--font-family);
    background-color: var(--bg-light);
    color: var(--text-primary);
}

/* Main Container */
.applications-container {
    min-height: 100vh;
    background: var(--bg-light);
}

/* Header Styles */
.header-gradient {
    background: #0055a5;
    padding: 20px 0;
    box-shadow: 0 4px 12px rgba(0, 85, 165, 0.3);
}

.header-content {
    width: 100%;
    padding: 0 20px;
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.header-left {
    display: flex;
    align-items: center;
    gap: 15px;
}

.back-button {
    color: white;
    font-size: 20px;
    text-decoration: none;
    padding: 8px;
    border-radius: 50%;
    transition: var(--transition-base);
}

.back-button:hover {
    background-color: rgba(255, 255, 255, 0.1);
    color: white;
    text-decoration: none;
}

.header-title {
    color: white;
    font-size: 24px;
    font-weight: 600;
    margin: 0;
}

.header-right {
    display: flex;
    align-items: center;
    gap: 15px;
}

.create-btn {
    background: white;
    color: var(--primary-color);
    border: none;
    padding: 12px 24px;
    border-radius: var(--radius-md);
    font-size: 16px;
    font-weight: 600;
    cursor: pointer;
    transition: var(--transition-base);
    box-shadow: var(--shadow-sm);
    text-decoration: none;
    display: inline-flex;
    align-items: center;
    gap: 8px;
}

.create-btn:hover {
    background: rgba(255, 255, 255, 0.9);
    transform: translateY(-2px);
    box-shadow: var(--shadow-md);
    text-decoration: none;
    color: var(--primary-color);
}

/* Main Content */
.main-content {
    width: 100%;
    padding: 30px 20px;
}

/* Form Styles */
.form-container {
    background: white;
    border-radius: var(--radius-lg);
    box-shadow: var(--shadow-md);
    border: 1px solid var(--border-light);
    overflow: hidden;
}

.form-section {
    padding: 30px;
    border-bottom: 1px solid var(--border-light);
}

.form-section:last-child {
    border-bottom: none;
}

.section-title {
    font-size: 20px;
    font-weight: 600;
    color: var(--text-primary);
    margin-bottom: 20px;
    display: flex;
    align-items: center;
    gap: 10px;
}

.section-title i {
    color: var(--primary-color);
}

/* Applications List */
.applications-list {
    display: flex;
    flex-direction: column;
    gap: 16px;
}

.application-card {
    background: white;
    border-radius: 12px;
    padding: 20px;
    box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
    border: 1px solid #e2e8f0;
    cursor: pointer;
    transition: all 0.2s ease;
}

.application-card:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
}

.card-content {
    display: flex;
    flex-direction: column;
    gap: 12px;
}

.applicant-info h3 {
    font-size: 18px;
    font-weight: 600;
    color: #1f2937;
    margin: 0 0 4px 0;
}

.applicant-info p {
    font-size: 14px;
    color: #6b7280;
    margin: 0;
}

.deceased-info p {
    font-size: 14px;
    color: #374151;
    margin: 0;
    font-weight: 500;
}

.card-footer {
    display: flex;
    align-items: center;
    justify-content: space-between;
    margin-top: 8px;
}

.status {
    font-size: 14px;
    font-weight: 600;
    padding: 4px 12px;
    border-radius: 20px;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.status.pending {
    background: #fef3c7;
    color: #d97706;
}

.status.approved {
    background: #d1fae5;
    color: #059669;
}

.status.rejected {
    background: #fee2e2;
    color: #dc2626;
}

.btn-menu {
    display: flex;
    align-items: center;
    justify-content: center;
    width: 32px;
    height: 32px;
    background: #f3f4f6;
    color: #6b7280;
    border: 1px solid #d1d5db;
    border-radius: 6px;
    cursor: pointer;
    transition: all 0.2s ease;
}

.btn-menu:hover {
    background: #e5e7eb;
    color: #374151;
}

/* Responsive Design */
@media (max-width: 768px) {
    .main-content {
        padding: 20px 15px;
    }
    
    .form-section {
        padding: 20px;
    }
    
    .header-content {
        padding: 0 15px;
    }
    
    .header-title {
        font-size: 20px;
    }
    
    .application-card {
        padding: 16px;
    }
    
    .applicant-info h3 {
        font-size: 16px;
    }
}

@media (max-width: 480px) {
    .header-content {
        flex-direction: column;
        gap: 15px;
        align-items: flex-start;
    }
    
    .header-right {
        width: 100%;
        justify-content: flex-end;
    }
    
    .main-content {
        padding: 15px 10px;
    }
    
    .form-section {
        padding: 15px;
    }
    
    .application-card {
        padding: 12px;
    }
    
    .applicant-info h3 {
        font-size: 15px;
    }
    
    .applicant-info p,
    .deceased-info p {
        font-size: 13px;
    }
    
    .status {
        font-size: 12px;
        padding: 3px 8px;
    }
}
</style>
@endsection

@section('content')
<div class="applications-container">
    <div class="header-gradient">
        <div class="header-content">
            <div class="header-left">
                <a href="{{ route('hr.admin.dashboard') }}" class="back-button">
                    <i class="fas fa-arrow-left"></i>
                </a>
                <h1 class="header-title">Funeral Applications</h1>
            </div>
            <div class="header-right">
                <a href="{{ route('hr.create.application') }}" class="create-btn">
                    <i class="fas fa-plus"></i>
                    New Funeral
                </a>
            </div>
        </div>
    </div>

    <div class="main-content">
        <div class="form-container">
            <div class="form-section">
                <h2 class="section-title">
                    <i class="fas fa-heart"></i>
                    Applications
                </h2>

                <div class="applications-list" id="applicationsList">
                    <!-- Loading indicator -->
                    <div id="loadingIndicator" class="loading-container">
                        <div class="loading-spinner"></div>
                        <p>Loading funeral applications...</p>
                    </div>
                    
                    <!-- Applications will be dynamically loaded here -->
                </div>
            </div>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    loadFuneralApplications();
});

// Load funeral applications from MongoDB
async function loadFuneralApplications() {
    try {
        console.log('Loading funeral applications from MongoDB...');
        const applicationsList = document.getElementById('applicationsList');
        const loadingIndicator = document.getElementById('loadingIndicator');
        
        // Show loading indicator
        if (loadingIndicator) loadingIndicator.style.display = 'flex';
        
        const response = await fetch('/api/hr-applications/funeral_grant_form');
        
        if (!response.ok) {
            throw new Error('Failed to fetch funeral applications');
        }
        
        const result = await response.json();
        console.log('MongoDB data received:', result);
        
        // Hide loading indicator
        if (loadingIndicator) loadingIndicator.style.display = 'none';
        
        if (!result.success || !result.data || result.data.length === 0) {
            showNoDataMessage();
            updateHomePageCounts([]);
            return;
        }
        
        // Update home page counts
        updateHomePageCounts(result.data);
        
        // Display applications
        displayApplications(result.data);
    } catch (error) {
        console.error('Error fetching funeral applications:', error);
        const loadingIndicator = document.getElementById('loadingIndicator');
        if (loadingIndicator) loadingIndicator.style.display = 'none';
        showErrorMessage('Failed to load funeral applications. Please try again.');
    }
}

// Display applications in the UI
function displayApplications(applications) {
    const applicationsList = document.getElementById('applicationsList');
    
    // Handle both array and object formats
    const applicationsArray = Array.isArray(applications) 
        ? applications.filter(app => app && typeof app === 'object')
        : Object.entries(applications)
            .filter(([key, value]) => value && typeof value === 'object')
            .map(([key, value]) => ({ key: key, ...value }));
    
    // Store applications in sessionStorage for later use
    const applicationsObject = {};
    applicationsArray.forEach(app => {
        applicationsObject[app.id || app.key || app._id] = app;
    });
    sessionStorage.setItem('funeralApplications', JSON.stringify(applicationsObject));
    
    applicationsList.innerHTML = '';
    
    if (applicationsArray.length === 0) {
        showNoDataMessage();
        return;
    }
    
    // Sort applications by submission date (newest first)
    const sortedApplications = applicationsArray.sort((a, b) => {
        const dateA = new Date(a.submittedAt || a.submissionDate || a.createdAt || 0);
        const dateB = new Date(b.submittedAt || b.submissionDate || b.createdAt || 0);
        return dateB - dateA;
    });
    
    sortedApplications.forEach(application => {
        const applicationCard = createApplicationCard(application);
        applicationsList.appendChild(applicationCard);
    });
    
    // Add event listeners to the new cards
    addCardEventListeners();
}

// Create application card HTML
function createApplicationCard(application) {
    const card = document.createElement('div');
    card.className = 'application-card';
    card.setAttribute('data-application-id', application.id || application.key || application._id);
    
    // Format submission date
    const submissionDate = application.submissionDate || application.createdAt || 'Unknown';
    const formattedDate = formatDate(submissionDate);
    
    // Get applicant name (prioritize fullName, then name, then firstName + lastName)
    const applicantName = application.fullName || 
                         application.name || 
                         `${application.firstName || ''} ${application.lastName || ''}`.trim() ||
                         'Unknown Applicant';
    
    // Get employee number
    const employeeNo = application.employeeNumber || 
                      application.employeeId || 
                      application.empId || 
                      'N/A';
    
    // Get deceased information
    const deceasedName = application.deceasedName || 'Unknown';
    const relationship = application.deceasedRelationship || 'Unknown';
    const relationshipType = application.selectedFuneralRelationshipType ? 
        application.selectedFuneralRelationshipType.replace('_', ' ').toUpperCase() : 'N/A';
    const funeralFromDate = application.funeralFromDate ? 
        new Date(application.funeralFromDate).toLocaleDateString('en-US', { year: 'numeric', month: 'short', day: 'numeric' }) : 'N/A';
    const funeralToDate = application.funeralToDate ? 
        new Date(application.funeralToDate).toLocaleDateString('en-US', { year: 'numeric', month: 'short', day: 'numeric' }) : 'N/A';
    
    // Get status
    const status = application.status || 
                  application.applicationStatus || 
                  'pending';
    
    card.innerHTML = `
        <div class="card-content">
            <div class="applicant-info">
                <h3 class="applicant-name">${applicantName}</h3>
                <p class="employee-no">Employee No: ${employeeNo}</p>
                <p class="submission-date">Submitted: ${formattedDate}</p>
            </div>
            <div class="deceased-info">
                <p class="deceased-details">
                    <strong>Deceased:</strong> ${deceasedName} | <strong>Relationship:</strong> ${relationship}<br>
                    <strong>Grant Type:</strong> ${relationshipType}<br>
                    <strong>Leave Period:</strong> ${funeralFromDate} - ${funeralToDate}<br>
                    <strong>Department:</strong> ${application.department || 'N/A'} | <strong>Project:</strong> ${application.project || 'N/A'}
                </p>
            </div>
            <div class="card-footer">
                <span class="status ${status.toLowerCase()}">Status: ${status.charAt(0).toUpperCase() + status.slice(1)}</span>
                <button class="btn-menu">
                    <i class="fas fa-ellipsis-v"></i>
                </button>
            </div>
        </div>
    `;
    
    return card;
}

// Format date for display
function formatDate(dateString) {
    try {
        const date = new Date(dateString);
        if (isNaN(date.getTime())) {
            return dateString; // Return original if not a valid date
        }
        return date.toLocaleDateString('en-US', {
            year: 'numeric',
            month: '2-digit',
            day: '2-digit'
        });
    } catch (error) {
        return dateString;
    }
}

// Show no data message
function showNoDataMessage() {
    const applicationsList = document.getElementById('applicationsList');
    applicationsList.innerHTML = `
        <div class="no-data-message">
            <i class="fas fa-inbox" style="font-size: 48px; margin-bottom: 16px; color: #9ca3af;"></i>
            <h3>No Funeral Applications Found</h3>
            <p>There are currently no funeral applications in the system.</p>
        </div>
    `;
}

// Show error message
function showErrorMessage(message) {
    const applicationsList = document.getElementById('applicationsList');
    applicationsList.innerHTML = `
        <div class="error-message">
            <i class="fas fa-exclamation-triangle" style="margin-right: 8px;"></i>
            ${message}
        </div>
    `;
}

// Add event listeners to application cards
function addCardEventListeners() {
    // Add click handlers for application cards
    const applicationCards = document.querySelectorAll('.application-card');
    
    applicationCards.forEach(card => {
        card.addEventListener('click', function(e) {
            // Don't trigger if clicking on menu button
            if (e.target.closest('.btn-menu')) return;
            
            const applicationId = this.getAttribute('data-application-id');
            
            // Navigate to dedicated application detail page
            window.location.href = `/application-detail?id=${applicationId}&type=funeral_grant_form`;
        });
    });

    // Add click handlers for menu buttons
    const menuButtons = document.querySelectorAll('.btn-menu');
    
    menuButtons.forEach(button => {
        button.addEventListener('click', function(e) {
            e.stopPropagation();
            const card = this.closest('.application-card');
            const applicationId = card.getAttribute('data-application-id');
            showActionMenu(this, applicationId);
        });
    });
}

// Update home page counts
function updateHomePageCounts(applications) {
    try {
        const applicationsArray = Array.isArray(applications) ? applications : Object.values(applications);
        
        const pendingCount = applicationsArray.filter(app => 
            (app.status || app.applicationStatus || 'pending').toLowerCase() === 'pending'
        ).length;
        
        const approvedCount = applicationsArray.filter(app => 
            (app.status || app.applicationStatus || '').toLowerCase() === 'approved'
        ).length;
        
        const rejectedCount = applicationsArray.filter(app => 
            (app.status || app.applicationStatus || '').toLowerCase() === 'rejected'
        ).length;
        
        const totalCount = applicationsArray.length;
        
        console.log('Funeral Applications Counts:', {
            total: totalCount,
            pending: pendingCount,
            approved: approvedCount,
            rejected: rejectedCount
        });
        
        localStorage.setItem('funeralApplicationsCount', JSON.stringify({
            total: totalCount,
            pending: pendingCount,
            approved: approvedCount,
            rejected: rejectedCount,
            lastUpdated: new Date().toISOString()
        }));
        
        if (window.opener && !window.opener.closed) {
            window.opener.postMessage({
                type: 'funeralApplicationsUpdate',
                counts: {
                    total: totalCount,
                    pending: pendingCount,
                    approved: approvedCount,
                    rejected: rejectedCount
                }
            }, '*');
        }
        
    } catch (error) {
        console.error('Error updating home page counts:', error);
    }
}

// Global variable to track open menu
let currentOpenMenu = null;

// Show action menu
function showActionMenu(button, applicationId) {
    // Close any existing menu
    if (currentOpenMenu) {
        currentOpenMenu.remove();
        currentOpenMenu = null;
    }

    const menu = document.createElement('div');
    menu.className = 'action-menu';
    menu.innerHTML = `
        <div class="action-menu-item" data-action="view">
            <i class="fas fa-eye"></i> View Details
        </div>
        <div class="action-menu-item" data-action="approve">
            <i class="fas fa-check"></i> Approve
        </div>
        <div class="action-menu-item" data-action="reject">
            <i class="fas fa-times"></i> Reject
        </div>
        <div class="action-menu-item action-menu-item-danger" data-action="delete">
            <i class="fas fa-trash"></i> Delete
        </div>
    `;

    // Position menu
    const buttonRect = button.getBoundingClientRect();
    menu.style.position = 'fixed';
    menu.style.top = `${buttonRect.bottom + 5}px`;
    menu.style.left = `${buttonRect.left - 150}px`;
    
    document.body.appendChild(menu);
    currentOpenMenu = menu;

    // Add event listeners to menu items
    menu.querySelectorAll('.action-menu-item').forEach(item => {
        item.addEventListener('click', (e) => {
            e.stopPropagation();
            const action = item.dataset.action;
            handleMenuAction(action, applicationId);
            menu.remove();
            currentOpenMenu = null;
        });
    });

    // Close menu when clicking outside
    setTimeout(() => {
        document.addEventListener('click', function closeMenu(e) {
            if (menu && !menu.contains(e.target)) {
                menu.remove();
                currentOpenMenu = null;
                document.removeEventListener('click', closeMenu);
            }
        });
    }, 10);
}

// Handle menu actions
async function handleMenuAction(action, applicationId) {
    const applications = JSON.parse(sessionStorage.getItem('funeralApplications') || '{}');
    const application = applications[applicationId];

    if (!application) {
        showNotification('Application not found', 'error');
        return;
    }

    switch(action) {
        case 'view':
            showApplicationDetails(application);
            break;
        case 'approve':
            showConfirmDialog('Approve Application', 'Are you sure you want to approve this funeral application?', () => {
                updateApplicationStatus(applicationId, 'approved');
            });
            break;
        case 'reject':
            showConfirmDialog('Reject Application', 'Are you sure you want to reject this funeral application?', () => {
                updateApplicationStatus(applicationId, 'rejected');
            });
            break;
        case 'delete':
            showConfirmDialog('Delete Application', 'Are you sure you want to delete this application? This action cannot be undone.', () => {
                deleteApplication(applicationId);
            }, true);
            break;
    }
}

// Update application status
async function updateApplicationStatus(applicationId, status) {
    try {
        const statusData = {
            status: status,
            comments: '',
            approverName: 'Admin',
            approverRole: 'ProAdmin',
            reason: `${status} by admin`,
            timestamp: new Date().toISOString()
        };
        
        const response = await fetch(`/api/hr-applications/${applicationId}/status`, {
            method: 'PUT',
            headers: { 
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.content || ''
            },
            body: JSON.stringify(statusData)
        });

        if (!response.ok) {
            throw new Error('Failed to update status');
        }
        
        const result = await response.json();
        if (result.success) {
            showNotification(`Application ${status} successfully`, 'success');
            loadFuneralApplications(); // Reload applications
        } else {
            throw new Error(result.error || 'Failed to update status');
        }
    } catch (error) {
        console.error('Error updating application status:', error);
        showNotification('Failed to update application status', 'error');
    }
}

// Delete application
async function deleteApplication(applicationId) {
    try {
        const response = await fetch(`/api/hr-applications/${applicationId}`, {
            method: 'DELETE',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.content || ''
            }
        });

        if (!response.ok) {
            throw new Error('Failed to delete application');
        }
        
        const result = await response.json();
        if (result.success) {
            showNotification('Application deleted successfully', 'success');
            loadFuneralApplications(); // Reload applications
        } else {
            throw new Error(result.error || 'Failed to delete application');
        }
    } catch (error) {
        console.error('Error deleting application:', error);
        showNotification('Failed to delete application', 'error');
    }
}

// Show application details modal
function showApplicationDetails(application) {
    const modal = document.createElement('div');
    modal.className = 'details-modal';
    
    const applicantName = application.fullName || application.name || 'Unknown Applicant';
    const employeeNo = application.employeeNo || application.employeeNumber || 'N/A';
    const deceasedName = application.deceasedName || 'Unknown';
    const relationship = application.deceasedRelationship || 'Unknown';
    const relationshipType = application.selectedFuneralRelationshipType ? 
        application.selectedFuneralRelationshipType.replace('_', ' ').toUpperCase() : 'N/A';
    const funeralFromDate = application.funeralFromDate ? 
        formatDate(application.funeralFromDate) : 'N/A';
    const funeralToDate = application.funeralToDate ? 
        formatDate(application.funeralToDate) : 'N/A';
    const deceasedDate = application.deceasedDate ? 
        formatDate(application.deceasedDate) : 'N/A';
    const submissionDate = formatDate(application.submittedAt || application.submissionDate || application.createdAt || '');
    
    modal.innerHTML = `
        <div class="details-modal-content">
            <div class="details-modal-header">
                <h2>Funeral Application Details</h2>
                <button class="details-modal-close">&times;</button>
            </div>
            <div class="details-modal-body">
                <div class="detail-row"><span class="detail-label">Application No:</span><span class="detail-value">${application.applicationNumber || 'N/A'}</span></div>
                <div class="detail-row"><span class="detail-label">Applicant Name:</span><span class="detail-value">${applicantName}</span></div>
                <div class="detail-row"><span class="detail-label">Employee No:</span><span class="detail-value">${employeeNo}</span></div>
                <div class="detail-row"><span class="detail-label">Department:</span><span class="detail-value">${application.department || 'N/A'}</span></div>
                <div class="detail-row"><span class="detail-label">Country:</span><span class="detail-value">${application.country || 'N/A'}</span></div>
                <div class="detail-row"><span class="detail-label">Project:</span><span class="detail-value">${application.project || 'N/A'}</span></div>
                <div class="detail-row"><span class="detail-label">NRC No:</span><span class="detail-value">${application.nrcNo || 'N/A'}</span></div>
                <div class="detail-row"><span class="detail-label">Grant Type:</span><span class="detail-value">${relationshipType}</span></div>
                <div class="detail-row"><span class="detail-label">Deceased Name:</span><span class="detail-value">${deceasedName}</span></div>
                <div class="detail-row"><span class="detail-label">Relationship:</span><span class="detail-value">${relationship}</span></div>
                <div class="detail-row"><span class="detail-label">Deceased Date:</span><span class="detail-value">${deceasedDate}</span></div>
                <div class="detail-row"><span class="detail-label">Leave From:</span><span class="detail-value">${funeralFromDate}</span></div>
                <div class="detail-row"><span class="detail-label">Leave To:</span><span class="detail-value">${funeralToDate}</span></div>
                <div class="detail-row"><span class="detail-label">Status:</span><span class="detail-value"><span class="status ${(application.status || application.applicationStatus || 'pending').toLowerCase()}">${application.status || application.applicationStatus || 'Pending'}</span></span></div>
                <div class="detail-row"><span class="detail-label">Submitted At:</span><span class="detail-value">${submissionDate}</span></div>
            </div>
        </div>
    `;

    document.body.appendChild(modal);

    // Close modal handlers
    modal.querySelector('.details-modal-close').addEventListener('click', () => modal.remove());
    modal.addEventListener('click', (e) => {
        if (e.target === modal) modal.remove();
    });
}

// Show confirm dialog
function showConfirmDialog(title, message, onConfirm, isDanger = false) {
    const dialog = document.createElement('div');
    dialog.className = 'confirm-dialog';
    dialog.innerHTML = `
        <div class="confirm-dialog-content">
            <h3>${title}</h3>
            <p>${message}</p>
            <div class="confirm-dialog-buttons">
                <button class="btn-cancel">Cancel</button>
                <button class="btn-confirm ${isDanger ? 'btn-danger' : ''}">${isDanger ? 'Delete' : 'Confirm'}</button>
            </div>
        </div>
    `;

    document.body.appendChild(dialog);

    dialog.querySelector('.btn-cancel').addEventListener('click', () => dialog.remove());
    dialog.querySelector('.btn-confirm').addEventListener('click', () => {
        onConfirm();
        dialog.remove();
    });
    dialog.addEventListener('click', (e) => {
        if (e.target === dialog) dialog.remove();
    });
}

// Notification system
function showNotification(message, type = 'info') {
    const notification = document.createElement('div');
    notification.className = `app-notification app-notification-${type}`;
    notification.textContent = message;
    
    notification.style.cssText = `
        position: fixed;
        top: 80px;
        right: 20px;
        padding: 12px 20px;
        border-radius: 8px;
        color: white;
        font-weight: 500;
        font-size: 14px;
        z-index: 10001;
        opacity: 0;
        transform: translateX(100%);
        transition: all 0.3s ease;
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
        font-family: 'Inter', sans-serif;
    `;
    
    if (type === 'success') {
        notification.style.background = 'linear-gradient(135deg, #10b981 0%, #059669 100%)';
    } else if (type === 'error') {
        notification.style.background = 'linear-gradient(135deg, #ef4444 0%, #dc2626 100%)';
    } else if (type === 'warning') {
        notification.style.background = 'linear-gradient(135deg, #f59e0b 0%, #d97706 100%)';
    } else {
        notification.style.background = 'linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%)';
    }
    
    document.body.appendChild(notification);
    
    setTimeout(() => {
        notification.style.opacity = '1';
        notification.style.transform = 'translateX(0)';
    }, 100);
    
    setTimeout(() => {
        notification.style.opacity = '0';
        notification.style.transform = 'translateX(100%)';
        setTimeout(() => {
            if (notification.parentNode) {
                document.body.removeChild(notification);
            }
        }, 300);
    }, 3000);
}
</script>

<style>
/* Import Google Fonts */
@import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');

/* ===== APPLICATIONS CONTAINER ===== */
.applications-container {
    min-height: 100vh;
}

/* ===== HEADER SECTION ===== */
.applications-header {
    background: #0055a5;
    padding: 16px 20px;
    box-shadow: 0 2px 8px rgba(0, 85, 165, 0.3);
}

.header-content {
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.btn-back {
    display: flex;
    align-items: center;
    justify-content: center;
    width: 40px;
    height: 40px;
    background: rgba(255, 255, 255, 0.1);
    color: white;
    text-decoration: none;
    border-radius: 8px;
    transition: all 0.2s ease;
    border: 1px solid rgba(255, 255, 255, 0.2);
}

.btn-back:hover {
    background: rgba(255, 255, 255, 0.2);
    color: white;
    text-decoration: none;
    transform: translateY(-1px);
}

.page-title h1 {
    color: white;
    font-size: 20px;
    font-weight: 600;
    margin: 0;
    text-align: center;
    flex: 1;
}

.btn-notification {
    display: flex;
    align-items: center;
    justify-content: center;
    width: 40px;
    height: 40px;
    background: rgba(255, 255, 255, 0.1);
    color: white;
    border: 1px solid rgba(255, 255, 255, 0.2);
    border-radius: 8px;
    cursor: pointer;
    transition: all 0.2s ease;
}

.btn-notification:hover {
    background: rgba(255, 255, 255, 0.2);
    transform: translateY(-1px);
}

/* ===== MAIN CONTENT ===== */
.applications-content {
    padding: 20px;
}

.section-header {
    display: flex;
    align-items: center;
    justify-content: space-between;
    margin-bottom: 20px;
}

.section-header h2 {
    color: #0055a5;
    font-size: 24px;
    font-weight: 700;
    margin: 0;
}

.btn-lock {
    display: flex;
    align-items: center;
    justify-content: center;
    width: 40px;
    height: 40px;
    background: #f3f4f6;
    color: #6b7280;
    border: 1px solid #d1d5db;
    border-radius: 8px;
    cursor: pointer;
    transition: all 0.2s ease;
}

.btn-lock:hover {
    background: #e5e7eb;
    color: #374151;
}

/* ===== APPLICATIONS LIST ===== */
.applications-list {
    display: flex;
    flex-direction: column;
    gap: 16px;
}

.application-card {
    background: white;
    border-radius: 12px;
    padding: 20px;
    box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
    border: 1px solid #e2e8f0;
    cursor: pointer;
    transition: all 0.2s ease;
}

.application-card:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
}

.card-content {
    display: flex;
    flex-direction: column;
    gap: 12px;
}

.applicant-info h3 {
    font-size: 18px;
    font-weight: 600;
    color: #1f2937;
    margin: 0 0 4px 0;
}

.applicant-info p {
    font-size: 14px;
    color: #6b7280;
    margin: 0;
}

.deceased-info p {
    font-size: 14px;
    color: #374151;
    margin: 0;
    font-weight: 500;
}

.card-footer {
    display: flex;
    align-items: center;
    justify-content: space-between;
    margin-top: 8px;
}

.status {
    font-size: 14px;
    font-weight: 600;
    padding: 4px 12px;
    border-radius: 20px;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.status.pending {
    background: #fef3c7;
    color: #d97706;
}

.status.approved {
    background: #d1fae5;
    color: #059669;
}

.status.rejected {
    background: #fee2e2;
    color: #dc2626;
}

.btn-menu {
    display: flex;
    align-items: center;
    justify-content: center;
    width: 32px;
    height: 32px;
    background: #f3f4f6;
    color: #6b7280;
    border: 1px solid #d1d5db;
    border-radius: 6px;
    cursor: pointer;
    transition: all 0.2s ease;
}

.btn-menu:hover {
    background: #e5e7eb;
    color: #374151;
}

/* ===== RESPONSIVE DESIGN ===== */
@media (max-width: 768px) {
    .applications-content {
        padding: 16px;
    }
    
    .section-header h2 {
        font-size: 20px;
    }
    
    .application-card {
        padding: 16px;
    }
    
    .applicant-info h3 {
        font-size: 16px;
    }
}

@media (max-width: 576px) {
    .applications-header {
        padding: 12px 16px;
    }
    
    .page-title h1 {
        font-size: 18px;
    }
    
    .applications-content {
        padding: 12px;
    }
    
    .section-header h2 {
        font-size: 18px;
    }
    
    .application-card {
        padding: 12px;
    }
    
    .applicant-info h3 {
        font-size: 15px;
    }
    
    .applicant-info p,
    .deceased-info p {
        font-size: 13px;
    }
    
    .status {
        font-size: 12px;
        padding: 3px 8px;
    }
}

/* Loading Styles */
.loading-container {
    display: flex;
    flex-direction: column;
    align-items: center;
    justify-content: center;
    padding: 40px 20px;
    text-align: center;
}

.loading-spinner {
    width: 40px;
    height: 40px;
    border: 4px solid #f3f4f6;
    border-top: 4px solid #0055a5;
    border-radius: 50%;
    animation: spin 1s linear infinite;
    margin-bottom: 16px;
}

@keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}

.loading-container p {
    color: #6b7280;
    font-size: 16px;
    margin: 0;
}

.error-message {
    background: #fee2e2;
    color: #dc2626;
    padding: 16px;
    border-radius: 8px;
    text-align: center;
    margin: 20px 0;
    border: 1px solid #fecaca;
}

.no-data-message {
    background: #f3f4f6;
    color: #6b7280;
    padding: 40px 20px;
    border-radius: 8px;
    text-align: center;
    margin: 20px 0;
}

/* Action Menu */
.action-menu {
    position: fixed;
    background: white;
    border-radius: 8px;
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
    min-width: 180px;
    z-index: 10000;
    overflow: hidden;
    animation: slideIn 0.2s ease;
}

@keyframes slideIn {
    from {
        opacity: 0;
        transform: translateY(-10px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.action-menu-item {
    padding: 12px 16px;
    cursor: pointer;
    display: flex;
    align-items: center;
    gap: 10px;
    color: #374151;
    font-size: 14px;
    transition: all 0.2s ease;
}

.action-menu-item:hover {
    background: #f3f4f6;
}

.action-menu-item i {
    width: 16px;
    font-size: 14px;
}

.action-menu-item-danger {
    color: #dc2626;
}

.action-menu-item-danger:hover {
    background: #fee2e2;
}

/* Details Modal */
.details-modal {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0, 0, 0, 0.5);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 10002;
    padding: 20px;
}

.details-modal-content {
    background: white;
    border-radius: 12px;
    max-width: 600px;
    width: 100%;
    max-height: 80vh;
    overflow-y: auto;
    box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.1);
}

.details-modal-header {
    display: flex;
    align-items: center;
    justify-content: space-between;
    padding: 20px;
    border-bottom: 1px solid #e5e7eb;
}

.details-modal-header h2 {
    margin: 0;
    font-size: 20px;
    font-weight: 600;
    color: #0055a5;
}

.details-modal-close {
    background: none;
    border: none;
    font-size: 28px;
    color: #6b7280;
    cursor: pointer;
    width: 32px;
    height: 32px;
    display: flex;
    align-items: center;
    justify-content: center;
    border-radius: 4px;
    transition: all 0.2s ease;
}

.details-modal-close:hover {
    background: #f3f4f6;
    color: #374151;
}

.details-modal-body {
    padding: 20px;
}

.detail-row {
    display: flex;
    padding: 12px 0;
    border-bottom: 1px solid #f3f4f6;
}

.detail-row:last-child {
    border-bottom: none;
}

.detail-label {
    font-weight: 600;
    color: #374151;
    min-width: 150px;
    font-size: 14px;
}

.detail-value {
    color: #6b7280;
    flex: 1;
    font-size: 14px;
}

/* Confirm Dialog */
.confirm-dialog {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0, 0, 0, 0.5);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 10003;
    padding: 20px;
}

.confirm-dialog-content {
    background: white;
    border-radius: 12px;
    padding: 24px;
    max-width: 400px;
    width: 100%;
    box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.1);
}

.confirm-dialog-content h3 {
    margin: 0 0 12px 0;
    font-size: 18px;
    font-weight: 600;
    color: #1f2937;
}

.confirm-dialog-content p {
    margin: 0 0 20px 0;
    color: #6b7280;
    font-size: 14px;
    line-height: 1.5;
}

.confirm-dialog-buttons {
    display: flex;
    gap: 12px;
    justify-content: flex-end;
}

.btn-cancel,
.btn-confirm {
    padding: 10px 20px;
    border-radius: 6px;
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.2s ease;
    border: none;
}

.btn-cancel {
    background: #f3f4f6;
    color: #374151;
}

.btn-cancel:hover {
    background: #e5e7eb;
}

.btn-confirm {
    background: #0055a5;
    color: white;
}

.btn-confirm:hover {
    background: #003f7d;
}

.btn-confirm.btn-danger {
    background: #dc2626;
}

.btn-confirm.btn-danger:hover {
    background: #b91c1c;
}

@media (max-width: 640px) {
    .details-modal-content {
        max-width: 100%;
    }
    
    .detail-row {
        flex-direction: column;
        gap: 4px;
    }
    
    .detail-label {
        min-width: auto;
    }
}
</style>

@endsection
