@extends('welcome')

@section('title', 'Leave Applications')

@section('styles')
<meta name="csrf-token" content="{{ csrf_token() }}">
<link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
<link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
<style>
/* CSS Variables for consistent theming */
:root {
    --primary-color: #0055a5;
    --primary-hover: #003f7d;
    --secondary-color: #6c757d;
    --success-color: #28a745;
    --warning-color: #ffc107;
    --danger-color: #dc3545;
    --info-color: #17a2b8;
    --text-primary: #212529;
    --text-secondary: #6c757d;
    --bg-primary: #ffffff;
    --bg-secondary: #f8f9fa;
    --bg-light: #f8fafc;
    --border-color: #dee2e6;
    --border-light: #e2e8f0;
    --shadow-sm: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
    --shadow-md: 0 0.5rem 1rem rgba(0, 0, 0, 0.15);
    --radius-sm: 0.25rem;
    --radius-md: 0.375rem;
    --radius-lg: 0.5rem;
    --transition-base: all 0.15s ease-in-out;
    --font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
}

/* Reset and base styles */
* {
    margin: 0;
    padding: 0;
    box-sizing: border-box;
}

body {
    font-family: var(--font-family);
    background-color: var(--bg-light);
    color: var(--text-primary);
}

/* Main Container */
.applications-container {
    min-height: 100vh;
    background: var(--bg-light);
}

/* Header Styles */
.header-gradient {
    background: #0055a5;
    padding: 20px 0;
    box-shadow: 0 4px 12px rgba(0, 85, 165, 0.3);
}

.header-content {
    width: 100%;
    padding: 0 20px;
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.header-left {
    display: flex;
    align-items: center;
    gap: 15px;
}

.back-button {
    color: white;
    font-size: 20px;
    text-decoration: none;
    padding: 8px;
    border-radius: 50%;
    transition: var(--transition-base);
}

.back-button:hover {
    background-color: rgba(255, 255, 255, 0.1);
    color: white;
    text-decoration: none;
}

.header-title {
    color: white;
    font-size: 24px;
    font-weight: 600;
    margin: 0;
}

.header-right {
    display: flex;
    align-items: center;
    gap: 15px;
}

.create-btn {
    background: white;
    color: var(--primary-color);
    border: none;
    padding: 12px 24px;
    border-radius: var(--radius-md);
    font-size: 16px;
    font-weight: 600;
    cursor: pointer;
    transition: var(--transition-base);
    box-shadow: var(--shadow-sm);
    text-decoration: none;
    display: inline-flex;
    align-items: center;
    gap: 8px;
}

.create-btn:hover {
    background: rgba(255, 255, 255, 0.9);
    transform: translateY(-2px);
    box-shadow: var(--shadow-md);
    text-decoration: none;
    color: var(--primary-color);
}

/* Main Content */
.main-content {
    width: 100%;
    padding: 30px 20px;
}

/* Form Styles */
.form-container {
    background: white;
    border-radius: var(--radius-lg);
    box-shadow: var(--shadow-md);
    border: 1px solid var(--border-light);
    overflow: hidden;
}

.form-section {
    padding: 30px;
    border-bottom: 1px solid var(--border-light);
}

.form-section:last-child {
    border-bottom: none;
}

.section-title {
    font-size: 20px;
    font-weight: 600;
    color: var(--text-primary);
    margin-bottom: 20px;
    display: flex;
    align-items: center;
    gap: 10px;
}

.section-title i {
    color: var(--primary-color);
}

/* Applications List */
.applications-list {
    display: flex;
    flex-direction: column;
    gap: 16px;
}

.application-card {
    background: white;
    border-radius: 12px;
    padding: 20px;
    box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
    border: 1px solid #e2e8f0;
    cursor: pointer;
    transition: all 0.2s ease;
}

.application-card:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
}

.card-content {
    display: flex;
    flex-direction: column;
    gap: 12px;
}

.applicant-info h3 {
    font-size: 18px;
    font-weight: 600;
    color: #1f2937;
    margin: 0 0 4px 0;
}

.applicant-info p {
    font-size: 14px;
    color: #6b7280;
    margin: 0;
}

.leave-info p {
    font-size: 14px;
    color: #374151;
    margin: 0;
    font-weight: 500;
}

.card-footer {
    display: flex;
    align-items: center;
    justify-content: space-between;
    margin-top: 8px;
}

.status {
    font-size: 14px;
    font-weight: 600;
    padding: 4px 12px;
    border-radius: 20px;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.status.pending {
    background: #fef3c7;
    color: #d97706;
}

.status.approved {
    background: #d1fae5;
    color: #059669;
}

.status.rejected {
    background: #fee2e2;
    color: #dc2626;
}

/* Loading State */
.loading-state {
    display: flex;
    flex-direction: column;
    align-items: center;
    justify-content: center;
    padding: 60px 20px;
    text-align: center;
}

.loading-spinner {
    width: 40px;
    height: 40px;
    border: 4px solid #e5e7eb;
    border-top: 4px solid #0055a5;
    border-radius: 50%;
    animation: spin 1s linear infinite;
    margin-bottom: 16px;
}

@keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}

.loading-state p {
    color: #6b7280;
    font-size: 16px;
    margin: 0;
}

/* Empty State */
.empty-state {
    display: flex;
    flex-direction: column;
    align-items: center;
    justify-content: center;
    padding: 60px 20px;
    text-align: center;
}

.empty-icon {
    color: #9ca3af;
    margin-bottom: 16px;
}

.empty-state h3 {
    color: #374151;
    font-size: 18px;
    font-weight: 600;
    margin: 0 0 8px 0;
}

.empty-state p {
    color: #6b7280;
    font-size: 14px;
    margin: 0;
}

.btn-menu {
    display: flex;
    align-items: center;
    justify-content: center;
    width: 32px;
    height: 32px;
    background: #f3f4f6;
    color: #6b7280;
    border: 1px solid #d1d5db;
    border-radius: 6px;
    cursor: pointer;
    transition: all 0.2s ease;
}

.btn-menu:hover {
    background: #e5e7eb;
    color: #374151;
}

/* Responsive Design */
@media (max-width: 768px) {
    .main-content {
        padding: 20px 15px;
    }
    
    .form-section {
        padding: 20px;
    }
    
    .header-content {
        padding: 0 15px;
    }
    
    .header-title {
        font-size: 20px;
    }
    
    .application-card {
        padding: 16px;
    }
    
    .applicant-info h3 {
        font-size: 16px;
    }
}

@media (max-width: 480px) {
    .header-content {
        flex-direction: column;
        gap: 15px;
        align-items: flex-start;
    }
    
    .header-right {
        width: 100%;
        justify-content: flex-end;
    }
    
    .main-content {
        padding: 15px 10px;
    }
    
    .form-section {
        padding: 15px;
    }
    
    .application-card {
        padding: 12px;
    }
    
    .applicant-info h3 {
        font-size: 15px;
    }
    
    .applicant-info p,
    .leave-info p {
        font-size: 13px;
    }
    
    .status {
        font-size: 12px;
        padding: 3px 8px;
    }
}
</style>
@endsection

@section('content')
<div class="applications-container">
    <div class="header-gradient">
        <div class="header-content">
            <div class="header-left">
                <a href="{{ route('hr.admin.dashboard') }}" class="back-button">
                    <i class="fas fa-arrow-left"></i>
                </a>
                <h1 class="header-title">Leave Applications</h1>
            </div>
            <div class="header-right">
                <a href="{{ route('hr.create.application') }}" class="create-btn">
                    <i class="fas fa-plus"></i>
                    New Leave
                </a>
            </div>
        </div>
    </div>

    <div class="main-content">
        <div class="form-container">
            <div class="form-section">
                <h2 class="section-title">
                    <i class="fas fa-calendar-alt"></i>
                    Applications
                </h2>

                <div class="applications-list" id="applicationsList">
                    <!-- Loading State -->
                    <div class="loading-state" id="loadingState">
                        <div class="loading-spinner"></div>
                        <p>Loading leave applications...</p>
                    </div>
                    
                    <!-- Empty State -->
                    <div class="empty-state" id="emptyState" style="display: none;">
                        <div class="empty-icon">
                            <i class="fas fa-calendar-alt" style="font-size: 48px;"></i>
                        </div>
                        <h3>No Leave Applications</h3>
                        <p>There are no leave applications to display at the moment.</p>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Check if user role is set, if not show role selector
    checkUserRole();
    loadLeaveApplications();
});

// Check and set user role
function checkUserRole() {
    const currentUser = JSON.parse(localStorage.getItem('currentUser') || '{}');
    
    if (!currentUser.role) {
        console.log('⚠️ No user role set. Please select your role.');
        showRoleSelector();
    } else {
        console.log('✅ User role:', currentUser.role);
    }
}

// Show role selector modal
function showRoleSelector() {
    const modal = document.createElement('div');
    modal.style.cssText = 'position: fixed; top: 0; left: 0; right: 0; bottom: 0; background: rgba(0,0,0,0.8); z-index: 999999; display: flex; align-items: center; justify-content: center;';
    modal.innerHTML = `
        <div style="background: white; padding: 30px; border-radius: 12px; max-width: 500px; width: 90%;">
            <h2 style="color: #0055a5; margin-bottom: 20px; font-size: 24px;">👤 Select Your Role</h2>
            <p style="margin-bottom: 20px; color: #666;">Please select your role to test the approval system:</p>
            <div style="display: flex; flex-direction: column; gap: 12px;">
                <button onclick="setUserRole('ProAdmin', 'Admin User')" style="padding: 15px; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; border: none; border-radius: 8px; cursor: pointer; font-weight: 600; font-size: 16px;">
                    👑 ProAdmin (Can approve all levels)
                </button>
                <button onclick="setUserRole('Supervisor', 'Test Supervisor')" style="padding: 15px; background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%); color: white; border: none; border-radius: 8px; cursor: pointer; font-weight: 600; font-size: 16px;">
                    ⭐ Supervisor
                </button>
                <button onclick="setUserRole('HrManager', 'Test HR Manager')" style="padding: 15px; background: linear-gradient(135deg, #4facfe 0%, #00f2fe 100%); color: white; border: none; border-radius: 8px; cursor: pointer; font-weight: 600; font-size: 16px;">
                    ⭐ HR Manager
                </button>
                <button onclick="setUserRole('FinanceManager', 'Test Finance Manager')" style="padding: 15px; background: linear-gradient(135deg, #43e97b 0%, #38f9d7 100%); color: white; border: none; border-radius: 8px; cursor: pointer; font-weight: 600; font-size: 16px;">
                    ⭐ Finance Manager
                </button>
            </div>
        </div>
    `;
    document.body.appendChild(modal);
}

// Set user role in localStorage
function setUserRole(role, name) {
    localStorage.setItem('currentUser', JSON.stringify({
        _id: 'test_' + Date.now(),
        name: name,
        role: role
    }));
    console.log('✅ Role set to:', role);
    location.reload();
}

// Add floating role switcher button
window.addEventListener('load', function() {
    const currentUser = JSON.parse(localStorage.getItem('currentUser') || '{}');
    if (currentUser.role) {
        const roleBtn = document.createElement('button');
        roleBtn.innerHTML = `<i class="fas fa-user-circle"></i> ${currentUser.role}`;
        roleBtn.style.cssText = 'position: fixed; bottom: 20px; right: 20px; z-index: 9999; padding: 12px 20px; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; border: none; border-radius: 50px; cursor: pointer; font-weight: 600; box-shadow: 0 4px 15px rgba(0,0,0,0.2); display: flex; align-items: center; gap: 8px; font-size: 14px;';
        roleBtn.onclick = function() {
            if (confirm('Change your role? This will reload the page.')) {
                localStorage.removeItem('currentUser');
                location.reload();
            }
        };
        document.body.appendChild(roleBtn);
    }
});

// Load leave applications from MongoDB
async function loadLeaveApplications() {
    try {
        console.log('Loading leave applications from MongoDB...');
        
        const response = await fetch('/api/hr-applications/leave_application');
        
        if (!response.ok) {
            throw new Error('Failed to fetch leave applications');
        }
        
        const result = await response.json();
        console.log('MongoDB data received:', result);
        
        // Hide loading state
        document.getElementById('loadingState').style.display = 'none';
        
        if (!result.success || !result.data || result.data.length === 0) {
            // Show empty state
            document.getElementById('emptyState').style.display = 'block';
            updateHomePageCounts([]);
            return;
        }
        
        // Display applications
        displayApplications(result.data);
        
        // Update home page counts
        updateHomePageCounts(result.data);
        
    } catch (error) {
        console.error('Error loading leave applications:', error);
        document.getElementById('loadingState').style.display = 'none';
        showNotification('Failed to load leave applications. Please try again.', 'error');
    }
}

// Display applications in the UI
function displayApplications(applications) {
    const applicationsList = document.getElementById('applicationsList');
    
    // Handle both array and object formats
    const applicationsArray = Array.isArray(applications) 
        ? applications 
        : Object.entries(applications).map(([key, value]) => ({ id: key, ...value }));
    
    // Store applications in sessionStorage for later use
    const applicationsObject = {};
    applicationsArray.forEach(app => {
        applicationsObject[app.id || app._id] = app;
    });
    sessionStorage.setItem('leaveApplications', JSON.stringify(applicationsObject));
    
    // Clear existing content
    applicationsList.innerHTML = '';
    
    // Sort by submission date
    const sortedApplications = applicationsArray.sort((a, b) => 
        new Date(b.submittedAt || b.createdAt) - new Date(a.submittedAt || a.createdAt)
    );
    
    console.log('Applications array:', sortedApplications);
    
    sortedApplications.forEach(application => {
        const applicationCard = createApplicationCard(application);
        applicationsList.appendChild(applicationCard);
    });
    
    // Add event listeners to new cards
    addCardEventListeners();
}

// Create application card element
function createApplicationCard(application) {
    const card = document.createElement('div');
    card.className = 'application-card';
    card.dataset.applicationId = application.id || application._id;
    
    // Calculate leave duration
    const startDate = new Date(application.leaveStartDate || application.dateFrom);
    const endDate = new Date(application.leaveEndDate || application.dateTo);
    const duration = Math.ceil((endDate - startDate) / (1000 * 60 * 60 * 24)) + 1;
    
    // Format submission date
    const submissionDate = new Date(application.submittedAt || application.createdAt).toLocaleDateString('en-US', {
        year: 'numeric',
        month: 'short',
        day: 'numeric'
    });
    
    // Format leave dates
    const leaveStartFormatted = new Date(application.leaveStartDate || application.dateFrom).toLocaleDateString('en-US', {
        year: 'numeric',
        month: 'short',
        day: 'numeric'
    });
    
    const leaveEndFormatted = new Date(application.leaveEndDate || application.dateTo).toLocaleDateString('en-US', {
        year: 'numeric',
        month: 'short',
        day: 'numeric'
    });
    
    card.innerHTML = `
        <div class="card-content">
            <div class="applicant-info">
                <h3 class="applicant-name">${application.fullName || application.applicantName || 'N/A'}</h3>
                <p class="employee-no">Employee No: ${application.employeeNo || application.employeeNumber || 'N/A'}</p>
                <p class="submission-date">Submitted: ${submissionDate}</p>
            </div>
            <div class="leave-info">
                <p class="leave-details">
                    <strong>Leave Period:</strong> ${leaveStartFormatted} - ${leaveEndFormatted}<br>
                    <strong>Duration:</strong> ${duration} day${duration > 1 ? 's' : ''}<br>
                    <strong>Reason:</strong> ${application.reasonOfLeave || application.reason || 'N/A'}<br>
                    <strong>Department:</strong> ${application.department || 'N/A'}<br>
                    <strong>Project:</strong> ${application.project || 'N/A'}
                </p>
            </div>
            <div class="card-footer">
                <span class="status ${application.status ? application.status.toLowerCase() : 'pending'}">Status: ${application.status || 'Pending'}</span>
                <button class="btn-menu">
                    <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <circle cx="12" cy="12" r="1"/>
                        <circle cx="12" cy="5" r="1"/>
                        <circle cx="12" cy="19" r="1"/>
                    </svg>
                </button>
            </div>
        </div>
    `;
    
    return card;
}

// Add event listeners to application cards
function addCardEventListeners() {
    const applicationCards = document.querySelectorAll('.application-card');
    
    applicationCards.forEach(card => {
        card.addEventListener('click', function(e) {
            // Don't trigger if clicking on menu button
            if (e.target.closest('.btn-menu')) return;
            
            const applicationId = this.dataset.applicationId;
            const applications = JSON.parse(sessionStorage.getItem('leaveApplications') || '{}');
            const application = applications[applicationId];
            
            if (application) {
                // Navigate to dedicated application detail page
                window.location.href = `/application-detail?id=${applicationId}&type=leave_application`;
            }
        });
    });

    const menuButtons = document.querySelectorAll('.btn-menu');
    
    menuButtons.forEach(button => {
        button.addEventListener('click', function(e) {
            e.stopPropagation();
            const card = this.closest('.application-card');
            const applicationId = card.dataset.applicationId;
            showActionMenu(this, applicationId);
        });
    });
}

// Update home page counts
function updateHomePageCounts(applications) {
    try {
        const applicationsArray = Array.isArray(applications) ? applications : Object.values(applications);
        
        // Count by status
        const pendingCount = applicationsArray.filter(app => 
            (app.status || 'pending').toLowerCase() === 'pending'
        ).length;
        
        const approvedCount = applicationsArray.filter(app => 
            (app.status || '').toLowerCase() === 'approved'
        ).length;
        
        const rejectedCount = applicationsArray.filter(app => 
            (app.status || '').toLowerCase() === 'rejected'
        ).length;
        
        const totalCount = applicationsArray.length;
        
        console.log('Leave Applications Counts:', {
            total: totalCount,
            pending: pendingCount,
            approved: approvedCount,
            rejected: rejectedCount
        });
        
        // Update counts in localStorage for home page access
        localStorage.setItem('leaveApplicationsCount', JSON.stringify({
            total: totalCount,
            pending: pendingCount,
            approved: approvedCount,
            rejected: rejectedCount,
            lastUpdated: new Date().toISOString()
        }));
        
        // If there's a specific element on the home page, update it
        // This will work if the user navigates back to the home page
        if (window.opener && !window.opener.closed) {
            window.opener.postMessage({
                type: 'leaveApplicationsUpdate',
                counts: {
                    total: totalCount,
                    pending: pendingCount,
                    approved: approvedCount,
                    rejected: rejectedCount
                }
            }, '*');
        }
        
    } catch (error) {
        console.error('Error updating home page counts:', error);
    }
}

// Global variable to track open menu
let currentOpenMenu = null;

// Show action menu
function showActionMenu(button, applicationId) {
    // Close any existing menu
    if (currentOpenMenu) {
        currentOpenMenu.remove();
        currentOpenMenu = null;
    }

    const menu = document.createElement('div');
    menu.className = 'action-menu';
    menu.innerHTML = `
        <div class="action-menu-item" data-action="view">
            <i class="fas fa-eye"></i> View Details
        </div>
        <div class="action-menu-item" data-action="approve">
            <i class="fas fa-check"></i> Approve
        </div>
        <div class="action-menu-item" data-action="reject">
            <i class="fas fa-times"></i> Reject
        </div>
        <div class="action-menu-item action-menu-item-danger" data-action="delete">
            <i class="fas fa-trash"></i> Delete
        </div>
    `;

    // Position menu
    const buttonRect = button.getBoundingClientRect();
    menu.style.position = 'fixed';
    menu.style.top = `${buttonRect.bottom + 5}px`;
    menu.style.left = `${buttonRect.left - 150}px`;
    
    document.body.appendChild(menu);
    currentOpenMenu = menu;

    // Add event listeners to menu items
    menu.querySelectorAll('.action-menu-item').forEach(item => {
        item.addEventListener('click', (e) => {
            e.stopPropagation();
            const action = item.dataset.action;
            handleMenuAction(action, applicationId);
            menu.remove();
            currentOpenMenu = null;
        });
    });

    // Close menu when clicking outside
    setTimeout(() => {
        document.addEventListener('click', function closeMenu(e) {
            if (menu && !menu.contains(e.target)) {
                menu.remove();
                currentOpenMenu = null;
                document.removeEventListener('click', closeMenu);
            }
        });
    }, 10);
}

// Handle menu actions
async function handleMenuAction(action, applicationId) {
    const applications = JSON.parse(sessionStorage.getItem('leaveApplications') || '{}');
    const application = applications[applicationId];

    if (!application) {
        showNotification('Application not found', 'error');
        return;
    }

    switch(action) {
        case 'view':
            showApplicationDetails(application);
            break;
        case 'approve':
            showConfirmDialog('Approve Application', 'Are you sure you want to approve this leave application?', () => {
                updateApplicationStatus(applicationId, 'approved');
            });
            break;
        case 'reject':
            showConfirmDialog('Reject Application', 'Are you sure you want to reject this leave application?', () => {
                updateApplicationStatus(applicationId, 'rejected');
            });
            break;
        case 'delete':
            showConfirmDialog('Delete Application', 'Are you sure you want to delete this application? This action cannot be undone.', () => {
                deleteApplication(applicationId);
            }, true);
            break;
    }
}

// Update application status
async function updateApplicationStatus(applicationId, status) {
    try {
        const statusData = {
            status: status,
            comments: '',
            approverName: 'Admin',
            approverRole: 'ProAdmin',
            reason: `${status} by admin`,
            timestamp: new Date().toISOString()
        };
        
        const response = await fetch(`/api/hr-applications/${applicationId}/status`, {
            method: 'PUT',
            headers: { 
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.content || ''
            },
            body: JSON.stringify(statusData)
        });

        if (!response.ok) {
            throw new Error('Failed to update status');
        }
        
        const result = await response.json();
        if (result.success) {
            showNotification(`Application ${status} successfully`, 'success');
            loadLeaveApplications(); // Reload applications
        } else {
            throw new Error(result.error || 'Failed to update status');
        }
    } catch (error) {
        console.error('Error updating application status:', error);
        showNotification('Failed to update application status', 'error');
    }
}

// Delete application
async function deleteApplication(applicationId) {
    try {
        const response = await fetch(`/api/hr-applications/${applicationId}`, {
            method: 'DELETE',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.content || ''
            }
        });

        if (!response.ok) {
            throw new Error('Failed to delete application');
        }
        
        const result = await response.json();
        if (result.success) {
            showNotification('Application deleted successfully', 'success');
            loadLeaveApplications(); // Reload applications
        } else {
            throw new Error(result.error || 'Failed to delete application');
        }
    } catch (error) {
        console.error('Error deleting application:', error);
        showNotification('Failed to delete application', 'error');
    }
}

// Show application details modal
function showApplicationDetails(application) {
    // Get current user role from localStorage
    const currentUser = JSON.parse(localStorage.getItem('currentUser') || '{}');
    const userRole = (currentUser.role || '').toLowerCase();
    const isSupervisor = userRole === 'supervisor' || userRole === 'supervisormanager';
    const isHrManager = userRole === 'hrmanager' || userRole === 'hr manager';
    const isFinanceManager = userRole === 'financemanager' || userRole === 'finance manager';
    const isProAdmin = userRole === 'proadmin' || userRole === 'pro admin';
    
    console.log('Current user role:', userRole, { isSupervisor, isHrManager, isFinanceManager, isProAdmin });
    
    const startDate = new Date(application.leaveStartDate || application.dateFrom);
    const endDate = new Date(application.leaveEndDate || application.dateTo);
    const reportDate = application.reportDate ? new Date(application.reportDate).toLocaleDateString('en-US', { year: 'numeric', month: 'short', day: 'numeric' }) : 'N/A';
    const duration = Math.ceil((endDate - startDate) / (1000 * 60 * 60 * 24)) + 1;
    const addressWhileOnLeave = application.addressWhileOnLeave || 'N/A';
    const otherContactName = application.otherContactName || 'N/A';
    const otherContactCell = application.otherContactCell || 'N/A';
    
    // Build approval status HTML
    const approvalStatusHtml = buildApprovalStatusHTML(application, {
        isSupervisor,
        isHrManager,
        isFinanceManager,
        isProAdmin,
        applicationType: 'leave_application'
    });
    
    const modal = document.createElement('div');
    modal.className = 'details-modal';
    modal.innerHTML = `
        <div class="details-modal-content" style="max-width: 800px;">
            <div class="details-modal-header">
                <h2>Leave Application Details</h2>
                <button class="details-modal-close">&times;</button>
            </div>
            <div class="details-modal-body">
                <div class="detail-row"><span class="detail-label">App Number:</span><span class="detail-value">${application.applicationNumber || 'N/A'}</span></div>
                <div class="detail-row"><span class="detail-label">Full Name:</span><span class="detail-value">${application.fullName || application.applicantName || 'N/A'}</span></div>
                <div class="detail-row"><span class="detail-label">Employee No:</span><span class="detail-value">${application.employeeNo || application.employeeNumber || 'N/A'}</span></div>
                <div class="detail-row"><span class="detail-label">Department:</span><span class="detail-value">${application.department || 'N/A'}</span></div>
                <div class="detail-row"><span class="detail-label">Country:</span><span class="detail-value">${application.country || 'N/A'}</span></div>
                <div class="detail-row"><span class="detail-label">Project:</span><span class="detail-value">${application.project || 'N/A'}</span></div>
                <div class="detail-row"><span class="detail-label">NRC No:</span><span class="detail-value">${application.nrcNo || 'N/A'}</span></div>
                <div class="detail-row"><span class="detail-label">Leave Start Date:</span><span class="detail-value">${startDate.toLocaleDateString('en-US', { year: 'numeric', month: 'short', day: 'numeric' })}</span></div>
                <div class="detail-row"><span class="detail-label">Leave End Date:</span><span class="detail-value">${endDate.toLocaleDateString('en-US', { year: 'numeric', month: 'short', day: 'numeric' })}</span></div>
                <div class="detail-row"><span class="detail-label">Report Date:</span><span class="detail-value">${reportDate}</span></div>
                <div class="detail-row"><span class="detail-label">Duration:</span><span class="detail-value">${duration} day${duration > 1 ? 's' : ''}</span></div>
                <div class="detail-row"><span class="detail-label">Reason:</span><span class="detail-value" style="white-space: pre-wrap;">${application.reasonOfLeave || application.reason || 'N/A'}</span></div>
                <div class="detail-row"><span class="detail-label">Address While On Leave:</span><span class="detail-value">${addressWhileOnLeave}</span></div>
                <div class="detail-row"><span class="detail-label">Other Contact Name:</span><span class="detail-value">${otherContactName}</span></div>
                <div class="detail-row"><span class="detail-label">Other Contact Cell:</span><span class="detail-value">${otherContactCell}</span></div>
                <div class="detail-row"><span class="detail-label">Submitted At:</span><span class="detail-value">${new Date(application.submittedAt || application.createdAt).toLocaleString('en-US', { year: 'numeric', month: 'short', day: 'numeric', hour: '2-digit', minute: '2-digit' })}</span></div>
                
                <hr style="margin: 20px 0; border: none; border-top: 1px solid #e0e0e0;">
                
                ${approvalStatusHtml}
            </div>
        </div>
    `;

    document.body.appendChild(modal);

    // Close modal handlers
    modal.querySelector('.details-modal-close').addEventListener('click', () => modal.remove());
    modal.addEventListener('click', (e) => {
        if (e.target === modal) modal.remove();
    });
    
    // Add event listeners for approval buttons
    attachApprovalListeners(modal, application);
}

// Show confirm dialog
function showConfirmDialog(title, message, onConfirm, isDanger = false) {
    const dialog = document.createElement('div');
    dialog.className = 'confirm-dialog';
    dialog.innerHTML = `
        <div class="confirm-dialog-content">
            <h3>${title}</h3>
            <p>${message}</p>
            <div class="confirm-dialog-buttons">
                <button class="btn-cancel">Cancel</button>
                <button class="btn-confirm ${isDanger ? 'btn-danger' : ''}">${isDanger ? 'Delete' : 'Confirm'}</button>
            </div>
        </div>
    `;

    document.body.appendChild(dialog);

    dialog.querySelector('.btn-cancel').addEventListener('click', () => dialog.remove());
    dialog.querySelector('.btn-confirm').addEventListener('click', () => {
        onConfirm();
        dialog.remove();
    });
    dialog.addEventListener('click', (e) => {
        if (e.target === dialog) dialog.remove();
    });
}

function showNotification(message, type = 'info') {
    const notification = document.createElement('div');
    notification.className = `app-notification app-notification-${type}`;
    notification.textContent = message;
    
    notification.style.cssText = `
        position: fixed;
        top: 80px;
        right: 20px;
        padding: 12px 20px;
        border-radius: 8px;
        color: white;
        font-weight: 500;
        font-size: 14px;
        z-index: 10001;
        opacity: 0;
        transform: translateX(100%);
        transition: all 0.3s ease;
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
        font-family: 'Inter', sans-serif;
    `;
    
    if (type === 'success') {
        notification.style.background = 'linear-gradient(135deg, #10b981 0%, #059669 100%)';
    } else if (type === 'error') {
        notification.style.background = 'linear-gradient(135deg, #ef4444 0%, #dc2626 100%)';
    } else if (type === 'warning') {
        notification.style.background = 'linear-gradient(135deg, #f59e0b 0%, #d97706 100%)';
    } else {
        notification.style.background = 'linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%)';
    }
    
    document.body.appendChild(notification);
    
    setTimeout(() => {
        notification.style.opacity = '1';
        notification.style.transform = 'translateX(0)';
    }, 100);
    
    setTimeout(() => {
        notification.style.opacity = '0';
        notification.style.transform = 'translateX(100%)';
        setTimeout(() => {
            if (notification.parentNode) {
                document.body.removeChild(notification);
            }
        }, 300);
    }, 3000);
}

// ============ APPROVAL SYSTEM FUNCTIONS ============

// Build approval status HTML timeline
function buildApprovalStatusHTML(application, userContext) {
    const { isSupervisor, isHrManager, isFinanceManager, isProAdmin, applicationType } = userContext;
    
    let html = '<div style="margin-top: 30px;"><h3 style="color: #0055a5; margin-bottom: 20px;">Approval Status</h3>';
    
    // Supervisor Section
    if (application.Supervisor || isSupervisor || isProAdmin) {
        const supervisorStatus = application.Supervisor?.status || 'Pending';
        const supervisorName = application.Supervisor?.name && application.Supervisor.name !== 'Pending:Supervisor' ? application.Supervisor.name : 'N/A';
        const supervisorReason = application.Supervisor?.reason || application.Supervisor?.comments || '';
        const canApproveAsSupervisor = (isSupervisor || isProAdmin) && (!application.Supervisor || !application.Supervisor.status || application.Supervisor.status === 'Pending');
        
        html += `
            <div style="border-left: 3px solid ${supervisorStatus === 'Approved' ? '#34C759' : supervisorStatus === 'Rejected' ? '#FF3B30' : '#e0e0e0'}; padding-left: 15px; margin-bottom: 20px;">
                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 8px;">
                    <strong style="color: #333; font-size: 16px;">Supervisor${isProAdmin ? ' (Admin can approve)' : ''}</strong>
                    <span style="font-size: 14px; color: #666;">${supervisorName}${isSupervisor ? ' ⭐' : ''}${isProAdmin && !isSupervisor ? ' 👑' : ''}</span>
                </div>
                <div style="margin-bottom: 8px;">
                    <span class="status ${supervisorStatus.toLowerCase()}" style="display: inline-block;">${supervisorStatus}</span>
                    ${supervisorReason ? `<div style="margin-top: 8px; padding: 10px; background: #f9f9f9; border-radius: 4px; font-size: 14px; color: #666;">💬 ${supervisorReason}</div>` : ''}
                </div>
                ${canApproveAsSupervisor ? `
                    <div style="margin-top: 12px;">
                        <button class="approval-btn approve-btn" data-role="Supervisor" data-action="Approved" style="background: #34C759; color: white; border: none; padding: 8px 16px; border-radius: 4px; margin-right: 8px; cursor: pointer; font-weight: 600;">✓ Approve</button>
                        <button class="approval-btn reject-btn" data-role="Supervisor" data-action="Rejected" style="background: #FF3B30; color: white; border: none; padding: 8px 16px; border-radius: 4px; cursor: pointer; font-weight: 600;">✕ Reject</button>
                    </div>
                ` : ''}
            </div>
        `;
    }
    
    // HR Manager Section
    if (application.HrManager || isHrManager || isProAdmin) {
        const hrStatus = application.HrManager?.status || 'Pending';
        const hrName = application.HrManager?.name && application.HrManager.name !== 'Pending:HrManager' ? application.HrManager.name : 'N/A';
        const hrReason = application.HrManager?.reason || application.HrManager?.comments || '';
        const supervisorApproved = application.Supervisor?.status === 'Approved';
        const canApproveAsHrManager = (isHrManager || isProAdmin) && (!application.HrManager || !application.HrManager.status || application.HrManager.status === 'Pending');
        
        html += `
            <div style="border-left: 3px solid ${hrStatus === 'Approved' ? '#34C759' : hrStatus === 'Rejected' ? '#FF3B30' : '#e0e0e0'}; padding-left: 15px; margin-bottom: 20px;">
                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 8px;">
                    <strong style="color: #333; font-size: 16px;">HR Manager${isProAdmin ? ' (Admin can approve)' : ''}</strong>
                    <span style="font-size: 14px; color: #666;">${hrName}${isHrManager ? ' ⭐' : ''}${isProAdmin && !isHrManager ? ' 👑' : ''}</span>
                </div>
                <div style="margin-bottom: 8px;">
                    <span class="status ${hrStatus.toLowerCase()}" style="display: inline-block;">${hrStatus}</span>
                    ${hrReason ? `<div style="margin-top: 8px; padding: 10px; background: #f9f9f9; border-radius: 4px; font-size: 14px; color: #666;">💬 ${hrReason}</div>` : ''}
                </div>
                ${!supervisorApproved && (isHrManager || isProAdmin) && !isProAdmin ? `<div style="margin-top: 12px; color: #F57C00; font-style: italic; font-size: 14px;">⏳ Waiting for Supervisor approval</div>` : ''}
                ${!supervisorApproved && isProAdmin ? `<div style="margin-top: 12px; color: #F57C00; font-style: italic; font-size: 14px;">⏳ Supervisor not approved yet (Admin can approve anyway)</div>` : ''}
                ${canApproveAsHrManager && (supervisorApproved || isProAdmin) ? `
                    <div style="margin-top: 12px;">
                        <button class="approval-btn approve-btn" data-role="HrManager" data-action="Approved" style="background: #34C759; color: white; border: none; padding: 8px 16px; border-radius: 4px; margin-right: 8px; cursor: pointer; font-weight: 600;">✓ Approve</button>
                        <button class="approval-btn reject-btn" data-role="HrManager" data-action="Rejected" style="background: #FF3B30; color: white; border: none; padding: 8px 16px; border-radius: 4px; cursor: pointer; font-weight: 600;">✕ Reject</button>
                    </div>
                ` : ''}
            </div>
        `;
    }
    
    // Finance Manager Section (only for salary_advance_application)
    if (applicationType === 'salary_advance_application' && (application.FinanceManager || isFinanceManager)) {
        const financeStatus = application.FinanceManager?.status || 'Pending';
        const financeName = application.FinanceManager?.name && application.FinanceManager.name !== 'Pending:FinanceManager' ? application.FinanceManager.name : 'N/A';
        const financeReason = application.FinanceManager?.reason || application.FinanceManager?.comments || '';
        const hrApproved = application.HrManager?.status === 'Approved';
        
        html += `
            <div style="border-left: 3px solid ${financeStatus === 'Approved' ? '#34C759' : financeStatus === 'Rejected' ? '#FF3B30' : '#e0e0e0'}; padding-left: 15px; margin-bottom: 20px;">
                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 8px;">
                    <strong style="color: #333; font-size: 16px;">Finance Manager</strong>
                    <span style="font-size: 14px; color: #666;">${financeName}${isFinanceManager ? ' ⭐' : ''}</span>
                </div>
                <div style="margin-bottom: 8px;">
                    <span class="status ${financeStatus.toLowerCase()}" style="display: inline-block;">${financeStatus}</span>
                    ${financeReason ? `<div style="margin-top: 8px; padding: 10px; background: #f9f9f9; border-radius: 4px; font-size: 14px; color: #666;">💬 ${financeReason}</div>` : ''}
                </div>
                ${!hrApproved && isFinanceManager ? `<div style="margin-top: 12px; color: #F57C00; font-style: italic; font-size: 14px;">⏳ Waiting for HR Manager approval</div>` : ''}
                ${isFinanceManager && hrApproved && (!application.FinanceManager || !application.FinanceManager.status || application.FinanceManager.status === 'Pending') ? `
                    <div style="margin-top: 12px;">
                        <button class="approval-btn approve-btn" data-role="FinanceManager" data-action="Approved" style="background: #34C759; color: white; border: none; padding: 8px 16px; border-radius: 4px; margin-right: 8px; cursor: pointer; font-weight: 600;">✓ Approve</button>
                        <button class="approval-btn reject-btn" data-role="FinanceManager" data-action="Rejected" style="background: #FF3B30; color: white; border: none; padding: 8px 16px; border-radius: 4px; cursor: pointer; font-weight: 600;">✕ Reject</button>
                    </div>
                ` : ''}
            </div>
        `;
    }
    
    html += '</div>';
    return html;
}

// Attach approval button listeners
function attachApprovalListeners(modal, application) {
    const approvalButtons = modal.querySelectorAll('.approval-btn');
    
    approvalButtons.forEach(btn => {
        btn.addEventListener('click', (e) => {
            e.stopPropagation();
            const role = btn.dataset.role;
            const action = btn.dataset.action;
            showApprovalModal(application, role, action, modal);
        });
    });
}

// Show approval/rejection modal
function showApprovalModal(application, approverRole, action, parentModal) {
    const isApproval = action === 'Approved';
    
    const reasonModal = document.createElement('div');
    reasonModal.className = 'details-modal';
    reasonModal.style.zIndex = '10001';
    reasonModal.innerHTML = `
        <div class="details-modal-content" style="max-width: 600px;">
            <div class="details-modal-header">
                <h2>${isApproval ? '✓ Approve' : '✕ Reject'} Application</h2>
                <button class="details-modal-close">&times;</button>
            </div>
            <div class="details-modal-body">
                <div style="background: ${isApproval ? '#f0fdf4' : '#fef2f2'}; border-left: 4px solid ${isApproval ? '#34C759' : '#FF3B30'}; padding: 15px; margin-bottom: 20px; border-radius: 4px;">
                    <strong style="color: ${isApproval ? '#166534' : '#991b1b'}; display: block; margin-bottom: 8px;">
                        ${isApproval ? 'Approving' : 'Rejecting'} as ${approverRole === 'HrManager' ? 'HR Manager' : approverRole === 'FinanceManager' ? 'Finance Manager' : approverRole}
                    </strong>
                    <div style="font-size: 14px; color: #666;">
                        <strong>Employee:</strong> ${application.fullName} (#${application.employeeNo})<br>
                        <strong>Application:</strong> ${application.applicationNumber}
                    </div>
                </div>
                
                <div style="margin-bottom: 20px;">
                    <label style="display: block; margin-bottom: 8px; font-weight: 600; color: #333;">
                        Reason ${isApproval ? 'for Approval' : 'for Rejection'} <span style="color: #FF3B30;">*</span>
                    </label>
                    <textarea 
                        id="approvalReason" 
                        style="width: 100%; min-height: 120px; padding: 12px; border: 1px solid #e0e0e0; border-radius: 4px; font-family: inherit; font-size: 14px; resize: vertical;"
                        placeholder="Enter detailed reason (required)..."
                        maxlength="500"
                    ></textarea>
                    <div style="text-align: right; font-size: 12px; color: #999; margin-top: 4px;">
                        <span id="reasonCharCount">0</span>/500 characters
                    </div>
                </div>
                
                <div style="display: flex; gap: 12px; justify-content: flex-end;">
                    <button class="btn-cancel" style="padding: 10px 20px; border: none; border-radius: 4px; cursor: pointer; font-weight: 600;">Cancel</button>
                    <button class="btn-submit" style="padding: 10px 20px; background: ${isApproval ? '#34C759' : '#FF3B30'}; color: white; border: none; border-radius: 4px; cursor: pointer; font-weight: 600;">
                        ${isApproval ? '✓ Approve' : '✕ Reject'}
                    </button>
                </div>
            </div>
        </div>
    `;
    
    document.body.appendChild(reasonModal);
    
    const reasonTextarea = reasonModal.querySelector('#approvalReason');
    const charCount = reasonModal.querySelector('#reasonCharCount');
    
    // Character counter
    reasonTextarea.addEventListener('input', () => {
        charCount.textContent = reasonTextarea.value.length;
    });
    
    // Cancel button
    reasonModal.querySelector('.btn-cancel').addEventListener('click', () => {
        reasonModal.remove();
    });
    
    // Close button
    reasonModal.querySelector('.details-modal-close').addEventListener('click', () => {
        reasonModal.remove();
    });
    
    // Submit button
    reasonModal.querySelector('.btn-submit').addEventListener('click', async () => {
        const reason = reasonTextarea.value.trim();
        
        if (!reason) {
            showNotification('Please enter a reason', 'error');
            return;
        }
        
        reasonModal.querySelector('.btn-submit').disabled = true;
        reasonModal.querySelector('.btn-submit').textContent = 'Processing...';
        
        await submitApprovalDecision(application, approverRole, action, reason);
        
        reasonModal.remove();
        if (parentModal) parentModal.remove();
        
        // Reload applications after 1 second
        setTimeout(() => {
            loadLeaveApplications();
        }, 1000);
    });
    
    // Focus textarea
    setTimeout(() => reasonTextarea.focus(), 100);
}

// Submit approval/rejection decision
async function submitApprovalDecision(application, approverRole, action, reason) {
    try {
        console.log('🔄 Submitting approval decision:', { approverRole, action, reason });
        
        // Get current user data
        const currentUser = JSON.parse(localStorage.getItem('currentUser') || '{}');
        const userName = currentUser.name || currentUser.fullName || approverRole;
        const userUid = currentUser._id || currentUser.id || 'unknown';
        
        // Call MongoDB API to update status
        const response = await fetch(`http://168.231.113.219:5013/hr-applications/${application.id || application._id}/status`, {
            method: 'PUT',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                status: action,
                comments: reason,
                approverUid: userUid,
                approverName: userName,
                approverRole: approverRole,
                reason: reason,
                timestamp: new Date().toISOString()
            }),
        });
        
        if (!response.ok) {
            throw new Error('Failed to update application status');
        }
        
        const result = await response.json();
        console.log('✅ Status update result:', result);
        
        if (result.success) {
            // Send notifications
            setTimeout(async () => {
                try {
                    console.log('🔔 Sending notifications...');
                    
                    if (approverRole === 'Supervisor' && action === 'Approved') {
                        // Supervisor approved → Notify HR Managers
                        console.log('📧 Notifying HR Managers');
                        await fetch('http://168.231.113.219:5027/api/hr-notifications/supervisor-approval', {
                            method: 'POST',
                            headers: { 'Content-Type': 'application/json' },
                            body: JSON.stringify({
                                applicationData: {
                                    _id: application._id || application.id,
                                    id: application.id || application._id,
                                    applicationNumber: application.applicationNumber,
                                    applicationType: application.applicationType || 'leave_application',
                                    fullName: application.fullName,
                                    employeeNo: application.employeeNo,
                                    department: application.department,
                                    country: application.country,
                                    project: application.project,
                                    status: 'Approved by Supervisor',
                                },
                                approverName: userName,
                            }),
                        });
                    } else {
                        // All other cases → Notify Applicant
                        console.log('📧 Notifying applicant');
                        await fetch('http://168.231.113.219:5027/api/hr-notifications/application-decision', {
                            method: 'POST',
                            headers: { 'Content-Type': 'application/json' },
                            body: JSON.stringify({
                                applicationData: {
                                    _id: application._id || application.id,
                                    id: application.id || application._id,
                                    applicationNumber: application.applicationNumber,
                                    applicationType: application.applicationType || 'leave_application',
                                    fullName: application.fullName,
                                    employeeNo: application.employeeNo,
                                    department: application.department,
                                    country: application.country,
                                    project: application.project,
                                },
                                status: action,
                                approverName: userName,
                                approverRole: approverRole,
                                reason: reason,
                                applicantId: application.adminUid || application.userUid,
                            }),
                        });
                    }
                    console.log('✅ Notifications sent');
                } catch (notifError) {
                    console.error('❌ Error sending notifications:', notifError);
                }
            }, 500);
            
            showNotification(`Application ${action.toLowerCase()} successfully`, 'success');
        } else {
            throw new Error(result.error || 'Failed to update status');
        }
    } catch (error) {
        console.error('❌ Error submitting approval:', error);
        showNotification('Failed to update application status', 'error');
    }
}

</script>

<style>
@import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');

.applications-container {
    min-height: 100vh;
}

.applications-header {
    background: #0055a5;
    padding: 16px 20px;
    box-shadow: 0 2px 8px rgba(0, 85, 165, 0.3);
}

.header-content {
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.btn-back {
    display: flex;
    align-items: center;
    justify-content: center;
    width: 40px;
    height: 40px;
    background: rgba(255, 255, 255, 0.1);
    color: white;
    text-decoration: none;
    border-radius: 8px;
    transition: all 0.2s ease;
    border: 1px solid rgba(255, 255, 255, 0.2);
}

.btn-back:hover {
    background: rgba(255, 255, 255, 0.2);
    color: white;
    text-decoration: none;
    transform: translateY(-1px);
}

.page-title h1 {
    color: white;
    font-size: 20px;
    font-weight: 600;
    margin: 0;
    text-align: center;
    flex: 1;
}

.btn-notification {
    display: flex;
    align-items: center;
    justify-content: center;
    width: 40px;
    height: 40px;
    background: rgba(255, 255, 255, 0.1);
    color: white;
    border: 1px solid rgba(255, 255, 255, 0.2);
    border-radius: 8px;
    cursor: pointer;
    transition: all 0.2s ease;
}

.btn-notification:hover {
    background: rgba(255, 255, 255, 0.2);
    transform: translateY(-1px);
}

.applications-content {
    padding: 20px;
}

.section-header {
    display: flex;
    align-items: center;
    justify-content: space-between;
    margin-bottom: 20px;
}

.section-header h2 {
    color: #0055a5;
    font-size: 24px;
    font-weight: 700;
    margin: 0;
}

.btn-lock {
    display: flex;
    align-items: center;
    justify-content: center;
    width: 40px;
    height: 40px;
    background: #f3f4f6;
    color: #6b7280;
    border: 1px solid #d1d5db;
    border-radius: 8px;
    cursor: pointer;
    transition: all 0.2s ease;
}

.btn-lock:hover {
    background: #e5e7eb;
    color: #374151;
}

.applications-list {
    display: flex;
    flex-direction: column;
    gap: 16px;
}

.application-card {
    background: white;
    border-radius: 12px;
    padding: 20px;
    box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
    border: 1px solid #e2e8f0;
    cursor: pointer;
    transition: all 0.2s ease;
}

.application-card:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
}

.card-content {
    display: flex;
    flex-direction: column;
    gap: 12px;
}

.applicant-info h3 {
    font-size: 18px;
    font-weight: 600;
    color: #1f2937;
    margin: 0 0 4px 0;
}

.applicant-info p {
    font-size: 14px;
    color: #6b7280;
    margin: 0;
}

.leave-info p {
    font-size: 14px;
    color: #374151;
    margin: 0;
    font-weight: 500;
}

.card-footer {
    display: flex;
    align-items: center;
    justify-content: space-between;
    margin-top: 8px;
}

.status {
    font-size: 14px;
    font-weight: 600;
    padding: 4px 12px;
    border-radius: 20px;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.status.pending {
    background: #fef3c7;
    color: #d97706;
}

.status.approved {
    background: #d1fae5;
    color: #059669;
}

    .status.rejected {
        background: #fee2e2;
        color: #dc2626;
    }

/* Loading State */
.loading-state {
    display: flex;
    flex-direction: column;
    align-items: center;
    justify-content: center;
    padding: 60px 20px;
    text-align: center;
}

.loading-spinner {
    width: 40px;
    height: 40px;
    border: 4px solid #e5e7eb;
    border-top: 4px solid #0055a5;
    border-radius: 50%;
    animation: spin 1s linear infinite;
    margin-bottom: 16px;
}

@keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}

.loading-state p {
    color: #6b7280;
    font-size: 16px;
    margin: 0;
}

/* Empty State */
.empty-state {
    display: flex;
    flex-direction: column;
    align-items: center;
    justify-content: center;
    padding: 60px 20px;
    text-align: center;
}

.empty-icon {
    color: #9ca3af;
    margin-bottom: 16px;
}

.empty-state h3 {
    color: #374151;
    font-size: 18px;
    font-weight: 600;
    margin: 0 0 8px 0;
}

.empty-state p {
    color: #6b7280;
    font-size: 14px;
    margin: 0;
}

.btn-menu {
    display: flex;
    align-items: center;
    justify-content: center;
    width: 32px;
    height: 32px;
    background: #f3f4f6;
    color: #6b7280;
    border: 1px solid #d1d5db;
    border-radius: 6px;
    cursor: pointer;
    transition: all 0.2s ease;
}

.btn-menu:hover {
    background: #e5e7eb;
    color: #374151;
}

@media (max-width: 768px) {
    .applications-content {
        padding: 16px;
    }
    
    .section-header h2 {
        font-size: 20px;
    }
    
    .application-card {
        padding: 16px;
    }
    
    .applicant-info h3 {
        font-size: 16px;
    }
}

@media (max-width: 576px) {
    .applications-header {
        padding: 12px 16px;
    }
    
    .page-title h1 {
        font-size: 18px;
    }
    
    .applications-content {
        padding: 12px;
    }
    
    .section-header h2 {
        font-size: 18px;
    }
    
    .application-card {
        padding: 12px;
    }
    
    .applicant-info h3 {
        font-size: 15px;
    }
    
    .applicant-info p,
    .leave-info p {
        font-size: 13px;
    }
    
    .status {
        font-size: 12px;
        padding: 3px 8px;
    }
}

/* Action Menu */
.action-menu {
    position: fixed;
    background: white;
    border-radius: 8px;
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
    min-width: 180px;
    z-index: 10000;
    overflow: hidden;
    animation: slideIn 0.2s ease;
}

@keyframes slideIn {
    from {
        opacity: 0;
        transform: translateY(-10px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.action-menu-item {
    padding: 12px 16px;
    cursor: pointer;
    display: flex;
    align-items: center;
    gap: 10px;
    color: #374151;
    font-size: 14px;
    transition: all 0.2s ease;
}

.action-menu-item:hover {
    background: #f3f4f6;
}

.action-menu-item i {
    width: 16px;
    font-size: 14px;
}

.action-menu-item-danger {
    color: #dc2626;
}

.action-menu-item-danger:hover {
    background: #fee2e2;
}

/* Details Modal */
.details-modal {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0, 0, 0, 0.5);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 10002;
    padding: 20px;
}

.details-modal-content {
    background: white;
    border-radius: 12px;
    max-width: 600px;
    width: 100%;
    max-height: 80vh;
    overflow-y: auto;
    box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.1);
}

.details-modal-header {
    display: flex;
    align-items: center;
    justify-content: space-between;
    padding: 20px;
    border-bottom: 1px solid #e5e7eb;
}

.details-modal-header h2 {
    margin: 0;
    font-size: 20px;
    font-weight: 600;
    color: #0055a5;
}

.details-modal-close {
    background: none;
    border: none;
    font-size: 28px;
    color: #6b7280;
    cursor: pointer;
    width: 32px;
    height: 32px;
    display: flex;
    align-items: center;
    justify-content: center;
    border-radius: 4px;
    transition: all 0.2s ease;
}

.details-modal-close:hover {
    background: #f3f4f6;
    color: #374151;
}

.details-modal-body {
    padding: 20px;
}

.detail-row {
    display: flex;
    padding: 12px 0;
    border-bottom: 1px solid #f3f4f6;
}

.detail-row:last-child {
    border-bottom: none;
}

.detail-label {
    font-weight: 600;
    color: #374151;
    min-width: 150px;
    font-size: 14px;
}

.detail-value {
    color: #6b7280;
    flex: 1;
    font-size: 14px;
}

/* Confirm Dialog */
.confirm-dialog {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0, 0, 0, 0.5);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 10003;
    padding: 20px;
}

.confirm-dialog-content {
    background: white;
    border-radius: 12px;
    padding: 24px;
    max-width: 400px;
    width: 100%;
    box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.1);
}

.confirm-dialog-content h3 {
    margin: 0 0 12px 0;
    font-size: 18px;
    font-weight: 600;
    color: #1f2937;
}

.confirm-dialog-content p {
    margin: 0 0 20px 0;
    color: #6b7280;
    font-size: 14px;
    line-height: 1.5;
}

.confirm-dialog-buttons {
    display: flex;
    gap: 12px;
    justify-content: flex-end;
}

.btn-cancel,
.btn-confirm {
    padding: 10px 20px;
    border-radius: 6px;
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.2s ease;
    border: none;
}

.btn-cancel {
    background: #f3f4f6;
    color: #374151;
}

.btn-cancel:hover {
    background: #e5e7eb;
}

.btn-confirm {
    background: #0055a5;
    color: white;
}

.btn-confirm:hover {
    background: #003f7d;
}

.btn-confirm.btn-danger {
    background: #dc2626;
}

.btn-confirm.btn-danger:hover {
    background: #b91c1c;
}

@media (max-width: 640px) {
    .details-modal-content {
        max-width: 100%;
    }
    
    .detail-row {
        flex-direction: column;
        gap: 4px;
    }
    
    .detail-label {
        min-width: auto;
    }
}
</style>

@endsection

