@extends('welcome')

@section('title', 'Logbook')

@section('content')
<div class="logbook-container">
    <!-- Header Section - matching Create New Workorder style -->
    <header class="header-gradient">
        <div class="header-content">
            <div class="header-left">
                <div>
                    <h1 class="header-title">Logbook</h1>
                    <p class="header-subtitle">Track and manage your daily activities, maintenance logs, and operational records</p>
                </div>
            </div>
            <div class="header-right">
                <div class="header-search-box">
                    <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <circle cx="11" cy="11" r="8"></circle>
                        <path d="m21 21-4.35-4.35"></path>
                    </svg>
                    <input type="text" id="searchInput" placeholder="Search entries..." />
                </div>
                <div class="action-buttons">
                    <button onclick="exportToExcel()" class="btn-export">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"></path>
                            <polyline points="7 10 12 15 17 10"></polyline>
                            <line x1="12" y1="15" x2="12" y2="3"></line>
                        </svg>
                        Export
                    </button>
                    <a href="{{ route('logbook.create') }}" class="btn-primary">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <line x1="12" y1="5" x2="12" y2="19"></line>
                            <line x1="5" y1="12" x2="19" y2="12"></line>
                        </svg>
                        Add New Logbook
                    </a>
                </div>
            </div>
        </div>
    </header>
    
    <!-- Logbook Content -->
    <div class="logbook-content">
        <!-- Table Section -->
        <div class="table-container">
            <table class="logbook-table" id="logbookTable">
                <thead>
                    <tr>
                        <th>Entry Number</th>
                        <th>Equipment</th>
                        <th>Country</th>
                        <th>Project</th>
                        <th>Start Date/Time</th>
                        <th>End Date/Time</th>
                        <th>Start Reading</th>
                        <th>End Reading</th>
                        <th>Meter Reading</th>
                        <th>Status</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody id="logbookTableBody">
                    <!-- Data will be populated here -->
                </tbody>
            </table>
            
            <!-- Pagination -->
            <div class="pagination-container" id="paginationContainer" style="display: none;">
                <div class="pagination-info">
                    Showing <span id="showingStart">0</span> to <span id="showingEnd">0</span> of <span id="totalEntries">0</span> entries
                </div>
                <div class="pagination-controls" id="paginationControls">
                    <!-- Pagination buttons will be populated here -->
                </div>
            </div>
            
            <!-- Empty State -->
            <div id="emptyState" class="empty-state" style="display: none;">
                <div class="empty-state-icon">
                    <svg xmlns="http://www.w3.org/2000/svg" width="64" height="64" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round">
                        <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
                        <polyline points="14,2 14,8 20,8"/>
                        <line x1="16" y1="13" x2="8" y2="13"/>
                        <line x1="16" y1="17" x2="8" y2="17"/>
                    </svg>
                </div>
                <h3>No Logbook Entries Found</h3>
                <p>No logbook entries match your search criteria. Try adjusting your search or create a new entry.</p>
            </div>
        </div>
    </div>

    <!-- Detail Modal -->
    <div id="detailModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h2>Logbook Entry Details</h2>
                <button class="close-btn" onclick="closeDetailModal()">&times;</button>
            </div>
            <div class="modal-body" id="modalBody">
                <!-- Details will be populated here -->
            </div>
        </div>
    </div>
</div>

<style>
/* CSS Variables for consistent theming - matching Create New Workorder */
@import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');
:root {
    --primary-color: #0055a5;
    --primary-hover: #003f7d;
    --secondary-color: #6c757d;
    --success-color: #28a745;
    --warning-color: #ffc107;
    --danger-color: #dc3545;
    --info-color: #17a2b8;
    --text-primary: #212529;
    --text-secondary: #6c757d;
    --bg-primary: #ffffff;
    --bg-secondary: #f8f9fa;
    --bg-light: #f8fafc;
    --border-color: #dee2e6;
    --border-light: #e2e8f0;
    --shadow-sm: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
    --shadow-md: 0 0.5rem 1rem rgba(0, 0, 0, 0.15);
    --radius-sm: 0.25rem;
    --radius-md: 0.375rem;
    --radius-lg: 0.5rem;
    --transition-base: all 0.15s ease-in-out;
    --font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
}

/* Reset and base styles */
* {
    margin: 0;
    padding: 0;
    box-sizing: border-box;
}

body {
    font-family: var(--font-family);
    background-color: var(--bg-light);
    color: var(--text-primary);
}

/* Main Container */
.logbook-container {
    min-height: 100vh;
    background: var(--bg-light);
}

/* Header Styles - matching Create New Workorder */
.header-gradient {
    background: #0055a5;
    padding: 20px 0;
    box-shadow: 0 4px 12px rgba(0, 85, 165, 0.3);
    margin-bottom: 30px;
}

.header-content {
    width: 100%;
    padding: 0 20px;
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.header-left {
    display: flex;
    align-items: center;
    gap: 15px;
}

.header-title {
    color: white;
    font-size: 28px;
    font-weight: 700;
    margin: 0;
}

.header-subtitle {
    color: rgba(255, 255, 255, 0.9);
    font-size: 16px;
    margin: 0;
}

.header-right {
    display: flex;
    align-items: center;
    gap: 15px;
    flex-wrap: wrap;
}

/* Header Search Box */
.header-search-box {
    display: flex;
    align-items: center;
    gap: 10px;
    padding: 8px 15px;
    background: rgba(255, 255, 255, 0.2);
    border: 1px solid rgba(255, 255, 255, 0.3);
    border-radius: var(--radius-md);
    min-width: 250px;
}

.header-search-box svg {
    color: white;
    flex-shrink: 0;
}

.header-search-box input {
    flex: 1;
    border: none;
    background: transparent;
    font-size: 14px;
    outline: none;
    font-family: var(--font-family);
    color: white;
}

.header-search-box input::placeholder {
    color: rgba(255, 255, 255, 0.7);
}

/* Action Button Styles */
.action-buttons {
    display: flex;
    gap: 12px;
    align-items: center;
}

.btn-primary {
    display: inline-flex;
    align-items: center;
    gap: 8px;
    padding: 12px 20px;
    background: white;
    color: var(--primary-color);
    border: none;
    border-radius: var(--radius-md);
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    box-shadow: var(--shadow-sm);
    transition: var(--transition-base);
    font-family: var(--font-family);
    text-decoration: none;
}

.btn-primary:hover {
    background: rgba(255, 255, 255, 0.9);
    color: var(--primary-color);
    text-decoration: none;
    transform: translateY(-1px);
    box-shadow: var(--shadow-md);
}

/* Export Button */
.btn-export {
    display: inline-flex;
    align-items: center;
    gap: 8px;
    padding: 12px 20px;
    background: #00B894;
    color: white;
    border: none;
    border-radius: var(--radius-md);
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    box-shadow: var(--shadow-sm);
    transition: var(--transition-base);
    font-family: var(--font-family);
}

.btn-export:hover {
    background: #00a080;
    transform: translateY(-1px);
    box-shadow: var(--shadow-md);
}

/* Content Section */
.logbook-content {
    margin: 0 20px 30px 20px;
}

/* Table Container */
.table-container {
    background: var(--bg-primary);
    border-radius: var(--radius-lg);
    box-shadow: var(--shadow-md);
    overflow: hidden;
    border: 1px solid var(--border-light);
}

/* Table Styles */
.logbook-table {
    width: 100%;
    border-collapse: collapse;
}

.logbook-table thead {
    background: linear-gradient(135deg, #0055a5 0%, #003f7d 100%);
}

.logbook-table th {
    padding: 15px;
    text-align: left;
    font-size: 13px;
    font-weight: 600;
    color: white;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    white-space: nowrap;
}

.logbook-table tbody tr {
    border-bottom: 1px solid var(--border-light);
    transition: var(--transition-base);
}

.logbook-table tbody tr:hover {
    background: var(--bg-light);
}

.logbook-table tbody tr:last-child {
    border-bottom: none;
}

.logbook-table td {
    padding: 15px;
    font-size: 14px;
    color: var(--text-primary);
    white-space: nowrap;
}

/* Status Badge */
.status-badge {
    display: inline-block;
    padding: 5px 12px;
    border-radius: 20px;
    font-size: 12px;
    font-weight: 500;
    text-transform: capitalize;
}

.status-badge.available {
    background: #d4edda;
    color: #155724;
}

.status-badge.in-use {
    background: #fff3cd;
    color: #856404;
}

.status-badge.maintenance {
    background: #f8d7da;
    color: #721c24;
}

/* Action Buttons */
.action-btn {
    padding: 6px 12px;
    border: none;
    border-radius: var(--radius-md);
    font-size: 12px;
    font-weight: 500;
    cursor: pointer;
    transition: var(--transition-base);
    font-family: var(--font-family);
}

.action-btn.view {
    background: var(--primary-color);
    color: white;
}

.action-btn.view:hover {
    background: var(--primary-hover);
}

/* Empty State */
.empty-state {
    padding: 60px 40px;
    text-align: center;
}

.empty-state-icon {
    color: var(--primary-color);
    margin-bottom: 24px;
    opacity: 0.7;
}

.empty-state h3 {
    font-size: 1.5rem;
    font-weight: 600;
    color: var(--primary-color);
    margin: 0 0 16px 0;
    font-family: var(--font-family);
}

.empty-state p {
    color: var(--text-secondary);
    font-size: 1rem;
    line-height: 1.6;
    margin: 0;
}

/* Pagination Styles */
.pagination-container {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 20px;
    border-top: 1px solid var(--border-light);
    flex-wrap: wrap;
    gap: 15px;
}

.pagination-info {
    color: var(--text-secondary);
    font-size: 14px;
}

.pagination-info span {
    font-weight: 600;
    color: var(--text-primary);
}

.pagination-controls {
    display: flex;
    gap: 5px;
    align-items: center;
}

.pagination-btn {
    padding: 8px 12px;
    border: 1px solid var(--border-light);
    background: var(--bg-primary);
    color: var(--text-primary);
    border-radius: var(--radius-md);
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: var(--transition-base);
    font-family: var(--font-family);
    min-width: 36px;
    text-align: center;
}

.pagination-btn:hover:not(:disabled) {
    background: var(--primary-color);
    color: white;
    border-color: var(--primary-color);
}

.pagination-btn.active {
    background: var(--primary-color);
    color: white;
    border-color: var(--primary-color);
}

.pagination-btn:disabled {
    opacity: 0.5;
    cursor: not-allowed;
}

.pagination-ellipsis {
    padding: 8px 12px;
    color: var(--text-secondary);
    font-size: 14px;
}

/* Modal Styles */
.modal {
    display: none;
    position: fixed;
    z-index: 1000;
    left: 0;
    top: 0;
    width: 100%;
    height: 100%;
    background-color: rgba(0, 0, 0, 0.5);
    overflow: auto;
}

.modal-content {
    background-color: var(--bg-primary);
    margin: 5% auto;
    border-radius: var(--radius-lg);
    width: 90%;
    max-width: 800px;
    box-shadow: 0 4px 20px rgba(0, 0, 0, 0.2);
}

.modal-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 20px 30px;
    background: linear-gradient(135deg, #0055a5 0%, #003f7d 100%);
    border-radius: var(--radius-lg) var(--radius-lg) 0 0;
}

.modal-header h2 {
    color: white;
    font-size: 1.5rem;
    margin: 0;
    font-family: var(--font-family);
}

.close-btn {
    background: none;
    border: none;
    color: white;
    font-size: 28px;
    font-weight: bold;
    cursor: pointer;
    line-height: 1;
    padding: 0;
    width: 30px;
    height: 30px;
    display: flex;
    align-items: center;
    justify-content: center;
}

.close-btn:hover {
    opacity: 0.8;
}

.modal-body {
    padding: 30px;
}

.detail-grid {
    display: grid;
    grid-template-columns: repeat(2, 1fr);
    gap: 20px;
}

.detail-item {
    border-bottom: 1px solid var(--border-light);
    padding-bottom: 10px;
}

.detail-label {
    font-size: 12px;
    font-weight: 600;
    color: var(--text-secondary);
    text-transform: uppercase;
    letter-spacing: 0.5px;
    margin-bottom: 5px;
}

.detail-value {
    font-size: 14px;
    color: var(--text-primary);
    word-break: break-word;
}

.detail-item.full-width {
    grid-column: 1 / -1;
}

/* Responsive Design */
@media (max-width: 768px) {
    .header-content {
        flex-direction: column;
        align-items: stretch;
        gap: 20px;
        padding: 0 15px;
    }
    
    .header-left {
        text-align: center;
    }
    
    .header-right {
        flex-direction: column;
        gap: 15px;
        align-items: stretch;
    }
    
    .header-search-box {
        min-width: 100%;
    }
    
    .action-buttons {
        justify-content: center;
        flex-wrap: wrap;
    }
    
    .logbook-content {
        margin: 0 15px 30px 15px;
    }
    
    .detail-grid {
        grid-template-columns: 1fr;
    }
    
    .table-container {
        overflow-x: auto;
    }
    
    .modal-content {
        width: 95%;
        margin: 10% auto;
    }
    
    .pagination-container {
        flex-direction: column;
        align-items: center;
        text-align: center;
    }
    
    .pagination-controls {
        flex-wrap: wrap;
        justify-content: center;
    }
}

@media (max-width: 480px) {
    .header-title {
        font-size: 24px;
    }
    
    .header-subtitle {
        font-size: 14px;
    }
    
    .logbook-content {
        margin: 0 10px 20px 10px;
    }
    
    .empty-state-icon svg {
        width: 48px;
        height: 48px;
    }
    
    .modal-body {
        padding: 20px;
    }
    
    .btn-export,
    .btn-primary {
        width: 100%;
        justify-content: center;
    }
}
</style>

<script>
let allLogbookData = [];
let filteredData = [];
let currentPage = 1;
const itemsPerPage = 10;

// Format date and time
function formatDateTime(dateString) {
    if (!dateString) return '-';
    const date = new Date(dateString);
    const day = String(date.getDate()).padStart(2, '0');
    const month = String(date.getMonth() + 1).padStart(2, '0');
    const year = date.getFullYear();
    const hours = String(date.getHours()).padStart(2, '0');
    const minutes = String(date.getMinutes()).padStart(2, '0');
    return `${day}/${month}/${year} ${hours}:${minutes}`;
}

// Get status class
function getStatusClass(status) {
    if (!status) return '';
    const statusLower = status.toLowerCase();
    if (statusLower.includes('available')) return 'available';
    if (statusLower.includes('use') || statusLower.includes('running')) return 'in-use';
    if (statusLower.includes('maintenance')) return 'maintenance';
    return '';
}

// Fetch data from Firebase
async function fetchLogbookData() {
    const tableBody = document.getElementById('logbookTableBody');
    const emptyState = document.getElementById('emptyState');
    
    emptyState.style.display = 'none';
    
    try {
        const response = await fetch('https://titan-drilling-1f8e9-default-rtdb.firebaseio.com/GlobalLogbook/admins.json');
        const data = await response.json();
        
        console.log('Firebase data:', data);
        
        allLogbookData = [];
        
        if (data && typeof data === 'object') {
            // Iterate through each admin UID folder
            Object.keys(data).forEach(adminUID => {
                const adminData = data[adminUID];
                
                if (adminData && typeof adminData === 'object') {
                    // Iterate through each logbook entry inside the admin folder
                    Object.keys(adminData).forEach(entryKey => {
                        const entry = adminData[entryKey];
                        
                        if (entry && typeof entry === 'object' && entry.id) {
                            allLogbookData.push({
                                ...entry,
                                adminUID: adminUID,
                                entryKey: entryKey
                            });
                        }
                    });
                }
            });
        }
        
        // Sort by created date (newest first)
        allLogbookData.sort((a, b) => {
            const dateA = new Date(a.createdAt || 0);
            const dateB = new Date(b.createdAt || 0);
            return dateB - dateA;
        });
        
        console.log('Processed logbook data:', allLogbookData);
        
        // Display data
        filteredData = [...allLogbookData];
        displayData(filteredData);
        
    } catch (error) {
        console.error('Error fetching logbook data:', error);
        emptyState.style.display = 'block';
        emptyState.querySelector('h3').textContent = 'Error Loading Data';
        emptyState.querySelector('p').textContent = 'Failed to load logbook entries. Please try again later.';
    }
}

// Display data in table with pagination
function displayData(data) {
    const tableBody = document.getElementById('logbookTableBody');
    const emptyState = document.getElementById('emptyState');
    const table = document.getElementById('logbookTable');
    const paginationContainer = document.getElementById('paginationContainer');
    
    tableBody.innerHTML = '';
    
    if (data.length === 0) {
        table.style.display = 'none';
        paginationContainer.style.display = 'none';
        emptyState.style.display = 'block';
        return;
    }
    
    table.style.display = 'table';
    emptyState.style.display = 'none';
    paginationContainer.style.display = 'flex';
    
    // Calculate pagination
    const totalPages = Math.ceil(data.length / itemsPerPage);
    const startIndex = (currentPage - 1) * itemsPerPage;
    const endIndex = Math.min(startIndex + itemsPerPage, data.length);
    const pageData = data.slice(startIndex, endIndex);
    
    // Display data for current page
    pageData.forEach(entry => {
        const row = document.createElement('tr');
        row.innerHTML = `
            <td>${entry.entryNumber || '-'}</td>
            <td>${entry.selectedEquipment || '-'}</td>
            <td>${entry.country || '-'}</td>
            <td>${entry.project || '-'}</td>
            <td>${formatDateTime(entry.startDateTime)}</td>
            <td>${formatDateTime(entry.endDateTime)}</td>
            <td>${entry.startReading || '-'}</td>
            <td>${entry.endReading || '-'}</td>
            <td>${entry.currentMeterReading || '-'}</td>
            <td><span class="status-badge ${getStatusClass(entry.equipmentStatus)}">${entry.equipmentStatus || '-'}</span></td>
            <td><button class="action-btn view" onclick="showDetails('${entry.id}')">View</button></td>
        `;
        tableBody.appendChild(row);
    });
    
    // Update pagination info and controls
    updatePaginationInfo(startIndex + 1, endIndex, data.length);
    updatePaginationControls(totalPages);
}

// Update pagination info text
function updatePaginationInfo(start, end, total) {
    document.getElementById('showingStart').textContent = start;
    document.getElementById('showingEnd').textContent = end;
    document.getElementById('totalEntries').textContent = total;
}

// Update pagination controls
function updatePaginationControls(totalPages) {
    const paginationControls = document.getElementById('paginationControls');
    paginationControls.innerHTML = '';
    
    // Previous button
    const prevBtn = document.createElement('button');
    prevBtn.className = 'pagination-btn';
    prevBtn.textContent = 'Previous';
    prevBtn.disabled = currentPage === 1;
    prevBtn.onclick = () => changePage(currentPage - 1);
    paginationControls.appendChild(prevBtn);
    
    // Page numbers
    const maxVisiblePages = 5;
    let startPage = Math.max(1, currentPage - Math.floor(maxVisiblePages / 2));
    let endPage = Math.min(totalPages, startPage + maxVisiblePages - 1);
    
    if (endPage - startPage < maxVisiblePages - 1) {
        startPage = Math.max(1, endPage - maxVisiblePages + 1);
    }
    
    // First page
    if (startPage > 1) {
        const firstBtn = document.createElement('button');
        firstBtn.className = 'pagination-btn';
        firstBtn.textContent = '1';
        firstBtn.onclick = () => changePage(1);
        paginationControls.appendChild(firstBtn);
        
        if (startPage > 2) {
            const ellipsis = document.createElement('span');
            ellipsis.className = 'pagination-ellipsis';
            ellipsis.textContent = '...';
            paginationControls.appendChild(ellipsis);
        }
    }
    
    // Page number buttons
    for (let i = startPage; i <= endPage; i++) {
        const pageBtn = document.createElement('button');
        pageBtn.className = 'pagination-btn' + (i === currentPage ? ' active' : '');
        pageBtn.textContent = i;
        pageBtn.onclick = () => changePage(i);
        paginationControls.appendChild(pageBtn);
    }
    
    // Last page
    if (endPage < totalPages) {
        if (endPage < totalPages - 1) {
            const ellipsis = document.createElement('span');
            ellipsis.className = 'pagination-ellipsis';
            ellipsis.textContent = '...';
            paginationControls.appendChild(ellipsis);
        }
        
        const lastBtn = document.createElement('button');
        lastBtn.className = 'pagination-btn';
        lastBtn.textContent = totalPages;
        lastBtn.onclick = () => changePage(totalPages);
        paginationControls.appendChild(lastBtn);
    }
    
    // Next button
    const nextBtn = document.createElement('button');
    nextBtn.className = 'pagination-btn';
    nextBtn.textContent = 'Next';
    nextBtn.disabled = currentPage === totalPages;
    nextBtn.onclick = () => changePage(currentPage + 1);
    paginationControls.appendChild(nextBtn);
}

// Change page
function changePage(page) {
    currentPage = page;
    displayData(filteredData);
    // Scroll to top of table
    document.getElementById('logbookTable').scrollIntoView({ behavior: 'smooth', block: 'start' });
}

// Filter data
function filterData() {
    const searchTerm = document.getElementById('searchInput').value.toLowerCase();
    
    filteredData = allLogbookData.filter(entry => {
        return !searchTerm || 
            (entry.entryNumber && entry.entryNumber.toLowerCase().includes(searchTerm)) ||
            (entry.selectedEquipment && entry.selectedEquipment.toLowerCase().includes(searchTerm)) ||
            (entry.country && entry.country.toLowerCase().includes(searchTerm)) ||
            (entry.project && entry.project.toLowerCase().includes(searchTerm)) ||
            (entry.operatorName && entry.operatorName.toLowerCase().includes(searchTerm));
    });
    
    // Reset to first page when filtering
    currentPage = 1;
    displayData(filteredData);
}

// Export to Excel
function exportToExcel() {
    if (filteredData.length === 0) {
        alert('No data to export');
        return;
    }
    
    // Create CSV content
    const headers = ['Entry Number', 'Equipment', 'Country', 'Project', 'Start Date/Time', 'End Date/Time', 'Start Reading', 'End Reading', 'Meter Reading', 'Status', 'Operator', 'Work Type', 'Work Location'];
    const rows = filteredData.map(entry => [
        entry.entryNumber || '',
        entry.selectedEquipment || '',
        entry.country || '',
        entry.project || '',
        formatDateTime(entry.startDateTime),
        formatDateTime(entry.endDateTime),
        entry.startReading || '',
        entry.endReading || '',
        entry.currentMeterReading || '',
        entry.equipmentStatus || '',
        entry.operatorName || '',
        entry.workType || '',
        entry.workLocation || ''
    ]);
    
    let csvContent = headers.join(',') + '\n';
    rows.forEach(row => {
        csvContent += row.map(cell => `"${cell}"`).join(',') + '\n';
    });
    
    // Create download link
    const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
    const link = document.createElement('a');
    const url = URL.createObjectURL(blob);
    link.setAttribute('href', url);
    link.setAttribute('download', `logbook_entries_${new Date().toISOString().split('T')[0]}.csv`);
    link.style.visibility = 'hidden';
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
}

// Show details modal
function showDetails(entryId) {
    const entry = allLogbookData.find(item => item.id === entryId);
    if (!entry) return;
    
    const modal = document.getElementById('detailModal');
    const modalBody = document.getElementById('modalBody');
    
    modalBody.innerHTML = `
        <div class="detail-grid">
            <div class="detail-item">
                <div class="detail-label">Entry Number</div>
                <div class="detail-value">${entry.entryNumber || '-'}</div>
            </div>
            <div class="detail-item">
                <div class="detail-label">Equipment ID</div>
                <div class="detail-value">${entry.equipmentId || '-'}</div>
            </div>
            <div class="detail-item">
                <div class="detail-label">Selected Equipment</div>
                <div class="detail-value">${entry.selectedEquipment || '-'}</div>
            </div>
            <div class="detail-item">
                <div class="detail-label">Equipment Status</div>
                <div class="detail-value"><span class="status-badge ${getStatusClass(entry.equipmentStatus)}">${entry.equipmentStatus || '-'}</span></div>
            </div>
            <div class="detail-item">
                <div class="detail-label">Country</div>
                <div class="detail-value">${entry.country || '-'}</div>
            </div>
            <div class="detail-item">
                <div class="detail-label">Project</div>
                <div class="detail-value">${entry.project || '-'}</div>
            </div>
            <div class="detail-item">
                <div class="detail-label">Start Date/Time</div>
                <div class="detail-value">${formatDateTime(entry.startDateTime)}</div>
            </div>
            <div class="detail-item">
                <div class="detail-label">End Date/Time</div>
                <div class="detail-value">${formatDateTime(entry.endDateTime)}</div>
            </div>
            <div class="detail-item">
                <div class="detail-label">Start Reading</div>
                <div class="detail-value">${entry.startReading || '-'}</div>
            </div>
            <div class="detail-item">
                <div class="detail-label">End Reading</div>
                <div class="detail-value">${entry.endReading || '-'}</div>
            </div>
            <div class="detail-item">
                <div class="detail-label">Current Meter Reading</div>
                <div class="detail-value">${entry.currentMeterReading || '-'}</div>
            </div>
            <div class="detail-item">
                <div class="detail-label">Operator Name</div>
                <div class="detail-value">${entry.operatorName || '-'}</div>
            </div>
            <div class="detail-item">
                <div class="detail-label">Fuel Opening</div>
                <div class="detail-value">${entry.fuelOpening || '-'}</div>
            </div>
            <div class="detail-item">
                <div class="detail-label">Fuel Consumption</div>
                <div class="detail-value">${entry.fuelConsumption || '-'}</div>
            </div>
            <div class="detail-item">
                <div class="detail-label">Fuel Unit</div>
                <div class="detail-value">${entry.fuelUnit || '-'}</div>
            </div>
            <div class="detail-item">
                <div class="detail-label">Material Name</div>
                <div class="detail-value">${entry.materialName || '-'}</div>
            </div>
            <div class="detail-item">
                <div class="detail-label">Material Quantity</div>
                <div class="detail-value">${entry.materialQuantity || '-'}</div>
            </div>
            <div class="detail-item">
                <div class="detail-label">Material Unit</div>
                <div class="detail-value">${entry.materialUnit || '-'}</div>
            </div>
            <div class="detail-item full-width">
                <div class="detail-label">Material Description</div>
                <div class="detail-value">${entry.materialDescription || '-'}</div>
            </div>
            <div class="detail-item full-width">
                <div class="detail-label">Work Type</div>
                <div class="detail-value">${entry.workType || '-'}</div>
            </div>
            <div class="detail-item full-width">
                <div class="detail-label">Work Location</div>
                <div class="detail-value">${entry.workLocation || '-'}</div>
            </div>
            <div class="detail-item full-width">
                <div class="detail-label">Work Description</div>
                <div class="detail-value">${entry.workDescription || '-'}</div>
            </div>
            <div class="detail-item">
                <div class="detail-label">Issue Name</div>
                <div class="detail-value">${entry.issueName || '-'}</div>
            </div>
            <div class="detail-item">
                <div class="detail-label">Issue Report</div>
                <div class="detail-value">${entry.issueReport || '-'}</div>
            </div>
            <div class="detail-item full-width">
                <div class="detail-label">Issue Details</div>
                <div class="detail-value">${entry.issueDetails || '-'}</div>
            </div>
            <div class="detail-item">
                <div class="detail-label">Created At</div>
                <div class="detail-value">${formatDateTime(entry.createdAt)}</div>
            </div>
            <div class="detail-item">
                <div class="detail-label">Updated At</div>
                <div class="detail-value">${formatDateTime(entry.updatedAt)}</div>
            </div>
        </div>
    `;
    
    modal.style.display = 'block';
}

// Close details modal
function closeDetailModal() {
    const modal = document.getElementById('detailModal');
    modal.style.display = 'none';
}

// Event listeners
document.addEventListener('DOMContentLoaded', function() {
    // Fetch data on page load
    fetchLogbookData();
    
    // Search input
    document.getElementById('searchInput').addEventListener('input', filterData);
    
    // Close modal when clicking outside
    window.onclick = function(event) {
        const modal = document.getElementById('detailModal');
        if (event.target === modal) {
            closeDetailModal();
        }
    };
});
</script>
@endsection
