import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  TextInput,
  StatusBar,
  Modal,
  ScrollView,
  Platform,
  Alert,
  Dimensions,
  Animated,
  KeyboardAvoidingView,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import Ionicons from 'react-native-vector-icons/Ionicons';
import AsyncStorage from '@react-native-async-storage/async-storage';
import { useNavigation } from '@react-navigation/native';
import DropDownPicker from 'react-native-dropdown-picker';
import Toast from 'react-native-toast-message';
import NetInfo from '@react-native-community/netinfo';
import DateTimePicker from '@react-native-community/datetimepicker';
import { MONGODB_API_URL } from '../../../config/api.config';

const { width, height } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.25;

// Custom Input Field Component
const CustomInputField = ({
  label,
  value,
  onChangeText,
  isRequired = false,
  showError = false,
  errorMessage = '',
  icon,
  onIconPress,
  editable = true,
  onPress,
  inputRef,
  containerRef,
  ...props
}) => {
  return (
    <View ref={containerRef} style={inputStyles.container}>
      <Text style={inputStyles.label}>
        {label}
        {isRequired && <Text style={inputStyles.requiredError}> *</Text>}
      </Text>
      <TouchableOpacity
        style={[inputStyles.inputContainer, showError ? inputStyles.inputError : null]}
        onPress={onPress}
        disabled={!onPress}
      >
        <TextInput
          ref={inputRef}
          style={[inputStyles.input, { flex: 1 }]}
          value={value}
          onChangeText={onChangeText}
          placeholderTextColor="#888"
          editable={editable}
          pointerEvents={onPress ? 'none' : 'auto'}
          autoCapitalize="none"
          autoCorrect={false}
          {...props}
        />
        {icon && (
          <TouchableOpacity style={inputStyles.iconButton} onPress={onIconPress}>
            {icon}
          </TouchableOpacity>
        )}
      </TouchableOpacity>
      {showError && <Text style={inputStyles.errorText}>{errorMessage}</Text>}
    </View>
  );
};

const Add_equipment = () => {
  const navigation = useNavigation();
  const [mainCategoryModalVisible, setMainCategoryModalVisible] = useState(false);
  const [loadingModalVisible, setLoadingModalVisible] = useState(false);
  const rotateAnim = useRef(new Animated.Value(0)).current;
  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });
  const slideAnim = useRef(new Animated.Value(300)).current;
  const scrollViewRef = useRef(null);
  const lastServiceMeterContainerRef = useRef(null);
  const equipmentNameContainerRef = useRef(null);

  const dynamicField1Ref = useRef(null);
  const dynamicField2Ref = useRef(null);
  const dynamicField3Ref = useRef(null);
  const equipmentNumberRef = useRef(null);
  const equipmentNameRef = useRef(null);
  const meterReadingRef = useRef(null);
  const modelRef = useRef(null);
  const lastServiceMeterRef = useRef(null);
  const lastServiceDateRef = useRef(null);
  const vinRef = useRef(null);
  const fleetValueRef = useRef(null);
  const purchaseDateRef = useRef(null);
  const warrantyExpDateRef = useRef(null);
  const operatorNameRef = useRef(null);

  const [countryOpen, setCountryOpen] = useState(false);
  const [selectedCountry, setSelectedCountry] = useState(null);
  const [isCountrySelectable, setIsCountrySelectable] = useState(false);
  const [countryItems, setCountryItems] = useState([
    { label: 'Zambia', value: 'Zambia' },
    { label: 'DRC', value: 'DRC' },
  ]);

  const [mainCategoryOpen, setMainCategoryOpen] = useState(false);
  const [selectedMainCategory, setSelectedMainCategory] = useState(null);
  const [mainCategoryItems, setMainCategoryItems] = useState([
    { label: 'Light Vehicles', value: 'Light Vehicles' },
    { label: 'Heavy Vehicles', value: 'Heavy Vehicles' },
    { label: 'Rigs', value: 'Rigs' },
    { label: 'Heavy Plant Equipment', value: 'Heavy Plant Equipment' },
    { label: 'Auxiliary Equipment', value: 'Auxiliary Equipment' },
    { label: 'Trailers', value: 'Trailers' },
    { label: 'Lathes', value: 'Lathes' },
  ]);

  const [dynamicField1, setDynamicField1] = useState('');
  const [dynamicField2, setDynamicField2] = useState('');
  const [dynamicField3, setDynamicField3] = useState('');

  const [rigTypeOpen, setRigTypeOpen] = useState(false);
  const [rigType, setRigType] = useState(null);
  const [rigTypeItems, setRigTypeItems] = useState([
    { label: 'Track Mounted', value: 'Track Mounted' },
    { label: 'Trailer Mounted', value: 'Trailer Mounted' },
    { label: 'Truck Mounted', value: 'Truck Mounted' },
  ]);

  const [meterOpen, setMeterOpen] = useState(false);
  const [meterUnit, setMeterUnit] = useState('Miles');
  const [meterItems, setMeterItems] = useState([
    { label: 'Miles', value: 'Miles' },
    { label: 'KM', value: 'Kilometers' },
    { label: 'Hours', value: 'Hours' },
  ]);

  const [priorityOpen, setPriorityOpen] = useState(false);
  const [maintenancePriority, setMaintenancePriority] = useState('1');
  const [priorityItems, setPriorityItems] = useState([
    { label: '1', value: '1' },
    { label: '2', value: '2' },
    { label: '3', value: '3' },
  ]);

  const [ownershipOpen, setOwnershipOpen] = useState(false);
  const [ownershipMode, setOwnershipMode] = useState('Owned Asset');
  const [ownershipItems, setOwnershipItems] = useState([
    { label: 'Owned Asset', value: 'Owned Asset' },
    { label: 'Leased', value: 'Leased' },
    { label: 'Rented', value: 'Rented' },
  ]);

  const [optionalFieldsVisible, setOptionalFieldsVisible] = useState(false);
  const [errors, setErrors] = useState({});
  const [formData, setFormData] = useState({
    equipmentNumber: '',
    equipmentName: '',
    meterReading: '',
    model: '',
    lastServiceMeter: '',
    lastServiceDate: '',
    vin: '',
    fleetValue: '',
    purchaseDate: '',
    warrantyExpDate: '',
    operatorName: '',
  });

  // Date picker states
  const [lastServiceDate, setLastServiceDate] = useState(null);
  const [purchaseDate, setPurchaseDate] = useState(null);
  const [warrantyExpDate, setWarrantyExpDate] = useState(null);
  const [tempDate, setTempDate] = useState(new Date());
  const [datePickerVisible, setDatePickerVisible] = useState(false);
  const [currentDateField, setCurrentDateField] = useState('');
  const [showDatePicker, setShowDatePicker] = useState(false);
  const [datePickerValue, setDatePickerValue] = useState(new Date());
  const [currentEditingDateField, setCurrentEditingDateField] = useState('');
  const [datePickerMaxDate, setDatePickerMaxDate] = useState(null);
  const [datePickerMinDate, setDatePickerMinDate] = useState(new Date(1900, 0, 1));

  const uniqueEquipmentNumber = `EQ-${Math.floor(Date.now() / 1000)}`;

  // Toast helper function
  const showToast = (type, text1, text2) => {
    Toast.show({ type, text1, text2, position: 'bottom' });
  };

  // Date formatting function
  const formatDate = (date) => {
    if (!date || isNaN(date.getTime())) return 'Select Date';
    return date.toLocaleDateString('en-GB', {
      day: '2-digit',
      month: 'short',
      year: 'numeric',
    }).replace(/ /g, '-');
  };

  // Open date picker for specific field
  const openDatePicker = (field) => {
    setCurrentDateField(field);
    let initialDate;
    switch (field) {
      case 'lastServiceDate':
        initialDate = lastServiceDate || new Date();
        break;
      case 'purchaseDate':
        initialDate = purchaseDate || new Date();
        break;
      case 'warrantyExpDate':
        initialDate = warrantyExpDate || new Date();
        break;
      default:
        initialDate = new Date();
    }
    setTempDate(initialDate);
    setDatePickerVisible(true);
  };

  // New date picker functions for iOS compatibility
  const onDateChange = (event, selectedDate) => {
    if (Platform.OS === 'android') {
      setShowDatePicker(false);
      if (event.type === 'set' && selectedDate) {
        setDatePickerValue(selectedDate);
        handleDatePickerDone();
      }
    } else {
      // iOS - just update the value, don't close yet
      if (selectedDate) {
        setDatePickerValue(selectedDate);
      }
    }
  };

  const showMode = (currentMode, field) => {
    try {
      setCurrentEditingDateField(field);
      
      // Set appropriate date limits based on field
      const today = new Date();
      switch (field) {
        case 'lastServiceDate':
          setDatePickerMaxDate(today);
          setDatePickerMinDate(new Date(1900, 0, 1));
          setDatePickerValue(lastServiceDate || today);
          break;
        case 'purchaseDate':
          setDatePickerMaxDate(today);
          setDatePickerMinDate(new Date(1900, 0, 1));
          setDatePickerValue(purchaseDate || today);
          break;
        case 'warrantyExpDate':
          setDatePickerMaxDate(new Date(today.getFullYear() + 10, 11, 31));
          setDatePickerMinDate(today);
          setDatePickerValue(warrantyExpDate || today);
          break;
        default:
          setDatePickerMaxDate(null);
          setDatePickerMinDate(new Date(1900, 0, 1));
          setDatePickerValue(today);
      }
      
      setShowDatePicker(true);
    } catch (error) {
      console.error('Error in showMode:', error);
      showToast('error', 'Date Picker Error', 'Failed to open date picker. Please try again.');
    }
  };

  const handleDatePickerDone = () => {
    try {
      const selectedDate = datePickerValue;
      if (!selectedDate || isNaN(selectedDate.getTime())) {
        showToast('error', 'Date Error', 'Please select a valid date');
        return;
      }

      switch (currentEditingDateField) {
        case 'lastServiceDate':
          setLastServiceDate(selectedDate);
          setFormData(prevData => ({ ...prevData, lastServiceDate: formatDate(selectedDate) }));
          break;
        case 'purchaseDate':
          setPurchaseDate(selectedDate);
          setFormData(prevData => ({ ...prevData, purchaseDate: formatDate(selectedDate) }));
          break;
        case 'warrantyExpDate':
          setWarrantyExpDate(selectedDate);
          setFormData(prevData => ({ ...prevData, warrantyExpDate: formatDate(selectedDate) }));
          break;
      }
      
      setShowDatePicker(false);
      setCurrentEditingDateField('');
      setDatePickerMaxDate(null);
      setDatePickerMinDate(new Date(1900, 0, 1));
    } catch (error) {
      console.error('Error in handleDatePickerDone:', error);
      showToast('error', 'Date Error', 'Failed to save date. Please try again.');
    }
  };

  const showDatePickerDialog = (field) => {
    try {
      showMode('date', field);
    } catch (error) {
      console.error('Error in showDatePickerDialog:', error);
      showToast('error', 'Date Picker Error', 'Failed to open date picker. Please try again.');
    }
  };

  // Handle date confirmation
  const handleDateConfirm = () => {
    switch (currentDateField) {
      case 'lastServiceDate':
        setLastServiceDate(tempDate);
        setFormData(prevData => ({ ...prevData, lastServiceDate: formatDate(tempDate) }));
        break;
      case 'purchaseDate':
        setPurchaseDate(tempDate);
        setFormData(prevData => ({ ...prevData, purchaseDate: formatDate(tempDate) }));
        break;
      case 'warrantyExpDate':
        setWarrantyExpDate(tempDate);
        setFormData(prevData => ({ ...prevData, warrantyExpDate: formatDate(tempDate) }));
        break;
    }
    setDatePickerVisible(false);
    setCurrentDateField('');
  };

  // Handle date cancellation
  const handleDateCancel = () => {
    setDatePickerVisible(false);
    setCurrentDateField('');
  };

  useEffect(() => {
    const loadCountry = async () => {
      try {
        const storedCountry = await AsyncStorage.getItem('selectedCountry');
        if (storedCountry && ['Zambia', 'DRC', 'Both Countries'].includes(storedCountry)) {
          setSelectedCountry(storedCountry === 'Both Countries' ? 'Zambia' : storedCountry);
          setIsCountrySelectable(storedCountry === 'Both Countries');
        } else {
          setIsCountrySelectable(true);
          Toast.show({
            type: 'error',
            text1: 'Error',
            text2: 'Please select a country.',
            position: 'bottom',
          });
        }
      } catch (error) {
        console.error('Error loading country:', error);
        setIsCountrySelectable(true);
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: 'Failed to load country.',
          position: 'bottom',
        });
      }
    };
    loadCountry();
  }, []);

  const getFieldMappings = (category) => {
    switch (category) {
      case 'Light Vehicles':
      case 'Heavy Vehicles':
        return {
          field1Key: 'vehicleNumber',
          field2Key: 'registrationNumber',
          field3Key: 'vinNumber',
          field1Label: 'Vehicle Number',
          field2Label: 'Registration Number',
          field3Label: 'VIN Number',
        };
      case 'Rigs':
        return {
          field1Key: 'rigType',
          field2Key: 'chassisNumber',
          field3Key: 'serialNumber',
          field1Label: 'Rig Type',
          field2Label: 'Chassis Number',
          field3Label: 'Serial Number',
        };
      case 'Heavy Plant Equipment':
        return {
          field1Key: 'equipmentNumberDynamic',
          field2Key: 'equipmentNameDynamic',
          field3Key: 'vinNumber',
          field1Label: 'Equipment Number',
          field2Label: 'Equipment Name',
          field3Label: 'VIN Number',
        };
      case 'Auxiliary Equipment':
        return {
          field1Key: 'type',
          field2Key: 'modelDynamic',
          field3Key: 'serialNumber',
          field1Label: 'Type',
          field2Label: 'Model',
          field3Label: 'Serial Number',
        };
      case 'Trailers':
        return {
          field1Key: 'type',
          field2Key: 'chassisNumber',
          field3Key: 'chassisNumberDuplicate',
          field1Label: 'Type',
          field2Label: 'Chassis Number',
          field3Label: 'Chassis Number',
        };
      case 'Lathes':
        return {
          field1Key: 'type',
          field2Key: 'modelDynamic',
          field3Key: 'serialNumber',
          field1Label: 'Type',
          field2Label: 'Model',
          field3Label: 'Serial Number',
        };
      default:
        return {
          field1Key: 'field1',
          field2Key: 'field2',
          field3Key: 'field3',
          field1Label: 'Field 1',
          field2Label: 'Field 2',
          field3Label: 'Field 3',
        };
    }
  };

  useEffect(() => {
    if (selectedMainCategory) {
      setDynamicField1('');
      setDynamicField2('');
      setDynamicField3('');
      if (selectedMainCategory === 'Rigs') {
        setRigType(null);
      }
    }
  }, [selectedMainCategory]);

  const checkNetworkLoader = async () => {
    try {
      const netInfo = await NetInfo.fetch();
      return netInfo.isConnected;
    } catch (error) {
      console.error('Error checking network status:', error);
      return false;
    }
  };

  const showLoadingModal = () => {
    setLoadingModalVisible(true);
    Animated.loop(
      Animated.timing(rotateAnim, {
        toValue: 1,
        duration: 1000,
        useNativeDriver: true,
      })
    ).start();
  };

  const hideLoadingModal = () => {
    setLoadingModalVisible(false);
    rotateAnim.setValue(0);
  };

  const handleInputChange = (name, value) => {
    setFormData({ ...formData, [name]: value });
    setErrors({ ...errors, [name]: '' });
  };

  const showEquipmentInfoAlert = () => {
    Alert.alert(
      'Information',
      'E.g., Plate Number, Serial Number, any Company Unique number, etc.',
      [{ text: 'OK', style: 'cancel' }],
      { cancelable: true }
    );
  };

  const handleSave = async () => {
    const requiredFields = {
      country: 'Country',
      mainCategory: 'Main Category',
      dynamicField1: 'Field 1',
      dynamicField2: 'Field 2',
    };

    const nonMandatoryCategories = ['Light Vehicles', 'Heavy Vehicles', 'Rigs'];

    if (!nonMandatoryCategories.includes(selectedMainCategory)) {
      requiredFields.equipmentNumber = 'Equipment Number';
      requiredFields.equipmentName = 'Equipment Name';
    }

    const newErrors = {};
    if (!selectedCountry) {
      newErrors.country = isCountrySelectable
        ? 'Please select a country'
        : 'Please select a country in Home screen';
    }
    if (!selectedMainCategory) {
      newErrors.mainCategory = 'Please select a main category';
    }
    if (!dynamicField1 && selectedMainCategory !== 'Rigs') {
      newErrors.dynamicField1 = `Please enter ${getFieldMappings(selectedMainCategory).field1Label}`;
    }
    if (!rigType && selectedMainCategory === 'Rigs') {
      newErrors.dynamicField1 = `Please select ${getFieldMappings(selectedMainCategory).field1Label}`;
    }
    if (!dynamicField2) {
      newErrors.dynamicField2 = `Please enter ${getFieldMappings(selectedMainCategory).field2Label}`;
    }

    Object.entries(requiredFields).forEach(([key, label]) => {
      if (
        key !== 'country' &&
        key !== 'mainCategory' &&
        key !== 'dynamicField1' &&
        key !== 'dynamicField2' &&
        !formData[key]
      ) {
        newErrors[key] = `Please fill "${label}"`;
      }
    });

    if (Object.keys(newErrors).length > 0) {
      setErrors(newErrors);
      showToast('error', 'Error', 'Please fill all required fields.');
      return;
    }

    const fieldMappings = getFieldMappings(selectedMainCategory);
    const equipmentDataToSave = {
      ...formData,
      country: selectedCountry,
      meterUnit,
      id: uniqueEquipmentNumber,
      maintenancePriority,
      ownershipMode,
      mainCategory: selectedMainCategory,
      [fieldMappings.field1Key]: selectedMainCategory === 'Rigs' ? rigType : dynamicField1,
      [fieldMappings.field2Key]: dynamicField2,
      [fieldMappings.field3Key]: dynamicField3 || 'N/A',
      createdAt: new Date().toISOString(),
      updatedAt: new Date().toISOString(),
      status: 'Available',
      equipmentCategory: selectedMainCategory,
      firebaseKey: uniqueEquipmentNumber,
      databaseName: 'titan_drilling',
    };

    showLoadingModal();

    try {
      const isOnline = await checkNetworkLoader();

      if (isOnline) {
        // Save to MongoDB via API
        const response = await fetch(`${MONGODB_API_URL}/add`, {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
          },
          body: JSON.stringify(equipmentDataToSave),
        });

        if (!response.ok) {
          throw new Error(`HTTP error! status: ${response.status}`);
        }

        const result = await response.json();
        console.log('Equipment saved to MongoDB:', result);

        // Save to local storage
        const existing = await AsyncStorage.getItem('equipment');
        const equipment = existing ? JSON.parse(existing) : [];
        equipment.push(result.data || equipmentDataToSave);
        await AsyncStorage.setItem('equipment', JSON.stringify(equipment));

        hideLoadingModal();
        showToast('success', 'Success', 'Equipment saved successfully to MongoDB!');
        navigation.goBack();
      } else {
        // Offline - save locally
        const existing = await AsyncStorage.getItem('equipment');
        const equipment = existing ? JSON.parse(existing) : [];
        equipment.push({ ...equipmentDataToSave, pendingSync: true });
        await AsyncStorage.setItem('equipment', JSON.stringify(equipment));

        hideLoadingModal();
        showToast('info', 'Offline', 'Equipment saved locally. Will sync when online.');
        navigation.goBack();
      }
    } catch (error) {
      console.error('Error saving equipment:', error);
      
      // Save locally on error
      const existing = await AsyncStorage.getItem('equipment');
      const equipment = existing ? JSON.parse(existing) : [];
      equipment.push({ ...equipmentDataToSave, pendingSync: true });
      await AsyncStorage.setItem('equipment', JSON.stringify(equipment));
      
      hideLoadingModal();
      showToast('info', 'Offline', 'Equipment saved locally. Will sync when online.');
      navigation.goBack();
    }
  };

  useEffect(() => {
    const syncPendingEquipment = async () => {
      const isOnline = await checkNetworkLoader();
      if (!isOnline) return;

      try {
        const existing = await AsyncStorage.getItem('equipment');
        const equipment = existing ? JSON.parse(existing) : [];
        const pending = equipment.filter((item) => item.pendingSync);

        if (pending.length > 0) {
          showLoadingModal();

          // Sync each pending item to MongoDB
          for (const item of pending) {
            try {
              const response = await fetch(`${MONGODB_API_URL}/add`, {
                method: 'POST',
                headers: {
                  'Content-Type': 'application/json',
                },
                body: JSON.stringify({ ...item, pendingSync: false }),
              });

              if (!response.ok) {
                console.error('Failed to sync item:', item.id);
              }
            } catch (syncError) {
              console.error('Error syncing item:', item.id, syncError);
            }
          }

          // Remove pendingSync flag from all items
          const updatedEquipment = equipment.map((item) =>
            item.pendingSync ? { ...item, pendingSync: false } : item
          );
          await AsyncStorage.setItem('equipment', JSON.stringify(updatedEquipment));

          hideLoadingModal();
          showToast('success', 'Success', 'Pending equipment synced to MongoDB!');
        }
      } catch (error) {
        console.error('Error syncing pending equipment:', error);
        hideLoadingModal();
        showToast('error', 'Error', 'Failed to sync pending equipment.');
      }
    };

    const unsubscribe = NetInfo.addEventListener((state) => {
      if (state.isConnected) {
        syncPendingEquipment();
      }
    });

    return () => unsubscribe();
  }, []);

  const handleRefresh = () => {
    setFormData({
      equipmentNumber: '',
      equipmentName: '',
      meterReading: '',
      model: '',
      lastServiceMeter: '',
      lastServiceDate: '',
      vin: '',
      fleetValue: '',
      purchaseDate: '',
      warrantyExpDate: '',
      operatorName: '',
    });
    setLastServiceDate(null);
    setPurchaseDate(null);
    setWarrantyExpDate(null);
    setMeterUnit('Miles');
    setMaintenancePriority('1');
    setOwnershipMode('Owned Asset');
    setSelectedMainCategory(null);
    setDynamicField1('');
    setDynamicField2('');
    setDynamicField3('');
    setRigType(null);
    setErrors({});
    setOptionalFieldsVisible(false);
    slideAnim.setValue(300);
    if (isCountrySelectable) {
      setSelectedCountry(null);
    }
  };

  const selectMainCategory = (category) => {
    setSelectedMainCategory(category);
    setMainCategoryModalVisible(false);
    setErrors({ ...errors, mainCategory: '' });
  };

  const handleMainCategoryPlusClick = () => {
    if (!selectedCountry) {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: isCountrySelectable
          ? 'Please select a country'
          : 'Please select a country in Home screen',
        position: 'bottom',
      });
    } else {
      setMainCategoryModalVisible(true);
    }
  };

  const toggleOptionalFields = () => {
    const willBeVisible = !optionalFieldsVisible;
    setOptionalFieldsVisible(willBeVisible);

    if (lastServiceMeterRef.current) {
      lastServiceMeterRef.current.blur();
    }
    Animated.timing(slideAnim, {
      toValue: willBeVisible ? 0 : 300,
      duration: 400,
      useNativeDriver: true,
    }).start();

    setTimeout(() => {
      if (!scrollViewRef.current) return;

      if (willBeVisible) {
        if (lastServiceMeterContainerRef.current) {
          lastServiceMeterContainerRef.current.measureLayout(
            scrollViewRef.current.getScrollResponder(),
            (x, y) => scrollViewRef.current.scrollTo({ y, animated: true }),
            (error) => console.error('measureLayout error:', error)
          );
        }
      } else {
        if (equipmentNameContainerRef.current) {
          equipmentNameContainerRef.current.measureLayout(
            scrollViewRef.current.getScrollResponder(),
            (x, y) => scrollViewRef.current.scrollTo({ y, animated: true }),
            (error) => console.error('measureLayout error:', error)
          );
        }
      }
    }, 400);
  };

  const { field1Label, field2Label, field3Label } = selectedMainCategory
    ? getFieldMappings(selectedMainCategory)
    : { field1Label: 'Field 1', field2Label: 'Field 2', field3Label: 'Field 3' };

  return (
    <View style={styles.container}>
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity onPress={() => navigation.goBack()}>
              <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>Add Equipment</Text>
          </View>
          <View style={styles.headerRight}>
            <TouchableOpacity onPress={handleRefresh} style={styles.iconButton}>
              <MaterialIcons name="refresh" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <TouchableOpacity onPress={handleSave}>
              <View style={styles.savebtn01}>
                <Text style={styles.saveText}>Save</Text>
              </View>
            </TouchableOpacity>
          </View>
        </View>
      </LinearGradient>

      <KeyboardAvoidingView
        behavior={Platform.OS === 'ios' ? 'padding' : 'height'}
        style={styles.keyboardAvoidingView}
      >
        <ScrollView
          ref={scrollViewRef}
          style={styles.mainContent}
          keyboardShouldPersistTaps="always"
          showsVerticalScrollIndicator={false}
        >
          <View style={styles.equipmentIdRow}>
            <Text style={styles.equipmentIdLabel}>Equipment#</Text>
            <Text style={styles.equipmentIdValue}>{uniqueEquipmentNumber}</Text>
          </View>

          {errors.general && (
            <Text style={styles.generalErrorText}>{errors.general}</Text>
          )}

          <View style={styles.inputContainer}>
            <View style={inputStyles.container}>
              <Text style={inputStyles.label}>
                Country <Text style={inputStyles.requiredError}>*</Text>
              </Text>
              <DropDownPicker
                open={countryOpen}
                value={selectedCountry}
                items={countryItems}
                setOpen={setCountryOpen}
                setValue={setSelectedCountry}
                setItems={setCountryItems}
                placeholder="Select Country"
                style={[styles.dropdown, !isCountrySelectable && styles.disabledDropdown]}
                dropDownContainerStyle={styles.dropdownContainer}
                textStyle={styles.dropdownText}
                placeholderStyle={styles.placeholderStyle}
                disabled={!isCountrySelectable}
                disabledStyle={{ opacity: 0.6 }}
                zIndex={5000}
                dropDownDirection="AUTO"
                onChangeValue={() => setErrors({ ...errors, country: '' })}
              />
              {errors.country && (
                <Text style={inputStyles.errorText}>{errors.country}</Text>
              )}
            </View>

            <CustomInputField
              label="Main Category"
              value={selectedMainCategory || ''}
              onChangeText={() => {}}
              isRequired
              showError={!!errors.mainCategory}
              errorMessage={errors.mainCategory}
              editable={false}
              placeholder="Select Main Category"
              icon={<Ionicons name="add" size={width * 0.05} color="#015185" />}
              onIconPress={handleMainCategoryPlusClick}
              onPress={handleMainCategoryPlusClick}
            />

            {selectedMainCategory && (
              <>
                {selectedMainCategory === 'Rigs' ? (
                  <View style={inputStyles.container}>
                    <Text style={inputStyles.label}>
                      {field1Label} <Text style={inputStyles.requiredError}>*</Text>
                    </Text>
                    <DropDownPicker
                      open={rigTypeOpen}
                      value={rigType}
                      items={rigTypeItems}
                      setOpen={setRigTypeOpen}
                      setValue={setRigType}
                      setItems={setRigTypeItems}
                      placeholder={`Select ${field1Label}`}
                      style={[styles.dropdown, errors.dynamicField1 ? inputStyles.inputError : null]}
                      dropDownContainerStyle={styles.dropdownContainer}
                      textStyle={styles.dropdownText}
                      placeholderStyle={styles.placeholderStyle}
                      zIndex={4000}
                      dropDownDirection="AUTO"
                      onChangeValue={() => setErrors({ ...errors, dynamicField1: '' })}
                    />
                    {errors.dynamicField1 && (
                      <Text style={inputStyles.errorText}>{errors.dynamicField1}</Text>
                    )}
                  </View>
                ) : (
                  <CustomInputField
                    label={field1Label}
                    value={dynamicField1}
                    onChangeText={(text) => {
                      setDynamicField1(text);
                      setErrors({ ...errors, dynamicField1: '' });
                    }}
                    isRequired
                    showError={!!errors.dynamicField1}
                    errorMessage={errors.dynamicField1}
                    placeholder={`Enter ${field1Label}`}
                    inputRef={dynamicField1Ref}
                    returnKeyType="next"
                    onSubmitEditing={() => dynamicField2Ref.current?.focus()}
                  />
                )}

                <CustomInputField
                  label={field2Label}
                  value={dynamicField2}
                  onChangeText={(text) => {
                    setDynamicField2(text);
                    setErrors({ ...errors, dynamicField2: '' });
                  }}
                  isRequired
                  showError={!!errors.dynamicField2}
                  errorMessage={errors.dynamicField2}
                  placeholder={`Enter ${field2Label}`}
                  inputRef={dynamicField2Ref}
                  returnKeyType="next"
                  onSubmitEditing={() => dynamicField3Ref.current?.focus()}
                />

                <CustomInputField
                  label={`${field3Label} (Optional)`}
                  value={dynamicField3}
                  onChangeText={(text) => setDynamicField3(text)}
                  placeholder={`Enter ${field3Label}`}
                  inputRef={dynamicField3Ref}
                  returnKeyType="next"
                  onSubmitEditing={() => equipmentNumberRef.current?.focus()}
                />

                <CustomInputField
                  label="Equipment #"
                  value={formData.equipmentNumber}
                  onChangeText={(text) => handleInputChange('equipmentNumber', text)}
                  isRequired={
                    !['Light Vehicles', 'Heavy Vehicles', 'Rigs'].includes(selectedMainCategory)
                  }
                  showError={!!errors.equipmentNumber}
                  errorMessage={errors.equipmentNumber}
                  icon={<MaterialIcons name="help-outline" size={width * 0.05} color="#015185" />}
                  onIconPress={showEquipmentInfoAlert}
                  placeholder="Enter Equipment Number"
                  inputRef={equipmentNumberRef}
                  returnKeyType="next"
                  onSubmitEditing={() => equipmentNameRef.current?.focus()}
                />

                <CustomInputField
                  label="Equipment Name"
                  value={formData.equipmentName}
                  onChangeText={(text) => handleInputChange('equipmentName', text)}
                  isRequired={
                    !['Light Vehicles', 'Heavy Vehicles', 'Rigs'].includes(selectedMainCategory)
                  }
                  showError={!!errors.equipmentName}
                  errorMessage={errors.equipmentName}
                  placeholder="Equipment Name"
                  inputRef={equipmentNameRef}
                  containerRef={equipmentNameContainerRef}
                  returnKeyType="next"
                  onSubmitEditing={() => meterReadingRef.current?.focus()}
                />

                <View style={inputStyles.container}>
                  <Text style={inputStyles.label}>Meter Reading</Text>
                  <View style={quantityStyles.row}>
                    <View style={[inputStyles.inputContainer, errors.meterReading ? inputStyles.inputError : null, quantityStyles.inputWrapper]}>
                      <TextInput
                        ref={meterReadingRef}
                        style={[inputStyles.input, { flex: 1 }]}
                        placeholder="Enter Meter Reading"
                        keyboardType="numeric"
                        value={formData.meterReading}
                        onChangeText={(text) => handleInputChange('meterReading', text)}
                        placeholderTextColor="#888"
                        autoCapitalize="none"
                        autoCorrect={false}
                        returnKeyType="next"
                        onSubmitEditing={() => modelRef.current?.focus()}
                      />
                    </View>
                    <View style={quantityStyles.unitBox}>
                      <DropDownPicker
                        open={meterOpen}
                        value={meterUnit}
                        items={meterItems}
                        setOpen={setMeterOpen}
                        setValue={setMeterUnit}
                        setItems={setMeterItems}
                        style={[styles.dropdown, quantityStyles.unitDropdown]}
                        dropDownContainerStyle={styles.dropdownContainer}
                        textStyle={styles.dropdownText}
                        placeholderStyle={styles.placeholderStyle}
                        zIndex={2000}
                        dropDownDirection="AUTO"
                      />
                    </View>
                  </View>
                  {errors.meterReading && (
                    <Text style={inputStyles.errorText}>{errors.meterReading}</Text>
                  )}
                </View>

                <CustomInputField
                  label="Model"
                  value={formData.model}
                  onChangeText={(text) => handleInputChange('model', text)}
                  showError={!!errors.model}
                  errorMessage={errors.model}
                  placeholder="Enter Model Number"
                  inputRef={modelRef}
                  returnKeyType="next"
                  onSubmitEditing={() => lastServiceMeterRef.current?.focus()}
                />
              </>
            )}
            <View style={styles.optionalRow}>
              <TouchableOpacity onPress={toggleOptionalFields} activeOpacity={0.7}>
                <Text style={styles.optionalText}>Optional Fields</Text>
              </TouchableOpacity>
              <TouchableOpacity onPress={toggleOptionalFields} activeOpacity={0.7}>
                <MaterialIcons
                  name={optionalFieldsVisible ? 'arrow-drop-up' : 'arrow-drop-down'}
                  size={width * 0.05}
                  color="#888"
                />
              </TouchableOpacity>
            </View>

            {optionalFieldsVisible && (
              <Animated.View
                style={[
                  styles.optionalFieldsContainer,
                  {
                    transform: [{ translateY: slideAnim }],
                    opacity: slideAnim.interpolate({
                      inputRange: [0, 300],
                      outputRange: [1, 0],
                    }),
                  },
                ]}
              >
                <CustomInputField
                  label="Last Service Meter Reading"
                  value={formData.lastServiceMeter}
                  onChangeText={(text) => handleInputChange('lastServiceMeter', text)}
                  keyboardType="numeric"
                  placeholder="Last Service Meter Reading"
                  inputRef={lastServiceMeterRef}
                  containerRef={lastServiceMeterContainerRef}
                  returnKeyType="next"
                  onSubmitEditing={() => lastServiceDateRef.current?.focus()}
                />

                <CustomInputField
                  label="Last Service Date"
                  value={formatDate(lastServiceDate)}
                  onChangeText={() => {}}
                  placeholder="Select Last Service Date"
                  editable={false}
                  onPress={() => showDatePickerDialog('lastServiceDate')}
                  inputRef={lastServiceDateRef}
                  icon={<MaterialIcons name="calendar-today" size={width * 0.05} color="#015185" />}
                  onIconPress={() => showDatePickerDialog('lastServiceDate')}
                  returnKeyType="next"
                  onSubmitEditing={() => vinRef.current?.focus()}
                />

                <CustomInputField
                  label="VIN #"
                  value={formData.vin}
                  onChangeText={(text) => handleInputChange('vin', text)}
                  placeholder="VIN #"
                  inputRef={vinRef}
                  returnKeyType="next"
                  onSubmitEditing={() => fleetValueRef.current?.focus()}
                />

                <CustomInputField
                  label="Fleet Value"
                  value={formData.fleetValue}
                  onChangeText={(text) => handleInputChange('fleetValue', text)}
                  keyboardType="numeric"
                  placeholder="Fleet Value"
                  inputRef={fleetValueRef}
                  returnKeyType="next"
                  onSubmitEditing={() => purchaseDateRef.current?.focus()}
                />

                <CustomInputField
                  label="Purchase Date"
                  value={formatDate(purchaseDate)}
                  onChangeText={() => {}}
                  placeholder="Select Purchase Date"
                  editable={false}
                  onPress={() => showDatePickerDialog('purchaseDate')}
                  inputRef={purchaseDateRef}
                  icon={<MaterialIcons name="calendar-today" size={width * 0.05} color="#015185" />}
                  onIconPress={() => showDatePickerDialog('purchaseDate')}
                  returnKeyType="next"
                  onSubmitEditing={() => warrantyExpDateRef.current?.focus()}
                />

                <CustomInputField
                  label="Warranty Exp Date"
                  value={formatDate(warrantyExpDate)}
                  onChangeText={() => {}}
                  placeholder="Select Warranty Exp Date"
                  editable={false}
                  onPress={() => showDatePickerDialog('warrantyExpDate')}
                  inputRef={warrantyExpDateRef}
                  icon={<MaterialIcons name="calendar-today" size={width * 0.05} color="#015185" />}
                  onIconPress={() => showDatePickerDialog('warrantyExpDate')}
                  returnKeyType="next"
                  onSubmitEditing={() => operatorNameRef.current?.focus()}
                />

                <CustomInputField
                  label="Operator Name"
                  value={formData.operatorName}
                  onChangeText={(text) => handleInputChange('operatorName', text)}
                  placeholder="Operator Name"
                  inputRef={operatorNameRef}
                  returnKeyType="done"
                />

                <View style={[inputStyles.container, { flexDirection: 'row', alignItems: 'center' }]}>
                  <Text style={[inputStyles.label, { flex: 1 }]}>Maintenance Priority</Text>
                  <View style={styles.dropdownWrapper}>
                    <DropDownPicker
                      open={priorityOpen}
                      value={maintenancePriority}
                      items={priorityItems}
                      setOpen={setPriorityOpen}
                      setValue={setMaintenancePriority}
                      setItems={setPriorityItems}
                      style={[styles.dropdownButton, { width: width * 0.2 }]}
                      dropDownContainerStyle={[styles.dropdownContainer, { maxHeight: height * 0.15 }]}
                      textStyle={styles.dropdownText}
                      placeholderStyle={styles.placeholderStyle}
                      zIndex={1000}
                      arrowIconStyle={{ tintColor: '#888' }}
                      dropDownDirection="TOP"
                    />
                  </View>
                </View>

                <View style={[inputStyles.container, { flexDirection:'row', alignItems: 'center' }]}>
                  <Text style={[inputStyles.label, { flex: 1 }]}>Ownership Mode</Text>
                  <View style={styles.dropdownWrapper}>
                    <DropDownPicker
                      open={ownershipOpen}
                      value={ownershipMode}
                      items={ownershipItems}
                      setOpen={setOwnershipOpen}
                      setValue={setOwnershipMode}
                      setItems={setOwnershipItems}
                      style={[styles.dropdownButton, { width: width * 0.3 }]}
                      dropDownContainerStyle={[styles.dropdownContainer, { maxHeight: height * 0.15 }]}
                      textStyle={styles.dropdownText}
                      placeholderStyle={styles.placeholderStyle}
                      zIndex={900}
                      arrowIconStyle={{ tintColor: '#888' }}
                      dropDownDirection="TOP"
                    />
                  </View>
                </View>
              </Animated.View>
            )}
            <View style={{ height: width * 0.3 }} />
          </View>
        </ScrollView>
      </KeyboardAvoidingView>

      <Modal
        visible={mainCategoryModalVisible}
        transparent={true}
        animationType="slide"
        onRequestClose={() => setMainCategoryModalVisible(false)}
      >
        <View style={styles.modalContainer}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>Select Main Category</Text>
            {mainCategoryItems.map((item) => (
              <TouchableOpacity
                key={item.value}
                style={styles.modalItem}
                onPress={() => selectMainCategory(item.value)}
              >
                <Text style={styles.modalItemText}>{item.label}</Text>
              </TouchableOpacity>
            ))}
            <TouchableOpacity
              style={styles.modalCloseButton}
              onPress={() => setMainCategoryModalVisible(false)}
            >
              <Text style={styles.modalCloseText}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      <Modal
        visible={loadingModalVisible}
        transparent={true}
        animationType="fade"
      >
        <View style={styles.syncModalContainer}>
          <View style={styles.syncModalContent}>
            <Text style={styles.syncModalText}>Please Wait</Text>
            <View style={styles.syncIcondiv}>
              <Animated.Image
                source={require('../../../Images/adminlogin/sync.png')}
                style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
              />
              <Text style={styles.syncModalTextadd}>Processing Equipment Data</Text>
            </View>
          </View>
        </View>
      </Modal>

      {/* iOS Date Picker Modal */}
      <Modal
        visible={showDatePicker}
        animationType="slide"
        transparent={true}
        onRequestClose={() => {
          setShowDatePicker(false);
          setCurrentEditingDateField('');
          setDatePickerMaxDate(null);
          setDatePickerMinDate(new Date(1900, 0, 1));
        }}
        presentationStyle="pageSheet"
      >
        <View style={styles.datePickerContainer}>
          <View style={styles.datePickerHeader}>
            <Text style={styles.datePickerTitle}>
              Select {currentEditingDateField ? currentEditingDateField.replace(/([A-Z])/g, ' $1').replace(/^./, str => str.toUpperCase()) : 'Date'}
            </Text>
            <TouchableOpacity
              style={styles.datePickerCloseButton}
              onPress={handleDatePickerDone}
            >
              <Text style={styles.datePickerCloseText}>Done</Text>
            </TouchableOpacity>
          </View>
          <View style={styles.datePickerContent}>
            <DateTimePicker
              testID="dateTimePicker"
              value={datePickerValue}
              mode="date"
              display="spinner"
              onChange={onDateChange}
              maximumDate={datePickerMaxDate}
              minimumDate={datePickerMinDate}
              accentColor="#025383"
              textColor="#333"
              style={styles.datePicker}
              themeVariant="light"
            />
          </View>
        </View>
      </Modal>

      {/* Native Android Date Picker - Only for Android */}
      {Platform.OS === 'android' && showDatePicker && (
        <DateTimePicker
          testID="dateTimePicker"
          value={datePickerValue}
          mode="date"
          display="default"
          onChange={onDateChange}
          maximumDate={datePickerMaxDate}
          minimumDate={datePickerMinDate}
          accentColor="#025383"
          textColor="#333"
        />
      )}
      <Toast />
    </View>
  );
};

const inputStyles = StyleSheet.create({
  container: {
    marginBottom: width * 0.04,
  },
  label: {
    fontSize: width * 0.038,
    color: '#015185',
    marginBottom: width * 0.01,
    fontWeight: '500',
  },
  requiredError: {
    color: 'red',
    fontWeight: 'bold',
  },
  inputContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: width * 0.012,
    backgroundColor: '#fff',
    minHeight: width * 0.12,
  },
  input: {
    paddingHorizontal: width * 0.03,
    paddingVertical: width * 0.035,
    fontSize: width * 0.04,
    color: '#000',
    minHeight: width * 0.12,
  },
  inputError: {
    borderColor: 'red',
  },
  errorText: {
    color: 'red',
    fontSize: width * 0.032,
    marginTop: width * 0.005,
    marginLeft: width * 0.005,
  },
  iconButton: {
    padding: width * 0.025,
  },
});

const quantityStyles = StyleSheet.create({
  row: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  inputWrapper: {
    flex: 1,
    marginRight: width * 0.02,
  },
  unitBox: {
    width: width * 0.25,
  },
  unitDropdown: {
    height: width * 0.12,
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: width * 0.012,
    backgroundColor: '#fff',
    paddingHorizontal: width * 0.02,
    minHeight: width * 0.12,
  },
});

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  keyboardAvoidingView: {
    flex: 1,
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.11,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.05,
    fontWeight: '400',
    marginLeft: width * 0.025,
  },
  saveText: {
    color: '#0055A5',
    fontSize: width * 0.045,
    fontWeight: '600',
  },
  savebtn01: {
    backgroundColor: 'white',
    paddingHorizontal: width * 0.035,
    paddingVertical: width * 0.007,
    borderRadius: width * 0.007,
  },
  mainContent: {
    flex: 1,
    padding: width * 0.04,
  },
  equipmentIdRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    backgroundColor: '#F2F2F2',
    borderRadius: width * 0.015,
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.035,
    marginBottom: width * 0.04,
  },
  equipmentIdLabel: {
    fontSize: width * 0.038,
    color: '#015185',
    fontWeight: '500',
  },
  equipmentIdValue: {
    fontSize: width * 0.038,
    color: '#025383',
    fontWeight: 'bold',
  },
  generalErrorText: {
    color: '#d32f2f',
    fontSize: width * 0.035,
    marginBottom: width * 0.04,
    textAlign: 'center',
  },
  inputContainer: {
    width: '100%',
  },
  dropdown: {
    backgroundColor: '#fff',
    borderRadius: width * 0.012,
    borderWidth: 1,
    borderColor: '#015185',
    height: width * 0.12,
    paddingHorizontal: width * 0.03,
  },
  disabledDropdown: {
    backgroundColor: '#e0e0e0',
  },
  dropdownContainer: {
    backgroundColor: '#fff',
    borderRadius: width * 0.012,
    borderWidth: 1,
    borderColor: '#015185',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 2,
    maxHeight: height * 0.25,
  },
  dropdownButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    backgroundColor: '#fff',
    height: width * 0.12,
    borderRadius: width * 0.012,
    paddingHorizontal: width * 0.03,
    borderWidth: 1,
    borderColor: '#015185',
  },
  dropdownText: {
    fontSize: width * 0.04,
    color: '#333',
  },
  placeholderStyle: {
    fontSize: width * 0.04,
    color: '#888',
  },
  iconButton: {
    padding: width * 0.012,
  },
  optionalRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    marginTop: width * 0.025,
    paddingVertical: width * 0.025,
    borderTopWidth: 1,
    borderTopColor: '#eee',
  },
  optionalText: {
    fontSize: width * 0.04,
    color: '#888',
  },
  optionalFieldsContainer: {
    marginTop: width * 0.025,
    borderTopWidth: 1,
    borderTopColor: '#eee',
    paddingTop: width * 0.025,
  },
  modalContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: 'rgba(0,0,0,0.5)',
  },
  modalContent: {
    width: width * 0.9,
    backgroundColor: 'white',
    borderRadius: width * 0.025,
    padding: width * 0.05,
    maxHeight: height * 0.8,
  },
  modalTitle: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    marginBottom: width * 0.037,
  },
  modalItem: {
    padding: width * 0.025,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
  },
  modalItemText: {
    fontSize: width * 0.04,
  },
  modalCloseButton: {
    marginTop: width * 0.037,
    alignItems: 'center',
  },
  modalCloseText: {
    color: '#3481BC',
    fontSize: width * 0.04,
    fontWeight: 'bold',
  },
  dropdownWrapper: {
    position: 'relative',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: width * 0.7,
    backgroundColor: 'white',
    borderRadius: width * 0.025,
    padding: width * 0.05,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: width * 0.045,
    marginBottom: width * 0.025,
    color: '#333',
  },
  syncIcon: {
    width: width * 0.075,
    height: width * 0.075,
    marginRight: width * 0.025,
  },
  syncModalTextadd: {
    fontSize: width * 0.035,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  modalOverlay: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: 'rgba(0,0,0,0.5)',
  },
  datePickerContainer: {
    backgroundColor: '#fff',
    borderRadius: width * 0.025,
    marginHorizontal: width * 0.05,
    marginVertical: Platform.OS === 'ios' ? width * 0.125 : width * 0.25,
    maxHeight: Platform.OS === 'ios' ? '80%' : undefined,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.25,
    shadowRadius: 4,
    overflow: 'hidden',
  },
  datePicker: {
    width: Platform.OS === 'ios' ? '100%' : width * 0.75,
    height: Platform.OS === 'ios' ? width * 0.5 : undefined,
  },
  datePickerCloseButton: {
    paddingVertical: width * 0.02,
    paddingHorizontal: width * 0.04,
    backgroundColor: '#015185',
    borderRadius: width * 0.015,
    minWidth: width * 0.15,
    alignItems: 'center',
  },
  datePickerCloseText: {
    color: '#fff',
    fontSize: width * 0.035,
    fontWeight: 'bold',
  },
  datePickerHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    padding: width * 0.0375,
    borderBottomWidth: 1,
    borderBottomColor: '#e0e0e0',
    backgroundColor: '#f8f9fa',
  },
  datePickerTitle: {
    fontSize: width * 0.04,
    fontWeight: 'bold',
    color: '#015185',
    flex: 1,
  },
  datePickerContent: {
    padding: width * 0.05,
    alignItems: 'center',
    justifyContent: 'center',
  },
  datePickerButtons: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    width: '100%',
    marginTop: 10,
  },
  datePickerButton: {
    flex: 1,
    borderRadius: 5,
    paddingVertical: 10,
    alignItems: 'center',
    marginHorizontal: 5,
  },
  cancelButton: {
    backgroundColor: '#A0A0A0',
  },
  doneButton: {
    backgroundColor: '#015185',
  },
  datePickerButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: 'bold',
  },
});

export default Add_equipment;