const express = require('express');
const { mongoose } = require('../dbConnection'); // Use shared connection
const cors = require('cors');
const nodemailer = require('nodemailer');
const http = require('http');

const app = express();

// Middleware
app.use(cors());
app.use(express.json());

// MongoDB configuration
const MONGO_HOST = process.env.MONGO_HOST || 'api.titandrillingzm.com';
const MONGO_PORT = process.env.MONGO_PORT || '27017';
const MONGO_USER = process.env.MONGO_USER || 'titandrillingadminuser';
const MONGO_PASSWORD = process.env.MONGO_PASSWORD || 'Titandrilling_@2024';
const MONGO_DB = process.env.MONGO_DB || 'titan_drilling';

// Using shared MongoDB connection from dbConnection.js
console.log('✅ [DVIR Notifications] Using shared MongoDB connection');

// GlobalUser Schema
const GlobalUserSchema = new mongoose.Schema({
  _id: String,
  name: String,
  email: String,
  role: String,
  countries: [String],
  country: String,
  project: String,
  fcmTokens: [{
    token: String,
    device: String,
    updatedAt: Date
  }],
  fcmToken: String,
}, { strict: false, collection: 'GlobalUsers' });

// Check if model exists to prevent OverwriteModelError
let GlobalUser;
try {
  GlobalUser = mongoose.model('GlobalUser');
  console.log('✅ [DVIRNotifications] Using existing GlobalUser model');
} catch (error) {
  GlobalUser = mongoose.model('GlobalUser', GlobalUserSchema);
  console.log('✅ [DVIRNotifications] Created new GlobalUser model');
}

// Email transporter configuration
const emailTransporter = nodemailer.createTransport({
  host: 'titandrillingzm.com',
  port: 587,
  secure: false,
  auth: {
    user: 'info@titandrillingzm.com',
    pass: 'L!A3Ys1Am(N(b)jU',
  },
});

// Helper function to send push notifications
const sendPushNotifications = async (dvirData, createdByName, recipients) => {
  try {
    console.log('📱 Sending push notifications for DVIR report...');
    
    const notificationData = {
      userIds: recipients.map(r => r._id),
      title: '🚛 New DVIR Report',
      body: `${dvirData.dvirNumber}\nEquipment: ${dvirData.equipmentName || 'N/A'}\nProject: ${dvirData.project || 'N/A'}\n\nInspected by: ${createdByName}`,
      data: {
        type: 'dvir_report',
        action: 'view_dvir',
        dvirId: dvirData._id || dvirData.id,
        requestNumber: dvirData.requestNumber,
        dvirNumber: dvirData.dvirNumber,
        equipmentName: dvirData.equipmentName || '',
        project: dvirData.project || '',
        country: dvirData.country || '',
        inspector: createdByName,
        timestamp: new Date().toISOString(),
      },
      // iOS specific settings
      apns: {
        payload: {
          aps: {
            alert: {
              title: '🚛 New DVIR Report',
              body: `${dvirData.dvirNumber}\nEquipment: ${dvirData.equipmentName || 'N/A'}\n\nInspected by: ${createdByName}`
            },
            sound: 'default',
            badge: 1,
            'content-available': 1,
          }
        }
      },
      // Android specific settings
      android: {
        priority: 'high',
        notification: {
          sound: 'default',
          channelId: 'dvir_notifications',
        }
      }
    };
    
    // Send push via PushNotificationHandler (port 5016)
    const result = await new Promise((resolve, reject) => {
      const postData = JSON.stringify(notificationData);
      const options = {
        hostname: 'api.titandrillingzm.com',
        port: 5016,
        path: '/push-notifications/send-to-users',
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Content-Length': Buffer.byteLength(postData),
        },
      };
      
      const req = http.request(options, (res) => {
        let data = '';
        res.on('data', (chunk) => { data += chunk; });
        res.on('end', () => {
          try {
            const response = JSON.parse(data);
            resolve(response);
          } catch (e) {
            reject(new Error('Invalid JSON response from push service'));
          }
        });
      });
      
      req.on('error', (error) => {
        reject(error);
      });
      
      req.write(postData);
      req.end();
    });
    
    console.log(`✅ Push notifications sent to ${result.successCount || 0} device(s)`);
    return result;
  } catch (error) {
    console.error('❌ Error sending push notifications:', error);
    return { success: false, error: error.message };
  }
};

// Helper function to save notifications to MongoDB
const saveNotificationsToMongoDB = async (dvirData, createdByName, recipients) => {
  try {
    console.log('💾 Saving DVIR notifications to MongoDB...');
    
    const notifications = recipients.map(recipient => ({
      notificationId: `dvir_${Date.now()}_${Math.random().toString(36).substr(2, 9)}_${recipient._id}`,
      userId: recipient._id,
      title: '🚛 New DVIR Report',
      body: `${dvirData.dvirNumber}\nEquipment: ${dvirData.equipmentName || 'N/A'}\nProject: ${dvirData.project || 'N/A'}\n\nInspected by: ${createdByName}`,
      type: 'dvir_report',
      timestamp: Date.now(),
      isRead: false,
      data: {
        type: 'dvir_report',
        action: 'view_dvir',
        dvirId: dvirData._id || dvirData.id,
        requestNumber: dvirData.requestNumber,
        dvirNumber: dvirData.dvirNumber,
        equipmentName: dvirData.equipmentName || '',
        project: dvirData.project || '',
        country: dvirData.country || '',
        inspector: createdByName,
        faultyItems: dvirData.faultyItems || 0,
        timestamp: new Date().toISOString(),
      }
    }));
    
    // Save to MongoDB via Notifications API (port 5017)
    const result = await new Promise((resolve, reject) => {
      const postData = JSON.stringify({ notifications });
      const options = {
        hostname: 'api.titandrillingzm.com',
        port: 5017,
        path: '/api/notifications/bulk',
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Content-Length': Buffer.byteLength(postData),
        },
      };
      
      const req = http.request(options, (res) => {
        let data = '';
        res.on('data', (chunk) => { data += chunk; });
        res.on('end', () => {
          try {
            const response = JSON.parse(data);
            resolve(response);
          } catch (e) {
            reject(new Error('Invalid JSON response from notifications service'));
          }
        });
      });
      
      req.on('error', (error) => {
        reject(error);
      });
      
      req.write(postData);
      req.end();
    });
    
    console.log(`✅ Saved ${result.count || notifications.length} notification(s) to MongoDB`);
    return result;
  } catch (error) {
    console.error('❌ Error saving notifications to MongoDB:', error);
    return { success: false, error: error.message };
  }
};

// Helper function to send email to inspector with PDF attachment
const sendInspectorEmail = async (dvirData, createdByName, inspectorEmail, pdfUrl) => {
  try {
    if (!inspectorEmail) {
      console.log('⚠️ No email address for inspector, skipping inspector email');
      return { success: false, error: 'No email address' };
    }
    
    console.log(`📧 Sending confirmation email to inspector: ${createdByName} (${inspectorEmail})`);
    
    // Prepare attachments if PDF URL is provided
    const attachments = [];
    if (pdfUrl) {
      try {
        // Download PDF from URL and attach
        const https = require('https');
        const http = require('http');
        
        const pdfBuffer = await new Promise((resolve, reject) => {
          const protocol = pdfUrl.startsWith('https') ? https : http;
          protocol.get(pdfUrl, (response) => {
            const chunks = [];
            response.on('data', (chunk) => chunks.push(chunk));
            response.on('end', () => resolve(Buffer.concat(chunks)));
            response.on('error', reject);
          }).on('error', reject);
        });
        
        attachments.push({
          filename: `DVIR_Report_${dvirData.dvirNumber}.pdf`,
          content: pdfBuffer,
          contentType: 'application/pdf'
        });
        
        console.log(`📎 PDF attachment prepared for inspector: ${dvirData.dvirNumber}.pdf`);
      } catch (pdfError) {
        console.error('⚠️ Failed to download PDF for inspector email:', pdfError.message);
        // Continue without attachment
      }
    }
    
    const emailSubject = `DVIR Report Submitted: ${dvirData.dvirNumber}`;
    const emailHtml = `
<!DOCTYPE html>
<html>
<head>
  <meta charset="UTF-8">
  <title>DVIR Report Submitted - Titan Drilling</title>
</head>
<body style="font-family: Arial, sans-serif; background-color: #f5f6f8; margin: 0; padding: 0;">
  <div style="max-width: 700px; margin: 40px auto; background: #ffffff; box-shadow: 0 3px 8px rgba(0,0,0,0.05); overflow: hidden; border: 1px solid #e5e5e5;">
    
    <!-- Header -->
    <div style="background-color: #004a93; padding: 20px; text-align: center; border-bottom: 3px solid #003366;">
      <img src="https://titandrillingzm.com/images/Sidebar-logo.png" alt="Titan Drilling Logo" style="width: 120px;">
    </div>

    <!-- Content -->
    <div style="padding: 30px; color: #333;">
      <h2 style="color: #003366; margin-bottom: 20px; font-size: 22px;">✅ DVIR Report Submitted Successfully</h2>

      <p style="line-height: 1.6; margin-bottom: 10px;">Dear ${createdByName},</p>
      <p style="line-height: 1.6; margin-bottom: 20px;">Your DVIR report has been successfully submitted and saved in the system.</p>
      
      <!-- Table -->
      <div style="margin-top: 20px; background: #f9fafc; border: 1px solid #e5e5e5; border-radius: 8px; padding: 15px;">
        <table style="width: 100%; border-collapse: collapse; margin-top: 10px;">
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">DVIR Number:</th>
            <td style="padding: 10px; text-align: left;">${dvirData.dvirNumber}</td>
          </tr>
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Request Number:</th>
            <td style="padding: 10px; text-align: left;">${dvirData.requestNumber}</td>
          </tr>
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Equipment:</th>
            <td style="padding: 10px; text-align: left;">${dvirData.equipmentName || 'N/A'}</td>
          </tr>
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Country:</th>
            <td style="padding: 10px; text-align: left;">${dvirData.country || 'N/A'}</td>
          </tr>
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Project:</th>
            <td style="padding: 10px; text-align: left;">${dvirData.project || 'N/A'}</td>
          </tr>
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Inspector:</th>
            <td style="padding: 10px; text-align: left;">${createdByName}</td>
          </tr>
          ${dvirData.faultyItems > 0 ? `
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Faulty Items:</th>
            <td style="padding: 10px; text-align: left;"><strong style="color:#d9534f;">${dvirData.faultyItems}</strong></td>
          </tr>` : ''}
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Date:</th>
            <td style="padding: 10px; text-align: left;">${new Date().toLocaleDateString('en-GB', { day: '2-digit', month: '2-digit', year: 'numeric', hour: '2-digit', minute: '2-digit' })}</td>
          </tr>
        </table>
      </div>

      <p style="line-height: 1.6; margin-top: 20px; margin-bottom: 10px;">
        Thank you for completing this DVIR inspection. The report has been sent to all relevant supervisors and managers.
      </p>

      <p style="line-height: 1.6; margin-top: 20px; margin-bottom: 10px;">
        If you need assistance, please contact our support team at 
        <a href="mailto:support@titandrillingzm.com" style="text-decoration: none; color: #003366; font-weight: bold;">support@titandrillingzm.com</a>.
      </p>

      <p style="line-height: 1.6; margin-bottom: 10px;">Best regards,</p>
      <p style="line-height: 1.6; margin-bottom: 10px;"><strong>Titan Drilling Team</strong></p>
    </div>

    <!-- Footer -->
    <div style="background-color: #f4f4f4; text-align: center; padding: 20px; font-size: 13px; color: #555;">
      © ${new Date().getFullYear()} Titan Drilling. All Rights Reserved.
    </div>
  </div>
</body>
</html>
    `;
    
    const mailOptions = {
      from: '"Titan Drilling" <info@titandrillingzm.com>',
      to: inspectorEmail,
      subject: emailSubject,
      html: emailHtml,
    };
    
    // Add PDF attachment if available
    if (attachments.length > 0) {
      mailOptions.attachments = attachments;
    }
    
    await emailTransporter.sendMail(mailOptions);
    
    console.log(`✅ Confirmation email sent to inspector ${createdByName} (${inspectorEmail})${attachments.length > 0 ? ' with PDF' : ''}`);
    return { success: true };
  } catch (error) {
    console.error('❌ Error sending inspector email:', error);
    return { success: false, error: error.message };
  }
};

// Helper function to send emails with PDF attachment
const sendEmails = async (dvirData, createdByName, recipients, pdfUrl) => {
  try {
    console.log('📧 Sending email notifications for DVIR report...');
    
    let successCount = 0;
    let failCount = 0;
    
    // Prepare attachments if PDF URL is provided
    const attachments = [];
    if (pdfUrl) {
      try {
        // Download PDF from URL and attach
        const https = require('https');
        const http = require('http');
        
        const pdfBuffer = await new Promise((resolve, reject) => {
          const protocol = pdfUrl.startsWith('https') ? https : http;
          protocol.get(pdfUrl, (response) => {
            const chunks = [];
            response.on('data', (chunk) => chunks.push(chunk));
            response.on('end', () => resolve(Buffer.concat(chunks)));
            response.on('error', reject);
          }).on('error', reject);
        });
        
        attachments.push({
          filename: `DVIR_Report_${dvirData.dvirNumber}.pdf`,
          content: pdfBuffer,
          contentType: 'application/pdf'
        });
        
        console.log(`📎 PDF attachment prepared: ${dvirData.dvirNumber}.pdf`);
      } catch (pdfError) {
        console.error('⚠️ Failed to download PDF for attachment:', pdfError.message);
        // Continue without attachment
      }
    }
    
    // Email template
    const emailSubject = `New DVIR Report: ${dvirData.dvirNumber}`;
    const emailHtml = `
<!DOCTYPE html>
<html>
<head>
  <meta charset="UTF-8">
  <title>New DVIR Report - Titan Drilling</title>
</head>
<body style="font-family: Arial, sans-serif; background-color: #f5f6f8; margin: 0; padding: 0;">
  <div style="max-width: 700px; margin: 40px auto; background: #ffffff; box-shadow: 0 3px 8px rgba(0,0,0,0.05); overflow: hidden; border: 1px solid #e5e5e5;">
    
    <!-- Header -->
    <div style="background-color: #004a93; padding: 20px; text-align: center; border-bottom: 3px solid #003366;">
      <img src="https://titandrillingzm.com/images/Sidebar-logo.png" alt="Titan Drilling Logo" style="width: 120px;">
    </div>

    <!-- Content -->
    <div style="padding: 30px; color: #333;">
      <h2 style="color: #003366; margin-bottom: 20px; font-size: 22px;">🚛 New DVIR Report</h2>

      <p style="line-height: 1.6; margin-bottom: 10px;">Dear Recipient,</p>
      <p style="line-height: 1.6; margin-bottom: 20px;">A new Driver Vehicle Inspection Report (DVIR) has been submitted in the Titan Drilling system.</p>
      
      <!-- Table -->
      <div style="margin-top: 20px; background: #f9fafc; border: 1px solid #e5e5e5; border-radius: 8px; padding: 15px;">
        <table style="width: 100%; border-collapse: collapse; margin-top: 10px;">
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">DVIR Number:</th>
            <td style="padding: 10px; text-align: left;">${dvirData.dvirNumber}</td>
          </tr>
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Request Number:</th>
            <td style="padding: 10px; text-align: left;">${dvirData.requestNumber}</td>
          </tr>
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Equipment:</th>
            <td style="padding: 10px; text-align: left;">${dvirData.equipmentName || 'N/A'}</td>
          </tr>
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Country:</th>
            <td style="padding: 10px; text-align: left;">${dvirData.country || 'N/A'}</td>
          </tr>
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Project:</th>
            <td style="padding: 10px; text-align: left;">${dvirData.project || 'N/A'}</td>
          </tr>
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Inspector:</th>
            <td style="padding: 10px; text-align: left;">${createdByName}</td>
          </tr>
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Status:</th>
            <td style="padding: 10px; text-align: left;"><strong style="color:#4CAF50;">Completed</strong></td>
          </tr>
          ${dvirData.faultyItems > 0 ? `
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Faulty Items:</th>
            <td style="padding: 10px; text-align: left;"><strong style="color:#d9534f;">${dvirData.faultyItems}</strong></td>
          </tr>` : ''}
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Date:</th>
            <td style="padding: 10px; text-align: left;">${new Date().toLocaleDateString('en-GB', { day: '2-digit', month: '2-digit', year: 'numeric', hour: '2-digit', minute: '2-digit' })}</td>
          </tr>
        </table>
      </div>

      <p style="line-height: 1.6; margin-top: 20px; margin-bottom: 10px;">
        Please review the DVIR report in the Titan Drilling application.
      </p>

      <p style="line-height: 1.6; margin-top: 20px; margin-bottom: 10px;">
        If you need assistance, please contact our support team at 
        <a href="mailto:support@titandrillingzm.com" style="text-decoration: none; color: #003366; font-weight: bold;">support@titandrillingzm.com</a>.
      </p>

      <p style="line-height: 1.6; margin-bottom: 10px;">Best regards,</p>
      <p style="line-height: 1.6; margin-bottom: 10px;"><strong>Titan Drilling Team</strong></p>
    </div>

    <!-- Footer -->
    <div style="background-color: #f4f4f4; text-align: center; padding: 20px; font-size: 13px; color: #555;">
      © ${new Date().getFullYear()} Titan Drilling. All Rights Reserved.
    </div>
  </div>
</body>
</html>
    `;
    
    // Send email to each recipient
    for (const recipient of recipients) {
      if (recipient.email) {
        try {
          const mailOptions = {
            from: '"Titan Drilling" <info@titandrillingzm.com>',
            to: recipient.email,
            subject: emailSubject,
            html: emailHtml,
          };
          
          // Add PDF attachment if available
          if (attachments.length > 0) {
            mailOptions.attachments = attachments;
          }
          
          await emailTransporter.sendMail(mailOptions);
          
          console.log(`✅ Email sent to ${recipient.name} (${recipient.email})${attachments.length > 0 ? ' with PDF' : ''}`);
          successCount++;
        } catch (emailError) {
          console.error(`❌ Failed to send email to ${recipient.name}:`, emailError.message);
          failCount++;
        }
      }
    }
    
    console.log(`📧 Email summary: ${successCount} sent, ${failCount} failed`);
    return { success: true, sent: successCount, failed: failCount, total: recipients.length };
  } catch (error) {
    console.error('❌ Error sending emails:', error);
    return { success: false, error: error.message };
  }
};

/**
 * POST /api/dvir-notifications/new-dvir
 * Handle notifications when a new DVIR report is created
 */
app.post('/api/dvir-notifications/new-dvir', async (req, res) => {
  try {
    const { dvirData, createdByName, inspectorEmail, pdfUrl } = req.body;
    
    if (!dvirData || !dvirData.dvirNumber) {
      return res.status(400).json({
        success: false,
        error: 'DVIR data is required'
      });
    }
    
    console.log(`📎 PDF URL received: ${pdfUrl || 'Not provided'}`);
    
    console.log(`\n🔔 Processing notifications for DVIR: ${dvirData.dvirNumber}`);
    console.log(`📋 Country: ${dvirData.country}, Project: ${dvirData.project}`);
    console.log(`👤 Inspector: ${createdByName}`);
    
    // Manager roles to notify
    const managerRoles = [
      'HrManager',
      'Workshop Manager',
      'Maintenance Manager',
      'FinanceManager',
      'Project Manager',
      'Site Manager'
    ];
    
    // Get all ProAdmins (no filtering)
    const proAdmins = await GlobalUser.find({
      role: 'ProAdmin'
    }).select('_id name email fcmToken fcmTokens');
    
    console.log(`📋 Found ${proAdmins.length} ProAdmin(s)`);
    
    // Get Supervisors filtered by country AND project
    // Note: countries and projects are stored as ARRAYS in MongoDB
    const supervisors = await GlobalUser.find({
      role: 'Supervisor',
      countries: dvirData.country,  // MongoDB matches if array contains value
      projects: dvirData.project
    }).select('_id name email fcmToken fcmTokens role countries projects');
    
    console.log(`📋 Found ${supervisors.length} Supervisor(s) for ${dvirData.country} - ${dvirData.project}`);
    
    // Get Managers filtered by country AND project
    // Note: countries and projects are stored as ARRAYS in MongoDB
    const managers = await GlobalUser.find({
      role: { $in: managerRoles },
      countries: dvirData.country,  // MongoDB matches if array contains value
      projects: dvirData.project
    }).select('_id name email fcmToken fcmTokens role countries projects');
    
    console.log(`📋 Found ${managers.length} Manager(s) for ${dvirData.country} - ${dvirData.project}`);
    
    // Combine all recipients
    const allRecipients = [...proAdmins, ...supervisors, ...managers];
    
    // Remove duplicates based on _id
    const uniqueRecipients = allRecipients.filter((recipient, index, self) =>
      index === self.findIndex((r) => r._id === recipient._id)
    );
    
    console.log(`📋 Total unique recipients: ${uniqueRecipients.length}`);
    
    if (uniqueRecipients.length === 0) {
      console.log('⚠️ No recipients found');
      return res.json({
        success: true,
        message: 'No recipients to notify',
        pushSent: 0,
        emailsSent: 0,
        notificationsSaved: 0,
      });
    }
    
    // Execute all notification tasks in parallel
    const [mongoResult, pushResult, emailResult, inspectorEmailResult] = await Promise.all([
      saveNotificationsToMongoDB(dvirData, createdByName, uniqueRecipients),
      sendPushNotifications(dvirData, createdByName, uniqueRecipients),
      sendEmails(dvirData, createdByName, uniqueRecipients, pdfUrl),
      sendInspectorEmail(dvirData, createdByName, inspectorEmail, pdfUrl),
    ]);
    
    // Log results
    console.log('\n📊 Notification Results:');
    console.log(`  💾 MongoDB: ${mongoResult.success ? '✅' : '❌'} (${mongoResult.count || 0} notifications)`);
    console.log(`  📱 Push: ${pushResult.success ? '✅' : '❌'} (${pushResult.successCount || 0} devices)`);
    console.log(`  📧 Emails to Recipients: ${emailResult.success ? '✅' : '❌'} (${emailResult.sent || 0} sent)`);
    console.log(`  📧 Inspector Email: ${inspectorEmailResult.success ? '✅' : '❌'}\n`);
    
    res.json({
      success: true,
      message: `Notifications sent for DVIR: ${dvirData.dvirNumber}`,
      results: {
        mongodb: {
          success: mongoResult.success,
          saved: mongoResult.count || 0,
          error: mongoResult.error || null,
        },
        push: {
          success: pushResult.success,
          sent: pushResult.successCount || 0,
          failed: pushResult.failureCount || 0,
          error: pushResult.error || null,
        },
        emails: {
          success: emailResult.success,
          sent: emailResult.sent || 0,
          failed: emailResult.failed || 0,
          total: emailResult.total || 0,
          error: emailResult.error || null,
        },
        inspectorEmail: {
          success: inspectorEmailResult.success,
          error: inspectorEmailResult.error || null,
        },
      },
      recipients: {
        proAdmins: proAdmins.length,
        supervisors: supervisors.length,
        managers: managers.length,
        total: uniqueRecipients.length,
      }
    });
  } catch (error) {
    console.error('❌ Error processing DVIR notifications:', error);
    res.status(500).json({
      success: false,
      error: error.message,
    });
  }
});

// Test endpoint to check supervisor data structure
app.get('/test-supervisors/:country/:project', async (req, res) => {
  try {
    const { country, project } = req.params;
    
    console.log(`🔍 Testing supervisor query for: ${country} - ${project}`);
    
    // Get sample supervisors with all fields
    const sampleSupervisor = await GlobalUser.findOne({ role: 'Supervisor' }).limit(1);
    
    console.log('📊 Sample Supervisor structure:', JSON.stringify(sampleSupervisor, null, 2));
    
    // Try current query
    const currentQuery = await GlobalUser.find({
      role: 'Supervisor',
      $or: [
        { country: country, project: project },
        { countries: country, project: project }
      ]
    }).select('_id name email role country countries project projects');
    
    console.log(`Current query found: ${currentQuery.length} supervisors`);
    
    // Try alternative queries
    const altQuery1 = await GlobalUser.find({
      role: 'Supervisor',
      countries: country,
      projects: project
    }).select('_id name email role country countries project projects');
    
    console.log(`Alt query 1 (countries + projects) found: ${altQuery1.length} supervisors`);
    
    // Try checking if countries is an array
    const altQuery2 = await GlobalUser.find({
      role: 'Supervisor',
      countries: country
    }).select('_id name email role country countries project projects');
    
    console.log(`Alt query 2 (countries only) found: ${altQuery2.length} supervisors`);
    
    // Try with projects only
    const altQuery3 = await GlobalUser.find({
      role: 'Supervisor',
      projects: project
    }).select('_id name email role country countries project projects');
    
    console.log(`Alt query 3 (projects only) found: ${altQuery3.length} supervisors`);
    
    res.json({
      success: true,
      searchCriteria: { country, project },
      sampleSupervisor: sampleSupervisor ? {
        _id: sampleSupervisor._id,
        name: sampleSupervisor.name,
        role: sampleSupervisor.role,
        country: sampleSupervisor.country,
        countries: sampleSupervisor.countries,
        project: sampleSupervisor.project,
        projects: sampleSupervisor.projects,
        countryType: typeof sampleSupervisor.country,
        countriesType: typeof sampleSupervisor.countries,
        countriesIsArray: Array.isArray(sampleSupervisor.countries),
        projectType: typeof sampleSupervisor.project,
        projectsType: typeof sampleSupervisor.projects,
      } : null,
      results: {
        currentQuery: currentQuery.length,
        altQuery1: altQuery1.length,
        altQuery2: altQuery2.length,
        altQuery3: altQuery3.length,
        currentQueryData: currentQuery.map(u => ({
          name: u.name,
          country: u.country,
          countries: u.countries,
          project: u.project,
          projects: u.projects,
        })),
        altQuery1Data: altQuery1.map(u => ({
          name: u.name,
          country: u.country,
          countries: u.countries,
          project: u.project,
          projects: u.projects,
        })),
        altQuery2Data: altQuery2.slice(0, 3).map(u => ({
          name: u.name,
          countries: u.countries,
          projects: u.projects,
        })),
        altQuery3Data: altQuery3.slice(0, 3).map(u => ({
          name: u.name,
          countries: u.countries,
          projects: u.projects,
        })),
      }
    });
  } catch (error) {
    console.error('Test endpoint error:', error);
    res.status(500).json({ success: false, error: error.message });
  }
});

// Health check
app.get('/health', (req, res) => {
  res.json({
    status: 'OK',
    service: 'DVIR Notifications Service',
    mongodb: mongoose.connection.readyState === 1 ? 'Connected' : 'Disconnected',
    timestamp: new Date().toISOString(),
  });
});

// Start server
const PORT = process.env.DVIR_NOTIFICATIONS_PORT || 5020;
app.listen(PORT, () => {
  console.log(`\n🚀 DVIR Notifications Service running on port ${PORT}`);
  console.log(`📊 MongoDB: titan_drilling`);
  console.log(`🔗 Health check: https://api.titandrillingzm.com:${PORT}/health\n`);
});

// Graceful shutdown
process.on('SIGTERM', () => {
  console.log('SIGTERM signal received: closing HTTP server');
  app.close(() => {
    console.log('HTTP server closed');
    mongoose.connection.close(false, () => {
      console.log('MongoDB connection closed');
      process.exit(0);
    });
  });
});

module.exports = app;

