const express = require('express');
const { mongoose } = require('./dbConnection'); // Use shared connection
const multer = require('multer');
const path = require('path');
const fs = require('fs');
const cors = require('cors');

const app = express();
const PORT = process.env.INSPECTION_PORT || 5004;

// Middleware
app.use(cors());
app.use(express.json({ limit: '50mb' }));
app.use(express.urlencoded({ extended: true, limit: '50mb' }));

// Using shared MongoDB connection from dbConnection.js
console.log('✅ [Inspection Handler] Using shared MongoDB connection');

// Inspection Report Schema
const inspectionReportSchema = new mongoose.Schema({
  _id: { type: String, required: true },
  userId: { type: String, required: true },
  requestNumber: { type: String, required: true },
  status: { 
    type: String, 
    enum: ['draft', 'completed', 'archived'], 
    default: 'draft' 
  },
  
  // Step 1: Inspection Info
  step1: {
    id: String,
    requestNumber: String,
    country: String,
    project: String,
    inspector: String,
    checklistTitle: String,
    checklistDocumentId: String,
    selectedChecklist: {
      document_id: String,
      documentId: String,
      document_name: String,
      documentName: String,
      version: String,
      version_date: String,
      approved_by: String,
      title: String,
      checklist: mongoose.Schema.Types.Mixed
    },
    selectedEquipment: {
      id: String,
      equipmentName: String,
      mainCategory: String,
      model: String,
      vehicleNumber: String,
      meterReading: Number,
      status: String,
      lastKnownCoordinates: String,
      lastKnownLocation: String,
      gpsCoordinates: String,
      gpsAddress: String,
      locationUpdatedAt: Date
    },
    meterReading: Number,
    currentMeterReading: Number,
    selectedChecklistId: String,
    checklistTitle: String,
    coordinates: String,
    gpsAddress: String,
    createdAt: Date
  },
  
  // Step 2: Do Inspection
  step2: {
    selections: mongoose.Schema.Types.Mixed, // Object with item keys and their selected values
    selectedFiles: mongoose.Schema.Types.Mixed, // Object with file data
    numericInputs: mongoose.Schema.Types.Mixed, // Object with numeric input values
    notes: mongoose.Schema.Types.Mixed, // Object with notes
    sections: [{
      name: String,
      subItems: [{
        key: String,
        itemName: String,
        inputType: String,
        description: String,
        instruction: String,
        sortOrder: Number,
        isInspectionRequired: Boolean,
        category: String
      }]
    }],
    checklistTitle: String,
    headerInputValues: mongoose.Schema.Types.Mixed,
    footerInputValues: mongoose.Schema.Types.Mixed,
    faultyItems: Number,
    timestamp: Date
  },
  
  // Step 3: Summary
  step3: {
    faultyItems: Number,
    overallCondition: String,
    equipmentStatus: String,
    priority: String,
    safeToUse: String,
    additionalNote: String,
    signatureStatus: String,
    signatureData: mongoose.Schema.Types.Mixed,
    signatureDownloadUrl: String,
    signatureFields: [{
      key: String,
      label: String,
      roleName: String,
      required: Boolean,
      showInput: Boolean,
      showSignature: Boolean,
      nameKey: String,
      signatureKey: String
    }],
    signatureValues: mongoose.Schema.Types.Mixed,
    selectedSignatureType: String,
    selectedSignatureTypes: [String],
    individualSignatures: mongoose.Schema.Types.Mixed,
    individualSignatureStatus: mongoose.Schema.Types.Mixed,
    pdfDownloadUrl: String,
    createdAt: Date
  },
  
  // Corrective Action Data
  correctiveActionData: {
    subItemStatuses: mongoose.Schema.Types.Mixed, // Object with item keys and their status
    faultyCount: Number,
    isSatisfactory: Boolean,
    role: String, // 'Mechanic', 'Operator', 'Both'
    
    // Mechanic data
    mechanicName: String,
    mechanicSignatureUri: String,
    mechanicSignatureDownloadUrl: String,
    
    // Operator data
    operatorName: String,
    operatorSignatureUri: String,
    operatorSignatureDownloadUrl: String,
    
    // PDF data
    pdfDownloadUrl: String,
    
    // Metadata
    createdAt: { type: Date, default: Date.now },
    updatedAt: { type: Date, default: Date.now },
    createdBy: String,
    lastModifiedBy: String
  },
  
  // User Permissions and Access Control
  userPermissions: [String],
  userCountries: [String],
  userProjects: [String],
  
  // Metadata
  createdAt: { type: Date, default: Date.now },
  updatedAt: { type: Date, default: Date.now },
  createdBy: String,
  lastModifiedBy: String
});

// Create indexes for better performances
inspectionReportSchema.index({ userId: 1, createdAt: -1 });
inspectionReportSchema.index({ requestNumber: 1 });
inspectionReportSchema.index({ 'step1.country': 1, 'step1.project': 1 });
inspectionReportSchema.index({ 'step1.inspector': 1 });
inspectionReportSchema.index({ status: 1 });

const InspectionReport = mongoose.model('InspectionReport', inspectionReportSchema, 'inspectionreports');

// Create a more flexible model for the /inspections/all endpoint
const FlexibleInspectionReport = mongoose.model('FlexibleInspectionReport', new mongoose.Schema({}, { strict: false }), 'inspectionreports');

// Inspection Checklist Schema
const inspectionChecklistSchema = new mongoose.Schema({
  _id: { type: String, required: true },
  documentId: { type: String, required: true, unique: true },
  headerTitle: String,
  approvedBy: String,
  checklist: mongoose.Schema.Types.Mixed,
  documentName: String,
  documentPdfName: String,
  version: String,
  versionDate: String,
  header_inputs: mongoose.Schema.Types.Mixed,
  footer_inputs: mongoose.Schema.Types.Mixed,
  uploadedAt: { type: Date, default: Date.now },
  uploadedBy: String
}, { 
  strict: false  // Allow additional fields not defined in schema
});

// Create indexes for checklist
inspectionChecklistSchema.index({ documentId: 1 });
inspectionChecklistSchema.index({ headerTitle: 1 });

const InspectionChecklist = mongoose.model('InspectionChecklist', inspectionChecklistSchema);

// Corrective Action Schema
const correctiveActionSchema = new mongoose.Schema({
  _id: { type: String, required: true },
  reportId: { type: String, required: true },
  userId: { type: String, required: true },
  
  // Step 1: Faulty Items Status
  subItemStatuses: mongoose.Schema.Types.Mixed, // Object with item keys and their status
  faultyCount: Number,
  
  // Step 2: Signature Data
  isSatisfactory: Boolean,
  role: String, // 'Mechanic', 'Operator', 'Both'
  
  // Mechanic data
  mechanicName: String,
  mechanicSignatureUri: String,
  mechanicSignatureDownloadUrl: String,
  
  // Operator data
  operatorName: String,
  operatorSignatureUri: String,
  operatorSignatureDownloadUrl: String,
  
  // PDF data
  pdfDownloadUrl: String,
  
  // Metadata
  createdAt: { type: Date, default: Date.now },
  updatedAt: { type: Date, default: Date.now },
  createdBy: String,
  lastModifiedBy: String
});

// Create indexes for corrective actions
correctiveActionSchema.index({ reportId: 1 });
correctiveActionSchema.index({ userId: 1, createdAt: -1 });
correctiveActionSchema.index({ createdAt: -1 });

const CorrectiveAction = mongoose.model('CorrectiveAction', correctiveActionSchema, 'correctiveactions');

// Risk Management Schema
const riskManagementSchema = new mongoose.Schema({
  _id: { type: String, required: true },
  adminId: { type: String, required: true },
  inspectionId: { type: String, required: true },
  createdBy: String,
  lastModifiedBy: String,
  linkedInspection: {
    requestNumber: String,
    inspectionId: String,
    jobCardRequestNumber: String,
    jobCardId: String,
    inspector: String,
  },
  step1: {
    selectedHazardIds: [String],
    hazardRiskSelections: mongoose.Schema.Types.Mixed,
    updatedAt: Date,
  },
  step2: {
    rows: [{
      step: String,
      hazard: String,
      assess: String,
      control: String,
    }],
    guidanceNotes: [String],
    updatedAt: Date,
  },
  step3: {
    form: mongoose.Schema.Types.Mixed,
    pdfDownloadUrl: String,
    pdfFileName: String,
    generatedAt: Date,
    updatedAt: Date,
  },
  pdfDownloadUrl: String,
  pdfFileName: String,
  createdAt: { type: Date, default: Date.now },
  updatedAt: { type: Date, default: Date.now },
});

riskManagementSchema.index({ adminId: 1, inspectionId: 1 }, { unique: true });
riskManagementSchema.index({ inspectionId: 1 });

const RiskManagementRecord = mongoose.model('RiskManagementRecord', riskManagementSchema, 'riskmanagementrecords');

const getRiskManagementDocumentId = (adminId, inspectionId) => `risk_${adminId}_${inspectionId}`;

// Multer configuration for file uploads - Same as other backend services
const inspectionStorage = multer.diskStorage({
  destination: function (req, file, cb) {
    // Store in temp directory first, then move to user-specific folder
    const tempDir = path.join(__dirname, 'uploads', 'temp');
    
    // Ensure temp directory exists
    if (!fs.existsSync(tempDir)) {
      fs.mkdirSync(tempDir, { recursive: true });
      console.log(`📁 Created temp directory: ${tempDir}`);
    }
    
    console.log(`📁 Storing file temporarily in: ${tempDir}`);
    cb(null, tempDir);
  },
  filename: function (req, file, cb) {
    // Generate unique filename with timestamp and original extension
    const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
    const fileExtension = path.extname(file.originalname);
    cb(null, file.fieldname + '-' + uniqueSuffix + fileExtension);
  }
});

const fileUpload = multer({ 
  storage: inspectionStorage,
  limits: { fileSize: 50 * 1024 * 1024 } // 50MB limit
});

// Routes

// Health check
app.get('/inspectionreports', (req, res) => {
  res.json({ 
    status: 'OK', 
    service: 'Inspection Handler',
    port: PORT,
    timestamp: new Date().toISOString()
  });
});

// Debug endpoint to check database connection and collections
app.get('/debug/db', async (req, res) => {
  try {
    const db = mongoose.connection.db;
    const collections = await db.listCollections().toArray();
    const inspectionReportsCount = await db.collection('inspectionreports').countDocuments({});
    const sampleDoc = await db.collection('inspectionreports').findOne({});
    
    res.json({
      success: true,
      database: mongoose.connection.name,
      connectionState: mongoose.connection.readyState,
      collections: collections.map(c => c.name),
      inspectionReportsCount,
      sampleDocument: sampleDoc ? {
        _id: sampleDoc._id,
        userId: sampleDoc.userId,
        requestNumber: sampleDoc.requestNumber
      } : null
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

// Get total count of inspection reports (optimized endpoint)
app.get('/inspections/count', async (req, res) => {
  try {
    console.log('🔢 /inspections/count endpoint called');
    
    const db = mongoose.connection.db;
    const collection = db.collection('inspectionreports');
    
    console.log('📊 Database connection state:', mongoose.connection.readyState);
    console.log('📊 Database name:', mongoose.connection.name);
    console.log('📊 Collection name: inspectionreports');
    
    // Check if collection exists
    const collections = await db.listCollections({ name: 'inspectionreports' }).toArray();
    console.log('📊 Collections found:', collections.length);
    
    // Get total count only
    const total = await collection.countDocuments({});
    console.log('📊 Total documents in collection:', total);
    
    // Also try with mongoose model
    const modelCount = await InspectionReport.countDocuments({});
    console.log('📊 Model count:', modelCount);
    
    // Get a sample document to verify structure
    const sampleDoc = await collection.findOne({});
    console.log('📊 Sample document:', sampleDoc ? {
      _id: sampleDoc._id,
      userId: sampleDoc.userId,
      requestNumber: sampleDoc.requestNumber
    } : 'No documents found');
    
    res.json({
      success: true,
      total,
      modelCount,
      message: `Found ${total} inspection reports in database`,
      collectionExists: collections.length > 0,
      sampleDocument: sampleDoc ? {
        _id: sampleDoc._id,
        userId: sampleDoc.userId,
        requestNumber: sampleDoc.requestNumber
      } : null
    });
  } catch (error) {
    console.error('Error fetching inspection reports count:', error);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to fetch inspection reports count: ' + error.message 
    });
  }
});

// Get all inspection reports from all users (for global request number generation)
// IMPORTANT: This route MUST come BEFORE /inspections/:userId to avoid "all" being treated as a userId
app.get('/inspections/all', async (req, res) => {
  try {
    // Parse pagination parameters properly
    const page = parseInt(req.query.page) || 1;
    const limit = parseInt(req.query.limit) || 10000;
    
    console.log('🔍 /inspections/all endpoint called with params:', { page, limit });
    
    // Use direct MongoDB collection access for flexibility
    const db = mongoose.connection.db;
    const collection = db.collection('inspectionreports');
    
    console.log('📊 Database connection state:', mongoose.connection.readyState);
    console.log('📊 Database name:', mongoose.connection.name);
    
    // Get total count first
    const total = await collection.countDocuments({});
    console.log('📊 Total documents in collection:', total);
    
    // Calculate skip for pagination
    const skip = (page - 1) * limit;
    console.log('📊 Skip:', skip, 'Limit:', limit);
    
    // Fetch paginated documents directly from MongoDB
    const paginatedReports = await collection.find({})
      .sort({ createdAt: -1 })
      .skip(skip)
      .limit(limit)
      .toArray();
    
    console.log('📊 Reports fetched:', paginatedReports.length);
    
    if (paginatedReports.length > 0) {
      console.log('📊 Sample document keys:', Object.keys(paginatedReports[0]));
      console.log('📊 Sample document _id:', paginatedReports[0]._id);
      console.log('📊 Sample document userId:', paginatedReports[0].userId);
    }
    
    // Get corrective actions for all reports to check status
    const reportIds = paginatedReports.map(report => report._id);
    const correctiveActions = {};
    
    try {
      const correctiveActionsData = await CorrectiveAction.find({ 
        reportId: { $in: reportIds } 
      }).select('reportId pdfDownloadUrl isSatisfactory role faultyCount mechanicSignatureUri mechanicSignatureDownloadUrl operatorSignatureUri operatorSignatureDownloadUrl createdAt updatedAt');
      
      correctiveActionsData.forEach(ca => {
        const hasPdf = !!ca.pdfDownloadUrl;
        const hasSignatures = !!(
          ca.mechanicSignatureUri ||
          ca.mechanicSignatureDownloadUrl ||
          ca.operatorSignatureUri ||
          ca.operatorSignatureDownloadUrl
        );
        const hasData = hasPdf || hasSignatures || typeof ca.faultyCount === 'number' || typeof ca.isSatisfactory === 'boolean';
        
        let status = 'not_started';
        if (hasPdf) {
          status = 'completed';
        } else if (hasData) {
          status = 'in_progress';
        }
        
        correctiveActions[ca.reportId] = {
          hasCorrectiveAction: hasData,
          correctiveActionStatus: status,
          correctiveActionNumber: `CA-${ca.reportId}`,
          correctiveActionCreatedAt: ca.createdAt,
          correctiveActionSummary: {
            pdfDownloadUrl: ca.pdfDownloadUrl || null,
            isSatisfactory: typeof ca.isSatisfactory === 'boolean' ? ca.isSatisfactory : null,
            role: ca.role || null,
            faultyCount: typeof ca.faultyCount === 'number' ? ca.faultyCount : null,
            hasSignatures,
            createdAt: ca.createdAt || null,
            updatedAt: ca.updatedAt || null
          },
          correctiveActionPdfUrl: ca.pdfDownloadUrl || null
        };
      });
    } catch (error) {
      console.error('Error fetching corrective actions:', error);
    }
    
    // Enhance reports with corrective action status
    const enhancedReports = paginatedReports.map(report => ({
      ...report,
      // Add corrective action status if exists
      ...(correctiveActions[report._id] || {})
    }));
    
    console.log('✅ Sending response with', enhancedReports.length, 'reports');
    
    res.json({
      success: true,
      data: enhancedReports,
      pagination: {
        page: page,
        limit: limit,
        total: total,
        pages: Math.ceil(total / limit)
      }
    });
  } catch (error) {
    console.error('Error fetching all inspection reports:', error);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to fetch all inspection reports: ' + error.message 
    });
  }
});

// Get all inspection reports for a user
app.get('/inspections/:userId', async (req, res) => {
  try {
    const { userId } = req.params;
    const { page = 1, limit = 50, status, country, project } = req.query;
    
    const query = { userId };
    if (status) query.status = status;
    if (country) query['step1.country'] = country;
    if (project) query['step1.project'] = project;
    
    const skip = (page - 1) * limit;
    
    const reports = await InspectionReport.find(query)
      .sort({ createdAt: -1 })
      .skip(skip)
      .limit(parseInt(limit))
      .select('-__v');
    
    const total = await InspectionReport.countDocuments(query);
    
    res.json({
      success: true,
      data: reports,
      pagination: {
        page: parseInt(page),
        limit: parseInt(limit),
        total,
        pages: Math.ceil(total / limit)
      }
    });
  } catch (error) {
    console.error('Error fetching inspection reports:', error);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to fetch inspection reports' 
    });
  }
});

// Get a specific inspection report by reportId only (for cross-user viewing)
app.get('/inspections/report/:reportId', async (req, res) => {
  try {
    const { reportId } = req.params;
    
    const report = await InspectionReport.findOne({ 
      _id: reportId 
    }).select('-__v');
    
    if (!report) {
      return res.status(404).json({ 
        success: false, 
        error: 'Inspection report not found' 
      });
    }
    
    res.json({
      success: true,
      data: report
    });
  } catch (error) {
    console.error('Error fetching inspection report:', error);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to fetch inspection report' 
    });
  }
});

// Get a specific inspection report (original endpoint for user's own reports)
app.get('/inspections/:userId/:reportId', async (req, res) => {
  try {
    const { userId, reportId } = req.params;
    
    const report = await InspectionReport.findOne({ 
      _id: reportId, 
      userId 
    }).select('-__v');
    
    if (!report) {
      return res.status(404).json({ 
        success: false, 
        error: 'Inspection report not found' 
      });
    }
    
    res.json({
      success: true,
      data: report
    });
  } catch (error) {
    console.error('Error fetching inspection report:', error);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to fetch inspection report' 
    });
  }
});

// Create a new inspection report
app.post('/inspections', async (req, res) => {
  try {
    const { userId, reportData } = req.body;
    
    // Debug logging (can be removed in production)
    console.log('📝 Creating inspection report for userId:', userId);
    
    if (!userId || !reportData) {
      return res.status(400).json({ 
        success: false, 
        error: 'userId and reportData are required' 
      });
    }
    
    // Generate unique report ID if not provided
    const reportId = reportData._id || reportData.id || Date.now().toString();
    
    console.log('📝 Creating inspection report with data:', {
      _id: reportId,
      userId,
      requestNumber: reportData.requestNumber || reportData.step1?.requestNumber || reportId,
      status: reportData.status || 'draft',
      step1Keys: Object.keys(reportData.step1 || {}),
      step2Keys: Object.keys(reportData.step2 || {}),
      step3Keys: Object.keys(reportData.step3 || {}),
      createdAt: reportData.createdAt
    });

    // Prepare step1 data and persist checklist metadata for reporting
    const step1Data = reportData.step1 ? { ...reportData.step1 } : {};
    if (step1Data?.selectedChecklist) {
      const checklist = step1Data.selectedChecklist;
      step1Data.checklistTitle =
        step1Data.checklistTitle ||
        checklist.document_name ||
        checklist.documentName ||
        checklist.title ||
        checklist.document_id ||
        checklist.documentId ||
        '';
      step1Data.checklistDocumentId =
        step1Data.checklistDocumentId ||
        checklist.document_id ||
        checklist.documentId ||
        '';
    }

    // Use ALL data from reportData (important for offline-synced reports)
    const inspectionReport = new InspectionReport({
      _id: reportId,
      userId,
      requestNumber: reportData.requestNumber || reportData.step1?.requestNumber || reportId,
      status: reportData.status || 'completed', // Use provided status or default to completed
      step1: step1Data,
      step2: reportData.step2 || {},
      step3: reportData.step3 || {},
      createdAt: reportData.createdAt || new Date(),
      updatedAt: reportData.updatedAt || new Date(),
      createdBy: reportData.createdBy || userId,
      lastModifiedBy: reportData.lastModifiedBy || userId
    });
    
    console.log('💾 Attempting to save inspection report to MongoDB...');
    await inspectionReport.save();
    console.log('✅ Inspection report saved successfully to MongoDB');
    
    res.status(201).json({
      success: true,
      data: inspectionReport,
      message: 'Inspection report created successfully'
    });
  } catch (error) {
    console.error('❌ Error creating inspection report:', error);
    console.error('❌ Error details:', {
      message: error.message,
      name: error.name,
      code: error.code,
      stack: error.stack
    });
    res.status(500).json({ 
      success: false, 
      error: 'Failed to create inspection report: ' + error.message 
    });
  }
});

// Update an inspection report
app.put('/inspections/:userId/:reportId', async (req, res) => {
  try {
    const { userId, reportId } = req.params;
    const updateData = req.body;
    
    // Remove _id and userId from update data to prevent conflicts
    delete updateData._id;
    delete updateData.userId;
    
    updateData.updatedAt = new Date();
    updateData.lastModifiedBy = userId;
    
    const report = await InspectionReport.findOneAndUpdate(
      { _id: reportId, userId },
      { $set: updateData },
      { new: true, runValidators: true }
    ).select('-__v');
    
    if (!report) {
      return res.status(404).json({ 
        success: false, 
        error: 'Inspection report not found' 
      });
    }
    
    res.json({
      success: true,
      data: report,
      message: 'Inspection report updated successfully'
    });
  } catch (error) {
    console.error('Error updating inspection report:', error);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to update inspection report' 
    });
  }
});

// Update specific step of an inspection report
app.put('/inspections/:userId/:reportId/:step', async (req, res) => {
  try {
    const { userId, reportId, step } = req.params;
    const stepData = req.body;
    
    if (!['step1', 'step2', 'step3'].includes(step)) {
      return res.status(400).json({ 
        success: false, 
        error: 'Invalid step. Must be step1, step2, or step3' 
      });
    }
    
    const updateData = {
      [`${step}`]: stepData,
      updatedAt: new Date(),
      lastModifiedBy: userId
    };
    
    const report = await InspectionReport.findOneAndUpdate(
      { _id: reportId, userId },
      { $set: updateData },
      { new: true, runValidators: true }
    ).select('-__v');
    
    if (!report) {
      return res.status(404).json({ 
        success: false, 
        error: 'Inspection report not found' 
      });
    }
    
    res.json({
      success: true,
      data: report,
      message: `${step} updated successfully`
    });
  } catch (error) {
    console.error(`Error updating ${req.params.step}:`, error);
    res.status(500).json({ 
      success: false, 
      error: `Failed to update ${req.params.step}` 
    });
  }
});

// Complete an inspection report
app.put('/inspections/:userId/:reportId/complete', async (req, res) => {
  try {
    const { userId, reportId } = req.params;
    const { pdfDownloadUrl, signatureDownloadUrl } = req.body;
    
    const updateData = {
      status: 'completed',
      updatedAt: new Date(),
      lastModifiedBy: userId
    };
    
    if (pdfDownloadUrl) {
      updateData['step3.pdfDownloadUrl'] = pdfDownloadUrl;
    }
    
    if (signatureDownloadUrl) {
      updateData['step3.signatureDownloadUrl'] = signatureDownloadUrl;
    }
    
    const report = await InspectionReport.findOneAndUpdate(
      { _id: reportId, userId },
      { $set: updateData },
      { new: true, runValidators: true }
    ).select('-__v');
    
    if (!report) {
      return res.status(404).json({ 
        success: false, 
        error: 'Inspection report not found' 
      });
    }
    
    res.json({
      success: true,
      data: report,
      message: 'Inspection report completed successfully'
    });
  } catch (error) {
    console.error('Error completing inspection report:', error);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to complete inspection report' 
    });
  }
});

// Delete an inspection report by ID only (for admin/cross-user deletion)
app.delete('/inspections/delete-by-id/:reportId', async (req, res) => {
  try {
    const { reportId } = req.params;
    
    console.log('🗑️ Delete by ID request:', reportId);
    console.log('📊 Report ID type:', typeof reportId);
    console.log('📊 Report ID value:', reportId);
    
    // Strategy 1: Try with Mongoose model first
    let report = await InspectionReport.findOneAndDelete({ 
      _id: reportId
    });
    
    console.log('📊 Strategy 1 (Mongoose) result:', report ? 'Found and deleted' : 'Not found');
    
    // Strategy 2: If Mongoose fails, try direct MongoDB collection access
    if (!report) {
      console.log('🔄 Trying direct MongoDB collection access...');
      const db = mongoose.connection.db;
      const collection = db.collection('inspectionreports');
      
      // First, check if document exists
      const existingDoc = await collection.findOne({ _id: reportId });
      console.log('📊 Document exists in collection:', existingDoc ? 'Yes' : 'No');
      
      if (existingDoc) {
        // Delete directly from collection
        const deleteResult = await collection.deleteOne({ _id: reportId });
        console.log('📊 Direct delete result:', deleteResult);
        
        if (deleteResult.deletedCount > 0) {
          console.log('✅ Report deleted successfully via direct collection access');
          
          return res.json({
            success: true,
            message: 'Inspection report deleted successfully',
            data: {
              deletedId: reportId,
              deletedRequestNumber: existingDoc.requestNumber || 'N/A',
              method: 'direct-collection'
            }
          });
        }
      }
    } else {
      console.log('✅ Report deleted successfully via Mongoose:', reportId);
      
      return res.json({
        success: true,
        message: 'Inspection report deleted successfully',
        data: {
          deletedId: reportId,
          deletedRequestNumber: report.requestNumber || 'N/A',
          method: 'mongoose'
        }
      });
    }
    
    // If we reach here, the report was not found with any strategy
    console.log('❌ Report not found with any strategy:', reportId);
    
    // Get some debug info about what IDs exist
    const db = mongoose.connection.db;
    const collection = db.collection('inspectionreports');
    const allDocs = await collection.find({}).limit(20).project({ _id: 1, requestNumber: 1 }).toArray();
    console.log('📊 Available inspection IDs:', allDocs.map(doc => ({ _id: doc._id, requestNumber: doc.requestNumber })));
    
    return res.status(404).json({ 
      success: false, 
      error: 'Inspection report not found in database',
      debug: {
        searchedId: reportId,
        searchedIdType: typeof reportId,
        availableIds: allDocs.map(doc => doc._id),
        totalReports: allDocs.length
      }
    });
    
  } catch (error) {
    console.error('❌ Error deleting inspection report by ID:', error);
    console.error('❌ Error details:', error.message, error.stack);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to delete inspection report: ' + error.message 
    });
  }
});

// Delete an inspection report (original endpoint with userId verification)
app.delete('/inspections/:userId/:reportId', async (req, res) => {
  try {
    const { userId, reportId } = req.params;
    
    const report = await InspectionReport.findOneAndDelete({ 
      _id: reportId, 
      userId 
    });
    
    if (!report) {
      return res.status(404).json({ 
        success: false, 
        error: 'Inspection report not found' 
      });
    }
    
    res.json({
      success: true,
      message: 'Inspection report deleted successfully'
    });
  } catch (error) {
    console.error('Error deleting inspection report:', error);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to delete inspection report' 
    });
  }
});

// Get inspection statistics
app.get('/inspections/:userId/stats', async (req, res) => {
  try {
    const { userId } = req.params;
    const { country, project, startDate, endDate } = req.query;
    
    const matchQuery = { userId };
    if (country) matchQuery['step1.country'] = country;
    if (project) matchQuery['step1.project'] = project;
    if (startDate || endDate) {
      matchQuery.createdAt = {};
      if (startDate) matchQuery.createdAt.$gte = new Date(startDate);
      if (endDate) matchQuery.createdAt.$lte = new Date(endDate);
    }
    
    const stats = await InspectionReport.aggregate([
      { $match: matchQuery },
      {
        $group: {
          _id: null,
          totalReports: { $sum: 1 },
          completedReports: {
            $sum: { $cond: [{ $eq: ['$status', 'completed'] }, 1, 0] }
          },
          draftReports: {
            $sum: { $cond: [{ $eq: ['$status', 'draft'] }, 1, 0] }
          },
          totalFaultyItems: { $sum: '$step3.faultyItems' },
          avgFaultyItems: { $avg: '$step3.faultyItems' }
        }
      }
    ]);
    
    const countryStats = await InspectionReport.aggregate([
      { $match: matchQuery },
      {
        $group: {
          _id: '$step1.country',
          count: { $sum: 1 }
        }
      },
      { $sort: { count: -1 } }
    ]);
    
    const projectStats = await InspectionReport.aggregate([
      { $match: matchQuery },
      {
        $group: {
          _id: '$step1.project',
          count: { $sum: 1 }
        }
      },
      { $sort: { count: -1 } }
    ]);
    
    res.json({
      success: true,
      data: {
        overview: stats[0] || {
          totalReports: 0,
          completedReports: 0,
          draftReports: 0,
          totalFaultyItems: 0,
          avgFaultyItems: 0
        },
        byCountry: countryStats,
        byProject: projectStats
      }
    });
  } catch (error) {
    console.error('Error fetching inspection statistics:', error);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to fetch inspection statistics' 
    });
  }
});

// Upload file for inspection report
app.post('/inspections/:userId/:reportId/upload', fileUpload.single('file'), async (req, res) => {
  try {
    const { userId, reportId } = req.params;
    const { step, itemKey } = req.body;
    
    if (!req.file) {
      return res.status(400).json({ 
        success: false, 
        error: 'No file uploaded' 
      });
    }
    
    const fileUrl = `/inspections/files/${req.file.filename}`;
    
    // Update the report with file information
    const updateData = {
      [`${step}.selectedFiles.${itemKey}`]: {
        filename: req.file.filename,
        originalName: req.file.originalname,
        url: fileUrl,
        uploadedAt: new Date()
      },
      updatedAt: new Date(),
      lastModifiedBy: userId
    };
    
    await InspectionReport.findOneAndUpdate(
      { _id: reportId, userId },
      { $set: updateData }
    );
    
    res.json({
      success: true,
      data: {
        filename: req.file.filename,
        originalName: req.file.originalname,
        url: fileUrl,
        uploadedAt: new Date()
      },
      message: 'File uploaded successfully'
    });
  } catch (error) {
    console.error('Error uploading file:', error);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to upload file' 
    });
  }
});

// Serve uploaded files
app.use('/inspections/files', express.static(path.join(__dirname, 'uploads', 'inspections')));

// Search inspection reports
app.get('/inspections/:userId/search', async (req, res) => {
  try {
    const { userId } = req.params;
    const { q, page = 1, limit = 50 } = req.query;
    
    if (!q) {
      return res.status(400).json({ 
        success: false, 
        error: 'Search query is required' 
      });
    }
    
    const searchQuery = {
      userId,
      $or: [
        { requestNumber: { $regex: q, $options: 'i' } },
        { 'step1.inspector': { $regex: q, $options: 'i' } },
        { 'step1.country': { $regex: q, $options: 'i' } },
        { 'step1.project': { $regex: q, $options: 'i' } },
        { 'step1.selectedEquipment.equipmentName': { $regex: q, $options: 'i' } }
      ]
    };
    
    const skip = (page - 1) * limit;
    
    const reports = await InspectionReport.find(searchQuery)
      .sort({ createdAt: -1 })
      .skip(skip)
      .limit(parseInt(limit))
      .select('-__v');
    
    const total = await InspectionReport.countDocuments(searchQuery);
    
    res.json({
      success: true,
      data: reports,
      pagination: {
        page: parseInt(page),
        limit: parseInt(limit),
        total,
        pages: Math.ceil(total / limit)
      }
    });
  } catch (error) {
    console.error('Error searching inspection reports:', error);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to search inspection reports' 
    });
  }
});

// ========== FILE UPLOAD ENDPOINTS ==========

// Upload PDF to local storage
app.post('/inspections/upload-pdf/:userId', fileUpload.single('pdf'), async (req, res) => {
  try {
    console.log('📤 PDF Upload Request received:', {
      userId: req.params.userId,
      hasFile: !!req.file,
      fileInfo: req.file ? {
        originalname: req.file.originalname,
        mimetype: req.file.mimetype,
        size: req.file.size,
        path: req.file.path
      } : null,
      headers: req.headers
    });
    
    if (!req.file) {
      console.log('❌ No PDF file in request');
      return res.status(400).json({
        success: false,
        error: 'No PDF file uploaded'
      });
    }

    const userId = req.params.userId;
    const fileName = `inspection_${userId}_${Date.now()}.pdf`;
    
    // Create user-specific directory
    const userDir = path.join(__dirname, 'uploads', 'documents', 'inspectionreports', userId);
    if (!fs.existsSync(userDir)) {
      fs.mkdirSync(userDir, { recursive: true });
      console.log(`📁 Created user directory: ${userDir}`);
    }
    
    // Move file from temp to user directory
    const finalPath = path.join(userDir, fileName);
    
    try {
      fs.renameSync(req.file.path, finalPath);
      console.log(`📁 File moved to user directory: ${finalPath}`);
    } catch (moveError) {
      console.error('❌ Error moving file to user directory:', moveError);
      // If move fails, use the temp path
      const tempPath = req.file.path;
      finalPath = tempPath;
    }
    
    res.json({
      success: true,
      fileName: fileName,
      filePath: finalPath,
      downloadUrl: `/inspections/download-file/${userId}/${fileName}`,
      message: 'PDF uploaded successfully'
    });

  } catch (error) {
    console.error('PDF upload error:', error);
    if (req.file && fs.existsSync(req.file.path)) {
      fs.unlinkSync(req.file.path); // Clean up temp file
    }
    res.status(500).json({
      success: false,
      error: 'Failed to upload PDF: ' + error.message
    });
  }
});

// Upload signature image to local storage
app.post('/inspections/upload-signature/:userId', fileUpload.single('signature'), async (req, res) => {
  try {
    if (!req.file) {
      return res.status(400).json({
        success: false,
        error: 'No signature image uploaded'
      });
    }

    const userId = req.params.userId;
    const fileName = `signature_${userId}_${Date.now()}.png`;
    
    // Create user-specific directory
    const userDir = path.join(__dirname, 'uploads', 'documents', 'inspectionreports', userId);
    if (!fs.existsSync(userDir)) {
      fs.mkdirSync(userDir, { recursive: true });
      console.log(`📁 Created user directory: ${userDir}`);
    }
    
    // Move file from temp to user directory
    const finalPath = path.join(userDir, fileName);
    
    try {
      fs.renameSync(req.file.path, finalPath);
      console.log(`📁 File moved to user directory: ${finalPath}`);
    } catch (moveError) {
      console.error('❌ Error moving file to user directory:', moveError);
      // If move fails, use the temp path
      const tempPath = req.file.path;
      finalPath = tempPath;
    }
    
    res.json({
      success: true,
      fileName: fileName,
      filePath: finalPath,
      downloadUrl: `/inspections/download-file/${userId}/${fileName}`,
      message: 'Signature uploaded successfully'
    });

  } catch (error) {
    console.error('Signature upload error:', error);
    if (req.file && fs.existsSync(req.file.path)) {
      fs.unlinkSync(req.file.path); // Clean up temp file
    }
    res.status(500).json({
      success: false,
      error: 'Failed to upload signature: ' + error.message
    });
  }
});

// Download file from local storage
app.get('/inspections/download-file/:userId/:fileName', (req, res) => {
  try {
    const { userId, fileName } = req.params;
    const filePath = path.join(__dirname, 'uploads', 'documents', 'inspectionreports', userId, fileName);
    
    // Check if file exists
    if (!fs.existsSync(filePath)) {
      return res.status(404).json({
        success: false,
        error: 'File not found'
      });
    }

    // Set appropriate headers
    const ext = path.extname(fileName).toLowerCase();
    let contentType = 'application/octet-stream';
    
    if (ext === '.pdf') {
      contentType = 'application/pdf';
    } else if (ext === '.png') {
      contentType = 'image/png';
    } else if (ext === '.jpg' || ext === '.jpeg') {
      contentType = 'image/jpeg';
    }
    
    res.set({
      'Content-Type': contentType,
      'Content-Disposition': `attachment; filename="${fileName}"`
    });

    // Stream the file
    const fileStream = fs.createReadStream(filePath);
    fileStream.pipe(res);

    fileStream.on('error', (error) => {
      console.error('File download error:', error);
      res.status(500).json({
        success: false,
        error: 'Failed to download file'
      });
    });

  } catch (error) {
    console.error('Download error:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to download file: ' + error.message
    });
  }
});

// ========== CUSTOM CHECKLIST ENDPOINTS (MUST BE FIRST) ==========
console.log('🔧 Registering custom checklist routes...');

// Create unique index for custom checklists to prevent duplicates
mongoose.connection.once('open', async () => {
  try {
    const collection = mongoose.connection.db.collection('customchecklists');
    await collection.createIndex({ _id: 1 }, { unique: true });
    await collection.createIndex({ createdBy: 1, createdAt: -1 });
    console.log('✅ Custom checklists indexes created');
  } catch (error) {
    console.error('❌ Error creating custom checklists indexes:', error);
  }
});

// Test custom checklist route
app.get('/checklists/custom/test', (req, res) => {
  console.log('🧪 CUSTOM CHECKLIST TEST ROUTE HIT');
  res.json({ success: true, message: 'Custom checklist test route works' });
});
console.log('✅ Registered: GET /checklists/custom/test');

// Debug endpoint to check MongoDB connection and data
app.get('/checklists/custom/debug', async (req, res) => {
  try {
    console.log('🔍 DEBUG: Checking MongoDB connection and custom checklists...');
    
    const collection = mongoose.connection.db.collection('customchecklists');
    console.log('🔍 Collection name: customchecklists');
    console.log('🔍 Database name:', mongoose.connection.name);
    console.log('🔍 Connection state:', mongoose.connection.readyState);
    
    // Get total count
    const totalCount = await collection.countDocuments({});
    console.log('🔍 Total documents in customchecklists:', totalCount);
    
    // Get all documents
    const allDocs = await collection.find({}).limit(10).toArray();
    console.log('🔍 Sample documents:', allDocs.length);
    
    if (allDocs.length > 0) {
      console.log('🔍 First document keys:', Object.keys(allDocs[0]));
      console.log('🔍 First document _id:', allDocs[0]._id);
      console.log('🔍 First document title:', allDocs[0].title);
    }
    
    res.json({
      success: true,
      debug: {
        databaseName: mongoose.connection.name,
        connectionState: mongoose.connection.readyState,
        totalCount,
        sampleDocs: allDocs.length,
        firstDoc: allDocs[0] || null
      }
    });
  } catch (error) {
    console.error('❌ Debug error:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});
console.log('✅ Registered: GET /checklists/custom/debug');

// Create a new custom checklist
app.post('/checklists/custom', async (req, res) => {
  try {
    console.log('📝 Creating custom checklist...');
    const { checklistData, userId } = req.body;
    
    if (!checklistData || !userId) {
      return res.status(400).json({
        success: false,
        error: 'checklistData and userId are required'
      });
    }

    const collection = mongoose.connection.db.collection('customchecklists');
    
    // FIRST: Check for duplicate checklist by title and user (prevent multiple checklists with same title)
    console.log('🔍 Checking for duplicate checklist with title:', checklistData.title, 'and userId:', userId);
    const duplicateChecklist = await collection.findOne({ 
      title: checklistData.title,
      createdBy: userId 
    });
    
    console.log('🔍 Duplicate check result:', duplicateChecklist ? 'Found duplicate' : 'No duplicate found');
    
    if (duplicateChecklist) {
      console.log('⚠️ Duplicate checklist found with same title, updating existing one:', duplicateChecklist._id);
      
      // Update the existing checklist instead of creating a new one
      const updateData = {
        description: checklistData.description || duplicateChecklist.description,
        items: checklistData.items || duplicateChecklist.items,
        updatedAt: new Date().toISOString(),
        status: 'active'
      };
      
      const result = await collection.updateOne(
        { _id: duplicateChecklist._id, createdBy: userId },
        { $set: updateData }
      );
      
      console.log('🔍 Update result:', result);
      
      if (result.matchedCount > 0) {
        const updatedChecklist = await collection.findOne({ 
          _id: duplicateChecklist._id, 
          createdBy: userId 
        });
        
        console.log('✅ Duplicate checklist updated instead of creating new one');
        
        return res.json({
          success: true,
          data: updatedChecklist,
          message: 'Checklist updated (duplicate prevented)'
        });
      }
    }
    
    // SECOND: Check if this is an update to existing checklist by ID (for editing existing checklists)
    if (checklistData.id || checklistData._id) {
      const existingId = checklistData.id || checklistData._id;
      console.log('🔍 Checking for existing checklist by ID:', existingId);
      
      const existingChecklist = await collection.findOne({ 
        _id: existingId, 
        createdBy: userId 
      });
      
      if (existingChecklist) {
        console.log('📝 Updating existing checklist by ID:', existingId);
        
        // Update existing checklist
        const updateData = {
          title: checklistData.title || existingChecklist.title,
          description: checklistData.description || existingChecklist.description,
          items: checklistData.items || existingChecklist.items,
          updatedAt: new Date().toISOString(),
          status: 'active'
        };
        
        const result = await collection.updateOne(
          { _id: existingId, createdBy: userId },
          { $set: updateData }
        );
        
        if (result.matchedCount > 0) {
          const updatedChecklist = await collection.findOne({ 
            _id: existingId, 
            createdBy: userId 
          });
          
          console.log('✅ Custom checklist updated by ID:', existingId);
          
          return res.json({
            success: true,
            data: updatedChecklist,
            message: 'Custom checklist updated successfully'
          });
        }
      }
    }
    
    // Generate unique checklist ID for new checklist
    const checklistId = checklistData.id || `custom_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
    
    // Check if ID already exists (prevent duplicates)
    const existingWithId = await collection.findOne({ _id: checklistId });
    if (existingWithId) {
      console.log('⚠️ Checklist ID already exists, generating new one');
      const newChecklistId = `custom_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
      checklistId = newChecklistId;
    }
    
    const customChecklist = {
      _id: checklistId,
      documentId: checklistId,
      type: 'custom',
      title: checklistData.title || 'Untitled Checklist',
      description: checklistData.description || '',
      items: checklistData.items || [],
      createdBy: userId,
      createdAt: new Date().toISOString(),
      updatedAt: new Date().toISOString(),
      status: 'active'
    };

    await collection.insertOne(customChecklist);

    console.log('✅ Custom checklist created:', checklistId);

    res.json({
      success: true,
      data: customChecklist,
      message: 'Custom checklist created successfully'
    });

  } catch (error) {
    console.error('❌ Error creating custom checklist:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to create custom checklist: ' + error.message
    });
  }
});

// Get ALL custom checklists (for all users)
app.get('/checklists/custom/all', async (req, res) => {
  try {
    const { page = 1, limit = 100, search } = req.query;
    
    console.log('📋 CUSTOM CHECKLIST ALL ROUTE HIT - Fetching ALL custom checklists');

    const query = {};
    if (search) {
      query.$or = [
        { title: { $regex: search, $options: 'i' } },
        { description: { $regex: search, $options: 'i' } }
      ];
    }

    const skip = (page - 1) * limit;
    
    const collection = mongoose.connection.db.collection('customchecklists');
    const checklists = await collection.find(query)
      .sort({ createdAt: -1 })
      .skip(skip)
      .limit(parseInt(limit))
      .toArray();

    const total = await collection.countDocuments(query);

    console.log(`✅ Found ${checklists.length} custom checklists total`);

    res.json({
      success: true,
      data: checklists,
      pagination: {
        page: parseInt(page),
        limit: parseInt(limit),
        total,
        pages: Math.ceil(total / limit)
      }
    });

  } catch (error) {
    console.error('❌ Error fetching all custom checklists:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to fetch all custom checklists: ' + error.message
    });
  }
});

// Get user's custom checklists
app.get('/checklists/custom/:userId', async (req, res) => {
  try {
    const { userId } = req.params;
    const { page = 1, limit = 50, search } = req.query;
    
    console.log('📋 CUSTOM CHECKLIST ROUTE HIT - Fetching custom checklists for user:', userId);

    const query = { createdBy: userId };
    if (search) {
      query.$or = [
        { title: { $regex: search, $options: 'i' } },
        { description: { $regex: search, $options: 'i' } }
      ];
    }

    const skip = (page - 1) * limit;
    
    const collection = mongoose.connection.db.collection('customchecklists');
    const checklists = await collection.find(query)
      .sort({ createdAt: -1 })
      .skip(skip)
      .limit(parseInt(limit))
      .toArray();

    const total = await collection.countDocuments(query);

    console.log(`✅ Found ${checklists.length} custom checklists for user ${userId}`);

    res.json({
      success: true,
      data: checklists,
      pagination: {
        page: parseInt(page),
        limit: parseInt(limit),
        total,
        pages: Math.ceil(total / limit)
      }
    });

  } catch (error) {
    console.error('❌ Error fetching custom checklists:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to fetch custom checklists: ' + error.message
    });
  }
});

// Get specific custom checklist
app.get('/checklists/custom/:userId/:checklistId', async (req, res) => {
  try {
    const { userId, checklistId } = req.params;
    
    console.log('📋 Fetching custom checklist:', checklistId, 'for user:', userId);

    const collection = mongoose.connection.db.collection('customchecklists');
    const checklist = await collection.findOne({ 
      _id: checklistId, 
      createdBy: userId 
    });

    if (!checklist) {
      return res.status(404).json({
        success: false,
        error: 'Custom checklist not found'
      });
    }

    console.log('✅ Custom checklist found');

    res.json({
      success: true,
      data: checklist
    });

  } catch (error) {
    console.error('❌ Error fetching custom checklist:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to fetch custom checklist: ' + error.message
    });
  }
});

// Update custom checklist
app.put('/checklists/custom/:userId/:checklistId', async (req, res) => {
  try {
    const { userId, checklistId } = req.params;
    const updateData = req.body;
    
    console.log('📝 Updating custom checklist:', checklistId);

    // Remove fields that shouldn't be updated
    delete updateData._id;
    delete updateData.createdBy;
    delete updateData.createdAt;
    
    // Set updated timestamp
    updateData.updatedAt = new Date().toISOString();

    const collection = mongoose.connection.db.collection('customchecklists');
    const result = await collection.updateOne(
      { _id: checklistId, createdBy: userId },
      { $set: updateData }
    );

    if (result.matchedCount === 0) {
      return res.status(404).json({
        success: false,
        error: 'Custom checklist not found'
      });
    }

    // Fetch updated checklist
    const updatedChecklist = await collection.findOne({ 
      _id: checklistId, 
      createdBy: userId 
    });

    console.log('✅ Custom checklist updated');

    res.json({
      success: true,
      data: updatedChecklist,
      message: 'Custom checklist updated successfully'
    });

  } catch (error) {
    console.error('❌ Error updating custom checklist:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to update custom checklist: ' + error.message
    });
  }
});

// Delete custom checklist
app.delete('/checklists/custom/:userId/:checklistId', async (req, res) => {
  try {
    const { userId, checklistId } = req.params;
    
    console.log('🗑️ Deleting custom checklist:', checklistId);

    const collection = mongoose.connection.db.collection('customchecklists');
    const result = await collection.deleteOne({ 
      _id: checklistId, 
      createdBy: userId 
    });

    if (result.deletedCount === 0) {
      return res.status(404).json({
        success: false,
        error: 'Custom checklist not found'
      });
    }

    console.log('✅ Custom checklist deleted');

    res.json({
      success: true,
      message: 'Custom checklist deleted successfully'
    });

  } catch (error) {
    console.error('❌ Error deleting custom checklist:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to delete custom checklist: ' + error.message
    });
  }
});

// Get all custom checklists (for admin/global view)
app.get('/checklists/custom/all', async (req, res) => {
  try {
    const { page = 1, limit = 50, search } = req.query;
    
    console.log('📋 Fetching all custom checklists');

    const query = {};
    if (search) {
      query.$or = [
        { title: { $regex: search, $options: 'i' } },
        { description: { $regex: search, $options: 'i' } },
        { createdBy: { $regex: search, $options: 'i' } }
      ];
    }

    const skip = (page - 1) * limit;
    
    const collection = mongoose.connection.db.collection('customchecklists');
    const checklists = await collection.find(query)
      .sort({ createdAt: -1 })
      .skip(skip)
      .limit(parseInt(limit))
      .toArray();

    const total = await collection.countDocuments(query);

    console.log(`✅ Found ${checklists.length} custom checklists`);

    res.json({
      success: true,
      data: checklists,
      pagination: {
        page: parseInt(page),
        limit: parseInt(limit),
        total,
        pages: Math.ceil(total / limit)
      }
    });

  } catch (error) {
    console.error('❌ Error fetching all custom checklists:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to fetch custom checklists: ' + error.message
    });
  }
});

// ========== GENERAL CHECKLIST ENDPOINTS ==========

// Test endpoint
app.post('/checklists/test', (req, res) => {
  console.log('🧪 TEST ENDPOINT CALLED');
  res.json({ success: true, message: 'Test endpoint works' });
});

// Get checklist library documents from server folders
app.get('/checklists/library', async (req, res) => {
  try {
    console.log('📚 Fetching checklist library documents...');
    
    const fs = require('fs');
    const path = require('path');
    
    // Base directory for checklist documents
    const baseDir = '/root/node-mongo-api/titandrillingchecklistsModule';
    
    // Check if base directory exists
    if (!fs.existsSync(baseDir)) {
      return res.status(404).json({
        success: false,
        error: 'Checklist library directory not found'
      });
    }
    
    const categories = [];
    const allDocuments = [];
    
    // Read all folders in the base directory
    const folders = fs.readdirSync(baseDir, { withFileTypes: true })
      .filter(dirent => dirent.isDirectory())
      .map(dirent => dirent.name);
    
    console.log('📁 Found folders:', folders);
    
    for (const folder of folders) {
      const folderPath = path.join(baseDir, folder);
      const documents = [];
      
      try {
        // Read files in each folder
        const files = fs.readdirSync(folderPath, { withFileTypes: true })
          .filter(dirent => dirent.isFile())
          .map(dirent => dirent.name);
        
        console.log(`📄 Files in ${folder}:`, files);
        
        for (const file of files) {
          const filePath = path.join(folderPath, file);
          const stats = fs.statSync(filePath);
          
          // Get file extension
          const ext = path.extname(file).toLowerCase();
          let fileType = 'Document';
          let icon = 'insert-drive-file';
          
          if (['.pdf'].includes(ext)) {
            fileType = 'PDF';
            icon = 'picture-as-pdf';
          } else if (['.doc', '.docx'].includes(ext)) {
            fileType = 'Word';
            icon = 'description';
          } else if (['.xls', '.xlsx'].includes(ext)) {
            fileType = 'Excel';
            icon = 'table-chart';
          } else if (['.ppt', '.pptx'].includes(ext)) {
            fileType = 'PowerPoint';
            icon = 'slideshow';
          }
          
          const document = {
            id: `${folder}_${file}`,
            name: file,
            category: folder,
            filePath: filePath,
            size: stats.size,
            type: fileType,
            icon: icon,
            createdAt: stats.birthtime,
            modifiedAt: stats.mtime,
            extension: ext,
            downloaded: false,
            selected: false
          };
          
          documents.push(document);
          allDocuments.push(document);
        }
        
        if (documents.length > 0) {
          categories.push({
            name: folder,
            count: documents.length,
            documents: documents
          });
        }
        
      } catch (error) {
        console.error(`Error reading folder ${folder}:`, error);
      }
    }
    
    console.log(`✅ Found ${allDocuments.length} documents in ${categories.length} categories`);
    
    res.json({
      success: true,
      data: {
        categories: categories,
        allDocuments: allDocuments,
        totalDocuments: allDocuments.length,
        totalCategories: categories.length
      }
    });
    
  } catch (error) {
    console.error('Error fetching checklist library:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to fetch checklist library: ' + error.message
    });
  }
});

// Download checklist library document
app.get('/checklists/library/download/:category/:filename', (req, res) => {
  try {
    const { category, filename } = req.params;
    const filePath = path.join('/root/node-mongo-api/titandrillingchecklistsModule', category, filename);
    
    console.log('📥 Download request for:', filePath);
    
    // Check if file exists
    if (!fs.existsSync(filePath)) {
      return res.status(404).json({
        success: false,
        error: 'File not found'
      });
    }
    
    // Set appropriate headers based on file extension
    const ext = path.extname(filename).toLowerCase();
    let contentType = 'application/octet-stream';
    
    if (ext === '.pdf') {
      contentType = 'application/pdf';
    } else if (ext === '.doc') {
      contentType = 'application/msword';
    } else if (ext === '.docx') {
      contentType = 'application/vnd.openxmlformats-officedocument.wordprocessingml.document';
    } else if (ext === '.xls') {
      contentType = 'application/vnd.ms-excel';
    } else if (ext === '.xlsx') {
      contentType = 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet';
    } else if (ext === '.ppt') {
      contentType = 'application/vnd.ms-powerpoint';
    } else if (ext === '.pptx') {
      contentType = 'application/vnd.openxmlformats-officedocument.presentationml.presentation';
    }
    
    res.set({
      'Content-Type': contentType,
      'Content-Disposition': `attachment; filename="${filename}"`
    });
    
    // Stream the file
    const fileStream = fs.createReadStream(filePath);
    fileStream.pipe(res);
    
    fileStream.on('error', (error) => {
      console.error('File download error:', error);
      res.status(500).json({
        success: false,
        error: 'Failed to download file'
      });
    });
    
  } catch (error) {
    console.error('Download error:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to download file: ' + error.message
    });
  }
});

// Upload all checklists from JSON data
app.post('/checklists/upload-all', async (req, res) => {
  console.log('🚀 UPLOAD ENDPOINT CALLED');
  try {
    const { checklistsData, uploadedBy = 'system' } = req.body;
    console.log('📦 Received data keys:', Object.keys(checklistsData || {}));
    
    if (!checklistsData || typeof checklistsData !== 'object') {
      console.log('❌ Invalid checklists data');
      return res.status(400).json({ 
        success: false, 
        error: 'Checklists data is required' 
      });
    }
    
    // Debug: Check first checklist structure
    const firstKey = Object.keys(checklistsData)[0];
    const firstChecklist = checklistsData[firstKey];
    console.log('🔍 First checklist key:', firstKey);
    console.log('🔍 First checklist header_inputs:', JSON.stringify(firstChecklist.header_inputs));
    console.log('🔍 First checklist footer_inputs:', JSON.stringify(firstChecklist.footer_inputs));
    
    const results = [];
    const errors = [];
    
    // Get MongoDB collection
    const collection = mongoose.connection.db.collection('inspectionchecklists');
    
    for (const [documentId, checklistData] of Object.entries(checklistsData)) {
      try {
        // Delete existing if it exists
        await collection.deleteOne({ documentId });
        
        // Create document with ALL fields including header_inputs and footer_inputs
        const document = {
          _id: documentId,
          documentId: documentId,
          headerTitle: checklistData.Header_title || checklistData.headerTitle,
          approvedBy: checklistData.approved_by || checklistData.approvedBy,
          checklist: checklistData.checklist,
          documentName: checklistData.document_name || checklistData.documentName,
          documentPdfName: checklistData.document_pdf_name || checklistData.documentPdfName,
          version: checklistData.version,
          versionDate: checklistData.version_date || checklistData.versionDate,
          header_inputs: checklistData.header_inputs || {},
          footer_inputs: checklistData.footer_inputs || {},
          uploadedBy: uploadedBy,
          uploadedAt: new Date()
        };
        
        // Insert the document
        const result = await collection.insertOne(document);
        
        // Verify the inserted document
        const insertedDoc = await collection.findOne({ _id: documentId });
        
        results.push({
          documentId,
          action: 'created',
          data: insertedDoc
        });
      } catch (error) {
        errors.push({
          documentId,
          error: error.message
        });
      }
    }
    
    console.log('✅ Upload completed successfully');
    res.json({
      success: true,
      message: `Processed ${results.length} checklists successfully`,
      data: {
        processed: results.length,
        errors: errors.length,
        results,
        errors
      }
    });
  } catch (error) {
    console.error('❌ Error uploading checklists:', error);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to upload checklists: ' + error.message 
    });
  }
});

// ========== GENERAL CHECKLIST ENDPOINTS ==========

// Get all checklists
app.get('/checklists', async (req, res) => {
  try {
    const { page = 1, limit = 50, search } = req.query;
    
    const query = {};
    if (search) {
      query.$or = [
        { documentId: { $regex: search, $options: 'i' } },
        { headerTitle: { $regex: search, $options: 'i' } },
        { documentName: { $regex: search, $options: 'i' } }
      ];
    }
    
    const skip = (page - 1) * limit;
    
    const checklists = await InspectionChecklist.find(query)
      .sort({ uploadedAt: -1 })
      .skip(skip)
      .limit(parseInt(limit))
      .select('-__v');
    
    const total = await InspectionChecklist.countDocuments(query);
    
    res.json({
      success: true,
      data: checklists,
      pagination: {
        page: parseInt(page),
        limit: parseInt(limit),
        total,
        pages: Math.ceil(total / limit)
      }
    });
  } catch (error) {
    console.error('Error fetching checklists:', error);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to fetch checklists' 
    });
  }
});

// Get a specific checklist (skip if documentId is "custom", "library", or "test")
app.get('/checklists/:documentId', async (req, res, next) => {
  try {
    const { documentId } = req.params;
    
    // Skip this route if documentId is a reserved keyword - pass to next route
    if (documentId === 'custom' || documentId === 'library' || documentId === 'test' || documentId === 'upload-all') {
      return next();
    }
    
    const checklist = await InspectionChecklist.findOne({ documentId }).select('-__v');
    
    if (!checklist) {
      return res.status(404).json({ 
        success: false, 
        error: 'Checklist not found' 
      });
    }
    
    res.json({
      success: true,
      data: checklist
    });
  } catch (error) {
    console.error('Error fetching checklist:', error);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to fetch checklist' 
    });
  }
});

// Delete a checklist
app.delete('/checklists/:documentId', async (req, res) => {
  try {
    const { documentId } = req.params;
    
    const checklist = await InspectionChecklist.findOneAndDelete({ documentId });
    
    if (!checklist) {
      return res.status(404).json({ 
        success: false, 
        error: 'Checklist not found' 
      });
    }
    
    res.json({
      success: true,
      message: 'Checklist deleted successfully'
    });
  } catch (error) {
    console.error('Error deleting checklist:', error);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to delete checklist' 
    });
  }
});

// ========== CORRECTIVE ACTION ENDPOINTS ==========

// Get corrective action by report ID
// ========== RISK MANAGEMENT ENDPOINTS ==========

app.get('/risk-management/:adminId/:inspectionId', async (req, res) => {
  try {
    const { adminId, inspectionId } = req.params;

    const record = await RiskManagementRecord.findOne({ adminId, inspectionId }).select('-__v');

    if (!record) {
      return res.status(404).json({
        success: false,
        error: 'Risk management record not found',
      });
    }

    res.json({
      success: true,
      data: record,
    });
  } catch (error) {
    console.error('Error fetching risk management record:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to fetch risk management record',
    });
  }
});

const upsertRiskManagementRecord = async ({
  adminId,
  inspectionId,
  setData,
  createdBy,
}) => {
  const now = new Date();
  const documentId = getRiskManagementDocumentId(adminId, inspectionId);

  const updateSet = {
    ...setData,
    adminId,
    inspectionId,
    updatedAt: now,
  };

  const setOnInsert = {
    _id: documentId,
    adminId,
    inspectionId,
    createdAt: now,
  };

  if (createdBy) {
    setOnInsert.createdBy = createdBy;
  }

  const record = await RiskManagementRecord.findOneAndUpdate(
    { _id: documentId },
    {
      $set: updateSet,
      $setOnInsert: setOnInsert,
    },
    {
      upsert: true,
      new: true,
      runValidators: true,
    }
  ).select('-__v');

  return record;
};

app.put('/risk-management/:adminId/:inspectionId/step1', async (req, res) => {
  try {
    const { adminId, inspectionId } = req.params;
    const payload = req.body || {};
    const now = new Date();

    const step1Data = {
      selectedHazardIds: Array.isArray(payload.selectedHazardIds) ? payload.selectedHazardIds : [],
      hazardRiskSelections: payload.hazardRiskSelections || {},
      updatedAt: payload.updatedAt ? new Date(payload.updatedAt) : now,
    };

    const setData = {
      step1: step1Data,
    };

    if (payload.linkedInspection) {
      setData.linkedInspection = payload.linkedInspection;
    }

    if (payload.userId || payload.updatedBy || payload.lastModifiedBy) {
      setData.lastModifiedBy = payload.updatedBy || payload.lastModifiedBy || payload.userId;
    }

    const record = await upsertRiskManagementRecord({
      adminId,
      inspectionId,
      setData,
      createdBy: payload.createdBy || payload.userId || payload.updatedBy,
    });

    res.json({
      success: true,
      data: record,
      message: 'Risk management step 1 saved',
    });
  } catch (error) {
    console.error('Error saving risk management step1:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to save risk management step1',
    });
  }
});

app.put('/risk-management/:adminId/:inspectionId/step2', async (req, res) => {
  try {
    const { adminId, inspectionId } = req.params;
    const payload = req.body || {};
    const now = new Date();

    const rows = Array.isArray(payload.rows) ? payload.rows : [];

    const step2Data = {
      rows,
      guidanceNotes: Array.isArray(payload.guidanceNotes) ? payload.guidanceNotes : undefined,
      updatedAt: payload.updatedAt ? new Date(payload.updatedAt) : now,
    };

    if (!step2Data.guidanceNotes) {
      delete step2Data.guidanceNotes;
    }

    const setData = {
      step2: step2Data,
    };

    if (payload.linkedInspection) {
      setData.linkedInspection = payload.linkedInspection;
    }

    if (payload.userId || payload.updatedBy || payload.lastModifiedBy) {
      setData.lastModifiedBy = payload.updatedBy || payload.lastModifiedBy || payload.userId;
    }

    const record = await upsertRiskManagementRecord({
      adminId,
      inspectionId,
      setData,
      createdBy: payload.createdBy || payload.userId || payload.updatedBy,
    });

    res.json({
      success: true,
      data: record,
      message: 'Risk management step 2 saved',
    });
  } catch (error) {
    console.error('Error saving risk management step2:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to save risk management step2',
    });
  }
});

app.put('/risk-management/:adminId/:inspectionId/step3', async (req, res) => {
  try {
    const { adminId, inspectionId } = req.params;
    const payload = req.body || {};
    const now = new Date();

    const step3Data = {
      form: payload.form || {},
      pdfDownloadUrl: payload.pdfDownloadUrl || null,
      pdfFileName: payload.pdfFileName || null,
      generatedAt: payload.generatedAt ? new Date(payload.generatedAt) : now,
      updatedAt: now,
    };

    const setData = {
      step3: step3Data,
    };

    if (payload.pdfDownloadUrl) {
      setData.pdfDownloadUrl = payload.pdfDownloadUrl;
    }
    if (payload.pdfFileName) {
      setData.pdfFileName = payload.pdfFileName;
    }

    if (payload.linkedInspection) {
      setData.linkedInspection = payload.linkedInspection;
    }

    if (payload.userId || payload.updatedBy || payload.lastModifiedBy) {
      setData.lastModifiedBy = payload.updatedBy || payload.lastModifiedBy || payload.userId;
    }

    const record = await upsertRiskManagementRecord({
      adminId,
      inspectionId,
      setData,
      createdBy: payload.createdBy || payload.userId || payload.updatedBy,
    });

    res.json({
      success: true,
      data: record,
      message: 'Risk management step 3 saved',
    });
  } catch (error) {
    console.error('Error saving risk management step3:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to save risk management step3',
    });
  }
});

app.post('/risk-management/upload-pdf/:adminId/:inspectionId', fileUpload.single('pdf'), async (req, res) => {
  try {
    const { adminId, inspectionId } = req.params;
    const { userId } = req.body;

    if (!req.file) {
      return res.status(400).json({
        success: false,
        error: 'No PDF file uploaded',
      });
    }

    const directoryPath = path.join(__dirname, 'uploads', 'documents', 'riskmanagement', adminId || 'unknown');
    if (!fs.existsSync(directoryPath)) {
      fs.mkdirSync(directoryPath, { recursive: true });
      console.log(`📁 Created risk management directory: ${directoryPath}`);
    }

    const fileName = req.file.originalname && req.file.originalname.endsWith('.pdf')
      ? req.file.originalname
      : `riskmanagement_${inspectionId}_${Date.now()}.pdf`;

    const finalPath = path.join(directoryPath, fileName);
    try {
      fs.renameSync(req.file.path, finalPath);
    } catch (moveError) {
      console.error('❌ Error moving risk management PDF:', moveError);
      fs.copyFileSync(req.file.path, finalPath);
      fs.unlinkSync(req.file.path);
    }

    const downloadUrl = `/risk-management/download-pdf/${adminId}/${fileName}`;

    const setData = {
      pdfDownloadUrl: downloadUrl,
      pdfFileName: fileName,
      'step3.pdfDownloadUrl': downloadUrl,
      'step3.pdfFileName': fileName,
      'step3.updatedAt': new Date(),
    };

    const record = await upsertRiskManagementRecord({
      adminId,
      inspectionId,
      setData,
      createdBy: userId,
    });

    res.json({
      success: true,
      fileName,
      filePath: finalPath,
      downloadUrl,
      data: record,
      message: 'Risk management PDF uploaded successfully',
    });
  } catch (error) {
    console.error('Risk management PDF upload error:', error);
    if (req.file && fs.existsSync(req.file.path)) {
      fs.unlinkSync(req.file.path);
    }
    res.status(500).json({
      success: false,
      error: 'Failed to upload risk management PDF: ' + error.message,
    });
  }
});

app.get('/risk-management/download-pdf/:adminId/:fileName', (req, res) => {
  try {
    const { adminId, fileName } = req.params;
    const filePath = path.join(__dirname, 'uploads', 'documents', 'riskmanagement', adminId || 'unknown', fileName);

    if (!fs.existsSync(filePath)) {
      return res.status(404).json({
        success: false,
        error: 'Risk management PDF not found',
      });
    }

    res.set({
      'Content-Type': 'application/pdf',
      'Content-Disposition': `attachment; filename="${fileName}"`,
    });

    const fileStream = fs.createReadStream(filePath);
    fileStream.pipe(res);

    fileStream.on('error', (error) => {
      console.error('Risk management PDF download error:', error);
      res.status(500).json({
        success: false,
        error: 'Failed to download risk management PDF',
      });
    });
  } catch (error) {
    console.error('Risk management download error:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to download risk management PDF: ' + error.message,
    });
  }
});

// ========== CORRECTIVE ACTION ENDPOINTS ==========

app.get('/corrective-actions/:reportId', async (req, res) => {
  try {
    const { reportId } = req.params;
    
    const correctiveAction = await CorrectiveAction.findOne({ reportId }).select('-__v');
    
    if (!correctiveAction) {
      return res.status(404).json({ 
        success: false, 
        error: 'Corrective action not found' 
      });
    }
    
    res.json({
      success: true,
      data: correctiveAction
    });
  } catch (error) {
    console.error('Error fetching corrective action:', error);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to fetch corrective action' 
    });
  }
});

// Create or update corrective action
app.post('/corrective-actions', async (req, res) => {
  console.log('🔧 CORRECTIVE ACTIONS POST ENDPOINT CALLED!');
  console.log('🔧 Request body:', req.body);
  try {
    const { reportId, userId, correctiveActionData } = req.body;
    
    if (!reportId || !userId || !correctiveActionData) {
      return res.status(400).json({ 
        success: false, 
        error: 'reportId, userId, and correctiveActionData are required' 
      });
    }
    
    // Generate unique corrective action ID if not provided
    const correctiveActionId = correctiveActionData._id || `ca_${reportId}_${Date.now()}`;
    
    console.log('Creating/updating corrective action:', {
      _id: correctiveActionId,
      reportId,
      userId,
      hasSubItemStatuses: !!correctiveActionData.subItemStatuses,
      hasPdfUrl: !!correctiveActionData.pdfDownloadUrl
    });

    // Check if corrective action already exists
    const existingAction = await CorrectiveAction.findOne({ reportId });
    
    let correctiveAction;
    if (existingAction) {
      // Update existing
      const updateData = {
        ...correctiveActionData,
        updatedAt: new Date(),
        lastModifiedBy: userId
      };
      
      correctiveAction = await CorrectiveAction.findOneAndUpdate(
        { reportId },
        { $set: updateData },
        { new: true, runValidators: true }
      ).select('-__v');
      
      console.log('Updated existing corrective action');
    } else {
      // Create new
      correctiveAction = new CorrectiveAction({
        _id: correctiveActionId,
        reportId,
        userId,
        ...correctiveActionData,
        createdBy: userId,
        lastModifiedBy: userId
      });
      
      await correctiveAction.save();
      console.log('Created new corrective action');
    }
    
    // Also update the inspection report with correctiveActionData
    try {
      const inspectionReport = await InspectionReport.findOne({ _id: reportId });
      if (inspectionReport) {
        const correctiveDataForReport = {
          subItemStatuses: correctiveActionData.subItemStatuses,
          faultyCount: correctiveActionData.faultyCount,
          isSatisfactory: correctiveActionData.isSatisfactory,
          role: correctiveActionData.role,
          mechanicName: correctiveActionData.mechanicName,
          mechanicSignatureUri: correctiveActionData.mechanicSignatureUri,
          mechanicSignatureDownloadUrl: correctiveActionData.mechanicSignatureDownloadUrl,
          operatorName: correctiveActionData.operatorName,
          operatorSignatureUri: correctiveActionData.operatorSignatureUri,
          operatorSignatureDownloadUrl: correctiveActionData.operatorSignatureDownloadUrl,
          pdfDownloadUrl: correctiveActionData.pdfDownloadUrl,
          updatedAt: new Date(),
          lastModifiedBy: userId
        };
        
        // Add createdAt if this is a new corrective action
        if (!existingAction) {
          correctiveDataForReport.createdAt = new Date();
          correctiveDataForReport.createdBy = userId;
        }
        
        await InspectionReport.findOneAndUpdate(
          { _id: reportId },
          { 
            $set: { 
              correctiveActionData: correctiveDataForReport,
              updatedAt: new Date(),
              lastModifiedBy: userId
            } 
          }
        );
        
        console.log('✅ Updated inspection report with correctiveActionData');
      } else {
        console.warn('⚠️ Inspection report not found for corrective action data update');
      }
    } catch (reportUpdateError) {
      console.error('❌ Error updating inspection report with corrective action data:', reportUpdateError);
      // Don't fail the entire operation if report update fails
    }
    
    res.status(201).json({
      success: true,
      data: correctiveAction,
      message: 'Corrective action saved successfully'
    });
  } catch (error) {
    console.error('Error saving corrective action:', error);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to save corrective action: ' + error.message 
    });
  }
});

// Update corrective action step (for step-by-step updates)
app.put('/corrective-actions/:reportId/step/:stepNumber', async (req, res) => {
  try {
    const { reportId, stepNumber } = req.params;
    const stepData = req.body;
    
    if (!['1', '2'].includes(stepNumber)) {
      return res.status(400).json({ 
        success: false, 
        error: 'Invalid step number. Must be 1 or 2' 
      });
    }
    
    const updateData = {
      ...stepData,
      updatedAt: new Date()
    };
    
    const correctiveAction = await CorrectiveAction.findOneAndUpdate(
      { reportId },
      { $set: updateData },
      { new: true, runValidators: true, upsert: true }
    ).select('-__v');
    
    res.json({
      success: true,
      data: correctiveAction,
      message: `Step ${stepNumber} updated successfully`
    });
  } catch (error) {
    console.error(`Error updating corrective action step ${req.params.stepNumber}:`, error);
    res.status(500).json({ 
      success: false, 
      error: `Failed to update step ${req.params.stepNumber}` 
    });
  }
});

// Upload corrective action PDF
app.post('/corrective-actions/:reportId/upload-pdf', fileUpload.single('pdf'), async (req, res) => {
  try {
    const { reportId } = req.params;
    const { userId } = req.body;
    
    if (!req.file) {
      return res.status(400).json({
        success: false,
        error: 'No PDF file uploaded'
      });
    }

    // Create user-specific directory - correctiveactionreports/uid structure
    const userDir = path.join(__dirname, 'uploads', 'documents', 'correctiveactionreports', userId || 'unknown');
    if (!fs.existsSync(userDir)) {
      fs.mkdirSync(userDir, { recursive: true });
      console.log(`📁 Created corrective action directory: ${userDir}`);
    }
    
    // Use the filename from the uploaded file if available, otherwise generate one
    let fileName = req.file.originalname;
    if (!fileName || !fileName.endsWith('.pdf')) {
      const timestamp = Date.now();
      fileName = `corrective_action_${reportId}_${timestamp}.pdf`;
    }
    
    // Move file from temp to user directory
    const finalPath = path.join(userDir, fileName);
    
    try {
      fs.renameSync(req.file.path, finalPath);
      console.log(`📁 Corrective action PDF moved to: ${finalPath}`);
    } catch (moveError) {
      console.error('❌ Error moving corrective action PDF:', moveError);
      const tempPath = req.file.path;
      finalPath = tempPath;
    }
    
    res.json({
      success: true,
      fileName: fileName,
      filePath: finalPath,
      downloadUrl: `/corrective-actions/download-pdf/${fileName}?userId=${userId}`,
      message: 'Corrective action PDF uploaded successfully'
    });

  } catch (error) {
    console.error('Corrective action PDF upload error:', error);
    if (req.file && fs.existsSync(req.file.path)) {
      fs.unlinkSync(req.file.path); // Clean up temp file
    }
    res.status(500).json({
      success: false,
      error: 'Failed to upload corrective action PDF: ' + error.message
    });
  }
});

// Upload corrective action signature
app.post('/corrective-actions/:reportId/upload-signature', fileUpload.single('signature'), async (req, res) => {
  try {
    const { reportId } = req.params;
    const { signatureType, userId } = req.body; // 'mechanic' or 'operator'
    
    if (!req.file) {
      return res.status(400).json({
        success: false,
        error: 'No signature image uploaded'
      });
    }

    if (!signatureType || !['mechanic', 'operator'].includes(signatureType)) {
      return res.status(400).json({
        success: false,
        error: 'signatureType must be "mechanic" or "operator"'
      });
    }

    // Use the filename from the uploaded file if available, otherwise generate one
    let fileName = req.file.originalname;
    if (!fileName || !fileName.endsWith('.png')) {
      fileName = `signature_${reportId}_${signatureType}_${Date.now()}.png`;
    }
    
    // Create user-specific directory - correctiveactionreports/uid structure
    const userDir = path.join(__dirname, 'uploads', 'documents', 'correctiveactionreports', userId || 'unknown');
    if (!fs.existsSync(userDir)) {
      fs.mkdirSync(userDir, { recursive: true });
      console.log(`📁 Created corrective action directory: ${userDir}`);
    }
    
    // Move file from temp to user directory
    const finalPath = path.join(userDir, fileName);
    
    try {
      fs.renameSync(req.file.path, finalPath);
      console.log(`📁 Corrective action signature moved to: ${finalPath}`);
    } catch (moveError) {
      console.error('❌ Error moving corrective action signature:', moveError);
      const tempPath = req.file.path;
      finalPath = tempPath;
    }
    
    res.json({
      success: true,
      fileName: fileName,
      filePath: finalPath,
      downloadUrl: `/corrective-actions/download-signature/${fileName}?userId=${userId}`,
      signatureType: signatureType,
      message: 'Corrective action signature uploaded successfully'
    });

  } catch (error) {
    console.error('Corrective action signature upload error:', error);
    if (req.file && fs.existsSync(req.file.path)) {
      fs.unlinkSync(req.file.path); // Clean up temp file
    }
    res.status(500).json({
      success: false,
      error: 'Failed to upload corrective action signature: ' + error.message
    });
  }
});

// Download corrective action PDF
app.get('/corrective-actions/download-pdf/:fileName', (req, res) => {
  try {
    const { fileName } = req.params;
    const { userId } = req.query;
    const filePath = path.join(__dirname, 'uploads', 'documents', 'correctiveactionreports', userId || 'unknown', fileName);
    
    console.log('🔍 Looking for PDF file at:', filePath);
    console.log('🔍 User ID:', userId);
    console.log('🔍 File name:', fileName);
    
    // Check if file exists
    if (!fs.existsSync(filePath)) {
      console.log('❌ File not found at:', filePath);
      return res.status(404).json({
        success: false,
        error: 'PDF file not found'
      });
    }
    
    console.log('✅ File found at:', filePath);

    // Set appropriate headers
    res.set({
      'Content-Type': 'application/pdf',
      'Content-Disposition': `attachment; filename="${fileName}"`
    });

    // Stream the file
    const fileStream = fs.createReadStream(filePath);
    fileStream.pipe(res);

    fileStream.on('error', (error) => {
      console.error('Corrective action PDF download error:', error);
      res.status(500).json({
        success: false,
        error: 'Failed to download PDF'
      });
    });

  } catch (error) {
    console.error('Download error:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to download PDF: ' + error.message
    });
  }
});

// Download corrective action signature
app.get('/corrective-actions/download-signature/:fileName', (req, res) => {
  try {
    const { fileName } = req.params;
    const { userId } = req.query;
    const filePath = path.join(__dirname, 'uploads', 'documents', 'correctiveactionreports', userId || 'unknown', fileName);
    
    // Check if file exists
    if (!fs.existsSync(filePath)) {
      return res.status(404).json({
        success: false,
        error: 'Signature file not found'
      });
    }

    // Set appropriate headers
    res.set({
      'Content-Type': 'image/png',
      'Content-Disposition': `attachment; filename="${fileName}"`
    });

    // Stream the file
    const fileStream = fs.createReadStream(filePath);
    fileStream.pipe(res);

    fileStream.on('error', (error) => {
      console.error('Corrective action signature download error:', error);
      res.status(500).json({
        success: false,
        error: 'Failed to download signature'
      });
    });

  } catch (error) {
    console.error('Download error:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to download signature: ' + error.message
    });
  }
});

// Delete corrective action
app.delete('/corrective-actions/:reportId', async (req, res) => {
  try {
    const { reportId } = req.params;
    
    const correctiveAction = await CorrectiveAction.findOneAndDelete({ reportId });
    
    if (!correctiveAction) {
      return res.status(404).json({ 
        success: false, 
        error: 'Corrective action not found' 
      });
    }
    
    res.json({
      success: true,
      message: 'Corrective action deleted successfully'
    });
  } catch (error) {
    console.error('Error deleting corrective action:', error);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to delete corrective action' 
    });
  }
});

// 404 handler - MUST BE BEFORE ERROR HANDLER BUT AFTER ALL ROUTES
app.use((req, res, next) => {
  console.log('❌ 404 - Endpoint not found:', req.method, req.originalUrl);
  res.status(404).json({ 
    success: false, 
    error: 'Endpoint not found' 
  });
});

// Error handling middleware
app.use((error, req, res, next) => {
  console.error('Unhandled error:', error);
  res.status(500).json({ 
    success: false, 
    error: 'Internal server error' 
  });
});

// Start server
app.listen(PORT, '0.0.0.0', () => {
  console.log(`🔍 Inspection Handler server running on port ${PORT}`);
  console.log(`📊 Health check: http://localhost:${PORT}/inspectionreports`);
  console.log(`📊 External access: https://api.titandrillingzm.com:${PORT}/inspectionreports`);
  console.log(`🔢 Total count: https://api.titandrillingzm.com:${PORT}/inspections/count`);
  console.log(`📁 PDF upload: https://api.titandrillingzm.com:${PORT}/inspections/upload-pdf/:userId`);
  console.log(`📁 Signature upload: https://api.titandrillingzm.com:${PORT}/inspections/upload-signature/:userId`);
  console.log(`📁 File download: https://api.titandrillingzm.com:${PORT}/inspections/download-file/:userId/:fileName`);
  console.log(`📋 Checklist upload: https://api.titandrillingzm.com:${PORT}/checklists/upload-all`);
  console.log(`📚 Checklist library: https://api.titandrillingzm.com:${PORT}/checklists/library`);
  console.log(`📝 Custom checklists:`);
  console.log(`   📋 Create: https://api.titandrillingzm.com:${PORT}/checklists/custom`);
  console.log(`   📋 Get user's: https://api.titandrillingzm.com:${PORT}/checklists/custom/:userId`);
  console.log(`   📋 Get specific: https://api.titandrillingzm.com:${PORT}/checklists/custom/:userId/:checklistId`);
  console.log(`   📝 Update: https://api.titandrillingzm.com:${PORT}/checklists/custom/:userId/:checklistId`);
  console.log(`   🗑️ Delete: https://api.titandrillingzm.com:${PORT}/checklists/custom/:userId/:checklistId`);
  console.log(`   📋 Get all: https://api.titandrillingzm.com:${PORT}/checklists/custom/all`);
  console.log(`🔍 Get report by ID: https://api.titandrillingzm.com:${PORT}/inspections/report/:reportId`);
  console.log(`🔧 Corrective Actions:`);
  console.log(`   📋 Get corrective action: https://api.titandrillingzm.com:${PORT}/corrective-actions/:reportId`);
  console.log(`   💾 Save corrective action: https://api.titandrillingzm.com:${PORT}/corrective-actions`);
  console.log(`   📁 Upload corrective PDF: https://api.titandrillingzm.com:${PORT}/corrective-actions/:reportId/upload-pdf`);
  console.log(`   ✍️ Upload signature: https://api.titandrillingzm.com:${PORT}/corrective-actions/:reportId/upload-signature`);
  console.log(`   📄 Download PDF: https://api.titandrillingzm.com:${PORT}/corrective-actions/download-pdf/:fileName`);
  console.log(`   🖋️ Download signature: https://api.titandrillingzm.com:${PORT}/corrective-actions/download-signature/:fileName`);
});

module.exports = app;
