const express = require('express');
const { mongoose } = require('./dbConnection'); // Use shared connection
const cors = require('cors');
const { Server } = require('socket.io');
const http = require('http');

const app = express();
const server = http.createServer(app);
const io = new Server(server, {
  cors: {
    origin: '*',
    methods: ['GET', 'POST', 'PUT', 'DELETE'],
  },
});

// Middleware
app.use(cors());
app.use(express.json());

// MongoDB configuration
const MONGO_HOST = process.env.MONGO_HOST || 'api.titandrillingzm.com';
const MONGO_PORT = process.env.MONGO_PORT || '27017';
const MONGO_USER = process.env.MONGO_USER || 'titandrillingadminuser';
const MONGO_PASSWORD = process.env.MONGO_PASSWORD || 'Titandrilling_@2024';
const MONGO_DB = process.env.MONGO_DB || 'titan_drilling';

// Using shared MongoDB connection from dbConnection.js
console.log('✅ [Notifications Handler] Using shared MongoDB connection');

// Notification Schema
const NotificationSchema = new mongoose.Schema({
  notificationId: {
    type: String,
    required: true,
    unique: true,
  },
  userId: {
    type: String,
    required: true,
    index: true,
  },
  title: {
    type: String,
    required: true,
  },
  body: {
    type: String,
    required: true,
  },
  type: {
    type: String,
    required: true,
    enum: [
      'welcome',
      'safety_alert',
      'maintenance',
      'update',
      'test',
      'notice_board',
      'user_created',
      'equipment_added',
      'dvir_report',
      'inspection_report',
      'hse_inspection_report',
      'incident_report',
      'request_maintenance_created',
      'work_order_created',
      'job_card_created',
      'service_schedule_created',
      'hr_application',
      'hr_application_approval_rejection',
      'hr_application_supervisor_approved',
      'general',
    ],
  },
  data: {
    type: mongoose.Schema.Types.Mixed,
    default: {},
  },
  isRead: {
    type: Boolean,
    default: false,
  },
  timestamp: {
    type: Number,
    required: true,
    index: true,
  },
  createdAt: {
    type: Date,
    default: Date.now,
    index: true,
  },
  // Additional fields for different notification types
  requestNumber: String,
  equipmentName: String,
  mainCategory: String,
  country: String,
  project: String,
  equipment: String,
  incidentNumber: String,
  adminName: String,
  creatorName: String,
  inspectionType: String,
  applicationType: String,
  employeeName: String,
  employeeNo: String,
}, {
  collection: 'notifications',
  timestamps: true,
});

// Indexes for better performance
NotificationSchema.index({ userId: 1, timestamp: -1 });
NotificationSchema.index({ userId: 1, isRead: 1 });
NotificationSchema.index({ userId: 1, createdAt: -1 });

const Notification = mongoose.model('Notification', NotificationSchema);

// ==================== API ROUTES ====================

/**
 * GET /api/notifications/user/:userId
 * Get all notifications for a user
 */
app.get('/api/notifications/user/:userId', async (req, res) => {
  try {
    const { userId } = req.params;
    const { limit = 100, skip = 0 } = req.query;

    console.log(`📱 Fetching notifications for user: ${userId}`);

    const notifications = await Notification.find({ userId })
      .sort({ timestamp: -1 }) // Newest first
      .limit(parseInt(limit))
      .skip(parseInt(skip))
      .lean();

    console.log(`✅ Found ${notifications.length} notifications`);

    res.json({
      success: true,
      data: notifications,
      count: notifications.length,
    });
  } catch (error) {
    console.error('❌ Error fetching notifications:', error);
    res.status(500).json({
      success: false,
      error: error.message,
    });
  }
});

/**
 * GET /api/notifications/user/:userId/unread-count
 * Get unread notification count
 */
app.get('/api/notifications/user/:userId/unread-count', async (req, res) => {
  try {
    const { userId } = req.params;

    const count = await Notification.countDocuments({
      userId,
      isRead: false,
    });

    res.json({
      success: true,
      unreadCount: count,
    });
  } catch (error) {
    console.error('❌ Error counting unread notifications:', error);
    res.status(500).json({
      success: false,
      unreadCount: 0,
      error: error.message,
    });
  }
});

/**
 * POST /api/notifications
 * Create a new notification
 */
app.post('/api/notifications', async (req, res) => {
  try {
    const notificationData = req.body;

    console.log('📝 Creating notification:', notificationData.title);

    // Generate notificationId if not provided
    if (!notificationData.notificationId) {
      notificationData.notificationId = `notif_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
    }

    // Set timestamp if not provided
    if (!notificationData.timestamp) {
      notificationData.timestamp = Date.now();
    }

    const notification = new Notification(notificationData);
    await notification.save();

    console.log(`✅ Notification created: ${notification.notificationId}`);

    // Emit real-time update via Socket.IO
    io.emit('notification:created', {
      userId: notification.userId,
      notification: notification.toObject(),
    });

    res.json({
      success: true,
      data: notification,
      message: 'Notification created successfully',
    });
  } catch (error) {
    console.error('❌ Error creating notification:', error);
    res.status(500).json({
      success: false,
      error: error.message,
    });
  }
});

/**
 * POST /api/notifications/bulk
 * Create multiple notifications at once
 */
app.post('/api/notifications/bulk', async (req, res) => {
  try {
    const { notifications } = req.body;

    if (!Array.isArray(notifications)) {
      return res.status(400).json({
        success: false,
        error: 'Notifications must be an array',
      });
    }

    console.log(`📝 Creating ${notifications.length} notifications`);

    // Add notificationId and timestamp to each
    const notificationsWithIds = notifications.map(notif => ({
      ...notif,
      notificationId: notif.notificationId || `notif_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`,
      timestamp: notif.timestamp || Date.now(),
    }));

    const createdNotifications = await Notification.insertMany(notificationsWithIds);

    console.log(`✅ Created ${createdNotifications.length} notifications`);

    // Emit real-time updates
    createdNotifications.forEach(notification => {
      io.emit('notification:created', {
        userId: notification.userId,
        notification: notification.toObject(),
      });
    });

    res.json({
      success: true,
      data: createdNotifications,
      count: createdNotifications.length,
      message: `${createdNotifications.length} notifications created successfully`,
    });
  } catch (error) {
    console.error('❌ Error creating bulk notifications:', error);
    res.status(500).json({
      success: false,
      error: error.message,
    });
  }
});

/**
 * PUT /api/notifications/:notificationId/read
 * Mark notification as read
 */
app.put('/api/notifications/:notificationId/read', async (req, res) => {
  try {
    const { notificationId } = req.params;
    const { userId } = req.body;

    console.log(`✅ Marking notification as read: ${notificationId}`);

    const notification = await Notification.findOneAndUpdate(
      { notificationId, userId },
      { isRead: true },
      { new: true }
    );

    if (!notification) {
      return res.status(404).json({
        success: false,
        error: 'Notification not found',
      });
    }

    // Emit real-time update
    io.emit('notification:updated', {
      userId,
      notification: notification.toObject(),
    });

    res.json({
      success: true,
      data: notification,
      message: 'Notification marked as read',
    });
  } catch (error) {
    console.error('❌ Error marking notification as read:', error);
    res.status(500).json({
      success: false,
      error: error.message,
    });
  }
});

/**
 * PUT /api/notifications/user/:userId/mark-all-read
 * Mark all notifications as read for a user
 */
app.put('/api/notifications/user/:userId/mark-all-read', async (req, res) => {
  try {
    const { userId } = req.params;

    console.log(`✅ Marking all notifications as read for user: ${userId}`);

    const result = await Notification.updateMany(
      { userId, isRead: false },
      { isRead: true }
    );

    console.log(`✅ Marked ${result.modifiedCount} notifications as read`);

    // Emit real-time update
    io.emit('notifications:all-read', { userId });

    res.json({
      success: true,
      modifiedCount: result.modifiedCount,
      message: `${result.modifiedCount} notifications marked as read`,
    });
  } catch (error) {
    console.error('❌ Error marking all notifications as read:', error);
    res.status(500).json({
      success: false,
      error: error.message,
    });
  }
});

/**
 * DELETE /api/notifications/:notificationId
 * Delete a notification
 */
app.delete('/api/notifications/:notificationId', async (req, res) => {
  try {
    const { notificationId } = req.params;
    const { userId } = req.body;

    console.log(`🗑️ Deleting notification: ${notificationId}`);

    const notification = await Notification.findOneAndDelete({
      notificationId,
      userId,
    });

    if (!notification) {
      return res.status(404).json({
        success: false,
        error: 'Notification not found',
      });
    }

    // Emit real-time update
    io.emit('notification:deleted', {
      userId,
      notificationId,
    });

    res.json({
      success: true,
      message: 'Notification deleted successfully',
    });
  } catch (error) {
    console.error('❌ Error deleting notification:', error);
    res.status(500).json({
      success: false,
      error: error.message,
    });
  }
});

/**
 * DELETE /api/notifications/user/:userId/all
 * Delete all notifications for a user
 */
app.delete('/api/notifications/user/:userId/all', async (req, res) => {
  try {
    const { userId } = req.params;

    console.log(`🗑️ Deleting all notifications for user: ${userId}`);

    const result = await Notification.deleteMany({ userId });

    console.log(`✅ Deleted ${result.deletedCount} notifications`);

    // Emit real-time update
    io.emit('notifications:all-deleted', { userId });

    res.json({
      success: true,
      deletedCount: result.deletedCount,
      message: `${result.deletedCount} notifications deleted`,
    });
  } catch (error) {
    console.error('❌ Error deleting all notifications:', error);
    res.status(500).json({
      success: false,
      error: error.message,
    });
  }
});

// Health check
app.get('/health', (req, res) => {
  res.json({
    status: 'OK',
    service: 'Notifications Service',
    mongodb: mongoose.connection.readyState === 1 ? 'Connected' : 'Disconnected',
    timestamp: new Date().toISOString(),
  });
});

// Socket.IO connection handling
io.on('connection', (socket) => {
  console.log('🔌 Client connected to notifications service');

  socket.on('disconnect', () => {
    console.log('🔌 Client disconnected from notifications service');
  });
});

// Start server
const PORT = process.env.NOTIFICATIONS_PORT || 5017;
server.listen(PORT, () => {
  console.log(`\n🚀 Notifications Service running on port ${PORT}`);
  console.log(`📊 MongoDB: titan_drilling`);
  console.log(`🔗 Health check: https://api.titandrillingzm.com:6017/health\n`);
});

// Graceful shutdown
process.on('SIGTERM', () => {
  console.log('SIGTERM signal received: closing HTTP server');
  server.close(() => {
    console.log('HTTP server closed');
    mongoose.connection.close(false, () => {
      console.log('MongoDB connection closed');
      process.exit(0);
    });
  });
});

module.exports = app;

