const express = require('express');
const mongoose = require('mongoose');
const cors = require('cors');
const nodemailer = require('nodemailer');
const http = require('http');

const app = express();

// Middleware
app.use(cors());
app.use(express.json());

// MongoDB configuration
const MONGO_HOST = process.env.MONGO_HOST || '168.231.113.219';
const MONGO_PORT = process.env.MONGO_PORT || '27017';
const MONGO_USER = process.env.MONGO_USER || 'titandrillingadminuser';
const MONGO_PASSWORD = process.env.MONGO_PASSWORD || 'Titandrilling_@2024';
const MONGO_DB = process.env.MONGO_DB || 'titan_drilling';

// MongoDB connection - use existing connection if available
if (mongoose.connection.readyState === 0) {
  const mongoURI = `mongodb://${MONGO_USER}:${MONGO_PASSWORD}@${MONGO_HOST}:${MONGO_PORT}/${MONGO_DB}?authSource=admin`;

  mongoose.connect(mongoURI, {
    useNewUrlParser: true,
    useUnifiedTopology: true,
  })
  .then(() => {
    console.log('✅ MongoDB connected successfully for Incident Notifications');
  })
  .catch((err) => {
    console.error('❌ MongoDB connection error:', err);
    process.exit(1);
  });
} else {
  console.log('✅ Incident Notifications: Using existing MongoDB connection');
}

// GlobalUser Schema
const GlobalUserSchema = new mongoose.Schema({
  _id: String,
  name: String,
  email: String,
  role: String,
  countries: [String],
  country: String,
  projects: [String],
  project: String,
  fcmTokens: [{
    token: String,
    device: String,
    updatedAt: Date
  }],
  fcmToken: String,
}, { strict: false, collection: 'GlobalUsers' });

// Check if model exists to prevent OverwriteModelError
let GlobalUser;
try {
  GlobalUser = mongoose.model('GlobalUser');
  console.log('✅ [IncidentNotifications] Using existing GlobalUser model');
} catch (error) {
  GlobalUser = mongoose.model('GlobalUser', GlobalUserSchema);
  console.log('✅ [IncidentNotifications] Created new GlobalUser model');
}

// Email transporter configuration
const emailTransporter = nodemailer.createTransport({
  host: 'titandrillingzm.com',
  port: 587,
  secure: false,
  auth: {
    user: 'info@titandrillingzm.com',
    pass: 'L!A3Ys1Am(N(b)jU',
  },
});

// Helper function to send push notifications
const sendPushNotifications = async (incidentData, createdByName, recipients) => {
  try {
    console.log('📱 Sending push notifications for incident report...');
    
    const notificationData = {
      userIds: recipients.map(r => r._id),
      title: '🚨 New Incident Report',
      body: `${incidentData.incidentNumber}\nSeverity: ${incidentData.severity || 'N/A'}\nProject: ${incidentData.project || 'N/A'}\n\nReported by: ${createdByName}`,
      data: {
        type: 'incident_report',
        action: 'view_incident',
        incidentId: incidentData._id || incidentData.id,
        incidentNumber: incidentData.incidentNumber,
        severity: incidentData.severity || '',
        incidentCategory: incidentData.incidentCategory || '',
        project: incidentData.project || '',
        country: incidentData.country || '',
        reporter: createdByName,
        timestamp: new Date().toISOString(),
      },
      // iOS specific settings
      apns: {
        payload: {
          aps: {
            alert: {
              title: '🚨 New Incident Report',
              body: `${incidentData.incidentNumber}\nSeverity: ${incidentData.severity || 'N/A'}\n\nReported by: ${createdByName}`
            },
            sound: 'default',
            badge: 1,
            'content-available': 1,
          }
        }
      },
      // Android specific settings
      android: {
        priority: 'high',
        notification: {
          sound: 'default',
          channelId: 'incident_notifications',
        }
      }
    };
    
    // Send push via PushNotificationHandler (port 5016)
    const result = await new Promise((resolve, reject) => {
      const postData = JSON.stringify(notificationData);
      const options = {
        hostname: '168.231.113.219',
        port: 5016,
        path: '/push-notifications/send-to-users',
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Content-Length': Buffer.byteLength(postData),
        },
      };
      
      const req = http.request(options, (res) => {
        let data = '';
        res.on('data', (chunk) => { data += chunk; });
        res.on('end', () => {
          try {
            const response = JSON.parse(data);
            resolve(response);
          } catch (e) {
            reject(new Error('Invalid JSON response from push service'));
          }
        });
      });
      
      req.on('error', (error) => {
        reject(error);
      });
      
      req.write(postData);
      req.end();
    });
    
    console.log(`✅ Push notifications sent to ${result.successCount || 0} device(s)`);
    return result;
  } catch (error) {
    console.error('❌ Error sending push notifications:', error);
    return { success: false, error: error.message };
  }
};

// Helper function to save notifications to MongoDB
const saveNotificationsToMongoDB = async (incidentData, createdByName, recipients) => {
  try {
    console.log('💾 Saving incident notifications to MongoDB...');
    
    const notifications = recipients.map(recipient => ({
      notificationId: `incident_${Date.now()}_${Math.random().toString(36).substr(2, 9)}_${recipient._id}`,
      userId: recipient._id,
      title: '🚨 New Incident Report',
      body: `${incidentData.incidentNumber}\nSeverity: ${incidentData.severity || 'N/A'}\nProject: ${incidentData.project || 'N/A'}\n\nReported by: ${createdByName}`,
      type: 'incident_report',
      timestamp: Date.now(),
      isRead: false,
      data: {
        type: 'incident_report',
        action: 'view_incident',
        incidentId: incidentData._id || incidentData.id,
        incidentNumber: incidentData.incidentNumber,
        severity: incidentData.severity || '',
        incidentCategory: incidentData.incidentCategory || '',
        project: incidentData.project || '',
        country: incidentData.country || '',
        reporter: createdByName,
        timestamp: new Date().toISOString(),
      }
    }));
    
    // Save to MongoDB via Notifications API (port 5017)
    const result = await new Promise((resolve, reject) => {
      const postData = JSON.stringify({ notifications });
      const options = {
        hostname: '168.231.113.219',
        port: 5017,
        path: '/api/notifications/bulk',
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Content-Length': Buffer.byteLength(postData),
        },
      };
      
      const req = http.request(options, (res) => {
        let data = '';
        res.on('data', (chunk) => { data += chunk; });
        res.on('end', () => {
          try {
            const response = JSON.parse(data);
            resolve(response);
          } catch (e) {
            reject(new Error('Invalid JSON response from notifications service'));
          }
        });
      });
      
      req.on('error', (error) => {
        reject(error);
      });
      
      req.write(postData);
      req.end();
    });
    
    console.log(`✅ Saved ${result.count || notifications.length} notification(s) to MongoDB`);
    return result;
  } catch (error) {
    console.error('❌ Error saving notifications to MongoDB:', error);
    return { success: false, error: error.message };
  }
};

// Helper function to send email to reporter with PDF attachment
const sendReporterEmail = async (incidentData, createdByName, reporterEmail, pdfUrl) => {
  try {
    if (!reporterEmail) {
      console.log('⚠️ No email address for reporter, skipping reporter email');
      return { success: false, error: 'No email address' };
    }
    
    console.log(`📧 Sending confirmation email to reporter: ${createdByName} (${reporterEmail})`);
    
    // Prepare attachments if PDF URL is provided
    const attachments = [];
    if (pdfUrl) {
      try {
        // Download PDF from URL and attach
        const https = require('https');
        const http = require('http');
        
        const pdfBuffer = await new Promise((resolve, reject) => {
          const protocol = pdfUrl.startsWith('https') ? https : http;
          protocol.get(pdfUrl, (response) => {
            const chunks = [];
            response.on('data', (chunk) => chunks.push(chunk));
            response.on('end', () => resolve(Buffer.concat(chunks)));
            response.on('error', reject);
          }).on('error', reject);
        });
        
        attachments.push({
          filename: `Incident_Report_${incidentData.incidentNumber}.pdf`,
          content: pdfBuffer,
          contentType: 'application/pdf'
        });
        
        console.log(`📎 PDF attachment prepared for reporter: ${incidentData.incidentNumber}.pdf`);
      } catch (pdfError) {
        console.error('⚠️ Failed to download PDF for reporter email:', pdfError.message);
        // Continue without attachment
      }
    }
    
    const emailSubject = `Incident Report Submitted: ${incidentData.incidentNumber}`;
    const emailHtml = `
<!DOCTYPE html>
<html>
<head>
  <style>
    body { font-family: Arial, sans-serif; background-color: #f4f4f4; margin: 0; padding: 20px; }
    .container { max-width: 600px; margin: 0 auto; background-color: #ffffff; border-radius: 8px; overflow: hidden; box-shadow: 0 2px 4px rgba(0,0,0,0.1); }
    .header { background: linear-gradient(135deg, #4CAF50 0%, #45a049 100%); color: #ffffff; padding: 40px 20px; text-align: center; }
    .header h1 { margin: 0; font-size: 28px; font-weight: bold; }
    .success-box { background-color: #f1f8f4; border-left: 4px solid #4CAF50; padding: 20px; margin: 20px 0; border-radius: 4px; }
    .info-box { background-color: #f8f9fa; padding: 15px; margin: 20px 0; border-radius: 4px; }
    .info-row { margin: 8px 0; }
    .info-label { font-weight: bold; color: #333; }
    .info-value { color: #666; }
    .severity-badge { display: inline-block; padding: 4px 12px; border-radius: 12px; font-size: 12px; font-weight: bold; }
    .severity-high { background-color: #f44336; color: white; }
    .severity-medium { background-color: #ff9800; color: white; }
    .severity-low { background-color: #ffc107; color: white; }
    .footer { background-color: #f8f9fa; padding: 20px; text-align: center; font-size: 12px; color: #999; }
    .icon { font-size: 64px; margin-bottom: 10px; }
  </style>
</head>
<body>
  <div class="container">
    <div class="header">
      <div class="icon">✅</div>
      <h1>Incident Report Submitted</h1>
    </div>
    <div class="content" style="padding: 30px 20px;">
      <div class="success-box">
        <h2 style="margin-top: 0; color: #4CAF50;">Report Submitted Successfully!</h2>
        <p>Your incident report has been successfully submitted and saved.</p>
      </div>
      
      <h3 style="color: #333;">Report Details:</h3>
      <div class="info-box">
        <div class="info-row">
          <span class="info-label">Incident Number:</span>
          <span class="info-value">${incidentData.incidentNumber}</span>
        </div>
        <div class="info-row">
          <span class="info-label">Severity:</span>
          <span class="severity-badge severity-${incidentData.severity?.toLowerCase() || 'low'}">${incidentData.severity || 'N/A'}</span>
        </div>
        <div class="info-row">
          <span class="info-label">Category:</span>
          <span class="info-value">${incidentData.incidentCategory || 'N/A'}</span>
        </div>
        <div class="info-row">
          <span class="info-label">Country:</span>
          <span class="info-value">${incidentData.country || 'N/A'}</span>
        </div>
        <div class="info-row">
          <span class="info-label">Project:</span>
          <span class="info-value">${incidentData.project || 'N/A'}</span>
        </div>
        <div class="info-row">
          <span class="info-label">Reported by:</span>
          <span class="info-value">${createdByName}</span>
        </div>
        <div class="info-row">
          <span class="info-label">Date:</span>
          <span class="info-value">${new Date().toLocaleDateString('en-GB', { day: '2-digit', month: '2-digit', year: 'numeric', hour: '2-digit', minute: '2-digit' })}</span>
        </div>
      </div>
      
      <p style="color: #666;">Thank you for reporting this incident. The report has been sent to all relevant supervisors and managers for review.</p>
    </div>
    <div class="footer">
      <p>Titan Drilling - Incident Management System</p>
      <p>&copy; ${new Date().getFullYear()} Titan Drilling. All rights reserved.</p>
    </div>
  </div>
</body>
</html>
    `;
    
    const mailOptions = {
      from: '"Titan Drilling" <mandeep.owninfotech@gmail.com>',
      to: reporterEmail,
      subject: emailSubject,
      html: emailHtml,
    };
    
    // Add PDF attachment if available
    if (attachments.length > 0) {
      mailOptions.attachments = attachments;
    }
    
    await emailTransporter.sendMail(mailOptions);
    
    console.log(`✅ Confirmation email sent to reporter ${createdByName} (${reporterEmail})${attachments.length > 0 ? ' with PDF' : ''}`);
    return { success: true };
  } catch (error) {
    console.error('❌ Error sending reporter email:', error);
    return { success: false, error: error.message };
  }
};

// Helper function to send emails with PDF attachment
const sendEmails = async (incidentData, createdByName, recipients, pdfUrl) => {
  try {
    console.log('📧 Sending email notifications for incident report...');
    
    let successCount = 0;
    let failCount = 0;
    
    // Prepare attachments if PDF URL is provided
    const attachments = [];
    if (pdfUrl) {
      try {
        // Download PDF from URL and attach
        const https = require('https');
        const http = require('http');
        
        const pdfBuffer = await new Promise((resolve, reject) => {
          const protocol = pdfUrl.startsWith('https') ? https : http;
          protocol.get(pdfUrl, (response) => {
            const chunks = [];
            response.on('data', (chunk) => chunks.push(chunk));
            response.on('end', () => resolve(Buffer.concat(chunks)));
            response.on('error', reject);
          }).on('error', reject);
        });
        
        attachments.push({
          filename: `Incident_Report_${incidentData.incidentNumber}.pdf`,
          content: pdfBuffer,
          contentType: 'application/pdf'
        });
        
        console.log(`📎 PDF attachment prepared: ${incidentData.incidentNumber}.pdf`);
      } catch (pdfError) {
        console.error('⚠️ Failed to download PDF for attachment:', pdfError.message);
        // Continue without attachment
      }
    }
    
    // Email template
    const emailSubject = `New Incident Report: ${incidentData.incidentNumber}`;
    const emailHtml = `
<!DOCTYPE html>
<html>
<head>
  <style>
    body { font-family: Arial, sans-serif; background-color: #f4f4f4; margin: 0; padding: 20px; }
    .container { max-width: 600px; margin: 0 auto; background-color: #ffffff; border-radius: 8px; overflow: hidden; box-shadow: 0 2px 4px rgba(0,0,0,0.1); }
    .header { background: linear-gradient(135deg, #F44336 0%, #D32F2F 100%); color: #ffffff; padding: 30px 20px; text-align: center; }
    .header h1 { margin: 0; font-size: 24px; font-weight: bold; }
    .content { padding: 30px 20px; }
    .incident-info { background-color: #f8f9fa; border-left: 4px solid #F44336; padding: 15px; margin: 20px 0; border-radius: 4px; }
    .info-row { margin: 10px 0; }
    .info-label { font-weight: bold; color: #333; display: inline-block; width: 150px; }
    .info-value { color: #666; }
    .severity-badge { display: inline-block; padding: 4px 12px; border-radius: 12px; font-size: 12px; font-weight: bold; }
    .severity-high { background-color: #f44336; color: white; }
    .severity-medium { background-color: #ff9800; color: white; }
    .severity-low { background-color: #ffc107; color: white; }
    .footer { background-color: #f8f9fa; padding: 20px; text-align: center; font-size: 12px; color: #999; }
    .icon { font-size: 48px; margin-bottom: 10px; }
    .urgent { background-color: #ffebee; border: 2px solid #f44336; padding: 15px; margin: 20px 0; border-radius: 4px; }
    .urgent h3 { color: #f44336; margin: 0 0 10px 0; }
  </style>
</head>
<body>
  <div class="container">
    <div class="header">
      <div class="icon">🚨</div>
      <h1>New Incident Report</h1>
    </div>
    <div class="content">
      ${incidentData.severity === 'High' || incidentData.severity === 'Critical' ? `
      <div class="urgent">
        <h3>⚠️ URGENT ATTENTION REQUIRED</h3>
        <p>This is a high severity incident that requires immediate attention.</p>
      </div>` : ''}
      
      <p>Hello,</p>
      <p>A new incident report has been submitted in the Titan Drilling system.</p>
      
      <div class="incident-info">
        <h3 style="margin-top: 0; color: #F44336;">Incident Details:</h3>
        <div class="info-row">
          <span class="info-label">Incident Number:</span>
          <span class="info-value">${incidentData.incidentNumber}</span>
        </div>
        <div class="info-row">
          <span class="info-label">Severity:</span>
          <span class="severity-badge severity-${incidentData.severity?.toLowerCase() || 'low'}">${incidentData.severity || 'N/A'}</span>
        </div>
        <div class="info-row">
          <span class="info-label">Category:</span>
          <span class="info-value">${incidentData.incidentCategory || 'N/A'}</span>
        </div>
        <div class="info-row">
          <span class="info-label">Country:</span>
          <span class="info-value">${incidentData.country || 'N/A'}</span>
        </div>
        <div class="info-row">
          <span class="info-label">Project:</span>
          <span class="info-value">${incidentData.project || 'N/A'}</span>
        </div>
        <div class="info-row">
          <span class="info-label">Reported by:</span>
          <span class="info-value">${createdByName}</span>
        </div>
        ${incidentData.incidentArea ? `
        <div class="info-row">
          <span class="info-label">Incident Area:</span>
          <span class="info-value">${incidentData.incidentArea}</span>
        </div>` : ''}
        <div class="info-row">
          <span class="info-label">Date:</span>
          <span class="info-value">${new Date().toLocaleDateString('en-GB', { day: '2-digit', month: '2-digit', year: 'numeric', hour: '2-digit', minute: '2-digit' })}</span>
        </div>
      </div>
      
      <p style="color: #666;">Please review the incident report in the Titan Drilling application and take appropriate action.</p>
    </div>
    <div class="footer">
      <p>Titan Drilling - Incident Management System</p>
      <p>&copy; ${new Date().getFullYear()} Titan Drilling. All rights reserved.</p>
    </div>
  </div>
</body>
</html>
    `;
    
    // Send email to each recipient
    for (const recipient of recipients) {
      if (recipient.email) {
        try {
          const mailOptions = {
            from: '"Titan Drilling" <mandeep.owninfotech@gmail.com>',
            to: recipient.email,
            subject: emailSubject,
            html: emailHtml,
          };
          
          // Add PDF attachment if available
          if (attachments.length > 0) {
            mailOptions.attachments = attachments;
          }
          
          await emailTransporter.sendMail(mailOptions);
          
          console.log(`✅ Email sent to ${recipient.name} (${recipient.email})${attachments.length > 0 ? ' with PDF' : ''}`);
          successCount++;
        } catch (emailError) {
          console.error(`❌ Failed to send email to ${recipient.name}:`, emailError.message);
          failCount++;
        }
      }
    }
    
    console.log(`📧 Email summary: ${successCount} sent, ${failCount} failed`);
    return { success: true, sent: successCount, failed: failCount, total: recipients.length };
  } catch (error) {
    console.error('❌ Error sending emails:', error);
    return { success: false, error: error.message };
  }
};

/**
 * POST /api/incident-notifications/new-incident
 * Handle notifications when a new incident report is created
 */
app.post('/api/incident-notifications/new-incident', async (req, res) => {
  try {
    const { incidentData, createdByName, reporterEmail, pdfUrl } = req.body;
    
    if (!incidentData || !incidentData.incidentNumber) {
      return res.status(400).json({
        success: false,
        error: 'Incident data is required'
      });
    }
    
    console.log(`📎 PDF URL received: ${pdfUrl || 'Not provided'}`);
    
    console.log(`\n🔔 Processing notifications for incident: ${incidentData.incidentNumber}`);
    console.log(`📋 Country: ${incidentData.country}, Project: ${incidentData.project}`);
    console.log(`👤 Reporter: ${createdByName}`);
    console.log(`🚨 Severity: ${incidentData.severity}`);
    
    // Manager roles to notify
    const managerRoles = [
      'HrManager',
      'Workshop Manager',
      'Maintenance Manager',
      'FinanceManager',
      'Project Manager',
      'Site Manager'
    ];
    
    // Get all ProAdmins (no filtering)
    const proAdmins = await GlobalUser.find({
      role: 'ProAdmin'
    }).select('_id name email fcmToken fcmTokens');
    
    console.log(`📋 Found ${proAdmins.length} ProAdmin(s)`);
    
    // Get Supervisors filtered by country AND project
    // Note: countries and projects are stored as ARRAYS in MongoDB
    const supervisors = await GlobalUser.find({
      role: 'Supervisor',
      countries: incidentData.country,  // MongoDB matches if array contains value
      projects: incidentData.project
    }).select('_id name email fcmToken fcmTokens role countries projects');
    
    console.log(`📋 Found ${supervisors.length} Supervisor(s) for ${incidentData.country} - ${incidentData.project}`);
    
    // Get Managers filtered by country AND project
    // Note: countries and projects are stored as ARRAYS in MongoDB
    const managers = await GlobalUser.find({
      role: { $in: managerRoles },
      countries: incidentData.country,  // MongoDB matches if array contains value
      projects: incidentData.project
    }).select('_id name email fcmToken fcmTokens role countries projects');
    
    console.log(`📋 Found ${managers.length} Manager(s) for ${incidentData.country} - ${incidentData.project}`);
    
    // Combine all recipients
    const allRecipients = [...proAdmins, ...supervisors, ...managers];
    
    // Remove duplicates based on _id
    const uniqueRecipients = allRecipients.filter((recipient, index, self) =>
      index === self.findIndex((r) => r._id === recipient._id)
    );
    
    console.log(`📋 Total unique recipients: ${uniqueRecipients.length}`);
    
    if (uniqueRecipients.length === 0) {
      console.log('⚠️ No recipients found');
      return res.json({
        success: true,
        message: 'No recipients to notify',
        pushSent: 0,
        emailsSent: 0,
        notificationsSaved: 0,
      });
    }
    
    // Execute all notification tasks in parallel
    const [mongoResult, pushResult, emailResult, reporterEmailResult] = await Promise.all([
      saveNotificationsToMongoDB(incidentData, createdByName, uniqueRecipients),
      sendPushNotifications(incidentData, createdByName, uniqueRecipients),
      sendEmails(incidentData, createdByName, uniqueRecipients, pdfUrl),
      sendReporterEmail(incidentData, createdByName, reporterEmail, pdfUrl),
    ]);
    
    // Log results
    console.log('\n📊 Notification Results:');
    console.log(`  💾 MongoDB: ${mongoResult.success ? '✅' : '❌'} (${mongoResult.count || 0} notifications)`);
    console.log(`  📱 Push: ${pushResult.success ? '✅' : '❌'} (${pushResult.successCount || 0} devices)`);
    console.log(`  📧 Emails to Recipients: ${emailResult.success ? '✅' : '❌'} (${emailResult.sent || 0} sent)`);
    console.log(`  📧 Reporter Email: ${reporterEmailResult.success ? '✅' : '❌'}\n`);
    
    res.json({
      success: true,
      message: `Notifications sent for incident: ${incidentData.incidentNumber}`,
      results: {
        mongodb: {
          success: mongoResult.success,
          saved: mongoResult.count || 0,
          error: mongoResult.error || null,
        },
        push: {
          success: pushResult.success,
          sent: pushResult.successCount || 0,
          failed: pushResult.failureCount || 0,
          error: pushResult.error || null,
        },
        emails: {
          success: emailResult.success,
          sent: emailResult.sent || 0,
          failed: emailResult.failed || 0,
          total: emailResult.total || 0,
          error: emailResult.error || null,
        },
        reporterEmail: {
          success: reporterEmailResult.success,
          error: reporterEmailResult.error || null,
        },
      },
      recipients: {
        proAdmins: proAdmins.length,
        supervisors: supervisors.length,
        managers: managers.length,
        total: uniqueRecipients.length,
      }
    });
  } catch (error) {
    console.error('❌ Error processing incident notifications:', error);
    res.status(500).json({
      success: false,
      error: error.message,
    });
  }
});

// Health check
app.get('/health', (req, res) => {
  res.json({
    status: 'OK',
    service: 'Incident Notifications Service',
    mongodb: mongoose.connection.readyState === 1 ? 'Connected' : 'Disconnected',
    timestamp: new Date().toISOString(),
  });
});

// Start server
const PORT = process.env.INCIDENT_NOTIFICATIONS_PORT || 5021;
app.listen(PORT, () => {
  console.log(`\n🚀 Incident Notifications Service running on port ${PORT}`);
  console.log(`📊 MongoDB: titan_drilling`);
  console.log(`🔗 Health check: http://168.231.113.219:${PORT}/health\n`);
});

// Graceful shutdown
process.on('SIGTERM', () => {
  console.log('SIGTERM signal received: closing HTTP server');
  app.close(() => {
    console.log('HTTP server closed');
    mongoose.connection.close(false, () => {
      console.log('MongoDB connection closed');
      process.exit(0);
    });
  });
});

module.exports = app;

