const express = require('express');
const cors = require('cors');
const fs = require('fs');
const path = require('path');
const { mongoose } = require('./dbConnection');

const app = express();
const PORT = process.env.VERSION_MANAGER_PORT || 5036;

app.use(cors());
app.use(express.json());

const VersionSchema = new mongoose.Schema(
  {},
  { collection: 'versions', strict: false, timestamps: true }
);

let VersionModel;
try {
  VersionModel = mongoose.model('AppVersion');
} catch (error) {
  VersionModel = mongoose.model('AppVersion', VersionSchema);
}

const readIOSBundleVersion = () => {
  try {
    const plistPath = path.resolve(__dirname, '../ios/titan_drilling/Info.plist');
    const plistContent = fs.readFileSync(plistPath, 'utf8');
    const match = plistContent.match(
      /<key>CFBundleShortVersionString<\/key>\s*<string>([^<]+)<\/string>/
    );
    return match ? match[1].trim() : null;
  } catch (error) {
    console.warn('⚠️ Unable to read iOS bundle version:', error.message);
    return null;
  }
};

const readAndroidVersionName = () => {
  try {
    const gradlePath = path.resolve(__dirname, '../android/app/build.gradle');
    const gradleContent = fs.readFileSync(gradlePath, 'utf8');
    const match = gradleContent.match(/versionName\s+"([^"]+)"/);
    return match ? match[1].trim() : null;
  } catch (error) {
    console.warn('⚠️ Unable to read Android version name:', error.message);
    return null;
  }
};

const iosBundleVersion = readIOSBundleVersion();
const androidVersionName = readAndroidVersionName();

const normalizeVersionValue = (doc, keys) => {
  for (const key of keys) {
    if (doc[key] !== undefined && doc[key] !== null && doc[key] !== '') {
      return doc[key];
    }
  }
  return null;
};

const buildResponsePayload = (doc) => {
  let androidVersion = normalizeVersionValue(doc, [
    'Android_version',
    'android_version',
    'androidVersion',
    'android',
  ]);

  let iosVersion = normalizeVersionValue(doc, [
    'IOS_version',
    'iOS_version',
    'ios_version',
    'iosVersion',
    'ios',
  ]);

  if (androidVersionName) {
    androidVersion = androidVersionName;
  }

  if (iosBundleVersion) {
    iosVersion = iosBundleVersion;
  }

  const releaseNotes = normalizeVersionValue(doc, [
    'releaseNotes',
    'changelog',
    'notes',
  ]);

  return {
    success: true,
    data: {
      android: {
        latestVersion: androidVersion,
      },
      ios: {
        latestVersion: iosVersion,
      },
    },
    updatedAt: doc.updatedAt || doc.lastUpdated || doc.createdAt,
    metadata: {
      releaseNotes,
      source: 'mongo:versions',
      documentId: doc._id,
      updatedBy: doc.updatedBy || null,
    },
  };
};

app.get('/', (req, res) => {
  res.json({
    status: 'Version Manager is running',
    port: PORT,
    collection: 'versions',
  });
});

app.get('/app/version', async (req, res) => {
  try {
    const latestDoc = await VersionModel.findOne({})
      .sort({ updatedAt: -1, lastUpdated: -1, createdAt: -1 })
      .lean();

    if (!latestDoc) {
      return res.status(404).json({
        success: false,
        error: 'Version document not found',
      });
    }

    return res.json(buildResponsePayload(latestDoc));
  } catch (error) {
    console.error('❌ Version Manager error:', error);
    return res.status(500).json({
      success: false,
      error: 'Failed to fetch version information',
      details: error.message,
    });
  }
});

app.listen(PORT, () => {
  console.log(`🆕 Version Manager server running on port ${PORT}`);
});

