const BASE_URL = 'https://api.titandrillingzm.com:6032';

const jsonRequest = async (endpoint, { method = 'GET', body, headers = {} } = {}) => {
  try {
    const response = await fetch(`${BASE_URL}${endpoint}`, {
      method,
      headers: {
        'Content-Type': 'application/json',
        ...headers,
      },
      body,
    });

    const data = await response.json().catch(() => ({ success: false, error: 'Invalid server response' }));

    if (!response.ok) {
      return {
        success: false,
        status: response.status,
        error: data.error || data.message || `Request failed with status ${response.status}`,
      };
    }

    return data;
  } catch (error) {
    console.error('[HSECorrectiveActionAPI] Network error:', error);
    return {
      success: false,
      error: error.message || 'Network request failed',
    };
  }
};

const HSECorrectiveActionAPI = {
  getCorrectiveAction: async (reportId) => {
    if (!reportId) {
      return { success: false, error: 'reportId is required' };
    }
    const result = await jsonRequest(`/hse-corrective-actions/${reportId}`);
    if (result.status === 404) {
      return { success: false, error: 'Corrective action not found' };
    }
    return result;
  },

  saveCorrectiveAction: async (reportId, userId, correctiveActionData) => {
    if (!reportId || !userId) {
      return { success: false, error: 'reportId and userId are required' };
    }
    return await jsonRequest('/hse-corrective-actions', {
      method: 'POST',
      body: JSON.stringify({
        reportId,
        userId,
        correctiveActionData,
      }),
    });
  },

  uploadCorrectiveActionPdf: async (reportId, formData, userId, fileName) => {
    if (!reportId || !formData) {
      return { success: false, error: 'reportId and formData are required' };
    }
    if (userId) {
      formData.append('userId', userId);
    }
    if (fileName) {
      formData.append('fileName', fileName);
    }

    try {
      const response = await fetch(`${BASE_URL}/hse-corrective-actions/${reportId}/upload-pdf`, {
        method: 'POST',
        headers: {
          'Content-Type': 'multipart/form-data',
        },
        body: formData,
      });

      const data = await response.json().catch(() => ({ success: false, error: 'Invalid server response' }));

      if (!response.ok) {
        return {
          success: false,
          error: data.error || data.message || `Request failed with status ${response.status}`,
        };
      }

      if (data.downloadUrl && !data.downloadUrl.startsWith('http')) {
        data.downloadUrl = `${BASE_URL}${data.downloadUrl}`;
      }

      return data;
    } catch (error) {
      console.error('[HSECorrectiveActionAPI] PDF upload error:', error);
      return { success: false, error: error.message || 'Failed to upload PDF' };
    }
  },

  deleteCorrectiveAction: async (reportId) => {
    if (!reportId) {
      return { success: false, error: 'reportId is required' };
    }
    return await jsonRequest(`/hse-corrective-actions/${reportId}`, {
      method: 'DELETE',
    });
  },

  getPdfDownloadUrl: (userId, fileName) => {
    if (!userId || !fileName) return null;
    return `${BASE_URL}/hse-corrective-actions/download-pdf/${userId}/${fileName}`;
  },
};

export default HSECorrectiveActionAPI;

