// Push Notification API Configuration
// Reads FCM tokens from MongoDB, sends via Firebase Admin SDK

const BASE_URL = 'https://api.titandrillingzm.com:6016';

const PushNotificationAPI = {
  // Send push notification to a single user
  sendToUser: async (userId, title, body, data = {}) => {
    try {
      console.log('📤 Sending push notification via MongoDB + Firebase...');
      console.log('👤 User:', userId);
      console.log('📋 Title:', title);
      
      const response = await fetch(`${BASE_URL}/push-notifications/send`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          userId,
          title,
          body,
          data
        }),
      });

      const result = await response.json();
      console.log('📥 Push notification response:', result);

      return result;
    } catch (error) {
      console.error('❌ Error sending push notification:', error);
      return { success: false, error: error.message };
    }
  },

  // Send push notification to multiple users
  sendToMultipleUsers: async (userIds, title, body, data = {}) => {
    try {
      console.log(`📤 Sending push notification to ${userIds.length} users...`);
      
      const response = await fetch(`${BASE_URL}/push-notifications/send-multiple`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          userIds,
          title,
          body,
          data
        }),
      });

      const result = await response.json();
      console.log('📥 Push notification response:', result);

      return result;
    } catch (error) {
      console.error('❌ Error sending push notifications:', error);
      return { success: false, error: error.message };
    }
  },

  // Send welcome notification
  sendWelcomeNotification: async (userId, userName) => {
    const title = `Welcome ${userName}! 👋`;
    const body = `You've successfully logged in to Titan Drilling. Have a productive day!`;
    const data = {
      type: 'welcome',
      action: 'view_home',
      timestamp: new Date().toISOString(),
    };

    return await PushNotificationAPI.sendToUser(userId, title, body, data);
  },

  // Send notification to ALL users
  sendToAllUsers: async (title, body, data = {}) => {
    try {
      console.log('📤 Sending push notification to all users...');
      console.log('🌐 Target URL:', `${BASE_URL}/push-notifications/send-to-all`);
      
      // Create AbortController for timeout
      const controller = new AbortController();
      const timeoutId = setTimeout(() => controller.abort(), 30000); // 30 second timeout
      
      const response = await fetch(`${BASE_URL}/push-notifications/send-to-all`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Accept': 'application/json',
        },
        body: JSON.stringify({
          title,
          body,
          data
        }),
        signal: controller.signal,
      });

      clearTimeout(timeoutId);

      console.log('📡 Response received - Status:', response.status);
      console.log('📡 Response headers:', {
        contentType: response.headers.get('content-type'),
        contentLength: response.headers.get('content-length')
      });

      // Check if response is OK
      if (!response.ok) {
        const text = await response.text();
        console.error('❌ Server error (HTTP ' + response.status + '):', text.substring(0, 200));
        throw new Error(`Server returned ${response.status}: ${text.substring(0, 100)}`);
      }

      // Check if response is JSON
      const contentType = response.headers.get('content-type');
      if (!contentType || !contentType.includes('application/json')) {
        const text = await response.text();
        console.error('❌ Non-JSON response received:', text.substring(0, 200));
        throw new Error('Server returned non-JSON response (got HTML or plain text)');
      }

      const result = await response.json();
      console.log('📥 Push notification sent successfully:', result);

      return result;
    } catch (error) {
      console.error('❌ Error sending push notification to all users:', error);
      
      // Provide more specific error messages
      if (error.name === 'AbortError') {
        console.error('⏱️ Request timeout after 30 seconds');
        return { success: false, error: 'Request timeout - server did not respond in time' };
      } else if (error.message.includes('Network request failed')) {
        console.error('🌐 Network error - cannot reach server');
        return { success: false, error: 'Network error - check if backend server is running' };
      }
      
      return { success: false, error: error.message };
    }
  },

  // Send new notice notification to all users
  sendNewNoticeNotification: async (noticeTitle, noticeDescription, noticeId, creatorName) => {
    const title = `📢 New Notice Coming`;
    const description = noticeDescription.length > 80 
      ? noticeDescription.substring(0, 80) + '...' 
      : noticeDescription;
    const body = `${description}\n\nBy-- ${creatorName}`;
    const data = {
      type: 'new_notice',
      action: 'view_notice',
      noticeId: noticeId,
      noticeTitle: noticeTitle,
      creatorName: creatorName,
      timestamp: new Date().toISOString(),
    };

    return await PushNotificationAPI.sendToAllUsers(title, body, data);
  },

  // Send notification to admins and supervisors only
  sendToAdminsAndSupervisors: async (title, body, data = {}) => {
    try {
      console.log('📤 Sending push notification to admins and supervisors...');
      
      const response = await fetch(`${BASE_URL}/push-notifications/send-to-roles`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Accept': 'application/json',
        },
        body: JSON.stringify({
          roles: ['ProAdmin', 'Supervisor'],
          title,
          body,
          data
        }),
      });

      const contentType = response.headers.get('content-type');
      if (!contentType || !contentType.includes('application/json')) {
        const text = await response.text();
        console.error('❌ Non-JSON response:', text.substring(0, 200));
        throw new Error('Server returned non-JSON response');
      }

      const result = await response.json();
      console.log('📥 Push notification to admins/supervisors response:', result);

      return result;
    } catch (error) {
      console.error('❌ Error sending push notification to admins/supervisors:', error);
      return { success: false, error: error.message };
    }
  },

  // Send new equipment notification to admins and supervisors
  sendNewEquipmentNotification: async (equipmentName, mainCategory, country, creatorName) => {
    const title = `🚜 New Equipment Added`;
    const body = `${equipmentName || mainCategory} has been added to the system.\n\nAdded by: ${creatorName}`;
    const data = {
      type: 'equipment_added',
      action: 'view_equipment',
      equipmentName: equipmentName,
      mainCategory: mainCategory,
      country: country,
      creatorName: creatorName,
      timestamp: new Date().toISOString(),
    };

    return await PushNotificationAPI.sendToAdminsAndSupervisors(title, body, data);
  },
};

export default PushNotificationAPI;

