import React from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  FlatList,
  Dimensions,
} from 'react-native';
import { useSafeAreaInsets } from 'react-native-safe-area-context';

const STATUS_OPTIONS = [
  { label: 'Fixed', value: 'Fixed', color: '#38a169' },
  { label: 'Not Fixed', value: 'Not Fixed', color: '#e53e3e' },
  { label: 'No Need', value: 'No Need', color: '#718096' },
];

const { width } = Dimensions.get('window');

const HSEcorrectiveactionStep1 = ({
  checklistTitle,
  groupedSections = [],
  unsafeItems = [],
  unsafeTotal = 0,
  subItemStatuses = {},
  onChangeStatus = () => {},
  onNext,
  onCancel,
}) => {
  const insets = useSafeAreaInsets();
  const sections = Array.isArray(groupedSections) ? groupedSections : [];
  const flattenedUnsafeItems = sections.reduce((acc, group) => {
    const sectionName = group?.section || 'General Area';
    const items = Array.isArray(group?.items) ? group.items : [];
    items.forEach((item) => {
      acc.push({
        ...item,
        sectionName,
      });
    });
    return acc;
  }, []);
  const totalUnsafeItems = flattenedUnsafeItems.length;
  const allAnswered =
    totalUnsafeItems === 0 ||
    flattenedUnsafeItems.every((item) => subItemStatuses[item.id]);

  const renderHeader = () => (
    <View style={styles.headerContainer}>
      <Text style={styles.mainHeading}>{checklistTitle || 'HSE Inspection'}</Text>
      <Text style={styles.title}>Corrective Action</Text>
      <Text style={styles.subtitle}>
        Review each unsafe observation and capture the corrective action status before proceeding.
      </Text>

      <View style={styles.summaryCard}>
        <Text style={styles.summaryLabel}>Total Unsafe Items</Text>
        <Text style={styles.summaryValue}>{totalUnsafeItems}</Text>
        <Text style={styles.summaryHint}>
          {totalUnsafeItems === 0
            ? 'No unsafe items were recorded in this inspection.'
            : `Update the status for all ${totalUnsafeItems} unsafe item${totalUnsafeItems === 1 ? '' : 's'} to continue.`}
        </Text>
      </View>
    </View>
  );

  const renderEmptyComponent = () => (
    <View style={styles.emptyState}>
      <Text style={styles.emptyText}>No unsafe items require corrective action.</Text>
    </View>
  );

  const renderItem = ({ item, index }) => {
    const selected = subItemStatuses[item.id];
    return (
      <View style={styles.itemCard}>
        <View style={styles.itemHeader}>
          <View style={styles.badge}>
            <Text style={styles.badgeText}>{index + 1}</Text>
          </View>
          <View style={styles.itemTextWrapper}>
            <Text style={styles.sectionPill}>{item.sectionName || 'General Area'}</Text>
            <Text style={styles.itemTitle}>{item.title}</Text>
            <View style={styles.itemStatsRow}>
              <View style={[styles.statBadge, styles.safeBadge]}>
                <Text style={styles.statLabel}>Safe</Text>
                <Text style={styles.statValue}>{item.safeCount ?? 0}</Text>
              </View>
              <View style={[styles.statBadge, styles.unsafeBadge]}>
                <Text style={styles.statLabel}>Unsafe</Text>
                <Text style={styles.statValue}>{item.unsafeCount ?? 0}</Text>
              </View>
              {Number(item.naCount) > 0 && (
                <View style={[styles.statBadge, styles.naBadge]}>
                  <Text style={styles.statLabel}>N/A</Text>
                  <Text style={styles.statValue}>{item.naCount}</Text>
                </View>
              )}
            </View>
            {!!item.selection && (
              <Text style={styles.itemSelection}>Recorded status: {String(item.selection)}</Text>
            )}
          </View>
        </View>

        <View style={styles.statusRow}>
          {STATUS_OPTIONS.map((option) => {
            const isSelected = selected === option.value;
            return (
              <TouchableOpacity
                key={option.value}
                style={[
                  styles.statusButton,
                  isSelected && { backgroundColor: option.color, borderColor: option.color, shadowColor: option.color },
                ]}
                onPress={() => onChangeStatus(item.id, option.value)}
                activeOpacity={0.9}
              >
                <Text style={[styles.statusButtonText, isSelected && styles.statusButtonTextSelected]}>
                  {option.label}
                </Text>
              </TouchableOpacity>
            );
          })}
        </View>
      </View>
    );
  };

  return (
    <View style={styles.container}>
      <FlatList
        data={flattenedUnsafeItems}
        keyExtractor={(item, idx) => item.id || `${item.sectionName}-${idx}`}
        renderItem={renderItem}
        contentContainerStyle={[styles.listContent, { paddingBottom: insets.bottom + 96 }]}
        showsVerticalScrollIndicator={false}
        ListHeaderComponent={renderHeader}
        ListEmptyComponent={renderEmptyComponent}
      />

      <View style={[styles.bottomBar, { paddingBottom: Math.max(insets.bottom, 12) }]}>
        <TouchableOpacity style={styles.secondaryButton} onPress={onCancel}>
          <Text style={styles.secondaryButtonText}>Cancel</Text>
        </TouchableOpacity>
        <TouchableOpacity
          style={[styles.primaryButton, !allAnswered && styles.primaryButtonDisabled]}
          onPress={onNext}
          disabled={!allAnswered}
        >
          <Text style={styles.primaryButtonText}>Next</Text>
        </TouchableOpacity>
      </View>
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  listContent: {
    paddingTop: width * 0.05,
    paddingHorizontal: width * 0.06,
  },
  headerContainer: {
    marginBottom: width * 0.04,
  },
  mainHeading: {
    fontSize: width * 0.045,
    fontWeight: '700',
    color: '#195c85',
    textTransform: 'uppercase',
    letterSpacing: 0.6,
    marginBottom: width * 0.01,
  },
  title: {
    fontSize: width * 0.055,
    fontWeight: '700',
    color: '#05375a',
  },
  subtitle: {
    fontSize: width * 0.035,
    color: '#4a5568',
    marginTop: width * 0.02,
    lineHeight: width * 0.05,
  },
  summaryCard: {
    marginTop: width * 0.05,
    // backgroundColor: '#f7f9fc',
    // borderRadius: width * 0.04,
    paddingVertical: width * 0.045,
    paddingHorizontal: width * 0.05,
    borderWidth: 1,
    borderColor: '#e2e8f0',
   
  },
  summaryLabel: {
    fontSize: width * 0.035,
    color: '#718096',
    fontWeight: '600',
  },
  summaryValue: {
    marginTop: width * 0.02,
    fontSize: 22,
    fontWeight: '700',
    color: '#1a202c',
  },
  summaryHint: {
    marginTop: width * 0.025,
    fontSize: width * 0.032,
    color: '#4a5568',
    lineHeight: width * 0.045,
  },
  itemCard: {
    marginBottom: width * 0.05,
    // backgroundColor: '#fff',
    // borderRadius: width * 0.035,
    padding: width * 0.045,
    borderWidth: 1,
    borderColor: '#e2e8f0',
  
  },
  itemHeader: {
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: width * 0.035,
  },
  badge: {
    width: width * 0.085,
    height: width * 0.085,
    borderRadius: width * 0.0425,
    backgroundColor: '#195c85',
    alignItems: 'center',
    justifyContent: 'center',
    marginRight: width * 0.025,
  },
  badgeText: {
    color: '#fff',
    fontWeight: '700',
    fontSize: width * 0.035,
  },
  itemTextWrapper: {
    flex: 1,
  },
  sectionPill: {
    alignSelf: 'flex-start',
    backgroundColor: '#eef7ff',
    color: '#195c85',
    fontWeight: '600',
    fontSize: width * 0.032,
    paddingHorizontal: width * 0.03,
    paddingVertical: width * 0.015,
    borderRadius: width * 0.025,
    marginBottom: width * 0.02,
  },
  itemTitle: {
    fontSize: width * 0.042,
    fontWeight: '600',
    color: '#1a202c',
  },
  itemStatsRow: {
    flexDirection: 'row',
    gap: 8,
    marginTop: width * 0.025,
  },
  statBadge: {
    flexDirection: 'row',
    alignItems: 'center',
    gap: 4,
    paddingHorizontal: width * 0.03,
    paddingVertical: width * 0.015,
    borderRadius: width * 0.03,
  },
  safeBadge: {
    backgroundColor: '#e6fffa',
  },
  unsafeBadge: {
    backgroundColor: '#fff5f5',
  },
  naBadge: {
    backgroundColor: '#edf2f7',
  },
  statLabel: {
    fontSize: width * 0.03,
    fontWeight: '600',
    color: '#4a5568',
    textTransform: 'uppercase',
  },
  statValue: {
    fontSize: width * 0.032,
    fontWeight: '700',
    color: '#2d3748',
  },
  itemSelection: {
    marginTop: width * 0.02,
    fontSize: width * 0.032,
    color: '#4a5568',
    fontStyle: 'italic',
  },
  statusRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    gap: width * 0.025,
  },
  statusButton: {
    flex: 1,
    borderWidth: 1,
    paddingVertical: width * 0.015,
    alignItems: 'center',
    justifyContent: 'center',
    backgroundColor: '#f2f6fa',
    borderColor: '#d0d7e2',
    shadowColor: '#222',
    shadowOpacity: 0.08,
    shadowRadius: 4,
    elevation: 1,
  },
  statusButtonText: {
    fontSize: width * 0.030,
    fontWeight: '600',
    color: '#1a365d',
    textTransform: 'uppercase',
    letterSpacing: 0.4,
  },
  statusButtonTextSelected: {
    color: '#fff',
  },
  emptyState: {
    marginTop: width * 0.08,
    padding: width * 0.06,
    borderRadius: width * 0.03,
    backgroundColor: '#eef2ff',
    borderWidth: 1,
    borderColor: '#c3dafe',
  },
  emptyText: {
    fontSize: width * 0.038,
    color: '#4c51bf',
    textAlign: 'center',
  },
  bottomBar: {
    position: 'absolute',
    left: 0,
    right: 0,
    bottom: 0,
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.06,
    paddingTop: width * 0.035,
    backgroundColor: '#ffffff',
    borderTopWidth: 1,
    borderTopColor: '#dde5ee',
  },
  secondaryButton: {
    flex: 1,
    marginRight: width * 0.03,
  
    borderColor: '#195c85',
    paddingVertical: width * 0.035,
    alignItems: 'center',
    justifyContent: 'center',
  },
  secondaryButtonText: {
    fontSize: width * 0.045,
    fontWeight: '600',
    color: '#195c85',
  },
  primaryButton: {
    flex: 1,
    // backgroundColor: '#195c85',
    borderRadius: width * 0.025,
    paddingVertical: width * 0.035,
    alignItems: 'center',
    justifyContent: 'center',
   
  },
  primaryButtonDisabled: {
    // backgroundColor: '#a0aec0',
    shadowOpacity: 0,
    color: '#195c85',
  },
  primaryButtonText: {
    fontSize: width * 0.045,
    fontWeight: '700',
    color: '#195c85',
  },
});

export default HSEcorrectiveactionStep1;