
import React, { useState, useEffect } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  TextInput,
  ScrollView,
  Dimensions,
  Alert,
  KeyboardAvoidingView,
  Platform,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';

const { width } = Dimensions.get('window');

const HSE_077_078Step2 = ({ setActiveStep }) => {
  const navigation = useNavigation();
  const [formData, setFormData] = useState({
    date: new Date().toISOString().split('T')[0],
    name: '',
    department: '',
    description: '',
    hazardClass: '',
    location: ''
  });

  // Load saved data on component mount
  useEffect(() => {
    loadSavedData();
  }, []);

  const loadSavedData = async () => {
    try {
      const savedData = await AsyncStorage.getItem('hseStep2');
      if (savedData) {
        const data = JSON.parse(savedData);
        setFormData(prevData => ({ ...prevData, ...data }));
      }
    } catch (error) {
      console.error('Error loading saved data:', error);
    }
  };

  const saveData = async () => {
    try {
      await AsyncStorage.setItem('hseStep2', JSON.stringify(formData));
    } catch (error) {
      console.error('Error saving data:', error);
    }
  };

  const updateFormData = (field, value) => {
    setFormData(prev => ({
      ...prev,
      [field]: value
    }));
  };

  const selectHazardClass = (classType) => {
    setFormData(prev => ({
      ...prev,
      hazardClass: classType
    }));
  };

  const validateForm = () => {
    if (!formData.date.trim()) {
      Alert.alert('Error', 'Please enter the date');
      return false;
    }
    if (!formData.name.trim()) {
      Alert.alert('Error', 'Please enter the name of person reporting');
      return false;
    }
    if (!formData.department.trim()) {
      Alert.alert('Error', 'Please enter the department');
      return false;
    }
    if (!formData.description.trim()) {
      Alert.alert('Error', 'Please enter the NM / Hazard description');
      return false;
    }
    if (!formData.hazardClass) {
      Alert.alert('Error', 'Please select a hazard class');
      return false;
    }
    if (!formData.location.trim()) {
      Alert.alert('Error', 'Please enter the area/location');
      return false;
    }
    return true;
  };

  const handleNext = async () => {
    if (!validateForm()) {
      return;
    }

    try {
      await saveData();
      setActiveStep(3);
    } catch (error) {
      console.error('Error in handleNext:', error);
      Alert.alert('Error', 'Failed to save data. Please try again.');
    }
  };

  const handleBack = () => {
    setActiveStep(1);
  };

  return (
    <KeyboardAvoidingView 
      style={styles.container} 
      behavior={Platform.OS === 'ios' ? 'padding' : 'height'}
      keyboardVerticalOffset={Platform.OS === 'ios' ? 0 : 20}
    >
      <ScrollView style={styles.mainContent} contentContainerStyle={styles.scrollContent} showsVerticalScrollIndicator={false}>
        
        {/* Header Section */}
        <View style={styles.section}>
          <Text style={styles.sectionTitle}>Near Miss / Hazard Report</Text>
        </View>

        {/* Date Field */}
        <View style={styles.section}>
          <View style={styles.inputContainer}>
            <Text style={styles.label}>Date *</Text>
            <TextInput
              style={styles.textInput}
              value={formData.date}
              onChangeText={(text) => updateFormData('date', text)}
              placeholder="Enter date (YYYY-MM-DD)"
              placeholderTextColor="#888"
            />
          </View>
        </View>

        {/* Name Field */}
        <View style={styles.section}>
          <View style={styles.inputContainer}>
            <Text style={styles.label}>Name of Person Reporting *</Text>
            <TextInput
              style={styles.textInput}
              value={formData.name}
              onChangeText={(text) => updateFormData('name', text)}
              placeholder="Enter your name"
              placeholderTextColor="#888"
            />
          </View>
        </View>

        {/* Department Field */}
        <View style={styles.section}>
          <View style={styles.inputContainer}>
            <Text style={styles.label}>Department *</Text>
            <TextInput
              style={styles.textInput}
              value={formData.department}
              onChangeText={(text) => updateFormData('department', text)}
              placeholder="Enter department"
              placeholderTextColor="#888"
            />
          </View>
        </View>

        {/* Description Field */}
        <View style={styles.section}>
          <View style={styles.inputContainer}>
            <Text style={styles.label}>NM / Hazard Description *</Text>
            <TextInput
              style={[styles.textInput, styles.textArea]}
              value={formData.description}
              onChangeText={(text) => updateFormData('description', text)}
              placeholder="Describe the near miss or hazard in detail..."
              placeholderTextColor="#888"
              multiline
              numberOfLines={6}
              textAlignVertical="top"
            />
          </View>
        </View>

        {/* Hazard Class Selection */}
        <View style={styles.section}>
          <View style={styles.inputContainer}>
            <Text style={styles.label}>Hazard Class A B C (circle) *</Text>
            <View style={styles.hazardClassContainer}>
              <TouchableOpacity
                style={[
                  styles.hazardClassButton,
                  formData.hazardClass === 'A' && styles.hazardClassButtonSelected
                ]}
                onPress={() => selectHazardClass('A')}
              >
                <Text style={[
                  styles.hazardClassText,
                  formData.hazardClass === 'A' && styles.hazardClassTextSelected
                ]}>A</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[
                  styles.hazardClassButton,
                  formData.hazardClass === 'B' && styles.hazardClassButtonSelected
                ]}
                onPress={() => selectHazardClass('B')}
              >
                <Text style={[
                  styles.hazardClassText,
                  formData.hazardClass === 'B' && styles.hazardClassTextSelected
                ]}>B</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[
                  styles.hazardClassButton,
                  formData.hazardClass === 'C' && styles.hazardClassButtonSelected
                ]}
                onPress={() => selectHazardClass('C')}
              >
                <Text style={[
                  styles.hazardClassText,
                  formData.hazardClass === 'C' && styles.hazardClassTextSelected
                ]}>C</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>

        {/* Location Field */}
        <View style={styles.section}>
          <View style={styles.inputContainer}>
            <Text style={styles.label}>Area (location) *</Text>
            <TextInput
              style={styles.textInput}
              value={formData.location}
              onChangeText={(text) => updateFormData('location', text)}
              placeholder="Enter area or location"
              placeholderTextColor="#888"
            />
          </View>
        </View>

      </ScrollView>

      <View style={styles.buttonContainer}>
        <TouchableOpacity style={styles.cancelButton} onPress={handleBack}>
          <MaterialIcons style={styles.bottomicon} name="chevron-left" size={16} color="#015185" />
          <Text style={styles.cancelButtonText}>Back</Text>
        </TouchableOpacity>
        <TouchableOpacity
          style={styles.nextButton}
          onPress={handleNext}
        >
          <Text style={styles.nextButtonText}>Next</Text>
          <MaterialIcons style={styles.bottomicon1} name="chevron-right" size={16} color="#015185" />
        </TouchableOpacity>
      </View>
    </KeyboardAvoidingView>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  mainContent: {
    flex: 1,
  },
  scrollContent: {
    paddingBottom: 100,
  },
  section: {
    marginBottom: width * 0.04,
    paddingHorizontal: width * 0.04,
  },
  sectionTitle: {
    fontSize: width * 0.045,
    fontWeight: '600',
    color: '#015185',
    marginBottom: 8,
    textAlign: 'center',
  },
  inputContainer: {
    marginBottom: width * 0.04,
  },
  label: {
    fontSize: width * 0.038,
    color: '#015185',
    marginBottom: 4,
    fontWeight: '500',
  },
  textInput: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    paddingHorizontal: 12,
    paddingVertical: 14,
    fontSize: width * 0.04,
    backgroundColor: '#fff',
  },
  textArea: {
    minHeight: 120,
    textAlignVertical: 'top',
  },
  hazardClassContainer: {
    flexDirection: 'row',
    justifyContent: 'space-around',
    marginTop: 10,
  },
  hazardClassButton: {
    width: 60,
    height: 60,
    borderRadius: 30,
    borderWidth: 2,
    borderColor: '#ddd',
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: '#fff',
  },
  hazardClassButtonSelected: {
    borderColor: '#015185',
    backgroundColor: '#015185',
  },
  hazardClassText: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#333',
  },
  hazardClassTextSelected: {
    color: '#fff',
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: 15,
    paddingVertical: 15,
    backgroundColor: '#F2F2F2',
    borderTopWidth: 1,
    borderTopColor: '#ddd',
    position: 'absolute',
    bottom: 0,
    left: 0,
    right: 0,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: -2 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
    paddingBottom: 30,
  },
  cancelButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 15,
    paddingBottom: 30,
  },
  nextButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingBottom: 18,
    borderRadius: 5,
  },
  cancelButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  nextButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  bottomicon: {
    marginTop: 2,
  },
  bottomicon1: {
    marginTop: 4,
  },
});

export default HSE_077_078Step2;