import React, { useState, useEffect } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  ScrollView,
  TextInput,
  Dimensions,
  Modal,
  Alert,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import HSE_132Json from './HSE_132_Json.js';

const { width, height } = Dimensions.get('window');

const HSE_132_step2 = ({ setActiveStep }) => {
  const navigation = useNavigation();
  const [checklistData, setChecklistData] = useState({});
  const [selectedChecklist, setSelectedChecklist] = useState(null);
  const [showErrors, setShowErrors] = useState(false);
  const [commentModalVisible, setCommentModalVisible] = useState(false);
  const [currentCommentItem, setCurrentCommentItem] = useState(null);
  const [tempComment, setTempComment] = useState('');

  useEffect(() => {
    loadSavedData();
  }, []);

  const loadSavedData = async () => {
    try {
      const savedHse = await AsyncStorage.getItem('hseStep1');
      if (savedHse) {
        const hseData = JSON.parse(savedHse);
        if (hseData.selectedChecklist) {
          setSelectedChecklist(hseData.selectedChecklist);
        }
      }

      const savedStep2 = await AsyncStorage.getItem('hseStep2');
      if (savedStep2) {
        setChecklistData(JSON.parse(savedStep2));
      }
    } catch (error) {
      console.error('Error loading saved data:', error);
    }
  };

  const saveStep2Data = async (data) => {
    try {
      await AsyncStorage.setItem('hseStep2', JSON.stringify(data));
    } catch (error) {
      console.error('Error saving step 2 data:', error);
    }
  };

  const handleInputChange = (key, value) => {
    const newData = { ...checklistData, [key]: value };
    setChecklistData(newData);
    saveStep2Data(newData);
  };

  const handleChecklistChange = (itemKey, value) => {
    const newData = { ...checklistData, [itemKey]: value };
    setChecklistData(newData);
    saveStep2Data(newData);
  };

  const handleCommentPress = (itemKey) => {
    setCurrentCommentItem(itemKey);
    setTempComment(checklistData[`${itemKey}_comments`] || '');
    setCommentModalVisible(true);
  };

  const handleCommentSave = () => {
    if (currentCommentItem) {
      const newData = { ...checklistData, [`${currentCommentItem}_comments`]: tempComment };
      setChecklistData(newData);
      saveStep2Data(newData);
    }
    setCommentModalVisible(false);
    setCurrentCommentItem(null);
    setTempComment('');
  };

  const handleCommentCancel = () => {
    setCommentModalVisible(false);
    setCurrentCommentItem(null);
    setTempComment('');
  };

  const handleBack = () => {
    setActiveStep(1);
  };

  const handleNext = async () => {
    setShowErrors(true);
    
    // Validate required fields
    const requiredFields = [
      'oilSpillsCleaned',
      'freeFromChemicalSpills',
      'sumpClosed',
      'freeFromOtherWaste',
      'surroundingArea'
    ];

    const missingFields = requiredFields.filter(field => !checklistData[field]);
    
    if (missingFields.length > 0) {
      const fieldLabels = {
        oilSpillsCleaned: 'Oil Spills Cleaned',
        freeFromChemicalSpills: 'Free from Chemical Spills',
        sumpClosed: 'Sump Closed',
        freeFromOtherWaste: 'Free from Other Waste',
        surroundingArea: 'Surrounding Area'
      };

      const missingLabels = missingFields.map(field => fieldLabels[field]).join(', ');
      alert(`Please fill in the following required fields: ${missingLabels}`);
      return;
    }

    // Save final step 2 data before proceeding
    try {
      await AsyncStorage.setItem('hseStep2', JSON.stringify(checklistData));
      console.log('Step 2 data saved successfully:', checklistData);
    } catch (error) {
      console.error('Error saving step 2 data:', error);
    }

    setActiveStep(3);
  };

  const renderInputField = (item, isRequired = false) => {
    const value = checklistData[item.key] || '';
    
    if (item.inputType === 'Yes/No') {
      return (
        <View style={styles.checklistItemContainer}>
          <Text style={styles.checklistItemLabel}>
            {item.itemName}
            {isRequired && <Text style={styles.requiredAsterisk}> *</Text>}
          </Text>
          <View style={styles.itemRow}>
            <View style={styles.optionsContainer}>
              <TouchableOpacity
                style={[
                  styles.optionButton,
                  value === 'Yes' && styles.selectedOptionButton,
                  value === 'Yes' && styles.yesButton,
                ]}
                onPress={() => handleChecklistChange(item.key, 'Yes')}
              >
                <Text style={[
                  styles.optionText,
                  value === 'Yes' && styles.optionTextSelected,
                ]}>
                  Yes
                </Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[
                  styles.optionButton,
                  value === 'No' && styles.selectedOptionButton,
                  value === 'No' && styles.noButton,
                ]}
                onPress={() => handleChecklistChange(item.key, 'No')}
              >
                <Text style={[
                  styles.optionText,
                  value === 'No' && styles.optionTextSelected,
                ]}>
                  No
                </Text>
              </TouchableOpacity>
            </View>
            <View style={styles.iconsContainer}>
              <TouchableOpacity
                style={[
                  styles.iconButton,
                  checklistData[`${item.key}_comments`] && styles.iconButtonActive
                ]}
                onPress={() => handleCommentPress(item.key)}
              >
                <MaterialIcons 
                  name={checklistData[`${item.key}_comments`] ? "comment" : "add-comment"} 
                  size={24} 
                  color={checklistData[`${item.key}_comments`] ? "#0078D4" : "#666"} 
                />
              </TouchableOpacity>
            </View>
          </View>
          {checklistData[`${item.key}_comments`] && (
            <View style={styles.commentContainer}>
              <MaterialIcons name="comment" size={16} color="#0078D4" style={styles.commentIcon} />
              <Text style={styles.commentText}>
                {checklistData[`${item.key}_comments`]}
              </Text>
            </View>
          )}
        </View>
      );
    }

    if (item.inputType === 'date') {
      return (
        <View style={styles.inputContainer}>
          <Text style={styles.inputLabel}>
            {item.itemName}
            {isRequired && <Text style={styles.requiredAsterisk}> *</Text>}
          </Text>
          <TextInput
            style={[styles.input, showErrors && isRequired && !value && styles.inputError]}
            placeholder="YYYY-MM-DD"
            value={value}
            onChangeText={(text) => handleInputChange(item.key, text)}
          />
        </View>
      );
    }

    if (item.inputType === 'time') {
      return (
        <View style={styles.inputContainer}>
          <Text style={styles.inputLabel}>
            {item.itemName}
            {isRequired && <Text style={styles.requiredAsterisk}> *</Text>}
          </Text>
          <TextInput
            style={[styles.input, showErrors && isRequired && !value && styles.inputError]}
            placeholder="HH:MM"
            value={value}
            onChangeText={(text) => handleInputChange(item.key, text)}
          />
        </View>
      );
    }

    if (item.inputType === 'signature') {
      return (
        <View style={styles.inputContainer}>
          <Text style={styles.inputLabel}>
            {item.itemName}
            {isRequired && <Text style={styles.requiredAsterisk}> *</Text>}
          </Text>
          <TouchableOpacity style={styles.signatureButton}>
            <MaterialIcons name="edit" size={20} color="#0078D4" />
            <Text style={styles.signatureButtonText}>Add Signature</Text>
          </TouchableOpacity>
        </View>
      );
    }

    // Default text input
    return (
      <View style={styles.inputContainer}>
        <Text style={styles.inputLabel}>
          {item.itemName}
          {isRequired && <Text style={styles.requiredAsterisk}> *</Text>}
        </Text>
        <TextInput
          style={[styles.input, showErrors && isRequired && !value && styles.inputError]}
          placeholder={item.placeholder || `Enter ${item.itemName.toLowerCase()}`}
          value={value}
          onChangeText={(text) => handleInputChange(item.key, text)}
        />
      </View>
    );
  };

  if (!selectedChecklist) {
    return (
      <View style={styles.container}>
        <Text style={styles.errorText}>No checklist selected. Please go back to step 1.</Text>
        <TouchableOpacity style={styles.backButton} onPress={handleBack}>
          <Text style={styles.backButtonText}>Go Back</Text>
        </TouchableOpacity>
      </View>
    );
  }

  return (
    <View style={styles.container}>
      <ScrollView style={styles.mainContent} contentContainerStyle={styles.scrollContent}>
        <View style={styles.checklistInfo}>
          <Text style={styles.checklistTitle}>{selectedChecklist.document_name}</Text>
          <Text style={styles.checklistSubtitle}>Document ID: {selectedChecklist.document_id}</Text>
        </View>

        {/* Environmental Rehabilitation Checklist */}
        <View style={styles.section}>
          <Text style={styles.sectionTitle}>Environmental Rehabilitation Checklist</Text>
          {HSE_132Json.sections[1].items.map((item, index) => (
            <View key={index}>
              {renderInputField(item, true)}
            </View>
          ))}
        </View>
      </ScrollView>

      {/* Bottom Buttons */}
      <View style={styles.buttonContainer}>
        <TouchableOpacity style={styles.backButtonBottom} onPress={handleBack}>
          <MaterialIcons name="arrow-back" size={20} color="#015185" />
          <Text style={styles.backButtonTextBottom}>Back</Text>
        </TouchableOpacity>
        <TouchableOpacity style={styles.nextButton} onPress={handleNext}>
          <Text style={styles.nextButtonText}>Next</Text>
          <MaterialIcons name="arrow-forward" size={20} color="#fff" />
        </TouchableOpacity>
      </View>

      {/* Comment Modal */}
      <Modal
        visible={commentModalVisible}
        transparent={true}
        animationType="slide"
        onRequestClose={handleCommentCancel}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>Add Comment</Text>
            <TextInput
              style={styles.modalTextInput}
              placeholder="Enter your comment..."
              value={tempComment}
              onChangeText={setTempComment}
              multiline
              numberOfLines={4}
            />
            <View style={styles.modalButtons}>
              <TouchableOpacity style={styles.modalCancelButton} onPress={handleCommentCancel}>
                <Text style={styles.modalCancelButtonText}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity style={styles.modalSaveButton} onPress={handleCommentSave}>
                <Text style={styles.modalSaveButtonText}>Save</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#f8f9fa',
  },
  mainContent: {
    flex: 1,
    paddingHorizontal: 20,
    paddingTop: 20,
  },
  scrollContent: {
    paddingBottom: 100,
  },
  checklistInfo: {
    backgroundColor: '#fff',
    padding: 20,
    borderRadius: 10,
    marginBottom: 20,
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
  },
  checklistTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: 5,
  },
  checklistSubtitle: {
    fontSize: 14,
    color: '#666',
  },
  section: {
    backgroundColor: '#fff',
    padding: 20,
    borderRadius: 10,
    marginBottom: 20,
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
  },
  sectionTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: 20,
    borderBottomWidth: 1,
    borderBottomColor: '#e0e0e0',
    paddingBottom: 10,
  },
  inputContainer: {
    marginBottom: 20,
  },
  inputLabel: {
    fontSize: 16,
    fontWeight: '600',
    color: '#333',
    marginBottom: 8,
  },
  requiredAsterisk: {
    color: 'red',
    fontWeight: 'bold',
  },
  input: {
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 8,
    paddingHorizontal: 15,
    paddingVertical: 12,
    fontSize: 16,
    backgroundColor: '#fff',
  },
  inputError: {
    borderColor: 'red',
  },
  checklistItemContainer: {
    marginBottom: 20,
    backgroundColor: '#fff',
    padding: 15,
    borderRadius: 8,
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
  },
  checklistItemLabel: {
    fontSize: 16,
    fontWeight: '600',
    color: '#333',
    marginBottom: 12,
  },
  itemRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
  },
  optionsContainer: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    flex: 1,
  },
  optionButton: {
    paddingHorizontal: 12,
    paddingVertical: 8,
    marginHorizontal: 3,
    marginVertical: 2,
    borderRadius: 5,
    minWidth: 60,
    backgroundColor: '#ECF0F3',
    alignItems: 'center',
  },
  selectedOptionButton: {
    backgroundColor: '#333',
  },
  yesButton: {
    backgroundColor: '#4CAF50',
  },
  noButton: {
    backgroundColor: '#FF4B2B',
  },
  optionText: {
    fontSize: 14,
    color: '#333',
    fontWeight: '500',
  },
  optionTextSelected: {
    color: '#fff',
    fontWeight: '600',
  },
  iconsContainer: {
    flexDirection: 'row',
    marginLeft: 10,
  },
  iconButton: {
    padding: 10,
    backgroundColor: '#f0f0f0',
    borderRadius: 5,
  },
  iconButtonActive: {
    backgroundColor: '#e3f2fd',
    borderWidth: 1,
    borderColor: '#0078D4',
  },
  commentContainer: {
    flexDirection: 'row',
    alignItems: 'flex-start',
    marginTop: 10,
    backgroundColor: '#f9f9f9',
    padding: 10,
    borderRadius: 8,
    borderLeftWidth: 3,
    borderLeftColor: '#0078D4',
  },
  commentIcon: {
    marginRight: 8,
    marginTop: 2,
  },
  commentText: {
    fontSize: 14,
    color: '#666',
    fontStyle: 'italic',
    flex: 1,
    lineHeight: 20,
  },
  signatureButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingVertical: 12,
    paddingHorizontal: 20,
    borderRadius: 8,
    borderWidth: 2,
    borderColor: '#0078D4',
    backgroundColor: '#fff',
  },
  signatureButtonText: {
    fontSize: 16,
    fontWeight: '600',
    color: '#0078D4',
    marginLeft: 8,
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: 20,
    paddingVertical: 15,
    backgroundColor: '#fff',
    borderTopWidth: 1,
    borderTopColor: '#e0e0e0',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: -2 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
  },
  backButtonBottom: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 12,
    paddingHorizontal: 20,
    borderRadius: 8,
    borderWidth: 2,
    borderColor: '#015185',
    backgroundColor: '#fff',
  },
  backButtonTextBottom: {
    fontSize: 16,
    fontWeight: '600',
    color: '#015185',
    marginLeft: 8,
  },
  nextButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 12,
    paddingHorizontal: 25,
    borderRadius: 8,
    backgroundColor: '#0078D4',
  },
  nextButtonText: {
    fontSize: 16,
    fontWeight: '600',
    color: '#fff',
    marginRight: 8,
  },
  errorText: {
    fontSize: 18,
    color: 'red',
    textAlign: 'center',
    marginTop: 100,
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    width: '90%',
    maxWidth: 400,
  },
  modalTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: 15,
    textAlign: 'center',
  },
  modalTextInput: {
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 8,
    padding: 15,
    fontSize: 16,
    minHeight: 100,
    textAlignVertical: 'top',
    marginBottom: 20,
  },
  modalButtons: {
    flexDirection: 'row',
    justifyContent: 'space-between',
  },
  modalCancelButton: {
    flex: 1,
    paddingVertical: 12,
    paddingHorizontal: 20,
    borderRadius: 8,
    borderWidth: 2,
    borderColor: '#666',
    backgroundColor: '#fff',
    marginRight: 10,
    alignItems: 'center',
  },
  modalCancelButtonText: {
    fontSize: 16,
    fontWeight: '600',
    color: '#666',
  },
  modalSaveButton: {
    flex: 1,
    paddingVertical: 12,
    paddingHorizontal: 20,
    borderRadius: 8,
    backgroundColor: '#0078D4',
    alignItems: 'center',
  },
  modalSaveButtonText: {
    fontSize: 16,
    fontWeight: '600',
    color: '#fff',
  },
});

export default HSE_132_step2;