import { View, Text, TextInput, TouchableOpacity, StyleSheet, Alert, ScrollView, ActivityIndicator, Clipboard } from 'react-native'
import React, { useState } from 'react'
import { useNavigation } from '@react-navigation/native'

const TestLoginCredFCM = () => {
  const navigation = useNavigation()
  const [email, setEmail] = useState('')
  const [loading, setLoading] = useState(false)
  const [testResult, setTestResult] = useState('')
  const [fcmStatus, setFcmStatus] = useState(null)

  const checkFCMTokenByEmail = async () => {
    if (!email.trim()) {
      Alert.alert('Error', 'Please enter an email address')
      return
    }

    setLoading(true)
    setFcmStatus(null)
    
    try {
      console.log('🔍 Checking FCM token for email:', email.trim())
      
      // Check if email has FCM tokens stored
      const response = await fetch('https://us-central1-titan-drilling-1f8e9.cloudfunctions.net/getFCMTokenByEmail', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          email: email.trim(),
        }),
      })

      const result = await response.json()
      console.log('📋 FCM token check response:', result)
      
      if (result.success) {
        const statusInfo = {
          hasToken: true,
          userId: result.userId,
          totalDevices: result.totalDevices || 0,
          fcmToken: result.fcmToken,
          message: `✅ FCM Token Found!\n\n📧 Email: ${email.trim()}\n👤 User ID: ${result.userId}\n📱 Total Devices: ${result.totalDevices || 'Unknown'}`
        }
        
        setFcmStatus(statusInfo)
        setTestResult(statusInfo.message)
        Alert.alert('FCM Token Found', `This email has FCM tokens stored.\nUser ID: ${result.userId}\nTotal Devices: ${result.totalDevices || 'Unknown'}`)
      } else {
        const statusInfo = {
          hasToken: false,
          message: `❌ No FCM Token Found\n\n📧 Email: ${email.trim()}\n\nThis email has not logged into the app yet or has no FCM tokens stored.`
        }
        
        setFcmStatus(statusInfo)
        setTestResult(statusInfo.message)
        Alert.alert('No FCM Token', 'This email has no FCM tokens stored. The user needs to login to the app first.')
      }
    } catch (error) {
      console.error('❌ Error checking FCM token:', error)
      const errorMessage = `❌ Error checking FCM token: ${error.message || 'Unknown error'}`
      setTestResult(errorMessage)
      Alert.alert('Error', 'Failed to check FCM token. Check console for details.')
    } finally {
      setLoading(false)
    }
  }

  const sendTestNotification = async () => {
    if (!email.trim()) {
      Alert.alert('Error', 'Please enter an email address')
      return
    }

    // If we haven't checked FCM status yet, check it first
    if (!fcmStatus) {
      Alert.alert('Info', 'Please check FCM token first to verify the email has registered devices.')
      return
    }

    if (!fcmStatus.hasToken) {
      Alert.alert('No FCM Token', 'This email has no FCM tokens stored. Cannot send test notification.')
      return
    }

    setLoading(true)
    
    try {
      console.log('🧪 Sending test notification to email:', email.trim())
      console.log('📱 Using FCM token:', fcmStatus.fcmToken ? fcmStatus.fcmToken.substring(0, 20) + '...' : 'No token')
      console.log('👤 User ID:', fcmStatus.userId)
      
      let result = null
      let functionUsed = ''
      
      // Try multiple approaches to send the notification
      
      // Approach 1: Try sendNotificationToAllUserDevices (recommended)
      try {
        console.log('🔄 Trying sendNotificationToAllUserDevices...')
        const response1 = await fetch('https://us-central1-titan-drilling-1f8e9.cloudfunctions.net/sendNotificationToAllUserDevices', {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
          },
          body: JSON.stringify({
            userId: fcmStatus.userId,
            title: '🧪 Test Notification',
            body: 'This is a test notification from Titan Drilling App',
            data: {
              type: 'test_notification',
              email: email.trim(),
              userId: fcmStatus.userId,
              timestamp: new Date().toISOString(),
              testId: `test_${Date.now()}`,
            },
          }),
        })
        
        result = await response1.json()
        functionUsed = 'sendNotificationToAllUserDevices'
        console.log('✅ sendNotificationToAllUserDevices response:', result)
        
        if (result.success) {
          // Success with first approach
          console.log('✅ Notification sent successfully using sendNotificationToAllUserDevices')
        } else {
          throw new Error(result.error || 'sendNotificationToAllUserDevices failed')
        }
        
      } catch (error1) {
        console.log('⚠️ sendNotificationToAllUserDevices failed, trying sendTestPushNotification...')
        
        // Approach 2: Try sendTestPushNotification with FCM token
        try {
          const response2 = await fetch('https://us-central1-titan-drilling-1f8e9.cloudfunctions.net/sendTestPushNotification', {
            method: 'POST',
            headers: {
              'Content-Type': 'application/json',
            },
            body: JSON.stringify({
              title: '🧪 Test Notification',
              body: 'This is a test notification from Titan Drilling App',
              fcmToken: fcmStatus.fcmToken,
              email: email.trim(),
              userId: fcmStatus.userId,
              timestamp: new Date().toISOString(),
              testId: `test_${Date.now()}`,
            }),
          })
          
          result = await response2.json()
          functionUsed = 'sendTestPushNotification'
          console.log('✅ sendTestPushNotification response:', result)
          
          if (result.success) {
            console.log('✅ Notification sent successfully using sendTestPushNotification')
          } else {
            throw new Error(result.error || 'sendTestPushNotification failed')
          }
          
        } catch (error2) {
          console.log('⚠️ sendTestPushNotification also failed, trying sendPushNotificationByEmail...')
          
          // Approach 3: Try sendPushNotificationByEmail
          try {
            const response3 = await fetch('https://us-central1-titan-drilling-1f8e9.cloudfunctions.net/sendPushNotificationByEmail', {
              method: 'POST',
              headers: {
                'Content-Type': 'application/json',
              },
              body: JSON.stringify({
                email: email.trim(),
                title: '🧪 Test Notification',
                body: 'This is a test notification from Titan Drilling App',
                data: {
                  type: 'test_notification',
                  email: email.trim(),
                  userId: fcmStatus.userId,
                  timestamp: new Date().toISOString(),
                  testId: `test_${Date.now()}`,
                },
              }),
            })
            
            result = await response3.json()
            functionUsed = 'sendPushNotificationByEmail'
            console.log('✅ sendPushNotificationByEmail response:', result)
            
            if (result.success) {
              console.log('✅ Notification sent successfully using sendPushNotificationByEmail')
            } else {
              throw new Error(result.error || 'sendPushNotificationByEmail failed')
            }
            
          } catch (error3) {
            // All approaches failed
            console.error('❌ All notification functions failed:', {
              error1: error1.message,
              error2: error2.message,
              error3: error3.message
            })
            throw new Error(`All notification functions failed. Last error: ${error3.message}`)
          }
        }
      }
      
      // If we reach here, one of the approaches succeeded
      if (result && result.success) {
        const successMessage = `✅ Test notification sent successfully!\n\n📧 Email: ${email.trim()}\n👤 User ID: ${fcmStatus.userId}\n🔧 Function Used: ${functionUsed}\n📱 Sent to: ${result.successfulDevices || 'Unknown'} devices\n❌ Failed: ${result.failedDevices || 0} devices\n📊 Total devices: ${result.totalDevices || 'Unknown'}`
        
        setTestResult(successMessage)
        Alert.alert('Success', `Test notification sent successfully using ${functionUsed}!`)
      } else {
        const errorMessage = `❌ Failed to send test notification: ${result?.error || 'Unknown error'}`
        setTestResult(errorMessage)
        Alert.alert('Error', result?.error || 'Failed to send test notification')
      }
      
    } catch (error) {
      console.error('❌ Error sending test notification:', error)
      const errorMessage = `❌ Error: ${error.message || 'Failed to send test notification'}`
      setTestResult(errorMessage)
      Alert.alert('Error', 'Failed to send test notification. Check console for details.')
    } finally {
      setLoading(false)
    }
  }

  const clearResults = () => {
    setTestResult('')
    setEmail('')
    setFcmStatus(null)
  }

  const sendSimpleTestNotification = async () => {
    if (!email.trim()) {
      Alert.alert('Error', 'Please enter an email address')
      return
    }

    setLoading(true)
    
    try {
      console.log('🧪 Sending simple test notification to email:', email.trim())
      
      // Try the simplest approach - sendPushNotificationByEmail
      const response = await fetch('https://us-central1-titan-drilling-1f8e9.cloudfunctions.net/sendPushNotificationByEmail', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          email: email.trim(),
          title: '🧪 Simple Test',
          body: 'This is a simple test notification',
          data: {
            type: 'simple_test',
            timestamp: new Date().toISOString(),
          },
        }),
      })

      const result = await response.json()
      console.log('📋 Simple test response:', result)
      
      if (result.success) {
        const successMessage = `✅ Simple test notification sent!\n\n📧 Email: ${email.trim()}\n📱 Function: sendPushNotificationByEmail\n\nThis confirms the basic FCM system is working.`
        setTestResult(successMessage)
        Alert.alert('Success', 'Simple test notification sent successfully!')
      } else {
        const errorMessage = `❌ Simple test failed: ${result.error}`
        setTestResult(errorMessage)
        Alert.alert('Error', result.error || 'Simple test failed')
      }
    } catch (error) {
      console.error('❌ Simple test error:', error)
      const errorMessage = `❌ Simple test error: ${error.message || 'Unknown error'}`
      setTestResult(errorMessage)
      Alert.alert('Error', 'Simple test failed. Check console for details.')
    } finally {
      setLoading(false)
    }
  }

  const testFirebaseFunctions = async () => {
    if (!email.trim()) {
      Alert.alert('Error', 'Please enter an email address')
      return
    }

    setLoading(true)
    
    try {
      console.log('🧪 Testing Firebase Cloud Functions...')
      
      // Test 1: getFCMTokenByEmail
      console.log('🔄 Testing getFCMTokenByEmail...')
      const response1 = await fetch('https://us-central1-titan-drilling-1f8e9.cloudfunctions.net/getFCMTokenByEmail', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          email: email.trim(),
        }),
      })
      
      const result1 = await response1.json()
      console.log('📋 getFCMTokenByEmail result:', result1)
      
      // Test 2: sendPushNotificationByEmail
      console.log('🔄 Testing sendPushNotificationByEmail...')
      const response2 = await fetch('https://us-central1-titan-drilling-1f8e9.cloudfunctions.net/sendPushNotificationByEmail', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
        email: email.trim(),
          title: '🧪 Function Test',
          body: 'Testing Firebase Cloud Functions',
        data: {
            type: 'function_test',
          timestamp: new Date().toISOString(),
          },
        }),
      })
      
      const result2 = await response2.json()
      console.log('📋 sendPushNotificationByEmail result:', result2)
      
      // Test 3: sendNotificationToAllUserDevices (if we have userId)
      if (fcmStatus && fcmStatus.userId) {
        console.log('🔄 Testing sendNotificationToAllUserDevices...')
        try {
          const response3 = await fetch('https://us-central1-titan-drilling-1f8e9.cloudfunctions.net/sendNotificationToAllUserDevices', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
            body: JSON.stringify({
              userId: fcmStatus.userId,
              title: '🧪 Function Test',
              body: 'Testing sendNotificationToAllUserDevices',
              data: {
                type: 'function_test',
                timestamp: new Date().toISOString(),
              },
            }),
          })
          
          const result3 = await response3.json()
          console.log('📋 sendNotificationToAllUserDevices result:', result3)
        } catch (error3) {
          console.log('⚠️ sendNotificationToAllUserDevices test failed:', error3.message)
        }
      }
      
      const testMessage = `🔍 Firebase Functions Test Results:\n\n📧 Email: ${email.trim()}\n\n1️⃣ getFCMTokenByEmail: ${result1.success ? '✅ Success' : '❌ Failed'}\n2️⃣ sendPushNotificationByEmail: ${result2.success ? '✅ Success' : '❌ Failed'}\n\nCheck console for detailed results.`
      
      setTestResult(testMessage)
      Alert.alert('Function Test Complete', 'Firebase Cloud Functions test completed. Check console and results for details.')
      
    } catch (error) {
      console.error('❌ Function test error:', error)
      const errorMessage = `❌ Function test error: ${error.message || 'Unknown error'}`
      setTestResult(errorMessage)
      Alert.alert('Error', 'Function test failed. Check console for details.')
    } finally {
      setLoading(false)
    }
  }

  const debugFCMTokenStorage = async () => {
    if (!fcmStatus || !fcmStatus.userId) {
      Alert.alert('Error', 'Please check FCM token first to get user ID')
      return
    }

    setLoading(true)
    
    try {
      console.log('🔍 Debugging FCM token storage for user:', fcmStatus.userId)
      
      const response = await fetch('https://us-central1-titan-drilling-1f8e9.cloudfunctions.net/debugFCMTokenStorage', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          userId: fcmStatus.userId,
        }),
      })
      
      const result = await response.json()
      console.log('📋 Debug FCM token storage result:', result)
      
      if (result.success) {
        const debugInfo = result.debugInfo
        const debugMessage = `🔍 FCM Token Storage Debug:\n\n👤 User ID: ${debugInfo.userId}\n📱 Has Devices: ${debugInfo.hasDevices ? 'Yes' : 'No'}\n📊 Devices Count: ${debugInfo.devicesCount}\n✅ Active Devices: ${debugInfo.activeDevicesCount}\n🔑 Has Legacy Token: ${debugInfo.hasLegacyToken ? 'Yes' : 'No'}\n\n📋 Device Details:\n${Object.entries(debugInfo.devices).map(([key, device]) => 
          `${key}: Token=${device.hasToken ? 'Yes' : 'No'}, Status=${device.fcmStatus}, Active=${device.isActive}`
        ).join('\n')}`
        
        setTestResult(debugMessage)
        Alert.alert('Debug Complete', 'FCM token storage debug completed. Check results for details.')
      } else {
        const errorMessage = `❌ Debug failed: ${result.error}`
        setTestResult(errorMessage)
        Alert.alert('Debug Failed', result.error || 'Failed to debug FCM token storage')
      }
      
    } catch (error) {
      console.error('❌ Debug error:', error)
      const errorMessage = `❌ Debug error: ${error.message || 'Unknown error'}`
      setTestResult(errorMessage)
      Alert.alert('Error', 'Debug failed. Check console for details.')
    } finally {
      setLoading(false)
    }
  }

  const manuallyStoreFCMToken = async () => {
    if (!fcmStatus || !fcmStatus.userId || !fcmStatus.fcmToken) {
      Alert.alert('Error', 'Please check FCM token first to get user ID and token')
      return
    }

    setLoading(true)
    
    try {
      console.log('💾 Manually storing FCM token for user:', fcmStatus.userId)
      
      const response = await fetch('https://us-central1-titan-drilling-1f8e9.cloudfunctions.net/storeFCMToken', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          userId: fcmStatus.userId,
          fcmToken: fcmStatus.fcmToken,
        }),
      })
      
      const result = await response.json()
      console.log('📋 Manual store FCM token result:', result)
      
      if (result.success) {
        const successMessage = `✅ FCM Token Manually Stored!\n\n👤 User ID: ${fcmStatus.userId}\n🔑 Token: ${fcmStatus.fcmToken.substring(0, 20)}...\n💾 Status: ${result.message}`
        setTestResult(successMessage)
        Alert.alert('Success', 'FCM token manually stored successfully!')
      } else {
        const errorMessage = `❌ Manual store failed: ${result.error}`
        setTestResult(errorMessage)
        Alert.alert('Store Failed', result.error || 'Failed to manually store FCM token')
      }
      
    } catch (error) {
      console.error('❌ Manual store error:', error)
      const errorMessage = `❌ Manual store error: ${error.message || 'Unknown error'}`
      setTestResult(errorMessage)
      Alert.alert('Error', 'Manual store failed. Check console for details.')
    } finally {
      setLoading(false)
    }
  }

  const copyFCMToken = async () => {
    if (fcmStatus && fcmStatus.fcmToken) {
      try {
        await Clipboard.setString(fcmStatus.fcmToken)
        Alert.alert('Copied!', 'FCM token copied to clipboard')
        console.log('📋 FCM token copied to clipboard')
      } catch (error) {
        console.error('❌ Failed to copy FCM token:', error)
        Alert.alert('Error', 'Failed to copy FCM token')
      }
    } else {
      Alert.alert('No Token', 'No FCM token available to copy')
    }
  }

  const showFullFCMToken = () => {
    if (fcmStatus && fcmStatus.fcmToken) {
      Alert.alert(
        'Full FCM Token',
        fcmStatus.fcmToken,
        [
          { text: 'Copy Token', onPress: copyFCMToken },
          { text: 'Close', style: 'cancel' }
        ]
      )
    } else {
      Alert.alert('No Token', 'No FCM token available')
    }
  }

  return (
    <ScrollView style={styles.container}>
      <Text style={styles.title}>🧪 Test FCM Push Notifications</Text>
      
      <View style={styles.section}>
        <Text style={styles.sectionTitle}>Step 1: Check FCM Token</Text>
        <Text style={styles.description}>
          First, check if the email address has FCM tokens stored (meaning the user has logged into the app).
        </Text>
        
        <TextInput
          style={styles.input}
          placeholder="Enter email address"
          value={email}
          onChangeText={setEmail}
          keyboardType="email-address"
          autoCapitalize="none"
          autoCorrect={false}
        />
        
        <TouchableOpacity 
          style={[styles.button, styles.infoButton]} 
          onPress={checkFCMTokenByEmail}
          disabled={loading || !email.trim()}
        >
          <Text style={styles.buttonText}>
            {loading ? 'Checking...' : '🔍 Check FCM Token'}
          </Text>
        </TouchableOpacity>
      </View>

      {fcmStatus && (
        <View style={styles.section}>
          <Text style={styles.sectionTitle}>FCM Status:</Text>
          <Text style={[styles.statusText, fcmStatus.hasToken ? styles.successText : styles.errorText]}>
            {fcmStatus.message}
          </Text>
          
          {fcmStatus.hasToken && (
            <>
          <TouchableOpacity 
            style={[styles.button, styles.primaryButton]} 
                onPress={sendTestNotification}
            disabled={loading}
          >
            <Text style={styles.buttonText}>
                  {loading ? 'Sending...' : '📤 Send Test Notification'}
            </Text>
          </TouchableOpacity>
              
              <View style={styles.debugSection}>
                <Text style={styles.debugTitle}>🔍 Debug Info:</Text>
                <Text style={styles.debugText}>
                  User ID: {fcmStatus.userId}{'\n'}
                  Total Devices: {fcmStatus.totalDevices}{'\n'}
                  Has Token: {fcmStatus.hasToken ? 'Yes' : 'No'}
                </Text>
                
                {fcmStatus.fcmToken && (
                  <View style={styles.tokenSection}>
                    <Text style={styles.tokenLabel}>FCM Token:</Text>
                    <View style={styles.tokenContainer}>
                      <Text style={styles.tokenText} numberOfLines={3}>
                        {fcmStatus.fcmToken}
          </Text>
                    </View>
                    <View style={styles.tokenButtons}>
                      <TouchableOpacity 
                        style={[styles.button, styles.smallButton, styles.copyButton]} 
                        onPress={copyFCMToken}
                      >
                        <Text style={styles.buttonText}>📋 Copy Token</Text>
                      </TouchableOpacity>
                      <TouchableOpacity 
                        style={[styles.button, styles.smallButton, styles.viewButton]} 
                        onPress={showFullFCMToken}
                      >
                        <Text style={styles.buttonText}>👁️ View Full</Text>
                      </TouchableOpacity>
                    </View>
        </View>
      )}
              </View>
              
              {/* Notification Status */}
              <View style={styles.notificationStatusSection}>
                <Text style={styles.notificationStatusTitle}>📱 Notification Status:</Text>
                <View style={styles.statusRow}>
                  <Text style={styles.statusLabel}>FCM Token Valid:</Text>
                  <Text style={[styles.statusValue, { color: fcmStatus.fcmToken ? '#28a745' : '#dc3545' }]}>
                    {fcmStatus.fcmToken ? '✅ Valid' : '❌ Invalid'}
          </Text>
                </View>
                <View style={styles.statusRow}>
                  <Text style={styles.statusLabel}>Device Registered:</Text>
                  <Text style={[styles.statusValue, { color: fcmStatus.hasToken ? '#28a745' : '#dc3545' }]}>
                    {fcmStatus.hasToken ? '✅ Yes' : '❌ No'}
                  </Text>
                </View>
                <View style={styles.statusRow}>
                  <Text style={styles.statusLabel}>Ready for Notifications:</Text>
                  <Text style={[styles.statusValue, { color: fcmStatus.hasToken ? '#28a745' : '#dc3545' }]}>
                    {fcmStatus.hasToken ? '✅ Ready' : '❌ Not Ready'}
                  </Text>
                </View>
              </View>
            </>
          )}
        </View>
      )}

      {testResult && (
        <View style={styles.section}>
          <Text style={styles.sectionTitle}>Test Results:</Text>
          <Text style={styles.resultsText}>
            {testResult}
            </Text>
          
          {/* Success Indicator - Show when notification is sent successfully */}
          {testResult.includes('✅') && (
            <View style={styles.successIndicator}>
              <Text style={styles.successIndicatorTitle}>🎉 Push Notification Sent!</Text>
              <Text style={styles.successIndicatorText}>
                Check your device for the notification.{'\n'}
                If you received it, tap the button below to confirm.
              </Text>
            <TouchableOpacity 
                style={[styles.button, styles.successButton]} 
                onPress={() => {
                  Alert.alert(
                    'Notification Received!',
                    'Great! The FCM system is working correctly. Your device successfully received the push notification.',
                    [{ text: 'Excellent!', style: 'default' }]
                  )
                  console.log('✅ User confirmed notification received')
                }}
              >
                <Text style={styles.buttonText}>✅ I Received the Notification!</Text>
            </TouchableOpacity>
          </View>
          )}
        </View>
      )}

      <View style={styles.section}>
        <Text style={styles.sectionTitle}>How it works:</Text>
        <Text style={styles.infoText}>
          1. Enter the email address of a user{'\n'}
          2. Click "Check FCM Token" to verify if they have registered devices{'\n'}
          3. If tokens exist, you can send a test notification{'\n'}
          4. The notification will be sent to all their registered devices{'\n'}
          5. You'll see detailed results showing success/failure counts
        </Text>
      </View>

        <TouchableOpacity 
        style={[styles.button, styles.secondaryButton]} 
        onPress={clearResults}
        disabled={loading}
      >
        <Text style={styles.buttonText}>🗑️ Clear All Results</Text>
      </TouchableOpacity>

      <TouchableOpacity 
        style={[styles.button, styles.infoButton]} 
        onPress={sendSimpleTestNotification}
        disabled={loading || !email.trim()}
      >
        <Text style={styles.buttonText}>
          {loading ? 'Sending Simple Test...' : '📤 Send Simple Test Notification'}
        </Text>
      </TouchableOpacity>
      
      <TouchableOpacity 
        style={[styles.button, styles.warningButton]} 
        onPress={testFirebaseFunctions}
        disabled={loading || !email.trim()}
      >
        <Text style={styles.buttonText}>
          {loading ? 'Testing Functions...' : '🔧 Test Firebase Functions'}
        </Text>
      </TouchableOpacity>

      <TouchableOpacity 
        style={[styles.button, styles.infoButton]} 
        onPress={debugFCMTokenStorage}
        disabled={loading || !fcmStatus || !fcmStatus.userId}
      >
        <Text style={styles.buttonText}>
          {loading ? 'Debugging...' : '🔍 Debug FCM Token Storage'}
        </Text>
      </TouchableOpacity>

      <TouchableOpacity 
        style={[styles.button, styles.infoButton]} 
        onPress={manuallyStoreFCMToken}
        disabled={loading || !fcmStatus || !fcmStatus.userId || !fcmStatus.fcmToken}
        >
          <Text style={styles.buttonText}>
          {loading ? 'Storing Token...' : '💾 Manually Store FCM Token'}
          </Text>
        </TouchableOpacity>

      <TouchableOpacity 
        style={[styles.button, styles.secondaryButton]} 
        onPress={() => navigation.navigate('FirebaseFCMNotificaion')}
        disabled={loading}
      >
        <Text style={styles.buttonText}>🔙 Back to Main Testing</Text>
      </TouchableOpacity>
    </ScrollView>
  )
}

const styles = StyleSheet.create({
  container: {
    flex: 1,
    padding: 20,
    backgroundColor: '#f5f5f5',
  },
  title: {
    fontSize: 24,
    fontWeight: 'bold',
    textAlign: 'center',
    marginBottom: 30,
    color: '#333',
  },
  section: {
    backgroundColor: 'white',
    padding: 20,
    borderRadius: 10,
    marginBottom: 20,
    shadowColor: '#000',
    shadowOffset: {
      width: 0,
      height: 2,
    },
    shadowOpacity: 0.1,
    shadowRadius: 3.84,
    elevation: 5,
  },
  sectionTitle: {
    fontSize: 18,
    fontWeight: '600',
    marginBottom: 15,
    color: '#333',
  },
  description: {
    fontSize: 14,
    color: '#666',
    marginBottom: 20,
    lineHeight: 20,
  },
  input: {
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 8,
    padding: 15,
    fontSize: 16,
    marginBottom: 20,
    backgroundColor: 'white',
  },
  button: {
    padding: 15,
    borderRadius: 8,
    alignItems: 'center',
    justifyContent: 'center',
    marginBottom: 10,
  },
  primaryButton: {
    backgroundColor: '#007AFF',
  },
  infoButton: {
    backgroundColor: '#5856D6',
  },
  secondaryButton: {
    backgroundColor: '#6c757d',
  },
  buttonText: {
    color: 'white',
    fontSize: 16,
    fontWeight: '600',
  },
  statusText: {
    fontSize: 14,
    backgroundColor: '#f8f8f8',
    padding: 15,
    borderRadius: 8,
    fontFamily: 'monospace',
    lineHeight: 20,
    marginBottom: 15,
  },
  successText: {
    color: '#28a745',
    borderLeftWidth: 4,
    borderLeftColor: '#28a745',
  },
  errorText: {
    color: '#dc3545',
    borderLeftWidth: 4,
    borderLeftColor: '#dc3545',
  },
  resultsText: {
    fontSize: 14,
    color: '#333',
    backgroundColor: '#f8f8f8',
    padding: 15,
    borderRadius: 8,
    fontFamily: 'monospace',
    lineHeight: 20,
  },
  infoText: {
    fontSize: 14,
    color: '#666',
    lineHeight: 20,
  },
  debugSection: {
    backgroundColor: '#f0f0f0',
    padding: 15,
    borderRadius: 8,
    marginTop: 15,
    borderLeftWidth: 4,
    borderLeftColor: '#007bff',
  },
  debugTitle: {
    fontSize: 16,
    fontWeight: '600',
    marginBottom: 10,
    color: '#333',
  },
  debugText: {
    fontSize: 14,
    color: '#555',
    lineHeight: 20,
  },
  warningButton: {
    backgroundColor: '#ffc107', // Yellow color for warning
  },
  tokenSection: {
    marginTop: 15,
    paddingTop: 10,
    borderTopWidth: 1,
    borderTopColor: '#ccc',
  },
  tokenLabel: {
    fontSize: 14,
    fontWeight: '600',
    color: '#333',
    marginBottom: 5,
  },
  tokenContainer: {
    backgroundColor: '#e0e0e0',
    borderRadius: 5,
    padding: 10,
    minHeight: 50,
    justifyContent: 'center',
    alignItems: 'center',
  },
  tokenText: {
    fontSize: 12,
    color: '#555',
    textAlign: 'center',
  },
  tokenButtons: {
    flexDirection: 'row',
    justifyContent: 'space-around',
    marginTop: 10,
  },
  smallButton: {
    paddingVertical: 8,
    paddingHorizontal: 15,
    borderRadius: 6,
  },
  copyButton: {
    backgroundColor: '#4CAF50', // Green color for copy
  },
  viewButton: {
    backgroundColor: '#007bff', // Blue color for view
  },
  smallButtonText: {
    color: 'white',
    fontSize: 12,
    fontWeight: '600',
  },
  successIndicator: {
    backgroundColor: '#e8f5e9', // Light green background
    padding: 15,
    borderRadius: 8,
    marginTop: 15,
    borderLeftWidth: 4,
    borderLeftColor: '#4CAF50', // Green border
  },
  successIndicatorTitle: {
    fontSize: 16,
    fontWeight: '600',
    color: '#28a745',
    marginBottom: 10,
    textAlign: 'center',
  },
  successIndicatorText: {
    fontSize: 14,
    color: '#333',
    textAlign: 'center',
    marginBottom: 15,
  },
  successButton: {
    backgroundColor: '#28a745', // Darker green for the button
  },
  notificationStatusSection: {
    marginTop: 20,
    paddingTop: 15,
    borderTopWidth: 1,
    borderTopColor: '#eee',
  },
  notificationStatusTitle: {
    fontSize: 16,
    fontWeight: '600',
    color: '#333',
    marginBottom: 10,
  },
  statusRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 5,
  },
  statusLabel: {
    fontSize: 14,
    color: '#555',
  },
  statusValue: {
    fontSize: 14,
    fontWeight: '600',
  },
})

export default TestLoginCredFCM