import React, { useState, useCallback, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  TextInput,
  Platform,
  Modal,
  Animated,
  PermissionsAndroid,
  Dimensions,
  StatusBar,
  KeyboardAvoidingView,
  ScrollView,
  Alert,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import { launchCamera, launchImageLibrary } from 'react-native-image-picker';
import NoticeAPI from '../../ConfigMongo/NoticeMongoConfig';
import RNFS from 'react-native-fs'; // Added for persistent storage
import sanitizeHtml from 'sanitize-html';
import Toast from 'react-native-toast-message';
import ToastAndroid from 'react-native/Libraries/Components/ToastAndroid/ToastAndroid';
import { useSafeAreaInsets } from 'react-native-safe-area-context';
import PushNotificationAPI from '../../ConfigMongo/PushNotificationConfig';

// Move Dimensions outside component to prevent hooks order issues
const { width, height } = Dimensions.get('window');
const scale = (size) => (width / 375) * size;
const TOTAL_HEADER_HEIGHT = width * 0.25;

// SyncModal Component
const SyncModal = ({ visible, onClose }) => {
  const rotateAnim = React.useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  React.useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
            />
            <Text style={styles.syncModalTextadd}>Saving Notice</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

// Helper function to show toast messages
const showToast = (type, title, message) => {
  if (Platform.OS === 'android') {
    ToastAndroid.show(message, ToastAndroid.LONG, ToastAndroid.BOTTOM);
  } else {
    Toast.show({
      type: type,
      text1: title,
      text2: message,
      position: 'bottom',
    });
  }
};

// Request camera permission
const requestCameraPermission = async () => {
  if (Platform.OS === 'android') {
    try {
      const granted = await PermissionsAndroid.request(
        PermissionsAndroid.PERMISSIONS.CAMERA,
        {
          title: 'Camera Permission',
          message: 'This app needs access to your camera to take photos.',
          buttonNeutral: 'Ask Me Later',
          buttonNegative: 'Cancel',
          buttonPositive: 'OK',
        }
      );
      return granted === PermissionsAndroid.RESULTS.GRANTED;
    } catch (err) {
      console.warn(err);
      return false;
    }
  }
  return true; // iOS handles permissions via ImagePicker
};

// Request gallery permission
const requestGalleryPermission = async () => {
  // Android 13+ uses Photo Picker which doesn't require permissions
  // Only request permission for Android 12 and below
  if (Platform.OS === 'android' && Platform.Version < 33) {
    try {
      const permission = PermissionsAndroid.PERMISSIONS.READ_EXTERNAL_STORAGE;
      const granted = await PermissionsAndroid.request(
        permission,
        {
          title: 'Gallery Permission',
          message: 'This app needs access to your gallery to upload images.',
          buttonNeutral: 'Ask Me Later',
          buttonNegative: 'Cancel',
          buttonPositive: 'OK',
        }
      );
      return granted === PermissionsAndroid.RESULTS.GRANTED;
    } catch (err) {
      console.warn(err);
      return false;
    }
  }
  return true; // iOS handles permissions via ImagePicker
};

const getMimeTypeFromExtension = (extension) => {
  switch (extension) {
    case 'jpg':
    case 'jpeg':
      return 'image/jpeg';
    case 'png':
      return 'image/png';
    case 'gif':
      return 'image/gif';
    case 'webp':
      return 'image/webp';
    default:
      return 'application/octet-stream';
  }
};

// Copy file to persistent storage
const copyFileToPersistentStorage = async (uri, fileName) => {
  try {
    const extension = fileName.split('.').pop().toLowerCase();
    const newFileName = `image_${Date.now()}.${extension}`;
    const destPath = `${RNFS.DocumentDirectoryPath}/${newFileName}`;
    await RNFS.copyFile(uri, destPath);
    const normalizedUri = Platform.OS === 'android'
      ? `file://${destPath}`
      : destPath;
    return { 
      uri: normalizedUri,
      path: destPath,
      name: newFileName,
      type: getMimeTypeFromExtension(extension),
    };
  } catch (error) {
    console.error('Error copying file to persistent storage:', error);
    throw new Error('Failed to copy file to storage');
  }
};

const Add_NoticeONNB = () => {
  const navigation = useNavigation();
  const insets = useSafeAreaInsets();
  const [title, setTitle] = useState('');
  const [description, setDescription] = useState('');
  const [selectedImage, setSelectedImage] = useState(null); // Updated to match DocumentuploadS
  const [errors, setErrors] = useState({ title: '', description: '', image: '' });
  const [loading, setLoading] = useState(false);
  const [showImageModal, setShowImageModal] = useState(false);

  const slideAnim = useRef(new Animated.Value(Dimensions.get('window').height)).current;

  // Animate modal slide-in/slide-out
  React.useEffect(() => {
    if (showImageModal) {
      Animated.spring(slideAnim, {
        toValue: 0,
        useNativeDriver: true,
      }).start();
    } else {
      Animated.timing(slideAnim, {
        toValue: Dimensions.get('window').height,
        duration: 300,
        useNativeDriver: true,
      }).start();
    }
  }, [showImageModal]);

  // Sanitize input
  const sanitizeInput = useCallback((text) => {
    return sanitizeHtml(text, {
      allowedTags: [],
      allowedAttributes: {},
    });
  }, []);

  // Validate form
  const validateForm = useCallback(() => {
    const newErrors = {
      title: title.trim() ? '' : 'Title is required',
      description: description.trim() ? '' : 'Description is required',
      image: '',
    };
    setErrors(newErrors);
    return Object.values(newErrors).every((error) => !error);
  }, [title, description]);

  // Handle image selection (camera or gallery)
  const handleImageSelection = async (option) => {
    try {
      let imageInfo;
      if (option === 'camera') {
        const granted = await requestCameraPermission();
        if (!granted) {
          showToast('error', 'Permission Denied', 'Camera permission denied');
          return;
        }
        const options = {
          mediaType: 'photo',
          maxWidth: 1024,
          maxHeight: 1024,
          quality: 0.8,
        };
        const response = await launchCamera(options);
        if (response.didCancel) {
          console.log('User cancelled camera');
          return;
        } else if (response.errorCode) {
          throw new Error(response.errorMessage || 'Camera error');
        }
        
        if (!response || !response.assets || response.assets.length === 0) {
          throw new Error('No image captured or invalid response from camera');
        }
        
        const asset = response.assets[0];
        if (!asset || !asset.uri) {
          throw new Error('Invalid image data received from camera');
        }
        
        const uri = asset.uri;
        const fileName = asset.fileName || `photo_${Date.now()}.jpg`;
        const extension = fileName.split('.').pop().toLowerCase();
        if (!['jpg', 'jpeg', 'png', 'gif'].includes(extension)) {
          throw new Error('Unsupported image format. Please select JPG, JPEG, PNG, or GIF.');
        }
        // Validate file size (max 5MB)
        if (asset.fileSize && asset.fileSize > 5 * 1024 * 1024) {
          throw new Error('Image size exceeds 5MB limit.');
        }
        imageInfo = await copyFileToPersistentStorage(uri, fileName);
      } else if (option === 'gallery') {
        const granted = await requestGalleryPermission();
        if (!granted) {
          showToast('error', 'Permission Denied', 'Gallery permission denied');
          return;
        }
        const options = {
          mediaType: 'photo',
          maxWidth: 1024,
          maxHeight: 1024,
          quality: 0.8,
        };
        const response = await launchImageLibrary(options);
        if (response.didCancel) {
          console.log('User cancelled gallery');
          return;
        } else if (response.errorCode) {
          throw new Error(response.errorMessage || 'Gallery error');
        }
        const uri = response.assets[0].uri;
        const fileName = response.assets[0].fileName || `image_${Date.now()}.jpg`;
        const extension = fileName.split('.').pop().toLowerCase();
        if (!['jpg', 'jpeg', 'png', 'gif'].includes(extension)) {
          throw new Error('Unsupported image format. Please select JPG, JPEG, PNG, or GIF.');
        }
        // Validate file size (max 5MB)
        if (response.assets[0].fileSize > 5 * 1024 * 1024) {
          throw new Error('Image size exceeds 5MB limit.');
        }
        imageInfo = await copyFileToPersistentStorage(uri, fileName);
      }

      setSelectedImage(imageInfo);
      setErrors((prevErrors) => ({ ...prevErrors, image: '' }));
      setShowImageModal(false);
      showToast('success', 'Success', 'Image selected successfully.');
    } catch (error) {
      console.error('Image selection error:', error);
      showToast('error', 'Error', error.message || 'Failed to select image');
    }
  };

  // Upload image to MongoDB
  const uploadImageToMongoDB = useCallback(async (imageInfo, userUid) => {
    try {
      console.log('📤 Uploading image to MongoDB...');
      
      const uploadResponse = await NoticeAPI.uploadImage(userUid, {
        uri: imageInfo.uri,
        type: imageInfo.type || getMimeTypeFromExtension(imageInfo.name.split('.').pop().toLowerCase()),
        name: imageInfo.name,
      });
      
      if (!uploadResponse.success) {
        throw new Error(uploadResponse.error || 'Failed to upload image');
      }
      
      console.log('✅ Image uploaded successfully:', uploadResponse.imageUrl);
      return uploadResponse.imageUrl;
    } catch (error) {
      console.error('❌ Error uploading image to MongoDB:', error);
      throw error;
    }
  }, []);

  // Save notice to MongoDB
  const handleAddNotice = useCallback(async () => {
    if (!validateForm()) return;

    setLoading(true);
    try {
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) {
        showToast('error', 'Error', 'No user logged in. Please log in again.');
        navigation.replace('LoginScreens');
        return;
      }

      // Get user name from AsyncStorage or MongoDB
      let userName = 'Admin';
      try {
        const userDataStr = await AsyncStorage.getItem('userData');
        if (userDataStr) {
          const userData = JSON.parse(userDataStr);
          userName = userData.name || 'Admin';
        }
      } catch (err) {
        console.warn('⚠️ Could not get user name, using default:', err);
      }

      const noticeId = Date.now().toString();
      let imageUrl = null;

      if (selectedImage) {
        imageUrl = await uploadImageToMongoDB(selectedImage, userUid);
      }

      const notice = {
        noticeId,
        title: sanitizeInput(title),
        description: sanitizeInput(description),
        imageUrl,
        createdBy: userUid,
        createdAt: new Date().toISOString(),
      };

      console.log('📝 Saving notice to MongoDB:', notice);
      
      const saveResponse = await NoticeAPI.createNotice(notice);
      
      if (!saveResponse.success) {
        throw new Error(saveResponse.error || 'Failed to save notice');
      }
      
      console.log('✅ Notice saved successfully');

      // Send push notification to all users (background task)
      setTimeout(() => {
        (async () => {
          try {
            console.log('🔔 Sending push notification to all users about new notice...');
            console.log('📋 Notification details:', {
              title: sanitizeInput(title),
              description: sanitizeInput(description),
              noticeId,
              createdBy: userName
            });
            
            const notificationResult = await PushNotificationAPI.sendNewNoticeNotification(
              sanitizeInput(title),
              sanitizeInput(description),
              noticeId,
              userName
            );
            
            if (notificationResult && notificationResult.success) {
              console.log(`✅ Push notification sent to ${notificationResult.successCount} device(s) across ${notificationResult.totalUsers} user(s)`);
            } else {
              console.error('❌ Failed to send push notification:', notificationResult?.error || 'Unknown error');
              console.error('📋 Full response:', JSON.stringify(notificationResult));
            }
          } catch (notifError) {
            console.error('❌ Error sending push notification:', notifError);
            console.error('📋 Error details:', {
              message: notifError.message,
              stack: notifError.stack
            });
            // Don't fail notice creation if notification fails
          }
        })();
      }, 500);

      // Show success message
      showToast('success', 'Notice Added', 'Notice added and notifications sent to all users!');

      // Clear form and navigate back
      setTitle('');
      setDescription('');
      setSelectedImage(null);
      navigation.goBack();
      
    } catch (error) {
      console.error('❌ Error adding notice:', error);
      showToast('error', 'Error', `Failed to add notice: ${error.message}`);
    } finally {
      setLoading(false);
    }
  }, [validateForm, title, description, selectedImage, sanitizeInput, uploadImageToMongoDB, navigation]);

  return (
    <View style={styles.screen}>
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <TouchableOpacity
            style={styles.backButton}
            onPress={() => navigation.goBack()}
            accessibilityRole="button"
            accessibilityLabel="Go Back"
          >
            <MaterialIcons name="arrow-back" size={24} color="#fff" />
          </TouchableOpacity>
          <Text style={styles.headerTitle}>Add Notice</Text>
          <View style={{ width: 24 }} />
        </View>
      </LinearGradient>
      <KeyboardAvoidingView
        style={{ flex: 1 }}
        behavior={Platform.OS === 'ios' ? 'padding' : undefined}
        keyboardVerticalOffset={Platform.OS === 'ios' ? 0 : 0}
      >
        <ScrollView contentContainerStyle={styles.content} keyboardShouldPersistTaps="handled">
          {/* Title Input */}
          <Text style={styles.inputLabel}>Title</Text>
          <TextInput
            style={styles.input}
            value={title}
            onChangeText={setTitle}
            placeholder="Enter notice title"
            placeholderTextColor="#888"
            accessibilityLabel="Notice Title"
          />
          {errors.title ? <Text style={styles.errorText}>{errors.title}</Text> : null}

          {/* Description Editor */}
          <Text style={styles.inputLabel}>Description</Text>
          <TextInput
            style={styles.editor}
            value={description}
            onChangeText={setDescription}
            placeholder="Write your notice description..."
            placeholderTextColor="#888"
            multiline
            numberOfLines={6}
            textAlignVertical="top"
            accessibilityLabel="Notice Description"
          />
          {errors.description ? <Text style={styles.errorText}>{errors.description}</Text> : null}

          {/* Image Upload Button */}
          <TouchableOpacity
            style={[styles.imageUploadBtn, errors.image ? { borderColor: '#FF0000' } : null]}
            onPress={() => setShowImageModal(true)}
            accessibilityRole="button"
            accessibilityLabel="Upload Image"
          >
            <MaterialIcons name="image" size={32} color="#3481BC" />
            <Text style={styles.imageUploadText}>
              {selectedImage ? 'Image Selected' : 'Upload Image'}
            </Text>
          </TouchableOpacity>
          {selectedImage && (
            <Text style={styles.selectedText} numberOfLines={1} ellipsizeMode="middle">
              {selectedImage.name}
            </Text>
          )}
          {errors.image ? <Text style={styles.errorText}>{errors.image}</Text> : null}

          {/* Add Notice Button - moved out of ScrollView for fixed position */}
        </ScrollView>
        <View style={[styles.addNoticeBtnWrapFixed, { paddingBottom: Math.max(insets.bottom || 0, 16) + (Platform.OS === 'android' ? 60 : 0) }]}>
          <TouchableOpacity
            style={styles.addNoticeBtn}
            activeOpacity={0.85}
            onPress={handleAddNotice}
            accessibilityRole="button"
            accessibilityLabel="Add Notice"
          >
            <LinearGradient colors={['#3481BC', '#025383']} style={styles.addNoticeGradient}>
              <Text style={styles.addNoticeText}>Add Notice</Text>
            </LinearGradient>
          </TouchableOpacity>
        </View>
      </KeyboardAvoidingView>

      {/* Image Selection Modal */}
      <Modal
        transparent
        visible={showImageModal}
        animationType="none"
        onRequestClose={() => setShowImageModal(false)}
      >
        <TouchableOpacity
          style={styles.modalOverlay}
          activeOpacity={1}
          onPress={() => setShowImageModal(false)}
        >
          <Animated.View
            style={[styles.imageModalContainer, { transform: [{ translateY: slideAnim }] }]}
          >
            <TouchableOpacity style={styles.imageModalOption} onPress={() => handleImageSelection('camera')}>
              <View style={styles.imageModalOptionContent}>
                <MaterialIcons name="camera-alt" size={24} color="#3481BC" style={styles.imageModalIcon} />
                <Text style={styles.imageModalText}>Take Picture</Text>
              </View>
            </TouchableOpacity>
            <TouchableOpacity style={styles.imageModalOption} onPress={() => handleImageSelection('gallery')}>
              <View style={styles.imageModalOptionContent}>
                <MaterialIcons name="photo-library" size={24} color="#3481BC" style={styles.imageModalIcon} />
                <Text style={styles.imageModalText}>Choose from Gallery</Text>
              </View>
            </TouchableOpacity>
            <TouchableOpacity style={styles.imageModalOption} onPress={() => setShowImageModal(false)}>
              <Text style={[styles.imageModalText, { color: '#FF0000' }]}>Cancel</Text>
            </TouchableOpacity>
          </Animated.View>
        </TouchableOpacity>
      </Modal>

      <SyncModal visible={loading} onClose={() => setLoading(false)} />
    </View>
  );
};

const styles = StyleSheet.create({
  screen: {
    flex: 1,
    backgroundColor: '#f6f7f9',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
    elevation: 4,
    shadowColor: '#000',
    shadowOpacity: 0.1,
    shadowRadius: 8,
    shadowOffset: { width: 0, height: 4 },
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: Platform.OS === 'ios' ? width * 0.08 : width * 0.04,
    height: '100%',
  },
  backButton: {
    padding: 4,
    marginRight: 8,
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.05,
    fontWeight: 'bold',
    letterSpacing: 0.5,
    flex: 1,
    textAlign: 'center',
    
  },
  content: {
    padding: 22,
    alignItems: 'stretch',
    justifyContent: 'flex-start',
    paddingBottom: 120,
  },
  inputLabel: {
    fontSize: 15,
    color: '#29486A',
    fontWeight: 'bold',
    marginBottom: 6,
    marginLeft: 2,
  },
  input: {
    borderWidth: 1,
    borderColor: '#bcd2ee',
    borderRadius: 6,
    paddingHorizontal: 12,
    paddingVertical: 10,
    fontSize: 16,
    color: '#222',
    backgroundColor: '#fff',
    marginBottom: 18,
  },
  editor: {
    borderWidth: 1,
    borderColor: '#bcd2ee',
    borderRadius: 8,
    paddingHorizontal: 12,
    paddingVertical: 12,
    fontSize: 16,
    color: '#222',
    backgroundColor: '#fff',
    minHeight: 120,
    marginBottom: 22,
  },
  imageUploadBtn: {
    flexDirection: 'row',
    alignItems: 'center',
    alignSelf: 'flex-start',
    backgroundColor: '#eaf2fb',
    borderRadius: 8,
    paddingVertical: 10,
    paddingHorizontal: 18,
    borderWidth: 1,
    borderColor: '#3481BC',
  },
  imageUploadText: {
    color: '#3481BC',
    fontSize: 16,
    fontWeight: 'bold',
    marginLeft: 10,
  },
  selectedText: {
    fontSize: 14,
    color: '#3481BC',
    marginTop: 4,
    marginBottom: 16,
    marginLeft: 12,
  },
  addNoticeBtnWrap: {
    marginTop: 32,
    alignItems: 'center',
    width: '100%',
  },
  addNoticeBtnWrapFixed: {
    position: 'absolute',
    left: 0,
    right: 0,
    bottom: 0,
    backgroundColor: 'transparent',
    paddingHorizontal: 16,
    alignItems: 'center',
    zIndex: 10,
  },
  addNoticeBtn: {
    width: '100%',
    borderRadius: 1,
    overflow: 'hidden',
    height: 35,
  },
  addNoticeGradient: {
    height: 35,
    alignItems: 'center',
    justifyContent: 'center',
    borderRadius: 1,
    width: '100%',
  },
  addNoticeText: {
    color: '#fff',
    fontSize: 17,
    fontWeight: 'bold',
    letterSpacing: 0.5,
  },
  errorText: {
    color: '#FF0000',
    fontSize: 12,
    marginBottom: 10,
    marginLeft: 5,
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 30,
    height: 30,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'flex-end',
  },
  imageModalContainer: {
    backgroundColor: '#fff',
    borderTopLeftRadius: 20,
    borderTopRightRadius: 20,
    padding: 20,
  },
  imageModalOption: {
    paddingVertical: 15,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
    alignItems: 'center',
  },
  imageModalOptionContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  imageModalIcon: {
    marginRight: 10,
  },
  imageModalText: {
    fontSize: 18,
    color: '#3481BC',
  },
});

export default Add_NoticeONNB;
