import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  TextInput,
  Dimensions,
  FlatList,
  ActivityIndicator,
  Modal,
  Animated,
  Image,
  Linking,
  Platform,
  PermissionsAndroid,
  Share,
  ScrollView,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import Toast from 'react-native-toast-message';
import NetInfo from '@react-native-community/netinfo';
import ReactNativeBlobUtil from 'react-native-blob-util';

const { width } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.45;

const Checklist_library = () => {
  const navigation = useNavigation();
  const [searchQuery, setSearchQuery] = useState('');
  const [categories, setCategories] = useState([]);
  const [allDocuments, setAllDocuments] = useState([]);
  const [filteredDocuments, setFilteredDocuments] = useState([]);
  const [showLoaderModal, setShowLoaderModal] = useState(false);
  const [isOnline, setIsOnline] = useState(true);
  const [isLoading, setIsLoading] = useState(true);
  const [downloadingDocuments, setDownloadingDocuments] = useState(new Set());
  const [selectedTab, setSelectedTab] = useState('All');
  const [totalCount, setTotalCount] = useState(0);
  const [debugModalVisible, setDebugModalVisible] = useState(false);

  // MongoDB API URL
  const MONGODB_CHECKLIST_API_URL = 'https://api.titandrillingzm.com:6004';

  // Animation setup for loader icon
  const rotateAnim = useRef(new Animated.Value(0)).current;
  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  // Start animation when loader modal is shown
  useEffect(() => {
    if (showLoaderModal) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [showLoaderModal]);

  // Network status listener
  useEffect(() => {
    const unsubscribe = NetInfo.addEventListener(state => {
      setIsOnline(state.isConnected);
    });
    return unsubscribe;
  }, []);

  // Load documents from MongoDB server
  const loadDocuments = async () => {
    if (!isOnline) return;

    try {
      setShowLoaderModal(true);
      setIsLoading(true);

      console.log('📚 Fetching checklist library documents from MongoDB...');

      const response = await fetch(`${MONGODB_CHECKLIST_API_URL}/checklists/library`);
      
      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`);
      }
      
      const result = await response.json();
      
      if (result.success && result.data) {
        const { categories, allDocuments, totalDocuments } = result.data;
        console.log(`✅ Fetched ${totalDocuments} documents from ${categories.length} categories`);
        
        setCategories(categories);
        setAllDocuments(allDocuments);
        setFilteredDocuments(allDocuments);
        setTotalCount(totalDocuments);
        
        // Check if documents are already downloaded
        const storedDownloads = await AsyncStorage.getItem('downloadedDocuments');
        const downloadedItems = storedDownloads ? JSON.parse(storedDownloads) : {};
        
        setAllDocuments(prev => prev.map(doc => ({
          ...doc,
          downloaded: !!downloadedItems[doc.id]
        })));
        
        setFilteredDocuments(prev => prev.map(doc => ({
          ...doc,
          downloaded: !!downloadedItems[doc.id]
        })));
        
        Toast.show({
          type: 'success',
          text1: 'Success',
          text2: `Loaded ${totalDocuments} documents from ${categories.length} categories.`,
        });
      } else {
        setCategories([]);
        setAllDocuments([]);
        setFilteredDocuments([]);
        setTotalCount(0);
        Toast.show({
          type: 'info',
          text1: 'No Data',
          text2: 'No documents found in library.',
        });
      }
    } catch (error) {
      console.error('Error loading documents from MongoDB:', error);
      let errorMessage = 'Failed to load documents.';
      if (error.message.includes('Internet connection not available')) {
        errorMessage = 'No internet connection. Please connect to the internet.';
      }
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: errorMessage,
      });
    } finally {
      setShowLoaderModal(false);
      setIsLoading(false);
    }
  };

  // Load documents when component mounts or when online status changes
  useEffect(() => {
    if (isOnline) {
      loadDocuments();
    }
  }, [isOnline]);

  // Handle tab selection
  const handleTabSelect = (tabName) => {
    setSelectedTab(tabName);
    if (tabName === 'All') {
      setFilteredDocuments(allDocuments);
    } else {
      const category = categories.find(cat => cat.name === tabName);
      if (category) {
        setFilteredDocuments(category.documents);
      }
    }
  };

  // Handle document view
  const handleViewDocument = async (document) => {
    try {
      // Create download URL for viewing
      const downloadUrl = `${MONGODB_CHECKLIST_API_URL}/checklists/library/download/${encodeURIComponent(document.category)}/${encodeURIComponent(document.name)}`;
      
      // Navigate to ChecklistViewfile to view the document
      navigation.navigate('ChecklistViewfile', { 
        url: downloadUrl,
        title: document.name,
        name: document.name
      });

    } catch (error) {
      console.error('Error navigating to document view:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to open document view.',
      });
    }
  };

  // Handle document download
  const handleDownload = async (document) => {
    try {
      setDownloadingDocuments(prev => new Set(prev).add(document.id));
      setShowLoaderModal(true);

      // Request storage permission for Android
      // Android 13+ doesn't require permissions for Downloads folder
      if (Platform.OS === 'android' && Platform.Version < 33) {
        try {
            const granted = await PermissionsAndroid.request(
              PermissionsAndroid.PERMISSIONS.WRITE_EXTERNAL_STORAGE,
              {
                title: 'Storage Permission',
                message: 'App needs access to storage to download files.',
                buttonNeutral: 'Ask Me Later',
                buttonNegative: 'Cancel',
                buttonPositive: 'OK',
              }
            );
            if (granted !== PermissionsAndroid.RESULTS.GRANTED) {
              Toast.show({
                type: 'error',
                text1: 'Permission Denied',
                text2: 'Storage permission is required to download files',
                position: 'bottom',
                visibilityTime: 3000,
              });
              setShowLoaderModal(false);
              return;
            }
        } catch (err) {
          console.warn(err);
          Toast.show({
            type: 'error',
            text1: 'Permission Error',
            text2: 'Failed to request storage permission',
            position: 'bottom',
            visibilityTime: 3000,
          });
          return;
        }
      }

      // Download the document using ReactNativeBlobUtil
      try {
        const fileName = `${document.name}`;
        const downloadUrl = `${MONGODB_CHECKLIST_API_URL}/checklists/library/download/${encodeURIComponent(document.category)}/${encodeURIComponent(document.name)}`;
        let downloadDest;

        if (Platform.OS === 'android') {
          // On Android, use Download Manager to save to Downloads folder
          downloadDest = `${ReactNativeBlobUtil.fs.dirs.DownloadDir}/${fileName}`;
          const config = {
            fileCache: true,
            path: downloadDest,
            addAndroidDownloads: {
              useDownloadManager: true, // Use Android Download Manager
              notification: true, // Show notification
              title: fileName,
              description: 'Downloading document...',
              mime: document.type === 'PDF' ? 'application/pdf' : 'application/octet-stream',
              mediaScannable: true, // Make file visible in Downloads app
            },
          };

          const res = await ReactNativeBlobUtil.config(config).fetch('GET', downloadUrl);
          console.log('Document downloaded to:', res.path());

          Toast.show({
            type: 'success',
            text1: 'Download Complete',
            text2: `${document.name} downloaded to Downloads folder`,
            position: 'bottom',
            visibilityTime: 3000,
          });
        } else {
          // On iOS, save to DocumentDirectoryPath and offer to share
          downloadDest = `${ReactNativeBlobUtil.fs.dirs.DocumentDir}/${fileName}`;
          const config = {
            fileCache: true,
            path: downloadDest,
          };

          const res = await ReactNativeBlobUtil.config(config).fetch('GET', downloadUrl);
          console.log('Document downloaded to:', res.path());

          // Share the file on iOS to allow saving to Files app
          await Share.share({
            url: `file://${res.path()}`,
            message: 'Here is your downloaded document.',
          });

          Toast.show({
            type: 'success',
            text1: 'Download Complete',
            text2: `${document.name} downloaded and ready to share`,
            position: 'bottom',
            visibilityTime: 3000,
          });
        }

        // Store download status in AsyncStorage
        const stored = await AsyncStorage.getItem('downloadedDocuments');
        const downloadedItems = stored ? JSON.parse(stored) : {};
        downloadedItems[document.id] = {
          downloadedAt: new Date().toISOString(),
          fileSize: document.size,
          fileType: document.type
        };
        await AsyncStorage.setItem(`document_${document.id}`, JSON.stringify(document));
        await AsyncStorage.setItem('downloadedDocuments', JSON.stringify(downloadedItems));

        // Update document status
        setAllDocuments(prev => prev.map(doc => 
          doc.id === document.id ? { ...doc, downloaded: true } : doc
        ));

        // Update filtered documents as well
        setFilteredDocuments(prev => prev.map(doc => 
          doc.id === document.id ? { ...doc, downloaded: true } : doc
        ));

      } catch (downloadError) {
        console.error('Error downloading file:', downloadError);
        Toast.show({
          type: 'error',
          text1: 'Download Failed',
          text2: 'Failed to download file from server.',
          position: 'bottom',
          visibilityTime: 3000,
        });
      }

    } catch (error) {
      console.error('Error in download process:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to download document.',
        position: 'bottom',
        visibilityTime: 3000,
      });
    } finally {
      setShowLoaderModal(false);
      setDownloadingDocuments(prev => {
        const newSet = new Set(prev);
        newSet.delete(document.id);
        return newSet;
      });
    }
  };

  // Handle search
  const handleSearch = text => {
    setSearchQuery(text);
    if (text) {
      const filtered = allDocuments.filter(
        document =>
          document.name.toLowerCase().includes(text.toLowerCase()) ||
          document.category.toLowerCase().includes(text.toLowerCase()) ||
          document.type.toLowerCase().includes(text.toLowerCase())
      );
      setFilteredDocuments(filtered);
    } else {
      // Reset to current tab selection
      handleTabSelect(selectedTab);
    }
  };

  // Handle document selection
  const toggleSelection = id => {
    const updatedDocuments = allDocuments.map(document =>
      document.id === id ? { ...document, selected: !document.selected } : document
    );
    setAllDocuments(updatedDocuments);
    
    // Update filtered documents based on current tab
    if (selectedTab === 'All') {
      setFilteredDocuments(updatedDocuments);
    } else {
      const category = categories.find(cat => cat.name === selectedTab);
      if (category) {
        const categoryDocs = updatedDocuments.filter(doc => doc.category === selectedTab);
        setFilteredDocuments(categoryDocs);
      }
    }
    
    // Persist selection to AsyncStorage
    const selectedDocuments = updatedDocuments.filter(item => item.selected);
    AsyncStorage.setItem('selectedDocuments', JSON.stringify(selectedDocuments.map(item => item.id)));
  };

  // Format file size
  const formatFileSize = (bytes) => {
    if (bytes === 0) return '0 Bytes';
    const k = 1024;
    const sizes = ['Bytes', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
  };

  // Format date
  const formatDate = (dateString) => {
    return new Date(dateString).toLocaleDateString();
  };

  const renderItem = ({ item, index }) => (
    <View style={styles.cardWrapper}>
      <View style={styles.cardRow}>
        <View style={styles.cardLeft}>
          <View style={styles.circleNumber}>
            <Text style={styles.circleNumberText}>{index + 1}</Text>
          </View>
          <View style={styles.fileTypeIndicator}>
            <MaterialIcons name={item.icon} size={20} color="#015185" />
          </View>
        </View>
        <View style={styles.cardCenter}>
          <Text style={styles.cardTitle}>{item.name}</Text>
          <Text style={styles.cardSubtitle}>{item.type} Document</Text>
          <Text style={styles.cardDetail}>Category: {item.category}</Text>
          <Text style={styles.cardDetail}>Size: {formatFileSize(item.size)}</Text>
          <Text style={styles.cardDetail}>Created: {formatDate(item.createdAt)}</Text>
          <Text style={styles.cardDetail}>Modified: {formatDate(item.modifiedAt)}</Text>
          {item.downloaded && (
            <Text style={styles.downloadedStatus}>✓ Downloaded</Text>
          )}
          {downloadingDocuments.has(item.id) && (
            <Text style={styles.downloadingStatus}>⏳ Downloading...</Text>
          )}
          
          <View style={styles.actionsRow}>
            <TouchableOpacity
              style={styles.viewButton}
              onPress={() => handleViewDocument(item)}
            >
              <Text style={styles.viewButtonText}>View</Text>
            </TouchableOpacity>
            <TouchableOpacity
              style={[
                styles.downloadButton, 
                item.downloaded && styles.downloadedButton,
                downloadingDocuments.has(item.id) && styles.downloadingButton
              ]}
              onPress={() => handleDownload(item)}
              disabled={downloadingDocuments.has(item.id)}
            >
              <MaterialIcons
                name={
                  item.downloaded ? "check-circle" : 
                  downloadingDocuments.has(item.id) ? "hourglass-empty" : "file-download"
                }
                size={20}
                color={
                  item.downloaded ? "#4CAF50" : 
                  downloadingDocuments.has(item.id) ? "#888" : "#015185"
                }
              />
            </TouchableOpacity>
          </View>
        </View>
        <View style={styles.cardRight}>
          <TouchableOpacity onPress={() => toggleSelection(item.id)}>
            {item.selected ? (
              <MaterialIcons name="check" size={22} color="#015185" style={styles.checkIcon} />
            ) : (
              <View style={styles.checkIcon} />
            )}
          </TouchableOpacity>
        </View>
      </View>
    </View>
  );

  // Render tab buttons
  const renderTabs = () => {
    const tabNames = ['All', ...categories.map(cat => cat.name)];
    
    return (
      <View style={styles.tabsContainer}>
        <ScrollView 
          horizontal 
          showsHorizontalScrollIndicator={false}
          contentContainerStyle={styles.tabsContent}
        >
          {tabNames.map((tabName, index) => (
            <TouchableOpacity
              key={index}
              style={[
                styles.tabButton,
                selectedTab === tabName && styles.activeTabButton
              ]}
              onPress={() => handleTabSelect(tabName)}
            >
              <Text style={[
                styles.tabText,
                selectedTab === tabName && styles.activeTabText
              ]}>
                {tabName}
              </Text>
              {tabName !== 'All' && (
                <Text style={[
                  styles.tabCount,
                  selectedTab === tabName && styles.activeTabCount
                ]}>
                  {categories.find(cat => cat.name === tabName)?.count || 0}
                </Text>
              )}
            </TouchableOpacity>
          ))}
        </ScrollView>
      </View>
    );
  };

  return (
    <View style={styles.container}>
      {/* Header */}
      <LinearGradient colors={['#3481BC', '#025383']} style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}>
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity onPress={() => navigation.goBack()}>
              <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle} numberOfLines={1} ellipsizeMode="tail">
              Checklist Library
            </Text>
          </View>
          <View style={styles.headerRight}>
            <TouchableOpacity onPress={loadDocuments} style={styles.refreshButton}>
              <MaterialIcons name="refresh" size={width * 0.06} color={isOnline ? '#fff' : '#888'} />
            </TouchableOpacity>
            <TouchableOpacity 
              style={styles.infoButton}
              onPress={() => setDebugModalVisible(true)}
            >
              <MaterialIcons name="info" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
          </View>
        </View>
        <Text style={styles.subHeaderText}>
          Total Documents: {totalCount} | Categories: {categories.length}
        </Text>

        {/* Search Bar */}
        <View style={styles.searchContainer}>
          <View style={styles.searchWrapper}>
            <TextInput
              style={styles.searchInput}
              placeholder="Search documents..."
              placeholderTextColor="#888"
              value={searchQuery}
              onChangeText={handleSearch}
            />
            <MaterialIcons name="search" size={width * 0.05} color="#888" style={styles.searchIcon} />
          </View>
        </View>
      </LinearGradient>

      {/* Tabs */}
      {renderTabs()}

      {/* Custom Loader Modal */}
      <Modal
        transparent={true}
        visible={showLoaderModal}
        animationType="fade"
        onRequestClose={() => setShowLoaderModal(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Text style={styles.modalText}>Please Wait</Text>
            <View style={styles.syncIconDiv}>
              <Animated.Image
                source={require('../../../Images/adminlogin/sync.png')}
                style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
              />
              <Text style={styles.modalTextAdd}>Processing Documents</Text>
            </View>
          </View>
        </View>
      </Modal>

      {/* Offline Indicator */}
      {!isOnline && (
        <View style={styles.offlineIndicator}>
          <Text style={styles.offlineText}>Offline Mode - Data may not be up to date</Text>
        </View>
      )}

      {/* Document List */}
      {isLoading ? (
        <View style={styles.loadingContainer}>
          <ActivityIndicator size="large" color="#015185" />
          <Text style={styles.loadingText}>Loading Documents...</Text>
        </View>
      ) : (
        <View style={styles.contentContainer}>
          <FlatList
            data={filteredDocuments}
            renderItem={renderItem}
            keyExtractor={item => item.id}
            contentContainerStyle={styles.mainContent}
            showsVerticalScrollIndicator={false}
            ListEmptyComponent={
              <View style={styles.emptyContainer}>
                <MaterialIcons name="folder-open" size={64} color="#ccc" />
                <Text style={styles.emptyTitle}>
                  {searchQuery ? 'No documents found' : 'No documents in this category'}
                </Text>
                <Text style={styles.emptySubtitle}>
                  {searchQuery 
                    ? 'Try adjusting your search terms' 
                    : categories.length === 0 
                      ? 'No categories available. Check your connection.' 
                      : 'This category is empty'
                  }
                </Text>
                {categories.length === 0 && (
                  <TouchableOpacity style={styles.retryButton} onPress={loadDocuments}>
                    <Text style={styles.retryButtonText}>Retry</Text>
                  </TouchableOpacity>
                )}
              </View>
            }
          />
        </View>
      )}

      {/* Debug Modal */}
      <Modal
        visible={debugModalVisible}
        transparent
        animationType="fade"
        onRequestClose={() => setDebugModalVisible(false)}
      >
        <View style={styles.debugModalOverlay}>
          <View style={styles.debugModalContent}>
            <Text style={styles.debugModalTitle}>Debug Information</Text>
            
            <Text style={styles.debugModalLabel}>API URL:</Text>
            <Text style={styles.debugModalValue}>{MONGODB_CHECKLIST_API_URL}/checklists/library</Text>
            
            <Text style={styles.debugModalLabel}>Online Status:</Text>
            <Text style={styles.debugModalValue}>{isOnline ? 'Online' : 'Offline'}</Text>
            
            <Text style={styles.debugModalLabel}>Loading Status:</Text>
            <Text style={styles.debugModalValue}>{isLoading ? 'Loading...' : 'Loaded'}</Text>
            
            <Text style={styles.debugModalLabel}>Categories Found:</Text>
            <Text style={styles.debugModalValue}>{categories.length}</Text>
            
            <Text style={styles.debugModalLabel}>Total Documents:</Text>
            <Text style={styles.debugModalValue}>{totalCount}</Text>
            
            <Text style={styles.debugModalLabel}>Filtered Documents:</Text>
            <Text style={styles.debugModalValue}>{filteredDocuments.length}</Text>
            
            <Text style={styles.debugModalLabel}>Selected Tab:</Text>
            <Text style={styles.debugModalValue}>{selectedTab}</Text>
            
            {categories.length > 0 && (
              <>
                <Text style={styles.debugModalLabel}>Category Names:</Text>
                <Text style={styles.debugModalValue}>
                  {categories.map(cat => cat.name).join(', ')}
                </Text>
              </>
            )}
            
            <TouchableOpacity
              style={styles.debugCloseButton}
              onPress={() => setDebugModalVisible(false)}
            >
              <Text style={styles.debugCloseButtonText}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      {/* Toast Component */}
      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
    marginTop: 20,
    flex: 1,
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.05,
    fontWeight: 'bold',
    marginLeft: width * 0.025,
    flex: 1,
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
    marginTop: 20,
  },
  subHeaderText: {
    color: '#fff',
    fontSize: width * 0.035,
    paddingHorizontal: width * 0.04,
    paddingTop: 5,
    paddingBottom: 10,
  },
  searchContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    marginHorizontal: 16,
    marginTop: 8,
    marginBottom: 12,
    backgroundColor: '#fff',
    borderRadius: 25,
    paddingHorizontal: 16,
    borderWidth: 1,
    borderColor: '#E0E0E0',
    height: 44,
  },
  searchWrapper: {
    flex: 1,
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#fff',
    borderRadius: 25,
    paddingHorizontal: width * 0.025,
  },
  searchInput: {
    flex: 1,
    fontSize: 16,
    color: '#222',
    backgroundColor: 'transparent',
    paddingVertical: 0,
  },
  searchIcon: {
    marginLeft: 8,
  },
  mainContent: {
    paddingHorizontal: 0,
    paddingVertical: 0,
    paddingBottom: 30,
  },
  defaultText: {
    fontSize: 16,
    color: '#888',
    textAlign: 'center',
    marginVertical: 20,
  },
  cardWrapper: {
    backgroundColor: '#fff',
    borderTopWidth: 2,
    borderTopColor: '#fff',
    borderBottomWidth: 2,
    borderBottomColor: '#fff',
    marginBottom: 0,
  },
  cardRow: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#F2F2F2',
    borderRadius: 12,
    marginHorizontal: 16,
    marginTop: 18,
    paddingVertical: 18,
    paddingHorizontal: 12,
    minHeight: 120,
  },
  cardLeft: {
    alignItems: 'center',
    justifyContent: 'center',
    width: 40,
  },
  cardCenter: {
    flex: 1,
    alignItems: 'center',
    justifyContent: 'center',
  },
  cardRight: {
    alignItems: 'center',
    justifyContent: 'center',
    width: 48,
  },
  circleNumber: {
    width: 32,
    height: 32,
    borderRadius: 16,
    backgroundColor: '#015185',
    alignItems: 'center',
    justifyContent: 'center',
  },
  circleNumberText: {
    color: '#fff',
    fontWeight: 'bold',
    fontSize: 18,
  },
  fileTypeIndicator: {
    marginTop: 8,
  },
  checkIcon: {
    marginBottom: 8,
  },
  cardTitle: {
    fontSize: 15,
    fontWeight: 'bold',
    color: '#222',
    textAlign: 'center',
    marginTop: 8,
  },
  cardSubtitle: {
    fontSize: 13,
    color: '#888',
    textAlign: 'center',
    marginBottom: 5,
  },
  cardDetail: {
    fontSize: 11,
    color: '#666',
    textAlign: 'center',
    marginBottom: 2,
  },
  downloadedStatus: {
    fontSize: 11,
    color: '#4CAF50',
    textAlign: 'center',
    marginBottom: 2,
    fontWeight: 'bold',
  },
  downloadingStatus: {
    fontSize: 11,
    color: '#888',
    textAlign: 'center',
    marginBottom: 2,
    fontStyle: 'italic',
  },
  actionsRow: {
    flexDirection: 'row',
    justifyContent: 'center',
    marginTop: 4,
  },
  viewButton: {
    backgroundColor: '#015185',
    borderRadius: 4,
    paddingHorizontal: 10,
    paddingVertical: 4,
    marginHorizontal: 4,
  },
  viewButtonText: {
    color: '#fff',
    fontWeight: 'bold',
    fontSize: 13,
  },
  downloadButton: {
    backgroundColor: '#fff',
    borderRadius: 4,
    paddingHorizontal: 10,
    paddingVertical: 4,
    marginHorizontal: 4,
    borderWidth: 1,
    borderColor: '#015185',
  },
  downloadedButton: {
    borderColor: '#4CAF50',
    backgroundColor: '#E8F5E8',
  },
  downloadingButton: {
    borderColor: '#888',
    backgroundColor: '#f5f5f5',
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  modalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  modalTextAdd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcon: {
    width: 24,
    height: 24,
    marginRight: 10,
  },
  syncIconDiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  offlineIndicator: {
    backgroundColor: '#f8d7da',
    padding: 10,
    alignItems: 'center',
  },
  offlineText: {
    color: '#721c24',
    fontSize: 14,
  },
  loadingContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    paddingVertical: 20,
  },
  loadingText: {
    fontSize: 16,
    color: '#666',
    marginTop: 10,
  },
  loadMoreContainer: {
    paddingVertical: 20,
    alignItems: 'center',
  },
  loadMoreText: {
    fontSize: 14,
    color: '#666',
    marginTop: 8,
  },
  // Tab styles
  tabsContainer: {
    backgroundColor: '#f8f9fa',
    borderBottomWidth: 1,
    borderBottomColor: '#e0e0e0',
    paddingVertical: 8,
  },
  tabsContent: {
    paddingHorizontal: 16,
    flexDirection: 'row',
    alignItems: 'center',
  },
  tabButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingHorizontal: 12,
    paddingVertical: 6,
    marginRight: 8,
    borderRadius: 16,
    backgroundColor: '#fff',
    borderWidth: 1,
    borderColor: '#e0e0e0',
    minWidth: 60,
    justifyContent: 'center',
  },
  activeTabButton: {
    backgroundColor: '#015185',
    borderColor: '#015185',
  },
  tabText: {
    fontSize: 12,
    color: '#666',
    fontWeight: '500',
    textAlign: 'center',
  },
  activeTabText: {
    color: '#fff',
    fontWeight: '600',
  },
  tabCount: {
    fontSize: 10,
    color: '#999',
    marginLeft: 4,
    backgroundColor: '#f0f0f0',
    paddingHorizontal: 4,
    paddingVertical: 1,
    borderRadius: 8,
    minWidth: 16,
    textAlign: 'center',
  },
  activeTabCount: {
    color: '#015185',
    backgroundColor: '#e6f0fa',
  },
  // Content container styles
  contentContainer: {
    flex: 1,
    backgroundColor: '#fff',
  },
  emptyContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    paddingVertical: 60,
    paddingHorizontal: 20,
  },
  emptyTitle: {
    fontSize: 18,
    color: '#666',
    fontWeight: '600',
    marginTop: 16,
    textAlign: 'center',
  },
  emptySubtitle: {
    fontSize: 14,
    color: '#999',
    marginTop: 8,
    textAlign: 'center',
    lineHeight: 20,
  },
  retryButton: {
    marginTop: 20,
    backgroundColor: '#015185',
    paddingHorizontal: 24,
    paddingVertical: 12,
    borderRadius: 8,
  },
  retryButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: '600',
  },
  refreshButton: {
    padding: width * 0.015,
    marginRight: width * 0.02,
  },
  infoButton: {
    padding: width * 0.015,
  },
  // Debug modal styles
  debugModalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  debugModalContent: {
    backgroundColor: '#fff',
    borderRadius: 12,
    padding: 20,
    width: '90%',
    maxHeight: '80%',
  },
  debugModalTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: 16,
    textAlign: 'center',
  },
  debugModalLabel: {
    fontSize: 14,
    color: '#015185',
    fontWeight: '600',
    marginTop: 8,
  },
  debugModalValue: {
    fontSize: 12,
    color: '#333',
    marginBottom: 4,
    backgroundColor: '#f5f5f5',
    padding: 8,
    borderRadius: 4,
  },
  debugCloseButton: {
    alignSelf: 'center',
    marginTop: 16,
    backgroundColor: '#015185',
    borderRadius: 6,
    paddingHorizontal: 24,
    paddingVertical: 8,
  },
  debugCloseButtonText: {
    color: '#fff',
    fontWeight: 'bold',
    fontSize: 16,
  },
});

export default Checklist_library;