import React, { useState, useEffect } from 'react';
import {
  View,
  Text,
  TouchableOpacity,
  StatusBar,
  StyleSheet,
  Dimensions,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import AsyncStorage from '@react-native-async-storage/async-storage';
import NetInfo from '@react-native-community/netinfo';
import Toast from 'react-native-toast-message';
import Inspection_info_Checklist from './Inspection_info_Checklist';
import Do_inspection_checklist from './Do_inspection_checklist';
import Add_inspection_item from './Add_inspection_item';

const { height } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = height * 0.19 - 3;
const MONGODB_API_URL = 'https://api.titandrillingzm.com:6004';

const CreatechecklistHeader = ({ navigation, route }) => {
  const [activeStep, setActiveStep] = useState(1);
  const [addItemParams, setAddItemParams] = useState(null);
  const [userData, setUserData] = useState(null);
  const [userId, setUserId] = useState(null);
  const [isSaving, setIsSaving] = useState(false);
  const [lastSaveTime, setLastSaveTime] = useState(0);
  const [checklistData, setChecklistData] = useState({
    id: route.params?.checklist?._id || route.params?.checklist?.id || `custom_${Date.now()}`,
    title: route.params?.checklist?.title || '',
    description: route.params?.checklist?.description || '',
    items: route.params?.checklist?.items || [],
    createdAt: route.params?.checklist?.createdAt || new Date().toISOString(),
    updatedAt: route.params?.checklist?.updatedAt || new Date().toISOString(),
    createdBy: route.params?.checklist?.createdBy || null,
  });

  useEffect(() => {
    const fetchUserData = async () => {
      try {
        const userDataString = await AsyncStorage.getItem('userData');
        if (userDataString) {
          const parsedUserData = JSON.parse(userDataString);
          setUserData(parsedUserData);
          setUserId(parsedUserData.uid);
          
          // Set createdBy if not already set
          if (!checklistData.createdBy) {
            setChecklistData(prev => ({ ...prev, createdBy: parsedUserData.uid }));
          }
          
          console.log('✅ User data loaded:', parsedUserData.uid);
        } else {
          Toast.show({
            type: 'error',
            text1: 'Error',
            text2: 'User not logged in.',
          });
        }
      } catch (error) {
        console.error('❌ Error loading user data:', error);
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: 'Failed to load user data.',
        });
      }
    };

    fetchUserData();
  }, []);

  useEffect(() => {
    if (route.params?.checklist) {
      setChecklistData({
        id: route.params.checklist._id || route.params.checklist.id || `custom_${Date.now()}`,
        title: route.params.checklist.title || '',
        description: route.params.checklist.description || '',
        items: route.params.checklist.items || [],
        createdAt: route.params.checklist.createdAt || new Date().toISOString(),
        updatedAt: route.params.checklist.updatedAt || new Date().toISOString(),
        createdBy: route.params.checklist.createdBy || userId,
      });
    }
  }, [route.params?.checklist]);

  // Function to update local state only (no backend save)
  const updateChecklistData = (updatedData) => {
    console.log('📝 Updating local checklist data:', updatedData.title);
    setChecklistData(updatedData);
  };

  // Function to save to backend
  const saveChecklistToBackend = async (updatedData) => {
    if (!userId) {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'User not logged in.',
      });
      return;
    }

    // Prevent multiple simultaneous saves
    if (isSaving) {
      console.log('⚠️ Save already in progress, skipping duplicate call');
      return;
    }

    // Add debounce mechanism to prevent rapid-fire requests
    const currentTime = Date.now();
    const timeSinceLastSave = currentTime - lastSaveTime;
    
    if (timeSinceLastSave < 2000) { // 2 seconds debounce
      console.log('⚠️ Too soon since last save, skipping duplicate call');
      return;
    }

    // Add a small delay to prevent rapid-fire requests
    await new Promise(resolve => setTimeout(resolve, 100));

    // Check again after delay
    if (isSaving) {
      console.log('⚠️ Save already in progress after delay, skipping duplicate call');
      return;
    }

    setIsSaving(true);
    setLastSaveTime(currentTime);

    try {
      console.log('💾 Saving checklist to backend:', updatedData.id);
      
      // Check if this is an existing checklist (has _id or was loaded from DB)
      const isExisting = route.params?.checklist?._id || route.params?.checklist?.id;
      
      if (isExisting) {
        // Update existing checklist
        const response = await fetch(`${MONGODB_API_URL}/checklists/custom/${userId}/${updatedData.id}`, {
          method: 'PUT',
          headers: {
            'Content-Type': 'application/json',
          },
          body: JSON.stringify(updatedData),
        });

        const result = await response.json();
        console.log('📝 Update result:', result);

        if (result.success) {
          setChecklistData(result.data || updatedData);
          Toast.show({
            type: 'success',
            text1: 'Success',
            text2: 'Checklist updated successfully.',
          });
        } else {
          throw new Error(result.error || 'Update failed');
        }
      } else {
        // Create new checklist
        const response = await fetch(`${MONGODB_API_URL}/checklists/custom`, {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
          },
          body: JSON.stringify({
            checklistData: updatedData,
            userId: userId,
          }),
        });

        const result = await response.json();
        console.log('📝 Create result:', result);

        if (result.success) {
          setChecklistData(result.data || updatedData);
          Toast.show({
            type: 'success',
            text1: 'Success',
            text2: 'Checklist created successfully.',
          });
        } else {
          throw new Error(result.error || 'Creation failed');
        }
      }

      // Update local cache
      try {
        const cachedData = await AsyncStorage.getItem('customChecklists');
        let checklists = cachedData ? JSON.parse(cachedData) : [];
        
        const index = checklists.findIndex((c) => c.id === updatedData.id || c._id === updatedData.id);
        if (index >= 0) {
          checklists[index] = updatedData;
        } else {
          checklists.push(updatedData);
        }
        
        await AsyncStorage.setItem('customChecklists', JSON.stringify(checklists));
      } catch (cacheError) {
        console.error('❌ Error updating cache:', cacheError);
      }

    } catch (error) {
      console.error('❌ Error saving checklist:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: error.message || 'Failed to save checklist.',
      });
    } finally {
      setIsSaving(false);
    }
  };

  // Legacy function for backward compatibility (now only updates local state)
  const handleChecklistDataUpdate = (updatedData) => {
    updateChecklistData(updatedData);
  };

  const isStep1Complete = () => {
    return checklistData.title.trim() !== '' && checklistData.description.trim() !== '';
  };

  const steps = [
    {
      id: 1,
      title: 'Step 1',
      subtitle: 'Inspection Info',
      component: (
        <Inspection_info_Checklist
          checklistData={checklistData}
          setChecklistData={updateChecklistData}
          setActiveStep={setActiveStep}
        />
      ),
    },
    {
      id: 2,
      title: 'Step 2',
      subtitle: 'Do Inspection',
      component: addItemParams ? (
        <Add_inspection_item
          {...addItemParams}
          onSave={(subItem) => {
            addItemParams.onSave(subItem);
            setAddItemParams(null);
          }}
          onCancel={() => {
            addItemParams.onCancel();
            setAddItemParams(null);
          }}
        />
      ) : (
        <Do_inspection_checklist
          checklistData={checklistData}
          setChecklistData={updateChecklistData}
          setActiveStep={setActiveStep}
          setAddItemParams={setAddItemParams}
          saveChecklistToBackend={saveChecklistToBackend}
        />
      ),
    },
  ];

  if (!userId) {
    return (
      <View style={styles.container}>
        <LinearGradient
          colors={['#3481BC', '#025383']}
          style={styles.header}
        >
          <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
          <View style={styles.stepsRow}>
            <View style={styles.stepContainer}>
              <View style={styles.stepBadgeInactive}>
                <Text style={styles.stepBadgeTextInactive}>1</Text>
              </View>
              <View style={styles.stepTextBlock}>
                <Text style={styles.stepLabelWhite}>Step 1</Text>
                <Text style={styles.stepSubtitle}>Inspection Info</Text>
              </View>
            </View>
            <View style={styles.stepContainer}>
              <View style={styles.stepBadgeInactive}>
                <Text style={styles.stepBadgeTextInactive}>2</Text>
              </View>
              <View style={styles.stepTextBlock}>
                <Text style={styles.stepLabelWhite}>Step 2</Text>
                <Text style={styles.stepSubtitle}>Do Inspection</Text>
              </View>
            </View>
          </View>
        </LinearGradient>
        <View style={styles.mainContent}>
          <Text style={styles.errorText}>Loading user data...</Text>
        </View>
      </View>
    );
  }

  return (
    <View style={styles.container}>
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={styles.header}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <TouchableOpacity
          style={styles.homeButton}
          onPress={() => navigation.goBack()}
        >
          <MaterialIcons name="arrow-back" size={24} color="#fff" />
          <Text style={styles.headerHomeText}>Back</Text>
        </TouchableOpacity>
        <View style={styles.stepsRow}>
          <View style={styles.stepContainer}>
            {activeStep === 2 ? (
              <View style={styles.stepBadgeActive}>
                <MaterialIcons name="check" size={18} color="#fff" />
              </View>
            ) : (
              <View style={styles.stepBadgeActive}>
                <Text style={styles.stepBadgeTextActive}>1</Text>
              </View>
            )}
            <View style={styles.stepTextBlock}>
              <Text style={styles.stepLabelWhite}>Step 1</Text>
              <Text style={styles.stepSubtitle}>Inspection Info</Text>
            </View>
          </View>
          <View style={styles.stepContainer}>
            <View style={[styles.stepBadgeInactive, activeStep === 2 && { backgroundColor: '#4CAF50' }]}>
              <Text style={[styles.stepBadgeTextInactive, activeStep === 2 && { color: '#fff' }]}>2</Text>
            </View>
            <View style={styles.stepTextBlock}>
              <Text style={styles.stepLabelWhite}>Step 2</Text>
              <Text style={styles.stepSubtitle}>Do Inspection</Text>
            </View>
          </View>
        </View>
      </LinearGradient>
      <View style={styles.mainContent}>
        {steps.find((step) => step.id === activeStep).component}
      </View>
      <Toast position="bottom" />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    width: '100%',
    height: TOTAL_HEADER_HEIGHT,
    paddingTop: 32,
    paddingBottom: 0,
    paddingHorizontal: 0,
    justifyContent: 'flex-start',
    borderBottomLeftRadius: 0,
    borderBottomRightRadius: 0,
    top:"-4%",
  },
  homeButton: {
    position: 'absolute',
    top: 54,
    left: 10,
    flexDirection: 'row',
    alignItems: 'center',
    zIndex: 2,
  },
  headerHomeText: {
    
    color: '#fff',
    fontSize: 16,
    fontWeight: 'bold',
    marginLeft: 2,
  },
  stepsRow: {
    top: 24,
    flexDirection: 'row',
    justifyContent: 'center',
    alignItems: 'flex-end',
    marginTop: 30,
    width: '100%',
  },
  stepContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    marginHorizontal: 32,
  },
  stepBadgeActive: {
    width: 28,
    height: 28,
    borderRadius: 14,
    backgroundColor: '#4CAF50',
    alignItems: 'center',
    justifyContent: 'center',
    marginRight: 6,
  },
  stepBadgeInactive: {
    width: 28,
    height: 28,
    borderRadius: 14,
    backgroundColor: '#e0e6ed',
    alignItems: 'center',
    justifyContent: 'center',
    marginRight: 6,
  },
  stepBadgeTextActive: {
    color: '#fff',
    fontWeight: 'bold',
    fontSize: 16,
  },
  stepBadgeTextInactive: {
    color: '#3481BC',
    fontWeight: 'bold',
    fontSize: 16,
  },
  stepTextBlock: {
    flexDirection: 'column',
    alignItems: 'flex-start',
  },
  stepLabelWhite: {
    color: '#fff',
    fontWeight: 'bold',
    fontSize: 16,
    lineHeight: 18,
  },
  stepSubtitle: {
    color: '#fff',
    fontSize: 12,
    marginTop: -2,
  },
  mainContent: {
    flex: 1,
  },
  errorText: {
    fontSize: 16,
    color: 'red',
    textAlign: 'center',
    marginTop: 20,
  },
});

export default CreatechecklistHeader;