import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TextInput,
  TouchableOpacity,
  Dimensions,
  KeyboardAvoidingView,
  Platform,
  Modal,
  Alert,
  Animated,
  ScrollView,
  SafeAreaView,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { Dropdown } from 'react-native-element-dropdown';
import AsyncStorage from '@react-native-async-storage/async-storage';
// Removed Firebase imports - now using MongoDB and local file storage
import NetInfo from '@react-native-community/netinfo';
import Toast from 'react-native-toast-message';
import RNHTMLtoPDF from 'react-native-html-to-pdf';
import RNFS from 'react-native-fs';
import { check, request, PERMISSIONS, RESULTS } from 'react-native-permissions';
import { useNavigation } from '@react-navigation/native';
import DateTimePicker from '@react-native-community/datetimepicker';

const { width } = Dimensions.get('window');

const SyncModal = ({ visible }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
            />
            <Text style={styles.syncModalTextadd}>Generating Report</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

const DailyInsSummary = ({ setActiveStep }) => {
  const navigation = useNavigation();
  const [downtimeNote, setDowntimeNote] = useState('');
  const [isOnline, setIsOnline] = useState(true);
  const [inspectionId, setInspectionId] = useState(null);
  const [equipmentId, setEquipmentId] = useState(null);
  const [currentInspectionData, setCurrentInspectionData] = useState(null);
  const [checklistData, setChecklistData] = useState(null);
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [showSuccessModal, setShowSuccessModal] = useState(false);
  const [reportId, setReportId] = useState(null);
  const [inspectionDate, setInspectionDate] = useState(new Date());
  const [showDatePicker, setShowDatePicker] = useState(false);
  const [allReports, setAllReports] = useState([]);
  const [errors, setErrors] = useState({});

  useEffect(() => {
    let isMounted = true;

    const loadData = async () => {
      try {
        // Get user UID from AsyncStorage
        const userUid = await AsyncStorage.getItem('userUid');
        if (!userUid) {
          throw new Error('User UID not found. Please log in again.');
        }

        const savedInspection = await AsyncStorage.getItem('inspectionStep1');
        if (!savedInspection) throw new Error('Inspection data not found');
        const inspectionData = JSON.parse(savedInspection);
        // Patch: If equipmentId is missing but selectedEquipment exists, set it
        if ((!inspectionData.equipmentId || inspectionData.equipmentId === '') && inspectionData.selectedEquipment && inspectionData.selectedEquipment.id) {
          inspectionData.equipmentId = inspectionData.selectedEquipment.id;
          await AsyncStorage.setItem('inspectionStep1', JSON.stringify(inspectionData));
        }
        // Validate required fields with better error messages
        if (!inspectionData.id) {
          throw new Error('Inspection ID missing. Please go back to Step 1.');
        }
        if (!inspectionData.equipmentId) {
          throw new Error('Equipment ID missing. Please go back to Step 1.');
        }
        if (!inspectionData.requestNumber) {
          throw new Error('Request Number missing. Please go back to Step 1.');
        }
        if (!inspectionData.inspector) {
          throw new Error('Inspector data missing. Please go back to Step 1.');
        }
        if (!inspectionData.selectedEquipment) {
          throw new Error('Equipment data missing. Please go back to Step 1.');
        }
        
        setInspectionId(inspectionData.id);
        setEquipmentId(inspectionData.equipmentId);
        setCurrentInspectionData(inspectionData);
        // Use inspectionDate from inspectionStep1, not new Date()
        if (inspectionData.inspectionDate) {
          setInspectionDate(new Date(inspectionData.inspectionDate));
        } else {
          setInspectionDate(new Date());
        }

        let checklist = await AsyncStorage.getItem('checklistDetailsStep2');
        if (!checklist) {
          // Try to get checklist from step1 data first
          if (inspectionData.selectedChecklistId) {
            try {
              const checklistRef = database().ref(`/DailyInspectionReportJsonChecklists/AllDailyReportChecklistsJson/${inspectionData.selectedChecklistId}`);
              const snapshot = await checklistRef.once('value');
              const checklistData = snapshot.val();
              if (checklistData) {
                checklist = {
                  id: inspectionData.selectedChecklistId,
                  formTitle: checklistData.formTitle || 'Inspection Items',
                  inspectionItems: checklistData.inspectionItems || [],
                };
                await AsyncStorage.setItem('checklistDetailsStep2', JSON.stringify(checklist));
              } else {
                throw new Error('Checklist not found in Firebase');
              }
            } catch (error) {
              console.error('Error loading checklist from Firebase:', error);
              throw new Error('Checklist data not found. Please go back to Step 1.');
            }
          } else {
            // Fallback to selectedChecklistIds
          const savedChecklistIds = await AsyncStorage.getItem('selectedChecklistIds');
          const checklistIds = savedChecklistIds ? JSON.parse(savedChecklistIds) : [];
            if (!checklistIds.length) throw new Error('No checklist selected. Please go back to Step 1.');
            
            try {
          const checklistRef = database().ref(`/DailyInspectionReportJsonChecklists/AllDailyReportChecklistsJson/${checklistIds[0]}`);
          const snapshot = await checklistRef.once('value');
          const checklistData = snapshot.val();
          if (!checklistData) throw new Error('Checklist not found in Firebase');
          checklist = {
            id: checklistIds[0],
            formTitle: checklistData.formTitle || 'Inspection Items',
            inspectionItems: checklistData.inspectionItems || [],
          };
          await AsyncStorage.setItem('checklistDetailsStep2', JSON.stringify(checklist));
            } catch (error) {
              console.error('Error loading checklist:', error);
              throw new Error('Checklist data not found. Please go back to Step 1.');
            }
          }
        } else {
          checklist = JSON.parse(checklist);
        }
        
        // Validate checklist data
        if (!checklist || !checklist.inspectionItems || !Array.isArray(checklist.inspectionItems)) {
          throw new Error('Invalid checklist data. Please go back to Step 1.');
        }
        
        setChecklistData(checklist);

        const step2Data = await AsyncStorage.getItem('inspectionStep2');
        if (step2Data) {
          const step2 = JSON.parse(step2Data);
          // Validate step2 data
          if (step2 && step2.results) {
          setCurrentInspectionData((prev) => ({ ...prev, step2 }));
          } else {
            console.warn('Step 2 data is incomplete, proceeding without it');
          }
        } else {
          console.warn('No Step 2 data found, user may need to complete Step 2');
        }

        await fetchAllReports(inspectionData.equipmentId, isMounted);

        const netInfo = await NetInfo.fetch();
        setIsOnline(netInfo.isConnected);
      } catch (error) {
        console.error('Error loading data:', error);
        
        // Log detailed information about what data is available
        try {
          const step1Data = await AsyncStorage.getItem('inspectionStep1');
          const step2Data = await AsyncStorage.getItem('inspectionStep2');
          const checklistData = await AsyncStorage.getItem('checklistDetailsStep2');
          
          console.log('Available data:', {
            step1Data: step1Data ? 'Present' : 'Missing',
            step2Data: step2Data ? 'Present' : 'Missing',
            checklistData: checklistData ? 'Present' : 'Missing',
            userUid: await AsyncStorage.getItem('userUid') ? 'Present' : 'Missing'
          });
          
          if (step1Data) {
            const step1 = JSON.parse(step1Data);
            console.log('Step1 data keys:', Object.keys(step1));
          }
          
          if (step2Data) {
            const step2 = JSON.parse(step2Data);
            console.log('Step2 data keys:', Object.keys(step2));
          }
        } catch (debugError) {
          console.error('Error in debug logging:', debugError);
        }
        
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: error.message || 'Failed to load inspection data.',
          position: 'top',
        });
      }
    };

    const fetchAllReports = async (equipId, mounted) => {
      try {
        const userUid = await AsyncStorage.getItem('userUid');
        if (!userUid) return;
        let reports = [];
        if (isOnline) {
          const reportsRef = database().ref(`/GlobalDailyreportdata/admins/${userUid}`);
          const snapshot = await reportsRef.once('value');
          const data = snapshot.val() || {};
          reports = Object.values(data).filter((report) => report.step1?.equipmentId === equipId);
        } else {
          const storedReports = await AsyncStorage.getItem('dailyInspections');
          reports = storedReports
            ? JSON.parse(storedReports).filter((r) => r.equipmentId === equipId)
            : [];
        }
        if (mounted) {
          setAllReports(reports.sort((a, b) => new Date(b.step3.createdAt) - new Date(a.step3.createdAt)));
        }
      } catch (error) {
        console.error('Error fetching reports:', error);
      }
    };

    loadData();

    const unsubscribe = NetInfo.addEventListener((state) => {
      setIsOnline(state.isConnected);
      if (state.isConnected && equipmentId) {
        fetchAllReports(equipmentId, isMounted);
      }
    });

    return () => {
      isMounted = false;
      unsubscribe();
    };
  }, [equipmentId]);

  const validateFields = () => {
    // No validation needed since we removed the input fields
    return true;
  };

  const validateAllData = async () => {
    try {
      // Check step1 data
      const step1Data = await AsyncStorage.getItem('inspectionStep1');
      if (!step1Data) {
        throw new Error('Step 1 data not found');
      }
      const step1 = JSON.parse(step1Data);
      
      if (!step1.inspector) {
        throw new Error('Inspector data missing');
      }
      if (!step1.selectedEquipment) {
        throw new Error('Equipment data missing');
      }
      if (!step1.selectedChecklistId) {
        throw new Error('Checklist data missing');
      }
      
      // Check step2 data
      const step2Data = await AsyncStorage.getItem('inspectionStep2');
      if (!step2Data) {
        throw new Error('Step 2 data not found');
      }
      const step2 = JSON.parse(step2Data);
      
      if (!step2.results || Object.keys(step2.results).length === 0) {
        throw new Error('Step 2 inspection results missing');
      }
      
      return true;
    } catch (error) {
      console.error('Data validation error:', error);
      Toast.show({
        type: 'error',
        text1: 'Data Validation Error',
        text2: error.message + '. Please go back and complete all steps.',
        position: 'top',
        visibilityTime: 4000,
      });
      return false;
    }
  };

  const requestStoragePermission = async () => {
    try {
      if (Platform.OS === 'ios') {
        return true; // iOS doesn't need explicit storage permission for PDF generation
      }
      
      // Android 13+ uses Photo Picker which doesn't require READ_MEDIA_IMAGES
      // Only request legacy storage permissions for Android 12 and below
      const permissions = Platform.Version < 33
        ? [PERMISSIONS.ANDROID.READ_EXTERNAL_STORAGE, PERMISSIONS.ANDROID.WRITE_EXTERNAL_STORAGE]
        : [];
      
      for (const permission of permissions) {
        const result = await check(permission);
        if (result === RESULTS.DENIED) {
          const requestResult = await request(permission);
          if (requestResult !== RESULTS.GRANTED) {
            Alert.alert(
              'Permission Required', 
              'Storage permission is required to generate PDF reports. Please grant the permission to continue.',
              [
                { text: 'Cancel', style: 'cancel' },
                { text: 'Settings', onPress: () => Alert.alert('Go to Settings > Apps > Your App > Permissions') }
              ]
            );
            return false;
          }
        } else if (result === RESULTS.BLOCKED) {
          Alert.alert(
            'Permission Blocked', 
            'Storage permission is blocked. Please enable it in your device settings to generate PDF reports.',
            [
              { text: 'Cancel', style: 'cancel' },
              { text: 'Settings', onPress: () => Alert.alert('Go to Settings > Apps > Your App > Permissions') }
            ]
          );
          return false;
        }
      }
      return true;
    } catch (error) {
      console.error('Error requesting permission:', error);
      Alert.alert('Error', 'Failed to request storage permissions. Please try again.');
      return false;
    }
  };

  const uploadFileToLocalStorage = async (filePath, userId) => {
    try {
      console.log('📤 Starting PDF upload...', { filePath, userId });
      
      // Check if file exists
      const fileExists = await RNFS.exists(filePath);
      if (!fileExists) {
        throw new Error('PDF file does not exist at path: ' + filePath);
      }
      
      // Ensure file path has proper URI format for React Native
      let fileUri = filePath;
      if (Platform.OS === 'android' && !filePath.startsWith('file://')) {
        fileUri = `file://${filePath}`;
      } else if (Platform.OS === 'ios' && !filePath.startsWith('file://')) {
        fileUri = filePath.replace(/^\//, 'file:///');
      }
      
      const fileName = `Equipment_Inspection_${Date.now()}.pdf`;
      const formData = new FormData();
      
      // Correct FormData format for React Native
      formData.append('pdf', {
        uri: fileUri,
        type: 'application/pdf',
        name: fileName,
      });
      
      console.log('📤 Uploading PDF to server...', { fileName, fileUri });
      
      // IMPORTANT: Don't set Content-Type header - let fetch set it automatically with boundary
      const response = await fetch(`https://api.titandrillingzm.com:6008/daily-reports/upload-pdf/${userId}`, {
        method: 'POST',
        body: formData,
        // No headers - FormData will set the correct Content-Type with boundary
      });
      
      console.log('📤 Upload response status:', response.status);
      
      if (!response.ok) {
        const errorText = await response.text();
        console.error('📤 Upload error response:', errorText);
        throw new Error(`Upload failed: ${response.status} - ${errorText}`);
      }
      
      const result = await response.json();
      console.log('📤 Upload result:', result);
      
      if (!result.success) {
        throw new Error(result.error || 'Failed to upload PDF');
      }
      
      console.log('✅ PDF uploaded successfully:', result.pdfUrl);
      return result.pdfUrl;
    } catch (error) {
      console.error('❌ Error uploading PDF:', error);
      console.error('❌ Error details:', {
        message: error.message,
        filePath: filePath,
        userId: userId
      });
      throw error;
    }
  };

  const saveReport = async (userUid, report, pdfDownloadUrl = null) => {
    try {
      if (isOnline) {
        // Save to MongoDB DailyReports table
        const reportData = {
          _id: report.id,
          id: report.id,
          adminUid: userUid,
          requestNumber: report.step1.requestNumber,
          selectedEquipment: report.step1.selectedEquipment,
          step1: report.step1,
          step2: report.step2,
          step3: {
            ...report.step3,
            pdfDownloadUrl,
            inspector: report.step1.inspector || 'Unknown Inspector',
          },
          pdfDownloadUrl,
          createdAt: new Date().toISOString(),
          updatedAt: new Date().toISOString(),
        };

        // Save daily report data to MongoDB
        const dailyReportResponse = await fetch('https://api.titandrillingzm.com:6008/daily-reports', {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
          },
          body: JSON.stringify(reportData)
        });

        if (!dailyReportResponse.ok) {
          const errorData = await dailyReportResponse.json();
          throw new Error(errorData.error || `HTTP error! status: ${dailyReportResponse.status}`);
        }

        const dailyReportResult = await dailyReportResponse.json();
        
        if (!dailyReportResult.success) {
          throw new Error(dailyReportResult.error || 'Failed to save daily report to MongoDB');
        }

        console.log('✅ Daily report saved to MongoDB successfully');
      }

      // Also save to local storage for offline access
      let storedReports = await AsyncStorage.getItem('dailyInspections');
      storedReports = storedReports ? JSON.parse(storedReports) : [];
      storedReports = storedReports.filter((r) => r.id !== report.id);
      storedReports.push({
        ...report,
        equipmentId: report.step1.equipmentId,
        createdAt: report.step3.createdAt,
        pdfDownloadUrl: pdfDownloadUrl || null,
      });
      await AsyncStorage.setItem('dailyInspections', JSON.stringify(storedReports));
    } catch (error) {
      console.error('Error saving report:', error);
      throw error;
    }
  };

  const syncOfflineReports = async (userUid) => {
    try {
      let storedReports = await AsyncStorage.getItem('dailyInspections');
      if (!storedReports) return;
      storedReports = JSON.parse(storedReports);
      
      const pendingReports = storedReports.filter((report) => !report.synced);
      
      for (const report of pendingReports) {
        try {
          // Save to MongoDB DailyReports
          const reportData = {
            _id: report.id,
            id: report.id,
            adminUid: userUid,
            requestNumber: report.step1.requestNumber,
            selectedEquipment: report.step1.selectedEquipment,
            step1: report.step1,
            step2: report.step2,
            step3: report.step3,
            pdfDownloadUrl: report.pdfDownloadUrl,
            createdAt: new Date().toISOString(),
            updatedAt: new Date().toISOString(),
          };

          const dailyReportResponse = await fetch('https://api.titandrillingzm.com:6008/daily-reports', {
            method: 'POST',
            headers: {
              'Content-Type': 'application/json',
            },
            body: JSON.stringify(reportData)
          });

          if (dailyReportResponse.ok) {
            const dailyReportResult = await dailyReportResponse.json();
            if (dailyReportResult.success) {
              // Mark as synced
              report.synced = true;
              await AsyncStorage.setItem('dailyInspections', JSON.stringify(storedReports));
              console.log(`✅ Synced offline report ${report.id} to MongoDB DailyReports`);
            }
          }
          
          // Handle PDF generation and upload if needed
          if (!report.pdfDownloadUrl) {
            const pdfPath = await generatePDFWithHtml(report);
            const pdfDownloadUrl = await uploadFileToLocalStorage(pdfPath, userUid);
            await saveReport(userUid, report, pdfDownloadUrl);
            await RNFS.unlink(pdfPath).catch((err) => console.warn('Error deleting temp PDF:', err));
          }
        } catch (error) {
          console.error(`Error syncing report ${report.id}:`, error);
        }
      }
    } catch (error) {
      console.error('Error syncing offline reports:', error);
    }
  };

  const onDateChange = (event, selectedDate) => {
    setShowDatePicker(Platform.OS === 'ios');
    if (selectedDate) {
      setInspectionDate(selectedDate);
      AsyncStorage.getItem('inspectionStep1').then((savedInspection) => {
        if (savedInspection) {
          const inspectionData = JSON.parse(savedInspection);
          inspectionData.inspectionDate = selectedDate.toISOString();
          AsyncStorage.setItem('inspectionStep1', JSON.stringify(inspectionData));
          setCurrentInspectionData(inspectionData);
        }
      });
    }
  };



  const generatePDFWithHtml = async (report) => {
    try {
      const step1 = report.step1 || {};
      const step2 = report.step2 || {};
      const step3 = report.step3 || {};
      const checklistItems = report.checklist?.inspectionItems || [];
      const inspectionDate = new Date(step1.inspectionDate || new Date());
      const inspectorName = step1.inspector || 'Unknown Inspector';
      
      // Get header inputs from step2 if available
      const headerInputs = step2.headerInputs || {};
      
      // Fetch latest equipment data from MongoDB for dynamic updates
      let latestEquipmentData = null;
      try {
        if (isOnline) {
          const equipmentResponse = await fetch(`https://api.titandrillingzm.com:6001/equipment/${step1.equipmentId}`);
          if (equipmentResponse.ok) {
            const equipmentResult = await equipmentResponse.json();
            if (equipmentResult.success && equipmentResult.data) {
              latestEquipmentData = equipmentResult.data;
            }
          }
        }
      } catch (error) {
        console.warn('Error fetching latest equipment data from MongoDB:', error);
      }
      
      // Priority order: Step 2 header inputs (user filled) > Latest database data > Step 1 data > Fallback
      const machineName = headerInputs.machineName?.value || 
                         latestEquipmentData?.equipmentName || 
                         step1.selectedEquipment?.equipmentName || 'N/A';
      const model = headerInputs.model?.value || 
                   latestEquipmentData?.model || 
                   step1.selectedEquipment?.model || 'N/A';
      const serialNumber = headerInputs.serialNumber?.value || 
                          latestEquipmentData?.serialNumber || 
                          step1.selectedEquipment?.serialNumber || 'N/A';
      const headerMonth = headerInputs.month?.value || inspectionDate.toLocaleString('en-US', { month: 'long' });
      
      // Log what data source is being used for each field
      console.log('PDF Data Sources:', {
        machineName: {
          source: headerInputs.machineName?.value ? 'Step 2 Header Input' : 
                 latestEquipmentData?.equipmentName ? 'Database' : 
                 step1.selectedEquipment?.equipmentName ? 'Step 1' : 'Fallback',
          value: machineName
        },
        model: {
          source: headerInputs.model?.value ? 'Step 2 Header Input' : 
                 latestEquipmentData?.model ? 'Database' : 
                 step1.selectedEquipment?.model ? 'Step 1' : 'Fallback',
          value: model
        },
        serialNumber: {
          source: headerInputs.serialNumber?.value ? 'Step 2 Header Input' : 
                 latestEquipmentData?.serialNumber ? 'Database' : 
                 step1.selectedEquipment?.serialNumber ? 'Step 1' : 'Fallback',
          value: serialNumber
        }
      });
      
      // Update equipment data if it's different from what's stored
      if (latestEquipmentData && step1.selectedEquipment) {
        const needsUpdate = (
          latestEquipmentData.equipmentName !== step1.selectedEquipment.equipmentName ||
          latestEquipmentData.model !== step1.selectedEquipment.model ||
          latestEquipmentData.serialNumber !== step1.selectedEquipment.serialNumber
        );
        
        if (needsUpdate) {
          console.log('Equipment data updated from database:', {
            old: {
              equipmentName: step1.selectedEquipment.equipmentName,
              model: step1.selectedEquipment.model,
              serialNumber: step1.selectedEquipment.serialNumber
            },
            new: {
              equipmentName: latestEquipmentData.equipmentName,
              model: latestEquipmentData.model,
              serialNumber: latestEquipmentData.serialNumber
            }
          });
          
          // Update the stored equipment data with priority: Step 2 header inputs > Database > Step 1
          const updatedEquipment = {
            ...step1.selectedEquipment,
            equipmentName: headerInputs.machineName?.value || latestEquipmentData.equipmentName || step1.selectedEquipment.equipmentName,
            model: headerInputs.model?.value || latestEquipmentData.model || step1.selectedEquipment.model,
            serialNumber: headerInputs.serialNumber?.value || latestEquipmentData.serialNumber || step1.selectedEquipment.serialNumber,
            updatedAt: new Date().toISOString()
          };
          
          // Update AsyncStorage
          try {
            const updatedStep1 = { ...step1, selectedEquipment: updatedEquipment };
            await AsyncStorage.setItem('inspectionStep1', JSON.stringify(updatedStep1));
            
            // Update global equipment list
            const storedEquipment = await AsyncStorage.getItem('equipment');
            if (storedEquipment) {
              let equipmentList = JSON.parse(storedEquipment);
              const index = equipmentList.findIndex(eq => eq.id === updatedEquipment.id);
              if (index !== -1) {
                equipmentList[index] = updatedEquipment;
                await AsyncStorage.setItem('equipment', JSON.stringify(equipmentList));
              }
            }
            
            // Update equipment in MongoDB database
            if (isOnline) {
              try {
                const updatedEquipmentData = {
                  equipmentName: headerInputs.machineName?.value || latestEquipmentData.equipmentName || step1.selectedEquipment.equipmentName,
                  model: headerInputs.model?.value || latestEquipmentData.model || step1.selectedEquipment.model,
                  serialNumber: headerInputs.serialNumber?.value || latestEquipmentData.serialNumber || step1.selectedEquipment.serialNumber,
                  updatedAt: new Date().toISOString()
                };
                
                const equipmentResponse = await fetch(`https://api.titandrillingzm.com:6001/equipment/${updatedEquipment.id}`, {
                  method: 'PUT',
                  headers: {
                    'Content-Type': 'application/json',
                  },
                  body: JSON.stringify(updatedEquipmentData)
                });
                
                if (equipmentResponse.ok) {
                  console.log('Equipment updated in MongoDB database during PDF generation:', updatedEquipment.id);
                } else {
                  console.warn('Failed to update equipment in MongoDB during PDF generation');
                }
              } catch (mongodbError) {
                console.error('Error updating equipment in MongoDB during PDF generation:', mongodbError);
              }
            }
          } catch (error) {
            console.warn('Error updating stored equipment data:', error);
          }
        }
      }

      if (!checklistItems.length) {
        console.warn('No checklist items available, using fallback');
      }

      // Determine number of days in the month
      const year = inspectionDate.getFullYear();
      const month = inspectionDate.getMonth();
      const daysInMonth = new Date(year, month + 1, 0).getDate();
      const currentDay = inspectionDate.getDate();

      // Initialize checklist and inspector status
      const checklistStatus = checklistItems.reduce((acc, item) => {
        acc[item] = Array(daysInMonth).fill('');
        return acc;
      }, {});
      const inspectorStatus = Array(daysInMonth).fill('');
      const meterReadings = Array(daysInMonth).fill('');

      // Load all reports for this equipment, project, country, and checklist from the entire month
      let allMonthlyReports = [];
      try {
        const isOnline = await NetInfo.fetch().then((state) => state.isConnected);
        if (isOnline) {
          // Fetch all reports from MongoDB DailyReports
          const reportsResponse = await fetch('https://api.titandrillingzm.com:6008/daily-reports');
          if (reportsResponse.ok) {
            const reportsResult = await reportsResponse.json();
            if (reportsResult.success && reportsResult.data) {
              allMonthlyReports = reportsResult.data.filter((r) => {
                const reportDate = new Date(r.step1?.inspectionDate || r.createdAt);
                return r.step1?.equipmentId === step1.equipmentId &&
                       r.step1?.selectedChecklistId === step1.selectedChecklistId &&
                       r.step1?.country === step1.country &&
                       r.step1?.project === step1.project &&
                       reportDate.getFullYear() === year &&
                       reportDate.getMonth() === month;
              });
            }
          }
        }
      } catch (error) {
        console.warn('Error fetching online reports from MongoDB:', error);
      }

      // Also check local storage
      try {
        const storedReports = await AsyncStorage.getItem('dailyInspections');
        if (storedReports) {
          const localReports = JSON.parse(storedReports).filter((r) => {
            const reportDate = new Date(r.step1?.inspectionDate || r.step3?.createdAt);
            return r.equipmentId === step1.equipmentId &&
                   r.step1?.selectedChecklistId === step1.selectedChecklistId &&
                   r.step1?.country === step1.country &&
                   r.step1?.project === step1.project &&
                   reportDate.getFullYear() === year &&
                   reportDate.getMonth() === month;
          });
          allMonthlyReports = [...allMonthlyReports, ...localReports];
        }
      } catch (error) {
        console.warn('Error loading local reports:', error);
      }

      // Build a map of reports by day for fast lookup
      const reportByDay = {};
      allMonthlyReports.forEach((r) => {
        const reportDate = new Date(r.step1?.inspectionDate || r.step3?.createdAt);
        const day = reportDate.getDate();
        // If multiple reports for the same day, use the latest
        if (!reportByDay[day] || new Date(r.step3?.createdAt || r.step1?.inspectionDate) > new Date(reportByDay[day].step3?.createdAt || reportByDay[day].step1?.inspectionDate)) {
          reportByDay[day] = r;
        }
      });
      // Also include the current report
      reportByDay[currentDay] = report;

      // Fill checklistStatus, inspectorStatus, meterReadings, downtimeData, runtimeData for each day
      for (let d = 1; d <= daysInMonth; d++) {
        const r = reportByDay[d];
        if (r) {
          const rStep1 = r.step1 || {};
          const rStep2 = r.step2 || {};
          const rStep3 = r.step3 || {};
          checklistItems.forEach((item, idx) => {
            const itemId = `item_0_${idx}`;
            const status = rStep2.results?.[itemId] || 'Not Set';
            checklistStatus[item][d - 1] = status === 'Okay' ? '✓' : status === 'Not Okay' ? '✗' : 'N/A';
          });
          inspectorStatus[d - 1] = rStep1.inspector || 'Unknown';
          meterReadings[d - 1] = rStep1.meterReading || '';
        }
      }

      // Generate table rows
      const checklistRows = checklistItems.map((item) => {
        const statuses = checklistStatus[item].map((status, idx) => {
          const day = idx + 1;
          const cellStyle = day === currentDay ? 'background-color: #e0f7fa;' : status ? 'background-color: #f0f0f0;' : '';
          return `<td style="${cellStyle}">${status}</td>`;
        }).join('');
        return `
          <tr>
            <td style="text-align: left; font-weight: bold;">${item}</td>
            ${statuses}
          </tr>
        `;
      }).join('');

      // Additional data rows
      const meterReadingCells = meterReadings.map((reading, idx) => {
        const day = idx + 1;
        const cellStyle = day === currentDay ? 'background-color: #e0f7fa;' : reading ? 'background-color: #f0f0f0;' : '';
        return `<td style="${cellStyle}">${reading}</td>`;
      }).join('');

      // Inspector row
      const inspectorCells = inspectorStatus.map((inspector, idx) => {
        const day = idx + 1;
        const cellStyle = day === currentDay ? 'background-color: #e0f7fa;' : inspector ? 'background-color: #f0f0f0;' : '';
        return `<td style="${cellStyle}">${inspector}</td>`;
      }).join('');

      const html = `
        <!DOCTYPE html>
        <html lang="en">
        <head>
          <meta charset="UTF-8">
          <title>Equipment Inspection Report</title>
          <style>
            body { font-family: Arial, sans-serif; padding: 20px; font-size: 12px; line-height: 1.4; }
            .header-section {
              display: flex;
              justify-content: space-between;
              align-items: flex-start;
              margin-bottom: 20px;
            }
            .logo-section {
              flex: 1;
            }
            .logo-section img {
              width: 140px;
              height: 50px;
              object-fit: contain;
            }
            .document-info {
              flex: 1;
              margin-left: 20px;
            }
            .document-info table {
              width: 100%;
              border: 1px solid #222;
              border-collapse: collapse;
              font-size: 11px;
            }
            .document-info td {
              border: 1px solid #222;
              padding: 4px 6px;
              text-align: left;
            }
            .document-info td:first-child {
              font-weight: bold;
              background-color: #f5f5f5;
              width: 40%;
            }
            .title-section {
              text-align: center;
              margin: 20px 0;
            }
            .title-section h1 {
              margin: 0;
              font-size: 18px;
              font-weight: bold;
              color: #333;
            }
            .machine-details-section {
              margin: 20px 0;
              padding: 15px;
              border: 1px solid #222;
              background-color: #f9f9f9;
            }
            .machine-info {
              display: flex;
              justify-content: space-between;
              margin-bottom: 15px;
            }
            .info-row {
              display: flex;
              align-items: center;
              margin-right: 20px;
            }
            .info-label {
              font-weight: bold;
              margin-right: 10px;
              font-size: 12px;
              color: #333;
            }
            .info-value {
              border-bottom: 2px solid #000;
              min-width: 120px;
              padding: 2px 5px;
              text-align: center;
              background: #fff;
            }
            .month-info {
              display: flex;
              justify-content: center;
              align-items: center;
              margin-top: 10px;
            }
            .month-info .info-label {
              margin-right: 15px;
            }
            .month-info .info-value {
              min-width: 200px;
            }
            table { width: 100%; border-collapse: collapse; margin-bottom: 20px; }
            th, td { border: 1px solid #000; padding: 5px; text-align: center; }
            th { background-color: #f0f0f0; font-weight: bold; }
            .summary { margin-top: 20px; }
            .summary p { margin: 5px 0; }
            .current-day { background-color: #e0f7fa; }
            .data-row { background-color: #f9f9f9; }
          </style>
        </head>
        <body>
          <div class="header-section">
            <div class="logo-section">
              <img src="https://www.titanafricadrilling.com/wp-content/uploads/2022/10/titan-drlling_blue.png" alt="Titan Drilling Logo"/>
            </div>
            <div class="document-info">
              <table>
                <tr>
                  <td>Document Name</td>
                  <td>${step1.checklistTitle || report.checklist?.formTitle || 'Daily Inspection Report'}</td>
                </tr>
                <tr>
                  <td>Document ID</td>
                  <td>${report.checklist?.document?.number || 'HSE-001'}</td>
                </tr>
                <tr>
                  <td>Version Date</td>
                  <td>${report.checklist?.document?.versionDate || '01/01/2025'}</td>
                </tr>
                <tr>
                  <td>Version</td>
                  <td>${report.checklist?.document?.version || '1'}</td>
                </tr>
                <tr>
                  <td>Approved</td>
                  <td>${report.checklist?.document?.approvedBy || 'HSM'}</td>
                </tr>
                <tr>
                  <td>NO</td>
                  <td>${step1.requestNumber || '0001'}</td>
                </tr>
              </table>
            </div>
          </div>

          <div class="title-section">
            <h1>${report.checklist?.formTitle || 'DAILY EQUIPMENT INSPECTION REPORT'}</h1>
          </div>

          <div class="machine-details-section">
            <div class="machine-info">
              <div class="info-row">
                <span class="info-label">MACHINE NAME:</span>
                <span class="info-value">${machineName}</span>
              </div>
              <div class="info-row">
                <span class="info-label">MODEL:</span>
                <span class="info-value">${model}</span>
              </div>
              <div class="info-row">
                <span class="info-label">SERIAL #:</span>
                <span class="info-value">${serialNumber}</span>
              </div>
            </div>
            <div class="month-info">
              <span class="info-label">MONTH:</span>
              <span class="info-value">${headerMonth}</span>
            </div>
          </div>
          <table>
            <tr>
              <th>Item</th>
              ${Array.from({ length: daysInMonth }, (_, i) => `<th>${i + 1}</th>`).join('')}
            </tr>
            ${checklistRows}
            <tr class="data-row">
              <td style="text-align: left; font-weight: bold;">Meter Reading (${step1.selectedEquipment?.meterUnit || 'Miles'})</td>
              ${meterReadingCells}
            </tr>
            <tr>
              <td style="text-align: left; font-weight: bold;">Inspector</td>
              ${inspectorCells}
            </tr>
          </table>
          <div class="summary">
        
          </div>
        </body>
        </html>
      `;

      // <h3>Summary</h3>
      // <p><strong>Request Number:</strong> ${step1.requestNumber || 'N/A'}</p>
      // <p><strong>Date:</strong> ${inspectionDate.toLocaleDateString()}</p>
      // <p><strong>MACHINE NAME:</strong> ${machineName}</p>
      // <p><strong>MODEL:</strong> ${model}</p>
      // <p><strong>SERIAL #:</strong> ${serialNumber}</p>
      // <p><strong>MONTH:</strong> ${headerMonth}</p>
      // <p><strong>Meter Reading:</strong> ${step1.meterReading || 'N/A'} ${step1.selectedEquipment?.meterUnit || ''}</p>
      // <p><strong>Country:</strong> ${step1.country || 'N/A'}</p>
      // <p><strong>Project:</strong> ${step1.project || 'N/A'}</p>
      // <p><strong>Shift:</strong> ${step1.shift || 'N/A'}</p>
      // <p><strong>Checklist:</strong> ${step1.checklistTitle || 'N/A'}</p>
      // <p><strong>Coordinates:</strong> ${step1.coordinates || 'N/A'}</p>
      // <p><strong>GPS Address:</strong> ${step1.gpsAddress || 'N/A'}</p>
      // <p><strong>Work Log:</strong> ${step3.worklog || 'N/A'} ${step3.worklogUnit || ''}</p>

      const options = {
        html,
        fileName: `Equipment_Inspection_${report.id}`,
        directory: 'Documents',
      };

      console.log('Generating PDF with options:', options);
      const file = await RNHTMLtoPDF.convert(options);
      
      if (!file || !file.filePath) {
        throw new Error('PDF generation failed: No file path returned');
      }
      
      console.log('PDF generated successfully at:', file.filePath);
      return file.filePath;
    } catch (error) {
      console.error('Error generating PDF:', error);
      throw new Error(`PDF generation failed: ${error.message}`);
    }
  };

  const handleGenerateReport = async () => {
    if (!validateFields()) return;

    // Validate all required data before proceeding
    const isDataValid = await validateAllData();
    if (!isDataValid) return;

    const hasPermission = await requestStoragePermission();
    if (!hasPermission) return;

    setSyncModalVisible(true);
    try {
      // Get user ID from local storage instead of Firebase
      let userId = null;
      const storedUser = await AsyncStorage.getItem('userData');
      if (storedUser) {
        const userData = JSON.parse(storedUser);
        userId = userData.uid || userData.id;
        console.log('👤 User authenticated via local storage:', userId);
      } else {
        // Fallback: generate a temporary user ID for offline use
        userId = `temp_user_${Date.now()}`;
        console.log('👤 Using temporary user ID for offline mode:', userId);
      }

      if (!userId) {
        Alert.alert('Error', 'User not authenticated. Please log in again.');
        navigation.replace('LoginScreens');
        return;
      }

      const inspectionStep1Raw = await AsyncStorage.getItem('inspectionStep1');
      const inspectionStep2Raw = await AsyncStorage.getItem('inspectionStep2');

      if (!inspectionStep1Raw || !inspectionStep2Raw) {
        Alert.alert('Error', 'Inspection data is missing.');
        return;
      }

      const inspectionStep1 = JSON.parse(inspectionStep1Raw);
      const inspectionStep2 = JSON.parse(inspectionStep2Raw);

      if (!inspectionStep1.country || !inspectionStep1.project || !inspectionStep1.inspector) {
        Alert.alert('Error', 'Incomplete data from Step 1.');
        return;
      }
      if (!inspectionStep2.results || Object.keys(inspectionStep2.results).length === 0) {
        Alert.alert('Error', 'Incomplete data from Step 2.');
        return;
      }

      const reportId = Date.now().toString();
      setReportId(reportId);

      // Load checklist data
      const checklistDataStep2 = await AsyncStorage.getItem('checklistDetailsStep2');
      let checklistData = checklistDataStep2 ? JSON.parse(checklistDataStep2) : null;
      
      // Ensure checklist data has required fields
      if (!checklistData) {
        console.warn('No checklist data available, creating fallback');
        checklistData = {
          formTitle: 'Daily Equipment Inspection Report',
          inspectionItems: ['General Inspection', 'Safety Check', 'Equipment Status'],
          document: {
            number: 'HSE-001',
            versionDate: '01/01/2025',
            version: '1',
            approvedBy: 'HSM'
          }
        };
      }
      
      // Ensure inspection items exist
      if (!checklistData.inspectionItems || checklistData.inspectionItems.length === 0) {
        console.warn('No inspection items found, adding default items');
        checklistData.inspectionItems = [
          'Equipment Condition',
          'Safety Features',
          'Operational Status',
          'Maintenance Check',
          'Documentation Review'
        ];
      }

      const report = {
        id: reportId,
        step1: inspectionStep1,
        step2: inspectionStep2,
        step3: {
          downtime: 0,
          downtimeUnit: 'Hour',
          downtimeNote,
          runtime: 0,
          runtimeUnit: 'Hour',
          worklog: 0,
          worklogUnit: 'Hour',
          createdAt: inspectionDate.toISOString(),
        },
        checklist: checklistData,
      };

      // Save report to MongoDB DailyReports table
      if (isOnline) {
        try {
          console.log('🚀 Starting report generation process...');
          
          // Step 1: Generate PDF
          console.log('📄 Generating PDF...');
          const pdfPath = await generatePDFWithHtml(report);
          console.log('✅ PDF generated successfully:', pdfPath);
          
          // Step 2: Upload PDF to server (with fallback)
          console.log('📤 Uploading PDF to server...');
          let pdfDownloadUrl = null;
          try {
            pdfDownloadUrl = await uploadFileToLocalStorage(pdfPath, userId);
            console.log('✅ PDF uploaded successfully:', pdfDownloadUrl);
          } catch (uploadError) {
            console.warn('⚠️ PDF upload failed, proceeding without PDF URL:', uploadError.message);
            // Continue without PDF URL - the report will still be saved
            pdfDownloadUrl = null;
          }

          // Step 3: Prepare report data for MongoDB DailyReports
          const reportData = {
            _id: reportId,
            id: reportId,
            adminUid: userId,
            requestNumber: report.step1.requestNumber,
            selectedEquipment: report.step1.selectedEquipment,
            step1: report.step1,
            step2: report.step2,
            step3: {
              ...report.step3,
              pdfDownloadUrl,
              inspector: report.step1.inspector || 'Unknown Inspector',
            },
            pdfDownloadUrl,
            createdAt: new Date().toISOString(),
            updatedAt: new Date().toISOString(),
          };

          console.log('💾 Saving report to MongoDB...', { reportId, pdfDownloadUrl });

          // Step 4: Save to MongoDB DailyReports collection
          const response = await fetch('https://api.titandrillingzm.com:6008/daily-reports', {
            method: 'POST',
            headers: {
              'Content-Type': 'application/json',
            },
            body: JSON.stringify(reportData),
          });

          if (!response.ok) {
            const errorText = await response.text();
            console.error('❌ MongoDB save error response:', errorText);
            throw new Error(`HTTP error! status: ${response.status} - ${errorText}`);
          }

          const result = await response.json();
          if (!result.success) {
            throw new Error(result.error || 'Failed to save report to MongoDB');
          }

          console.log('✅ Report saved to MongoDB DailyReports collection:', result.data?._id || reportId);

          // Step 5: Clean up temp PDF file
          await RNFS.unlink(pdfPath).catch((err) => console.warn('Error deleting temp PDF:', err));
          
          // Step 6: Show success modal
          setSyncModalVisible(false);
          setShowSuccessModal(true);
          console.log('🎉 Success modal should be visible now');

          Toast.show({
            type: 'success',
            text1: 'Report Generated Successfully!',
            text2: pdfDownloadUrl ? 'Report saved to MongoDB with PDF' : 'Report saved to MongoDB (PDF upload failed)',
            position: 'top',
            visibilityTime: 3000,
          });

        } catch (error) {
          console.error('❌ Error in report generation process:', error);
          setSyncModalVisible(false);
          
          // Show specific error message
          let errorMessage = 'Failed to generate report.';
          if (error.message.includes('PDF generation failed')) {
            errorMessage = 'Failed to generate PDF. Please try again.';
          } else if (error.message.includes('upload')) {
            errorMessage = 'Failed to upload PDF. Please check your connection.';
          } else if (error.message.includes('MongoDB')) {
            errorMessage = 'Failed to save to database. Report saved locally.';
            // Save locally as fallback
            await saveReport(userId, report);
          }
          
          Alert.alert(
            'Error',
            errorMessage,
            [{ text: 'OK', onPress: () => navigation.replace('DailyIns_report_home') }]
          );
        }
      } else {
        await saveReport(userId, report);
        Alert.alert(
          'Success',
          'Report saved locally. It will sync when online.',
          [{ text: 'OK', onPress: () => navigation.replace('DailyIns_report_home') }]
        );
      }

      // Update equipment data in global equipment list and MongoDB database if it was modified during inspection
      if (currentInspectionData?.selectedEquipment) {
        try {
          // Get header inputs from step 2 to ensure we have the latest data
          const step2Data = await AsyncStorage.getItem('inspectionStep2');
          let headerInputs = {};
          if (step2Data) {
            const step2 = JSON.parse(step2Data);
            headerInputs = step2.headerInputs || {};
          }
          
          // Update local storage
          const storedEquipment = await AsyncStorage.getItem('equipment');
          if (storedEquipment) {
            let equipmentList = JSON.parse(storedEquipment);
            const index = equipmentList.findIndex(eq => eq.id === currentInspectionData.selectedEquipment.id);
            if (index !== -1) {
              // Update with the latest data from the inspection and step 2 header inputs
              equipmentList[index] = {
                ...equipmentList[index],
                ...currentInspectionData.selectedEquipment,
                // Use header input values if available (from step 2), otherwise use selectedEquipment data
                equipmentName: headerInputs.machineName?.value || currentInspectionData.selectedEquipment.equipmentName,
                model: headerInputs.model?.value || currentInspectionData.selectedEquipment.model,
                serialNumber: headerInputs.serialNumber?.value || currentInspectionData.selectedEquipment.serialNumber,
                meterReading: currentInspectionData.selectedEquipment.meterReading,
                updatedAt: new Date().toISOString()
              };
              await AsyncStorage.setItem('equipment', JSON.stringify(equipmentList));
              console.log('Updated equipment data in global list after report generation');
            }
          }
          
          // Update equipment in MongoDB (non-blocking)
          if (isOnline) {
            try {
              const updatedEquipmentData = {
                // Use header input values if available (from step 2), otherwise use selectedEquipment data
                equipmentName: headerInputs.machineName?.value || currentInspectionData.selectedEquipment.equipmentName,
                model: headerInputs.model?.value || currentInspectionData.selectedEquipment.model,
                serialNumber: headerInputs.serialNumber?.value || currentInspectionData.selectedEquipment.serialNumber,
                meterReading: currentInspectionData.selectedEquipment.meterReading,
                updatedAt: new Date().toISOString()
              };

              // Update equipment in MongoDB
              const equipmentResponse = await fetch(`https://api.titandrillingzm.com:6001/equipment/${currentInspectionData.selectedEquipment.id}`, {
                method: 'PUT',
                headers: {
                  'Content-Type': 'application/json',
                },
                body: JSON.stringify(updatedEquipmentData)
              });

              if (equipmentResponse.ok) {
                const equipmentResult = await equipmentResponse.json();
                if (equipmentResult.success) {
                  console.log('✅ Equipment updated in MongoDB after report generation:', currentInspectionData.selectedEquipment.id);
                } else {
                  console.warn('⚠️ Equipment update failed (non-blocking):', equipmentResult.error);
                }
              } else {
                console.warn('⚠️ Equipment update failed with status (non-blocking):', equipmentResponse.status);
              }
            } catch (mongodbError) {
              console.warn('⚠️ Equipment update error (non-blocking):', mongodbError.message);
              // Don't throw error or show error toast - this is non-blocking
            }
          }
        } catch (error) {
          console.warn('Error updating global equipment list:', error);
        }
      }

      // Clean up all AsyncStorage data after successful submission
      await AsyncStorage.removeItem('inspectionStep1');
      await AsyncStorage.removeItem('inspectionStep2');
      await AsyncStorage.removeItem('checklistDetailsStep2');
      await AsyncStorage.removeItem('selectedChecklistIds');
      await AsyncStorage.removeItem('globalSelectedEquipment');
      await AsyncStorage.removeItem('selectedChecklistData');
    } catch (error) {
      console.error('Error generating report:', error);
      Alert.alert('Error', 'Failed to generate or save report.');
    } finally {
      setSyncModalVisible(false);
    }
  };

  const handleViewPdf = async () => {
    try {
      console.log('📄 Opening PDF viewer...');
      
      // Get user ID from local storage
      let userId = null;
      const storedUser = await AsyncStorage.getItem('userData');
      if (storedUser) {
        const userData = JSON.parse(storedUser);
        userId = userData.uid || userData.id;
      }
      
      if (!userId) {
        Alert.alert('Error', 'User not authenticated. Please log in again.');
        navigation.replace('LoginScreens');
        return;
      }

      // Fetch the report from MongoDB to get the PDF URL
      const response = await fetch(`https://api.titandrillingzm.com:6008/daily-reports/${userId}/${reportId}`);
      
      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`);
      }

      const result = await response.json();
      if (!result.success || !result.data) {
        throw new Error('Report not found in database');
      }

      const reportData = result.data;
      const pdfUrl = reportData.pdfDownloadUrl || reportData.step3?.pdfDownloadUrl;
      
      console.log('📄 PDF URL from database:', pdfUrl);
      
      if (!pdfUrl) {
        Alert.alert('Error', 'PDF URL not available in the report.');
        return;
      }

      navigation.replace('PdfviewPage', {
        pdfUrl,
        onGoBack: () => navigation.replace('DailyIns_report_home'),
      });
    } catch (error) {
      console.error('❌ Error fetching PDF URL:', error);
      Alert.alert('Error', 'Could not fetch the PDF URL. Please try again.');
    } finally {
      setShowSuccessModal(false);
    }
  };

  const handleBackPress = () => {
    setActiveStep(2);
  };



  return (
    <SafeAreaView style={{ flex: 1, backgroundColor: '#fff' }}>
      <KeyboardAvoidingView style={{ flex: 1 }} behavior={Platform.OS === 'ios' ? 'padding' : undefined}>
        <ScrollView style={styles.container} contentContainerStyle={{ paddingBottom: 120 }}>
          <View style={{ marginBottom: 20 }}>
          
          </View>
          <View style={styles.downtimeNoteBox}>
            <View style={styles.downtimeNoteHeader}>
              <Text style={styles.downtimeNoteTitle}>Note</Text>
              {/* <MaterialIcons name="mic" size={24} color="#fff" /> */}
            </View>
            <TextInput
              style={styles.downtimeNoteInput}
              value={downtimeNote}
              onChangeText={setDowntimeNote}
              placeholder="Enter downtime notes"
              placeholderTextColor="#888"
              multiline
              numberOfLines={5}
              textAlignVertical="top"
            />
          </View>
          {checklistData?.inspectionItems?.length > 0 && (
            <View style={{ marginVertical: 20 }}>
          
            </View>
          )}
        </ScrollView>
        <View style={styles.bottomBarCustom}>
          <TouchableOpacity style={styles.backBtnCustom} onPress={handleBackPress}>
            <MaterialIcons name="arrow-back" size={24} color="#015185" />
            <Text style={styles.backTextCustom}>BACK</Text>
          </TouchableOpacity>
          <TouchableOpacity style={styles.generateBtnCustom} onPress={handleGenerateReport}>
            <Text style={styles.generateTextCustom}>GENERATE REPORT</Text>
          </TouchableOpacity>
        </View>
        <SyncModal visible={syncModalVisible} onClose={() => setSyncModalVisible(false)} />
        {/* Update the success modal to ensure it is always centered, overlays the screen, and cannot be dismissed by tapping outside. */}
        <Modal
          visible={showSuccessModal}
          transparent
          animationType="fade"
          onRequestClose={() => setShowSuccessModal(false)}
        >
          <View style={styles.modalOverlay}> 
            <View style={styles.successModalContent}>
              <Text style={styles.successTextStyle}>Report Generated Successfully!</Text>
              <Text style={styles.successSubText}>Report saved to MongoDB successfully.</Text>
              <View style={{ flexDirection: 'row', justifyContent: 'space-around', marginTop: 20 }}>
                <TouchableOpacity
                  style={styles.popupButton}
                  onPress={() => {
                    setShowSuccessModal(false);
                    navigation.replace('DailyIns_report_home');
                  }}
                >
                  <Text style={styles.popupButtonText}>Go Back</Text>
                </TouchableOpacity>
                {reportId && (
                  <TouchableOpacity style={styles.popupButton} onPress={handleViewPdf}>
                    <Text style={styles.popupButtonText}>View Report</Text>
                  </TouchableOpacity>
                )}
              </View>
            </View>
          </View>
        </Modal>
        <Toast />
      </KeyboardAvoidingView>
    </SafeAreaView>
  );
};

const styles = StyleSheet.create({
  container: { flex: 1, paddingHorizontal: 20, paddingVertical: 20, backgroundColor: '#fff' },
  sectionTitle: { fontSize: 18, fontWeight: '600', marginBottom: 10, color: '#333' },

  downtimeNoteBox: { backgroundColor: '#46607A', borderRadius: 8, marginBottom: 20, overflow: 'hidden' },
  downtimeNoteHeader: { flexDirection: 'row', alignItems: 'center', justifyContent: 'space-between', padding: 10, backgroundColor: '#46607A' },
  downtimeNoteTitle: { color: '#fff', fontSize: 18, fontStyle: 'italic', fontWeight: 'bold' },
  downtimeNoteInput: { backgroundColor: '#fff', minHeight: 100, padding: 10, fontSize: 16, borderBottomLeftRadius: 8, borderBottomRightRadius: 8 },
  datePickerButton: { flex: 1, flexDirection: 'row', alignItems: 'center', justifyContent: 'space-between', borderWidth: 1, borderColor: '#ccc', borderRadius: 5, padding: 10, backgroundColor: '#fff' },
  datePickerText: { fontSize: 16, color: '#333' },
  bottomBarCustom: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    padding: 10,
    backgroundColor: '#fff', // white background
    borderTopWidth: 1,
    borderTopColor: '#eee', // lighter border
  },
  backBtnCustom: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 10,
    paddingHorizontal: 16,
    borderRadius: 4,
  },
  backTextCustom: {
    color: '#015185',
    fontWeight: 'bold',
    fontSize: 16,
    marginLeft: 5,
  },
  generateBtnCustom: {
    // backgroundColor: '#D32F2F',
    borderRadius: 4,
    paddingVertical: 10,
    paddingHorizontal: 24,
    alignItems: 'center',
    justifyContent: 'center',
  },
  generateTextCustom: {
    color: '#015185',
    fontWeight: 'bold',
    fontSize: 16,
  },
  errorText: { color: '#FF0000', fontSize: 12, marginTop: 5 },
  modalOverlay: { 
    flex: 1, 
    backgroundColor: 'rgba(0, 0, 0, 0.5)', 
    justifyContent: 'center', 
    alignItems: 'center',
    position: 'absolute',
    top: 0,
    left: 0,
    right: 0,
    bottom: 0,
  },
  successModalContent: { backgroundColor: '#fff', borderRadius: 10, padding: 20, width: '80%' },
  successTextStyle: { fontSize: 18, fontWeight: '600', color: '#015185', marginBottom: 15, textAlign: 'center' },
  successSubText: { fontSize: 14, color: '#333', textAlign: 'center' },
  popupButton: { backgroundColor: '#0078D4', paddingVertical: 10, paddingHorizontal: 20, borderRadius: 5 },
  popupButtonText: { color: '#fff', fontSize: 14, fontWeight: 'bold' },
  syncModalContainer: { flex: 1, backgroundColor: 'rgba(0,0,0,0.5)', justifyContent: 'center', alignItems: 'center' },
  syncModalContent: { width: '70%', backgroundColor: 'white', borderRadius: width * 0.025, padding: width * 0.05, alignItems: 'center', elevation: 5, shadowColor: '#000', shadowOffset: { width: 0, height: 2 }, shadowOpacity: 0.25, shadowRadius: 4 },
  syncModalText: { fontSize: width * 0.045, marginBottom: width * 0.025, color: '#333', fontWeight: '600' },
  syncIcon: { width: width * 0.05, height: width * 0.05, marginRight: width * 0.02 },
  syncModalTextadd: { fontSize: width * 0.035, color: '#333', fontWeight: '500' },
  syncIcondiv: { flexDirection: 'row', alignItems: 'center', justifyContent: 'center', width: '100%', marginBottom: width * 0.025 },
});

export default DailyInsSummary;