import React, { useMemo, useRef, useState, useEffect, useCallback } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  Dimensions,
  Platform,
  StatusBar,
  Modal,
  Animated,
  Image,
  Linking,
  Share,
  PermissionsAndroid,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import Pdf from 'react-native-pdf';
import WebView from 'react-native-webview';
import Toast from 'react-native-toast-message';
import RNFS from 'react-native-fs';
import ReactNativeBlobUtil from 'react-native-blob-util';

const { width, height } = Dimensions.get('window');

const getExtension = (nameOrUrl = '') => {
  const match = (nameOrUrl || '').match(/\.([0-9a-z]+)(?:[\?#]|$)/i);
  return match ? (match[1] || '').toLowerCase() : '';
};

// Helper function to get file extension from MIME type
const getExtensionFromMimeType = (mimeType = '') => {
  if (!mimeType) return '';
  
  const mimeToExt = {
    'image/jpeg': 'jpg',
    'image/jpg': 'jpg',
    'image/png': 'png',
    'image/gif': 'gif',
    'image/bmp': 'bmp',
    'image/webp': 'webp',
    'application/pdf': 'pdf',
    'application/msword': 'doc',
    'application/vnd.openxmlformats-officedocument.wordprocessingml.document': 'docx',
    'application/vnd.ms-excel': 'xls',
    'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet': 'xlsx',
    'text/csv': 'csv',
    'text/plain': 'txt',
    'application/rtf': 'rtf',
    'application/vnd.ms-powerpoint': 'ppt',
    'application/vnd.openxmlformats-officedocument.presentationml.presentation': 'pptx',
  };
  
  return mimeToExt[mimeType.toLowerCase()] || '';
};



  const ViewDocuments = ({ route, navigation }) => {
    const { url, title, name, mimeType, fileType } = route.params || {};
    const [loading, setLoading] = useState(false);
    const [localPdfPath, setLocalPdfPath] = useState(null);
    const [connectionError, setConnectionError] = useState(null);
    const [isOnline, setIsOnline] = useState(true);
    const rotateAnim = useRef(new Animated.Value(0)).current;

    // Convert old /documents/ URLs to /image/ URLs
    const correctedUrl = url && url.includes('/documents/') ? url.replace('/documents/', '/image/') : url;

    // Debug logging
    console.log('📄 ViewDocuments - Route params:', { url, title, name, mimeType, fileType });
    console.log('📄 ViewDocuments - Corrected URL:', correctedUrl);
    console.log('📄 ViewDocuments - File extension:', getExtension(name || correctedUrl || ''));

    // Cleanup function for iOS navigation
    const cleanup = useCallback(() => {
      setLoading(false);
      setConnectionError(null);
      if (rotateAnim) {
        rotateAnim.setValue(0);
      }
      // Clean up local PDF path
      if (localPdfPath) {
        RNFS.unlink(localPdfPath).catch(() => {});
      }
    }, [localPdfPath, rotateAnim]);

  const fileTitle = title || name || 'Document';
  const extension = useMemo(() => {
    // Try to get extension from name first, then from URL, then from MIME type
    const nameExt = getExtension(name || '');
    const urlExt = getExtension(correctedUrl || '');
    const mimeExt = getExtensionFromMimeType(mimeType);
    const directFileType = fileType; // Direct file type from backend
    
    console.log('📄 Extension detection:', {
      nameExt,
      urlExt,
      mimeExt,
      directFileType,
      mimeType,
      correctedUrl
    });
    
    return nameExt || urlExt || mimeExt || directFileType || '';
  }, [name, correctedUrl, mimeType, fileType]);
  
  const isPdf = extension === 'pdf';
  const isDocx = extension === 'docx' || extension === 'doc';
  const isExcel = extension === 'xls' || extension === 'xlsx' || extension === 'csv';
  const isImage = ['png', 'jpg', 'jpeg', 'gif', 'bmp', 'webp'].includes(extension) || 
                  (mimeType && mimeType.startsWith('image/'));

  // Debug logging for file type detection
  console.log('📄 File type detection:', {
    name,
    url,
    extension,
    mimeType,
    fileType,
    isPdf,
    isDocx,
    isExcel,
    isImage
  });

  // Download function
  const handleDownload = async () => {
    if (!correctedUrl) {
      Toast.show({ type: 'error', text1: 'Error', text2: 'No file URL found.' });
      return;
    }

    try {
      setLoading(true);
      
      // Generate proper filename with correct extension
      let filename = name;
      if (!filename || !filename.includes('.')) {
        // If no filename or no extension, create one based on detected type
        const timestamp = Date.now();
        if (isPdf) {
          filename = `document_${timestamp}.pdf`;
        } else if (isDocx) {
          filename = `document_${timestamp}.docx`;
        } else if (isExcel) {
          filename = `document_${timestamp}.xlsx`;
        } else if (isImage) {
          filename = `image_${timestamp}.${extension}`;
        } else {
          filename = `document_${timestamp}.${extension || 'dat'}`;
        }
      } else {
        // Ensure filename has correct extension
        const currentExt = getExtension(filename);
        if (currentExt !== extension && extension) {
          // Replace extension if it doesn't match detected type
          const nameWithoutExt = filename.replace(/\.[^/.]+$/, '');
          filename = `${nameWithoutExt}.${extension}`;
        }
      }
      
      // For Android, use ReactNativeBlobUtil for better download experience
      if (Platform.OS === 'android') {
        // Android 13+ doesn't require permissions for Downloads folder
        if (Platform.Version < 33) {
          const granted = await PermissionsAndroid.request(
            PermissionsAndroid.PERMISSIONS.WRITE_EXTERNAL_STORAGE
          );
          if (granted !== PermissionsAndroid.RESULTS.GRANTED) {
            Toast.show({ type: 'error', text1: 'Permission denied', text2: 'Storage permission required for download' });
            return;
          }
        }

        const dest = `${ReactNativeBlobUtil.fs.dirs.DownloadDir}/${filename}`;
        
        const config = {
          fileCache: true,
          path: dest,
          addAndroidDownloads: {
            useDownloadManager: true,
            notification: true,
            title: filename,
            description: 'Downloading file...',
            mime: getMimeType(extension),
            mediaScannable: true,
          },
        };
        
        await ReactNativeBlobUtil.config(config).fetch('GET', correctedUrl);
        Toast.show({ type: 'success', text1: 'Downloaded', text2: `${filename} saved to Downloads folder` });
      } else {
        // For iOS, save to app documents and share
        const dest = `${RNFS.DocumentDirectoryPath}/${filename}`;
        
        const res = await RNFS.downloadFile({ fromUrl: correctedUrl, toFile: dest }).promise;
        if (res.statusCode >= 200 && res.statusCode < 300) {
          // Fix: Check if res.path exists and handle it properly
          const filePath = res.path || res.filePath || dest;
          const fileUrl = Platform.OS === 'ios' ? `file://${filePath}` : filePath;
          
          try {
            await Share.share({ url: fileUrl, message: filename });
            Toast.show({ type: 'success', text1: 'Downloaded', text2: `${filename} saved and shared` });
          } catch (shareError) {
            console.log('Share error:', shareError);
            // If sharing fails, just show success message
            Toast.show({ type: 'success', text1: 'Downloaded', text2: `${filename} saved successfully` });
          }
        } else {
          Toast.show({ type: 'error', text1: 'Download failed', text2: `Status ${res.statusCode}` });
        }
      }
    } catch (error) {
      console.error('Download error:', error);
      console.log('Error details:', {
        message: error.message,
        stack: error.stack,
        url: correctedUrl,
        platform: Platform.OS,
        extension: extension,
        filename: name
      });
      
      let errorMessage = 'Download failed';
      if (error.message) {
        if (error.message.includes('permission')) {
          errorMessage = 'Storage permission denied';
        } else if (error.message.includes('network')) {
          errorMessage = 'Network error occurred';
        } else if (error.message.includes('timeout')) {
          errorMessage = 'Download timeout';
        } else {
          errorMessage = error.message;
        }
      }
      
      Toast.show({ type: 'error', text1: 'Download failed', text2: errorMessage });
    } finally {
      setLoading(false);
    }
  };

  // Helper function to get MIME type
  const getMimeType = (ext) => {
    if (!ext) return 'application/octet-stream';
    
    const extLower = ext.toLowerCase();
    switch (extLower) {
      case 'pdf': return 'application/pdf';
      case 'png': return 'image/png';
      case 'jpg': case 'jpeg': return 'image/jpeg';
      case 'gif': return 'image/gif';
      case 'bmp': return 'image/bmp';
      case 'webp': return 'image/webp';
      case 'doc': return 'application/msword';
      case 'docx': return 'application/vnd.openxmlformats-officedocument.wordprocessingml.document';
      case 'xls': return 'application/vnd.ms-excel';
      case 'xlsx': return 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet';
      case 'csv': return 'text/csv';
      case 'txt': return 'text/plain';
      case 'rtf': return 'application/rtf';
      case 'ppt': return 'application/vnd.ms-powerpoint';
      case 'pptx': return 'application/vnd.openxmlformats-officedocument.presentationml.presentation';
      default: return 'application/octet-stream';
    }
  };

  // Check connection and validate URL with improved retry mechanism
  useEffect(() => {
    let isMounted = true;
    let timeoutIds = [];
    
    const checkConnection = async (retryCount = 0) => {
      if (!isMounted) return;
      
      try {
        if (!correctedUrl) {
          if (isMounted) setConnectionError('No file URL provided');
          return;
        }

        if (isMounted) {
          setLoading(true);
          setConnectionError(null); // Clear any previous errors
        }

        // For images, try a different approach - test with a small range request
        if (isImage) {
          await tryLoadImageContent(retryCount);
          return;
        }

        // For PDFs, try a different approach - download to local first for better reliability
        if (isPdf) {
          await tryLoadPdfContent(retryCount);
          return;
        }

        // For other files, try HEAD request first
        try {
          const controller = new AbortController();
          const timeoutId = setTimeout(() => controller.abort(), 8000);
          timeoutIds.push(timeoutId);

          const response = await fetch(correctedUrl, { 
            method: 'HEAD',
            signal: controller.signal,
            mode: 'cors'
          });
          
          if (isMounted) {
            clearTimeout(timeoutId);
            timeoutIds = timeoutIds.filter(id => id !== timeoutId);
          }
          
          if (response.ok) {
            if (isMounted) {
              setConnectionError(null);
              setLoading(false);
            }
            return;
          } else if (retryCount < 1 && isMounted) {
            // Auto-retry once without showing error
            const retryTimeoutId = setTimeout(() => checkConnection(retryCount + 1), 1000);
            timeoutIds.push(retryTimeoutId);
            return;
          } else if (isMounted) {
            // Second attempt failed, try content loading
            await tryLoadFileContent(retryCount);
            return;
          }
        } catch (fetchError) {
          if (fetchError.name === 'AbortError') {
            if (retryCount < 1 && isMounted) {
              // Auto-retry once without showing error
              const retryTimeoutId = setTimeout(() => checkConnection(retryCount + 1), 1000);
              timeoutIds.push(retryTimeoutId);
              return;
            } else if (isMounted) {
              // Second attempt failed, try content loading
              await tryLoadFileContent(retryCount);
              return;
            }
          } else {
            throw fetchError;
          }
        }
      } catch (error) {
        if (retryCount < 1 && isMounted) {
          // Auto-retry once without showing error
          const retryTimeoutId = setTimeout(() => checkConnection(retryCount + 1), 1000);
          timeoutIds.push(retryTimeoutId);
          return;
        } else if (isMounted) {
          // Second attempt failed, try content loading
          await tryLoadFileContent(retryCount);
        }
      }
    };

    // Special function for images - test with a small range request
    const tryLoadImageContent = async (retryCount = 0) => {
      try {
        console.log('📄 Testing image URL:', correctedUrl);
        const controller = new AbortController();
        const timeoutId = setTimeout(() => controller.abort(), 15000); // 15 seconds for images

        const response = await fetch(correctedUrl, { 
          method: 'GET',
          signal: controller.signal,
          mode: 'cors',
          headers: {
            'Range': 'bytes=0-1023' // Only download first 1KB to check accessibility
          }
        });
        
        clearTimeout(timeoutId);
        
        if (response.ok || response.status === 206) { // 206 = Partial Content
          console.log('📄 Image URL is accessible');
          setConnectionError(null);
          setLoading(false);
          return;
        } else if (retryCount < 1) {
          // Auto-retry once without showing error
          console.log('📄 Image URL test failed, retrying...');
          setTimeout(() => tryLoadImageContent(retryCount + 1), 1000);
          return;
        } else {
          // Second attempt failed, show error
          console.log('📄 Image URL not accessible:', response.status);
          setConnectionError(`Image not accessible (${response.status})`);
        }
      } catch (error) {
        if (retryCount < 1) {
          // Auto-retry once without showing error
          console.log('📄 Image URL test error, retrying...', error.message);
          setTimeout(() => tryLoadImageContent(retryCount + 1), 1000);
          return;
        }
        
        // Second attempt failed, show appropriate error
        console.log('📄 Image URL test failed:', error.message);
        if (error.name === 'AbortError') {
          setConnectionError('Image loading timeout. Please try again.');
        } else if (error.message.includes('Network request failed') || error.message.includes('fetch')) {
          setConnectionError('Network connection issue. Please check your internet.');
          setIsOnline(false);
        } else {
          setConnectionError('Failed to connect to image. Please check your network.');
        }
      } finally {
        setLoading(false);
      }
    };

    // Special function for PDFs - download to local for better reliability
    const tryLoadPdfContent = async (retryCount = 0) => {
      try {
        // For PDFs, we'll try to download a small portion first to verify accessibility
        const controller = new AbortController();
        const timeoutId = setTimeout(() => controller.abort(), 25000); // 25 seconds for PDF

        const response = await fetch(correctedUrl, { 
          method: 'GET',
          signal: controller.signal,
          mode: 'cors',
          headers: {
            'Range': 'bytes=0-1024' // Only download first 1KB to check accessibility
          }
        });
        
        clearTimeout(timeoutId);
        
        if (response.ok || response.status === 206) { // 206 = Partial Content
          setConnectionError(null);
          setLoading(false);
          return;
        } else if (retryCount < 1) {
          // Auto-retry once without showing error
          setTimeout(() => tryLoadPdfContent(retryCount + 1), 1000);
          return;
        } else {
          // Second attempt failed, show error
          setConnectionError(`PDF not accessible (${response.status})`);
        }
      } catch (error) {
        if (retryCount < 1) {
          // Auto-retry once without showing error
          setTimeout(() => tryLoadPdfContent(retryCount + 1), 1000);
          return;
        }
        
        // Second attempt failed, show appropriate error
        if (error.name === 'AbortError') {
          setConnectionError('Connection timeout. Please try again.');
        } else if (error.message.includes('Network request failed') || error.message.includes('fetch')) {
          setConnectionError('Network connection issue. Please check your internet.');
          setIsOnline(false);
        } else {
          setConnectionError('Failed to connect to PDF. Please check your network.');
        }
      } finally {
        setLoading(false);
      }
    };

    // Fallback function to try loading actual file content for non-PDF files
    const tryLoadFileContent = async (retryCount = 0) => {
      try {
        const controller = new AbortController();
        const timeoutId = setTimeout(() => controller.abort(), 20000);

        const response = await fetch(correctedUrl, { 
          method: 'GET',
          signal: controller.signal,
          mode: 'cors'
        });
        
        clearTimeout(timeoutId);
        
        if (response.ok) {
          setConnectionError(null);
        } else if (retryCount < 1) {
          // Auto-retry content loading once without showing error
          setTimeout(() => tryLoadFileContent(retryCount + 1), 1000);
          return;
        } else {
          // Second attempt failed, show error
          setConnectionError(`File not accessible (${response.status})`);
        }
      } catch (error) {
        if (retryCount < 1) {
          // Auto-retry content loading once without showing error
          setTimeout(() => tryLoadFileContent(retryCount + 1), 1000);
          return;
        }
        
        // Second attempt failed, show appropriate error
        if (error.name === 'AbortError') {
          setConnectionError('Connection timeout. Please try again.');
        } else if (error.message.includes('Network request failed') || error.message.includes('fetch')) {
          setConnectionError('Network connection issue. Please check your internet.');
          setIsOnline(false);
        } else {
          setConnectionError('Failed to connect to file. Please check your network.');
        }
      } finally {
        setLoading(false);
      }
    };

    // Add a small delay before first check to ensure page is fully loaded
    const initialDelay = setTimeout(() => {
      checkConnection();
    }, 500);
    timeoutIds.push(initialDelay);

    return () => {
      isMounted = false;
      // Clear all timeouts to prevent memory leaks
      timeoutIds.forEach(id => clearTimeout(id));
    };
  }, [correctedUrl, isPdf]);

  useEffect(() => {
    let animationLoop = null;
    
    if (loading) {
      animationLoop = Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      );
      animationLoop.start();
    } else {
      rotateAnim.setValue(0);
    }
    
    return () => {
      if (animationLoop) {
        animationLoop.stop();
      }
    };
  }, [loading]);

  // iOS Navigation Focus Effect - Prevents crashes when navigating back
  useEffect(() => {
    if (Platform.OS === 'ios') {
      const unsubscribe = navigation.addListener('focus', () => {
        // Reset states when screen comes into focus
        cleanup();
      });

      return unsubscribe;
    }
  }, [navigation, cleanup]);

  // Component unmount cleanup for iOS
  useEffect(() => {
    return () => {
      cleanup();
    };
  }, [cleanup]);

  useEffect(() => {
    // If PDF over http(s), prefetch to temp for smoother viewing
    const shouldPrefetchPdf = isPdf && correctedUrl && (correctedUrl.startsWith('http://') || correctedUrl.startsWith('https://'));
    if (!shouldPrefetchPdf) return;
    
    let isMounted = true;
    let tempPath = null;
    
    (async () => {
      if (!isMounted) return;
      setLoading(true);
      try {
        tempPath = `${RNFS.DocumentDirectoryPath}/temp_${Date.now()}.pdf`;
        const res = await RNFS.downloadFile({ fromUrl: correctedUrl, toFile: tempPath }).promise;
        if (isMounted && res.statusCode >= 200 && res.statusCode < 300) {
          setLocalPdfPath(tempPath);
        } else if (isMounted) {
          Toast.show({ type: 'error', text1: 'Error', text2: 'Failed to load PDF.' });
        }
      } catch (e) {
        if (isMounted) {
          Toast.show({ type: 'error', text1: 'Error', text2: 'Failed to load PDF.' });
        }
      } finally {
        if (isMounted) {
          setLoading(false);
        }
      }
    })();
    
    return () => {
      isMounted = false;
      // Clean up temp file on unmount
      if (tempPath) {
        RNFS.unlink(tempPath).catch(() => {});
      }
      if (localPdfPath) {
        RNFS.unlink(localPdfPath).catch(() => {});
      }
    };
  }, [isPdf, correctedUrl]);



  const renderViewer = () => {
    // Show connection error if any
    if (connectionError) {
      return (
        <View style={styles.errorContainer}>
          <MaterialIcons 
            name={isOnline ? "cloud-off" : "wifi-off"} 
            size={width * 0.2} 
            color="#e74c3c" 
          />
          <Text style={styles.errorTitle}>Connection Error</Text>
          <Text style={styles.errorMessage}>{connectionError}</Text>
          <Text style={styles.errorSubtitle}>The app will automatically retry connecting to the file.</Text>
          <TouchableOpacity 
            style={styles.retryButton}
            onPress={() => {
              setConnectionError(null);
              setIsOnline(true);
              setLoading(true);
              // Re-check connection with improved retry mechanism
              const checkConnection = async (retryCount = 0) => {
                try {
                  setLoading(true);
                  setConnectionError(null);

                  // First try HEAD request
                  try {
                    const controller = new AbortController();
                    const timeoutId = setTimeout(() => controller.abort(), 8000);

                    const response = await fetch(url, { 
                      method: 'HEAD',
                      signal: controller.signal,
                      mode: 'cors'
                    });
                    
                    clearTimeout(timeoutId);
                    
                    if (response.ok) {
                      setConnectionError(null);
                      setLoading(false);
                      return;
                    } else if (retryCount < 1) {
                      // Auto-retry once without showing error
                      setTimeout(() => checkConnection(retryCount + 1), 1000);
                      return;
                    } else {
                      // Second attempt failed, try content loading
                      await tryLoadFileContent(retryCount);
                      return;
                    }
                  } catch (fetchError) {
                    if (fetchError.name === 'AbortError') {
                      if (retryCount < 1) {
                        // Auto-retry once without showing error
                        setTimeout(() => checkConnection(retryCount + 1), 1000);
                        return;
                      }
                      // Second attempt failed, try content loading
                      await tryLoadFileContent(retryCount);
                      return;
                    } else {
                      throw fetchError;
                    }
                  }
                } catch (error) {
                  if (retryCount < 1) {
                    // Auto-retry once without showing error
                    setTimeout(() => checkConnection(retryCount + 1), 1000);
                    return;
                  }
                  // Second attempt failed, try content loading
                  await tryLoadFileContent(retryCount);
                }
              };

              // Special function for PDFs in retry
              const tryLoadPdfContent = async (retryCount = 0) => {
                try {
                  const controller = new AbortController();
                  const timeoutId = setTimeout(() => controller.abort(), 25000);

                  const response = await fetch(url, { 
                    method: 'GET',
                    signal: controller.signal,
                    mode: 'cors',
                    headers: {
                      'Range': 'bytes=0-1024'
                    }
                  });
                  
                  clearTimeout(timeoutId);
                  
                  if (response.ok || response.status === 206) {
                    setConnectionError(null);
                    setLoading(false);
                    return;
                  } else if (retryCount < 1) {
                    setTimeout(() => tryLoadPdfContent(retryCount + 1), 1000);
                    return;
                  } else {
                    setConnectionError(`PDF not accessible (${response.status})`);
                  }
                } catch (error) {
                  if (retryCount < 1) {
                    setTimeout(() => tryLoadPdfContent(retryCount + 1), 1000);
                    return;
                  }
                  
                  if (error.name === 'AbortError') {
                    setConnectionError('Connection timeout. Please try again.');
                  } else if (error.message.includes('Network request failed') || error.message.includes('fetch')) {
                    setConnectionError('Network connection issue. Please check your internet.');
                    setIsOnline(false);
                  } else {
                    setConnectionError('Failed to connect to PDF. Please check your network.');
                  }
                } finally {
                  setLoading(false);
                }
              };

              // Fallback function to try loading actual file content for non-PDF files
              const tryLoadFileContent = async (retryCount = 0) => {
                try {
                  const controller = new AbortController();
                  const timeoutId = setTimeout(() => controller.abort(), 20000);

                  const response = await fetch(url, { 
                    method: 'GET',
                    signal: controller.signal,
                    mode: 'cors'
                  });
                  
                  clearTimeout(timeoutId);
                  
                  if (response.ok) {
                    setConnectionError(null);
                  } else if (retryCount < 1) {
                    // Auto-retry content loading once without showing error
                    setTimeout(() => tryLoadFileContent(retryCount + 1), 1000);
                    return;
                  } else {
                    // Second attempt failed, show error
                    setConnectionError(`File not accessible (${response.status})`);
                  }
                } catch (error) {
                  if (retryCount < 1) {
                    // Auto-retry content loading once without showing error
                    setTimeout(() => tryLoadFileContent(retryCount + 1), 1000);
                    return;
                  }
                  
                  // Second attempt failed, show appropriate error
                  if (error.name === 'AbortError') {
                    setConnectionError('Connection timeout. Please try again.');
                  } else if (error.message.includes('Network request failed') || error.message.includes('fetch')) {
                    setConnectionError('Network connection issue. Please check your internet.');
                    setIsOnline(false);
                  } else {
                    setConnectionError('Failed to connect to file. Please check your network.');
                  }
                } finally {
                  setLoading(false);
                }
              };
              
              // Start the connection check
              checkConnection();
            }}
          >
            <Text style={styles.retryButtonText}>Retry</Text>
          </TouchableOpacity>
        </View>
      );
    }

    if (isImage) {
      console.log('📄 Rendering image:', { url, correctedUrl, isImage, extension, mimeType });
      
      // Test if URL is accessible before rendering
      if (connectionError && connectionError.includes('Failed to load image')) {
        return (
          <View style={styles.errorContainer}>
            <MaterialIcons 
              name="broken-image" 
              size={width * 0.2} 
              color="#e74c3c" 
            />
            <Text style={styles.errorTitle}>Image Not Available</Text>
            <Text style={styles.errorMessage}>The image could not be loaded. This might be due to:</Text>
            <Text style={styles.errorSubtitle}>
              • Network connection issues{'\n'}
              • File not found on server{'\n'}
              • Invalid image URL{'\n'}
              • Unsupported image format
            </Text>
            <TouchableOpacity 
              style={styles.retryButton}
              onPress={() => {
                console.log('📄 Retrying image load for URL:', correctedUrl);
                setConnectionError(null);
                setLoading(true);
                // Force re-render by updating state
                setLoading(false);
              }}
            >
              <Text style={styles.retryButtonText}>Retry</Text>
            </TouchableOpacity>
            <TouchableOpacity 
              style={[styles.retryButton, { backgroundColor: '#3498db', marginTop: 10 }]}
              onPress={async () => {
                console.log('📄 Testing image URL directly:', correctedUrl);
                try {
                  const response = await fetch(correctedUrl, { method: 'HEAD' });
                  console.log('📄 URL test response:', response.status, response.statusText);
                  if (response.ok) {
                    Toast.show({ type: 'success', text1: 'URL Test', text2: 'Image URL is accessible!' });
                  } else {
                    Toast.show({ type: 'error', text1: 'URL Test', text2: `Server returned ${response.status}` });
                  }
                } catch (error) {
                  console.log('📄 URL test error:', error);
                  Toast.show({ type: 'error', text1: 'URL Test', text2: error.message });
                }
              }}
            >
              <Text style={styles.retryButtonText}>Test URL</Text>
            </TouchableOpacity>
          </View>
        );
      }
      
      return (
        <View style={styles.imageContainer}>
          {loading && (
            <View style={styles.imageLoadingContainer}>
              <Animated.Image
                source={require('../../../Images/adminlogin/sync.png')}
                style={[styles.syncIcon, { transform: [{ rotate: rotateAnim.interpolate({ inputRange: [0, 1], outputRange: ['0deg', '360deg'] }) }] }]}
              />
              <Text style={styles.imageLoadingText}>Loading image...</Text>
            </View>
          )}
          <Image
            source={{ 
              uri: correctedUrl,
              cache: 'force-cache' // Force cache for better performance
            }}
            style={styles.imageViewer}
            onError={(error) => {
              console.log('📄 Image load error:', error);
              console.log('📄 Failed URL:', correctedUrl);
              setConnectionError('Failed to load image. Please check your connection.');
              setLoading(false);
            }}
            onLoad={() => {
              console.log('📄 Image loaded successfully');
              setConnectionError(null);
              setLoading(false);
            }}
            onLoadStart={() => {
              console.log('📄 Image load started for URL:', correctedUrl);
              setLoading(true);
            }}
            resizeMode="contain"
          />
        </View>
      );
    }
    if (isPdf) {
      const pdfSource = localPdfPath
        ? { uri: Platform.OS === 'android' ? `file://${localPdfPath}` : localPdfPath }
        : { uri: url, cache: false };
      return (
        <Pdf
          source={pdfSource}
          style={styles.pdf}
          onLoadComplete={() => {
            setConnectionError(null);
            setLoading(false);
          }}
          onError={(error) => {
            console.log('PDF Error:', error);
            // Only show error if we're not in loading state
            if (!loading) {
              setConnectionError('Failed to display PDF. Please check if the file is accessible.');
            }
          }}
          onPageChanged={(page, numberOfPages) => {
            // PDF is working, clear any connection errors
            setConnectionError(null);
          }}
          // iOS specific props to prevent crashes
          {...(Platform.OS === 'ios' && {
            enablePaging: true,
            enableRTL: false,
            enableAnnotationRendering: false,
          })}
        />
      );
    }
    if (isDocx) {
      const gviewUrl = `https://docs.google.com/gview?embedded=1&url=${encodeURIComponent(url)}`;
      return (
        <WebView 
          source={{ uri: gviewUrl }} 
          style={styles.webview}
          onError={() => setConnectionError('Failed to load document. Please check your connection.')}
          // iOS specific props to prevent crashes
          {...(Platform.OS === 'ios' && {
            allowsBackForwardNavigationGestures: false,
            allowsLinkPreview: false,
            incognito: true,
          })}
        />
      );
    }
    if (isExcel) {
      // For Excel files, show a message and button to open in browser
      return (
        <View style={styles.excelContainer}>
          <View style={styles.excelIconContainer}>
            <MaterialIcons name="table-chart" size={width * 0.2} color="#3481BC" />
          </View>
          <Text style={styles.excelTitle}>Excel File</Text>
          <Text style={styles.excelSubtitle}>This file format cannot be displayed directly in the app.</Text>
          <TouchableOpacity 
            style={styles.openInBrowserButton}
            onPress={() => Linking.openURL(url)}
          >
            <MaterialIcons name="open-in-browser" size={width * 0.05} color="#fff" />
            <Text style={styles.openInBrowserText}>Open in Browser</Text>
          </TouchableOpacity>
        </View>
      );
    }
    return (
      <WebView 
        source={{ uri: url }} 
        style={styles.webview}
        onError={() => setConnectionError('Failed to load file. Please check your connection.')}
        // iOS specific props to prevent crashes
        {...(Platform.OS === 'ios' && {
          allowsBackForwardNavigationGestures: false,
          allowsLinkPreview: false,
          incognito: true,
        })}
      />
    );
  };

  return (
    <View style={styles.container}>
      <LinearGradient colors={['#3481BC', '#025383']} style={styles.header}>
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity style={styles.backButton} onPress={() => navigation.goBack()}>
              <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle} numberOfLines={1}>{fileTitle}</Text>
          </View>
          <View style={styles.headerRight}>
            <TouchableOpacity style={styles.headerIconButton} onPress={handleDownload}>
              <MaterialIcons name="download" size={width * 0.055} color="#fff" />
            </TouchableOpacity>
          </View>
        </View>
      </LinearGradient>

      {renderViewer()}

      <Modal visible={loading} transparent animationType="fade">
        <View style={styles.syncModalContainer}>
          <View style={styles.syncModalContent}>
            <Text style={styles.syncModalText}>Please Wait</Text>
            <View style={styles.syncIcondiv}>
              <Animated.Image
                source={require('../../../Images/adminlogin/sync.png')}
                style={[styles.syncIcon, { transform: [{ rotate: rotateAnim.interpolate({ inputRange: [0, 1], outputRange: ['0deg', '360deg'] }) }] }]}
              />
              <Text style={styles.syncModalTextadd}>Loading...</Text>
            </View>
          </View>
        </View>
      </Modal>
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
    elevation: 3,
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.1,
    paddingBottom: width * 0.025,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
    flex: 1,
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  backButton: {
    padding: width * 0.012,
    borderRadius: 20,
    backgroundColor: 'rgba(255, 255, 255, 0.15)',
    borderWidth: 1,
    borderColor: 'rgba(255, 255, 255, 0.2)',
  },
  headerIconButton: {
    padding: width * 0.012,
    borderRadius: 20,
    backgroundColor: 'rgba(255, 255, 255, 0.15)',
    borderWidth: 1,
    borderColor: 'rgba(255, 255, 255, 0.2)',
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.045,
    fontWeight: '500',
    marginLeft: width * 0.025,
    flexShrink: 1,
  },
  pdf: {
    flex: 1,
    width,
    height: height - (width * 0.35 + (StatusBar.currentHeight || width * 0.1)),
  },
  webview: {
    flex: 1,
    width,
    height: height - (width * 0.35 + (StatusBar.currentHeight || width * 0.1)),
  },
  imageContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: '#f8f9fa',
    padding: 10,
  },
  imageViewer: {
    width: '100%',
    height: '100%',
    maxHeight: height - (width * 0.35 + (StatusBar.currentHeight || width * 0.1)),
  },
  imageLoadingContainer: {
    position: 'absolute',
    top: 0,
    left: 0,
    right: 0,
    bottom: 0,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: 'rgba(248, 249, 250, 0.9)',
    zIndex: 1,
  },
  imageLoadingText: {
    marginTop: 10,
    fontSize: 16,
    color: '#666',
  },


  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 22,
    height: 22,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  errorContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    paddingHorizontal: width * 0.05,
    backgroundColor: '#f8f9fa',
  },
  errorTitle: {
    fontSize: width * 0.065,
    fontWeight: 'bold',
    color: '#e74c3c',
    marginTop: width * 0.03,
    marginBottom: width * 0.025,
    textAlign: 'center',
  },
  errorMessage: {
    fontSize: width * 0.042,
    color: '#7f8c8d',
    textAlign: 'center',
    marginBottom: width * 0.03,
    lineHeight: width * 0.06,
    paddingHorizontal: width * 0.03,
  },
  errorSubtitle: {
    fontSize: width * 0.035,
    color: '#95a5a6',
    textAlign: 'center',
    marginBottom: width * 0.06,
    lineHeight: width * 0.05,
    paddingHorizontal: width * 0.05,
    fontStyle: 'italic',
  },
  retryButton: {
    backgroundColor: '#e74c3c',
    paddingVertical: width * 0.03,
    paddingHorizontal: width * 0.06,
    borderRadius: width * 0.025,
    shadowColor: '#000',
    shadowOffset: {
      width: 0,
      height: 2,
    },
    shadowOpacity: 0.25,
    shadowRadius: 3.84,
    elevation: 5,
    minWidth: width * 0.3,
  },
  retryButtonText: {
    color: '#fff',
    fontSize: width * 0.045,
    fontWeight: '600',
    textAlign: 'center',
  },
  excelContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    paddingHorizontal: width * 0.05,
  },
  excelTitle: {
    fontSize: width * 0.06,
    fontWeight: 'bold',
    color: '#333',
    marginTop: width * 0.03,
    marginBottom: width * 0.02,
  },
  excelSubtitle: {
    fontSize: width * 0.04,
    color: '#666',
    textAlign: 'center',
    marginBottom: width * 0.05,
    lineHeight: width * 0.06,
  },
  openInBrowserButton: {
    backgroundColor: '#3481BC',
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.05,
    borderRadius: width * 0.012,
  },
  openInBrowserText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: 'bold',
    marginLeft: width * 0.012,
  },
});

export default ViewDocuments;


