import React, { useState, useEffect, useRef, useCallback } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  TextInput,
  ScrollView,
  Dimensions,
  Image,
  Platform,
  SafeAreaView,
  Animated,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import AsyncStorage from '@react-native-async-storage/async-storage';
import CheckBox from '@react-native-community/checkbox';
import Toast from 'react-native-toast-message';
// Firebase imports removed - using MongoDB only
import { SafeAreaProvider, useSafeAreaInsets } from 'react-native-safe-area-context';
import NetInfo from '@react-native-community/netinfo';
import Modal from 'react-native-modal';
import RNHTMLtoPDF from 'react-native-html-to-pdf';
import RNFS from 'react-native-fs';
import debounce from 'lodash.debounce';

const { width, height } = Dimensions.get('window');

// MongoDB DVIR API URL
const MONGODB_DVIR_API_URL = 'https://api.titandrillingzm.com:6006';

const DvirCorrectiveActionStep2 = ({
  report,
  setActiveStep,
  navigation,
  subItemStatuses,
  setSubItemStatuses,
  resetSubItemStatuses,
}) => {
  const [sessionState, setSessionState] = useState({
    isSatisfactory: null,
    role: null,
    mechanicName: '',
    operatorName: '',
  });
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [mechanicSignatureUri, setMechanicSignatureUri] = useState(null);
  const [operatorSignatureUri, setOperatorSignatureUri] = useState(null);
  const [signatureRefreshKey, setSignatureRefreshKey] = useState(0);
  const [savePdf, setSavePdf] = useState(true);
  const [errors, setErrors] = useState({
    isSatisfactory: '',
    role: '',
    mechanicName: '',
    operatorName: '',
  });
  const [isOnline, setIsOnline] = useState(true);
  const [showSuccessModal, setShowSuccessModal] = useState(false);
  const [combinedPdfDownloadUrl, setCombinedPdfDownloadUrl] = useState(null);
  const [isLoading, setIsLoading] = useState(false);
  const isMounted = useRef(true);
  const insets = useSafeAreaInsets();

  useEffect(() => {
    isMounted.current = true;
    const loadStoredData = async () => {
      setIsLoading(true);
      if (!report?.id) {
        console.warn('No report ID provided for loading data');
        Toast.show({ type: 'error', text1: 'Error', text2: 'Missing report information.' });
        return;
      }
      try {
        const userUid = await AsyncStorage.getItem('userUid');
        if (!userUid) {
          Toast.show({ type: 'error', text1: 'Authentication Error', text2: 'Please log in again.' });
          navigation.replace('LoginScreens');
          return;
        }
        
        console.log('Loading corrective action data for report:', report.id);
        
        // Try MongoDB first
        try {
          console.log('🔍 Loading corrective action data from MongoDB...');
          const response = await fetch(`${MONGODB_DVIR_API_URL}/dvir-corrective-actions/report/${report.id}?userUid=${userUid}`, {
            method: 'GET',
            headers: {
              'Content-Type': 'application/json',
            },
          });
          
          if (response.ok) {
            const result = await response.json();
            if (result.success && result.data) {
              const correctiveData = result.data;
              console.log('✅ Corrective action data loaded from MongoDB:', correctiveData);
              
              console.log('Setting session state with MongoDB data:', {
                isSatisfactory: correctiveData.isSatisfactory,
                role: correctiveData.role,
                mechanicName: correctiveData.mechanicName,
                operatorName: correctiveData.operatorName,
                subItemStatusesCount: Object.keys(correctiveData.subItemStatuses || {}).length
              });
              
              setSessionState({
                isSatisfactory: correctiveData.isSatisfactory !== undefined ? correctiveData.isSatisfactory : null,
                role: correctiveData.role || null,
                mechanicName: correctiveData.mechanicName || '',
                operatorName: correctiveData.operatorName || '',
              });
              setMechanicSignatureUri(correctiveData.mechanicSignatureUri || null);
              setOperatorSignatureUri(correctiveData.operatorSignatureUri || null);
              
              // Only update subItemStatuses if we have data and it's different from current
              if (correctiveData.subItemStatuses && Object.keys(correctiveData.subItemStatuses).length > 0) {
                console.log('Updating subItemStatuses with MongoDB data:', correctiveData.subItemStatuses);
                setSubItemStatuses(correctiveData.subItemStatuses);
              }
              
              setSignatureRefreshKey((prev) => prev + 1);
              setCombinedPdfDownloadUrl(correctiveData.pdfDownloadUrl || null);
              return; // Success, exit early
            }
          } else {
            console.log('⚠️ MongoDB corrective action fetch failed with status:', response.status);
          }
        } catch (mongoError) {
          console.log('❌ MongoDB corrective action fetch failed:', mongoError.message);
        }

        // No fallback - MongoDB only
        console.log('No corrective action data found in MongoDB, keeping existing state');
      } catch (error) {
        console.error('Error loading corrective action data:', error);
        setIsOnline(false);
        Toast.show({ type: 'error', text1: 'Error', text2: 'Failed to load stored data. Working offline.' });
      } finally {
        setIsLoading(false);
      }
    };
    loadStoredData();

    const unsubscribe = NetInfo.addEventListener((state) => {
      setIsOnline(state.isConnected && state.isInternetReachable !== false);
    });

    return () => {
      isMounted.current = false;
      unsubscribe();
    };
  }, [report?.id, report?.userUid, setSubItemStatuses, navigation]);

  const isValidSignatureUri = useCallback((uri) => uri && typeof uri === 'string' && uri.startsWith('data:image'), []);

  // Handle session state changes with logging
  const handleSessionStateChange = useCallback((updates) => {
    console.log('Session state changing from:', sessionState, 'to:', { ...sessionState, ...updates });
    setSessionState(prev => ({ ...prev, ...updates }));
  }, [sessionState]);

  const validateFields = useCallback(() => {
    const newErrors = {
      isSatisfactory: sessionState.isSatisfactory === null ? 'Please select if the corrective action is satisfactory' : '',
      role: sessionState.role === null ? 'Please select your role' : '',
      mechanicName: (sessionState.role === 'Mechanic' || sessionState.role === 'Both') && !sessionState.mechanicName.trim() ? 'Mechanic name is required' : '',
      operatorName: (sessionState.role === 'Operator' || sessionState.role === 'Both') && !sessionState.operatorName.trim() ? 'Operator name is required' : '',
    };
    if ((sessionState.role === 'Mechanic' || sessionState.role === 'Both') && !isValidSignatureUri(mechanicSignatureUri)) {
      newErrors.mechanicName = 'Mechanic signature is required';
    }
    if ((sessionState.role === 'Operator' || sessionState.role === 'Both') && !isValidSignatureUri(operatorSignatureUri)) {
      newErrors.operatorName = 'Operator signature is required';
    }
    setErrors(newErrors);
    return Object.values(newErrors).every((error) => error === '');
  }, [sessionState, mechanicSignatureUri, operatorSignatureUri]);

  const debouncedValidate = useCallback(debounce(validateFields, 300), [validateFields]);

  useEffect(() => {
    debouncedValidate();
    return () => debouncedValidate.cancel();
  }, [sessionState, mechanicSignatureUri, operatorSignatureUri, debouncedValidate]);

  // Note: Auto-save removed - using MongoDB only with manual save on Done button

  const generateCombinedPDFContent = () => {
    console.log('🔍 Generating PDF content for report:', report);
    console.log('🔍 Report step2 data:', report.step2);
    console.log('🔍 Report step2 items:', report.step2?.items);
    console.log('🔍 Report step2 itemStates:', report.step2?.itemStates);
    console.log('🔍 SubItemStatuses:', subItemStatuses);
    
    // Inspection Items - Use the same structure as main DVIR report
    let inspectionItemsHtml = '';
    let lastSection = null;
    
    if (Array.isArray(report.step2?.items)) {
      // Filter items that have status and are not N/A - show all items that were inspected
      const filteredItems = report.step2.items.filter((item) => item.status && item.status !== '' && item.status !== 'N/A');
      
      inspectionItemsHtml = filteredItems
        .map((item, index) => {
          // Add section header if section changed
          let sectionHeader = '';
          if (item.section && item.section !== lastSection) {
            sectionHeader = `
              <tr style="background-color: #e0e0e0;">
                <td colspan="7"><strong>${item.section}</strong></td>
              </tr>
            `;
            lastSection = item.section;
          }
          
          // Determine status display
          const isYes = item.status && (item.status.toLowerCase() === 'yes' || item.status.toLowerCase() === 'fixed' || item.status.toLowerCase() === 'pass' || item.status.toLowerCase() === 'okay');
          const isNo = item.status && (item.status.toLowerCase() === 'no' || item.status.toLowerCase() === 'not fixed' || item.status.toLowerCase() === 'fail' || item.status.toLowerCase() === 'faulty');
          const isNA = item.status && (item.status.toLowerCase() === 'na' || item.status.toLowerCase() === 'no need');
          
          return sectionHeader + `
            <tr>
              <td>${index + 1}</td>
              <td>${item.label || item.name || item.text || 'N/A'}</td>
              <td>A</td>
              <td style="text-align: center;">${isYes ? '✓' : ''}</td>
              <td style="text-align: center;">${isNo ? 'X' : ''}</td>
              <td style="text-align: center;">${isNA ? '—' : ''}</td>
              <td>${item.note || item.description || ''}</td>
            </tr>
          `;
        })
        .join('');
    } else if (report.step2?.itemStates) {
      // Fallback for old itemStates structure
      const filteredKeys = Object.keys(report.step2.itemStates).filter((key) => {
        const status = report.step2.itemStates[key].status;
        return status && status !== '' && status !== 'N/A';
      });
      inspectionItemsHtml = filteredKeys
        .map((key, index) => {
          const [sectionKey, itemId] = key.split('_');
          const itemName = report.step2.itemStates[key].name || report.step2.itemStates[key].text || `${sectionKey}_${itemId}`;
          const status = report.step2.itemStates[key].status;
          const isYes = status && (status.toLowerCase() === 'yes' || status.toLowerCase() === 'fixed' || status.toLowerCase() === 'pass');
          const isNo = status && (status.toLowerCase() === 'no' || status.toLowerCase() === 'not fixed' || status.toLowerCase() === 'fail' || status.toLowerCase() === 'faulty');
          const isNA = status && (status.toLowerCase() === 'na' || status.toLowerCase() === 'no need');
          
          return `
            <tr>
              <td>${index + 1}</td>
              <td>${itemName}</td>
              <td>A</td>
              <td style="text-align: center;">${isYes ? '✓' : ''}</td>
              <td style="text-align: center;">${isNo ? 'X' : ''}</td>
              <td style="text-align: center;">${isNA ? '—' : ''}</td>
              <td>${report.step2.itemStates[key].note || ''}</td>
            </tr>
          `;
        })
        .join('');
    }
    
    // If no items found, show a message
    if (!inspectionItemsHtml) {
      inspectionItemsHtml = `
        <tr>
          <td colspan="7">No inspection items available</td>
        </tr>
      `;
    }

    // Corrective Action Items - Show only faulty items that need corrective action
    let correctiveItemsHtml = '';
    if (Array.isArray(report.step2?.items)) {
      // Filter only items that were marked as faulty/No/Fail
      const faultyItems = report.step2.items.filter((item) => {
        const status = item.status && item.status.toLowerCase();
        return status === 'no' || status === 'fail' || status === 'faulty' || status === 'not fixed';
      });
      
      correctiveItemsHtml = faultyItems
        .map((item, index) => `
            <tr>
              <td>${index + 1}</td>
              <td>${item.label || item.name || item.text || 'N/A'}</td>
              <td>${item.section || item.category || 'General'}</td>
              <td>${subItemStatuses[item.itemId || item.id] || 'Pending'}</td>
            </tr>
          `
        )
        .join('');
    } else if (report.step2?.itemStates) {
      // Fallback for old itemStates structure
      const faultyKeys = Object.keys(report.step2.itemStates).filter((key) => {
        const status = report.step2.itemStates[key].status;
        const statusLower = status && status.toLowerCase();
        return statusLower === 'no' || statusLower === 'fail' || statusLower === 'faulty' || statusLower === 'not fixed';
      });
      correctiveItemsHtml = faultyKeys
        .map((key, index) => {
          const [sectionKey, itemId] = key.split('_');
          const itemName = report.step2.itemStates[key].name || report.step2.itemStates[key].text || `${sectionKey}_${itemId}`;
          return `
            <tr>
              <td>${index + 1}</td>
              <td>${itemName}</td>
              <td>${sectionKey || report.step2.itemStates[key].category || 'General'}</td>
              <td>${subItemStatuses[key] || 'Pending'}</td>
            </tr>
          `;
        })
        .join('');
    }
    
    // If no corrective action items found, show a message
    if (!correctiveItemsHtml) {
      correctiveItemsHtml = `
        <tr>
          <td colspan="4">No corrective action items required</td>
        </tr>
      `;
    }

    return `
      <!DOCTYPE html>
      <html lang="en">
      <head>
        <meta charset="UTF-8"/>
        <meta name="viewport" content="width=device-width, initial-scale=1.0"/>
        <title>DVIR Corrective Action Report</title>
        <style>
          body {
            font-family: Arial, sans-serif;
            margin: 20px;
            font-size: 12px;
          }
          .header-row {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            margin-bottom: 10px;
          }
          .logo-left {
            display: flex;
            align-items: flex-start;
          }
          .logo-left img {
            width: 140px;
            height: 50px;
            object-fit: contain;
          }
          .doc-info-box {
            width: 340px;
          }
          .doc-info-box table {
            width: 100%;
            border: 1px solid #222;
            border-collapse: collapse;
            font-size: 12px;
          }
          .doc-info-box td {
            border: 1px solid #222;
            padding: 4px 8px;
            text-align: left;
          }
          .title-section {
            text-align: center;
            margin: 10px 0 0 0;
          }
          .title-section h2 {
            margin: 0;
            font-size: 18px;
            font-weight: bold;
          }
          .request-number {
            font-size: 14px;
            font-weight: bold;
            margin-top: 4px;
          }
          .info-boxes-row {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            margin: 16px 0 10px 0;
          }
          .info-box {
            width: 45%;
          }
          .info-box table {
            width: 100%;
            border: 1px solid #222;
            border-collapse: collapse;
            font-size: 12px;
          }
          .info-box td {
            border: 1px solid #222;
            padding: 4px 8px;
            text-align: left;
          }
          .section-break {
            margin-top: 20px;
            border-top: 2px solid #015185;
            padding-top: 10px;
          }
          table.checklist, table.corrective-table {
            width: 100%;
            border-collapse: collapse;
            margin-bottom: 20px;
          }
          table.checklist th, table.checklist td, table.corrective-table th, table.corrective-table td {
            border: 1px solid #222;
            padding: 8px;
            text-align: center;
          }
          table.checklist th, table.corrective-table th {
            background-color: #f2f2f2;
            font-weight: bold;
          }
          .summary-section-table {
            width: 100%;
            margin-top: 30px;
            border: none;
          }
          .summary-section-table td {
            border: none;
            padding: 8px;
            font-size: 14px;
            text-align: left;
          }
          .summary-section-table .label {
            font-weight: bold;
            width: 180px;
          }
          .summary-section-table .signature-cell {
            text-align: right;
            vertical-align: middle;
            padding-right: 30px;
          }
          .signature img {
            max-width: 200px;
            height: auto;
          }
        </style>
      </head>
      <body>
        <div class="header-row">
          <div class="logo-left">
            <img src="https://www.titanafricadrilling.com/wp-content/uploads/2022/10/titan-drlling_blue.png" alt="Logo"/>
          </div>
          <div class="doc-info-box">
            <table>
              <tr><td>Document Name</td><td>DVIR Corrective Action Report</td></tr>
              <tr><td>Report ID</td><td>CA-${report.step1?.requestNumber || 'N/A'}</td></tr>
              <tr><td>Date</td><td>${new Date().toLocaleDateString()}</td></tr>
              <tr><td>Country</td><td>${report.step1?.country || 'N/A'}</td></tr>
              <tr><td>Project</td><td>${report.step1?.project || 'N/A'}</td></tr>
            </table>
          </div>
        </div>
        <div class="title-section">
          <h2>DVIR CORRECTIVE ACTION REPORT</h2>
          <div class="request-number">NO: CA-${report.step1?.requestNumber || 'N/A'}</div>
        </div>
        <div class="info-boxes-row">
          <div class="info-box">
            <table>
              <tr><td>DATE</td><td>${new Date().toLocaleDateString()}</td></tr>
              <tr><td>SITE / PROJECT</td><td>${report.step1?.project || 'N/A'}</td></tr>
            </table>
          </div>
          <div class="info-box">
            <table>
              <tr><td>VEHICLE No</td><td>${report.step1?.selectedEquipment?.equipmentName || 'N/A'}</td></tr>
              <tr><td>HOURS</td><td>${report.step1?.meterReading || 'N/A'} ${report.step1?.selectedEquipment?.meterUnit || 'Hours'}</td></tr>
            </table>
          </div>
        </div>
        <div class="section-break"></div>
        <h2>Inspection Items</h2>
        <table class="checklist">
          <tr>
            <th>S.No</th>
            <th>Item</th>
            <th>Class</th>
            <th>Okay</th>
            <th>Faulty</th>
            <th>N/A</th>
            <th>Comment</th>
          </tr>
          ${inspectionItemsHtml || '<tr><td colspan="7">No inspection items available</td></tr>'}
        </table>
        <div class="section-break"></div>
        <h2>Corrective Action Items</h2>
        <div>
          <p><strong>Equipment Condition:</strong> ${sessionState.isSatisfactory ? 'Satisfactory' : 'Unsatisfactory'}</p>
          <p><strong>Role:</strong> ${sessionState.role || 'N/A'}</p>
          <p><strong>Mechanic Name:</strong> ${sessionState.mechanicName || 'N/A'}</p>
          <p><strong>Operator Name:</strong> ${sessionState.operatorName || 'N/A'}</p>
        </div>
        <table class="corrective-table">
          <tr>
            <th>S.No</th>
            <th>Item</th>
            <th>Category</th>
            <th>Status</th>
          </tr>
          ${correctiveItemsHtml || '<tr><td colspan="4">No corrective action items available</td></tr>'}
        </table>
        <table class="summary-section-table">
          <tr>
            <td class="label">Mechanic Signature</td>
            <td>
              ${(sessionState.role === 'Mechanic' || sessionState.role === 'Both') && isValidSignatureUri(mechanicSignatureUri)
                ? `<img src="${mechanicSignatureUri}" alt="Mechanic Signature" style="width: 120px; height: 60px;"/>`
                : 'Not Provided'}
            </td>
            <td class="label">Operator Signature</td>
            <td>
              ${(sessionState.role === 'Operator' || sessionState.role === 'Both') && isValidSignatureUri(operatorSignatureUri)
                ? `<img src="${operatorSignatureUri}" alt="Operator Signature" style="width: 120px; height: 60px;"/>`
                : 'Not Provided'}
            </td>
          </tr>
        </table>
      </body>
      </html>
    `;
  };

  const generateAndSavePDF = async () => {
    if (!validateFields()) {
      Toast.show({ type: 'error', text1: 'Incomplete Data', text2: 'Please fill all required fields.' });
      return null;
    }
    setIsLoading(true);
    let timeoutId;
    try {
      console.log('📄 Starting PDF generation...');
      const combinedHtml = generateCombinedPDFContent();
      const options = {
        html: combinedHtml,
        fileName: `DVIRCorrectiveAction_${report.step1?.requestNumber || 'Unknown'}_${Date.now()}`,
        directory: Platform.OS === 'android' ? 'Documents' : 'DocumentDirectory',
      };
      
      console.log('📄 Converting HTML to PDF...');
      const { filePath } = await RNHTMLtoPDF.convert(options);
      console.log('✅ DVIR PDF generated successfully at:', filePath);

      // Check if file exists
      const fileExists = await RNFS.exists(filePath);
      if (!fileExists) {
        throw new Error('PDF file was not created successfully');
      }

      const fileStats = await RNFS.stat(filePath);
      console.log('📄 PDF file stats:', {
        size: fileStats.size,
        path: filePath
      });

      const userId = await AsyncStorage.getItem('userUid');
      if (!userId) {
        throw new Error('User ID not found');
      }

      console.log('📤 Preparing to upload PDF to backend...');
      
      // Create FormData with correct structure for React Native
      const formData = new FormData();
      
      // Add PDF file - use the correct structure for React Native
      formData.append('pdf', {
        uri: Platform.OS === 'ios' ? filePath : `file://${filePath}`,
        type: 'application/pdf',
        name: `corrective-action_${userId}_${report.id}_${Date.now()}.pdf`,
      });
      
      formData.append('reportId', report.id);
      formData.append('userUid', userId);
      
      console.log('📤 FormData prepared:', {
        reportId: report.id,
        userUid: userId,
        fileUri: Platform.OS === 'ios' ? filePath : `file://${filePath}`,
        fileName: `corrective-action_${userId}_${report.id}_${Date.now()}.pdf`
      });
      
      // Create AbortController for timeout
      const controller = new AbortController();
      timeoutId = setTimeout(() => {
        console.log('⏰ Upload timeout reached, aborting...');
        controller.abort();
      }, 30000); // 30 second timeout
      
      console.log('📤 Uploading PDF to MongoDB backend...');
      const uploadResponse = await fetch(`${MONGODB_DVIR_API_URL}/dvir-corrective-actions/upload-pdf`, {
        method: 'POST',
        body: formData,
        headers: {
          'Content-Type': 'multipart/form-data',
        },
        signal: controller.signal,
      });
      
      clearTimeout(timeoutId);
      
      console.log('📥 Upload response received:', {
        status: uploadResponse.status,
        statusText: uploadResponse.statusText,
        ok: uploadResponse.ok
      });
      
      if (!uploadResponse.ok) {
        const errorText = await uploadResponse.text();
        console.error('❌ Upload failed with response:', errorText);
        throw new Error(`PDF upload failed: ${uploadResponse.status} - ${errorText}`);
      }
      
      const uploadResult = await uploadResponse.json();
      console.log('✅ Upload successful:', uploadResult);
      
      if (!uploadResult.success) {
        throw new Error(uploadResult.error || 'Upload failed');
      }
      
      const downloadUrl = uploadResult.downloadUrl;
      console.log('✅ PDF upload completed, download URL:', downloadUrl);
      
      // Clean up local PDF file
      await RNFS.unlink(filePath).catch((err) => console.warn('Failed to clean up local PDF:', err));

      return downloadUrl;
    } catch (error) {
      // Clear timeout if it exists
      if (typeof timeoutId !== 'undefined') {
        clearTimeout(timeoutId);
      }
      
      console.error('❌ Error generating or uploading PDF:', error);
      console.error('❌ Error details:', {
        name: error.name,
        message: error.message,
        code: error.code,
        stack: error.stack
      });
      
      let errorMessage = 'Failed to generate or upload PDF.';
      if (error.name === 'AbortError') {
        errorMessage = 'Upload timeout. Please check your connection and try again.';
      } else if (error.message.includes('Network request failed')) {
        errorMessage = 'Network error. Please check your internet connection and try again.';
      } else if (error.message.includes('PDF upload failed')) {
        errorMessage = 'Failed to upload PDF to server. Please try again.';
      } else if (error.message.includes('User ID not found')) {
        errorMessage = 'Authentication error. Please log in again.';
      } else if (error.message.includes('PDF file was not created')) {
        errorMessage = 'PDF generation failed. Please try again.';
      }
      
      Toast.show({ 
        type: 'error', 
        text1: 'PDF Generation Error', 
        text2: errorMessage 
      });
      return null;
    } finally {
      setIsLoading(false);
    }
  };

  const queueOperation = async (operation) => {
    try {
      let pendingOperations = [];
      const pendingOperationsRaw = await AsyncStorage.getItem('pendingOperations');
      pendingOperations = pendingOperationsRaw ? JSON.parse(pendingOperationsRaw) : [];
      pendingOperations.push(operation);
      await AsyncStorage.setItem('pendingOperations', JSON.stringify(pendingOperations));
      Toast.show({
        type: 'info',
        text1: 'Offline Mode',
        text2: 'Corrective action data queued for sync when online.',
      });
    } catch (error) {
      console.error('Error queuing operation:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to queue corrective action data.',
      });
    }
  };

  const saveToMongoDB = async (pdfUrl) => {
    try {
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid || !report?.id) return false;

      const correctiveActionData = {
        reportId: report.id,
        userUid: userUid,
        dvirNumber: report.dvirNumber || report.step1?.requestNumber,
        linkedDvirReportId: report.id,
        subItemStatuses,
        isSatisfactory: sessionState.isSatisfactory,
        role: sessionState.role,
        mechanicName: (sessionState.role === 'Mechanic' || sessionState.role === 'Both') ? sessionState.mechanicName : null,
        operatorName: (sessionState.role === 'Operator' || sessionState.role === 'Both') ? sessionState.operatorName : null,
        mechanicSignatureUri,
        operatorSignatureUri,
        pdfDownloadUrl: pdfUrl,
        createdAt: new Date().toISOString(),
        updatedAt: new Date().toISOString(),
        faultyCount: Object.keys(subItemStatuses).length,
        status: 'completed',
        // Add tracking fields
        originalReportData: {
          step1: report.step1,
          step2: report.step2,
          step3: report.step3,
        }
      };

      console.log('🔄 Saving corrective action to MongoDB...', correctiveActionData);

      // Try MongoDB first
      try {
        const response = await fetch(`${MONGODB_DVIR_API_URL}/dvir-corrective-actions`, {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
          },
          body: JSON.stringify(correctiveActionData),
        });

        if (response.ok) {
          const result = await response.json();
          console.log('✅ Corrective action saved to MongoDB successfully:', result);
        } else {
          throw new Error(`MongoDB API error: ${response.status}`);
        }
      } catch (mongoError) {
        console.error('❌ MongoDB save failed:', mongoError.message);
        throw new Error(`Failed to save corrective action to MongoDB: ${mongoError.message}`);
      }

      // Update local storage
      let storedReports = [];
      const storedData = await AsyncStorage.getItem('inspectionReports');
      storedReports = storedData ? JSON.parse(storedData) : [];
      
      const updatedReports = storedReports.map((r) =>
        r.id === report.id
          ? { 
              ...r, 
              correctiveAction: { 
                ...r.correctiveAction, 
                hasCorrectiveAction: true,
                correctiveActionId: report.id,
                correctiveActionNumber: `CA-${report.id}`,
                pdfDownloadUrl: pdfUrl,
                status: 'completed',
                updatedAt: new Date().toISOString(),
                faultyCount: Object.keys(subItemStatuses).length,
                isSatisfactory: sessionState.isSatisfactory,
                role: sessionState.role,
                mechanicName: (sessionState.role === 'Mechanic' || sessionState.role === 'Both') ? sessionState.mechanicName : null,
                operatorName: (sessionState.role === 'Operator' || sessionState.role === 'Both') ? sessionState.operatorName : null,
              } 
            }
          : r
      );
      
      if (!storedReports.some((r) => r.id === report.id)) {
        updatedReports.push({ 
          ...report, 
          correctiveAction: {
            hasCorrectiveAction: true,
            correctiveActionId: report.id,
            correctiveActionNumber: `CA-${report.id}`,
            pdfDownloadUrl: pdfUrl,
            status: 'completed',
            updatedAt: new Date().toISOString(),
            faultyCount: Object.keys(subItemStatuses).length,
            isSatisfactory: sessionState.isSatisfactory,
            role: sessionState.role,
            mechanicName: (sessionState.role === 'Mechanic' || sessionState.role === 'Both') ? sessionState.mechanicName : null,
            operatorName: (sessionState.role === 'Operator' || sessionState.role === 'Both') ? sessionState.operatorName : null,
          }
        });
      }
      
      await AsyncStorage.setItem('inspectionReports', JSON.stringify(updatedReports));

      // Clear any pending operations since we're using MongoDB only
      await AsyncStorage.setItem('pendingOperations', JSON.stringify([]));

      return true;
    } catch (error) {
      console.error('Error in saveToFirebase:', error);
      setIsOnline(false);
      await queueOperation({
        type: 'correctiveAction',
        reportId: report.id,
        userUid: await AsyncStorage.getItem('userUid') || 'unknown', // Fixed: use userUid instead of userId
        correctiveAction: { 
          reportId: report.id,
          userUid: await AsyncStorage.getItem('userUid'),
          dvirNumber: report.dvirNumber || report.step1?.requestNumber,
          subItemStatuses, 
          pdfDownloadUrl: pdfUrl,
          createdAt: new Date().toISOString(),
          updatedAt: new Date().toISOString(),
          faultyCount: Object.keys(subItemStatuses).length,
          status: 'completed'
        }
      });
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to save corrective action data. Saved offline.',
      });
      return false;
    }
  };

  const handleDone = async () => {
    if (isSubmitting || !isMounted.current) {
      console.log('❌ Cannot submit: isSubmitting=', isSubmitting, 'isMounted=', isMounted.current);
      return;
    }
    
    if (!validateFields()) {
      console.log('❌ Validation failed');
      Toast.show({ type: 'error', text1: 'Validation Error', text2: 'Please fill all required fields.' });
      return;
    }
    
    console.log('🚀 Starting corrective action submission...');
    console.log('📊 Current state:', {
      savePdf,
      isOnline,
      sessionState,
      hasMechanicSignature: !!mechanicSignatureUri,
      hasOperatorSignature: !!operatorSignatureUri,
      subItemStatusesCount: Object.keys(subItemStatuses).length
    });
    
    setIsSubmitting(true);
    setIsLoading(true);
    
    try {
      let pdfUrl = null;
      let pdfGenerated = false;
      
      // Try to generate and upload PDF if savePdf is enabled and online
      if (savePdf && isOnline) {
        console.log('📄 PDF generation enabled - starting PDF generation...');
        try {
          pdfUrl = await generateAndSavePDF();
          if (pdfUrl) {
            pdfGenerated = true;
            setCombinedPdfDownloadUrl(pdfUrl);
            console.log('✅ PDF generated and uploaded successfully:', pdfUrl);
          } else {
            console.log('⚠️ PDF generation returned null - continuing without PDF');
          }
        } catch (pdfError) {
          console.error('❌ PDF generation failed:', pdfError);
          console.log('⚠️ Continuing without PDF due to error');
        }
      } else if (!isOnline) {
        console.log('⚠️ Offline mode - skipping PDF generation');
      } else {
        console.log('📄 PDF generation disabled by user (savePdf = false)');
      }
      
      // Save corrective action data to MongoDB
      console.log('💾 Saving corrective action data to MongoDB...');
      const saveSuccess = await saveToMongoDB(pdfUrl);
      
      if (saveSuccess) {
        console.log('✅ Corrective action saved successfully');
        console.log('📊 Final result:', {
          pdfGenerated,
          pdfUrl: pdfUrl || 'none',
          hasCombinedPdfUrl: !!combinedPdfDownloadUrl
        });
        
        // Show success message
        const successMessage = pdfGenerated 
          ? 'Report saved and PDF uploaded successfully!' 
          : 'Report saved successfully!';
        
        Toast.show({
          type: 'success',
          text1: 'Success',
          text2: successMessage,
        });
        
        // Show success modal with proper buttons
        if (isMounted.current) {
          console.log('📱 Showing success modal with PDF URL:', pdfUrl || 'none');
          setShowSuccessModal(true);
        }
      } else {
        console.error('❌ Failed to save corrective action data');
        Toast.show({ 
          type: 'error', 
          text1: 'Error', 
          text2: 'Failed to save corrective action data. Please try again.' 
        });
      }
    } catch (error) {
      console.error('❌ Error submitting corrective action:', error);
      console.error('❌ Error details:', {
        name: error.name,
        message: error.message,
        stack: error.stack
      });
      Toast.show({ 
        type: 'error', 
        text1: 'Error', 
        text2: 'Failed to submit corrective action. Please try again.' 
      });
    } finally {
      if (isMounted.current) {
        console.log('🏁 Submission process completed');
        setIsSubmitting(false);
        setIsLoading(false);
      }
    }
  };

  const handleMechanicSignature = () => {
    navigation.navigate('SignatureCaptureScreen', {
      role: 'Mechanic',
      onSave: (signature) => {
        setMechanicSignatureUri(signature);
        setSignatureRefreshKey((prev) => prev + 1);
      },
    });
  };

  const handleOperatorSignature = () => {
    navigation.navigate('SignatureCaptureScreen', {
      role: 'Operator',
      onSave: (signature) => {
        setOperatorSignatureUri(signature);
        setSignatureRefreshKey((prev) => prev + 1);
      },
    });
  };

  const handleClearSignature = (type) => {
    if (type === 'Mechanic') {
      setMechanicSignatureUri(null);
      setSessionState((prev) => ({ ...prev, mechanicName: '' }));
      setErrors((prev) => ({ ...prev, mechanicName: '' }));
    } else if (type === 'Operator') {
      setOperatorSignatureUri(null);
      setSessionState((prev) => ({ ...prev, operatorName: '' }));
      setErrors((prev) => ({ ...prev, operatorName: '' }));
    }
    setSignatureRefreshKey((prev) => prev + 1);
  };

  const handleImageError = (type) => {
    console.error(`${type} signature image failed to load`);
    Toast.show({ type: 'error', text1: 'Error', text2: `Failed to load ${type.toLowerCase()} signature. Please try again.` });
    handleClearSignature(type);
  };

  const handleRolePress = useCallback((value) => {
    const newRole = sessionState.role === value ? null : value;
    const updates = {
      role: newRole,
      mechanicName: newRole !== 'Mechanic' && newRole !== 'Both' ? '' : sessionState.mechanicName,
      operatorName: newRole !== 'Operator' && newRole !== 'Both' ? '' : sessionState.operatorName,
    };
    
    handleSessionStateChange(updates);
    setErrors((prev) => ({ ...prev, role: '' }));
    if (value !== 'Mechanic' && value !== 'Both') setMechanicSignatureUri(null);
    if (value !== 'Operator' && value !== 'Both') setOperatorSignatureUri(null);
    setSignatureRefreshKey((prev) => prev + 1);
  }, [sessionState.role, sessionState.mechanicName, sessionState.operatorName, handleSessionStateChange]);

  const handleSatisfactoryPress = useCallback((value) => {
    const newValue = sessionState.isSatisfactory === value ? null : value;
    handleSessionStateChange({ isSatisfactory: newValue });
    setErrors((prev) => ({ ...prev, isSatisfactory: '' }));
  }, [sessionState.isSatisfactory, handleSessionStateChange]);

  const showMechanicSection = sessionState.role === 'Mechanic' || sessionState.role === 'Both';
  const showOperatorSection = sessionState.role === 'Operator' || sessionState.role === 'Both';

  const handleViewPdf = () => {
    console.log('🔍 Attempting to view PDF:', combinedPdfDownloadUrl);
    
    if (!combinedPdfDownloadUrl) {
      console.log('❌ No PDF URL available');
      Toast.show({ 
        type: 'error', 
        text1: 'Error', 
        text2: 'PDF not available for viewing. Please try generating the report again.' 
      });
      return;
    }
    
    console.log('📄 Navigating to PDF viewer with URL:', combinedPdfDownloadUrl);
    
    // Close the success modal first
    setShowSuccessModal(false);
    
    // Navigate to PDF viewer
    navigation.navigate('PdfviewPage', { 
      pdfUrl: combinedPdfDownloadUrl, 
      onGoBack: () => {
        console.log('📄 PDF viewer closed, navigating back to DVIR home');
        navigation.reset({
          index: 1,
          routes: [
            { name: 'MainApp' },
            { name: 'DviranddotInHome' },
          ],
        });
      } 
    });
  };

  const ProcessorModal = ({ visible }) => {
    const rotateAnim = useRef(new Animated.Value(0)).current;

    useEffect(() => {
      if (visible) {
        Animated.loop(
          Animated.timing(rotateAnim, {
            toValue: 1,
            duration: 900,
            useNativeDriver: true,
          })
        ).start();
      } else {
        rotateAnim.setValue(0);
      }
    }, [visible]);

    const rotate = rotateAnim.interpolate({
      inputRange: [0, 1],
      outputRange: ['0deg', '360deg'],
    });

    return (
      <Modal isVisible={visible} style={{ margin: 0 }} backdropOpacity={0.4}>
        <View style={styles.processorModalOuterCenter}>
          <View style={styles.processorModalContainer}>
            <Text style={styles.processorModalTitle}>Please Wait</Text>
            <View style={styles.processorRowCenter}>
              <Animated.Image
                source={require('../../../../Images/adminlogin/sync.png')}
                style={[styles.syncIcon, { transform: [{ rotate }] }]}
              />
              <Text style={styles.processorModalText}>Loading</Text>
            </View>
          </View>
        </View>
      </Modal>
    );
  };

  if (!report || !report.step1 || !report.step2) {
    return (
      <SafeAreaProvider>
        <SafeAreaView style={styles.container} edges={['top', 'left', 'right']}>
          <Text style={styles.noDataText}>No report data available</Text>
        </SafeAreaView>
      </SafeAreaProvider>
    );
  }

  return (
    <SafeAreaProvider>
      <SafeAreaView style={styles.container} edges={['top', 'left', 'right']}>
        <ScrollView
          contentContainerStyle={[styles.contentContainer, { paddingBottom: insets.bottom + 60 }]}
          showsVerticalScrollIndicator={false}
          keyboardShouldPersistTaps="handled"
        >
          <View style={styles.sectionNoBG}>
            <Text style={styles.sectionTitleLargeBlue}>Corrective of vehicle is satisfactory</Text>
            <View style={styles.buttonRowFullWidth}>
              <TouchableOpacity
                style={[
                  styles.fullWidthButton,
                  sessionState.isSatisfactory === true && styles.fullWidthButtonSelected,
                ]}
                onPress={() => handleSatisfactoryPress(true)}
              >
                <Text
                  style={[
                    styles.fullWidthButtonText,
                    sessionState.isSatisfactory === true && styles.fullWidthButtonTextSelected,
                  ]}
                >
                  Yes
                </Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[
                  styles.fullWidthButton,
                  sessionState.isSatisfactory === false && styles.fullWidthButtonSelected,
                ]}
                onPress={() => handleSatisfactoryPress(false)}
              >
                <Text
                  style={[
                    styles.fullWidthButtonText,
                    sessionState.isSatisfactory === false && styles.fullWidthButtonTextSelected,
                  ]}
                >
                  No
                </Text>
              </TouchableOpacity>
            </View>
            {errors.isSatisfactory ? <Text style={styles.errorText}>{errors.isSatisfactory}</Text> : null}
          </View>

          <View style={styles.sectionNoBG}>
            <Text style={styles.sectionTitleLargeBlue}>Are you?</Text>
            <View style={styles.buttonRowFullWidth3}>
              {['Mechanic', 'Operator', 'Both'].map((role) => (
                <TouchableOpacity
                  key={role}
                  style={[styles.fullWidthButton3, sessionState.role === role && styles.fullWidthButtonSelected]}
                  onPress={() => handleRolePress(role)}
                >
                  <Text
                    style={[
                      styles.fullWidthButtonText,
                      sessionState.role === role && styles.fullWidthButtonTextSelected,
                    ]}
                  >
                    {role}
                  </Text>
                </TouchableOpacity>
              ))}
            </View>
            {errors.role ? <Text style={styles.errorText}>{errors.role}</Text> : null}
          </View>

          {showMechanicSection && (
            <View style={styles.sectionNoBG}>
              <Text style={styles.signatureLabelCenteredBlue}>Mechanic Signature</Text>
              <TextInput
                style={styles.signatureInput}
                placeholder="Mechanic Name"
                placeholderTextColor="#888"
                value={sessionState.mechanicName}
                onChangeText={(text) => handleSessionStateChange({ mechanicName: text })}
              />
              {errors.mechanicName ? <Text style={styles.errorText}>{errors.mechanicName}</Text> : null}
              <View style={{ height: 12 }} />
              <View style={styles.signatureBoxImage}>
                {isValidSignatureUri(mechanicSignatureUri) ? (
                  <TouchableOpacity style={styles.plusIconCircleContainer} onPress={handleMechanicSignature}>
                    <Image
                      source={{ uri: mechanicSignatureUri }}
                      style={styles.signatureImage}
                      resizeMode="contain"
                      onError={() => handleImageError('Mechanic')}
                    />
                  </TouchableOpacity>
                ) : (
                  <TouchableOpacity style={styles.plusIconCircleContainer} onPress={handleMechanicSignature}>
                    <View style={styles.plusCircle}>
                      <MaterialIcons name="add" size={width * 0.13} color="#015185" />
                    </View>
                  </TouchableOpacity>
                )}
              </View>
            </View>
          )}

          {showMechanicSection && (
            <View style={styles.acknowledgeRowCenteredColumn}>
              <View style={styles.acknowledgeRowCenteredRow}>
                <MaterialIcons name="check-box" size={22} color="#28a745" style={{ alignSelf: 'center' }} />
                <Text style={styles.acknowledgeTitleCentered}>MECHANIC'S ACKNOWLEDGEMENT</Text>
              </View>
              <Text style={styles.acknowledgeDescCentered}>
                By signing, I acknowledge that I have checked all corrective action taken on vehicle.
              </Text>
            </View>
          )}

          {showOperatorSection && (
            <View style={styles.sectionNoBG}>
              <Text style={styles.signatureLabelCenteredBlue}>Operator Signature</Text>
              <TextInput
                style={styles.signatureInput}
                placeholder="Operator Name"
                placeholderTextColor="#888"
                value={sessionState.operatorName}
                onChangeText={(text) => handleSessionStateChange({ operatorName: text })}
              />
              {errors.operatorName ? <Text style={styles.errorText}>{errors.operatorName}</Text> : null}
              <View style={{ height: 12 }} />
              <View style={styles.signatureBoxImage}>
                {isValidSignatureUri(operatorSignatureUri) ? (
                  <TouchableOpacity style={styles.plusIconCircleContainer} onPress={handleOperatorSignature}>
                    <Image
                      source={{ uri: operatorSignatureUri }}
                      style={styles.signatureImage}
                      resizeMode="contain"
                      onError={() => handleImageError('Operator')}
                    />
                  </TouchableOpacity>
                ) : (
                  <TouchableOpacity style={styles.plusIconCircleContainer} onPress={handleOperatorSignature}>
                    <View style={styles.plusCircle}>
                      <MaterialIcons name="add" size={width * 0.13} color="#015185" />
                    </View>
                  </TouchableOpacity>
                )}
              </View>
            </View>
          )}

          <View style={styles.section}>
            <View style={styles.checkboxRow}>
              <CheckBox value={savePdf} onValueChange={setSavePdf} tintColors={{ true: '#015185', false: '#888' }} />
              <Text style={styles.checkboxLabel}>Save as PDF</Text>
            </View>
          </View>

          <View style={styles.buttonContainer}>
            <TouchableOpacity
              style={styles.cancelButton}
              onPress={() => setActiveStep(1)}
              disabled={isSubmitting}
            >
              <Text style={styles.buttonText}>Back</Text>
            </TouchableOpacity>
            <TouchableOpacity
              style={[styles.submitButton, isSubmitting && styles.disabledButton]}
              onPress={handleDone}
              disabled={isSubmitting}
            >
              <Text style={[styles.buttonText, { color: '#fff' }]}>Done</Text>
            </TouchableOpacity>
          </View>
        </ScrollView>

        <Toast />

        <Modal isVisible={showSuccessModal} style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>Success</Text>
            <MaterialIcons
              name="check"
              size={width * 0.1}
              color="#015185"
              style={{ alignSelf: 'center', marginVertical: width * 0.02 }}
            />
            <Text style={styles.modalSubtitle}>
              {combinedPdfDownloadUrl ? 'Report Generated and Saved' : 'Report Saved Successfully'}
            </Text>
            <View style={styles.modalButtonContainer}>
              <TouchableOpacity
                style={styles.modalButton}
                onPress={() => {
                  setShowSuccessModal(false);
                  navigation.reset({
                    index: 1,
                    routes: [
                      { name: 'MainApp' },
                      { name: 'DviranddotInHome' },
                    ],
                  });
                }}
              >
                <Text style={styles.modalButtonText}>Back</Text>
              </TouchableOpacity>
              {combinedPdfDownloadUrl && (
                <TouchableOpacity 
                  style={styles.modalButton} 
                  onPress={handleViewPdf}
                >
                  <Text style={styles.modalButtonText}>View PDF</Text>
                </TouchableOpacity>
              )}
            </View>
          </View>
        </Modal>

        <ProcessorModal visible={isLoading} />
      </SafeAreaView>
    </SafeAreaProvider>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  contentContainer: {
    paddingHorizontal: width * 0.04,
    paddingTop: width * 0.025,
    paddingBottom: width * 0.1,
  },
  section: {
    marginVertical: width * 0.025,
    padding: width * 0.025,
    backgroundColor: '#f0f0f0',
    borderRadius: width * 0.012,
  },
  sectionNoBG: {
    marginVertical: width * 0.025,
  },
  sectionTitleLargeBlue: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#1676b7',
    marginBottom: width * 0.01,
  },
  buttonRowFullWidth: {
    flexDirection: 'row',
    width: '100%',
    marginBottom: width * 0.01,
    gap: 8,
  },
  fullWidthButton: {
    flex: 1,
    backgroundColor: '#f2f6fa',
    borderRadius: 2,
    alignItems: 'center',
    justifyContent: 'center',
    paddingVertical: width * 0.025,
  },
  fullWidthButtonSelected: {
    backgroundColor: '#1676b7',
  },
  fullWidthButtonText: {
    color: '#222',
    fontWeight: 'bold',
    fontSize: width * 0.04,
  },
  fullWidthButtonTextSelected: {
    color: '#fff',
  },
  buttonRowFullWidth3: {
    flexDirection: 'row',
    width: '100%',
    marginBottom: width * 0.01,
    gap: 8,
  },
  fullWidthButton3: {
    flex: 1,
    backgroundColor: '#f2f6fa',
    borderRadius: 2,
    alignItems: 'center',
    justifyContent: 'center',
    paddingVertical: width * 0.025,
  },
  signatureLabelCenteredBlue: {
    color: '#1676b7',
    fontWeight: 'bold',
    fontSize: width * 0.045,
    textAlign: 'center',
    marginBottom: 4,
    marginTop: width * 0.04,
  },
  signatureInput: {
    backgroundColor: '#f0f0f0',
    borderRadius: 2,
    paddingHorizontal: width * 0.025,
    paddingVertical: width * 0.025,
    fontSize: width * 0.038,
    color: '#333',
    marginBottom: 4,
    borderWidth: 0,
    width: '100%',
    alignSelf: 'center',
  },
  signatureBoxImage: {
    height: height / 5,
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: width * 0.012,
    justifyContent: 'center',
    alignItems: 'center',
    marginVertical: width * 0.025,
  },
  plusIconCircleContainer: {
    justifyContent: 'center',
    alignItems: 'center',
    width: '100%',
    height: '100%',
  },
  plusCircle: {
    width: '100%',
    height: '100%',
    justifyContent: 'center',
    alignItems: 'center',
  },
  signatureImage: {
    width: '100%',
    height: '100%',
    borderRadius: 2,
    backgroundColor: '#f5f5f5',
  },
  acknowledgeRowCenteredColumn: {
    alignItems: 'center',
  },
  acknowledgeRowCenteredRow: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  acknowledgeTitleCentered: {
    fontSize: width * 0.04,
    fontWeight: 'bold',
    color: '#28a745',
    marginLeft: width * 0.02,
  },
  acknowledgeDescCentered: {
    fontSize: width * 0.035,
    color: '#333',
    textAlign: 'center',
    marginTop: width * 0.015,
  },
  checkboxRow: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: width * 0.025,
  },
  checkboxLabel: {
    fontSize: width * 0.04,
    color: '#666',
    marginLeft: width * 0.02,
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.02,
  },
  cancelButton: {
    paddingVertical: width * 0.012,
    paddingHorizontal: width * 0.025,
    borderRadius: width * 0.008,
    borderWidth: 1,
    borderColor: '#015185',
    alignItems: 'center',
    width: '40%',
  },
  submitButton: {
    backgroundColor: '#015185',
    paddingVertical: width * 0.012,
    paddingHorizontal: width * 0.025,
    borderRadius: width * 0.008,
    alignItems: 'center',
    width: '40%',
  },
  disabledButton: {
    backgroundColor: '#ccc',
  },
  buttonText: {
    fontSize: width * 0.032,
    fontWeight: '500',
    color: '#333',
  },
  errorText: {
    color: '#ff4444',
    fontSize: width * 0.035,
    marginLeft: width * 0.015,
    marginBottom: width * 0.015,
  },
  noDataText: {
    fontSize: width * 0.04,
    color: '#666',
    textAlign: 'center',
    marginTop: width * 0.05,
  },
  modalOverlay: {
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
  },
  modalContent: {
    backgroundColor: '#fff',
    borderRadius: width * 0.012,
    padding: width * 0.025,
    width: '80%',
    alignItems: 'center',
  },
  modalTitle: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: width * 0.015,
  },
  modalSubtitle: {
    fontSize: width * 0.035,
    color: '#333',
    textAlign: 'center',
    marginBottom: width * 0.025,
  },
  modalButtonContainer: {
    flexDirection: 'row',
    justifyContent: 'center',
    width: '100%',
    flexWrap: 'nowrap',
    gap: 10,
  },
  modalButton: {
    backgroundColor: '#015185',
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.03,
    borderRadius: width * 0.012,
    alignItems: 'center',
    flex: 1,
    marginHorizontal: 2,
    marginBottom: width * 0.015,
    minWidth: 0,
  },
  modalButtonText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: '500',
  },
  processorModalOuterCenter: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
  },
  processorModalContainer: {
    backgroundColor: '#fff',
    borderRadius: 14,
    padding: 28,
    alignItems: 'center',
    justifyContent: 'center',
    alignSelf: 'center',
    width: width * 0.7,
  },
  processorModalTitle: {
    fontSize: 20,
    fontWeight: '500',
    color: '#222',
    marginBottom: 10,
    textAlign: 'center',
  },
  processorModalText: {
    fontSize: 16,
    color: '#015185',
    marginTop: 6,
    textAlign: 'center',
  },
  syncIcon: {
    width: 24,
    height: 24,
    marginRight: 12,
  },
});

export default DvirCorrectiveActionStep2;