import React, { useRef, useState } from 'react';
import {
  View,
  Text,
  TouchableOpacity,
  StyleSheet,
  Dimensions,
  StatusBar,
  Platform,
} from 'react-native';
import SignatureCanvas from 'react-native-signature-canvas';
import { useNavigation, useRoute } from '@react-navigation/native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import Toast from 'react-native-toast-message';

const { width, height } = Dimensions.get('window');

// Helper function to show toast messages
const showToast = (type, title, message) => {
  Toast.show({
    type: type,
    text1: title,
    text2: message,
    position: 'bottom',
  });
};

const SignatureHRappl = () => {
  const navigation = useNavigation();
  const route = useRoute();
  const { signatureTitle = 'Signature', signatureType, initialSignature } = route.params || {};

  const signatureRef = useRef(null);
  const [isSaving, setIsSaving] = useState(false);
  const [isClearing, setIsClearing] = useState(false);

  // Handle saving the signature
  const handleSave = () => {
    if (isSaving || isClearing) return; // Prevent multiple clicks
    setIsSaving(true);
    signatureRef.current.readSignature();
  };

  // Handle clearing the signature
  const handleClear = () => {
    if (isSaving || isClearing) return; // Prevent multiple clicks
    setIsClearing(true);
    signatureRef.current.clearSignature();
    setTimeout(() => {
      showToast('info', 'Signature Cleared', 'The signature area has been cleared.');
      setIsClearing(false);
    }, 500); // Small delay to ensure clear action completes
  };

  // Callback when signature is read (returns base64 data URL)
  const handleSignature = (signature) => {
    setIsSaving(false);
    if (signature) {
      // Call the callback function passed from the previous screen
      if (route.params?.onSignatureSave) {
        route.params.onSignatureSave(signature);
      }
      // Go back to the previous screen
      navigation.goBack();
      showToast('success', 'Success', 'Signature saved successfully.');
    } else {
      showToast('error', 'Error', 'Failed to capture signature.');
    }
  };

  // Handle empty signature check
  const handleEmpty = () => {
    setIsSaving(false);
    showToast('error', 'Error', 'Signature is empty. Please sign before saving.');
  };

  // Signature canvas style
  const webStyle = `
    .m-signature-pad {
      width: ${width * 0.9}px;
      height: ${height * 0.4}px;
      border: 1px solid #ccc;
      border-radius: ${width * 0.02}px;
      background-color: #fff;
    }
    .m-signature-pad--body {
      width: 100%;
      height: 100%;
    }
    canvas {
      width: 100%;
      height: 100%;
    }
  `;

  return (
    <View style={styles.container}>
      {/* Header Section */}
      <LinearGradient
        colors={['#025383', '#015185']}
        style={styles.header}
        start={{ x: 0, y: 0 }}
        end={{ x: 1, y: 0 }}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <TouchableOpacity
            style={styles.backButton}
            onPress={() => navigation.goBack()}
            accessibilityLabel="Go back"
            accessibilityRole="button"
          >
            <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
          </TouchableOpacity>
          <Text style={styles.headerTitle}>{signatureTitle}</Text>
          <View style={styles.menuButton} />
        </View>
      </LinearGradient>

      {/* Signature Area */}
      <View style={styles.signatureContainer}>
        <Text style={styles.title}>Please Sign Below</Text>
        <SignatureCanvas
          ref={signatureRef}
          onOK={handleSignature}
          onEmpty={handleEmpty}
          descriptionText="Sign here"
          clearText={isClearing ? 'Clearing...' : 'Clear'}
          confirmText={isSaving ? 'Saving...' : 'Save'}
          webStyle={webStyle}
          penColor="#000"
          backgroundColor="#fff"
          dataURL={initialSignature} // Load initial signature if provided
          accessibilityLabel="Signature canvas"
          accessibilityRole="none"
        />
      </View>

      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#f5f5f5',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  backButton: {
    padding: width * 0.02,
  },
  headerTitle: {
    flex: 1,
    fontSize: width * 0.05,
    fontWeight: 'bold',
    color: '#fff',
    textAlign: 'center',
  },
  menuButton: {
    padding: width * 0.02,
  },
  signatureContainer: {
    flex: 1,
    alignItems: 'center',
    paddingTop: height * 0.02,
    paddingHorizontal: width * 0.04,
  },
  title: {
    fontSize: width * 0.045,
    fontWeight: '600',
    color: '#333',
    marginBottom: height * 0.02,
  },
});

export default SignatureHRappl;