import React, { useState, useEffect, useCallback, useRef, useMemo } from 'react';
import {
  View,
  Text,
  Image,
  StyleSheet,
  TouchableOpacity,
  ScrollView,
  Modal,
  Animated,
  Dimensions,
  Alert,
  BackHandler,
  Linking,
  Platform,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation, useFocusEffect } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import NetInfo from '@react-native-community/netinfo';
import Toast from 'react-native-toast-message';
import { MongoAPI, USERS_API } from '../ConfigMongo/MongoDBConfig';
import HomaUploadbtn from './AAdataa/HomaUploadbtn';
import DeviceInfo from 'react-native-device-info';

// SyncModal Component
const SyncModal = ({ visible, onClose, message }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
            />
            <Text style={styles.syncModalTextadd}>{message || 'Processing...'}</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

// Skeleton Component for Loading State
const Skeleton = () => {
  const deviceWidth = Dimensions.get('window').width;
  const boxSize = Math.floor((deviceWidth - 5 * 20) / 4);
  const minBoxSize = Math.max(boxSize, 70);
  const fadeGray = '#f5f6fa';
  const fadeGrayDark = '#e6e6e6';

  return (
    <ScrollView style={{ flex: 1, backgroundColor: '#fff' }}>
              <View style={{ paddingHorizontal: 5, paddingVertical: 10, alignItems: 'center', width: '100%' }}>
        {[
          { title: 'Master Data', items: 3 },
          { title: 'Inspection and Compliances', items: 5 },
          { title: 'Maintenance & Premium Services', items: 3 },
          { title: 'Logbook', items: 2 },
          { title: 'HR', items: 3 },
        ].map((section, index) => {
          const itemsPerRow = 4;
          const rows = Math.ceil(section.items / itemsPerRow);
          const itemsArray = Array.from({ length: section.items });
          let renderedRows = [];
          for (let row = 0; row < rows; row++) {
            const startIdx = row * itemsPerRow;
            const endIdx = startIdx + itemsPerRow;
            const rowItems = itemsArray.slice(startIdx, endIdx);
            renderedRows.push(
            <View
                key={row}
              style={{
                  flexDirection: 'row',
                  justifyContent: 'center',
                  alignItems: 'flex-start',
                  marginBottom: 18,
                  width: '100%',
                }}
              >
                {rowItems.map((_, idx) => (
                                  <View
                    key={idx}
                    style={{
                      alignItems: 'center',
                        marginHorizontal: 16,
                      width: minBoxSize,
                    }}
                  >
                    <View
                      style={{
                          backgroundColor: fadeGray,
                        borderRadius: 12,
                        width: minBoxSize,
                        height: minBoxSize,
                        justifyContent: 'center',
                        alignItems: 'center',
                        marginBottom: 8,
                      }}
                    />
                  <View
                    style={{
                      height: 13,
                      width: '80%',
                        backgroundColor: fadeGrayDark,
                      borderRadius: 4,
                      alignSelf: 'center',
                    }}
                  />
                </View>
                ))}
            </View>
            );
          }
          return (
            <View key={section.title} style={{ width: '100%' }}>
                <View
                  style={{
                  height: 24,
                  width: '60%',
                  backgroundColor: fadeGrayDark,
                      borderRadius: 4,
                      alignSelf: 'center',
                  marginBottom: 13,
                    }}
                  />
              {renderedRows}
              <View style={{ alignItems: 'center', marginVertical: 15, width: '100%' }}>
              <View
                style={{
                    width: 160,
                  height: 40,
                  borderRadius: 5,
                    backgroundColor: fadeGrayDark,
                    alignSelf: 'center',
                }}
              />
            </View>
          </View>
          );
        })}
      </View>
    </ScrollView>
  );
};

const APP_UPDATE_REMIND_INFO_KEY = 'titan.appUpdateRemindInfo';
const PLAY_STORE_URL = 'https://play.google.com/store/apps/details?id=com.titan_drilling&hl=en_IN';
const APP_STORE_URL = 'https://apps.apple.com/us/app/titan-drilling/id6751226650';
const UPDATE_MODAL_HEADER_HEIGHT = Dimensions.get('window').width * 0.25;

const VersionUpdateModal = ({ visible, info, onUpdate, onRemindLater }) => {
  if (!info) return null;

  const { width: windowWidth } = Dimensions.get('window');
  const cardWidth = Math.min(windowWidth - 48, 360);

  const releaseNotesSource = info.releaseNotes;
  let releaseNotes = [];

  if (Array.isArray(releaseNotesSource)) {
    releaseNotes = releaseNotesSource.map((note) =>
      typeof note === 'string' ? note.trim() : ''
    );
  } else if (typeof releaseNotesSource === 'string') {
    releaseNotes = releaseNotesSource
      .split('\n')
      .map((note) => note.trim())
      .filter(Boolean);
  } else if (releaseNotesSource && typeof releaseNotesSource === 'object') {
    releaseNotes = Object.values(releaseNotesSource)
      .flatMap((value) =>
        typeof value === 'string' ? value.split('\n') : Array.isArray(value) ? value : []
      )
      .map((note) => (typeof note === 'string' ? note.trim() : ''))
      .filter(Boolean);
  }

  releaseNotes = releaseNotes.filter(Boolean);

  return (
    <Modal
      visible={visible}
      transparent
      animationType="fade"
      hardwareAccelerated
      onRequestClose={() => {
        if (info.forceUpdate) {
          return;
        }
        onRemindLater?.();
      }}
    >
      <View style={styles.updateModalBackdrop}>
        <View style={[styles.updateModalCard, { width: cardWidth }]}>
          <LinearGradient
            colors={['#3481BC', '#025383']}
            start={{ x: 0, y: 0 }}
            end={{ x: 1, y: 1 }}
            style={[styles.updateModalHeader, { height: UPDATE_MODAL_HEADER_HEIGHT }]}
          >
            <View style={styles.updateModalHeaderContent}>
              <View style={styles.updateModalHeaderIconWrapper}>
                <MaterialIcons name="system-update" size={26} color="#fff" />
              </View>
              <View style={styles.updateModalHeaderTextContainer}>
                <Text style={styles.updateModalTitle}>Update Available</Text>
                <Text style={styles.updateModalSubtitle}>
                  Please update to keep Titan Drilling running smoothly.
                </Text>
              </View>
            </View>
          </LinearGradient>

          <View style={styles.updateModalBody}>
            <View style={styles.updateVersionRow}>
              <Text style={styles.updateVersionLabel}>Current</Text>
              <Text style={styles.updateVersionValue}>
                {info.currentVersion}
                {info.currentBuild ? ` (Build ${info.currentBuild})` : ''}
              </Text>
            </View>
            <View style={styles.updateVersionRow}>
              <Text style={[styles.updateVersionLabel, styles.updateVersionLabelLatest]}>Latest</Text>
              <Text style={[styles.updateVersionValue, styles.updateVersionValueLatest]}>
                {info.latestVersion}
                {info.latestBuild ? ` (Build ${info.latestBuild})` : ''}
              </Text>
            </View>
            {info.forceUpdate && (
              <View style={styles.updateMandatoryBadge}>
                <Text style={styles.updateMandatoryText}>Required update</Text>
              </View>
            )}
            {releaseNotes.length > 0 && (
              <View style={styles.updateNotesContainer}>
                <Text style={styles.updateNotesTitle}>What's new</Text>
                {releaseNotes.map((note, index) => (
                  <View key={`${note}-${index}`} style={styles.updateNotesItem}>
                    <View style={styles.updateNotesBullet} />
                    <Text style={styles.updateNotesText}>{note}</Text>
                  </View>
                ))}
              </View>
            )}
          </View>

          <View style={styles.updateModalActions}>
            <TouchableOpacity
              onPress={onUpdate}
              activeOpacity={0.85}
              style={styles.updatePrimaryButton}
            >
              <LinearGradient
                colors={['#3481BC', '#025383']}
                style={styles.updatePrimaryButtonGradient}
              >
                <Text style={styles.updatePrimaryButtonText}>Update Now</Text>
              </LinearGradient>
            </TouchableOpacity>
            {!info.forceUpdate && (
              <TouchableOpacity
                onPress={onRemindLater}
                activeOpacity={0.85}
                style={styles.updateSecondaryButton}
              >
                <Text style={styles.updateSecondaryButtonText}>Remind Me Later</Text>
              </TouchableOpacity>
            )}
          </View>
        </View>
      </View>
    </Modal>
  );
};

// Get screen dimensions outside component
const { width: screenWidth, height: screenHeight } = Dimensions.get('window');

const HomeScreen = React.memo(() => {
  const width = screenWidth;
  const height = screenHeight;
  
  const navigation = useNavigation();
  const [selectedCountry, setSelectedCountry] = useState(null);
  const [modalVisible, setModalVisible] = useState(false);
  const [userId, setUserId] = useState(null);
  const [userData, setUserData] = useState(null);
  const [loading, setLoading] = useState(true);
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [syncModalMessage, setSyncModalMessage] = useState('Processing...');
  const [dataLoaded, setDataLoaded] = useState(false);
  const [isOnline, setIsOnline] = useState(true);
  const [hasShownOfflineToast, setHasShownOfflineToast] = useState(false);
  const [hasShownOnlineToast, setHasShownOnlineToast] = useState(false);
  const [modules, setModules] = useState([]);
  const [selectedCountries, setSelectedCountries] = useState([]);
  const [lastSyncTime, setLastSyncTime] = useState(null);
  const [refreshing, setRefreshing] = useState(false);
  const [forceOfflineMode, setForceOfflineMode] = useState(false);
  const [pendingChanges, setPendingChanges] = useState(false);
  const [isUploadingEquipment, setIsUploadingEquipment] = useState(false);
  const [isUploadingUsers, setIsUploadingUsers] = useState(false);
  const [updateInfo, setUpdateInfo] = useState(null);
  const [showUpdateModal, setShowUpdateModal] = useState(false);
  const isMounted = useRef(true);
  const hasLoadedOnce = useRef(false);
  const lastNetworkState = useRef(null);
  const isPageFocused = useRef(false);
  const backgroundSyncTimeout = useRef(null);
  const versionCheckAttempted = useRef(false);
  const remindLaterTimeoutRef = useRef(null);

  // Valid country options
  const countryOptions = ['Zambia', 'DRC', 'Both Countries'];
  const memoizedCountryOptions = useMemo(() => countryOptions, []);

  // Module definitions with categories, navigation routes, and icons
  const moduleDefinitions = {
    Equipments: {
      category: 'Master Data',
      label: 'Equipments',
      navigation: 'Equipment_one',
      icon: require('../../Images/adminhome/icosn/eq101.png'),
    },
    Documents: {
      category: 'Master Data',
      label: 'Documents',
      navigation: 'Documenthome',
      icon: require('../../Images/adminhome/icosn/eq103.png'),
    },
    teamEmploy: {
      category: 'Master Data',
      label: 'Team/Employee',
      navigation: 'Team_employment_Home',
      icon: require('../../Images/adminhome/icosn/eq102.png'),
    },
    inspectionReport: {
      category: 'Inspection and Compliances',
      label: 'Inspection Report',
      navigation: 'Inspection_Report_Home',
      icon: require('../../Images/adminhome/icosn/ins0101.png'),
    },
    hseInspection: {
      category: 'Inspection and Compliances',
      label: 'HSE Inspection',
      navigation: 'HSEInspectionHome',
      icon: require('../../Images/adminhome/icosn/ins0101.png'),
    },
    dvirReport: {
      category: 'Inspection and Compliances',
      label: 'DVIR Inspection',
      navigation: 'DviranddotInHome',
      icon: require('../../Images/adminhome/icosn/eq103.png'),
    },
    checklist: {
      category: 'Inspection and Compliances',
      label: 'Check List',
      navigation: 'Checklist_home',
      icon: require('../../Images/adminhome/icosn/checklist1.png'),
    },
    incidentReport: {
      category: 'Inspection and Compliances',
      label: 'Incident Report',
      navigation: 'IncodentReport',
      icon: require('../../Images/adminhome/icosn/ins0101.png'),
    },
    dailyReport: {
      category: 'Inspection and Compliances',
      label: 'Daily Inspection',
      navigation: 'DailyIns_report_home',
      icon: require('../../Images/adminhome/icosn/solar_calendar-linear.png'),
    },
    requestMaintenance: {
      category: 'Maintenance & Premium Services',
      label: 'Request Maintenance',
      navigation: 'RequestmaintenanceHome',
      icon: require('../../Images/adminhome/icosn/RM01.png'),
    },
    jobcard: {
      category: 'Maintenance & Premium Services',
      label: 'Job Card',
      navigation: 'WorkOrdertab',
      icon: require('../../Images/adminhome/icosn/jobcard1.png'),
    },
    serviceSchedule: {
      category: 'Maintenance & Premium Services',
      label: 'Service Schedule',
      navigation: 'ServiceScheduleHome',
      icon: require('../../Images/adminhome/icosn/schdule01.png'),
    },
    fuelLog: {
      category: 'Logbook',
      label: 'Fuel Log',
      navigation: 'Fuel_Log_List',
      icon: require('../../Images/adminhome/icosn/fuel-log.png'),
    },
    logbook: {
      category: 'Logbook',
      label: 'Logbook',
      navigation: 'LogbookList',
      icon: require('../../Images/adminhome/icosn/logbook1.png'),
    },
    hrForm: {
      category: 'HR',
      label: 'HR Form',
      navigation: 'AdminCheckAllApli',
      icon: require('../../Images/adminhome/icosn/hrform01.png'),
    },
    hrApplications: {
      category: 'HR',
      label: 'HR Applications',
      navigation: 'AdminHrDocument',
      icon: require('../../Images/adminhome/icosn/hrdocuments01.png'),
    },
    teamKpi: {
      category: 'HR',
      label: 'Team KPI',
      navigation: 'HRReportingHome',
      icon: require('../../Images/adminhome/icosn/Hrreporting01.png'),
    },
  };

  const compareVersions = useCallback((source, target) => {
    if (!source || !target) return 0;

    const toSegments = (value) =>
      value
        .toString()
        .split('.')
        .map((segment) => Number.parseInt(segment, 10) || 0);

    const sourceSegments = toSegments(source);
    const targetSegments = toSegments(target);
    const maxLength = Math.max(sourceSegments.length, targetSegments.length);

    for (let index = 0; index < maxLength; index += 1) {
      const a = sourceSegments[index] ?? 0;
      const b = targetSegments[index] ?? 0;
      if (a > b) return 1;
      if (a < b) return -1;
    }

    return 0;
  }, []);

  const normalizeVersionPayload = useCallback((payload) => {
    if (!payload) return null;

    const base = payload.data ?? payload;
    if (!base) return null;

    const platformKey = Platform.OS === 'ios' ? 'ios' : 'android';
    const altPlatformKey = Platform.OS === 'ios' ? 'IOS' : 'Android';

    let platformData =
      base[platformKey] ??
      base[altPlatformKey] ??
      base?.platforms?.[platformKey] ??
      base?.platforms?.[altPlatformKey] ??
      null;

    if (typeof platformData === 'string') {
      platformData = { latestVersion: platformData };
    }

    if (Array.isArray(platformData)) {
      platformData = platformData[0];
    }

    const latestVersion =
      platformData?.latestVersion ??
      platformData?.version ??
      platformData?.latest ??
      platformData?.current ??
      base.latestVersion ??
      base.version ??
      null;

    const minimumVersion =
      platformData?.minimumVersion ??
      platformData?.minSupportedVersion ??
      platformData?.minimum ??
      platformData?.requiredVersion ??
      base.minimumVersion ??
      null;

    const latestBuild =
      platformData?.latestBuild ??
      platformData?.buildNumber ??
      platformData?.build ??
      base.latestBuild ??
      base.buildNumber ??
      null;

    const minimumBuild =
      platformData?.minimumBuildNumber ??
      platformData?.minBuild ??
      platformData?.minSupportedBuild ??
      base.minimumBuildNumber ??
      null;

    const storeUrl =
      platformData?.storeUrl ??
      platformData?.url ??
      platformData?.downloadUrl ??
      base.storeUrl ??
      base.url ??
      null;

    const releaseNotes =
      platformData?.releaseNotes ??
      platformData?.notes ??
      platformData?.changelog ??
      base.releaseNotes ??
      base.notes ??
      null;

    const forceUpdate = Boolean(
      platformData?.forceUpdate ||
        platformData?.mandatory ||
        platformData?.required ||
        base.forceUpdate ||
        base.mandatory
    );

    return {
      latestVersion: latestVersion ? latestVersion.toString() : null,
      minimumVersion: minimumVersion ? minimumVersion.toString() : null,
      latestBuild: latestBuild ? latestBuild.toString() : null,
      minimumBuild: minimumBuild ? minimumBuild.toString() : null,
      forceUpdate,
      storeUrl,
      releaseNotes,
    };
  }, []);

  const checkAppVersion = useCallback(async () => {
    try {
      if (!isMounted.current) return;

      const [currentVersion, currentBuild] = await Promise.all([
        DeviceInfo.getVersion(),
        DeviceInfo.getBuildNumber(),
      ]);

      let response;
      let fetchError = null;

      try {
        response = await MongoAPI.getMobileAppVersion();
      } catch (error) {
        fetchError = error;
        console.warn('⚠️ Mongo version fetch failed:', error.message);
      }

      if (!response) {
        try {
          const localVersionData = require('../../app-version.json');
          response = {
            android: localVersionData.android,
            ios: localVersionData.ios,
          };
          console.log('ℹ️ Using local app-version.json as fallback');
        } catch (fallbackError) {
          console.error('❌ Local version fallback failed:', fallbackError);
          if (fetchError) {
            console.error('❌ Original version fetch error:', fetchError);
          }
          return;
        }
      }

      const normalized = normalizeVersionPayload(response);
      if (!normalized || !normalized.latestVersion) {
        return;
      }

      const remoteBuild = normalized.latestBuild ? parseInt(normalized.latestBuild, 10) : null;
      const localBuild = currentBuild ? parseInt(currentBuild, 10) : null;

      const needsVersionUpdate = compareVersions(normalized.latestVersion, currentVersion) > 0;
      const needsBuildUpdate =
        !needsVersionUpdate &&
        remoteBuild !== null &&
        localBuild !== null &&
        remoteBuild > localBuild;

      const minimumVersionRequired =
        normalized.minimumVersion && compareVersions(normalized.minimumVersion, currentVersion) > 0;

      const minimumBuildRequired =
        normalized.minimumBuild &&
        localBuild !== null &&
        parseInt(normalized.minimumBuild, 10) > localBuild;

      const shouldPrompt =
        needsVersionUpdate || needsBuildUpdate || minimumVersionRequired || minimumBuildRequired;

      if (!shouldPrompt) {
        if (remindLaterTimeoutRef.current) {
          clearTimeout(remindLaterTimeoutRef.current);
          remindLaterTimeoutRef.current = null;
        }
        if (remindInfo) {
          await AsyncStorage.removeItem(APP_UPDATE_REMIND_INFO_KEY);
        }
        return;
      }

      const remindInfoRaw = await AsyncStorage.getItem(APP_UPDATE_REMIND_INFO_KEY);
      let remindInfo = null;
      if (remindInfoRaw) {
        try {
          remindInfo = JSON.parse(remindInfoRaw);
        } catch (error) {
          remindInfo = null;
        }
      }

      const reminderMatchesCurrent =
        remindInfo &&
        remindInfo.version === normalized.latestVersion &&
        (remindInfo.build ?? null) === (normalized.latestBuild ?? null);

      if (
        reminderMatchesCurrent &&
        typeof remindInfo.remindAt === 'number' &&
        remindInfo.remindAt > Date.now()
      ) {
        const delay = remindInfo.remindAt - Date.now();
        if (remindLaterTimeoutRef.current) {
          clearTimeout(remindLaterTimeoutRef.current);
        }
        remindLaterTimeoutRef.current = setTimeout(() => {
          if (!isMounted.current) return;
          versionCheckAttempted.current = false;
          checkAppVersion().catch((error) =>
            console.error('❌ Scheduled version check failed:', error)
          );
        }, delay);
        return;
      }

      if (!isMounted.current) return;

      const fallbackStoreUrl = Platform.OS === 'ios' ? APP_STORE_URL : PLAY_STORE_URL;

      setUpdateInfo({
        currentVersion,
        currentBuild,
        latestVersion: normalized.latestVersion,
        latestBuild: normalized.latestBuild,
        forceUpdate: normalized.forceUpdate || minimumVersionRequired || minimumBuildRequired,
        storeUrl: normalized.storeUrl || fallbackStoreUrl,
        releaseNotes: normalized.releaseNotes,
      });
      setShowUpdateModal(true);
      if (remindLaterTimeoutRef.current) {
        clearTimeout(remindLaterTimeoutRef.current);
        remindLaterTimeoutRef.current = null;
      }
    } catch (error) {
      console.error('❌ Version check error:', error);
    }
  }, [compareVersions, normalizeVersionPayload]);

  const handleUpdateNow = useCallback(() => {
    if (!updateInfo) return;

    if (!updateInfo.storeUrl) {
      showToast('info', 'Update', 'Update link not available yet. Please contact support.');
      return;
    }

    AsyncStorage.removeItem(APP_UPDATE_REMIND_INFO_KEY).catch((error) =>
      console.warn('⚠️ Failed to clear reminder info:', error)
    );

    Linking.openURL(updateInfo.storeUrl).catch((error) => {
      console.error('❌ Failed to open update link:', error);
      showToast('error', 'Error', 'Unable to open update link');
    });
  }, [updateInfo, showToast]);

  const handleRemindLater = useCallback(async () => {
    if (!updateInfo) return;

    try {
      const remindAt = Date.now() + 2 * 60 * 1000;
      const payload = {
        version: updateInfo.latestVersion,
        build: updateInfo.latestBuild ?? null,
        remindAt,
      };
      await AsyncStorage.setItem(APP_UPDATE_REMIND_INFO_KEY, JSON.stringify(payload));
      if (remindLaterTimeoutRef.current) {
        clearTimeout(remindLaterTimeoutRef.current);
      }
      remindLaterTimeoutRef.current = setTimeout(() => {
        if (!isMounted.current) return;
        versionCheckAttempted.current = false;
        checkAppVersion().catch((error) =>
          console.error('❌ Scheduled version check failed:', error)
        );
      }, 2 * 60 * 1000);
    } catch (error) {
      console.error('❌ Failed to store update reminder preference:', error);
    } finally {
      setShowUpdateModal(false);
      setUpdateInfo(null);
    }
  }, [updateInfo, checkAppVersion]);

  // Show toast notification
  const showToast = useCallback((type, text1, text2) => {
    if (!isMounted.current) return;
    Toast.show({
      type,
      text1,
      text2,
      position: 'bottom',
      visibilityTime: 3000,
      autoHide: true,
    });
  }, []);

  // Helper: Get module permissions from AsyncStorage (for offline use)
  const getModulePermissions = useCallback(async (moduleName) => {
    try {
      // Try permission map first (fastest)
      const storedPermissions = await AsyncStorage.getItem('modulePermissions');
      if (storedPermissions) {
        const permissionMap = JSON.parse(storedPermissions);
        if (permissionMap[moduleName]) {
          return permissionMap[moduleName];
        }
      }
      
      // Fallback to userModules
      const storedUserModules = await AsyncStorage.getItem('userModules');
      if (storedUserModules) {
        const fullModules = JSON.parse(storedUserModules);
        const moduleData = fullModules.find(m => m.module === moduleName);
        if (moduleData && moduleData.permissions) {
          return moduleData.permissions;
        }
      }
      
      return [];
    } catch (error) {
      console.error('❌ Permission load error:', error.message);
      return [];
    }
  }, []);

  // Check network status (respecting force offline mode)
  const checkNetworkStatus = useCallback(async () => {
    try {
      if (forceOfflineMode) {
        return false; // Force offline
      }
      const netInfo = await NetInfo.fetch();
      // Check both isConnected and isInternetReachable for better accuracy
      const hasConnection = netInfo.isConnected && netInfo.isInternetReachable !== false;
      return hasConnection;
    } catch (error) {
      console.error('❌ Network check error:', error.message);
      return false;
    }
  }, [forceOfflineMode]);

  // Save pending changes to AsyncStorage
  const savePendingChanges = useCallback(async (changeType, data) => {
    try {
      const existingChanges = await AsyncStorage.getItem('pendingChanges');
      const changes = existingChanges ? JSON.parse(existingChanges) : [];
      
      changes.push({
        type: changeType,
        data: data,
        timestamp: new Date().toISOString(),
      });
      
      await AsyncStorage.setItem('pendingChanges', JSON.stringify(changes));
      setPendingChanges(true);
    } catch (error) {
      console.error('❌ Save pending error:', error.message);
    }
  }, []);

  // Sync pending changes to server
  const syncPendingChanges = useCallback(async () => {
    try {
      const existingChanges = await AsyncStorage.getItem('pendingChanges');
      if (!existingChanges) {
        setPendingChanges(false);
        return;
      }

      const changes = JSON.parse(existingChanges);
      if (changes.length === 0) {
        setPendingChanges(false);
        return;
      }

      setSyncModalVisible(true);
      setSyncModalMessage(`Syncing ${changes.length} changes...`);

      // Sync each change
      for (const change of changes) {
        try {
          if (change.type === 'countryUpdate' && userId) {
            await MongoAPI.updateUser(userId, {
              countries: change.data.countries,
            });
          }
        } catch (error) {
          console.error('❌ Sync change failed:', error.message);
        }
      }

      // Clear pending changes
      await AsyncStorage.removeItem('pendingChanges');
      setPendingChanges(false);
      showToast('success', 'Synced', 'All changes synced to server');
    } catch (error) {
      console.error('❌ Pending sync error:', error.message);
      showToast('error', 'Sync Failed', 'Some changes could not be synced');
    } finally {
      setSyncModalVisible(false);
      setSyncModalMessage('Processing...');
    }
  }, [userId, showToast]);

  // Fetch user data from MongoDB and update permissions
  const fetchUserDataFromMongoDB = useCallback(
    async (uid, showSyncModal = true) => {
      if (!isMounted.current) return { success: false };
      try {
        if (showSyncModal) {
          setSyncModalVisible(true);
          setSyncModalMessage('Syncing with Server...');
        }

        // Use MongoAPI to fetch latest user data
        const result = await MongoAPI.getUserById(uid);
        
        if (result.success && result.data) {
          const user = result.data;
          setUserData(user);
          
          // Extract modules based on permissions
          let userModules = [];
          if (user.modules && Array.isArray(user.modules) && user.modules.length > 0) {
            userModules = user.modules
              .map((mod) => mod.module)
              .filter((moduleName) => !!moduleDefinitions[moduleName]);
          } else {
            userModules = ['Equipments', 'Documents', 'teamEmploy', 'hseInspection'];
          }
          
          // Store everything in AsyncStorage for offline access
          const syncTime = new Date().toISOString();
          const dataWithTimestamp = { ...user, lastSyncTime: syncTime };
          
          await AsyncStorage.setItem('userData', JSON.stringify(dataWithTimestamp));
          await AsyncStorage.setItem('modules', JSON.stringify(userModules));
          await AsyncStorage.setItem('userModules', JSON.stringify(user.modules || []));
          
          // Create module-permission map for easy offline access
          const modulePermissionMap = {};
          (user.modules || []).forEach(mod => {
            modulePermissionMap[mod.module] = mod.permissions || [];
          });
          await AsyncStorage.setItem('modulePermissions', JSON.stringify(modulePermissionMap));
          await AsyncStorage.setItem('lastSyncTimestamp', syncTime);
          setLastSyncTime(syncTime);
          
          // Update UI state
          setModules(userModules);

          // Store countries
          if (user.countries && Array.isArray(user.countries)) {
            setSelectedCountries(user.countries);
            await AsyncStorage.setItem('selectedCountries', JSON.stringify(user.countries));
          }
          
          console.log(`✅ Sync Complete: ${user.name} | ${userModules.length} modules | ${user.countries?.length || 0} countries`);
          
          if (showSyncModal) {
            showToast('success', 'Synced', `Loaded ${userModules.length} modules from server`);
          }
          
          return { success: true, data: user, modules: userModules };
        } else {
          return { success: false, error: 'No user data found' };
        }
      } catch (error) {
        console.error('❌ Sync failed:', error.message);
        
        // Fallback to AsyncStorage
        const storedData = await AsyncStorage.getItem('userData');
        const storedModules = await AsyncStorage.getItem('modules');
        const storedCountries = await AsyncStorage.getItem('selectedCountries');
        
        if (storedData && storedModules) {
          const user = JSON.parse(storedData);
          const moduleNames = JSON.parse(storedModules);
          
          setUserData(user);
          setModules(moduleNames);
          
          if (storedCountries) {
            setSelectedCountries(JSON.parse(storedCountries));
          }
          
          showToast('info', 'Offline Mode', 'Using locally stored data');
          return { success: true, source: 'local', modules: moduleNames };
        }
        
        showToast('error', 'Error', 'Failed to load user data');
        return { success: false, error: error.message };
      } finally {
        if (isMounted.current && showSyncModal) {
          setSyncModalVisible(false);
          setSyncModalMessage('Processing...');
        }
      }
    },
    [showToast, moduleDefinitions]
  );

  // Toggle online/offline mode
  const toggleOfflineMode = useCallback(async () => {
    const newOfflineMode = !forceOfflineMode;
    
    if (newOfflineMode) {
      // Switching to offline
      setForceOfflineMode(true);
      await AsyncStorage.setItem('forceOfflineMode', JSON.stringify(true));
      setIsOnline(false);
      showToast('info', 'Offline Mode', 'Working offline - using cached modules');
    } else {
      // Trying to switch to online - check actual network first
      try {
        const netInfo = await NetInfo.fetch();
        const hasInternet = netInfo.isConnected && netInfo.isInternetReachable !== false;
        
        if (hasInternet) {
          // Network is available - switch to online
          setForceOfflineMode(false);
          await AsyncStorage.setItem('forceOfflineMode', JSON.stringify(false));
          setIsOnline(true);
          showToast('success', 'Online Mode', 'Syncing modules & permissions...');
          
          // Auto-sync user data (modules/permissions) and pending changes
          if (userId) {
            await fetchUserDataFromMongoDB(userId, true);
            await syncPendingChanges();
          }
        } else {
          // No network available - keep offline mode ON
          setForceOfflineMode(true);
          await AsyncStorage.setItem('forceOfflineMode', JSON.stringify(true));
          setIsOnline(false);
          showToast('error', 'No Internet', 'No network. Staying offline.');
        }
      } catch (error) {
        console.error('❌ Network toggle error:', error.message);
        setForceOfflineMode(true);
        await AsyncStorage.setItem('forceOfflineMode', JSON.stringify(true));
        setIsOnline(false);
        showToast('error', 'Network Error', 'Cannot detect network. Staying offline.');
      }
    }
  }, [forceOfflineMode, showToast, userId, fetchUserDataFromMongoDB, syncPendingChanges]);

  // Update country in MongoDB
  const updateCountryInMongoDB = useCallback(
    async (userId, countries) => {
      if (!isMounted.current || !userId) return;

      setLoading(true);
      setSyncModalVisible(true);
      setSyncModalMessage('Updating Countries');

      try {
        const onlineStatus = await checkNetworkStatus();
        
        if (onlineStatus && !forceOfflineMode) {
          // Online - update directly
          const result = await MongoAPI.updateUser(userId, { countries });

          if (result.success) {
            await AsyncStorage.setItem('selectedCountries', JSON.stringify(countries));
            showToast('success', 'Success', 'Countries updated successfully');
          }
        } else {
          // Offline - save to local and pending changes
          await AsyncStorage.setItem('selectedCountries', JSON.stringify(countries));
          await savePendingChanges('countryUpdate', { countries });
          showToast('info', 'Saved Offline', 'Will sync when online');
        }
      } catch (error) {
        console.error('❌ Country update error:', error.message);
        
        // Save to local as fallback
        await AsyncStorage.setItem('selectedCountries', JSON.stringify(countries));
        await savePendingChanges('countryUpdate', { countries });
        showToast('info', 'Saved Offline', 'Will sync when online');
      } finally {
        if (isMounted.current) {
          setLoading(false);
          setSyncModalVisible(false);
          setSyncModalMessage('Processing...');
        }
      }
    },
    [showToast, checkNetworkStatus, forceOfflineMode, savePendingChanges]
  );

  // Handle country selection
  const handleCountrySelect = useCallback(
    async (country) => {
      if (!isMounted.current) return;
      try {
        if (!countryOptions.includes(country)) {
          showToast('error', 'Error', 'Invalid country selected');
          return;
        }

        setSelectedCountry(country);
        setModalVisible(false);
        await AsyncStorage.setItem('selectedCountry', country);
        showToast('success', 'Success', `Country updated to ${country}`);
      } catch (error) {
        console.error('❌ Country selection error:', error.message);
        showToast('error', 'Error', error.message);
      }
    },
    [showToast]
  );

  // Handle module press
  const handlePress = useCallback(
    (navigationRoute, actionName) => {
      try {
        if (!navigationRoute) {
          showToast('error', 'Error', 'Navigation route is missing');
          return;
        }
        
        navigation.navigate(navigationRoute);
      } catch (navError) {
        console.error('❌ Navigation error:', navError.message);
        showToast('error', 'Navigation Error', navError.message || 'Failed to navigate');
      }
    },
    [navigation, showToast]
  );

  // Handle back button press - Reload ONLY on first focus
  useFocusEffect(
    useCallback(() => {
      // Mark page as focused
      isPageFocused.current = true;
      
      const backAction = () => {
        Alert.alert(
          'Exit App',
          'Are you sure you want to exit the app?',
          [
            {
              text: 'Cancel',
              style: 'cancel',
            },
            {
              text: 'Exit',
              style: 'destructive',
              onPress: () => BackHandler.exitApp(),
            },
          ]
        );
        return true;
      };

      const backHandler = BackHandler.addEventListener('hardwareBackPress', backAction);
      
      // Cleanup when leaving the page
      return () => {
        isPageFocused.current = false;
        backHandler.remove();
        
        // Clear any background sync timeouts
        if (backgroundSyncTimeout.current) {
          clearTimeout(backgroundSyncTimeout.current);
          backgroundSyncTimeout.current = null;
        }
      };
    }, [])
  );

  // Load user data from AsyncStorage (OFFLINE MODE)
  const loadUserDataFromLocal = useCallback(async () => {
    try {
      // Get stored data
      const storedData = await AsyncStorage.getItem('userData');
      const storedModules = await AsyncStorage.getItem('modules');
      const storedUserModules = await AsyncStorage.getItem('userModules');
      const storedPermissions = await AsyncStorage.getItem('modulePermissions');
      const storedCountries = await AsyncStorage.getItem('selectedCountries');
      const storedCountry = await AsyncStorage.getItem('selectedCountry');
      const storedSyncTime = await AsyncStorage.getItem('lastSyncTimestamp');
      
      if (storedSyncTime) {
        setLastSyncTime(storedSyncTime);
      }
      
      // Load modules
      let loadedModules = [];
      
      if (storedModules) {
        const moduleNames = JSON.parse(storedModules);
        if (Array.isArray(moduleNames) && moduleNames.length > 0) {
          const validModules = moduleNames.filter((moduleName) => !!moduleDefinitions[moduleName]);
          
          if (validModules.length > 0) {
            loadedModules = validModules;
            setModules(validModules);
          }
        }
      }
      
      // Extract from userData if modules not loaded
      if (loadedModules.length === 0 && storedData) {
        const user = JSON.parse(storedData);
        setUserData(user);
        
        if (user.modules && Array.isArray(user.modules)) {
          const moduleNames = user.modules
            .map((mod) => mod.module)
            .filter((moduleName) => moduleDefinitions[moduleName]);
          
          if (moduleNames.length > 0) {
            loadedModules = moduleNames;
            setModules(moduleNames);
            await AsyncStorage.setItem('modules', JSON.stringify(moduleNames));
            
            const modulePermissionMap = {};
            user.modules.forEach(mod => {
              modulePermissionMap[mod.module] = mod.permissions || [];
            });
            await AsyncStorage.setItem('modulePermissions', JSON.stringify(modulePermissionMap));
          }
        }
      }
      
      // Use defaults if still no modules
      if (loadedModules.length === 0) {
        const defaultModules = ['Equipments', 'Documents', 'teamEmploy', 'hseInspection'];
        loadedModules = defaultModules;
        setModules(defaultModules);
        await AsyncStorage.setItem('modules', JSON.stringify(defaultModules));
      }
      
      // Load countries
      if (storedCountries) {
        const countries = JSON.parse(storedCountries);
        if (Array.isArray(countries) && countries.length > 0) {
          setSelectedCountries(countries);
        }
      } else if (storedData) {
        const user = JSON.parse(storedData);
        if (user.countries && Array.isArray(user.countries)) {
          setSelectedCountries(user.countries);
          await AsyncStorage.setItem('selectedCountries', JSON.stringify(user.countries));
        }
      }
      
      // Load selected country
      if (storedCountry && countryOptions.includes(storedCountry)) {
        setSelectedCountry(storedCountry);
      } else if (selectedCountries && selectedCountries.length > 0) {
        setSelectedCountry(selectedCountries[0]);
      } else {
        setSelectedCountry('Zambia');
      }
      
      // Load user data
      if (storedData) {
        const user = JSON.parse(storedData);
        setUserData(user);
      }
      
      return { success: true, modules: loadedModules };
    } catch (error) {
      console.error('❌ AsyncStorage load error:', error.message);
      
      // Set defaults on error
      const defaultModules = ['Equipments', 'Documents', 'teamEmploy', 'hseInspection'];
      setModules(defaultModules);
      setSelectedCountry('Zambia');
      return { success: false, error: error.message };
    }
  }, [moduleDefinitions, countryOptions]);

  // Load data on mount
  // OFFLINE-FIRST STRATEGY:
  // 1. Always load from AsyncStorage first (instant, works offline)
  // 2. If online, sync from MongoDB in background (doesn't block UI)
  // 3. Stored data includes: userData, modules, countries, selectedCountry
  // 4. This allows app to work fully offline after first online login
  useEffect(() => {
    // Ensure mounted state is true
    isMounted.current = true;
    
    let netInfoUnsubscribe;
    let loadingTimeout;

    const loadData = async () => {
      try {
        // Mark page as focused for initial load
        isPageFocused.current = true;
        
        // Load force offline mode preference
        const savedOfflineMode = await AsyncStorage.getItem('forceOfflineMode');
        if (savedOfflineMode) {
          setForceOfflineMode(JSON.parse(savedOfflineMode));
        }
        
        // Check for pending changes
        const savedPendingChanges = await AsyncStorage.getItem('pendingChanges');
        if (savedPendingChanges) {
          const changes = JSON.parse(savedPendingChanges);
          setPendingChanges(changes.length > 0);
        }
        
        // Get user ID first
        const uid = await AsyncStorage.getItem('userUid');
        setUserId(uid);

        if (!uid) {
          setDataLoaded(true);
          setLoading(false);
          return;
        }

        // Check network status
        const netInfo = await NetInfo.fetch();
        const onlineStatus = forceOfflineMode ? false : (netInfo.isConnected && netInfo.isInternetReachable !== false);
        setIsOnline(onlineStatus);
        lastNetworkState.current = onlineStatus;
        
        // Auto-set offline mode if no internet
        if (!onlineStatus && !forceOfflineMode) {
          setForceOfflineMode(true);
          await AsyncStorage.setItem('forceOfflineMode', JSON.stringify(true));
        }

        // Load from AsyncStorage first (instant)
        const localResult = await loadUserDataFromLocal();
        
        if (localResult.success && localResult.modules && localResult.modules.length > 0) {
          setDataLoaded(true);
          setLoading(false);
          
          // Show console log ONCE on first load only
          if (!hasLoadedOnce.current) {
            console.log('═══════════════════════════════════════');
            console.log('🏠 HOME SCREEN LOADED');
            console.log(`📦 ${localResult.modules.length} modules | ${onlineStatus ? '🌐 Online' : '📴 Offline'}`);
            console.log('═══════════════════════════════════════');
            hasLoadedOnce.current = true;
            
            // Show offline toast ONCE on first load if offline
            if (!onlineStatus) {
              showToast('info', 'Offline Mode', `${localResult.modules.length} modules loaded`);
              setHasShownOfflineToast(true);
            }
          }
          
          return;
        }
        
        // No local data - fetch from server
        if (onlineStatus) {
          const result = await fetchUserDataFromMongoDB(uid, false);
          
          if (result.success && result.modules) {
            showToast('success', 'Welcome', `Loaded ${result.modules.length} modules`);
          } else {
            showToast('error', 'Error', 'Failed to load user data');
            const defaultModules = ['Equipments', 'Documents', 'teamEmploy', 'hseInspection'];
            setModules(defaultModules);
            await AsyncStorage.setItem('modules', JSON.stringify(defaultModules));
            setSelectedCountry('Zambia');
          }
        } else {
          showToast('error', 'Error', 'No offline data. Connect to internet for first login.');
          const defaultModules = ['Equipments', 'Documents', 'teamEmploy', 'hseInspection'];
          setModules(defaultModules);
          await AsyncStorage.setItem('modules', JSON.stringify(defaultModules));
          setSelectedCountry('Zambia');
        }

        setDataLoaded(true);
        setLoading(false);
      } catch (error) {
        console.error('❌ Load error:', error.message);
        await loadUserDataFromLocal();
        
        if (!selectedCountry && (!selectedCountries || selectedCountries.length === 0)) {
          setSelectedCountry('Zambia');
        }
        
        setDataLoaded(true);
        setLoading(false);
        showToast('error', 'Error', 'Failed to load user data');
      }
    };

    // Set timeout to prevent infinite loading
    loadingTimeout = setTimeout(() => {
      if (isMounted.current && loading) {
        if (!selectedCountry && (!selectedCountries || selectedCountries.length === 0)) {
          setSelectedCountry('Zambia');
        }
        
        setDataLoaded(true);
        setLoading(false);
        showToast('info', 'Timeout', 'Loading took too long, using cached data');
      }
    }, 15000);

    loadData().catch((error) => {
      console.error('❌ Load failed:', error.message);
      setDataLoaded(true);
      setLoading(false);
    });

    // Network listener - Auto sync ONLY when page is focused
    netInfoUnsubscribe = NetInfo.addEventListener((state) => {
      if (!isMounted.current || forceOfflineMode || !isPageFocused.current) return;

      // Check both isConnected and isInternetReachable
      const newOnlineStatus = state.isConnected && state.isInternetReachable !== false;
      const currentState = lastNetworkState.current;
      
      // Only process if state actually changed
      if (currentState !== null && currentState === newOnlineStatus) {
        return; // No change, skip
      }
      
      lastNetworkState.current = newOnlineStatus;
      setIsOnline(newOnlineStatus);

      // Connection lost - show toast ONCE
      if (!newOnlineStatus && !hasShownOfflineToast) {
        showToast('info', 'Offline', 'No internet connection');
        setHasShownOfflineToast(true);
        setHasShownOnlineToast(false);
      } 
      // Connection restored - show toast ONCE and auto-sync (only if on home page)
      else if (newOnlineStatus && !hasShownOnlineToast && isPageFocused.current) {
        showToast('success', 'Online', 'Connection restored');
        setHasShownOnlineToast(true);
        setHasShownOfflineToast(false);
        
        // Auto-sync when back online - ONLY if on home page
        if (userId) {
          fetchUserDataFromMongoDB(userId, false)
            .then(() => syncPendingChanges())
            .catch((error) => {
              console.error('❌ Auto-sync failed:', error.message);
            });
        }
      }
    });

    return () => {
      isMounted.current = false;
      isPageFocused.current = false;
      if (netInfoUnsubscribe) netInfoUnsubscribe();
      if (loadingTimeout) clearTimeout(loadingTimeout);
      if (backgroundSyncTimeout.current) clearTimeout(backgroundSyncTimeout.current);
      if (remindLaterTimeoutRef.current) {
        clearTimeout(remindLaterTimeoutRef.current);
        remindLaterTimeoutRef.current = null;
      }
    };
  }, []); // Empty dependencies - only run once on mount

  useEffect(() => {
    if (!dataLoaded || !isOnline) return;
    if (versionCheckAttempted.current) return;

    versionCheckAttempted.current = true;
    checkAppVersion()
      .catch((error) => {
        console.error('❌ Failed to check mobile app version:', error);
        versionCheckAttempted.current = false;
      });
  }, [dataLoaded, isOnline, checkAppVersion]);

  useEffect(() => {
    if (!isOnline) {
      versionCheckAttempted.current = false;
    }
  }, [isOnline]);

  // Handler for toggling country selection
  const handleToggleCountry = (country) => {
    setSelectedCountries((prev) => {
      if (country === 'Both Countries') {
        const hasBoth = prev.includes('Zambia') && prev.includes('DRC');
        if (hasBoth) {
          return prev.filter((c) => c !== 'Zambia' && c !== 'DRC' && c !== 'Both Countries');
        } else {
          return ['Zambia', 'DRC'];
        }
      } else {
        let newSelected;
        if (prev.includes(country)) {
          newSelected = prev.filter((c) => c !== country && c !== 'Both Countries');
        } else {
          newSelected = [...prev.filter((c) => c !== 'Both Countries'), country];
        }
        if (newSelected.includes('Zambia') && newSelected.includes('DRC')) {
          return ['Zambia', 'DRC'];
        }
        return newSelected;
      }
    });
  };

  // Handler for saving selected countries
  const handleSaveCountries = async () => {
    if (!userId) return;
    setSyncModalVisible(true);
    setSyncModalMessage('Updating Countries...');
    try {
      await updateCountryInMongoDB(userId, selectedCountries);
      setModalVisible(false);
    } catch (error) {
      showToast('error', 'Error', 'Failed to update countries');
    } finally {
      setSyncModalVisible(false);
      setSyncModalMessage('Processing...');
    }
  };

  // Manual sync handler - Only works when page is focused
  const handleManualSync = async () => {
    if (!userId) {
      showToast('error', 'Error', 'No user ID found');
      return;
    }

    if (!isPageFocused.current) {
      console.log('⚠️ Manual sync blocked - page not focused');
      return;
    }

    const onlineStatus = await checkNetworkStatus();
    if (!onlineStatus) {
      showToast('error', 'Offline', 'Cannot sync while offline');
      return;
    }

    setRefreshing(true);
    try {
      const result = await fetchUserDataFromMongoDB(userId, true);
      if (!result.success) {
        showToast('error', 'Sync Failed', 'Could not sync data');
      }
    } catch (error) {
      console.error('❌ Manual sync error:', error.message);
      showToast('error', 'Sync Error', error.message);
    } finally {
      setRefreshing(false);
    }
  };

  const handleUploadEquipmentSeedData = useCallback(async () => {
    if (isUploadingEquipment) return;
    try {
      setIsUploadingEquipment(true);
      setSyncModalMessage('Uploading equipment data...');
      setSyncModalVisible(true);
      showToast('info', 'Uploading Equipment', 'Please wait (up to 5 minutes)');

      const response = await MongoAPI.uploadEquipmentSeedData();
      const success = response?.success;

      if (success) {
        const equipmentSummary = response.summary?.equipment || {};
        const created = equipmentSummary.created || 0;
        const updated = equipmentSummary.updated || 0;
        const skipped = equipmentSummary.skipped || 0;
        showToast(
          'success',
          'Equipment Uploaded',
          `Created: ${created} • Updated: ${updated} • Skipped: ${skipped}`
        );
      } else {
        showToast('error', 'Upload Failed', response?.error || 'Unknown error');
      }
    } catch (error) {
      console.error('❌ Equipment bulk upload error:', error.message);
      showToast('error', 'Upload Failed', error.message || 'Unexpected error');
    } finally {
      setIsUploadingEquipment(false);
      setSyncModalVisible(false);
      setSyncModalMessage('Processing...');
    }
  }, [isUploadingEquipment, showToast]);

  const handleUploadUsersSeedData = useCallback(async () => {
    if (isUploadingUsers) return;
    try {
      setIsUploadingUsers(true);
      setSyncModalMessage('Uploading GlobalUsers data...');
      setSyncModalVisible(true);
      showToast('info', 'Uploading Users', 'Please wait (up to 5 minutes)');

      const response = await MongoAPI.uploadUsersSeedData();
      const success = response?.success;

      if (success) {
        const usersSummary = response.summary?.users || {};
        const created = usersSummary.created || 0;
        const updated = usersSummary.updated || 0;
        const skipped = usersSummary.skipped || 0;
        showToast(
          'success',
          'Users Uploaded',
          `Created: ${created} • Updated: ${updated} • Skipped: ${skipped}`
        );
      } else {
        showToast('error', 'Upload Failed', response?.error || 'Unknown error');
      }
    } catch (error) {
      console.error('❌ Users bulk upload error:', error.message);
      showToast('error', 'Upload Failed', error.message || 'Unexpected error');
    } finally {
      setIsUploadingUsers(false);
      setSyncModalVisible(false);
      setSyncModalMessage('Processing...');
    }
  }, [isUploadingUsers, showToast]);

  // Format last sync time
  const getLastSyncText = () => {
    if (!lastSyncTime) return 'Never synced';
    
    const now = new Date();
    const syncDate = new Date(lastSyncTime);
    const diffInMinutes = Math.floor((now - syncDate) / (1000 * 60));
    
    if (diffInMinutes < 1) return 'Just now';
    if (diffInMinutes < 60) return `${diffInMinutes}m ago`;
    
    const diffInHours = Math.floor(diffInMinutes / 60);
    if (diffInHours < 24) return `${diffInHours}h ago`;
    
    const diffInDays = Math.floor(diffInHours / 24);
    return `${diffInDays}d ago`;
  };

  // Render category with dynamic modules
  const renderCategory = (category, navigationRoute) => {
    const boxSize = Math.floor((width - 5 * 10) / 4);
    const minBoxSize = Math.max(boxSize, 70);

    // Get modules for this category
    const categoryModules = modules
      .filter((mod) => moduleDefinitions[mod]?.category === category)
      .map((mod) => ({
        icon: moduleDefinitions[mod].icon,
        label: moduleDefinitions[mod].label,
        navigationRoute: moduleDefinitions[mod].navigation,
        moduleName: mod,
        onPress: () => handlePress(moduleDefinitions[mod].navigation, mod),
      }));

    // Special handling for Inspection and Compliances category
    if (category === 'Inspection and Compliances') {
      const hseModule = {
        icon: moduleDefinitions.hseInspection.icon,
        label: moduleDefinitions.hseInspection.label,
        navigationRoute: moduleDefinitions.hseInspection.navigation,
        moduleName: 'hseInspection',
        onPress: () => handlePress(moduleDefinitions.hseInspection.navigation, 'hseInspection'),
      };

      const otherModules = categoryModules.filter(mod => mod.label !== 'HSE Inspection');
      const allModules = [hseModule, ...otherModules];

      if (allModules.length === 0) return null;

      const firstRow = allModules.slice(0, 4);
      const secondRow = allModules.slice(4);

      return (
        <>
          <Text style={styles.heading}>{category}</Text>
          <View style={{ alignItems: 'center', width: '100%', paddingHorizontal: 5 }}>
            <View style={{
              flexDirection: 'row',
              justifyContent: 'center',
              alignItems: 'flex-start',
              marginBottom: 12,
              width: '100%',
            }}>
              {firstRow.map((item) => (
                <TouchableOpacity
                  key={item.label}
                  style={{ alignItems: 'center', marginHorizontal: 8 }}
                  onPress={item.onPress}
                  activeOpacity={0.8}
                >
                  <View style={{
                    backgroundColor: '#035484',
                    borderRadius: 12,
                    width: minBoxSize,
                    height: minBoxSize,
                    justifyContent: 'center',
                    alignItems: 'center',
                    marginBottom: 8,
                  }}>
                    <Image source={item.icon} style={{
                      width: item.label === 'Daily Report' ? 42 : 48,
                      height: item.label === 'Daily Report' ? 42 : 48,
                      resizeMode: 'contain',
                    }} />
                  </View>
                  <Text style={{
                    color: '#29486A',
                    fontWeight: 'bold',
                    fontSize: 12,
                    textAlign: 'center',
                    width: minBoxSize,
                    lineHeight: 16,
                    flexWrap: 'wrap',
                    numberOfLines: 2,
                  }} numberOfLines={2} adjustsFontSizeToFit={true} minimumFontScale={0.8}>
                    {item.label}
                  </Text>
                </TouchableOpacity>
              ))}
            </View>
            {secondRow.length > 0 && (
              <View style={{
                flexDirection: 'row',
                justifyContent: 'center',
                alignItems: 'flex-start',
                marginBottom: 12,
                width: '100%',
              }}>
                {secondRow.map((item) => (
                  <TouchableOpacity
                    key={item.label}
                    style={{ alignItems: 'center', marginHorizontal: 8 }}
                    onPress={item.onPress}
                    activeOpacity={0.8}
                  >
                    <View style={{
                      backgroundColor: '#035484',
                      borderRadius: 12,
                      width: minBoxSize,
                      height: minBoxSize,
                      justifyContent: 'center',
                      alignItems: 'center',
                      marginBottom: 8,
                    }}>
                      <Image source={item.icon} style={{
                        width: item.label === 'Daily Report' ? 42 : 48,
                        height: item.label === 'Daily Report' ? 42 : 48,
                        resizeMode: 'contain',
                      }} />
                    </View>
                    <Text style={{
                      color: '#29486A',
                      fontWeight: 'bold',
                      fontSize: 12,
                      textAlign: 'center',
                      width: minBoxSize,
                      lineHeight: 16,
                      flexWrap: 'wrap',
                      numberOfLines: 2,
                    }} numberOfLines={2} adjustsFontSizeToFit={true} minimumFontScale={0.8}>
                      {item.label}
                    </Text>
                  </TouchableOpacity>
                ))}
              </View>
            )}
            <View style={styles.buttonContainer}>
              <TouchableOpacity
                style={styles.gradientButton}
                onPress={() => handlePress(navigationRoute, `View${category.replace(/ & /g, '').replace(/\s/g, '')}KPI`)}
                activeOpacity={0.8}
              >
                <LinearGradient colors={['#3481BC', '#025383']} style={styles.gradientButtonContent}>
                  <Text style={styles.buttonText}>View KPI</Text>
                </LinearGradient>
              </TouchableOpacity>
            </View>
          </View>
        </>
      );
    }

    // For other categories
    if (categoryModules.length === 0) return null;

    const firstRow = categoryModules.slice(0, 4);
    const secondRow = categoryModules.slice(4);

    return (
      <>
        <Text style={styles.heading}>{category}</Text>
        <View style={{ alignItems: 'center', width: '100%', paddingHorizontal: 5 }}>
          <View style={{
            flexDirection: 'row',
            justifyContent: 'center',
            alignItems: 'flex-start',
            marginBottom: 12,
            width: '100%',
          }}>
            {firstRow.map((item, idx) => (
              <TouchableOpacity
                key={item.label}
                style={{ alignItems: 'center', marginHorizontal: 8 }}
                onPress={item.onPress}
                activeOpacity={0.8}
              >
                <View style={{
                  backgroundColor: '#035484',
                  borderRadius: 12,
                  width: minBoxSize,
                  height: minBoxSize,
                  justifyContent: 'center',
                  alignItems: 'center',
                  marginBottom: 8,
                }}>
                  <Image source={item.icon} style={{
                    width: item.label === 'Daily Report' ? 42 : 48,
                    height: item.label === 'Daily Report' ? 42 : 48,
                    resizeMode: 'contain'
                  }} />
                </View>
                <Text style={{
                  color: '#29486A',
                  fontWeight: 'bold',
                  fontSize: 12,
                  textAlign: 'center',
                  width: minBoxSize,
                  lineHeight: 16,
                  flexWrap: 'wrap',
                  numberOfLines: 2,
                }} numberOfLines={2} adjustsFontSizeToFit={true} minimumFontScale={0.8}>
                  {item.label}
                </Text>
              </TouchableOpacity>
            ))}
          </View>
          {secondRow.length > 0 && (
            <View style={{
              flexDirection: 'row',
              justifyContent: 'center',
              alignItems: 'flex-start',
              marginBottom: 12,
              width: '100%',
            }}>
              {secondRow.map((item, idx) => (
                <TouchableOpacity
                  key={item.label}
                  style={{ alignItems: 'center', marginHorizontal: 8 }}
                  onPress={item.onPress}
                  activeOpacity={0.8}
                >
                  <View style={{
                    backgroundColor: '#035484',
                    borderRadius: 12,
                    width: minBoxSize,
                    height: minBoxSize,
                    justifyContent: 'center',
                    alignItems: 'center',
                    marginBottom: 8,
                  }}>
                    <Image source={item.icon} style={{
                      width: item.label === 'Daily Report' ? 42 : 48,
                      height: item.label === 'Daily Report' ? 42 : 48,
                      resizeMode: 'contain'
                    }} />
                  </View>
                  <Text style={{
                    color: '#29486A',
                    fontWeight: 'bold',
                    fontSize: 12,
                    textAlign: 'center',
                    width: minBoxSize,
                    lineHeight: 16,
                    flexWrap: 'wrap',
                    numberOfLines: 2,
                  }} numberOfLines={2} adjustsFontSizeToFit={true} minimumFontScale={0.8}>
                    {item.label}
                  </Text>
                </TouchableOpacity>
              ))}
            </View>
          )}
          <View style={styles.buttonContainer}>
            <TouchableOpacity
              style={styles.gradientButton}
              onPress={() => handlePress(navigationRoute, `View${category.replace(/ & /g, '').replace(/\s/g, '')}KPI`)}
              activeOpacity={0.8}
            >
              <LinearGradient colors={['#3481BC', '#025383']} style={styles.gradientButtonContent}>
                <Text style={styles.buttonText}>View KPI</Text>
              </LinearGradient>
            </TouchableOpacity>
          </View>
        </View>
      </>
    );
  };

  // Country dropdown
  const CountryDropdown = useMemo(() => (
    <TouchableOpacity
      style={{
        borderWidth: 2,
        borderColor: '#035484',
        paddingHorizontal: 7,
        paddingVertical: 3,
        minWidth: '40%',
        backgroundColor: '#fff',
        flexDirection: 'row',
        alignItems: 'center',
        justifyContent: 'space-between',
        shadowColor: '#000',
        shadowOpacity: 0.08,
        shadowRadius: 4,
        shadowOffset: { width: 0, height: 2 },
        elevation: 2,
        marginBottom: 10,
      }}
      onPress={() => setModalVisible(true)}
      activeOpacity={0.8}
    >
      <Text style={{ color: selectedCountries.length > 0 ? '#222' : '#888', fontSize: 16, flex: 1 }}>
        {selectedCountries.includes('Zambia') && selectedCountries.includes('DRC')
          ? 'Both Countries'
          : selectedCountries.length > 0
            ? selectedCountries.join(', ')
            : selectedCountry
              ? selectedCountry
            : 'Choose Country'}
      </Text>
      <MaterialIcons name="keyboard-arrow-down" size={28} color="#035484" />
    </TouchableOpacity>
  ), [selectedCountries, selectedCountry]);

  // Country modal
  const CountryModal = useMemo(() => (
    <Modal
      animationType="fade"
      transparent={true}
      visible={modalVisible}
      onRequestClose={() => setModalVisible(false)}
    >
      <View style={{ flex: 1, backgroundColor: 'rgba(0,0,0,0.10)', justifyContent: 'center', alignItems: 'center' }}>
        <View style={{ backgroundColor: '#fff', borderRadius: 14, width: 320, paddingBottom: 18, shadowColor: '#000', shadowOpacity: 0.12, shadowRadius: 12, shadowOffset: { width: 0, height: 4 }, elevation: 8 }}>
          <View style={{ backgroundColor: '#F6F7F9', borderTopLeftRadius: 14, borderTopRightRadius: 14, paddingVertical: 16, alignItems: 'center', borderBottomWidth: 1, borderBottomColor: '#eee' }}>
            <Text style={{ color: '#035484', fontWeight: 'bold', fontSize: 20 }}>Choose Countries</Text>
          </View>
          {memoizedCountryOptions.map((option, idx) => (
            <TouchableOpacity
              key={option}
              style={{ flexDirection: 'row', alignItems: 'center', justifyContent: 'space-between', paddingVertical: 16, paddingHorizontal: 22, borderBottomWidth: idx !== memoizedCountryOptions.length - 1 ? 1 : 0, borderBottomColor: '#eee', backgroundColor: (option === 'Both Countries' ? (selectedCountries.includes('Zambia') && selectedCountries.includes('DRC')) : selectedCountries.includes(option)) ? '#F6F7F9' : '#fff' }}
              onPress={() => handleToggleCountry(option)}
              activeOpacity={0.8}
            >
              <Text style={{ color: (option === 'Both Countries' ? (selectedCountries.includes('Zambia') && selectedCountries.includes('DRC')) : selectedCountries.includes(option)) ? '#035484' : '#222', fontWeight: (option === 'Both Countries' ? (selectedCountries.includes('Zambia') && selectedCountries.includes('DRC')) : selectedCountries.includes(option)) ? 'bold' : 'normal', fontSize: 18 }}>{option}</Text>
              <View style={{ width: 28, height: 28, borderRadius: 14, borderWidth: 2, borderColor: '#035484', alignItems: 'center', justifyContent: 'center', backgroundColor: '#fff' }}>
                {(option === 'Both Countries' ? (selectedCountries.includes('Zambia') && selectedCountries.includes('DRC')) : selectedCountries.includes(option)) ? (
                  <View style={{ width: 16, height: 16, borderRadius: 8, backgroundColor: '#035484' }} />
                ) : null}
              </View>
            </TouchableOpacity>
          ))}
          <View style={{ flexDirection: 'row', justifyContent: 'center', marginTop: 10 }}>
            <TouchableOpacity
              style={{ backgroundColor: '#035484', borderRadius: 6, paddingVertical: 8, paddingHorizontal: 36, marginTop: 6, marginRight: 10 }}
              onPress={handleSaveCountries}
              activeOpacity={0.8}
            >
              <Text style={{ color: '#fff', fontWeight: 'bold', fontSize: 16 }}>Save</Text>
            </TouchableOpacity>
            <TouchableOpacity
              style={{ backgroundColor: '#888', borderRadius: 6, paddingVertical: 8, paddingHorizontal: 36, marginTop: 6 }}
              onPress={() => setModalVisible(false)}
              activeOpacity={0.8}
            >
              <Text style={{ color: '#fff', fontWeight: 'bold', fontSize: 16 }}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </View>
    </Modal>
  ), [modalVisible, selectedCountries, memoizedCountryOptions, handleToggleCountry, handleSaveCountries]);

  // Conditional rendering
  if (loading || !dataLoaded) {
    return <Skeleton />;
  }

  if (modules.length === 0) {
    return <Skeleton />;
  }

  return (
    <>
      <ScrollView style={styles.scrollContainer}>
        <View style={styles.container}>
          {/* Online/Offline Toggle & Sync Status Bar */}
          <View style={{
            flexDirection: 'row',
            alignItems: 'center',
            justifyContent: 'space-between',
            paddingHorizontal: 10,
            paddingVertical: 8,
            backgroundColor: isOnline ? '#E8F5E9' : '#FFF3E0',
            borderRadius: 8,
            marginBottom: 10,
            borderWidth: 1,
            borderColor: isOnline ? '#4CAF50' : '#FF9800',
          }}>
            <View style={{ flexDirection: 'row', alignItems: 'center', flex: 1 }}>
              {/* Status Indicator */}
              <View style={{
                width: 8,
                height: 8,
                borderRadius: 4,
                backgroundColor: isOnline ? '#4CAF50' : '#FF9800',
                marginRight: 8,
              }} />
              <View style={{ flex: 1 }}>
                <Text style={{ fontSize: 12, color: '#555', fontWeight: 'bold' }}>
                  {isOnline ? '🌐 Online' : '📴 Offline'}
                </Text>
                <Text style={{ fontSize: 10, color: '#777' }}>
                  {lastSyncTime ? `Synced ${getLastSyncText()}` : 'Not synced'}
                  {pendingChanges && ' • Changes pending'}
                </Text>
              </View>
            </View>

            {/* Sync Button */}
            <TouchableOpacity
              onPress={handleManualSync}
              disabled={!isOnline || refreshing}
              style={{
                backgroundColor: isOnline ? '#035484' : '#ccc',
                paddingHorizontal: 10,
                paddingVertical: 6,
                borderRadius: 5,
                flexDirection: 'row',
                alignItems: 'center',
                marginRight: 8,
              }}
              activeOpacity={0.8}
            >
              <MaterialIcons 
                name={refreshing ? "hourglass-empty" : "sync"} 
                size={16} 
                color="#fff" 
                style={{ marginRight: 4 }}
              />
              <Text style={{ color: '#fff', fontSize: 11, fontWeight: 'bold' }}>
                {refreshing ? 'Syncing' : 'Sync'}
              </Text>
            </TouchableOpacity>

            {/* Online/Offline Toggle */}
            <TouchableOpacity
              onPress={toggleOfflineMode}
              style={{
                width: 50,
                height: 28,
                borderRadius: 14,
                backgroundColor: forceOfflineMode ? '#ccc' : '#4CAF50',
                justifyContent: 'center',
                padding: 2,
              }}
              activeOpacity={0.8}
            >
              <View style={{
                width: 24,
                height: 24,
                borderRadius: 12,
                backgroundColor: '#fff',
                alignSelf: forceOfflineMode ? 'flex-start' : 'flex-end',
                shadowColor: '#000',
                shadowOpacity: 0.3,
                shadowRadius: 2,
                shadowOffset: { width: 0, height: 1 },
                elevation: 3,
              }} />
            </TouchableOpacity>
          </View>

          {/* Top bar with country dropdown */}
          {CountryDropdown}

          {/* Country Selection Modal */}
          {CountryModal}

          {/* SyncModal */}
          <SyncModal visible={syncModalVisible} onClose={() => setSyncModalVisible(false)} message={syncModalMessage} />

          {/* App Update Modal */}
          <VersionUpdateModal
            visible={showUpdateModal}
            info={updateInfo}
            onUpdate={handleUpdateNow}
            onRemindLater={handleRemindLater}
          />

          {/* Render Categories Dynamically */}
          {renderCategory('Master Data', 'masterdatakpi')}
          {renderCategory('Inspection and Compliances', 'InspectionKpi')}
          {renderCategory('Maintenance & Premium Services', 'MaintenanceKpiHome')}
          {renderCategory('Logbook', 'LogbookKPI')}
          {renderCategory('HR', 'HrModuleKpI')}

          {/* Quick Access: Speech to Text button */}
          {/* <View style={{ width: '100%', alignItems: 'center', marginTop: 12, marginBottom: 18 }}>
            <TouchableOpacity
              style={[styles.initializeButton, { flexDirection: 'row', alignItems: 'center', paddingHorizontal: 18 }]}
              onPress={() => handlePress('SpeechToText', 'SpeechToText')}
              activeOpacity={0.8}
            >
              <MaterialIcons name="keyboard-voice" size={20} color="#fff" />
              <Text style={[styles.initializeButtonText, { marginLeft: 10, fontWeight: '700' }]}>Speech To Text</Text>
            </TouchableOpacity>
          </View>

          <View style={{ width: '100%', alignItems: 'center', marginTop: 12, marginBottom: 18 }}>
            <TouchableOpacity
              style={[styles.initializeButton, { flexDirection: 'row', alignItems: 'center', paddingHorizontal: 18, backgroundColor: '#0055A5' }]}
              onPress={() => handlePress('UploadEquipmentToMongo', 'UploadEquipmentToMongo')}
              activeOpacity={0.8}
            >
              <MaterialIcons name="cloud-upload" size={20} color="#fff" />
              <Text style={[styles.initializeButtonText, { marginLeft: 10, fontWeight: '700' }]}>Equipment Mongo</Text>
            </TouchableOpacity>
          </View> */}
          {/* <HomaUploadbtn
            onUploadEquipment={handleUploadEquipmentSeedData}
            onUploadUsers={handleUploadUsersSeedData}
            loadingEquipment={isUploadingEquipment}
            loadingUsers={isUploadingUsers}
          /> */}
        </View>
      </ScrollView>
      <Toast />
    </>
  );
});

const styles = StyleSheet.create({
  scrollContainer: {
    flex: 1,
    backgroundColor: '#fff',
    width: '100%',
  },
  container: {
    paddingHorizontal: 5,
    paddingVertical: 10,
    alignItems: 'center',
    width: '100%',
  },
  heading: {
    fontSize: 24,
    fontWeight: 'bold',
    textAlign: 'center',
    marginBottom: 10,
    color: '#015185',
    alignSelf: 'center',
  },
  buttonContainer: {
    alignItems: 'center',
    marginVertical: 10,
    width: '100%',
  },
  gradientButton: {
    width: 160,
    borderRadius: 5,
    overflow: 'hidden',
    alignSelf: 'center',
  },
  gradientButtonContent: {
    flexDirection: 'row',
    justifyContent: 'center',
    alignItems: 'center',
    height: 40,
  },
  buttonText: {
    color: 'white',
    fontSize: 16,
    fontWeight: 'bold',
    marginRight: 5,
    textAlign: 'center',
    alignSelf: 'center',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 24,
    height: 24,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  updateModalBackdrop: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.45)',
    justifyContent: 'center',
    alignItems: 'center',
    paddingHorizontal: 20,
  },
  updateModalCard: {
    width: '100%',
    maxWidth: 360,
    borderRadius: 16,
    backgroundColor: '#fff',
    overflow: 'hidden',
    shadowColor: '#000',
    shadowOpacity: 0.2,
    shadowRadius: 12,
    shadowOffset: { width: 0, height: 8 },
    elevation: 10,
  },
  updateModalHeader: {
    width: '100%',
    justifyContent: 'center',
    elevation: 6,
    shadowColor: '#002a55',
    shadowOpacity: Platform.OS === 'ios' ? 0.35 : 0.2,
    shadowRadius: 12,
    shadowOffset: { width: 0, height: 6 },
  },
  updateModalHeaderContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 24,
    paddingTop: Platform.OS === 'ios' ? 34 : 18,
    paddingBottom: 18,
    height: '100%',
  },
  updateModalHeaderIconWrapper: {
    position: 'absolute',
    left: 24,
    padding: 8,
    top:10,
  },
  updateModalHeaderTextContainer: {
    flex: 1,
    alignItems: 'center',
  },
  updateModalTitle: {
    color: '#fff',
    fontSize: 18,
    fontWeight: '700',
    marginBottom: 2,
  },
  updateModalSubtitle: {
    color: '#e3ecf7',
    fontSize: 12,
    lineHeight: 16,
    textAlign: 'center',
  },
  updateModalBody: {
    paddingHorizontal: 20,
    paddingVertical: 18,
    backgroundColor: '#f8fafc',
  },
  updateVersionRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 8,
  },
  updateVersionLabel: {
    color: '#4a5d78',
    fontSize: 13,
    fontWeight: '600',
  },
  updateVersionLabelLatest: {
    color: '#0055A5',
  },
  updateVersionValue: {
    color: '#1f2a44',
    fontSize: 13,
    fontWeight: '600',
  },
  updateVersionValueLatest: {
    color: '#0055A5',
  },
  updateMandatoryBadge: {
    alignSelf: 'flex-start',
    backgroundColor: '#ffd205',
    borderRadius: 12,
    paddingHorizontal: 10,
    paddingVertical: 4,
    marginTop: 4,
    marginBottom: 12,
  },
  updateMandatoryText: {
    color: '#0055A5',
    fontSize: 11,
    fontWeight: '700',
    letterSpacing: 0.2,
  },
  updateNotesContainer: {
    marginTop: 8,
    backgroundColor: '#fff',
    borderRadius: 12,
    padding: 12,
  },
  updateNotesTitle: {
    color: '#0055A5',
    fontSize: 13,
    fontWeight: '700',
    marginBottom: 6,
  },
  updateNotesItem: {
    flexDirection: 'row',
    alignItems: 'flex-start',
    marginBottom: 6,
  },
  updateNotesBullet: {
    width: 6,
    height: 6,
    borderRadius: 3,
    backgroundColor: '#0055A5',
    marginTop: 6,
    marginRight: 8,
  },
  updateNotesText: {
    flex: 1,
    color: '#1f2a44',
    fontSize: 12,
    lineHeight: 16,
  },
  updateModalActions: {
    paddingHorizontal: 20,
    paddingBottom: 20,
    paddingTop: 12,
    backgroundColor: '#fff',
  },
  updatePrimaryButton: {
    borderRadius: 8,
    overflow: 'hidden',
  },
  updatePrimaryButtonGradient: {
    height: 46,
    alignItems: 'center',
    justifyContent: 'center',
  },
  updatePrimaryButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: '700',
    letterSpacing: 0.3,
  },
  updateSecondaryButton: {
    marginTop: 12,
    height: 44,
    borderRadius: 8,
    borderWidth: 2,
    borderColor: '#ffd205',
    backgroundColor: '#fff',
    alignItems: 'center',
    justifyContent: 'center',
  },
  updateSecondaryButtonText: {
    color: '#0055A5',
    fontSize: 15,
    fontWeight: '700',
  },
  initializeButton: {
    backgroundColor: '#035484',
    borderRadius: 6,
    paddingVertical: 8,
    paddingHorizontal: 15,
    marginLeft: 10,
  },
  initializeButtonText: {
    color: '#fff',
    fontSize: 14,
    fontWeight: 'bold',
  },
});

export default HomeScreen;
