# 🚨 Incident Report Module - Offline & MongoDB Sync

## Overview
The Incident Report module supports **full offline functionality** with manual MongoDB synchronization. Create, view, and manage incident reports with or without internet connection.

---

## 🎯 Key Features

✅ **Full Offline Mode** - Create incident reports without internet  
✅ **Auto Request Numbers** - Sequential numbering based on MongoDB count  
✅ **PDF Storage** - Local PDFs viewable offline  
✅ **Manual Sync Control** - User controls when to sync (no auto-sync)  
✅ **Duplicate Prevention** - Smart deduplication by report ID  
✅ **Image Upload** - Photos stored locally and synced to MongoDB  

---

## 🔄 How It Works

### **1. Online Mode (Internet Available)**

```
User Creates Report → MongoDB → Assigned Request Number
                                 ↓
                         DRC-MUS-INC-001
                                 ↓
                    Cached Locally for Offline Access
```

**Request Number Format:** `{COUNTRY}-{PROJECT}-INC-{NUMBER}`
- Example: `DRC-MUS-INC-001`, `ZAM-KAL-INC-002`
- Number is based on **total count in MongoDB**

### **2. Offline Mode (No Internet)**

```
User Creates Report → Offline Storage → Temporary Request Number
                                        ↓
                                DRC-MUS-INC-001-OFFLINE
                                        ↓
                               Local PDF Generated
                                        ↓
                            Stored in AsyncStorage
```

**Features:**
- Reports saved with `-OFFLINE` suffix
- PDFs stored locally using React Native FS
- Images stored locally (not uploaded yet)
- Yellow **SYNC** badge shown on report
- Viewable offline with all details

### **3. Sync Process (Manual)**

```
User Clicks Cloud Button → Fetch MongoDB Total Count
                                    ↓
                            Calculate Next Number
                                    ↓
                         Remove -OFFLINE Suffix
                                    ↓
                    Create Report in MongoDB with Clean Number
                         (DRC-MUS-INC-002)
                                    ↓
                         Upload Local PDF to MongoDB
                                    ↓
                         Upload Images to MongoDB
                                    ↓
                         Remove from Offline Storage
                                    ↓
                          UI Refreshes Automatically
                           (SYNC badge removed)
```

---

## 📁 Storage Architecture

### **AsyncStorage Keys**

| Key | Purpose | Format |
|-----|---------|--------|
| `incidentReports` | Cached reports from MongoDB (WITH permissions) | Array of reports with permissions |
| `offlineIncidentReports` | Reports created offline (pending sync) | Array of offline reports with permissions |
| `offlineIncidentPDFs` | Local PDF file paths | `{ reportId: { pdfPath, savedAt } }` |
| `incidentModuleUserInfo` | User permissions & access for incident module | `{ countries, projects, incidentReportPermissions }` |
| `incidentModulePermissions` | Just permissions array | Array of permission strings |
| `IncidentReportStep1Data` | Temporary step 1 data | Report details object |
| `IncidentReportStep2State` | Temporary step 2 state | Images & descriptions |

### **Offline Report Structure**

```javascript
{
  id: "DRC-MUS-INC-001-OFFLINE",
  _id: "DRC-MUS-INC-001-OFFLINE",
  incidentNumber: "DRC-MUS-INC-001-OFFLINE",
  incidentDate: "2025-10-29T20:12:00.000Z",
  country: "DRC",
  project: "Musompo",
  incidentArea: "Workshop",
  severity: "High",
  incidentCategory: "Equipment",
  vehicleId: "TD-001",
  vehicleName: "Drill Rig",
  personInvolved: "John Doe",
  coordinates: "-12.345, 28.678",
  gpsAddress: "Workshop Area, Musompo Site",
  selectedIncidents: {
    "Equipment Failure": true,
    "Oil Spill": true
  },
  step1: { /* step 1 data */ },
  step2: {
    pdfLocalPath: "/data/.../Incident_Report_DRC-MUS-INC-001-OFFLINE.pdf",
    incidentImages: ["file:///...", "file:///..."],
    incidentCause: "Equipment malfunction",
    equipmentDamage: "Hydraulic hose burst",
    additionalComment: "Immediate shutdown",
    immediateCorrectiveAction: "Replaced hose"
  },
  offlineCreated: true,
  offlineSyncPending: true,
  adminUid: "user_123",
  createdAt: "2025-10-29T20:12:00.000Z",
  // CRITICAL: Permissions stored with each report for offline access
  userPermissions: ["create", "view", "delete"],
  userCountries: ["DRC", "Zambia"],
  userProjects: ["Musompo", "Lubumbashi", "Kalumbila"],
  syncedAt: "2025-10-29T20:15:00.000Z"
}
```

---

## 🔐 Permissions & Offline Access

### **Permission Types**
- `create` - Create new incident reports
- `view` - View all incident reports (filtered by country/project)
- `onlyMineView` - View only own incident reports
- `edit` - Edit incident reports
- `delete` - Delete incident reports

### **Offline Permission Caching**

Permissions are stored in **3 locations** for reliability:

1. **`incidentModuleUserInfo`** - Dedicated incident module permissions
2. **`incidentModulePermissions`** - Just the permissions array
3. **With each cached report** - Embedded in report objects

```javascript
// Permission loading priority:
// 1. Try incidentModuleUserInfo (most specific)
// 2. Try cached reports (fallback)
// 3. Fetch from MongoDB if online

const cachedIncidentUserInfo = await AsyncStorage.getItem('incidentModuleUserInfo');
// Contains: { countries, projects, incidentReportPermissions, employeeNumber, name }
```

### **Each Report Stores Permissions**

```javascript
const report = {
  // ... report data ...
  userPermissions: ["create", "view"],
  userCountries: ["DRC", "Zambia"],
  userProjects: ["Musompo", "Lubumbashi"],
  syncedAt: "2025-10-29T20:00:00.000Z"
};
```

**Offline Benefits:**
- User can view reports based on their permissions
- Country/project filtering works offline
- No need to re-authenticate
- Create permission works offline
- Delete permission works offline (queued)

---

## 🚀 Creating an Offline Incident Report

### **Step 1: Create Incident Report (Step 1)**

```javascript
// User clicks "Create Incident Report" button
// Works online OR offline

// Select Country & Project
// System auto-generates incident number:
// - Online: DRC-MUS-INC-002 (based on MongoDB count)
// - Offline: DRC-MUS-INC-001-OFFLINE (temporary)

// Fill details:
- Incident Date & Time
- Incident Category
- Incident Area
- Severity Rating
- Type of Event (checkboxes)
- Equipment involvement
- GPS Location
```

### **Step 2: Add Photos (Step 3 in UI)**

```javascript
// Click Incident Images
// Take photos or select from gallery
// Images saved locally in incidentImages array
// Up to 9 images supported

// Offline mode: Images stored locally
// Online mode: Images uploaded to MongoDB immediately
```

### **Step 3: Add Details (Step 2 in UI)**

```javascript
// Enter text details:
- Description of incident
- Details of damage/injury/loss
- Possible direct causes
- Immediate corrective action taken

// Offline mode: All data saved locally
```

### **Step 4: Generate Report**

```javascript
// ONLINE MODE:
const pdfPath = await generatePDFWithHtml(reportData);
await uploadImagesToServer(imageUris, reportId);
await saveReportToMongoDB(userUid, report, pdfUrl);

// OFFLINE MODE:
const pdfPath = await generatePDFWithHtml(reportData);
await OfflineIncidentHelper.saveOfflineReport(report);
await OfflineIncidentHelper.saveOfflinePDF(reportId, pdfPath);

// Result: Report appears with yellow SYNC badge (offline)
```

---

## 🔄 Syncing Offline Reports

### **Manual Sync Process**

1. **User clicks Cloud button** (top-right corner)
2. **System checks for pending offline reports**
3. **For each offline report:**
   - Removes `-OFFLINE` suffix
   - Fetches MongoDB total count
   - Generates next sequential number
   - Creates report in MongoDB
   - Uploads local PDF
   - Uploads local images
   - Updates report with URLs
   - Removes from offline storage
4. **UI auto-refreshes**
   - SYNC badges disappear
   - Reports show as synced
   - Counts update

### **Sync Code Flow**

```javascript
// 1. Remove -OFFLINE suffix
let newRequestNumber = report.incidentNumber.replace('-OFFLINE', '');

// 2. Get MongoDB count and calculate next number
const response = await fetch('https://api.titandrillingzm.com:6007/incident-reports');
const result = await response.json();

const maxNum = Math.max(...result.data.map(r => 
  parseInt(r.incidentNumber.match(/\d+$/)[0], 10)
));

newRequestNumber = `DRC-MUS-INC-${String(maxNum + 1).padStart(3, '0')}`;
// Result: DRC-MUS-INC-002

// 3. Update report
report.incidentNumber = newRequestNumber;
report.id = newRequestNumber;

// 4. Create in MongoDB
await fetch('https://api.titandrillingzm.com:6007/incident-reports', {
  method: 'POST',
  headers: { 'Content-Type': 'application/json' },
  body: JSON.stringify(report)
});

// 5. Upload PDF
const formData = new FormData();
formData.append('pdf', {
  uri: report.step2.pdfLocalPath,
  type: 'application/pdf',
  name: `incident_${newRequestNumber}.pdf`
});

await fetch(`https://api.titandrillingzm.com:6007/incident-reports/${userId}/${newRequestNumber}/upload-pdf`, {
  method: 'POST',
  body: formData
});

// 6. Remove from offline storage (automatic)
// 7. Reload UI to remove SYNC badge
await fetchIncidentReports(false);
await updateReportCounts();
```

---

## 📊 Request Number Generation

### **Country Codes**
```javascript
const countryCodeMap = {
  'Zambia': 'ZAM',
  'DRC': 'DRC'
};
```

### **Project Codes**
```javascript
const projectCodeMap = {
  'Kobold': 'KOB',
  'Kalumbila': 'KAL',
  'Kansanshi': 'KAN',
  'Kimiteto': 'KIM',
  'Mimosa': 'MIM',
  'Musompo': 'MUS',
  'IME': 'IME'
};
```

### **Examples**
```
DRC-MUS-INC-001  → DRC, Musompo, Incident #1
ZAM-KAL-INC-002  → Zambia, Kalumbila, Incident #2
DRC-LUB-INC-003  → DRC, Lubumbashi, Incident #3
```

### **Offline Suffix**
```
DRC-MUS-INC-001-OFFLINE → Created offline
        ↓ (after sync)
DRC-MUS-INC-002 → Clean number in MongoDB
```

---

## 📱 UI Indicators

### **Offline Report Badges**

```javascript
{item.offlineSyncPending && (
  <View style={styles.offlineBadge}>
    <MaterialIcons name="cloud-upload" size={12} color="#fff" />
    <Text style={styles.offlineBadgeText}>SYNC</Text>
  </View>
)}
```

**Yellow SYNC Badge** = Report created offline, needs sync

### **Status Counter Cards**

| Card | Description | Tap Action |
|------|-------------|------------|
| **Offline Stored** | Total reports available offline (cached + pending) | Shows breakdown of cached vs pending |
| **Online in DB** | Reports synced to MongoDB | Shows sync metadata |
| **Storage** | Total AsyncStorage size | Shows storage details |

---

## 🛠️ Important Functions

### **OfflineIncidentHelper.js**

```javascript
// Save offline incident report
static async saveOfflineReport(report)

// Get all offline incident reports
static async getOfflineReports()

// Sync offline reports (with callback for each)
static async syncOfflineReports(uploadCallback)

// Save PDF locally
static async saveOfflinePDF(reportId, pdfPath)

// Get offline PDF path
static async getOfflinePDF(reportId)

// Clear all offline data
static async clearOfflineData()
```

### **IncidentReport.js (Home)**

```javascript
// Update offline/online counts
const updateReportCounts = async ()

// Fetch incident reports (cache + offline)
const fetchIncidentReports = async (showModal = false)

// Sync offline reports to MongoDB
const syncOfflineReports = async ()

// Manual cloud sync (user-triggered)
const handleCloudSync = async ()

// View PDF (online or offline)
const handleViewPDF = async (report)
```

### **CreateIncidentReport.js (Step 1)**

```javascript
// Generate incident number (online or offline)
const generateIncidentNumber = async (country, project)
// Returns: "DRC-MUS-INC-002" (online) or "DRC-MUS-INC-001-OFFLINE" (offline)

// Save incident data to AsyncStorage
const saveIncidentReportToAsyncStorage = async ()
```

### **IncidentReportStep2.js (Steps 2 & 3)**

```javascript
// Generate offline report (local save)
const generateOfflineReport = async ()

// Generate PDF from HTML
const generatePDFWithHtml = async (report)

// Upload images to server
const uploadImagesToServer = async (imageUris, reportId)

// Save report to MongoDB
const saveReportToMongoDB = async (userUid, report, pdfDownloadUrl)

// Update report with files
const updateReportWithFiles = async (userUid, incidentNumber, imageUrls, pdfDownloadUrl)
```

---

## 🚫 What's Disabled (No Auto-Sync)

### **Removed Auto-Sync Triggers:**

❌ **No sync on screen focus** - Prevents duplicate uploads  
❌ **No sync on app load** - User controls when to sync  
❌ **No sync on connection restore** - Prevents accidental syncs  

### **Why Manual Sync?**

✅ **User control** - Sync only when ready  
✅ **No duplicates** - Single sync operation per click  
✅ **Network efficiency** - Sync during good connection  
✅ **Battery saving** - No background sync operations  

---

## 📝 Example Workflow

### **Scenario: Field Incident (No Internet)**

1. **2:00 PM** - Safety officer at remote site (no internet)
2. **2:05 PM** - Incident occurs (Equipment failure)
3. **2:10 PM** - Opens app, clicks "Create Incident Report"
4. **2:15 PM** - Fills Step 1:
   - Country: DRC
   - Project: Musompo
   - Gets GPS location
   - Incident number: `DRC-MUS-INC-001-OFFLINE`
5. **2:25 PM** - Takes 3 photos of incident (Step 3)
6. **2:35 PM** - Fills details (Step 2):
   - Description of incident
   - Damage details
   - Immediate action taken
7. **2:40 PM** - Clicks "Generate Report"
   - PDF generated locally
   - Saved to AsyncStorage
   - Yellow SYNC badge appears
8. **3:00 PM** - Returns to office (internet available)
9. **3:05 PM** - Opens Incident Reports list
10. **3:06 PM** - Sees report with SYNC badge
11. **3:07 PM** - Clicks **Cloud button**
12. **3:08 PM** - Report synced → `DRC-MUS-INC-002`
    - PDF uploaded to MongoDB
    - Images uploaded to server
    - SYNC badge disappears
13. **3:09 PM** - Report accessible to all authorized users

**Result:**
- Report in MongoDB with clean number
- PDF URL: `https://api.titandrillingzm.com:6007/pdfs/incident_DRC-MUS-INC-002.pdf`
- Images URL: `https://api.titandrillingzm.com:6007/uploads/...`
- No `-OFFLINE` suffix
- Available for viewing by other users

---

## 🎯 Incident Report Steps

### **Step 1: Incident Details**

| Field | Required | Description |
|-------|----------|-------------|
| Incident Number | ✅ | Auto-generated (online/offline) |
| Incident Date | ✅ | Date & time picker |
| Country | ✅ | Zambia or DRC |
| Project | ✅ | Based on country |
| Incident Category | ✅ | Personnel / Equipment / Environmental / Other |
| Incident Area | ✅ | Free text |
| Person Involved | ❌ | Free text |
| Severity Rating | ❌ | High / Medium / Low |
| GPS Location | ❌ | Auto-fetch coordinates & address |
| Involves Equipment | ✅ | Yes/No checkbox |
| Equipment Details | ❌ | If equipment involved |
| Type of Event | ✅ | Multi-select checkboxes |

### **Step 2: Description (Step 2 in UI)**

| Field | Required | Description |
|-------|----------|-------------|
| Description of incident | ❌ | Multi-line text |
| Details of damage/injury/loss | ❌ | Multi-line text |
| Possible direct causes | ❌ | Multi-line text |
| Immediate corrective action | ❌ | Multi-line text |

### **Step 3: Images (Step 3 in UI)**

- Up to 9 incident photos
- Camera or gallery selection
- Stored locally (offline) or uploaded (online)
- Displayed in PDF report

---

## 🔑 Key Differences from Inspection Reports

| Feature | Incident Report | Inspection Report |
|---------|----------------|-------------------|
| **Number Prefix** | `-INC-` | `-IR-` |
| **Images** | Up to 9 photos | Checklist-based images |
| **Offline Storage** | `offlineIncidentReports` | `offlineReports` |
| **PDF Helper** | `OfflineIncidentHelper` | `OfflineDataHelper` |
| **Severity** | High/Medium/Low | Equipment status |
| **API Endpoint** | `:5007/incident-reports` | `:5005/inspections` |

---

## 📚 Related Files

| File | Purpose |
|------|---------|
| `IncidentReport.js` | Main listing, sync, offline logic |
| `OfflineIncidentHelper.js` | Offline storage utilities |
| `CreateIncidentReport.js` | Step 1 - Incident details |
| `IncidentReportStep2.js` | Steps 2 & 3 - Details & photos, PDF generation |
| `IncidentReportHeader.js` | Header navigation |

---

## 🐛 Troubleshooting

### **SYNC Badge Not Disappearing**
- Press **Cloud button** to sync
- UI auto-refreshes after successful sync
- Check console logs for sync errors

### **Request Numbers Not Sequential**
- Request numbers based on **MongoDB total count**
- If count is 5, next report is #6
- Offline suffix removed during sync

### **Images Not Uploading**
- Check local image paths in report
- Ensure internet connection during sync
- Check server logs for upload errors

### **PDF Not Viewable Offline**
- Check `step2.pdfLocalPath` exists in offline report
- Verify PDF file exists on device
- Use OfflineIncidentHelper.getOfflinePDF(reportId)

---

## 🎯 Best Practices

### **For Users**

1. **Create reports offline** - Works anywhere in the field
2. **Take photos immediately** - Before leaving incident site
3. **Sync regularly** - Click cloud button when back online
4. **Check SYNC badges** - Yellow badge = needs sync
5. **Verify upload** - SYNC badge should disappear after sync

### **For Developers**

1. **Always check network status** - Before MongoDB operations
2. **Use MongoDB count** - For sequential numbering
3. **Clean offline suffix** - Remove `-OFFLINE` before MongoDB
4. **Deduplicate reports** - Use Map by report ID
5. **Manual sync only** - No auto-sync triggers
6. **Refresh UI after sync** - Remove SYNC badges
7. **Handle images properly** - Local storage then upload
8. **Validate PDF paths** - Support local and server URLs

---

## 🔑 Key Takeaways

✅ **Offline-First Design** - Works without internet  
✅ **Manual Sync Control** - User decides when to sync  
✅ **Sequential Numbering** - Based on MongoDB count  
✅ **Clean Request Numbers** - No `-OFFLINE` in MongoDB  
✅ **Image Support** - Local photos synced to server  
✅ **Local PDF Generation** - HTML to PDF conversion  
✅ **Automatic UI Updates** - SYNC badges removed after sync  
✅ **Duplicate Prevention** - Smart deduplication by ID  

---

## ⚠️ Important Notes

### **Network Handling**

```javascript
// Always check network before sync
const netInfo = await NetInfo.fetch();
const isOnline = netInfo.isConnected && netInfo.isInternetReachable !== false;

if (!isOnline) {
  // Save offline
  await generateOfflineReport();
} else {
  // Save to MongoDB
  await saveReportToMongoDB();
}
```

### **Image Storage**

**Offline:**
```javascript
incidentImages: [
  "file:///data/user/0/.../image1.jpg",
  "file:///data/user/0/.../image2.jpg"
]
```

**After Sync:**
```javascript
incidentImages: [
  "https://api.titandrillingzm.com:6007/uploads/incident_img_1.jpg",
  "https://api.titandrillingzm.com:6007/uploads/incident_img_2.jpg"
]
```

### **PDF Storage**

**Offline:**
```javascript
pdfLocalPath: "/data/user/0/.../Incident_Report_DRC-MUS-INC-001-OFFLINE.pdf"
```

**After Sync:**
```javascript
pdfDownloadUrl: "https://api.titandrillingzm.com:6007/pdfs/incident_DRC-MUS-INC-002.pdf"
```

---

**Last Updated:** October 29, 2025  
**Module:** Incident Report  
**Storage:** AsyncStorage + MongoDB  
**PDF Storage:** React Native FS (offline) + MongoDB (online)  
**Image Storage:** Local FS (offline) + Server (online)

