import React, { useState, useEffect, useRef, useCallback } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  Dimensions,
  Platform,
  StatusBar,
  Modal,
  Animated,
  BackHandler,
} from 'react-native';
import Pdf from 'react-native-pdf';
import LinearGradient from 'react-native-linear-gradient';
import RNFS from 'react-native-fs';
import Toast from 'react-native-toast-message';
import { PermissionsAndroid, Share } from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import ReactNativeBlobUtil from 'react-native-blob-util'; // Import react-native-blob-util
import { useFocusEffect, CommonActions } from '@react-navigation/native';

const { width, height } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.25;

const PdfviewPage = ({ route, navigation }) => {
  const { pdfUrl, userId, onGoBack } = route.params || {};
  const [localPdfPath, setLocalPdfPath] = useState(null);
  const [loading, setLoading] = useState(false);
  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  const isNavigatingRef = useRef(false);
  const canNavigateBackRef = useRef(false);

  const navigateToInspectionHome = useCallback(() => {
    if (isNavigatingRef.current || canNavigateBackRef.current) {
      return;
    }
    isNavigatingRef.current = true;
    canNavigateBackRef.current = true;
    // Reset navigation stack with MainApp and Inspection_Report_Home
    navigation.dispatch(
      CommonActions.reset({
        index: 1,
        routes: [
          { name: 'MainApp' },
          { name: 'Inspection_Report_Home' },
        ],
      })
    );
  }, [navigation]);

  useFocusEffect(
    useCallback(() => {
      isNavigatingRef.current = false;
      canNavigateBackRef.current = false;

      const backHandler = BackHandler.addEventListener('hardwareBackPress', () => {
        if (canNavigateBackRef.current) {
          return false; // Allow default back behavior
        }
        navigateToInspectionHome();
        return true;
      });

      const unsubscribe = navigation.addListener('beforeRemove', (e) => {
        // If navigation is already in progress or allowed, don't prevent it
        if (canNavigateBackRef.current || isNavigatingRef.current) {
          return; // Allow navigation
        }
        // Prevent default behavior
        e.preventDefault();
        // Navigate to Inspection_Report_Home
        navigateToInspectionHome();
      });

      return () => {
        isNavigatingRef.current = false;
        canNavigateBackRef.current = false;
        backHandler.remove();
        unsubscribe();
      };
    }, [navigation, navigateToInspectionHome])
  );

  useEffect(() => {
    console.log('📄 PDF URL received:', pdfUrl); // Debug the URL
    console.log('📄 PDF URL type:', typeof pdfUrl);
    console.log('📄 PDF URL length:', pdfUrl ? pdfUrl.length : 'null');
    
    if (pdfUrl) {
      // Check if it's a local file path
      const isLocalPath = pdfUrl.startsWith('file://') || 
                         pdfUrl.startsWith(RNFS.DocumentDirectoryPath) ||
                         pdfUrl.startsWith(RNFS.CachesDirectoryPath) ||
                         pdfUrl.startsWith(RNFS.TemporaryDirectoryPath) ||
                         (!pdfUrl.startsWith('http://') && !pdfUrl.startsWith('https://') && !pdfUrl.startsWith('/'));
      
      if (isLocalPath) {
        // LOCAL FILE PATH: Use directly without downloading
        console.log('📱 Detected local PDF path:', pdfUrl);
        
        // Clean up the path (remove file:// prefix if present for Android)
        let cleanPath = pdfUrl;
        if (pdfUrl.startsWith('file://')) {
          cleanPath = pdfUrl.replace('file://', '');
        }
        
        // Verify file exists
        RNFS.exists(cleanPath)
          .then((exists) => {
            if (exists) {
              console.log('✅ Local PDF file exists:', cleanPath);
              setLocalPdfPath(cleanPath);
            } else {
              console.error('❌ Local PDF file not found:', cleanPath);
              Toast.show({
                type: 'error',
                text1: 'Error',
                text2: 'Local PDF file not found.',
                position: 'bottom',
                visibilityTime: 3000,
              });
            }
          })
          .catch((error) => {
            console.error('❌ Error checking local PDF file:', error);
            Toast.show({
              type: 'error',
              text1: 'Error',
              text2: 'Failed to access local PDF file.',
              position: 'bottom',
              visibilityTime: 3000,
            });
          });
        return;
      }
      
      // ONLINE URL: Handle HTTP/HTTPS and relative MongoDB URLs
      if (pdfUrl.startsWith('http://') || pdfUrl.startsWith('https://')) {
        // Handle full URLs - check if it's a corrective action URL and add userId if missing
        if (pdfUrl.includes('/corrective-actions/download-pdf/') && !pdfUrl.includes('userId=')) {
          const separator = pdfUrl.includes('?') ? '&' : '?';
          const fullUrl = `${pdfUrl}${separator}userId=${userId || 'test_proadmin_amit_001'}`;
          console.log('🌐 Adding userId to corrective action URL:', fullUrl);
          downloadPdfForViewing(fullUrl);
        } else if (pdfUrl.includes('https://api.titandrillingzm.com:6006/reports/download-file/')) {
          // Handle DVIR new format full URLs (port 5006)
          console.log('🌐 Processing DVIR new format PDF URL:', pdfUrl);
          downloadPdfForViewing();
        } else if (pdfUrl.includes('https://api.titandrillingzm.com:6006/download/')) {
          // Handle DVIR legacy full URLs (port 5006)
          console.log('🌐 Processing DVIR legacy PDF URL:', pdfUrl);
          downloadPdfForViewing();
        } else if (pdfUrl.includes('https://api.titandrillingzm.com:6006/dvir-corrective-actions/download-file/')) {
          // Handle DVIR corrective action full URLs (port 5006)
          console.log('🌐 Processing DVIR corrective action PDF URL:', pdfUrl);
          downloadPdfForViewing();
        } else {
          downloadPdfForViewing();
        }
      } else if (pdfUrl.startsWith('/inspections/download-file/')) {
        // Convert MongoDB relative URL to full URL
        const fullUrl = `https://api.titandrillingzm.com:6004${pdfUrl}`;
        console.log('🌐 Converting MongoDB URL to full URL:', fullUrl);
        downloadPdfForViewing(fullUrl);
      } else if (pdfUrl.startsWith('/corrective-actions/download-pdf/')) {
        // Convert corrective action relative URL to full URL (old format)
        let fullUrl = `https://api.titandrillingzm.com:6004${pdfUrl}`;
        // Add userId parameter if not present (for backward compatibility)
        if (!fullUrl.includes('userId=')) {
          fullUrl += `?userId=${userId || 'test_proadmin_amit_001'}`;
        }
        console.log('🌐 Converting corrective action URL to full URL:', fullUrl);
        downloadPdfForViewing(fullUrl);
      } else if (pdfUrl.startsWith('/dvir-corrective-actions/download-file/')) {
        // Convert DVIR corrective action relative URL to full URL (new format - port 5006)
        const fullUrl = `https://api.titandrillingzm.com:6006${pdfUrl}`;
        console.log('🌐 Converting DVIR corrective action URL to full URL:', fullUrl);
        downloadPdfForViewing(fullUrl);
      } else if (pdfUrl.startsWith('/jobcards/download-file/')) {
        // Convert Job Card relative URL to full URL
        const fullUrl = `https://api.titandrillingzm.com:6005${pdfUrl}`;
        console.log('🌐 Converting Job Card URL to full URL:', fullUrl);
        downloadPdfForViewing(fullUrl);
      } else if (pdfUrl.startsWith('/reports/download-file/')) {
        // Convert DVIR new format relative URL to full URL (port 5006)
        const fullUrl = `https://api.titandrillingzm.com:6006${pdfUrl}`;
        console.log('🌐 Converting DVIR new format URL to full URL:', fullUrl);
        downloadPdfForViewing(fullUrl);
      } else if (pdfUrl.startsWith('/download/')) {
        // Convert DVIR legacy relative URL to full URL (port 5006)
        const fullUrl = `https://api.titandrillingzm.com:6006${pdfUrl}`;
        console.log('🌐 Converting DVIR legacy URL to full URL:', fullUrl);
        downloadPdfForViewing(fullUrl);
      } else {
        console.error('❌ Invalid PDF URL format:', pdfUrl);
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: 'Invalid PDF URL format.',
          position: 'bottom',
          visibilityTime: 3000,
        });
      }
    }
    
    return () => {
      // Only clean up temp downloaded files, not local PDFs that may be needed offline
      if (localPdfPath && localPdfPath.includes('/temp_report_')) {
        RNFS.unlink(localPdfPath).catch((err) => console.log('Error deleting temp file:', err));
      }
    };
  }, [pdfUrl]);

  useEffect(() => {
    if (loading) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [loading]);

  const requestStoragePermission = async () => {
    if (Platform.OS !== 'android') return true;

    try {
      // Android 13+ doesn't require permissions for Downloads folder
      if (Platform.Version >= 33) {
        return true; // No permission needed for Downloads on Android 13+
      } else {
        const granted = await PermissionsAndroid.request(
          PermissionsAndroid.PERMISSIONS.WRITE_EXTERNAL_STORAGE,
          {
            title: 'Storage Permission Required',
            message: 'This app needs access to your storage to download the PDF.',
            buttonNeutral: 'Ask Me Later',
            buttonNegative: 'Cancel',
            buttonPositive: 'OK',
          }
        );
        return granted === PermissionsAndroid.RESULTS.GRANTED;
      }
    } catch (err) {
      console.warn('Permission request error:', err);
      return false;
    }
  };

  const downloadPdfForViewing = async (urlToDownload = null) => {
    const url = urlToDownload || pdfUrl;
    
    console.log('📥 Downloading PDF from URL:', url);
    console.log('🔍 URL ends with slash:', url ? url.endsWith('/') : 'null');
    
    // Remove trailing slash if present
    const cleanUrl = url ? url.replace(/\/$/, '') : url;
    console.log('🧹 Clean URL:', cleanUrl);
    
    if (!cleanUrl) {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'No PDF URL found.',
        position: 'bottom',
        visibilityTime: 3000,
      });
      return;
    }

    if (!cleanUrl.startsWith('http://') && !cleanUrl.startsWith('https://')) {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Invalid PDF URL.',
        position: 'bottom',
        visibilityTime: 3000,
      });
      return;
    }

    setLoading(true);
    try {
      // First, test if the URL is accessible with a HEAD request
      console.log('🔍 Testing URL accessibility:', cleanUrl);
      try {
        const testResponse = await fetch(cleanUrl, {
          method: 'HEAD',
          timeout: 10000, // 10 second timeout
        });
        
        console.log('🧪 URL test response status:', testResponse.status);
        
        if (testResponse.status === 404) {
          throw new Error(`PDF file not found on server. The file may have been moved or deleted. Status: ${testResponse.status}`);
        } else if (!testResponse.ok) {
          throw new Error(`Server error: ${testResponse.status} ${testResponse.statusText}`);
        }
        
        console.log('✅ URL is accessible, proceeding with download');
      } catch (testError) {
        console.error('URL accessibility test failed:', testError);
        if (testError.message.includes('404')) {
          throw new Error('PDF file not found on server. Please try regenerating the report.');
        } else if (testError.message.includes('Network')) {
          throw new Error('Network error. Please check your internet connection.');
        } else {
          throw new Error(`Cannot access PDF URL: ${testError.message}`);
        }
      }

      const hasPermission = await requestStoragePermission();
      if (!hasPermission) {
        throw new Error('Storage permission denied');
      }

      const tempPath = `${RNFS.DocumentDirectoryPath}/temp_report_${Date.now()}.pdf`;
      console.log('💾 Starting download to:', tempPath);
      console.log('🌐 Downloading from URL:', cleanUrl);

      const options = {
        fromUrl: cleanUrl,
        toFile: tempPath,
        headers: {
          'User-Agent': 'Mozilla/5.0 (compatible; ReactNative/1.0)',
        },
        connectionTimeout: 15000, // 15 second timeout
        readTimeout: 15000,
      };
      
      const result = await RNFS.downloadFile(options).promise;
      console.log('📊 Download result:', { statusCode: result.statusCode, bytesWritten: result.bytesWritten });
      
      if (result.statusCode === 200) {
        setLocalPdfPath(tempPath);
        console.log('PDF downloaded successfully to:', tempPath);
      } else {
        throw new Error(`Failed to download PDF. Server response: ${result.statusCode}`);
      }
    } catch (error) {
      console.error(' Error downloading PDF for viewing:', error);
      console.error('Full error details:', {
        message: error.message,
        name: error.name,
        stack: error.stack
      });
      
      let errorMessage = 'Failed to download PDF for viewing.';
      
      if (error.message.includes('404') || error.message.includes('not found')) {
        errorMessage = 'PDF file not found on server. Please try regenerating the report.';
      } else if (error.message.includes('permission denied')) {
        errorMessage = 'Storage permission denied. Please grant permission to view the PDF.';
      } else if (error.message.includes('Network') || error.message.includes('timeout')) {
        errorMessage = 'Network error. Please check your connection and try again.';
      } else if (error.message.includes('Cannot access PDF URL')) {
        errorMessage = error.message;
      } else {
        errorMessage = `Download failed: ${error.message}`;
      }
      
      console.error('Showing error to user:', errorMessage);
      
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: errorMessage,
        position: 'bottom',
        visibilityTime: 5000,
      });
      
      // Don't automatically go back, let user see the error
      // navigation.goBack();
    } finally {
      setLoading(false);
    }
  };

  const handleDownloadPDF = async () => {
    // If we already have a local PDF path, share/copy it directly
    if (localPdfPath) {
      try {
        const hasPermission = await requestStoragePermission();
        if (!hasPermission && Platform.OS === 'android') {
          throw new Error('Storage permission denied');
        }

        if (Platform.OS === 'android') {
          // On Android, copy to Downloads folder
          const fileName = `report_${Date.now()}.pdf`;
          const downloadDest = `${ReactNativeBlobUtil.fs.dirs.DownloadDir}/${fileName}`;
          
          // Clean path (remove file:// if present)
          const sourcePath = localPdfPath.startsWith('file://') 
            ? localPdfPath.replace('file://', '') 
            : localPdfPath;
          
          // Copy file to Downloads
          await RNFS.copyFile(sourcePath, downloadDest);
          
          Toast.show({
            type: 'success',
            text1: 'Success',
            text2: 'PDF saved to Downloads folder.',
            position: 'bottom',
            visibilityTime: 3000,
          });
        } else {
          // On iOS, share the file
          const sourcePath = localPdfPath.startsWith('file://') 
            ? localPdfPath.replace('file://', '') 
            : localPdfPath;
          
          await Share.share({
            url: `file://${sourcePath}`,
            message: 'Here is your PDF report.',
          });
          
          Toast.show({
            type: 'success',
            text1: 'Success',
            text2: 'PDF ready to share.',
            position: 'bottom',
            visibilityTime: 3000,
          });
        }
        return;
      } catch (error) {
        console.error('Error sharing local PDF:', error);
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: `Failed to save PDF: ${error.message}`,
          position: 'bottom',
          visibilityTime: 3000,
        });
        return;
      }
    }
    
    // Online URL download logic
    let downloadUrl = pdfUrl;
    
    // Remove trailing slash if present
    if (downloadUrl) {
      downloadUrl = downloadUrl.replace(/\/$/, '');
    }
    
    if (!downloadUrl) {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'No PDF URL found.',
        position: 'bottom',
        visibilityTime: 3000,
      });
      return;
    }

    // Convert MongoDB relative URL to full URL if needed
    if (downloadUrl.startsWith('/inspections/download-file/')) {
      downloadUrl = `https://api.titandrillingzm.com:6004${downloadUrl}`;
      console.log('Converting MongoDB download URL to full URL:', downloadUrl);
    } else if (downloadUrl.startsWith('/corrective-actions/download-pdf/')) {
      downloadUrl = `https://api.titandrillingzm.com:6004${downloadUrl}`;
      console.log('Converting corrective action download URL to full URL:', downloadUrl);
    } else if (downloadUrl.startsWith('/dvir-corrective-actions/download-file/')) {
      // Convert DVIR corrective action relative URL to full URL (port 5006)
      downloadUrl = `https://api.titandrillingzm.com:6006${downloadUrl}`;
      console.log(' Converting DVIR corrective action download URL to full URL:', downloadUrl);
    } else if (downloadUrl.startsWith('/reports/download-file/')) {
      // Convert DVIR new format relative URL to full URL (port 5006)
      downloadUrl = `https://api.titandrillingzm.com:6006${downloadUrl}`;
      console.log('Converting DVIR new format download URL to full URL:', downloadUrl);
    } else if (downloadUrl.startsWith('/download/')) {
      // Convert DVIR legacy relative URL to full URL (port 5006)
      downloadUrl = `https://api.titandrillingzm.com:6006${downloadUrl}`;
      console.log('Converting DVIR legacy download URL to full URL:', downloadUrl);
    }

    if (!downloadUrl.startsWith('http://') && !downloadUrl.startsWith('https://')) {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Invalid PDF URL.',
        position: 'bottom',
        visibilityTime: 3000,
      });
      return;
    }

    try {
      const hasPermission = await requestStoragePermission();
      if (!hasPermission && Platform.OS === 'android') {
        throw new Error('Storage permission denied');
      }

      const fileName = `report_${Date.now()}.pdf`;
      let downloadDest;

      if (Platform.OS === 'android') {
        // On Android, use Download Manager to save to Downloads folder
        downloadDest = `${ReactNativeBlobUtil.fs.dirs.DownloadDir}/${fileName}`;
        const config = {
          fileCache: true,
          path: downloadDest,
          addAndroidDownloads: {
            useDownloadManager: true, // Use Android Download Manager
            notification: true, // Show notification
            title: fileName,
            description: 'Downloading PDF...',
            mime: 'application/pdf',
            mediaScannable: true, // Make file visible in Downloads app
          },
        };

        const res = await ReactNativeBlobUtil.config(config).fetch('GET', downloadUrl);
        console.log('PDF downloaded to:', res.path());

        Toast.show({
          type: 'success',
          text1: 'Success',
          text2: 'PDF downloaded to Downloads folder.',
          position: 'bottom',
          visibilityTime: 3000,
        });
      } else {
        // On iOS, save to DocumentDirectoryPath and offer to share
        downloadDest = `${ReactNativeBlobUtil.fs.dirs.DocumentDir}/${fileName}`;
        const config = {
          fileCache: true,
          path: downloadDest,
        };

        const res = await ReactNativeBlobUtil.config(config).fetch('GET', downloadUrl);
        console.log('PDF downloaded to:', res.path());

        // Share the file on iOS to allow saving to Files app
        await Share.share({
          url: `file://${res.path()}`,
          message: 'Here is your downloaded PDF.',
        });

        Toast.show({
          type: 'success',
          text1: 'Success',
          text2: 'PDF downloaded and ready to share.',
          position: 'bottom',
          visibilityTime: 3000,
        });
      }
    } catch (error) {
      console.error('Error downloading PDF:', error);
      let errorMessage = 'Failed to download PDF.';
      if (error.message.includes('permission denied')) {
        errorMessage = 'Storage permission denied. Please grant permission to download.';
      } else if (error.message.includes('Network')) {
        errorMessage = 'Network error. Please check your connection and try again.';
      } else {
        errorMessage = `Download error: ${error.message}`;
      }
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: errorMessage,
        position: 'bottom',
        visibilityTime: 3000,
      });
    }
  };

  return (
    <View style={styles.container}>
      {loading ? (
        <Modal visible={loading} transparent animationType="fade">
          <View style={styles.syncModalContainer}>
            <View style={styles.syncModalContent}>
              <Text style={styles.syncModalText}>Please Wait</Text>
              <View style={styles.syncIcondiv}>
                <Animated.Image
                  source={require('../../../Images/adminlogin/sync.png')}
                  style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
                />
                <Text style={styles.syncModalTextadd}>Downloading PDF...</Text>
              </View>
            </View>
          </View>
        </Modal>
      ) : localPdfPath ? (
        <>
          <LinearGradient colors={['#3481BC', '#025383']} style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}>
            <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
            <View style={styles.headerContent}>
              <View style={{ width: 40, paddingLeft: width * 0.04,top:10 }}>
                <TouchableOpacity onPress={navigateToInspectionHome} style={styles.backButton}>
                  <MaterialIcons name="arrow-back" size={width * 0.07} color="white" />
                </TouchableOpacity>
              </View>

              <Text style={styles.headerTitle}>PDF Viewer</Text>

              <View style={{ width: 40, paddingRight: width * 0.04 }} />
            </View>
          </LinearGradient>
          <Pdf
            source={{
              uri: Platform.OS === 'android' 
                ? (localPdfPath.startsWith('file://') ? localPdfPath : `file://${localPdfPath}`)
                : localPdfPath.startsWith('file://') 
                  ? localPdfPath.replace('file://', '') 
                  : localPdfPath,
              cache: false
            }}
            style={styles.pdf}
            onError={(error) => {
              console.error('❌ PDF rendering error:', error);
              console.error('❌ PDF path:', localPdfPath);
              
              let errorMessage = 'Failed to load PDF.';
              if (error.message) {
                if (error.message.includes('not found') || error.message.includes('404')) {
                  errorMessage = 'PDF file not found.';
                } else if (error.message.includes('Network')) {
                  errorMessage = 'Network error loading PDF.';
                } else {
                  errorMessage = `PDF error: ${error.message}`;
                }
              }
              
              Toast.show({
                type: 'error',
                text1: 'Error',
                text2: errorMessage,
                position: 'bottom',
                visibilityTime: 4000,
              });
              
              // Don't automatically go back - let user see the error and try again
              // navigation.goBack();
            }}
            onLoadComplete={(numberOfPages, filePath) => {
              console.log(`✅ PDF loaded successfully with ${numberOfPages} pages at ${filePath}`);
            }}
            onPageChanged={(page, numberOfPages) => {
              console.log(`📄 Page ${page} of ${numberOfPages}`);
            }}
          />
          <View style={styles.buttonContainer}>
            <LinearGradient colors={['#3481BC', '#025383']} style={styles.downloadButtonGradient}>
              <TouchableOpacity style={styles.downloadButton} onPress={handleDownloadPDF}>
                <MaterialIcons name="download" size={width * 0.05} color="#fff" />
                <Text style={styles.downloadButtonText}>Download PDF</Text>
              </TouchableOpacity>
            </LinearGradient>
          </View>
        </>
      ) : (
        <View style={styles.errorContainer}>
          <Text style={styles.errorText}>Invalid or unavailable PDF URL</Text>
          <TouchableOpacity onPress={navigateToInspectionHome}>
            <Text style={styles.closeText}>Go Back</Text>
          </TouchableOpacity>
        </View>
      )}
      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
    elevation: 4,
    shadowColor: '#000',
    shadowOpacity: 0.1,
    shadowRadius: 8,
    shadowOffset: { width: 0, height: 4 },
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.05,
    fontWeight: 'bold',
    marginLeft: width * 0.025,
    top:10
  },
  pdf: {
    flex: 1,
    width: width,
    // subtract header height so PDF content sits below header
    height: height - (TOTAL_HEADER_HEIGHT + (StatusBar.currentHeight || width * 0.1)),
  },
  buttonContainer: {
    position: 'absolute',
    bottom: '5%',
    width: '100%',
    alignItems: 'center',
  },
  downloadButtonGradient: {
    borderRadius: width * 0.012,
    overflow: 'hidden',
  },
  downloadButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.05,
  },
  downloadButtonText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: 'bold',
    marginLeft: width * 0.012,
  },
  errorContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    padding: width * 0.05,
  },
  errorText: {
    fontSize: width * 0.045,
    color: '#FF0000',
    marginBottom: width * 0.025,
    textAlign: 'center',
  },
  closeText: {
    fontSize: width * 0.04,
    color: '#0078D4',
    fontWeight: '600',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 22,
    height: 22,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
});

export default PdfviewPage;