import React, { useState, useEffect, useRef } from 'react';
import { View, Text, StyleSheet, FlatList, TouchableOpacity, Dimensions } from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import RiskManagementHeader from './RiskManagementHeader';
import { useNavigation, useRoute } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import InspectionAPI from '../../../ConfigMongo/InspectionMongoConfig.js';

const TASKS = [
  { id: '1', title: 'Abrasive Surface' },
  { id: '2', title: 'Confined Space' },
  { id: '3', title: 'Dust' },
  { id: '4', title: 'Electricity  LV' },
  { id: '5', title: 'Electricity HV' },
  { id: '6', title: 'Elevated Work Area' },
  { id: '7', title: 'Falling Objects' },
  { id: '8', title: 'Flying Objects' },
  { id: '9', title: 'Fumes' },
  { id: '10', title: 'Hazardous Chemicals' },
  { id: '11', title: 'Heavy Objects' },
  { id: '12', title: 'High Pressure -  Hydraulic' },
  { id: '13', title: 'High Pressure – Air' },
  { id: '14', title: 'High Pressure – Water' },
  { id: '15', title: 'Hot Objects' },
  { id: '16', title: 'Hydraulic Failure' },
  { id: '17', title: 'Loud Noise' },
  { id: '18', title: 'Low Oxygen' },
  { id: '19', title: 'Poor Housekeeping' },
  { id: '20', title: 'Radiation' },
  { id: '21', title: 'Rotating Parts' },
  { id: '22', title: 'Sharp Objects' },
  { id: '23', title: 'Steam' },
  { id: '24', title: 'Uneven Surface' },
  { id: '25', title: 'Unexpected Movement' },
];

// Global risks list to show in a dropdown for the selected hazard
const RISKS = [
  { id: 'r1', title: 'Amputation' },
  { id: 'r2', title: 'Burns' },
  { id: 'r3', title: 'Caught Between' },
  { id: 'r4', title: 'Choking' },
  { id: 'r5', title: 'Contact With / Between (Pinch)' },
  { id: 'r6', title: 'Contamination – Soil' },
  { id: 'r7', title: 'Contamination – Oil' },
  { id: 'r8', title: 'Crush' },
  { id: 'r9', title: 'Drowning' },
  { id: 'r10', title: 'Electrocution' },
  { id: 'r11', title: 'Entangled' },
  { id: 'r12', title: 'Fall from Heights' },
  { id: 'r13', title: 'Impairment' },
  { id: 'r14', title: 'Laceration' },
  { id: 'r15', title: 'Over Exertion' },
  { id: 'r16', title: 'Penetrated by' },
  { id: 'r17', title: 'Poisoning' },
  { id: 'r18', title: 'Slip & Fall' },
  { id: 'r19', title: 'Sprain' },
  { id: 'r20', title: 'Strain' },
  { id: 'r21', title: 'Struck Against' },
  { id: 'r22', title: 'Struck By' },
  { id: 'r23', title: 'Thermal Burns' },
  { id: 'r24', title: 'Trip' },
];

const Hazards_step1 = (props) => {
  const navigation = useNavigation();
  const route = useRoute();
  const linkedReport = route.params?.report || null;
  const existingRiskRecord = route.params?.existingRiskRecord || null;
  const [checked, setChecked] = useState({});
  const [currentStep, setCurrentStep] = useState(1);
  const [selectedHazardId, setSelectedHazardId] = useState(null);
  // Which hazard's risk list is expanded
  const [expandedHazardId, setExpandedHazardId] = useState(null);
  // Map: hazardId -> { riskId: true }
  const [hazardRiskSelections, setHazardRiskSelections] = useState({});
  const [isRiskOpen, setIsRiskOpen] = useState(false);
  const riskMetaRef = useRef(null);

  const getRiskMeta = async () => {
    if (riskMetaRef.current) {
      return riskMetaRef.current;
    }

    try {
      if (linkedReport) {
        const reportId = linkedReport.id || linkedReport._id;
        const adminId =
          linkedReport.adminId ||
          linkedReport.userId ||
          linkedReport.createdBy ||
          (await AsyncStorage.getItem('userUid'));

        if (adminId && reportId) {
          const meta = { adminId, reportId };
          riskMetaRef.current = meta;
          await AsyncStorage.setItem('riskMgmt_meta', JSON.stringify(meta));
          return meta;
        }
      }

      const cached = await AsyncStorage.getItem('riskMgmt_meta');
      if (cached) {
        try {
          const parsed = JSON.parse(cached);
          if (parsed?.adminId && parsed?.reportId) {
            riskMetaRef.current = parsed;
            return parsed;
          }
        } catch (e) {
          // ignore parse error
        }
      }

      const userUid = await AsyncStorage.getItem('userUid');
      const step1Raw = await AsyncStorage.getItem('inspectionStep1');
      const step1Data = step1Raw ? JSON.parse(step1Raw) : null;

      if (userUid && step1Data?.id) {
        const meta = { adminId: userUid, reportId: step1Data.id };
        riskMetaRef.current = meta;
        await AsyncStorage.setItem('riskMgmt_meta', JSON.stringify(meta));
        return meta;
      }
    } catch (metaError) {
      console.warn('Unable to resolve risk management meta:', metaError?.message || metaError);
    }

    return null;
  };

  // Persist Step 1 data locally
  const persistStep1 = async (nextChecked = null, nextSelections = null) => {
    try {
      const effectiveChecked = nextChecked ?? checked;
      const effectiveSelections = nextSelections ?? hazardRiskSelections;
      const selectedHazardIds = Object.keys(effectiveChecked).filter((id) => effectiveChecked[id]);
      const toStore = {
        selectedHazardIds,
        hazardRiskSelections: effectiveSelections,
      };
      await AsyncStorage.setItem('riskMgmt_step1', JSON.stringify(toStore));

      const meta = await getRiskMeta();
      if (!meta) {
        return;
      }

      const userUid = await AsyncStorage.getItem('userUid');
      const inspectionStep1Raw = await AsyncStorage.getItem('inspectionStep1');
      const inspectionStep1 = inspectionStep1Raw ? JSON.parse(inspectionStep1Raw) : {};

      const linkedInspection = {
        requestNumber:
          inspectionStep1?.requestNumber ||
          linkedReport?.step1?.requestNumber ||
          linkedReport?.requestNumber ||
          null,
        inspectionId: meta.reportId,
        inspector: inspectionStep1?.inspector || linkedReport?.step1?.inspector || null,
        jobCardRequestNumber: linkedReport?.workOrder?.requestNumber || null,
        jobCardId: linkedReport?.workOrder?.id || null,
      };

      await InspectionAPI.saveRiskManagementStep(meta.adminId, meta.reportId, 'step1', {
        selectedHazardIds,
        hazardRiskSelections: effectiveSelections,
        linkedInspection,
        updatedAt: new Date().toISOString(),
        userId: userUid,
      });
    } catch (error) {
      console.warn('Unable to persist risk management step1:', error?.message || error);
    }
  };

  // Prefill from navigation param if provided
  useEffect(() => {
    if (existingRiskRecord?.step1) {
      const remoteStep1 = existingRiskRecord.step1;
      if (Array.isArray(remoteStep1.selectedHazardIds)) {
        const nextChecked = {};
        remoteStep1.selectedHazardIds.forEach((id) => (nextChecked[id] = true));
        setChecked(nextChecked);
      }
      if (remoteStep1.hazardRiskSelections) {
        setHazardRiskSelections(remoteStep1.hazardRiskSelections);
      }
    }
  }, [existingRiskRecord]);

  // Load existing (MongoDB first, fallback to Async storage)
  useEffect(() => {
    (async () => {
      try {
        const meta = await getRiskMeta();
        if (meta) {
          try {
            const response = await InspectionAPI.getRiskManagementRecord(meta.adminId, meta.reportId);
            if (response?.success && response.data?.step1) {
              const remoteStep1 = response.data.step1;
              if (Array.isArray(remoteStep1.selectedHazardIds)) {
                const nextChecked = {};
                remoteStep1.selectedHazardIds.forEach((id) => (nextChecked[id] = true));
                setChecked(nextChecked);
              }
              if (remoteStep1.hazardRiskSelections) {
                setHazardRiskSelections(remoteStep1.hazardRiskSelections);
              }
              return;
            }
          } catch (remoteError) {
            if (!(remoteError?.status === 404 || remoteError?.message?.includes('not found'))) {
              console.warn('Unable to load risk management step1 from MongoDB:', remoteError?.message || remoteError);
            }
          }
        }

        const saved = await AsyncStorage.getItem('riskMgmt_step1');
        if (saved) {
          const parsed = JSON.parse(saved);
          if (parsed?.selectedHazardIds) {
            const nextChecked = {};
            parsed.selectedHazardIds.forEach((id) => (nextChecked[id] = true));
            setChecked(nextChecked);
          }
          if (parsed?.hazardRiskSelections) setHazardRiskSelections(parsed.hazardRiskSelections);
        }
      } catch (e) {
        // ignore
      }
    })();
  }, []);

  const toggleCheck = (id) => {
    const willSelect = !checked[id];
    const nextChecked = { ...checked };

    if (willSelect) {
      nextChecked[id] = true;
    } else {
      delete nextChecked[id];
    }

    let nextSelections = hazardRiskSelections;
    if (!willSelect && hazardRiskSelections[id]) {
      nextSelections = { ...hazardRiskSelections };
      delete nextSelections[id];
      setHazardRiskSelections(nextSelections);
    }

    setChecked(nextChecked);
    setSelectedHazardId(willSelect ? id : null);
    if (willSelect) {
      setExpandedHazardId(id);
      setIsRiskOpen(true);
    } else if (expandedHazardId === id) {
      setExpandedHazardId(null);
      setIsRiskOpen(false);
    }

    persistStep1(nextChecked, nextSelections);
  };

  const toggleRiskForSelectedHazard = (riskId) => {
    if (!selectedHazardId) return;
    const currentSelections = hazardRiskSelections[selectedHazardId] || {};
    const nextForHazard = { ...currentSelections };

    if (nextForHazard[riskId]) {
      delete nextForHazard[riskId];
    } else {
      nextForHazard[riskId] = true;
    }

    const nextSelections = {
      ...hazardRiskSelections,
      [selectedHazardId]: nextForHazard,
    };

    setHazardRiskSelections(nextSelections);
    persistStep1(null, nextSelections);
  };

  const renderRisksDropdown = () => {
    if (!selectedHazardId) return null;
    const selectedHazard = TASKS.find(t => t.id === selectedHazardId);
    const selectedMap = hazardRiskSelections[selectedHazardId] || {};
    return (
      <View style={styles.dropdownContainer}>
        <TouchableOpacity style={styles.dropdownHeader} onPress={() => setIsRiskOpen((v) => !v)}>
          <Text style={styles.dropdownTitle}>Risks</Text>
          <View style={{ flexDirection: 'row', alignItems: 'center' }}>
            <Text style={styles.dropdownSubTitle}>For: {selectedHazard?.title || ''}  ·  Tap to {isRiskOpen ? 'hide' : 'show'}</Text>
            <MaterialIcons name={isRiskOpen ? 'expand-less' : 'expand-more'} size={22} color="#015185" />
          </View>
        </TouchableOpacity>
        {isRiskOpen && (
          <>
            <View style={styles.divider} />
            {RISKS.map((risk) => {
              const isSelected = !!selectedMap[risk.id];
              return (
                <TouchableOpacity key={risk.id} onPress={() => toggleRiskForSelectedHazard(risk.id)} style={styles.dropdownRow}>
                  <Text style={styles.dropdownText}>{risk.title}</Text>
                  <View style={styles.checkboxOuterSmall}>
                    {isSelected ? (
                      <View style={styles.checkboxInnerSmall}>
                        <MaterialIcons name="check" size={14} color="#3481BC" />
                      </View>
                    ) : null}
                  </View>
                </TouchableOpacity>
              );
            })}
          </>
        )}
      </View>
    );
  };

  const renderItem = ({ item }) => {
    const isSelected = !!checked[item.id];
    const isActive = selectedHazardId === item.id;
    const isExpanded = expandedHazardId === item.id;
    const selectedMap = hazardRiskSelections[item.id] || {};
    const selectedRiskNames = Object.keys(selectedMap)
      .filter((rid) => selectedMap[rid])
      .map((rid) => (RISKS.find((r) => r.id === rid)?.title || rid));
    return (
      <View>
        <TouchableOpacity onPress={() => toggleCheck(item.id)} style={styles.card}>
          <View style={{ flex: 1 }}>
            <Text style={styles.cardTitle} numberOfLines={1}>{item.title}</Text>
            {item.subtitle && <Text style={styles.cardSubtitle}>{item.subtitle}</Text>}
          </View>
          <View style={styles.checkboxOuter}>
            {isSelected ? (
              <View style={styles.checkboxInner}>
                <MaterialIcons name="check" size={16} color="#3481BC" />
              </View>
            ) : null}
          </View>
        </TouchableOpacity>

        {isSelected && (
          <View style={styles.dropdownContainer}>
            <TouchableOpacity
              style={styles.dropdownHeader}
              onPress={() => {
                setSelectedHazardId(item.id);
                setExpandedHazardId((cur) => (cur === item.id ? null : item.id));
              }}
            >
              <Text style={styles.dropdownTitle}>Risks</Text>
              <View style={{ flexDirection: 'row', alignItems: 'center' }}>
                <Text style={styles.dropdownSubTitle}>For: {item.title}  ·  Tap to {isExpanded ? 'hide' : 'show'}</Text>
                <MaterialIcons name={isExpanded ? 'expand-less' : 'expand-more'} size={22} color="#015185" />
              </View>
            </TouchableOpacity>
            {!!selectedRiskNames.length && (
              <View style={styles.chipsRow}>
                {selectedRiskNames.slice(0, 3).map((n) => (
                  <View key={n} style={styles.chip}><Text style={styles.chipText}>{n}</Text></View>
                ))}
                {selectedRiskNames.length > 3 && (
                  <View style={styles.chip}><Text style={styles.chipText}>+{selectedRiskNames.length - 3} more</Text></View>
                )}
              </View>
            )}
            {isExpanded && (
              <>
                <View style={styles.divider} />
                {RISKS.map((risk) => {
                  const isRiskChecked = !!selectedMap[risk.id];
                  return (
                    <TouchableOpacity key={risk.id} onPress={() => toggleRiskForSelectedHazard(risk.id)} style={styles.dropdownRow}>
                      <Text style={styles.dropdownText}>{risk.title}</Text>
                      <View style={styles.checkboxOuterSmall}>
                        {isRiskChecked ? (
                          <View style={styles.checkboxInnerSmall}>
                            <MaterialIcons name="check" size={14} color="#3481BC" />
                          </View>
                        ) : null}
                      </View>
                    </TouchableOpacity>
                  );
                })}
              </>
            )}
          </View>
        )}
      </View>
    );
  };

  return (
    <View style={styles.container}>
      {/* Risk Management Header */}
      <RiskManagementHeader 
        currentStep={currentStep} 
        setCurrentStep={setCurrentStep} 
      />
      
      {/* Content */}
      <View style={styles.content}>
        {/* Header */}
        <View style={styles.headerRow}>
          <Text style={[styles.headerTitle, { color: '#3481BC' }]}>Hazards</Text>
          {/* Removed the icons next to the Hazards heading */}
        </View>
        {/* Task List */}
        <FlatList
          data={TASKS}
          renderItem={renderItem}
          keyExtractor={(item) => item.id}
          contentContainerStyle={{ paddingBottom: 100 }}
          showsVerticalScrollIndicator={false}
        />
      </View>
      {/* Bottom Buttons */}
      <View style={styles.buttonContainer}>
        <TouchableOpacity style={styles.cancelButton} onPress={() => navigation.goBack()}>
          <MaterialIcons style={styles.bottomicon} name="chevron-left" size={16} color="#015185" />
          <Text style={styles.cancelButtonText}>Cancel</Text>
        </TouchableOpacity>
        <TouchableOpacity
          style={styles.nextButton}
          onPress={() => navigation.navigate('Risk_step2', {
            selectedHazards: TASKS.filter(t => checked[t.id]),
            hazardRiskSelections,
            report: linkedReport || null,
          })}
        >
          <Text style={styles.nextButtonText}>Next</Text>
          <MaterialIcons style={styles.bottomicon} name="chevron-right" size={16} color="#015185" />
        </TouchableOpacity>
      </View>
    </View>
  );
};

const { width } = Dimensions.get('window');

const styles = StyleSheet.create({
  container: {
    flex: 1,
  },
  content: {
    flex: 1,
  },
  headerRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: 20,
    paddingTop: 16,
    paddingBottom: 16,
  },
  headerTitle: {
    fontSize: 22,
    fontWeight: 'bold',
  },
  headerIcons: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  card: {
    backgroundColor: '#e0e0e0', // grey background
    borderRadius: 16,
    flexDirection: 'row',
    alignItems: 'center',
    padding: 16,
    marginHorizontal: 16,
    marginBottom: 12,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.15,
    shadowRadius: 4,
    elevation: 2,
    justifyContent: 'space-between',
  },
  dropdownContainer: {
    marginHorizontal: 16,
    marginTop: -4,
    marginBottom: 12,
    backgroundColor: '#fff',
    borderRadius: 12,
    borderWidth: 1,
    borderColor: '#e0e0e0',
  },
  dropdownHeader: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: 14,
    paddingVertical: 10,
  },
  dropdownTitle: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '700',
  },
  dropdownSubTitle: {
    color: '#707173',
    fontSize: 13,
    fontWeight: '500',
  },
  divider: {
    height: 1,
    backgroundColor: '#eee',
    marginHorizontal: 10,
  },
  dropdownRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: 14,
    paddingVertical: 10,
    borderBottomWidth: 1,
    borderBottomColor: '#f2f2f2',
  },
  dropdownText: {
    color: '#333',
    fontSize: 14,
    flex: 1,
    paddingRight: 10,
  },
  chipsRow: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    paddingHorizontal: 14,
    paddingBottom: 8,
    gap: 6,
  },
  chip: {
    backgroundColor: '#e6f0fa',
    borderRadius: 12,
    paddingHorizontal: 10,
    paddingVertical: 4,
    borderWidth: 1,
    borderColor: '#c7d8ee',
  },
  chipText: {
    color: '#015185',
    fontSize: 12,
    fontWeight: '600',
  },
  checkboxOuter: {
    width: 24,
    height: 24,
    borderRadius: 12,
    borderWidth: 2,
    borderColor: '#3481BC',
    marginRight: 0,
    alignItems: 'center',
    justifyContent: 'center',
    backgroundColor: 'transparent', // remove background
  },
  checkboxInner: {
    width: 20,
    height: 20,
    borderRadius: 10,
    backgroundColor: 'transparent', // remove background
    alignItems: 'center',
    justifyContent: 'center',
  },
  checkboxOuterSmall: {
    width: 20,
    height: 20,
    borderRadius: 10,
    borderWidth: 2,
    borderColor: '#3481BC',
    alignItems: 'center',
    justifyContent: 'center',
    backgroundColor: 'transparent',
  },
  checkboxInnerSmall: {
    width: 16,
    height: 16,
    borderRadius: 8,
    backgroundColor: 'transparent',
    alignItems: 'center',
    justifyContent: 'center',
  },
  cardTitle: {
    color: '#3481BC', // blue like heading
    fontSize: 16,
    fontWeight: '600',
  },
  cardSubtitle: {
    color: '#3481BC', // blue like heading
    fontSize: 12,
    marginTop: 2,
  },
  fab: {
    position: 'absolute',
    bottom: 24,
    right: 24,
    backgroundColor: '#fff',
    width: 56,
    height: 56,
    borderRadius: 28,
    alignItems: 'center',
    justifyContent: 'center',
    elevation: 6,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 4,
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: 15,
    paddingVertical: 15,
    backgroundColor: '#F2F2F2',
    borderTopWidth: 1,
    borderTopColor: '#ddd',
    position: 'absolute',
    bottom: 0,
    left: 0,
    right: 0,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: -2 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
    paddingBottom: 30,
  },
  cancelButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingBottom: 30,
    paddingHorizontal: 15,
  },
  cancelButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  nextButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingBottom: 30,
    borderRadius: 5,
  },
  nextButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  bottomicon: {
    marginTop: 2,
  },
});

export default Hazards_step1;