import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  FlatList,
  StatusBar,
  Dimensions,
  Image,
  ScrollView,
  Animated,
  Modal,
  Platform,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import DateTimePicker from '@react-native-community/datetimepicker';
import LinearGradient from 'react-native-linear-gradient';
import Toast from 'react-native-toast-message';
import NetInfo from '@react-native-community/netinfo';
import JobCardAPI from '../../ConfigMongo/JobCardMongoConfig.js';
import { MongoAPI } from '../../ConfigMongo/MongoDBConfig';

const { width, height } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.25;

// Enhanced Loader Modal with modern design
const SyncModal = ({ visible, message = 'Loading Maintenance KPI Data...' }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
            />
            <Text style={styles.syncModalTextadd}>{message}</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

const MaintenanceKpiHome = () => {
  const navigation = useNavigation();
  const [activeTab, setActiveTab] = useState('Today');
  const [isLoading, setIsLoading] = useState(true);
  const [modalVisible, setModalVisible] = useState(false);
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [isOnline, setIsOnline] = useState(true);

  // Enhanced user info state
  const [userInfo, setUserInfo] = useState({
    countries: [],
    projects: [],
    requestMaintenancePermissions: [],
    jobCardPermissions: [],
    name: '',
    email: '',
    uid: '',
  });

  // Enhanced KPI data state
  const [kpiData, setKpiData] = useState({
    totalMaintenanceRequests: 0,
    pendingApproval: 0,
    worksInProgress: 0,
    onHold: 0,
    emergency: 0,
    upcomingSchedules: 0,
    totalServiceSchedules: 0,
    totalJobCards: 0,
    completedJobs: 0,
    averageCompletionTime: 0,
    equipmentUtilization: 0,
  });

  // State to toggle between all-time and date-range for maintenance requests
  const [showAllMaintenanceRequests, setShowAllMaintenanceRequests] = useState(true);
  // Enhanced data fetching and KPI calculation
  const [maintenanceRequests, setMaintenanceRequests] = useState([]);
  const [jobCards, setJobCards] = useState([]);
  const [serviceSchedules, setServiceSchedules] = useState([]);

  // Calculate KPIs from real data
  const calculateKPIs = () => {
    try {
      // Filter data based on active tab
      let filteredMaintenanceRequests = maintenanceRequests;
      let filteredJobCards = jobCards;
      let filteredServiceSchedules = serviceSchedules;

      // Apply date filtering based on active tab
      const now = new Date();
      let startDate = new Date();
      
      switch (activeTab) {
        case 'Today':
          startDate.setHours(0, 0, 0, 0);
          break;
        case '7 days':
          startDate.setDate(startDate.getDate() - 7);
          break;
        case '30 days':
          startDate.setDate(startDate.getDate() - 30);
          break;
        case '3 months':
          startDate.setMonth(startDate.getMonth() - 3);
          break;
        default:
          startDate = new Date(0); // All time
      }

      if (activeTab !== 'All Time') {
        filteredMaintenanceRequests = maintenanceRequests.filter(req => 
          req.createdAt && new Date(req.createdAt) >= startDate
        );
        filteredJobCards = jobCards.filter(jc => 
          jc.date && jc.date >= startDate
        );
        filteredServiceSchedules = serviceSchedules.filter(ss => 
          ss.date && ss.date >= startDate
        );
      }

      // Calculate KPIs
      const totalMaintenanceRequests = filteredMaintenanceRequests.length;
      const pendingApproval = filteredMaintenanceRequests.filter(req => 
        !req.status || req.status.toLowerCase().includes('pending')
      ).length;
      const worksInProgress = filteredJobCards.filter(jc => 
        jc.status && jc.status.toLowerCase().includes('progress')
      ).length;
      const onHold = filteredJobCards.filter(jc => 
        jc.status && jc.status.toLowerCase().includes('hold')
      ).length;
      const emergency = filteredMaintenanceRequests.filter(req => 
        req.priority && req.priority.toLowerCase().includes('emergency')
      ).length;
      // Upcoming schedules - schedules that are upcoming/pending
      const upcomingSchedules = filteredServiceSchedules.filter(ss => 
        ss.status && (ss.status.toLowerCase().includes('upcoming') || ss.status.toLowerCase().includes('pending'))
      ).length;
      const totalServiceSchedules = filteredServiceSchedules.length; // Total service schedule requests
      const totalJobCards = filteredJobCards.length;
      const completedJobs = filteredJobCards.filter(jc => 
        jc.status && jc.status.toLowerCase().includes('completed')
      ).length;
      
      // Calculate average completion time
      const completedJobCards = filteredJobCards.filter(jc => 
        jc.status && jc.status.toLowerCase().includes('completed') && jc.completedAt
      );
      const totalCompletionTime = completedJobCards.reduce((sum, jc) => {
        const start = new Date(jc.date);
        const end = new Date(jc.completedAt);
        return sum + (end - start);
      }, 0);
      const averageCompletionTime = completedJobCards.length > 0 ? 
        Math.round(totalCompletionTime / (completedJobCards.length * 24 * 60 * 60 * 1000)) : 0;

      // Calculate equipment utilization (simplified)
      const uniqueEquipments = new Set(filteredJobCards.map(jc => jc.equipment)).size;
      const equipmentUtilization = uniqueEquipments > 0 ? 
        Math.round((filteredJobCards.length / uniqueEquipments) * 10) : 0;

      setKpiData({
        totalMaintenanceRequests,
        pendingApproval,
        worksInProgress,
        onHold,
        emergency,
        upcomingSchedules,
        totalServiceSchedules,
        totalJobCards,
        completedJobs,
        averageCompletionTime,
        equipmentUtilization,
      });

    } catch (error) {
      console.error('Error calculating KPIs:', error);
    }
  };

  // Enhanced data fetching with better structure - MongoDB version
  const fetchJobCardsWithConnections = async () => {
    try {
      setJobCardsLoading(true);
      console.log('🔄 Fetching job cards from MongoDB...');
      
      const response = await JobCardAPI.getAllJobCards();
      const jobCardsArr = [];
      
      if (response.success && response.data) {
        response.data.forEach((card) => {
          const createdAt = card?.createdAt;
          if (createdAt) {
            const cardDate = new Date(createdAt);
            if (cardDate >= fromDate && cardDate <= toDate) {
              // Enhanced job card object with all connections
              jobCardsArr.push({
                id: card._id || card.id,
                equipment: card?.equipment || {},
                status: card?.status || '-',
                date: cardDate,
                requestNumber: card?.requestNumber || card?.title || '',
                title: card?.title || '',
                assignTo: card?.assignTo || [],
                items: card?.items || [],
                tasks: card?.tasks || {},
                linkInspection: card?.linkInspection || null,
                linkedWorkOrder: card?.linkedWorkOrder || null,
                serviceScheduleId: card?.serviceScheduleId || null,
                priority: card?.priority || 'Normal',
                workOrderType: card?.workOrderType || 'General',
                memo: card?.memo || '',
                startDate: card?.startDate || null,
                dueDate: card?.dueDate || null,
                additionalDetails: card?.additionalDetails || '',
                estimatedCost: card?.estimatedCost || '',
                estimatedLaborTime: card?.estimatedLaborTime || '',
                partMaterialCost: card?.partMaterialCost || '',
                totalLaborCost: card?.totalLaborCost || '',
                additionalCost: card?.additionalCost || '',
                taxAmount: card?.taxAmount || '',
                taxPercentage: card?.taxPercentage || '',
                totalWoCost: card?.totalWoCost || '',
                checklistItems: card?.checklistItems || [],
                pdfDownloadUrl: card?.pdfDownloadUrl || null,
                adminUid: card?.adminUid || card?.userId,
                country: card?.location?.country || card?.country || '',
                project: card?.location?.project || card?.project || '',
                // Enhanced equipment details
                equipmentDetails: {
                  equipmentNumber: card?.equipment?.equipmentNumber || '',
                  equipmentName: card?.equipment?.equipmentName || '',
                  model: card?.equipment?.model || '',
                  status: card?.equipment?.status || '',
                  category: card?.equipment?.equipmentCategory || card?.equipment?.mainCategory || '',
                  location: card?.equipment?.location || '',
                  manufacturer: card?.equipment?.manufacturer || '',
                  year: card?.equipment?.year || '',
                }
              });
            }
          }
        });
        
        console.log('✅ Loaded job cards from MongoDB:', jobCardsArr.length);
      }

      // Apply permission-based filtering
      let filteredJobCards = jobCardsArr;
      
      // Check if user has onlyMineView permission
      if (userInfo.jobCardPermissions.includes('onlyMineView') && userInfo.uid) {
        // Show only job cards created by the current user
        filteredJobCards = jobCardsArr.filter(jc => jc.adminUid === userInfo.uid);
        console.log('🔒 Filtered to user\'s own job cards:', filteredJobCards.length);
      } else {
        // Filter by user's assigned countries and projects
        if (userInfo.countries.length > 0 || userInfo.projects.length > 0) {
          filteredJobCards = jobCardsArr.filter(jc => {
            const countryMatch = userInfo.countries.length === 0 || 
              userInfo.countries.includes(jc.country);
            const projectMatch = userInfo.projects.length === 0 || 
              userInfo.projects.includes(jc.project);
            return countryMatch && projectMatch;
          });
          console.log('🌐 Filtered by permissions:', filteredJobCards.length);
        }
      }

      setJobCards(filteredJobCards);
      
      // Cache job cards
      await AsyncStorage.setItem('maintenanceJobCards', JSON.stringify(filteredJobCards));
    } catch (e) {
      console.error('❌ Error fetching job cards from MongoDB:', e);
      
      // Try to load from cache
      try {
        const cachedJobCards = await AsyncStorage.getItem('maintenanceJobCards');
        if (cachedJobCards) {
          setJobCards(JSON.parse(cachedJobCards));
          Toast.show({
            type: 'info',
            text1: 'Offline Mode',
            text2: 'Using cached job cards',
            position: 'bottom',
          });
        } else {
          setJobCards([]);
        }
      } catch (cacheError) {
        console.error('❌ Cache fallback failed:', cacheError);
        setJobCards([]);
      }
    } finally {
      setJobCardsLoading(false);
    }
  };

  // Enhanced service schedules fetching - MongoDB version
  const fetchServiceSchedulesWithConnections = async () => {
    try {
      setServiceSchedulesLoading(true);
      console.log('🔄 Fetching service schedules from MongoDB...');
      
      // Fetch from MongoDB API
      const response = await fetch('https://api.titandrillingzm.com:6010/service-schedules', {
        method: 'GET',
        headers: {
          'Content-Type': 'application/json',
        },
      });
      
      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`);
      }
      
      const data = await response.json();
      const schedulesArr = [];
      
      if (data.success && data.data) {
        data.data.forEach((schedule) => {
          const createdAt = schedule?.createdAt;
          if (createdAt) {
            const scheduleDate = new Date(createdAt);
            // Apply date filter
            if (scheduleDate >= fromDate && scheduleDate <= toDate) {
              schedulesArr.push({
                id: schedule._id || schedule.id,
                date: scheduleDate,
                equipment: schedule?.equipment || {},
                status: schedule?.status || 'upcoming',
                serviceOrderNumber: schedule?.scheduleId || schedule?.title || schedule?.id || '',
                title: schedule?.title || '',
                serviceType: schedule?.repeatType || '',
                frequency: schedule?.intervalUnit || '',
                lastServiceDate: schedule?.lastServiceDate || null,
                meterReading: schedule?.meterReading || null,
                meterUnit: schedule?.meterUnit || '',
                notifyBefore: schedule?.notifyBefore || null,
                tasks: schedule?.tasks || [],
                dueHours: schedule?.dueHours || '',
                hoursToGo: schedule?.hoursToGo || '',
                daysToGo: schedule?.daysToGo || '',
                jobCardId: schedule?.jobCardId || null,
                jobCardRequestNumber: schedule?.jobCardRequestNumber || null,
                jobCardTitle: schedule?.jobCardTitle || null,
                jobCardStatus: schedule?.jobCardStatus || null,
                note: schedule?.note || '',
                adminUid: schedule?.adminUid,
                country: schedule?.country || '',
                project: schedule?.project || '',
                // Enhanced equipment details
                equipmentDetails: {
                  equipmentNumber: schedule?.equipment?.equipmentNumber || '',
                  equipmentName: schedule?.equipment?.equipmentName || '',
                  entryNumber: schedule?.equipment?.entryNumber || '',
                  model: schedule?.equipment?.model || '',
                  meterReading: schedule?.equipment?.meterReading || '',
                  meterUnit: schedule?.equipment?.meterUnit || '',
                }
              });
            }
          }
        });
        
        console.log('✅ Loaded service schedules from MongoDB:', schedulesArr.length);
      }

      // Apply permission-based filtering for service schedules
      let filteredSchedules = schedulesArr;
      
      // Check if user has onlyMineView permission
      if (userInfo.jobCardPermissions.includes('onlyMineView') && userInfo.uid) {
        filteredSchedules = schedulesArr.filter(ss => ss.adminUid === userInfo.uid);
        console.log('🔒 Filtered to user\'s own schedules:', filteredSchedules.length);
      } else {
        // Filter by user's assigned countries and projects
        if (userInfo.countries.length > 0 || userInfo.projects.length > 0) {
          filteredSchedules = schedulesArr.filter(ss => {
            const countryMatch = userInfo.countries.length === 0 || 
              userInfo.countries.includes(ss.country);
            const projectMatch = userInfo.projects.length === 0 || 
              userInfo.projects.includes(ss.project);
            return countryMatch && projectMatch;
          });
          console.log('🌐 Filtered by permissions:', filteredSchedules.length);
        }
      }

      setServiceSchedules(filteredSchedules);
      
      // Cache service schedules
      await AsyncStorage.setItem('maintenanceServiceSchedules', JSON.stringify(filteredSchedules));
    } catch (e) {
      console.error('❌ Error fetching service schedules from MongoDB:', e);
      
      // Try to load from cache
      try {
        const cachedSchedules = await AsyncStorage.getItem('maintenanceServiceSchedules');
        if (cachedSchedules) {
          setServiceSchedules(JSON.parse(cachedSchedules));
          Toast.show({
            type: 'info',
            text1: 'Offline Mode',
            text2: 'Using cached service schedules',
            position: 'bottom',
          });
        } else {
          setServiceSchedules([]);
        }
      } catch (cacheError) {
        console.error('❌ Cache fallback failed:', cacheError);
        setServiceSchedules([]);
      }
    } finally {
      setServiceSchedulesLoading(false);
    }
  };

  // Enhanced maintenance requests fetching - MongoDB version
  const fetchMaintenanceRequestsWithDetails = async () => {
    try {
      setSyncModalVisible(true);
      console.log('🔄 Fetching maintenance requests from MongoDB...');
      
      // Fetch from MongoDB API
      const response = await fetch('https://api.titandrillingzm.com:6009/request-maintenance', {
        method: 'GET',
        headers: {
          'Content-Type': 'application/json',
        },
      });
      
      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`);
      }
      
      const data = await response.json();
      const requestsArr = [];
      
      if (data.success && data.data) {
        data.data.forEach((req) => {
          const createdAt = req?.createdAt;
          if (createdAt) {
            const reqDate = new Date(createdAt);
            requestsArr.push({
              id: req._id || req.id,
              requestNumber: req?.requestNumber || req?.id || '',
              workOrderTitle: req?.workOrderTitle || req?.title || '',
              title: req?.workOrderTitle || req?.title || '',
              requestType: req?.requestType || '',
              priority: req?.priority || 'Normal',
              description: req?.description || '',
              status: req?.status || 'Requested',
              tasks: req?.tasks || [],
              assignTo: req?.assignTo || [],
              date: reqDate,
              createdAt: reqDate,
              updatedAt: req?.updatedAt ? new Date(req.updatedAt) : reqDate,
              adminUid: req?.adminUid || req?.userId,
              userId: req?.userId || req?.adminUid,
              country: req?.country || '',
              project: req?.project || '',
              isArchived: req?.isArchived || false,
              // Enhanced equipment details
              equipment: req?.equipment || {},
              equipmentDetails: {
                equipmentNumber: req?.equipment?.equipmentNumber || '',
                equipmentName: req?.equipment?.equipmentName || '',
                model: req?.equipment?.model || '',
                serialNumber: req?.equipment?.serialNumber || '',
                status: req?.equipment?.status || '',
                category: req?.equipment?.equipmentCategory || req?.equipment?.mainCategory || '',
                entryNumber: req?.equipment?.entryNumber || '',
              }
            });
          }
        });
        
        console.log('✅ Loaded maintenance requests from MongoDB:', requestsArr.length);
      }

      // Apply permission-based filtering for maintenance requests
      let filteredRequests = requestsArr;
      
      // Check if user has onlyMineView permission for maintenance requests
      if (userInfo.requestMaintenancePermissions.includes('onlyMineView') && userInfo.uid) {
        filteredRequests = requestsArr.filter(req => req.adminUid === userInfo.uid);
        console.log('🔒 Filtered to user\'s own requests:', filteredRequests.length);
      } else {
        // Filter by user's assigned countries and projects
        if (userInfo.countries.length > 0 || userInfo.projects.length > 0) {
          filteredRequests = requestsArr.filter(req => {
            const countryMatch = userInfo.countries.length === 0 || 
              userInfo.countries.includes(req.country);
            const projectMatch = userInfo.projects.length === 0 || 
              userInfo.projects.includes(req.project);
            return countryMatch && projectMatch;
          });
          console.log('🌐 Filtered by permissions:', filteredRequests.length);
        }
      }

      setMaintenanceRequests(filteredRequests);
      
      // Cache maintenance requests
      await AsyncStorage.setItem('maintenanceRequests', JSON.stringify(filteredRequests));
    } catch (e) {
      console.error('❌ Error fetching maintenance requests from MongoDB:', e);
      
      // Try to load from cache
      try {
        const cachedRequests = await AsyncStorage.getItem('maintenanceRequests');
        if (cachedRequests) {
          setMaintenanceRequests(JSON.parse(cachedRequests));
          Toast.show({
            type: 'info',
            text1: 'Offline Mode',
            text2: 'Using cached maintenance requests',
            position: 'bottom',
          });
        } else {
          setMaintenanceRequests([]);
        }
      } catch (cacheError) {
        console.error('❌ Cache fallback failed:', cacheError);
        setMaintenanceRequests([]);
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: 'Failed to load maintenance requests',
        });
      }
    } finally {
      setSyncModalVisible(false);
    }
  };

  // Enhanced render functions for better UI
  const renderJobCardItem = (jc, index) => (
    <TouchableOpacity 
      key={jc.id} 
      style={styles.dataCard}
      onPress={() => { setSelectedJobCard(jc); setJobCardModalVisible(true); }}
      activeOpacity={0.7}
    >
      <View style={styles.cardHeader}>
        <View style={styles.cardTitleSection}>
          <Text style={styles.cardTitle}>{jc.requestNumber || jc.title || jc.id}</Text>
          <View style={[styles.statusBadge, { backgroundColor: getStatusColor(jc.status) }]}>
            <Text style={styles.statusText}>{jc.status}</Text>
          </View>
        </View>
        <Text style={styles.cardDate}>{jc.date.toLocaleDateString()}</Text>
      </View>
      
      <View style={styles.cardContent}>
        <View style={styles.cardRow}>
          <Text style={styles.cardLabel}>Equipment:</Text>
          <Text style={styles.cardValue}>
            {jc.equipmentDetails?.equipmentName || jc.equipmentDetails?.equipmentNumber || 'N/A'}
          </Text>
        </View>
        
        <View style={styles.cardRow}>
          <Text style={styles.cardLabel}>Type:</Text>
          <Text style={styles.cardValue}>{jc.workOrderType || 'General'}</Text>
        </View>
        
        <View style={styles.cardRow}>
          <Text style={styles.cardLabel}>Priority:</Text>
          <Text style={[styles.cardValue, { color: getPriorityColor(jc.priority) }]}>
            {jc.priority || 'Normal'}
          </Text>
        </View>
        
        {jc.assignTo && jc.assignTo.length > 0 && (
          <View style={styles.cardRow}>
            <Text style={styles.cardLabel}>Assigned:</Text>
            <Text style={styles.cardValue}>
              {jc.assignTo.map(user => user.name || user.fullName || user.email).join(', ')}
            </Text>
          </View>
        )}
        
        {/* Show connections */}
        {(jc.linkInspection || jc.linkedWorkOrder || jc.serviceScheduleId) && (
          <View style={styles.connectionsSection}>
            <Text style={styles.connectionsTitle}>Connections:</Text>
            {jc.linkInspection && (
              <View style={styles.connectionItem}>
                <MaterialIcons name="assignment" size={16} color="#015185" />
                <Text style={styles.connectionText}>
                  Inspection: {jc.linkInspection.name || jc.linkInspection.id}
                </Text>
              </View>
            )}
            {jc.linkedWorkOrder && (
              <View style={styles.connectionItem}>
                <MaterialIcons name="work" size={16} color="#015185" />
                <Text style={styles.connectionText}>
                  Work Order: {jc.linkedWorkOrder.requestNumber || jc.linkedWorkOrder.id}
                </Text>
              </View>
            )}
            {jc.serviceScheduleId && (
              <View style={styles.connectionItem}>
                <MaterialIcons name="schedule" size={16} color="#015185" />
                <Text style={styles.connectionText}>
                  Service Schedule: {jc.serviceScheduleId}
                </Text>
              </View>
            )}
          </View>
        )}
      </View>
    </TouchableOpacity>
  );

  // Render equipment item with job cards
  const renderEquipmentItem = (eq, index) => {
    // Get job cards for this equipment
    const equipmentJobCards = filteredJobCards.filter(jc => 
      jc.equipmentDetails?.equipmentNumber === eq.equipmentNumber ||
      jc.equipmentDetails?.equipmentName === eq.equipmentName
    );

    return (
      <View key={eq.equipmentNumber || index} style={styles.dataCard}>
        <View style={styles.cardHeader}>
          <View style={styles.cardTitleSection}>
            <Text style={styles.cardTitle}>{eq.equipmentName || eq.equipmentNumber || 'Unknown Equipment'}</Text>
            <View style={[styles.statusBadge, { backgroundColor: getStatusColor(eq.status) }]}>
              <Text style={styles.statusText}>{eq.status || 'Active'}</Text>
            </View>
          </View>
        </View>
        
        <View style={styles.cardContent}>
          <View style={styles.cardRow}>
            <Text style={styles.cardLabel}>Number:</Text>
            <Text style={styles.cardValue}>{eq.equipmentNumber || '-'}</Text>
          </View>
          
          <View style={styles.cardRow}>
            <Text style={styles.cardLabel}>Model:</Text>
            <Text style={styles.cardValue}>{eq.model || '-'}</Text>
          </View>
          
          <View style={styles.cardRow}>
            <Text style={styles.cardLabel}>Category:</Text>
            <Text style={styles.cardValue}>{eq.equipmentCategory || eq.mainCategory || '-'}</Text>
          </View>
          
          <View style={styles.cardRow}>
            <Text style={styles.cardLabel}>Job Cards:</Text>
            <Text style={styles.cardValue}>{equipmentJobCards.length}</Text>
          </View>
          
          {/* Show job cards for this equipment */}
          {equipmentJobCards.length > 0 && (
            <View style={styles.connectionsSection}>
              <Text style={styles.connectionsTitle}>Related Job Cards:</Text>
              {equipmentJobCards.slice(0, 3).map((jc, jcIndex) => (
                <TouchableOpacity 
                  key={jc.id} 
                  style={styles.connectionItem}
                  onPress={() => { setSelectedJobCard(jc); setJobCardModalVisible(true); }}
                >
                  <MaterialIcons name="work" size={16} color="#015185" />
                  <Text style={styles.connectionText}>
                    {jc.requestNumber || jc.title} ({jc.status})
                  </Text>
                </TouchableOpacity>
              ))}
              {equipmentJobCards.length > 3 && (
                <Text style={[styles.connectionText, { color: '#666', fontStyle: 'italic' }]}>
                  +{equipmentJobCards.length - 3} more job cards
                </Text>
              )}
            </View>
          )}
        </View>
      </View>
    );
  };

  // Render assigned user item
  const renderAssignedUserItem = (au, index) => (
    <View key={index} style={styles.dataCard}>
      <View style={styles.cardHeader}>
        <View style={styles.cardTitleSection}>
          <Text style={styles.cardTitle}>{au.user.fullName || au.user.name || au.user.email || au.user.employeeNumber || 'Unknown User'}</Text>
        </View>
        <Text style={styles.cardDate}>{au.jobCard.date.toLocaleDateString()}</Text>
      </View>
      
      <View style={styles.cardContent}>
        <View style={styles.cardRow}>
          <Text style={styles.cardLabel}>Job Card:</Text>
          <TouchableOpacity onPress={() => { setSelectedJobCard(au.jobCard); setJobCardModalVisible(true); }}>
            <Text style={[styles.cardValue, { color: '#015185', textDecorationLine: 'underline' }]}>
              {au.jobCard.requestNumber || au.jobCard.title}
            </Text>
          </TouchableOpacity>
        </View>
        
        <View style={styles.cardRow}>
          <Text style={styles.cardLabel}>Status:</Text>
          <Text style={[styles.cardValue, { color: getStatusColor(au.jobCard.status) }]}>
            {au.jobCard.status}
          </Text>
        </View>
        
        <View style={styles.cardRow}>
          <Text style={styles.cardLabel}>Equipment:</Text>
          <Text style={styles.cardValue}>
            {au.jobCard.equipmentDetails?.equipmentName || au.jobCard.equipmentDetails?.equipmentNumber || 'N/A'}
          </Text>
        </View>
      </View>
    </View>
  );

  // Render inspection item
  const renderInspectionItem = (ins, index) => {
    // Get job cards for this inspection
    const inspectionJobCards = filteredJobCards.filter(jc => 
      jc.linkInspection && jc.linkInspection.id === ins.inspectionId
    );

    return (
      <View key={ins.inspectionId || index} style={styles.dataCard}>
        <View style={styles.cardHeader}>
          <View style={styles.cardTitleSection}>
            <Text style={styles.cardTitle}>{ins.name || ins.inspectionId || 'Unknown Inspection'}</Text>
          </View>
        </View>
        
        <View style={styles.cardContent}>
          <View style={styles.cardRow}>
            <Text style={styles.cardLabel}>ID:</Text>
            <Text style={styles.cardValue}>{ins.inspectionId || ins.id || '-'}</Text>
          </View>
          
          <View style={styles.cardRow}>
            <Text style={styles.cardLabel}>Job Cards:</Text>
            <Text style={styles.cardValue}>{inspectionJobCards.length}</Text>
          </View>
          
          {/* Show job cards for this inspection */}
          {inspectionJobCards.length > 0 && (
            <View style={styles.connectionsSection}>
              <Text style={styles.connectionsTitle}>Related Job Cards:</Text>
              {inspectionJobCards.map((jc, jcIndex) => (
                <TouchableOpacity 
                  key={jc.id} 
                  style={styles.connectionItem}
                  onPress={() => { setSelectedJobCard(jc); setJobCardModalVisible(true); }}
                >
                  <MaterialIcons name="work" size={16} color="#015185" />
                  <Text style={styles.connectionText}>
                    {jc.requestNumber || jc.title} ({jc.status})
                  </Text>
                </TouchableOpacity>
              ))}
            </View>
          )}
        </View>
      </View>
    );
  };

  const renderMaintenanceRequestItem = (rm, index) => (
    <View key={rm.id} style={styles.dataCard}>
      <View style={styles.cardHeader}>
        <View style={styles.cardTitleSection}>
          <Text style={styles.cardTitle}>{rm.requestNumber || rm.id}</Text>
          <View style={[styles.statusBadge, { backgroundColor: getStatusColor(rm.status) }]}>
            <Text style={styles.statusText}>{rm.status || 'Requested'}</Text>
          </View>
        </View>
        <Text style={styles.cardDate}>{rm.date ? rm.date.toLocaleDateString() : '-'}</Text>
      </View>
      
      <View style={styles.cardContent}>
        <View style={styles.cardRow}>
          <Text style={styles.cardLabel}>Title:</Text>
          <Text style={styles.cardValue}>{rm.workOrderTitle || rm.title || '-'}</Text>
        </View>
        
        {rm.requestType && (
          <View style={styles.cardRow}>
            <Text style={styles.cardLabel}>Request Type:</Text>
            <Text style={styles.cardValue}>{rm.requestType}</Text>
          </View>
        )}
        
        <View style={styles.cardRow}>
          <Text style={styles.cardLabel}>Priority:</Text>
          <Text style={[styles.cardValue, { color: getPriorityColor(rm.priority) }]}>
            {rm.priority || 'Normal'}
          </Text>
        </View>
        
        <View style={styles.cardRow}>
          <Text style={styles.cardLabel}>Equipment:</Text>
          <Text style={styles.cardValue}>
            {rm.equipmentDetails?.equipmentName || rm.equipmentDetails?.equipmentNumber || 'N/A'}
          </Text>
        </View>
        
        {rm.equipmentDetails?.model && (
          <View style={styles.cardRow}>
            <Text style={styles.cardLabel}>Model:</Text>
            <Text style={styles.cardValue}>{rm.equipmentDetails.model}</Text>
          </View>
        )}
        
        {rm.country && (
          <View style={styles.cardRow}>
            <Text style={styles.cardLabel}>Country:</Text>
            <Text style={styles.cardValue}>{rm.country}</Text>
          </View>
        )}
        
        {rm.project && (
          <View style={styles.cardRow}>
            <Text style={styles.cardLabel}>Project:</Text>
            <Text style={styles.cardValue}>{rm.project}</Text>
          </View>
        )}
        
        {rm.description && (
          <View style={styles.cardRow}>
            <Text style={styles.cardLabel}>Description:</Text>
            <Text style={styles.cardValue} numberOfLines={2}>{rm.description}</Text>
          </View>
        )}
        
        {rm.assignTo && rm.assignTo.length > 0 && (
          <View style={styles.cardRow}>
            <Text style={styles.cardLabel}>Assigned To:</Text>
            <Text style={styles.cardValue}>
              {rm.assignTo.map(user => user.fullName || user.name || user.email).join(', ')}
            </Text>
          </View>
        )}
        
        {rm.tasks && rm.tasks.length > 0 && (
          <View style={styles.connectionsSection}>
            <Text style={styles.connectionsTitle}>Tasks ({rm.tasks.length}):</Text>
            {rm.tasks.slice(0, 3).map((task, idx) => (
              <View key={task.id || idx} style={styles.connectionItem}>
                <MaterialIcons name="check-circle" size={16} color="#015185" />
                <Text style={styles.connectionText}>
                  {task.title || task.description || task.note || 'Task'}
                  {task.status && ` (${task.status})`}
                </Text>
              </View>
            ))}
            {rm.tasks.length > 3 && (
              <Text style={[styles.connectionText, { color: '#666', fontStyle: 'italic', marginLeft: 8 }]}>
                +{rm.tasks.length - 3} more task(s)
              </Text>
            )}
          </View>
        )}
      </View>
    </View>
  );

  const renderServiceScheduleItem = (ss, index) => (
    <View key={ss.id} style={styles.dataCard}>
      <View style={styles.cardHeader}>
        <View style={styles.cardTitleSection}>
          <Text style={styles.cardTitle}>{ss.serviceOrderNumber || ss.title || ss.id}</Text>
          <View style={[styles.statusBadge, { backgroundColor: getStatusColor(ss.status) }]}>
            <Text style={styles.statusText}>{ss.status}</Text>
          </View>
        </View>
        <Text style={styles.cardDate}>{ss.date.toLocaleDateString()}</Text>
      </View>
      
      <View style={styles.cardContent}>
        {ss.title && (
          <View style={styles.cardRow}>
            <Text style={styles.cardLabel}>Title:</Text>
            <Text style={styles.cardValue}>{ss.title}</Text>
          </View>
        )}
        
        <View style={styles.cardRow}>
          <Text style={styles.cardLabel}>Equipment:</Text>
          <Text style={styles.cardValue}>
            {ss.equipmentDetails?.equipmentName || ss.equipmentDetails?.equipmentNumber || 'N/A'}
          </Text>
        </View>
        
        <View style={styles.cardRow}>
          <Text style={styles.cardLabel}>Repeat Type:</Text>
          <Text style={styles.cardValue}>{ss.serviceType || '-'}</Text>
        </View>
        
        <View style={styles.cardRow}>
          <Text style={styles.cardLabel}>Frequency:</Text>
          <Text style={styles.cardValue}>{ss.frequency || '-'}</Text>
        </View>
        
        {ss.lastServiceDate && (
          <View style={styles.cardRow}>
            <Text style={styles.cardLabel}>Last Service:</Text>
            <Text style={styles.cardValue}>{new Date(ss.lastServiceDate).toLocaleDateString()}</Text>
          </View>
        )}
        
        {ss.meterReading && (
          <View style={styles.cardRow}>
            <Text style={styles.cardLabel}>Meter Reading:</Text>
            <Text style={styles.cardValue}>{ss.meterReading} {ss.meterUnit || ''}</Text>
          </View>
        )}
        
        {ss.tasks && ss.tasks.length > 0 && (
          <View style={styles.cardRow}>
            <Text style={styles.cardLabel}>Tasks:</Text>
            <Text style={styles.cardValue}>{ss.tasks.length} task(s)</Text>
          </View>
        )}
        
        {ss.daysToGo && (
          <View style={styles.cardRow}>
            <Text style={styles.cardLabel}>Days To Go:</Text>
            <Text style={styles.cardValue}>{ss.daysToGo}</Text>
          </View>
        )}
        
        {ss.country && (
          <View style={styles.cardRow}>
            <Text style={styles.cardLabel}>Country:</Text>
            <Text style={styles.cardValue}>{ss.country}</Text>
          </View>
        )}
        
        {ss.project && (
          <View style={styles.cardRow}>
            <Text style={styles.cardLabel}>Project:</Text>
            <Text style={styles.cardValue}>{ss.project}</Text>
          </View>
        )}
        
        {/* Show job card connection */}
        {(ss.jobCardId || ss.jobCardRequestNumber) && (
          <View style={styles.connectionsSection}>
            <Text style={styles.connectionsTitle}>Connected Job Card:</Text>
            <View style={styles.connectionItem}>
              <MaterialIcons name="work" size={16} color="#015185" />
              <Text style={styles.connectionText}>
                {ss.jobCardRequestNumber || ss.jobCardId}
                {ss.jobCardTitle && ` - ${ss.jobCardTitle}`}
                {ss.jobCardStatus && ` (${ss.jobCardStatus})`}
              </Text>
            </View>
          </View>
        )}
      </View>
    </View>
  );

  // Helper functions for colors
  const getStatusColor = (status) => {
    const statusLower = (status || '').toLowerCase();
    if (statusLower.includes('completed') || statusLower.includes('complete')) return '#4CAF50';
    if (statusLower.includes('progress')) return '#FF9800';
    if (statusLower.includes('hold')) return '#F44336';
    if (statusLower.includes('initiated')) return '#2196F3';
    if (statusLower.includes('requested')) return '#2196F3';
    if (statusLower.includes('pending')) return '#FF9800';
    if (statusLower.includes('upcoming')) return '#9C27B0';
    if (statusLower.includes('overdue')) return '#F44336';
    if (statusLower.includes('cancelled')) return '#9E9E9E';
    if (statusLower.includes('approved')) return '#4CAF50';
    if (statusLower.includes('assigned')) return '#00BCD4';
    return '#9E9E9E';
  };

  const getPriorityColor = (priority) => {
    const priorityLower = (priority || '').toLowerCase();
    if (priorityLower.includes('emergency')) return '#F44336';
    if (priorityLower.includes('high')) return '#FF9800';
    if (priorityLower.includes('medium')) return '#FFC107';
    if (priorityLower.includes('low')) return '#4CAF50';
    return '#9E9E9E';
  };

  // Enhanced KPI cards data with real-time calculations
  const kpiCardsData = [
    {
      id: '1',
      title: 'Total Maintenance Requests',
      value: kpiData.totalMaintenanceRequests,
      change: '+12%',
      changeType: 'positive',
      icon: 'build',
      color: '#3481BC',
      image: require('../../../Images/adminhome/maintananceKPI/carbon_license-maintenance-draft.png')
    },
    {
      id: '2',
      title: 'Pending Approval',
      value: kpiData.pendingApproval,
      change: '+5%',
      changeType: 'positive',
      icon: 'schedule',
      color: '#FF9800',
      image: require('../../../Images/adminhome/maintananceKPI/pendingApproval.png')
    },
    {
      id: '3',
      title: 'Works in Progress',
      value: kpiData.worksInProgress,
      change: '+8%',
      changeType: 'positive',
      icon: 'engineering',
      color: '#4CAF50',
      image: require('../../../Images/adminhome/maintananceKPI/workInProgress.png')
    },
    {
      id: '4',
      title: 'On Hold',
      value: kpiData.onHold,
      change: '-3%',
      changeType: 'negative',
      icon: 'pause-circle-outline',
      color: '#F44336'
    },
    {
      id: '5',
      title: 'Emergency Requests',
      value: kpiData.emergency,
      change: '+15%',
      changeType: 'positive',
      icon: 'warning',
      color: '#FF5722',
      image: require('../../../Images/adminhome/maintananceKPI/emergency.png')
    },
    {
      id: '6',
      title: 'Upcoming Schedules',
      value: kpiData.upcomingSchedules,
      change: '+20%',
      changeType: 'positive',
      icon: 'event',
      color: '#9C27B0',
      image: require('../../../Images/adminhome/maintananceKPI/upcoming_schedules.png')
    },
    {
      id: '7',
      title: 'Total Service Schedule Requests',
      value: kpiData.totalServiceSchedules,
      change: '+7%',
      changeType: 'positive',
      icon: 'event',
      color: '#9C27B0',
      image: require('../../../Images/adminhome/maintananceKPI/upcoming_schedules.png')
    },
    {
      id: '8',
      title: 'Equipment Utilization',
      value: `${kpiData.equipmentUtilization}%`,
      change: '+3%',
      changeType: 'positive',
      icon: 'analytics',
      color: '#00BCD4'
    }
  ];

  // Enhanced KPI card renderer
  const renderKpiCard = (item, index) => (
    <TouchableOpacity
      key={item.id}
      style={styles.kpiCard}
      activeOpacity={0.8}
    >
      <View style={styles.kpiCardGradient}>
        <View style={styles.kpiCardHeader}>
          {item.image ? (
            <Image source={item.image} style={styles.kpiCardImage} />
          ) : (
            <MaterialIcons name={item.icon} size={28} color="#015185" />
          )}
          <View style={styles.changeBadge}>
            <Text style={styles.changeText}>{item.change}</Text>
          </View>
        </View>
        <Text style={styles.kpiCardValue}>{item.value}</Text>
        <Text style={styles.kpiCardTitle}>{item.title}</Text>
      </View>
    </TouchableOpacity>
  );

  // Update useEffect to use enhanced fetching
  useEffect(() => {
    if (userInfo.uid) {
      fetchJobCardsWithConnections();
    }
  }, [fromDate, toDate, userInfo]);

  useEffect(() => {
    if (userInfo.uid) {
      fetchServiceSchedulesWithConnections();
    }
  }, [fromDate, toDate, userInfo]);

  useEffect(() => {
    if (userInfo.uid) {
      fetchMaintenanceRequestsWithDetails();
    }
  }, [userInfo]);

  // Date range state
  const [fromDate, setFromDate] = useState(() => {
    const d = new Date();
    d.setDate(d.getDate() - 6); // last 7 days
    d.setHours(0,0,0,0);
    return d;
  });
  const [toDate, setToDate] = useState(() => {
    const d = new Date();
    d.setHours(23,59,59,999);
    return d;
  });
  const [showFromPicker, setShowFromPicker] = useState(false);
  const [showToPicker, setShowToPicker] = useState(false);

  // Module dropdown state
  const moduleOptions = [
    'Job Cards Total',
    'Jobs in Progress',
    'Jobs On Hold',
    'Request Maintenance Total',
    'RM Pending Approval',
    'Service Schedule Total',
  ];
  const [selectedModule, setSelectedModule] = useState(moduleOptions[0]);
  const [moduleDropdownVisible, setModuleDropdownVisible] = useState(false);

  // Loading states
  const [jobCardsLoading, setJobCardsLoading] = useState(false);
  const [serviceSchedulesLoading, setServiceSchedulesLoading] = useState(false);

  // Filter by date range
  const isInRange = (date) => {
    return date >= fromDate && date <= toDate;
  };
  const filteredJobCards = jobCards.filter(jc => isInRange(jc.date));
  // Show all maintenance requests by default (no date filter)
  let filteredMaintenanceRequests = maintenanceRequests;
  if (fromDate && toDate) {
    filteredMaintenanceRequests = maintenanceRequests.filter(rm => {
      const date = rm.createdAt ? new Date(rm.createdAt) : null;
      return date && date >= fromDate && date <= toDate;
    });
  }
  const filteredServiceSchedules = serviceSchedules.filter(ss => isInRange(ss.date));

  // KPIs from real data
  const uniqueEquipments = Array.from(new Set(filteredJobCards.map(jc => jc.equipment)));
  const equipmentsUsed = uniqueEquipments.length;
  const jobCardsTotal = filteredJobCards.length;
  const jobsInProgress = filteredJobCards.filter(jc => jc.status === 'In Progress').length;
  const jobsOnHold = filteredJobCards.filter(jc => jc.status === 'On Hold').length;
  const serviceScheduleTotal = filteredServiceSchedules.length;

  // Helper to get a display title for job cards
  const getJobCardTitle = (jc) => jc.requestNumber || jc.title || jc.id;
  // Helper to get a display title for service schedules
  const getServiceScheduleTitle = (ss) => ss.serviceOrderNumber || ss.title || ss.id;

  // Fetch user info (country, project, permissions) on mount - MongoDB version
  useEffect(() => {
    const loadUserData = async () => {
      try {
        // Get user data from AsyncStorage (set during login)
        const userData = await AsyncStorage.getItem('userData');
        if (!userData) {
          Toast.show({
            type: 'error',
            text1: 'Error',
            text2: 'User not authenticated. Please log in.',
            position: 'bottom',
          });
          navigation.replace('AdminLogin');
          return;
        }
        
        const user = JSON.parse(userData);
        const uid = await AsyncStorage.getItem('userUid');

        if (!uid) {
          console.error('No user UID found');
          return;
        }

        console.log('👤 Loading user data for UID:', uid);

        // Extract requestMaintenance permissions
        const requestMaintenanceModule = user.modules?.find(mod => mod.module === 'requestMaintenance');
        const requestMaintenancePermissions = requestMaintenanceModule?.permissions || [];

        // Extract job card permissions
        const jobCardModule = user.modules?.find(mod => mod.module === 'jobCard' || mod.module === 'workOrder');
        const jobCardPermissions = jobCardModule?.permissions || [];

        const userInfoData = {
          countries: Array.isArray(user.countries) ? user.countries : [],
          projects: Array.isArray(user.projects) ? user.projects : [],
          requestMaintenancePermissions: requestMaintenancePermissions,
          jobCardPermissions: jobCardPermissions,
          name: user.name || user.fullName || 'Admin',
          email: user.email || '',
          uid: uid,
        };

        console.log('✅ User info loaded:', {
          countries: userInfoData.countries.length,
          projects: userInfoData.projects.length,
          requestMaintenancePermissions: userInfoData.requestMaintenancePermissions,
          jobCardPermissions: userInfoData.jobCardPermissions
        });

        setUserInfo(userInfoData);
      } catch (error) {
        console.error('❌ Error loading user data:', error);
        
        // Try to load from cache
        try {
          const cachedUserInfo = await AsyncStorage.getItem('userInfo');
          if (cachedUserInfo) {
            const parsed = JSON.parse(cachedUserInfo);
            setUserInfo({
              countries: Array.isArray(parsed.countries) ? parsed.countries : [],
              projects: Array.isArray(parsed.projects) ? parsed.projects : [],
              requestMaintenancePermissions: parsed.requestMaintenancePermissions || [],
              jobCardPermissions: parsed.jobCardPermissions || [],
              name: parsed.name || 'Admin',
              email: parsed.email || '',
              uid: parsed.userUid || parsed.uid || '',
            });
            console.log('✅ User info loaded from cache');
          }
        } catch (cacheError) {
          console.error('❌ Cache fallback failed:', cacheError);
        }
      }
    };
    loadUserData();
  }, []);

  // Recalculate KPIs when data or active tab changes
  useEffect(() => {
    calculateKPIs();
  }, [maintenanceRequests, jobCards, serviceSchedules, activeTab]);

  // Simulate data loading
  useEffect(() => {
    const timer = setTimeout(() => setIsLoading(false), 1200);
    return () => clearTimeout(timer);
  }, []);

  // State for Job Card tabs and modal
  const [jobCardTab, setJobCardTab] = useState('Job Cards');
  const [selectedJobCard, setSelectedJobCard] = useState(null);
  const [jobCardModalVisible, setJobCardModalVisible] = useState(false);

  // Get unique equipment objects from filtered job cards
  const equipmentMap = {};
  filteredJobCards.forEach(jc => {
    if (jc.equipment && typeof jc.equipment === 'object' && jc.equipment.equipmentNumber) {
      equipmentMap[jc.equipment.equipmentNumber] = jc.equipment;
    } else if (jc.equipment) {
      // fallback for string equipment
      equipmentMap[jc.equipment] = { equipmentNumber: jc.equipment };
    }
  });
  const jobCardEquipments = Object.values(equipmentMap);

  // Get assigned users from filtered job cards (if available)
  const assignedUsers = filteredJobCards.flatMap(jc => {
    // If you have a field like assignedTo or assignedUsers, use it. Otherwise, fallback to empty.
    if (jc.assignTo) {
      return Array.isArray(jc.assignTo) ? jc.assignTo.map(u => ({ jobCard: jc, user: u })) : [{ jobCard: jc, user: jc.assignTo }];
    }
    return [];
  });

  // Get unique inspections from filtered job cards
  const jobCardInspections = Array.from(new Set(filteredJobCards.map(jc => jc.linkInspection && jc.linkInspection.id))).filter(Boolean).map(inspectId => {
    // Find the first job card with this inspection to get more info
    const jc = filteredJobCards.find(j => j.linkInspection && j.linkInspection.id === inspectId);
    return jc ? { inspectionId: inspectId, ...jc.linkInspection } : { inspectionId: inspectId };
  });

  return (
    <View style={styles.container}>
      <StatusBar
        translucent
        backgroundColor="transparent"
        barStyle="light-content"
      />
      
      {/* Modern Header */}
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <View style={styles.headerContent}>
          <TouchableOpacity 
            onPress={() => navigation.goBack()}
            style={styles.backButton}
            activeOpacity={0.8}
          >
            <MaterialIcons name="arrow-back" size={24} color="white" />
          </TouchableOpacity>
          
          <Text style={styles.headerTitle}>Maintenance KPI Dashboard</Text>
          
          <TouchableOpacity 
            onPress={() => setModalVisible(true)} 
            style={styles.lockButton}
            activeOpacity={0.8}
          >
            <MaterialIcons name="lock" size={24} color="white" />
          </TouchableOpacity>
        </View>
      </LinearGradient>

      {/* Sync Modal */}
      <SyncModal visible={isLoading} message="Loading Maintenance KPI Data..." />

      {/* Permission Modal */}
      <Modal visible={modalVisible} transparent animationType="fade">
        <View style={{ flex: 1, backgroundColor: 'rgba(0,0,0,0.5)', justifyContent: 'center', alignItems: 'center' }}>
          <View style={{ backgroundColor: '#fff', borderRadius: 12, padding: 24, width: '80%', alignItems: 'flex-start', elevation: 5 }}>
            <Text style={{ fontSize: 20, fontWeight: 'bold', color: '#015185', marginBottom: 16, alignSelf: 'center', width: '100%', textAlign: 'center' }}>User Permissions</Text>
            <Text style={{ fontSize: 14, color: '#015185', fontWeight: '600', marginTop: 10 }}>Job Card Permissions:</Text>
            <Text style={{ fontSize: 14, color: '#333', marginBottom: 4 }}>{userInfo.requestMaintenancePermissions.length > 0 ? userInfo.requestMaintenancePermissions.join(', ') : 'None'}</Text>
            <Text style={{ fontSize: 14, color: '#015185', fontWeight: '600', marginTop: 10 }}>Countries:</Text>
            <Text style={{ fontSize: 14, color: '#333', marginBottom: 4 }}>{userInfo.countries.length > 0 ? userInfo.countries.join(', ') : 'None'}</Text>
            <Text style={{ fontSize: 14, color: '#015185', fontWeight: '600', marginTop: 10 }}>Projects:</Text>
            <Text style={{ fontSize: 14, color: '#333', marginBottom: 4 }}>{userInfo.projects.length > 0 ? userInfo.projects.join(', ') : 'None'}</Text>
            <TouchableOpacity style={{ alignSelf: 'center', marginTop: 18, backgroundColor: '#015185', borderRadius: 6, paddingHorizontal: 24, paddingVertical: 8 }} onPress={() => setModalVisible(false)}>
              <Text style={{ color: '#fff', fontWeight: 'bold', fontSize: 16 }}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      {/* Main Scrollable Content */}
      <ScrollView contentContainerStyle={{ paddingBottom: 24 }} showsVerticalScrollIndicator={false}>
        {/* Modern Time Period Tabs */}
        <View style={styles.tabContainer}>
          {['Today', '7 days', '30 days', '3 months'].map((tab) => (
          <TouchableOpacity 
              key={tab}
              style={[styles.tab, activeTab === tab && styles.activeTab]}
              onPress={() => setActiveTab(tab)}
          >
              <Text style={[styles.tabText, activeTab === tab && styles.activeTabText]}>
                {tab}
              </Text>
          </TouchableOpacity>
          ))}
        </View>

        {/* Modern KPI Cards Grid */}
        <View style={styles.kpiSection}>
          <Text style={styles.sectionTitle}>Performance Overview</Text>
          <Text style={styles.sectionSubtitle}>Maintenance metrics for {activeTab}</Text>
          
          <View style={styles.kpiGrid}>
            {kpiCardsData.map((item, index) => renderKpiCard(item, index))}
          </View>
        </View>

        {/* Date Range Picker (dynamic only) */}
        <View style={styles.dateRangeContainer}>
          <View style={{ flexDirection: 'row', alignItems: 'center', marginBottom: 6 }}>
            <TouchableOpacity
              style={{ backgroundColor: showAllMaintenanceRequests ? '#4682B4' : '#FFF', borderColor: '#4682B4', borderWidth: 1, borderRadius: 6, paddingHorizontal: 12, paddingVertical: 4, marginRight: 8 }}
              onPress={() => setShowAllMaintenanceRequests(true)}
            >
              <Text style={{ color: showAllMaintenanceRequests ? '#FFF' : '#4682B4', fontWeight: 'bold' }}>All Time</Text>
            </TouchableOpacity>
            <TouchableOpacity
              style={{ backgroundColor: !showAllMaintenanceRequests ? '#4682B4' : '#FFF', borderColor: '#4682B4', borderWidth: 1, borderRadius: 6, paddingHorizontal: 12, paddingVertical: 4 }}
              onPress={() => setShowAllMaintenanceRequests(false)}
            >
              <Text style={{ color: !showAllMaintenanceRequests ? '#FFF' : '#4682B4', fontWeight: 'bold' }}>Date Range</Text>
            </TouchableOpacity>
          </View>
          <View style={{ flexDirection: 'row', alignItems: 'center' }}>
            <TouchableOpacity
              style={styles.dateSelector}
              onPress={() => setShowFromPicker(true)}
              activeOpacity={0.7}
            >
              <MaterialIcons name="calendar-today" size={18} color="#4682B4" style={{ marginRight: 4 }} />
              <Text style={styles.dateSelectorText}>From {fromDate.toLocaleDateString()}</Text>
            </TouchableOpacity>
            <Text style={{ color: '#4682B4', fontSize: 16, marginHorizontal: 8 }}>to</Text>
            <TouchableOpacity
              style={styles.dateSelector}
              onPress={() => setShowToPicker(true)}
              activeOpacity={0.7}
            >
              <MaterialIcons name="calendar-today" size={18} color="#4682B4" style={{ marginRight: 4 }} />
              <Text style={styles.dateSelectorText}>{toDate.toLocaleDateString()}</Text>
            </TouchableOpacity>
          </View>
        </View>
        {showFromPicker && (
          <DateTimePicker
            value={fromDate}
            mode="date"
            display="default"
            onChange={(event, selectedDate) => {
              setShowFromPicker(false);
              if (selectedDate) setFromDate(new Date(selectedDate.setHours(0,0,0,0)));
            }}
          />
        )}
        {showToPicker && (
          <DateTimePicker
            value={toDate}
            mode="date"
            display="default"
            onChange={(event, selectedDate) => {
              setShowToPicker(false);
              if (selectedDate) setToDate(new Date(selectedDate.setHours(23,59,59,999)));
            }}
          />
        )}

        {/* KPIs List */}
        <View style={styles.kpiListContainer}>
          <View style={styles.kpiRow}><Text style={styles.kpiLabel}>Job Cards Total</Text><Text style={styles.kpiValue}>{jobCardsTotal}</Text></View>
          <View style={styles.kpiRow}><Text style={styles.kpiLabel}>Jobs in Progress</Text><Text style={styles.kpiValue}>{jobsInProgress}</Text></View>
          <View style={styles.kpiRow}><Text style={styles.kpiLabel}>Jobs On Hold</Text><Text style={styles.kpiValue}>{jobsOnHold}</Text></View>
          <View style={styles.kpiRow}><Text style={styles.kpiLabel}>Request Maintenance Total</Text><Text style={styles.kpiValue}>{maintenanceRequests.length}</Text></View>
          <View style={styles.kpiRow}><Text style={styles.kpiLabel}>RM Pending Approval</Text><Text style={styles.kpiValue}>{filteredMaintenanceRequests.filter(rm => (!rm.status || rm.status.trim() === '' || (rm.status && rm.status.toLowerCase() === 'pending approval'))).length}</Text></View>
          <View style={styles.kpiRow}><Text style={styles.kpiLabel}>Service Schedule Total</Text><Text style={styles.kpiValue}>{serviceScheduleTotal}</Text></View>
        </View>

        {/* Module Dropdown (Pure JS) */}
        <View style={{ marginHorizontal: 16, marginBottom: 16, backgroundColor: '#FFF', borderRadius: 8, borderWidth: 1, borderColor: '#EEE', padding: 8 }}>
          <Text style={{ color: '#4682B4', fontWeight: 'bold', marginBottom: 4 }}>Select Module:</Text>
          <TouchableOpacity
            style={{ borderWidth: 1, borderColor: '#4682B4', borderRadius: 6, padding: 10, backgroundColor: '#F5F7FA' }}
            onPress={() => setModuleDropdownVisible(true)}
          >
            <Text style={{ color: '#333' }}>{selectedModule}</Text>
          </TouchableOpacity>
          <Modal
            visible={moduleDropdownVisible}
            transparent
            animationType="fade"
            onRequestClose={() => setModuleDropdownVisible(false)}
          >
            <TouchableOpacity
              style={{ flex: 1, backgroundColor: 'rgba(0,0,0,0.3)', justifyContent: 'center', alignItems: 'center' }}
              activeOpacity={1}
              onPressOut={() => setModuleDropdownVisible(false)}
            >
              <View style={{ backgroundColor: '#FFF', borderRadius: 8, padding: 16, minWidth: 220 }}>
                {moduleOptions.map(mod => (
                  <TouchableOpacity key={mod} onPress={() => { setSelectedModule(mod); setModuleDropdownVisible(false); }}>
                    <View style={{ flexDirection: 'row', justifyContent: 'space-between', alignItems: 'center' }}>
                      <Text style={{ padding: 10, color: '#333' }}>{mod}</Text>
                      <Text style={{ padding: 10, color: '#015185', fontWeight: 'bold' }}>
                        {mod === 'Job Cards Total' ? filteredJobCards.length :
                         mod === 'Jobs in Progress' ? filteredJobCards.filter(jc => jc.status === 'In Progress').length :
                         mod === 'Jobs On Hold' ? filteredJobCards.filter(jc => jc.status === 'On Hold').length :
                         mod === 'Request Maintenance Total' ? maintenanceRequests.length :
                         mod === 'RM Pending Approval' ? filteredMaintenanceRequests.filter(rm => (!rm.status || rm.status.trim() === '' || (rm.status && rm.status.toLowerCase() === 'pending approval'))).length :
                         mod === 'Service Schedule Total' ? filteredServiceSchedules.length : 0}
                      </Text>
                    </View>
                  </TouchableOpacity>
                ))}
              </View>
            </TouchableOpacity>
          </Modal>
        </View>

        {/* Selected Module Count Row (below dropdown) */}
        <View style={styles.kpiListContainer}>
          <View style={styles.kpiRow}>
            <Text style={styles.kpiLabel}>{selectedModule}</Text>
            <Text style={styles.kpiValue}>
              {selectedModule === 'Job Cards Total' ? filteredJobCards.length :
               selectedModule === 'Jobs in Progress' ? filteredJobCards.filter(jc => jc.status === 'In Progress').length :
               selectedModule === 'Jobs On Hold' ? filteredJobCards.filter(jc => jc.status === 'On Hold').length :
               selectedModule === 'Request Maintenance Total' ? maintenanceRequests.length :
               selectedModule === 'RM Pending Approval' ? filteredMaintenanceRequests.filter(rm => (!rm.status || rm.status.trim() === '' || (rm.status && rm.status.toLowerCase() === 'pending approval'))).length :
               selectedModule === 'Service Schedule Total' ? filteredServiceSchedules.length : 0}
            </Text>
          </View>
        </View>

        {/* Module-specific data display */}
        {selectedModule === 'Request Maintenance Total' && (
          <View style={{ marginHorizontal: 16, marginBottom: 16 }}>
            <View style={styles.sectionHeader}>
              <Text style={styles.sectionTitle}>Request Maintenance List</Text>
              <Text style={styles.sectionCount}>({maintenanceRequests.length})</Text>
                    </View>
            {/* Show filtering mode */}
            <View style={styles.filterModeContainer}>
              <MaterialIcons name="filter-list" size={16} color="#015185" />
              <Text style={styles.filterModeText}>
                {userInfo.requestMaintenancePermissions.includes('onlyMineView') 
                  ? 'Showing only your requests' 
                  : `Filtered by: ${userInfo.countries.join(', ') || 'All Countries'} | ${userInfo.projects.join(', ') || 'All Projects'}`}
              </Text>
                    </View>
            {maintenanceRequests.length === 0 ? (
              <View style={styles.emptyState}>
                <MaterialIcons name="assignment" size={48} color="#CCC" />
                <Text style={styles.emptyText}>No maintenance requests available</Text>
                </View>
            ) : (
              maintenanceRequests.map((rm, index) => renderMaintenanceRequestItem(rm, index))
            )}
          </View>
        )}
        {selectedModule === 'Job Cards Total' && (
          <View style={{ marginHorizontal: 16, marginBottom: 16 }}>
            <View style={styles.sectionHeader}>
              <Text style={styles.sectionTitle}>Job Cards</Text>
              <Text style={styles.sectionCount}>({filteredJobCards.length})</Text>
            </View>
            {/* Show filtering mode */}
            <View style={styles.filterModeContainer}>
              <MaterialIcons name="filter-list" size={16} color="#015185" />
              <Text style={styles.filterModeText}>
                {userInfo.jobCardPermissions.includes('onlyMineView') 
                  ? 'Showing only your job cards' 
                  : `Filtered by: ${userInfo.countries.join(', ') || 'All Countries'} | ${userInfo.projects.join(', ') || 'All Projects'}`}
              </Text>
            </View>
            
            {/* Enhanced Tabs for Job Cards module */}
            <View style={styles.tabBar}>
              {['Job Cards', 'Assigned To User', 'Inspections'].map(tab => (
                <TouchableOpacity
                  key={tab}
                  style={[styles.tabButton, jobCardTab === tab && styles.activeTabButton]}
                  onPress={() => setJobCardTab(tab)}
                >
                  <Text style={[styles.tabButtonText, jobCardTab === tab && styles.activeTabButtonText]}>
                    {tab}
                  </Text>
                  <Text style={[styles.tabCount, jobCardTab === tab && styles.activeTabCount]}>
                    {tab === 'Job Cards' ? filteredJobCards.length :
                     tab === 'Assigned To User' ? assignedUsers.length :
                     tab === 'Inspections' ? jobCardInspections.length : 0}
                  </Text>
                </TouchableOpacity>
              ))}
            </View>
            
            {/* Tab content */}
            {jobCardTab === 'Job Cards' && (
              <View style={styles.tabContent}>
                {filteredJobCards.length === 0 ? (
                  <View style={styles.emptyState}>
                    <MaterialIcons name="work" size={48} color="#CCC" />
                    <Text style={styles.emptyText}>No job cards available</Text>
                      </View>
                ) : (
                  filteredJobCards.map((jc, index) => renderJobCardItem(jc, index))
                )}
              </View>
            )}
            {jobCardTab === 'Assigned To User' && (
              <View style={styles.tabContent}>
                {assignedUsers.length === 0 ? (
                  <View style={styles.emptyState}>
                    <MaterialIcons name="people" size={48} color="#CCC" />
                    <Text style={styles.emptyText}>No assigned users found</Text>
                    </View>
                ) : (
                  assignedUsers.map((au, index) => renderAssignedUserItem(au, index))
                )}
              </View>
            )}
            {jobCardTab === 'Inspections' && (
              <View style={styles.tabContent}>
                {jobCardInspections.length === 0 ? (
                  <View style={styles.emptyState}>
                    <MaterialIcons name="assignment" size={48} color="#CCC" />
                    <Text style={styles.emptyText}>No inspections found</Text>
                    </View>
                ) : (
                  jobCardInspections.map((ins, index) => renderInspectionItem(ins, index))
                )}
              </View>
            )}
            {/* Job Card Detail Modal */}
            <Modal visible={jobCardModalVisible} transparent animationType="slide" onRequestClose={() => setJobCardModalVisible(false)}>
              <View style={{ flex: 1, backgroundColor: 'rgba(0,0,0,0.4)', justifyContent: 'center', alignItems: 'center' }}>
                <View style={{ backgroundColor: '#FFF', borderRadius: 12, padding: 24, width: '90%', maxWidth: 400 }}>
                  {selectedJobCard && (
                    <ScrollView>
                      <Text style={{ fontSize: 18, fontWeight: 'bold', color: '#015185', marginBottom: 8 }}>Job Card Details</Text>
                      <Text style={{ color: '#015185', fontWeight: 'bold' }}>Request Number:</Text>
                      <Text style={{ color: '#333', marginBottom: 4 }}>{selectedJobCard.requestNumber || selectedJobCard.id}</Text>
                      <Text style={{ color: '#015185', fontWeight: 'bold' }}>Title:</Text>
                      <Text style={{ color: '#333', marginBottom: 4 }}>{selectedJobCard.title || '-'}</Text>
                      <Text style={{ color: '#015185', fontWeight: 'bold' }}>Status:</Text>
                      <Text style={{ color: '#333', marginBottom: 4 }}>{selectedJobCard.status}</Text>
                      <Text style={{ color: '#015185', fontWeight: 'bold' }}>Date:</Text>
                      <Text style={{ color: '#333', marginBottom: 4 }}>{selectedJobCard.date.toLocaleDateString()}</Text>
                      {/* Equipment Details */}
                      <Text style={{ color: '#015185', fontWeight: 'bold', marginTop: 8 }}>Equipment Details:</Text>
                      {selectedJobCard.equipment && typeof selectedJobCard.equipment === 'object' ? (
                        <View style={{ marginLeft: 8, marginBottom: 4 }}>
                          <Text style={{ color: '#333' }}>Name: {selectedJobCard.equipment.equipmentName || '-'}</Text>
                          <Text style={{ color: '#333' }}>Number: {selectedJobCard.equipment.equipmentNumber || '-'}</Text>
                          <Text style={{ color: '#333' }}>Model: {selectedJobCard.equipment.model || '-'}</Text>
                          <Text style={{ color: '#333' }}>Category: {selectedJobCard.equipment.equipmentCategory || '-'}</Text>
                          <Text style={{ color: '#333' }}>Status: {selectedJobCard.equipment.status || '-'}</Text>
                        </View>
                      ) : (
                        <Text style={{ color: '#333', marginLeft: 8, marginBottom: 4 }}>{selectedJobCard.equipment || '-'}</Text>
                      )}
                      {/* Assigned Users (root) */}
                      {selectedJobCard.assignTo && Array.isArray(selectedJobCard.assignTo) && selectedJobCard.assignTo.length > 0 && (
                        <View style={{ marginTop: 8 }}>
                          <Text style={{ color: '#015185', fontWeight: 'bold' }}>Assigned Users:</Text>
                          {selectedJobCard.assignTo.map((user, idx) => (
                            <Text key={user.id || idx} style={{ color: '#333', marginLeft: 8 }}>
                              - {user.fullName || user.name || user.email || user.employeeNumber || user.id}
                            </Text>
                          ))}
                        </View>
                      )}
                      {/* Tasks/Items (array) */}
                      {selectedJobCard.items && selectedJobCard.items.length > 0 && (
                        <View style={{ marginTop: 8 }}>
                          <Text style={{ color: '#015185', fontWeight: 'bold' }}>Tasks/Items:</Text>
                          {selectedJobCard.items.map((item, idx) => (
                            <View key={item.id || idx} style={{ marginBottom: 4, marginLeft: 8 }}>
                              <Text style={{ color: '#333' }}>- {item.text || item.note || item.category || ''} ({item.status || ''})</Text>
                              {/* Assigned users for this item */}
                              {item.assignTo && Array.isArray(item.assignTo) && item.assignTo.length > 0 && (
                                <View style={{ marginLeft: 8 }}>
                                  <Text style={{ color: '#015185' }}>Assigned To:</Text>
                                  {item.assignTo.map((user, uidx) => (
                                    <Text key={user.id || uidx} style={{ color: '#333', marginLeft: 8 }}>
                                      - {user.fullName || user.name || user.email || user.employeeNumber || user.id}
                                    </Text>
                                  ))}
                                </View>
                              )}
                            </View>
                          ))}
                        </View>
                      )}
                      {/* Tasks (object) */}
                      {selectedJobCard.tasks && Object.values(selectedJobCard.tasks).length > 0 && (
                        <View style={{ marginTop: 8 }}>
                          <Text style={{ color: '#015185', fontWeight: 'bold' }}>Tasks:</Text>
                          {Object.values(selectedJobCard.tasks).map((task, idx) => (
                            <View key={task.id || idx} style={{ marginBottom: 4, marginLeft: 8 }}>
                              <Text style={{ color: '#333' }}>- {task.text || task.note || task.category || ''} ({task.status || ''})</Text>
                            </View>
                          ))}
                        </View>
                      )}
                      {/* Linked Inspection */}
                      {selectedJobCard.linkInspection && (
                        <View style={{ marginTop: 8 }}>
                          <Text style={{ color: '#015185', fontWeight: 'bold' }}>Linked Inspection:</Text>
                          <Text style={{ color: '#333', marginLeft: 8 }}>{selectedJobCard.linkInspection.name || selectedJobCard.linkInspection.id}</Text>
                        </View>
                      )}
                      {/* Linked Work Order */}
                      {selectedJobCard.linkedWorkOrder && (
                        <View style={{ marginTop: 8 }}>
                          <Text style={{ color: '#015185', fontWeight: 'bold' }}>Linked Work Order:</Text>
                          <Text style={{ color: '#333', marginLeft: 8 }}>{selectedJobCard.linkedWorkOrder.requestNumber || selectedJobCard.linkedWorkOrder.id}</Text>
                          <Text style={{ color: '#333', marginLeft: 8 }}>{selectedJobCard.linkedWorkOrder.title || ''}</Text>
                        </View>
                      )}
                      <TouchableOpacity style={{ marginTop: 18, backgroundColor: '#015185', borderRadius: 6, paddingHorizontal: 24, paddingVertical: 8, alignSelf: 'center' }} onPress={() => setJobCardModalVisible(false)}>
                        <Text style={{ color: '#fff', fontWeight: 'bold', fontSize: 16 }}>Close</Text>
                      </TouchableOpacity>
                    </ScrollView>
                  )}
                </View>
              </View>
            </Modal>
          </View>
        )}
        {selectedModule === 'Jobs in Progress' && (
          <View style={{ marginHorizontal: 16, marginBottom: 16 }}>
            <View style={styles.sectionHeader}>
              <Text style={styles.sectionTitle}>Jobs In Progress</Text>
              <Text style={styles.sectionCount}>({filteredJobCards.filter(jc => jc.status === 'In Progress').length})</Text>
            </View>
            {filteredJobCards.filter(jc => jc.status === 'In Progress').length === 0 ? (
              <View style={styles.emptyState}>
                <MaterialIcons name="engineering" size={48} color="#CCC" />
                <Text style={styles.emptyText}>No jobs in progress</Text>
                </View>
            ) : (
              filteredJobCards.filter(jc => jc.status === 'In Progress').map((jc, index) => renderJobCardItem(jc, index))
            )}
          </View>
        )}
        {selectedModule === 'Jobs On Hold' && (
          <View style={{ marginHorizontal: 16, marginBottom: 16 }}>
            <View style={styles.sectionHeader}>
              <Text style={styles.sectionTitle}>Jobs On Hold</Text>
              <Text style={styles.sectionCount}>({filteredJobCards.filter(jc => jc.status === 'On Hold').length})</Text>
            </View>
            {filteredJobCards.filter(jc => jc.status === 'On Hold').length === 0 ? (
              <View style={styles.emptyState}>
                <MaterialIcons name="pause-circle-outline" size={48} color="#CCC" />
                <Text style={styles.emptyText}>No jobs on hold</Text>
                </View>
            ) : (
              filteredJobCards.filter(jc => jc.status === 'On Hold').map((jc, index) => renderJobCardItem(jc, index))
            )}
          </View>
        )}
        {selectedModule === 'RM Pending Approval' && (
          <View style={{ marginHorizontal: 16, marginBottom: 16 }}>
            <View style={styles.sectionHeader}>
              <Text style={styles.sectionTitle}>Pending Approval Requests</Text>
              <Text style={styles.sectionCount}>({filteredMaintenanceRequests.filter(rm => (!rm.status || rm.status.trim() === '' || (rm.status && rm.status.toLowerCase() === 'pending approval'))).length})</Text>
            </View>
            {filteredMaintenanceRequests.filter(rm => (!rm.status || rm.status.trim() === '' || (rm.status && rm.status.toLowerCase() === 'pending approval'))).length === 0 ? (
              <View style={styles.emptyState}>
                <MaterialIcons name="pending" size={48} color="#CCC" />
                <Text style={styles.emptyText}>No pending approval requests</Text>
                    </View>
                  ) : (
              filteredMaintenanceRequests.filter(rm => (!rm.status || rm.status.trim() === '' || (rm.status && rm.status.toLowerCase() === 'pending approval'))).map((rm, index) => renderMaintenanceRequestItem(rm, index))
            )}
                    </View>
                  )}
        {selectedModule === 'Service Schedule Total' && (
          <View style={{ marginHorizontal: 16, marginBottom: 16 }}>
            <View style={styles.sectionHeader}>
              <Text style={styles.sectionTitle}>Service Schedules</Text>
              <Text style={styles.sectionCount}>({filteredServiceSchedules.length})</Text>
                </View>
            {/* Show filtering mode */}
            <View style={styles.filterModeContainer}>
              <MaterialIcons name="filter-list" size={16} color="#015185" />
              <Text style={styles.filterModeText}>
                {userInfo.jobCardPermissions.includes('onlyMineView') 
                  ? 'Showing only your service schedules' 
                  : `Filtered by: ${userInfo.countries.join(', ') || 'All Countries'} | ${userInfo.projects.join(', ') || 'All Projects'}`}
              </Text>
          </View>
            {filteredServiceSchedules.length === 0 ? (
              <View style={styles.emptyState}>
                <MaterialIcons name="schedule" size={48} color="#CCC" />
                <Text style={styles.emptyText}>No service schedules available</Text>
                </View>
            ) : (
              filteredServiceSchedules.map((ss, index) => renderServiceScheduleItem(ss, index))
            )}
          </View>
        )}
      </ScrollView>
      
      {/* Sync Modal */}
      <SyncModal visible={syncModalVisible} message="Syncing data..." />
      
      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#f5f6fa',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
    elevation: 4,
    shadowColor: '#000',
    shadowOpacity: 0.1,
    shadowRadius: 8,
    shadowOffset: { width: 0, height: 4 },
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 16,
    paddingTop: Platform.OS === 'ios' ? 32 : 16,
    height: '100%',
    top: 7,
  },
  backButton: {
    position: 'absolute',
    left: 16,
    padding: 8,
    top: 44,
  },
  headerTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    color: 'white',
  },
  lockButton: {
    position: 'absolute',
    right: 16,
    padding: 8,
    top: 44,
  },
  buttonContainer: {
    flexDirection: 'column', // Buttons in one column
    alignItems: 'center', // Center the buttons
    paddingHorizontal: 8,
    marginVertical: 20,
  },
  button: {
    backgroundColor: '#4682B4', // Solid color for buttons
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
    width: '40%', // Adjusted width to fit the column layout
    marginVertical: 10, // Space between buttons
  },
  buttonText: {
    color: '#FFF',
    fontSize: 16,
    marginTop: 10,
    textAlign: 'center',
  },
  // Modern KPI Section Styles
  kpiSection: {
    padding: 20,
    marginBottom: 20,
  },
  sectionTitle: {
    fontSize: 24,
    fontWeight: 'bold',
    color: '#035484',
    marginBottom: 5,
    textAlign: 'center',
  },
  sectionSubtitle: {
    fontSize: 14,
    color: '#666',
    marginBottom: 20,
    textAlign: 'center',
  },
  kpiGrid: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    justifyContent: 'space-between',
  },
  kpiCard: {
    width: (width - 60) / 2,
    marginBottom: 15,
    borderRadius: 12,
    overflow: 'hidden',
    backgroundColor: '#FFF', // ensure white
    borderColor: '#EEE',
    borderWidth: 1,
    shadowColor: '#000',
    shadowOffset: {
      width: 0,
      height: 2,
    },
    shadowOpacity: 0.1,
    shadowRadius: 3.84,
    elevation: 5,
  },
  kpiCardGradient: {
    padding: 20,
    alignItems: 'center',
    backgroundColor: '#FFF', // ensure white
  },
  kpiCardHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    width: '100%',
    marginBottom: 10,
  },
  kpiCardImage: {
    width: 28,
    height: 28,
    resizeMode: 'contain',
  },
  changeBadge: {
    paddingHorizontal: 8,
    paddingVertical: 4,
    borderRadius: 12,
    borderWidth: 1,
    borderColor: '#EEE',
    backgroundColor: '#F8F8F8', // light gray, not green/red
  },
  changeText: {
    color: '#333',
    fontSize: 12,
    fontWeight: 'bold',
  },
  kpiCardValue: {
    fontSize: 28,
    fontWeight: 'bold',
    color: '#015185', // or #333
    marginBottom: 5,
  },
  kpiCardTitle: {
    fontSize: 12,
    color: '#333',
    textAlign: 'center',
    lineHeight: 16,
  },
  tabContainer: {
    flexDirection: 'row',
    justifyContent: 'center',
    alignItems: 'center',
    paddingHorizontal: 20,
    marginVertical: 20,
    backgroundColor: 'white',
    marginHorizontal: 20,
    borderRadius: 12,
    paddingVertical: 8,
    shadowColor: '#000',
    shadowOffset: {
      width: 0,
      height: 2,
    },
    shadowOpacity: 0.1,
    shadowRadius: 3.84,
    elevation: 5,
  },
  tab: {
    paddingVertical: 12,
    paddingHorizontal: 16,
    borderRadius: 8,
    backgroundColor: 'transparent',
    alignItems: 'center',
    justifyContent: 'center',
    minWidth: 80,
  },
  activeTab: {
    backgroundColor: '#035484',
  },
  tabText: {
    color: '#035484',
    fontSize: 14,
    fontWeight: '500',
  },
  activeTabText: {
    color: 'white',
  },
  // Enhanced Loader Styles
  loaderOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  loaderContainer: {
    width: '70%',
    maxWidth: 300,
  },
  loaderGradient: {
    borderRadius: 12,
    padding: 30,
    alignItems: 'center',
  },
  loaderIcon: {
    width: 40,
    height: 40,
    marginBottom: 15,
  },
  loaderText: {
    fontSize: 16,
    color: 'white',
    fontWeight: '500',
    textAlign: 'center',
  },
  dateRangeContainer: {
    backgroundColor: '#F5F7FA',
    padding: 10,
    marginHorizontal: 20,
    borderRadius: 10,
    alignItems: 'center',
    marginBottom: 10,
  },
  dateRangeText: {
    fontSize: 14,
    color: '#4682B4',
  },
  list: {
    flex: 1,
    backgroundColor: '#FFF',
    borderTopLeftRadius: 20,
    borderTopRightRadius: 20,
    paddingHorizontal: 20,
    paddingTop: 10,
  },
  listItem: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 10,
    borderBottomWidth: 1,
    borderBottomColor: '#EEE',
  },
  listIcon: {
    marginRight: 15,
  },
  listImage: {
    width: 24,
    height: 24,
    marginRight: 15,
  },
  listText: {
    flex: 1,
    fontSize: 16,
    color: '#333',
  },
  countBadge: {
    backgroundColor: '#EEE',
    borderRadius: 15,
    paddingHorizontal: 10,
    paddingVertical: 5,
  },
  countText: {
    fontSize: 14,
    color: '#333',
  },
  kpiListContainer: {
    backgroundColor: '#FFF',
    borderRadius: 12,
    margin: 16,
    padding: 8,
    elevation: 1,
  },
  kpiRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingVertical: 10,
    borderBottomWidth: 1,
    borderBottomColor: '#EEE',
  },
  kpiLabel: {
    fontSize: 15,
    color: '#4682B4',
    fontWeight: '500',
  },
  kpiValue: {
    fontSize: 16,
    color: '#333',
    fontWeight: 'bold',
  },
  // New styles for data cards
  dataCard: {
    backgroundColor: '#FFF',
    borderRadius: 12,
    marginBottom: 12,
    padding: 16,
    borderWidth: 1,
    borderColor: '#EEE',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.05,
    shadowRadius: 4,
    elevation: 2,
  },
  cardHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 8,
  },
  cardTitleSection: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  cardTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#015185',
    marginRight: 8,
  },
  statusBadge: {
    paddingHorizontal: 8,
    paddingVertical: 4,
    borderRadius: 6,
  },
  statusText: {
    color: 'white',
    fontSize: 12,
    fontWeight: 'bold',
  },
  cardDate: {
    fontSize: 14,
    color: '#888',
  },
  cardContent: {
    marginTop: 8,
  },
  cardRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 4,
  },
  cardLabel: {
    fontSize: 14,
    color: '#666',
    fontWeight: '500',
  },
  cardValue: {
    fontSize: 15,
    color: '#333',
    fontWeight: 'bold',
  },
  connectionsSection: {
    marginTop: 12,
    paddingTop: 12,
    borderTopWidth: 1,
    borderTopColor: '#EEE',
  },
  connectionsTitle: {
    fontSize: 14,
    color: '#015185',
    fontWeight: 'bold',
    marginBottom: 8,
  },
  connectionItem: {
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: 4,
    paddingVertical: 4,
    paddingHorizontal: 8,
    borderRadius: 6,
    backgroundColor: '#F5F7FA',
  },
  connectionText: {
    fontSize: 13,
    color: '#333',
    marginLeft: 8,
  },
  // Enhanced section styles
  sectionHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 16,
    paddingHorizontal: 4,
  },
  sectionTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#015185',
  },
  sectionCount: {
    fontSize: 16,
    color: '#666',
    fontWeight: '500',
  },
  // Enhanced tab styles
  tabBar: {
    flexDirection: 'row',
    backgroundColor: '#F5F7FA',
    borderRadius: 12,
    padding: 4,
    marginBottom: 16,
  },
  tabButton: {
    flex: 1,
    paddingVertical: 12,
    paddingHorizontal: 8,
    borderRadius: 8,
    alignItems: 'center',
  },
  activeTabButton: {
    backgroundColor: '#015185',
  },
  tabButtonText: {
    fontSize: 14,
    fontWeight: '500',
    color: '#666',
  },
  activeTabButtonText: {
    color: 'white',
  },
  tabCount: {
    fontSize: 12,
    color: '#999',
    marginTop: 2,
  },
  activeTabCount: {
    color: 'rgba(255,255,255,0.8)',
  },
  tabContent: {
    marginTop: 8,
  },
  // Empty state styles
  emptyState: {
    alignItems: 'center',
    paddingVertical: 40,
    backgroundColor: '#F5F7FA',
    borderRadius: 12,
  },
  emptyText: {
    fontSize: 16,
    color: '#999',
    marginTop: 12,
    textAlign: 'center',
  },
  // Filter mode styles
  filterModeContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#F0F8FF',
    paddingHorizontal: 12,
    paddingVertical: 8,
    borderRadius: 8,
    marginBottom: 12,
  },
  filterModeText: {
    fontSize: 12,
    color: '#015185',
    marginLeft: 6,
    fontStyle: 'italic',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 30,
    height: 30,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  dateSelector: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#FFF',
    borderColor: '#4682B4',
    borderWidth: 1,
    borderRadius: 8,
    paddingHorizontal: 10,
    paddingVertical: 6,
    marginVertical: 4,
    minWidth: 120,
    justifyContent: 'center',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 1 },
    shadowOpacity: 0.05,
    shadowRadius: 2,
    elevation: 1,
  },
  dateSelectorText: {
    color: '#4682B4',
    fontWeight: 'bold',
    fontSize: 15,
  },
});

export default MaintenanceKpiHome;