import React, { useState, useEffect } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  TextInput,
  ScrollView,
  Dimensions,
  StatusBar,
  Modal,
  Alert,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation, useRoute } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import DateTimePicker from '@react-native-community/datetimepicker';
import { Dropdown } from 'react-native-element-dropdown';

const { width } = Dimensions.get('window');

const UNIT_OPTIONS = ['Miles', 'KM', 'Hours'];
const TIME_UNIT_OPTIONS = ['Day', 'Week', 'Month'];

const EditServiceSchedule = () => {
  const navigation = useNavigation();
  const route = useRoute();
  const { schedule } = route.params || {};

  // Define parseDate function to handle DD-MMM-YYYY format
  const parseDate = (dateString) => {
    try {
      const [day, monthStr, year] = dateString.split('-');
      const month = new Date(Date.parse(monthStr + ' 1, 2020')).getMonth();
      const parsedDate = new Date(Number(year), month, Number(day));
      return isNaN(parsedDate.getTime()) ? new Date() : parsedDate;
    } catch (error) {
      console.error('Error parsing date:', error);
      return new Date();
    }
  };

  const [repeatType, setRepeatType] = useState(schedule?.repeatType || 'Repeat');
  const [intervals, setIntervals] = useState(
    schedule?.intervals?.length > 0
      ? schedule.intervals
      : [{ checked: true, value: '', unit: 'Miles' }, { checked: false, value: '', unit: 'Miles' }]
  );
  const [intervalUnit, setIntervalUnit] = useState(schedule?.intervalUnit || 'Day');
  const [notifyBefore, setNotifyBefore] = useState(
    schedule?.notifyBefore || { value: '', unit: schedule?.meterUnit || 'Miles', time: 'Day' }
  );
  const [selectedEquipment, setSelectedEquipment] = useState(
    schedule?.equipment
      ? {
          equipmentName: schedule.equipment,
          meterReading: schedule.meterReading,
          meterUnit: schedule.meterUnit || 'Miles',
        }
      : null
  );
  const [scheduleTitle, setScheduleTitle] = useState(schedule?.title || '');
  const [note, setNote] = useState(schedule?.note || '');
  const [lastServiceDate, setLastServiceDate] = useState(
    schedule?.lastServiceDate ? parseDate(schedule.lastServiceDate) : new Date()
  );
  const [tempDate, setTempDate] = useState(lastServiceDate);
  const [datePickerVisible, setDatePickerVisible] = useState(false);
  const [scheduleId, setScheduleId] = useState(schedule?.scheduleId || '');
  const [meterReading, setMeterReading] = useState(schedule?.meterReading || '');
  const [meterUnit, setMeterUnit] = useState(schedule?.meterUnit || 'Miles');
  const [tasks, setTasks] = useState(schedule?.tasks || []); // Initialize tasks from schedule
  const [taskModalVisible, setTaskModalVisible] = useState(false);
  const [newTask, setNewTask] = useState('');

  useEffect(() => {
    if (!scheduleId && !schedule) {
      const generateScheduleId = async () => {
        try {
          const existingSchedules = await AsyncStorage.getItem('schedules');
          const schedules = existingSchedules ? JSON.parse(existingSchedules) : [];
          const newId = `SCH${1001 + schedules.length}-${schedules.length + 1}`;
          setScheduleId(newId);
        } catch (error) {
          console.error('Error generating schedule ID:', error);
          setScheduleId('SCH1001-1');
        }
      };
      generateScheduleId();
    }
    setTempDate(lastServiceDate);
  }, [scheduleId, schedule, lastServiceDate]);

  const handleEquipmentSelect = (equipmentData) => {
    const unit = equipmentData?.meterUnit || 'Miles';
    setSelectedEquipment({ ...equipmentData, meterUnit: unit });
    setMeterReading(equipmentData?.meterReading || '');
    setMeterUnit(unit);
    setNotifyBefore((prev) => ({ ...prev, unit }));
    setIntervals((prev) =>
      prev.map((interval) => ({
        ...interval,
        unit,
      }))
    );
  };

  const formatDate = (date) => {
    if (!date || isNaN(date.getTime())) return 'Select Date';
    return date.toLocaleDateString('en-GB', {
      day: '2-digit',
      month: 'short',
      year: 'numeric',
    }).replace(/ /g, '-');
  };

  const openDatePicker = () => {
    setTempDate(lastServiceDate || new Date());
    setDatePickerVisible(true);
  };

  const handleDateChange = (event, selectedDate) => {
    if (event.type === 'set' && selectedDate) {
      setTempDate(selectedDate);
      setLastServiceDate(selectedDate);
    }
    setDatePickerVisible(false);
  };

  const openTaskModal = () => {
    setNewTask('');
    setTaskModalVisible(true);
  };

  const handleTaskConfirm = () => {
    if (newTask.trim()) {
      setTasks([...tasks, newTask.trim()]);
    }
    setTaskModalVisible(false);
    setNewTask('');
  };

  const handleTaskCancel = () => {
    setTaskModalVisible(false);
    setNewTask('');
  };

  const handleUnitSelect = (unit, index) => {
    setIntervals((prev) =>
      prev.map((item, i) => (i === index ? { ...item, unit } : item))
    );
  };

  const handleMeterUnitSelect = (unit) => {
    setMeterUnit(unit);
    setNotifyBefore((prev) => ({ ...prev, unit }));
    setIntervals((prev) => prev.map((interval) => ({ ...interval, unit })));
  };

  const handleUpdateSchedule = async () => {
    // Validate required fields
    if (!lastServiceDate || isNaN(lastServiceDate.getTime())) {
      Alert.alert('Error', 'Please select a valid last service date.');
      return;
    }
    if (!scheduleTitle.trim()) {
      Alert.alert('Error', 'Please enter a schedule title.');
      return;
    }
    if (!selectedEquipment) {
      Alert.alert('Error', 'Please select an equipment.');
      return;
    }

    const updatedSchedule = {
      ...schedule,
      id: schedule?.id || Date.now().toString(),
      scheduleId: scheduleId || schedule?.scheduleId,
      title: scheduleTitle.trim(),
      equipment: selectedEquipment.equipmentName,
      note: note.trim(),
      lastServiceDate: formatDate(lastServiceDate),
      meterReading: meterReading.trim(),
      meterUnit,
      repeatType,
      intervals: intervals.filter((interval) => interval.checked && interval.value.trim()),
      intervalUnit,
      notifyBefore: {
        ...notifyBefore,
        value: notifyBefore.value.trim(),
      },
      tasks, // Include tasks in the updated schedule
      status: schedule?.status || 'upcoming',
      createdAt: schedule?.createdAt || new Date().toISOString(),
      dueHours: schedule?.dueHours || '12',
      hoursToGo: schedule?.hoursToGo || '10',
      daysToGo: schedule?.daysToGo || '69',
    };

    try {
      console.log('Attempting to update schedule:', updatedSchedule);

      // Retrieve existing schedules
      const existingSchedules = await AsyncStorage.getItem('schedules');
      let schedules = existingSchedules ? JSON.parse(existingSchedules) : [];

      // Update or add schedule
      let isUpdated = false;
      if (schedule && schedule.id) {
        schedules = schedules.map((item) => {
          if (item.id === schedule.id || item.scheduleId === schedule.scheduleId) {
            isUpdated = true;
            return updatedSchedule;
          }
          return item;
        });
      }

      if (!isUpdated) {
        schedules.push(updatedSchedule);
      }

      // Save to AsyncStorage
      await AsyncStorage.setItem('schedules', JSON.stringify(schedules));
      console.log('Schedule updated successfully:', updatedSchedule);
      console.log('All schedules in storage:', schedules);

      // Show success message
      Alert.alert('Success', 'Schedule updated successfully!', [
        {
          text: 'OK',
          onPress: () => navigation.navigate('ServiceScheduleHome'),
        },
      ]);
    } catch (error) {
      console.error('Error updating schedule:', error);
      Alert.alert('Error', 'Failed to update schedule. Please try again.');
    }
  };

  return (
    <View style={styles.container}>
      <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
      <View style={styles.header}>
        <TouchableOpacity onPress={() => navigation.goBack()} style={styles.headerLeft}>
          <MaterialIcons name="arrow-back" size={24} color="#fff" />
          <Text style={styles.headerTitle}>Update Schedule</Text>
        </TouchableOpacity>
        <TouchableOpacity style={styles.createButton} onPress={handleUpdateSchedule}>
          <Text style={styles.createButtonText}>Update</Text>
        </TouchableOpacity>
      </View>

      <ScrollView style={styles.formContainer} contentContainerStyle={{ paddingBottom: 30 }}>
        <View style={styles.rowBetween}>
          <Text style={styles.label}>Schedule ID</Text>
          <Text style={styles.value}>{scheduleId}</Text>
        </View>

        <Text style={styles.label}>Write Schedule</Text>
        <TextInput
          style={styles.input}
          placeholder="Write Request Title"
          placeholderTextColor="#A0A0A0"
          value={scheduleTitle}
          onChangeText={setScheduleTitle}
        />

        <Text style={styles.label}>Write Note</Text>
        <View style={styles.inputWithIcon}>
          <TextInput
            style={[styles.input, { flex: 1, marginBottom: 0 }]}
            placeholder="Write Note"
            placeholderTextColor="#A0A0A0"
            value={note}
            onChangeText={setNote}
          />
          <MaterialIcons name="keyboard-voice" size={22} color="#015185" style={{ marginLeft: 8 }} />
        </View>

        <View style={styles.rowBetween}>
          <Text style={styles.label}>Equipments</Text>
          <TouchableOpacity
            onPress={() => navigation.navigate('Select_Equipment', { source: 'EditSchedule', onSelect: handleEquipmentSelect })}
          >
            <Text style={styles.selectText}>Select</Text>
          </TouchableOpacity>
        </View>
        {selectedEquipment ? (
          <View style={styles.selectedEquipmentContainer}>
            <Text style={styles.selectedEquipmentText}>{selectedEquipment.equipmentName}</Text>
          </View>
        ) : (
          <Text style={styles.placeholderText}>No equipment selected</Text>
        )}

        <View style={styles.row}>
          <TextInput
            style={[styles.input, { flex: 1 }]}
            placeholder="Last Service Meter Reading"
            placeholderTextColor="#A0A0A0"
            value={meterReading}
            onChangeText={setMeterReading}
            keyboardType="numeric"
          />
          <Dropdown
            style={styles.unitDropdown}
            placeholderStyle={styles.unitDropdownText}
            selectedTextStyle={styles.unitDropdownText}
            data={UNIT_OPTIONS.map((unit) => ({ label: unit, value: unit }))}
            maxHeight={150}
            labelField="label"
            valueField="value"
            placeholder="Unit"
            value={meterUnit}
            onChange={(item) => handleMeterUnitSelect(item.value)}
            renderRightIcon={() => (
              <MaterialIcons name="arrow-drop-down" size={22} color="#015185" />
            )}
          />
        </View>

        <View style={styles.row}>
          <TouchableOpacity
            style={[styles.input, { flex: 1, justifyContent: 'center' }]}
            onPress={openDatePicker}
          >
            <Text style={{ color: lastServiceDate && !isNaN(lastServiceDate.getTime()) ? '#222' : '#A0A0A0', fontSize: 15 }}>
              {formatDate(lastServiceDate)}
            </Text>
          </TouchableOpacity>
          <TouchableOpacity onPress={openDatePicker}>
            <MaterialIcons name="calendar-today" size={22} color="#015185" style={{ marginLeft: 8, marginBottom: width * 0.04 }} />
          </TouchableOpacity>
        </View>

        {datePickerVisible && (
          <DateTimePicker
            value={tempDate || new Date()}
                mode="date"
            display="default"
                maximumDate={new Date()}
            onChange={handleDateChange}
          />
        )}

        {/* Task Section */}
        <View style={styles.taskHeader}>
          <Text style={styles.label}>Task</Text>
          <TouchableOpacity style={styles.addTaskButton} onPress={openTaskModal}>
            <Text style={styles.addTaskText}>+ Add Task</Text>
          </TouchableOpacity>
        </View>
        {tasks.length > 0 ? (
          <View style={styles.taskList}>
            {tasks.map((task, idx) => (
              <Text key={idx} style={styles.taskItem}>- {task}</Text>
            ))}
          </View>
        ) : (
          <Text style={styles.placeholderText}>No tasks added</Text>
        )}

        <Modal visible={taskModalVisible} transparent={true} animationType="fade" onRequestClose={handleTaskCancel}>
          <View style={styles.modalOverlay}>
            <View style={styles.taskModalContainer}>
              <Text style={styles.modalTitle}>Add New Task</Text>
              <TextInput
                style={styles.modalInput}
                placeholder="Enter task name"
                placeholderTextColor="#A0A0A0"
                value={newTask}
                onChangeText={setNewTask}
              />
              <View style={styles.modalButtons}>
                <TouchableOpacity style={[styles.modalButton, styles.cancelButton]} onPress={handleTaskCancel}>
                  <Text style={styles.modalButtonText}>Cancel</Text>
                </TouchableOpacity>
                <TouchableOpacity style={[styles.modalButton, styles.doneButton]} onPress={handleTaskConfirm}>
                  <Text style={styles.modalButtonText}>Done</Text>
                </TouchableOpacity>
              </View>
            </View>
          </View>
        </Modal>

        <View style={styles.row}>
          <Text style={styles.label}>Repeat Type</Text>
          <TouchableOpacity
            style={[styles.repeatButton, repeatType === 'Repeat' && styles.repeatButtonActive]}
            onPress={() => setRepeatType('Repeat')}
          >
            <Text style={[styles.repeatButtonText, repeatType === 'Repeat' && styles.repeatButtonTextActive]}>Repeat</Text>
          </TouchableOpacity>
          <TouchableOpacity
            style={[styles.repeatButton, repeatType === 'One time' && styles.repeatButtonActive]}
            onPress={() => setRepeatType('One time')}
          >
            <Text style={[styles.repeatButtonText, repeatType === 'One time' && styles.repeatButtonTextActive]}>One time</Text>
          </TouchableOpacity>
        </View>

        <Text style={styles.labelSmall}>Intervals (whichever comes first)</Text>
        {intervals.map((interval, idx) => (
          <View key={idx} style={styles.row}>
            <TouchableOpacity
              style={styles.checkbox}
              onPress={() => {
                setIntervals(intervals.map((item, i) => (i === idx ? { ...item, checked: !item.checked } : item)));
              }}
            >
              <MaterialIcons
                name={interval.checked ? 'check-box' : 'check-box-outline-blank'}
                size={22}
                color={interval.checked ? '#015185' : '#A0A0A0'}
              />
            </TouchableOpacity>
            <Text style={styles.everyText}>Every</Text>
            <TextInput
              style={[styles.input, { width: 50, marginBottom: 0 }]}
              value={interval.value}
              onChangeText={(text) => setIntervals(intervals.map((item, i) => (i === idx ? { ...item, value: text } : item)))}
              keyboardType="numeric"
              placeholder="0"
              placeholderTextColor="#A0A0A0"
              editable={interval.checked}
            />
            <Dropdown
              style={styles.intervalUnitDropdown}
              placeholderStyle={styles.unitDropdownText}
              selectedTextStyle={styles.unitDropdownText}
              data={UNIT_OPTIONS.map((unit) => ({ label: unit, value: unit }))}
              maxHeight={150}
              labelField="label"
              valueField="value"
              placeholder="Unit"
              value={interval.unit}
              onChange={(item) => handleUnitSelect(item.value, idx)}
              renderRightIcon={() => (
                <MaterialIcons name="arrow-drop-down" size={22} color="#015185" />
              )}
              disable={!interval.checked}
            />
          </View>
        ))}
        <View style={styles.row}>
          {['Day', 'Week', 'Month', 'Year'].map((unit) => (
            <TouchableOpacity
              key={unit}
              style={[styles.unitButton, intervalUnit === unit && styles.unitButtonActive]}
              onPress={() => setIntervalUnit(unit)}
            >
              <Text style={[styles.unitButtonText, intervalUnit === unit && styles.unitButtonTextActive]}>{unit}</Text>
            </TouchableOpacity>
          ))}
        </View>

        <Text style={styles.label}>Notify Before</Text>
        <View style={styles.row}>
          <TextInput
            style={[styles.input, { width: 50, marginBottom: 0 }]}
            value={notifyBefore.value}
            onChangeText={(text) => setNotifyBefore({ ...notifyBefore, value: text })}
            keyboardType="numeric"
            placeholder="0"
            placeholderTextColor="#A0A0A0"
          />
          <Dropdown
            style={styles.intervalUnitDropdown}
            placeholderStyle={styles.unitDropdownText}
            selectedTextStyle={styles.unitDropdownText}
            data={UNIT_OPTIONS.map((unit) => ({ label: unit, value: unit }))}
            maxHeight={150}
            labelField="label"
            valueField="value"
            placeholder="Unit"
            value={notifyBefore.unit}
            onChange={(item) => setNotifyBefore({ ...notifyBefore, unit: item.value })}
            renderRightIcon={() => (
              <MaterialIcons name="arrow-drop-down" size={22} color="#015185" />
            )}
          />
          {TIME_UNIT_OPTIONS.map((time) => (
            <TouchableOpacity
              key={time}
              style={[styles.unitButton, notifyBefore.time === time && styles.unitButtonActive]}
              onPress={() => setNotifyBefore({ ...notifyBefore, time })}
            >
              <Text style={[styles.unitButtonText, notifyBefore.time === time && styles.unitButtonTextActive]}>{time}</Text>
            </TouchableOpacity>
          ))}
        </View>

        <Text style={styles.infoText}>
          The schedule will be marked as 'Due soon' once any of the above threshold is crossed.
        </Text>
      </ScrollView>
    </View>
  );
};

const styles = StyleSheet.create({
  container: { flex: 1, backgroundColor: '#F2F2F2' },
  header: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    backgroundColor: '#025383',
    paddingTop: StatusBar.currentHeight || 60,
    paddingHorizontal: 16,
    paddingBottom: 12,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: 18,
    fontWeight: 'bold',
    marginLeft: 10,
  },
  createButton: {
    backgroundColor: '#fff',
    borderRadius: 4,
    paddingHorizontal: 18,
    paddingVertical: 6,
  },
  createButtonText: { color: '#015185', fontWeight: 'bold', fontSize: 15 },
  formContainer: { padding: 18 },
  rowBetween: { flexDirection: 'row', justifyContent: 'space-between', alignItems: 'center', marginBottom: 12 },
  value: {
    color: '#025383',
    fontSize: width * 0.034,
  },
  label: {
    fontSize: width * 0.034,
    color: '#025383',
    marginTop: width * 0.01,
    marginBottom: width * 0.01,
  },
  labelSmall: {
    color: '#015185',
    marginTop: 10,
    marginBottom: 6,
    fontSize: width * 0.034,
  },
  input: {
    backgroundColor: '#fff',
    borderWidth: 1,
    borderColor: '#B0C4DE',
    borderRadius: 6,
    paddingHorizontal: 12,
    paddingVertical: 8,
    fontSize: width * 0.034,
    marginBottom: 12,
    color: '#222',
  },
  inputWithIcon: { flexDirection: 'row', alignItems: 'center', marginBottom: 12 },
  selectText: {
    fontSize: width * 0.034,
    color: '#025383',
    marginTop: width * 0.01,
    marginBottom: width * 0.01,
  },
  row: { flexDirection: 'row', alignItems: 'center', marginBottom: 12 },
  unitText: {
    fontSize: width * 0.034,
    color: '#025383',
    marginBottom: width * 0.04,
    marginLeft: width * 0.02,
  },
  unitText1: {
    fontSize: width * 0.034,
    color: '#025383',
    marginLeft: width * 0.02,
  },
  unitDropdown: {
    width: 110,
    marginLeft: width * 0.02,
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 6,
    paddingHorizontal: 8,
    backgroundColor: '#fff',
  },
  unitDropdownText: {
    fontSize: width * 0.034,
    color: '#015185',
  },
  intervalUnitDropdown: {
    width: 110,
    marginLeft: width * 0.02,
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 6,
    paddingHorizontal: 8,
    backgroundColor: '#fff',
  },
  everyText: { color: '#015185', fontSize: 15, marginHorizontal: 8 },
  checkbox: { marginRight: 4 },
  taskHeader: { flexDirection: 'row', justifyContent: 'space-between', alignItems: 'center', marginTop: 18, marginBottom: 6 },
  addTaskButton: {
    backgroundColor: '#015185',
    borderRadius: 6,
    paddingHorizontal: 18,
    paddingVertical: 8,
  },
  addTaskText: {
    color: '#fff',
    fontSize: width * 0.034,
  },
  repeatButton: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 6,
    paddingHorizontal: 18,
    paddingVertical: 8,
    marginLeft: 10,
    backgroundColor: '#fff',
  },
  repeatButtonActive: { backgroundColor: '#015185' },
  repeatButtonText: {
    color: '#015185',
    fontSize: width * 0.034,
  },
  repeatButtonTextActive: { color: '#fff' },
  unitButton: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 4,
    paddingHorizontal: 12,
    paddingVertical: 8,
    marginLeft: 8,
    backgroundColor: '#fff',
  },
  unitButtonActive: { backgroundColor: '#015185' },
  unitButtonText: {
    color: '#015185',
    fontSize: width * 0.034,
  },
  unitButtonTextActive: { color: '#fff' },
  infoText: { color: '#015185', fontSize: 13, marginTop: 10, marginBottom: 20 },
  selectedEquipmentContainer: {
    marginBottom: 10,
  },
  selectedEquipmentText: {
    fontSize: 15,
    color: '#015185',
    marginBottom: 4,
  },
  placeholderText: {
    fontSize: 15,
    color: '#A0A0A0',
    marginBottom: 12,
  },
  modalOverlay: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: 'rgba(0,0,0,0.5)',
  },
  datePickerContainer: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  datePickerButtons: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    width: '100%',
    marginTop: 10,
  },
  datePickerButton: {
    flex: 1,
    borderRadius: 5,
    paddingVertical: 10,
    alignItems: 'center',
    marginHorizontal: 5,
  },
  taskModalContainer: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
    width: '80%',
  },
  modalTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#025383',
    marginBottom: 15,
  },
  modalInput: {
    backgroundColor: '#fff',
    borderWidth: 1,
    borderColor: '#B0C4DE',
    borderRadius: 6,
    paddingHorizontal: 12,
    paddingVertical: 8,
    fontSize: 16,
    width: '100%',
    marginBottom: 15,
    color: '#222',
  },
  modalButtons: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    width: '100%',
  },
  modalButton: {
    flex: 1,
    borderRadius: 5,
    paddingVertical: 10,
    alignItems: 'center',
    marginHorizontal: 5,
  },
  cancelButton: {
    backgroundColor: '#A0A0A0',
  },
  doneButton: {
    backgroundColor: '#015185',
  },
  modalButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: 'bold',
  },
  taskList: {
    marginBottom: 12,
  },
  taskItem: {
    fontSize: 15,
    color: '#015185',
    marginBottom: 4,
  },
});

export default EditServiceSchedule;