import React, { useState, useRef, useEffect } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  TextInput,
  Image,
  Dimensions,
  Animated,
  Modal,
  FlatList,
  ScrollView,
  KeyboardAvoidingView,
  Platform,
  ActivityIndicator,
  Alert,
  InteractionManager,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';

import { useNavigation, useRoute } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import RNHTMLtoPDF from 'react-native-html-to-pdf';
import RNFS from 'react-native-fs';
import database from '@react-native-firebase/database';
import storage from '@react-native-firebase/storage';
import Toast from 'react-native-toast-message';
import NetInfo from '@react-native-community/netinfo';

import auth from '@react-native-firebase/auth';
import DropDownPicker from 'react-native-dropdown-picker';
import DateTimePicker from '@react-native-community/datetimepicker';
import { getFilteredTeamMembers } from '../../Utils/teamMemberUtils';
import { filterEquipmentByPermission } from '../../Utils/normalizeEquipmentData';
import * as ElementDropdown from 'react-native-element-dropdown';
import JobCardAPI from '../../ConfigMongo/JobCardMongoConfig.js';
import InspectionAPI from '../../ConfigMongo/InspectionMongoConfig';
import { MongoAPI } from '../../ConfigMongo/MongoDBConfig';
import { EquipmentAPI } from '../../ConfigMongo/EquipmentMongoConfig';

const { width, height } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.3;

// Helper for chunked state updates
const chunkArray = (array, chunkSize) => {
  const result = [];
  for (let i = 0; i < array.length; i += chunkSize) {
    result.push(array.slice(i, i + chunkSize));
  }
  return result;
};

// Add SyncModal loader component (after imports, before CreateInsWorkOrder)
const SyncModal = ({ visible, message = 'Loading...' }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
            />
            <Text style={styles.syncModalTextadd}>{message}</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

const ServiceSWorkOrderJob = () => {
  const navigation = useNavigation();
  const route = useRoute();
  const scrollViewRef = useRef(null);
  const [workOrderTitle, setWorkOrderTitle] = useState('');
  const [memo, setMemo] = useState('');
  const [priority, setPriority] = useState(null);
  const [selectedEquipment, setSelectedEquipment] = useState(null);
  const [assignTo, setAssignTo] = useState([]);


  const [taskStatusMenuVisibleId, setTaskStatusMenuVisibleId] = useState(null);
  const [statusDropdownAnchor, setStatusDropdownAnchor] = useState(null);



  const [teamMembers, setTeamMembers] = useState([]);
  const [startDate, setStartDate] = useState(null);
  const [dueDate, setDueDate] = useState(null);
  const [showStartDatePicker, setShowStartDatePicker] = useState(false);
  const [showDueDatePicker, setShowDueDatePicker] = useState(false);
  const [additionalDetails, setAdditionalDetails] = useState('');
  const [isAdditionalDetailsOpen, setIsAdditionalDetailsOpen] = useState(false);
  const [workOrderType, setWorkOrderType] = useState('General');
  const [workOrderTypeModalVisible, setWorkOrderTypeModalVisible] = useState(false);
  const [estimatedCost, setEstimatedCost] = useState('');
  const [estimatedLaborTime, setEstimatedLaborTime] = useState('');
  const [linkInspection, setLinkInspection] = useState(null);
  const [linkInspectionModalVisible, setLinkInspectionModalVisible] = useState(false);
  const [linkedWorkOrder, setLinkedWorkOrder] = useState(null);
  const [linkedWorkOrderModalVisible, setLinkedWorkOrderModalVisible] = useState(false);
  const [serviceScheduleLink, setServiceScheduleLink] = useState(null);
  const [isCostDetailsOpen, setIsCostDetailsOpen] = useState(false);
  const [partMaterialCost, setPartMaterialCost] = useState('');
  const [totalLaborCost, setTotalLaborCost] = useState('');
  const [additionalCost, setAdditionalCost] = useState('');
  const [taxAmount, setTaxAmount] = useState('');
  const [taxPercentage, setTaxPercentage] = useState('');
  const [totalWoCost, setTotalWoCost] = useState('');
  const [selectedCategory, setSelectedCategory] = useState('Task');
  const [addItemModalVisible, setAddItemModalVisible] = useState(false);
  const [newItemText, setNewItemText] = useState('');
  const [items, setItems] = useState([]);
  const [taskType, setTaskType] = useState('Repair');
  const [note, setNote] = useState('');
  const [tempAssignTo, setTempAssignTo] = useState([]);
  const [statusModalVisible, setStatusModalVisible] = useState(false);
  const [selectedItemId, setSelectedItemId] = useState(null);
  const [editModalVisible, setEditModalVisible] = useState(false);
  const [editItem, setEditItem] = useState(null);
  const [editMenuVisible, setEditMenuVisible] = useState(false);
  const [addLaborModalVisible, setAddLaborModalVisible] = useState(false);
  const [addPartsModalVisible, setAddPartsModalVisible] = useState(false);
  const [addAdditionalCostModalVisible, setAddAdditionalCostModalVisible] = useState(false);
  const [addRemarksModalVisible, setAddRemarksModalVisible] = useState(false);
  const [laborWorker, setLaborWorker] = useState('');
  const [laborCode, setLaborCode] = useState('');
  const [laborRate, setLaborRate] = useState('');
  const [laborHours, setLaborHours] = useState('');
  const [laborMinutes, setLaborMinutes] = useState('');
  const [laborNote, setLaborNote] = useState('');
  const [partsItemNumber, setPartsItemNumber] = useState('');
  const [partsPartNumber, setPartsPartNumber] = useState('');
  const [partsItemName, setPartsItemName] = useState('');
  const [partsQuantity, setPartsQuantity] = useState('');
  const [partsUnit, setPartsUnit] = useState('Piece');
  const [partsItemCost, setPartsItemCost] = useState('');
  const [partsSite, setPartsSite] = useState('');
  const [partsNote, setPartsNote] = useState('');
  const [additionalCostDescription, setAdditionalCostDescription] = useState('');
  const [additionalCostAmount, setAdditionalCostAmount] = useState('');
  const [remarksDescription, setRemarksDescription] = useState('');
  const [faultListModalVisible, setFaultListModalVisible] = useState(false);
  const [selectedFaults, setSelectedFaults] = useState([]);
  const [isLoading, setIsLoading] = useState(false); // Start as false
  const [showSuccessModal, setShowSuccessModal] = useState(false);
  const [workOrderId, setWorkOrderId] = useState(null);
  const [savedPdfPath, setSavedPdfPath] = useState(null);
  const [requestNumber, setRequestNumber] = useState('');
  const [inspections, setInspections] = useState([]);
  const [workOrders, setWorkOrders] = useState([]);
  const [faultListData, setFaultListData] = useState([]);
  const [errors, setErrors] = useState({ title: '', equipment: '' });
  const [checklistItems, setChecklistItems] = useState([]);
  const [selectedTaskId, setSelectedTaskId] = useState(null);
  const [assignMenuVisible, setAssignMenuVisible] = useState(false);
  const [showUserSelectionModal, setShowUserSelectionModal] = useState(false);
  const [availableUsers, setAvailableUsers] = useState([]);
  const [taskAssignments, setTaskAssignments] = useState({});
  
  // Add state for user permissions and admin profile
  const [userPermissions, setUserPermissions] = useState([]);
  const [adminProfile, setAdminProfile] = useState(null);
  
  // Country and Project state
  const [country, setCountry] = useState(null);
  const [countryOpen, setCountryOpen] = useState(false);
  const [countryItems] = useState([{ label: 'Zambia', value: 'Zambia' }, { label: 'DRC', value: 'DRC' }]);
  const [project, setProject] = useState(null);
  const [projectOpen, setProjectOpen] = useState(false);
  const [projectItems, setProjectItems] = useState([]);

  // Add state to store corrective action statuses
  const [correctiveStatuses, setCorrectiveStatuses] = useState({});

  // 1. Add state for firebaseTasks
  const [firebaseTasks, setFirebaseTasks] = useState([]);

  const [inspectionFaultyTasks, setInspectionFaultyTasks] = useState([]);
  
  // Add state for total job card count
  const [totalJobCardCount, setTotalJobCardCount] = useState(0);

  const rotateAnim = useRef(new Animated.Value(0)).current;
  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });
  const extractedProjectRef = useRef(null);

  const priorityOptions = [
    { label: 'Low', image: require('../../../Images/adminhome/inspreport/Priority1.png') },
    { label: 'Medium', image: require('../../../Images/adminhome/inspreport/Priority2.png') },
    { label: 'High', image: require('../../../Images/adminhome/inspreport/Priority3.png') },
    { label: 'Emergency', image: require('../../../Images/adminhome/inspreport/Priority4.png') },
  ];

  const workOrderTypes = [
    { id: '1', name: 'General' },
    { id: '2', name: 'Maintenance' },
    { id: '3', name: 'Repair' },
    { id: '4', name: 'Installation' },
  ];

  const categories = ['Task', 'Parts', 'Labor', 'Additional Cost', 'Remarks'];
  const taskTypes = ['Repair', 'Replace'];
  const statuses = [
    { label: 'Initiated', color: '#A9A9A9' },
    { label: 'In Progress', color: '#FFD700' },
    { label: 'Hold', color: '#FF4500' },
    { label: 'Complete', color: '#008000' },
  ];
  const units = ['Piece', 'Unit', 'Box'];

  const showToast = (type, title, message) => {
    // Map 'warning' to 'info' since 'warning' type doesn't exist
    const toastType = type === 'warning' ? 'info' : type;
    
    Toast.show({
      type: toastType,
      text1: title,
      text2: message,
      position: 'bottom',
      visibilityTime: 3000,
      autoHide: true,
    });
  };

  // Helper function to get status color
  const getStatusColor = (status) => {
    switch (status?.toLowerCase()) {
      case 'completed':
      case 'complete':
        return '#28a745';
      case 'in progress':
      case 'pending':
        return '#ffc107';
      case 'cancelled':
      case 'cancelled':
        return '#dc3545';
      case 'on hold':
        return '#fd7e14';
      default:
        return '#6c757d';
    }
  };

  const generateRequestNumber = (country, project, totalCount) => {
    if (country && project) {
      // Get country code
      const countryCode = country === 'Zambia' ? 'ZAM' : country === 'DRC' ? 'DRC' : 'XXX';
      // Get project code (first 3 letters, uppercased)
      const projectCode = project.slice(0, 3).toUpperCase();
      // Calculate next number (total count + 1) and format with leading zeros
      const nextNumber = totalCount + 1;
      return `${countryCode}-${projectCode}-${nextNumber.toString().padStart(3, '0')}`;
    }
    // Fallback if either is missing
    return `WO${totalCount + 1}`;
  };

  const extractNumberFromRequestNumber = (requestNumber) => {
    if (!requestNumber) return 0;
    
    // Try to extract number from new format: ZAM-KAN-001 (Country-Project-XXX)
    const newFormatMatch = requestNumber.match(/-(\d{3})$/);
    if (newFormatMatch) {
      const number = parseInt(newFormatMatch[1], 10);
      console.log('Extracted number from new format:', { requestNumber, number });
      return number;
    }
    
    // Try to extract number from old format with JC: ZAM-KAN-JC-0001
    const oldJcFormatMatch = requestNumber.match(/-JC-(\d+)$/);
    if (oldJcFormatMatch) {
      const number = parseInt(oldJcFormatMatch[1], 10);
      console.log('Extracted number from old JC format:', { requestNumber, number });
      return number;
    }
    
    // Try to extract number from old format: WO0001
    const oldFormatMatch = requestNumber.match(/WO(\d+)/);
    if (oldFormatMatch) {
      const number = parseInt(oldFormatMatch[1], 10);
      console.log('Extracted number from old WO format:', { requestNumber, number });
      return number;
    }
    
    // Fallback
    console.log('Using fallback number for:', requestNumber);
    return 0;
  };

  // Handle selectedEquipment param
  useEffect(() => {
    if (
      route.params?.selectedEquipment &&
      (selectedEquipment == null || route.params.selectedEquipment.id !== selectedEquipment.id)
    ) {
      setSelectedEquipment(route.params.selectedEquipment);
      setErrors && setErrors(prev => ({ ...prev, equipment: '' }));
    }
  }, [route.params?.selectedEquipment]);

  // Handle schedule param from service schedule
  useEffect(() => {
    if (route.params?.schedule) {
      const schedule = route.params.schedule;
      console.log('📋 Auto-selecting service schedule data:', schedule);
      
      // Auto-populate work order title from schedule title
      if (schedule.title) {
        setWorkOrderTitle(schedule.title);
      }
      
      // Auto-populate equipment from schedule
      if (schedule.equipment) {
        console.log('📋 Auto-selecting equipment from schedule:', schedule.equipment);
        setSelectedEquipment(schedule.equipment);
      }
      
      // Auto-populate tasks from schedule
      if (schedule.tasks && Array.isArray(schedule.tasks) && schedule.tasks.length > 0) {
        console.log('📋 Auto-populating tasks from schedule:', schedule.tasks);
        const autoTasks = schedule.tasks.map((task, index) => ({
          id: `auto_task_${Date.now()}_${index}`,
          text: task,
          status: 'Pending',
          assignedTo: null,
          category: 'Task',
          type: 'Repair',
          note: '',
          isAutoGenerated: true
        }));
        setItems(autoTasks);
      }
      
      // Auto-populate country and project
      if (schedule.country) {
        console.log('📋 Auto-selecting country from schedule:', schedule.country);
        setCountry(schedule.country);
      }
      
      if (schedule.project) {
        console.log('📋 Auto-selecting project from schedule:', schedule.project);
        setProject(schedule.project);
      }
      
      // Auto-populate memo/note from schedule note
      if (schedule.note) {
        setMemo(schedule.note);
      }
      
      // Set service schedule link
      const serviceScheduleData = {
        id: schedule.id,
        scheduleId: schedule.scheduleId,
        title: schedule.title,
        country: schedule.country,
        project: schedule.project,
        equipment: schedule.equipment,
        tasks: schedule.tasks,
        note: schedule.note
      };
      
      console.log('📋 Created service schedule link data:', serviceScheduleData);
      setServiceScheduleLink(serviceScheduleData);
    }
  }, [route.params?.schedule?.id]); // Only re-run if schedule ID changes

  // Handle selectedReport param (MongoDB with Firebase fallback)
  useEffect(() => {
    if (route.params?.selectedReport) {
      const report = route.params.selectedReport;
      console.log('📊 Auto-selecting inspection data:', report);
      console.log('📊 Report ID fields:', { id: report.id, _id: report._id });
      
      // Set inspection link - use _id if id is not available
      const inspectionData = {
        id: report.id || report._id,
        name: report.step1?.requestNumber || 'Inspection Report',
        step1: report.step1,
        step2: report.step2,
        step3: report.step3,
      };
      
      console.log('📊 Created inspection data:', inspectionData);
      setLinkInspection(inspectionData);
      
      // Auto-create tasks from faulty items
      console.log('📊 Auto-creating tasks from inspection...');
      const faultyTasks = extractFaultyItemsFromInspection(inspectionData);
      console.log('📊 Extracted faulty tasks:', faultyTasks.length, faultyTasks);
      
      // Add tasks immediately
      if (faultyTasks.length > 0) {
        setItems(prevItems => {
          // Remove any existing auto-generated tasks for this inspection
          const filteredItems = prevItems.filter(item => 
            !item.note?.includes('Auto-generated from inspection')
          );
          
          // Add new tasks
          const updatedItems = [...filteredItems, ...faultyTasks];
          console.log('📊 Updated items with auto-created tasks:', updatedItems.length);
          return updatedItems;
        });
        
        showToast('success', 'Tasks Created', `${faultyTasks.length} tasks auto-created from faulty items`);
      }
      
      // Also call the handler for any additional processing
      setTimeout(() => {
        handleInspectionSelected(inspectionData);
      }, 500); // Reduced delay
      
      // Set work order title
      setWorkOrderTitle(`Job Card for ${report.step1?.requestNumber || 'Inspection'}`);
      
      // Set memo from inspection notes
      setMemo(report.step2?.notes || report.step1?.notes || '');
      
      // Set priority based on equipment status
      const equipmentStatus = report.step3?.equipmentStatus || report.step2?.equipmentStatus;
      if (equipmentStatus === 'Critical' || equipmentStatus === 'Faulty') {
        setPriority('High');
      } else if (equipmentStatus === 'Good' || equipmentStatus === 'OK') {
        setPriority('Low');
      } else {
        setPriority('Medium');
      }
      
      // Auto-set country and project from inspection
      if (report.step1?.country) {
        setCountry(report.step1.country);
        console.log('✅ Auto-selected country:', report.step1.country);
      }
      
      if (report.step1?.project) {
        setProject(report.step1.project);
        console.log('✅ Auto-selected project:', report.step1.project);
      }

      // Fetch the latest equipment from MongoDB for this inspection
      (async () => {
        try {
          console.log('📊 Fetching equipment from inspection in MongoDB...');
          
          // Try MongoDB first
          const userId = await AsyncStorage.getItem('userUid');
          if (!userId) return;
          const inspectionResponse = await InspectionAPI.getInspection(userId, report.id);
          
          if (inspectionResponse.success && inspectionResponse.data) {
            const reportData = inspectionResponse.data;
            const inspectionEquipment = reportData?.step1?.selectedEquipment || reportData?.step1?.equipment;
            if (inspectionEquipment) {
              console.log('✅ Equipment found in MongoDB inspection:', inspectionEquipment);
              setSelectedEquipment(inspectionEquipment);
              setErrors && setErrors(prev => ({ ...prev, equipment: '' }));
            }
          } else {
            console.log('❌ Inspection not found in MongoDB, trying Firebase fallback...');
            // Fallback to Firebase
            const adminUid = await findAdminUidForInspection(report.id);
            if (!adminUid) return;
            const reportRef = database().ref(`/GlobalInspectionReport/admins/${adminUid}/${report.id}`);
            const snapshot = await reportRef.once('value');
            const reportData = snapshot.val();
            const inspectionEquipment = reportData?.step1?.selectedEquipment || reportData?.step1?.equipment;
            if (inspectionEquipment) {
              console.log('✅ Equipment found in Firebase inspection:', inspectionEquipment);
              setSelectedEquipment(inspectionEquipment);
              setErrors && setErrors(prev => ({ ...prev, equipment: '' }));
            }
          }
        } catch (err) {
          console.error('❌ Error fetching inspection equipment:', err);
        }
      })();

      // Auto-fill country and project from requestNumber if possible
      const requestNumber = report.step1?.requestNumber || '';
      
      // First try to get from the report data directly
      let extractedCountry = report.step1?.country || null;
      let extractedProject = report.step1?.project || null;
      
      // If not available in report data, try to extract from requestNumber
      if (!extractedCountry || !extractedProject) {
        const match = requestNumber.match(/^([^-]+)-([^-]+)-IR-/);
        if (match) {
          const countryCode = match[1];
          const projectCode = match[2];
          
          // Map country code to country name
          if (!extractedCountry) {
            if (countryCode === 'ZAM') {
              extractedCountry = 'Zambia';
            } else if (countryCode === 'DRC') {
              extractedCountry = 'DRC';
            }
          }
          
          // Map project code to project name
          if (!extractedProject && extractedCountry) {
            if (extractedCountry === 'Zambia') {
              switch (projectCode) {
                case 'KAN': extractedProject = 'Kansanshi'; break;
                case 'KAL': extractedProject = 'Kalumbila'; break;
                case 'KIM': extractedProject = 'Kimiteto'; break;
                case 'KOB': extractedProject = 'Kobold'; break;
                case 'FQM': extractedProject = 'FQM Exploration'; break;
                case 'MIM': extractedProject = 'Mimosa'; break;
                default: 
                  // Fallback: try to find a project that starts with the code
                  const zambiaProjects = ['Kobold', 'Kalumbila', 'Kansanshi', 'Kimiteto', 'Mimosa'];
                  extractedProject = zambiaProjects.find(p => p.toUpperCase().startsWith(projectCode)) || null;
                  break;
              }
            } else if (extractedCountry === 'DRC') {
              switch (projectCode) {
                case 'MUS': extractedProject = 'Musompo'; break;
                case 'IME': extractedProject = 'IME'; break;
                default:
                  // Fallback: try to find a project that starts with the code
                  const drcProjects = ['Musompo', 'IME'];
                  extractedProject = drcProjects.find(p => p.toUpperCase().startsWith(projectCode)) || null;
                  break;
              }
            }
          }
        }
      }
      
      if (extractedCountry) {
        setCountry(extractedCountry);
      }
      
      if (extractedProject) {
        // Store the extracted project in a ref to set it after projectItems are populated
        if (!extractedProjectRef.current) {
          extractedProjectRef.current = extractedProject;
          console.log('Extracted project from requestNumber:', extractedProject);
        }
      }
      
      console.log('Report data:', {
        requestNumber,
        reportCountry: report.step1?.country,
        reportProject: report.step1?.project,
        extractedCountry,
        extractedProject
      });
    }
  }, [route.params?.selectedReport]);

  // Initial data fetch and loading animation
  useEffect(() => {
    const load = async () => {
      setIsLoading(true);
      try {
        await fetchInitialData();
      } finally {
        setIsLoading(false);
      }
    };
    load();
  }, []);

  useEffect(() => {
    let animation;
    if (isLoading) {
      rotateAnim.setValue(0); // Always reset before starting
      animation = Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 900,
          useNativeDriver: true,
          easing: undefined,
        })
      );
      animation.start();
    } else {
      rotateAnim.stopAnimation && rotateAnim.stopAnimation();
      rotateAnim.setValue(0);
      if (animation && animation.stop) animation.stop();
    }
    return () => {
      if (animation && animation.stop) animation.stop();
    };
  }, [isLoading]);

  useEffect(() => {
    if (!linkInspection) return;
    (async () => {
      try {
        console.log('📊 Fetching inspection data from MongoDB for faulty items...');
        
        // Fetch inspection from MongoDB
        const adminUid = await AsyncStorage.getItem('userUid');
        if (!adminUid) return;
        const inspectionResponse = await InspectionAPI.getInspection(adminUid, linkInspection.id);
        
        if (!inspectionResponse.success || !inspectionResponse.data) {
          console.log('❌ Inspection not found in MongoDB, trying Firebase fallback...');
          // Fallback to Firebase if not in MongoDB
          const adminUid = await findAdminUidForInspection(linkInspection.id);
          if (!adminUid) return;
          const reportRef = database().ref(`/GlobalInspectionReport/admins/${adminUid}/${linkInspection.id}`);
          const snapshot = await reportRef.once('value');
          const reportData = snapshot.val();
          if (!reportData?.step2?.sections || !reportData?.step2?.selections) return;
          
          // Process Firebase data
          const faultyTasks = [];
          reportData.step2.sections.forEach(section => {
            section.subItems.forEach(subItem => {
              const sel = reportData.step2.selections[subItem.key];
              if ((sel === 'Not Ok' || sel === 'Reject' || sel === 'Faulty') && subItem.itemName && subItem.itemName.trim() !== '') {
                faultyTasks.push({
                  id: `task_${subItem.key}`,
                  category: 'Task',
                  text: subItem.itemName,
                  type: 'Repair',
                  note: `Auto-generated from inspection: ${linkInspection.name}`,
                  assignTo: [],
                  status: 'Initiated',
                  section: section.name,
                  inspectionId: linkInspection.id
                });
              }
            });
          });
          
          setItems(prev => {
            const existingTaskIds = new Set(prev.filter(i => i.category === 'Task').map(i => i.id));
            const newTasks = faultyTasks.filter(t => !existingTaskIds.has(t.id));
            const filteredPrev = prev.filter(i => i.category !== 'Task' || !i.id.startsWith('task_'));
            return [...filteredPrev, ...newTasks];
          });
          return;
        }
        
        // Process MongoDB data
        const reportData = inspectionResponse.data;
        console.log('✅ Inspection data from MongoDB:', reportData);
        
        if (!reportData?.step2?.sections || !reportData?.step2?.selections) {
          console.log('❌ No step2 data in inspection');
          return;
        }
        
        const faultyTasks = [];
        reportData.step2.sections.forEach(section => {
          section.subItems.forEach(subItem => {
            const sel = reportData.step2.selections[subItem.key];
            if ((sel === 'Not Ok' || sel === 'Reject' || sel === 'Faulty') && subItem.itemName && subItem.itemName.trim() !== '') {
              faultyTasks.push({
                id: `task_${subItem.key}`,
                category: 'Task',
                text: subItem.itemName,
                type: 'Repair',
                note: `Auto-generated from inspection: ${linkInspection.name}`,
                assignTo: [],
                status: 'Initiated',
                section: section.name,
                inspectionId: linkInspection.id
              });
            }
          });
        });
        
        console.log('✅ Found faulty tasks:', faultyTasks.length);
        
        // Convert to proper task format and add to items
        const tasksFromFaultyItems = faultyTasks.map(task => ({
          ...task,
          category: 'Task', // Ensure they appear in the Task tab
          type: 'Repair',
          status: 'Initiated',
          assignTo: [],
          note: `Auto-generated from inspection: ${linkInspection.name || linkInspection.step1?.requestNumber || 'Inspection'}`,
        }));
        
        console.log('📊 Converting faulty items to tasks:', tasksFromFaultyItems.length);
        
        // Only add new tasks that are not already present
        setItems(prev => {
          const existingTaskIds = new Set(prev.filter(i => i.category === 'Task').map(i => i.id));
          const newTasks = tasksFromFaultyItems.filter(t => !existingTaskIds.has(t.id));
          const filteredPrev = prev.filter(i => i.category !== 'Task' || !i.id.startsWith('task_'));
          console.log('📊 Adding new tasks to items:', newTasks.length);
          return [...filteredPrev, ...newTasks];
        });
      } catch (err) {
        console.error('❌ Error fetching inspection faulty items:', err);
      }
    })();
  }, [linkInspection]);

  // Auto-populate Task items from inspection checklist (MongoDB with Firebase fallback)
  useEffect(() => {
    const fetchChecklistTasksFromInspection = async () => {
      if (!linkInspection) return;
      try {
        console.log('📊 Fetching inspection checklist tasks from MongoDB...');
        
        // Try MongoDB first
        const adminUid = await AsyncStorage.getItem('userUid');
        if (!adminUid) return;
        const inspectionResponse = await InspectionAPI.getInspection(adminUid, linkInspection.id);
        
        if (!inspectionResponse.success || !inspectionResponse.data) {
          console.log('❌ Inspection not found in MongoDB, trying Firebase fallback...');
          // Fallback to Firebase
          const reportRef = database().ref(`/GlobalInspectionReport/admins/${adminUid}/${linkInspection.id}`);
          const snapshot = await reportRef.once('value');
          const reportData = snapshot.val();
          if (!reportData?.step2?.sections || !reportData?.step2?.selections) return;
          
          const checklistTasks = [];
          reportData.step2.sections.forEach(section => {
            section.subItems.forEach(subItem => {
              const sel = reportData.step2.selections[subItem.key];
              if ((sel === 'Not Ok' || sel === 'Reject') && subItem.itemName && subItem.itemName.trim() !== '') {
                checklistTasks.push({
                  id: `task_${subItem.key}`,
                  category: 'Task',
                  text: `Fix ${subItem.itemName}`,
                  type: 'Repair',
                  note: `From inspection: ${linkInspection.name}`,
                  assignTo: [],
                  status: 'Initiated',
                  section: section.name,
                  inspectionId: linkInspection.id
                });
              }
            });
          });
          
          setItems(prev => {
            const existingTaskIds = new Set(prev.filter(i => i.category === 'Task').map(i => i.id));
            const newTasks = checklistTasks.filter(t => !existingTaskIds.has(t.id));
            const filteredPrev = prev.filter(i => i.category !== 'Task' || !i.id.startsWith('task_'));
            return [...filteredPrev, ...newTasks];
          });
          return;
        }
        
        // Process MongoDB data
        const reportData = inspectionResponse.data;
        console.log('✅ Processing MongoDB inspection data for checklist tasks');
        
        if (!reportData?.step2?.sections || !reportData?.step2?.selections) return;
        
        const checklistTasks = [];
        reportData.step2.sections.forEach(section => {
          section.subItems.forEach(subItem => {
            const sel = reportData.step2.selections[subItem.key];
            if ((sel === 'Not Ok' || sel === 'Reject') && subItem.itemName && subItem.itemName.trim() !== '') {
              checklistTasks.push({
                id: `task_${subItem.key}`,
                category: 'Task',
                text: `Fix ${subItem.itemName}`,
                type: 'Repair',
                note: `From inspection: ${linkInspection.name}`,
                assignTo: [],
                status: 'Initiated',
                section: section.name,
                inspectionId: linkInspection.id
              });
            }
          });
        });
        
        console.log('✅ Found checklist tasks:', checklistTasks.length);
        
        // Convert to proper task format
        const tasksFromChecklist = checklistTasks.map(task => ({
          ...task,
          category: 'Task', // Ensure they appear in the Task tab
          type: 'Repair',
          status: 'Initiated',
          assignTo: [],
          note: `Auto-generated from inspection checklist: ${linkInspection.name || linkInspection.step1?.requestNumber || 'Inspection'}`,
        }));
        
        console.log('📊 Converting checklist items to tasks:', tasksFromChecklist.length);
        
        setItems(prev => {
          const existingTaskIds = new Set(prev.filter(i => i.category === 'Task').map(i => i.id));
          const newTasks = tasksFromChecklist.filter(t => !existingTaskIds.has(t.id));
          const filteredPrev = prev.filter(i => i.category !== 'Task' || !i.id.startsWith('task_'));
          console.log('📊 Adding new checklist tasks to items:', newTasks.length);
          return [...filteredPrev, ...newTasks];
        });
      } catch (err) {
        console.error('❌ Error fetching checklist tasks:', err);
      }
    };
    fetchChecklistTasksFromInspection();
  }, [linkInspection]);

  useEffect(() => {
    if (!linkInspection) return;

    // This should be the result of your faulty items fetch
    const faultyTasks = inspectionFaultyTasks.map(fault => ({
      id: `task_${fault.id}`,
      category: 'Task',
      text: `Fix ${fault.text}`,
      type: 'Repair',
      note: `Faulty item from inspection: ${linkInspection.name}`,
      assignTo: [],
      status: 'Initiated',
      section: fault.section,
      inspectionId: linkInspection.id
    }));

    // Only add new tasks that don't already exist
    setItems(prev => {
      const existingIds = new Set(prev.filter(i => i.category === 'Task').map(i => i.id));
      const newOnes = faultyTasks.filter(t => !existingIds.has(t.id));
      return [...prev, ...newOnes];
    });
  }, [inspectionFaultyTasks, linkInspection]);

  // --- CHUNKED STATE FOR LARGE LISTS ---
  const [chunkedChecklistItems, setChunkedChecklistItems] = useState([]);
  const [chunkedItems, setChunkedItems] = useState([]);
  const [chunkedTeamMembers, setChunkedTeamMembers] = useState([]);
  const [chunkedWorkOrders, setChunkedWorkOrders] = useState([]);
  const [chunkedInspections, setChunkedInspections] = useState([]);
  const [chunkedFaultListData, setChunkedFaultListData] = useState([]);
  const CHUNK_SIZE = 5;

  // Helper to chunk and set state in small pieces
  const setChunkedState = (data, setState) => {
    if (!Array.isArray(data) || data.length <= CHUNK_SIZE) {
      setState(data ? [data] : []);
      return;
    }
    const chunks = chunkArray(data, CHUNK_SIZE);
    let i = 0;
    setState([]);
    const addChunk = () => {
      setState(prev => [...prev, chunks[i]]);
      i++;
      if (i < chunks.length) {
        setTimeout(addChunk, 10);
      }
    };
    addChunk();
  };

  // --- MONGODB fetchInitialData ---
  const [loaderTimeout, setLoaderTimeout] = useState(null);
  const firebaseRefs = useRef([]);

  const fetchInitialData = async () => {
    try {
      setIsLoading(true);
      const adminUid = await AsyncStorage.getItem('userUid');
      if (!adminUid) {
        showToast('error', 'Error', 'Admin not authenticated. Please log in again.');
        navigation.navigate('AdminLoginScreen');
        return;
      }

      console.log('📊 Fetching initial data from MongoDB...');

      // Fetch all data from MongoDB
      await Promise.all([
        fetchTeamMembersFromMongo(adminUid),
        fetchInspectionsFromMongo(adminUid),
        fetchWorkOrdersFromMongo(adminUid),
        fetchFaultsFromMongo(adminUid),
      ]);

      setIsLoading(false);
      console.log('✅ All data loaded successfully');
    } catch (error) {
      console.error('❌ Error loading data:', error);
      setIsLoading(false);
      showToast('error', 'Error', 'Failed to load data.');
    }
  };

  // Fetch team members from MongoDB
  const fetchTeamMembersFromMongo = async (adminUid) => {
    try {
      console.log('📊 Fetching team members from MongoDB...');
      const teamResponse = await MongoAPI.getUsersFast(1, 8000);
      
      if (teamResponse && teamResponse.success && teamResponse.data && teamResponse.data.length > 0) {
        const membersArray = teamResponse.data.map((user) => ({
          id: user._id || user.id,
          fullName: user.name || user.fullName || 'N/A',
          name: user.name || user.fullName || 'N/A',
          employeeNumber: user.employeeNumber || 'N/A',
          email: user.email || user.employeeNumber || 'N/A',
          projects: user.projects || [],
          countries: user.countries || [],
          role: user.role || 'Operator',
          isBlocked: user.isBlocked || false,
          profilePic: user.profilePic || null,
        }));
        
        console.log('✅ Loaded team members from MongoDB:', membersArray.length);
        setTeamMembers(membersArray);
      } else {
        console.log('❌ No team members found');
        setTeamMembers([]);
      }
    } catch (error) {
      console.error('❌ Error fetching team members:', error);
      setTeamMembers([]);
    }
  };

  // Fetch inspections from MongoDB
  const fetchInspectionsFromMongo = async (adminUid) => {
    try {
      console.log('📊 Fetching inspections from MongoDB...');
      const inspectionResponse = await InspectionAPI.getAllReports();
      
      if (inspectionResponse && inspectionResponse.success && inspectionResponse.data) {
        const inspectionsArray = inspectionResponse.data.map((inspection) => ({
          id: inspection._id || inspection.id,
          name: inspection.step1?.requestNumber || `Inspection ${inspection._id}`,
          equipmentName: inspection.step1?.selectedEquipment?.equipmentName || inspection.step1?.equipment?.name,
          createdAt: inspection.createdAt || inspection.step1?.date,
          step1: inspection.step1,
          step2: inspection.step2,
          step3: inspection.step3,
        }));
        
        console.log('✅ Loaded inspections from MongoDB:', inspectionsArray.length);
        setInspections(inspectionsArray);
      } else {
        console.log('❌ No inspections found');
        setInspections([]);
      }
    } catch (error) {
      console.error('❌ Error fetching inspections:', error);
      setInspections([]);
    }
  };

  // Fetch work orders from MongoDB
  const fetchWorkOrdersFromMongo = async (adminUid) => {
    try {
      console.log('📊 Fetching work orders from MongoDB...');
      const workOrderResponse = await JobCardAPI.getAllJobCards();
      
      if (workOrderResponse && workOrderResponse.success && workOrderResponse.data) {
        const workOrdersArray = workOrderResponse.data.map((wo) => ({
          id: wo._id || wo.id,
          title: wo.title,
          requestNumber: wo.requestNumber || wo.title,
          equipment: wo.equipment,
          linkInspection: wo.linkInspection,
          status: wo.status,
          createdAt: wo.createdAt,
        }));
        
        // Extract total count from pagination if available
        const totalCount = workOrderResponse.pagination?.total || workOrderResponse.data?.length || 0;
        
        console.log('✅ Loaded work orders from MongoDB:', workOrdersArray.length);
        console.log('📊 Total job card count:', totalCount);
        setWorkOrders(workOrdersArray);
        setTotalJobCardCount(totalCount);
      } else {
        console.log('❌ No work orders found');
        setWorkOrders([]);
        setTotalJobCardCount(0);
      }
    } catch (error) {
      console.error('❌ Error fetching work orders:', error);
      setWorkOrders([]);
      setTotalJobCardCount(0);
    }
  };

  // Fetch faults from MongoDB (keeping Firebase for now as faults might not be in MongoDB yet)
  const fetchFaultsFromMongo = async (adminUid) => {
    try {
      console.log('📊 Fetching faults...');
      // For now, keeping Firebase for faults as they might not be migrated yet
      const faultsRef = database().ref(`/faults/${adminUid}`);
      const snapshot = await faultsRef.once('value');
      const faultsData = snapshot.val();
      
      if (faultsData) {
        const faultsArray = Object.keys(faultsData).map(key => ({
          id: key,
          ...faultsData[key]
        }));
        console.log('✅ Loaded faults:', faultsArray.length);
        setFaultListData(faultsArray);
      } else {
        setFaultListData([]);
      }
    } catch (error) {
      console.error('❌ Error fetching faults:', error);
      setFaultListData([]);
    }
  };

  // Clean up all Firebase listeners on unmount
  useEffect(() => {
    return () => {
      firebaseRefs.current.forEach(ref => ref.off && ref.off());
      firebaseRefs.current = [];
      if (loaderTimeout) clearTimeout(loaderTimeout);
    };
  }, []);

  // --- OPTIMIZED checklistItems and items chunking ---
  useEffect(() => {
    setChunkedState(checklistItems, setChunkedChecklistItems);
  }, [checklistItems]);
  useEffect(() => {
    setChunkedState(items, setChunkedItems);
  }, [items]);

  const validateFields = () => {
    console.log('🔍 Validating fields...');
    console.log('📊 Field values:', {
      workOrderTitle: workOrderTitle?.trim(),
      selectedEquipment: selectedEquipment?.equipmentName,
      country,
      project
    });
    
    let isValid = true;
    const newErrors = { title: '', equipment: '', country: '', project: '' };

    if (!workOrderTitle.trim()) {
      newErrors.title = 'Job Card Title is required';
      isValid = false;
      console.log('❌ Title validation failed');
    }
    if (!selectedEquipment) {
      newErrors.equipment = 'Equipment is required';
      isValid = false;
      console.log('❌ Equipment validation failed');
    }
    if (!country) {
      newErrors.country = 'Country is required';
      isValid = false;
      console.log('❌ Country validation failed');
    }
    if (!project) {
      newErrors.project = 'Project is required';
      isValid = false;
      console.log('❌ Project validation failed');
    }

    console.log('📊 Validation result:', { isValid, errors: newErrors });
    setErrors(newErrors);
    return isValid;
  };

  // Test API connection
  const testAPIConnection = async () => {
    try {
      console.log('🔍 Testing API connection...');
      
      // Create a timeout promise
      const timeoutPromise = new Promise((_, reject) => {
        setTimeout(() => reject(new Error('Connection timeout')), 5000); // 5 second timeout
      });
      
      // Test the job cards endpoint directly with timeout
      const testPromise = fetch('https://api.titandrillingzm.com:6005/jobcards', {
        method: 'GET',
        headers: {
          'Accept': 'application/json',
          'Content-Type': 'application/json'
        }
      });
      
      const response = await Promise.race([testPromise, timeoutPromise]);
      
      if (response && response.ok) {
        console.log('✅ API connection test successful');
        return true;
      } else {
        console.log('❌ API connection test failed - response not ok');
        return false;
      }
    } catch (error) {
      console.error('❌ API connection test failed:', error);
      console.error('❌ Error details:', {
        name: error.name,
        message: error.message,
        code: error.code
      });
      
      // If connection fails, we'll still try to proceed with a warning
      console.log('⚠️ Proceeding despite connection test failure...');
      return true; // Allow proceeding even if test fails
    }
  };

  const testDatabaseConnection = async () => {
    try {
      console.log('🧪 Testing database connection...');
      const response = await fetch('https://api.titandrillingzm.com:6005/jobcards/test-db', {
        method: 'GET',
        timeout: 15000
      });
      
      if (response.ok) {
        const data = await response.json();
        console.log('✅ Database connection test successful:', data);
        return true;
      } else {
        console.log('❌ Database connection test failed:', response.status, response.statusText);
        return false;
      }
    } catch (error) {
      console.error('❌ Database connection test error:', error);
      return false;
    }
  };

  const validateTaskData = (tasks) => {
    console.log('🧪 Validating task data before save...');
    
    const validationResults = {
      totalTasks: tasks.length,
      validTasks: 0,
      invalidTasks: 0,
      errors: [],
      categoryBreakdown: {}
    };
    
    tasks.forEach((task, index) => {
      const taskErrors = [];
      
      // Check required fields
      if (!task.id) taskErrors.push('Missing id');
      if (!task.category) taskErrors.push('Missing category');
      if (!task.text) taskErrors.push('Missing text');
      if (!task.status) taskErrors.push('Missing status');
      if (!task.source) taskErrors.push('Missing source');
      
      // Check status enum values
      const validStatuses = ['Initiated', 'In Progress', 'Completed', 'Cancelled'];
      if (task.status && !validStatuses.includes(task.status)) {
        taskErrors.push(`Invalid status: ${task.status}`);
      }
      
      // Check category enum values
      const validCategories = ['Task', 'Parts', 'Labor', 'Additional Cost', 'Remarks', 'Attach'];
      if (task.category && !validCategories.includes(task.category)) {
        taskErrors.push(`Invalid category: ${task.category}`);
      }

      // Validate category-specific fields
      if (task.category === 'Parts') {
        if (!task.itemName && !task.partNumber) taskErrors.push('Parts: Missing itemName or partNumber');
        if (!task.quantity) taskErrors.push('Parts: Missing quantity');
        if (!task.itemCost) taskErrors.push('Parts: Missing itemCost');
      }
      
      if (task.category === 'Labor') {
        if (!task.worker) taskErrors.push('Labor: Missing worker');
        if (!task.rate) taskErrors.push('Labor: Missing rate');
        if (!task.hours && !task.minutes) taskErrors.push('Labor: Missing hours or minutes');
      }
      
      if (task.category === 'Additional Cost') {
        if (!task.description) taskErrors.push('Additional Cost: Missing description');
        if (!task.amount) taskErrors.push('Additional Cost: Missing amount');
      }
      
      if (task.category === 'Attach') {
        if (!task.fileName && !task.fileUrl) taskErrors.push('Attach: Missing fileName or fileUrl');
      }
      
      // Track category breakdown
      if (!validationResults.categoryBreakdown[task.category]) {
        validationResults.categoryBreakdown[task.category] = 0;
      }
      validationResults.categoryBreakdown[task.category]++;
      
      if (taskErrors.length === 0) {
        validationResults.validTasks++;
      } else {
        validationResults.invalidTasks++;
        validationResults.errors.push({
          taskIndex: index,
          taskId: task.id,
          category: task.category,
          errors: taskErrors
        });
      }
    });
    
    console.log('🧪 Task validation results:', validationResults);
    return validationResults;
  };

  // Test function to verify saved data in MongoDB
  const testSavedJobCardData = async (jobCardId) => {
    try {
      console.log('🧪 Testing saved job card data...');
      const response = await JobCardAPI.getJobCard(jobCardId);
      
      if (response.success && response.data) {
        const jobCard = response.data;
        console.log('🧪 Retrieved job card from MongoDB:', {
          id: jobCard._id,
          title: jobCard.title,
          itemsCount: jobCard.items?.length || 0,
          categories: [...new Set(jobCard.items?.map(item => item.category) || [])],
          categoryBreakdown: jobCard.items?.reduce((acc, item) => {
            acc[item.category] = (acc[item.category] || 0) + 1;
            return acc;
          }, {}) || {},
          sampleItems: jobCard.items?.slice(0, 3).map(item => ({
            id: item.id,
            category: item.category,
            text: item.text?.substring(0, 30),
            status: item.status,
            // Show category-specific data
            ...(item.category === 'Parts' && {
              itemNumber: item.itemNumber,
              partNumber: item.partNumber,
              quantity: item.quantity,
              itemCost: item.itemCost
            }),
            ...(item.category === 'Labor' && {
              worker: item.worker,
              rate: item.rate,
              hours: item.hours,
              minutes: item.minutes
            }),
            ...(item.category === 'Additional Cost' && {
              description: item.description,
              amount: item.amount
            }),
            ...(item.category === 'Attach' && {
              fileName: item.fileName,
              fileUrl: item.fileUrl
            })
          }))
        });
        
        return {
          success: true,
          data: jobCard,
          summary: {
            totalItems: jobCard.items?.length || 0,
            categories: [...new Set(jobCard.items?.map(item => item.category) || [])],
            categoryCounts: jobCard.items?.reduce((acc, item) => {
              acc[item.category] = (acc[item.category] || 0) + 1;
              return acc;
            }, {}) || {}
          }
        };
      } else {
        console.log('❌ Failed to retrieve job card for testing');
        return { success: false, error: 'Failed to retrieve job card' };
      }
    } catch (error) {
      console.error('❌ Error testing saved job card data:', error);
      return { success: false, error: error.message };
    }
  };

  const uploadFileToMongoDB = async (filePath, adminUid) => {
    const Platform = require('react-native').Platform;
    
    try {
      console.log('📁 Creating FormData for PDF upload...');
      console.log('📁 File path:', filePath);
      console.log('👤 Admin UID:', adminUid);
      console.log('📱 Platform:', Platform.OS);
      
      // Check if file exists
      const fileExists = await RNFS.exists(filePath);
      console.log('📁 File exists:', fileExists);
      
      if (!fileExists) {
        throw new Error('PDF file does not exist at path: ' + filePath);
      }
      
      // Create FormData for file upload with Android-specific handling
      const formData = new FormData();
      
      // Android-specific file handling
      const fileObject = Platform.OS === 'android' ? {
        uri: filePath.startsWith('file://') ? filePath : `file://${filePath}`,
        type: 'application/pdf',
        name: `jobcard_${adminUid}_${Date.now()}.pdf`
      } : {
        uri: filePath,
        type: 'application/pdf',
        name: `jobcard_${adminUid}_${Date.now()}.pdf`
      };
      
      formData.append('pdf', fileObject);
      
      console.log('📤 FormData created for', Platform.OS, ', calling uploadJobCardPdf...');
      
      // Add timeout for Android network requests
      const uploadPromise = JobCardAPI.uploadJobCardPdf(adminUid, formData);
      const timeoutPromise = new Promise((_, reject) => 
        setTimeout(() => reject(new Error('Upload timeout')), Platform.OS === 'android' ? 30000 : 15000)
      );
      
      const response = await Promise.race([uploadPromise, timeoutPromise]);
      
      console.log('📊 Upload response:', response);
      console.log('📊 Upload response details:', {
        success: response.success,
        downloadUrl: response.downloadUrl,
        fileName: response.fileName,
        filePath: response.filePath,
        message: response.message
      });
      
      if (response.success) {
        console.log('✅ PDF upload successful, download URL:', response.downloadUrl);
        return response.downloadUrl;
      } else {
        throw new Error(response.error || 'Failed to upload PDF');
      }
    } catch (error) {
      console.error('❌ Error uploading file to MongoDB:', error);
      console.error('❌ Upload error details:', {
        name: error.name,
        message: error.message,
        stack: error.stack,
        platform: Platform.OS
      });
      
      // Check if it's a network-related error
      if (error.message?.includes('Network request failed') || 
          error.message?.includes('timeout') ||
          error.message?.includes('connection')) {
        console.log('📱 Network error detected on', Platform.OS, '- will be handled by caller');
      }
      
      throw error;
    }
  };

  // Simple fallback PDF generation function with Android fixes
  const generateSimplePDF = async () => {
    try {
      const Platform = require('react-native').Platform;
      
      const simpleHtml = `
        <!DOCTYPE html>
        <html>
        <head>
          <meta charset="UTF-8">
          <title>Job Card: ${requestNumber || 'Untitled'}</title>
          <style>
            body { font-family: Arial, sans-serif; margin: 20px; }
            h1 { color: #333; }
            p { margin: 10px 0; }
          </style>
        </head>
        <body>
          <h1>Job Card</h1>
          <p><strong>Request Number:</strong> ${requestNumber || 'N/A'}</p>
          <p><strong>Title:</strong> ${workOrderTitle || 'N/A'}</p>
          <p><strong>Created:</strong> ${new Date().toLocaleString()}</p>
          <p><strong>Status:</strong> Hold</p>
          <p><strong>Priority:</strong> ${priority || 'Medium'}</p>
          <p><strong>Items Count:</strong> ${items?.length || 0}</p>
        </body>
        </html>
      `;

      // Android-specific options
      const options = {
        html: simpleHtml,
        fileName: `Simple_JobCard_${requestNumber || Date.now()}`,
        directory: Platform.OS === 'android' ? 'Download' : 'Documents',
        height: Platform.OS === 'android' ? 842 : 842,
        width: Platform.OS === 'android' ? 595 : 595,
        padding: Platform.OS === 'android' ? 24 : 24,
        bgColor: '#FFFFFF',
        ...(Platform.OS === 'android' && {
          base64: false,
          type: 'application/pdf',
          quality: 1.0,
          orientation: 'portrait',
        }),
      };

      console.log('🎯 Generating simple PDF for', Platform.OS, 'with options:', options);
      const file = await RNHTMLtoPDF.convert(options);
      console.log('✅ Simple PDF generated:', file);
      
      // Verify file exists on Android
      if (Platform.OS === 'android' && file.filePath) {
        const fileExists = await RNFS.exists(file.filePath);
        console.log('✅ Simple PDF file exists on Android:', fileExists);
        
        if (fileExists) {
          const fileStats = await RNFS.stat(file.filePath);
          console.log('✅ Simple PDF file stats:', {
            size: fileStats.size,
            isFile: fileStats.isFile(),
            path: file.filePath
          });
        }
      }
      
      return file.filePath;
    } catch (error) {
      console.error('❌ Simple PDF generation failed:', error);
      console.error('❌ Simple PDF Error details:', {
        name: error.name,
        message: error.message,
        stack: error.stack,
        platform: require('react-native').Platform.OS
      });
      throw error;
    }
  };

  const generatePDFWithHtml = async () => {
    try {
      let checklistHtml = '';
      const safeChecklistItems = Array.isArray(checklistItems) ? checklistItems : [];
      if (safeChecklistItems.length > 0) {
        checklistHtml = safeChecklistItems.map(section => `
          <div style="margin-bottom: 10px;">
            <h3 style="color: #015185; font-size: 16px; margin-bottom: 6px; border-bottom: 1.5px solid #e0e0e0; padding-bottom: 2px;">${section.sectionName}</h3>
            ${(Array.isArray(section.subItems) ? section.subItems : []).map(subItem => `
              <div style="display: flex; justify-content: space-between; padding: 5px 0; border-bottom: 1px solid #f0f0f0;">
                <span style="font-size: 14px; color: #222;">${subItem.text}</span>
                <span style="font-size: 14px; color: ${subItem.status === 'Reject' ? '#D32F2F' : '#FBC02D'}; font-weight: 600;">${subItem.status}</span>
              </div>
            `).join('')}
          </div>
        `).join('');
      } else {
        checklistHtml = '<p>No checklist items.</p>';
      }

      const safeItems = Array.isArray(items) ? items : [];
      let taskRows = safeItems.filter(item => item.category === 'Task').map((item, index) => `
        <tr>
          <td>${index + 1}</td>
          <td>${item.text || 'N/A'}</td>
          <td>${item.type || 'N/A'}</td>
          <td>${item.status || 'N/A'}</td>
          <td>${item.note || 'N/A'}</td>
        </tr>
      `).join('') || '<tr><td colspan="5">No tasks available</td></tr>';

      let laborRows = safeItems.filter(item => item.category === 'Labor').map((item, index) => `
        <tr>
          <td>${index + 1}</td>
          <td>${item.worker || 'N/A'}</td>
          <td>${item.code || 'N/A'}</td>
          <td>${item.rate || 'N/A'}</td>
          <td>${item.hours || '0'}h ${item.minutes || '0'}m</td>
        </tr>
      `).join('') || '<tr><td colspan="5">No labor items available</td></tr>';

      let partsRows = safeItems.filter(item => item.category === 'Parts').map((item, index) => `
        <tr>
          <td>${index + 1}</td>
          <td>${item.itemNumber || 'N/A'}</td>
          <td>${item.partNumber || 'N/A'}</td>
          <td>${item.itemName || 'N/A'}</td>
          <td>${item.quantity || '0'} ${item.unit || 'N/A'}</td>
          <td>${item.itemCost || 'N/A'}</td>
        </tr>
      `).join('') || '<tr><td colspan="6">No parts available</td></tr>';

      let additionalCostRows = safeItems.filter(item => item.category === 'Additional Cost').map((item, index) => `
        <tr>
          <td>${index + 1}</td>
          <td>${item.description || 'N/A'}</td>
          <td>${item.amount || 'N/A'}</td>
        </tr>
      `).join('') || '<tr><td colspan="3">No additional costs available</td></tr>';

      let remarksRows = safeItems.filter(item => item.category === 'Remarks').map((item, index) => `
        <tr>
          <td>${index + 1}</td>
          <td>${item.description || 'N/A'}</td>
        </tr>
      `).join('') || '<tr><td colspan="2">No remarks available</td></tr>';

      const html = `
        <!DOCTYPE html>
        <html lang="en">
        <head>
          <meta charset="UTF-8" />
          <meta name="viewport" content="width=device-width, initial-scale=1.0" />
          <title>Job Card: ${workOrderTitle || 'Untitled'}</title>
          <style>
            body { font-family: 'Segoe UI', Arial, sans-serif; padding: 24px; font-size: 13px; background: #f8f9fa; color: #222; }
            .header, .section { border-radius: 8px; background: #fff; box-shadow: 0 1px 4px #e0e0e0; padding: 16px 20px; margin-bottom: 18px; }
            .header-table, .checklist-table { width: 100%; border-collapse: collapse; margin-bottom: 10px; }
            .header-table td { border: none; padding: 3px 0; }
            .checklist-table th, .checklist-table td { border: 1px solid #e0e0e0; padding: 7px 6px; text-align: left; }
            .title { text-align: center; font-weight: bold; font-size: 22px; margin: 24px 0 18px 0; color: #015185; letter-spacing: 1px; }
            .notes { font-size: 11px; margin-bottom: 10px; color: #888; }
            .checklist-table th { background-color: #015185; color: #fff; font-weight: 600; font-size: 14px; }
            .summary-section { margin-top: 24px; background: #fff; border-radius: 8px; box-shadow: 0 1px 4px #e0e0e0; padding: 18px 20px; }
            .summary-section h3 { color: #015185; font-size: 16px; margin-bottom: 10px; border-bottom: 1.5px solid #e0e0e0; padding-bottom: 2px; }
            .summary-section p { margin: 5px 0; }
            .section strong { color: #015185; }
            .section label { color: #888; font-weight: 400; }
          </style>
        </head>
        <body>
          <div class="header">
            <table class="header-table">
              <tr>
                <td rowspan="5"><img src="https://www.titanafricadrilling.com/wp-content/uploads/2022/10/titan-drlling_blue.png" alt="Logo" width="100"></td>
                <td><strong>Document Name:</strong> HSE-FOR-8.1.2 Job Card</td>
              </tr>
              <tr><td><strong>Document ID:</strong> HSE-029</td></tr>
              <tr><td><strong>Version Date:</strong> 05/08/2021</td></tr>
              <tr><td><strong>Version:</strong> 1</td></tr>
              <tr><td><strong>Approved:</strong> HSM</td></tr>
            </table>
          </div>

          <div class="title">JOB CARD</div>
          <div class="section">
            <div style="display: flex; flex-wrap: wrap; gap: 18px 32px;">
              <div><strong>NO:</strong> ${requestNumber}</div>
              <div><strong>DATE:</strong> ${new Date().toLocaleDateString('en-US')}</div>
              <div><strong>COUNTRY:</strong> ${country || 'N/A'}</div>
              <div><strong>PROJECT:</strong> ${project || 'N/A'}</div>
              <div><strong>EQUIPMENT:</strong> ${selectedEquipment?.equipmentName || 'N/A'}</div>
              <div><strong>CREATED AT:</strong> ${new Date().toLocaleString()}</div>
            </div>
          </div>

          <div class="summary-section">
            <h3>Checklist Items</h3>
            ${checklistHtml}
          </div>

          <div class="summary-section">
            <h3>Job Card Details</h3>
            <div style="display: flex; flex-wrap: wrap; gap: 12px 32px;">
              <div><strong>Title:</strong> ${workOrderTitle || 'N/A'}</div>
              <div><strong>Country:</strong> ${country || 'N/A'}</div>
              <div><strong>Project:</strong> ${project || 'N/A'}</div>
              <div><strong>Memo:</strong> ${memo || 'N/A'}</div>
              <div><strong>Priority:</strong> ${priority || 'N/A'}</div>
                              <div><strong>Assigned To:</strong> ${assignTo.map(a => a.fullName || a.name || 'Unknown User').join(', ') || 'None'}</div>
              <div><strong>Start Date:</strong> ${startDate ? new Date(startDate).toLocaleString() : 'N/A'}</div>
              <div><strong>Due Date:</strong> ${dueDate ? new Date(dueDate).toLocaleString() : 'N/A'}</div>
              <div><strong>Job Card Type:</strong> ${workOrderType || 'N/A'}</div>
              <div><strong>Estimated Cost:</strong> ${estimatedCost || 'N/A'}</div>
              <div><strong>Estimated Labor Time:</strong> ${estimatedLaborTime || 'N/A'}</div>
              <div><strong>Linked Service Schedule:</strong> ${serviceScheduleLink?.scheduleId || 'None'}</div>
              <div><strong>Linked Inspection:</strong> ${linkInspection?.name || 'None'}</div>
              <div><strong>Linked Job Card:</strong> ${linkedWorkOrder?.title || 'None'}</div>
            </div>
          </div>

          <div class="summary-section">
            <h3>Cost Details</h3>
            <div style="display: flex; flex-wrap: wrap; gap: 12px 32px;">
              <div><strong>Part & Material Cost:</strong> ${partMaterialCost || '0.00'}</div>
              <div><strong>Total Labor Cost:</strong> ${totalLaborCost || '0.00'}</div>
              <div><strong>Additional Cost:</strong> ${additionalCost || '0.00'}</div>
              <div><strong>Tax Amount:</strong> ${taxAmount || '0.00'} (${taxPercentage}%)</div>
              <div><strong>Total WO Cost:</strong> ${totalWoCost || '0.00'}</div>
            </div>
          </div>

          <div class="summary-section">
            <h3>Tasks</h3>
            <table class="checklist-table">
              <tr>
                <th>#</th>
                <th>Description</th>
                <th>Type</th>
                <th>Status</th>
                <th>Note</th>
              </tr>
              ${taskRows}
            </table>
          </div>

          <div class="summary-section">
            <h3>Labor</h3>
            <table class="checklist-table">
              <tr>
                <th>#</th>
                <th>Worker</th>
                <th>Code</th>
                <th>Rate</th>
                <th>Time</th>
              </tr>
              ${laborRows}
            </table>
          </div>

          <div class="summary-section">
            <h3>Parts</h3>
            <table class="checklist-table">
              <tr>
                <th>#</th>
                <th>Item No</th>
                <th>Part No</th>
                <th>Name</th>
                <th>Quantity</th>
                <th>Cost</th>
              </tr>
              ${partsRows}
            </table>
          </div>

          <div class="summary-section">
            <h3>Additional Costs</h3>
            <table class="checklist-table">
              <tr>
                <th>#</th>
                <th>Description</th>
                <th>Amount</th>
              </tr>
              ${additionalCostRows}
            </table>
          </div>

          <div class="summary-section">
            <h3>Remarks</h3>
            <table class="checklist-table">
              <tr>
                <th>#</th>
                <th>Description</th>
              </tr>
              ${remarksRows}
            </table>
          </div>
        </body>
        </html>
      `;

      // Platform-specific PDF generation options
      const Platform = require('react-native').Platform;
      
      const options = {
        html,
        fileName: `JobCard_${requestNumber}`,
        directory: Platform.OS === 'android' ? 'Download' : 'Documents',
        height: Platform.OS === 'android' ? 842 : 842, // A4 height
        width: Platform.OS === 'android' ? 595 : 595,  // A4 width
        padding: Platform.OS === 'android' ? 24 : 24,
        bgColor: Platform.OS === 'android' ? '#FFFFFF' : '#FFFFFF',
        // Android-specific options
        ...(Platform.OS === 'android' && {
          base64: false,
          type: 'application/pdf',
          quality: 1.0,
          orientation: 'portrait',
        }),
      };

      console.log('🎯 PDF generation options for', Platform.OS, ':', options);
      console.log('🎯 HTML length:', html.length);
      console.log('🎯 Request number:', requestNumber);
      console.log('🎯 Platform:', Platform.OS);

      const file = await RNHTMLtoPDF.convert(options);
      console.log('✅ PDF generated successfully:', file);
      console.log('✅ PDF file path:', file.filePath);
      
      // Verify the file exists
      const fileExists = await RNFS.exists(file.filePath);
      console.log('✅ PDF file exists:', fileExists);
      
      if (!fileExists) {
        throw new Error('PDF file was not created at expected path: ' + file.filePath);
      }
      
      return file.filePath;
    } catch (error) {
      console.error('❌ Error generating PDF:', error);
      console.error('❌ PDF Error details:', {
        name: error.name,
        message: error.message,
        stack: error.stack
      });
      throw error;
    }
  };

  // 2. Add function to fetch tasks from Firebase for the current work order
  const fetchFirebaseTasks = async (woId) => {
    try {
      const userId = await AsyncStorage.getItem('userUid');
      if (!userId || !woId) return;
      const tasksRef = database().ref(`/GlobalWorkOrders/admins/${userId}/${woId}/tasks`);
      const snapshot = await tasksRef.once('value');
      const tasksData = snapshot.val();
      if (tasksData) {
        const tasksArray = Object.keys(tasksData).map(key => ({ id: key, ...tasksData[key] }));
        setFirebaseTasks(tasksArray);
      } else {
        setFirebaseTasks([]);
      }
    } catch (error) {
      setFirebaseTasks([]);
    }
  };

  // Add function to save tasks to Firebase real-time database
  const saveTasksToFirebase = async (workOrderId, tasks, userId) => {
    try {
      console.log('📊 Saving tasks to Firebase real-time database...');
      console.log('📊 Work Order ID:', workOrderId);
      console.log('📊 Tasks to save:', tasks.length);
      console.log('📊 User ID:', userId);
      
      const workOrderRef = database().ref(`/GlobalWorkOrders/admins/${userId}/${workOrderId}`);
      
      // Save work order basic info
      const workOrderBasicInfo = {
        id: workOrderId,
        title: `${requestNumber} - ${workOrderTitle}`,
        status: 'Hold',
        priority: priority || 'Medium',
        workOrderType: workOrderType,
        country: country || '',
        project: project || '',
        equipment: selectedEquipment ? {
          equipmentName: selectedEquipment.equipmentName || '',
          mainCategory: selectedEquipment.mainCategory || '',
          model: selectedEquipment.model || '',
          vehicleNumber: selectedEquipment.vehicleNumber || '',
          meterReading: selectedEquipment.meterReading || 0,
          status: selectedEquipment.status || ''
        } : null,
        assignTo: assignTo.map(user => ({
          name: user.name || user.fullName || '',
          email: user.email || '',
          uid: user.uid || user.id || ''
        })),
        startDate: startDate ? startDate.toISOString() : null,
        dueDate: dueDate ? dueDate.toISOString() : null,
        description: memo,
        notes: memo,
        createdAt: new Date().toISOString(),
        updatedAt: new Date().toISOString(),
        createdBy: userId,
        lastModifiedBy: userId,
        linkInspection: linkInspection,
        linkedWorkOrder: linkedWorkOrder
      };
      
      console.log('📊 Saving work order basic info to Firebase...');
      await workOrderRef.update(workOrderBasicInfo);
      
      // Save tasks to Firebase
      const tasksRef = workOrderRef.child('tasks');
      const tasksData = {};
      
      tasks.forEach((task, index) => {
        const taskKey = task.id || `task_${index}_${Date.now()}`;
        
        // Ensure all required fields are present
        const firebaseTask = {
          ...task,
          id: taskKey,
          workOrderId: workOrderId,
          createdAt: task.createdAt || new Date().toISOString(),
          updatedAt: task.updatedAt || new Date().toISOString(),
          createdBy: task.createdBy || userId,
          updatedBy: task.updatedBy || userId,
          // Ensure status is properly set
          status: task.status || 'Initiated',
          // Ensure category is properly set
          category: task.category || 'Task',
          // Ensure source is properly set
          source: task.source || 'unknown'
        };
        
        tasksData[taskKey] = firebaseTask;
        
        console.log('📊 Preparing task for Firebase:', {
          key: taskKey,
          category: firebaseTask.category,
          text: firebaseTask.text?.substring(0, 50),
          status: firebaseTask.status,
          source: firebaseTask.source,
          autoGenerated: firebaseTask.note?.includes('Auto-generated')
        });
      });
      
      console.log('📊 Saving tasks data to Firebase...');
      await tasksRef.set(tasksData);
      
      console.log('✅ Tasks saved to Firebase successfully');
      console.log('📊 Total tasks saved:', Object.keys(tasksData).length);
      return true;
    } catch (error) {
      console.error('❌ Error saving tasks to Firebase:', error);
      console.error('❌ Firebase error details:', {
        code: error.code,
        message: error.message,
        stack: error.stack
      });
      return false;
    }
  };

  // Add function to update individual task in Firebase
  const updateTaskInFirebase = async (taskId, updates) => {
    try {
      if (!workOrderId) return;
      
      const userId = await AsyncStorage.getItem('userUid');
      if (!userId) return;
      
      const taskRef = database().ref(`/GlobalWorkOrders/admins/${userId}/${workOrderId}/tasks/${taskId}`);
      await taskRef.update(updates);
      
      console.log('✅ Task updated in Firebase:', taskId, updates);
    } catch (error) {
      console.error('❌ Error updating task in Firebase:', error);
    }
  };

  // Add function to delete task from Firebase
  const deleteTaskFromFirebase = async (taskId) => {
    try {
      if (!workOrderId) return;
      
      const userId = await AsyncStorage.getItem('userUid');
      if (!userId) return;
      
      const taskRef = database().ref(`/GlobalWorkOrders/admins/${userId}/${workOrderId}/tasks/${taskId}`);
      await taskRef.remove();
      
      console.log('✅ Task deleted from Firebase:', taskId);
    } catch (error) {
      console.error('❌ Error deleting task from Firebase:', error);
    }
  };

  // 3. Fetch firebase tasks when workOrderId is set (after creation)
  useEffect(() => {
    if (workOrderId) {
      fetchFirebaseTasks(workOrderId);
    }
  }, [workOrderId]);

  // Save work order to MongoDB
  const saveWorkOrder = async () => {
    console.log('🚀 Create button clicked - saveWorkOrder function called');
    
    // Show immediate feedback that button was pressed
    showToast('info', 'Creating...', 'Starting job card creation...');
    
    console.log('📊 Current form state:', {
      workOrderTitle,
      country,
      project,
      selectedEquipment: selectedEquipment?.equipmentName,
      assignTo: assignTo.length,
      isLoading
    });
    
    if (!validateFields()) {
      console.log('❌ Validation failed');
      showToast('error', 'Error', 'Please fill all required fields.');
      return;
    }
    
    console.log('✅ Validation passed, proceeding with save...');

    setIsLoading(true);
    setPdfModalVisible(true);
    
    // Skip API connection test for now to avoid hanging
    console.log('🚀 Skipping API connection test, proceeding directly to save...');
    showToast('info', 'Creating...', 'Generating PDF and saving job card...');
    
    try {
      let userId = route.params?.userId;
      if (!userId) {
        userId = await AsyncStorage.getItem('userUid');
      }
      if (!userId) {
        throw new Error('Admin not authenticated. Please log in again.');
      }

      console.log('📊 Creating job card in MongoDB...');

      // Check if inspection already has a job card
      if (linkInspection) {
        const existingWorkOrder = workOrders.find(wo => wo.linkInspection?.id === linkInspection.id);
        if (existingWorkOrder) {
          throw new Error('A work order already exists for this inspection.');
        }
      }

      // Generate request number based on total count + 1
      const finalRequestNumber = generateRequestNumber(country, project, totalJobCardCount);
      setRequestNumber(finalRequestNumber);

      // Generate work order ID (using timestamp for uniqueness)
      const workOrderIdLocal = Date.now().toString();

      // Log items before saving
      console.log('📊 Items to be saved to database:', {
        totalItems: items.length,
        items: items.map(item => ({
          id: item.id,
          category: item.category,
          text: item.text,
          status: item.status,
          type: item.type,
          note: item.note,
          autoGenerated: item.note?.includes('Auto-generated from inspection')
        }))
      });

      // Validate task data before saving
      const taskValidation = validateTaskData(items);
      if (taskValidation.invalidTasks > 0) {
        console.warn('⚠️ Some tasks have validation errors:', taskValidation.errors);
        // Continue anyway, but log the issues
      }

      // Prepare work order data - matching CreateWorkOrder.js structure exactly
      const workOrderData = {
        _id: workOrderIdLocal,
        adminUid: userId,
        title: `${finalRequestNumber} - ${workOrderTitle}`,
        status: 'Hold',
        priority: priority || 'Medium',
        workOrderType: workOrderType,
        equipment: selectedEquipment ? {
          equipmentName: selectedEquipment.equipmentName || '',
          mainCategory: selectedEquipment.mainCategory || '',
          model: selectedEquipment.model || '',
          vehicleNumber: selectedEquipment.vehicleNumber || '',
          meterReading: selectedEquipment.meterReading || 0,
          status: selectedEquipment.status || ''
        } : null,
        assignTo: assignTo.map(user => ({
          name: user.name || user.fullName || '',
          email: user.email || '',
          uid: user.uid || user.id || ''
        })),
        startDate: startDate ? startDate.toISOString() : null,
        dueDate: dueDate ? dueDate.toISOString() : null,
        completedDate: null,
        description: memo,
        notes: memo,
        location: {
          country: country || '',
          project: project || '',
          coordinates: '',
          address: ''
        },
        serviceScheduleId: serviceScheduleLink?.id || linkInspection?.id || '',
        serviceType: workOrderType,
        attachments: [],
        pdfDownloadUrl: '',
        signatureDownloadUrl: '',
        tracking: {
          currentStep: 'Initiated',
          progress: 0,
          lastUpdated: new Date().toISOString(),
          history: [{
            step: 'Initiated',
            timestamp: new Date().toISOString(),
            updatedBy: userId,
            notes: 'Job Card created'
          }]
        },
        createdAt: new Date().toISOString(),
        updatedAt: new Date().toISOString(),
        createdBy: userId,
        lastModifiedBy: userId,
        // Additional fields for compatibility
        requestNumber: finalRequestNumber,
        estimatedCost: estimatedCost || '',
        estimatedLaborTime: estimatedLaborTime || '',
        linkInspection: linkInspection,
        linkedWorkOrder: linkedWorkOrder,
        partMaterialCost: partMaterialCost || '',
        totalLaborCost: totalLaborCost || '',
        additionalCost: additionalCost || '',
        taxAmount: taxAmount || '',
        taxPercentage: taxPercentage || '',
        totalWoCost: totalWoCost || '',
        // Store detailed task data in items field (MongoDB schema expects this)
        items: Array.isArray(items) ? items.map((item, index) => {
          // Ensure all required fields are present for MongoDB storage
          const processedItem = {
            ...item,
            id: item.id || `item_${Date.now()}_${index}`,
            category: item.category || 'Task',
            text: item.text || item.itemName || item.worker || item.description || `Item ${index + 1}`,
            type: item.type || 'Repair',
            status: item.status || 'Initiated',
            note: item.note || '',
            assignTo: Array.isArray(item.assignTo) ? item.assignTo : [],
            createdAt: item.createdAt || new Date().toISOString(),
            createdBy: item.createdBy || userId,
            updatedAt: new Date().toISOString(),
            updatedBy: userId,
            // Preserve all additional fields based on category
            ...(item.category === 'Parts' && {
              itemNumber: item.itemNumber || '',
              partNumber: item.partNumber || '',
              itemName: item.itemName || item.text || '',
              quantity: item.quantity || '1',
              unit: item.unit || 'pcs',
              itemCost: item.itemCost || '0',
              site: item.site || ''
            }),
            ...(item.category === 'Labor' && {
              worker: item.worker || item.text || '',
              laborCode: item.laborCode || '',
              rate: item.rate || '0',
              hours: item.hours || '0',
              minutes: item.minutes || '0'
            }),
            ...(item.category === 'Additional Cost' && {
              description: item.description || item.text || '',
              amount: item.amount || '0'
            }),
            ...(item.category === 'Attach' && {
              fileName: item.fileName || '',
              fileUrl: item.fileUrl || ''
            }),
            ...(item.category === 'Remarks' && {
              userEmail: item.userEmail || '',
              userRole: item.userRole || '',
              description: item.description || item.text || ''
            })
          };
          
          console.log('📊 Processing item for MongoDB:', {
            id: processedItem.id,
            category: processedItem.category,
            text: processedItem.text,
            status: processedItem.status,
            type: processedItem.type,
            autoGenerated: processedItem.note?.includes('Auto-generated'),
            // Log category-specific data
            ...(item.category === 'Parts' && {
              itemNumber: processedItem.itemNumber,
              partNumber: processedItem.partNumber,
              quantity: processedItem.quantity,
              itemCost: processedItem.itemCost
            }),
            ...(item.category === 'Labor' && {
              worker: processedItem.worker,
              rate: processedItem.rate,
              hours: processedItem.hours,
              minutes: processedItem.minutes
            }),
            ...(item.category === 'Additional Cost' && {
              description: processedItem.description,
              amount: processedItem.amount
            }),
            ...(item.category === 'Attach' && {
              fileName: processedItem.fileName,
              fileUrl: processedItem.fileUrl
            })
          });
          
          return processedItem;
        }) : [],
        checklistItems: Array.isArray(checklistItems) ? checklistItems : []
      };

      // Generate PDF with Android-specific handling
      const Platform = require('react-native').Platform;
      console.log('🎯 Generating PDF for platform:', Platform.OS);
      console.log('🎯 PDF generation dependencies check:');
      console.log('🎯 RNHTMLtoPDF available:', typeof RNHTMLtoPDF);
      console.log('🎯 RNFS available:', typeof RNFS);
      
      let pdfPath;
      let pdfDownloadUrl = '';
      
      try {
        console.log('🎯 Starting PDF generation...');
        console.log('🎯 Work order data for PDF:', {
          title: workOrderTitle,
          requestNumber,
          itemsCount: items?.length || 0,
          checklistItemsCount: checklistItems?.length || 0
        });
        
        // Check if PDF generation dependencies are available
        if (typeof RNHTMLtoPDF === 'undefined' || !RNHTMLtoPDF) {
          throw new Error('RNHTMLtoPDF is not available - PDF generation dependency missing');
        }
        
        if (typeof RNFS === 'undefined' || !RNFS) {
          throw new Error('RNFS is not available - File system access dependency missing');
        }
        
        // Platform-specific PDF generation
        if (Platform.OS === 'android') {
          console.log('📱 Android PDF generation starting...');
          // Try main PDF first
          try {
            pdfPath = await generatePDFWithHtml();
            console.log('✅ Android PDF generated successfully at:', pdfPath);
          } catch (androidError) {
            console.log('⚠️ Android main PDF generation failed, trying simple fallback:', androidError.message);
            pdfPath = await generateSimplePDF();
            console.log('✅ Android simple PDF generated successfully at:', pdfPath);
          }
        } else {
          pdfPath = await generatePDFWithHtml();
          console.log('✅ iOS PDF generated successfully at:', pdfPath);
        }
        
        if (!pdfPath) {
          throw new Error('PDF generation returned null path');
        }
        
        // Enhanced file verification for Android
        console.log('🔍 Verifying PDF file creation...');
        const fileExists = await RNFS.exists(pdfPath);
        console.log('🔍 File exists check:', fileExists);
        
        if (!fileExists) {
          throw new Error('PDF file was not created at expected path: ' + pdfPath);
        }
        
        const fileStats = await RNFS.stat(pdfPath);
        console.log('✅ PDF file stats:', {
          size: fileStats.size,
          isFile: fileStats.isFile(),
          path: pdfPath,
          platform: Platform.OS
        });
        
        if (fileStats.size === 0) {
          throw new Error('PDF file was created but is empty (0 bytes)');
        }
        
        // Upload PDF to MongoDB with retry for Android
        console.log('📤 Uploading PDF to MongoDB...');
        console.log('📤 PDF path:', pdfPath);
        console.log('📤 User ID:', userId);
        console.log('📤 Platform:', Platform.OS);
        
        let uploadAttempts = 0;
        const maxUploadAttempts = Platform.OS === 'android' ? 3 : 1;
        
        while (uploadAttempts < maxUploadAttempts && !pdfDownloadUrl) {
          try {
            uploadAttempts++;
            console.log(`📤 Upload attempt ${uploadAttempts}/${maxUploadAttempts} for ${Platform.OS}...`);
            
            pdfDownloadUrl = await uploadFileToMongoDB(pdfPath, userId);
            
            if (pdfDownloadUrl) {
              console.log('✅ PDF uploaded successfully:', pdfDownloadUrl);
              break;
            } else {
              throw new Error('Upload returned empty URL');
            }
          } catch (uploadError) {
            console.error(`❌ Upload attempt ${uploadAttempts} failed:`, uploadError.message);
            
            if (uploadAttempts >= maxUploadAttempts) {
              throw uploadError;
            }
            
            // Wait before retry on Android
            if (Platform.OS === 'android') {
              console.log('⏳ Waiting 2 seconds before retry...');
              await new Promise(resolve => setTimeout(resolve, 2000));
            }
          }
        }
        
        // Ensure we have a valid download URL
        if (!pdfDownloadUrl) {
          throw new Error('PDF upload returned empty download URL after all attempts');
        }
        
        console.log('✅ PDF generation and upload completed successfully for', Platform.OS);
      } catch (pdfError) {
        console.error('❌ PDF generation/upload failed:', pdfError);
        console.error('❌ PDF Error details:', {
          name: pdfError.name,
          message: pdfError.message,
          stack: pdfError.stack
        });
        console.log('⚠️ PDF generation failed, but continuing with job card creation...');
        
        // Continue without PDF rather than failing completely
        pdfPath = null;
        pdfDownloadUrl = '';
        
        // Show user-friendly error message
        showToast('warning', 'PDF Generation Failed', 'Job card created successfully, but PDF generation failed. You can generate PDF later.');
        
        // Log the specific error for debugging
        console.log('🔍 PDF Generation Error Analysis:');
        console.log('🔍 Error type:', typeof pdfError);
        console.log('🔍 Error message:', pdfError.message);
        console.log('🔍 Error name:', pdfError.name);
        console.log('🔍 Error code:', pdfError.code);
        console.log('🔍 RNHTMLtoPDF status:', typeof RNHTMLtoPDF);
        console.log('🔍 RNFS status:', typeof RNFS);
        
        // Try fallback simple PDF generation with Android-specific handling
        console.log('🔄 Attempting fallback simple PDF generation...');
        try {
          pdfPath = await generateSimplePDF();
          console.log('✅ Fallback PDF generated successfully at:', pdfPath);
          
          // Try to upload the fallback PDF with better error handling for Android
          try {
            pdfDownloadUrl = await uploadFileToMongoDB(pdfPath, userId);
            console.log('✅ Fallback PDF uploaded successfully:', pdfDownloadUrl);
            
            if (pdfDownloadUrl) {
              console.log('✅ Fallback PDF generation and upload completed successfully');
              showToast('success', 'PDF Generated', 'Job card created successfully with fallback PDF.');
            } else {
              throw new Error('Fallback PDF upload returned empty download URL');
            }
          } catch (uploadError) {
            console.error('❌ Fallback PDF upload failed (network issue):', uploadError);
            
            // For Android, handle network errors gracefully
            if (Platform.OS === 'android' && uploadError.message?.includes('Network request failed')) {
              console.log('📱 Android network issue detected - will save job card first and retry PDF upload later');
              
              // Set a placeholder URL that indicates PDF needs to be uploaded later
              pdfDownloadUrl = '';
              pdfPath = null; // Keep path for potential later upload
              
              showToast('warning', 'Network Issue', 'Job card will be created. PDF upload will be retried automatically.');
            } else {
              throw uploadError; // Re-throw if it's not a network issue
            }
          }
        } catch (fallbackError) {
          console.error('❌ Fallback PDF generation also failed:', fallbackError);
          
          // Check if it's a network issue on Android
          if (Platform.OS === 'android' && fallbackError.message?.includes('Network request failed')) {
            console.log('📱 Android: PDF generation failed due to network - continuing without PDF');
            pdfPath = null;
            pdfDownloadUrl = '';
            
            showToast('warning', 'Network Issue', 'Job card created successfully. PDF generation will be retried when network is available.');
          } else {
            // Other types of errors
            console.error('❌ Non-network fallback error:', fallbackError);
            pdfPath = null;
            pdfDownloadUrl = '';
          }
        }
      }
      
      // Always set PDF URLs, even if empty (for debugging)
      workOrderData.pdfDownloadUrl = pdfDownloadUrl;
      workOrderData.pdfUrl = pdfDownloadUrl;
      workOrderData.documentUrl = pdfDownloadUrl;
      
      console.log('📊 Final PDF URLs set:', {
        pdfDownloadUrl: workOrderData.pdfDownloadUrl,
        pdfUrl: workOrderData.pdfUrl,
        documentUrl: workOrderData.documentUrl
      });

      setPdfModalVisible(false);

      // Save to MongoDB with retry mechanism
      console.log('📊 Saving job card to MongoDB...');
      console.log('📊 Work order data to save:', {
        title: workOrderData.title,
        adminUid: workOrderData.adminUid,
        itemsCount: workOrderData.items?.length || 0,
        hasPdf: !!workOrderData.pdfDownloadUrl,
        itemsPreview: workOrderData.items?.slice(0, 3).map(item => ({
          id: item.id,
          category: item.category,
          text: item.text?.substring(0, 50),
          status: item.status,
          source: item.source,
          // Show category-specific fields
          ...(item.category === 'Parts' && {
            itemNumber: item.itemNumber,
            partNumber: item.partNumber,
            quantity: item.quantity,
            itemCost: item.itemCost
          }),
          ...(item.category === 'Labor' && {
            worker: item.worker,
            rate: item.rate,
            hours: item.hours,
            minutes: item.minutes
          }),
          ...(item.category === 'Additional Cost' && {
            description: item.description,
            amount: item.amount
          }),
          ...(item.category === 'Attach' && {
            fileName: item.fileName,
            fileUrl: item.fileUrl
          })
        }))
      });
      
      let response;
      let retryCount = 0;
      const maxRetries = 2; // Reduced retries for faster failure
      
      while (retryCount < maxRetries) {
        try {
          console.log(`📊 Attempt ${retryCount + 1} to save job card...`);
          
          // First test the API and database connections
          if (retryCount === 0) {
            console.log('🧪 Testing API and database connections...');
            try {
              const apiTest = await testAPIConnection();
              const dbTest = await testDatabaseConnection();
              console.log('🧪 Connection tests - API:', apiTest ? 'SUCCESS' : 'FAILED', 'DB:', dbTest ? 'SUCCESS' : 'FAILED');
            } catch (testError) {
              console.log('🧪 Connection tests failed:', testError.message);
            }
          }
          
          console.log('📊 About to call JobCardAPI.createJobCard with:', {
            userId,
            workOrderDataType: typeof workOrderData,
            itemsType: typeof workOrderData.items,
            itemsIsArray: Array.isArray(workOrderData.items),
            itemsLength: workOrderData.items?.length
          });
          
          response = await JobCardAPI.createJobCard(userId, workOrderData);
          console.log('📊 Save response:', response);
          console.log('📊 Response data PDF URLs:', {
            hasPdfDownloadUrl: !!response?.data?.pdfDownloadUrl,
            hasPdfUrl: !!response?.data?.pdfUrl,
            hasDocumentUrl: !!response?.data?.documentUrl,
            pdfDownloadUrl: response?.data?.pdfDownloadUrl,
            pdfUrl: response?.data?.pdfUrl,
            documentUrl: response?.data?.documentUrl
          });
          
          if (response && response.success) {
            console.log('✅ Job card saved successfully on attempt', retryCount + 1);
            break;
          } else {
            throw new Error(response?.error || 'Unknown error from API');
          }
        } catch (error) {
          retryCount++;
          console.error(`❌ Attempt ${retryCount} failed:`, error);
          console.error(`❌ Error details:`, {
            name: error.name,
            message: error.message,
            stack: error.stack
          });
          
          if (retryCount >= maxRetries) {
            console.log('❌ All retry attempts failed, but continuing...');
            // Don't throw error, just log it and continue
            response = { success: false, error: error.message };
            break;
          }
          
          // Wait before retrying
          console.log(`⏳ Waiting 2 seconds before retry ${retryCount + 1}...`);
          await new Promise(resolve => setTimeout(resolve, 2000));
        }
      }

      if (response && response.success) {
        // Set workOrderId AFTER successful save
        setWorkOrderId(workOrderIdLocal);
        
        console.log('✅ Job card saved to MongoDB:', workOrderIdLocal);
        console.log('📊 Setting success modal and PDF path...');
        
        // Update service schedule with job card information if this was created from a service schedule
        if (serviceScheduleLink && serviceScheduleLink.id) {
          console.log('📋 Updating service schedule with job card info...');
          console.log('📋 Service schedule ID:', serviceScheduleLink.id);
          console.log('📋 Job card ID:', workOrderIdLocal);
          console.log('📋 Job card request number:', finalRequestNumber);
          console.log('📋 Job card title:', workOrderTitle);
          
          try {
            const MONGODB_SERVICE_SCHEDULE_API_URL = 'https://api.titandrillingzm.com:6010';
            const updatePayload = {
              jobCardId: workOrderIdLocal,
              jobCardRequestNumber: finalRequestNumber,
              jobCardTitle: workOrderTitle,
              jobCardStatus: 'Initiated',
              jobCardCreatedAt: new Date().toISOString()
            };
            
            console.log('📋 Update payload:', updatePayload);
            
            const updateResponse = await fetch(`${MONGODB_SERVICE_SCHEDULE_API_URL}/service-schedules/${serviceScheduleLink.id}`, {
              method: 'PUT',
              headers: {
                'Content-Type': 'application/json',
              },
              body: JSON.stringify(updatePayload),
            });
            
            console.log('📋 Update response status:', updateResponse.status);
            
            if (!updateResponse.ok) {
              const errorText = await updateResponse.text();
              console.error(`⚠️ Failed to update service schedule: ${updateResponse.status}`, errorText);
            } else {
              const updateResult = await updateResponse.json();
              console.log('📋 Update result:', updateResult);
              
              if (updateResult.success) {
                console.log('✅ Service schedule updated successfully with job card info');
                console.log('✅ Updated fields:', {
                  jobCardId: updateResult.data?.jobCardId,
                  jobCardRequestNumber: updateResult.data?.jobCardRequestNumber,
                  jobCardTitle: updateResult.data?.jobCardTitle,
                  jobCardStatus: updateResult.data?.jobCardStatus,
                  jobCardCreatedAt: updateResult.data?.jobCardCreatedAt
                });
                showToast('success', 'Success', 'Service schedule updated with job card info');
              } else {
                console.log('⚠️ Service schedule update returned error:', updateResult.error);
                showToast('warning', 'Warning', 'Job card created but service schedule not updated');
              }
            }
          } catch (updateError) {
            console.error('❌ Error updating service schedule with job card info:', updateError);
            console.error('❌ Error details:', {
              name: updateError.name,
              message: updateError.message,
              stack: updateError.stack
            });
            showToast('warning', 'Warning', 'Job card created but service schedule update failed');
            // Don't fail the job card creation if service schedule update fails
          }
        } else {
          console.log('📋 No service schedule linked to this job card');
          console.log('📋 serviceScheduleLink:', serviceScheduleLink);
        }
        
        // Save to Firebase real-time database as well
        console.log('📊 Saving to Firebase real-time database...');
        console.log('📊 Tasks to save to Firebase:', items.length, items.map(item => ({
          id: item.id,
          category: item.category,
          text: item.text?.substring(0, 50),
          status: item.status,
          source: item.source
        })));
        
        const firebaseSuccess = await saveTasksToFirebase(workOrderIdLocal, items, userId);
        if (firebaseSuccess) {
          console.log('✅ Tasks saved to Firebase successfully');
        } else {
          console.log('⚠️ Firebase save failed, but MongoDB save succeeded');
        }
        
        // Update job card with PDF URL (always attempt, even if empty for debugging)
        console.log('📤 Updating job card with PDF URL:', pdfDownloadUrl);
        console.log('📤 Update parameters:', {
          userId,
          workOrderIdLocal,
          pdfDownloadUrl,
          pdfUrl: workOrderData.pdfUrl,
          documentUrl: workOrderData.documentUrl
        });
        
        try {
          const updateResponse = await JobCardAPI.updateJobCardWithAdmin(userId, workOrderIdLocal, {
            pdfDownloadUrl: pdfDownloadUrl,
            pdfUrl: workOrderData.pdfUrl,
            documentUrl: workOrderData.documentUrl
          });
          
          console.log('📤 Update response:', updateResponse);
          
          if (updateResponse && updateResponse.success) {
            console.log('✅ Job card updated with PDF URL successfully');
            console.log('✅ Updated job card data:', updateResponse.data);
          } else {
            console.log('⚠️ Failed to update job card with PDF URL:', updateResponse?.error);
          }
        } catch (updateError) {
          console.error('❌ Error updating job card with PDF URL:', updateError);
          console.error('❌ Update error details:', {
            name: updateError.name,
            message: updateError.message,
            stack: updateError.stack
          });
        }

        // Test the saved data to verify all task categories are stored correctly
        console.log('🧪 Testing saved job card data integrity...');
        setTimeout(async () => {
          try {
            const testResult = await testSavedJobCardData(workOrderIdLocal);
            if (testResult.success) {
              console.log('✅ Job card data integrity test passed:', testResult.summary);
              showToast('success', 'Data Integrity', `All ${testResult.summary.totalItems} tasks saved correctly with ${testResult.summary.categories.length} categories`);
            } else {
              console.log('❌ Job card data integrity test failed:', testResult.error);
            }
          } catch (testError) {
            console.error('❌ Error during data integrity test:', testError);
          }
        }, 1000);
        
        // Save to AsyncStorage for offline access
        await AsyncStorage.setItem(`workOrder_${finalRequestNumber}`, JSON.stringify(workOrderData));

        setSavedPdfPath(pdfPath);
        
        console.log('🎯 Setting success modal to TRUE');
        console.log('🎯 Current showSuccessModal state:', showSuccessModal);
        console.log('🎯 PDF path:', pdfPath);
        
        // Show success modal first
        setShowSuccessModal(true);
        
        console.log('🎯 Success modal state updated to TRUE');
        console.log('✅ Success modal should now be visible');
        
        // Add a small delay to ensure modal renders before showing toast
        setTimeout(() => {
          showToast('success', 'Success', 'Job Card created successfully in MongoDB and Firebase.');
        }, 200);
        
        // Refresh data in background without affecting modal
        setTimeout(async () => {
          await fetchInitialData();
          if (route.params?.onRefresh) route.params.onRefresh();
        }, 100);

        // Android-specific: Retry PDF upload if it failed due to network issues
        if (Platform.OS === 'android' && !pdfDownloadUrl && pdfPath) {
          console.log('📱 Android: Retrying PDF upload after job card save...');
          setTimeout(async () => {
            try {
              console.log('🔄 Retrying PDF upload for Android...');
              const retryPdfUrl = await uploadFileToMongoDB(pdfPath, userId);
              
              if (retryPdfUrl) {
                console.log('✅ Android PDF upload retry successful:', retryPdfUrl);
                
                // Update job card with the PDF URL
                try {
                  await JobCardAPI.updateJobCardWithAdmin(userId, workOrderIdLocal, {
                    pdfDownloadUrl: retryPdfUrl,
                    pdfUrl: retryPdfUrl,
                    documentUrl: retryPdfUrl
                  });
                  console.log('✅ Android job card updated with retry PDF URL');
                  showToast('success', 'PDF Uploaded', 'PDF has been uploaded successfully.');
                } catch (updateError) {
                  console.error('❌ Failed to update job card with retry PDF URL:', updateError);
                }
              }
            } catch (retryError) {
              console.error('❌ Android PDF upload retry failed:', retryError);
              console.log('📱 Will try again later when user views PDF');
            }
          }, 3000); // Wait 3 seconds after job card save
        }

        // Don't clean up PDF file immediately - let the success modal handle it
      } else {
        console.log('❌ MongoDB save failed, but saving locally and showing success...');
        
        // Try to save to Firebase even if MongoDB failed
        console.log('📊 Attempting to save to Firebase as fallback...');
        const firebaseSuccess = await saveTasksToFirebase(workOrderIdLocal, items, userId);
        if (firebaseSuccess) {
          console.log('✅ Tasks saved to Firebase as fallback');
        } else {
          console.log('⚠️ Firebase save also failed');
        }
        
        // Save locally as fallback
        try {
          await AsyncStorage.setItem(`workOrder_${finalRequestNumber}`, JSON.stringify(workOrderData));
          console.log('✅ Job card saved locally as fallback');
        } catch (localError) {
          console.error('❌ Failed to save locally:', localError);
        }
        
        // Set workOrderId for local reference
        setWorkOrderId(workOrderIdLocal);
        
        // Still show success modal
        setSavedPdfPath(pdfPath);
        setShowSuccessModal(true);
        const successMessage = firebaseSuccess 
          ? 'Job Card created in Firebase. MongoDB sync pending.' 
          : 'Job Card created locally. Will sync when connection is restored.';
        showToast('warning', 'Partial Success', successMessage);
        
        console.log('✅ Success modal shown with local save');
      }
    } catch (error) {
      console.error('❌ Error saving work order:', error);
      console.error('❌ Error details:', {
        name: error.name,
        message: error.message,
        code: error.code,
        stack: error.stack
      });
      
      // Provide more specific error messages
      let errorMessage = 'Failed to create job card';
      if (error.message.includes('Network request failed')) {
        errorMessage = 'Network error: Please check your internet connection and try again';
      } else if (error.message.includes('timeout')) {
        errorMessage = 'Request timeout: Server is taking too long to respond';
      } else if (error.message.includes('AbortError')) {
        errorMessage = 'Request cancelled: Please try again';
      } else if (error.message) {
        errorMessage = `Failed to create job card: ${error.message}`;
      }
      
      showToast('error', 'Error', errorMessage);
      setPdfModalVisible(false);
    } finally {
      setIsLoading(false);
    }
  };

  const handleViewPDF = async (fromSuccessModal = false) => {
    if (!workOrderId) {
      showToast('error', 'Error', 'Job Card ID not available.');
      return;
    }
    try {
      console.log('📊 Fetching PDF URL from MongoDB for work order:', workOrderId);
      
      // Fetch from MongoDB using getJobCard
      const response = await JobCardAPI.getJobCard(workOrderId);
      
      if (!response.success || !response.data) {
        console.log('❌ Work order not found in MongoDB');
        throw new Error('Work order not found in MongoDB.');
      }
      
      const workOrderData = response.data;
      console.log('📊 Work order data retrieved:', {
        hasPdfDownloadUrl: !!workOrderData.pdfDownloadUrl,
        hasPdfUrl: !!workOrderData.pdfUrl,
        hasDocumentUrl: !!workOrderData.documentUrl,
        pdfDownloadUrl: workOrderData.pdfDownloadUrl,
        pdfUrl: workOrderData.pdfUrl,
        documentUrl: workOrderData.documentUrl
      });
      
      const pdfUrl = workOrderData.pdfDownloadUrl || workOrderData.pdfUrl || workOrderData.documentUrl;
      
      if (!pdfUrl) {
        console.log('❌ PDF URL not available in work order data');
        console.log('📊 Attempting to generate PDF on demand...');
        
        // Try to generate PDF on demand with Android-specific handling
        try {
          const Platform = require('react-native').Platform;
          
          // Try different PDF generation approaches based on platform
          let pdfPath;
          if (Platform.OS === 'android') {
            console.log('📱 Android: Trying simple PDF generation first...');
            try {
              pdfPath = await generateSimplePDF();
              console.log('✅ Android simple PDF generated on demand at:', pdfPath);
            } catch (simpleError) {
              console.log('⚠️ Android simple PDF failed, trying main generator:', simpleError.message);
              pdfPath = await generatePDFWithHtml();
              console.log('✅ Android main PDF generated on demand at:', pdfPath);
            }
          } else {
            pdfPath = await generatePDFWithHtml();
            console.log('✅ iOS PDF generated on demand at:', pdfPath);
          }
          
          const userId = await AsyncStorage.getItem('userUid');
          
          // Try uploading with retry for Android
          let pdfDownloadUrl;
          try {
            pdfDownloadUrl = await uploadFileToMongoDB(pdfPath, userId);
            console.log('✅ PDF uploaded on demand:', pdfDownloadUrl);
          } catch (uploadError) {
            console.error('❌ PDF upload failed on demand:', uploadError);
            
            // For Android network issues, show informative message
            if (Platform.OS === 'android' && uploadError.message?.includes('Network request failed')) {
              showToast('warning', 'Network Issue', 'PDF is ready but upload failed due to network. Will retry automatically.');
              pdfDownloadUrl = null; // Will be handled below
            } else {
              throw uploadError;
            }
          }
          
          if (pdfDownloadUrl) {
            // Update the job card with the new PDF URL
            try {
              await JobCardAPI.updateJobCardWithAdmin(userId, workOrderId, {
                pdfDownloadUrl: pdfDownloadUrl,
                pdfUrl: pdfDownloadUrl,
                documentUrl: pdfDownloadUrl
              });
              console.log('✅ Job card updated with PDF URL');
            } catch (updateError) {
              console.error('❌ Failed to update job card with PDF URL:', updateError);
            }
            
            // Navigate to PDF viewer with the new URL
            navigation.navigate('PdfviewPage', {
              pdfUrl: pdfDownloadUrl,
              onGoBack: () => navigation.navigate('ServiceScheduleHome'),
            });
            return;
          }
          
          // If upload failed but PDF was generated, still try to show it
          throw new Error('PDF generated but upload failed due to network issues.');
          
        } catch (pdfGenError) {
          console.error('❌ Failed to generate PDF on demand:', pdfGenError);
          
          // For Android network issues, provide better user feedback
          if (Platform.OS === 'android' && pdfGenError.message?.includes('Network request failed')) {
            showToast('warning', 'Network Issue', 'Unable to generate PDF due to network connectivity. Please check your connection and try again.');
            return;
          }
          
          // Try to get the existing PDF URL from the work order
          if (workOrderData.pdfDownloadUrl || workOrderData.pdfUrl || workOrderData.documentUrl) {
            const existingPdfUrl = workOrderData.pdfDownloadUrl || workOrderData.pdfUrl || workOrderData.documentUrl;
            console.log('✅ Using existing PDF URL:', existingPdfUrl);
            
            navigation.navigate('PdfviewPage', {
              pdfUrl: existingPdfUrl,
              onGoBack: () => navigation.navigate('ServiceScheduleHome'),
            });
            return;
          }
          
          throw new Error('PDF could not be generated. Please try again later.');
        }
      }
      
      console.log('✅ PDF URL found:', pdfUrl);
      
      navigation.navigate('PdfviewPage', {
        pdfUrl: pdfUrl,
        onGoBack: () => navigation.navigate('ServiceScheduleHome'),
      });
    } catch (error) {
      console.error('❌ Error fetching PDF URL:', error);
      
      let errorMessage = 'Could not fetch the PDF URL.';
      if (error.message.includes('PDF could not be generated')) {
        errorMessage = 'PDF is still being generated. Please wait a moment and try again.';
      } else if (error.message.includes('not found')) {
        errorMessage = 'Job card not found. Please refresh and try again.';
      } else if (error.message.includes('Network')) {
        errorMessage = 'Network error. Please check your connection and try again.';
      }
      
      showToast('error', 'Error', errorMessage);
      // Don't hide the modal on error - let user try again
    }
  };

  const saveItem = () => {
    if (!newItemText.trim()) {
      showToast('error', 'Error', 'Task description is required.');
      return;
    }
    
    const currentTime = new Date().toISOString();
    const itemId = `custom_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
    
    const newItem = {
      id: itemId,
      category: selectedCategory || 'Task',
      text: newItemText.trim(),
      type: taskType || 'Repair',
      note: note || '',
      assignTo: Array.isArray(tempAssignTo) ? tempAssignTo : [],
      status: 'Initiated',
      createdAt: currentTime,
      createdBy: 'User',
      updatedAt: currentTime,
      updatedBy: 'User',
      source: 'custom',
      // Ensure all fields are present for consistency
      itemNumber: selectedCategory === 'Parts' ? '' : '',
      partNumber: selectedCategory === 'Parts' ? '' : '',
      itemName: selectedCategory === 'Parts' ? newItemText.trim() : '',
      quantity: selectedCategory === 'Parts' ? '1' : '',
      unit: selectedCategory === 'Parts' ? 'pcs' : '',
      itemCost: selectedCategory === 'Parts' ? '0' : '',
      site: selectedCategory === 'Parts' ? '' : '',
      worker: selectedCategory === 'Labor' ? newItemText.trim() : '',
      laborCode: selectedCategory === 'Labor' ? '' : '',
      rate: selectedCategory === 'Labor' ? '0' : '',
      hours: selectedCategory === 'Labor' ? '0' : '',
      minutes: selectedCategory === 'Labor' ? '0' : '',
      description: selectedCategory === 'Additional Cost' || selectedCategory === 'Remarks' ? newItemText.trim() : '',
      amount: selectedCategory === 'Additional Cost' ? '0' : '',
      fileName: selectedCategory === 'Attach' ? newItemText.trim() : '',
      fileUrl: selectedCategory === 'Attach' ? '' : '',
      userEmail: selectedCategory === 'Remarks' ? '' : '',
      userRole: selectedCategory === 'Remarks' ? '' : '',
      // Additional tracking fields
      inspectionId: '',
      inspectionName: '',
      originalItemKey: '',
      sectionIndex: 0,
      itemIndex: 0
    };
    
    console.log('📊 Creating custom item:', {
      id: newItem.id,
      category: newItem.category,
      text: newItem.text,
      type: newItem.type,
      source: newItem.source
    });
    
    // Use InteractionManager for better iOS performance
    InteractionManager.runAfterInteractions(() => {
      setItems(prev => [...prev, newItem]);
    });
    
    // Always reset modal state and close modal
    setNewItemText('');
    setTaskType('Repair');
    setNote('');
    setTempAssignTo([]);
    setAddItemModalVisible(false);
    setEditModalVisible(false);
    setEditItem(null);
    // Hard refresh after adding a task
    handleRefresh && handleRefresh();
  };

  const saveLabor = () => {
    if (!laborWorker || !laborCode || !laborRate || !laborHours || !laborMinutes) {
      showToast('error', 'Error', 'All labor fields are required.');
      return;
    }

    const currentTime = new Date().toISOString();
    const laborId = `labor_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;

    const newLabor = {
      id: laborId,
      category: 'Labor',
      text: laborWorker, // Main text for display
      worker: laborWorker,
      laborCode: laborCode,
      rate: laborRate,
      hours: laborHours,
      minutes: laborMinutes,
      note: laborNote || '',
      status: 'Initiated',
      assignTo: [],
      createdAt: currentTime,
      createdBy: 'User',
      updatedAt: currentTime,
      updatedBy: 'User',
      source: 'custom',
      // Ensure all fields are present for consistency
      itemNumber: '', partNumber: '', itemName: '', quantity: '', unit: '', itemCost: '', site: '',
      description: '', amount: '', fileName: '', fileUrl: '', userEmail: '', userRole: '',
      inspectionId: '', inspectionName: '', originalItemKey: '', sectionIndex: 0, itemIndex: 0
    };
    
    console.log('📊 Creating labor item:', {
      id: newLabor.id,
      category: newLabor.category,
      worker: newLabor.worker,
      source: newLabor.source
    });
    
    setItems([...items, newLabor]);
    resetLaborFields();
  };

  const saveParts = () => {
    if (!partsItemNumber || !partsPartNumber || !partsItemName || !partsQuantity || !partsItemCost) {
      showToast('error', 'Error', 'All parts fields are required.');
      return;
    }

    const currentTime = new Date().toISOString();
    const partsId = `parts_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;

    const newPart = {
      id: partsId,
      category: 'Parts',
      text: partsItemName, // Main text for display
      itemNumber: partsItemNumber,
      partNumber: partsPartNumber,
      itemName: partsItemName,
      quantity: partsQuantity,
      unit: partsUnit || 'pcs',
      itemCost: partsItemCost,
      site: partsSite || '',
      note: partsNote || '',
      status: 'Initiated',
      assignTo: [],
      createdAt: currentTime,
      createdBy: 'User',
      updatedAt: currentTime,
      updatedBy: 'User',
      source: 'custom',
      // Ensure all fields are present for consistency
      worker: '', laborCode: '', rate: '', hours: '', minutes: '',
      description: '', amount: '', fileName: '', fileUrl: '', userEmail: '', userRole: '',
      inspectionId: '', inspectionName: '', originalItemKey: '', sectionIndex: 0, itemIndex: 0
    };
    
    console.log('📊 Creating parts item:', {
      id: newPart.id,
      category: newPart.category,
      itemName: newPart.itemName,
      source: newPart.source
    });
    
    setItems([...items, newPart]);
    resetPartsFields();
  };

  const saveAdditionalCost = () => {
    if (!additionalCostDescription || !additionalCostAmount) {
      showToast('error', 'Error', 'All additional cost fields are required.');
      return;
    }

    const currentTime = new Date().toISOString();
    const costId = `cost_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;

    const newAdditionalCost = {
      id: costId,
      category: 'Additional Cost',
      text: additionalCostDescription, // Main text for display
      description: additionalCostDescription,
      amount: additionalCostAmount,
      status: 'Initiated',
      assignTo: [],
      createdAt: currentTime,
      createdBy: 'User',
      updatedAt: currentTime,
      updatedBy: 'User',
      source: 'custom',
      // Ensure all fields are present for consistency
      itemNumber: '', partNumber: '', itemName: '', quantity: '', unit: '', itemCost: '', site: '',
      worker: '', laborCode: '', rate: '', hours: '', minutes: '',
      fileName: '', fileUrl: '', userEmail: '', userRole: '',
      inspectionId: '', inspectionName: '', originalItemKey: '', sectionIndex: 0, itemIndex: 0
    };
    
    console.log('📊 Creating additional cost item:', {
      id: newAdditionalCost.id,
      category: newAdditionalCost.category,
      description: newAdditionalCost.description,
      source: newAdditionalCost.source
    });
    
    setItems([...items, newAdditionalCost]);
    resetAdditionalCostFields();
  };

  const saveRemarks = () => {
    if (!remarksDescription) {
      showToast('error', 'Error', 'Remark description is required.');
      return;
    }

    const currentTime = new Date().toISOString();
    const remarkId = `remark_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;

    const newRemark = {
      id: remarkId,
      category: 'Remarks',
      text: remarksDescription, // Main text for display
      description: remarksDescription,
      status: 'Initiated',
      assignTo: [],
      createdAt: currentTime,
      createdBy: 'User',
      updatedAt: currentTime,
      updatedBy: 'User',
      source: 'custom',
      userEmail: '', // Will be filled when saving to MongoDB
      userRole: '', // Will be filled when saving to MongoDB
      // Ensure all fields are present for consistency
      itemNumber: '', partNumber: '', itemName: '', quantity: '', unit: '', itemCost: '', site: '',
      worker: '', laborCode: '', rate: '', hours: '', minutes: '',
      amount: '', fileName: '', fileUrl: '',
      inspectionId: '', inspectionName: '', originalItemKey: '', sectionIndex: 0, itemIndex: 0
    };
    
    console.log('📊 Creating remark item:', {
      id: newRemark.id,
      category: newRemark.category,
      description: newRemark.description,
      source: newRemark.source
    });
    
    setItems([...items, newRemark]);
    resetRemarksFields();
  };

  const updateItemStatus = (itemId, newStatus) => {
    const updatedItems = items.map(item => 
      item.id === itemId 
        ? { 
            ...item, 
            status: newStatus,
            updatedAt: new Date().toISOString(),
            updatedBy: 'User'
          } 
        : item
    );
    setItems(updatedItems);
    setStatusModalVisible(false);
    
    // Update in Firebase if workOrderId exists
    if (workOrderId) {
      updateTaskInFirebase(itemId, { status: newStatus, updatedAt: new Date().toISOString(), updatedBy: 'User' });
    }
  };

  const deleteItem = (itemId) => {
    setItems(items.filter(item => item.id !== itemId));
    setEditModalVisible(false);
    setEditMenuVisible(false);
    
    // Delete from Firebase if workOrderId exists
    if (workOrderId) {
      deleteTaskFromFirebase(itemId);
    }
    setSelectedItemId(null);
  };

  const editItemSave = () => {
    if (!newItemText.trim()) {
      showToast('error', 'Error', 'Task description is required.');
      return;
    }
    if (editItem) {
      const updatedItem = { 
        ...editItem, 
        text: newItemText, 
        note, 
        assignTo: tempAssignTo, 
        type: taskType,
        updatedAt: new Date().toISOString(),
        updatedBy: 'User'
      };
      
      setItems(items.map(item =>
        item.id === editItem.id ? updatedItem : item
      ));
      
      // Update in Firebase if workOrderId exists
      if (workOrderId) {
        updateTaskInFirebase(editItem.id, {
          text: newItemText,
          note,
          assignTo: tempAssignTo,
          type: taskType,
          updatedAt: new Date().toISOString(),
          updatedBy: 'User'
        });
      }
      
      resetModalFields();
    }
  };

  const resetModalFields = () => {
    setNewItemText('');
    setTaskType('Repair');
    setNote('');
    setTempAssignTo([]);
    setAddItemModalVisible(false);
    setEditModalVisible(false);
    setEditItem(null);
  };

  const resetLaborFields = () => {
    setLaborWorker('');
    setLaborCode('');
    setLaborRate('');
    setLaborHours('');
    setLaborMinutes('');
    setLaborNote('');
    setAddLaborModalVisible(false);
    setEditLaborId(null);
  };

  const resetPartsFields = () => {
    setPartsItemNumber('');
    setPartsPartNumber('');
    setPartsItemName('');
    setPartsQuantity('');
    setPartsUnit('Piece');
    setPartsItemCost('');
    setPartsSite('');
    setPartsNote('');
    setAddPartsModalVisible(false);
    setEditPartsId(null);
  };

  const resetAdditionalCostFields = () => {
    setAdditionalCostDescription('');
    setAdditionalCostAmount('');
    setAddAdditionalCostModalVisible(false);
    setEditAdditionalCostId(null);
  };

  const resetRemarksFields = () => {
    setRemarksDescription('');
    setAddRemarksModalVisible(false);
    setEditRemarksId(null);
  };

  // Add missing state variable for search functionality
  const [assignToSearchQuery, setAssignToSearchQuery] = useState('');
  const [linkedWorkOrderSearchQuery, setLinkedWorkOrderSearchQuery] = useState('');

  const filteredTeamMembers = teamMembers.filter(member =>
    member.userId?.toLowerCase().includes(assignToSearchQuery.toLowerCase()) ||
    member.name?.toLowerCase().includes(assignToSearchQuery.toLowerCase()) ||
    member.role?.toLowerCase().includes(assignToSearchQuery.toLowerCase())
  );

  const handlePrioritySelect = selectedPriority => {
    setPriority(selectedPriority);
  };

  const handleEquipmentSelect = equipmentData => {
    setSelectedEquipment(equipmentData);
    setErrors({ ...errors, equipment: '' });
  };

  const handleAssignToSelect = item => {
    setTempAssignTo(prev => prev.some(selected => selected.id === item.id)
      ? prev.filter(selected => selected.id !== item.id)
      : [...prev, item]);
  };





  const handleRemoveAssignTo = itemToRemove => {
    setAssignTo(assignTo.filter(item => item.id !== itemToRemove.id));
  };

  const handleWorkOrderTypeSelect = type => {
    setWorkOrderType(type.name);
    setWorkOrderTypeModalVisible(false);
  };

  const handleInspectionSelect = inspection => {
    setLinkInspection({ 
      id: inspection.id, 
      name: inspection.name || inspection.step1?.requestNumber || `Inspection ${inspection.id}`,
      equipment: inspection.equipment || inspection.step1?.equipment,
      date: inspection.date || inspection.step1?.date,
      type: inspection.type || 'Inspection',
      step1: inspection.step1,
      step2: inspection.step2
    });
    setLinkInspectionModalVisible(false);
    
    // Extract faulty items from inspection and add them as tasks
    const faultyTasks = extractFaultyItemsFromInspection(inspection);
    if (faultyTasks.length > 0) {
      setTasks(faultyTasks);
      console.log('✅ Added faulty items as tasks:', faultyTasks.length);
      showToast('success', 'Linked', `Inspection linked and ${faultyTasks.length} faulty items added as tasks`);
    } else {
      console.log('📊 No faulty items found in inspection');
      showToast('success', 'Linked', `Inspection "${inspection.name || inspection.step1?.requestNumber || `Inspection ${inspection.id}`}" linked successfully`);
    }
  };



  const handleWorkOrderSelect = workOrder => {
    setLinkedWorkOrder(workOrder);
    setLinkedWorkOrderModalVisible(false);
  };

  const handleCategorySelect = category => {
    setSelectedCategory(category);
  };

  const formatDate = date => date ? new Date(date).toLocaleString('en-US', {
    year: 'numeric',
    month: 'short',
    day: 'numeric',
    hour: '2-digit',
    minute: '2-digit',
  }) : 'Select Date';

  const handleFaultSelect = fault => {
    setSelectedFaults(prev => prev.some(f => f.id === fault.id)
      ? prev.filter(f => f.id !== fault.id)
      : [...prev, fault]);
  };

  const handleSaveFaults = () => {
    setFaultListModalVisible(false);
  };

  const handleEditItem = (item) => {
    setSelectedItem(item);
    setIsEditing(true);
    setShowModal(true); // or whatever opens your modal
  };

  const handleEditLaborItem = item => {
    setLaborWorker(item.worker || '');
    setLaborCode(item.code || '');
    setLaborRate(item.rate || '');
    setLaborHours(item.hours || '');
    setLaborMinutes(item.minutes || '');
    setLaborNote(item.note || '');
    setAddLaborModalVisible(true);
    setEditMenuVisible(false);
    setSelectedItemId(null);
  };

  const handleEditPartsItem = item => {
    setPartsItemNumber(item.itemNumber || '');
    setPartsPartNumber(item.partNumber || '');
    setPartsItemName(item.itemName || '');
    setPartsQuantity(item.quantity || '');
    setPartsUnit(item.unit || 'Piece');
    setPartsItemCost(item.itemCost || '');
    setPartsSite(item.site || '');
    setPartsNote(item.note || '');
    setAddPartsModalVisible(true);
    setEditMenuVisible(false);
    setSelectedItemId(null);
    setEditPartsId(item.id);
  };

  const handleEditAdditionalCostItem = item => {
    setAdditionalCostDescription(item.description || '');
    setAdditionalCostAmount(item.amount || '');
    setAddAdditionalCostModalVisible(true);
    setEditMenuVisible(false);
    setSelectedItemId(null);
  };

  const handleEditRemarksItem = item => {
    setRemarksDescription(item.description || '');
    setAddRemarksModalVisible(true);
    setEditMenuVisible(false);
    setSelectedItemId(null);
  };

  const renderChecklistItem = ({ item: subItem }) => {
    if (!subItem.itemName || subItem.itemName.trim() === '') {
      return null; // Don't render if itemName is empty or null
    }
    const isAssigned = taskAssignments[subItem.id];
    const assignedUser = isAssigned ? availableUsers.find(u => u.id === isAssigned.assignedTo) : null;
    // Get corrective status for this subItem
    const correctiveStatus = correctiveStatuses[subItem.id.replace(/^task_/, '')];

    return (
      <View style={styles.checklistItemContainer}>
        <View style={styles.checklistItemHeader}>
          <Text style={styles.checklistItemText}>{subItem.itemName}</Text>
          <View style={styles.rightSection}>
            {isAssigned ? (
              <Text style={styles.assignedUserText}>{assignedUser ? assignedUser.name : 'N/A'}</Text>
            ) : (
              <Text style={styles.notAssignedText}>Not assigned</Text>
            )}
            <TouchableOpacity 
              style={styles.threeDotsButton}
              onPress={() => {
                setSelectedTaskId(subItem.id);
                setAssignMenuVisible(true);
              }}
            >
              <MaterialIcons name="more-vert" size={24} color="#015185" />
            </TouchableOpacity>
          </View>
        </View>
        {/* Show corrective status if available */}
        {correctiveStatus && (
          <Text style={{ color: correctiveStatus === 'Fixed' ? '#28a745' : correctiveStatus === 'Not Fixed' ? '#dc3545' : '#808080', fontWeight: 'bold', marginTop: 4 }}>
            Status: {correctiveStatus}
          </Text>
        )}
      </View>
    );
  };

  const renderChecklistSection = ({ item }) => (
    <View style={styles.sectionContainer}>
      <Text style={styles.sectionName}>{item.sectionName}</Text>
      <FlatList
        data={item.subItems}
        renderItem={renderChecklistItem}
        keyExtractor={subItem => subItem.id}
      />
    </View>
  );

  const fetchAvailableUsers = async () => {
    try {
      const user = auth().currentUser;
      if (!user) return;

      const usersRef = database().ref('users');
      const snapshot = await usersRef.once('value');
      const users = snapshot.val() || {};
      
      const formattedUsers = Object.entries(users).map(([id, userData]) => ({
        id,
        name: userData.name || 'Unknown User',
        email: userData.email,
        role: userData.role || 'User'
      }));

      setAvailableUsers(formattedUsers);
    } catch (error) {
      console.error('Error fetching users:', error);
      showToast('error', 'Error', 'Failed to fetch users');
    }
  };

  const handleTaskAssignment = async (taskId, userId) => {
    try {
      const user = auth().currentUser;
      if (!user) return;

      const task = items.find(item => item.id === taskId);
      if (task?.status === 'Down Assign') {
        showToast('error', 'Cannot Assign', 'Task is in Down Assign state and cannot be reassigned');
        return;
      }

      const existingAssignment = taskAssignments[taskId];
      if (existingAssignment && existingAssignment.assignedTo === userId) {
        showToast('info', 'Already Assigned', 'This user is already assigned to this task');
        return;
      }

      const newAssignment = {
        taskId,
        assignedTo: userId,
        assignedAt: new Date().toISOString(),
        assignedBy: user.uid,
        status: 'Pending'
      };

      setTaskAssignments(prev => ({
        ...prev,
        [taskId]: newAssignment
      }));

      const assignmentRef = database().ref(`workOrders/${workOrderId}/taskAssignments/${taskId}`);
      await assignmentRef.set(newAssignment);

      showToast('success', 'Success', 'Task assigned successfully');
      setAssignMenuVisible(false);
    } catch (error) {
      console.error('Error assigning task:', error);
      showToast('error', 'Error', 'Failed to assign task');
    }
  };

  const handleRemoveAssignment = async (taskId) => {
    try {
      const user = auth().currentUser;
      if (!user) return;

      const task = items.find(item => item.id === taskId);
      if (task?.status === 'Down Assign') {
        showToast('error', 'Cannot Modify', 'Task is in Down Assign state and cannot be modified');
        return;
      }

      setTaskAssignments(prev => {
        const newAssignments = { ...prev };
        delete newAssignments[taskId];
        return newAssignments;
      });

      const assignmentRef = database().ref(`workOrders/${workOrderId}/taskAssignments/${taskId}`);
      await assignmentRef.remove();

      showToast('success', 'Success', 'Assignment removed successfully');
      setAssignMenuVisible(false);
    } catch (error) {
      console.error('Error removing assignment:', error);
      showToast('error', 'Error', 'Failed to remove assignment');
    }
  };

  const renderUserSelectionModal = () => (
    <Modal
      visible={showUserSelectionModal}
      transparent={true}
      animationType="fade"
      onRequestClose={() => setShowUserSelectionModal(false)}
    >
      <View style={styles.modalOverlay}>
        <View style={styles.userSelectionModal}>
          <View style={styles.modalHeader}>
            <Text style={styles.modalTitle}>Assign Task</Text>
            <TouchableOpacity onPress={() => setShowUserSelectionModal(false)}>
              <MaterialIcons name="close" size={24} color="#666" />
            </TouchableOpacity>
          </View>
          
          {/* Search input for team members */}
          <View style={styles.searchContainer}>
            <TextInput
              style={styles.searchInput}
              placeholder="Search team members..."
              placeholderTextColor="#888"
              value={assignToSearchQuery}
              onChangeText={setAssignToSearchQuery}
            />
            <MaterialIcons name="search" size={20} color="#888" />
          </View>
          <FlatList
            data={availableUsers}
            keyExtractor={(item) => item.id}
            renderItem={({ item }) => (
              <TouchableOpacity
                style={styles.userItem}
                onPress={() => handleTaskAssignment(selectedTaskId, item.id)}
              >
                <View style={styles.userInfo}>
                  <Text style={styles.userName}>{item.name}</Text>
                  <Text style={styles.userRole}>{item.role}</Text>
                </View>
                <MaterialIcons name="chevron-right" size={24} color="#666" />
              </TouchableOpacity>
            )}
          />
        </View>
      </View>
    </Modal>
  );

  useEffect(() => {
    fetchAvailableUsers();
  }, []);

  // Clear extracted project ref when component unmounts or route params change
  useEffect(() => {
    return () => {
      extractedProjectRef.current = null;
    };
  }, [route.params?.selectedReport]);

  // Handle setting project when extractedProjectRef is available and projectItems are populated
  useEffect(() => {
    if (extractedProjectRef.current && projectItems.length > 0 && country) {
      const isValidProject = projectItems.some(item => item.value === extractedProjectRef.current);
      console.log('Setting project from ref:', {
        extractedProject: extractedProjectRef.current,
        projectItems: projectItems.map(item => item.value),
        isValidProject
      });
      
      if (isValidProject) {
        console.log('✅ Setting project from ref:', extractedProjectRef.current);
        setProject(extractedProjectRef.current);
        console.log('Project set from ref to:', extractedProjectRef.current);
        extractedProjectRef.current = null;
      } else {
        console.log('❌ Invalid project from ref:', extractedProjectRef.current, 'Available projects:', projectItems);
      }
    }
  }, [extractedProjectRef.current, projectItems, country]);

  // Handle project items based on country selection
  useEffect(() => {
    if (country === 'Zambia') {
      setProjectItems([
        { label: 'Kobold', value: 'Kobold' },
        { label: 'Kalumbila', value: 'Kalumbila' },
        { label: 'Kansanshi', value: 'Kansanshi' },
        { label: 'Kimiteto', value: 'Kimiteto' },
        { label: 'Mimosa', value: 'Mimosa' },
      ]);
    } else if (country === 'DRC') {
      setProjectItems([
        { label: 'Musompo', value: 'Musompo' },
        { label: 'IME', value: 'IME' },
      ]);
    } else {
      setProjectItems([]);
    }
    
    // Clear project if country is not set or if current project is not valid for the new country
    if (!country || (project && !projectItems.some((item) => item.value === project))) {
      setProject(null);
    }
  }, [country]);

  // Update request number when country, project, or total count changes
  useEffect(() => {
    const updateRequestNumber = async () => {
      try {
        console.log('📊 Generating request number for country:', country, 'project:', project, 'totalCount:', totalJobCardCount);
        
        if (country && project) {
          // Generate request number based on total job card count + 1
          const newRequestNumber = generateRequestNumber(country, project, totalJobCardCount);
          setRequestNumber(newRequestNumber);
          console.log('✅ Request number generated:', newRequestNumber);
        } else {
          setRequestNumber('');
          console.log('📊 Waiting for country and project selection');
        }
      } catch (error) {
        console.error('❌ Error generating request number:', error);
        setRequestNumber(generateRequestNumber(country, project, totalJobCardCount));
      }
    };
    updateRequestNumber();
  }, [country, project, totalJobCardCount]);

  const [datePickerVisible, setDatePickerVisible] = useState(false);
  const [datePickerMode, setDatePickerMode] = useState('start'); // 'start' or 'due'
  
  // iOS-specific modal state management
  const [modalKey, setModalKey] = useState(0);

  // Monitor modal state changes
  useEffect(() => {
    console.log('addItemModalVisible changed to:', addItemModalVisible);
    console.log('iosModalVisible changed to:', iosModalVisible);
    
    // Update ref when modal state changes
    const isModalOpen = addItemModalVisible || iosModalVisible;
    console.log('Modal is open:', isModalOpen);
  }, [addItemModalVisible, iosModalVisible]);

  // Add focus listener to handle modal state when returning from other screens
  useEffect(() => {
    const unsubscribe = navigation.addListener('focus', () => {
      console.log('Screen focused - checking modal states');
      console.log('wasModalOpenBeforeNavigation:', wasModalOpenBeforeNavigation);
      console.log('isNavigating:', isNavigating);
      
      // Force a re-render of modals if needed
      if (Platform.OS === 'ios') {
        forceModalUpdate();
      }
      
      // Focus listener is now disabled to prevent double reopening
      console.log('Focus listener: Disabled to prevent conflicts');
    });

    return unsubscribe;
  }, [navigation, wasModalOpenBeforeNavigation, isNavigating]);

  // Add blur listener to ensure modal is closed when leaving the screen
  useEffect(() => {
    const unsubscribe = navigation.addListener('blur', () => {
      console.log('Screen blurred - ensuring modal is closed');
      // Ensure modal is closed when leaving the screen
      setAddItemModalVisible(false);
      setIosModalVisible(false);
    });

    return unsubscribe;
  }, [navigation]);


  const [shouldReopenAddTaskModal, setShouldReopenAddTaskModal] = useState(false);
  const [wasModalOpenBeforeNavigation, setWasModalOpenBeforeNavigation] = useState(false);
  const [iosModalVisible, setIosModalVisible] = useState(false);
  const [isNavigating, setIsNavigating] = useState(false);
  const modalStateRef = useRef({ wasOpen: false, shouldReopen: false });
  
  // Force modal re-render on iOS to prevent issues
  const forceModalUpdate = () => {
    console.log('forceModalUpdate called, current modalKey:', modalKey);
    setModalKey(prev => {
      const newKey = prev + 1;
      console.log('New modalKey:', newKey);
      return newKey;
    });
  };

  // Fetch and filter team members from MongoDB
  const fetchFilteredTeamMembers = async () => {
    try {
      const adminUid = await AsyncStorage.getItem('userUid');
      console.log('📊 Admin UID:', adminUid);
      if (!adminUid) {
        console.log('❌ No admin UID found');
        return setTeamMembers([]);
      }

      // Fetch admin's data from MongoDB
      console.log('📊 Fetching admin data from MongoDB...');
      const adminResponse = await MongoAPI.getUserById(adminUid);
      
      if (!adminResponse.success || !adminResponse.data) {
        console.log('❌ Admin data not found in MongoDB');
        return setTeamMembers([]);
      }

      const userData = adminResponse.data;
      console.log('✅ User data from MongoDB:', userData);

      // Set admin profile
      const adminProfileData = {
        ...userData,
        id: adminUid,
        fullName: userData.name || 'N/A',
        employeeNumber: userData.employeeNumber || 'N/A',
        projects: userData.projects || [],
        countries: userData.countries || [],
        role: userData.role || 'Operator',
        isBlocked: userData.isBlocked || false,
        password: userData.password || 'N/A',
        profilePic: userData.profilePic || null,
      };
      setAdminProfile(adminProfileData);
      console.log('✅ Admin profile set:', adminProfileData);

      // Get permissions
      const teamEmployModule = userData.modules?.find(m => m.module === 'teamEmploy');
      const permissions = teamEmployModule?.permissions || [];
      const userCountry = userData.countries || [];
      const userProject = userData.projects || [];
      
      console.log('📊 Admin countries:', userCountry);
      console.log('📊 Admin projects:', userProject);
      console.log('📊 Admin permissions:', permissions);

      setUserPermissions(permissions);

      // Check for OnlyMineView permission
      const hasOnlyMineView = permissions.some(p => p.toLowerCase() === 'onlymineview');
      setHasOnlyMineView(hasOnlyMineView);
      
      console.log('📊 Has OnlyMineView permission:', hasOnlyMineView);

      if (hasOnlyMineView) {
        console.log('📊 User has OnlyMineView permission, showing only admin profile');
        setTeamMembers([]);
        return;
      }

      // Fetch all team members from MongoDB
      console.log('📊 Fetching all team members from MongoDB...');
      const teamResponse = await MongoAPI.getUsersFast(1, 8000);
      
      if (teamResponse && teamResponse.success && teamResponse.data && teamResponse.data.length > 0) {
        const membersArray = teamResponse.data.map((user) => ({
          id: user._id || user.id,
          fullName: user.name || user.fullName || 'N/A',
          name: user.name || user.fullName || 'N/A',
          employeeNumber: user.employeeNumber || 'N/A',
          email: user.email || user.employeeNumber || 'N/A',
          projects: user.projects || [],
          countries: user.countries || [],
          role: user.role || 'Operator',
          isBlocked: user.isBlocked || false,
          profilePic: user.profilePic || null,
        }));

        console.log('📊 Total members found:', membersArray.length);

        // Filter team members by country and project
        let filteredMembers = membersArray.filter(member => {
          // Skip the current admin
          if (member.id === adminUid) return false;
          
          // Apply country and project filtering
          const countryMatch = userData.countries.length === 0 || 
            (member.countries && member.countries.some(c => userData.countries.includes(c)));
          const projectMatch = userData.projects.length === 0 || 
            (member.projects && member.projects.some(p => userData.projects.includes(p)));
          
          return countryMatch && projectMatch;
        });
        
        console.log('✅ After filtering:', filteredMembers.length, 'members');
        setTeamMembers(filteredMembers);
      } else {
        console.log('❌ No team members found');
        setTeamMembers([]);
      }

    } catch (error) {
      console.error('❌ Error fetching team members:', error);
      setTeamMembers([]);
    }
  };





  // Fetch corrective action statuses when linkInspection changes (MongoDB with Firebase fallback)
  useEffect(() => {
    const fetchCorrectiveStatuses = async () => {
      if (!linkInspection) return setCorrectiveStatuses({});
      try {
        console.log('📊 Fetching corrective action statuses from MongoDB...');
        
        // Try MongoDB first
        const adminUid = await AsyncStorage.getItem('userUid');
        if (!adminUid) return;
        const inspectionResponse = await InspectionAPI.getInspection(adminUid, linkInspection.id);
        
        if (inspectionResponse.success && inspectionResponse.data) {
          const reportData = inspectionResponse.data;
          if (reportData?.correctiveAction?.subItemStatuses) {
            console.log('✅ Found corrective statuses in MongoDB');
            setCorrectiveStatuses(reportData.correctiveAction.subItemStatuses);
          } else {
            setCorrectiveStatuses({});
          }
        } else {
          console.log('❌ Inspection not found in MongoDB, trying Firebase fallback...');
          // Fallback to Firebase
          const adminUid = await AsyncStorage.getItem('userUid');
          if (!adminUid) return setCorrectiveStatuses({});
          const reportRef = database().ref(`/GlobalInspectionReport/admins/${adminUid}/${linkInspection.id}`);
          const snapshot = await reportRef.once('value');
          const reportData = snapshot.val();
          if (reportData?.correctiveAction?.subItemStatuses) {
            console.log('✅ Found corrective statuses in Firebase');
            setCorrectiveStatuses(reportData.correctiveAction.subItemStatuses);
          } else {
            setCorrectiveStatuses({});
          }
        }
      } catch (err) {
        console.error('❌ Error fetching corrective statuses:', err);
        setCorrectiveStatuses({});
      }
    };
    fetchCorrectiveStatuses();
  }, [linkInspection]);

  const findAdminUidForInspection = async (inspectionId) => {
    const adminsRef = database().ref('/GlobalInspectionReport/admins');
    const snapshot = await adminsRef.once('value');
    const adminsData = snapshot.val();
    if (!adminsData) return null;
    for (const [adminUid, reports] of Object.entries(adminsData)) {
      if (reports && reports[inspectionId]) {
        return adminUid;
      }
    }
    return null;
  };

  // --- Refactor handleRefresh ---
  const handleRefresh = async () => {
    setRefreshing(true);
    setIsLoading(true);
    try {
      await fetchInitialData();
    } catch (e) {
      // ignore
    } finally {
      setIsLoading(false);
      setRefreshing(false);
    }
  };

  const [refreshing, setRefreshing] = useState(false);

  // Add state for equipment selection modal and equipment list
  const [equipmentModalVisible, setEquipmentModalVisible] = useState(false);
  const [equipmentList, setEquipmentList] = useState([]);
  const [equipmentSearch, setEquipmentSearch] = useState('');
  const [equipmentLoading, setEquipmentLoading] = useState(false);
  const [equipmentUserInfo, setEquipmentUserInfo] = useState(null);

  // Fetch equipment list from MongoDB
  const fetchEquipmentList = async () => {
    setEquipmentLoading(true);
    try {
      console.log('📊 Fetching equipment from MongoDB...');
      const userUid = await AsyncStorage.getItem('userUid');
      
      if (!userUid) {
        console.log('❌ No user UID found');
        setEquipmentList([]);
        return;
      }

      // Fetch user data from MongoDB
      const userResponse = await MongoAPI.getUserById(userUid);
      let userData = null;
      
      if (userResponse.success && userResponse.data) {
        userData = userResponse.data;
        console.log('✅ User data from MongoDB:', userData);
      } else {
        console.log('❌ User data not found in MongoDB');
        setEquipmentList([]);
        return;
      }
      
      setEquipmentUserInfo(userData);

      // Fetch equipment from MongoDB
      const equipmentResponse = await EquipmentAPI.getAllEquipment();
      
      if (equipmentResponse.success && equipmentResponse.data) {
        let parsedData = Array.isArray(equipmentResponse.data)
          ? equipmentResponse.data.map((item, index) => ({
              ...item,
              id: item._id || item.id || `equip_${index}`,
              equipmentName: item.equipmentName || item.name,
              equipmentNumber: item.equipmentNumber || item.number,
            }))
          : [];
        
        console.log('✅ Loaded equipment from MongoDB:', parsedData.length);

        // Filter by country and project if set
        let filtered = filterEquipmentByPermission(parsedData, userData);
        if (country) {
          filtered = filtered.filter(eq => eq.country === country);
          console.log('📊 Filtered by country:', filtered.length);
        }
        if (project) {
          filtered = filtered.filter(eq => eq.project === project);
          console.log('📊 Filtered by project:', filtered.length);
        }
        
        console.log('✅ Final equipment list:', filtered.length);
        setEquipmentList(filtered);
      } else {
        console.log('❌ No equipment found in MongoDB, trying Firebase fallback...');
        // Fallback to Firebase
        const globalRef = database().ref('EquipmentsDataJSON/GlobalEquipmentJsonData');
        const snapshot = await globalRef.once('value');
        const data = snapshot.val() || [];
        let parsedData = Array.isArray(data)
          ? data.map((item, index) => ({ ...item, firebaseKey: `equip_${index}` }))
          : [];
        
        let filtered = filterEquipmentByPermission(parsedData, userData);
        if (country) filtered = filtered.filter(eq => eq.country === country);
        if (project) filtered = filtered.filter(eq => eq.project === project);
        setEquipmentList(filtered);
      }
    } catch (err) {
      console.error('❌ Error fetching equipment:', err);
      setEquipmentList([]);
    } finally {
      setEquipmentLoading(false);
    }
  };

  // Equipment selection modal UI
  const renderEquipmentModal = () => (
    <Modal
      visible={equipmentModalVisible}
      transparent={true}
      animationType="slide"
      onRequestClose={() => setEquipmentModalVisible(false)}
    >
      <View style={[styles.modalOverlay, { justifyContent: 'flex-end' }]}> 
        <View style={[styles.modalContent, { maxHeight: '80%' }]}> 
          <Text style={styles.modalTitle}>Select Equipment</Text>
          <TextInput
            style={[styles.inputmain, { marginBottom: 10 }]}
            placeholder="Search equipment..."
            placeholderTextColor="#888"
            value={equipmentSearch}
            onChangeText={setEquipmentSearch}
          />
          {equipmentLoading ? (
            <View style={{ alignItems: 'center', marginVertical: 20 }}>
              <Animated.View style={{ transform: [{ rotate: rotateInterpolate }] }}>
                <MaterialIcons name="autorenew" size={32} color="#015185" />
              </Animated.View>
              <Text style={{ color: '#015185', marginTop: 8 }}>Loading...</Text>
            </View>
          ) : (
            <FlatList
              data={equipmentList.filter(eq =>
                !equipmentSearch ||
                (eq.equipmentName?.toLowerCase().includes(equipmentSearch.toLowerCase()) ||
                  eq.equipmentNumber?.toLowerCase().includes(equipmentSearch.toLowerCase()) ||
                  eq.mainCategory?.toLowerCase().includes(equipmentSearch.toLowerCase()))
              )}
              keyExtractor={item => item.id}
              ListHeaderComponent={selectedEquipment ? (
                <TouchableOpacity
                  style={{ backgroundColor: '#e6f0ff', borderRadius: 6, padding: 10, marginBottom: 10, borderWidth: 1, borderColor: '#015185' }}
                  onPress={() => {
                    setSelectedEquipment(selectedEquipment);
                    setEquipmentModalVisible(false);
                  }}
                >
                  <Text style={{ color: '#015185', fontWeight: 'bold' }}>Selected: {selectedEquipment.equipmentName || selectedEquipment.name}</Text>
                  <Text style={{ color: '#888', fontSize: 12 }}>{selectedEquipment.equipmentNumber}</Text>
                  <Text style={{ color: '#888', fontSize: 12 }}>{selectedEquipment.mainCategory}</Text>
                </TouchableOpacity>
              ) : null}
              renderItem={({ item }) => (
                <TouchableOpacity
                  style={{ padding: 10, borderBottomWidth: 1, borderBottomColor: '#eee', backgroundColor: selectedEquipment?.id === item.id ? '#e6f0ff' : '#fff' }}
                  onPress={() => {
                    setSelectedEquipment(item);
                    setEquipmentModalVisible(false);
                    setErrors({ ...errors, equipment: '' });
                  }}
                >
                  <Text style={{ color: '#015185', fontWeight: selectedEquipment?.id === item.id ? 'bold' : '500' }}>{item.equipmentName || item.name}</Text>
                  <Text style={{ color: '#888', fontSize: 12 }}>{item.equipmentNumber}</Text>
                  <Text style={{ color: '#888', fontSize: 12 }}>{item.mainCategory}</Text>
                  {selectedEquipment?.id === item.id && (
                    <Text style={{ color: '#28a745', fontWeight: 'bold', fontSize: 12 }}>Selected</Text>
                  )}
                </TouchableOpacity>
              )}
              ListEmptyComponent={<Text style={{ color: '#888', textAlign: 'center', marginTop: 20 }}>No equipment found.</Text>}
              style={{ marginBottom: 10 }}
            />
          )}
          <TouchableOpacity
            style={[styles.modalCloseButton, { marginTop: 8 }]}
            onPress={() => setEquipmentModalVisible(false)}
          >
            <Text style={styles.modalCloseText}>Close</Text>
          </TouchableOpacity>
        </View>
      </View>
    </Modal>
  );



  // Fetch and filter job cards for the popup (MongoDB)
  const fetchFilteredWorkOrders = async () => {
    try {
      console.log('📊 Fetching filtered work orders from MongoDB...');
      const userId = await AsyncStorage.getItem('userUid');
      if (!userId) {
        console.log('❌ No user ID found');
        return setWorkOrders([]);
      }

      // Fetch all work orders from MongoDB
      const response = await JobCardAPI.getAllJobCards();
      
      if (!response.success || !response.data) {
        console.log('❌ No work orders found in MongoDB');
        return setWorkOrders([]);
      }
      
      let workOrdersArray = response.data.map(wo => ({
        id: wo._id || wo.id,
        title: wo.title,
        requestNumber: wo.requestNumber || wo.title,
        equipment: wo.equipment,
        linkInspection: wo.linkInspection,
        status: wo.status,
        createdAt: wo.createdAt,
      }));
      
      // Filter by equipment if selected
      if (selectedEquipment && selectedEquipment.id) {
        workOrdersArray = workOrdersArray.filter(wo => 
          wo.equipment && (wo.equipment.id === selectedEquipment.id || wo.equipment._id === selectedEquipment.id)
        );
        console.log('📊 Filtered work orders by equipment:', workOrdersArray.length);
      }
      
      console.log('✅ Loaded work orders from MongoDB:', workOrdersArray.length);
      setWorkOrders(workOrdersArray);
    } catch (err) {
      console.error('❌ Error fetching work orders:', err);
      setWorkOrders([]);
    }
  };

  // Add a case-insensitive check for onlyMineView permission
  const hasOnlyMineView = Array.isArray(userPermissions)
    ? userPermissions.some(p => p && typeof p === 'string' && p.toLowerCase() === 'onlymineview')
    : false;

  // --- Add state for loader when opening assign-to modal ---


  // --- Refactor open assign-to modal for Add Task ---
  const openAssignToModal = async () => {
    console.log('Navigating to UsersAssignToJC...');
    // Navigate to UsersAssignToJC with current selected users and callback
    navigation.navigate('UsersAssignToJC', {
      selectedUsers: assignTo,
      onUserSelection: (selectedUsers) => {
        console.log('Received selected users from callback:', selectedUsers);
        console.log('Selected users details:', selectedUsers.map(user => ({
          id: user.id,
          name: user.name,
          fullName: user.fullName,
          employeeNumber: user.employeeNumber,
          role: user.role
        })));
        
        // Ensure all users have consistent data structure
        const normalizedUsers = selectedUsers.map(user => ({
          ...user,
          name: user.name || user.fullName || 'Unknown User',
          fullName: user.fullName || user.name || 'Unknown User',
          employeeNumber: user.employeeNumber || 'N/A',
          role: user.role || 'Operator'
        }));
        
        console.log('Normalized users:', normalizedUsers);
        setAssignTo(normalizedUsers);
        
        // Show success message
        if (normalizedUsers.length > 0) {
          showToast('success', 'Users Selected', `${normalizedUsers.length} user(s) have been assigned to this work order.`);
        } else {
          showToast('info', 'No Users Selected', 'No users were selected for this work order.');
        }
      }
    });
  };

  // --- Open assign-to modal for task (separate from main assign-to) ---
  const openTaskAssignToModal = async () => {
    console.log('openTaskAssignToModal called - navigating to UsersAssignToJC');
    
    // Store the current modal state before navigation
    const modalWasOpen = addItemModalVisible || iosModalVisible;
    modalStateRef.current = { wasOpen: modalWasOpen, shouldReopen: modalWasOpen };
    console.log('Stored modal state before navigation:', modalWasOpen);
    
    // Close the modal before navigation
    setAddItemModalVisible(false);
    setIosModalVisible(false);
    
    // Navigate to UsersAssignToJC with current selected users and callback
    navigation.navigate('UsersAssignToJC', {
      selectedUsers: tempAssignTo,
      onUserSelection: (selectedUsers) => {
        console.log('Received selected users from callback for task:', selectedUsers);
        
        // Ensure all users have consistent data structure
        const normalizedUsers = selectedUsers.map(user => ({
          ...user,
          name: user.name || user.fullName || 'Unknown User',
          fullName: user.fullName || user.name || 'Unknown User',
          employeeNumber: user.employeeNumber || 'N/A',
          role: user.role || 'Operator'
        }));
        
        setTempAssignTo(normalizedUsers);
        
        // Show success message
        if (normalizedUsers.length > 0) {
          showToast('success', 'Users Selected', `${normalizedUsers.length} user(s) have been assigned to this task.`);
        } else {
          showToast('info', 'No Users Selected', 'No users were selected for this task.');
        }
        
        // Reopen modal if it was open before navigation
        if (modalStateRef.current.shouldReopen) {
          console.log('Reopening modal after user selection');
          setTimeout(() => {
            setAddItemModalVisible(true);
            if (Platform.OS === 'ios') {
              setIosModalVisible(true);
            }
            console.log('Modal reopened successfully');
          }, 300);
        }
      }
    });
  };

  // Loader shows for 3.5 seconds, then hides automatically
  useEffect(() => {
    let timer;
    if (isLoading) {
      timer = setTimeout(() => setIsLoading(false), 3500);
    }
    return () => clearTimeout(timer);
  }, [isLoading]);

  // Monitor success modal state changes
  useEffect(() => {
    console.log('🎯 Success modal state changed:', showSuccessModal);
    if (showSuccessModal) {
      console.log('✅ Success modal is now VISIBLE');
    } else {
      console.log('❌ Success modal is now HIDDEN');
    }
  }, [showSuccessModal]);

  // iOS-specific performance optimization
  useEffect(() => {
    if (Platform.OS === 'ios') {
      // Pre-warm the component for better performance
      const preWarm = () => {
        // This helps iOS prepare the component for rendering
        requestAnimationFrame(() => {
          // Force a layout pass
          if (scrollViewRef.current) {
            scrollViewRef.current.scrollTo({ y: 0, animated: false });
          }
        });
      };
      
      // Run after component mounts
      const timer = setTimeout(preWarm, 100);
      return () => clearTimeout(timer);
    }
  }, []);

  useEffect(() => {
    if (!linkInspection) return;
    const fetchInspectionTasks = async () => {
      try {
        console.log('📊 Fetching inspection tasks from MongoDB...');
        console.log('📊 LinkInspection object:', linkInspection);
        
        // Try MongoDB first
        const adminUid = await AsyncStorage.getItem('userUid');
        if (!adminUid) {
          console.log('❌ No admin UID found');
          return;
        }
        
        if (!linkInspection || !linkInspection.id) {
          console.log('❌ No linkInspection or linkInspection.id found');
          return;
        }
        
        console.log('📊 Fetching inspection with adminUid:', adminUid, 'inspectionId:', linkInspection.id);
        const inspectionResponse = await InspectionAPI.getInspection(adminUid, linkInspection.id);
        
        if (!inspectionResponse.success || !inspectionResponse.data) {
          console.log('❌ Inspection not found in MongoDB, trying Firebase fallback...');
          // Fallback to Firebase
          const reportRef = database().ref(`/GlobalInspectionReport/admins/${adminUid}/${linkInspection.id}`);
          const snapshot = await reportRef.once('value');
          const reportData = snapshot.val();
          if (!reportData?.step2?.sections || !reportData?.step2?.selections) return;
          
          const inspectionTasks = [];
          reportData.step2.sections.forEach(section => {
            section.subItems.forEach(subItem => {
              const sel = reportData.step2.selections[subItem.key];
              if (
                (sel === 'Not Ok' || sel === 'Reject' || sel === 'Faulty') &&
                subItem.itemName && subItem.itemName.trim() !== ''
              ) {
                inspectionTasks.push({
                  id: `task_${subItem.key}`,
                  category: 'Task',
                  text: `Fix ${subItem.itemName}`,
                  type: 'Repair',
                  note: `From inspection: ${linkInspection.name}`,
                  assignTo: [],
                  status: 'Initiated',
                  section: section.name,
                  inspectionId: linkInspection.id
                });
              }
            });
          });
          
          setItems(prev => {
            const existingTaskIds = new Set(prev.filter(i => i.category === 'Task').map(i => i.id));
            const newTasks = inspectionTasks.filter(t => !existingTaskIds.has(t.id));
            const filteredPrev = prev.filter(i => i.category !== 'Task' || !i.id.startsWith('task_'));
            return [...filteredPrev, ...newTasks];
          });
          return;
        }
        
        // Process MongoDB data
        const reportData = inspectionResponse.data;
        console.log('✅ Processing MongoDB inspection data for tasks');
        
        if (!reportData?.step2?.sections || !reportData?.step2?.selections) return;
        
        const inspectionTasks = [];
        reportData.step2.sections.forEach(section => {
          section.subItems.forEach(subItem => {
            const sel = reportData.step2.selections[subItem.key];
            if (
              (sel === 'Not Ok' || sel === 'Reject' || sel === 'Faulty') &&
              subItem.itemName && subItem.itemName.trim() !== ''
            ) {
              inspectionTasks.push({
                id: `task_${subItem.key}`,
                category: 'Task',
                text: `Fix ${subItem.itemName}`,
                type: 'Repair',
                note: `From inspection: ${linkInspection.name}`,
                assignTo: [],
                status: 'Initiated',
                section: section.name,
                inspectionId: linkInspection.id
              });
            }
          });
        });
        
        console.log('✅ Found inspection tasks:', inspectionTasks.length);
        
        // Convert to proper task format
        const tasksFromInspection = inspectionTasks.map(task => ({
          ...task,
          category: 'Task', // Ensure they appear in the Task tab
          type: 'Repair',
          status: 'Initiated',
          assignTo: [],
          note: `Auto-generated from inspection: ${linkInspection.name || linkInspection.step1?.requestNumber || 'Inspection'}`,
        }));
        
        console.log('📊 Converting inspection items to tasks:', tasksFromInspection.length);
        
        setItems(prev => {
          const existingTaskIds = new Set(prev.filter(i => i.category === 'Task').map(i => i.id));
          const newTasks = tasksFromInspection.filter(t => !existingTaskIds.has(t.id));
          const filteredPrev = prev.filter(i => i.category !== 'Task' || !i.id.startsWith('task_'));
          console.log('📊 Adding new inspection tasks to items:', newTasks.length);
          return [...filteredPrev, ...newTasks];
        });
      } catch (err) {
        console.error('❌ Error fetching inspection tasks:', err);
      }
    };
    fetchInspectionTasks();
  }, [linkInspection]);

  const handleSaveInspections = () => {
    setLinkInspectionModalVisible(false);
  };

  // Add a useEffect to add faulty items as tasks when linkInspection changes
  useEffect(() => {
    if (!linkInspection || !linkInspection.step2 || !linkInspection.step2.sections || !linkInspection.step2.selections) return;
    // Find all faulty items
    const faultyTasks = linkInspection.step2.sections.flatMap(section =>
      (section.subItems || []).filter(subItem =>
        ['Faulty', 'Not Ok', 'Reject'].includes(linkInspection.step2.selections[subItem.key])
      ).map(subItem => ({
        id: `faulty_${linkInspection.id}_${subItem.key}`,
        category: 'Task',
        text: subItem.itemName,
        type: 'Repair',
        note: `Auto-generated from inspection: ${linkInspection.name || linkInspection.step1?.requestNumber || ''}`,
        assignTo: [],
        status: 'Initiated',
        section: section.name,
        inspectionId: linkInspection.id,
      }))
    );
    // Only add new faulty tasks that are not already present
    setChecklistItems(prev => {
      const existingIds = new Set(prev.filter(i => i.category === 'Task').map(i => i.id));
      const newOnes = faultyTasks.filter(t => !existingIds.has(t.id));
      return [...prev, ...newOnes];
    });
  }, [linkInspection]);

  // In the modal, after the FlatList, show a preview of faulty items for the selected inspection
  // ...
  {linkInspection && linkInspection.step2 && linkInspection.step2.sections && linkInspection.step2.selections && (
    <View style={{ marginTop: 16, backgroundColor: '#f9f9f9', borderRadius: 8, padding: 12 }}>
      <Text style={{ fontWeight: 'bold', fontSize: 16, marginBottom: 8 }}>Faulty Items / Tasks</Text>
      {(() => {
        const faultyItems = linkInspection.step2.sections.flatMap(section =>
          (section.subItems || []).filter(subItem =>
            ['Faulty', 'Not Ok', 'Reject'].includes(linkInspection.step2.selections[subItem.key])
          ).map(subItem => ({
            itemName: subItem.itemName,
            section: section.name,
          }))
        );
        if (faultyItems.length === 0) {
          return <Text style={styles.defaultText}>No faulty items found in this inspection.</Text>;
        }
        return faultyItems.map((item, idx) => (
          <View key={idx} style={{ flexDirection: 'row', alignItems: 'center', marginBottom: 6 }}>
            <MaterialIcons name="error-outline" size={20} color="#e74c3c" style={{ marginRight: 8 }} />
            <Text style={{ color: '#333', fontSize: 15 }}>{item.itemName} <Text style={{ color: '#888' }}>({item.section})</Text></Text>
          </View>
        ));
      })()}
    </View>
  )}
  // ... existing code ...

  {serviceScheduleLink && (
    <View style={{ marginTop: 16, backgroundColor: '#e8f4fd', borderRadius: 8, padding: 12, borderLeftWidth: 4, borderLeftColor: '#3481BC' }}>
      <Text style={{ fontWeight: 'bold', fontSize: 16, marginBottom: 8, color: '#3481BC' }}>📋 Service Schedule Details</Text>
      <Text style={{ color: '#015185', fontWeight: 'bold' }}>Schedule ID: <Text style={{ color: '#333', fontWeight: 'normal' }}>{serviceScheduleLink.scheduleId || 'N/A'}</Text></Text>
      <Text style={{ color: '#015185', fontWeight: 'bold' }}>Title: <Text style={{ color: '#333', fontWeight: 'normal' }}>{serviceScheduleLink.title || 'N/A'}</Text></Text>
      <Text style={{ color: '#015185', fontWeight: 'bold' }}>Equipment: <Text style={{ color: '#333', fontWeight: 'normal' }}>{serviceScheduleLink.equipment?.equipmentName || 'N/A'}</Text></Text>
      <Text style={{ color: '#015185', fontWeight: 'bold' }}>Country: <Text style={{ color: '#333', fontWeight: 'normal' }}>{serviceScheduleLink.country || 'N/A'}</Text></Text>
      <Text style={{ color: '#015185', fontWeight: 'bold' }}>Project: <Text style={{ color: '#333', fontWeight: 'normal' }}>{serviceScheduleLink.project || 'N/A'}</Text></Text>
      <Text style={{ color: '#015185', fontWeight: 'bold' }}>Tasks: <Text style={{ color: '#2ecc40', fontWeight: 'bold' }}>{serviceScheduleLink.tasks?.length || 0} tasks auto-populated</Text></Text>
      {serviceScheduleLink.note && (
        <Text style={{ color: '#015185', fontWeight: 'bold' }}>Note: <Text style={{ color: '#333', fontWeight: 'normal' }}>{serviceScheduleLink.note}</Text></Text>
      )}
    </View>
  )}

  {linkInspection && (
    <View style={{ marginTop: 16, backgroundColor: '#f9f9f9', borderRadius: 8, padding: 12 }}>
      <Text style={{ fontWeight: 'bold', fontSize: 16, marginBottom: 8 }}>Inspection Details</Text>
      <Text style={{ color: '#015185', fontWeight: 'bold' }}>Request #: <Text style={{ color: '#333', fontWeight: 'normal' }}>{linkInspection.name || linkInspection.step1?.requestNumber || 'N/A'}</Text></Text>
      <Text style={{ color: '#015185', fontWeight: 'bold' }}>Equipment: <Text style={{ color: '#333', fontWeight: 'normal' }}>{linkInspection.equipmentName || linkInspection.step1?.selectedEquipment?.equipmentName || 'N/A'}</Text></Text>
      <Text style={{ color: '#015185', fontWeight: 'bold' }}>Date: <Text style={{ color: '#333', fontWeight: 'normal' }}>{linkInspection.createdAt ? new Date(linkInspection.createdAt).toLocaleDateString('en-US', { year: 'numeric', month: 'short', day: 'numeric' }) : 'N/A'}</Text></Text>
      <Text style={{ color: '#015185', fontWeight: 'bold' }}>Inspector: <Text style={{ color: '#333', fontWeight: 'normal' }}>{linkInspection.step1?.inspector || 'N/A'}</Text></Text>
      <Text style={{ color: '#015185', fontWeight: 'bold' }}>Project: <Text style={{ color: '#333', fontWeight: 'normal' }}>{linkInspection.step1?.project || 'N/A'}</Text></Text>
      <Text style={{ color: '#015185', fontWeight: 'bold' }}>Faulty Items: <Text style={{ color: linkInspection.step3?.faultyItems > 0 ? '#e74c3c' : '#2ecc40', fontWeight: 'bold' }}>{linkInspection.step3?.faultyItems || 0}</Text></Text>
      <Text style={{ fontWeight: 'bold', fontSize: 15, marginTop: 8, marginBottom: 4 }}>Faulty Item List:</Text>
      {linkInspection.step2 && linkInspection.step2.sections && linkInspection.step2.selections ? (
        (() => {
          const faultyItems = linkInspection.step2.sections.flatMap(section =>
            (section.subItems || []).filter(subItem =>
              ['Faulty', 'Not Ok', 'Reject'].includes(linkInspection.step2.selections[subItem.key])
            ).map(subItem => ({
              itemName: subItem.itemName,
              section: section.name,
            }))
          );
          if (faultyItems.length === 0) {
            return <Text style={{ color: '#888', fontSize: 14 }}>No faulty items found in this inspection.</Text>;
          }
          return faultyItems.map((item, idx) => (
            <Text key={idx} style={{ color: '#e74c3c', fontSize: 14 }}>
              {item.itemName} <Text style={{ color: '#888' }}>({item.section})</Text>
            </Text>
          ));
        })()
      ) : (
        <Text style={{ color: '#888', fontSize: 14 }}>No faulty items found in this inspection.</Text>
      )}
    </View>
  )}
  // ... existing code ...

  const extractFaultyItemsFromInspection = (inspection) => {
    console.log('📊 Extracting faulty items from inspection:', inspection);
    let items = [];
    
    try {
      // Check different possible data structures
      console.log('📊 Inspection structure check:');
      console.log('📊 - step2 exists:', !!inspection?.step2);
      console.log('📊 - step2.sections exists:', !!inspection?.step2?.sections);
      console.log('📊 - step2.selections exists:', !!inspection?.step2?.selections);
      console.log('📊 - step3 exists:', !!inspection?.step3);
      console.log('📊 - faultyItems exists:', !!inspection?.step3?.faultyItems);
      
      // Method 1: Try step2 structure (most common)
      if (inspection?.step2?.sections && inspection?.step2?.selections) {
        console.log('📊 Using step2 structure');
        const selections = inspection.step2.selections;
        const sections = inspection.step2.sections;
        
        console.log('📊 Selections:', selections);
        console.log('📊 Sections count:', sections.length);
        
        sections.forEach((section, sectionIndex) => {
          console.log(`📊 Processing section ${sectionIndex}:`, section.name);
          if (Array.isArray(section.subItems)) {
            section.subItems.forEach((subItem, itemIndex) => {
              const sel = selections[subItem.key];
              const isFaulty = sel === 'Faulty' || sel === 'Not Ok' || sel === 'Reject' || sel === 'Fail';
              console.log(`📊 Item ${itemIndex}: ${subItem.itemName} - Selection: ${sel} - Faulty: ${isFaulty}`);
              
              if (isFaulty && subItem.itemName && subItem.itemName.trim() !== '') {
                const taskId = `task_${subItem.key || `section_${sectionIndex}_item_${itemIndex}_${Date.now()}`}`;
                const taskText = `Fix ${subItem.itemName}`;
                const inspectionRef = inspection.name || inspection.step1?.requestNumber || inspection.id || 'Inspection';
                
                items.push({
                  id: taskId,
                  text: taskText,
                  fileName: subItem.fileName || '',
                  category: 'Task',
                  sectionName: section.name || 'Unknown Section',
                  status: 'Initiated',
                  type: 'Repair',
                  assignTo: [],
                  note: `Auto-generated from inspection: ${inspectionRef}`,
                  createdAt: new Date().toISOString(),
                  createdBy: 'System',
                  updatedAt: new Date().toISOString(),
                  updatedBy: 'System',
                  // Additional fields for better tracking
                  source: 'inspection',
                  inspectionId: inspection.id || inspection._id,
                  inspectionName: inspectionRef,
                  originalItemKey: subItem.key,
                  sectionIndex: sectionIndex,
                  itemIndex: itemIndex,
                  // Ensure all category-specific fields are present (empty for Task category)
                  itemNumber: '',
                  partNumber: '',
                  itemName: '',
                  quantity: '',
                  unit: '',
                  itemCost: '',
                  site: '',
                  worker: '',
                  laborCode: '',
                  rate: '',
                  hours: '',
                  minutes: '',
                  description: '',
                  amount: '',
                  fileUrl: '',
                  userEmail: '',
                  userRole: ''
                });
                
                console.log('📊 Auto-generated task created:', {
                  id: taskId,
                  text: taskText,
                  inspection: inspectionRef,
                  section: section.name
                });
              }
            });
          }
        });
      }
      
      // Method 2: Try step3.faultyItems structure
      if (items.length === 0 && inspection?.step3?.faultyItems) {
        console.log('📊 Using step3.faultyItems structure');
        const faultyItems = inspection.step3.faultyItems;
        
        if (Array.isArray(faultyItems)) {
          faultyItems.forEach((item, index) => {
            if (item.itemName || item.name || item.text) {
              items.push({
                id: `task_${item.id || item.key || `faulty_${index}`}`,
                text: `Fix ${item.itemName || item.name || item.text}`,
                fileName: item.fileName || 'No file',
                category: 'Task',
                sectionName: item.sectionName || item.section || 'Unknown Section',
                status: 'Initiated',
                type: 'Repair',
                assignTo: [],
                note: `Auto-generated from inspection: ${inspection.name || inspection.step1?.requestNumber || 'Inspection'}`,
                createdAt: new Date().toISOString(),
                createdBy: 'System',
                updatedAt: new Date().toISOString(),
                updatedBy: 'System',
                source: 'inspection',
                inspectionId: inspection.id || inspection._id,
                inspectionName: inspection.name || inspection.step1?.requestNumber || 'Inspection',
                originalItemKey: item.key || item.id,
                sectionIndex: 0,
                itemIndex: index,
                // Ensure all category-specific fields are present
                itemNumber: '', partNumber: '', itemName: '', quantity: '', unit: '', itemCost: '', site: '',
                worker: '', laborCode: '', rate: '', hours: '', minutes: '',
                description: '', amount: '', fileUrl: '', userEmail: '', userRole: ''
              });
            }
          });
        } else if (typeof faultyItems === 'number' && faultyItems > 0) {
          // If faultyItems is just a number, create placeholder tasks
          console.log('📊 faultyItems is a number, creating placeholder tasks');
          for (let i = 0; i < faultyItems; i++) {
            items.push({
              id: `task_placeholder_${i}`,
              text: `Fix Faulty Item ${i + 1}`,
              fileName: 'No file',
              category: 'Task',
              sectionName: 'Unknown Section',
              status: 'Initiated',
              type: 'Repair',
              assignTo: [],
              note: `Auto-generated from inspection: ${inspection.name || inspection.step1?.requestNumber || 'Inspection'}`,
              createdAt: new Date().toISOString(),
              createdBy: 'System',
              updatedAt: new Date().toISOString(),
              updatedBy: 'System',
              source: 'inspection',
              inspectionId: inspection.id || inspection._id,
              inspectionName: inspection.name || inspection.step1?.requestNumber || 'Inspection',
              originalItemKey: `placeholder_${i}`,
              sectionIndex: 0,
              itemIndex: i,
              // Ensure all category-specific fields are present
              itemNumber: '', partNumber: '', itemName: '', quantity: '', unit: '', itemCost: '', site: '',
              worker: '', laborCode: '', rate: '', hours: '', minutes: '',
              description: '', amount: '', fileUrl: '', userEmail: '', userRole: ''
            });
          }
        }
      }
      
      // Method 3: Try to extract from step2.selections directly
      if (items.length === 0 && inspection?.step2?.selections) {
        console.log('📊 Trying to extract from step2.selections directly');
        const selections = inspection.step2.selections;
        
        Object.keys(selections).forEach((key, index) => {
          const selection = selections[key];
          if (selection === 'Faulty' || selection === 'Not Ok' || selection === 'Reject' || selection === 'Fail') {
            items.push({
              id: `task_${key}`,
              text: `Fix Item ${index + 1}`,
              fileName: 'No file',
              category: 'Task',
              sectionName: 'Unknown Section',
              status: 'Initiated',
              type: 'Repair',
              assignTo: [],
              note: `Auto-generated from inspection: ${inspection.name || inspection.step1?.requestNumber || 'Inspection'}`,
              createdAt: new Date().toISOString(),
              createdBy: 'System',
              updatedAt: new Date().toISOString(),
              updatedBy: 'System',
              source: 'inspection',
              inspectionId: inspection.id || inspection._id,
              inspectionName: inspection.name || inspection.step1?.requestNumber || 'Inspection',
              originalItemKey: key,
              sectionIndex: 0,
              itemIndex: index,
              // Ensure all category-specific fields are present
              itemNumber: '', partNumber: '', itemName: '', quantity: '', unit: '', itemCost: '', site: '',
              worker: '', laborCode: '', rate: '', hours: '', minutes: '',
              description: '', amount: '', fileUrl: '', userEmail: '', userRole: ''
            });
          }
        });
      }
      
      // Method 4: Create placeholder items based on faulty count if available
      if (items.length === 0) {
        console.log('📊 No faulty items found, checking for faulty count...');
        const faultyCount = inspection?.step3?.faultyItems || inspection?.step2?.faultyItems || 0;
        if (faultyCount > 0) {
          console.log(`📊 Creating ${faultyCount} placeholder tasks based on faulty count`);
          for (let i = 0; i < faultyCount; i++) {
            items.push({
              id: `task_placeholder_${i}`,
              text: `Fix Faulty Item ${i + 1}`,
              fileName: 'No file',
              category: 'Task',
              sectionName: 'Unknown Section',
              status: 'Initiated',
              type: 'Repair',
              assignTo: [],
              note: `Auto-generated from inspection: ${inspection.name || inspection.step1?.requestNumber || 'Inspection'}`,
              createdAt: new Date().toISOString(),
              createdBy: 'System',
              updatedAt: new Date().toISOString(),
              updatedBy: 'System',
              source: 'inspection',
              inspectionId: inspection.id || inspection._id,
              inspectionName: inspection.name || inspection.step1?.requestNumber || 'Inspection',
              originalItemKey: `count_placeholder_${i}`,
              sectionIndex: 0,
              itemIndex: i,
              // Ensure all category-specific fields are present
              itemNumber: '', partNumber: '', itemName: '', quantity: '', unit: '', itemCost: '', site: '',
              worker: '', laborCode: '', rate: '', hours: '', minutes: '',
              description: '', amount: '', fileUrl: '', userEmail: '', userRole: ''
            });
          }
        }
      }
      
    } catch (error) {
      console.error('❌ Error extracting faulty items:', error);
    }
    
    console.log('📊 Final extracted faulty items:', items.length, items);
    return items;
  };

  const handleInspectionSelected = (inspection) => {
    console.log('📊 Inspection selected, extracting faulty items...');
    const faultyTasks = extractFaultyItemsFromInspection(inspection);
    
    console.log('📊 Created tasks from faulty items:', faultyTasks.length);
    
    // Add these tasks to the existing items
    setItems(prevItems => {
      // Remove any existing auto-generated tasks for this inspection
      const filteredItems = prevItems.filter(item => 
        !item.note?.includes('Auto-generated from inspection')
      );
      
      // Add new tasks
      return [...filteredItems, ...faultyTasks];
    });
    
    // Show toast notification about auto-created tasks
    if (faultyTasks.length > 0) {
      showToast('success', 'Tasks Created', `${faultyTasks.length} tasks auto-created from faulty items`);
    } else {
      showToast('info', 'No Faulty Items', 'No faulty items found in this inspection');
    }
  };

  // Function to manually extract and create tasks from linked inspection
  const handleExtractTasksFromInspection = () => {
    if (!linkInspection) {
      showToast('error', 'No Inspection', 'Please link an inspection first');
      return;
    }
    
    console.log('📊 Manually extracting tasks from linked inspection...');
    const faultyTasks = extractFaultyItemsFromInspection(linkInspection);
    
    if (faultyTasks.length > 0) {
      // Add these tasks to the existing items
      setItems(prevItems => {
        // Remove any existing auto-generated tasks for this inspection
        const filteredItems = prevItems.filter(item => 
          !item.note?.includes('Auto-generated from inspection')
        );
        
        // Add new tasks
        const updatedItems = [...filteredItems, ...faultyTasks];
        console.log('📊 Updated items count:', updatedItems.length);
        return updatedItems;
      });
      
      showToast('success', 'Tasks Created', `${faultyTasks.length} tasks created from inspection faulty items`);
      console.log('✅ Tasks successfully added to items list');
    } else {
      showToast('info', 'No Faulty Items', 'No faulty items found in the linked inspection');
      console.log('ℹ️ No faulty items found in inspection');
    }
  };

  // ... existing code ...
  // 1. Add state to track which task's menu is open:
  const [taskMenuVisibleId, setTaskMenuVisibleId] = useState(null);
  const [taskDropdownAnchor, setTaskDropdownAnchor] = useState(null);

  // 2. In the Task tab's FlatList renderItem, add the three-dot icon and dropdown:
  const renderTaskItem = ({ item }) => {
    const statusObj = statuses.find(opt => opt.label === item.status) || statuses[0];
    return (
      <View style={{ flexDirection: 'row', alignItems: 'center', justifyContent: 'space-between', paddingVertical: 8 }}>
        <View style={{ flex: 1 }}>
          <Text style={styles.itemText}>{item.text}</Text>
          {/* Show type (Repair/Replace) in red */}
          {item.type && (
            <Text style={{ color: 'red', fontWeight: 'bold', fontSize: 14, marginTop: 2 }}>
              {item.type}
            </Text>
          )}
          {item.note && <Text style={styles.itemNote}>{item.note}</Text>}
        </View>
        {/* Status button just left of three-dot menu */}
        <TouchableOpacity
          style={{
            paddingVertical: 4,
            paddingHorizontal: 10,
            borderRadius: 12,
            backgroundColor: statusObj.color,
            marginRight: 4,
          }}
          onPress={event => {
            setTaskStatusMenuVisibleId(taskStatusMenuVisibleId === item.id ? null : item.id);
            if (event && event.nativeEvent) {
              setStatusDropdownAnchor({ x: event.nativeEvent.pageX, y: event.nativeEvent.pageY });
            }
          }}
        >
          <Text style={{ color: '#fff', fontWeight: 'bold', fontSize: 13 }}>{item.status}</Text>
        </TouchableOpacity>
        {/* Status dropdown as Modal */}
        {taskStatusMenuVisibleId === item.id && statusDropdownAnchor && (
          <Modal
            visible={true}
            transparent
            animationType="fade"
            onRequestClose={() => setTaskStatusMenuVisibleId(null)}
          >
            <TouchableOpacity
              style={{ flex: 1 }}
              activeOpacity={1}
              onPress={() => setTaskStatusMenuVisibleId(null)}
            >
              <View
                style={{
                  position: 'absolute',
                  left: statusDropdownAnchor.x - 120, // Adjust width as needed
                  top: statusDropdownAnchor.y - 160, // Move further up
                  backgroundColor: '#fff',
                  borderRadius: 8,
                  elevation: 5,
                  zIndex: 200,
                  shadowColor: '#000',
                  shadowOpacity: 0.1,
                  shadowRadius: 4,
                  minWidth: 120,
                }}
              >
                {statuses.map(opt => (
                  <TouchableOpacity
                    key={opt.label}
                    style={{ flexDirection: 'row', alignItems: 'center', padding: 10 }}
                    onPress={() => {
                      updateItemStatus(item.id, opt.label);
                      setTaskStatusMenuVisibleId(null);
                    }}
                  >
                    <View style={{ width: 16, height: 16, borderRadius: 8, backgroundColor: opt.color, marginRight: 8, borderWidth: item.status === opt.label ? 2 : 0, borderColor: '#333' }} />
                    <Text style={{ color: opt.color, fontWeight: item.status === opt.label ? 'bold' : 'normal' }}>{opt.label}</Text>
                  </TouchableOpacity>
                ))}
              </View>
            </TouchableOpacity>
          </Modal>
        )}
        {/* Three-dot menu button */}
        <TouchableOpacity
          style={styles.threeDotsButton}
          onPress={event => {
            setTaskMenuVisibleId(taskMenuVisibleId === item.id ? null : item.id);
            if (event && event.nativeEvent) {
              setTaskDropdownAnchor({ x: event.nativeEvent.pageX, y: event.nativeEvent.pageY });
            }
          }}
        >
          <MaterialIcons name="more-vert" size={24} color="#015185" />
        </TouchableOpacity>
        {taskMenuVisibleId === item.id && taskDropdownAnchor && (
          <Modal
            visible={true}
            transparent
            animationType="fade"
            onRequestClose={() => setTaskMenuVisibleId(null)}
          >
            <TouchableOpacity
              style={{ flex: 1 }}
              activeOpacity={1}
              onPress={() => setTaskMenuVisibleId(null)}
            >
              <View
                style={{
                  position: 'absolute',
                  left: taskDropdownAnchor.x - 120,
                  top: taskDropdownAnchor.y - 90,
                  backgroundColor: '#fff',
                  borderRadius: 8,
                  elevation: 5,
                  zIndex: 200,
                  shadowColor: '#000',
                  shadowOpacity: 0.1,
                  shadowRadius: 4,
                  minWidth: 120,
                }}
              >
                <TouchableOpacity
                  style={{ padding: 12 }}
                  onPress={() => {
                    setTaskMenuVisibleId(null);
                    setEditItem(item);
                    setNewItemText(item.text);
                    setNote(item.note || '');
                    setEditModalVisible(true);
                  }}
                >
                  <Text style={{ color: '#015185', fontWeight: 'bold' }}>Edit</Text>
                </TouchableOpacity>
                <TouchableOpacity
                  style={{ padding: 12 }}
                  onPress={() => {
                    setTaskMenuVisibleId(null);
                    deleteItem(item.id);
                  }}
                >
                  <Text style={{ color: '#dc3545', fontWeight: 'bold' }}>Delete</Text>
                </TouchableOpacity>
              </View>
            </TouchableOpacity>
          </Modal>
        )}
      </View>
    );
  };

  // 3. Use renderTaskItem only in the Task tab's FlatList:
  {selectedCategory === 'Task' && (
    <FlatList
      data={items.filter(i => i.category === 'Task')}
      keyExtractor={item => item.id}
      renderItem={renderTaskItem}
      ListEmptyComponent={<Text style={styles.emptyText}>No tasks added yet.</Text>}
      onScrollBeginDrag={() => { setTaskMenuVisibleId(null); setTaskStatusMenuVisibleId(null); }}
    />
  )}
  // ... existing code ...
  // 4. In the edit modal, use editItem to pre-fill and update the task on save.
  // ... existing code ...

  // Add state to track which menu is open for each category
  const [partsMenuVisibleId, setPartsMenuVisibleId] = useState(null);
  const [partsDropdownAnchor, setPartsDropdownAnchor] = useState(null);
  const [laborMenuVisibleId, setLaborMenuVisibleId] = useState(null);
  const [laborDropdownAnchor, setLaborDropdownAnchor] = useState(null);
  const [additionalCostMenuVisibleId, setAdditionalCostMenuVisibleId] = useState(null);
  const [additionalCostDropdownAnchor, setAdditionalCostDropdownAnchor] = useState(null);
  const [remarksMenuVisibleId, setRemarksMenuVisibleId] = useState(null);
  const [remarksDropdownAnchor, setRemarksDropdownAnchor] = useState(null);

  // Add editItemId state for each category
  const [editPartsId, setEditPartsId] = useState(null);
  const [editLaborId, setEditLaborId] = useState(null);
  const [editAdditionalCostId, setEditAdditionalCostId] = useState(null);
  const [editRemarksId, setEditRemarksId] = useState(null);

  const handleUpdateParts = () => {
    setItems(items.map(item =>
      item.id === editPartsId
        ? {
            ...item,
            itemNumber: partsItemNumber,
            partNumber: partsPartNumber,
            itemName: partsItemName,
            quantity: partsQuantity,
            unit: partsUnit,
            itemCost: partsItemCost,
            site: partsSite,
            note: partsNote,
          }
        : item
    ));
    resetPartsFields();
  };

  const [pdfModalVisible, setPdfModalVisible] = useState(false);

  // Add these state variables at the top, with the other useState hooks:
  const [partsStatusMenuVisibleId, setPartsStatusMenuVisibleId] = useState(null);
  const [partsStatusDropdownAnchor, setPartsStatusDropdownAnchor] = useState(null);
  const [laborStatusMenuVisibleId, setLaborStatusMenuVisibleId] = useState(null);
  const [laborStatusDropdownAnchor, setLaborStatusDropdownAnchor] = useState(null);
  const [additionalCostStatusMenuVisibleId, setAdditionalCostStatusMenuVisibleId] = useState(null);
  const [additionalCostStatusDropdownAnchor, setAdditionalCostStatusDropdownAnchor] = useState(null);
  const [remarksStatusMenuVisibleId, setRemarksStatusMenuVisibleId] = useState(null);
  const [remarksStatusDropdownAnchor, setRemarksStatusDropdownAnchor] = useState(null);

  return (
    <>
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity onPress={() => navigation.goBack()}>
              <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>New Job Card</Text>
          </View>
          <View style={styles.headerRight}>
            <TouchableOpacity 
              style={[styles.saveButtoncreate, isLoading && { opacity: 0.5 }]} 
              onPress={() => {
                console.log('🎯 Create button pressed, isLoading:', isLoading);
                if (!isLoading) {
                  saveWorkOrder();
                } else {
                  console.log('⚠️ Button is disabled due to loading state');
                }
              }} 
              disabled={isLoading}
            >
              <Text style={styles.saveButtonText0}>
                {isLoading ? 'Creating...' : 'Create'}
              </Text>
            </TouchableOpacity>
          </View>
        </View>
      </LinearGradient>

      <KeyboardAvoidingView
        behavior={Platform.OS === 'ios' ? 'padding' : 'height'}
        style={styles.keyboardAvoidingView}
        keyboardVerticalOffset={Platform.OS === 'ios' ? TOTAL_HEADER_HEIGHT + 20 : 0}
        enabled={Platform.OS === 'ios'}
      >
        <ScrollView
          ref={scrollViewRef}
          keyboardShouldPersistTaps="always"
          showsVerticalScrollIndicator={false}
          style={styles.mainContent}
          contentContainerStyle={Platform.OS === 'ios' ? { paddingBottom: 100 } : undefined}
          bounces={Platform.OS === 'ios'}
          alwaysBounceVertical={Platform.OS === 'ios'}
        >
          {/* Total Job Card Count Display */}
          <View style={styles.totalCountContainer}>
            <View style={styles.totalCountContent}>
              <MaterialIcons name="assignment" size={20} color="#015185" />
              <Text style={styles.totalCountLabel}>Total Job Cards:</Text>
              <Text style={styles.totalCountValue}>{totalJobCardCount}</Text>
            </View>
          </View>

          <View style={styles.formGroup}>
            <View style={styles.requestNumberRow}>
              <Text style={styles.requestNumberLabel}>Job Card #</Text>
              <Text style={styles.requestNumberValue}>
                {requestNumber || (country && project ? 'Generating...' : 'Select Country & Project')}
              </Text>
            </View>
          </View>

          <View style={styles.formGroup}>
            <Text style={styles.label}>
              Job Card Title
              <Text style={styles.requiredError}> *</Text>
            </Text>
            <TextInput
              style={[styles.inputmain, errors.title && styles.inputError]}
              placeholder="Job Card Title"
              placeholderTextColor="#888"
              value={workOrderTitle}
              onChangeText={text => {
                setWorkOrderTitle(text);
                setErrors({ ...errors, title: text.trim() ? '' : 'Job Card Title is required' });
              }}
            />
            {errors.title && <Text style={styles.errorText}>{errors.title}</Text>}
          </View>

          <View style={styles.formGroup}>
            <Text style={styles.label}>
              Country
              <Text style={styles.requiredError}> *</Text>
            </Text>
            {/* Defensive: Show error if Dropdown is not available */}
            {ElementDropdown.Dropdown ? (
              <ElementDropdown.Dropdown
                style={[styles.inputmain, errors.country && styles.inputError]}
                containerStyle={[styles.dropdownContainer, { backgroundColor: '#fff', borderWidth: 1, borderColor: '#015185', borderRadius: 8 }]}
                dropdownStyle={styles.dropdownList}
                data={countryItems}
                labelField="label"
                valueField="value"
                placeholder="Select Country"
                value={country}
                onChange={item => {
                  setCountry(item.value);
                  setErrors({ ...errors, country: item.value ? '' : 'Country is required' });
                }}
                placeholderStyle={styles.placeholderStyle}
                selectedTextStyle={styles.dropdownText}
              />
            ) : (
              <Text style={{ color: 'red', fontWeight: 'bold' }}>Dropdown component is not available. Please check your react-native-element-dropdown installation.</Text>
            )}
            {errors.country && <Text style={styles.errorText}>Country is required</Text>}
          </View>

          <View style={styles.formGroup}>
            <Text style={styles.label}>
              Project
              <Text style={styles.requiredError}> *</Text>
            </Text>
            {ElementDropdown.Dropdown ? (
              <ElementDropdown.Dropdown
                style={[styles.inputmain, (!country && styles.disabledDropdown), errors.project && styles.inputError]}
                containerStyle={[styles.dropdownContainer, { backgroundColor: '#fff', borderWidth: 1, borderColor: '#015185', borderRadius: 8 }]}
                dropdownStyle={styles.dropdownList}
                data={projectItems}
                labelField="label"
                valueField="value"
                placeholder="Select Project"
                value={project}
                onChange={item => {
                  console.log('📊 Project selected:', item);
                  setProject(item.value);
                  setErrors({ ...errors, project: item.value ? '' : 'Project is required' });
                }}
                placeholderStyle={styles.placeholderStyle}
                selectedTextStyle={styles.dropdownText}
                disable={!country}
              />
            ) : (
              <Text style={{ color: 'red', fontWeight: 'bold' }}>Dropdown component is not available. Please check your react-native-element-dropdown installation.</Text>
            )}
            {errors.project && <Text style={styles.errorText}>Project is required</Text>}
          </View>

          <View style={styles.formGroup}>
            <Text style={styles.label}>Write Memo</Text>
            <TextInput
              style={[styles.inputmain]}
              placeholder="Check within a day"
              placeholderTextColor="#888"
              value={memo}
              onChangeText={setMemo}
              multiline
              numberOfLines={4}
            />
          </View>

          <View style={styles.formGroup}>
            <Text style={styles.label}>Priority</Text>
            <View style={styles.priorityContainer}>
              {priorityOptions.map(option => {
                const isSelected = priority === option.label;
                return (
                  <View key={option.label} style={styles.priorityItem}>
                    <TouchableOpacity
                      style={[
                        styles.priorityButton,
                        isSelected ? styles.priorityButtonSelected : styles.priorityButtonUnselected,
                      ]}
                      onPress={() => handlePrioritySelect(option.label)}
                    >
                      <Image source={option.image} style={styles.priorityImage} />
                      <Text style={isSelected ? styles.priorityTextSelected : styles.priorityText}>
                        {option.label}
                      </Text>
                    </TouchableOpacity>
                  </View>
                );
              })}
            </View>
          </View>

          <View style={[styles.formGroup, styles.dropdownContainer]}>
            <View style={styles.rowCentered}>
              <View style={styles.equipmentContainer}>
                <Text style={styles.dropdownLabel}>
                  Equipments
                  <Text style={styles.requiredError}> *</Text>
                </Text>
                {selectedEquipment && (
                  <Text style={styles.selectedEquipmentText}>
                    {selectedEquipment.equipmentName || selectedEquipment.name}
                  </Text>
                )}
                {errors.equipment && <Text style={styles.errorText}>{errors.equipment}</Text>}
              </View>
              {/* Equipment selection UI (find the button or TouchableOpacity for equipment selection) */}
              <TouchableOpacity
                style={styles.equipmentSelectButton}
                onPress={() => {
                  navigation.navigate('Select_EquipmentWo', {
                    selectedEquipment,
                    onSelect: (equipment) => {
                      setSelectedEquipment(equipment);
                      setErrors && setErrors(prev => ({ ...prev, equipment: '' }));
                    },
                  });
                }}
              >
                <Text style={styles.equipmentSelectButtonText}>
                  {selectedEquipment ? selectedEquipment.equipmentName : 'Select Equipment'}
                </Text>
              </TouchableOpacity>
            </View>
          </View>
          {renderEquipmentModal()}

          <View style={[styles.formGroup, styles.dropdownContainer]}>
            <View style={styles.assignToContainer}>
              <View style={styles.assignToHeader}>
                <Text style={styles.dropdownLabel}>Assign To</Text>
                <TouchableOpacity
                  style={styles.selectButton}
                  onPress={openAssignToModal}
                >
                  <Text style={styles.selectButtonText}>
                    {assignTo.length > 0 ? `${assignTo.length} Selected` : 'Select'}
                  </Text>
                  <MaterialIcons name="chevron-right" size={width * 0.05} color="#015185" />
                </TouchableOpacity>
              </View>
              {assignTo.length > 0 ? (
                <View style={styles.selectedItemsContainer}>
                  <ScrollView
                    horizontal={true}
                    showsHorizontalScrollIndicator={false}
                    contentContainerStyle={styles.selectedItemsRow}
                  >
                    {assignTo.map((item, index) => (
                      <View key={`${item.id || index}`} style={styles.selectedItem}>
                        <View style={styles.selectedItemContent}>
                          <Text style={styles.selectedItemText}>
                            {item.fullName || item.name || 'Unknown User'}
                          </Text>
                          <Text style={styles.selectedItemSubText}>
                            {item.employeeNumber || 'N/A'} • {item.role || 'Operator'}
                          </Text>
                        </View>
                        <TouchableOpacity
                          style={styles.closeButton}
                          onPress={() => handleRemoveAssignTo(item)}
                        >
                          <View style={styles.closeCircle}>
                            <MaterialIcons name="close" size={width * 0.04} color="#fff" />
                          </View>
                        </TouchableOpacity>
                      </View>
                    ))}
                  </ScrollView>
                </View>
              ) : (
                <View style={styles.noUsersSelected}>
                  <Text style={styles.noUsersSelectedText}>No users selected</Text>
                  <Text style={styles.noUsersSelectedSubText}>Tap "Select" to choose users to assign</Text>
                </View>
              )}
            </View>
          </View>

          <View style={styles.formsubdat}>
            <View style={styles.formGroupdates0}>
              <View style={styles.row}>
                <Text style={styles.label}>Start Date</Text>
                <TouchableOpacity
                  style={styles.dateInputButton}
                  onPress={() => {
                    setDatePickerMode('start');
                    setDatePickerVisible(true);
                    if (Platform.OS === 'ios') {
                      forceModalUpdate();
                    }
                  }}
                >
                  <Text style={styles.dateInputText}>{formatDate(startDate)}</Text>
                  <MaterialIcons name="calendar-today" size={width * 0.045} color="#015185" />
                </TouchableOpacity>
              </View>
            </View>

            <View style={styles.mainspace0}></View>

            <View style={styles.formGroupdates1}>
              <View style={styles.row}>
                <Text style={styles.label}>Due Date</Text>
                <TouchableOpacity
                  style={styles.dateInputButton}
                  onPress={() => {
                    setDatePickerMode('due');
                    setDatePickerVisible(true);
                    if (Platform.OS === 'ios') {
                      forceModalUpdate();
                    }
                  }}
                >
                  <Text style={styles.dateInputText}>{formatDate(dueDate)}</Text>
                  <MaterialIcons name="calendar-today" size={width * 0.045} color="#015185" />
                </TouchableOpacity>
              </View>
            </View>
          </View>

          <View style={[styles.formGroup, styles.dropdownContaineradditional]}>
            <View style={styles.rowCentered}>
              <Text style={styles.label}>Additional Details</Text>
              <TouchableOpacity
                style={styles.selectButton}
                onPress={() => setIsAdditionalDetailsOpen(!isAdditionalDetailsOpen)}
              >
                <MaterialIcons
                  name={isAdditionalDetailsOpen ? 'keyboard-arrow-up' : 'keyboard-arrow-down'}
                  size={width * 0.05}
                  color="#015185"
                />
              </TouchableOpacity>
            </View>
            {isAdditionalDetailsOpen && (
              <View style={styles.additionalDetailsContainer}>
                <View style={styles.formGroup}>
                  <View style={styles.row}>
                    <Text style={styles.labelwot}>Job Card Type</Text>
                    <TouchableOpacity
                      style={styles.selectButton}
                      onPress={() => setWorkOrderTypeModalVisible(true)}
                    >
                      <Text style={styles.selectButtonText}>{workOrderType}</Text>
                      <MaterialIcons name="chevron-right" size={width * 0.05} color="#015185" />
                    </TouchableOpacity>
                  </View>
                  <View style={styles.wotclono}></View>
                </View>

                <View style={styles.formGroup}>
                  <TextInput
                    style={styles.input}
                    placeholder="Enter estimated cost"
                    placeholderTextColor="#888"
                    value={estimatedCost}
                    onChangeText={setEstimatedCost}
                    keyboardType="numeric"
                  />
                </View>

                <View style={styles.formGroup}>
                  <TextInput
                    style={styles.input}
                    placeholder="Enter labor time in hours"
                    placeholderTextColor="#888"
                    value={estimatedLaborTime}
                    onChangeText={setEstimatedLaborTime}
                    keyboardType="numeric"
                  />
                </View>

                <View style={styles.formGroup}>
                  <View style={styles.row}>
                    <Text style={styles.labelwot}>Link Inspection</Text>
                    <TouchableOpacity
                      style={styles.selectButton}
                      onPress={() => setLinkInspectionModalVisible(true)}
                    >
                      <Text style={styles.selectButtonText}>
                        {linkInspection ? linkInspection.name : 'Select'}
                      </Text>
                      <MaterialIcons name="chevron-right" size={width * 0.05} color="#015185" />
                    </TouchableOpacity>
                  </View>
                  {linkInspection && (
                    <View style={styles.linkedInspectionDetails}>
                      <Text style={styles.linkedInspectionText}>
                        Equipment: {linkInspection.equipment?.equipmentName || 'N/A'} | 
                        Date: {linkInspection.date ? new Date(linkInspection.date).toLocaleDateString() : 'N/A'}
                      </Text>
                      
                      {/* Auto-create tasks button */}
                      <TouchableOpacity
                        style={[styles.selectButton, { backgroundColor: '#ffd205', marginTop: 10 }]}
                        onPress={handleExtractTasksFromInspection}
                      >
                        <MaterialIcons name="auto-fix-high" size={width * 0.04} color="#333" />
                        <Text style={[styles.selectButtonText, { color: '#333', marginLeft: 8 }]}>
                          Auto-Create Tasks from Faulty Items
                        </Text>
                      </TouchableOpacity>
                    </View>
                  )}
                  <View style={styles.wotclono}></View>
                </View>

                <View style={styles.formGroup}>
                  <View style={styles.row}>
                    <Text style={styles.labelwot}>Job Card</Text>
                    <TouchableOpacity
                      style={styles.selectButton}
                      onPress={async () => {
                        await fetchFilteredWorkOrders();
                        setLinkedWorkOrderModalVisible(true);
                      }}
                    >
                      <Text style={styles.selectButtonText}>
                        {linkedWorkOrder ? linkedWorkOrder.title : 'Select'}
                      </Text>
                      <MaterialIcons name="chevron-right" size={width * 0.05} color="#015185" />
                    </TouchableOpacity>
                  </View>
                  <View style={styles.wotclono}></View>
                </View>
              </View>
            )}
          </View>

          <View style={[styles.formGroup, styles.dropdownContainerCostd]}>
            <View style={styles.rowCentered}>
              <Text style={styles.label}>Cost Details</Text>
              <TouchableOpacity
                style={styles.selectButton}
                onPress={() => setIsCostDetailsOpen(!isCostDetailsOpen)}
              >
                <MaterialIcons
                  name={isCostDetailsOpen ? 'keyboard-arrow-up' : 'keyboard-arrow-down'}
                  size={width * 0.05}
                  color="#015185"
                />
              </TouchableOpacity>
            </View>
            {isCostDetailsOpen && (
              <View style={styles.additionalDetailsContainer}>
                <View style={styles.formGroup}>
                  <View style={styles.row}>
                    <Text style={styles.labelwot}>Part & Material Cost</Text>
                    <TextInput
                      style={[styles.inputcost0, styles.costInput]}
                      placeholder="0.00"
                      placeholderTextColor="#888"
                      value={partMaterialCost}
                      onChangeText={setPartMaterialCost}
                      keyboardType="numeric"
                    />
                  </View>
                </View>

                <View style={styles.formGroup}>
                  <View style={styles.row}>
                    <Text style={styles.labelwot}>Total Labor Cost</Text>
                    <TextInput
                      style={[styles.inputcost0, styles.costInput]}
                      placeholder="0.00"
                      placeholderTextColor="#888"
                      value={totalLaborCost}
                      onChangeText={setTotalLaborCost}
                      keyboardType="numeric"
                    />
                  </View>
                </View>

                <View style={styles.formGroup}>
                  <View style={styles.row}>
                    <Text style={styles.labelwot}>Additional Cost</Text>
                    <TextInput
                      style={[styles.inputcost0, styles.costInput]}
                      placeholder="0.00"
                      placeholderTextColor="#888"
                      value={additionalCost}
                      onChangeText={setAdditionalCost}
                      keyboardType="numeric"
                    />
                  </View>
                </View>

                <View style={styles.formGroup}>
                  <View style={styles.row}>
                    <Text style={styles.labelwot}>Tax Amount</Text>
                    <View style={styles.taxContainer}>
                      <TextInput
                        style={[styles.inputcost0tax, styles.taxInput]}
                        placeholder="0.00"
                        placeholderTextColor="#888"
                        value={taxAmount}
                        onChangeText={setTaxAmount}
                        keyboardType="numeric"
                      />
                      <Text style={styles.taxLabel}>Tax(%)</Text>
                      <TextInput
                        style={[styles.inputcost0tax, styles.taxInput]}
                        placeholder="0.00"
                        placeholderTextColor="#888"
                        value={taxPercentage}
                        onChangeText={setTaxPercentage}
                        keyboardType="numeric"
                      />
                    </View>
                  </View>
                </View>

                <View style={styles.formGroup}>
                  <View style={styles.row}>
                    <Text style={styles.labelwot}>Total WO Cost</Text>
                    <TextInput
                      style={[styles.input, styles.costInput]}
                      placeholder="0.00"
                      placeholderTextColor="#888"
                      value={totalWoCost}
                      onChangeText={setTotalWoCost}
                      keyboardType="numeric"
                    />
                  </View>
                </View>
              </View>
            )}
          </View>

          <View style={{ marginTop: 16 }}>
            <Text style={styles.sectionHeader}>Tasks & Items</Text>
            
            {/* Debug: Show items that will be saved */}
            {__DEV__ && (
              <View style={{ backgroundColor: '#f0f8ff', padding: 10, marginBottom: 10, borderRadius: 8 }}>
                <Text style={{ color: '#015185', fontWeight: 'bold', fontSize: 14 }}>
                  Debug: Items to be saved ({items.length})
                </Text>
                {items.map((item, index) => (
                  <Text key={index} style={{ fontSize: 11, color: '#666', marginTop: 2 }}>
                    {index + 1}. {item.category}: {item.text} 
                    {item.note?.includes('Auto-generated') ? ' (Auto-created)' : ''}
                  </Text>
                ))}
                {items.length === 0 && (
                  <Text style={{ fontSize: 11, color: '#e74c3c' }}>
                    No items to save
                  </Text>
                )}
              </View>
            )}
            
            <View style={styles.tabRow}>
              {categories.map((cat) => (
                <TouchableOpacity
                  key={cat}
                  style={[styles.tabButton, selectedCategory === cat && styles.tabButtonActive]}
                  onPress={() => setSelectedCategory(cat)}
                >
                  <Text style={[styles.tabButtonText, selectedCategory === cat && styles.tabButtonTextActive]}>{cat}</Text>
                </TouchableOpacity>
              ))}
            </View>
            {/* Render only items for the selected tab/category */}
            <View style={{ marginTop: 10 }}>
              {selectedCategory === 'Task' && (
                <FlatList
                  data={items.filter(item => item.category === 'Task')}
                  keyExtractor={item => item.id}
                  renderItem={renderTaskItem}
                  ListEmptyComponent={<Text style={styles.emptyText}>No tasks added.</Text>}
                />
              )}
              {selectedCategory === 'Parts' && (
                <FlatList
                  data={items.filter(item => item.category === 'Parts')}
                  keyExtractor={item => item.id}
                  renderItem={({ item }) => (
                    <View style={[styles.taskCard, { flexDirection: 'row', alignItems: 'center', position: 'relative' }]}> 
                      <View style={{ flex: 1 }}>
                        <Text style={styles.taskTitle}>{item.itemName}</Text>
                        <Text style={styles.taskNote}>Qty: {item.quantity} {item.unit}</Text>
                        <Text style={styles.taskNote}>Cost: {item.itemCost}</Text>
                      </View>
                      {/* Status button */}
                      <TouchableOpacity
                        style={{
                          paddingVertical: 4,
                          paddingHorizontal: 10,
                          borderRadius: 12,
                          backgroundColor: (statuses.find(opt => opt.label === item.status) || statuses[0]).color,
                          marginRight: 4,
                        }}
                        onPress={event => {
                          setPartsStatusMenuVisibleId(partsStatusMenuVisibleId === item.id ? null : item.id);
                          if (event && event.nativeEvent) {
                            setPartsStatusDropdownAnchor({ x: event.nativeEvent.pageX, y: event.nativeEvent.pageY });
                          }
                        }}
                      >
                        <Text style={{ color: '#fff', fontWeight: 'bold', fontSize: 13 }}>{item.status}</Text>
                      </TouchableOpacity>
                      {/* Status dropdown as Modal */}
                      {partsStatusMenuVisibleId === item.id && partsStatusDropdownAnchor && (
                        <Modal
                          visible={true}
                          transparent
                          animationType="fade"
                          onRequestClose={() => setPartsStatusMenuVisibleId(null)}
                        >
                          <TouchableOpacity
                            style={{ flex: 1 }}
                            activeOpacity={1}
                            onPress={() => setPartsStatusMenuVisibleId(null)}
                          >
                            <View
                              style={{
                                position: 'absolute',
                                left: partsStatusDropdownAnchor.x - 120,
                                top: partsStatusDropdownAnchor.y - 160,
                                backgroundColor: '#fff',
                                borderRadius: 8,
                                elevation: 5,
                                zIndex: 200,
                                shadowColor: '#000',
                                shadowOpacity: 0.1,
                                shadowRadius: 4,
                                minWidth: 120,
                              }}
                            >
                              {statuses.map(opt => (
                                <TouchableOpacity
                                  key={opt.label}
                                  style={{ flexDirection: 'row', alignItems: 'center', padding: 10 }}
                                  onPress={() => {
                                    updateItemStatus(item.id, opt.label);
                                    setPartsStatusMenuVisibleId(null);
                                  }}
                                >
                                  <View style={{ width: 16, height: 16, borderRadius: 8, backgroundColor: opt.color, marginRight: 8, borderWidth: item.status === opt.label ? 2 : 0, borderColor: '#333' }} />
                                  <Text style={{ color: opt.color, fontWeight: item.status === opt.label ? 'bold' : 'normal' }}>{opt.label}</Text>
                                </TouchableOpacity>
                              ))}
                            </View>
                          </TouchableOpacity>
                        </Modal>
                      )}
                      {/* Three-dot menu button and modal (existing code) */}
                      <TouchableOpacity
                        style={styles.threeDotsButton}
                        onPress={event => {
                          setPartsMenuVisibleId(partsMenuVisibleId === item.id ? null : item.id);
                          if (event && event.nativeEvent) {
                            setPartsDropdownAnchor({ x: event.nativeEvent.pageX, y: event.nativeEvent.pageY });
                          }
                        }}
                      >
                        <MaterialIcons name="more-vert" size={24} color="#015185" />
                      </TouchableOpacity>
                      {partsMenuVisibleId === item.id && partsDropdownAnchor && (
                        <Modal
                          visible={true}
                          transparent
                          animationType="fade"
                          onRequestClose={() => setPartsMenuVisibleId(null)}
                        >
                          <TouchableOpacity
                            style={{ flex: 1 }}
                            activeOpacity={1}
                            onPress={() => setPartsMenuVisibleId(null)}
                          >
                            <View
                              style={{
                                position: 'absolute',
                                left: partsDropdownAnchor.x - 120,
                                top: partsDropdownAnchor.y - 90,
                                backgroundColor: '#fff',
                                borderRadius: 8,
                                elevation: 5,
                                zIndex: 200,
                                shadowColor: '#000',
                                shadowOpacity: 0.1,
                                shadowRadius: 4,
                                minWidth: 120,
                              }}
                            >
                              <TouchableOpacity
                                style={{ padding: 12 }}
                                onPress={() => {
                                  setPartsMenuVisibleId(null);
                                  setEditPartsId(item.id);
                                  setPartsItemNumber(item.itemNumber || '');
                                  setPartsPartNumber(item.partNumber || '');
                                  setPartsItemName(item.itemName || '');
                                  setPartsQuantity(item.quantity || '');
                                  setPartsUnit(item.unit || 'Piece');
                                  setPartsItemCost(item.itemCost || '');
                                  setPartsSite(item.site || '');
                                  setPartsNote(item.note || '');
                                  setAddPartsModalVisible(true);
                                }}
                              >
                                <Text style={{ color: '#015185', fontWeight: 'bold' }}>Edit</Text>
                              </TouchableOpacity>
                              <TouchableOpacity
                                style={{ padding: 12 }}
                                onPress={() => {
                                  setPartsMenuVisibleId(null);
                                  setItems(items.filter(i => i.id !== item.id));
                                }}
                              >
                                <Text style={{ color: '#dc3545', fontWeight: 'bold' }}>Delete</Text>
                              </TouchableOpacity>
                            </View>
                          </TouchableOpacity>
                        </Modal>
                      )}
                    </View>
                  )}
                  ListEmptyComponent={<Text style={styles.emptyText}>No parts added.</Text>}
                />
              )}
              {selectedCategory === 'Labor' && (
                <FlatList
                  data={items.filter(item => item.category === 'Labor')}
                  keyExtractor={item => item.id}
                  renderItem={({ item }) => (
                    <View style={[styles.taskCard, { flexDirection: 'row', alignItems: 'center', position: 'relative' }]}> 
                      <View style={{ flex: 1 }}>
                        <Text style={styles.taskTitle}>{item.worker}</Text>
                        <Text style={styles.taskNote}>Hours: {item.hours}h {item.minutes}m</Text>
                        <Text style={styles.taskNote}>Rate: {item.rate}</Text>
                      </View>
                      {/* Status button */}
                      <TouchableOpacity
                        style={{
                          paddingVertical: 4,
                          paddingHorizontal: 10,
                          borderRadius: 12,
                          backgroundColor: (statuses.find(opt => opt.label === item.status) || statuses[0]).color,
                          marginRight: 4,
                        }}
                        onPress={event => {
                          setLaborStatusMenuVisibleId(laborStatusMenuVisibleId === item.id ? null : item.id);
                          if (event && event.nativeEvent) {
                            setLaborStatusDropdownAnchor({ x: event.nativeEvent.pageX, y: event.nativeEvent.pageY });
                          }
                        }}
                      >
                        <Text style={{ color: '#fff', fontWeight: 'bold', fontSize: 13 }}>{item.status}</Text>
                      </TouchableOpacity>
                      {/* Status dropdown as Modal */}
                      {laborStatusMenuVisibleId === item.id && laborStatusDropdownAnchor && (
                        <Modal
                          visible={true}
                          transparent
                          animationType="fade"
                          onRequestClose={() => setLaborStatusMenuVisibleId(null)}
                        >
                          <TouchableOpacity
                            style={{ flex: 1 }}
                            activeOpacity={1}
                            onPress={() => setLaborStatusMenuVisibleId(null)}
                          >
                            <View
                              style={{
                                position: 'absolute',
                                left: laborStatusDropdownAnchor.x - 120,
                                top: laborStatusDropdownAnchor.y - 160,
                                backgroundColor: '#fff',
                                borderRadius: 8,
                                elevation: 5,
                                zIndex: 200,
                                shadowColor: '#000',
                                shadowOpacity: 0.1,
                                shadowRadius: 4,
                                minWidth: 120,
                              }}
                            >
                              {statuses.map(opt => (
                                <TouchableOpacity
                                  key={opt.label}
                                  style={{ flexDirection: 'row', alignItems: 'center', padding: 10 }}
                                  onPress={() => {
                                    updateItemStatus(item.id, opt.label);
                                    setLaborStatusMenuVisibleId(null);
                                  }}
                                >
                                  <View style={{ width: 16, height: 16, borderRadius: 8, backgroundColor: opt.color, marginRight: 8, borderWidth: item.status === opt.label ? 2 : 0, borderColor: '#333' }} />
                                  <Text style={{ color: opt.color, fontWeight: item.status === opt.label ? 'bold' : 'normal' }}>{opt.label}</Text>
                                </TouchableOpacity>
                              ))}
                            </View>
                          </TouchableOpacity>
                        </Modal>
                      )}
                      {/* Three-dot menu button and modal (existing code) */}
                      <TouchableOpacity
                        style={styles.threeDotsButton}
                        onPress={event => {
                          setLaborMenuVisibleId(laborMenuVisibleId === item.id ? null : item.id);
                          if (event && event.nativeEvent) {
                            setLaborDropdownAnchor({ x: event.nativeEvent.pageX, y: event.nativeEvent.pageY });
                          }
                        }}
                      >
                        <MaterialIcons name="more-vert" size={24} color="#015185" />
                      </TouchableOpacity>
                      {laborMenuVisibleId === item.id && laborDropdownAnchor && (
                        <Modal
                          visible={true}
                          transparent
                          animationType="fade"
                          onRequestClose={() => setLaborMenuVisibleId(null)}
                        >
                          <TouchableOpacity
                            style={{ flex: 1 }}
                            activeOpacity={1}
                            onPress={() => setLaborMenuVisibleId(null)}
                          >
                            <View
                              style={{
                                position: 'absolute',
                                left: laborDropdownAnchor.x - 120,
                                top: laborDropdownAnchor.y - 90,
                                backgroundColor: '#fff',
                                borderRadius: 8,
                                elevation: 5,
                                zIndex: 200,
                                shadowColor: '#000',
                                shadowOpacity: 0.1,
                                shadowRadius: 4,
                                minWidth: 120,
                              }}
                            >
                              <TouchableOpacity
                                style={{ padding: 12 }}
                                onPress={() => {
                                  setLaborMenuVisibleId(null);
                                  setEditLaborId(item.id);
                                  setLaborWorker(item.worker || '');
                                  setLaborCode(item.code || '');
                                  setLaborRate(item.rate || '');
                                  setLaborHours(item.hours || '');
                                  setLaborMinutes(item.minutes || '');
                                  setLaborNote(item.note || '');
                                  setAddLaborModalVisible(true);
                                }}
                              >
                                <Text style={{ color: '#015185', fontWeight: 'bold' }}>Edit</Text>
                              </TouchableOpacity>
                              <TouchableOpacity
                                style={{ padding: 12 }}
                                onPress={() => {
                                  setLaborMenuVisibleId(null);
                                  setItems(items.filter(i => i.id !== item.id));
                                }}
                              >
                                <Text style={{ color: '#dc3545', fontWeight: 'bold' }}>Delete</Text>
                              </TouchableOpacity>
                            </View>
                          </TouchableOpacity>
                        </Modal>
                      )}
                    </View>
                  )}
                  ListEmptyComponent={<Text style={styles.emptyText}>No labor added.</Text>}
                />
              )}
              {selectedCategory === 'Additional Cost' && (
                <FlatList
                  data={items.filter(item => item.category === 'Additional Cost')}
                  keyExtractor={item => item.id}
                  renderItem={({ item }) => (
                    <View style={[styles.taskCard, { flexDirection: 'row', alignItems: 'center', position: 'relative' }]}> 
                      <View style={{ flex: 1 }}>
                        <Text style={styles.taskTitle}>{item.description}</Text>
                        <Text style={styles.taskNote}>Amount: {item.amount}</Text>
                      </View>
                      {/* Status button */}
                      <TouchableOpacity
                        style={{
                          paddingVertical: 4,
                          paddingHorizontal: 10,
                          borderRadius: 12,
                          backgroundColor: (statuses.find(opt => opt.label === item.status) || statuses[0]).color,
                          marginRight: 4,
                        }}
                        onPress={event => {
                          setAdditionalCostStatusMenuVisibleId(additionalCostStatusMenuVisibleId === item.id ? null : item.id);
                          if (event && event.nativeEvent) {
                            setAdditionalCostStatusDropdownAnchor({ x: event.nativeEvent.pageX, y: event.nativeEvent.pageY });
                          }
                        }}
                      >
                        <Text style={{ color: '#fff', fontWeight: 'bold', fontSize: 13 }}>{item.status}</Text>
                      </TouchableOpacity>
                      {/* Status dropdown as Modal */}
                      {additionalCostStatusMenuVisibleId === item.id && additionalCostStatusDropdownAnchor && (
                        <Modal
                          visible={true}
                          transparent
                          animationType="fade"
                          onRequestClose={() => setAdditionalCostStatusMenuVisibleId(null)}
                        >
                          <TouchableOpacity
                            style={{ flex: 1 }}
                            activeOpacity={1}
                            onPress={() => setAdditionalCostStatusMenuVisibleId(null)}
                          >
                            <View
                              style={{
                                position: 'absolute',
                                left: additionalCostStatusDropdownAnchor.x - 120,
                                top: additionalCostStatusDropdownAnchor.y - 160,
                                backgroundColor: '#fff',
                                borderRadius: 8,
                                elevation: 5,
                                zIndex: 200,
                                shadowColor: '#000',
                                shadowOpacity: 0.1,
                                shadowRadius: 4,
                                minWidth: 120,
                              }}
                            >
                              {statuses.map(opt => (
                                <TouchableOpacity
                                  key={opt.label}
                                  style={{ flexDirection: 'row', alignItems: 'center', padding: 10 }}
                                  onPress={() => {
                                    updateItemStatus(item.id, opt.label);
                                    setAdditionalCostStatusMenuVisibleId(null);
                                  }}
                                >
                                  <View style={{ width: 16, height: 16, borderRadius: 8, backgroundColor: opt.color, marginRight: 8, borderWidth: item.status === opt.label ? 2 : 0, borderColor: '#333' }} />
                                  <Text style={{ color: opt.color, fontWeight: item.status === opt.label ? 'bold' : 'normal' }}>{opt.label}</Text>
                                </TouchableOpacity>
                              ))}
                            </View>
                          </TouchableOpacity>
                        </Modal>
                      )}
                      {/* Three-dot menu button and modal (existing code) */}
                      <TouchableOpacity
                        style={styles.threeDotsButton}
                        onPress={event => {
                          setAdditionalCostMenuVisibleId(additionalCostMenuVisibleId === item.id ? null : item.id);
                          if (event && event.nativeEvent) {
                            setAdditionalCostDropdownAnchor({ x: event.nativeEvent.pageX, y: event.nativeEvent.pageY });
                          }
                        }}
                      >
                        <MaterialIcons name="more-vert" size={24} color="#015185" />
                      </TouchableOpacity>
                      {additionalCostMenuVisibleId === item.id && additionalCostDropdownAnchor && (
                        <Modal
                          visible={true}
                          transparent
                          animationType="fade"
                          onRequestClose={() => setAdditionalCostMenuVisibleId(null)}
                        >
                          <TouchableOpacity
                            style={{ flex: 1 }}
                            activeOpacity={1}
                            onPress={() => setAdditionalCostMenuVisibleId(null)}
                          >
                            <View
                              style={{
                                position: 'absolute',
                                left: additionalCostDropdownAnchor.x - 120,
                                top: additionalCostDropdownAnchor.y - 90,
                                backgroundColor: '#fff',
                                borderRadius: 8,
                                elevation: 5,
                                zIndex: 200,
                                shadowColor: '#000',
                                shadowOpacity: 0.1,
                                shadowRadius: 4,
                                minWidth: 120,
                              }}
                            >
                              <TouchableOpacity
                                style={{ padding: 12 }}
                                onPress={() => {
                                  setAdditionalCostMenuVisibleId(null);
                                  setEditAdditionalCostId(item.id);
                                  setAdditionalCostDescription(item.description || '');
                                  setAdditionalCostAmount(item.amount || '');
                                  setAddAdditionalCostModalVisible(true);
                                }}
                              >
                                <Text style={{ color: '#015185', fontWeight: 'bold' }}>Edit</Text>
                              </TouchableOpacity>
                              <TouchableOpacity
                                style={{ padding: 12 }}
                                onPress={() => {
                                  setAdditionalCostMenuVisibleId(null);
                                  setItems(items.filter(i => i.id !== item.id));
                                }}
                              >
                                <Text style={{ color: '#dc3545', fontWeight: 'bold' }}>Delete</Text>
                              </TouchableOpacity>
                            </View>
                          </TouchableOpacity>
                        </Modal>
                      )}
                    </View>
                  )}
                  ListEmptyComponent={<Text style={styles.emptyText}>No additional cost added.</Text>}
                />
              )}
              {selectedCategory === 'Remarks' && (
                <FlatList
                  data={items.filter(item => item.category === 'Remarks')}
                  keyExtractor={item => item.id}
                  renderItem={({ item }) => (
                    <View style={[styles.taskCard, { flexDirection: 'row', alignItems: 'center', position: 'relative' }]}> 
                      <View style={{ flex: 1 }}>
                        <Text style={styles.taskTitle}>{item.description}</Text>
                      </View>
                      {/* Status button */}
                      <TouchableOpacity
                        style={{
                          paddingVertical: 4,
                          paddingHorizontal: 10,
                          borderRadius: 12,
                          backgroundColor: (statuses.find(opt => opt.label === item.status) || statuses[0]).color,
                          marginRight: 4,
                        }}
                        onPress={event => {
                          setRemarksStatusMenuVisibleId(remarksStatusMenuVisibleId === item.id ? null : item.id);
                          if (event && event.nativeEvent) {
                            setRemarksStatusDropdownAnchor({ x: event.nativeEvent.pageX, y: event.nativeEvent.pageY });
                          }
                        }}
                      >
                        <Text style={{ color: '#fff', fontWeight: 'bold', fontSize: 13 }}>{item.status}</Text>
                      </TouchableOpacity>
                      {/* Status dropdown as Modal */}
                      {remarksStatusMenuVisibleId === item.id && remarksStatusDropdownAnchor && (
                        <Modal
                          visible={true}
                          transparent
                          animationType="fade"
                          onRequestClose={() => setRemarksStatusMenuVisibleId(null)}
                        >
                          <TouchableOpacity
                            style={{ flex: 1 }}
                            activeOpacity={1}
                            onPress={() => setRemarksStatusMenuVisibleId(null)}
                          >
                            <View
                              style={{
                                position: 'absolute',
                                left: remarksStatusDropdownAnchor.x - 120,
                                top: remarksStatusDropdownAnchor.y - 160,
                                backgroundColor: '#fff',
                                borderRadius: 8,
                                elevation: 5,
                                zIndex: 200,
                                shadowColor: '#000',
                                shadowOpacity: 0.1,
                                shadowRadius: 4,
                                minWidth: 120,
                              }}
                            >
                              {statuses.map(opt => (
                                <TouchableOpacity
                                  key={opt.label}
                                  style={{ flexDirection: 'row', alignItems: 'center', padding: 10 }}
                                  onPress={() => {
                                    updateItemStatus(item.id, opt.label);
                                    setRemarksStatusMenuVisibleId(null);
                                  }}
                                >
                                  <View style={{ width: 16, height: 16, borderRadius: 8, backgroundColor: opt.color, marginRight: 8, borderWidth: item.status === opt.label ? 2 : 0, borderColor: '#333' }} />
                                  <Text style={{ color: opt.color, fontWeight: item.status === opt.label ? 'bold' : 'normal' }}>{opt.label}</Text>
                                </TouchableOpacity>
                              ))}
                            </View>
                          </TouchableOpacity>
                        </Modal>
                      )}
                      {/* Three-dot menu button and modal (existing code) */}
                      <TouchableOpacity
                        style={styles.threeDotsButton}
                        onPress={event => {
                          setRemarksMenuVisibleId(remarksMenuVisibleId === item.id ? null : item.id);
                          if (event && event.nativeEvent) {
                            setRemarksDropdownAnchor({ x: event.nativeEvent.pageX, y: event.nativeEvent.pageY });
                          }
                        }}
                      >
                        <MaterialIcons name="more-vert" size={24} color="#015185" />
                      </TouchableOpacity>
                      {remarksMenuVisibleId === item.id && remarksDropdownAnchor && (
                        <Modal
                          visible={true}
                          transparent
                          animationType="fade"
                          onRequestClose={() => setRemarksMenuVisibleId(null)}
                        >
                          <TouchableOpacity
                            style={{ flex: 1 }}
                            activeOpacity={1}
                            onPress={() => setRemarksMenuVisibleId(null)}
                          >
                            <View
                              style={{
                                position: 'absolute',
                                left: remarksDropdownAnchor.x - 120,
                                top: remarksDropdownAnchor.y - 90,
                                backgroundColor: '#fff',
                                borderRadius: 8,
                                elevation: 5,
                                zIndex: 200,
                                shadowColor: '#000',
                                shadowOpacity: 0.1,
                                shadowRadius: 4,
                                minWidth: 120,
                              }}
                            >
                              <TouchableOpacity
                                style={{ padding: 12 }}
                                onPress={() => {
                                  setRemarksMenuVisibleId(null);
                                  setEditRemarksId(item.id);
                                  setRemarksDescription(item.description || '');
                                  setAddRemarksModalVisible(true);
                                }}
                              >
                                <Text style={{ color: '#015185', fontWeight: 'bold' }}>Edit</Text>
                              </TouchableOpacity>
                              <TouchableOpacity
                                style={{ padding: 12 }}
                                onPress={() => {
                                  setRemarksMenuVisibleId(null);
                                  setItems(items.filter(i => i.id !== item.id));
                                }}
                              >
                                <Text style={{ color: '#dc3545', fontWeight: 'bold' }}>Delete</Text>
                              </TouchableOpacity>
                            </View>
                          </TouchableOpacity>
                        </Modal>
                      )}
                    </View>
                  )}
                  ListEmptyComponent={<Text style={styles.emptyText}>No remarks added.</Text>}
                />
              )}
            </View>
            {/* Add button for the selected tab/category if needed */}
          </View>

          <TouchableOpacity
            style={styles.addTaskButton}
            onPress={() => {
              console.log('Add button pressed for category:', selectedCategory);
              console.log('Current modal states:', {
                addItemModalVisible,
                addLaborModalVisible,
                addPartsModalVisible,
                addAdditionalCostModalVisible,
                addRemarksModalVisible
              });
              
              // Force modal update for iOS
              if (Platform.OS === 'ios') {
                forceModalUpdate();
              }
              
              // Small delay for iOS to prevent freezing
              setTimeout(() => {
                console.log('Opening modal after delay');
                if (selectedCategory === 'Task') {
                  console.log('Setting addItemModalVisible to true');
                  setAddItemModalVisible(true);
                  if (Platform.OS === 'ios') {
                    setIosModalVisible(true);
                  }
                }
                else if (selectedCategory === 'Labor') setAddLaborModalVisible(true);
                else if (selectedCategory === 'Parts') setAddPartsModalVisible(true);
                else if (selectedCategory === 'Additional Cost') setAddAdditionalCostModalVisible(true);
                else if (selectedCategory === 'Remarks') setAddRemarksModalVisible(true);
              }, Platform.OS === 'ios' ? 100 : 0);
            }}
          >
            <Text style={styles.addTaskText}>+ Add {selectedCategory}</Text>
          </TouchableOpacity>
          
         
        </ScrollView>
      </KeyboardAvoidingView>

      {/* iOS Date Picker */}
      {datePickerVisible && Platform.OS === 'ios' && (
        <Modal
          key={modalKey}
          visible={datePickerVisible}
          transparent={true}
          animationType="slide"
          onRequestClose={() => setDatePickerVisible(false)}
        >
          <View style={styles.datePickerModalOverlay}>
            <View style={styles.datePickerModalContent}>
              <View style={styles.datePickerHeader}>
                <TouchableOpacity onPress={() => setDatePickerVisible(false)}>
                  <Text style={styles.datePickerCancelText}>Cancel</Text>
                </TouchableOpacity>
                <Text style={styles.datePickerTitle}>
                  {datePickerMode === 'start' ? 'Select Start Date' : 'Select Due Date'}
                </Text>
                <TouchableOpacity onPress={() => setDatePickerVisible(false)}>
                  <Text style={styles.datePickerDoneText}>Done</Text>
                </TouchableOpacity>
              </View>
              <View style={styles.datePickerContainer}>
                <DateTimePicker
                  value={datePickerMode === 'start' ? (startDate || new Date()) : (dueDate || new Date())}
                  mode="date"
                  display="spinner"
                  onChange={(event, selectedDate) => {
                    if (event.type === 'set' && selectedDate) {
                      if (datePickerMode === 'start') setStartDate(selectedDate);
                      else setDueDate(selectedDate);
                    }
                  }}
                  style={{ width: 300, height: 200 }}
                  textColor="#015185"
                />
              </View>
            </View>
          </View>
        </Modal>
      )}

      {/* Android Date Picker */}
      {datePickerVisible && Platform.OS === 'android' && (
        <DateTimePicker
          value={datePickerMode === 'start' ? (startDate || new Date()) : (dueDate || new Date())}
          mode="date"
          display="calendar"
          onChange={(event, selectedDate) => {
            setDatePickerVisible(false);
            if (event.type === 'set' && selectedDate) {
              if (datePickerMode === 'start') setStartDate(selectedDate);
              else setDueDate(selectedDate);
            }
          }}
        />
      )}



      <Modal
        visible={workOrderTypeModalVisible}
        transparent={true}
        animationType="slide"
        onRequestClose={() => setWorkOrderTypeModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>Select Job Card Type</Text>
            <FlatList
              data={workOrderTypes}
              keyExtractor={item => item.id}
              renderItem={({ item }) => (
                <TouchableOpacity
                  style={styles.modalOption}
                  onPress={() => handleWorkOrderTypeSelect(item)}
                >
                  <Text style={styles.modalOptionText}>{item.name}</Text>
                </TouchableOpacity>
              )}
            />
            <TouchableOpacity
              style={styles.modalCloseButton}
              onPress={() => setWorkOrderTypeModalVisible(false)}
            >
              <Text style={styles.modalCloseText}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      <Modal
        visible={linkInspectionModalVisible}
        transparent
        animationType="slide"
        onRequestClose={() => setLinkInspectionModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={[styles.modalContent, styles.inspectionModal]}>
            <Text style={styles.modalTitle}>Select Inspections</Text>
            <FlatList
              data={inspections}
              keyExtractor={(item) => item.id}
              renderItem={({ item }) => {
                // Calculate faulty count if available
                let faultyCount = 0;
                if (item.step3 && typeof item.step3.faultyItems === 'number') {
                  faultyCount = item.step3.faultyItems;
                } else if (item.step2 && Array.isArray(item.step2.sections) && item.step2.selections) {
                  faultyCount = item.step2.sections.flatMap(section =>
                    (section.subItems || []).filter(subItem =>
                      ['Faulty', 'Not Ok', 'Reject'].includes(item.step2.selections[subItem.key])
                    )
                  ).length;
                }
                const existingJobCard = workOrders.find(
                  wo => wo.linkInspection && wo.linkInspection.id === item.id
                );
                return (
                  <TouchableOpacity
                    style={styles.inspectionOption}
                    onPress={() => {
                      setLinkInspection(item);
                      handleInspectionSelected(item); // Auto-create tasks from faulty items
                      setLinkInspectionModalVisible(false); // Close modal after selection
                    }}
                  >
                    <View style={styles.inspectionInfo}>
                      <View style={styles.inspectionRow}>
                        <Text style={styles.inspectionLabel}>Request #</Text>
                        <Text style={styles.inspectionValue}>{item.name}</Text>
                      </View>
                      <View style={styles.inspectionRow}>
                        <Text style={styles.inspectionLabel}>Equipment</Text>
                        <Text style={styles.inspectionValue}>{item.equipmentName || (item.step1?.selectedEquipment?.equipmentName ?? 'N/A')}</Text>
                      </View>
                      <View style={styles.inspectionRow}>
                        <Text style={styles.inspectionLabel}>Date</Text>
                        <Text style={styles.inspectionValue}>
                          {item.createdAt && item.createdAt !== 'N/A'
                            ? new Date(item.createdAt).toLocaleDateString('en-US', {
                                year: 'numeric',
                                month: 'short',
                                day: 'numeric',
                              })
                            : 'N/A'}
                        </Text>
                      </View>
                      <View style={styles.inspectionRow}>
                        <Text style={styles.inspectionLabel}>Faulty</Text>
                        <Text style={[styles.inspectionValue, { color: faultyCount > 0 ? '#e74c3c' : '#2ecc40', fontWeight: 'bold' }]}>{faultyCount}</Text>
                      </View>
                      {/* Show red warning if job card exists */}
                      {existingJobCard && (
                        <Text style={{ color: 'red', fontSize: 13, marginTop: 2 }}>Job Card already created!</Text>
                      )}
                    </View>
                    {linkInspection && linkInspection.id === item.id && (
                      <View style={styles.checkmarkContainer}>
                        <MaterialIcons name="check-circle" size={24} color="#28a745" />
                      </View>
                    )}
                  </TouchableOpacity>
                );
              }}
              ListEmptyComponent={<Text style={styles.defaultText}>No inspections found</Text>}
              style={styles.inspectionList}
            />
            <View style={styles.modalButtonContainer}>
              <TouchableOpacity
                style={[styles.modalButton, styles.cancelButton]}
                onPress={() => setLinkInspectionModalVisible(false)}
              >
                <Text style={styles.modalButtonText}>Close</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.modalButton, styles.saveButton]}
                onPress={handleSaveInspections}
              >
                <Text style={styles.saveButtonText}>Save</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      <Modal
        visible={linkedWorkOrderModalVisible}
        transparent={true}
        animationType="slide"
        onRequestClose={() => setLinkedWorkOrderModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={[styles.modalContent, { maxHeight: height * 0.8 }]}>
            <Text style={styles.modalTitle}>Link Job Card</Text>
            
            {/* Search Input */}
            <View style={styles.searchContainer}>
              <TextInput
                style={styles.searchInput}
                placeholder="Search job cards..."
                value={linkedWorkOrderSearchQuery}
                onChangeText={setLinkedWorkOrderSearchQuery}
                placeholderTextColor="#666"
              />
              <MaterialIcons name="search" size={20} color="#666" style={styles.searchIcon} />
            </View>
            
            {/* Job Cards List */}
            <FlatList
              data={workOrders.filter(wo => {
                const matchesSearch = !linkedWorkOrderSearchQuery || 
                  wo.title?.toLowerCase().includes(linkedWorkOrderSearchQuery.toLowerCase()) ||
                  wo.requestNumber?.toLowerCase().includes(linkedWorkOrderSearchQuery.toLowerCase()) ||
                  wo.equipment?.equipmentName?.toLowerCase().includes(linkedWorkOrderSearchQuery.toLowerCase()) ||
                  wo.country?.toLowerCase().includes(linkedWorkOrderSearchQuery.toLowerCase()) ||
                  wo.project?.toLowerCase().includes(linkedWorkOrderSearchQuery.toLowerCase());
                
                return matchesSearch;
              })}
              keyExtractor={item => item.id}
              renderItem={({ item }) => (
                <TouchableOpacity
                  style={[styles.modalOption, linkedWorkOrder && linkedWorkOrder.id === item.id && styles.selectedOption]}
                  onPress={() => handleWorkOrderSelect(item)}
                >
                  <View style={styles.jobCardInfo}>
                    <View style={styles.jobCardHeader}>
                      <Text style={styles.modalOptionText}>{item.title || 'Untitled Job Card'}</Text>
                      <Text style={styles.jobCardNumber}>#{item.requestNumber || 'N/A'}</Text>
                    </View>
                    
                    <View style={styles.jobCardDetails}>
                      <Text style={styles.jobCardDetail}>
                        📍 Equipment: {item.equipment?.equipmentName || 'N/A'}
                      </Text>
                      <Text style={styles.jobCardDetail}>
                        🏢 Location: {item.country || 'N/A'} - {item.project || 'N/A'}
                      </Text>
                      <Text style={styles.jobCardDetail}>
                        📅 Created: {item.createdAt ? new Date(item.createdAt).toLocaleDateString() : 'N/A'}
                      </Text>
                      <Text style={[styles.jobCardDetail, { color: getStatusColor(item.status) }]}>
                        🔄 Status: {item.status || 'Unknown'}
                      </Text>
                    </View>
                  </View>
                  
                  {linkedWorkOrder && linkedWorkOrder.id === item.id && (
                    <View style={styles.checkmarkContainer}>
                      <MaterialIcons name="check-circle" size={24} color="#28a745" />
                    </View>
                  )}
                </TouchableOpacity>
              )}
              ListEmptyComponent={
                <View style={styles.emptyContainer}>
                  <MaterialIcons name="work" size={48} color="#ccc" />
                  <Text style={styles.emptyText}>No job cards found</Text>
                  <Text style={styles.emptySubtext}>
                    {linkedWorkOrderSearchQuery ? 'Try adjusting your search' : 'Create a job card first'}
                  </Text>
                </View>
              }
              style={styles.jobCardList}
              showsVerticalScrollIndicator={true}
            />
            
            {/* Action Buttons */}
            <View style={styles.modalButtonContainer}>
              <TouchableOpacity
                style={[styles.modalButton, styles.cancelButton]}
                onPress={() => setLinkedWorkOrderModalVisible(false)}
              >
                <Text style={styles.modalButtonText}>Close</Text>
              </TouchableOpacity>
              
              {linkedWorkOrder && (
                <TouchableOpacity
                  style={[styles.modalButton, styles.clearButton]}
                  onPress={() => {
                    setLinkedWorkOrder(null);
                    setLinkedWorkOrderModalVisible(false);
                  }}
                >
                  <Text style={styles.modalButtonText}>Clear Selection</Text>
                </TouchableOpacity>
              )}
            </View>
          </View>
        </View>
      </Modal>

      {/* Add Task Modal */}
      {console.log('Rendering Add Task Modal, visible:', addItemModalVisible, 'modalKey:', modalKey, 'iosModalVisible:', iosModalVisible)}
      <Modal
        key={`addItem-${modalKey}-${addItemModalVisible}-${iosModalVisible}`}
        visible={Platform.OS === 'ios' ? iosModalVisible : addItemModalVisible}
        transparent={true}
        animationType="slide"
        onRequestClose={() => {
          console.log('Modal onRequestClose called');
          setAddItemModalVisible(false);
          setIosModalVisible(false);
        }}
        onShow={() => {
          console.log('Add Item Modal is now visible');
        }}
        onDismiss={() => {
          console.log('Modal dismissed');
        }}
        statusBarTranslucent={true}
      >
        <View style={[styles.taskModalOverlayCustom, { backgroundColor: 'rgba(0,0,0,0.5)' }]}>
          <View style={[styles.taskModalContentCustom, { backgroundColor: '#fff' }]}>
            <Text style={styles.taskModalTitleCustom}>Add Task</Text>
            <View style={styles.taskModalRowCustom}>
              <Text style={styles.taskModalLabelCustom}>Task</Text>
              <TextInput
                style={styles.taskModalInputCustom}
                
                placeholderTextColor="#888"
                value={newItemText}
                onChangeText={setNewItemText}
              />
            </View>
            <View style={styles.taskModalDividerCustom} />
            <View style={styles.taskModalRowCustom}>
              <Text style={styles.taskModalLabelCustom}>Type</Text>
              <View style={styles.taskModalTypeButtonRowCustom}>
                {taskTypes.map(type => (
                  <TouchableOpacity
                    key={type}
                    style={[styles.taskModalTypeButtonCustom, taskType === type && styles.taskModalTypeButtonSelectedCustom]}
                    onPress={() => setTaskType(type)}
                  >
                    <Text style={[styles.taskModalTypeButtonTextCustom, taskType === type && styles.taskModalTypeButtonTextSelectedCustom]}>{type}</Text>
                  </TouchableOpacity>
                ))}
              </View>
            </View>
            <View style={styles.taskModalDividerCustom} />
            <View style={styles.taskModalRowCustom}>
              <Text style={styles.taskModalLabelCustom}>Assign to</Text>
              <TouchableOpacity
                style={styles.taskModalAssignButtonCustom}
                onPress={openTaskAssignToModal}
              >
                <Text style={styles.taskModalAssignButtonTextCustom}>
                  {tempAssignTo.length > 0 ? `${tempAssignTo.length} Assigned` : 'Set'}
                </Text>
                <MaterialIcons name="chevron-right" size={width * 0.05} color="#015185" />
              </TouchableOpacity>
            </View>
            
            {/* Selected Users Display */}
            {tempAssignTo.length > 0 && (
              <View style={styles.taskModalSelectedUsersContainer}>
                <Text style={styles.taskModalSelectedUsersTitle}>Selected Users:</Text>
                {tempAssignTo.map((user, index) => (
                  <View key={user.id} style={styles.taskModalSelectedUserItem}>
                    <View style={styles.taskModalSelectedUserInfo}>
                      <Text style={styles.taskModalSelectedUserName}>
                        {user.fullName || user.name || 'Unknown User'}
                      </Text>
                      <Text style={styles.taskModalSelectedUserDetails}>
                        Emp#: {user.employeeNumber || 'N/A'} | Role: {user.role || 'Operator'}
                      </Text>
                    </View>
                    <TouchableOpacity
                      style={styles.taskModalRemoveUserButton}
                      onPress={() => {
                        setTempAssignTo(prev => prev.filter(u => u.id !== user.id));
                      }}
                    >
                      <MaterialIcons name="close" size={16} color="#ff0000" />
                    </TouchableOpacity>
                  </View>
                ))}
              </View>
            )}
            
            <View style={styles.taskModalDividerCustom} />
            <View style={styles.taskModalRowCustom}>
              <Text style={styles.taskModalLabelCustom}>Note</Text>
              <TextInput
                style={styles.taskModalInputCustom}
                placeholder="Note"
                placeholderTextColor="#888"
                value={note}
                onChangeText={setNote}
              />
            </View>
            <View style={styles.taskModalDividerCustom} />
            {/* <View style={styles.taskModalRowCustom}>
              <Text style={styles.taskModalLabelCustom}>Fault List</Text>
              <TouchableOpacity
                style={styles.taskModalFaultButtonCustom}
                onPress={() => setFaultListModalVisible(true)}
              >
                <Text style={styles.taskModalFaultButtonTextCustom}>
                  {selectedFaults.length > 0 ? `${selectedFaults.length} Selected` : 'Select'}
                </Text>
                <MaterialIcons name="chevron-right" size={width * 0.05} color="#015185" />
              </TouchableOpacity>
            </View> */}
            <View style={styles.taskModalButtonRowCustom}>
              <TouchableOpacity
                style={[styles.taskModalButtonCustom, styles.taskModalCancelButtonCustom]}
                onPress={() => {
                  setAddItemModalVisible(false);
                  setIosModalVisible(false);
                }}
              >
                <Text style={styles.taskModalButtonTextCustom}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.taskModalButtonCustom, styles.taskModalSaveButtonCustom]}
                onPress={saveItem}
              >
                <Text style={styles.taskModalButtonTextCustom}>Save</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      <Modal
        visible={editModalVisible}
        transparent={true}
        animationType="slide"
        onRequestClose={() => setEditModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>Edit Task</Text>
            {/* Task Row */}
            <View style={[styles.rowCentered, { marginBottom: 12 }]}> 
              <Text style={[styles.label, { flex: 1 }]}>Task</Text>
              <TextInput
                style={[styles.input, { flex: 2 }]}
                placeholder="Task description"
                placeholderTextColor="#888"
                value={newItemText}
                onChangeText={setNewItemText}
              />
            </View>
            {/* Type Row */}
            <View style={[styles.rowCentered, { marginBottom: 12 }]}> 
              <Text style={[styles.label, { flex: 1 }]}>Type</Text>
              <View style={{ flex: 2, flexDirection: 'row', gap: 8 }}>
                {taskTypes.map(type => (
                  <TouchableOpacity
                    key={type}
                    style={[
                      styles.typeButton,
                      taskType === type && styles.typeButtonSelected,
                      { minWidth: 80, alignItems: 'center' },
                    ]}
                    onPress={() => setTaskType(type)}
                  >
                    <Text
                      style={[
                        styles.typeText,
                        taskType === type && styles.typeTextSelected,
                      ]}
                    >
                      {type}
                    </Text>
                  </TouchableOpacity>
                ))}
              </View>
            </View>
            {/* Assign To Row */}
            <View style={[styles.rowCentered, { marginBottom: 12 }]}> 
              <Text style={[styles.label, { flex: 1 }]}>Assign to</Text>
              <TouchableOpacity
                style={[styles.selectButton, { flex: 2, justifyContent: 'space-between' }]}
                onPress={() => setAssignToModalVisible(true)}
              >
                <Text style={styles.selectButtonText}>
                  {tempAssignTo.length > 0 ? `${tempAssignTo.length} Assigned` : 'Set'}
                </Text>
                <MaterialIcons name="chevron-right" size={width * 0.05} color="#015185" />
              </TouchableOpacity>
            </View>
            {/* Note Row */}
            <View style={[styles.rowCentered, { marginBottom: 12 }]}> 
              <Text style={[styles.label, { flex: 1 }]}>Note</Text>
              <TextInput
                style={[styles.input, { flex: 2 }]}
                placeholder="Note"
                placeholderTextColor="#888"
                value={note}
                onChangeText={setNote}
              />
            </View>
            {/* Fault List Row */}
            {/* <View style={[styles.rowCentered, { marginBottom: 18 }]}> 
              <Text style={[styles.label, { flex: 1 }]}>Fault List</Text>
              <TouchableOpacity
                style={[styles.selectButton, { flex: 2, justifyContent: 'space-between' }]}
                onPress={() => setFaultListModalVisible(true)}
              >
                <Text style={styles.selectButtonText}>
                  {selectedFaults.length > 0 ? `${selectedFaults.length} Selected` : 'Select'}
                </Text>
                <MaterialIcons name="chevron-right" size={width * 0.05} color="#015185" />
              </TouchableOpacity>
            </View> */}
            {/* Buttons */}
            <View style={[styles.rowCentered, { justifyContent: 'flex-end', marginTop: 10 }]}> 
              <TouchableOpacity
                style={[styles.saveButton, { backgroundColor: '#0078D4', marginRight: 10, minWidth: 90, borderRadius: 4, paddingVertical: 8 }]}
                onPress={() => setEditModalVisible(false)}
              >
                <Text style={[styles.saveButtonText, { color: '#fff', textAlign: 'center' }]}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.saveButton, { minWidth: 90, borderRadius: 4, paddingVertical: 8 }]}
                onPress={editItemSave}
              >
                <Text style={[styles.saveButtonText, { color: '#fff', textAlign: 'center' }]}>Save</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      <Modal
        visible={addLaborModalVisible}
        transparent={true}
        animationType="slide"
        onRequestClose={() => setAddLaborModalVisible(false)}
      >
        <View style={styles.laborModalOverlayCustom}>
          <View style={styles.laborModalContentCustom}>
            <Text style={styles.laborModalTitleCustom}>Add Labor Details</Text>
            <TextInput
              style={styles.laborInputCustom}
              placeholder="Worker Name"
              placeholderTextColor="#888"
              value={laborWorker}
              onChangeText={setLaborWorker}
            />
            <TextInput
              style={styles.laborInputCustom}
              placeholder="Labor Code"
              placeholderTextColor="#888"
              value={laborCode}
              onChangeText={setLaborCode}
            />
            <TextInput
              style={styles.laborInputCustom}
              placeholder="Rate"
              placeholderTextColor="#888"
              value={laborRate}
              onChangeText={setLaborRate}
              keyboardType="numeric"
            />
            <View style={styles.laborHoursRowCustom}>
              <Text style={styles.laborHoursLabelCustom}>Work Hours</Text>
              <TextInput
                style={styles.laborHoursInputCustom}
                placeholder="Hours"
                placeholderTextColor="#888"
                value={laborHours}
                onChangeText={setLaborHours}
                keyboardType="numeric"
              />
              <TextInput
                style={styles.laborHoursInputCustom}
                placeholder="Minut...."
                placeholderTextColor="#888"
                value={laborMinutes}
                onChangeText={setLaborMinutes}
                keyboardType="numeric"
              />
            </View>
            <TextInput
              style={styles.laborNoteInputCustom}
              placeholder="Note"
              placeholderTextColor="#888"
              value={laborNote}
              onChangeText={setLaborNote}
              multiline
              numberOfLines={4}
            />
            <View style={styles.laborModalButtonRowCustom}>
              <TouchableOpacity
                style={[styles.laborModalButtonCustom, styles.laborModalCancelButtonCustom]}
                onPress={() => setAddLaborModalVisible(false)}
              >
                <Text style={styles.laborModalButtonTextCustom}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.laborModalButtonCustom, styles.laborModalSaveButtonCustom]}
                onPress={saveLabor}
              >
                <Text style={styles.laborModalButtonTextCustom}>Save</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      <Modal
        visible={addPartsModalVisible}
        transparent={true}
        animationType="slide"
        onRequestClose={() => setAddPartsModalVisible(false)}
      >
        <View style={styles.partsModalOverlayCustom}>
          <View style={styles.partsModalContentCustom}>
            <Text style={styles.partsModalTitleCustom}>Add Parts</Text>
            <TextInput
              style={styles.partsInputCustom}
              placeholder="Item Number"
              placeholderTextColor="#888"
              value={partsItemNumber}
              onChangeText={setPartsItemNumber}
            />
            <TextInput
              style={styles.partsInputCustom}
              placeholder="Part Number"
              placeholderTextColor="#888"
              value={partsPartNumber}
              onChangeText={setPartsPartNumber}
            />
            <TextInput
              style={styles.partsInputCustom}
              placeholder="Item Name"
              placeholderTextColor="#888"
              value={partsItemName}
              onChangeText={setPartsItemName}
            />
            <TextInput
              style={styles.partsInputCustom}
              placeholder="Quantity"
              placeholderTextColor="#888"
              value={partsQuantity}
              onChangeText={setPartsQuantity}
              keyboardType="numeric"
            />
            <View style={styles.partsUnitContainerCustom}>
              <Text style={styles.partsUnitLabelCustom}>Unit</Text>
            </View>
            <TextInput
              style={styles.partsInputCustom}
              placeholder="Item Cost"
              placeholderTextColor="#888"
              value={partsItemCost}
              onChangeText={setPartsItemCost}
              keyboardType="numeric"
            />
            <TextInput
              style={styles.partsInputCustom}
              placeholder="Site"
              placeholderTextColor="#888"
              value={partsSite}
              onChangeText={setPartsSite}
            />
            <TextInput
              style={styles.partsNoteInputCustom}
              placeholder="Note"
              placeholderTextColor="#888"
              value={partsNote}
              onChangeText={setPartsNote}
              multiline
              numberOfLines={4}
            />
            <View style={styles.partsModalButtonRowCustom}>
              <TouchableOpacity
                style={[styles.partsModalButtonCustom, styles.partsModalCancelButtonCustom]}
                onPress={() => setAddPartsModalVisible(false)}
              >
                <Text style={styles.partsModalButtonTextCustom}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.partsModalButtonCustom, styles.partsModalSaveButtonCustom]}
                onPress={editPartsId ? handleUpdateParts : saveParts}
              >
                <Text style={styles.partsModalButtonTextCustom}>{editPartsId ? 'Update' : 'Add'}</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      <Modal
        visible={addAdditionalCostModalVisible}
        transparent={true}
        animationType="slide"
        onRequestClose={() => setAddAdditionalCostModalVisible(false)}
      >
        <View style={styles.additionalCostModalOverlayCustom}>
          <View style={styles.additionalCostModalContentCustom}>
            <Text style={styles.additionalCostModalTitleCustom}>Add Additional Cost</Text>
            <TextInput
              style={styles.additionalCostDescriptionInputCustom}
              placeholder="Description"
              placeholderTextColor="#888"
              value={additionalCostDescription}
              onChangeText={setAdditionalCostDescription}
              multiline
              numberOfLines={4}
            />
            <TextInput
              style={styles.additionalCostInputCustom}
              placeholder="Select Labor Code"
              placeholderTextColor="#888"
              value={additionalCostAmount}
              onChangeText={setAdditionalCostAmount}
            />
            <View style={styles.additionalCostModalButtonRowCustom}>
              <TouchableOpacity
                style={[styles.additionalCostModalButtonCustom, styles.additionalCostModalCancelButtonCustom]}
                onPress={() => setAddAdditionalCostModalVisible(false)}
              >
                <Text style={styles.additionalCostModalButtonTextCustom}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.additionalCostModalButtonCustom, styles.additionalCostModalSaveButtonCustom]}
                onPress={saveAdditionalCost}
              >
                <Text style={styles.additionalCostModalButtonTextCustom}>Save</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      <Modal
        visible={addRemarksModalVisible}
        transparent={true}
        animationType="slide"
        onRequestClose={() => setAddRemarksModalVisible(false)}
      >
        <View style={styles.remarksModalOverlayCustom}>
          <View style={styles.remarksModalContentCustom}>
            <Text style={styles.remarksModalTitleCustom}>Add Remarks</Text>
            <TextInput
              style={styles.remarksDescriptionInputCustom}
              placeholder="Description"
              placeholderTextColor="#888"
              value={remarksDescription}
              onChangeText={setRemarksDescription}
              multiline
              numberOfLines={4}
            />
            <View style={styles.remarksModalButtonRowCustom}>
              <TouchableOpacity
                style={[styles.remarksModalButtonCustom, styles.remarksModalCancelButtonCustom]}
                onPress={() => setAddRemarksModalVisible(false)}
              >
                <Text style={styles.remarksModalButtonTextCustom}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.remarksModalButtonCustom, styles.remarksModalSaveButtonCustom]}
                onPress={saveRemarks}
              >
                <Text style={styles.remarksModalButtonTextCustom}>Save</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      <Modal
        visible={statusModalVisible}
        transparent={true}
        animationType="slide"
        onRequestClose={() => setStatusModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>Select Status</Text>
            <FlatList
              data={statuses}
              keyExtractor={item => item.label}
              renderItem={({ item }) => (
                <TouchableOpacity
                  style={styles.modalOption}
                  onPress={() => updateItemStatus(selectedItemId, item.label)}
                >
                  <Text style={[styles.modalOptionText, { color: item.color }]}>{item.label}</Text>
                </TouchableOpacity>
              )}
            />
            <TouchableOpacity
              style={styles.modalCloseButton}
              onPress={() => setStatusModalVisible(false)}
            >
              <Text style={styles.modalCloseText}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      <Modal
        visible={faultListModalVisible}
        transparent={true}
        animationType="slide"
        onRequestClose={() => setFaultListModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>Select Faults</Text>
            <FlatList
              data={chunkedFaultListData.flat()}
              keyExtractor={item => item.id}
              renderItem={({ item }) => (
                <TouchableOpacity
                  style={styles.modalOption}
                  onPress={() => handleFaultSelect(item)}
                >
                  <Text style={styles.modalOptionText}>{item.name}</Text>
                  {selectedFaults.some(f => f.id === item.id) && (
                    <MaterialIcons name="check" size={width * 0.05} color="#015185" />
                  )}
                </TouchableOpacity>
              )}
            />
            <TouchableOpacity
              style={styles.modalCloseButton}
              onPress={handleSaveFaults}
            >
              <Text style={styles.modalCloseText}>Save</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      <Modal
        visible={showSuccessModal}
        transparent={true}
        animationType="slide"
        onRequestClose={() => setShowSuccessModal(false)}
        statusBarTranslucent={true}
        presentationStyle="overFullScreen"
      >
        <View style={[styles.successModalOverlayCustom, { zIndex: 9999 }]}>
          <View style={[styles.successModalContentCustom, { zIndex: 10000 }]}>
            <Text style={styles.successModalTitleCustom}>Success</Text>
            <MaterialIcons name="check-circle" size={60} color="#28a745" style={{ marginVertical: 15 }} />
            <Text style={styles.successModalTextCustom}>Job Card Created Successfully!</Text>
            <View style={styles.successModalButtonRowCustom}>
              <TouchableOpacity
                style={[styles.successModalButtonCustom]}
                onPress={async () => {
                  console.log('Back button pressed');
                  setShowSuccessModal(false);
                  // Clean up PDF file
                  if (savedPdfPath) {
                    try {
                      await RNFS.unlink(savedPdfPath);
                    } catch (error) {
                      console.log('PDF cleanup error:', error);
                    }
                  }
                  await fetchInitialData();
                  navigation.reset({
                    index: 1,
                    routes: [
                      { name: 'MainApp' },
                      { name: 'Inspection_Report_Home' },
                    ],
                  });
                }}
              >
                <Text style={styles.successModalButtonTextCustom}>Back</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.successModalButtonCustom]}
                onPress={async () => {
                  console.log('View PDF button pressed');
                  try {
                    await handleViewPDF(true); // Pass a flag to indicate PDF navigation
                    // Only hide modal if PDF viewing is successful
                    setShowSuccessModal(false);
                    await fetchInitialData();
                  } catch (error) {
                    console.error('Error viewing PDF:', error);
                    // Keep modal open if there's an error
                  }
                }}
              >
                <Text style={styles.successModalButtonTextCustom}>View PDF</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>






      {/* Faulty Items (Auto Tasks) preview (commented out as per user request)
      {items.filter(item => item.category === 'Task').length > 0 && (
        <View style={{ marginTop: 20 }}>
          <Text style={{ fontWeight: 'bold', fontSize: 16 }}>Faulty Items (Auto Tasks):</Text>
          {items.filter(item => item.category === 'Task').map((item, idx) => (
            <Text key={idx} style={{ color: 'red', marginVertical: 2 }}>
              {item.text}
            </Text>
          ))}
        </View>
      )} */}

      {/* Start Date Picker Modal for iOS */}
      {Platform.OS === 'ios' && (
        <Modal
          visible={showStartDatePicker}
          transparent
          animationType="slide"
          onRequestClose={() => setShowStartDatePicker(false)}
        >
          <View style={{ flex: 1, justifyContent: 'center', alignItems: 'center', backgroundColor: 'rgba(0,0,0,0.3)' }}>
            <View style={{ backgroundColor: '#fff', borderRadius: 16, padding: 20, width: '85%', alignItems: 'center' }}>
              <DateTimePicker
                value={startDate || new Date()}
                mode="date"
                display="spinner"
                onChange={(event, date) => {
                  if (date) setStartDate(date);
                }}
                style={{ width: '100%' }}
              />
              <View style={{ flexDirection: 'row', justifyContent: 'space-between', width: '100%', marginTop: 16 }}>
                <TouchableOpacity onPress={() => setShowStartDatePicker(false)} style={{ flex: 1, alignItems: 'center' }}>
                  <Text style={{ color: '#dc3545', fontSize: 16 }}>Cancel</Text>
                </TouchableOpacity>
                <TouchableOpacity
                  onPress={() => setShowStartDatePicker(false)}
                  style={{ flex: 1, alignItems: 'center' }}
                >
                  <Text style={{ color: '#015185', fontWeight: 'bold', fontSize: 16 }}>Confirm</Text>
                </TouchableOpacity>
              </View>
            </View>
          </View>
        </Modal>
      )}

      {/* Due Date Picker Modal for iOS */}
      {Platform.OS === 'ios' && (
        <Modal
          visible={showDueDatePicker}
          transparent
          animationType="slide"
          onRequestClose={() => setShowDueDatePicker(false)}
        >
          <View style={{ flex: 1, justifyContent: 'center', alignItems: 'center', backgroundColor: 'rgba(0,0,0,0.3)' }}>
            <View style={{ backgroundColor: '#fff', borderRadius: 16, padding: 20, width: '85%', alignItems: 'center' }}>
              <DateTimePicker
                value={dueDate || new Date()}
                mode="date"
                display="spinner"
                onChange={(event, date) => {
                  if (date) setDueDate(date);
                }}
                style={{ width: '100%' }}
              />
              <View style={{ flexDirection: 'row', justifyContent: 'space-between', width: '100%', marginTop: 16 }}>
                <TouchableOpacity onPress={() => setShowDueDatePicker(false)} style={{ flex: 1, alignItems: 'center' }}>
                  <Text style={{ color: '#dc3545', fontSize: 16 }}>Cancel</Text>
                </TouchableOpacity>
                <TouchableOpacity
                  onPress={() => setShowDueDatePicker(false)}
                  style={{ flex: 1, alignItems: 'center' }}
                >
                  <Text style={{ color: '#015185', fontWeight: 'bold', fontSize: 16 }}>Confirm</Text>
                </TouchableOpacity>
              </View>
            </View>
          </View>
        </Modal>
      )}

      <TouchableOpacity onPress={handleRefresh} style={{position: 'absolute', top: 10, right: 10, zIndex: 100}}>
        <Text style={{color: '#015185', fontWeight: 'bold'}}>Refresh</Text>
      </TouchableOpacity>

      {/* {pdfModalVisible && (
        <Modal visible={pdfModalVisible} transparent animationType="fade">
          <View style={{ flex: 1, backgroundColor: 'rgba(0,0,0,0.3)', justifyContent: 'center', alignItems: 'center' }}>
            <View style={{ backgroundColor: '#fff', borderRadius: 12, padding: 32, alignItems: 'center' }}>
              <ActivityIndicator size="large" color="#015185" />
              <Text style={{ marginTop: 16, color: '#015185', fontWeight: 'bold', fontSize: 16 }}>Generating PDF, please wait...</Text>
            </View>
          </View>
        </Modal>
      )} */}
      <SyncModal visible={isLoading} message="Processing..." />
    </>
  );
};



const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  keyboardAvoidingView: {
    flex: 1,
  },
  // iOS Date Picker Styles
  datePickerModalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.3)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  datePickerModalContent: {
    backgroundColor: '#fff',
    borderRadius: 12,
    padding: 20,
    alignItems: 'center',
    width: 340,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 8,
    elevation: 8,
  },
  datePickerHeader: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    width: '100%',
    marginBottom: 10,
  },
  datePickerTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#015185',
    flex: 1,
    textAlign: 'center',
  },
  datePickerCancelText: {
    color: '#888',
    fontSize: 16,
    paddingHorizontal: 8,
  },
  datePickerDoneText: {
    color: '#015185',
    fontWeight: 'bold',
    fontSize: 16,
    paddingHorizontal: 8,
  },
  datePickerContainer: {
    marginTop: 10,
    alignItems: 'center',
  },

  // iOS Menu Styles
  menuOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  menuContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 10,
    minWidth: 150,
    shadowColor: '#000',
    shadowOffset: {
      width: 0,
      height: 2,
    },
    shadowOpacity: 0.25,
    shadowRadius: 3.84,
    elevation: 5,
  },
  menuItem: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 12,
    paddingHorizontal: 16,
    borderRadius: 6,
  },
  menuItemText: {
    marginLeft: 12,
    fontSize: 16,
    color: '#333',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.05,
    fontWeight: '500',
    marginLeft: width * 0.025,
  },
  saveButton: {
    backgroundColor: '#015185',
  },
  saveButtonText: {
    color: '#FFFFFF',
    fontSize: width * 0.04,
    fontWeight: '600',
  },
  mainContent: {
    flex: 1,
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
  },
  formGroup: {
    marginBottom: width * 0.02,
  },
  totalCountContainer: {
    backgroundColor: '#f8f9fa',
    borderRadius: 8,
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.04,
    marginBottom: width * 0.02,
    borderLeftWidth: 4,
    borderLeftColor: '#015185',
    shadowColor: '#000',
    shadowOffset: {
      width: 0,
      height: 1,
    },
    shadowOpacity: 0.1,
    shadowRadius: 2,
    elevation: 2,
  },
  totalCountContent: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  totalCountLabel: {
    fontSize: width * 0.038,
    color: '#015185',
    fontWeight: '600',
    marginLeft: width * 0.02,
  },
  totalCountValue: {
    fontSize: width * 0.042,
    color: '#0055A5',
    fontWeight: 'bold',
    marginLeft: width * 0.015,
  },
  mainspace0: {
    width: '100%',
    height: 0.1,
    backgroundColor: '#707173',
    color: '#707173',
  },
  formsubdat: {
    marginBottom: 14,
  },
  formGroupdates0: {
    backgroundColor: '#F2F2F2',
    paddingHorizontal: 10,
    paddingVertical: 2,
    borderBottomWidth: 1,
    borderBottomColor: '#707173',
  },
  formGroupdates1: {
    backgroundColor: '#F2F2F2',
    paddingHorizontal: 10,
    paddingVertical: 2,
  },
  dropdownContainer: {
    backgroundColor: '#F2F2F2',
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.03,
  },
  dropdownContaineradditional: {
    backgroundColor: '#F2F2F2',
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.03,
  },
  dropdownContainerCostd: {
    backgroundColor: '#F2F2F2',
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.03,
  },
  dropdownText: {
    fontSize: 16,
    color: '#333',
  },
  placeholderStyle: {
    fontSize: 16,
    color: '#888',
  },
  disabledDropdown: {
    backgroundColor: '#e0e0e0',
  },
  requiredError: {
    color: 'red',
    fontWeight: 'bold',
  },
  inputError: {
    borderColor: 'red',
  },
  errorText: {
    color: 'red',
    fontSize: width * 0.032,
    marginTop: 2,
    marginLeft: 2,
  },
  label: {
     fontSize: width * 0.038,
    color: '#015185',
    marginBottom: 4,
    fontWeight: '500',
  },
  dropdownLabel: {
    fontSize: width * 0.038,
    color: '#015185',
    marginBottom: 4,
    fontWeight: '500',
  },
  input: {
    borderBottomWidth: 1,
    borderColor: '#015185',
    borderRadius: 3,
    paddingVertical: width * 0.015,
    fontSize: width * 0.035,
    color: '#333',
    paddingHorizontal: width * 0.01,
  },
  inputmain: {
     borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    paddingHorizontal: 12,
    paddingVertical: 14,
    fontSize: width * 0.04,
    backgroundColor: '#fff',
  },
  inputSingleLine: {
    backgroundColor: '#fff',
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 3,
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.03,
    fontSize: width * 0.035,
    color: '#333',
  },
  hourInput: {
    flex: 1,
    marginRight: width * 0.01,
  },
  minuteInput: {
    flex: 1,
  },
  quantityInput: {
    flex: 2,
    marginRight: width * 0.01,
  },
  unitContainer: {
    flex: 1,
  },
  unitInput: {
    backgroundColor: '#fff',
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 3,
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.03,
    fontSize: width * 0.035,
    color: '#333',
  },
  multilineInput: {
    height: width * 0.09,
    textAlignVertical: 'top',
  },
  costInput: {
    width: width * 0.3,
    borderWidth: 1,
    borderColor: '#3481BC',
    borderRadius: 3,
    width: '55%',
  },
  taxContainer: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  taxInput: {
    width: width * 0.15,
    marginRight: width * 0.02,
  },
  taxLabel: {
    fontSize: width * 0.035,
    color: '#333',
    marginRight: width * 0.02,
  },
  additionalDetailsContainer: {
    marginTop: width * 0.025,
    paddingHorizontal: width * 0.03,
  },
  tabContainer: {
    marginVertical: width * 0.02,
  },
  tabRow: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  tabButton: {
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.03,
    marginRight: width * 0.02,
    borderBottomWidth: 2,
    borderBottomColor: 'transparent',
  },
  tabButtonActive: {
    borderBottomColor: '#015185',
  },
  tabText: {
    fontSize: width * 0.035,
    color: '#707173',
    fontWeight: '500',
  },
  tabTextActive: {
    color: '#015185',
    fontWeight: '600',
  },
  itemsContainer: {
    //marginVertical: width * 0.025,
    minHeight: width * 0.1,
  },
  itemsFlatList: {
    flexGrow: 0,
  },
  itemsFlatListContent: {
    paddingBottom: width * 0.025,
  },
  itemRow: {
    marginBottom: width * 0.02,
    padding: width * 0.015,
    borderRadius: width * 0.01,
    borderWidth: 1,
    borderColor: '#ddd',
  },
  itemContent: {
    flexDirection: 'column',
  },
  itemHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
  },
  itemText: {
    fontSize: width * 0.035,
    color: '#333',
    fontWeight: '600',
  },
  itemDetails: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    marginTop: width * 0.005,
  },
  itemType: {
    fontSize: width * 0.035,
    color: '#FF0000', // Red to match screenshot
    marginRight: width * 0.02,
  },
  itemNote: {
    fontSize: width * 0.035,
    color: '#707173',
    marginTop: width * 0.005,
  },
  statusButton: {
    paddingVertical: width * 0.01,
    paddingHorizontal: width * 0.02,
    borderRadius: width * 0.01,
  },
  statusButtonText: {
    color: '#fff',
    fontSize: width * 0.03,
    fontWeight: '600',
  },
  threeDotsButton: {
    padding: 4,
  },
  emptyItemsText: {
    fontSize: width * 0.035,
    color: '#888',
    textAlign: 'center',
    marginVertical: width * 0.025,
  },
  addTaskButton: {
  paddingVertical: width * 0.015,
  paddingHorizontal: width * 0.03,
  borderRadius: width * 0.015,
  backgroundColor: '#015185',
  alignSelf: 'flex-start',
  marginBottom: width * 0.04,
},
addTaskText: {
  color: '#fff',
  fontSize: width * 0.035,
  fontWeight: '600',
},
  priorityContainer: {
    flexDirection: 'row',
    justifyContent: 'space-around',
    marginBottom: 15,
    backgroundColor: '#fff',
    borderRadius: 5,
    width: '100%',
  },
  priorityItem: {
    alignItems: 'center',
  },
  priorityButton: {
    padding: 8,
    borderRadius: 5,
    borderWidth: 1,
    borderColor: '#015185',
    width: 85,
    height: 60,
    justifyContent: 'center',
    alignItems: 'center',
  },
  priorityButtonSelected: {
    backgroundColor: '#015185',
  },
  priorityButtonUnselected: {
    backgroundColor: '#FFFFFF',
  },
  priorityImage: {
    width: 24,
    height: 24,
  },
  priorityText: {
    fontSize: 10,
    color: '#333',
    marginTop: 3,
    textAlign: 'center',
  },
  priorityTextSelected: {
    fontSize: 10,
    color: '#fff',
    fontWeight: '600',
    marginTop: 3,
    textAlign: 'center',
  },
  row: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
  },
  rowCentered: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
  },
  assignToContainer: {
    flexDirection: 'column',
  },
  assignToHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
  },
  equipmentContainer: {
    flexDirection: 'column',
  },
  selectedEquipmentText: {
    fontSize: width * 0.035,
    color: '#015185',
    marginTop: width * 0.01,
  },
  selectButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.025,
    borderRadius: width * 0.01,
  },
  selectButtonText: {
    color: '#015185',
    fontSize: width * 0.035,
    marginRight: width * 0.01,
  },
  selectedItemsContainer: {
    marginTop: width * 0.01,
  },
  selectedItemsRow: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  selectedItem: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#e8f5e8',
    borderRadius: width * 0.02,
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.025,
    marginRight: width * 0.02,
    marginBottom: width * 0.01,
    borderWidth: 1,
    borderColor: '#4CAF50',
    minWidth: width * 0.4,
  },
  selectedItemContent: {
    flex: 1,
    marginRight: width * 0.015,
  },
  selectedItemText: {
    fontSize: width * 0.035,
    color: '#015185',
    fontWeight: 'bold',
  },
  selectedItemSubText: {
    fontSize: width * 0.03,
    color: '#666',
    marginTop: width * 0.005,
  },
  noUsersSelected: {
    alignItems: 'center',
    paddingVertical: width * 0.03,
    paddingHorizontal: width * 0.04,
    backgroundColor: '#f8f9fa',
    borderRadius: width * 0.02,
    marginTop: width * 0.01,
    borderWidth: 1,
    borderColor: '#e9ecef',
    borderStyle: 'dashed',
  },
  noUsersSelectedText: {
    fontSize: width * 0.035,
    color: '#6c757d',
    fontWeight: '500',
  },
  noUsersSelectedSubText: {
    fontSize: width * 0.03,
    color: '#adb5bd',
    marginTop: width * 0.005,
    textAlign: 'center',
  },
  closeButton: {
    padding: width * 0.005,
  },
  closeCircle: {
    width: width * 0.04,
    height: width * 0.04,
    borderRadius: (width * 0.04) / 2,
    backgroundColor: '#d32f2f',
    justifyContent: 'center',
    alignItems: 'center',
  },
  dateButton: {
    flexDirection: 'row',
    alignItems: 'center',
    padding: width * 0.03,
  },
  dateText: {
    fontSize: width * 0.035,
    color: '#333',
    marginRight: width * 0.02,
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 22,
    height: 22,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    backgroundColor: '#fff',
    borderRadius: width * 0.025,
    padding: width * 0.05,
    width: '80%',
    maxHeight: '80%',
  },
  modalTitle: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: width * 0.025,
    textAlign: 'center',
  },
  modalOption: {
    paddingVertical: width * 0.025,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
  },
  statusOption: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: width * 0.015,
  },
  statusColor: {
    width: width * 0.03,
    height: width * 0.03,
    borderRadius: (width * 0.03) / 2,
    marginRight: width * 0.02,
  },
  statusOptionText: {
    fontSize: width * 0.035,
    color: '#333',
  },
  optionRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'flex-start',
  },
  optionDetails: {
    flex: 1,
  },
  modalOptionText: {
    fontSize: width * 0.04,
    color: '#333',
  },
  modalOptionSubtext: {
    fontSize: width * 0.035,
    color: '#666',
    marginTop: 2,
  },
  modalOptionSelected: {
    backgroundColor: '#e3f2fd',
    borderColor: '#015185',
  },
  modalDivider: {
    height: 1,
    backgroundColor: '#e0e0e0',
    marginVertical: 10,
  },
  linkedInspectionDetails: {
    backgroundColor: '#f8f9fa',
    padding: 8,
    borderRadius: 4,
    marginTop: 4,
  },
  linkedInspectionText: {
    fontSize: width * 0.03,
    color: '#666',
    fontStyle: 'italic',
  },
  inspectionTaskRow: {
    backgroundColor: '#FFF8F0',
    borderLeftWidth: 3,
    borderLeftColor: '#FF6B35',
  },
  inspectionTaskLabel: {
    fontSize: width * 0.025,
    color: '#FF6B35',
    fontStyle: 'italic',
    marginTop: 2,
    fontWeight: '500',
  },
  modalOptionSubText: {
    fontSize: width * 0.035,
    color: '#707173',
  },
  checkmarkContainer: {
    marginRight: width * 0.03,
  },
  roleContainer: {
    backgroundColor: '#28a745',
    borderRadius: width * 0.02,
    paddingHorizontal: width * 0.03,
    paddingVertical: width * 0.01,
  },
  roleText: {
    fontSize: width * 0.035,
    color: '#fff',
  },
  modalCloseButton: {
    marginTop: width * 0.025,
    alignItems: 'center',
  },
  modalCloseText: {
    fontSize: width * 0.04,
    color: '#0078D4',
    fontWeight: '600',
  },
  modalButtonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-around',
    marginTop: width * 0.025,
  },
  modalButton: {
    paddingVertical: width * 0.02,
    paddingHorizontal: width * 0.05,
    borderRadius: width * 0.02,
    alignItems: 'center',
    flex: 1,
    marginHorizontal: width * 0.01,
  },
  modalTextButton: {
    paddingVertical: width * 0.02,
    paddingHorizontal: width * 0.05,
  },
  modalTextButtonText: {
    fontSize: width * 0.04,
    color: '#015185',
    fontWeight: '600',
    textTransform: 'uppercase',
  },
  cancelButton: {
   backgroundColor: '#fff',
    borderWidth: 1,
    borderColor: '#015185',
  },
  saveButton: {
    backgroundColor: '#015185',
  },
  deleteButton: {
    backgroundColor: '#d32f2f',
  },
  modalButtonText: {
    color: '#015185',
    fontSize: width * 0.04,
    fontWeight: '600',
  },
  defaultText: {
    fontSize: width * 0.045,
    color: '#888',
    textAlign: 'center',
    marginVertical: width * 0.05,
  },
  labelwot: {
    color: 'black',
    fontWeight: '400',
    fontSize: width * 0.032,
    color: '#707173',
    paddingHorizontal: 2,
  },
  wotclono: {
    borderBottomWidth: 1,
    borderColor: '#015185',
    width: '98%',
    height: 1,
    alignSelf: 'center',
    left: -1,
  },
  inputcost0: {
    paddingHorizontal: 10,
    paddingVertical: 5,
    borderRadius: 3,
  },
  inputcost0tax: {
    borderWidth: 1,
    borderColor: '#3481BC',
    paddingVertical: 4,
    paddingHorizontal: 4,
    borderRadius: 3,
  },
  typeContainer: {
    flexDirection: 'row',
    justifyContent: 'flex-start',
    marginBottom: width *    0.01,
  },
  typeButton: {
    paddingVertical: width * 0.01,
    paddingHorizontal: width * 0.03,
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: width * 0.01,
    marginRight: width * 0.02,
  },
  typeButtonSelected: {
    backgroundColor: '#015185',
    borderColor: '#015185',
  },
  typeText: {
    fontSize: width * 0.035,
    color: '#333',
  },
  typeTextSelected: {
    fontSize: width * 0.035,
    color: '#fff',
    fontWeight: '600',
  },
  searchInput: {
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 3,
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.03,
    fontSize: width * 0.035,
    color: '#333',
    marginBottom: width * 0.025,
  },
  updateTaskModal: {
    backgroundColor: '#fff',
    width: '80%',
    borderRadius: 5,
    padding: width * 0.04,
  },
  addTaskTitle: {
    fontSize: width * 0.045,
    color: '#015185',
    textAlign: 'center',
    marginBottom: width * 0.04,
  },
  taskFormGroup: {
    marginBottom: width * 0.03,
  },
  taskLabel: {
    fontSize: width * 0.035,
    color: '#000',
    marginBottom: width * 0.01,
  },
  taskTextInput: {
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 3,
    paddingHorizontal: width * 0.03,
    paddingVertical: width * 0.02,
    fontSize: width * 0.035,
    backgroundColor: '#fff',
  },
  typeButtonContainer: {
    flexDirection: 'row',
    gap: 10,
  },
  repairButtonSelected: {
    backgroundColor: '#FFA500', // Orange/Amber color Orange
    borderColor: '#FFA500',
  },
  assignToButton: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    borderBottomWidth: 1,
    borderColor: '#ddd',
    paddingVertical: width * 0.02,
  },
  assignToButtonText: {
    fontSize: width * 0.035,
    color: '#666',
  },
  assignToArrow: {
    fontSize: width * 0.04,
    color: '#015185',
  },
  taskButtonContainer: {
    flexDirection: 'row',
    justifyContent: 'flex-end',
    gap: 10,
    marginTop: width * 0.04,
  },
  taskCancelButton: {
    backgroundColor: '#6c757d',
    paddingVertical: width * 0.02,
    paddingHorizontal: width * 0.04,
    borderRadius: 3,
  },
  taskSaveButton: {
    backgroundColor: '#015185',
    paddingVertical: width * 0.02,
    paddingHorizontal: width * 0.04,
    borderRadius: 3,
  },
  taskButtonText: {
    color: '#fff',
    fontSize: width * 0.035,
  },
  laborModal: {
    backgroundColor: '#fff',
    padding: width * 0.05,
  },
  laborTitle: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#333',
    marginBottom: width * 0.04,
    textAlign: 'center',
  },
  laborRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    borderBottomWidth: 1,
    borderBottomColor: '#ddd',
    paddingVertical: width * 0.02,
  },
  laborLabel: {
    fontSize: width * 0.035,
    color: '#333',
  },
  laborSelectButton: {
    padding: width * 0.02,
  },
  laborInput: {
    borderBottomWidth: 1,
    borderBottomColor: '#ddd',
    borderRadius: 3,
    paddingVertical: width * 0.02,
    fontSize: width * 0.035,
    color: '#333',
  },
  laborSubLabel: {
    fontSize: width * 0.035,
    color: '#333',
    marginBottom: width * 0.02,
  },
  hoursContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
  },
  hoursInput: {
    flex: 1,
    marginRight: width * 0.02,
  },
  minutesInput: {
    flex: 1,
    marginLeft: width * 0.02,
  },
  partsModal: {
    backgroundColor: '#fff',
    padding: width * 0.05,
  },
  partsTitle: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#333',
    marginBottom: width * 0.04,
    textAlign: 'center',
  },
  partsRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    borderBottomWidth: 1,
    borderBottomColor: '#ddd',
    paddingVertical: width * 0.02,
  },
  partsLabel: {
    fontSize: width * 0.035,
    color: '#333',
  },
  partsSelectButton: {
    padding: width * 0.02,
  },
  partsInput: {
    borderBottomWidth: 1,
    borderBottomColor: '#ddd',
    borderRadius: 3,
    paddingVertical: width * 0.02,
    fontSize: width * 0.035,
    color: '#333',
  },
  quantityContainer: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  quantityInput: {
    flex: 2,
    marginRight: width * 0.02,
  },
  unitSelector: {
    flex: 1,
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    borderBottomWidth: 1,
    borderBottomColor: '#ddd',
    paddingVertical: width * 0.02,
  },
  unitText: {
    fontSize: width * 0.035,
    color: '#333',
  },
  remarksModal: {
    backgroundColor: '#fff',
    padding: width * 0.05,
  },
  remarksTitle: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#333',
    marginBottom: width * 0.04,
    textAlign: 'center',
  },
  remarksInput: {
    borderBottomWidth: 1,
    borderBottomColor: '#ddd',
    borderRadius: 3,
    paddingVertical: width * 0.02,
    fontSize: width * 0.035,
    color: '#333',
  },
  descriptionInput: {
    height: width * 0.2,
    textAlignVertical: 'top',
  },
  typeRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
  },
  assignToRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
  },
  bottomBorderInput: {
    borderWidth: 0,
    borderBottomWidth: 1,
    borderColor: '#ddd',
  },
  assignToModal: {
    backgroundColor: '#fff',
    width: '90%',
    maxHeight: '80%',
    borderRadius: 10,
    padding: 20,
  },
  assignToModalTitle: {
    fontSize: 24,
    color: '#015185',
    fontWeight: '600',
    textAlign: 'center',
    marginBottom: 20,
  },
  searchContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#f5f5f5',
    borderRadius: 5,
    paddingHorizontal: 10,
    marginBottom: 15,
    borderWidth: 1,
    borderColor: '#ccc',
  },
  searchIcon: {
    marginRight: 10,
  },
  assignToSearchInput: {
    flex: 1,
    paddingVertical: 10,
    fontSize: 16,
    color: '#333',
  },
  assignToList: {
    marginBottom: 15,
  },
  assignToOption: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingVertical: 12,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
  },
  assignToUserInfo: {
    flex: 1,
  },
  userIdRow: {
    flexDirection: 'row',
    marginBottom: 4,
  },
  nameRow: {
    flexDirection: 'row',
  },
  userIdLabel: {
    width: 70,
    fontSize: 16,
    color: '#015185',
    fontWeight: '500',
  },
  nameLabel: {
    width: 70,
    fontSize: 16,
    color: '#015185',
    fontWeight: '500',
  },
  userIdValue: {
    fontSize: 16,
    color: '#666',
    marginLeft: 10,
  },
  nameValue: {
    fontSize: 16,
    color: '#666',
    marginLeft: 10,
  },
  technicianBadge: {
    backgroundColor: '#015185',
    paddingHorizontal: 12,
    paddingVertical: 6,
    borderRadius: 5,
    marginLeft: 10,
  },
  technicianText: {
    color: '#fff',
    fontSize: 14,
    fontWeight: '500',
  },
  noResultsText: {
    textAlign: 'center',
    color: '#666',
    fontSize: 16,
    marginTop: 20,
  },
  // selectButton: {
    // backgroundColor: '#015185',
    // paddingVertical: 12,
    // borderRadius: 5,
    // marginTop: 10,
  // },
  // selectButtonText: {
  //   color: '#fff',
  //   fontSize: 16,
  //   fontWeight: '600',
  //   textAlign: 'center',
  // },
  faultListButton: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    borderBottomWidth: 1,
    borderColor: '#ddd',
    paddingVertical: 8,
    marginTop: 5,
  },
  faultListButtonText: {
    fontSize: 14,
    color: '#666',
  },
  faultListModal: {
    backgroundColor: '#fff',
    width: '90%',
    maxHeight: '80%',
    borderRadius: 10,
    padding: 20,
  },
  faultListTitle: {
    fontSize: 20,
    color: '#015185',
    fontWeight: '600',
    marginBottom: 20,
  },
  faultList: {
    marginBottom: 15,
  },
  faultItem: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingVertical: 12,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
  },
  faultItemLeft: {
    flex: 1,
  },
  faultName: {
    fontSize: 16,
    color: '#015185',
    fontWeight: '500',
    marginBottom: 4,
  },
  faultTime: {
    fontSize: 12,
    color: '#666',
  },
  faultItemRight: {
    flexDirection: 'row',
    alignItems: 'center',
    gap: 10,
  },
  faultCode: {
    fontSize: 14,
    color: '#666',
  },
  faultListButtons: {
    flexDirection: 'row',
    justifyContent: 'flex-end',
    gap: 10,
    marginTop: 10,
  },
  faultListActionButton: {
    paddingVertical: 8,
    paddingHorizontal: 20,
    borderRadius: 5,
  },
  closeButton: {
    backgroundColor: '#f0f0f0',
  },
  saveButton: {
    backgroundColor: '#015185',
  },
  saveButtonText: {
    color: '#fff',
  },
  saveButtoncreate: {
    backgroundColor: '#FFFFFF',
    paddingVertical: 8,
    paddingHorizontal: 16,
    borderRadius: 4,
    marginRight: 15,
    marginTop: 5,
  },
  saveButtonText0: {
    color: '#015185',
    fontSize: 14,
    fontWeight: '500',
  },
  loadingModalOverlay: {
    position: 'absolute',
    top: 0,
    left: 0,
    right: 0,
    bottom: 0,
    backgroundColor: 'rgba(255, 255, 255, 0.9)',
    justifyContent: 'center',
    alignItems: 'center',
    zIndex: 9999,
  },
  loadingContainer: {
    backgroundColor: '#FFFFFF',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
    justifyContent: 'center',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.25,
    shadowRadius: 3.84,
    minWidth: 150,
    minHeight: 150,
  },
  spinner: {
    marginBottom: 15,
  },
  loadingText: {
    fontSize: 16,
    color: '#015185',
    fontWeight: '500',
    textAlign: 'center',
  },
  additionalCostModal: {
    backgroundColor: '#FFFFFF',
    width: '90%',
    borderRadius: 5,
    padding: 20,
  },
  additionalCostTitle: {
    color: '#015185',
    fontSize: 18,
    fontWeight: '600',
    marginBottom: 20,
    textAlign: 'center',
  },
  additionalCostInput: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    padding: 12,
    marginBottom: 15,
    color: '#333',
    backgroundColor: '#FFFFFF',
    fontSize: 16,
  },
  descriptionInput: {
    height: 120,
    textAlignVertical: 'top',
    paddingTop: 12,
  },
  laborCodeInput: {
    height: 50,
  },
  additionalCostButtons: {
    flexDirection: 'row',
    justifyContent: 'flex-end',
    gap: 10,
    marginTop: 10,
  },
  additionalCostButton: {
    paddingVertical: 8,
    paddingHorizontal: 20,
    borderRadius: 5,
    minWidth: 80,
  },
  cancelButton: {
    backgroundColor: '#fff',
    borderWidth: 1,
    borderColor: '#015185',
  },
  saveButton: {
    backgroundColor: '#015185',
  },
  cancelButtonText: {
    color: '#FFFFFF',
    fontSize: 14,
    fontWeight: '500',
    textAlign: 'center',
  },
  saveButtonText: {
    color: '#FFFFFF',
    fontSize: 14,
    fontWeight: '500',
    textAlign: 'center',
  },
  successModalContent: {
    backgroundColor: '#FFFFFF',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
    width: '92%',
    maxWidth: width * 0.95,
  },
  successModalTitle: {
    fontSize: 24,
    color: '#015185',
    fontWeight: '600',
    marginBottom: 15,
  },
  successIcon: {
    marginVertical: 15,
  },
  successModalText: {
    fontSize: 18,
    color: '#333',
    marginBottom: 20,
  },
  successModalButtons: {
    flexDirection: 'row',
    justifyContent: 'center',
    gap: 10,
    width: '100%',
  },
  successModalButton: {
    paddingVertical: 5,
    paddingHorizontal: 10,
    borderRadius: 1,
    minWidth: 100,
    alignItems: 'center',
  },
  backButton: {
    backgroundColor: '#015185',
  },
  viewPDFButton: {
    backgroundColor: '#015185',
  },
  successModalButtonText: {
    color: '#FFFFFF',
    fontSize: 16,
    fontWeight: '400',
  },
    requestNumberRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    backgroundColor: '#F2F2F2',
    borderRadius: 6,
    paddingHorizontal: 16,
    paddingVertical: 14,
    marginBottom: width * 0.02,
  },
    requestNumberLabel: {
    fontSize: width * 0.038,
    color: '#015185',
    fontWeight: '500',
  },
    requestNumberValue: {
    fontSize: width * 0.038,
    color: '#025383',
    fontWeight: 'bold',
  },
   scrollContent: {
    flexGrow: 1,
    paddingBottom: height * 0.1,
  },
   keyboardAvoidingView: {
    flex: 1,
  },
  successModalOverlayCustom: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.6)',
    justifyContent: 'center',
    alignItems: 'center',
    zIndex: 9999,
  },
  successModalContentCustom: {
    backgroundColor: '#fff',
    borderRadius: 16,
    padding: 30,
    alignItems: 'center',
    width: '85%',
    maxWidth: 400,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 4 },
    shadowOpacity: 0.3,
    shadowRadius: 12,
    elevation: 10,
    zIndex: 10000,
  },
  successModalTitleCustom: {
    fontSize: 24,
    color: '#015185',
    fontWeight: '700',
    textAlign: 'center',
    marginBottom: 4,
  },
  successModalTextCustom: {
    fontSize: 20,
    color: '#015185',
    marginTop: 2,
    textAlign: 'center',
    fontWeight: '400',
    marginBottom: 2,
  },
  successModalButtonRowCustom: {
    flexDirection: 'row',
    justifyContent: 'center',
    marginTop: 4,
    alignItems: 'center',
  },
  successModalButtonCustom: {
    borderRadius: 4,
    paddingVertical: 8,
    paddingHorizontal: 16,
    marginLeft: 0,
    marginRight: 10,
    minWidth: 70,
    alignItems: 'center',
    backgroundColor: '#015185',
  },
  successModalButtonTextCustom: {
    color: '#fff',
    fontSize: 14,
    fontWeight: '500',
  },
  loadingModalOverlayCustom: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.3)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  loadingModalContentCustomSmall: {
    backgroundColor: '#fff',
    borderRadius: 16,
    paddingVertical: 18,
    paddingHorizontal: 18,
    alignItems: 'center',
    width: 140,
    maxWidth: '90%',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 8,
    elevation: 8,
  },
  loadingIconSmall: {
    width: 38,
    height: 38,
    marginVertical: 8,
    tintColor: '#1A4C8B',
  },
  loaderModalTitleCustomSmall: {
    fontSize: 16,
    color: '#015185',
    fontWeight: '700',
    textAlign: 'center',
    marginTop: 2,
    marginBottom: 2,
  },
  loaderModalTextCustomSmall: {
    fontSize: 13,
    color: '#222',
    marginTop: 2,
    textAlign: 'center',
    fontWeight: '400',
  },
  processorRowCenter: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  processorModalText: {
    fontSize: 16,
    color: '#015185',
    marginLeft: 12,
    textAlign: 'center',
    fontWeight: 'bold',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 22,
    height: 22,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  checklistItemContainer: {
    backgroundColor: '#000',
    borderRadius: 8,
    padding: 12,
    marginBottom: 8,
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 1 },
    shadowOpacity: 0.2,
    shadowRadius: 2,
  },
  checklistItemHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
  },
  checklistItemText: {
    fontSize: 16,
    color: '#fff',
    flex: 1,
  },
  menuButton: {
    padding: 4,
  },
  assignmentBadge: {
    backgroundColor: '#e6f4ea',
    borderWidth: 1,
    borderColor: '#28a745',
    borderRadius: 4,
    padding: 8,
    marginTop: 4,
  },
  assignmentText: {
    color: '#28a745',
    fontSize: 12,
    fontWeight: '500',
  },
  assignmentDate: {
    color: '#666',
    fontSize: 11,
    marginTop: 2,
  },
  disabledAssignmentBadge: {
    backgroundColor: '#f0f0f0',
    borderColor: '#ddd',
  },
  disabledAssignmentText: {
    color: '#999',
  },
  userSelectionModal: {
    backgroundColor: '#fff',
    borderRadius: 8,
    width: '90%',
    maxHeight: '80%',
    padding: 16,
  },
  modalHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 16,
  },
  userItem: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingVertical: 12,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
  },
  userInfo: {
    flex: 1,
  },
  userName: {
    fontSize: 16,
    color: '#333',
    fontWeight: '500',
  },
  userRole: {
    fontSize: 14,
    color: '#666',
    marginTop: 2,
  },
  disabledButton: {
    opacity: 0.5,
  },
  assignmentInfo: {
    marginTop: 8,
    padding: 8,
    backgroundColor: '#f8f9fa',
    borderRadius: 4,
  },
  rightSection: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
  },
  assignedUserText: {
    color: '#28a745',
    fontSize: 12,
    fontWeight: '500',
  },
  notAssignedText: {
    color: '#6c757d',
    fontSize: 12,
    fontWeight: '500',
  },
  dateInputButton: {
    flexDirection: 'row',
    alignItems: 'center',
    padding: width * 0.03,
  },
  dateInputText: {
    fontSize: width * 0.035,
    color: '#333',
    marginRight: width * 0.02,
  },
  laborModalOverlayCustom: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.3)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  laborModalContentCustom: {
    backgroundColor: '#fff',
    borderRadius: 8,
    padding: 24,
    width: '90%',
    maxWidth: 400,
    alignItems: 'stretch',
    elevation: 8,
  },
  laborModalTitleCustom: {
    fontSize: 22,
    color: '#015185',
    fontWeight: '700',
    textAlign: 'center',
    marginBottom: 18,
  },
  laborInputCustom: {
    borderWidth: 2,
    borderColor: '#015185',
    borderRadius: 7,
    paddingVertical: 10,
    paddingHorizontal: 14,
    fontSize: 16,
    marginBottom: 14,
    backgroundColor: '#fff',
  },
  laborHoursRowCustom: {
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: 14,
  },
  laborHoursLabelCustom: {
    fontSize: 15,
    color: '#015185',
    fontWeight: '500',
    marginRight: 10,
    minWidth: 80,
  },
  laborHoursInputCustom: {
    flex: 1,
    borderWidth: 2,
    borderColor: '#015185',
    borderRadius: 7,
    paddingVertical: 10,
    paddingHorizontal: 10,
    fontSize: 16,
    marginRight: 10,
    backgroundColor: '#fff',
  },
  laborNoteInputCustom: {
    borderWidth: 2,
    borderColor: '#015185',
    borderRadius: 7,
    paddingVertical: 10,
    paddingHorizontal: 14,
    fontSize: 16,
    marginBottom: 18,
    backgroundColor: '#fff',
    minHeight: 120,
    textAlignVertical: 'top',
  },
  laborModalButtonRowCustom: {
    flexDirection: 'row',
    justifyContent: 'flex-start',
    marginTop: 10,
  },
  laborModalButtonCustom: {
    borderRadius: 5,
    paddingVertical: 8,
    paddingHorizontal: 16,
    marginLeft: 10,
    minWidth: 70,
    alignItems: 'center',
  },
  laborModalCancelButtonCustom: {
    backgroundColor: '#015185',
  },
  laborModalSaveButtonCustom: {
    backgroundColor: '#015185',
  },
  laborModalButtonTextCustom: {
    color: '#fff',
    fontSize: 14,
    fontWeight: '600',
  },
  additionalCostModalOverlayCustom: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.3)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  additionalCostModalContentCustom: {
    backgroundColor: '#fff',
    borderRadius: 8,
    padding: 24,
    width: '90%',
    maxWidth: 400,
    alignItems: 'stretch',
    elevation: 8,
  },
  additionalCostModalTitleCustom: {
    fontSize: 22,
    color: '#015185',
    fontWeight: '700',
    textAlign: 'center',
    marginBottom: 18,
  },
  additionalCostDescriptionInputCustom: {
    borderWidth: 2,
    borderColor: '#015185',
    borderRadius: 7,
    paddingVertical: 10,
    paddingHorizontal: 14,
    fontSize: 16,
    marginBottom: 18,
    backgroundColor: '#fff',
    minHeight: 100,
    textAlignVertical: 'top',
  },
  additionalCostInputCustom: {
    borderWidth: 2,
    borderColor: '#015185',
    borderRadius: 7,
    paddingVertical: 10,
    paddingHorizontal: 14,
    fontSize: 16,
    marginBottom: 18,
    backgroundColor: '#fff',
  },
  additionalCostModalButtonRowCustom: {
    flexDirection: 'row',
    justifyContent: 'flex-start',
    marginTop: 10,
  },
  additionalCostModalButtonCustom: {
    borderRadius: 5,
    paddingVertical: 8,
    paddingHorizontal: 16,
    marginLeft: 0,
    marginRight: 10,
    minWidth: 70,
    alignItems: 'center',
    backgroundColor: '#015185',
  },
  additionalCostModalCancelButtonCustom: {
    // backgroundColor: '#015185',
  },
  additionalCostModalSaveButtonCustom: {
    // backgroundColor: '#015185',
  },
  additionalCostModalButtonTextCustom: {
    color: '#fff',
    fontSize: 14,
    fontWeight: '600',
  },
  taskModalOverlayCustom: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.3)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  taskModalContentCustom: {
    backgroundColor: '#fff',
    borderRadius: 8,
    padding: 24,
    width: '90%',
    maxWidth: 400,
    alignItems: 'stretch',
    elevation: 8,
  },
  taskModalTitleCustom: {
    fontSize: 22,
    color: '#015185',
    fontWeight: '700',
    textAlign: 'center',
    marginBottom: 18,
  },
  taskModalRowCustom: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    marginBottom: 0,
    paddingVertical: 10,
   
    paddingHorizontal: 0,
  },
  taskModalLabelCustom: {
    fontSize: 16,
    color: '#015185',
    fontWeight: '500',
    minWidth: 90,
  },
  taskModalInputCustom: {
    flex: 1,
    borderWidth: 0,
    borderRadius: 5,
    paddingVertical: 8,
    paddingHorizontal: 10,
    fontSize: 16,
    backgroundColor: '#fff',
    color: '#333',
    textAlign: 'left',
    marginLeft: 10,
  },
  taskModalTypeButtonRowCustom: {
    flexDirection: 'row',
    gap: 8,
  },
  taskModalTypeButtonCustom: {
    backgroundColor: '#e0e0e0',
    borderRadius: 5,
    paddingVertical: 6,
    paddingHorizontal: 18,
    marginLeft: 8,
  },
  taskModalTypeButtonSelectedCustom: {
    backgroundColor: '#015185',
  },
  taskModalTypeButtonTextCustom: {
    color: '#015185',
    fontSize: 15,
    fontWeight: '500',
  },
  taskModalTypeButtonTextSelectedCustom: {
    color: '#fff',
    fontWeight: '600',
  },
  taskModalAssignButtonCustom: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#fff',
    borderRadius: 5,
    paddingVertical: 6,
    paddingHorizontal: 12,
    marginLeft: 10,
  },
  taskModalAssignButtonTextCustom: {
    color: '#015185',
    fontSize: 15,
    fontWeight: '500',
    marginRight: 4,
  },
  taskModalFaultButtonCustom: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#fff',
    borderRadius: 5,
    paddingVertical: 6,
    paddingHorizontal: 12,
    marginLeft: 10,
  },
  taskModalFaultButtonTextCustom: {
    color: '#015185',
    fontSize: 15,
    fontWeight: '500',
    marginRight: 4,
  },
  taskModalDividerCustom: {
    height: 1,
    backgroundColor: '#e0e0e0',
    marginVertical: 0,
  },
  taskModalButtonRowCustom: {
    flexDirection: 'row',
    justifyContent: 'flex-end',
    marginTop: 18,
  },
  taskModalButtonCustom: {
    borderRadius: 5,
    paddingVertical: 8,
    paddingHorizontal: 16,
    marginLeft: 10,
    minWidth: 70,
    alignItems: 'center',
    backgroundColor: '#015185',
  },
  taskModalCancelButtonCustom: {
    // backgroundColor: '#015185',
  },
  taskModalSaveButtonCustom: {
    // backgroundColor: '#015185',
  },
  taskModalButtonTextCustom: {
    color: '#fff',
    fontSize: 14,
    fontWeight: '600',
  },
  remarksModalOverlayCustom: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.3)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  remarksModalContentCustom: {
    backgroundColor: '#fff',
    borderRadius: 8,
    padding: 24,
    width: '90%',
    maxWidth: 500,
    alignItems: 'stretch',
    elevation: 8,
  },
  remarksModalTitleCustom: {
    fontSize: 26,
    color: '#015185',
    fontWeight: '500',
    textAlign: 'center',
    marginBottom: 18,
  },
  remarksDescriptionInputCustom: {
    borderWidth: 2,
    borderColor: '#015185',
    borderRadius: 10,
    paddingVertical: 16,
    paddingHorizontal: 18,
    fontSize: 18,
    marginBottom: 24,
    backgroundColor: '#fff',
    minHeight: 120,
    textAlignVertical: 'top',
  },
  remarksModalButtonRowCustom: {
    flexDirection: 'row',
    justifyContent: 'flex-start',
    marginTop: 10,
  },
  remarksModalButtonCustom: {
    borderRadius: 5,
    paddingVertical: 8,
    paddingHorizontal: 16,
    marginLeft: 0,
    marginRight: 10,
    minWidth: 70,
    alignItems: 'center',
    backgroundColor: '#015185',
  },
  remarksModalCancelButtonCustom: {
    // backgroundColor: '#015185',
  },
  remarksModalSaveButtonCustom: {
    // backgroundColor: '#015185',
  },
  remarksModalButtonTextCustom: {
    color: '#fff',
    fontSize: 14,
    fontWeight: '400',
  },
  loaderModalOverlayCustom: {
    position: 'absolute',
    top: 0,
    left: 0,
    right: 0,
    bottom: 0,
    backgroundColor: 'rgba(0,0,0,0.2)',
    justifyContent: 'center',
    alignItems: 'center',
    zIndex: 9999,
  },
  loaderModalContentCustom: {
    backgroundColor: '#fff',
    borderRadius: 16,
    padding: 32,
    alignItems: 'center',
    width: 140,
    maxWidth: '90%',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 8,
    elevation: 8,
  },
  loaderModalTitleCustom: {
    fontSize: 28,
    color: '#015185',
    fontWeight: '500',
    textAlign: 'center',
    marginBottom: 8,
  },
  loaderModalTextCustom: {
    fontSize: 22,
    color: '#222',
    marginTop: 8,
    textAlign: 'center',
    fontWeight: '400',
  },
  partsModalOverlayCustom: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.3)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  partsModalContentCustom: {
    backgroundColor: '#fff',
    borderRadius: 8,
    padding: 24,
    width: '90%',
    maxWidth: 400,
    alignItems: 'stretch',
    elevation: 8,
  },
  partsModalTitleCustom: {
    fontSize: 22,
    color: '#015185',
    fontWeight: '700',
    textAlign: 'center',
    marginBottom: 18,
  },
  partsInputCustom: {
    borderWidth: 2,
    borderColor: '#015185',
    borderRadius: 7,
    paddingVertical: 10,
    paddingHorizontal: 14,
    fontSize: 16,
    marginBottom: 14,
    backgroundColor: '#fff',
  },
  partsUnitContainerCustom: {
    marginBottom: 14,
  },
  partsUnitLabelCustom: {
    fontSize: 15,
    color: '#015185',
    fontWeight: '500',
    marginBottom: 8,
  },
  partsUnitButtonRowCustom: {
    flexDirection: 'row',
    gap: 8,
  },
  partsUnitButtonCustom: {
    backgroundColor: '#e0e0e0',
    borderRadius: 5,
    paddingVertical: 6,
    paddingHorizontal: 18,
    marginRight: 8,
  },
  partsUnitButtonSelectedCustom: {
    backgroundColor: '#015185',
  },
  partsUnitButtonTextCustom: {
    color: '#015185',
    fontSize: 15,
    fontWeight: '500',
  },
  partsUnitButtonTextSelectedCustom: {
    color: '#fff',
    fontWeight: '600',
  },
  partsNoteInputCustom: {
    borderWidth: 2,
    borderColor: '#015185',
    borderRadius: 7,
    paddingVertical: 10,
    paddingHorizontal: 14,
    fontSize: 16,
    marginBottom: 18,
    backgroundColor: '#fff',
    minHeight: 120,
    textAlignVertical: 'top',
  },
  partsModalButtonRowCustom: {
    flexDirection: 'row',
    justifyContent: 'flex-start',
    marginTop: 10,
  },
  partsModalButtonCustom: {
    borderRadius: 5,
    paddingVertical: 8,
    paddingHorizontal: 16,
    marginLeft: 0,
    marginRight: 10,
    minWidth: 70,
    alignItems: 'center',
    backgroundColor: '#015185',
  },
  partsModalCancelButtonCustom: {
    // backgroundColor: '#015185',
  },
  partsModalSaveButtonCustom: {
    // backgroundColor: '#015185',
  },
  partsModalButtonTextCustom: {
    color: '#fff',
    fontSize: 14,
    fontWeight: '600',
  },
  selectEquipmentButton: {
    backgroundColor: '#015185',
    paddingVertical: 12,
    borderRadius: 5,
    marginTop: 10,
  },
  selectEquipmentButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: '600',
    textAlign: 'center',
    paddingHorizontal: 12,
  },
  // Inline Assign To Styles
  taskModalInlineAssignToContainer: {
    backgroundColor: '#f8f9fa',
    borderRadius: 8,
    marginVertical: 10,
    padding: 12,
    borderWidth: 1,
    borderColor: '#e9ecef',
  },
  taskModalInlineAssignToHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 10,
  },
  taskModalInlineAssignToTitle: {
    fontSize: 16,
    fontWeight: '600',
    color: '#015185',
  },
  taskModalInlineSearchInput: {
    backgroundColor: '#fff',
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 6,
    paddingHorizontal: 12,
    paddingVertical: 8,
    fontSize: 14,
    marginBottom: 10,
  },
  taskModalInlineAssignToList: {
    maxHeight: 200,
  },
  taskModalInlineAssignToItem: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 8,
    paddingHorizontal: 4,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
    backgroundColor: '#fff',
    marginBottom: 2,
    borderRadius: 4,
  },
  taskModalInlineProfilePic: {
    width: 32,
    height: 32,
    borderRadius: 16,
    marginRight: 10,
  },
  taskModalInlineProfilePicPlaceholder: {
    width: 32,
    height: 32,
    borderRadius: 16,
    backgroundColor: '#ddd',
    marginRight: 10,
    alignItems: 'center',
    justifyContent: 'center',
  },
  taskModalInlineUserInfo: {
    flex: 1,
  },
  taskModalInlineUserName: {
    fontWeight: '600',
    color: '#015185',
    fontSize: 14,
  },
  taskModalInlineUserDetails: {
    color: '#666',
    fontSize: 12,
  },
  // Assign To Modal Styles (matching TeamMemberHome)
  assignToModalItem: {
    flexDirection: 'row',
    padding: 12,
    borderBottomWidth: 1,
    borderBottomColor: '#E0E0E0',
    backgroundColor: '#fff',
    marginVertical: 2,
    borderRadius: 8,
    elevation: 1,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 1 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
  },
  assignToModalItemSelected: {
    backgroundColor: '#e6f0ff',
    borderColor: '#015185',
  },
  assignToModalItemLeft: {
    flex: 2,
    flexDirection: 'row',
    paddingRight: 12,
  },
  assignToModalItemRight: {
    flex: 1,
    alignItems: 'flex-end',
    justifyContent: 'space-between',
    flexDirection: 'column',
  },
  assignToModalProfilePic: {
    width: 40,
    height: 40,
    borderRadius: 20,
    marginRight: 10,
  },
  assignToModalProfilePicPlaceholder: {
    width: 40,
    height: 40,
    borderRadius: 20,
    backgroundColor: '#ddd',
    marginRight: 10,
    alignItems: 'center',
    justifyContent: 'center',
  },
  assignToModalUserInfo: {
    flex: 1,
  },
  assignToModalUserName: {
    fontSize: 16,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: 4,
  },
  assignToModalSelfText: {
    color: '#4CAF50',
    fontWeight: 'bold',
  },
  assignToModalUserDetails: {
    fontSize: 13,
    color: '#333',
    marginBottom: 2,
    lineHeight: 18,
  },
  assignToModalRoleBadge: {
    backgroundColor: '#4CAF50',
    paddingVertical: 6,
    paddingHorizontal: 12,
    borderRadius: 12,
    alignItems: 'center',
    marginBottom: 8,
  },
  assignToModalRoleText: {
    fontSize: 12,
    fontWeight: 'bold',
    color: '#fff',
  },
  // Inline Assign To Enhanced Styles
  taskModalInlineAssignToItemSelected: {
    backgroundColor: '#e6f0ff',
    borderColor: '#015185',
  },
  taskModalInlineAssignToItemLeft: {
    flex: 2,
    flexDirection: 'row',
    paddingRight: 8,
  },
  taskModalInlineAssignToItemRight: {
    flex: 1,
    alignItems: 'flex-end',
    justifyContent: 'space-between',
    flexDirection: 'column',
  },
  taskModalInlineSelfText: {
    color: '#4CAF50',
    fontWeight: 'bold',
  },
  taskModalInlineRoleBadge: {
    backgroundColor: '#4CAF50',
    paddingVertical: 4,
    paddingHorizontal: 8,
    borderRadius: 10,
    alignItems: 'center',
    marginBottom: 6,
  },
  taskModalInlineRoleText: {
    fontSize: 10,
    fontWeight: 'bold',
    color: '#fff',
  },
  labelText: {
    fontWeight: 'bold',
    color: '#333',
  },
  equipmentSelectButton: {
    backgroundColor: '#015185',
    paddingVertical: 12,
    borderRadius: 5,
    marginTop: 10,
  },
  equipmentSelectButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: '600',
    textAlign: 'center',
    paddingHorizontal: 12,
  },
  inspectionModal: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
  },
  inspectionInfo: {
    marginBottom: 15,
  },
  inspectionRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 5,
  },
  inspectionLabel: {
    fontSize: 14,
    color: '#333',
    fontWeight: '500',
  },
  inspectionValue: {
    fontSize: 14,
    color: '#666',
  },
  checkmarkContainer: {
    marginLeft: 10,
  },
  inspectionList: {
    maxHeight: 200,
    marginBottom: 15,
  },
  defaultText: {
    fontSize: 14,
    color: '#666',
    textAlign: 'center',
  },
  inspectionOption: {
    paddingVertical: 10,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
  },
  sectionHeader: {
    fontSize: 16,
    color: '#015185',
    fontWeight: 'bold',
    marginBottom: 10,
  },
  taskCard: {
    padding: 10,
    borderWidth: 2,
    borderColor: '#ddd',
    borderRadius: 5,
    marginBottom: 10,
  },
  taskTitle: {
    fontSize: 16,
    fontWeight: 'bold',
    marginBottom: 5,
  },
  taskType: {
    fontSize: 14,
    color: '#666',
    marginBottom: 5,
  },
  taskNote: {
    fontSize: 14,
    color: '#777',
    marginBottom: 5,
  },
  taskStatus: {
    fontSize: 14,
    fontWeight: 'bold',
    color: '#333',
  },
  emptyText: {
    fontSize: 14,
    color: '#999',
    textAlign: 'center',
    marginVertical: 10,
  },
  // Add state for which task's status menu is open
  taskStatusMenuVisibleId: {
    // Add your styles here
  },
  dropdownList: {
    backgroundColor: '#fff',
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#015185',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 4,
    elevation: 5,
  },
  dropdownMenu: {
    position: 'absolute',
    right: 10,
    top: 40,
    backgroundColor: '#fff',
    borderRadius: 6,
    elevation: 4,
    zIndex: 100,
    shadowColor: '#000',
    shadowOpacity: 0.1,
    shadowRadius: 4,
  },
  dropdownMenuItem: {
    padding: 12,
  },
  threeDotsButton: {
    padding: 8,
    marginLeft: 8,
  },
  searchContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#f5f5f5',
    borderRadius: 8,
    paddingHorizontal: 12,
    marginBottom: 15,
    height: 40,
  },
  searchInput: {
    flex: 1,
    fontSize: 14,
    color: '#333',
    paddingVertical: 0,
  },
  
  // Job Card Modal Styles
  searchContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#f8f9fa',
    borderRadius: 8,
    marginBottom: 15,
    paddingHorizontal: 12,
    borderWidth: 1,
    borderColor: '#e9ecef',
  },
  jobCardList: {
    maxHeight: height * 0.5,
    marginBottom: 15,
  },
  jobCardInfo: {
    flex: 1,
  },
  jobCardHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 8,
  },
  jobCardNumber: {
    fontSize: 14,
    color: '#015185',
    fontWeight: 'bold',
    backgroundColor: '#e3f2fd',
    paddingHorizontal: 8,
    paddingVertical: 2,
    borderRadius: 4,
  },
  jobCardDetails: {
    gap: 4,
  },
  jobCardDetail: {
    fontSize: 13,
    color: '#666',
    lineHeight: 18,
  },
  selectedOption: {
    backgroundColor: '#e8f5e8',
    borderColor: '#28a745',
    borderWidth: 1,
  },
  emptyContainer: {
    alignItems: 'center',
    paddingVertical: 40,
  },
  emptyText: {
    fontSize: 16,
    color: '#666',
    marginTop: 10,
    fontWeight: '500',
  },
  emptySubtext: {
    fontSize: 14,
    color: '#999',
    marginTop: 5,
    textAlign: 'center',
  },
  clearButton: {
    backgroundColor: '#dc3545',
  },
  
});

export default ServiceSWorkOrderJob;         

