import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  TextInput,
  ScrollView,
  Dimensions,
  StatusBar,
  Modal,
  FlatList,
  TouchableWithoutFeedback,
  Animated,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation, useRoute } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import DateTimePicker from '@react-native-community/datetimepicker';
import Toast from 'react-native-toast-message';
import NetInfo from '@react-native-community/netinfo';

const { width } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.3;
const MONGODB_SERVICE_SCHEDULE_API_URL = 'https://api.titandrillingzm.com:6010';
const OFFLINE_UPDATE_STORAGE_KEY = 'offlineScheduleUpdates';

const CustomInputField = ({
  label,
  value,
  onChangeText,
  isRequired = false,
  showError = false,
  errorMessage = '',
  style,
  ...props
}) => (
  <View style={[inputStyles.container, style]}>
    <Text style={inputStyles.label}>
      {label}
      {isRequired && <Text style={inputStyles.requiredError}> *</Text>}
    </Text>
    <TextInput
      style={[inputStyles.input, showError ? inputStyles.inputError : null]}
      value={value}
      onChangeText={onChangeText}
      placeholderTextColor="#888"
      {...props}
    />
    {showError && <Text style={inputStyles.errorText}>{errorMessage}</Text>}
  </View>
);

const UNIT_OPTIONS = ['Miles', 'Kilometers', 'Hours'];
const TIME_UNIT_OPTIONS = ['Day', 'Week', 'Month', 'Year'];

const UpdateServiceSchedule = () => {
  const navigation = useNavigation();
  const route = useRoute();
  const { schedule } = route.params || {};

  const parseScheduleDate = (dateStr) => {
    if (!dateStr) return new Date();
    try {
      const [day, month, year] = dateStr.split('-');
      const parsedDate = new Date(`${year}-${month}-${day}`);
      return isNaN(parsedDate.getTime()) ? new Date() : parsedDate;
    } catch (error) {
      console.warn('Invalid date format, defaulting to current date:', error);
      return new Date();
    }
  };

  const [repeatType, setRepeatType] = useState(schedule?.repeatType || 'Repeat');
  const [intervals, setIntervals] = useState(
    schedule?.intervals?.length
      ? schedule.intervals
      : [
          { checked: true, value: '', unit: schedule?.meterUnit || 'Miles' },
          { checked: false, value: '', unit: schedule?.meterUnit || 'Miles' },
        ]
  );
  const [intervalUnit, setIntervalUnit] = useState(schedule?.intervalUnit || 'Day');
  const [notifyBefore, setNotifyBefore] = useState(
    schedule?.notifyBefore || { value: '', unit: schedule?.meterUnit || 'Miles', time: 'Day' }
  );
  const [selectedEquipment, setSelectedEquipment] = useState(schedule?.equipment || null);
  const [scheduleTitle, setScheduleTitle] = useState(schedule?.title || '');
  const [note, setNote] = useState(schedule?.note || '');
  const [lastServiceDate, setLastServiceDate] = useState(
    schedule?.lastServiceDate ? parseScheduleDate(schedule.lastServiceDate) : new Date()
  );
  const [tempDate, setTempDate] = useState(lastServiceDate);
  const [datePickerVisible, setDatePickerVisible] = useState(false);
  const [scheduleId, setScheduleId] = useState(schedule?.scheduleId || '');
  const [showErrors, setShowErrors] = useState(false);
  const [meterReading, setMeterReading] = useState(schedule?.meterReading?.toString() || '');
  const [meterUnit, setMeterUnit] = useState(schedule?.meterUnit || 'Miles');
  const [tasks, setTasks] = useState(schedule?.tasks || []);
  const [taskModalVisible, setTaskModalVisible] = useState(false);
  const [newTask, setNewTask] = useState('');
  const [meterUnitDropdownVisible, setMeterUnitDropdownVisible] = useState(false);
  const [intervalUnitDropdownVisible, setIntervalUnitDropdownVisible] = useState(false);
  const [notifyUnitDropdownVisible, setNotifyUnitDropdownVisible] = useState(false);
  const [intervalDropdownIndex, setIntervalDropdownIndex] = useState(null);
  const [isLoading, setIsLoading] = useState(false);
  const rotateAnim = useRef(new Animated.Value(0)).current;
  const animationRef = useRef(null);

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    const unsubscribeNetInfo = NetInfo.addEventListener((state) => {
      if (state.isConnected && state.isInternetReachable) {
        syncOfflineSchedules();
      }
    });

    // Attempt sync on mount
    syncOfflineSchedules();

    return () => {
      unsubscribeNetInfo();
      if (animationRef.current) {
        animationRef.current.stop();
      }
    };
  }, []);

  const handleEquipmentSelect = (equipmentData) => {
    setSelectedEquipment(equipmentData);
    setMeterReading(equipmentData?.meterReading?.toString() || '0');
    setMeterUnit(equipmentData?.meterUnit || 'Miles');
    setNotifyBefore({ ...notifyBefore, unit: equipmentData?.meterUnit || 'Miles' });
    intervals.forEach((_, idx) => {
      handleUnitSelect(equipmentData?.meterUnit || 'Miles', idx);
    });
  };

  const formatDate = (date) => {
    if (!date || isNaN(date.getTime())) {
      return 'Select Date';
    }
    return date.toLocaleDateString('en-GB', {
      day: '2-digit',
      month: 'short',
      year: 'numeric',
    }).replace(/ /g, '-');
  };

  const openDatePicker = () => {
    setTempDate(lastServiceDate || new Date());
    setDatePickerVisible(true);
  };

  const handleDatePickerChange = (event, selectedDate) => {
    if (event.type === 'set' && selectedDate) {
      setTempDate(selectedDate);
      setLastServiceDate(selectedDate);
    }
    setDatePickerVisible(false);
  };

  const openTaskModal = () => {
    setNewTask('');
    setTaskModalVisible(true);
  };

  const handleTaskConfirm = () => {
    if (newTask.trim()) {
      setTasks([...tasks, newTask.trim()]);
    }
    setTaskModalVisible(false);
    setNewTask('');
  };

  const handleTaskCancel = () => {
    setTaskModalVisible(false);
    setNewTask('');
  };

  const handleUnitSelect = (unit, index) => {
    setIntervals(intervals.map((item, i) => (i === index ? { ...item, unit } : item)));
    setIntervalDropdownIndex(null);
  };

  const handleIntervalUnitSelect = (unit) => {
    setIntervalUnit(unit);
    setIntervalUnitDropdownVisible(false);
  };

  const handleNotifyUnitSelect = (time) => {
    setNotifyBefore({ ...notifyBefore, time });
    setNotifyUnitDropdownVisible(false);
  };

  const handleMeterUnitSelect = (unit) => {
    setMeterUnit(unit);
    setNotifyBefore({ ...notifyBefore, unit });
    intervals.forEach((_, idx) => {
      handleUnitSelect(unit, idx);
    });
    setMeterUnitDropdownVisible(false);
  };

  const showToast = (type, text1, text2) => {
    Toast.show({
      type,
      text1,
      text2,
      position: 'bottom',
      visibilityTime: 3000,
      autoHide: true,
    });
  };

  const storeScheduleOffline = async (scheduleData) => {
    try {
      const offlineData = await AsyncStorage.getItem(OFFLINE_UPDATE_STORAGE_KEY);
      const schedules = offlineData ? JSON.parse(offlineData) : [];
      const filteredSchedules = schedules.filter((s) => s.id !== scheduleData.id);
      filteredSchedules.push(scheduleData);
      await AsyncStorage.setItem(OFFLINE_UPDATE_STORAGE_KEY, JSON.stringify(filteredSchedules));
      showToast('success', 'Success', 'Schedule update stored offline');
      return true;
    } catch (error) {
      console.error('Error storing schedule offline:', error);
      showToast('error', 'Error', 'Failed to store schedule update offline');
      return false;
    }
  };

  const syncOfflineSchedules = async () => {
    try {
      const offlineData = await AsyncStorage.getItem(OFFLINE_UPDATE_STORAGE_KEY);
      if (!offlineData) return;
      const schedules = JSON.parse(offlineData);
      if (schedules.length === 0) return;

      const uid = await AsyncStorage.getItem('userUid');
      if (!uid) {
        console.warn('User ID not found. Cannot sync offline schedules.');
        return;
      }

      for (const schedule of schedules) {
        try {
          const response = await fetch(
            `${MONGODB_SERVICE_SCHEDULE_API_URL}/service-schedules/${schedule.id}`,
            {
              method: 'PUT',
              headers: {
                'Content-Type': 'application/json',
              },
              body: JSON.stringify({ ...schedule, adminUid: uid }),
            }
          );

          if (!response.ok) {
            throw new Error(`Failed to sync schedule ${schedule.id}: ${response.status}`);
          }
        } catch (error) {
          console.error('Error syncing schedule:', error);
          return;
        }
      }

      await AsyncStorage.removeItem(OFFLINE_UPDATE_STORAGE_KEY);
      showToast('success', 'Success', 'Offline schedule updates synced to database');
    } catch (error) {
      console.error('Error syncing offline schedules:', error);
      showToast('error', 'Error', 'Failed to sync offline schedules');
    }
  };

  const handleUpdateSchedule = async () => {
    setShowErrors(true);

    const validIntervals = intervals.filter((interval) => interval.checked && interval.value);
    if (
      !scheduleTitle ||
      !note ||
      !selectedEquipment ||
      !meterReading ||
      isNaN(parseFloat(meterReading)) ||
      !lastServiceDate ||
      isNaN(lastServiceDate.getTime()) ||
      !tasks.length ||
      !repeatType ||
      (repeatType === 'Repeat' && validIntervals.length === 0)
    ) {
      showToast('error', 'Validation Error', 'Please fill all required fields, including at least one interval for repeating schedules.');
      setIsLoading(false);
      return;
    }

    setIsLoading(true);
    animationRef.current = Animated.loop(
      Animated.timing(rotateAnim, {
        toValue: 1,
        duration: 1000,
        useNativeDriver: true,
      })
    );
    animationRef.current.start();

    const scheduleData = {
      id: schedule.id,
      scheduleId,
      country: schedule.country || '',
      project: schedule.project || '',
      title: scheduleTitle,
      equipment: {
        entryNumber: selectedEquipment.entryNumber || '',
        equipmentName: selectedEquipment.equipmentName || 'TYRE - TYRE',
        equipmentNumber: selectedEquipment.equipmentNumber || '',
        meterReading: parseFloat(meterReading) || 0,
        meterUnit,
      },
      note,
      lastServiceDate: formatDate(lastServiceDate),
      meterReading: parseFloat(meterReading) || 0,
      meterUnit,
      repeatType,
      intervals: validIntervals,
      intervalUnit,
      notifyBefore,
      tasks,
      status: schedule.status || 'upcoming',
      createdAt: schedule.createdAt || new Date().toISOString(),
      dueHours: schedule.dueHours || '12',
      hoursToGo: schedule.hoursToGo || '10',
      daysToGo: schedule.daysToGo || '69',
    };

    try {
      const uid = await AsyncStorage.getItem('userUid');
      if (!uid) {
        showToast('error', 'Error', 'User ID not found. Please log in again.');
        setIsLoading(false);
        return;
      }

      const schedulePayload = { ...scheduleData, adminUid: uid };
      const state = await NetInfo.fetch();
      if (state.isConnected && state.isInternetReachable) {
        const response = await fetch(
          `${MONGODB_SERVICE_SCHEDULE_API_URL}/service-schedules/${schedulePayload.id}`,
          {
            method: 'PUT',
            headers: {
              'Content-Type': 'application/json',
            },
            body: JSON.stringify(schedulePayload),
          }
        );

        if (!response.ok) {
          const errorText = await response.text();
          throw new Error(`Failed to update schedule: ${response.status} ${errorText}`);
        }

        showToast('success', 'Success', 'Schedule updated successfully.');
        navigation.replace('ServiceScheduleHome');
      } else {
        const success = await storeScheduleOffline(schedulePayload);
        if (success) {
          showToast('info', 'Offline', 'Schedule update stored offline. Will sync when online.');
          navigation.replace('ServiceScheduleHome');
        }
      }
    } catch (error) {
      console.error('Error updating schedule:', error);
      const uid = await AsyncStorage.getItem('userUid');
      const schedulePayload = {
        ...scheduleData,
        adminUid: uid || '',
      };
      const success = await storeScheduleOffline(schedulePayload);
      if (success) {
        showToast('info', 'Offline', 'Schedule update stored offline due to error. Will sync when online.');
        navigation.replace('ServiceScheduleHome');
      } else {
        showToast('error', 'Error', 'Failed to update schedule.');
      }
    } finally {
      setIsLoading(false);
      if (animationRef.current) {
        animationRef.current.stop();
        animationRef.current = null;
      }
      rotateAnim.setValue(0);
    }
  };

  return (
    <View style={styles.container}>
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity onPress={() => navigation.goBack()}>
              <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>Update Schedule</Text>
          </View>
          <View style={styles.headerRight}>
            <TouchableOpacity style={styles.iconButton} onPress={handleUpdateSchedule}>
              <Text style={styles.createButtonText}>Update</Text>
            </TouchableOpacity>
          </View>
        </View>
      </LinearGradient>

      <ScrollView style={styles.mainContent} keyboardShouldPersistTaps="handled">
        <View style={styles.scheduleIdRow}>
          <Text style={styles.scheduleIdLabel}>Schedule ID</Text>
          <Text style={styles.scheduleIdValue}>{scheduleId}</Text>
        </View>

        <CustomInputField
          label="Schedule Title"
          value={scheduleTitle}
          onChangeText={setScheduleTitle}
          isRequired
          showError={showErrors && !scheduleTitle}
          errorMessage="Schedule Title is required"
        />

        <CustomInputField
          label={<Text>Note<Text style={inputStyles.requiredError}> *</Text></Text>}
          value={note}
          onChangeText={setNote}
          isRequired
          showError={showErrors && !note}
          errorMessage="Note is required"
        />

        <View style={inputStyles.container}>
          <View style={styles.rowBetween}>
            <Text style={inputStyles.label}>
              Equipments
              <Text style={inputStyles.requiredError}> *</Text>
            </Text>
            <TouchableOpacity
              onPress={() =>
                navigation.navigate('Service_select_equipment', {
                  source: 'UpdateServiceSchedule',
                  onSelect: handleEquipmentSelect,
                })
              }
            >
              <Text style={styles.selectText}>Select</Text>
            </TouchableOpacity>
          </View>
          {showErrors && !selectedEquipment && (
            <Text style={inputStyles.errorText}>Equipment is required</Text>
          )}
          {selectedEquipment ? (
            <View>
              <Text style={styles.selectedEquipmentText}>{selectedEquipment.equipmentName}</Text>
              <Text style={styles.selectedEquipmentDetail}>
                Entry Number: {selectedEquipment.entryNumber || 'N/A'}
              </Text>
              <Text style={styles.selectedEquipmentDetail}>
                Meter: {selectedEquipment.meterReading} {selectedEquipment.meterUnit}
              </Text>
            </View>
          ) : (
            <Text style={styles.placeholderText}>No equipment selected</Text>
          )}
          <View style={quantityStyles.row}>
            <CustomInputField
              label={<Text>Last Service Meter Reading<Text style={inputStyles.requiredError}> *</Text></Text>}
              value={meterReading}
              onChangeText={(text) => {
                const sanitizedText = text
                  .replace(/[^0-9.]|^(.*\..*)\./g, '$1')
                  .replace(/(\.\d{2})\d+/, '$1');
                setMeterReading(sanitizedText);
              }}
              keyboardType="numeric"
              isRequired
              showError={showErrors && (!meterReading || isNaN(parseFloat(meterReading)))}
              errorMessage="Valid meter reading is required"
              style={quantityStyles.input}
            />
            <TouchableOpacity
              style={quantityStyles.unitBox}
              onPress={() => setMeterUnitDropdownVisible(true)}
              activeOpacity={0.7}
            >
              <Text style={quantityStyles.unitText}>{meterUnit}</Text>
              <MaterialIcons name="arrow-drop-down" size={22} color="#015185" />
            </TouchableOpacity>
          </View>
        </View>

        <View style={inputStyles.container}>
          <Text style={inputStyles.label}>
            Last Service Date
            <Text style={inputStyles.requiredError}> *</Text>
          </Text>
          <View style={quantityStyles.row}>
            <TouchableOpacity
              style={[quantityStyles.input1, { justifyContent: 'center' }]}
              onPress={openDatePicker}
            >
              <Text
                style={{
                  color: lastServiceDate && !isNaN(lastServiceDate.getTime()) ? '#222' : '#888',
                  fontSize: width * 0.04,
                }}
              >
                {formatDate(lastServiceDate)}
              </Text>
            </TouchableOpacity>
            <TouchableOpacity onPress={openDatePicker}>
              <MaterialIcons name="calendar-today" size={22} color="#015185" style={{ marginLeft: 8 }} />
            </TouchableOpacity>
          </View>
          {showErrors && (!lastServiceDate || isNaN(lastServiceDate.getTime())) && (
            <Text style={inputStyles.errorText}>Last Service Date is required</Text>
          )}
        </View>

        {datePickerVisible && (
          <DateTimePicker
            value={(tempDate && !isNaN(tempDate.getTime()) ? tempDate : new Date())}
            mode="date"
            display="default"
            maximumDate={new Date()}
            onChange={handleDatePickerChange}
          />
        )}

        <View style={inputStyles.container}>
          <View style={styles.taskHeader}>
            <Text style={inputStyles.label}>
              Task
              <Text style={inputStyles.requiredError}> *</Text>
            </Text>
            <TouchableOpacity style={styles.addTaskButton} onPress={openTaskModal}>
              <Text style={styles.addTaskText}>+ Add Task</Text>
            </TouchableOpacity>
          </View>
          {showErrors && !tasks.length && <Text style={inputStyles.errorText}>Task is required</Text>}
          {tasks.length > 0 ? (
            tasks.map((task, idx) => (
              <Text key={idx} style={styles.taskItem}>
                - {task}
              </Text>
            ))
          ) : (
            <Text style={styles.placeholderText}>No tasks added</Text>
          )}
        </View>

        <Modal visible={taskModalVisible} transparent animationType="fade" onRequestClose={handleTaskCancel}>
          <View style={styles.modalOverlay}>
            <View style={styles.taskModalContainer}>
              <Text style={styles.modalTitle}>Add New Task</Text>
              <CustomInputField
                label="Task Name"
                value={newTask}
                onChangeText={setNewTask}
                style={{ width: '100%' }}
              />
              <View style={styles.modalButtons}>
                <TouchableOpacity
                  style={[styles.modalButton, styles.cancelButton]}
                  onPress={handleTaskCancel}
                >
                  <Text style={styles.modalButtonText}>Cancel</Text>
                </TouchableOpacity>
                <TouchableOpacity style={[styles.modalButton, styles.doneButton]} onPress={handleTaskConfirm}>
                  <Text style={styles.modalButtonText}>Done</Text>
                </TouchableOpacity>
              </View>
            </View>
          </View>
        </Modal>

        <View style={inputStyles.container}>
          <View style={[styles.rowBetween, { marginBottom: 0 }]}>
            <Text style={inputStyles.label}>
              Repeat Type
              <Text style={inputStyles.requiredError}> *</Text>
            </Text>
            <View style={styles.repeatTypeRow}>
              <TouchableOpacity
                style={[styles.pillButton, repeatType === 'Repeat' && styles.pillButtonActive]}
                onPress={() => setRepeatType('Repeat')}
              >
                <Text
                  style={[styles.pillButtonText, repeatType === 'Repeat' && styles.pillButtonTextActive]}
                >
                  Repeat
                </Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.pillButton, repeatType === 'One time' && styles.pillButtonActive]}
                onPress={() => setRepeatType('One time')}
              >
                <Text
                  style={[styles.pillButtonText, repeatType === 'One time' && styles.pillButtonTextActive]}
                >
                  One time
                </Text>
              </TouchableOpacity>
            </View>
          </View>
          {showErrors && !repeatType && <Text style={inputStyles.errorText}>Repeat Type is required</Text>}
        </View>

        <View style={inputStyles.container}>
          <Text style={inputStyles.labelSmall}>Intervals (whichever comes first)</Text>
          {intervals.map((interval, idx) => (
            <View key={idx} style={quantityStyles.row}>
              <TouchableOpacity
                style={styles.checkbox}
                onPress={() => {
                  setIntervals(intervals.map((item, i) => (i === idx ? { ...item, checked: !item.checked } : item)));
                }}
              >
                <MaterialIcons
                  name={interval.checked ? 'check-box' : 'check-box-outline-blank'}
                  size={22}
                  color={interval.checked ? '#015185' : '#A0A0A0'}
                />
              </TouchableOpacity>
              <Text style={styles.everyText}>Every</Text>
              <CustomInputField
                label=""
                value={interval.value}
                onChangeText={(text) => {
                  const numericText = text.replace(/[^0-9]/g, '');
                  if (numericText.length <= 10) {
                    setIntervals(intervals.map((item, i) => (i === idx ? { ...item, value: numericText } : item)));
                  }
                }}
                keyboardType="numeric"
                placeholder="0"
                editable={interval.checked}
                maxLength={10}
                style={[quantityStyles.input, { width: 70 }, !interval.checked && inputStyles.inputDisabled]}
              />
              <TouchableOpacity
                style={quantityStyles.unitBox}
                onPress={() => setIntervalDropdownIndex(idx)}
                activeOpacity={0.7}
              >
                <Text style={quantityStyles.unitText}>{interval.unit}</Text>
                <MaterialIcons name="arrow-drop-down" size={22} color="#015185" />
              </TouchableOpacity>
            </View>
          ))}
          {showErrors && repeatType === 'Repeat' && !intervals.some((interval) => interval.checked && interval.value) && (
            <Text style={inputStyles.errorText}>At least one interval is required for repeating schedules</Text>
          )}
          <View style={styles.pillButtonGroup}>
            {['Day', 'Week', 'Month', 'Year'].map((unit) => (
              <TouchableOpacity
                key={unit}
                style={[styles.pillButton, intervalUnit === unit && styles.pillButtonActive]}
                onPress={() => setIntervalUnit(unit)}
              >
                <Text
                  style={[styles.pillButtonText, intervalUnit === unit && styles.pillButtonTextActive]}
                >
                  {unit}
                </Text>
              </TouchableOpacity>
            ))}
          </View>
        </View>

        <Modal
          visible={meterUnitDropdownVisible}
          transparent
          animationType="fade"
          onRequestClose={() => setMeterUnitDropdownVisible(false)}
        >
          <TouchableWithoutFeedback onPress={() => setMeterUnitDropdownVisible(false)}>
            <View style={quantityStyles.dropdownOverlay} />
          </TouchableWithoutFeedback>
          <View style={quantityStyles.dropdownContainer}>
            <FlatList
              data={UNIT_OPTIONS}
              keyExtractor={(item) => item}
              renderItem={({ item }) => (
                <TouchableOpacity
                  style={quantityStyles.dropdownItem}
                  onPress={() => handleMeterUnitSelect(item)}
                >
                  <Text style={quantityStyles.dropdownText}>{item}</Text>
                </TouchableOpacity>
              )}
            />
          </View>
        </Modal>

        <Modal
          visible={intervalDropdownIndex !== null}
          transparent
          animationType="fade"
          onRequestClose={() => setIntervalDropdownIndex(null)}
        >
          <TouchableWithoutFeedback onPress={() => setIntervalDropdownIndex(null)}>
            <View style={quantityStyles.dropdownOverlay} />
          </TouchableWithoutFeedback>
          <View style={quantityStyles.dropdownContainer}>
            <FlatList
              data={UNIT_OPTIONS}
              keyExtractor={(item) => item}
              renderItem={({ item }) => (
                <TouchableOpacity
                  style={quantityStyles.dropdownItem}
                  onPress={() => handleUnitSelect(item, intervalDropdownIndex)}
                >
                  <Text style={quantityStyles.dropdownText}>{item}</Text>
                </TouchableOpacity>
              )}
            />
          </View>
        </Modal>

        <Modal
          visible={intervalUnitDropdownVisible}
          transparent
          animationType="fade"
          onRequestClose={() => setIntervalUnitDropdownVisible(false)}
        >
          <TouchableWithoutFeedback onPress={() => setIntervalUnitDropdownVisible(false)}>
            <View style={quantityStyles.dropdownOverlay} />
          </TouchableWithoutFeedback>
          <View style={quantityStyles.dropdownContainer}>
            <FlatList
              data={TIME_UNIT_OPTIONS}
              keyExtractor={(item) => item}
              renderItem={({ item }) => (
                <TouchableOpacity
                  style={quantityStyles.dropdownItem}
                  onPress={() => handleIntervalUnitSelect(item)}
                >
                  <Text style={quantityStyles.dropdownText}>{item}</Text>
                </TouchableOpacity>
              )}
            />
          </View>
        </Modal>

        <View style={inputStyles.container}>
          <Text style={inputStyles.label}>Notify Before</Text>
          <View style={quantityStyles.row}>
            <CustomInputField
              label=""
              value={notifyBefore.value}
              onChangeText={(text) => setNotifyBefore({ ...notifyBefore, value: text.replace(/[^0-9]/g, '') })}
              keyboardType="numeric"
              placeholder="0"
              style={[quantityStyles.input, { width: 70 }]}
            />
            <TouchableOpacity
              style={quantityStyles.unitBox}
              onPress={() => setMeterUnitDropdownVisible(true)}
              activeOpacity={0.7}
            >
              <Text style={quantityStyles.unitText}>{notifyBefore.unit}</Text>
              <MaterialIcons name="arrow-drop-down" size={22} color="#015185" />
            </TouchableOpacity>
          </View>
          <View style={styles.pillButtonGroup}>
            {['Day', 'Week', 'Month'].map((unit) => (
              <TouchableOpacity
                key={unit}
                style={[styles.pillButton, notifyBefore.time === unit && styles.pillButtonActive]}
                onPress={() => setNotifyBefore({ ...notifyBefore, time: unit })}
              >
                <Text
                  style={[styles.pillButtonText, notifyBefore.time === unit && styles.pillButtonTextActive]}
                >
                  {unit}
                </Text>
              </TouchableOpacity>
            ))}
          </View>
          <Text style={styles.infoText}>
            The schedule will be marked as 'Due soon' once any of the above threshold is crossed.
          </Text>
        </View>

        <Modal
          visible={notifyUnitDropdownVisible}
          transparent
          animationType="fade"
          onRequestClose={() => setNotifyUnitDropdownVisible(false)}
        >
          <TouchableWithoutFeedback onPress={() => setNotifyUnitDropdownVisible(false)}>
            <View style={quantityStyles.dropdownOverlay} />
          </TouchableWithoutFeedback>
          <View style={quantityStyles.dropdownContainer}>
            <FlatList
              data={['Day', 'Week', 'Month']}
              keyExtractor={(item) => item}
              renderItem={({ item }) => (
                <TouchableOpacity
                  style={quantityStyles.dropdownItem}
                  onPress={() => handleNotifyUnitSelect(item)}
                >
                  <Text style={quantityStyles.dropdownText}>{item}</Text>
                </TouchableOpacity>
              )}
            />
          </View>
        </Modal>

        <Modal visible={isLoading} transparent animationType="fade">
          <View style={styles.syncModalContainer}>
            <View style={styles.syncModalContent}>
              <Text style={styles.syncModalText}>Please Wait</Text>
              <View style={styles.syncIconDiv}>
                <Text style={styles.syncModalTextAdd}>Updating Schedule</Text>
              </View>
            </View>
          </View>
        </Modal>

        <Toast />
      </ScrollView>
    </View>
  );
};

const inputStyles = StyleSheet.create({
  container: {
    marginBottom: width * 0.04,
  },
  label: {
    fontSize: width * 0.038,
    color: '#015185',
    marginBottom: 4,
    fontWeight: '500',
  },
  requiredError: {
    color: 'red',
    fontWeight: 'bold',
  },
  input: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    paddingHorizontal: 12,
    paddingVertical: 14,
    fontSize: width * 0.04,
    backgroundColor: '#fff',
  },
  inputError: {
    borderColor: 'red',
  },
  inputDisabled: {
    color: '#A0A0A0',
    borderColor: '#B0C4DE',
  },
  errorText: {
    color: 'red',
    fontSize: width * 0.032,
    marginTop: 2,
    marginLeft: 2,
  },
});

const quantityStyles = StyleSheet.create({
  row: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  input: {
    flex: 1,
    marginRight: 8,
  },
  input1: {
    flex: 1,
    marginRight: 8,
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    paddingHorizontal: 12,
    paddingVertical: 14,
    fontSize: width * 0.04,
    backgroundColor: '#fff',
  },
  unitBox: {
    flexDirection: 'row',
    alignItems: 'center',
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    paddingHorizontal: 12,
    paddingVertical: 14,
    backgroundColor: '#fff',
    minWidth: 70,
    justifyContent: 'space-between',
  },
  unitText: {
    color: '#015185',
    fontSize: width * 0.038,
    fontWeight: '500',
    marginRight: 2,
  },
  dropdownOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.2)',
  },
  dropdownContainer: {
    position: 'absolute',
    top: width * 0.5,
    right: width * 0.12,
    backgroundColor: '#fff',
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#015185',
    width: 120,
    zIndex: 1000,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 4,
    elevation: 5,
  },
  dropdownItem: {
    paddingVertical: 12,
    paddingHorizontal: 16,
  },
  dropdownText: {
    color: '#015185',
    fontSize: width * 0.038,
  },
});

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.05,
    fontWeight: 'bold',
    marginLeft: width * 0.025,
  },
  iconButton: {
    padding: width * 0.019,
    backgroundColor: '#fff',
    borderRadius: width * 0.01,
  },
  createButtonText: {
    color: '#015185',
    fontSize: width * 0.04,
    fontWeight: '500',
  },
  mainContent: {
    flex: 1,
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
  },
  rowBetween: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 12,
  },
  row: {
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: 12,
  },
  labelSmall: {
    color: '#015185',
    marginTop: 10,
    marginBottom: 6,
    fontSize: width * 0.034,
  },
  selectText: {
    fontSize: width * 0.034,
    color: '#025383',
  },
  everyText: {
    color: '#015185',
    fontSize: 15,
    marginHorizontal: 8,
  },
  checkbox: {
    marginRight: 4,
  },
  taskHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 6,
  },
  addTaskButton: {
    backgroundColor: '#015185',
    borderRadius: 6,
    paddingHorizontal: 18,
    paddingVertical: 8,
  },
  addTaskText: {
    color: '#fff',
    fontSize: width * 0.034,
  },
  repeatTypeRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    width: 180,
  },
  pillButtonGroup: {
    flexDirection: 'row',
    justifyContent: 'flex-start',
    alignItems: 'center',
    marginTop: 4,
    marginBottom: 8,
  },
  pillButton: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    paddingHorizontal: 18,
    paddingVertical: 7,
    marginRight: 8,
    backgroundColor: '#fff',
  },
  pillButtonActive: {
    backgroundColor: '#015185',
  },
  pillButtonText: {
    color: '#015185',
    fontSize: width * 0.034,
    fontWeight: '500',
  },
  pillButtonTextActive: {
    color: '#fff',
  },
  infoText: {
    color: '#015185',
    fontSize: 13,
    marginTop: 10,
  },
  selectedEquipmentText: {
    fontSize: 15,
    color: '#015185',
    marginBottom: 4,
    fontWeight: 'bold',
  },
  selectedEquipmentDetail: {
    fontSize: 14,
    color: '#015185',
    marginBottom: 4,
  },
  placeholderText: {
    fontSize: 15,
    color: '#A0A0A0',
    marginBottom: 12,
  },
  modalOverlay: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: 'rgba(0,0,0,0.5)',
  },
  datePickerContainer: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  datePickerButtons: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    width: '100%',
    marginTop: 10,
  },
  datePickerButton: {
    flex: 1,
    borderRadius: 5,
    paddingVertical: 10,
    alignItems: 'center',
    marginHorizontal: 5,
  },
  taskModalContainer: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
    width: '80%',
  },
  modalTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#025383',
    marginBottom: 15,
  },
  modalButtons: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    width: '100%',
  },
  modalButton: {
    flex: 1,
    borderRadius: 5,
    paddingVertical: 10,
    alignItems: 'center',
    marginHorizontal: 5,
  },
  cancelButton: {
    backgroundColor: '#A0A0A0',
  },
  doneButton: {
    backgroundColor: '#015185',
  },
  modalButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: 'bold',
  },
  taskItem: {
    fontSize: 15,
    color: '#333',
    marginBottom: 4,
  },
  scheduleIdRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    backgroundColor: '#F2F2F2',
    borderRadius: 6,
    paddingHorizontal: 16,
    paddingVertical: 14,
    marginBottom: width * 0.04,
  },
  scheduleIdLabel: {
    fontSize: width * 0.038,
    color: '#015185',
    fontWeight: '500',
  },
  scheduleIdValue: {
    fontSize: width * 0.038,
    color: '#025383',
    fontWeight: 'bold',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIconDiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  syncModalTextAdd: {
    fontSize: 14,
    color: '#333',
  },
});

export default UpdateServiceSchedule;